{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

module Epoll

where
import Chunk
import System
import System.IO
import System.Exit
import System.Posix hiding (Stop)

type EpollDevice = CInt
type EpollFD = CInt
data EPOLL_EVENT = EPOLL_READ | EPOLL_WRITE deriving (Eq,Show);

-----------------------------------------------------------------------------
-- Initialize some global stuff
-----------------------------------------------------------------------------
epoll_setup :: IO ()
epoll_setup = 
  do hSetBuffering stdout NoBuffering
     installHandler sigPIPE Ignore Nothing
     return ()

-----------------------------------------------------------------------------
-- Creates an epoll device
-- On failure: terminate
-----------------------------------------------------------------------------
epoll_create :: Int -> IO EpollDevice
epoll_create size = 
  c_epoll_create (fromIntegral size)

foreign import ccall unsafe "mod_epoll.h c_epoll_create" 
  c_epoll_create :: CInt -> IO CInt

-----------------------------------------------------------------------------
-- Add a file descriptor to an epoll device
-- An arbitrary Haskell memory object can be registered with the file descriptor
-- On failure: program will terminate
-----------------------------------------------------------------------------
epoll_add  :: EpollDevice -> EpollFD -> EPOLL_EVENT -> a -> IO ()
epoll_add dev fd ev ref =
  do sp <- newStablePtr (fd,ref)
     let p = castStablePtrToPtr sp
     let isread = if ev == EPOLL_READ then 1 else 0
     c_epoll_add dev fd isread p

foreign import ccall unsafe "mod_epoll.h c_epoll_add" 
  c_epoll_add :: CInt -> CInt -> CInt -> Ptr () -> IO ()


-----------------------------------------------------------------------------
-- Delete a file descriptor from an epoll device
-- On failure: program will terminate
-----------------------------------------------------------------------------
epoll_del  :: EpollDevice -> EpollFD -> IO ()
epoll_del = c_epoll_del
foreign import ccall unsafe "mod_epoll.h c_epoll_del" 
   c_epoll_del :: CInt -> CInt -> IO ()


-----------------------------------------------------------------------------
-- Wait for some epoll events
-- Input:  -1: wait forever
  --        0: return immediately
  --        x: wait at most x milliseconds
-- Returns: a list of registered objects
-- On error: program will terminate
-----------------------------------------------------------------------------
epoll_wait :: EpollDevice -> Int -> IO [a]
epoll_wait dev timeout =
  do res <- c_epoll_wait dev (fromIntegral timeout)
     cnt <- c_epoll_wait_getcnt res
     let handle idx = do {
        p <- c_epoll_wait_getevt res idx;
        let sp = castPtrToStablePtr p
        ;
        (fd,ref) <- deRefStablePtr sp;
        freeStablePtr sp;
        epoll_del dev fd;
        return ref;
      }
     results <- mapM handle [0..(cnt-1)]
     free res
     return results

foreign import ccall safe "mod_epoll.h c_epoll_wait" 
   c_epoll_wait :: CInt -> CInt  -> IO (Ptr())
foreign import ccall unsafe "mod_epoll.h c_epoll_wait_getcnt" 
   c_epoll_wait_getcnt :: Ptr () -> IO (CInt)
foreign import ccall unsafe "mod_epoll.h c_epoll_wait_getevt" 
   c_epoll_wait_getevt :: Ptr () -> CInt -> IO (Ptr())
