{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

module Chunk 
( module Chunk
, module Foreign
, module Foreign.C
)
where

import Foreign.C
import Foreign

--
-- Invariant: offset + left = size of buffer
--
data Chunk =
  Chunk
  { chk_pointer :: ForeignPtr CChar  -- a foreign pointer to the data
  , chk_offset  :: !Int                 -- an offset into this data
  , chk_left    :: !Int                 -- number of chars left in the data
  }

instance Show Chunk where
  show (Chunk fptr off len) = "Chunk("++(show off)++","++(show len)++")"

new_chunk :: Int -> IO Chunk
new_chunk n = 
  do ptr  <- mallocArray n
     fptr <- newForeignPtr finalizerFree ptr
     return $ Chunk fptr 0 n

-- This function is not portable, but it allows us to use Linux AIO...
foreign import ccall "static stdlib.h" 
  memalign :: CInt -> CInt -> IO (Ptr CChar)

-- This alignment works on Linux x86
chunk_alignment = 512

-- To use Linux AIO, the buffer must be aligned.
-- The following function allocates such a buffer
new_aligned_chunk :: Int -> IO Chunk
new_aligned_chunk n = 
  do ptr <- memalign (fromIntegral chunk_alignment) (fromIntegral n)
     fptr <- newForeignPtr finalizerFree ptr
     return $ Chunk fptr 0 n

chunk_size :: Chunk -> Int
chunk_size chk = (chk_left chk) + (chk_offset chk)

chunk_to_string :: Chunk -> IO String
chunk_to_string (Chunk fptr off left) =
  withForeignPtr fptr
    (\ptr ->
        do arr <- peekArray left (ptr `plusPtr` off)
           return $ map castCCharToChar arr
    )

-- Convert a Haskell string to a C byte array.
-- The C byte array cannot be used as a C string 
-- (No NULL is added at the end of the buffer)
string_to_chunk :: String -> IO Chunk
string_to_chunk s =
  do (ptr,n) <- newCStringLen s
     fptr    <- newForeignPtr finalizerFree ptr
     return (Chunk fptr 0 n)


-- Convert a Haskell string to a C string, 
-- adding a NULL character at the end of the C string.
string_to_chunk_withnull :: String -> IO Chunk
string_to_chunk_withnull s =
  do ptr <- newCString s
     fptr    <- newForeignPtr finalizerFree ptr
     return (Chunk fptr 0 (length s + 1))
