{--
Copyright (c) 2006, Peng Li
              2006, Stephan A. Zdancewic
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

* Redistributions of source code must retain the above copyright
  notice, this list of conditions and the following disclaimer.

* Redistributions in binary form must reproduce the above copyright
  notice, this list of conditions and the following disclaimer in the
  documentation and/or other materials provided with the distribution.

* Neither the name of the copyright owners nor the names of its
  contributors may be used to endorse or promote products derived from
  this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
--}

import System
import System.IO
import System.Posix hiding (Stop)
import Data.List
import Data.IORef
import Data.Maybe
import System.Random
import Numeric
import Control.Monad
import Control.Monad.Identity
import Control.Monad.ST
import GHC.Prim


data Trace = 
        SYS_NBIO (IO Trace)
      | SYS_HALT

newtype CPSMonad a = CPSMonad ((a-> Trace)-> Trace)
instance Monad CPSMonad where
   m >>= f = cps_bind m f
   return  = cps_return
   {-# INLINE (>>=) #-}
   {-# INLINE return #-}

cps_bind :: CPSMonad a -> (a -> CPSMonad b) -> CPSMonad b
cps_bind (CPSMonad x) f = CPSMonad(\y-> x (\z-> let CPSMonad f' = f z in f' y))
cps_return :: a -> CPSMonad a
cps_return x = CPSMonad(\c -> c x)
{-# INLINE cps_bind #-}
{-# INLINE cps_return #-}

sys_nbio :: IO a -> CPSMonad a
sys_nbio f = 
    CPSMonad $ \c -> SYS_NBIO( do x <- f; return (c x) )
{-# INLINE sys_nbio #-}

sys_halt :: CPSMonad ()
sys_halt = 
    CPSMonad $ \c -> SYS_HALT

sys_yield :: CPSMonad ()
sys_yield = 
    CPSMonad $ \c -> SYS_NBIO (return $ c ())

thread_to_trace :: (CPSMonad a) -> Trace
thread_to_trace (CPSMonad f) = f (\c-> SYS_HALT)

run_cps :: CPSMonad a -> IO ()
run_cps c = run_trace (thread_to_trace c)

run_trace :: Trace -> IO ()
run_trace t = 
  case t of
    SYS_NBIO iot -> do t2 <- iot
                       run_trace t2
    SYS_HALT -> return ()

--------------------------------------------------------------------

{-# INLINE fibmonad #-}
{-# INLINE fib #-}

fib :: Int -> Int
fib x = 
  if x < 3 then 1 else
     (fib (x-1)) + (fib  (x-2))

fibmonad :: Monad m => Int -> m Int
fibmonad x 
  | x `seq` False = undefined
  | x < 3 = return 1
  | otherwise = do
         a <- fibmonad (x-1)
         b <- fibmonad (x-2)
         return $! (a+b)


{-# SPECIALIZE fibmonad :: Int -> IO Int #-}
{-# SPECIALIZE fibmonad :: Int -> Identity Int #-}
{-# SPECIALIZE fibmonad :: Int -> ST RealWorld Int #-}
{-# SPECIALIZE fibmonad :: Int -> CPSMonad Int #-}

sump :: Int -> Int
sump x
  | x `seq` False = undefined
  | x == 0 = 0
  | otherwise = x + sump(x-1)
  
summonad :: Monad m => Int -> m Int
summonad x
  | x `seq` False = undefined
  | x == 0 = return 0
  | otherwise = do
      y <- summonad (x-1)
      return $! x+y

{-# SPECIALIZE summonad :: Int -> IO Int #-}
{-# SPECIALIZE summonad :: Int -> CPSMonad Int #-}

main = do 
  args <- getArgs
  when (length args /= 2) $ do
       putStrLn $ "Usage: monads task_num arg"
       undefined
  let arg = read $ args !! 1
  case read $ args !! 0 of
    1 -> do print "pure fib:"
            print $ fib arg
    2 -> do print "IO fib"
            x <- fibmonad arg; 
            print x
    3 -> do print "Identity fib"
            print (runIdentity $ fibmonad arg)
    4 -> do print "ST fib"
            let c :: ST RealWorld Int = fibmonad arg
            x <- stToIO c
            print x
    5 -> do print "CPSMonad fib"
            run_cps $ do x <- fibmonad arg; sys_nbio $ print x; sys_halt

    10 -> do print "pure sum"
             print $ sump $ arg*1000
    11 -> do print "IO sum"
             x <- summonad $ arg*1000
             print x
    12 -> do print "CPSMonad sum"
             run_cps $ do x <- summonad (arg * 1000); sys_nbio $ print x; sys_halt
