/*
 * simsys.c - top level C file for simulating pu16/pu17 after vtoc generation
 * (C) TenisonTech 1998.
 */
#include <stdio.h>
#include <stdlib.h>

#define uchar unsigned char

#define memsize 65536

uchar memory[memsize];

static FILE *fd;
int quietf = 1;
int steps;                    /* remaining cycles to simulate */
int break_address = 0x10000;  /* address to trap */


/*
 *  Read a hex byte
 */
int iread(len)
{
  int r = 0;
  int i;
  for (i=0; i < len; i++)
  {
    unsigned char c = getc(fd);
    c = (c <= '9') ? c-'0': c-('0'+7);
    r = (r<<4) + c;
  }
  return r;
}

/*
 * Load intel hex file into simulated RAM...
 */
int hexload(char *fn)
{
  int line = 1;
  int addr = -1;
  int bytes = 0;  
  int hwm = 0;
   
  fd = fopen(fn, "r");
  if (fd == 0) 
    {
      printf("%s ", fn);
      perror("cant open input file\n");
      exit(1);
    }
  while (feof(fd)==0)
  {
    uchar c = getc(fd);
    int llen, ltype;
    uchar lsum;
    int wq;

    if (c == '\n') line++;
    if (c != ':') continue;
    llen = iread(2);
    addr = iread(4);
    ltype = iread(2);
/*
    printf("Type %x  addr %x len %x \n", ltype, laddr, llen);
*/
    if (ltype != 0) 
    { 
      printf("Type %i encountered in line %i\n", ltype, line);
      continue;
    }

    lsum = llen + addr + ltype + (addr >> 8);
    for (wq=0; wq<llen; wq++)
    {
      uchar data = iread(2);
      lsum = lsum + data;
      memory[addr+wq] = data;
      bytes += 1;
    }
    if (addr+wq > hwm) hwm = addr+wq;

    lsum = lsum + iread(2);

    if (lsum != 0) 
          fprintf(stderr, "Checksum error in line %i, residue %i\n", line, lsum);    

    addr = addr + llen;
  }
  return hwm;
}


/*
 * Simulate keyboard input...
 */
char get_uart_char()
{
  static char *p = "mdw 104 5\nhelp\nmm b 6000\n21\n22\n.\nmd 6000\n";
  if (!quietf) printf("URead byte %c\n", *p);

  if (*p) return *p++;

  else 
    {
      breakk("end of input");
      return 13;  /* carriage return */
    }
}

/*
 *
 */

unsigned int abus16, dbus16_in, dbus16_out, clk, reset, opreq, irq, rwbar, wbyte, waitb;

void breakk(char *s)
{
  quietf = 0;
  steps = 6;
  printf("%s\n", s);
}

void step()
{
  int vd = 1-quietf; /* verbose */
  pu17core( &abus16, &dbus16_in, &dbus16_out, &clk, &reset, &opreq, &irq, &rwbar,
	    &wbyte, &waitb);
  

  if (abus16 == break_address) breakk("breakpoint/trap address hit");

  if (abus16 < 0xF800) /* if in memory */
    {
      dbus16_in = memory[abus16 & 0xFFFE] | (memory[(abus16 & 0xFFFE)+1]<<8);
      if (opreq && rwbar == 0)
	{

	  if (abus16 < 8192) 
	    {
	      breakk("Write to ROM");
	      }
	  if (wbyte) 
	    {
	      if (vd) printf("Wrote byte %X to address %04X\n", dbus16_out, abus16);
	      memory[abus16] = (abus16 & 1) ? dbus16_out >> 8: dbus16_out & 0xFF;
	    }
	  else
	    {
	      if (vd) printf("Wrote word %X to address %04X\n", dbus16_out, abus16);
	      memory[abus16] =  dbus16_out & 0xFF;
	      memory[abus16 | 1] =  dbus16_out >> 8;
	    }
	  
	}
    }
  else /* io space */
    {
      if (abus16 == 0xF880) dbus16_in = 3;  /* uart status register */
      else if (abus16 == 0xF882 && opreq && rwbar == 0)  /* write uart data register */
	{
	  if (quietf)
	    {
	      char c = (dbus16_out & 0xFF);
	      if (c != (char) 13) putchar(c);
	    }	 
	  else printf("UART byte %c\n", dbus16_out & 0xFF);
	}

      else if (abus16 == 0xF882 && opreq && rwbar == 1)  /* read uart data register */
	{
	  dbus16_in = get_uart_char();
	}
    }

  if (vd) printf("Addr=%04X DIN=%04X DOUT=%04X OPREQ=%i RWBAR=%i WBYTE=%i\n",
	 abus16, dbus16_in, dbus16_out, opreq, rwbar, wbyte); 
}



void coredump()
{
  int b = 0;
  int skip = 0;
  while (b < memsize)
  {
    int i;
    uchar *x = memory+b;

    for (i=0; i<16; i++) if (x[i]) break;
    if (i < 16)
      {
	printf("%X  ", b);
	for (i=0; i<16; i++)
	  {
	    printf("%02X ", x[i]);
	    if ((i & 3) == 3) printf(" ");
	  }
	printf("  ");
	for (i=0; i<16; i++)
	  {
	    uchar t = x[i];
	    
	    if (t <32 || t >= 127) t = '.';
	    printf("%c", t);
	  }
	skip = 0;
	printf("\n");
      }
    else 
      {
	if (skip == 0) printf("...\n");
	skip =1;
      }
    b += 16;
  }
}
    
int main(int argc, char **argv)
{
  steps = 600000;


  while(1)  /* arg parse loop */
    {
      if (!strcmp(argv[1], "-verbose"))
	{
	  quietf = 0;
	  argc --; argv ++;
	  continue;
	}
     
      if (!strcmp(argv[1], "-quiet"))
	{
	  quietf = 1;
	  argc --; argv ++;
	  continue;
	}
      if (!strcmp(argv[1], "-bp"))
	{
	  break_address = strtol(argv[2], NULL, 0);
	  argc -=2; argv +=2;
	  continue;
	}

      break;
    }

  hexload(argv[1]);
  printf("Reset...\n");
  reset = 1;
  waitb = 1;
  step();
  step();
  reset = 0;
  printf("Simulating...\n");
  while(steps-- >0)
    {
       step();
    }
  coredump();
  return 0;
}

  /* eof */
