/*
 * CBG empu16.c PU16 emulator
 *
 * Software Excellence from Tenison Technology.
 *
 * (C) 1995 Tenison Technology.
 * (C) 1995 DJ Greaves.
 *
 */

#include <stdio.h>
#include <fcntl.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>

#define MEMSIZE 65536

typedef unsigned int u32;
typedef unsigned char uchar;

int byteswapf;
int disasf;
int errorf;
#define bptype_bp  0  /* break and stop */
#define bptype_bpt 1  /* show regs then continue */
#define bptype_bpl 2  /* break and start listing each step */
char *find_a_symbol(int v);
char *symbols, *inputfile;
FILE *logfile;

static unsigned char memory[MEMSIZE];
static int gpregs[4], pc;
static int zf, nf, cf, vf;
static int mem_reads, mem_writes, steps = 0;



void coredump()
{
  int i;
  int *m = (int *) (&memory[0]);
  
  for (i=0;i<=MEMSIZE>>2; i+=4)
    {
      int j;
      fprintf(logfile, "%08X:  %08X %08X %08X %08X  ", i<<2, m[i], m[i+1], m[i+2
], m[i+3]);
      for (j=0;j<16;j++)
        {
          putc((isprint(memory[i*4+j])) ? (memory[i*4+j]): ' ', logfile); 

        }
      fprintf(logfile, "\n");
    }
}

void emulate_putchar()
{
  putc(gpregs[0], logfile);
}

void emulate_getchar()
{
  if (inputfile == NULL || *inputfile == 0)
    {
      printf("***End of inputfile reached\n");
      fprintf(logfile, "\n\n***End of inputfile reached\n");
      errorf = 1;
    }
  else 
    {
      char c =  *inputfile ++; 
      printf("Read input char 0x%02X\n", c);
      gpregs[0] = c;
    }
}


int cbgatoi(char *s)
{
  int r = 0;
  int base = 10;
  if (s == NULL) return 0;
  while (*s == ' ' || *s == 0) s++;  /* Skip leading whities */
  while (1)
   { char c = *(s++);
     if (c >= '0' && c <= '9')
      { r = r*base + c - '0';
        continue;
      }
     if (c >= 'a') c = c - 32;
     if (c =='X')
      { base = 16;
        continue;
      }
     if (c >= 'A' && c <= 'F')
      { r = r*base + c - 55;
        continue;
      }
     break;
   }
  return r;
}



char *find_a_symbol(int v)
{
  static char rline[132];
  char *r;
  int i = 0;
  char line[132];
  if (symbols)
    {
      sprintf(line, "%08X", v);
      r = (char *) strstr(symbols, line);
      if (r) 
	{
	  r += 8;
	  while (*r && *r != '\n') rline[i++] = *r++;
	}
    }
  rline[i] = (char) 0;
  return rline;
}



void read_symbols(char *f)
{
  int v = 100000;
  int fd = open(f, O_RDONLY);
  if (fd < 0)
    {
      printf ("could not open %s for symbols\n", f);
      return;
      symbols = NULL;
    }
  symbols = (char *) malloc(v);
  v = read(fd, symbols, v);
  symbols[v] = (char) 0;
}

void givestats()
{
  printf("Instructions %i,  memory_reads %i, memory_writes %i\n", steps, mem_reads, mem_writes);
}

void read_inputfile(char *f)
{
  int v = 100000;
  int fd = open(f, O_RDONLY);
  if (fd < 0)
    {
      printf ("could not open %s for simulated uart input\n", f);
      return;
      inputfile = NULL;
    }
  inputfile = (char *) malloc(v);
  v = read(fd, inputfile, v);
  inputfile[v] = (char) 0;
}


unsigned int byteswap(unsigned x)
{
  return ((x >> 8) & 0x00FF) + ((x << 8) & 0xFF00);
}



void showregs()
{
  printf("   R0=%08X  R1=%08X  R2=%08X  R3=%08X\n", gpregs[0], gpregs[1], gpregs[2], gpregs[3]);
  printf("   pc=%08X  ZNCV= %i %i %i %i\n", pc, zf, nf, cf, vf);

}

void update_zn_flags(unsigned int a)
{
  zf = (a == 0) ? 1 : 0;
  nf = (a >> 31);
}
 

int check_addr(int a, int bf)
{
  if ( a < 0 || a >= MEMSIZE)
    {
      printf("Emulated memory error, access address %x\n", a);
      printf("Fault PC = 0x%x\n", pc);
      if (1)
	{
	  showregs();
	  errorf = 1;
	  return 1;
	}
      else return 0;
    }
  else return 1;
}


int memory_read(int a, int bytef)
{
  mem_reads += 1;
  a = a & 0xFFFF;
  
  if (bytef) return memory[a];
  else 
    {
      int r;
      if (a & 1) printf("Unaligned access %x\n", a);
      r = (memory[a] << 8) | memory[a+1];
      return r;
    }
}



int next()
{
  int r = memory_read(pc, 0);
  pc += 2;
  return r;
}

void memory_write(int a, int d, int bytef)
{
  a = a & 0xFFFF;
  mem_writes += 1;
  if (!bytef) 
    {
      if (a & 1) printf("Unaligned access %x\n", a);
      memory[a] = d >> 8;
      memory[a+1] = d;
    }
  else memory[a] = d;
}


void emulate_printf()
{
  int p = gpregs[0];
  char c = memory_read(p++, 1);
  char format[132];
  int i = 0;
  printf("Printf >");
  while (c && i <132)
    {
      format[i++] = c;
      c = memory_read(p++, 1);
    }
  format[i] = 0;
  printf(format, gpregs[1], gpregs[2], gpregs[3], gpregs[4], gpregs[5]); 
  fprintf(logfile, format, gpregs[1], gpregs[2], gpregs[3], gpregs[4], gpregs[5]
); 
}

char *alunem(int v)
{
  if (v == 0) return "id";
  if (v == 1) return  "add";
  if (v == 2) return "sub";
  if (v == 3) return "addc";
  if (v == 4) return "subc";
  if (v == 5) return "cmp";
  if (v == 6) return "or";
  if (v == 7) return "and";
  if (v == 8) return "xor";
  if (v == 9) return "asl";
  if (v == 10) return "lsr";
  if (v == 11) return "lsr";
  if (v == 12) return "mov";
  return "???";
}

int alufunc(unsigned int v, int a, unsigned int b)
{
  unsigned int r;
  vf = 0;
  cf = 0;
  if (v == 0) r = a;
  if (v == 1)
    { /* ADD */
      r = a + b;
      cf = (r >> 16) & 1;
      zf = (((a & 0x7fff) + (b & 0x7fff)) >> 15) ^ cf;
    }
  if (v == 2 || v == 5) 
    {
      /* SUB and CMP */  /* carry is borrowbar */
      r = a + ~b + 1;
      cf = (r >> 16) & 1;
      zf = ((((a & 0x7fff) + (~b & 0x7fff) + 1) >> 15) & 1) ^ cf;
    }

  if (v == 3)  
    { /* ADC */
      int ncf = 0;
      r = a + b + cf;
      ncf = (r >> 16) & 1;
      zf = ((((a & 0x7fff) + (b & 0x7fff) + cf) >> 15) & 1) ^ ncf;
      cf = ncf;
    }
  if (v == 4)
    { /* SBC - subtract b from a and subtract one more if carry clear */
      int ncf = 0;
      r = a + ~b + cf;
      ncf = (r >> 16) & 1;
      zf = ((((a & 0x7fff) + (~b & 0x7fff) + cf) >> 15) & 1) ^ ncf;
      cf = ncf;
    }

  if (v == 6) r = a | b;
  if (v == 7) r = a & b;
  if (v == 8) r = a ^ b;
  if (v == 9) r = (a << 1);
  if (v == 10) r = a >> 1;
  if (v == 11) r = (a >> 1) | (a & 0x8000);
  if (v == 12) r = b;

  zf = (r & 0xFFFF) ? 0: 1;
  nf = (r >> 15) & 1;
  return r & 0xFFFF;
}


void pudis(unsigned int x, int exec, int print)
{
  int epc = pc;
  char line[132];

  u32 d = x >> 12;

  if (d == 0xA) /* load imed */
    {
      int rd = (x >> 8) & 3;
      int d;
      if (x & (1<<11))  d = next(); else d = x & 0xFF;
      sprintf(line, "\t lod  r%i,#0x%X", rd, d);
      if (exec) gpregs[rd] = d;
    }

  else if (d == 0x4) /* load  abs */
    {
      int rd = (x >> 8) & 3;
      int bf = (x & (1<<10));
      int ea = (x & (1<<11)) ? next(): (x & 0xFF);
      sprintf(line, " \t %s  r%i, 0x%x %s", 
	     (bf) ? "lodb": "lod",
	     rd, ea, find_a_symbol(ea));
      if (exec) gpregs[rd] = memory_read(ea, bf);
    }

  else if (d == 0x5) /* store abs/direct*/
    {
      int rd = (x >> 8) & 3;
      int bf = (x & (1<<10));
      int ea = (x & (1<<11)) ? next(): (x & 0xFF);
      sprintf(line, " \t %s  r%i, 0x%x %s", 
	      (bf) ? "strb": "str",
	      rd, ea, find_a_symbol(ea));
      if (exec) memory_write(ea, gpregs[rd], bf);
    }

  else if (d == 0x6) /* load indexed */
    {
      int rd = (x >> 8) & 3;
      int rx = (x >> 6) & 3;
      int bf = (x & (1<<10));
      int ea = (x & (1<<11)) ? next(): (x & 0x3F);
      sprintf(line, " \t %s  r%i, [r%i, #%i]", 
	     (bf) ? "lodb": "lod",
	     rd, rx, ea);
      if (exec) gpregs[rd] = memory_read(gpregs[rx]+ea, bf);
    }

  else if (d == 0x7) /* store indexed */
    {
      int rd = (x >> 8) & 3;
      int rx = (x >> 6) & 3;
      int bf = (x & (1<<10));
      int ea = (x & (1<<11)) ? next(): (x & 0x3F);
      sprintf(line, " \t %s  r%i, [r%i, #%i]", 
	     (bf) ? "strb": "str",
	     rd, rx, ea);
      if (exec) memory_write(gpregs[rx]+ea, gpregs[rd], bf);
    }

  else if (d == 0xA) /* load immediate  */
    {
      int rd = (x >> 8) & 3;
      int v = (x & (1<<11)) ? next(): (x & 0xFF);
      sprintf(line, " \t lod  r%i, #%i", 
	     rd, v);
      if (exec) gpregs[rd] = v;
    }    

  else if (d == 0xB) /* arith various */
    {
      int rd = (x >> 6) & 3;
      int fc = (x >> 8) & 0xF;
      char *nem = alunem(fc);
      int mode = (x >> 4) & 3;
      if (mode == 0)
	{
	  int ra = (x >> 0) & 3;
	  sprintf(line, " \t %s  r%i, r%i", nem, rd, ra);
	  if (exec) 
	    {
	      int r = alufunc(fc, gpregs[rd], gpregs[ra]);
	      if (fc != 5) gpregs[rd] = r;
	    }
	}
      else if (mode == 1)
	{
	  int v = next();
	  
	  sprintf(line, " \t %s  r%i, %i", nem, rd, v);
	  if (exec) 
	    {
	      int a = memory_read(v, 0);
	      int r = alufunc(fc, gpregs[rd], a);
	      if (fc != 5) gpregs[rd] = r; 

	    }
	}
      else if (mode == 2)
	{
	  int v = next();
	  sprintf(line, " \t %s  r%i, #%i", nem, rd, v);
	  if (exec) 
	    {
	      int r = alufunc(fc, gpregs[rd], v);
	      if (fc != 5) gpregs[rd] = r; 
	    }

	}
      /* else spare */
      else sprintf(line, "???");

    }
  else if (d == 0xC) /* arith abs short */
    {
      int fc = (x >> 8) & 0xF;
      int rd = (x >> 6) & 3;
      char *nem = alunem(fc);
      int v = x & 0x3F;
      sprintf(line, " \t %s  r%i, %i", nem, rd, v);
      if (exec) 
	{
	  int a = memory_read(v, 0);
	  int r = alufunc(fc, gpregs[rd], a);
	  if (fc != 5)  gpregs[rd] = r;

	}

    }

  else if (d == 0xD) /* arith immed 6 short */
    {
      int rd = (x >> 6) & 3;
      int fc = (x >> 8) & 0xF;
      char *nem = alunem(fc);
      int v = x & 0x3F;
      sprintf(line, " \t %s  r%i, #%i", nem, rd, v);

      if (exec) 
	{
	  int r = alufunc(fc, gpregs[rd], v);
	  if (fc != 5) gpregs[rd] = r;
	}

    }

  else if (d == 0xE) /* return or indirect jump */
    {
      int rd = (x >> 8) & 3;
      int v = x & 0x3F;
      if (rd == 3) sprintf(line, " \t ret #%i", v);
      else sprintf(line, " \t retr r%i, #%i", rd, v);
      if (exec) 
	{
	  int a = memory_read(gpregs[rd], 0);
	  printf("Return address %x\n", a);
	  gpregs[rd] += v;
	  pc = a;
	}

    }

  else if (d == 0xF) /* jumps */
    {
      char *conde;  
      int dest;
      int branch_yes;
      int c = ((x >> 8) & 15);
      if (x & 0xFF)
	{
	  int d = x & 0xFF;
	  if (d & 0x80) d |= 0xFFFFFF00;
	  dest = pc - 2 + (d << 1);
	  if (c == 15) dest += 2;
	}
      else dest = next();


      switch(c)
	{
	case 0:  branch_yes =  zf; break; 
	case 1:  branch_yes =  ~zf; break;
	case 2:  branch_yes = (nf ^ vf); break; 
	case 3:  branch_yes = ~(nf ^ vf) | zf; break; 
	case 4:   branch_yes = ~(nf ^ vf) & ~zf; break; 
	case 5:  branch_yes = (nf ^ vf) | zf; break; 
	case 6:  branch_yes = cf; break;
	case 7:  branch_yes = ~cf; break;
	case 8:  branch_yes = vf; break;
	case 9:  branch_yes = ~vf; break;
	case 10:  branch_yes = 1; break; 
	case 11:  branch_yes = ~cf & ~zf; break; 
	case 12:  branch_yes = cf | zf; break;   
	case 13:  branch_yes = ~nf; break; 
	case 14:  branch_yes = nf; break; 
	default:  branch_yes = 1; break; 
	}

      switch(c)
	{	
	  
	case 0: conde = "eq"; break;
	case 1: conde = "ne"; break;
	case 2: conde = "lt"; break;
	case 3: conde = "ge"; break;
	case 4: conde = "gt"; break;
	case 5: conde = "le"; break;
	case 6: conde = "cs"; break;
	case 7: conde = "cc"; break;
	case 8: conde = "vs"; break;
	case 9: conde = "vc"; break;
	case 10: conde = "ra"; break;
	case 11: conde = "hi"; break;
	case 12: conde = "ls"; break;
	case 13: conde = "mi"; break;
	case 14: conde = "pl"; break;
	case 15: conde = "sr"; break;
	}
  

      if (exec)
	{
	  sprintf(line, "\t b%s 0x%X %s", conde, dest, (branch_yes) ? "TAKEN": "nottaken"); 
	  if (c==15) 
	    {
	      memory_write(gpregs[3], pc, 0);
	    }
	  if (branch_yes) pc = dest;
	}
      else 
	{
	  sprintf(line, "\t b%s 0x%X", conde, dest);
	}
    }

  else sprintf(line, " ??? ");
  if (print)
    {
      printf("%s ", find_a_symbol(epc));
      if (epc == pc) printf("%04X %04X     %s\n", epc-2, x, line);
      else printf("%04X %04X %04X %s\n", epc-2, x, memory_read(pc-2, 0), line);
    }
}


static FILE *fd;

int iread(len)
{
  int r = 0;
  int i;
  for (i=0; i < len; i++)
  {
    unsigned char c = getc(fd);
    c = (c <= '9') ? c-'0': c-('0'+7);
    r = (r<<4) + c;
  }
  return r;
}

int hexload(char *fn)
{
  int line = 1;
  int addr = -1;
  int bytes = 0;  
  int hwm = 0;
   
  fd = fopen(fn, "r");
  if (fd == 0) 
    {
      printf("%s ", fn);
      perror("cant open input file");
      exit(1);
    }
  while (feof(fd)==0)
  {
    uchar c = getc(fd);
    int llen, ltype;
    uchar lsum;
    int wq;

    if (c == '\n') line++;
    if (c != ':') continue;
    llen = iread(2);
    addr = iread(4);
    ltype = iread(2);
/*
    printf("Type %x  addr %x len %x \n", ltype, laddr, llen);
*/
    if (ltype != 0) 
    { 
      printf("Type %i encountered in line %i\n", ltype, line);
      continue;
    }

    lsum = llen + addr + ltype + (addr >> 8);
    for (wq=0; wq<llen; wq++)
    {
      uchar data = iread(2);
      lsum = lsum + data;
      memory[addr+wq] = data;
      bytes += 1;
    }
    if (addr+wq > hwm) hwm = addr+wq;

    lsum = lsum + iread(2);


    if (lsum != 0) 
          fprintf(stderr, "Checksum error in line %i, residue %i\n", line, lsum);    

    addr = addr + llen;
  }
  return hwm;
}


int main(int argc, char **argv)
{
  while (argc > 1)
    {
      if (strcmp("-byteswap", argv[1])==0)
	{
	  argc --;
	  argv ++;
	  byteswapf = 1;
	  continue;
	}
      if (strcmp("-disas", argv[1])==0)
       
	{
	  argc --;
	  argv ++;
	  disasf = 1;
	  continue;
	}
      break;
    }
  if (argc == 1)
    {
      printf("usenix: empu16 [ -byteswap -disas ] <filename>\n");
      exit(0);
    }

  if (argc == 2)
    {
      int len;
      len = hexload(argv[1]);
      
      if (disasf)
	{
	  pc = 0;
	  while (pc < len) pudis(next(), 0, 1);
	  return 0;
	}


      read_symbols("symbols");
      read_inputfile("inputfile");
      logfile = fopen("coredump", "w");


      
      pc = 0;
      steps  = 0;
      while(!errorf)
	{
	  int vb = 1;
	  pudis(next(), 1, 1);
	  if (vb) showregs();
	  
	  if (steps > 0 && pc==0) break;  /* Exit on jump back to zero */
	  
	  if ((steps & 0xFFF) == 0) showregs();
	  if (steps++ > 1000) break;
	}
      givestats();

    }
  printf("Stopped emulation\n");
  return 0;
}






/* end of cbg carmage.c */
