/*Lynx is a lock-free single-producer/single-consumer software
queue for fine-grained communictaion. Copyright (C) 2016  
Konstantina Mitropoulou, Vasileios Porpodas, Xiaochun Zhang and 
Timothy M. Jones.

Lynx is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

Lynx is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  
02110-1301, USA. 
 
More information about Lynx can be found at 
https://www.repository.cam.ac.uk/handle/1810/255384
*/

#define QUEUE_SIZE (1<<21)	/* 2MB */
#include "../include/lynxq.h"
#include <stdio.h>
#include <stdlib.h>

#define DATA ((1 << 30))

volatile long push_sum;

void *push_f(void *arg)
{
  lynxQ *queue = (lynxQ *) arg;
  long i, sum = 0;
  for(i=0; i != DATA; i++) {
    queue->push(i);
    sum += i;
  }
  fprintf(stderr, "PUSH done. Sum=0x%lx\n", sum);
  
  push_sum = sum;
  queue->push_done();
  return NULL;
}

void *pop_f(void *arg)
{
  lynxQ *queue = (lynxQ *) arg;
  long qval, i, sum = 0;
  for (i = 0; i != DATA; ++i) {
    qval = queue->pop_long();
    sum += qval;
  }
  queue->pop_done();
  printf ("POP done.  Sum=0x%lx\n", sum);
  if (sum != push_sum) {
    fprintf (stderr, "\n\nERROR: sum != 0x%lx\n", push_sum);
  } else {
    fprintf (stderr, "SUCCESS!\n");
  }
  return NULL;
}

static void start_thread_and_pin (pthread_t *thread, void * (*func)(void *),
				  void *arg, int core) {
  cpu_set_t mask;
  CPU_ZERO(&mask);
  CPU_SET(core, &mask);
  pthread_attr_t attr;
  pthread_attr_init(&attr);
  pthread_attr_setaffinity_np(&attr, sizeof(cpu_set_t), &mask);
  pthread_create(thread, &attr, func, arg);
}

static void join_thread (pthread_t *thread, void **value_ptr, const char *msg) {
  pthread_join(*thread, value_ptr);
  fprintf (stderr, "%s", msg);
}

int main(int argc, char **argv)
{
  pthread_t push_thread, pop_thread;
  lynxQ *queue = new lynxQ(QUEUE_SIZE);

  // Start push thread and pin it to core 1
  start_thread_and_pin(&push_thread, (void *(*) (void *)) &push_f, (void *) queue, 1);
  // Start pop thread and pin it to core 3
  start_thread_and_pin(&pop_thread, (void *(*) (void *)) &pop_f, (void *) queue, 3);

  // Join threads 
  join_thread(&push_thread, NULL, "** PUSH THREAD JOINED\n");
  join_thread(&pop_thread, NULL, "** POP THREAD JOINED\n");

  queue->finalize();

  double time = queue->get_time();
  delete queue;
  
  fprintf (stderr, "Queue Bandwidth: %10.4f GBytes/s\n",
	   ((double)DATA * sizeof(long))/(time * 1024 * 1024 * 1024));
  fprintf (stderr, "Total data sent: %10.4f GBytes\n",
	   ((double)DATA * sizeof(long))/(1024 * 1024 * 1024));
  return 0;
}
