/*
*     TorTA: Perform traffic analysis experiments on Tor
*
*     Copyright (C) 2005 Steven J. Murdoch <http://www.cl.cam.ac.uk/users/sjm217/>
*
*     This program is free software; you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation; either version 2 of the License, or
*     (at your option) any later version.
*
*     This program is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with this program; if not, write to the Free Software
*     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*     $Id: victim.c 1190 2005-08-10 22:30:31Z sjm217 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <netdb.h>

#include <netinet/in.h>
#include <netinet/tcp.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>

// Minimum block size
#define MINSIZE 1

/**
 * Output an message to stderr, appending an \n
 */

void err(char *message) {
  // Output the message
  fprintf(stderr,"%s\n", message);
  // Don't check the error, since we are probably
  //  in error handing code anyway. If you can't 
  //  write to stderr, things are quite bad anyway
}


/** 
 * Receive data and log timing until end of file is received
 */

int recvloop(int sock, int bs, FILE *fh) {
  char *buf; // Buffer for storing received data
  int retval; // Temporary email for return value of functions
  int bytes; // Number of bytes received
  struct timeval time; // Time that data was received
  
  // Allocate the buffer
  //  sizeof(char) is defined to be 1, but this it clear what I mean
  buf=(char*)calloc(bs, sizeof(char));
  if (buf==NULL) {
    perror("Error allocating buffer space");
    return 1;
  }
  
  // Loop until socket is closed
  for (;;) {

    // Receive up to bs bytes
    bytes=recv(sock, buf, bs, 0);
    if (bytes<0) {
      perror("Error in receiving data");
      free(buf);
      return 2;
    }

    // Check for end of file
    if (bytes==0) {
      err("Connection closed, exiting cleanly");
      break;
    }

    // Get the time that the bytes were received
    retval=gettimeofday(&time, NULL);
    if (retval<0) {
      perror("Error reading current time");
      free(buf);
      return 3;
    }
    
    // Output "time(s) time(us) number_of_bytes" 
    retval=fprintf(fh,"%ld %ld %d\n",time.tv_sec, time.tv_usec, bytes);
    if (retval<0) {
      perror("Error writing data to file");
      free(buf);
      return 4;
    }
  }       
  
  // Cleanup and exit
  free(buf);
  return 0;
}

/**
 * Main function
 */

int main(int argc, char *argv[]) {
  int sock; // Socket to send on
  int bs, port; // Buffer size, port number to connect to
  int trueval=1; // True (used for setting socket options)
  int retval; // Return value of functions
  FILE *fh; // Log file handle

  struct hostent *he; // Result of doing DNS lookup
  struct sockaddr_in dest_addr; // Address of host to receive data from 

  // Check usage
  if (argc != 5) {
    printf("Usage: %s HOSTNAME PORT BLOCKSIZE FILENAME\n", argv[0]);
    return 1;
  }
	
  port=atoi(argv[2]);
  if (port<=0 || port >=65536) {
    fprintf(stderr, "Port must be between 1 and 65535\n");
    return 1;
  }

  bs=atoi(argv[3]);
  if (bs<MINSIZE || bs >=65536) {
    fprintf(stderr, "Block size must be between %d and 65535\n", MINSIZE);
    return 1;
  }

  // Set up the socket
  if ((sock = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
    perror("Error creating socket");
    return 2;
  }
	
  // Disable the Nagle algorithm (send segments as soon as possible)
  if (setsockopt(sock, SOL_TCP, TCP_NODELAY, &trueval, sizeof(int)) == -1) {
    perror("Error setting socket options");
    return 2;
  }

  // Look up the hostname
  if ((he=gethostbyname(argv[1])) == NULL) {
    // herror is marked as obsolete in gethostbyname (3),
    //  but it doesn't suggest an alternative
    herror("Error resolving hostname");
    return 3;
  }

  // Make the connection
  dest_addr.sin_family = AF_INET; 
  dest_addr.sin_port = htons(port); 
  dest_addr.sin_addr = *((struct in_addr *)he->h_addr);
	
  if (connect(sock, (struct sockaddr *)&dest_addr,
	      sizeof(struct sockaddr)) == -1) {
    perror("Error connecting to host");
    return 3;
  }

  err("Connected to server, receiving data...");

  // Open output file
  fh=fopen(argv[4], "w");
  if (NULL==fh) {
    perror("Error while opening output file");
    close(sock);
    return 4;
  }

  // Print file header
  retval=fprintf(fh,"sec usec bytes\n");
  if (retval<0) {
    perror("Error writing header to file");
    fclose(fh);
    close(sock);
    return 5;
  }

  // Receive data and log timing 
  retval=recvloop(sock, bs, fh); 
	
  // Clean up and exit
  fclose(fh);
  close(sock);
	
  // If there was an error in recvloop, return with an error
  //  otherwise signal success
  if (retval)
    return retval+6;
  else
    return 0;
} 
