/*
*     TorTA: Perform traffic analysis experiments on Tor
*
*     Copyright (C) 2005 Steven J. Murdoch <http://www.cl.cam.ac.uk/users/sjm217/>
*
*     This program is free software; you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation; either version 2 of the License, or
*     (at your option) any later version.
*
*     This program is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with this program; if not, write to the Free Software
*     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*     $Id: sender.c 1187 2005-08-10 22:25:28Z sjm217 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <netdb.h>

#include <netinet/in.h>
#include <netinet/tcp.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <sys/time.h>
#include <time.h>

// Minimum block size
#define MINSIZE (sizeof(long)*3+1)

/**
 * Output an message to stderr, appending an \n
 */

void err(char *message) {
  // Output the message
  fprintf(stderr,"%s\n", message);
  // Don't check the error, since we are probably
  //  in error handing code anyway. If you can't 
  //  write to stderr, things are quite bad anyway
}


/**
 * Send timing information on a socket
 */

int sendloop(int sock, int bs, struct timespec waittime) {
  char *buf; // Send buffer
  long *bufl; // Send buffer cast as long*
  long magic=0xC0DEC0B0L; // A magic number to include in packets to check
                          //  for corruption and loss of synchronisation

  struct timeval time; // Current time

  // Temporary variables
  int retval; // Return values from functions

  // Allocate buffer
  buf=calloc(bs, sizeof(char));
  buf[bs-1]='\n';

  // Cast buffer to long*
  bufl=(long*)buf;

  // Loop until interrupted
  for (;;) {

    // Get send time
    gettimeofday(&time, NULL);

    // Fill record
    bufl[0]=magic;
    bufl[1]=time.tv_sec;
    bufl[2]=time.tv_usec;

    // Send the data
    retval=send(sock, buf, bs, 0);

    if (retval!=bs) {
      err("Send truncated");
      retval=1;
      break;
    }

    // Sleep for requested time
    retval=nanosleep(&waittime, NULL);
    if (retval!=0) {
      perror("Sleep interrupted");
      retval=2;
      break;
    }
	 
    retval=0;
  }
       
  return retval;
}

int main(int argc, char *argv[]) {
  int sock; // Connection socket
  int bs, port; // Buffer size, port
  int pwlen; // Length of padded password
  char *password; // Password
  int true_val=1; // True (used for setting socket options)

  double sleept; // Amount of time to sleep (seconds)
  struct timespec waittime; // Time to sleep (nanoseconds)

  struct hostent *he; // Result of doing DNS lookup
  struct sockaddr_in server_addr; // Address of host to receive data from 

  // Temporary variables
  int retval; // Return values from functions

  // Check usage
  if (argc != 6) {
    printf("Usage: %s HOSTNAME PORT BLOCKSIZE SLEEP PASSWORD\n", argv[0]);
    return 1;
  }
	
  port=atoi(argv[2]);
  if (port<=0 || port >=65536) {
    fprintf(stderr, "Port must be between 1 and 65535\n");
    return 1;
  }

  bs=atoi(argv[3]);
  if (bs<MINSIZE || bs >=65536) {
    fprintf(stderr, "Buffer size must be between %d and 65535\n", MINSIZE);
    return 1;
  }

  sleept=atof(argv[4]);
  if (sleept<=0.01) {
    fprintf(stderr, "Sleep must be at least 0.01s\n");
    return 1;
  }
  // Convert time in seconds to a timespec
  waittime.tv_sec=(int)sleept;
  waittime.tv_nsec=(sleept-waittime.tv_sec)*1000000000L;

  // Set up the socket
  if ((sock = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
    perror("Error creating socket");
    return 2;
  }
	
  // Disable the Nagle algorithm (send segments as soon as possible)
  if (setsockopt(sock, SOL_TCP, TCP_NODELAY, &true_val, sizeof(int)) == -1) {
    perror("Error setting socket options");
    close(sock);
    return 2;
  }

  // Look up the hostname
  if ((he=gethostbyname(argv[1])) == NULL) {
    // herror is marked as obsolete in gethostbyname (3),
    //  but it doesn't suggest an alternative
    herror("Error resolving hostname");
    close(sock);
    return 3;
  }
	
  // Make the connection
  server_addr.sin_family = AF_INET;
  server_addr.sin_port = htons(port);
  server_addr.sin_addr = *((struct in_addr *)he->h_addr);
	
  if (connect(sock, (struct sockaddr *)&server_addr,
	      sizeof(struct sockaddr)) == -1) {

    perror("Error connecting to host");
    close(sock);
    return 3;
  }

  // Length of password+'\n', padded to a multiple of bs
  pwlen=((strlen(argv[5])+1+bs-1)/bs)*bs;

  // Expected password
  password=(char*)calloc(pwlen, sizeof(char));
  memset(password, '#', pwlen);
  memcpy(password, argv[5], strlen(argv[5]));
  password[pwlen-1]='\n';

  // Send the password
  retval=send(sock, password, pwlen, 0);
  printf(password);
  if (retval!=pwlen) {
    perror("Error sending password");
    free(password);
    close(sock);
    return 4;
  }

  // Send the data
  retval=sendloop(sock,  bs, waittime); 
  if (retval) {
    free(password);
    close(sock);
    return retval+4;
  }

  free(password);
  close(sock);
	
  return 0;
} 
