/*
*     TorTA: Perform traffic analysis experiments on Tor
*
*     Copyright (C) 2005 Steven J. Murdoch <http://www.cl.cam.ac.uk/users/sjm217/>
*
*     This program is free software; you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation; either version 2 of the License, or
*     (at your option) any later version.
*
*     This program is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with this program; if not, write to the Free Software
*     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*     $Id: receiver.c 1187 2005-08-10 22:25:28Z sjm217 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <sys/time.h>

// Minimum size of the buffer
#define MINSIZE (sizeof(long)*3+1)

/**
 * Output an message to stderr, appending an \n
 */

void err(char *message) {
  // Output the message
  fprintf(stderr,"%s\n", message);
  // Don't check the error, since we are probably
  //  in error handing code anyway. If you can't 
  //  write to stderr, things are quite bad anyway
}

/**
 * Try to receive a timing record until it is possible to get a full
 * record in one call to recv. Return the number of bytes that had to
 * be skipped to achive this
 */

int get_record(int new_fd, char *buf, int bs) {
  int remain=bs; // How many bytes remaining
  int ok; // Is the buffer synchronised on record boundries
  int badcounter=0; // Number of bad bytes encountered

  int retval; // Return value of functions

  // Loop until we get a full record in one recv call
  while (remain) {

    // If we are asking for a whole block, it is OK
    ok=(remain==bs);

    // Try to receive as much data as was needed
    retval=recv(new_fd, buf, remain, 0);
    if (retval==0) {
      return -1;
    } else if (retval<0) {
      return -2;
    }

    // We did not get everything we asked for, so skip
    //  this record
    if (retval!=remain) {
      err("Incomplete record");

      // Increment the count of bad bytes
      badcounter+=retval;
    }

    // Find out how many bytes are remaining
    remain-=retval;

    if (remain==0 && !ok)
      // If we have received all the bytes we are looking for
      //  however we weren't at the beginning, then try to get
      //  a whole record next time round
      remain=bs;

    // If remain==0 at this point, we asked for a whole record (ok==1)
    //  and got it
  }
  
  // Save how many bad bytes were encountered
  return badcounter;
}

int main(int argc, char *argv[])
{
  int sock, new_fd; // Listening socket, connection socket
  struct sockaddr_in server_addr; // Local address
  struct sockaddr_in client_addr; // Remote address

  int sin_size; // Size of sockaddr_in
  int true_val=1; // True (used for setting socket options)
  int port; // Port to listen on

  FILE *fh; // Log file handle
 
  int bs; // Block size

  int  pwlen; // Length of padded password
  char *password, *pwbuf; // Expected password, received password

  char *buf; // Receive buffer
  long *bufl; // Receive buffer cast as a log*

  int bad; // Number of bad bytes received

  // Temporary variables
  int i; // Loop counter
  int retval; // Return values from functions
  

  long magic=0xC0DEC0B0L; // A magic number to include in packets to check
                          //  for corruption and loss of synchronisation

  struct timeval time; // Current time

  // Check usage
  if (argc != 5) {
    printf("Usage: %s PORT BLOCKSIZE DATAFILE PASSWORD\n", argv[0]);
    return 1;
  }

  port=atoi(argv[1]);
  if (port<=0 || port >=65536) {
    err("Port must be between 1 and 65535");
    return 1;
  }
    
  bs=atoi(argv[2]);
  if (bs<MINSIZE || bs >=65536) {
    fprintf(stderr, "Buffer size must be between %d and 65535\n", MINSIZE);
    return 1;
  }

  // Set up the socket
  if ((sock = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
    perror("Error creating socket");
    return 2;
  }

  // Allow listening port to be reused, in case the previous instance
  //  of the program did not close the socket properly
  if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &true_val, sizeof(int)) == -1) {
    perror("Error setting socket to reuse address");
    return 2;
  }
    
  // Bind socket to port
  server_addr.sin_family = AF_INET;
  server_addr.sin_port = htons(port);
  server_addr.sin_addr.s_addr = INADDR_ANY;

  if (bind(sock, (struct sockaddr *)&server_addr, sizeof(struct sockaddr)) == -1) {
    perror("Error binding socket to port");
    close(sock);
    return 2;
  }

  if (listen(sock, 0) == -1) {
    perror("Error listening on socket");
    close(sock);
    return 2;
  }

  // Length of password+'\n', padded to a multiple of bs
  pwlen=((strlen(argv[4])+1+bs-1)/bs)*bs;

  // Expected password
  password=(char*)calloc(pwlen, sizeof(char));

  // Password received
  pwbuf=(char*)calloc(pwlen, sizeof(char));

  // Pad password with '#'
  memset(password, '#', pwlen);
  memcpy(password, argv[4], strlen(argv[4]));
  // Append a '\n'
  password[pwlen-1]='\n';

  // Allocate the buffer
  buf=(char*)calloc(bs, sizeof(char));

  // Cast it as long*
  bufl=(long*)buf;

  // Open the log file
  fh=fopen(argv[3], "w");
  if (NULL==fh) {
    perror("Error while opening output file");
    close(sock);
    free(password);
    free(pwbuf);
    free(buf);

    return 3;
  }

  // Print the header
  fprintf(fh, "ssec susec rsec rusec\n");    

  // Accept connections until one is received with correct magic
  while(1) {
    // Wait for a connection
    sin_size = sizeof(struct sockaddr_in);
    if ((new_fd = accept(sock, (struct sockaddr *)&client_addr, &sin_size)) == -1) {
      perror("Error accepting connection");
      continue;
    }

    fprintf(stderr, "server: got connection from %s\n",inet_ntoa(client_addr.sin_addr));

    // Receive the password (this will fail of the password is split over several
    //  recv calls, however this has not happend in practice)
    retval=recv(new_fd, pwbuf, pwlen, 0);
    if (retval!=pwlen) {
      close(new_fd);
      err("Bad password length");
      continue;
    }
    retval=memcmp(password, pwbuf, pwlen);
    if (0!=retval) {
      close(new_fd);
      err("Bad password");
      continue;
    }
    err("Good password, accepting connection");
	
    for (;;) {
      // Receive a record
      bad=get_record(new_fd, buf, bs);

      if (-2==bad) {
	perror("Error while receiving data");
	
	close(sock);
	close(new_fd);
	free(password);
	free(pwbuf);
	free(buf);
	
	return 4;
      } else if (-1==bad) {
	err("Connection closed, exiting...");
	break;
      }

      // Check if magic is as expected
      if (bufl[0]!=magic) {
	err("Bad magic in received record");

	close(sock);
	close(new_fd);
	free(password);
	free(pwbuf);
	free(buf);
	
	return 5;
      }

      // Get time that data was received
      retval=gettimeofday(&time, NULL);
      if (-1==retval) {
	perror("Error reading current time");

	close(sock);
	close(new_fd);
	free(password);
	free(pwbuf);
	free(buf);
	
	return 6;
      }

      // For every record which was missed, log a null result
      bad=bad/bs;
      for (i=0;i<bad;i++)
	fprintf(fh, "%ld %ld %ld %ld\n",0L,0L, (long)time.tv_sec, (long)time.tv_usec);

      // Log send time (sec, usec) and receive time (sec, usec) to file
      fprintf(fh, "%ld %ld %ld %ld\n",bufl[1],bufl[2], (long)time.tv_sec, (long)time.tv_usec);
    }

    // A valid password was received, so exit cleanly;
    break;
  }

  close(sock);
  close(new_fd);
  free(password);
  free(pwbuf);
  free(buf);
  
  return 0;
}

