/*
*     TorTA: Perform traffic analysis experiments on Tor
*
*     Copyright (C) 2005 Steven J. Murdoch <http://www.cl.cam.ac.uk/users/sjm217/>
*
*     This program is free software; you can redistribute it and/or modify
*     it under the terms of the GNU General Public License as published by
*     the Free Software Foundation; either version 2 of the License, or
*     (at your option) any later version.
*
*     This program is distributed in the hope that it will be useful,
*     but WITHOUT ANY WARRANTY; without even the implied warranty of
*     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
*     GNU General Public License for more details.
*
*     You should have received a copy of the GNU General Public License
*     along with this program; if not, write to the Free Software
*     Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*
*     $Id: evil-server.c 1187 2005-08-10 22:25:28Z sjm217 $
*/

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <arpa/inet.h>

#include <sys/time.h>

// Size of buffer used to send data
#define BUFSIZE 498

/**
 * Output an message to stderr, appending an \n
 */

void err(char *message) {
  // Output the message
  fprintf(stderr,"%s\n", message);
  // Don't check the error, since we are probably
  //  in error handing code anyway. If you can't 
  //  write to stderr, things are quite bad anyway
}

/**
 * Main function
 */

int main(int argc, char *argv[])
{
    int sock, new_fd; // listen socket, connection socket
    struct sockaddr_in server_addr; // local address
    struct sockaddr_in client_addr; // remote address

    int sin_size; // Size of a sockaddr_in
    int true_val=1; // True (used for setting socket options)
    int port; // Port to listen on

    FILE *fh; // Log file handle

    int seed, repeat; // PRNG seed, number of repetitions

    struct timeval time; // Current time

    char buf[BUFSIZE]; // Send buffer

    // Temporary variables
    int retval; // Return value of functions
    int i; // Loop counter
    int t; // A time

    long stoptime; // Time to stop sending

    int maxtime; // Initially stores the maximum time, but later stores the
                 //  difference between the maximum sending time and minimum sending time

    int mintime, sleepfact; // Minimum time, Factor to multiply random numbers to generate
                            //  sleep time

    int bs=BUFSIZE; // Size of buffer (used in setting socket options)

    // Check usage
    if (argc != 8) {
      printf("Usage: %s PORT DATAFILE SEED REPEAT MINTIME MAXTIME SLEEPFACT\n", argv[0]);
      return 1;
    }

    port=atoi(argv[1]);
    if (port<=0 || port >=65536) {
      err("Port must be between 1 and 65535");
      return 1;
    }

    repeat=atoi(argv[4]);
    if (repeat<=0 || repeat >1000) {
      err("Repeat must be between 1 and 1000");
      return 1;
    }

    maxtime=atoi(argv[6]);
    if (maxtime<=0 || maxtime >1000) {
      err("Maxtime must be between 1 and 1000");
      return 1;
    }

    mintime=atoi(argv[5]);
    if (mintime<=0 || mintime>maxtime) {
      err("Mintime must be between 1 and Maxtime");
      return 1;
    }

    sleepfact=atoi(argv[7]);
    if (sleepfact<=0 || sleepfact>100) {
      err("Sleepfact must be between 1 and 100");
      return 1;
    }

    seed=atoi(argv[3]);

    // Set up the socket
    if ((sock = socket(AF_INET, SOCK_STREAM, 0)) == -1) {
        perror("Error creating socket");
        return 2;
    }

    // Allow listening port to be reused, in case the previous instance
    //  of the program did not close the socket properly
    if (setsockopt(sock, SOL_SOCKET, SO_REUSEADDR, &true_val, sizeof(int)) == -1) {
        perror("Error setting socket to reuse address");
	close(sock);
        return 2;
    }

    // Bind socket to port
    server_addr.sin_family = AF_INET;
    server_addr.sin_port = htons(port);
    server_addr.sin_addr.s_addr = INADDR_ANY;

    if (bind(sock, (struct sockaddr *)&server_addr, sizeof(struct sockaddr)) == -1) {
        perror("Error binding socket to port");
	close(sock);
        return 2;
    }

    // Listen on socket
    if (listen(sock, 0) == -1) {
        perror("Error listening on socket");
	close(sock);
        return 2;
    }

    // Seed RNG from user-supplied seed
    srand(seed);

    // Fill buffer with pseudo-random data to prevent compression
    for (i=0;i<BUFSIZE;i++) {
      buf[i]=rand()%0xFF;
    }

    // Open output file
    fh=fopen(argv[2], "w");
    if (NULL==fh) {
      perror("Error while opening output file");
      close(sock);
      return 3;
    }
    
    // Print file header
    retval=fprintf(fh, "t\n");
    if (retval<0) {
      perror("Error writing header to file");
      fclose(fh);
      close(sock);
      return 3;
    }

    // Set maxtime to be the difference between mintime and maxtime
    maxtime-=mintime;

    err("Waiting for incoming connection...");

    // Wait for a connection
    sin_size = sizeof(struct sockaddr_in);
    if ((new_fd = accept(sock, (struct sockaddr *)&client_addr, &sin_size)) == -1) {
      perror("Error accepting connection");
      fclose(fh);
      close(sock);
      return 4;
    }
    
    // Set the maximum send buffer size to be BUFSIZE
    //  This is to stop there to be still data being sent after the stop time
    if (setsockopt(new_fd, SOL_SOCKET, SO_SNDBUF, &bs, sizeof(int)) == -1) {
      perror("Error setting socket buffer size");
      fclose(fh);
      close(sock);
      close(new_fd);
      return 4;
    }
    
    fprintf(stderr, "Got connection from %s\n", inet_ntoa(client_addr.sin_addr));
      
    // Send pattern
    for (i=1;i<=repeat;i++) {
      
      // Get start time of block
      retval=gettimeofday(&time, NULL);
      if (-1==retval) {
	perror("Error reading current time");
	fclose(fh);
	close(sock);
	close(new_fd);
	return 5;
      }
      
      // Log start time
      retval=fprintf(fh,"%ld\n", time.tv_sec);
      if (retval<0) {
	perror("Error writing data to file");
	fclose(fh);
	close(sock);
	close(new_fd);
	return 5;
      }
      
      // Calculate duration to send data
      if (maxtime) {
	// Maxtime is the difference between the minimum and maximum sending time
	//  so chose a random number between 0 and maxtime and add this to mintime
	t=mintime+(rand()%(maxtime+1));
      } else {
	// mintime==maxtime so there is no randomness to be added
	t=mintime;
      }
      
      // Time of day to stop sending
      stoptime=time.tv_sec+t;

      fprintf(stderr, "Sending for %ds\n",t);
      
      // Send as much data as possible until time is up
      for (;;) {

	// Get current time
	retval=gettimeofday(&time, NULL);
	if (-1==retval) {
	  perror("Error reading current time");
	  fclose(fh);
	  close(sock);
	  close(new_fd);
	  return 6;
	}
	
	// If time is up, stop sending
	if (time.tv_sec>=stoptime) {
	  break;
	}
	
	// Send the data
	retval=send(new_fd, buf, BUFSIZE, 0);
	if (retval<1) {
	  perror("Error in sending data");
	  fclose(fh);
	  close(sock);
	  close(new_fd);
	  return 7;
	}
      }
      
      // Get the stop time of block
      retval=gettimeofday(&time, NULL);
      if (-1==retval) {
	perror("Error reading current time");
	fclose(fh);
	close(sock);
	close(new_fd);
	return 5;
      }
      
      // Log end time
      retval=fprintf(fh,"%ld\n", (long)time.tv_sec);
      if (retval<0) {
	perror("Error writing data to file");
	fclose(fh);
	close(sock);
	close(new_fd);
	return 6;
      }
      
      // If we are not in the last block, wait before sending next one
      if (i<repeat) {
	
	// Calculate duration to sleep
	if (maxtime)
	  // Maxtime is the difference between the minimum and maximum sending time
	  //  so chose a random number between 0 and maxtime and add this to mintime
	  t=mintime+(rand()%maxtime);
	else
	  // mintime==maxtime so there is no randomness to be added
	  t=mintime;

	// If requested, sleep longer by a factor of sleepfact
	t*=sleepfact;
	
	fprintf(stderr, "Sleeping for %ds\n",t);
	
	// Sleep for t seconds
	while (t) {
	  err("Sleeping...");
	  t=sleep(t);
	  // If t is non-zero, sleep was interrupted so try again
	}
      }
    }

    err("Closing connection and exiting cleanly");

    // Clean up and exit
    fclose(fh);
    close(sock);
    close(new_fd);
    
    return 0;
}
