#!/usr/bin/env python

### TempCov: Toolkit for investigating temperature covert channels
### Copyright (C) 2006 Steven J. Murdoch <http://www.cl.cam.ac.uk/users/sjm217/>
###
### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; either version 2 of the License, or
### (at your option) any later version.
###
### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.
###
### You should have received a copy of the GNU General Public License along
### with this program; if not, write to the Free Software Foundation, Inc.,
### 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
###
### $Id: probe-sender.py 3381 2006-12-25 18:07:55Z sjm217 $

import sys, time, struct, socket, random

from scapy import sr, send, IP, TCP, interact, \
     ICMP, RandShort, PacketList, RandInt, Raw, Net

tDnsCheck=None
dest_ips=None

## Return a list of IP addresses, caching
##  hostname lookups
def get_dests():
    global tDnsCheck, dest_ips
    tNow=time.time()
    if dest_ips==None or (tNow-tDnsCheck)>5:
        tDnsCheck=tNow
        try:
            dest_ips=map(socket.gethostbyname,dests)
        except:
            write("H")

    return dest_ips

## Write a character (if specified)
##  Otherwise write one "." per 50 calls
cWriteIndex=0
def write(s=None):
    global cWriteIndex
    if s==None:
        cWriteIndex+=1
        if cWriteIndex==50:
            cWriteIndex=0
            sys.stdout.write(".")
    else:
        sys.stdout.write(s)
    sys.stdout.flush()


def send_packets(dests, i):
    # Construct the SYN
    syns=IP(dst=dests, id=RandShort())/ \
          TCP(flags="S", dport=pPort, \
              options=[("Timestamp",(i,0)), \
                       ("NOP",None),("NOP",None)],
              seq=i)
    
    # Construct the ICMP
    icmps=IP(dst=dests, id=RandShort())/ \
           ICMP(type="timestamp-request", \
                id=i%(2**16))/ \
                ("xxxx"*3)

    pl=PacketList([icmps,syns])

    # Send the packets
    write()

    ans, unans=sr(pl, timeout=1, verbose=False)
    return ans, unans

def send_rsts(ans, i):
    # Find the SYN ACKs
    ans = filter(lambda (sent,recvd): recvd.haslayer(IP) and \
                   recvd.haslayer(TCP), ans)
    
    # Find the sources
    acked_sources=[recvd.src for sent,recvd in ans]
    
    # Construct the RSTs
    rsts=IP(dst=acked_sources, id=RandShort())/ \
          TCP(flags="R", dport=pPort, \
              options=[("Timestamp",(i+1,0)), \
                       ("NOP",None),("NOP",None)],
              seq=i+1)
    
    # Send the RSTs
    write()
    send(rsts, verbose=False)

def log_unanswered(unans, cUnansweredHosts):
    # Log unanswered SYNs
    unans_dests=[sent.dst for sent in unans]
    for dst in unans_dests:
        write("L")
        print dst
        x=cUnansweredHosts.setdefault(dst, 0)
        cUnansweredHosts[dst]=x+1

    
# Send no more than 8000 packets per host and for
#  no longer than 12 hours

if len(sys.argv)!=5:
    print "Usage: probe-sender PREFIX PORT PKTCOUNT HOSTS"
    sys.exit()
        
prefix = sys.argv[1]
pPort = int(sys.argv[2])
cMaxPackets= int(sys.argv[3])
dests = list(sys.argv[4].split())

## Stop after a time limit
# now+14 hours
# tStop=time.time()+(24*60*60)
tStop = None

tInter=1 # Time between sending packets

cDelayed=0
cUnansweredHosts={}

i=1
while True:
    tSend=time.time()
    if tStop!=None and tSend>tStop:
        break

    ans, unans=send_packets(get_dests(), i)
    send_rsts(ans, i)
    log_unanswered(unans, cUnansweredHosts)

    # Jump out if the packetcount has been reached
    if i==cMaxPackets:
        break
    i+=1

    # Wait until next time
    tSent=time.time()
    if tSent-tSend < tInter:
        time.sleep(tInter+tSend-tSent+random.random())
    else:
        write("D")
        cDelayed+=1

# Save unanswered packet counts
fh=file(prefix+"unans.data","wt")
fh.write("host unans\n")
i=1
for host,unans in cUnansweredHosts.items():
    fh.write("%s '%s' %s\n"%(i,host,unans))
    i+=1
fh.close()

# Save information on delays
fh=file(prefix+"delayed.data","wt")
fh.write("delayed\n%s %s\n"%(1,cDelayed))
fh.close()

