### TempCov: Toolkit for investigating temperature covert channels
### Copyright (C) 2006 Steven J. Murdoch <http://www.cl.cam.ac.uk/users/sjm217/>
###
### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; either version 2 of the License, or
### (at your option) any later version.
###
### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.
###
### You should have received a copy of the GNU General Public License along
### with this program; if not, write to the Free Software Foundation, Inc.,
### 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
###
### $Id: plot_offset_skew.R 3381 2006-12-25 18:07:55Z sjm217 $

#####
## Options
#####

## Data analysis options

## What type of measurement to be plotted
mMeasurement <- c(TRUE,  # TCP timestamp (works best)
                  FALSE, # ICMP timestamp
                  FALSE) # TCP sequence number

## If cTrim is positive, only the first cTrim samples will be loaded
if (!exists("cTrim")) {
    cTrim <- -1 # Load all samples
}

## Length of a piece (us) in the piecewise fitting
if (!exists("tPiece")){
    tPiece <- 6000*1000000
}

## Number of times to oversample in one piece
if (!exists("cOversample")){
    cOversample <- 5
}

## Optional graph components

## If true, plot temperature on the graph
if (!exists("doTemperature"))
    doTemperature <- TRUE

## If true plot loading on/off on the graph
if (!exists("doOnOff"))
    doOnOff <- FALSE

## If true, remove the constant skew and only show how skew varies
if (!exists("doPieceWise"))
  doPieceWise <- TRUE

## Graphics options

## When plotting the piecewise skew, negate it (because skew increases as
## temperature decreases)
if (!exists("doNegateHostSlopes"))
  doNegateHostSlopes <- TRUE

## Set up margins, leaving enough room for axis labels
if (doTemperature) {
  ## Leave room  on the right for temperature
  mar <- c(5.1, 4.1, 2.6, 4.1)
} else {
  mar <- c(5.1, 4.1, 2.6, 2.1)
}

## If graph is for a presentation, then there will be no skew values printed, so
## don't leave space at the top
if (iAnimStep != 0)
  mar[3] <- mar[3] - 1.6

## Requested number of intervals on the X axis
if (!exists("cAxisCount")){
    cAxisCount <- 5
}

#####
## Load data
#####

## Number of hosts to be tested (always 1 for plot_offset_skew)
cHosts <- length(HostNames)
## Number of types of measurements (always 1 for plot_offset_skew)
cMeasurements <- length(which(mMeasurement))
## Number of (host, measurement) pairs (always 1 for plot_offset_skew)
cData <- cHosts*cMeasurements

## Filename based on measurement type
fnMeasurements <- c("_tcp_","_icmp_","_tcp_")[mMeasurement]
fnMeasurements <- rep(fnMeasurements, cHosts)

## Field names based on measurement type
sFields <- c("ts","ts","seq")[mMeasurement]
sFields <- rep(sFields,cHosts)

## Based on hostname, find out nominal clock speed
print (HostNames)
hzSpeeds <- c(sapply(HostNames,Clocks))[mMeasurement]

## Filenames based on hostnames
fnHosts <- rep(HostNames, each=cMeasurements)

#####
## Perform analysis
#####

## Calculate clock offsets
tmp <- ofCalcOffset()
xs <- tmp[[1]] # Sample times
ys <- tmp[[2]] # Offsets
tAbsStart <- tmp[[3]]
rtts <- tmp[[4]]

### Find breaks for sequence numbers
iBreaks <- list()
for (i in which(sFields=="seq")) {
    iBreaks[[i]] <- iFindBreaks(ys[[i]], 1000)
}

### TCP sequence numbers
for (i in which(sFields=="seq")) {
    tTimes <- xs[[i]]
    ofOffsets <- ys[[i]]
    iRekeys <- c(0,iBreaks[[i]],length(ofOffsets))

    # For each segment, calculate slope
    prevOffset <- 0
    prevTime <- 0
    for (j in 1:(length(iRekeys)-1)) {
        # Calculate segment bounds
        iStart=iRekeys[j]+1
        iEnd=iRekeys[j+1]

        # Calculate slope for that segment
        tmp <- CalcSlope(tTimes[iStart:iEnd],ofOffsets[iStart:iEnd])
        a <- tmp[1]
        b <- tmp[2]

        # Extend current slope back to last time of previous segment
        of <- b*prevTime + a

        # Calculate difference between the offset at that point (prevOffset)
        #  and the extended line
        ofDiff <- of - prevOffset

        # Remove this difference to join up the two lines
        ofOffsets[iStart:iEnd] <- ofOffsets[iStart:iEnd] - ofDiff

        # Save the end offset and time for next time around
        prevOffset <- ofOffsets[iEnd]
        prevTime <- tTimes[iEnd]
    }

    ys[[i]] <- ofOffsets
}

## Calculate overall slope
slopes <- list()

for (i in 1:length(sFields)) {
    tTimes <- xs[[i]]
    ofOffsets <- ys[[i]]

    tmp <- CalcSlope(tTimes,ofOffsets)
    slopes[[i]] <- tmp[2]

    ## Shift points so that overall slope goes through origin
    ys[[i]] <- ofOffsets - tmp[1]
}


if (doPieceWise) {
    ## Calculate piecewise slope
    pslopes <- list()
    pintercepts <- list()
    pieces <- list()

    for (i in 1:length(sFields)) {
        tTimes <- xs[[i]]

        ## Predicted from overall slope
        ofPredicted <- tTimes*slopes[[i]]
        ## Remove linear component
        ys[[i]] <- ys[[i]] - ofPredicted

        ys[[i]] <- CleanupData(ys[[i]])
        ofOffsets <- ys[[i]]

        tMax <- max(tTimes)
        cPiece <- floor((tMax-tPiece) / tPiece * cOversample) + 1 # Number of pieces
        tPieces <- (0:(cPiece-1)) * (tPiece/cOversample) # Timestamps of pieces

        ## Calculate indicies of segments
        ## Piece j runs from iSPieces[j]..iEPieces[j]
        iSPieces=c()
        iEPieces=c()
        for (j in 1:length(tPieces)) {
            ## Start indicies
            iSPieces[j] <- min(which(tTimes>=tPieces[j]))
            ## End indicies
            iEPieces[j] <- min(which(tTimes>=(tPieces[j]+tPiece)))
        }
        ## Trim end index
        iEPieces[length(tPieces)] <- length(tTimes)

        ## Calculate slope of each segment
        hostIntercepts=c()
        hostSlopes=c()
        for (j in 1:length(tPieces)) {
            iStart <- iSPieces[j]
            iEnd <- iEPieces[j]

            tmp <- CalcSlope(tTimes[iStart:iEnd],ofOffsets[iStart:iEnd])
            hostIntercepts[j] <- tmp[1]
            hostSlopes[j] <- tmp[2]
        }

        pslopes[[i]] <- hostSlopes
        pintercepts[[i]] <- hostIntercepts
        pieces[[i]] <- list(iSPieces, iEPieces)
    }
}

####
## Prepare for plotting
####

## Index of data series (always one for skew)
iSeries <- 1

if (doPieceWise) {
  ## Piecewise components
  iSPieces <- pieces[[iSeries]][[1]] # Indicies of piece starts
  iEPieces <- pieces[[iSeries]][[2]] # Indicies of piece ends
  hostSlopes <- pslopes[[iSeries]] # Slopes for each piece
  hostIntercepts <- pintercepts[[iSeries]] # Intercepts for each piece
}

## Overall components
tTimes <- xs[[iSeries]] # Time of each sample (relative to start)
hostSlope <- slopes[[iSeries]] # Absolute skew of test run

if (doPieceWise) {
  ## Calculate the predicted offset for each sample based on piecewise skew

  ofPredicted <- c()

  ## Loop over all pieces
  len <- length(iSPieces)
  for (k in 1:len) {
    iStart <- iSPieces[k] # Sample index of start of piece
    if (k == len)
      ## If this is the last piece, let it run up to the end of the data
      iEnd <- iEPieces[k]
    else
      ## Otherwise, let it run up till the start of the next piece
      iEnd <- iSPieces[k+1] - 1

    ## Calculate predicted offset for this piece
    ofPredicted[iStart:iEnd] <-
      tTimes[iStart:iEnd]*hostSlopes[k] + hostIntercepts[k]
  }

  ## Times of piece endpoints, for plotting segments
  tSegmentStart <- tTimes[iSPieces[1:len]] # Start
  tSegmentEnd <- tTimes[iEPieces[1:len]] # End
  ## Offsets of piece endpoints, for plotting segments
  ofSegmentStart <- tSegmentStart * hostSlopes[1:len] + hostIntercepts[1:len] # Start
  ofSegmentEnd <- tSegmentEnd * hostSlopes[1:len] + hostIntercepts[1:len] # End
}

## Times are normally since start of experiement, but for generating labels we
## need to add on the real time since start of experiment and convert to seconds
tLabTime <- xs[[iSeries]]/1000000+tAbsStart
class(tLabTime) <- "POSIXct"

## Set X range to cover all samples
xlim <- range(tLabTime)

## If caller has specified a function, use it to alter the range
if (exists("tweakRange")) {
  xlim <- tweakRange(1, xlim)
}

## Convert offset to milliseconds
y <- ys[[iSeries]]/1000

## Hide latencies outside the 99% window
ymin <- sort(y)[length(y)*0.001]
mask <- y>ymin & tLabTime>xlim[1] & tLabTime<xlim[2]

## If we have removed the absolute skew, note in the axis label
if (doPieceWise) {
  ylab <- "Non-linear offset component (ms)"
} else {
  ylab <- "Offset (ms)"
}

## FIXME label for the X axis
r <- diff(range(tLabTime[mask]))
print(r)
if (r < 60*60) {
  xlab <- "Time (mm:ss)"
} else if (r < 24*60*60) {
  xlab <- "Time (hh:mm)"
} else {
  xlab <- "Time"
}

####
## Do the plotting
####

## Configure margins and axes
par(mar=mar)
par(las=1) # Axis labels are always horizontal

## Remove any duplicate points to reduce file size
mUniq <- mRemoveDupes(tLabTime[mask], y[mask])

if (iAnimStep>=1 || iAnimStep==0)
  ## Plot offset against time
  plot(tLabTime[mask][mUniq],y[mask][mUniq],pch=".",
       xlab="",ylab="",
       frame.plot=FALSE,xlim=xlim, col=coOffset, axes=FALSE)

## Draw the X axis
tmp <- SimpleAxis(1, tLabTime[mask][mUniq], cIntervals=cAxisCount)

## Choose appropriate label based on unit axes
xFormat <- tmp[[2]]
if (xFormat=="%S") {
  xlab <- "Time (s)"
} else if (xFormat=="%M:%S") {
  xlab <- "Time (mm:ss)"
} else if (xFormat=="%H:%M") {
  xlab <- "Time (hh:mm)"
} else {
  xlab <- "Time"
}

## Show the X axis label
title(xlab=xlab)

## Show the left Y axis title
title(ylab=ylab, line=2.7)

## Find the point after the end of the Y axis title
x <- par("usr")[1] -
  par("cin")[2]/par("pin")[1]*diff(par("usr")[1:2])*3.2
y <- sum(par("usr")[3:4])/2 -
  (strwidth(ylab, "inches")/2+par("cin")[1])/par("pin")[2]*diff(par("usr")[3:4])

## Plot a line the in the same colour as offset slope segments
par(xpd=TRUE)
points(x,y,pch="-",col=coOffsetSlope, cex=2)
par(xpd=FALSE)

## Draw the Y axis
SimpleAxis(2)

if (doPieceWise) {

  if (iAnimStep>=2 || iAnimStep==0)
    ## Plot slope segments
    segments(tSegmentStart/1000000+tAbsStart,
             ofSegmentStart/1000,
             tSegmentEnd/1000000+tAbsStart,
             ofSegmentEnd/1000, col=coOffsetSlope,lwd=1.5)


  par(new=TRUE) ## Draw over offset, with same X but different Y range

  ## Middle of each piece
  tPieceTime <- (tTimes[iSPieces[1:length(iSPieces)]] +
                 tTimes[iEPieces[1:length(iEPieces)]]) /2

  ## Convert to seconds
  tPieceTime <- tPieceTime/1000000+tAbsStart

  ## If requested, negate slope of skew
  if (doNegateHostSlopes)
    hostSlopes <- -hostSlopes

  ## Remove the duplicates to reduce file size
  mUniq <- mRemoveDupes(tPieceTime, hostSlopes, xLimit=50, yLimit=25)
  mask <- tPieceTime>(xlim[1]) & tPieceTime<(xlim[2]) & mUniq

  ## Set Y scale to range over all samples
  ylim <- range(hostSlopes[mask])

  ## If caller has specified a function, use it to alter the range
  if (exists("tweakRange")) {
    ylim <- tweakRange(2, ylim)
  }

  ## Plot the skew
  if (iAnimStep>=3 || iAnimStep==0) {
    par(mar=mar)
    plot(tPieceTime[mask], hostSlopes[mask],
         axes=FALSE,col=coSkew,pch=2,type="b",ann=FALSE,xlim=xlim,ylim=ylim)
  }

  ## Plot slope values
  iMin <- which.min(hostSlopes[mask])
  iMax <- which.max(hostSlopes[mask])
  yMin <- hostSlopes[mask][iMin]
  yMax <- hostSlopes[mask][iMax]

  ## Print out actual values of the skew at the top
  if (iAnimStep==0) {
    par(xpd=TRUE)
    c <- format(hostSlope*1000000,digits=2)
    a <- format(yMin*1000000,digits=2)
    b <- format(yMax*1000000,digits=2)
    msg <- substitute(paste(hat(italic(s))[c] == sc,", ",
                            "min ",hat(italic(s))(italic(t)) == sa,", ",
                            "max ",hat(italic(s))(italic(t)) == sb,
                            " ppm"),
                      list(sc=c, sa=a, sb=b)
                      )
    y <- par("usr")[4] + par("cxy")[2]*1.4
    x <- par("usr")[2] - strwidth(msg) - par("cxy")[1]*1.5
    points(x,y,pch=2,col=coSkew)
    mtext(msg,3,adj=1,line=0.9)
    par(xpd=FALSE)
  }
}

if (doTemperature) {
  ## Plot measurements from temperature probe
  tmp <- CalcTemperature()

  tTempTime <- tmp[[1]] ## Time of each temperature sample
  tempT <- tmp[[2]] ## Temperature sample

  par(new=TRUE) ## Plot over existing graph, but with different Y values

  ## Remove duplicates to reduce file size
  mask <- (tTempTime>xlim[1] & tTempTime<xlim[2])
  mUniq <- mRemoveDupes(tTempTime[mask], tempT[mask],
                        xLimit=50, yLimit=25)

  ## Don't plot except in step 4
  if (iAnimStep>=4 || iAnimStep==0)
    type <- "b"
  else
    type <- "n"

  ## Plot the data
  par(mar=mar)
  plot(tTempTime[mask][mUniq],tempT[mask][mUniq],xlim=xlim,
       axes=FALSE,col=coTemperature,
       pch=1,type=type,ann=FALSE)

  ## Draw the temperature axis
  if (iAnimStep>=4 || iAnimStep==0) {
    SimpleAxis(4) # Draw right Y axis

    par(las=0) # Label is too long to be horizontal, use vertical

    ## The message text
    msg <- expression(paste("Temperature (", ""*degree, "C)"))

    ## Draw text in right Y axis
    mtext(msg,4,line=2.7)

    ## Find the coordinate at bottom of text
    x <- par("usr")[2] +
      par("cin")[2]/par("pin")[1]*diff(par("usr")[1:2])*3.3
    y <- sum(par("usr")[3:4])/2 -
      (strwidth(msg,
    "inches")/2+par("cin")[1])/par("pin")[2]*diff(par("usr")[3:4])

    ## Draw the same style/colour of data marker
    par(xpd=TRUE) ## Disable cropping
    points(x,y,pch=1,col=coTemperature)

    ## Restore cropping and text direction
    par(xpd=FALSE)
    par(las=1)
  }
}

if (doOnOff) {
  ## Plot times when load inducement was on and off
  xMin <- par("usr")[1]
  xMax <- par("usr")[2]

  ## FIXME Cast to date-time type so right axis
  class(xMin) <- "POSIXct"
  class(xMax) <- "POSIXct"

  ## Find ranges of load inducement
  tmp <- CalcOnOff(xMin, xMax)
  tStart <- tmp[[1]] # Load inducement started
  tEnd <- tmp[[2]] # Load inducment stopped

  ## Draw at top of diagram
  yPos <- par("usr")[4]

  par(xpd=NA) # Disable cropping
  par(lend=1) # For accuracy, use butt end style

  if (iAnimStep>=1 || iAnimStep==0)
    ## Actually draw the line segments
    segments(tStart,yPos,tEnd,yPos,col=coOnOff,lwd=5)

  ## Restore cropping
  par(xpd=FALSE)
}



