### TempCov: Toolkit for investigating temperature covert channels
### Copyright (C) 2006 Steven J. Murdoch <http://www.cl.cam.ac.uk/users/sjm217/>
###
### This program is free software; you can redistribute it and/or modify
### it under the terms of the GNU General Public License as published by
### the Free Software Foundation; either version 2 of the License, or
### (at your option) any later version.
###
### This program is distributed in the hope that it will be useful,
### but WITHOUT ANY WARRANTY; without even the implied warranty of
### MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
### GNU General Public License for more details.
###
### You should have received a copy of the GNU General Public License along
### with this program; if not, write to the Free Software Foundation, Inc.,
### 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
###
### $Id: common.R 3381 2006-12-25 18:07:55Z sjm217 $

## Find indicies of suspected breaks in a sequence
## A break is assumed if the change between two consecutive values
##  is greater than limit*interquartile distance
iFindBreaks <- function(ofOffsets, limit) {
  ## Calculate difference between consecutive values
  ofDiff=diff(ofOffsets)

  ## Calculate break criteria (interquartile distance * limit)
  OffsetSummary=summary(ofDiff)
  Interquartile=OffsetSummary[5]-OffsetSummary[2]
  ofCutoff=Interquartile*limit

  ## Find indicies with sufficiently large jumps
  iRekeys=which(ofDiff>ofCutoff)
  return(iRekeys)
}

removeOffset <- function(tTimes, ofOffsets, slope) {
  predicted <- tTimes*slope
  ofDiff <- max(ofOffsets-predicted)
  return(ofOffsets-ofDiff)
}

ofAdjust <- function(ofOffsets, iRekeys) {
  ofDiff=diff(ofOffsets)
  ofDiff[iRekeys] <- 0
  ofOffsets <- c(0, cumsum(ofDiff))

  return(ofOffsets)
}

## Find points where the counter overflowed and correct them
## Original counter is stTimestamps, period of counter is stToAdd
## A point is treates as an overflow if a difference is <dstLimit
## For monotonically non-decreasing counters, dstLimit can be 0
stAdjustOverflow <- function(stTimestamps, stToAdd, dstLimit=0) {
  ## Find the indicies of the overflows
  iOverflows <- which(diff(stTimestamps) < dstLimit) + 1
  ## Include first and last+1 index to form intervals
  iOverflows <- c(1, iOverflows, length(stTimestamps) + 1)

  ## Copy input vector so it is the right length
  stOffsets <- stTimestamps
  ## Amount to add on
  stOffset <- 0
  for (i in 1:(length(iOverflows) - 1)) {
    iStart=iOverflows[i]     # Start of interval
    iEnd=iOverflows[i+1] - 1 # End of interval
    ## Amount to add on
    stOffsets[iStart:iEnd] <- stOffset
    ## Increase amount to add by the period of the counter
    stOffset <- stOffset + stToAdd
  }

  ## Add the offsets to orignal and return
  return(stTimestamps+stOffsets)
}

stAdjustTCPTS <- function(stTimestamps) {
  return(stAdjustOverflow(stTimestamps,2^32))
}

stAdjustICMP <- function(stTimestamps) {
  return(stAdjustOverflow(stTimestamps,24*60*60*1000,TRUE))
}

stAdjustTCPSEQ <- function(stTimestamps){
  return(stAdjustOverflow(stTimestamps,2^32,-2^31))
}

## Find the X coordinate of the intersection of two lines, of slope "M" and Y intercept "C"
XIntersect <- function(M1, C1, M2, C2) {
  (C2-C1)/(M1-M2)
}

## Find a line above all (tTime, ofOffset) points, but with the minimum average
## distance to them. Returns y-intercept and slope
##
## Based on the algorithm described in "S. B. Moon and P. Skelly and D. Towsley:
##  Estimation and Removal of Clock Skew from Network Delay Measurements"
## Appears to be based on Andrew's Monotone Chain Algorithm
##  http://softsurfer.com/Archive/algorithm_0109/algorithm_0109.htm#MC%20Pseudo-Code
CalcSlope <- function(tTimes, ofOffsets) {
  ## Remove any NA values
  mNa <- is.na(tTimes) | is.na(ofOffsets)
  tTimes <- tTimes[!mNa]
  ofOffsets <- ofOffsets[!mNa]

  ## Calculating the slope from a a tiny number of samples is hopeless
  if (length(tTimes) < 10)
    return(c(NA,NA))

  ## Initialize stack (values are popped off/pushed onto the end)
  iStack <- c(1, 2)

  ## Find convex hull
  cLen <- length(tTimes)

  ## BEGIN PERFORMANCE CRITICAL CODE
  ## This section of code accounts for 68% of the running time (63s out of 97s)
  ##Rprof("prof.out", append=TRUE)
  for (i in (3:cLen)) {
    while (length(iStack)>=2) {
      cStackLen <- length(iStack)
      if (XIntersect(tTimes[i],
                     -ofOffsets[i],
                     tTimes[iStack[cStackLen]],
                     -ofOffsets[iStack[cStackLen]]) <
          XIntersect(tTimes[iStack[cStackLen]],
                     -ofOffsets[iStack[cStackLen]],
                     tTimes[iStack[cStackLen-1]],
                     -ofOffsets[iStack[cStackLen-1]]))
        break
      iStack <- iStack[-cStackLen] # Pop off last value
    }
    iStack <- c(iStack, i) # Push on new value
  }
  ##Rprof(NULL)
  ## END PERFORMANCE CRITICAL CODE

  ## Optimum time is the mean
  opt <- mean(tTimes)

  ## Calculate slope and intercept
  cStackLen <- length(iStack)

  for (i in (1:(cStackLen-1))) {
    ## Find time on convex hull that is closest to optimum
    if (tTimes[iStack[i]] < opt && opt < tTimes[iStack[i+1]]) {
      ## Use this to calculate the slope and intercept
      sSlope <- XIntersect(tTimes[iStack[i]],
                           -ofOffsets[iStack[i]],
                           tTimes[iStack[i+1]],
                           -ofOffsets[iStack[i+1]])
      ofIntercept <- -(tTimes[iStack[i]]*sSlope - ofOffsets[iStack[i]])
      break
    }
  }

  ## Return y-intercept and slope
  return(c(ofIntercept, sSlope))
}

CleanupData <- function(ofOffsets) {
  ##tmp <- running(ofOffsets,width=1000,fun=median)
  ##ofOffsets <- rep(NA,length(ofOffsets))
  ##ofOffsets[1:length(tmp)] <- tmp
  return(ofOffsets)
}

ofCalcOffset <- function() {
  ## X and Y coordiates of series
  xs=list()
  ys=list()
  rtts=list()

  for (i in 1:cData) {
    fnHost=fnHosts[i]
    fnMeasurement=fnMeasurements[i]

    fn <- paste(fnPrefix,fnMeasurement,fnHost,".data",sep="")

    tmp=read.table(fn,nrows=cTrim,header=TRUE,colClasses=c("numeric"))
    tbDat = na.omit(tmp)

    ## Time in sec of start of experiment
    ##  Max=~2**32 (will lose precision)
    tAbsStart <- tbDat$rSec[1]+tbDat$rUsec[1]/1000000

    ## Calculate RTT (in usec)
    ## Calculate seconds first, to reduce rounding errors
    tRtt <- tbDat$rSec-tbDat$sSec
    tRtt <- tRtt*1000000+(tbDat$rUsec-tbDat$sUsec)

    ## Subtract start time to reduce FP rounding errors
    ##
    ## Second component of time of each sample
    ##  since the start of experiment
    tSec <- tbDat$rSec-tbDat$rSec[1]

    ## Microsecond component of time of each sample
    ##  since the start of experiment
    tUsec <- tbDat$rUsec-tbDat$rUsec[1]

    ## Combine the two
    tTimes <- tSec*1000000+tUsec

    sField=sFields[i]
    if (sField=="seq") {
      stTimestamps=tbDat$seq
    } else {
      stTimestamps=tbDat$ts
    }

    if (sField=="ts" && fnMeasurement=="_icmp_") {
      stTimestamps <- stAdjustICMP(stTimestamps)
    } else if (sField=="ts" && fnMeasurement=="_tcp_") {
      stTimestamps <- stAdjustTCPTS(stTimestamps)
    } else if (sField=="seq" && fnMeasurement=="_tcp_") {
      stTimestamps <- stAdjustTCPSEQ(stTimestamps)
    }


    ## Convert to machine ticks

    ## Number of us per machine tick
    hzSpeed=hzSpeeds[i]

    ## Convert timestamps to microseconds
    stTimestamps <- stTimestamps/(hzSpeed/1000000)

    ## Offsets in microseconds
    ofOffsets=stTimestamps-tTimes

    xs[[i]]=tTimes
    ys[[i]]=ofOffsets
    rtts[[i]]=tRtt
  }

  return(list(xs,ys,tAbsStart,rtts))
}

CalcTemperature <- function() {
  fn <- paste(fnPrefix,"_temp.data",sep="")
  tbTemp <- read.csv(fn, header=FALSE,
                     col.names=c("date","unit","temp"),
                     colClasses=c("character","NULL","numeric"))

  ## Convert times from iButton format
  tTempTime <- strptime(tbTemp$date,"%d/%m/%y %H:%M:%S")

  ## Assume local time, due to suspected bug
  tTempTime <- as.POSIXct(tTempTime, "")
  tempT <- tbTemp$temp
  return(list(tTempTime, tempT))
}

CalcOnOff <- function(xMin, xMax) {
  fn <- paste(fnPrefix,"_burn.data",sep="")

  tbOnOff <- read.csv(fn,sep=";",colClasses="character",
                      col.names=c("SE","date"),header=FALSE)

  Times <-
    as.POSIXct(strptime(tbOnOff$date,"%Y-%m-%d %H:%M:%S BST"))
  Times[Times<xMin] <- xMin
  Times[Times>xMax] <- xMax

  tStart <- Times[tbOnOff$SE=="S"]
  tEnd <- Times[tbOnOff$SE=="E"]

  return (list(tStart, tEnd))
}
