import scala.io._
import scala.collection.immutable._

object TestResults {
  /**
   * Parse a .res file and create a mapping of idIn -> idOut
   */
  def parseResults(lines: Iterator[String]) = {
    def parseResults(lines: Iterator[String],
		     map: Map[String, String]): Map[String, String] = {
      if (lines.hasNext) {
	val line = List.fromString(lines.next.stripLineEnd, ',')
	val idIn = line(0)
	val idOut = line(1)

	val kv = (idIn, idOut)
	parseResults(lines, map + kv)
      } else {
	map
      }
    }

    parseResults(lines, Map.empty[String, String])
  }

  /*
   * Parse a .msg file and calculate scores. Returns (total, attempted, solved)
   */
  def calculateScore(lines: Iterator[String],
		     resultMapping: Map[String, String]) = {
    def calculateScore(lines: Iterator[String],
		       resultMapping: Map[String, String],
		       total: Int,
		       attempted: Int,
		       solved: Int): (Int, Int, Int) = {
      if (lines.hasNext) {
	val line = List.fromString(lines.next.stripLineEnd, ',')
	val idIn = line(0)
	val idOut = line(1)

	if (resultMapping.contains(idIn)) {
	  if (resultMapping(idIn) == idOut)
	    // Solved
	    calculateScore(lines, resultMapping, total + 1, attempted + 1, solved + 1)
	  else
	    // Failed to solve
	    calculateScore(lines, resultMapping, total + 1, attempted + 1, solved)
	} else {
	  // Not attempted
	  calculateScore(lines, resultMapping, total + 1, attempted, solved)
	}
      } else {
	(total, attempted, solved)
      }
    }

    lines.next // Skip the header line
    calculateScore(lines, resultMapping, 0, 0, 0)
  }

  /**
   * Entry point
   */
  def main(args: Array[String]) = {
    val resFile = args(0)
    val msgFile = args(1)

    Console.println("Testing results in " + resFile +
		    " against message file " + msgFile)

    // Calculate scores
    val resultMapping = parseResults(Source.fromFile(resFile).getLines)    
    val scores = calculateScore(Source.fromFile(msgFile).getLines,
				resultMapping)

    // Print scores
    Console.println("\nTOTAL: " + scores._1 +
		    "\tATTEMPTED: " + scores._2 +
		    "\tSOLVED: " + scores._3)
    Console.printf("Attempted Score: %f\tTotal Score: %f\n\n",
		   scores._3.toDouble/scores._2,
		   scores._3.toDouble/scores._1)
  }
}
