//===========
module LFSR56 (clk, reset, stopped, count);
//===========
//
// 56 bit LFSR counter (with initial reset)
//
// using (56,7,4,2,0)
//   (from Schneier, Applied Cryptography 2nd edition, page 376)
//   (note that many printings of this book have significant
//    errors in this area - check the errata on Schneier's website)
//
// if "reset" is asserted then the value of the counter is set to "1"
//
// otherwise when clk rises then the LFSR is rippled by 1
//
// "count" is the current value ... and is extended by 19 bits
// to the right (to allow old values to be known)
//
   input  clk;
   input  reset;
   input  stopped;
   output [74:0] count;

   wire clk;
   wire reset;
   wire stopped;
   reg  [74:0] count;

   always @(posedge clk or posedge reset)
   begin
      if (reset)
      begin
         count[74:19] <= 1;
         count[18: 0] <= 0;
      end
      else if (!stopped)
      begin
         count[73:0] <= count[74:1];
//
// counter contains 74..19 (56 bit LFSR)
//                  18..00 (extension)
//
// wish to tap 7,4,2,0, hence we tap (7+19),(4+19),(2+19),(0+19) = 26,23,21,19
// 
         count[74] <= count[26] ^ count[23] ^ count[21] ^ count[19];
      end
   end
endmodule
