(**************************************************************************)
(*         x86 Multiprocessor Machine Code Semantics: HOL sources         *)
(*                                                                        *)
(*                                                                        *)
(*  Susmit Sarkar (1), Peter Sewell (1), Francesco Zappa Nardelli (2),    *)
(*  Scott Owens (1), Tom Ridge (1), Thomas Braibant (2),                  *)
(*  Magnus Myreen (1), Jade Alglave (2)                                   *)
(*                                                                        *)
(*   (1) Computer Laboratory, University of Cambridge                     *)
(*   (2) Moscova project, INRIA Paris-Rocquencourt                        *)
(*                                                                        *)
(*    Copyright 2007-2008                                                 *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*                                                                        *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*     notice, this list of conditions and the following disclaimer.      *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*     notice, this list of conditions and the following disclaimer in    *)
(*     the documentation and/or other materials provided with the         *)
(*     distribution.                                                      *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*     products derived from this software without specific prior         *)
(*     written permission.                                                *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,          *)
(*  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING             *)
(*  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS    *)
(*  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.          *)
(*                                                                        *)
(**************************************************************************)

open HolKernel boolLib bossLib Parse;
open wordsTheory bit_listTheory (* opmonTheory*);

open x86_coretypesTheory x86_typesTheory x86_astTheory x86_event_monadTheory x86_event_opsemTheory ;
(* open x86_lts_opsemTheory;  *)
open x86_decoderTheory;
open x86_axiomatic_modelTheory;


open HolDoc;
val _ = new_theory "x86_program";

(* ---------------------------------------------------------------------------------- *>

  Here the decoder from x86_decoderTheory is put together with x86_execute
  from x86_event_opsemTheory and the event structure monad from x86_event_monadTheory.

<* ---------------------------------------------------------------------------------- *)


val _ = Define `DOMAIN f = { x | ~(f x = NONE) } `;



(* **************************** *)
(*   decoding                   *)
(* **************************** *)


val _ = type_abbrev("program_word8",  ``: (address -> word8 option) ``);
 
val _ = type_abbrev("program_Xinst",  ``: (address -> (Xinst#num) option) ``); 


(* try to decode an instruction from an address in a program_word8
memory (record the length too, to avoid having to define a separate
length_of_Xinst) *)

val read_mem_bytes_def = Define `
  read_mem_bytes n a m = 
    if n = 0 then [] else m a :: read_mem_bytes (n-1) (a+1w) m`;

(*
val x86_decode_bytes_def = Define `
  x86_decode_bytes b = x86_decode (FOLDR APPEND [] (MAP w2bits b))`;
*)

(* check that prog_Xinst is a correct decoding of prog_word8 (the
relational version allows the domain of prog_Xinst to be an arbitrary
subset of the domain of prog_word8, whereas the functional version
gives a decoding (if possible) for every address in the domain of
prog_word8) *)

val _ = Define `(decode_program_fun : program_word8 -> address -> (Xinst#num) option) prog_word8 a =
  let xs = MAP THE (read_mem_bytes 20 a prog_word8) in  (* read next 20 bytes *)
  let decode = x86_decode_bytes xs in                   (* attempt to decode *)
  case decode of
     NONE -> NONE                                      (* if decoding fails, then fail *)
  || SOME (i,w) ->                                     (* otherwise extract content *)
       let n = 20 - (LENGTH w DIV 8) in                (* calc length of instruction *)
       if EVERY (\x. ~(x = NONE)) (read_mem_bytes n a prog_word8) (* check the memory is there *)
       then SOME (i,n) 
       else NONE `;

val _ = Define `(decode_program_rel : program_word8 -> program_Xinst -> bool) 
       prog_word8 prog_Xinst  = 
   !a.  case prog_Xinst a of
           NONE -> T
        || SOME (inst,n) -> decode_program_fun prog_word8 a = SOME (inst,n)`;



(* **************************** *)
(*   event structure semantics  *)
(* **************************** *)

val _ = type_abbrev("run_skeleton", ``: (proc -> (program_order_index -> address option)) ``);

(* a run skeleton is well-formed if it is down-closed at each
processor and only mentions addresses in addrs*)

val _ = Define `(run_skeleton_wf:address set -> run_skeleton->bool) addrs rs = 
  (!p i i'.  ((~((rs p i') = NONE)) /\ (i<i')) ==>  (~((rs p i) = NONE))) /\
  (!p i a. (rs p i = SOME a) ==> a IN addrs) /\
  FINITE {p | ?i a. rs p i = SOME a}` ;

(* the event structure semantics of a single instruction *)
val _ = Define `x86_event_execute = x86_event_opsem$x86_execute`;

(* the event structure semantics of a single instruction, restricted to 
those for which any reads of EIP are the specified value *)
val _ = Define `x86_execute_with_eip_check ii inst len eip = 
     let s = (x86_event_execute ii inst len) {} in
        { E 
          | ?eiid_next x. 
            (eiid_next,x,E) IN s /\
            ! v  ev. ((ev.action=(Access R (Location_reg ii.proc RegEIP) v)) /\ 
                     ev IN E.events) ==> (v=eip) 
        } `;

             (* let eiid_start = if j=0 then 0 else (\x. case x of NONE -> 0 || SOME (eiid,x_unit,E) -> eiid) (E_eiid_s (j-1)) in *)

val _ = Define ` (event_structures_of_run_skeleton:program_Xinst -> run_skeleton -> event_structure set) 
                    prog_Xinst rs = 
  let Ess = {x86_execute_with_eip_check <|proc:=p;program_order_index:=i|> inst (n2w n) eip | 
             (rs p i = SOME eip) /\ (SOME (inst, n) = prog_Xinst eip)}
  in
    {event_structure_bigunion Es | Es IN all_choices Ess}`;



(* all such event structures are well-formed - cf x86_program_event_structure_wfScript.sml *)



(* TODO: we need to define an additional "completed" predicate when we want to talk
about completed executions *)


(* finally, putting all this together *)

val _ = Define `(x86_semantics : program_word8 -> state_constraint -> 
 (run_skeleton #  program_Xinst # ((event_structure # (execution_witness set)) set)) set)
     prog_word8 initial_state = 
   
   let x1 = { (rs,prog_Xinst) | rs,prog_Xinst | run_skeleton_wf (DOMAIN prog_Xinst) rs /\ 
                                                decode_program_rel prog_word8 prog_Xinst } in
 
   let x2 = { (rs,prog_Xinst,Es) | (rs,prog_Xinst) IN x1 /\
                                    (Es = event_structures_of_run_skeleton  prog_Xinst rs) } in

   let x3 = { (rs,prog_Xinst,{(E,Xs) | E IN Es /\
                                        (Xs = { X | valid_execution E X /\
                                                   (X.initial_state = initial_state) })})
            | (rs,prog_Xinst,Es) IN x2} in
   x3   `;



(* **************************** *)
(*   lts of event_structure     *)
(* **************************** *)



val _ = Define `(lts_po_of_event_structure (* : event_structure -> (event set,unit,machine_visible_label) LTS *)) E =
   <| states := POW E.events;
      initial := {};
      final := if FINITE E.events then { (E.events,()) } else {};
      trans := { (s,(Vis   <|mvl_event:=e;
                             mvl_iico:= { e' | e' IN s /\ (e',e) IN E.intra_causality } ;
                             mvl_first_of_instruction:= ~ (? (e'::s). e'.iiid = e.iiid);
                             mvl_last_of_instruction:= ~ (? (e'::(E.events DIFF s)). e'.iiid = e.iiid);
                             mvl_locked := locked E e
                           |>),s UNION {e}) | e IN E.events /\ ~(e IN s) /\ 
                                                    e IN minimal_elements (E.events DIFF s) (po_iico E)   } |> `;





(*
val test_def = Define `
test s = 
   (XREAD_REG EBX s = 0x6F5BE65Bw) ==>
   (XREAD_EIP s = 0x804848Bw) ==>
   (XREAD_MEM 0x804848Bw s = SOME 0x89w) ==>
   (XREAD_MEM 0x804848Cw s = SOME 0xD8w) ==>
   (XREAD_REG EAX (THE (X86_NEXT s)) = 0x6F5BE65Bw) /\
   (XREAD_REG EBX (THE (X86_NEXT s)) = 0x6F5BE65Bw) /\
   (XREAD_EIP (THE (X86_NEXT s)) = 0x804848Dw)`;
*)


val _ = export_theory ();
