(**************************************************************************)
(*         x86 Multiprocessor Machine Code Semantics: HOL sources         *)
(*                                                                        *)
(*                                                                        *)
(*  Susmit Sarkar (1), Peter Sewell (1), Francesco Zappa Nardelli (2),    *)
(*  Scott Owens (1), Tom Ridge (1), Thomas Braibant (2),                  *)
(*  Magnus Myreen (1), Jade Alglave (2)                                   *)
(*                                                                        *)
(*   (1) Computer Laboratory, University of Cambridge                     *)
(*   (2) Moscova project, INRIA Paris-Rocquencourt                        *)
(*                                                                        *)
(*    Copyright 2007-2008                                                 *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*                                                                        *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*     notice, this list of conditions and the following disclaimer.      *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*     notice, this list of conditions and the following disclaimer in    *)
(*     the documentation and/or other materials provided with the         *)
(*     distribution.                                                      *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*     products derived from this software without specific prior         *)
(*     written permission.                                                *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,          *)
(*  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING             *)
(*  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS    *)
(*  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.          *)
(*                                                                        *)
(**************************************************************************)

open HolKernel boolLib bossLib Parse;
open wordsTheory bit_listTheory listTheory pred_setTheory;

open x86_coretypesTheory (* ia32_monadtypesTheory*) ;


open utilTheory ;
open x86_typesTheory;
 
(* open ia32_monadtypesTheory; *)

open HolDoc;

val _ = new_theory "x86_event_monad";


(* ----------------------------------------------------------------------------- *>

  We define a monad for constructing an event structure version of the semantics.

<* ----------------------------------------------------------------------------- *)

(* the monad type *)



val _ = type_abbrev("M",``: eiid_state -> ((eiid_state # 'a # event_structure) set)``); 


(* operations on event structures *)

(* OCAML: empty_event_structure *)
val _ = Define `event_structure_empty = <| procs := {}; events := {}; intra_causality := {}; atomicity := {} |>`;
 
(* OCAML: lock_all_events *)
val _ = Define `event_structure_lock es = <| procs := es.procs; events := es.events; intra_causality := es.intra_causality; atomicity := if es.events = {} then {} else { es.events } |>`;
 
(* OCAML: =|= *)
val _ = Define 
            `event_structure_union es1 es2 = 
             <| procs := es1.procs UNION es2.procs;
                events := es1.events UNION es2.events;
                intra_causality := es1.intra_causality UNION es2.intra_causality;
                atomicity := es1.atomicity UNION es2.atomicity |>`;

val _ = Define 
            `event_structure_bigunion (ess : event_structure set) = 
             <| procs := BIGUNION {es.procs | es IN ess};
                events := BIGUNION {es.events | es IN ess};
                intra_causality := BIGUNION {es.intra_causality | es IN ess};
                atomicity := BIGUNION {es.atomicity | es IN ess} |> `;

(* OCAML: =*= *)
val _ = Define 
            `event_structure_seq_union es1 es2 = 
             <| procs := es1.procs UNION es2.procs;
                events := es1.events UNION es2.events;
                intra_causality := es1.intra_causality 
                                       UNION es2.intra_causality 
                                       UNION {(e1,e2) 
                                             | e1 IN (maximal_elements es1.events es1.intra_causality) 
                                               /\ e2 IN (minimal_elements es2.events es2.intra_causality)} ;
                atomicity := es1.atomicity UNION es2.atomicity |>`;


(* the monad operations *)

val mapT_ev_def = Define `
  (mapT_ev:  ('a -> 'b) -> 'a M -> 'b M) f s = 
    \eiid_next:eiid_state. 
        let t = s eiid_next in 
          { (eiid_next', f x, es)  
                       | (eiid_next',x,es) IN t }`;

val choiceT_ev_def = Define `
  (choiceT_ev:  'a M -> 'a M -> 'a M) s s' = 
    \eiid_next:eiid_state. s eiid_next UNION s' eiid_next`;

(* OCAML: unitT *)
val constT_ev_def = Define `
  (constT_ev: 'a -> 'a M) x = \eiid_next. {(eiid_next,x,event_structure_empty)}`;

(* OCAML: discardT *)
val discardT_ev_def = Define `
  (discardT_ev: 'a M -> unit M) s = 
    \eiid_next. let (t:(eiid_state # 'a # event_structure) set)= s eiid_next in 
       IMAGE (\ (eiid_next',v,es). (eiid_next',(),es)) t`;

(* OCAML: addT *)
val addT_ev_def = Define `
  (addT_ev: 'a -> 'b M -> ('a # 'b) M) x s = 
    \eiid_next. let (t:(eiid_state # 'b # event_structure) set)= s eiid_next in 
       IMAGE (\ (eiid_next',v,es). (eiid_next',(x,v),es)) t`;

(* OCAML: lockT *)
val lockT_ev_def = Define `
  (lockT_ev: 'a M -> 'a M) s = 
    \eiid_next. let (t:(eiid_state # 'a # event_structure) set)= s eiid_next in 
       IMAGE (\ (eiid_next',v,es). (eiid_next',v,event_structure_lock es)) t`;

(* OCAML: zeroT *)
val failureT_ev_def = Define `
  (failureT_ev: 'a M) = \eiid_next. {}`;  (* or we lift the result type to options and blow up in this case *)

(* OCAML: >>= *)
val seqT_ev_def = Define `
  (seqT_ev: 'a M -> ('a -> 'b M) -> 'b M) s f = 
    \eiid_next:eiid_state. 
    let t = s eiid_next in 
        BIGUNION { let t' = f x eiid_next' in 
                       { (eiid_next'',x',event_structure_seq_union es es')  
                       | (eiid_next'',x',es') IN t' } 
                 | (eiid_next',x,es) IN t }`;

(* OCAML: >>| *)
val parT_ev_def = Define `
  (parT_ev: 'a M -> 'b M -> ('a # 'b) M) s s' = 
   \eiid_next:eiid_state. 
    let t = s eiid_next in 
        BIGUNION { let t' = s' eiid_next' in 
                       { (eiid_next'',(x,x'),event_structure_union es es')  
                       | (eiid_next'',x',es') IN t' } 
                 | (eiid_next',x,es) IN t }`;


val parT_unit_ev_def = Define `
  (parT_unit_ev: unit M -> unit M -> unit M) s s' = 
   \eiid_next:eiid_state. 
    let t = s eiid_next in 
        BIGUNION { let t' = s' eiid_next' in 
                       { (eiid_next'',(),event_structure_union es es')  
                       | (eiid_next'',(),es') IN t' } 
                 | (eiid_next',(),es) IN t }`;


val write_location_ev_def = Define `(write_location_ev ii l x):unit M = 
  \eiid_next. { (eiid_next', 
                 () , 
                 <| procs := {ii.proc};
                    events := { <| eiid:=eiid';
                                  iiid:=ii;
                                  action:= Access W l x |> };
                    intra_causality := {};
                    atomicity := {} |> )  | (eiid',eiid_next') IN next_eiid eiid_next} `;

val read_location_ev_def = Define `(read_location_ev ii l):value M = 
  \eiid_next. { (eiid_next', 
                 x , 
                 <| procs := {ii.proc};
                    events := { <| eiid:=eiid';
                                  iiid:=ii;
                                  action:= Access R l x |> };
                    intra_causality := {};
                    atomicity := {} |> ) 
              | x IN UNIV /\ (eiid',eiid_next') IN next_eiid eiid_next} `;


(* register reads/writes and eip accesses always succeed. *)

val write_reg_ev_def = Define `(write_reg_ev ii r x):unit M = 
  write_location_ev ii (Location_reg ii.proc (Reg32 r)) x`;

val read_reg_ev_def = Define `(read_reg_ev ii r): value M = 
  read_location_ev ii (Location_reg ii.proc (Reg32 r))`;

val write_eip_ev_def = Define `(write_eip_ev ii x):unit M = 
  write_location_ev ii (Location_reg ii.proc RegEIP) x`;

val read_eip_ev_def = Define `(read_eip_ev ii): value M = 
  read_location_ev ii (Location_reg ii.proc RegEIP)`;

(* eflags *)

val write_eflag_ev_def = Define `(write_eflag_ev ii f bo):unit M = 
  case bo of
     SOME b ->
      (write_location_ev ii (Location_reg ii.proc (Reg1 f)) (if b then 1w else 0w))
  || NONE -> 
       choiceT_ev
         (write_location_ev ii (Location_reg ii.proc (Reg1 f)) 0w)
         (write_location_ev ii (Location_reg ii.proc (Reg1 f)) 1w)
    `;

val read_eflag_ev_def  = Define `(read_eflag_ev ii f): bool M = 
  mapT_ev (\x. (x=0w))
    (read_location_ev ii (Location_reg ii.proc (Reg1 f)))`;


(* memory writes are only allowed to modelled memory, i.e. locations containing SOME ... *)

(* P suspects that ia32_opsemScript.sml doesn't use the byte-sized memory accessors at present *)
(*

val write_mem_ev_def   = Define `(write_mem_ev ii a x):unit M = 
  write_location_ev ii (Location_mem a) x`;

(* a memory read to an unmodelled memory location causes a failure *)

val read_mem_ev_def  = Define `(read_mem_ev ii a):word8 M = 
  read_location_ev ii (Location_mem a)`;
*)



(* reading and writing 32-bit entities *)

(* For now, we take a num-indexed family of word32, but do not access
it except at aligned binaries. *)

val _ = Define `aligned32 a = ((a && 3w) = 0w)`;  

val write_m32_ev_def = Define `(write_m32_ev ii a x):unit M =
  if aligned32 a then 
    write_location_ev ii (Location_mem a) x
  else
    failureT_ev `;

(*
  discardT_ev 
    (let bs = word2bytes 4 w in
       parT_ev (write_mem_ev (a+0w) (EL 0 bs)) (parT_ev (write_mem_ev (a+1w) (EL 1 bs)) 
      (parT_ev (write_mem_ev (a+2w) (EL 2 bs)) (write_mem_ev (a+3w) (EL 3 bs)))))`;
*)

val read_m32_ev_def = Define `(read_m32_ev ii a):Ximm M =
  if aligned32 a then 
    read_location_ev ii (Location_mem a)
  else
    failureT_ev `;


(* export *)
val _ = Define `(constT: 'a -> 'a M)                                     = constT_ev`;
val _ = Define `(addT: 'a -> 'b M -> ('a # 'b) M)                        = addT_ev`;
val _ = Define `(lockT: unit M -> unit M)                                = lockT_ev`;
val _ = Define `(failureT: unit M)                                       = failureT_ev`;
val _ = Define `(seqT: 'a M -> (('a -> 'b M) -> 'b M))                   = seqT_ev`;
val _ = Define `(parT: 'a M -> 'b M -> ('a # 'b) M)                      = parT_ev`;
val _ = Define `(parT_unit: unit M -> unit M -> unit M)                  = parT_unit_ev`;
val _ = Define `(write_reg: iiid -> Xreg -> Ximm -> unit M)              = write_reg_ev`;
val _ = Define `(read_reg: iiid -> Xreg -> Ximm M)                       = read_reg_ev`;
val _ = Define `(write_eip: iiid -> Ximm -> unit M)                      = write_eip_ev`;
val _ = Define `(read_eip: iiid -> Ximm M)                               = read_eip_ev`;
val _ = Define `(write_eflag: iiid -> Xeflags -> bool option -> unit M)  = write_eflag_ev`;
val _ = Define `(read_eflag: iiid -> Xeflags -> bool M)                  = read_eflag_ev`;
val _ = Define `(write_m32: iiid -> Ximm -> Ximm-> unit M)               = write_m32_ev`;
val _ = Define `(read_m32: iiid -> Ximm -> Ximm M)                       = read_m32_ev`;



(*
  seqT_ev (parT_ev (read_mem_ev (a+0w)) (parT_ev (read_mem_ev (a+1w)) 
           (parT_ev (read_mem_ev (a+2w)) (read_mem_ev (a+3w)))))
       (\(x0,x1,x2,x3). constT_ev (bytes2word [x0;x1;x2;x3]))`;
*)


(*
val m_ev_def = Define ` m_ev = 
  <| 
  constT                        := constT_ev;  
  failureT			:= failureT_ev; 
  addT				:= addT_ev; 
  parT_unit			:= parT_unit_ev; 
  parT          		:= parT_ev;
  seqT          		:= seqT_ev; 
  lockT                       	:= lockT_ev;                       
  write_reg   			:= write_reg_ev  ;
  read_reg    			:= read_reg_ev   ;
  write_eip   			:= write_eip_ev  ;
  read_eip    			:= read_eip_ev   ;
  write_eflag 			:= write_eflag_ev;
  read_eflag  			:= read_eflag_ev ;
  read_m32    			:= read_m32_ev   ;
  write_m32   			:= write_m32_ev  
|>`;  
*)

(* a rewriter-friendly theorem *)

(*

val option_apply_def = Define `
  (option_apply NONE f = NONE) /\
  (option_apply (SOME x) f = f x)`;

val monad_simp_thm = store_thm("monad_simp_thm",
  ``(constT_ev x = \y. SOME (x,y)) /\ (failureT_ev = \y. NONE) /\  (lockT_ev s = s) /\
    (addT_ev q s = \y. option_apply (s y) (\t. SOME ((q,FST t),SND t))) /\
    (seqT_ev s f = \y. option_apply (s y) (\t. f (FST t) (SND t))) /\
    (parT_ev s t = \y. option_apply (s y) (\z. 
                    option_apply (t (SND z)) (\x. SOME ((FST z,FST x),SND x))))``,
  SRW_TAC [] [parT_ev_def,seqT_ev_def,failureT_ev_def,lockT_ev_def,addT_ev_def,constT_ev_def,FUN_EQ_THM]
  THEN Cases_on `s y` THEN POP_ASSUM MP_TAC THEN SRW_TAC [] [option_apply_def]
  THEN Cases_on `x` THEN POP_ASSUM MP_TAC THEN SRW_TAC [] [option_apply_def]
  THEN Cases_on `t r` THEN SRW_TAC [] [option_apply_def]
  THEN Cases_on `x` THEN SRW_TAC [] [option_apply_def]);
*)

val _ = export_theory ();
