(***********************************************************************)
(*                                                                     *)
(*                           Objective Caml                            *)
(*                                                                     *)
(*            Xavier Leroy, projet Cristal, INRIA Rocquencourt         *)
(*                                                                     *)
(*  Copyright 1996 Institut National de Recherche en Informatique et   *)
(*  en Automatique.  All rights reserved.  This file is distributed    *)
(*  under the terms of the GNU Library General Public License, with    *)
(*  the special exception on linking described in file LICENSE-OCAML.  *)
(*                                                                     *)
(***********************************************************************)

(* $Id: atcp.ml,v 1.5 2004/12/22 15:27:09 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)



type file_descr = int

type inet_addr = string

external inet_addr_of_string : string -> inet_addr
                                    = "unix_inet_addr_of_string"
external string_of_inet_addr : inet_addr -> string
                                    = "unix_string_of_inet_addr"

let inet_addr_any = inet_addr_of_string "0.0.0.0"

type socket_domain =
    PF_UNIX
  | PF_INET

type socket_type =
    SOCK_STREAM
  | SOCK_DGRAM
  | SOCK_RAW
  | SOCK_SEQPACKET

type sockaddr =
    ADDR_UNIX of string
  | ADDR_INET of inet_addr * int

type shutdown_command =
    SHUTDOWN_RECEIVE
  | SHUTDOWN_SEND
  | SHUTDOWN_ALL

type msg_flag =
    MSG_OOB
  | MSG_DONTROUTE
  | MSG_PEEK

type socket_bool_option =
    SO_DEBUG
  | SO_BROADCAST
  | SO_REUSEADDR
  | SO_KEEPALIVE
  | SO_DONTROUTE
  | SO_OOBINLINE
  | SO_ACCEPTCONN

type socket_int_option =
    SO_SNDBUF
  | SO_RCVBUF
  | SO_ERROR
  | SO_TYPE
  | SO_RCVLOWAT
  | SO_SNDLOWAT

type socket_optint_option = SO_LINGER

type socket_float_option =
    SO_RCVTIMEO
  | SO_SNDTIMEO

external socket : socket_domain -> socket_type -> int -> file_descr
                                  = "unix_socket"
external socketpair : socket_domain -> socket_type -> int -> file_descr * file_descr
                                  = "unix_socketpair"
external accept : file_descr -> file_descr * sockaddr
                                  = "unix_accept"
external bind : file_descr -> sockaddr -> unit
                                  = "unix_bind"
external connect : file_descr -> sockaddr -> unit
                                  = "unix_connect"
external listen : file_descr -> int -> unit
                                  = "unix_listen"
external shutdown : file_descr -> shutdown_command -> unit
                                  = "unix_shutdown"
external getsockname : file_descr -> sockaddr
                                  = "unix_getsockname"
external getpeername : file_descr -> sockaddr
                                  = "unix_getpeername"

external unsafe_recv : file_descr -> string -> int -> int -> msg_flag list -> int
                                  = "unix_recv"
external unsafe_recvfrom : file_descr -> string -> int -> int -> msg_flag list -> int * sockaddr
                                  = "unix_recvfrom"
external unsafe_send : file_descr -> string -> int -> int -> msg_flag list -> int
                                  = "unix_send"
external unsafe_sendto : file_descr -> string -> int -> int -> msg_flag list -> sockaddr -> int
                                  = "unix_sendto" "unix_sendto_native"

let recv fd buf ofs len flags =
  if ofs < 0 || len < 0 || ofs > String.length buf - len
  then invalid_arg "Unix.recv"
  else unsafe_recv fd buf ofs len flags
let recvfrom fd buf ofs len flags =
  if ofs < 0 || len < 0 || ofs > String.length buf - len
  then invalid_arg "Unix.recvfrom"
  else unsafe_recvfrom fd buf ofs len flags
let send fd buf ofs len flags =
  if ofs < 0 || len < 0 || ofs > String.length buf - len
  then invalid_arg "Unix.send"
  else unsafe_send fd buf ofs len flags
let sendto fd buf ofs len flags addr =
  if ofs < 0 || len < 0 || ofs > String.length buf - len
  then invalid_arg "Unix.sendto"
  else unsafe_sendto fd buf ofs len flags addr


(* ********************************************************** *)

type file_descr2 = int
type inet_addr2 = string

let inet_addr_of_string2 : string -> inet_addr2 = inet_addr_of_string
let string_of_inet_addr2 : inet_addr2 -> string = string_of_inet_addr
let inet_addr_any2 = fun () -> inet_addr_of_string "0.0.0.0"

type socket_domain2 = int
let sd2_sd = function
  | 0 -> PF_UNIX
  | 1 -> PF_INET
  | _ -> assert false

let sd_sd2 = function
  | PF_UNIX -> 0
  | PF_INET -> 1

type socket_type2 = int
let st2_st = function
  | 0 -> SOCK_STREAM
  | 1 -> SOCK_DGRAM
  | 2 -> SOCK_RAW
  | 3 -> SOCK_SEQPACKET
  | _ -> assert false

type sockaddr2 = inet_addr2 * int
let sa2_sa (ia,i) = ADDR_INET ((inet_addr_of_string ia), i)
let sa_sa2 x = match x with (ADDR_INET (ia,i)) -> (string_of_inet_addr ia, i) | _ -> assert false

let socket2 : socket_domain2 -> socket_type2 -> int -> file_descr
  = fun sd2 st2 i ->
    let sd = sd2_sd(sd2) in
    let st = st2_st(st2) in
    socket sd st i

let socketpair2 : socket_domain2 -> socket_type2 -> int -> file_descr * file_descr
  = fun sd2 st2 i ->
    let sd = sd2_sd(sd2) in
    let st = st2_st(st2) in
    socketpair sd st i

let accept2 : file_descr -> file_descr * sockaddr2
  = fun fd2 ->
    let fd,sa = accept fd2 in
    (fd, sa_sa2(sa))

let bind2 : file_descr -> sockaddr2 -> unit
  = fun fd sa2 ->
    let sa = sa2_sa(sa2) in
    bind fd sa

let connect2 : file_descr -> sockaddr2 -> unit
  = fun fd sa2 ->
    let sa = sa2_sa(sa2) in
    connect fd sa

let listen2 : file_descr -> int -> unit
  = listen

type shutdown_command2 = int
let sc2_sc = function
  | 0 -> SHUTDOWN_RECEIVE
  | 1 -> SHUTDOWN_SEND
  | 2 -> SHUTDOWN_ALL
  | _ -> assert false

let shutdown2 : file_descr -> shutdown_command2 -> unit
  = fun fd sc2 ->
    let sc = sc2_sc(sc2) in
    shutdown fd sc

let getsockname2 : file_descr -> sockaddr2
  = fun fd ->
    let sa = getsockname fd in
    sa_sa2(sa)

let getpeername2 : file_descr -> sockaddr2
  = fun fd ->
    let sa = getpeername fd in
    sa_sa2(sa)

type msg_flag2 = int
let mf2_mf = function
  | 0 -> MSG_OOB
  | 1 -> MSG_DONTROUTE
  | 2 -> MSG_PEEK
  | _ -> assert false

let recv2 : file_descr -> string -> int -> int -> msg_flag2 list -> int
  = fun fd s i1 i2 ml ->
    recv fd s i1 i2 (List.map mf2_mf ml)

let recvfrom2 : file_descr -> string -> int -> int -> msg_flag2 list -> int * sockaddr2
  = fun fd s i1 i2 ml ->
    let (i,sa) = recvfrom fd s i1 i2 (List.map mf2_mf ml) in
    (i, sa_sa2(sa))

let send2 : file_descr -> string -> int -> int -> msg_flag2 list -> int
  = fun fd s i1 i2 ml ->
    send fd s i1 i2 (List.map mf2_mf ml)

let sendto2 : file_descr -> string -> int -> int -> msg_flag2 list -> sockaddr2 -> int
  = fun fd s i1 i2 ml sa2 ->
    sendto fd s i1 i2 (List.map mf2_mf ml) (sa2_sa(sa2))


