(* tcp.ml
 *
 * $Id: tcp.ml,v 1.509 2004/12/22 15:27:08 zappa Exp $
 *
*** Copyright 2002-2004 The Netsem Team

    * Steve Bishop
    * Michael Compton
    * Matthew Fairbairn
    * Michael Norrish
    * Andrei Serjantov
    * Peter Sewell
    * Michael Smith
    * Keith Wansbrough

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

type error=int

let e2big = 0
  let eacces = 1
  let eaddrinuse = 2
  let eaddrnotavail = 3
  let eafnosupport = 4
  let eagain = 5
  let ewouldblock (* only if EWOULDBLOCK <> EAGAIN *) = 6
  let ealready = 7
  let ebadf = 8
  let ebadmsg = 9
  let ebusy = 10
  let ecanceled = 11
  let echild = 12
  let econnaborted = 13
  let econnrefused = 14
  let econnreset = 15
  let edeadlk = 16
  let edestaddrreq = 17
  let edom = 18
  let edquot = 19
  let eexist = 20
  let efault = 21
  let efbig = 22
  let ehostunreach = 23
  let eidrm = 24
  let eilseq = 25
  let einprogress = 26
  let eintr = 27
  let einval = 28
  let eio = 29
  let eisconn = 30
  let eisdir = 31
  let eloop = 32
  let emfile = 33
  let emlink = 34
  let emsgsize = 35
  let emultihop = 36
  let enametoolong = 37
  let enetdown = 38
  let enetreset = 39
  let enetunreach = 40
  let enfile = 41
  let enobufs = 42
  let enodata = 43
  let enodev = 44
  let enoent = 45
  let enoexec = 46
  let enolck = 47
  let enolink = 48
  let enomem = 49
  let enomsg = 50
  let enoprotoopt = 51
  let enospc = 52
  let enosr = 53
  let enostr = 54
  let enosys = 55
  let enotconn = 56
  let enotdir = 57
  let enotempty = 58
  let enotsock = 59
  let enotsup = 60
  let enotty = 61
  let enxio = 62
  let eopnotsupp = 63
  let eoverflow = 64
  let eperm = 65
  let epipe = 66
  let eproto = 67
  let eprotonosupport = 68
  let eprototype = 69
  let erange = 70
  let erofs = 71
  let espipe = 72
  let esrch = 73
  let estale = 74
  let etime = 75
  let etimedout = 76
  let etxtbsy = 77
  let exdev = 78
  let eshutdown = 79
  let ehostdown = 80
  let eunknown_unix_error = 81


exception Unix_error of error * string * string

let _ = Callback.register_exception "Ocamllib.Unix_error"
                                    (Unix_error(e2big, "", ""))

external error_message : error -> string = "unix_error_message"

let handle_unix_error f arg =
  try
    f arg
  with Unix_error(err, fun_name, arg) ->
    prerr_string Sys.argv.(0);
    prerr_string ": \"";
    prerr_string fun_name;
    prerr_string "\" failed";
    if String.length arg > 0 then begin
      prerr_string " on \"";
      prerr_string arg;
      prerr_string "\""
    end;
    prerr_string ": ";
    prerr_endline (error_message err);
    exit 2

(** {2 Signals types} *)
type signal =
    SIGABRT
  | SIGALRM
  | SIGBUS
  | SIGCHLD
  | SIGCONT
  | SIGFPE
  | SIGHUP
  | SIGILL
  | SIGINT
  | SIGKILL
  | SIGPIPE
  | SIGQUIT
  | SIGSEGV
  | SIGSTOP
  | SIGTERM
  | SIGTSTP
  | SIGTTIN
  | SIGTTOU
  | SIGUSR1
  | SIGUSR2
  | SIGPOLL    (* XSI only *)
  | SIGPROF    (* XSI only *)
  | SIGSYS     (* XSI only *)
  | SIGTRAP    (* XSI only *)
  | SIGURG
  | SIGVTALRM  (* XSI only *)
  | SIGXCPU    (* XSI only *)
  | SIGXFSZ    (* XSI only *)


(** {2 Socket types} *)

type fd=int
(** The abstract type of file descriptors *)

type ip=string
(** The abstract type of ip addresses *)

type port=string
(** The abstract type of inet ports *)

type tid
(** The abstract type of tids *)

type netmask
(** The abstract type of netmask *)

type ifid
(** The abstract type of ifids *)

type filebflag =
    O_NONBLOCK
  | O_ASYNC

type sockbflag =
    SO_BSDCOMPAT
  | SO_REUSEADDR
  | SO_KEEPALIVE
  | SO_OOBINLINE
  | SO_DONTROUTE
  | SO_BROADCAST

type socknflag =
    SO_SNDBUF
  | SO_RCVBUF
  | SO_SNDLOWAT
  | SO_RCVLOWAT

type socktflag =
    SO_LINGER
  | SO_SNDTIMEO
  | SO_RCVTIMEO

type msgbflag = int
let msg_peek = 0
let msg_oob = 1
let msg_waitall = 2
let msg_dontwait = 3

type sock_type =  int
let sock_dgram = 0
let sock_stream  = 1


(** {2 Useful socket functions} *)

external ip_of_string : string -> ip = "nssock_ipofstring"
(** Conversion between string with format |XXX.YYY.ZZZ.TTT|
  and ip addresses. [ip_of_string] raises [Failure] when
  given a string that does not match this format. *)

external string_of_ip : ip -> string = "nssock_stringofip"
(** See {!Nssock.ip_of_string}. *)

external port_of_int: int -> port = "nssock_portofint"
(** Conversion between an int and ports *)

external int_of_port: port -> int = "nssock_intofport"
(** Conversion between a port and an int *)

external fd_of_int_private: int -> fd = "nssock_fdofint"
(** Conversion between an int and a fd *)

external int_of_fd: fd -> int = "nssock_intoffd"
(** Conversion between a fd and an int *)

external gettid: unit -> tid = "nssock_gettid"
(** Returns the process id *)

external int_of_tid: tid -> int = "nssock_int_of_tid"
(** Converts the tid into an int *)

external tid_of_int_private: int -> tid = "nssock_tid_of_int"
(** Converts the int into a tid *)


external ifid_of_string : string -> ifid = "nssock_ifidofstring"
(** Converts the string into an ifid *)

external string_of_ifid : ifid -> string = "nssock_stringofifid"
(** Converts the ifid into a string *)

external netmask_of_int : int -> netmask = "nssock_netmaskofint"
(** Converts the int into a netmask *)

external int_of_netmask : netmask -> int = "nssock_intofnetmask"
(** Converts the netmask into an int *)


let ifid_of_string2 : string -> string = fun x->x
let string_of_ifid2 : string -> string = fun x->x
let netmask_of_int2 : int-> int = fun x->x
let int_of_netmask2 :int->int = fun x-> x

(** {2 Socket Calls} *)

external accept: fd -> fd * (ip * port) = "nssock_accept"

external bind: fd -> ip option -> port option -> unit = "nssock_bind"

external close: fd -> unit = "nssock_close"
external connect: fd -> ip -> port option -> unit = "nssock_connect"
external disconnect: fd -> unit = "nssock_disconnect"
external dup: fd -> fd = "nssock_dup"
external dupfd: fd -> int -> fd = "nssock_dup2"

external getfileflags: fd -> filebflag list = "nssock_getfileflags"
external setfileflags: fd -> filebflag list -> unit = "nssock_setfileflags"

external getifaddrs: unit -> (ifid * ip * ip list * netmask) list = "nssock_getifaddrs"

let getifaddrs2: unit -> (string * ip * ip list * int) list
  = fun u -> List.map (function (ifid,ip,ips,netmask) -> (string_of_ifid ifid,ip,ips,int_of_netmask netmask)) (getifaddrs ())

external getsockname: fd -> ip option * port option = "nssock_getsockname"
external getpeername: fd -> ip * port = "nssock_getpeername"
external getsockbopt: fd -> sockbflag -> bool = "nssock_getsockbopt"
external getsocknopt: fd -> socknflag -> int = "nssock_getsocknopt"
external getsocktopt: fd -> socktflag -> (int * int) option = "nssock_getsocktopt"
external getsockerr: fd -> unit = "nssock_getsockerr"
external getsocklistening: fd -> bool = "nssock_getsocklistening"
external listen: fd -> int -> unit = "nssock_listen"

external recv: fd -> int -> msgbflag list ->
  (string * ((ip option * port option)*bool) option) = "nssock_recv"

external pselect: fd list -> fd list -> fd list -> (int * int) option ->
  signal list option -> fd list * (fd list * fd list) = "nssock_pselect"

let pselect2: fd list -> fd list -> fd list -> (int * int) option ->
  (* signal list option  -> *) fd list * (fd list * fd list)
  = fun r w x ops -> pselect r w x ops None

external send: fd -> (ip * port) option -> string -> msgbflag list -> string = "nssock_send"

external setsockbopt: fd -> sockbflag -> bool -> unit = "nssock_setsockbopt"
external setsocknopt: fd -> socknflag -> int -> unit = "nssock_setsocknopt"
external setsocktopt: fd -> socktflag -> (int * int) option -> unit = "nssock_setsocktopt"

external shutdown: fd -> bool -> bool -> unit = "nssock_shutdown"
external sockatmark: fd -> bool = "nssock_sockatmark"
external socket: sock_type -> fd = "nssock_socket"

let tcp_socket () = socket sock_stream
let udp_socket () = socket sock_dgram
























