(* -=-- --*- caml -*---------------------------------------- --=- *
 *                                                                *
 * runtest_acute.ml                                               *
 *                                                                *
 * Version: $Id: runtest_acute.ml,v 1.15 2004/12/22 12:23:32 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

open Ast
open Util
open Exceptions
open Tysupp
open Pipeline

open Runtest_db
open Runtest_library

exception Never of string

let results = ref []

let thread_exit_hook v = results := v :: !results

let ps_empty_debug () = Pretty.initial_printer_state (Some Econst.string_of_ident) (Pretty.pm_debug ())

let print_expr_option = function
  | Some e -> Some (Pretty.print_prim_expr (ps_empty_debug ()) e)
  | None -> None

let get_error = function
  | Deadlock _ ->
      print_endline "-=[ ERRORS ]=-\nDEADLOCK";
      Some "DEADLOCK"
  | Failure f ->
      let err = Pipeline.print_err 4 0 f in
      print_endline "-=[ ERRORS ]=-";
      print_endline err;
      Some err
  | _ ->
      None

let get_results () =
  match !results with
  | [] -> []
  | l ->
      print_endline "-=[ VALUES ]=-";
      let r = List.map print_expr_option l in
      List.iter (function None -> print_endline "Killed" | Some s -> print_endline s) r;
      r

let rec prune_none = function
  | [] -> (0,0,[])
  | None::t -> let (nb,len,q) = prune_none t in (nb+1,len,q)
  | Some h::t -> let (nb,len,q) = prune_none t in (nb,len+1,h::q)

let reg_match re s = try let _ = Str.search_forward re s 0 in true with Not_found -> false

let match_result exp res =
  let has_re_a_match re lst =
    List.fold_left (function b -> function s -> b || reg_match re s) false lst
  in
  List.fold_left (function b -> function re -> b && has_re_a_match re res) true exp

let acute_do_test test opts =
  Debug.thread_exit_hook := thread_exit_hook;
  let f_name = (Filename.dirname test.file) ^ "/tmp_" ^ (test.name ^ "." ^ (string_of_int (Unix.getpid ())) ^ ".ac") in
  (
    match test.body with
    | None -> raise (Never "Test run with empty body !!")
    | Some s ->
        let out_d = Unix.openfile f_name [ Unix.O_CREAT; Unix.O_WRONLY; Unix.O_TRUNC ] 0o664 in
        let out = Unix.out_channel_of_descr out_d in
        output_string out (Runtest_pretty.acpp_body s);
        close_out out
  );

  let result =
    let opts' = (
      match test.expect with
      | Compile -> ("fake_an_argv_0" :: "-norun" :: opts@[f_name])
      | _ -> ("fake_an_argv_0" :: opts@[f_name])
      ) in
    (try Opts.custom_parse (Array.of_list opts') with e -> ());
    Threadpool.create_thread_pool ();
    seed ();
    main ()
  in
  Unix.unlink f_name;

  match test.expect with
  | Empty    -> raise (Never "expect Empty")
  | Group  _ -> raise (Never "expect Group")
  | Killed _ -> 1 (* here it's always bad : WE WANT to be killed, so if we are here, we were not ! *)
  | Compile  ->
      (
        match get_error result , get_results () with
        | None   , _ -> 0
        | Some _ , _ -> 1
      )
  | Fail  f  ->
      (
        match get_error result , get_results () with
        | None   , _ -> 1
        | Some s , _ -> if reg_match (Str.regexp f) s then 0 else 1
      )
  | Exits l  ->
      (
        match get_error result , get_results () with
        | Some _ , _ -> 1
        | None   , r ->
            let (res_nb_kill,res_nb_val,res) = prune_none r
            and (exp_nb_kill,exp_nb_val,exp) = prune_none ( List.map (function None -> None | Some s -> Some (Str.regexp s)) l ) in
            if ( exp_nb_kill > res_nb_kill ) || ( exp_nb_val > res_nb_val ) then
              1
            else
              if match_result exp res then 0 else 1
      )

let fork test opts =
  let (std_in, std_out) = Unix.pipe () in  (* data written to out will appear at in *)
  let (err_in, err_out) = Unix.pipe () in  (* data written to out will appear at in *)
  let pid = Unix.fork () in
  if pid <> 0 then (
      Unix.close std_out;
      Unix.close err_out;
      (pid, std_in, err_in)
  ) else (
    Unix.dup2 std_out Unix.stdout; Unix.close std_out; Unix.close std_in;
    Unix.dup2 err_out Unix.stderr; Unix.close err_out; Unix.close err_in;
    exit (acute_do_test test opts) (* kind of ungly, but exit codes can be also used to explain to the father what really happened *)
  )

