(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * opts.ml                                                        *
 *                                                                *
 * Version: $Id: opts.ml,v 1.539 2004/12/22 16:49:13 pes20 Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

(* command-line options *)

type optionclass =
   OC_output
  |OC_phase
  |OC_rttc
  |OC_semantic

let oc_alist = [
   OC_output, "output";
   OC_phase,  "phase";
   OC_rttc,   "rttc";
   OC_semantic,"semantic";
]

exception Fatal
exception No_args

type dumplevel = int
type evalname = int
let levelvar ="<level> (0-4)"
let leveldoc = "the <level>s are:\n  0 none\n  1 expression\n  2 expression, store\n  3 expression, store, userdefns\n  4 expression, store, userdefns, libdefns\n"

(* bit tasteless to use raw ints, but want the ordering*)

type debugclass =
    DBC_default
  | DBC_flattenclos
  | DBC_desugar
  | DBC_tcopt
  | DBC_mkhash
  | DBC_lexer
  | DBC_evalstep
  | DBC_marshal
  | DBC_hashify
  | DBC_tcquant
  | DBC_linkok
  | DBC_namecase
  | DBC_namevalues
  | DBC_reachable

let dbc_alist = [
  "default", DBC_default;
  "flattenclos", DBC_flattenclos;
  "desugar", DBC_desugar;
  "tcopt", DBC_tcopt;
  "mkhash", DBC_mkhash;
  "lexer", DBC_lexer;
  "evalstep", DBC_evalstep;
  "marshal", DBC_marshal;
  "hashify", DBC_hashify;
  "tcquant", DBC_tcquant;
  "linkok", DBC_linkok;
  "namecase", DBC_namecase;
  "namevalues", DBC_namevalues;
  "reachable", DBC_reachable;
]

let dbc_alist' = List.map (fun (x,y) -> (y,x)) dbc_alist
let dbc_all = List.map snd dbc_alist


let fail s = prerr_endline ("Fatal: "^s); exit 1

(* otheropts: *)
let output_filename_opt = ref None
let final_filename_opt = ref None
let err_filename_opt = ref None
let tcdepth = ref 4
let debugs = ref dbc_all
let printstepinterval = ref None
let dumpfrom = ref 0
let dumpstepinterval = ref 1

let emitobjectfile = ref None
let emitsourcefile = ref None

type filename = {
    fi_filename : string;
    fi_basename : string;
    fi_compiled : bool;
  };;

let mkfilename name =
  let n = try
    String.rindex name '.'
  with
    Not_found ->
      fail ("Input filename \"" ^ name ^ "\" has no extension.")
  in
  let (basename,ext) = (String.sub name 0 n,
                        String.sub name (n+1) (String.length name - (n+1))) in
  let compiled =
    match ext with
      "ac" ->
        false
    | "mini" ->
        prerr_endline ("Warning: input filename \""^name^"\" uses deprecated extension ."^ext^".\n"
                       ^"Please use the preferred extension .ac.");
        false
    | "aco" ->
        true
    | _ ->
        fail ("Input filename \""^name^"\" has an unrecognised extension ."^ext^".")
  in
  { fi_filename = name;
    fi_basename = basename;
    fi_compiled = compiled }
;;

let getfilename fi = fi.fi_filename

let dslib_filename_opt = ref (Some (mkfilename "definitions_lib_small.ac"))
let input_filenames = ref []

let final_output_filename_opt = ref None
let final_check_filename_opt = ref None


(* boolflags: *)
    (* overall control *)
let showpasses = ref true
let showtimes = ref true
let showprogress = ref false
    (* pipeline phases *)
let parsetest = ref false
let desugar = ref true
let compile = ref true
let typecheckcompiled = ref true
let run = ref true
    (* behaviour options *)
let showlocs = ref false
let default = ref true

   let bigstep = 0
    let smallstep = 1

(* let evaluator = ref smallstep *)
let printerrordeath = ref true
let printcleandeath = ref false
let nonunitthread = ref true

let rttc = ref true
let mttc = ref true
let terminate_on_rttc = ref true
let showtrailer = ref true
let suffixall = ref false
let shownames = ref false
let globalhashmap = ref true
let disable_import_typecheck = ref false
let disable_eqsok_typecheck = ref false
let internal_weqs = ref true
let linkok_sig_typecheck = ref true
let show_options = ref false
let hack_optimise = ref true
let really_hack_optimise = ref false
let abstract_existentials = ref true
let showtcenv = ref false
let emitobject = ref false
let lithash = ref false
let printenv = ref false
let printenvbodies = ref false
let printclos = ref false
let showfocussing = ref false
    (* dumping state after pipeline phases *)
let dumpparse = ref 0
let dumppreinf = ref 0
let dumppostinf = ref 0
let dumpdesugared = ref 0
let dumpcompiled = ref 3
let dumptrace = ref 1
let dumpfinal = ref 2
let dumptypefail = ref 3

let dumpall = ref false
let dumptex = ref false
let dumphuman = ref false
let marshaltex = ref false

let impl_options_alist = [   (* the negation below must be kept in sync *)
  ("norttc", fun () -> rttc := false);
  ("nomttc", fun () -> mttc := false);
  ("notypecheckcompiled", fun () -> typecheckcompiled := false);
  ("lithash", fun () -> lithash := true);
  ("nolinkok_sig_typecheck", fun () -> linkok_sig_typecheck := false);
  ("hack_optimise", fun () -> hack_optimise := true);
  ("dumptrace", fun () -> dumptrace := 0)
]

let not_impl_options_alist = [
  ("rttc", fun () -> rttc := true);
  ("mttc", fun () -> mttc := true);
  ("typecheckcompiled", fun () -> typecheckcompiled := true);
  ("nolithash", fun () -> lithash := false);
  ("linkok_sig_typecheck", fun () -> linkok_sig_typecheck := true);
  ("nohack_optimise", fun () -> hack_optimise := false);
  ("dumptrace", fun () -> dumptrace := 1)
]

let production = ref false

(*let writechecklevel = ref 1*)
    (* debug *)
let debug = ref false

let usage = "acute <options> <filename>\n  where\n" ^ leveldoc ^ "\nand options are:"

let file_argument name =
  input_filenames := !input_filenames @ [mkfilename name]
;;

let string_of_eval eval =
   if (eval = bigstep) then "bigstep"
     else if (eval = smallstep) then "smallstep"
     else fail ("No such evaluator")

let string_of_debugs debugs =
  String.concat "," (List.map (fun d -> List.assoc d dbc_alist') debugs)

let comma_re = Str.regexp ","
let debugs_of_string fail_cont s =
  if s = "all" then dbc_all else
  List.map (fun ds -> try List.assoc ds dbc_alist
                      with Not_found -> fail_cont ("Unknown debug class "^ds))
           (Str.split comma_re s)

(* Non-Boolean options: flag (including hyphen), Arg descriptor, arg description, description, (?) function to generate string from option *)
let otheropts = [

(* semantic options *)

  (OC_semantic,"-definitionslib", Arg.String (fun name -> dslib_filename_opt := Some (mkfilename name)),
    "<filename>", "Read the standard definitions from <filename> " ^
                  "\n(default: " ^ getfilename (Util.the !dslib_filename_opt) ^ "\n but use definitions_lib.ac for full set)",
    function () -> match !dslib_filename_opt with None-> "" | Some fi -> "-definitionslib "^getfilename fi);

  (OC_semantic,"-nodefinitionslib", Arg.Unit (fun () -> dslib_filename_opt := None),
    "", "No standard definitions",
    function () -> match !dslib_filename_opt with None-> "-nodefinitionslib" | Some s -> "");

(* phase control *)

(* P asks: are -o, -df, -err, -writefinal and -checkfinal still live? *)

  (OC_phase,"-o", Arg.String (fun name -> output_filename_opt := Some name),
    "<filename>", "Output to <filename> (default: " ^ (match !output_filename_opt with
                                                         None -> "<stdout>" | Some s -> s) ^ ")",
   function () -> (match !output_filename_opt with None -> "" | Some s -> "-o "^s));


  (OC_phase,"-df", Arg.String (fun name -> final_filename_opt := Some name),
    "<filename>", "Print final state dump to to <filename> (default: " ^ (match !final_filename_opt with
                                                         None -> "<stdout>" | Some s -> s) ^ ")",
   function () -> (match !err_filename_opt with None -> "" | Some s -> "-o "^s));


  (OC_phase,"-err", Arg.String (fun name -> err_filename_opt := Some name),
    "<filename>", "Print debug output to <filename> (default: " ^ (match !err_filename_opt with
                                                         None -> "<stderr>" | Some s -> s) ^ ")",


   function () -> (match !err_filename_opt with None -> "" | Some s -> "-err "^s));



  (OC_phase,"-writefinal", Arg.String (fun name -> final_output_filename_opt := Some name),
    "<filename>", "Pretty print result to <filename> (default: " ^ (match !final_output_filename_opt with
                                                         None -> "<stdout>" | Some s -> s) ^ ")",
     function () -> (match !final_output_filename_opt with None -> "" | Some s -> "-o "^s));

  (OC_phase,"-checkfinal", Arg.String (fun name -> final_check_filename_opt := Some name),
    "<filename>", "Check result against contents of <filename> (default: " ^ (match !final_check_filename_opt with
                                                         None -> "None" | Some s -> s) ^ ")",
     function () -> (match !final_check_filename_opt with None -> "" | Some s -> "-o "^s));

  (OC_phase,"-emitobjectfile", Arg.String (fun name -> emitobject:= false; emitobjectfile := Some name),
    "<filename>", "Emit compiled (object) code after compilation",
    function () -> match !emitobjectfile with None -> "" | Some a -> a);

  (OC_phase,"-emitsourcefile", Arg.String (fun name -> emitsourcefile := Some name),
    "<filename>", "Emit source code after compilation",
    function () -> match !emitsourcefile with None -> "" | Some a -> a);

(* output control *)

  (OC_output,"-debugs", Arg.String (fun debugs_str ->
    debug := true;
    debugs := debugs_of_string fail debugs_str),
   "<class>[,<class>..]",
   "Which classes of debug output to display \n(default: " ^ string_of_debugs !debugs ^ ")",
   function () -> "-debugs " ^ string_of_debugs !debugs );

  (OC_output,"-dumpstepinterval", Arg.Int (fun x -> dumpstepinterval := x),
    "<n>", "Print the configuration (at dumptrace level) every <n> steps ",
    function () -> match !dumpstepinterval with x -> "-dumpstepinterval "^(string_of_int x));

  (OC_output,"-dumpfrom", Arg.Int (fun x -> dumpfrom := x),
    "<n>", "Only print the configuration (at dumptrace level) after <n> steps ",
    function () -> match !dumpfrom with x -> "-dumpfrom "^(string_of_int x));

  (OC_output,"-printstepinterval", Arg.Int (fun x -> printstepinterval := Some x),
    "<n>", "Print the reduction step count every <n> steps ",
    function () -> match !printstepinterval with None-> "" | Some x -> "-printstepinterval "^(string_of_int x));

  (OC_output,"-noprintstepinterval", Arg.Unit (fun () -> printstepinterval := None),
    "", "Do not print the reduction step count",
    function () -> match !printstepinterval with None-> "-noprintstepinterval" | Some s -> "");

  (OC_rttc,"-production",Arg.Unit (fun () -> (production:=true;(List.iter (fun (_, f) -> f())  impl_options_alist))),"",("Set options used for a production implementation\n " ^
  (List.fold_right (fun a -> fun b -> a ^ b) (List.map (fun (s, _) -> "<"^s^">") impl_options_alist )  "")),function () -> "-production");

  (OC_rttc,"-noproduction",Arg.Unit (fun () -> (production:=false;(List.iter (fun (_, f) -> f())  not_impl_options_alist))),"",("Set options used for a non-production implementation\n " ^
  (List.fold_right (fun a -> fun b -> a ^ b) (List.map (fun (s, _) -> "<"^s^">") not_impl_options_alist )  "")),function () -> "-noproduction");

]

(* Boolean flags: name of flag (without hyphen), reference to modify, description string *)
let boolflags : (optionclass * string(*flag name*) * bool ref(*flag*) * string(*description*)) list = [

(* output control *)
  OC_output,"showpasses", showpasses, "Show names of compilation passes";
  OC_output,"showtimes", showtimes, "Show time taken per pass";
  OC_output,"showprogress", showprogress, "Show progress during type inference";
  OC_output,"showlocs", showlocs, "Show locations in dump output";
  OC_output,"showtrailer", showtrailer, "Show trailer information (e.g., hash values) when printing";
  OC_output,"suffixall", suffixall, "Always suffix names, even when unshadowed";
  OC_output,"shownames", shownames, "Show internal representation of bound names";
  OC_output,"globalhashmap", globalhashmap, "Use a common map for abbreviating hashes and abstract names";
  OC_output,"show_options", show_options, "Show the command line used, including default options";
  OC_output,"showtcenv", showtcenv, "Show environment in typecheck errors";
  OC_output,"emitobject", emitobject, "Emit compiled (object) code after compilation";
  OC_output,"printenv", printenv, "Print runtime environments";
  OC_output,"printenvbodies", printenvbodies, "Print runtime environment bodies (RHSs)";
  OC_output,"printclos", printclos, "Print closures as closures (rather than expanding)";
  OC_output,"printerrordeath", printerrordeath, "Print error message when a thread exits with an exception";
  OC_output,"printcleandeath", printcleandeath, "Print message when a thread exits cleanly";
  OC_output,"debug", debug, "Generate debug output (on stderr)";
  OC_output,"showfocussing", showfocussing, "Show focussing process in dumptrace";
  OC_output,"dumptex", dumptex, "Dump in tex format";
  OC_output,"dumphuman", dumphuman, "Dump for humans (no type annotations)";
  OC_output,"dumpall", dumpall, "Don't ever abbreviate traces to ...";


(* phase control *)
  OC_phase,"parsetest", parsetest, "Parser - pretty printer identity test";
  OC_phase,"desugar", desugar, "Desugar";
  OC_phase,"compile", compile, "Compile";
  OC_phase,"typecheckcompiled", typecheckcompiled, "Typecheck the compiled program";
  OC_phase,"run", run, "Run program";


(* run-time type checking *)
  OC_rttc,"lithash", lithash, "Emit literal 0#123ABC hashes in certain places";
  OC_rttc,"rttc", rttc, "Do runtime typechecking";
  OC_rttc,"mttc", mttc, "Do unmarshaltime typechecking";
  OC_rttc,"terminate_on_tc", terminate_on_rttc, "Terminate if typecheckcompiled or rttc is on and fails";

(* semantic options *)
  OC_semantic,"default", default, "Default underspecified types to unit";
  OC_semantic,"disable_import_typecheck", disable_import_typecheck, "Disable typechecking of import links";
  OC_semantic,"disable_eqsok_typecheck", disable_eqsok_typecheck, "Disable typechecking of |- eqs ok";
  OC_semantic,"internal_weqs", internal_weqs, "Allow use of with! equations inside modules\n (not just at boundary)";
  OC_semantic,"linkok_sig_typecheck", linkok_sig_typecheck, "Do full subsignature typecheck in linkok\n (not just syntactic check)";
  OC_semantic,"hack_optimise", hack_optimise, "Perform vacuous-bracket optimisation";
  OC_semantic, "really_hack_optimise", really_hack_optimise, "Erase all brackets";
  OC_semantic,"abstract_existentials", abstract_existentials, "Dynamically-abstract existentials";
  OC_semantic,"nonunitthread", nonunitthread, "Threads do not have to evaluate to unit";
  OC_semantic,"marshaltex", marshaltex, "Marshal in tex format (cannot be unmarshalled)";
]



(* Integer options: name of option, reference to modify, description string *)
let levelopts : (optionclass * string * int ref * string * string) list = [

 (* output control *)
  OC_output,"dumpparse", dumpparse, levelvar, "Dump result of parse";
  OC_output,"dumppreinf", dumppreinf, levelvar, "Dump input to inference";
  OC_output,"dumppostinf", dumppostinf, levelvar, "Dump output of inference";
  OC_output,"dumpdesugared", dumpdesugared, levelvar, "Dump output of desugaring";
  OC_output,"dumpcompiled", dumpcompiled, levelvar, "Dump output of compilation";
  OC_output,"dumptrace", dumptrace, levelvar, "Dump traced execution steps";
  OC_output,"dumpfinal", dumpfinal, levelvar, "Dump final state (if no type failure)";
  OC_output,"dumptypefail", dumptypefail, levelvar, "Dump on type failure (or unmarshalfail)";
(*  "writechecklevel", writechecklevel, levelvar, "Amount of detail to include when writing result/checking result"; *)
]
let intopts : (optionclass * string * int ref * string * string) list = [
  OC_output,"tcdepth",  tcdepth, "<depth>", "Context depth for typechecking errors";
] @ levelopts


(* check for conflicts *)
let conflict_check () =
  (* let fail s = prerr_endline ("Fatal: "^s); exit 1 in *) (* moved out *)
  if !input_filenames = [] then
    raise No_args
  else if !showtimes && not !showpasses then
    fail "-showtimes requires -showpasses"
  else if List.exists (fun (_,_,r,_,_) -> !r < 0 || !r > 4) levelopts then
    fail "Dump level must be in range 0..4"
  else if !printenvbodies && not !printenv then
    fail "-printenvbodies requires -printenv"
  else if !lithash && !rttc then
    fail "-rttc is not possible with -lithash"
  else if !lithash && !mttc then
    fail "-mttc is not possible with -lithash"
  else if !lithash && !typecheckcompiled then
    fail "-typecheckcompiled is not possible with -lithash"
  else
    ()

(* massage functions convert the above lists to the format expected by Arg for parsing *)
let massage_flag (oc,name,flag,desc) =
  [
   ("-"^name,   Arg.Set flag,   "");
   ("-no"^name, Arg.Clear flag, "")
  ]
let massage_int_opt (oc,name,rf,argname,desc) =
  ("-"^name, Arg.Int ((:=) rf), "")
let massage_otheropts (oc,name,spec,arg,desc,dump) =
  (name, spec, "")


(* dumpoption functions generate strings in the compiler option syntax from the above lists *)
let dumpoption_flag (oc,name,flag,desc) =
  if !flag then "-"^name else "-no"^name
let dumpoption_int_opt (oc,name,rf,argname,desc) =
  "-"^name^" "^(string_of_int !rf)
let dumpoption_otheropt (oc,name,spec,arg,desc,dump) =
  dump ()


(* format functions convert the above lists to the format expected by Arg's usage message generation (via align) *)
let string_of_oc oc = List.assoc oc oc_alist ^ ": "

let format_flag (oc,name,flag,desc) =
  ("-[no]"^name,
   Arg.Set flag,
    (if !flag then "(*)" else "( )") ^ " " ^ (string_of_oc oc) ^ desc)
let format_otheropts (oc,name,spec,arg,desc,dump) =
  (name ^ " " ^ arg, spec, (string_of_oc oc) ^ desc)
let format_int_opts (oc,name,rf,arg,desc) =
  ("-"^name ^ " " ^ arg, Arg.Int ((:=) rf), "("^(string_of_int !rf)^") "^ (string_of_oc oc) ^desc )

(* align all the descriptions in a neat column, ready for usage message generation *)
let align speclist
  = let maxlen = List.fold_right (fun (s,_,_) n -> max n (String.length s)) speclist 0 in
    let width = maxlen + 1 in
    List.map (fun(s,p,d) -> (s,p,String.make (width - String.length s) ' ' ^ d)) speclist

(* local version of Arg.parse that takes the various lists as arguments and massages/formats them appropriately *)
(* TODO: add in leveldoc somewhere here *)
let myparse otheropts intopts boolflags anon_fun usage_msg opts_array
  = let speclist = List.map massage_otheropts otheropts @
                   List.map massage_int_opt intopts @
                   Util.concat_map massage_flag boolflags in
    let help s =
      let formatted_opts = align (List.map format_otheropts otheropts @
                                  List.map format_int_opts intopts @
                                  List.map format_flag boolflags)
      in
      Arg.usage formatted_opts usage_msg; prerr_string s; exit 255
    in
    try
      Arg.parse_argv opts_array speclist anon_fun usage_msg
    with
      Arg.Bad(s)  -> help (String.sub s 0 (1 + String.index s '\n'))  (* UGH! s has undesired usage message as well *)
    | Arg.Help(_) -> help ""

(* main entry point: parse command-line arguments *)
let parse ()
    = begin
      myparse otheropts intopts boolflags file_argument usage Sys.argv;
      conflict_check ();
    end

let custom_parse opts_array
  = begin
      Arg.current := 0;
      myparse otheropts intopts boolflags file_argument usage opts_array;
      conflict_check ()
    end

(* entry point to echo command line, listing all the options *)
let do_show_options ()     =
  Sys.argv.(0)
  ^ " "
  ^ (List.fold_right
       (fun s1 s2 -> s1^" "^s2)
       (List.map dumpoption_otheropt otheropts @
        List.map dumpoption_int_opt intopts @
        List.map dumpoption_flag boolflags)
       "")
  ^ (List.fold_right
       (fun fi s -> " " ^ getfilename fi ^ s)
       !input_filenames
       "")
