(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * hashify.ml                                                     *
 *                                                                *
 * Version: $Id: hashify.ml,v 1.545 2004/12/22 12:23:31 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)

(* the meat of compilation *)
open Ast
open Util
open Tysupp


(* == EXCEPTIONS == *)
exception Bad_sourcedef_valuability of string
exception Withspec_equation_from_import of string
exception Withspec_wrt_bad_type_field of string
exception Withspec_types_not_equal of string



exception Hashify of Exceptions.hashify_exn

let fatal e =
  match e with
    Withspec_equation_from_import s->
      raise (Hashify (Exceptions.Withspec_equation_from_import s))
  | Withspec_wrt_bad_type_field s->
      raise (Hashify (Exceptions.Withspec_wrt_bad_type_field s))
  | Withspec_types_not_equal s->
      raise (Hashify (Exceptions.Withspec_types_not_equal s))
  | Linkok.Linkok_not s ->
      raise (Hashify (Exceptions.Linkok_not s))
  | Bad_sourcedef_valuability s ->
      raise (Hashify (Exceptions.Bad_sourcedef_valuability s))
  | e -> raise (Never_happen ("Unknown exception in hashify: " ^ Printexc.to_string e))

(* apply rho to the top level TXDot(Modname).  Use rho_type to recursively
rhoify a type*)
let rec rho_top_typ cm t =
  match t with
  | TXDot (Modname m, external_typname) ->
      let def_of_m = ModnameMap.find m cm in
      let sign_of_m = sign1_of_definition def_of_m in
      begin match the (lookup_signature_etn sign_of_m external_typname) with
      | KType -> None
      | KEq(t0) -> Some (t0)
      end
  | _ -> None

let rho_sign cm sign =
  try tmap_signature (early_tmap_typ (rho_top_typ cm)) sign
      with _ -> raise (Failure "moo1")
let rho_str cm str = try
  tmap_structure (early_tmap_typ (rho_top_typ cm)) str
      with _ -> raise (Failure "moo2")
let rho_expr cm expr = try
  tmap_expr (early_tmap_typ (rho_top_typ cm)) expr
      with e -> raise (Failure ("moo3: " ^ Printexc.to_string e ^ ": " ^  Pretty.print_expr (Dump.fresh_printer_state 1 0 false) expr))
let rho_type cm ty = try
  early_tmap_typ (rho_top_typ cm) ty
      with _ -> raise (Failure "moo4")
let rho_kind cm = function
    KType   -> KType
  | KEq(ty) -> KEq(rho_type cm ty)

let hashify_envi cm = function
    Env_val(iid,ty)  -> Env_val(iid,rho_type cm ty)
  | Env_loc(l,ty)    -> Env_loc(l,rho_type cm ty)
  | Env_mod(mn,sign) -> Env_mod(mn,
                         rho_sign cm  (* cm must have binding for mn already *)
                           (Tysupp.type_flatten_sign
                              (Tysupp.selfify_signature (Modname(mn))
                                 sign)))
  | Env_typ(itn,k)   -> Env_typ(itn,rho_kind cm k)
let hashify_env cm env =
  mktypeenv (List.map (hashify_envi cm) (typeenv_list_of_typeenv env))


let lookup_hash cm m =
  let def = try ModnameMap.find m cm
  with Not_found-> let (ext, int) = m in raise(Never_happen ("modname not found in lookup_hash: " ^ ext))

  in
  begin match def with
  | Mod_compile ((ext, int), body) -> body.mc_hash
  | Import_compile (_, body) -> body.ic_hash
  | _ -> raise (Never_happen ("there should only be compiled modules and interfaces in cm, got a " ^ Pretty.print_definition (Dump.fresh_printer_state 3 0 false) def))
  end

let lookup_typ cm (m, x) =
  let def = try ModnameMap.find m cm
  with Not_found ->  let (ext, int) = m in raise(Never_happen ("modname not found in lookup_typ: " ^ ext))
  in
  begin match def with
  | Mod_compile((ext, int), body) ->
      let sign = body.mc_sign1 in
      (match find_val_field sign x  with {loc=_; desc=SVal(x, t)} -> t
      |	_ -> assert false)

       | Import_compile(_, body) ->
       let sign = body.ic_sign1 in
      (match find_val_field sign x  with {loc=_; desc=SVal(x, t)} -> t
      |	_ -> assert false)

  | _ -> raise (Never_happen ("there should only be compiled modules and interfaces in cm, got a " ^ Pretty.print_definition (Dump.fresh_printer_state 3 0 false) def))
  end



let sigma cm = fun e -> match e.desc with
  | LocDot (m', xx) ->
      (Some (at_loc e.loc (LocHashDot (lookup_hash cm m', xx))))
  | LocHashMvf (Modname m',eid,t) ->
      Debug.print' Opts.DBC_hashify (fun () -> "Mvf");
      (Some (at_loc e.loc (LocHashMvf (Hash (lookup_hash cm m'), eid,t))))
(*  | LocTie (Modname m',eid) ->
      (Some (at_loc e.loc (LocTie (Hash (lookup_hash cm m'), eid)))) *)
  | _ -> None


let hashify_ties_and_hashes_expr cm = fun e -> match e.desc with
| LocTie(m, eid) ->
    let t' = lookup_typ cm (m, eid) in
    Some (at_loc e.loc (LocC1
		    (TieCon, at_loc e.loc (LocTup [
				  at_loc e.loc (LocHashMvf (Hash (lookup_hash cm m), eid, t'));
				  at_loc e.loc (LocDot(m, eid))]))))

| LocHashMvf(Modname m, eid, t) ->
    Some (at_loc e.loc (LocHashMvf(Hash (lookup_hash cm m), eid, t)))
| _ -> None

let hashify_ties_and_hashes cm =
  emap_structure false (early_emap_expr false (fun _ e -> hashify_ties_and_hashes_expr cm e) None)

let eval_atomic_version_number h avne =
  match avne with
  | VLiteral _ -> avne
  | VMyname -> VLiteral (VHash h)

let rec eval_version_number h vne =
  match vne with
  | VAtomic avne -> VAtomic (eval_atomic_version_number h avne)
  | VCons (avne, vne') -> VCons (eval_atomic_version_number h avne, eval_version_number h vne')

let rec eval_version_constraint cm = function
  | VCDotted (avce_list, tvce) ->
      VCDotted (List.map (eval_atomic_version_constraint cm) avce_list,
                eval_tail_version_constraint cm tvce)
  | VCNameEqual ahvce -> VCNameEqual (eval_atomic_hash_version_constraint cm ahvce)
and eval_atomic_hash_version_constraint cm = function
  | (AVCHHash _) as ahvce -> ahvce
  | AVCHModname m ->
      let h =
	try
	  lookup_hash cm m
	with e -> let (ext, int) =  m in raise (Never_happen ("lookuphash " ^ ext))
      in
      AVCHHash h
and eval_atomic_version_constraint cm = function
  | AVCHashVersion ahvce ->
      AVCHashVersion (eval_atomic_hash_version_constraint cm ahvce)
  | (AVCNumber _) as avce -> avce

and eval_tail_version_constraint cm tvce =
  match tvce with
  | TVCAtomic avce -> TVCAtomic (eval_atomic_version_constraint cm avce)
  | TVCBetween _ -> tvce
  | TVCBefore _ -> tvce
  | TVCAfter _ -> tvce
  | TVCAnything -> tvce

let rec hashify_def' en cm d vubs recurse =
    Debug.print' Opts.DBC_hashify (function () -> "hd': " ^ Pretty.print_source_definition (Dump.fresh_printer_state_all ()) d);
  match d with
  | Mod_user (m, mode, mu_body) ->
      let str = mu_body.mu_str in
      let sign = mu_body.mu_sign in
      let str1 = rho_str cm str in
      let str1' =
	hashify_ties_and_hashes cm str1 in
      let sign1 = rho_sign cm sign  in
      let str2 = Tysupp.type_flatten_str str1' in
      let sign2 = Tysupp.type_flatten_sign sign1 in
      let (en3,str3) = (match mode with
                          | MHash | MFresh | MBangHash-> (en,str2)
                          | MCfresh | MBangCfresh -> Evalsupp.evalcfresh_str(en,str2)) in

      let weqs = mu_body.mu_withspec in
      let eqs =
        let eq_of_weq (m', tt', ty) =
          EMod (m, tt', rho_type cm ty) in
        List.map eq_of_weq weqs in
      let check_weq (m', tt', ty) =
        let def' = try ModnameMap.find m' cm
	    with Not_found -> raise (Never_happen "modname not found in hashify_def'")
	in
        begin match def' with
        | Mod_compile (_, body') ->
            let tyrepfromC = Util.the (lookup_structure_etn body'.mc_str tt') in
            begin if not (Tysupp.typ_eq (rho_type cm ty) tyrepfromC) then
              fatal (Withspec_types_not_equal ("rep type doesn't match in weqs: " ^ Pretty.print_typ (Dump.fresh_printer_state 1 0 false) tyrepfromC ^ " and " ^
		                               Pretty.print_typ (Dump.fresh_printer_state 1 0 false) (rho_type cm ty)))
            end;
            begin match Util.the (lookup_signature_etn body'.mc_sign0 tt') with
            | KType -> ()
            | KEq (_) ->
                fatal(Withspec_wrt_bad_type_field "can't see through to a non abstract type in weqs")
            end
        | _ -> fatal (Withspec_equation_from_import "the lhs of a weqs equation can only refer to a module")
        end in
      List.iter check_weq weqs;
      let eqs' =
        let closed_eq_of_weq (m', tt', ty) =
          let (h'',tt'') = begin match rho_type cm (TXDot (Modname m', tt')) with
          | TXDot (Hash h'',tt'') -> (h'', tt'')
          | _ -> raise (Never_happen "internal error: not getting a type of the form h.t when building eqs'")
          end in
          begin if etn_compare tt'' tt' <> 0 then
            raise(Never_happen "internal error: not getting back the right type field when building eqs'")
          end;
          EHash (h'', tt'',  rho_type cm ty) in
        List.map closed_eq_of_weq weqs in

      let str5 =
	Debug.print' Opts.DBC_hashify (function () -> "about to map");
	emap_structure false (early_emap_expr false (fun _ e -> sigma cm e) None) str3 in
                                 (* no need to freshen; sigma touches only M.x occurrences *)
      	Debug.print' Opts.DBC_hashify (function () -> "mapped");
      let (en',h) =
        let mh_body = { mh_eqs = eqs';
                        mh_sign0 = sign2;
                        mh_vne = mu_body.mu_vne;
                        mh_str = str5; }
        in (match mode with
              | MHash | MBangHash -> (en3, Pretty.mkHashM Econst.string_of_ident (modname_ext m, mh_body))
              | MCfresh | MFresh | MBangCfresh ->
		  let n = new_abstract_name ()
                  in (addnameenv (NEnv_nmod(n,modname_ext(m),mh_body)) en3, HashName n) ) in

      let sign7 = Tysupp.type_flatten_sign
          (Tysupp.selfify_signature (Hash(h)) sign2) in
      let vn = eval_version_number h mu_body.mu_vne in
      let mc_body = { mc_hash = h;
                      mc_eqs = eqs';
                      mc_sign0 = sign2;
                      mc_vubs = vubs;
                      mc_sign1 = sign7;
                      mc_vn = vn;
                      mc_str = str3; }

      in (en', Mod_compile (m, mc_body))

  | Mod_alias (m, ma_body) ->
      let def_of_m' = try ModnameMap.find ma_body.ma_modname cm with Not_found -> assert false in
      ( match def_of_m' with
        | Mod_compile(_,body) -> (en, Mod_compile(m,body))
        | Import_compile(_,body) -> (en, Import_compile(m,body))
        | _ -> raise (Never_happen "hashify of modalias: something weird is going on") )

  | Import_user (m, mode, iu_body)  ->
      let sign = iu_body.iu_sign in
      let likestr_of_m' m' =
        let def_of_m' = try ModnameMap.find m' cm with Not_found -> assert false in
        begin match def_of_m' with
        | Mod_compile (_, body) -> body.mc_str.desc
        | Import_compile (_, body) -> body.ic_likestr.desc
        | _ ->  raise (Never_happen ("[1]there should only be compiled modules and interfaces in cm; got a " ^ Pretty.print_definition (Dump.fresh_printer_state 3 0 false) def_of_m'))
        end
      in
      begin match (recurse, iu_body.iu_vce) with
      |	(Some _, _)
      | (_, VCDotted (_)) ->
          let likestr = no_loc begin match iu_body.iu_likespec with
          | LikeNone ->
              ( match iu_body.iu_mo with
              | None -> []
              | Some m' -> likestr_of_m' m' )
          | LikeMod m' -> likestr_of_m' m'
          | LikeStr str -> (Tysupp.type_flatten_str(rho_str cm str)).desc
          end in
          let likestr' = filter_str_sign likestr (limitdom sign) in
          let sign0 = Tysupp.type_flatten_sign (rho_sign cm sign) in
          let vc = eval_version_constraint cm iu_body.iu_vce in

          let (en',h) =
            ( match recurse with
                | None ->
                    let ih_body = { ih_sign0 = sign0;
                                    ih_vce = vc;
                                    ih_likestr = likestr'; }
                    in (match mode with
                          | MHash | MBangHash -> (en, Pretty.mkHashI Econst.string_of_ident (modname_ext m, ih_body))
                          | MCfresh | MFresh | MBangCfresh ->
                              let n = new_abstract_name ()
                              in (addnameenv (NEnv_nimp(n,modname_ext(m),ih_body)) en, HashName n) )
                | Some h -> (en,h) ) in

          let sign1 = Tysupp.type_flatten_sign
              (Tysupp.selfify_signature (Hash(h)) sign0) in

          let ic_body = { ic_hash = h;
                          ic_sign0 = sign0;
                          ic_vubs = vubs;
                          ic_sign1 = sign1;
                          ic_vc = vc;
                          ic_likestr = likestr';
                          ic_resolvespec = iu_body.iu_resolvespec;
                          ic_mo = iu_body.iu_mo; } in
          let definition' = Import_compile (m, ic_body) in
          begin match iu_body.iu_mo with
          | None -> ()
          | Some modname'' ->
              let definition'' = try ModnameMap.find modname'' cm with Not_found -> assert false in
              Linkok.linkok en' definition'' definition'  (* CHECK *)
          end;
          (en', definition')

      | (None, VCNameEqual ahvce) ->
          begin if iu_body.iu_likespec <> LikeNone then
            raise (Never_happen ("Hashify: likespec must be empty and is " ^ Pretty.print_likespec (Dump.fresh_printer_state 3 0 false) iu_body.iu_likespec ^ "\n for definition:\n"
				 ^ Pretty.print_source_definition (Dump.fresh_printer_state 3 0 false) d))
          end;
          let m' = match ahvce with
          | AVCHHash _ -> raise (Never_happen ("Hashify:Cannot have hash = literal") 		(* TODO: does the parser get this? (in which case it's never_happen) Similarly empty thingy above.*))
          | AVCHModname m' -> m' in
          let h =
	    try lookup_hash cm m'
	    with _ -> let (ext, (iid, pn)) = m' in raise (Never_happen ("couldn't find modname " ^ ext))
	  in
          let iu_body1 = { iu_body with
                           iu_likespec = LikeMod m';
                           iu_vce = VCNameEqual (AVCHModname m');
                         } in
          let def1 = Import_user (m, mode, iu_body1) in
          let (en', def1') = hashify_def' en cm def1 vubs (Some h) in
            (en', def1')
      end

  | Mark_user mk -> assert false (* cannot happen *) (* TODO: raise never_happen? *)


let hashify_sdef en cm d vubs =
  match d with
    | Mark_user mk ->
	(en, Mark_compile mk)
    | Mod_user _ |  Mod_alias _ | Import_user _ ->
	hashify_def' en cm d vubs None

let hashify_fresh_def en cm d vubs =
  let d' = ( match d with
               | Mod_fresh (m,b) -> Mod_user (m,MFresh,b)
               | Import_fresh (m,b) -> Import_user (m,MFresh,b)
               | _ -> raise (Never_happen "hashify invoked on a definition already compiled") )
  in hashify_sdef en cm d' vubs

(*
let hashify_program cm (defs, expr) =
  let (defs', cm') = hashify_defs cm defs in
  let expr' = rho_expr cm' expr in
  (defs', expr')
*)
