(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * Evaluator support code                                         *
 *                                                                *
 * Version: $Id: evalsupp.ml,v 1.554 2004/12/22 12:23:31 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)


(* the code in this module was once in Ast, but due to module
   dependencies it has had to be separated off. *)


open Ast
open Tysupp
open Util
open Pretty


(* -=-- ---------- ------------------------------------ ---------- --=- *)
(*                     bracket optimisations                            *)

(* insert bracket only if necessary, if this HACK is allowed *)
let maybe_col : eqs -> prim_expr * eqs * typ -> prim_expr
    = fun eqs0 (e,eqs,ty) ->
      if !Opts.hack_optimise && eqs_eq eqs0 eqs then
        (* no need for bracket since outer and inner eqs are the same *)
        e
      else if !Opts.really_hack_optimise then e
      else
        Col(e,eqs,ty);;

(* insert unnecessary bracket (inner and outer eqs known to be the same) only if hack_optimise is off *)
let maybe_col_eqeq : prim_expr * eqs * typ -> prim_expr
    = fun (e,eqs,ty) ->
      if !Opts.hack_optimise then
        (* no need for bracket *)
        e
      else
        Col(e,eqs,ty);;

(* append bracket to sequence only if necessary, if this HACK is allowed *)
let maybe_cons_bs : eqs -> (eqs * typ) -> bracket_seq -> bracket_seq
    = fun eqs0 (eqs,ty) bs ->
      let eqs00 = match bs with
        (eqs00,_)::_ -> eqs00
      | []           -> eqs0
      in
      if !Opts.hack_optimise && eqs_eq eqs00 eqs then
        bs
      else
        (eqs,ty)::bs


(* -=-- ---------- ------------------------------------ ---------- --=- *)
(*                   expression catamorphisms                           *)

(* A freshening walk over the prim_expr, using an in-scope map for
 * efficiency rather than swaps.  The strategy is to create a fresh
 * name for each binder encountered, but not to replace occurrences
 * until they are encountered.  In the meantime, we keep an in-scope
 * map from old names to new names - in fact, we map instead to
 * *expressions* in order to (slightly) improve sharing.
 * Handily, this map can also be used for substitutions.
 *
 * For most expressions, we just freshen.  For InEnv, Clos, and TClos, we
 * invoke the argument "fe" function, since it's not clear what
 * else we could do, and different uses do different things here.
 *)

let new_bind : etysubst -> internal_ident -> etysubst * internal_ident
  = fun (menv,tenv) iid ->
    let iid' = fresh_prettyname iid in
    ((IIdentMap.add iid (Id(iid')) menv,tenv), iid')

let new_tybind : etysubst -> internal_typname -> etysubst * internal_typname
  = fun (menv,tenv) itn ->
    let itn' = fresh_prettyname itn in
    ((menv, ITypnameMap.add itn (TVar(itn')) tenv), itn')

let lookup_bind : etysubst -> internal_ident -> prim_expr option
  = fun (menv,_) iid -> maybe (IIdentMap.find iid) menv

let lookup_tybind : etysubst -> internal_typname -> typ option
  = fun (_,tenv) itn -> maybe (ITypnameMap.find itn) tenv

let rec map' : ('b -> 'a -> 'd -> 'a * 'c) -> 'b list -> 'a -> 'd -> 'a * 'c list
  = fun f xs st st2 ->
    match xs with
      [] -> (st, [])
    | (x::xs) -> let (st',x') = f x st st2 in
                 let (st'',xs') = map' f xs st' st2 in
                 (st'',x'::xs')

let rec early_emap_fresh_typ : 'env -> eqs option -> typ -> typ
  = fun env eqs ty -> match ty with
    | TTyCon0 _          -> ty
    | TTyCon1 (tc1,t0)   -> TTyCon1 (tc1,early_emap_fresh_typ env eqs t0)
    | TTup ts            -> TTup (List.map (early_emap_fresh_typ env eqs) ts)
    | TSum ts            -> TSum (List.map (early_emap_fresh_typ env eqs) ts)
    | TFunc (t1, t2)     -> TFunc (early_emap_fresh_typ env eqs t1, early_emap_fresh_typ env eqs t2)
    | TXDot _            -> ty
    | TTyName _          -> ty
    | TForall (itn, t0)  -> let (env',itn') = new_tybind env itn in
                            TForall (itn', early_emap_fresh_typ env' eqs t0)
    | TExists (itn, t0)  -> let (env',itn') = new_tybind env itn in
                            TExists (itn', early_emap_fresh_typ env' eqs t0)
    | TVar itn           -> (match lookup_tybind env itn with
                               Some ty' -> early_emap_fresh_typ env eqs ty'  (* might need more freshening *)
                             | None     -> ty)

let early_emap_fresh_con0 : 'env -> eqs option -> con0 -> con0
  = fun env eqs c0 ->
    match c0 with
    | Nil(ty)     -> Nil (early_emap_fresh_typ env eqs ty)
    | NONE(ty)    -> NONE (early_emap_fresh_typ env eqs ty)
    | B0 _        -> c0

let early_emap_fresh_con1 : 'env -> eqs option -> con1 -> con1
  = fun env eqs c1 ->
    match c1 with
    SOME
  | TieCon
  | Node          -> c1
  | Inj(i,ty)     -> Inj(i,early_emap_fresh_typ env eqs ty)
  | B1 _          -> c1

let early_emap_fresh_lazy_op : 'env -> eqs option -> lazy_op -> lazy_op
  = fun env eqs op ->
    match op with
    | LoAnd
    | LoOr   -> op

let early_emap_fresh_op : 'env -> eqs option -> op -> op
  = fun env eqs op ->
    match op with
    | Ref(ty)     -> Ref(early_emap_fresh_typ env eqs ty)
    | Deref(ty)   -> Deref(early_emap_fresh_typ env eqs ty)
    | Assign(ty)  -> Assign(early_emap_fresh_typ env eqs ty)
    | Assign'(ty) -> Assign'(early_emap_fresh_typ env eqs ty)
    | Equal(ty)   -> Equal(early_emap_fresh_typ env eqs ty)

    | Less | LessEq | Greater | GreaterEq | Mod | Land | Lor | Lxor
    | Lsl | Lsr | Asr | UMinus | Plus | Minus | Times
    | Divide -> op

    | ListConcat(ty)   -> ListConcat(early_emap_fresh_typ env eqs ty)
    | StrConcat        -> op
    | CompareName(ty)  -> CompareName(early_emap_fresh_typ env eqs ty)
    | NameToString(ty) -> NameToString(early_emap_fresh_typ env eqs ty)
    | CreateThread(ty) -> CreateThread(early_emap_fresh_typ env eqs ty)

    | Self | Kill | CreateMutex | Lock | TryLock | Unlock | CreateCVar
    | Wait | Waiting | Signal | Broadcast | Thunkify
    | Unthunkify       -> op

    | Exit(ty)         -> Exit(early_emap_fresh_typ env eqs ty)

let early_emap_fresh_oe : 'env -> eqs option -> op_or_econst -> op_or_econst
  = fun env eqs oe ->
    match oe with
    | OEOp(op)   -> OEOp(early_emap_fresh_op env eqs op)
    | OEEconst _ -> oe

(* freshen pattern binders *)
let rec early_emap_fresh0_pat : pat -> 'env -> eqs option -> 'env * pat
  = fun p env eqs ->
    match p with
    | PWild ty     -> (env, PWild (early_emap_fresh_typ env eqs ty))
    | PVar(x,ty)   -> let (env',x') = new_bind env x in (env',PVar(x', early_emap_fresh_typ env eqs ty))
    | PC0 c0       -> (env, PC0 (early_emap_fresh_con0 env eqs c0))
    | PC1(c1,p)    -> let (env',p') = early_emap_fresh0_pat p env eqs in (env',PC1(early_emap_fresh_con1 env eqs c1,p'))
    | PCons(p1,p2) -> let (env',p1') = early_emap_fresh0_pat p1 env eqs in
                      let (env'',p2') = early_emap_fresh0_pat p2 env' eqs in
                      (env'',PCons(p1',p2'))
    | PTup(ps)     -> let (env',ps') = map' early_emap_fresh0_pat ps env eqs in
                      (env',PTup(ps'))
    | PTyped(p,ty) -> let (env',p') = early_emap_fresh0_pat p env eqs in (env',PTyped(p', early_emap_fresh_typ env eqs ty))

let rec early_emap_fresh_prim_expr : 'env
                                  -> ('env -> eqs option -> prim_expr -> prim_expr)
                                  -> eqs option
                                  -> prim_expr
                                  -> prim_expr
= fun env fe eqs e ->
  begin
    let go env e0 = early_emap_fresh_prim_expr env fe eqs e0  (* I'm sick of typing that *)
    in
    let goT env t0 = early_emap_fresh_typ env eqs t0
    in
    match e with
      (* first, the cases we can't handle ourselves *)
    | InEnv _  -> fe env eqs e
    | Clos _   -> fe env eqs e
    | TClos _  -> fe env eqs e
      (* the binding constructs *)
    | Fn ({desc=[(PVar(x1,ty1),e1)]} as m) ->
        let (env',x1') = new_bind env x1 in
        Fn {m with desc=[(PVar(x1',goT env ty1),go env' e1)]}
    | TAbs(itn,e1) ->
        let (env',itn') = new_tybind env itn in
        TAbs(itn', go env' e1)
    | Letrec(ty1,(x1,({desc=[(PVar(x2,ty2),e2)]} as m,e1))) ->
        let (env',x1') = new_bind env x1 in
        let (env'',x2') = new_bind env' x2 in
        Letrec(goT env ty1,(x1',({m with desc=[(PVar(x2',goT env' ty2),go env'' e2)]},go env' e1)))
    | Namecase(e1,itn,x1,x2,e0,e2,e3) ->
        let (env',itn') = new_tybind env itn in
        let (env'',x1') = new_bind env' x1 in
        let (env''',x2') = new_bind env'' x2 in
        Namecase(go env e1,itn',x1',x2',go env e0,go env''' e2,go env e3)
    | Unpack(itn,x1,e1,e2) ->
        let (env',itn') = new_tybind env itn in
        let (env'',x1') = new_bind env' x1 in
        Unpack(itn',x1',go env e1,go env'' e2)
      (* bound occurrence of term variable *)
    | Id(iid) ->
        (match lookup_bind env iid with
          Some e' -> go env e'  (* might need more freshening *)
        | None -> e)
      (* sugared forms - not allowed *)
    | Fn _          -> raise (Never_happen "early_emap_fresh_prim_expr: sugared form Fn")
    | Fun _         -> raise (Never_happen "early_emap_fresh_prim_expr: sugared form Fun")
    | Letrec _      -> raise (Never_happen "early_emap_fresh_prim_expr: sugared form Letrec")
    | Let _         -> raise (Never_happen "early_emap_fresh_prim_expr: sugared form Let")
    | LetMulti _    -> raise (Never_happen "early_emap_fresh_prim_expr: sugared form LetMulti")
    | LetrecMulti _ -> raise (Never_happen "early_emap_fresh_prim_expr: sugared form Let")
      (* hoi polloi *)
    | C0 c0                  -> C0 (early_emap_fresh_con0 env eqs c0)
    | C1 (c1, e0)            -> C1 (early_emap_fresh_con1 env eqs c1, go env e0)
    | Cons (e1, e2)          -> Cons (go env e1, go env e2)
    | Tup es                 -> Tup (List.map (go env) es)
    | Op (oe,el)             -> Op (early_emap_fresh_oe env eqs oe, (List.map (go env) el))
    | Loc _                  -> e  (* TODO fix this when locations gain a type annotation *)
    | Dot _                  -> e
    | HashDot _              -> e
    | If (e1, e2, e3)        -> If (go env e1, go env e2, go env e3)
    | While (e1, e2)         -> While (go env e1, go env e2)
    | LazyOp (lo,es)         -> LazyOp (early_emap_fresh_lazy_op env eqs lo, (List.map (go env) es))
    | Seq (e1, e2)           -> Seq (go env e1, go env e2)
    | App (e1, e2)           -> App (go env e1, go env e2)
    | Match(e1,mtch)         -> Match(go env e1, early_emap_fresh_prim_mtch env fe eqs mtch)
    | Raise (e0)             -> Raise (go env e0)
    | Try(e1,mtch)           -> Try(go env e1, early_emap_fresh_prim_mtch env fe eqs mtch)
    | Marshal (e1, e2, t)    -> Marshal (go env e1, go env e2, goT env t)
    | Marshalz (mk, e2, t)   -> Marshalz (mk, early_emap_fresh_prim_expr env fe (Some []) e2, goT env t)
    | Unmarshal (e0, t)      -> Unmarshal (go env e0, goT env t)
    | RET ty                 -> RET (goT env ty)
    | Col (e0, eqs', t)      -> Col (early_emap_fresh_prim_expr env fe (Some eqs') e0, eqs', goT env t)  (* NB: we don't need to tysubst insides eqs; they are ground by construction *)
    | OP (n, oe, es)         -> OP (n, early_emap_fresh_oe env eqs oe, List.map (early_emap_fresh_prim_expr env fe (Some [])) es)
    | Resolve(e,mn,rs)       -> Resolve(go env e,mn,rs)
    | Resolve_blocked(e,mn,rs) -> Resolve_blocked(go env e,mn,rs)
    | ValOfTie e0            -> ValOfTie (go env e0)
    | NameOfTie e0           -> NameOfTie (go env e0)
    | Tie _                  -> e
    | Support(ty,e0)         -> Support(goT env ty, go env e0)
    | Freshfor(e1,e2)        -> Freshfor(go env e1, go env e2)
    | Swap(e1,e2,e3)         -> Swap(go env e1, go env e2, go env e3)
    | NameValue _            -> e  (* don't descend into types inside; they must be closed already *)
    | HashHts(ty1,e1,e2,ty2) -> HashHts(goT env ty1, go env e1, go env e2, goT env ty2)
    | HashTs(ty1,e1,ty2)     -> HashTs(goT env ty1, go env e1, goT env ty2)
    | HashMvf(x,eid,ty)      -> HashMvf(x,eid,goT env ty)
    | CFresh(ty)             -> CFresh(goT env ty)
    | Fresh(ty)              -> Fresh(goT env ty)
    | Par(e1,e2)             -> Par(go env e1, go env e2)
    | SLOWRET(ty)            -> SLOWRET(goT env ty)
    | Pack(ty1,e1,ty2)       -> Pack(goT env ty1, go env e1, goT env ty2)
    | TApp(e1,ty1)           -> TApp(go env e1, goT env ty1)

  end


and early_emap_fresh_prim_mtch env fe eqs mtch =
  {mtch with desc=List.map (fun (p,e) -> let (env',p') = early_emap_fresh0_pat p env eqs in (p', early_emap_fresh_prim_expr env' fe eqs e)) mtch.desc}



(* esub_expr perform a simultaneous internal ident to term expression
substitution on an expression; doesn't go inside hashes; doesn't go
inside closures or inenvs - on the basis that these are
supposed to be closed...  *)

(* helper - internal use only *)
let esub_fresh_expr0 : etysubst -> eqs option -> prim_expr -> prim_expr =
  fun esubs eqso ex ->
    let eqs = the' "esub_expr0" eqso in
    match ex with
    | Clos _  -> ex
    | TClos _ -> ex
    | InEnv _ -> ex
    | _       -> raise (Util.Never_happen "esub_expr0")



(* helper - internal use only *)
let esub_expr0 : etysubst -> eqs option -> prim_expr -> prim_expr =
  fun esubs eqso ex ->
    let eqs = the' "esub_expr0" eqso in
    match ex with
    | Clos _  -> ex
    | TClos _ -> ex
    | InEnv _ -> ex
    | _       -> raise (Util.Never_happen "esub_expr0")



let esub_expr subs eqs e =
  Debug.print(function () -> "Mapping...");
   early_emap_fresh_prim_expr subs esub_expr0 (Some eqs) e
  (* early_emap_prim_expr false (esub_expr0 subs) (Some eqs) e *)


(* apply a bracket sequence.  Note that the sequence is stored in reverse order,
   so the head bracket is the innermost *)
let rec apply_bs bs e =
  match bs with
  | []              -> e
  | ((eqs,ty)::bs') -> apply_bs bs' (Col (e,eqs,ty))

(* flatten all closures in an expression *)

(* OLD COMMENT:
   this substitution environment maps an ident to either an expression or a
   bracket_seq; in the former case, the expression is inserted and recursion
   continues latter case; in the latter, the bracket_seq is wrapped around the
   ident and recursion stops. *)
(* In fact, it simply takes an ordinary substitution environment; the
   use of a fresh identifier in the range of the map ensures we don't get
   into nasty infinite loops. *)

let rec flattenclos : etysubst -> eqs -> prim_expr -> prim_expr
  = fun env0 eqs0 e0
 -> (* bitrot - awaiting basic_print_tysubst
     *  Debug.print' Opts.DBC_flattenclos
     *  (fun () -> "flattenclos():\n" ^ !basic_print_esubst env0 ^"\n"
     *    ^ !basic_print_prim_expr e0);
     *)
    early_emap_fresh_prim_expr
     env0
     (fun env eqso e ->
       Debug.print' Opts.DBC_flattenclos (fun () -> "=>fc0:\n" ^ !basic_print_prim_expr e);
       let eqs = the' "flattenclos" eqso in
       match e with
       (* Ids get mapped automatically; we just maintain the right substitution *)
       | InEnv(env',e') ->
           flattenclos env' eqs e'
       | Clos(env',x2,ty2,bs2,e1,None) ->
           let ((menv',tenv') as env') = Lazy.force env' in
           let x2' = fresh_prettyname x2 in
           Fn {desc=[(PVar(x2',early_emap_fresh_typ env' eqso ty2),
                      flattenclos
                        (IIdentMap.add x2
                           (apply_bs bs2 (Id(x2'))) menv',tenv') eqs e1)
                   ];loc=Location.none}
       | Clos(env',x2,ty2,bs2,e1,Some(x1,ty1,bs1)) ->
           let ((menv',tenv') as env') = Lazy.force env' in
           let menv_del = IIdentMap.remove x1 menv' in
           let x1' = fresh_prettyname x1 in
           let x2' = fresh_prettyname x2 in
           Letrec(early_emap_fresh_typ env' eqso ty1,
                  (x1',(no_loc [(PVar(x2',early_emap_fresh_typ env' eqso ty2),
                                 flattenclos (IIdentMap.add x2 (apply_bs bs2 (Id(x2')))
                                                (IIdentMap.add x1 (apply_bs bs1 (Id(x1')))
                                                   menv_del), tenv')
                                             eqs
                                             e1)],
                        Id(x1'))))
       | TClos((menv',tenv'),itn,e') ->
           let itn' = fresh_prettyname itn in
           TAbs(itn',flattenclos (menv', ITypnameMap.add itn (TVar itn') tenv') eqs e')
       | _ -> raise (Util.Never_happen "flattenclos:fe")
     )
     (Some eqs0)
     e0




(* entry points *)
let flatten_all_closures eqs e = flattenclos (IIdentMap.empty,ITypnameMap.empty) eqs e

let rec def_flatten_all_closures d = (match d with (* TODO *)
    Mod_compile (mn, mcb)    -> Mod_compile (mn, { mcb with
                                  mc_str = flatten_struct mcb.mc_eqs mcb.mc_str})
  | Mod_imod (mn, mib)       -> Mod_imod (mn, { mib with
                                  mi_str_done = List.map (flatten_str_item mib.mi_eqs) mib.mi_str_done;
                                  mi_str_todo = List.map (flatten_str_item mib.mi_eqs) mib.mi_str_todo})
  | Import_compile (mn, icb) -> d  (* no closures in here! *)
  | Mod_fresh (mn, mub)      -> d  (* no closures in here! *)
  | Import_fresh (mn, iub)   -> d  (* no closures in here! *)
  | Mark_compile e           -> d)

and flatten_str_item_desc eqs (d: structure_item_desc) = match d with
       StrVal      (x, e)     -> StrVal(x, primtoexpr (flatten_all_closures eqs (exprtoprim e)))
     | StrValMulti (x, ps, e) -> StrValMulti(x, ps, primtoexpr (flatten_all_closures eqs (exprtoprim e)))
     | StrTyp _               -> d

and flatten_str_item eqs (s: structure_item) = { s with desc = flatten_str_item_desc eqs s.desc }
and flatten_struct eqs (s: structure) = { s with desc = List.map (flatten_str_item eqs) s.desc }


let printer_state = ref None

let ps = function () -> match !printer_state with
  None -> printer_state := Some (Dump.fresh_printer_state !Opts.dumptrace 0 !Opts.dumptex); Util.the(!printer_state)
| Some ps -> ps

let subst_and_flatten_all_closures env eqs e =
 Debug.print' Opts.DBC_evalstep (fun () ->
   "subst_and_flatten_all_closures:\n"
   ^ print_etysubst (ps ()) env ^ "\n"
   ^ print_eqs (ps ()) eqs ^ "\n"
   ^ print_prim_expr (ps ()) e);
  let r = flattenclos env eqs e in
  Debug.print' Opts.DBC_evalstep (fun () -> "subst_and_flatten_all_closures complete.");
  r

(* -=-- ---------- ------------------------------------ ---------- --=- *)
(*                     expression size                                  *)

(* number of AST nodes *)
let size_of_prim_expr : prim_expr -> int
  = fun e0
 -> let r = ref 0 in
    let rec f _ e =
      r := !r + 1;
      match e with
        InEnv(env,e')               -> Some (early_emap_prim_expr false f None e')
      | Clos(env,x2,ty2,bs2,e1,x1o) -> Some (early_emap_prim_expr false f None e1)
      |	TClos(env, x1, e')           -> Some (early_emap_prim_expr false f None e')
      | _                           -> None
    in
    let (_:prim_expr) = early_emap_prim_expr false f None e0 in
    !r


(* TODO: follow locations?
         currently caller deals with store (calling freenames on elements of the store it wants to)
         but perhaps cleaner to pass in store and follow it here *)

let free_names : prim_expr -> name_value list
    = fun e0 ->
      let r = ref [] in

      let f _ e = match e with
	NameValue(n) ->
	  r := (n::(!r));
	  None (* NB: if n is a VHashHts, we do not add the inner namevalue *)
      |	_ -> None
      in
      let e' = early_emap_prim_expr false f None e0 in
      !r

let rec print_list locs =
  match locs with [] -> ""
    | (e::es) -> string_of_int e ^ ", " ^ (print_list es)

(* doesn't go inside modules *)
let reachable_locs expr store =
  let rec reachable_locs es locs =
    match es with [] -> locs
    | es -> let locs' = List.concat (List.map (function e -> prim_free_locs (flatten_all_closures [] e)) es) in (* all the locations from the list of expressions *)
       Debug.print' Opts.DBC_marshal (function () -> "[1]Reachable store:" ^
      (print_list locs'));
      let locs0_dup = (remove_duplicates (function l1 -> function l2 -> if (l1 > l2) then 1 else if (l2 > l1) then -1 else 0) locs') in  (* filter out duplicates and anything that's in the original list *) Debug.print' Opts.DBC_marshal (function () -> "[2]Reachable store:" ^
      (print_list locs0_dup));
      let locs0 = List.filter (fun l -> not(mem_by l_eq l locs)) locs0_dup in
       Debug.print' Opts.DBC_marshal (function () -> "[2]Reachable store:" ^
      (print_list locs0));
      let exprs = List.map (fun l -> assoc_by l_eq l store) locs0 in
      let exprs0 = List.map (function e -> flatten_all_closures [] e) exprs in
       Debug.print' Opts.DBC_marshal (function () -> "[3] " ^ string_of_int (List.length exprs0) ^ " exprs");
      reachable_locs exprs0 (locs @ locs0)
  in reachable_locs [expr] []


  (* ---------- evalcfresh ------------ *)

let rec evalcfresh : nameenv * expr -> nameenv * expr
    = fun (en, e0) ->
      let l = e0.loc
      in match e0.desc with
	| LocCFresh ty ->
	    let (en',n) = new_name_value en ty
	    in (en', at_loc l (LocNameValue n))
	| LocC1 (c,e) ->
	    let (en',e1) = evalcfresh (en,e)
	    in (en', at_loc l (LocC1 (c, e1)))
	| LocHashTs (ty1,e,ty2) ->
	    let (en',e1) = evalcfresh (en,e)
	    in (en', at_loc l (LocHashTs (ty1, e1,ty2)))
	| LocPack (ty1, e, ty2) ->
	    let (en',e1) = evalcfresh (en,e)
	    in (en', at_loc l (LocPack (ty1, e1, ty2)))
	| LocCons (e1, e2) ->
	    let (en1',e1') = evalcfresh (en,e1)
	    in let (en2',e2') = evalcfresh (en1',e2)
	    in (en2', at_loc l (LocCons (e1', e2')))
	| LocHashHts (t1, e2, e1, t2) ->
	    let (en1',e2') = evalcfresh (en,e2)
	    in let (en2',e1') = evalcfresh (en1',e1)
	    in (en2', at_loc l (LocHashHts (t1, e2', e1', t2)))
	| LocTup (el) ->
	    let f (n,l) e =
	      let (en1,e') = evalcfresh (n,e)
	      in (en1,e'::l)
	    in let (en2,rel') = List.fold_left f (en,[]) el
	    in (en2, at_loc l (LocTup (el)))
	| _ -> (en, e0)

let evalcfresh_str_item : nameenv * structure_item -> nameenv * structure_item
  = fun (en, str_it) ->
    let l = str_it.loc
    in match str_it.desc with
      | StrVal (i,e) ->
          let (en1,e') = evalcfresh (en,e)
          in (en1, at_loc l (StrVal(i,e')))
     | StrValMulti (i,pl,e) ->    (* FZ: we do not need to recurse inside pat list *)
         let (en1,e') = evalcfresh (en,e)
         in (en1, at_loc l (StrValMulti(i,pl,e')))
     | StrTyp (tn, t) -> (en, at_loc l (StrTyp (tn,t)))

let evalcfresh_str : nameenv * structure -> nameenv * structure
  = fun (en, str) ->
    let l = str.loc
    in let f (n,l) s =
        let (en1,s') = evalcfresh_str_item (n,s)
        in (en1,s'::l)
    in let (en2,strlist_rev) = List.fold_left f (en,[]) str.desc
    in (en2, at_loc l (List.rev strlist_rev))

let evalcfresh_sdef : nameenv * source_definition -> nameenv * source_definition
  = fun (en, sdef) ->
    match sdef with
      | Mod_user (mn,mode,body) ->
          let (en',str') = evalcfresh_str (en,body.mu_str)
          in (en', Mod_user(mn,mode,{ body with mu_str = str' }))
      | _ -> (en, sdef)



(* FZ the code below is disgusting - I will rewrite it, promised *)
let compatible_nenv_entries
  = fun a b ->
    let ps = initial_printer_state None pm_hash in
    Digest.string(Pretty.wrap(Pretty.pp_print_nameenv_entry) ps a)
   =
    Digest.string(Pretty.wrap(Pretty.pp_print_nameenv_entry) ps b)

(* XXX inefficient: should try to make n log n rather than n^2;
   careful though not to change the order of the original environments...

   J suggested the following pseudocode:

 merge_nenv (nenv1, nenv2) =
   let nenv1' = sort nenv1 in
   let nenv2' = sort nenv2 in
   let d1 = make_set_of_domain nenv1 in
   if compatible nenv1' nevn2 then
     concatenate_nevs nenv1 (filter_out_set nenv2 d1)
   else
     error

*)

let merge_nameenv : nameenv * nameenv -> nameenv
  = fun (en1,en2) ->
    let (en1_l, en2_l) = (nameenv_list_of_nameenv en1, nameenv_list_of_nameenv en2)
    in let rec merge_env_aux tmp e2l =
      match e2l with
        | [] -> tmp
        | b::tl ->
            let nb = name_of_nameenv_entry b
            in try
                let a = List.find (fun x -> nb = (name_of_nameenv_entry x)) en1_l
                in if (compatible_nenv_entries a b)
                  then merge_env_aux tmp tl
                  else raise (Never_happen "merge of incompatible name environments")
                       (* TODO:
			  should raise some other error to be caught by caller
			  and turned into relevant failure *)
            with Not_found -> merge_env_aux (b::tmp) tl
    in let tmp = List.rev en1_l
    in nameenv_nameenv_of_list(List.rev(merge_env_aux tmp en2_l))


(* substitute type environment into type *)
let get_typ env eqs t =
  early_emap_fresh_typ env (Some eqs) t

(* substitute type environment into value *)
let get_val (menv,tenv) eqs v =
  esub_expr (IIdentMap.empty,tenv) eqs v
