(* -=-- ---------------------------------------------------- --=- *
 *                                                                *
 * desugarer                                                      *
 *                                                                *
 * Version: $Id: desugar.ml,v 1.512 2004/12/22 12:23:31 zappa Exp $
 *                                                                *
*** Copyright 2002-2004 The Acute Team

  Allen-Williams, Mair
  Bishop, Steven
  Fairbairn, Matthew
  Habouzit, Pierre [*]
  Leifer, James [*]
  Sewell, Peter
  Sjberg, Vilhelm
  Steinruecken, Christian
  Vafeiadis, Viktor
  Wansbrough, Keith
  Zappa Nardelli, Francesco [*]
  Institut National de Recherche en Informatique et en Automatique (INRIA)

  Contributions of authors marked [*] are copyright INRIA.

All rights reserved.

This file is distributed under the terms of the GNU Lesser General
Public License, with the special exception on linking described in
file NEW-LICENSE.

***
 * -=-- ---------------------------------------------------- --=- *)


open Ast
open Tysupp
open Basecon

(* Eta-expanding and desugaring expression are done in two separate *)
(* phases.  The algorithm used in eta-expansion relies on being *)
(* applied only once to an expression. *)

let rec eta_expand = fun e ->

  let expand_app e el =
    List.fold_right (fun e2 e1 -> gh_loc e.loc (LocApp(e1,e2))) (List.rev(el)) e
  in

  let create_var_list n tl =
    let rec create_var_list_int n vl al =
      if n=0
      then (vl,al)
      else
        let x = fresh_internal_ident "ds" in
        let t = List.nth tl (List.length(tl) - n) in
        create_var_list_int (n-1) (PVar(x,t)::vl) (no_loc (LocId(x))::al)
    in create_var_list_int n [] []
  in

  let expand_fun_op const op el n tl =
    let (vl,al) = create_var_list n tl
    in List.fold_right (fun v e -> no_loc (LocFn([v,e]))) (List.rev(vl)) (no_loc (const(op,el@List.rev(al))))
  in

  let expand_fun_econst id el n tl =
    let (vl,al) = create_var_list n tl
    in List.fold_right (fun v e -> no_loc (LocFn([v,e]))) (List.rev(vl)) (expand_app (no_loc (LocId(id))) (el@List.rev(al)))
  in

  let rec eta_expand_int e n el =
    match e.desc with
    | LocApp(e1,e2) -> eta_expand_int e1 (n+1) ((eta_expand e2)::el)
    | LocOp(oe,[]) ->
        let arity = arity_of_op_or_econst oe in
        if n>arity
        then
          let (el1,el2) = Util.splitAt arity el in
          Some(expand_app (at_loc e.loc (LocOp(oe,el1))) el2)
        else
          Some(expand_fun_op (fun (x,y) -> LocOp(x,y)) oe el (arity-n) (fst (info_of_op_or_econst oe)))
    | LocOp(_,h::t) ->
        raise (Util.Never_happen "desugar: op or econst applied to arguments in source")
    | LocLazyOp(lo,[]) ->
        let arity = 2 in
        if n>arity
        then
          let (el1,el2) = Util.splitAt arity el in
          Some(expand_app (at_loc e.loc (LocLazyOp(lo,el1))) el2)
        else
          Some(expand_fun_op (fun (x,y) -> LocLazyOp(x,y)) lo el (arity-n) [TTyCon0 TBool;TTyCon0 TBool])
    | LocLazyOp(_,h::t) ->
        raise (Util.Never_happen "desugar: lazyop applied to arguments in source")
    | _ ->
        match el with
        | [] -> None
        | _ -> Some(expand_app (eta_expand e) el)

  in early_emap_expr false (fun _ e -> eta_expand_int e 0 []) None e

(* a conservative approximation *)
let is_pattern_exhaustive p = match p with
  | PVar _        -> true
  | PWild _       -> true
  | PC0 (B0 Unit) -> true
  | _             -> false

(* a conservative approximation *)
let rec is_match_exhaustive ms = match ms with
    []         -> false
  | (p,_)::ms0 -> is_pattern_exhaustive p || is_match_exhaustive ms0


(* NB: no freshening required, because we do not alter the binding structure other
   than to introduce new globally-fresh binders *)

let rec desugar_expr : expr -> expr           (* performs also eta_expantion *)
  =fun e -> desugar_expr_really(eta_expand e)


and desugar_expr_really e =
  early_emap_expr false
    (fun _ e ->
      Util.option_lift (at_loc e.loc)
      (match e.desc with
        LocFn mtch ->
          (match desugarmtch true e.loc mtch with
            None -> None
          | Some mtch -> Some (LocFn mtch))
      | LocFun (pats, e') -> let fns = desugar_pats_expr e.loc (pats, e') in Some fns.desc
      | LocLet (e1,(p,e2)) ->
          let l = (p,desugar_expr_really e2) ::
                  (if is_pattern_exhaustive p then []
                   else [PVar(fresh_internal_ident "ds", patty p),
                         primtoexpr (Raise(C1(B1 MatchFail, expr_of_loc e.loc)))]) in
          Some (LocMatch(desugar_expr_really e1, l))
(* should never get a match failure here? *)
      | LocLetMulti (ty, (pats, e'), (x,e'')) ->
          let fns = desugar_pats_expr e.loc (pats, e') in
          Some (LocMatch (fns, [(PVar(x,ty), desugar_expr_really e'')]))
      | LocLetrecMulti (ty, (pat::pats, (x,(e',e'')))) ->
          let fns = desugar_pats_expr e.loc (pats, e') in
          let mtch = match desugarmtch false e.loc [(pat,fns)] with
            None -> [(pat,fns)]
          | Some mtch -> mtch in
          Some (LocLetrec (ty, (x,(mtch, desugar_expr_really e''))))
      | LocLetrecMulti (_, ([], _)) ->
          raise (Util.Never_happen "desugar_expr_really of LetrecMulti with empty pats")
      | LocLetrec(ty,(x,(m,e))) ->
              (match desugarmtch true e.loc m with
                None -> None  (* the expression gets treated later in the emap *)
              | Some m' -> Some (LocLetrec (ty,(x,(m', desugar_expr_really(e))))))
      | LocPar (e1, e2) ->
          let e1' = desugar_expr_really e1 in
          let e2' = desugar_expr_really e2 in
          Some (LocSeq({loc = e.loc;
                        desc = LocOp(OEOp (CreateThread (TTyCon0 TUnit)),
                          [{loc=e.loc; desc=LocFresh (TTyCon0 TThread) };
                           {loc=e.loc; desc=LocFn [PVar(fresh_internal_ident "u", TTyCon0 TUnit), e1']};
                           {loc=e.loc; desc=LocC0 (B0 Unit)}])},
                       e2'))

      | _ ->
          None))
    None
    e

and desugar_pats_expr : Location.t -> pat list * expr -> expr
  = fun loc (pats,e')
 -> (* K thinks in fact empty pats is just fine:
     * (match pats with [] -> raise (Util.Never_happen "desugar_pats_expr with empty pats") | _ -> ());
     *)
    let trivial = function PVar(_,_) -> true | _ -> false in
    let make_fn pati expr =
      if trivial pati then
        gh_loc loc (LocFn [(pati,expr)])
      else
        let x = fresh_internal_ident "ds" in
        let ty = patty pati in
        let l = (pati, expr) ::
                (if is_pattern_exhaustive pati then []
                 else [PVar(fresh_internal_ident "ds", ty),
                       primtoexpr (Raise(C1(B1 MatchFail, expr_of_loc loc)))]) in
        gh_loc loc (LocFn [(PVar(x,ty), no_loc (LocMatch (no_loc (LocId x), l)))]) in
    List.fold_right make_fn pats (desugar_expr_really e')

(* do_exprs: if false, then assume RHSs already desugared (don't do them again) *)
and desugarmtch : bool -> Location.t -> mtch -> mtch option
  = fun do_exprs loc mtch
 -> match mtch with
      [(PVar(_,_),_)] ->
        None
    | _ ->
        let x = fresh_internal_ident "ds" in
        let ty = matchty mtch in
	let mtch' = if is_match_exhaustive mtch then mtch
                    else mtch @ [(PVar(fresh_internal_ident "ds", ty),
                                 primtoexpr (Raise(C1(B1 MatchFail, expr_of_loc loc))))] in
        Some [(PVar(x,ty), gh_loc loc (LocMatch(no_loc (LocId(x)),
                                    (if do_exprs then
                                      List.map (fun (p,e) -> (p, desugar_expr_really e)) mtch'
                                    else
                                      mtch'))))]


let desugar_str : structure -> structure   (* performs also eta-expantion *)
  = fun str
 -> let go si = match si.desc with
                  StrVal(i,e) ->
                    at_loc si.loc (StrVal(i,desugar_expr e))
                | StrValMulti(i,pats,e) ->
                    at_loc si.loc (StrVal(i, desugar_pats_expr e.loc (pats,eta_expand e)))
                | StrTyp _    ->
                    si
    in
 {str with desc=List.map go str.desc}

let desugar_sdef : source_definition -> source_definition  (* performs also eta_expantion *)
  = fun d
 -> match d with
      Mod_user(mn,mi,mu)
       -> Mod_user(mn, mi, { mu with mu_str = desugar_str mu.mu_str })
    | Mod_alias _      -> d  (* nothing to desugar *)
    | Import_user _    -> d  (* nothing to desugar *)
    | Mark_user _      -> d  (* nothing to desugar *)


