/***************************************************************************/
/*                                                                         */
/* Program : wmi810 [<opts>]                                               */
/*                                                                         */
/* Function: 1) Adjust watermark register on i810 boards.                  */
/*           2) Set or clear bit 2 of register MEM_MODE                    */
/*                                                                         */
/* V1.00  2001-05-23 cr                                                    */
/*   - Initial version                                                     */
/*                                                                         */
/* V1.01  2001-05-29 cr                                                    */
/*   - Small text changes                                                  */
/*                                                                         */
/* V1.10  2001-06-01 cr                                                    */
/*   - Added options -2c and -2s (MEM_MODE initialization)                 */
/*                                                                         */
/***************************************************************************/

#define     VERS     "V1.10"              /* Version (== edition number)   */
#define     UTIL     "wmi810"             /* Utility name                  */
#define     AUTH     "cr"                 /* Author of utility             */
#define     DATE     "2001-06-01"         /* Date of source modification   */
#define     COPR     "(c)2001"            /* "Copyright" year or years     */


#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <fcntl.h>
#include <string.h>
#include <errno.h>
#include <sys/stat.h>
#include <getopt.h>
#include <asm/types.h>
#include <asm/io.h>
#include <sys/mman.h>


//
// Function prototypes
//
int getflags(int *, char ***, char *, ...);
int iopl(int);
u_long MapMem(u_long StartAddr, u_long ByteCount);
void show_wmbl(u_long wm, u_long mm);
void info();
void syntax();


//
// Globals
//
int f_help = 0;
int f_bit2c = 0;
int f_bit2s = 0;
int f_info = 0;
int f_update = 0;
int f_verbose = 0;
int g_FDmem = -1;


int main(int argc, char **argv)
{
    int flagstat;
    u_long dev;
    u_char i810found = 0;
    u_long i, n, n1, n2, val;
    u_long mmadr;
    u_long mmbl, mmwm;
    u_long WM_MappedAddr;
    u_long *mm, *wm;
    u_long WM_PhyAddr;


    //
    // Customize Args
    //
    flagstat = getflags(&argc, &argv, "?,2c,2s,h,i,u,v",
                        &f_help, &f_bit2c, &f_bit2s, &f_help, &f_info, &f_update, &f_verbose);


    if (f_help || flagstat == 0)    {
        syntax();
        if (f_help)
            exit(1);
        if (!flagstat)  {
            printf("%s: unknown option.\n", argv[0]);
            exit(1);
        }
    }
    if(f_info)    {
        info();
        exit(0);
    }

    if (iopl(3) < 0)    {
        fprintf(stderr, "This program must be run as root.\n");
        return errno;
    }


    outl(0x80000000, 0xCF8); n1 = inl(0xCFC);  // Host Bridge
    if (n1 == 0x71208086 || n1 == 0x71228086 || n1 == 0x71248086 )    {
        outl(0x80000800, 0xCF8); n2 = inl(0xCFC); // Intel i810 VGA
        if (n2 == 0x71218086 || n2 == 0x71238086 || n2 == 0x71258086)
        i810found = 1;
    }

    if (!i810found)    {
        fprintf(stderr, "%s: Board with Intel i810 chipset not found!\n", UTIL);
        exit(EXIT_FAILURE);
    }

    outl(0x80000814, 0xCF8);
    mmadr = inl(0xCFC);
    WM_PhyAddr = mmadr + 0x2000;

    if ((WM_MappedAddr = MapMem(WM_PhyAddr, 4096)) == 0)    {
        fprintf(stderr, "This program must be run as root.\n");
        return 0;
    }

    wm = (unsigned long *) WM_MappedAddr + (0xD8/4);
    mm = (unsigned long *) WM_MappedAddr + (0xDC/4);

    i = *wm;

    if (!f_update && !f_bit2c && !f_bit2s)    {
        fprintf(stderr, "\nCurrent register values:\n");
        show_wmbl(i, *mm);
        fprintf(stderr, "\nFor more information type # wmi810 -i\n");
        exit(0);
    }

    if (f_bit2c && f_bit2s)  {
        fprintf(stderr, "%s: Ambiguous option combination\n", UTIL);
        exit(EXIT_FAILURE);
    }
    if (f_bit2c)    *mm = 0x00000000;
    if (f_bit2s)    *mm = 0x00000004;

    if (f_update)  {
        if (argc == 1)  {
            fprintf(stderr, "%s: Aborted - Arguments missing\n", UTIL);
            exit(EXIT_FAILURE);
        }
    }

    if (argc > 1)   {
        sscanf(argv[1], "%d", &mmwm);
        if (mmwm >= 64) {
            fprintf(stderr, "%s: Display FIFO Watermark (1. Arg) must be in range 0-63\n", UTIL);
            exit(EXIT_FAILURE);
        }
        i = (i & 0xFFFC0FFF) | (mmwm << 12);
    }
    if (argc > 2)   {
        sscanf(argv[2], "%d", &mmbl);
        if (mmbl >= 8)  {
            fprintf(stderr, "%s: Display Burst Length (2. Arg) must be in range 0-7\n", UTIL);
            exit(EXIT_FAILURE);
        }
        i = (i & 0xFF8FFFFF) | (mmbl << 20);
    }

    *wm = i;

    fprintf(stderr, "\nUpdated register values:\n");
    show_wmbl(i, *mm);

    exit(0);
}


void show_wmbl(u_long wm, u_long mm)
{
    if (f_verbose)    {
        printf("\ni810 FIFO Watermark and Burst Length register (MMADR+0x20D8) : 0x%08X\n", wm);
        printf("i810 (undocumented) Memory Mode register      (MMADR+0x20DC) : 0x%08X\n\n", mm);
    }
    printf("MM Display FIFO Watermark   : %d\n", (wm >> 12) & 0x3F);
    printf("MM Display Burst Length     : %d\n", (wm >> 20) & 0x07);
    if (f_verbose)  {
        printf("LM Display FIFO Watermark   : %d\n", (wm >> 0) & 0x3F);
        printf("LM Display Burst Length     : %d\n", (wm >> 8) & 0x07);
    }
    printf("Bit 2 of register MEM_MODE  : %s\n", mm & 0x4 ? "Set (ok)" : "Cleared (should be set to avoid flicker)");
}


u_long MapMem(u_long StartAddr, u_long ByteCount)
{
#define DEV_MEM "/dev/mem"
    u_long vaddr;

    if ((g_FDmem = open(DEV_MEM, O_RDWR)) < 0)
        return 0;

    vaddr = (u_long) mmap((caddr_t)0, (size_t) ByteCount,
                PROT_READ|PROT_WRITE, MAP_SHARED, g_FDmem, (off_t) StartAddr);

    if (vaddr == (u_long) 0xFFFFFFFF)
        return 0;

    return(vaddr);
}


//
// Show syntax:
//
void syntax()
{
    fprintf(stderr, "Syntax: %s [<opts>] <FIFO_Watermark> <Burst_Length>\n", UTIL);
    fprintf(stderr, "Function: Adjust watermark register and bit 2 of reg. MEM_MODE on i810 boards\n");
    fprintf(stderr, "Options:\n");
    fprintf(stderr, "   -2c     Clear bit 2 of register MEM_MODE\n");
    fprintf(stderr, "   -2s     Set bit 2 of register MEM_MODE\n");
    fprintf(stderr, "   -i      Description/information about this utility\n");
    fprintf(stderr, "   -u      Update watermark register (needs two arguments: FIFO_Watermark and\n");
    fprintf(stderr, "           Burst_Length). Ranges: FIFO_Watermark=0-63, Burst_Length=0-7\n");
    fprintf(stderr, "   -v      Verbose (debug mode)\n");
    fprintf(stderr, "%s: %s, %s by EKF Elektronik GmbH, 59065 Hamm, Germany\n", UTIL, VERS, COPR);
    fprintf(stderr, "%s: Modified: %s by %s, Compiled: %s %s\n", UTIL, DATE, AUTH, __DATE__, __TIME__);
}


//
// Info:
//
void info()
{
    fprintf(stderr, "\n");
    fprintf(stderr, "Description of wmi810:\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "This utility has two functions:\n");
    fprintf(stderr, "The main function is to set a bit in an undocumented register in the built-in\n");
    fprintf(stderr, "graphics device of Intels i810 chipset (GMCH). As a secondary function it\n");
    fprintf(stderr, "allows to change the Watermark and Burst-Length register in the same device.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "The current i810 driver of the XFREE-Server 4.0.3 (and older) does not set\n");
    fprintf(stderr, "bit 2 of the undocumented GMCH register called MEM_MODE (MMADR+0x20DC).\n");
    fprintf(stderr, "If this bit is not set, noise and flicker will be seen on the screen (strength\n");
    fprintf(stderr, "of flicker depends on CPU activity, screen resolution, color depth and\n");
    fprintf(stderr, "miscellaneous H/V timings). Even with optimal settings of watermark and\n");
    fprintf(stderr, "burst-length, in some cases this flicker can only be partly reduced.\n");
    fprintf(stderr, "If bit 2 of MEM_MODE is set (# wmi810 -2s), the setting of watermark and\n");
    fprintf(stderr, "burst-length is not critical anymore and normally the default settings of\n");
    fprintf(stderr, "the i810 driver can be used.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Usage (call this utility after the X-Server has been started):\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Step 1: Run wmi810 without any options.\n");
    fprintf(stderr, "If this shows that bit 2 of register MEM_MODE is set, someone (e.g. the\n");
    fprintf(stderr, "BIOS) has already set this bit and normally nothing more is to do.\n");
    fprintf(stderr, "If the table shows that bit 2 of MEM_MODE is cleared, continue with step 2.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Step 2: Run wmi810 -2s  This call should remove flicker in almost all cases.\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "Note that this utility is only a workaround. Long term the better way would\n");
    fprintf(stderr, "be to modify the XFREE i810 driver sources, but this should be done by the\n");
    fprintf(stderr, "maintainer of the i810 driver, not by EKF (hopefully the next XFREE version\n");
    fprintf(stderr, "of the i810 driver will have the necessary initialization included).\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "For questions and comments please mail to cr@ekf.de (Christoph Rabe)\n");
    fprintf(stderr, "\n");
    fprintf(stderr, "2001-06-01, EKF Elektronik GmbH, 59065 Hamm, Germany, Internet: www.ekf.de\n");
}


//// End of File