
/*
This program attempts to disassemble the Cintcode generated by RCP
BCPL compiled code for the BBC Microcomputer. It is designed to work
with code in either BBC BCPL ROM or object modules.

Written by Martin Richards (c) 23 Feb 2017

History

14/02/2017
Began to disassemble 6402 machine code triggered by #m hints in
hints files. Flag M_f marks the op code bytes of Cintcode instructions
and M_m mark the op code bytes of 6502 instructions.
*/

GET "libhdr"

GLOBAL {
  stdin:ug
  stdout

  fromfilename
  fromstream
  tofilename
  tostream
  hintfilename
  hintstream
  flagsoption

  spacev
  spacet
  spacep

  ch
  sectbase    // Position of the first byte of the current section.
  sectend     // Position of the first byte after the current section.

  gvec        // Position in mem of the global vector (size 768 16-bit words)
  stackv      // Position in mem of the root stack (16-bit words)

  sectv       // Vector of section positions, first discovered section starts
              // at sectv!2 and ends at sectv!3
  sectp       // Latest entry in sectv
  sectt       // Size of sectv

  globnamev   // For the mapping between global numbers and names

  codevsize   // The number of bytes loaded into codev.

  codev       // 64K byte vector holding machine code and Cintcode
  flagsv      // vector of 32-bit words (one element per byte in codev)
              //     holding 32 one bit flags.
  argv        // vector holding the main argument assoc with codev
  globav      // vector holding non entry globals
  globbv      // vector holding entry global numbers
  strv        // vector holding string locations assoc with codev
  textv       // vector holding comment text strings
  casesv      // vector holding lists of case constants

  globiv      // 768 32-bit words holding flags about globals


  aval        // Indicates the value the Cintcode A register just loaded
              // by LLL or LLL$, zero otherwise.
              // This is useful since the compiled code sometimes contains
              // sequences like LLL 123;GOTO   or  LLL 234; K32 
  pcstack     // A stack of (pc,aval) pairs representing places in the
              // code that still need to be scanned
  pcstackp    // Subscript of the top item on the pc stack
              // The stack is empty when pcstackp=0
  pcstackt    // Subscript of the end of the pc stack

  pcpush      // pcpush(pc, aval, mark)   // mark = 0 or M_f or M_m etc
  pcpop       // pc, aval popped from the pc stack

  baseaddr    // Set to 32768 by the -r option

  hintsfilename

  bytecount

  tostream
  tracing
  debug

  findsections
  findstrings
  scan
  dumpcode

  indaddr
  indaddrmark
  reladdr

  markprog
  markprogarg
  markproggloba
  markprogglobb
  markprogstr

  prmarks
  chk
  gb
  gsb
  gw
  gsw

  fcounter
  mcounter
  notfcounter
  notmcounter

  prevpc
  prprevflags
  prflags
}

MANIFEST {
  // relocatable object blocks
  T_hunk  = 1000
  T_mc    = 1001
  T_reloc = 1002
  T_end   =  992

  Sectword  = #xFDDF      // Marker for a Section name
  Needsword = #xFEED      // Marker for a NEEDS string
  Entryword = #xDFDF      // Function 16-bit entrypoint marker

  Flagsvupb  = #x10000    // (=64K)
  Codevupb   = Flagsvupb/bytesperword // upb of codev in words

  GvecUpb    =     767    // The upb of the BBC BCPL global vector.

  PcstackUpb = 2*50000    // Allow 50000 items

  // Marks for flagsv
  Type_gentry      =  1   // Global entry point from global initialisation data
  Type_ngentry     =  2   // Non global entry points specified by #c statements
  Type_sectname    =  3   // Location of a section name marked by FDDF
  Type_needsname   =  4   // Location of NEEDS name marked by FEED
  Type_entryname   =  5   // Location of a function name marked by DFDF
  Type_hunkword    =  6   // Location of a BCPL Hunk marked by 03E0
  Type_mhunkword   =  7   // Location of a CODE Hunk marked by 03E1
  Type_relocword   =  8
  Type_relbdy      =  9
  Type_endword     = 10
  Type_gdataend    = 11
  Type_gdataentry  = 12
  Type_gdatagn     = 13
  Type_gdatamaxg   = 14
  Type_string      = 15
  Type_dataword    = 16
  Type_rw          = 17  // Indirect reference resolving word
  Type_swlen       = 18
  Type_swdefault   = 19
  Type_swlab       = 20
  Type_swconst     = 21
  Type_swmink      = 22
  Type_filler      = 23


  M_f        = #x00000020  // Certainly a Cintcode function byte
  M_m        = #x00000040  // Location of a 6502 function byte
  M_notf     = #x00000080  // Cannot be a Cintcode function byte
  M_notm     = #x00000100  // Cannot be a 6502 function byte
  M_scanned  = #x00000200  // This instruction has been or is being scanned
  M_lll      = #x00000400  // Address referenced by LLL or LLL$
  M_ll       = #x00000800  // Address referenced by LL or LL$
  M_sl       = #x00001000  // Address referenced by SL or SL$
  M_l        = #x00002000  // L LW LM LM1 Ln   arg is the constant loaded
  M_gn       = #x00004000  // A Cintcode instruction referring to a global

  M_lstr     = #x00010000  // Address of LLL or LLL$ loading a string constant
  M_lab      = #x00020000  // Destination of jumps such as J JEQ JNE JLS JLE JGR JGE
                           // or 6502 branch or jump instructions

  M_swl      = #x00040000  // A valid SWL instruction
  M_swb      = #x00080000  // A valid SWB instruction
  M_default  = #x00100000  // Position of a DEFAULT label in the code
  M_cint     = #x00200000  // All bytes in a BCPL section
  M_code     = #x00400000  // All bytes in a CODE section
  M_string   = #x00800000  // Possible packed characters of a string. The first
                           // word is referenced by LLL and it has a length
                           // between 1 and 255, and all the characters are
                           // in the range 32 to 127 plus *n, *s, *t *p and *e,
                           // padded with a space if the length is even.
                           // A string of length 0 is not recognised.
  M_char     = #x01000000  // ASCII data and possible length values
}

// BBC Cintcode function codes.
MANIFEST {
F_k0   =   0
F_brk  =   2
F_code1=  13
F_lm   =  14
F_lm1  =  15
F_l0   =  16
F_fhop =  27
F_jeq  =  28
F_jeq0 =  30

F_k    =  32
F_kw   =  33
F_s0g  =  34
F_k0g  =  32

F_l0g  =  45
F_l1g  =  46
F_l2g  =  47
F_lg   =  48
F_sg   =  49
F_llg  =  50
F_ag   =  51
F_mul  =  52
F_div  =  53
F_rem  =  54
F_xor  =  55
F_sl   =  56
F_ll   =  58
F_jne  =  60
F_jne0 =  62

F_llp  =  64
F_llpw =  65
F_s0g1 =  66

F_k0g1  =  32+32
F_l0g1  =  45+32
F_l1g1  =  46+32
F_l2g1  =  47+32
F_lg1   =  48+32
F_sg1   =  49+32
F_llg1  =  50+32
F_ag1   =  51+32

F_add  =  84
F_sub  =  85
F_lsh  =  86
F_rsh  =  87
F_and  =  88
F_or   =  89
F_lll  =  90
F_jls  =  92
F_jls0 =  94

F_l    =  96
F_lw   =  97
F_s0g2 =  98

F_k0g2  =  32+64
F_l0g2  =  45+64
F_l1g2  =  46+64
F_l2g2  =  47+64
F_lg2   =  48+64
F_sg2   =  49+64
F_llg2  =  50+64
F_ag2   =  51+64

F_rv   = 116
F_rtn  = 123
F_jgr  = 124
F_jgr0 = 126

F_lp   = 128
F_lpw  = 129

F_lp0  = 128

F_sys  = 145   // Added by MR
F_swb  = 146
F_swl  = 147
F_st   = 148
F_st0  = 148
F_stp0 = 149
F_goto = 155
F_jle  = 156
F_jle0 = 158

F_sp   = 160
F_spw  = 161

F_sp0  = 160
F_s0   = 176
F_xch  = 181
F_gbyt = 182
F_pbyt = 183
F_atc  = 184
F_atb  = 185
F_j    = 186
F_jge  = 188
F_jge0 = 190

F_ap   = 192
F_apw  = 193

F_ap0  = 192

F_code2= 207
F_nop  = 208
F_a0   = 208
F_rvp0 = 211
F_st0p0= 216
F_st1p0= 218

F_a    = 224
F_aw   = 225

F_l0p0 = 224

F_neg  = 241
F_not  = 242
F_l1p0 = 240
F_l2p0 = 244
F_l3p0 = 247
F_l4p0 = 249
}

LET chk(name, pc) BE
{ LET addr = 14
  LET flags = flagsv!addr
RETURN
//  IF (flags & M_f) ~= 0 & (flags & M_notf) ~= 0 |
//     (flags & M_m) ~= 0 & (flags & M_notm) ~= 0 |
//     (flags & M_f) ~= 0 & (flags & M_m) ~= 0    DO
  { writef("*n// %i5: chk: %x2 in %s pc=%n ", addr, gb(addr), name, pc)
    prmarks(flagsv!addr); newline()
    //IF (flags & M_notf) ~= 0 DO
    //  abort(9999)
  }
}

LET prprevflags(pc) BE IF flagsoption DO
  FOR p = prevpc+1 TO pc-1 DO
  { prflags(p)
    writef("// %i5: %x2*n", p+baseaddr, codev%p)
  }

AND prflags(pc) BE IF flagsoption DO
{
//sawritef("prflags: prevpc=%n pc=%n*n", prevpc, pc)
//abort(1000)
  LET flags = flagsv!pc
  //sawritef("prflags: addr=%n flags=%x8*n", addr, flags)
  wrch((flags & M_f)>0         -> 'F', '-')
  wrch((flags & M_notf)>0      -> 'f', '-')
  wrch((flags & M_m)>0         -> 'M', '-')
  wrch((flags & M_notm)>0      -> 'm', '-')
  wrch((flags & M_scanned)>0   -> 's', '-')
  wrch((flags & M_l)>0         -> 'k', '-')
  wrch((flags & M_gn)>0        -> 'G', '-')
  wrch((flags & (M_ll|M_sl))>0 -> 'W', '-')
  wrch((flags & M_string)>0    -> 'S', '-')
  wrch((flags & M_char)>0      -> 'C', '-')
  wrch((flags & M_lab)>0       -> 'L', '-')
  wrch((flags & M_default)>0   -> 'D', '-')
  wrch(' ')

  SWITCHON flags & #x1F INTO
  { DEFAULT:   writes("        "); ENDCASE
    CASE Type_gentry:      writes("Gentry  "); ENDCASE
    CASE Type_ngentry:     writes("NGentry "); ENDCASE
    CASE Type_sectname:    writes("Section "); ENDCASE
    CASE Type_needsname:   writes("Needs   "); ENDCASE
    CASE Type_entryname:   writes("Fn Name "); ENDCASE
    CASE Type_hunkword:    writes("Hunk    "); ENDCASE
    CASE Type_mhunkword:   writes("MHunk   "); ENDCASE
    CASE Type_relocword:   writes("RelocWrd"); ENDCASE
    CASE Type_relbdy:      writes("Relbody "); ENDCASE
    CASE Type_endword:     writes("End     "); ENDCASE
    CASE Type_gdataend:    writes("Gend    "); ENDCASE
    CASE Type_gdataentry:  writes("GlobL   "); ENDCASE
    CASE Type_gdatagn:     writes("GlobN   "); ENDCASE
    CASE Type_gdatamaxg:   writes("MaxGlog "); ENDCASE
    CASE Type_string:      writes("String  "); ENDCASE
    CASE Type_dataword:    writes("Data    "); ENDCASE
    CASE Type_rw:          writes("ResolveW"); ENDCASE
    CASE Type_swlen:       writes("Sw len  "); ENDCASE
    CASE Type_swdefault:   writes("Sw dlab "); ENDCASE
    CASE Type_swlab:       writes("Sw label"); ENDCASE
    CASE Type_swconst:     writes("Sw const"); ENDCASE
    CASE Type_swmink:      writes("Sw minK "); ENDCASE
    CASE Type_filler:      writes("Filler  "); ENDCASE

  }
  wrch(' ')
  prevpc := pc
}

LET start() = VALOF
{ LET globdata = -1
  LET rdargv = VEC 50

  stdin  := input()
  stdout := output()

  // Vectors allocated by getvec
  pcstack := 0
  flagsv, argv   := 0, 0
  globav, globbv := 0, 0
  casesv, strv, textv := 0, 0, textv

  codev, spacev := 0, 0
  gvec, globiv  := 0, 0

  //Vectors allocated by newvec
  sectv     := 0
  globnamev := 0
  baseaddr  := 0

  fromstream := 0
  tostream   := 0
  hintstream := 0

  UNLESS rdargs("file/a,to/k,hints/k,-r/n,-t/s,-f/s", rdargv, 50) DO
  { writef("Bad arguments for mapcode*n")
    GOTO fin
  }

  // Set default values
  fromfilename := 0
  tofilename := 0
  hintsfilename := 0
  tostream := stdout
  hintstream := 0
  debug := 0

  IF rdargv!0 DO fromfilename  := rdargv!0      // file/a
  IF rdargv!1 DO tofilename    := rdargv!1      // to/k
  IF rdargv!2 DO hintsfilename := rdargv!2      // hints/k
  IF rdargv!3 DO baseaddr := !rdargv!3          // -r/n  sideways rom mem addr
  tracing := rdargv!4                           // -t/s
  //IF rdargv!4 DO baseaddr := 32768              // -r/s  sideways rom code
  flagsoption := rdargv!5                       // -f/s  trace flags

  codev := getvec(Codevupb)      // 64K bytes of memory, Codevupb in words
                                 // Codevupb is in words not bytes

  UNLESS codev DO
  { writef("Insufficient space for codev*n")
    GOTO fin
  }
  FOR i = 0 TO Codevupb DO codev!i := 0

  spacev := getvec(10_000) // Space mainly for global names
  spacet := spacev+10_000
  spacep := spacet

  sectv := newvec(200) // Allow for 100 sections of code.
  sectp := 0
  sectt := 201

  // Read the code file into codev
  codevsize := readcode(fromfilename)
  // codevsize is the number of bytes of code

  UNLESS codevsize DO
  { writef("Cannot open the from file: %s*n", fromfilename)
    GOTO fin
  }

  gvec   := getvec(GvecUpb)
  globiv := getvec(GvecUpb)    // For flags about globals
  flagsv := getvec(codevsize)  // Flags vector -- one word per memory byte
  argv   := getvec(codevsize)  // Main arg vector -- ditto
  globav := getvec(codevsize)  // Vector holding non entry global numbers
  globbv := getvec(codevsize)  // Vector global numbers of function entry points
  casesv := getvec(codevsize)  // Vector of lists of case constants
  strv   := getvec(codevsize)  // Vector strings locations
  textv  := getvec(codevsize)  // Vector text strings

  // Allocate space for locations to be inspected later
  pcstack  := getvec(PcstackUpb)
  pcstackp := 0
  pcstackt := PcstackUpb

  UNLESS gvec & globiv & flagsv & argv &
         globav & globbv &
         casesv & strv & textv &
         pcstack DO
  { writef("Insufficient space*n")
    GOTO fin
  }

  //writef("Initialising the global vector*n")

  // Initialise the global vector
  FOR i = 0 TO GvecUpb DO
  { gvec!i := 0
    globiv!i := 0    // Flags about globals, eg was the target of SG.
  }
  gvec!0 := GvecUpb  // The size of the Cintcode global vector

  FOR pc = 0 TO codevsize DO
  { flagsv!pc, argv!pc           := 0, 0
    globav!pc, globbv!pc         := 0, 0
    casesv!pc, strv!pc, textv!pc := 0, 0, 0
  }

  // Read the hints file, if specified
  IF hintsfilename DO
  { writef("*nCalling readhints(%s)*n", hintsfilename)
    UNLESS readhints(hintsfilename) DO
    { writef("Trouble with the hints file: %s*n", hintsfilename)
      GOTO fin
    }
  }
//abort(9821)
  IF tofilename DO
  { tostream := findoutput(tofilename)
    UNLESS tostream DO
    { writef("Trouble with output file: %s*n", tofilename)
      RESULTIS 20
    }

    writef("Sending output to file: %s*n", tofilename)
    selectoutput(tostream)
  }

  sectbase, sectend := 0, 0

  fcounter    := 0 // Count of Cintcode opcode bytes
  notfcounter := 0 // Count of Cintcode operand of data bytes
  mcounter    := 0 // Count of 6502 opcode bytes
  notmcounter := 0 // Count of 6502 operand or data bytes

  writef("Before calling findsections*n")
  writef("fcounter=%n notfcounter=%n mcounter=%n notmcounter=%n*n",
          fcounter, notfcounter, mcounter, notmcounter)
  writef("codevsize-fcounter-notfcounter=%n*n",
          codevsize-fcounter-notfcounter)
  writef("codevsize-mcounter-notmcounter=%n*n",
          codevsize-mcounter-notmcounter)

  // Search for both cintcode and non cintcode sections.
  writef("*nCalling findsections*n")

  findsections()

  writef("After calling findsections*n")
  writef("fcounter=%n notfcounter=%n mcounter=%n notmcounter=%n*n",
          fcounter, notfcounter, mcounter, notmcounter)
  writef("codevsize-fcounter-notfcounter=%n*n",
          codevsize-fcounter-notfcounter)
  writef("codevsize-mcounter-notmcounter=%n*n",
          codevsize-mcounter-notmcounter)

  //abort(1010)

  writef("*nCalling findnames*n*n")
  findnames()
  writef("After calling findnames*n")
  writef("fcounter=%n notfcounter=%n mcounter=%n notmcounter=%n*n",
          fcounter, notfcounter, mcounter, notmcounter)
  writef("codevsize-fcounter-notfcounter=%n*n",
          codevsize-fcounter-notfcounter)
  writef("codevsize-mcounter-notmcounter=%n*n",
          codevsize-mcounter-notmcounter)

  //IF baseaddr DO pcpush(0, 0, M_m | M_notf)

  // Now scan all instructions reachable via entries in the pc stack
  // obtained from global initialisation data or specified by the hints file

  writef("*nCalling scancode*n*n")
  scancode()

  writef("After calling scancode*n")
  writef("fcounter=%n notfcounter=%n mcounter=%n notmcounter=%n*n",
          fcounter, notfcounter, mcounter, notmcounter)
  writef("codevsize-fcounter-notfcounter=%n*n",
          codevsize-fcounter-notfcounter)
  writef("codevsize-mcounter-notmcounter=%n*n",
          codevsize-mcounter-notmcounter)
//  abort(1020)


//abort(1030)

  writef("*nCalling findnotf*n*n")
  findnotf()
  writef("After calling findnotf*n")
  writef("fcounter=%n notfcounter=%n mcounter=%n notmcounter=%n*n",
          fcounter, notfcounter, mcounter, notmcounter)
  writef("codevsize-fcounter-notfcounter=%n*n",
          codevsize-fcounter-notfcounter)
  writef("codevsize-mcounter-notmcounter=%n*n",
          codevsize-mcounter-notmcounter)

//abort(1030)

  writef("*nCalling findstrings*n*n")
  findstrings()
  writef("After calling findstrings*n")
  writef("fcounter=%n notfcounter=%n mcounter=%n notmcounter=%n*n",
          fcounter, notfcounter, mcounter, notmcounter)
  writef("codevsize-fcounter-notfcounter=%n*n",
          codevsize-fcounter-notfcounter)
  writef("codevsize-mcounter-notmcounter=%n*n",
          codevsize-mcounter-notmcounter)

  //abort(1040)

  writef("About to call dumpcode()*n")
  //abort(1050)

  // Output the resulting disassembled code
  dumpcode()

  // Output the unclassified byte in codev -- not right yet
  { LET k = 0
    FOR pc = 0 TO codevsize-1 DO
    { LET flags = flagsv!pc
      IF flags=0 |
         (flags & (M_f | M_notf)) = (M_f | M_notf) |
         (flags & (M_m | M_notm)) = (M_m | M_notm) |
         (flags & (M_f | M_m)) = (M_f | M_m) DO k := k+1
    }

    TEST k=0
    THEN writef("*nNo unclassified bytes*n")
    ELSE writef("*n%n unclassified byte%ps*n", k, k)
  }

  FOR pc = 0 TO codevsize-1 DO
  { LET flags = flagsv!pc
    UNLESS flags=0 |
           (flags & (M_f | M_notf)) = (M_f | M_notf) |
           (flags & (M_m | M_notm)) = (M_m | M_notm) |
           (flags & (M_f | M_m)) = (M_f | M_m) LOOP

    writef("// %i5: %x2 ", pc+baseaddr, gb(pc))
    prinstr(pc)
    writef("   ")
    prmarks(flagsv!pc)
    newline()
//abort(6651)
  }

fin:

  UNLESS tostream=stdout DO endstream(tostream)
  selectoutput(stdout)

//writef("codev = %n*n", codev)
  IF codev       DO freevec(codev)
//writef("flagsv = %n*n", flagsv)
  IF flagsv      DO freevec(flagsv)
//writef("argv = %n*n", argv)
  IF argv        DO freevec(argv)
  IF globav      DO freevec(globav)
  IF globbv      DO freevec(globbv)
  IF casesv      DO freevec(casesv)
  IF strv        DO freevec(strv)
  IF textv       DO freevec(textv)
  IF gvec        DO freevec(gvec)
  IF globiv      DO freevec(globiv)
  IF pcstack     DO freevec(pcstack)
  IF spacev      DO freevec(spacev)

  RESULTIS 0
}

AND readcode(filename) = VALOF
{ LET count = 0
  LET oldin = input()
  LET fromstream = findinput(filename)

  UNLESS fromstream DO
  { writef("Unable to open file: %s*n", filename)
    RESULTIS 0
  }

  newline()
  writef("Code file: %s*n", filename)

  selectinput(fromstream)

  { ch := binrdch()
    //writef("%i5: ch = %x2 %i3*n", count, ch, ch)
//abort(1000)
    UNLESS 0<=ch<=255 BREAK
    codev%count := ch
    count := count+1
  } REPEAT

  writef("codev initialised     length=%n*n", count)

  //FOR i = 0 TO count-1 DO
  //{ LET ch = codev%i
  //  IF i MOD 32 = 0 DO writef("*n%i5: ", i)
  //  writef(" %x2", ch)
  //}
  //newline()

  endstream(fromstream)
  selectinput(oldin)

  RESULTIS count
}

AND readhints(filename) BE
{ LET oldin = input()
  LET hintsstream = findinput(filename)
  UNLESS hintsstream DO
  { writef("Unable to open file: %s*n", filename)
    RETURN
  }
  selectinput(hintsstream)

  writef("Hints file: %s*n", filename)

  globnamev := newvec(GvecUpb)
  FOR i = 0 TO GvecUpb DO globnamev!i := 0

  { ch := rdch()

    SWITCHON ch INTO
    { DEFAULT: LOOP

      CASE endstreamch: BREAK

      CASE '/': // Treat the rest of the line as a comment
                ch := rdch() REPEATUNTIL ch='*n' | ch=endstreamch
                LOOP

      CASE 'A':CASE 'B':CASE 'C':CASE 'D':CASE 'E':
      CASE 'F':CASE 'G':CASE 'H':CASE 'I':CASE 'J':
      CASE 'K':CASE 'L':CASE 'M':CASE 'N':CASE 'O':
      CASE 'P':CASE 'Q':CASE 'R':CASE 'S':CASE 'T':
      CASE 'U':CASE 'V':CASE 'W':CASE 'X':CASE 'Y':
      CASE 'Z':
      CASE 'a':CASE 'b':CASE 'c':CASE 'd':CASE 'e':
      CASE 'f':CASE 'g':CASE 'h':CASE 'i':CASE 'j':
      CASE 'k':CASE 'l':CASE 'm':CASE 'n':CASE 'o':
      CASE 'p':CASE 'q':CASE 'r':CASE 's':CASE 't':
      CASE 'u':CASE 'v':CASE 'w':CASE 'x':CASE 'y':
      CASE 'z':

            { LET name = rdname()
              LET num = 0
              IF name=0 BREAK
              num := rdnum()
              UNLESS 0<=num<=GvecUpb DO
              { writef("Global number %n out of range*n", num)
                writef("############## ERROR #####################*n")
sawritef("Global number %n out of range*n", num)
                abort(999)
endstream(tostream)
stop(0,0)
              }
              globnamev!num := name
              //writef("G%i3 = %s*n", num, name)
            }
            LOOP

      CASE '#':
          { LET op = rdch()
            AND pc1, pc2 = ?, ?

            ch := rdch()

            IF op='f' DO     // Location of a named non global Cintcode function
            { // #f addr name
              LET pc = rdnum() - baseaddr
              LET name = rdname()
              UNLESS 0 <= pc < codevsize DO
              { sawritef("Bad #f directive: pc=%n out of range*n")
                abort(999)
endstream(tostream)
stop(0,0)
                RETURN
              }
              // Mark pc as a non global named Cintcode op byte
              markprogglobb(pc, Type_ngentry | M_f | M_notm, name)
              pcpush(pc, 0, 0)
              LOOP
            }

            IF op='c' DO     // Location as a Cintcode op code byte.
            { // #c addr
              pc1 := rdnum() - baseaddr
//writef("#c %n*n", pc1)
              UNLESS 0 < pc1 < codevsize DO
              { sawritef("Bad #c directive*n")
                abort(999)
endstream(tostream)
stop(0,0)
                LOOP
              }
//writef("#c %n*n", pc1)
              // Mark pc1 as a Cintcode op code byte.
              pcpush(pc1, 0, M_f | M_notm)
              LOOP
            }

            IF op='m' DO // Location of a 6502 function byte
            { // #m addr
              pc1 := rdnum() - baseaddr
//writef("#m %n*n", pc1)
              IF 0 < pc1 < codevsize DO
              { // Mark the location as 6502 machine instructions
                pcpush(pc1, 0, M_m | M_notf)
              }
              LOOP
            }

            IF op='a' DO // Bytes of ASCII data
            { // #a addr1 addr2
              LET pc1 = rdnum() - baseaddr
              LET pc2 = rdnum() - baseaddr
              FOR p = pc1 TO pc2 DO markprog(p, M_char | M_notf | M_notm)
              LOOP
            }

            IF op='t' DO     // Location of some comment text
            { // #t text
              LET pc = rdnum() - baseaddr
              LET text = rdtext()
              UNLESS 0 <= pc < codevsize DO
              { writef("Bad #t directive: pc=%n out of range*n", pc)
sawritef("Bad #t directive: pc=%n out of range, baseaddr=%n*n", pc, baseaddr)
                abort(999)
endstream(tostream)
stop(0,0)
                RETURN
              }
              textv!pc := text
//writef("text for location %n: %s*n", pc+baseaddr, text)
//abort(4881)
              LOOP
            }

            IF op='w' DO        // Locations of static variables or table elements
                                // for iethe Cintcode or 6502 code
            { // #w addr1 addr2
              pc1 := rdnum() - baseaddr
              pc2 := rdnum() - baseaddr
//writef("#w %n %n*n", pc1, pc2)
              //IF ((pc1 | pc2) & 1) ~= 0 DO
              //{ writef("Bad #w directive*n")
              //  abort(999)
              //  LOOP
              //}
              IF 0 < pc1 <= pc2 < codevsize DO
              { // Treat these locations as 16-bit data word(s).
                markprog(pc1, M_lab)
                FOR p = pc1 TO pc2 BY 2 DO
                { markprog(p, Type_dataword | M_notf | M_notm)
                  markprog(p+1, M_notf | M_notm)
                }
              }
              LOOP
            }
          }
    }
  } REPEAT

  endstream(hintsstream)
  selectinput(oldin)

  //FOR gn = 0 TO 250 DO
  //{ LET name = globnamev!gn
  //  IF name DO writef("G%i3 %s*n", gn, name)
  //}

  writef("*nSpace used %n out of %n*n", spacet-spacep, spacet-spacev)
//abort(1111)
}

AND rdname() = VALOF
{ LET res = 0
  LET len, upb = 0, 0
  LET str = VEC 50

  FOR i = 0 TO 50 DO str!i := 0

  // Skip until a letter
  
  { IF 'A'<=ch<='Z' | 'a'<=ch<='z' BREAK
    IF ch=endstreamch RESULTIS 0  
    ch := rdch()
  } REPEAT

  { len := len+1
    str%len := ch
    ch := rdch()
  } REPEATWHILE 'A'<=ch<='Z' | 'a'<=ch<='z'|
                '0'<=ch<='9' | ch='.' | ch='_'

  str%0 := len

  upb := len / bytesperword
  res := newvec(upb)
  FOR i = 0 TO upb DO res!i := str!i
  RESULTIS res
}

AND rdtext() = VALOF
{ LET res = 0
  LET len, upb = 0, 0
  LET str = VEC 255

  FOR i = 0 TO 255 DO str!i := 0

  // Skip over spaces
  
  WHILE ch='*s' DO ch := rdch()

  UNTIL ch='*n' | ch=endstreamch | len>=255 DO
  { len := len+1
    str%len := ch
    ch := rdch()
  }

  str%0 := len

  upb := len / bytesperword
  res := newvec(upb)
  FOR i = 0 TO upb DO res!i := str!i
  RESULTIS res
}

AND rdnum() = VALOF
{ LET res = 0

  // Skip until a digit

  UNTIL ch=endstreamch DO
  { IF '0'<=ch<='9'  BREAK
    ch := rdch()
  }

  WHILE '0'<=ch<='9' DO
  { res := 10*res + ch - '0'
    ch := rdch()
  }

  RESULTIS res
}

AND newvec(upb) = VALOF
{ LET res = spacep - upb - 1
  IF res < spacev DO
  { sawritef("newvec: More space needed*n")
    abort(999)
endstream(tostream)
stop(0,0)
  }
  spacep := res
  RESULTIS res
}

AND mk2(a, b) = VALOF
{ LET p = newvec(1)
  p!0, p!1 := a, b
  RESULTIS p
}

AND markprog(pc, mark) = VALOF IF mark TEST 0 <= pc <= codevsize
THEN { // mark is bit pattern of 27 flag bits followed by 5 type bits
       // If the type is non zero the previous type setting must
       // be zero or equal to mark. 
       LET oldflags = flagsv!pc
       LET oldtype  = oldflags & #x1F
       LET newtype  = mark & #x1F
       LET newflags = oldflags | mark
       LET diff     = oldflags XOR newflags

       IF newtype & oldtype & newtype ~= oldtype DO
       { writef("// %i5: markprog: System error type changed from %i2 to %i2*n",
                pc+baseaddr, oldtype, newtype)
         newflags := (newflags & #xFFFFFFE0) | newtype
         //abort(3336)
       }

       IF (diff & M_f)    ~= 0 DO fcounter    := fcounter+1
       IF (diff & M_m)    ~= 0 DO mcounter    := mcounter+1
       IF (diff & M_notf) ~= 0 DO notfcounter := notfcounter+1
       IF (diff & M_notm) ~= 0 DO notmcounter := notmcounter+1

       flagsv!pc := newflags

       IF (newflags & (M_f | M_notf)) = (M_f | M_notf) |
          (newflags & (M_m | M_notm)) = (M_m | M_notm) |
          (newflags & (M_f | M_m))    = (M_f | M_m)    DO
       { writef("// %i5: markprog: Bad flags ", pc+baseaddr)
         prmarks(newflags);
         newline()
         //abort(3337)
       } 

       IF tracing IF pc=0 DO
       { writef("// %i5: markprog: mark=%x8            ", pc+baseaddr, mark)
         prmarks(mark)
         newline()
//abort(3338)
       }
//IF pc<128 DO abort(3411)
       RESULTIS TRUE
     }
ELSE { writef("markprog: pc = %n out of range for markprog, codevsize=%n*n", pc, codevsize)
       writef("############## ERROR #####################*n")
sawritef("markprog: pc = %n out of range for markprog, codevsize=%n*n", pc, codevsize)
       abort(999)
endstream(tostream)
stop(0,0)
       RESULTIS FALSE
     }

AND markprogarg(pc, mark, arg) BE
{ IF markprog(pc, mark) DO argv!pc := arg

  //IF tracing DO
  //{ writef("// %i5: markprogarg: mark=%x8 arg=%i5  ", pc+baseaddr, mark, arg)
  //  prmarks(mark)
  //  newline()
  //}
}

AND markproggloba(pc, mark, arg) BE
{ IF markprog(pc, mark) DO globav!pc := arg

  IF tracing DO
  { writef("// %i5: markproggloba: mark=%x8 G%z3  ", pc+baseaddr, mark, arg)
    prmarks(mark)
    newline()
  }
}

AND markprogglobb(pc, mark, arg) BE
{ IF markprog(pc, mark) DO globbv!pc := arg

  IF tracing DO
  { writef("// %i5: markprogglobb: mark=%x8 G%z3  ", pc+baseaddr, mark, arg)
    prmarks(mark)
    newline()
  }
}

AND markprogstr(pc, mark, arg) BE
{ IF markprog(pc, mark) DO strv!pc := arg

  //IF tracing DO
  //{ writef("// %i5: markprogstr: mark=%x8 arg=%i5  ", pc+baseaddr, mark, arg)
  //  prmarks(mark)
  //  newline()
  //}
}

AND markglob(gn, mark) BE globiv!gn := globiv!gn | mark

AND wrfcode(f) BE
{ LET s = VALOF SWITCHON f&31 INTO
   { DEFAULT:
      CASE  0: RESULTIS "     -     K   LLP     L    LP    SP    AP     A"
      CASE  1: RESULTIS "     -    KW  LLPW    LW   LPW   SPW   APW    AW"
      CASE  2: RESULTIS "   BRK   S0G  S0G1  S0G2     -     -     -     -"
      CASE  3: RESULTIS "    K3   K3G  K3G1  K3G2   LP3   SP3   AP3  L0P3"
      CASE  4: RESULTIS "    K4   K4G  K4G1  K4G2   LP4   SP4   AP4  L0P4"
      CASE  5: RESULTIS "    K5   K5G  K5G1  K5G2   LP5   SP5   AP5  L0P5"
      CASE  6: RESULTIS "    K6   K6G  K6G1  K6G2   LP6   SP6   AP6  L0P6"
      CASE  7: RESULTIS "    K7   K7G  K7G1  K7G2   LP7   SP7   AP7  L0P7"
      CASE  8: RESULTIS "    K8   K8G  K8G1  K8G2   LP8   SP8   AP8  L0P8"
      CASE  9: RESULTIS "    K9   K9G  K9G1  K9G2   LP9   SP9   AP9  L0P9"
      CASE 10: RESULTIS "   K10  K10G K10G1 K10G2  LP10  SP10  AP10 L0P10"
      CASE 11: RESULTIS "   K11  K11G K11G1 K11G2  LP11  SP11  AP11 L0P11"
      CASE 12: RESULTIS "   K12  K12G K12G1 K12G2  LP12  SP12  AP12 L0P12"
      CASE 13: RESULTIS " CODE1   L0G  L0G1  L0G2  LP13  SP13     -     -"
      CASE 14: RESULTIS "    LM   L1G  L1G1  L1G2  LP14  SP14     -     -"
      CASE 15: RESULTIS "   LM1   L2G  L2G1  L2G2  LP15  SP15 CODE2     -"
      CASE 16: RESULTIS "    L0    LG   LG1   LG2  LP16  SP16   NOP     -"
      CASE 17: RESULTIS "    L1    SG   SG1   SG2   SYS    S1    A1   NEG"
      CASE 18: RESULTIS "    L2   LLG  LLG1  LLG2   SWB    S2    A2   NOT"
      CASE 19: RESULTIS "    L3    AG   AG1   AG2   SWL    S3    A3  L1P3"
      CASE 20: RESULTIS "    L4   MUL   ADD    RV    ST    S4    A4  L1P4"
      CASE 21: RESULTIS "    L5   DIV   SUB   RV1   ST1   XCH    A5  L1P5"
      CASE 22: RESULTIS "    L6   REM   LSH   RV2   ST2  GBYT  RVP3  L1P6"
      CASE 23: RESULTIS "    L7   XOR   RSH   RV3   ST3  PBYT  RVP4  L2P3"
      CASE 24: RESULTIS "    L8    SL   AND   RV4  STP3   ATC  RVP5  L2P4"
      CASE 25: RESULTIS "    L9   SL$    OR   RV5  STP4   ATB  RVP6  L2P5"
      CASE 26: RESULTIS "   L10    LL   LLL   RV6  STP5     J  RVP7  L3P3"
      CASE 27: RESULTIS "  FHOP   LL$  LLL$   RTN  GOTO    J$ ST0P3  L3P4"
      CASE 28: RESULTIS "   JEQ   JNE   JLS   JGR   JLE   JGE ST0P4  L4P3"
      CASE 29: RESULTIS "  JEQ$  JNE$  JLS$  JGR$  JLE$  JGE$ ST1P3  L4P4"
      CASE 30: RESULTIS "  JEQ0  JNE0  JLS0  JGR0  JLE0  JGE0 ST1P4     -"
      CASE 31: RESULTIS " JEQ0$ JNE0$ JLS0$ JGR0$ JLE0$ JGE0$     -     -"
   }
   LET n = f>>5 & 7
   FOR i = 6*n+1 TO 6*(n+1) DO wrch(s%i)
}

AND instrtype(f) =  "??0000000000010000000000000RIRI*
                   *121111111111111111110000RIRIRIRI*
                   *12111111111111111111000000RIRIRI*
                   *1211111111111111111100000000RIRI*
                   *12?000000000000000BL00000000RIRI*
                   *12?00000000000000000000000RIRIRI*
                   *12?0000000000??0000000000000000?*
                   *12?0000000000????0000000000000??"%f

AND nextpc(pc) = VALOF SWITCHON instrtype(gb(pc)) INTO
                       { DEFAULT:
                         CASE 255: // The string length byte corresponding
                                   // op code 0, treat a a one byte instruction
                         CASE '0': // One byte instruction
                                   RESULTIS pc+1
                         CASE '1': // One byte operand, eg L 100
                         CASE 'R': // Direct relative address
                         CASE 'I': // Indirect relative address
                                   RESULTIS pc+2
                         CASE '2': // 2 by operand, eg LW
                                   RESULTIS pc+3
                         CASE 'B': // SWB instruction
                                 { LET data = (pc+2 & -2)
                                   LET n    = gw(data)
                                   LET tab  = data + 4
                                   // 0  1  SWB
                                   // 1     filler if SWB in an even location
                                   // 2  2  n     number of cases
                                   // 4  4  dlab
                                   // 6  6  Table of n (k,l) pairs
                                   RESULTIS tab + 4*n
                                 }
                         CASE 'L': // SWL instruction
                                 { LET data = (pc+2 & -2)
                                   LET n    = gw(data)
                                   LET dlab = data+2
                                   LET mink = data+4
                                   LET tab  = data+6
                                   // 0  1  SWL    Op code byte
                                   // 1     filler if SWL in an even location
                                   // 2  2  n      number of cases
                                   // 4  4  dlab
                                   // 6  6  kmin   Cases from kmin to kmin+n-1
                                   // 8  8  tab    Table of n labels
//writef("%i5: SWL n=%n dlab=%n mink=%n tab at %n next=%n*n",
//        pc, n, dlab, mink, tab, tab+2*n)
                                   RESULTIS tab + 2*n
                                 }
                       }

AND prinstr(pc) BE
{ LET a = 0
//   writef("%i5: ", pc+baseaddr)
   wrfcode(gb(pc))
   SWITCHON instrtype(codev%pc) INTO
   { DEFAULT:
     CASE '0': writef("      ");             RETURN
     CASE '1': a := gb(pc+1);                ENDCASE
     CASE '2': a := gsw(pc+1);               ENDCASE
     CASE 'R': a := reladdr(pc+1)+baseaddr;  ENDCASE
     CASE 'I': a := indaddr(pc+1)+baseaddr;  ENDCASE
   }
   writef(" %i5", a)
}

AND prentry(f) BE writef("*n%s:*n", codev+(f>>1)-4)  //??????

AND isfun(f) = VALOF
{ // Return TRUE if f is the byte address of a function entry point
  // that is preceeded by the function name.
  LET a = f-10
  UNLESS (f&1)=0 & gw(a)=Entryword & codev%(a+2)=7 RESULTIS FALSE
  RESULTIS TRUE 
}

// Get an unsigned codev byte
AND gb(pc) = codev%pc

// Get a signed codev byte
AND gsb(pc) = VALOF
{ LET res = gb(pc)
  IF res <= 127 RESULTIS res
  RESULTIS res - 256
}

// Get an unsigned codev 16-bit word.
AND gw(a) = VALOF
{ LET w = codev%(a+1)<<8 | codev%a
IF debug>=2 DO writef("gw(%i5) => %x4 %i5*n", a, w, w)
  RESULTIS w
}

AND putword(a, w) BE
{ codev%a := w & #xFF
  codev%(a+1) := (w>>8) & #xFF
}

// Get a signed codev 16-bit word.
AND gsw(a) = VALOF
{ LET w = codev%(a+1)<<8 | codev%a
  IF w > #x7FFF DO w := w - #x10000
IF debug>=2 DO writef("gsw(%i5) => %x4 %i5*n", a, w, w)
  RESULTIS w
}

AND pcpush(pc, val, mark) BE
{ //sawritef("pcpush: pc=%i5  val=%i5*n", pc+baseaddr, val)
  UNLESS 0 <= pc  < codevsize &
         0 <= val < codevsize DO
  { writef("pcpush: pc=%n or val=%n out of range*n", pc+baseaddr, val)
    writef("############## ERROR #####################*n")
    //abort(997)
    RETURN
  }

  // Check that there is room for two values.
  IF pcstackp>=pcstackt DO
  { sawritef("pcstack overflow*n")
    abort(999)
endstream(tostream)
stop(0,0)
  }
  pcstack!pcstackp, pcstack!(pcstackp+1) := pc, val
  pcstackp := pcstackp+2
  markprog(pc, mark)
}

AND pcpop() = VALOF
{ IF pcstackp<2 DO
  { sawritef("pcstack undeflow*n")
    abort(999)
endstream(tostream)
stop(0,0)
  }
  pcstackp := pcstackp - 2
  aval := pcstack!(pcstackp+1)

  //writef("pcpop => %i5  %i5*n", pcstack!pcstackp+baseaddr, aval)
  RESULTIS pcstack!pcstackp
}
  
AND findsections() BE
{ // Scan codev for sections and deal with their
  // global vector initialisation data, if any.
  // All global entry points are pushed into the pcstack.

  sectbase, sectend := 0, 0 // No section discovered yet.
  sectp := 0

  // sectv!0 and sectv!1 are unused
  // sectv!2 and sectv!3 are positions in codev of the start and end of
  // the first discovered section. The start value is the position of
  // the section word T_hunk, T_mc or T_reloc.
  // sectv!sectp and sectv!(sectp+1) are positions in codev of the start
  // and end of the last discovered section which should be the position
  // of the T_end word
  // Section, needs and function names will be discovered after all the
  // Cintcode and 6502 code has been found.

  FOR pc = 0 TO codevsize-1-4 BY 2 DO // Only look for word aligned words.
  { LET word = gw(pc)

    // We are only looking for word = T_hunk, T_mc, R_reloc or T_end

    //T_hunk  = 1000
    //T_mc    = 1001
    //T_reloc = 1002
    //T_end   =  992
chk("findsections", pc)

    IF word = T_hunk | word = T_mc DO
    { LET sectsize  = gw(pc+2) * 2
      LET sectsize2 = gw(pc+4) * 2 // In BCPL sections the first word of code
                                   // is also the hunk size.
      LET next = sectsize + pc + 4 // Position of the first word following the hunk.
      // The default settings
      LET hunkbit, bodybit, str = Type_hunkword | M_notf | M_notm, M_cint, "BCPL "
      LET sectok = TRUE
      sectbase := pc + 4
      sectend := next

//sawritef("findsections: word=%x4 sectbase=%n sectsize=%n sectsize2=%n next=%n*n",
//          word, sectbase, sectsize, sectsize2, next)
      //IF word=T_mc DO
      //{ writef("mc section found*n")
      //  //abort(999)
      //}

      // Unless sectsize=sectsize2 we have probably not found a BCPL section.
      UNLESS sectsize=sectsize2 DO
        hunkbit, bodybit, str := Type_mhunkword | M_notf | M_notm, M_code, "CODE "
//sawritef("word=%x4 sectbase=%n sectsize=%n sectsize2=%n next=%n nextword=%n*n",
//          word, sectbase, sectsize, sectsize2, next, gw(next))
//abort(1000)
      IF word=T_mc DO str := "MC   "

      IF next < codevsize DO
      { // Check that the word following this hunk is a valid hunk word
        // or T_end
        LET nextword = gw(next)

        IF nextword=T_hunk | nextword=T_end |
           nextword=T_mc   | nextword=T_reloc DO
        { writef("*n%i5: %x4 %s Section     size %i5 bytes*n",
                  pc+baseaddr, gw(pc), str, sectsize)
          markprogarg(pc, hunkbit, next)
          FOR i = 0 TO 3 DO markprog(pc+i, M_notf | M_notm)
          IF sectsize=sectsize2 FOR i = 4 TO 5 DO markprog(pc+i, M_notf | M_notm)
//abort(2281)
          sectend := next

          IF gw(next) = T_end DO markprog(next, Type_endword | M_notf | M_notm)
          IF sectp>=sectt DO
          { sawritef("sectv stack too small -- increase sectt*n")
            abort(999)
endstream(tostream)
stop(0,0)
          }
          // Store the section start and end in sectv*n
          sectp := sectp+2
          sectv!sectp := sectbase
          sectv!(sectp+1) := next
//writef("sectv!%i2: %i5 %i5*n", sectp, pc, next)
//abort(6655)

          // Mark every byte of the section as M_cint or M_code
          //newline()
          //writef("sectbase=%i5   sectend=%i5*n", sectbase, sectend)
          //writef("calling markprog to mark with ") prmarks(bodybit)
          //writef(" from ")
          //writef("sectbase=%i5 to sectend-1 = %i5*n",
          //        sectbase,       sectend-1)

          //abort(8264)
          FOR pc = sectbase TO sectend - 1 DO
          {  markprog(pc, bodybit)
          }
        }
//abort(6656)

        // If this hunk is followed by a Reloc hunk perform the required
        // relocation.

        IF nextword = T_reloc DO
        { // next is the locations of the reloc word T_reloc
          LET relocsize  = gw(next+2) * 2
          LET relocnext = relocsize + next + 4
          LET reloctok = TRUE

          IF relocnext < codevsize DO
          { LET relocnextword = gw(relocnext)

            writef("*n%i5: %x4 Reloc Section     size %i5 bytes*n",
                    next+baseaddr, nextword, relocsize)

            UNLESS relocnextword=T_hunk | relocnextword=T_end |
               relocnextword=T_mc DO
            { writef("// %i5: Reloc section not followed by end or a hunk*n", next)
              abort(9995)
            }

            markprogarg(next, Type_relocword | M_notf | M_notm, relocnext)
            FOR p = next+1 TO next+3 DO markprog(p, M_notf | M_notm)

            // Mark every byte of the section body as Type_relbody
            FOR p = next+4 TO relocnext - 1 DO markprog(p, Type_relbdy | M_notf | M_notm)
            //writef("%i5: Reloc section found, size=%i5*n", next, relocsize)
            //abort(3366)

//writef("Reloc section words*n")
//FOR p = next TO relocnext-2 BY 2 DO writef("// %i5: %x4*n", p, gw(p))

            FOR p = next+4 TO relocnext-2 BY 2 DO
            { LET item     = gw(p)
              LET itemtype = item>>13
              AND itemaddr = item & #x1FFF

              writef("// %i5: Applying relocation type %n address %n*n",
                     p, itemtype, itemaddr)
              IF itemtype=0 DO
              { // Add sectbase to word at location sectbase+itemaddr
                LET a = sectbase+itemaddr
                LET newval = gw(a) + sectbase
                putword(a, newval)
                writef("Updated word at %n with value %n*n", a, newval)
                LOOP
              }
              writef("Reloc item type %n not processed*n", itemtype)
            }
//abort(6661)
          }
        }

        // First check that the global initialisation data looks ok.
        { LET p = next - 2
          LET entry = 0
          LET gn = gw(p)  // The maximum global number of this section.
//sawritef("p=%n entry=%n max gn=%n sectok=%n*n", p, entry, gn, sectok)

          UNLESS 0<=gn<=GvecUpb DO sectok := FALSE // GvecUpb=3*256-1 = 767
          WHILE sectok DO
          { p := p-2
            entry := gw(p)     // Entry address relative to baseaddr
//sawritef("p=%n entry=%n sectok=%n*n", p, entry, sectok)
            UNLESS entry BREAK // End of initialisation data, if entry=0
            p := p-2
            gn := gw(p)        // The global number of this entry
//sawritef("p=%n entry=%n gn=%n sectsize=%n sectok=%n*n", p, entry, gn, sectsize, sectok)
            UNLESS 0 <= entry <= sectsize &
                   0 <= gn <= GvecUpb DO sectok := FALSE
//sawritef("sectok=%n*n", sectok)
          }
        }

        UNLESS sectok DO
        { sawritef("Global initialisation data not ok*n")
//abort(2281)
        }

        IF sectok DO
        { // The initialisation data has been checked and looks ok.
          LET p = next - 2
          LET entry = 0
          LET gn = gw(p)        // Max global referenced
//writef("Global initialisation data is ok*n")

          // The max global referenced 
          markprog(p,   Type_gdatamaxg | M_notf | M_notm)
          markprog(p+1, M_notf | M_notm)

writef("%i5: Max global number %n*n", p+baseaddr, gn)
//writef("%i5: gdata %i5*n", p+baseaddr, gw(p))

          { p := p-2
            entry := gw(p)

//writef("%i5: gdata %i5*n", p+baseaddr, entry)
            IF entry=0 DO
            { markprog(p,   Type_gdataend | M_notf | M_notm) // End mark
              markprog(p+1, M_notf | M_notm)
              BREAK   // End of initialisation data
            }

            entry := sectbase + entry // Global function entry address
            markprogarg(p, Type_gdataentry | M_notf | M_notm, entry)
            markprog(p+1, M_notf | M_notm)

            p := p-2
            gn := gw(p)
//writef("%i5: gdata %i5*n", p+baseaddr, gn)
            markprogglobb(p, Type_gdatagn | M_notf | M_notm, gn)  // The global number
            markprog(p+1, M_notf | M_notm)

            gvec!gn := entry      // Initialise the global variable

            // Mark the entry address as a global entry point.
            // Could the entry point to 6502 code????
            markprogglobb(entry, Type_gentry | M_f | M_notm, gn)
            pcpush(entry, 0, 0)  // Remember to scan it.

            //IF tracing DO
            { writef("%i5: G%z3 = %i5", p+baseaddr, gn, entry+baseaddr)
              IF globnamev!gn DO writef("   %s", globnamev!gn)
              newline()
            }
//abort(3392)
          } REPEAT // Deal with other global initialisations, if any

//abort(1004)
        }
      }
    }


    IF word = T_end DO markprog(pc, Type_endword | M_notf | M_notm)
  }
}

AND findnames() BE
{ // Find section, needs and entry names.
  // This is called after findsections and scancode.
  writef("Finding section, needs and entry names*n")

  // codevsize = the number of bytes in codev
  FOR pc = 0 TO codevsize-2 BY 2 DO // Only look at word aligned words.
  { LET word   = gw(pc)
    LET flags0 = flagsv!pc
    LET flags1 = flagsv!(pc+1)
    LET marks  = M_scanned |
                 M_lll | M_ll | M_sl |
                 M_lstr | M_lab |
                 M_swl | M_swb | M_default | M_string

    // Ignore words that cannot be section, needs or function name markers.
    IF ((flags0 | flags1) & marks) ~= 0 LOOP

    IF word = Sectword DO
    { LET len = gb(pc+2)
      LET ok = len=7      // Section names are all of length 7
//writef("%i5: %x4 Sectword found, len=%n*n", pc+baseaddr, gw(pc), len)
      IF ok FOR p = pc+3 TO pc+9 DO
      { LET ch = gb(p)
        UNLESS 'A'<=ch<='Z' |
               'a'<=ch<='z' |
               '0'<=ch<='9' |
               ch='*s' | ch='.' | ch='_' DO
        { ok := FALSE
          BREAK
        }
      }
      IF ok DO
      { markprog(pc, Type_sectname | M_notf | M_notm)
//writef("// %i5: called markprog(%n, Type_sectname)*n", pc+baseaddr, pc)
        markprog(pc+1, M_notf | M_notm)
        FOR i = 2 TO 9 DO markprog(pc+i, M_notf | M_notm | M_char)
        writef("%i5: SECTION Name: ", pc+baseaddr)
        FOR p = pc+3 TO pc+9 DO wrch(gb(p))
        newline()
//abort(4321)
      }
    }

    IF word = Needsword DO
    { LET len = gb(pc+2)
      LET ok = len=7      // Needs names are all of length 7
      IF ok FOR p = pc+3 TO pc+9 DO
      { LET ch = gb(p)
        UNLESS 'A'<=ch<='Z' |
               'a'<=ch<='z' |
               '0'<=ch<='9' |
               ch='*s' | ch='.' | ch='_' DO
        { ok := FALSE
          BREAK
        }
      }

      IF ok DO
      { markprog(pc, Type_needsname | M_notf | M_notm )
        markprog(pc+1, M_notf | M_notm)
        FOR i = 2 TO 9 DO markprog(pc+i, M_notf | M_notm | M_char)
        writef("%i5: Needs Name: ", pc+baseaddr)
        FOR p = pc+3 TO pc+9 DO wrch(gb(p))
        newline()
        //abort(5998)
      }
    }

    IF word = Entryword DO
    { LET len = gb(pc+2)
      LET ok = len=7      // Entry names are all of length 7
//writef("// %i5: %x4  Entryword len=%n*n", pc+baseaddr, gw(pc), len)
      IF ok FOR p = pc+3 TO pc+9 DO
      { LET ch = gb(p)
        UNLESS 'A'<=ch<='Z' |
               'a'<=ch<='z' |
               '0'<=ch<='9' |
               ch='*s' | ch='.' | ch='_' DO
        { ok := FALSE
          BREAK
        }
      }
//writef("ok=%n*n", ok)
      IF ok DO
      { markprog(pc, Type_entryname | M_notf | M_notm)
        markprog(pc+1, M_notf | M_notm)
        FOR i = 2 TO 9 DO markprog(pc+i, M_notf | M_notm | M_char)
        pcpush(pc+10, 0, M_f | M_notm)
        //writef("%i5: Entry Name: ", pc+baseaddr)
        //FOR p = pc+3 TO pc+9 DO wrch(gb(p))
        //newline()
        //abort(5588)
      }
//abort(5589)
    }
  }
}

AND findstrings() BE
{ // Find all strings of length one or mre loaded by LLL or LLL$.
  // The instructions must have been scanned and every byte of the string
  // must not be marked M_f or M_notf. Every byte of the string must be a
  // sensible ASCII character. This function must be called after scancode.
  LET p = 2  // Subscript of sectv
  sectbase, sectend := -1, -1

  FOR pc = 0 TO codevsize-1 DO
  { LET op = gb(pc)
    LET flags = flagsv!pc
chk("findstrings", pc)
//writef("findstrings: %i5: sectv!%i2 = %i5*n", pc+baseaddr, p, sectv!p)
//abort(1012)
//writef("findstrings: pc=%i5 sectv!%n=%i5*n", pc+baseaddr, p, sectv!p)
//writef("// %i5: %x2 %x2 op=%x2 ", pc+baseaddr, gb(pc), gb(pc+1), op); prinstr(pc); newline()

    // Set sectbase and sectend

    UNLESS sectbase <= pc < sectend DO
    { sectbase, sectend := 0, 0
      FOR i = 2 TO sectp BY 2 DO
      { //writef("%i5: %i2 base=%i5  end=%i5*n",
        //        pc+baseaddr, i, sectv!i+baseaddr, sectv!(i+1)+baseaddr)
        IF sectv!i <= pc < sectv!(i+1) DO
        { sectbase := sectv!i
          sectend  := sectv!(i+1)
          BREAK
        }
      }
      //writef("%i5: scan: new sectbase=%n  sectend=%i5*n",
      //        pc+baseaddr, sectbase+baseaddr, sectend+baseaddr)
      //abort(5385)
    } 


    IF op=F_lll | op=F_lll+1 IF (flags & M_scanned) ~= 0 DO
    { // See if it is loading of string with length > 0
      LET sa = op=F_lll -> reladdr(pc+1), indaddr(pc+1)
      LET len = gb(sa)
      LET ok = TRUE

//writef("// %i5: %x2 %x2 op=%x2 ", pc+baseaddr, gb(pc), gb(pc+1), op); prinstr(pc); newline()
//writef("// %i5: findstrings: len=%i3 sa=%i5 ", pc+baseaddr, len, sa+baseaddr)
//prmarks(flagsv!sa); newline()
//abort(8991)
      IF (flagsv!sa & M_f) ~= 0 DO ok := FALSE
//writef("ok=%n*n", ok)
      //TEST op=F_lll
      //THEN writef("%i5: %x2 %x2 LLL found, len=%n*n", pc+baseaddr, op, gsb(pc+1), len)
      //ELSE writef("%i5: %x2 %x2 LLL$ found, len=%n*n", pc+baseaddr, op, gb(pc+1), len)
      //writef(" sectbase=%n sa=%n sectend=%n*n", sectbase, sa, sectend)

      UNLESS len>0 & (sa & 1) = 0 & sectbase <= sa <= sectend DO ok := FALSE

//writef("findstrings: len=%i3 sectbase=%i5 sa=%i5 sectend=%n ok=%n*n",
//        len, sectbase+baseaddr, sa+baseaddr, sectend+baseaddr, ok)
//writef("// %i5: sa=%i5  ", sa+baseaddr)
//prmarks(flagsv!sa); newline()
      IF ok FOR i = 1 TO len DO
      { LET ch = gb(sa+i)
//writef("findstrings: %i2: %i3", i, ch)
//IF 32<=ch<=127 DO writef(" '%c'  ", ch); prmarks(flagsv!(sa+i))
//newline()
        IF (flagsv!(sa+i) & M_f) ~= 0 DO { ok := FALSE; BREAK }
        UNLESS 32 <= ch < 127 |
               ch='*n' | ch='*c' | ch='*p' | ch=7 |
               ch='*t' | ch='*e' DO { ok := FALSE; BREAK }
      }
//writef("ok=%n*n",ok)
//abort(2931)
      IF ok DO
      { IF tracing DO
        { writef("%i5: %x2 String: *"", sa+baseaddr, gb(sa))
          FOR i = 1 TO len DO wrch(gb(sa+i))
          writef("*" found, loaded by the following instruction*n")
          writef("%i5: ", pc+baseaddr); prinstr(pc); newline()
//abort(7427)
        }
        markprogstr(pc, M_lstr, sa)  // Make the string available to dumpcode
        markprog(sa, Type_string | M_char)
        FOR i = 0 TO len DO markprog(sa+i, M_notf | M_notm | M_char)
      }
//IF 950 <=sa=960 DO
//{ writef("// %i5: String len=%n ok=%n*n", sa, gb(sa), ok)
//writef("// %i5: ", sa); prmarks(flags!sa); newline()
//abort(8881)
//}
    }
  }
}

AND findnotf() BE
{ // Find all positions in codev that cannot be the op code of
  // a Cintcode instruction.
 LET count = notfcounter
chk("findnotf", -1)
 notfscan()
//writef("findnotf: notfcounter=%n*n", notfcounter)
//abort(8472)
 IF count=notfcounter RETURN
} REPEAT

AND notfscan() BE
{ // Mark all bytes in codev as notf that cannot be Cintcode op codes.

//writef("notscan: entered*n")

  FOR pc = codevsize-1 TO 0 BY -1 DO
  { // The bytes are scanned in reverse order since a seemingly valid
    // non jumping instruction must be marked notf if the following 
    // instruction is marked notf.
    LET op     = gb(pc)
    LET arg    = gb(pc+1)
    LET flags  = flagsv!pc
    LET flags1 = flagsv!(pc+1)
    LET flags2 = flagsv!(pc+2)
    LET type   = flags & #x1F
chk("notfscan", pc)
    //IF tracing DO
    //{ writef("notfscan: %i5: ", pc); prinstr(pc);
    //  writef("  "); prmarks(flags); newline()
    //  //abort(4561)
    //}

    IF (flags & M_notf) ~= 0 LOOP
    IF (flags & M_f) ~= 0 LOOP
    IF (flags & M_m) ~= 0 DO { markprog(pc, M_notf); LOOP }

    IF type = Type_dataword DO
    { markprog(pc,   M_notf | M_notm)
      markprog(pc+1, M_notf | M_notm)
      LOOP
    }

    SWITCHON op INTO
    { DEFAULT: // Not a Cintcode op code
               markprog(pc, M_notf)
               LOOP

      // One byte instructions
      CASE F_mul:
      CASE F_div:
      CASE F_rem:
      CASE F_add:
      CASE F_sub:
      CASE F_neg:
      CASE F_fhop:
      CASE F_lsh:
      CASE F_rsh:
      CASE F_not:
      CASE F_and:
      CASE F_or:
      CASE F_xor:

      CASE F_brk:
      CASE F_code1:
      CASE F_code2:
                 
      CASE F_gbyt:
      CASE F_pbyt:
      CASE F_atc:  
      CASE F_atb:
      CASE F_xch:

      CASE F_rv+6:
      CASE F_rv+5:
      CASE F_rv+4:
      CASE F_rv+3:
      CASE F_rv+2:
      CASE F_rv+1:
      CASE F_rv:
      CASE F_st+3:
      CASE F_st+2:
      CASE F_st+1:
      CASE F_st:  

      //CASE F_mdiv:
      CASE F_sp0+16:
      CASE F_sp0+15:
      CASE F_sp0+14:
      CASE F_sp0+13:
      CASE F_sp0+12:
      CASE F_sp0+11:
      CASE F_sp0+10:
      CASE F_sp0+9:
      CASE F_sp0+8:
      CASE F_sp0+7:
      CASE F_sp0+6:
      CASE F_sp0+5:
      CASE F_sp0+4:
      CASE F_sp0+3:

      CASE F_l0+10:
      CASE F_l0+9:
      CASE F_l0+8:
      CASE F_l0+7:
      CASE F_l0+6:
      CASE F_l0+5:
      CASE F_l0+4:
      CASE F_l0+3:
      CASE F_l0+2:
      CASE F_l0+1:
      CASE F_l0:
      CASE F_l0-1:

      CASE F_ap0+12:
      CASE F_ap0+11:
      CASE F_ap0+10:
      CASE F_ap0+9:
      CASE F_ap0+8:
      CASE F_ap0+7:
      CASE F_ap0+6:
      CASE F_ap0+5:
      CASE F_ap0+4:
      CASE F_ap0+3:

      CASE F_a0+5:
      CASE F_a0+4:
      CASE F_a0+3:
      CASE F_a0+2:
      CASE F_a0+1:

      CASE F_nop:

      CASE F_sys:
      CASE F_lp0+16:
      CASE F_lp0+15:
      CASE F_lp0+14:
      CASE F_lp0+13:
      CASE F_lp0+12:
      CASE F_lp0+11:
      CASE F_lp0+10:
      CASE F_lp0+9: 
      CASE F_lp0+8:
      CASE F_lp0+7:
      CASE F_lp0+6:
      CASE F_lp0+5:
      CASE F_lp0+4:
      CASE F_lp0+3:

      CASE F_s0+4:
      CASE F_s0+3:
      CASE F_s0+2:
      CASE F_s0+1:

      CASE F_l0p0+12:
      CASE F_l0p0+11:
      CASE F_l0p0+10:
      CASE F_l0p0+9:
      CASE F_l0p0+8:
      CASE F_l0p0+7:
      CASE F_l0p0+6:
      CASE F_l0p0+5:
      CASE F_l0p0+4:
      CASE F_l0p0+3:

      CASE F_l1p0+6:
      CASE F_l1p0+5:
      CASE F_l1p0+4:
      CASE F_l1p0+3:

      CASE F_l2p0+5:
      CASE F_l2p0+4:
      CASE F_l2p0+3:

      CASE F_l3p0+4:
      CASE F_l3p0+3:

      CASE F_l4p0+4:
      CASE F_l4p0+3:

      CASE F_stp0+5:
      CASE F_stp0+4:
      CASE F_stp0+3:

      CASE F_st0p0+4:
      CASE F_st0p0+3:

      CASE F_st1p0+4:
      CASE F_st1p0+3:
   
      CASE F_rvp0+7:
      CASE F_rvp0+6:
      CASE F_rvp0+5:
      CASE F_rvp0+4:
      CASE F_rvp0+3:

      CASE F_k0+12:
      CASE F_k0+11:
      CASE F_k0+10:
      CASE F_k0+9:
      CASE F_k0+8:
      CASE F_k0+7:
      CASE F_k0+6:
      CASE F_k0+5:
      CASE F_k0+4:
      CASE F_k0+3:

//writef("One byte instruction*n")
                IF pc+1 >= codevsize |
                   (flags1 & M_notf) ~= 0 |         // flags1 hold the flags of
                   (flags1 & #x1F) = Type_gentry DO // the byte at pc+1.
                  markprog(pc, M_notf)
                LOOP

      // Two byte instructions
      CASE F_lp:
      CASE F_llp:
      CASE F_sp:

      CASE F_lg2:
      CASE F_lg1:
      CASE F_lg:
      CASE F_sg2:
      CASE F_sg1:
      CASE F_sg:
      CASE F_llg2:
      CASE F_llg1:
      CASE F_llg:
      CASE F_l:
      CASE F_k0g2+12:
      CASE F_k0g2+11:
      CASE F_k0g2+10:
      CASE F_k0g2+9:
      CASE F_k0g2+8:
      CASE F_k0g2+7:
      CASE F_k0g2+6:
      CASE F_k0g2+5:
      CASE F_k0g2+4:
      CASE F_k0g2+3:

      CASE F_k0g1+12:
      CASE F_k0g1+11:
      CASE F_k0g1+10:
      CASE F_k0g1+9:
      CASE F_k0g1+8:
      CASE F_k0g1+7:
      CASE F_k0g1+6:
      CASE F_k0g1+5:
      CASE F_k0g1+4:
      CASE F_k0g1+3:
 
      CASE F_k0g+12:
      CASE F_k0g+11:
      CASE F_k0g+10:
      CASE F_k0g+9:
      CASE F_k0g+8:
      CASE F_k0g+7:
      CASE F_k0g+6:
      CASE F_k0g+5:
      CASE F_k0g+4:
      CASE F_k0g+3:

      CASE F_ap:
      CASE F_ag2:
      CASE F_ag1:
      CASE F_ag:

      CASE F_a:                  //markprog(pc, M_swb)

      CASE F_l0g2:
      CASE F_l1g2:
      CASE F_l2g2:
      CASE F_l0g1:
      CASE F_l1g1:
      CASE F_l2g1:

      CASE F_l0g:
      CASE F_l1g:
      CASE F_l2g:

      CASE F_s0g2:
      CASE F_s0g1:
      CASE F_s0g:
      CASE F_lm:

      CASE F_ll+1:
      CASE F_sl+1:
      CASE F_ll:
      CASE F_sl:
      CASE F_k:
                IF pc+2 >= codevsize |
                   (flags2 & M_notf) ~= 0 |
                   (flags2 & #x1F) = Type_gentry DO
                  markprog(pc, M_notf)
                LOOP

      // Three byte instructions
      CASE F_lpw:
      CASE F_llpw:
      CASE F_spw:
      CASE F_apw:
      CASE F_aw:
      CASE F_lw:
      CASE F_kw:
                IF (pc+3 >= codevsize | flagsv!(pc+3) & M_notf) ~= 0 DO
                  markprog(pc, M_notf)
                LOOP
  
      CASE F_rtn:
      CASE F_goto:
                LOOP

      CASE F_swb:
                { LET q = pc+2 & -2       // Position of length word
                  LET n = gw(q)           // Number of cases
                  LET lab = q+2 + gw(q+2) // Default destination
                  IF q + 4*n + 4 >= codevsize GOTO notf
                  UNLESS validaddr(lab) &
                         (flagsv!lab & M_notf) = 0 GOTO notf
                  FOR i = 1 TO n DO
                  { LET p = q + 4*i
                    lab := p+2 + gsw(p+2)
                    UNLESS validaddr(lab) &
                           (flagsv!lab & M_notf) = 0 GOTO notf
                  }
                  LOOP  // It was a valid switch
                }

notf:           markprog(pc, M_notf)
                LOOP

      CASE F_swl: 
                { LET q = pc+2 & -2   // Position of length word
                  LET n = gw(q)       // Number of cases
                  LET dlab = ?
                  LET min = ?
                  LET p = ?

                  IF q + 2*n + 4 >= codevsize GOTO notf

                  q := q+2
                  dlab := q + gsw(q)       // Default destination

                  // Test the default label
                  UNLESS validaddr(dlab) &
                         (flagsv!dlab & M_notf) = 0 GOTO notf

                  q := q+2
                  min := gsw(q)            // Minimum case constant
                  q := q+2
                  
                  // Scan the case labels
                  FOR i = 0 TO n-1 DO
                  { LET p = q + 2*i
                    LET lab = p + gsw(p)
                    UNLESS validaddr(lab) &
                           (flagsv!lab & M_notf) = 0 GOTO notf
                  }

                  LOOP  // It was a valid switch
                }

      CASE F_lll+1:
               { LET addr = indaddr(pc+1)
                 UNLESS 0 <= addr < codevsize DO
                 { markprog(pc, M_notf)
                   LOOP
                 }
                 IF pc+2 >= codevsize | (flagsv!(pc+2) & M_notf) ~= 0 DO
                   markprog(pc, M_notf)
                 LOOP
               }

      CASE F_lll:
               { LET addr = reladdr(pc+1)
                 UNLESS 0 <= addr < codevsize DO
                 { markprog(pc, M_notf)
                   LOOP
                 }
                 IF pc+2 >= codevsize | (flagsv!(pc+2) & M_notf) ~= 0 DO
                   markprog(pc, M_notf)
                 LOOP
               }

      CASE F_jeq:   CASE F_jne:
      CASE F_jls:   CASE F_jgr:
      CASE F_jle:   CASE F_jge:
      CASE F_jeq+2: CASE F_jne+2:
      CASE F_jls+2: CASE F_jgr+2:
      CASE F_jle+2: CASE F_jge+2:
               { LET addr = reladdr(pc+1)
                 UNLESS 0 <= addr < codevsize DO
                 { markprog(pc, M_notf)
                   LOOP
                 }
                 IF (flagsv!addr & M_notf) ~= 0 |
                    (pc+2 >= codevsize | flagsv!(pc+2) & M_notf) ~= 0 DO
                   markprog(pc, M_notf)
                 LOOP
               }

      CASE F_jeq+1: CASE F_jne+1:
      CASE F_jls+1: CASE F_jgr+1:
      CASE F_jle+1: CASE F_jge+1:
      CASE F_jeq+3: CASE F_jne+3:
      CASE F_jls+3: CASE F_jgr+3:
      CASE F_jle+3: CASE F_jge+3:
               { LET addr = indaddr(pc+1)
                 UNLESS 0 <= addr < codevsize DO
                 { markprog(pc, M_notf)
                   LOOP
                 }
                 IF (flagsv!addr & M_notf) ~= 0 |
                    (pc+2 >= codevsize | flagsv!(pc+2) & M_notf) ~= 0 DO
                   markprog(pc, M_notf)
                 LOOP
               }

      CASE F_j:
               { LET addr = reladdr(pc+1)
                 UNLESS 0 <= addr < codevsize DO
                   markprog(pc, M_notf)
                 LOOP
               }

      CASE F_j+1:
               { LET addr = indaddr(pc+1)
                 UNLESS 0 <= addr < codevsize DO
                   markprog(pc, M_notf)
                 LOOP
               }
    }
  }
}

AND validaddr(addr) = 0 <= addr < codevsize

// Find Cintcode instructions based on global initialisation
// data in pcstack
AND scancode() BE UNTIL pcstackp=0 DO
{ LET pc = pcpop()
  //writef("%i5: scancode entered*n", pc+baseaddr)
  //writef("%i5: ", pc+baseaddr); prinstr(pc)
  //writef("   "); prmarks(flagsv!pc); newline()
//abort(4444)
//

chk("scancode", pc)
  UNLESS 0 <= pc < codevsize DO
  { writef("pc out of range*n")
//abort(5555)
    LOOP // ##########################
  }

  sectbase, sectend := 0, 0

  { LET flags = flagsv!pc
    IF (flags & M_f) ~= 0 DO { scan(pc);     LOOP }
    IF (flags & M_m) ~= 0 DO { scan6502(pc); LOOP }
    writef("// %i5: Bad instruction type, either M_f or M_m must be set*n", pc+baseaddr)
    abort(7639)
  }    
}

AND scan(pc) BE WHILE 0 <= pc < codevsize DO
{ // pc is the position of the first byte of a Cintcode instruction.
  // Explore all reachable instructions from here.
  // aval is either zero or states that A holds an address set by LLL or LLL$
  // If non zero it is used as the destination address of GOTO and some
  // function calls.

  // This is the start of the main scan loop.
  LET op    = gb(pc)
  LET flags = flagsv!pc
chk("scan 1", pc)

  // Set sectbase and sectend if possible
//sawritef("%i5: scan: op=%i3 %x2  arg=%i3*n", pc, op, op, gb(pc+1)) 
  UNLESS sectbase <= pc < sectend DO
  { sectbase, sectend := 0, 0
    FOR i = 2 TO sectp BY 2 DO
    { //writef("%i5: %i2 base=%i5  end=%i5*n",
      //        pc+baseaddr, i, sectv!i+baseaddr, sectv!(i+1)+caseaddr)
      IF sectv!i <= pc < sectv!(i+1) DO
      { sectbase := sectv!i
        sectend  := sectv!(i+1)
        BREAK
      }
    }

    // If no valid section found sectbase and sectend are both zero

    IF tracing DO
      writef("// %i5: scan: new sectbase=%n  sectend=%i5*n",
              pc+baseaddr, sectbase+baseaddr, sectend+baseaddr)
    //abort(5385)
  } 

//abort(9876)
  IF tracing DO
  { //writef("// %i5: %x2 %x2  ",
    //        pc+baseaddr, gb(pc+1), op)
    //prmarks(flagsv!pc)
    //newline()
    writef("// %i5: %x2 %x2  ",
            pc+baseaddr, gb(pc+1), op)
    IF (flagsv!pc & M_f) ~= 0 DO
    { //writef("// %i5: calling printstr*n", pc+baseaddr)
      prinstr(pc)
    }
    IF (flagsv!pc & M_m) ~= 0 DO
    { //writef("// %i5: calling pr6502intstr*n", pc+baseaddr)
      pr6502instr(pc)
    }
    writef("   "); prmarks(flagsv!pc); newline()
//abort(1819)
  }
//abort(1000)

  IF FALSE DO
  { STATIC { oldflags=0 }
    LET addr = -1 //960

    UNLESS flagsv!addr=oldflags DO
    { writef("*n// %i5: addr=%i5  flagsv!%n changed:", pc+baseaddr, addr, addr)
      writef("*n// %i5: addr=%i5 Old flags: ", pc+baseaddr, addr); prmarks(oldflags)
      oldflags := flagsv!addr
      writef("*n// %i5: addr=%i5 New flags: ", pc+baseaddr, addr); prmarks(oldflags)
      newline()
//abort(3744)
    }
  }

  IF (flags & M_notf) ~= 0 DO
  { writef("// %i5: scan: ERROR: Trying to scan a byte with ", pc+baseaddr)
    prmarks(flags)
    newline()
    //abort(9551)
  }

  IF (flags & M_scanned) ~= 0 RETURN        // Already scanned

chk("scan 2", pc)

  markprog(pc, M_scanned)
  flags := flagsv!pc      // To add the scanned mark

chk("scan 3", pc)
//writef("// %i5: ", pc+baseaddr); prinstr(pc); newline()
//abort(8831)
  SWITCHON op INTO
  { DEFAULT:        writef("%i5: Unexpected Cintcode function code %n*n", pc+baseaddr, op)
                    markprog(pc, M_notf)
                    writef("// %i5: ", pc+baseaddr); prinstr(pc); newline()
                    writef("############## ERROR #####################*n")
sawritef("%i5: Unexpected Cintcode function code %n*n", pc+baseaddr, op)
                    //abort(999)
//endstream(tostream)
//stop(0,0)
                    RETURN

    CASE F_mul:
    CASE F_div:
    CASE F_rem:
    CASE F_add:
    CASE F_sub:
    CASE F_neg:
    CASE F_fhop:
    CASE F_lsh:
    CASE F_rsh:
    CASE F_not:
    CASE F_and:
    CASE F_or:
    CASE F_xor:

    CASE F_brk:
                 
    CASE F_rv+6:
    CASE F_rv+5:
    CASE F_rv+4:
    CASE F_rv+3:
    CASE F_rv+2:
    CASE F_rv+1:
    CASE F_rv:

    CASE F_st+3:
    CASE F_st+2:
    CASE F_st+1:
    CASE F_st:  

    //CASE F_chgco:
    //CASE F_mdiv:

    CASE F_gbyt:
    CASE F_pbyt:
    CASE F_atc:  
    CASE F_atb:
    CASE F_xch:     aval := 0
                    pc := markinstr(pc, 1)
                    LOOP

    CASE F_code1:
    CASE F_code2:   aval := 0
                    // Remember to scan 6502 code following code1 0r code2
                    pcpush(pc+1, 0, M_m | M_notf)
                    RETURN

    CASE F_rtn:     aval := 0
                    IF (pc&1)=0 & gb(pc+1)=F_nop DO markprog(pc+1, M_f) // Filler
                    RETURN

    CASE F_goto:    IF aval & sectbase<=aval<sectend DO
                      pcpush(aval, 0, M_f | M_notm | M_lab)
                    // The target location of a GOTO cannot always be determined.
                    RETURN

    CASE F_swb:   { LET q = pc+2 & -2         // Position of length word
                    LET n = gw(q)             // The number of cases
                    LET dlab = q+2 + gsw(q+2) // Default destination  -correct-
                    LET npc = nextpc(pc)
                    markprog(pc, M_swb)
                    FOR p = pc+1 TO npc-1 DO markprog(p, M_notf | M_notm)
                    IF (pc&1)=0 DO markprog(pc+1, Type_filler)
                    markprog(q, Type_swlen)
                    markprog(q+2, Type_swdefault)
                    // Remember to scan the DEFAULT label
                    pcpush(dlab, 0, M_f | M_notm | M_default | M_lab)
//writef("%i5: SWB n=%n default label at %i5*n", pc+baseaddr, n, dlab)
                    FOR i = 1 TO n DO
                    { LET p = q + 4*i
                      LET k = gsw(p)
                      LET lab = p+2 + gsw(p+2)    // This seems to be correct
//writef("%i5: SWB label at %i5 for CASE %n*n", pc+baseaddr, lab+baseaddr, k)
                      markprog(p, Type_swconst)
                      markprog(p+2, Type_swlab)
                      casesv!lab := mk2(casesv!lab, k)
                      pcpush(lab, 0, M_f | M_notm | M_lab) // Position of CASE label
                    }
//writef("sectbase=%i5  sectend=%i5*n", sectbase, sectend)
//abort(9756)
                    //RETURN
                    pc := npc
                    markprog(pc, M_f | M_notm)
                    LOOP
                  }

    CASE F_swl:   // n dlab mink L1 ... Ln

                  { LET q    = pc+2 & -2   // Position of length word
                    LET n    = gw(q)       // Number of cases
                    LET dlab = q+2 + gsw(q+2)
                    LET mink = gsw(q+4)
                    LET tab  = q+6
                    LET npc  = nextpc(pc)
//sawritef("%i5: SWL n=%n default label=%n mink=%n*n", pc+baseaddr, n, dlab+baseaddr, mink)
                    // Remember to scan the default label
                    pcpush(dlab, 0, M_default | M_f | M_notm | M_lab)

                    // Scan the case labels
                    FOR i = 0 TO n-1 DO
                    { LET p = tab + 2*i
                      LET lab = p + gsw(p)
                      markprog(p, Type_swlab)
//sawritef("%i5: SWL label at %n for CASE %n:*n", pc+baseaddr, lab+baseaddr, mink+i)
                      casesv!lab := mk2(casesv!lab, mink+i)
                      pcpush(lab, 0, M_f | M_notm | M_lab)
                    }
                    markprog(pc, M_swl)
                    FOR q = pc+1 TO npc-1 DO markprog(q, M_notf | M_notm)
                    IF (pc&1)=0 DO markprog(pc+1, Type_filler)
//sawritef("%i5: SWL Calling markprog(%n, %n)*n", pc+baseaddr, q, Type_swlen)
                    markprog(q, Type_swlen)
//sawritef("%i5: SWL Calling markprog(%n, %n)*n", pc+baseaddr, q+2, Type_swdefault)
                    markprog(q+2, Type_swdefault)
//sawritef("%i5: SWL Calling markprog(%n, %n)*n", pc+baseaddr, q+4, Type_swmink)
                    markprog(q+4, Type_swmink)
                    aval := 0
//abort(9981) 
                    pc := npc
                    markprog(pc, M_f | M_notm)
                    LOOP
                  }

    CASE F_sys:     aval := 0
                    pc := markinstr(pc, 1)
                    LOOP

    CASE F_lp0+16:
    CASE F_lp0+15:
    CASE F_lp0+14:
    CASE F_lp0+13:
    CASE F_lp0+12:
    CASE F_lp0+11:
    CASE F_lp0+10:
    CASE F_lp0+9:
    CASE F_lp0+8:
    CASE F_lp0+7:
    CASE F_lp0+6:
    CASE F_lp0+5:
    CASE F_lp0+4:
    CASE F_lp0+3:   aval := 0
                    pc := markinstr(pc, 1)
                    LOOP

    CASE F_lp:      aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_lpw:     aval := 0
                    pc := markinstr(pc, 3)
                    LOOP

    CASE F_llp:     aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_llpw:    aval := 0
                    pc := markinstr(pc, 3)
                    LOOP

    CASE F_sp0+16:
    CASE F_sp0+15:
    CASE F_sp0+14:
    CASE F_sp0+13:
    CASE F_sp0+12:
    CASE F_sp0+11:
    CASE F_sp0+10:
    CASE F_sp0+9:
    CASE F_sp0+8:
    CASE F_sp0+7:
    CASE F_sp0+6:
    CASE F_sp0+5:
    CASE F_sp0+4:
    CASE F_sp0+3:   aval := 0
                    pc := markinstr(pc, 1)
                    LOOP

    CASE F_sp:      aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_spw:     aval := 0
                    pc := markinstr(pc, 3)
                    LOOP

    CASE F_lg2:   { LET gn = gb(pc+1)+256
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_lg1:   { LET gn = gb(pc+1)+256
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_lg:    { LET gn = gb(pc+1)
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_sg2:   { LET gn = gb(pc+1) + 512
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_sg1:   { LET gn = gb(pc+1) + 256
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_sg:    { LET gn = gb(pc+1)
                    markproggloba(pc, M_gn, gn)
                    markglob(gn, M_gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_llg2:  { LET gn = gb(pc+1) + 512
                    markproggloba(pc, M_gn, gn)
                    markglob(gn, M_gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_llg1:  { LET gn = gb(pc+1) + 256
                    markproggloba(pc, M_gn, gn)
                    markglob(gn, M_gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_llg:   { LET gn = gb(pc+1)
                    markproggloba(pc, M_gn, gn)
                    markglob(gn, M_gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_ll+1:    markprog(indaddrmark(pc+1), M_ll)
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_ll:      markprog(reladdr(pc+1), M_ll)
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_sl+1:    markprog(indaddrmark(pc+1), M_sl)
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_sl:      markprog(reladdr(pc+1), M_sl)
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
   
    CASE F_lll+1:   aval := indaddrmark(pc+1)
                    markprog(aval, M_lll)
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_lll:     aval := reladdr(pc+1)
                    markprog(aval, M_lll)
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_l0+10:
    CASE F_l0+9:
    CASE F_l0+8:
    CASE F_l0+7:
    CASE F_l0+6:
    CASE F_l0+5:
    CASE F_l0+4:
    CASE F_l0+3:
    CASE F_l0+2:
    CASE F_l0+1:
    CASE F_l0:
    CASE F_l0-1:    aval := 0
                    pc := markinstr(pc, 1)
                    LOOP

    CASE F_l:       aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_lw:      aval := 0
                    pc := markinstr(pc, 3)
                    LOOP

    CASE F_lm:      aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_k0g2+12:
    CASE F_k0g2+11:
    CASE F_k0g2+10:
    CASE F_k0g2+9:
    CASE F_k0g2+8:
    CASE F_k0g2+7:
    CASE F_k0g2+6:
    CASE F_k0g2+5:
    CASE F_k0g2+4:
    CASE F_k0g2+3:
    applyg2:      { LET gn = gb(pc+1) + 512
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_k0g1+12:
    CASE F_k0g1+11:
    CASE F_k0g1+10:
    CASE F_k0g1+9:
    CASE F_k0g1+8:
    CASE F_k0g1+7:
    CASE F_k0g1+6:
    CASE F_k0g1+5:
    CASE F_k0g1+4:
    CASE F_k0g1+3:
    applyg1:      { LET gn = gb(pc+1) + 256
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }
 
    CASE F_k0g+12:
    CASE F_k0g+11:
    CASE F_k0g+10:
    CASE F_k0g+9:
    CASE F_k0g+8:
    CASE F_k0g+7:
    CASE F_k0g+6:
    CASE F_k0g+5:
    CASE F_k0g+4:
    CASE F_k0g+3:
    applyg:       { LET gn = gb(pc+1)
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s*n", globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_k0+12:
    CASE F_k0+11:
    CASE F_k0+10:
    CASE F_k0+9:
    CASE F_k0+8:
    CASE F_k0+7:
    CASE F_k0+6:
    CASE F_k0+5:
    CASE F_k0+4:
    CASE F_k0+3:
    applyk:         IF aval DO pcpush(aval, 0, M_f | M_notm | M_lab)
                    aval := 0
                    pc := markinstr(pc, 1)
                    LOOP

    CASE F_k:       IF aval DO pcpush(aval, 0, M_f | M_notm | M_lab)
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_kw:      IF aval DO pcpush(aval, 0, M_f | M_notm | M_lab)
                    aval := 0
                    pc := markinstr(pc, 3)
                    LOOP

    CASE F_jeq:   CASE F_jne:
    CASE F_jls:   CASE F_jgr:
    CASE F_jle:   CASE F_jge:
    CASE F_jeq+2: CASE F_jne+2:
    CASE F_jls+2: CASE F_jgr+2:
    CASE F_jle+2: CASE F_jge+2:
                  { LET lab = reladdr(pc+1)
                    // Mark the destination addr as labelled
                    pcpush(lab, 0, M_f | M_notm | M_lab)
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_jeq+1: CASE F_jne+1:
    CASE F_jls+1: CASE F_jgr+1:
    CASE F_jle+1: CASE F_jge+1:
    CASE F_jeq+3: CASE F_jne+3:
    CASE F_jls+3: CASE F_jgr+3:
    CASE F_jle+3: CASE F_jge+3:
                  { LET lab = indaddrmark(pc+1)
                    // Mark the destination addr as labelled
                    pcpush(lab, 0, M_f | M_notm | M_lab)
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_j:     { LET lab = reladdr(pc+1)
                    // Mark the destination addr as labelled
                    pcpush(lab, aval, M_f | M_notm | M_lab)
//writef("// %i5: J   jumping to %n*n", pc+baseaddr, lab+baseaddr) 
                    aval := 0
                    markprog(pc+1, M_notf | M_notm)
                    IF (pc&1)=1 & gb(pc+2)=F_nop DO
                      markprog(pc+2, M_f | M_notm)
                    RETURN
                  }

    CASE F_j+1:   chk("J$ 1", pc)
                  { LET lab = indaddrmark(pc+1)
chk("J$ 2", pc)
//writef("// %i5: %x2(=%n) %x2(=%n)*n", pc+baseaddr, gb(pc),gb(pc), gb(pc+1),gb(pc+1))
                    // Mark the destination addr as labelled
                    pcpush(lab, aval, M_f | M_notm | M_lab)
chk("J$ 3", pc)
//writef("// %i5: J$  jumping to %n*n", pc+baseaddr, lab+baseaddr) 
                    markprog(pc+1, M_notf | M_notm)
chk("J$ 4", pc)
                    aval := 0
                    IF (pc&1)=1 & gb(pc+2)=F_nop DO
                      markprog(pc+2, M_f | M_notm)
chk("J$ 5", pc)
//writef("// %i5: Location just after J$*n", pc+2)
//writef("// %i5: ", pc+2)
//prinstr(pc+2); writef("   "); prmarks(flagsv!(pc+2)); newline()
//abort(3334)
                    RETURN
                  }

    CASE F_ap0+12:
    CASE F_ap0+11:
    CASE F_ap0+10:
    CASE F_ap0+9:
    CASE F_ap0+8:
    CASE F_ap0+7:
    CASE F_ap0+6:
    CASE F_ap0+5:
    CASE F_ap0+4:
    CASE F_ap0+3:   aval := 0
                    pc := markinstr(pc, 1)
                    LOOP

    CASE F_ap:      aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_apw:     aval := 0
                    pc := markinstr(pc, 3)
                    LOOP

    CASE F_ag2:   { LET gn = gb(pc+1) + 512
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s", gn, globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_ag1:   { LET gn = gb(pc+1) + 256
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s", gn, globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_ag:    { LET gn = gb(pc+1)
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s", gn, globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_a0+5:
    CASE F_a0+4:
    CASE F_a0+3:
    CASE F_a0+2:
    CASE F_a0+1:

    CASE F_nop:     aval := 0
                    pc := markinstr(pc, 1)
                    LOOP

    CASE F_a:       aval := 0
                    pc := markinstr(pc, 2)
                    LOOP

    CASE F_aw:      aval := 0
                    pc := markinstr(pc, 3)
                    LOOP

    CASE F_s0+4:
    CASE F_s0+3:
    CASE F_s0+2:
    CASE F_s0+1:

    CASE F_l0p0+12:
    CASE F_l0p0+11:
    CASE F_l0p0+10:
    CASE F_l0p0+9:
    CASE F_l0p0+8:
    CASE F_l0p0+7:
    CASE F_l0p0+6:
    CASE F_l0p0+5:
    CASE F_l0p0+4:
    CASE F_l0p0+3:

    CASE F_l1p0+6:
    CASE F_l1p0+5:
    CASE F_l1p0+4:
    CASE F_l1p0+3:

    CASE F_l2p0+5:
    CASE F_l2p0+4:
    CASE F_l2p0+3:

    CASE F_l3p0+4:
    CASE F_l3p0+3:

    CASE F_l4p0+4:
    CASE F_l4p0+3:  aval := 0
                    pc := markinstr(pc, 1)
                    LOOP

    CASE F_l0g2:
    CASE F_l1g2:
    CASE F_l2g2:  { LET gn = gb(pc+1) + 512
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s", gn, globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }
                
    CASE F_l0g1:
    CASE F_l1g1:
    CASE F_l2g1:  { LET gn = gb(pc+1) + 256
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s", gn, globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_l0g:
    CASE F_l1g:
    CASE F_l2g:   { LET gn = gb(pc+1)
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s", gn, globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_s0g2:  { LET gn = gb(pc+1) + 512
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s", gn, globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_s0g1:  { LET gn = gb(pc+1) + 256
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s", gn, globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_s0g:   { LET gn = gb(pc+1)
                    markproggloba(pc, M_gn, gn)
                    //writef("// G%i3", gn)
                    //IF globnamev!gn DO
                    //  writef(" %s", gn, globnamev!gn)
                    //newline()
                    aval := 0
                    pc := markinstr(pc, 2)
                    LOOP
                  }

    CASE F_stp0+5:
    CASE F_stp0+4:
    CASE F_stp0+3:

    CASE F_st0p0+4:
    CASE F_st0p0+3:

    CASE F_st1p0+4:
    CASE F_st1p0+3:
   
    CASE F_rvp0+7:
    CASE F_rvp0+6:
    CASE F_rvp0+5:
    CASE F_rvp0+4:
    CASE F_rvp0+3:  aval := 0
                    pc := markinstr(pc, 1)
                    LOOP
  }
}

AND markinstr(pc, size) = VALOF
{ // Mark the arg bytes of the instruction at position pc
  // Mark the instruction at position pc+size as M_f
  // and return pc+size.
  LET npc = pc+size // Position of the next instruction
  FOR p = pc+1 TO npc-1 DO markprog(p, M_notf | M_notm)
  markprog(npc, M_f | M_notm)
  RESULTIS npc
}

AND scan6502(pc) BE
{ // Scan 6502 instructions starting at location pc
  // Start of main loop
  LET op = gb(pc)

//writef("// %i5: ", pc+baseaddr); pr6502instr(pc)
//writef("   "); prmarks(flagsv!pc); newline()
//abort(1117)

  IF (flagsv!pc & M_scanned) ~= 0 RETURN

  markprog(pc, M_scanned)

  SWITCHON op INTO
  { DEFAULT:   writef("// %i5: %x2   Unknown 6502 Instruction*n", pc+baseaddr, op)
               RETURN

    CASE #x00: //   BRK  imp
               RETURN

    CASE #x20: //   JSR  abs
             { LET dest = gw(pc+1)-baseaddr
               IF 0<dest<codevsize DO
                 pcpush(dest, 0, M_m | M_notf | M_lab)
               markprog(pc+1, M_notf | M_notm)
               markprog(pc+2, M_notf | M_notm)
               pc := pc+3;
               markprog(pc, M_m | M_notf)
               LOOP
             }

    CASE #x40: //   RTI  imp
    CASE #x60: //   RTS  imp
               RETURN

    CASE #x08: //   PHP  imp
    CASE #x28: //   PLP  imp
    CASE #x48: //   PHA  imp
    CASE #x68: //   PLA  imp
    CASE #x88: //   DEY  imp
    CASE #xA8: //   TAY  imp
    CASE #xC8: //   INY  imp
    CASE #xE8: //   INX  imp
    CASE #x18: //   CLC  imp
    CASE #x38: //   SEC  imp
    CASE #x58: //   CLI  imp
    CASE #x78: //   SEI  imp
    CASE #x98: //   TYA  imp
    CASE #xB8: //   CLV  imp
    CASE #xD8: //   CLD  imp
    CASE #xF8: //   SED  imp
    CASE #x8A: //   TXA  imp
    CASE #xAA: //   TAX  imp
    CASE #xCA: //   DEX  imp
    CASE #xEA: //   NOP  imp
    CASE #x12: // x STP  imp
    CASE #x32: // x STP  imp
    CASE #x52: // x STP  imp
    CASE #x72: // x STP  imp
    CASE #x92: // x STP  imp
    CASE #xB2: // x STP  imp
    CASE #xD2: // x STP  imp
    CASE #xF2: // x STP  imp
    CASE #x1A: // x NOP  imp
    CASE #x3A: // x NOP  imp
    CASE #x5A: // x NOP  imp
    CASE #x7A: // x NOP  imp
    CASE #x9A: //   TXS  imp
    CASE #xBA: //   TSX  imp
    CASE #xDA: // x NOP  imp
    CASE #xFA: // x NOP  imp
               pc := pc+1
               markprog(pc, M_m | M_notf)
               LOOP

    CASE #x80: // x NOP  imm
    CASE #xA0: //   LDY  imm
    CASE #xC0: //   CPY  imm
    CASE #xE0: //   CPX  imm
    CASE #x09: //   ORA  imm
    CASE #x29: //   AND  imm
    CASE #x49: //   EOR  imm
    CASE #x69: //   ADC  imm
    CASE #x89: // x NOP  imm
    CASE #xA9: //   LDA  imm
    CASE #xC9: //   CMP  imm
    CASE #xE9: //   SBC  imm
    CASE #x02: // x STP  imm
    CASE #x22: // x STP  imm
    CASE #x42: // x STP  imm
    CASE #x62: // x STP  imm
    CASE #x82: // x NOP  imm
    CASE #xA2: //   LDX  imm
    CASE #xC2: // x NOP  imm
    CASE #xE2: // x NOP  imm
               markprog(pc+1, M_notf | M_notm)
               pc := pc+2
               markprog(pc, M_m | M_notf)
               LOOP

    CASE #x04: // x NOP  zpg
    CASE #x24: //   BIT  zpg
    CASE #x44: // x NOP  zpg
    CASE #x64: // x NOP  zpg
    CASE #x84: //   STY  zpg
    CASE #xA4: //   LDY  zpg
    CASE #xC4: //   CPY  zpg
    CASE #xE4: //   CPX  zpg
    CASE #x05: //   ORA  zpg
    CASE #x25: //   AND  zpg
    CASE #x45: //   EOR  zpg
    CASE #x65: //   ADC  zpg
    CASE #x85: //   STA  zpg
    CASE #xA5: //   LDA  zpg
    CASE #xC5: //   CMP  zpg
    CASE #xE5: //   SBC  zpg
    CASE #x06: //   ASL  zpg
    CASE #x26: //   ROL  zpg
    CASE #x46: //   LSR  zpg
    CASE #x66: //   ROR  zpg
    CASE #x86: //   STX  zpg
    CASE #xA6: //   LDX  zpg
    CASE #xC6: //   DEC  zpg
    CASE #xE6: //   INC  zpg
               markprog(pc+1, M_notf | M_notm)
               pc := pc+2
               markprog(pc, M_m | M_notf)
               LOOP

    CASE #x0C: // x NOP  abs
    CASE #x2C: //   BIT  abs
    CASE #x8C: //   STY  abs
    CASE #xAC: //   LDY  abs
    CASE #xCC: //   CPY  abs
    CASE #xEC: //   CPX  abs
    CASE #x0D: //   ORA  abs
    CASE #x2D: //   AND  abs
    CASE #x4D: //   EOR  abs
    CASE #x6D: //   ADC  abs
    CASE #x8D: //   STA  abs
    CASE #xAD: //   LDA  abs
    CASE #xCD: //   CMP  abs
    CASE #xED: //   SBC  abs
    CASE #x0E: //   ASL  abs
    CASE #x2E: //   ROL  abs
    CASE #x4E: //   LSR  abs
    CASE #x6E: //   ROR  abs
    CASE #x8E: //   STX  abs
    CASE #xAE: //   LDX  abs
    CASE #xCE: //   DEC  abs
    CASE #xEE: //   INC  abs
               markprog(pc+1, M_notf | M_notm)
               markprog(pc+2, M_notf | M_notm)
               pc := pc+3;
               markprog(pc, M_m | M_notf)
               LOOP

    CASE #x4C: //   JMP  abs
    CASE #x6C: //   JMP  abs
             { LET dest = gw(pc+1)-baseaddr
               markprog(pc+1, M_notf | M_notm)
               markprog(pc+2, M_notf | M_notm)
               IF 0 <= dest < codevsize DO
                 pcpush(dest, 0, M_m | M_notf | M_lab)
               RETURN
             }

    CASE #x10: //   BPL  rel
    CASE #x30: //   BMI  rel
    CASE #x50: //   BVC  rel
    CASE #x70: //   BVS  rel
    CASE #x90: //   BCC  rel
    CASE #xB0: //   BCS  rel
    CASE #xD0: //   BNE  rel
    CASE #xF0: //   BEQ  rel
               pcpush(pc+2+gsb(pc+1), 0, M_m | M_notf | M_lab)
               markprog(pc+1, M_notf | M_notm)
               pc := pc+2;
               markprog(pc, M_m | M_notf)
               LOOP
               

    CASE #x01: //   ORA  indx
    CASE #x21: //   AND  indx
    CASE #x41: //   EOR  indx
    CASE #x61: //   ADC  indx
    CASE #x81: //   STA  indx
    CASE #xA1: //   LDA  indx
    CASE #xC1: //   CMP  indx
    CASE #xE1: //   SBC  indx
               markprog(pc+1, M_notf | M_notm)
               markprog(pc+2, M_notf | M_notm)
               pc := pc+3;
               markprog(pc, M_m | M_notf)
               LOOP

    CASE #x11: //   ORA  indy
    CASE #x31: //   AND  indy
    CASE #x51: //   EOR  indy
    CASE #x71: //   ADC  indy
    CASE #x91: //   STA  indy
    CASE #xB1: //   LDA  indy
    CASE #xD1: //   CMP  indy
    CASE #xF1: //   SBC  indy
               markprog(pc+1, M_notf | M_notm)
               pc := pc+2
               markprog(pc, M_m | M_notf)
               LOOP


    CASE #x19: //   ORA  absy
    CASE #x39: //   AND  absy
    CASE #x59: //   EOR  absy
    CASE #x79: //   ADC  absy
    CASE #x99: //   STA  absy
    CASE #xB9: //   LDA  absy
    CASE #xD9: //   CMP  absy
    CASE #xF9: //   SBC  absy
    CASE #xBE: //   LDX  absy
               markprog(pc+1, M_notf | M_notm)
               markprog(pc+2, M_notf | M_notm)
               pc := pc+3;
               markprog(pc, M_m | M_notf)
               LOOP

    CASE #x0A: //   ASL  acc
    CASE #x2A: //   ROL  acc
    CASE #x4A: //   LSR  acc
    CASE #x6A: //   ROR  acc
               pc := pc+1;
               markprog(pc, M_m | M_notf)
               LOOP

    CASE #x14: // x NOP  zpgx
    CASE #x34: // x NOP  zpgx
    CASE #x54: // x NOP  zpgx
    CASE #x74: // x NOP  zpgx
    CASE #x94: //   STY  zpgx
    CASE #xB4: //   LDY  zpgx
    CASE #xD4: // x NOP  zpgx
    CASE #xF4: // x NOP  zpgx
    CASE #x15: //   ORA  zpgx
    CASE #x35: //   AND  zpgx
    CASE #x55: //   EOR  zpgx
    CASE #x75: //   ADC  zpgx
    CASE #x95: //   STA  zpgx
    CASE #xB5: //   LDA  zpgx
    CASE #xD5: //   CMP  zpgx
    CASE #xF5: //   SBC  zpgx
    CASE #x16: //   ASL  zpgx
    CASE #x36: //   ROL  zpgx
    CASE #x56: //   LSR  zpgx
    CASE #x76: //   ROR  zpgx
    CASE #xB6: //   LDX  zpgx
    CASE #xD6: //   DEC  zpgx
    CASE #xF6: //   INC  zpgx
               markprog(pc+1, M_notf | M_notm)
               pc := pc+2
               markprog(pc, M_m | M_notf)
               LOOP


    CASE #x1C: // x NOP  absx
    CASE #x3C: // x NOP  absx
    CASE #x5C: // x NOP  absx
    CASE #x7C: // x NOP  absx
    CASE #x9C: // x SHY  absx
    CASE #xBC: //   LDY  absx
    CASE #xDC: // x NOP  absx
    CASE #xFC: // x NOP  absx
    CASE #x1D: //   ORA  absx
    CASE #x3D: //   AND  absx
    CASE #x5D: //   EOR  absx
    CASE #x7D: //   ADC  absx
    CASE #x9D: //   STA  absx
    CASE #xBD: //   LDA  absx
    CASE #xDD: //   CMP  absx
    CASE #xFD: //   SBC  absx
    CASE #x1E: //   ASL  absx
    CASE #x3E: //   ROL  absx
    CASE #x5E: //   LSR  absx
    CASE #x7E: //   ROR  absx
    CASE #x9E: //   STX  absx
    CASE #xDE: //   DEC  absx
    CASE #xFE: //   INC  absx
               markprog(pc+1, M_notf | M_notm)
               markprog(pc+2, M_notf | M_notm)
               pc := pc+3;
               markprog(pc, M_m | M_notf)
               LOOP
  }
} REPEAT

AND directdest(a) = a + gsb(a)

AND indirectdest(a) = VALOF
{ a := (a + gb(a) * 2 + 1) & #xFFFE // rounded up to even address
  RESULTIS a + gsw(a)
}

AND reladdr(pc) = pc + gb(pc) - 127

AND indaddr(pc) = VALOF
{ // Find the destination address of an indirect reference, but
  // do not mark the resolving word as rw.
  LET ra = (pc+2 & -2) + 2*gb(pc)
  LET t = ra + gsw(ra)
  //writef("*nindaddr: pc:%n gb(pc):%n ra:%n  gsw(ra):%n t=%n*n",
  //        pc+baseaddr,   gb(pc),   ra+baseaddr,    gsw(ra), t+baseaddr)

  //writef("*nsectbase=%n t=%n  sectend=%n*n", sectbase+baseaddr, t+baseaddr, sectend+baseaddr)
  //writef("*nindaddr: pc:%n gb(pc):%n ra:%n  gsw(ra):%n t=%n*n",
  //        pc+baseaddr,   gb(pc),   ra+baseaddr,    gsw(ra), t+baseaddr)
  IF FALSE UNLESS sectbase <= t < sectend DO
  { writef("*n################*n")
    writef("sectbase=%n t=%n  sectend=%n*n", sectbase+baseaddr, t+baseaddr, sectend+baseaddr)
    writef("*nindaddr: pc:%n gb(pc):%n ra:%n  gsw(ra):%n t=%n*n",
            pc+baseaddr,   gb(pc),   ra+baseaddr,    gsw(ra), t+baseaddr)
    //abort(998)
    RESULTIS -2
  }
//abort(8856)
  RESULTIS t
} 

AND indaddrmark(pc) = VALOF
{ // Return -1 or the destination address of an indirect reference.
  // If successful mark the resolving word as M_rw|M_notf|M_notm
  LET ra = (pc+2 & -2) + 2*gb(pc)  // Address of resolving word
  LET t  = ra + gsw(ra)
  //writef("*nindaddr: pc:%n gb(pc):%n ra:%n  gsw(ra):%n t=%n*n",
  //        pc+baseaddr,   gb(pc),   ra+baseaddr,    gsw(ra), t+baseaddr)
  UNLESS sectbase <= t < sectend DO
  { writef("*n################*n")
    writef("*nindaddr: pc:%n gb(pc):%n ra:%n  gsw(ra):%n t=%n*n",
           pc+baseaddr,   gb(pc),   ra+baseaddr,    gsw(ra), t+baseaddr)
    ///abort(998)
    RESULTIS 0
  }
//abort(8856)
  markprogarg(ra, M_notf | M_notm | Type_rw, t)
  markprog(ra+1, M_notf | M_notm, t)
  RESULTIS t
} 

AND relcondjump(f, pc) = reladdr(pc)
AND indcondjump(f, pc) = reladdr(pc)

AND dumpcode() BE
{ // Dump the whole of codev

  LET pc   = 0
  prevpc := pc

  // The memory bytes are held in codev%0 to codev%(codevsize-1)

  sectbase := 0
  sectend  := 0

  // sectbase is either 0 or points to the start of the body
  // of the current section.

  bytecount := 0   // For layout of byte data

  //writef("codevsize = %x4 (%n)*n", codevsize, codevsize)
  //newline()

  WHILE pc < codevsize DO
  { // Start of main loop
    LET op, word, a1 = gb(pc), gw(pc), gb(pc+1)
    LET flags, arg = flagsv!pc, argv!pc
    LET type = flags & #x1F
    LET globa, globb = globav!pc, globbv!pc
    UNLESS (pc&1)=0 DO word := 0  // Word aligned 16-bit word

    // Set sectbase and sectend

    UNLESS sectbase <= pc < sectend DO
    { sectbase, sectend := -1, -1
      FOR i = 2 TO sectp BY 2 DO
      { //writef("%i5: %i2 base=%i5  end=%i5*n", pc, i, sectv!i, sectv!(i+1))
        IF sectv!i <= pc < sectv!(i+1) DO
        { sectbase := sectv!i
          sectend  := sectv!(i+1)
          BREAK
        }
      }
    }

//abort(3376)
    IF tracing DO
    { writef("*n%i5: op=%i3 %i3   %x4  arg=%i5  globa=%i5 casesv!pc=%n",
              pc+baseaddr, op, a1, word, arg, globa, casesv!pc)

      //newline()
      //optnl()
      //writef("%i5: %x2 %x2 ", pc+baseaddr, gb(pc), gb(pc+1))
      prinstr(pc); writef("           "); prmarks(flags); newline()
//abort(1000)
    }

//IF 650<=pc DO
//{ sawritef("type=%n  flags=%x8*n", type, flagsv!pc)
//  abort(pc)
//}

    IF type = Type_sectname DO
    { optnl()
      prflags(pc)
      writef("// %i5: %x4                     Section name: ", pc+baseaddr, word)
      FOR i = 3 TO 9 DO wrch(gb(pc+i))
      newline()
      pc := pc+2
      prprevflags(pc)
      prflags(pc)
      writef("// %i5:", pc+baseaddr)
      FOR i = 0 TO 7 DO writef(" %x2", gb(pc+i))
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+8
      prprevflags(pc)
      LOOP
    }

    IF type = Type_needsname DO
    { optnl()
      prflags(pc)
      writef("// %i5: %x4                     Needs name: ", pc+baseaddr, word)
      FOR i = 3 TO 9 DO wrch(gb(pc+i))
      newline()
      pc := pc+2
      prprevflags(pc)
      prflags(pc)
      writef("// %i5:", pc+baseaddr)
      FOR i = 0 TO 7 DO writef(" %x2", gb(pc+i))
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+8
      prprevflags(pc)
      LOOP
    }

    IF type = Type_entryname DO
    { optnl()
      newline()
      prflags(pc)
      writef("// %i5: %x4                     Entry name: ", pc+baseaddr, word)
      FOR i = 3 TO 9 DO wrch(gb(pc+i))
      newline()
      pc := pc+2
      prprevflags(pc)
      prflags(pc)
      writef("// %i5:", pc+baseaddr)
      FOR i = 0 TO 7 DO writef(" %x2", gb(pc+i))
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+8
      prprevflags(pc)
      LOOP
    }

//writef("*n%i5: casesv!pc=%n*n", pc+baseaddr, casesv!pc)
//abort(9349)
    IF casesv!pc DO
    { LET p = casesv!pc      // List of case labels.
      optnl()
      newline()
      WHILE p DO
      { prflags(pc)
        writef("// %i5: CASE %n:", pc+baseaddr, p!1)
        IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
        newline()
        prprevflags(pc)
        p := !p
      }
    }

    IF (flags & M_default) ~= 0 DO
    { // Position of the default label in the code
      optnl()
      newline()
      prflags(pc)
      writef("// %i5: DEFAULT:", pc+baseaddr)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      prprevflags(pc)
    }

    IF type = Type_gentry DO
    { optnl()
      newline()
      prflags(pc)
      writef("// %i5: GLOBAL ENTRY             G%z3", pc+baseaddr, globb)
      IF globnamev!globb DO writef(" %s", globnamev!globb)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
//abort(6666)
    }

    IF type = Type_ngentry DO
    { optnl()
      newline()
      prflags(pc)
      writef("// %i5: NON GLOBAL ENTRY          %s", pc+baseaddr, globb)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
//abort(6666)
    }

    IF (flags & M_lab) ~= 0 DO
    { optnl()
      newline()
      prflags(pc)
      writef("// %i5: L%z5:", pc+baseaddr, pc+baseaddr)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
    }

    IF type = Type_endword DO
    { optnl()
      prflags(pc)
      writef("// %i5: %x4                  END", pc+baseaddr, gw(pc))
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      newline()
      sectend := arg
      pc := pc+2
      prprevflags(pc)
      LOOP
    }

    IF type = Type_hunkword DO
    { optnl()
      newline()
      prflags(pc)
      writef("// %i5: %x4 %x4                HUNK    size=%n*n",
             pc+baseaddr, gw(pc), gw(pc+2), gw(pc+2))
      pc := pc+4
      prprevflags(pc)
      prflags(pc)
      writef("// %i5: %x4", pc+baseaddr, gw(pc))
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      aval := 0
      pc := pc+2
      prprevflags(pc)
      LOOP
    }

    IF type = Type_mhunkword DO
    { optnl()
      newline()
      prflags(pc)
      writef("// %i5: %x4 %x4                MHUNK    size=%n",
              pc+baseaddr, gw(pc), gw(pc+2), gw(pc+2))
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      aval := 0
      pc := pc+4
      prprevflags(pc)
      LOOP
    }

    IF type = Type_gdataend DO
    { // First word of the global initialisation data
      optnl()
      newline()
      prflags(pc)
      writef("// %i5: %x4      Ginit end mark", pc+baseaddr, gw(pc))
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+2
      prprevflags(pc)
      LOOP
    }

    IF type = Type_gdatagn DO
    { // First word of the global initialisation (gn,entry) pair
      LET gn  = gw(pc)
      LET lab = gsw(pc+2)
      LET entry = sectbase + lab
      optnl()
      prflags(pc)
      writef("// %i5: %x4 %x4 Init  G%z3 = %n",
              pc+baseaddr, gn, lab, gn, entry+baseaddr)
      IF globnamev!gn DO writef("  %s", globnamev!gn)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+4
      prprevflags(pc)
      LOOP
    }

    IF type = Type_gdatamaxg DO
    { // The higher global number used in this section
      LET gn  = gw(pc)
      optnl()
      prflags(pc)
      writef("// %i5: %x4      Highest Global  G%n", pc+baseaddr, gn, gn)
      IF globnamev!gn DO writef(" %s", globnamev!gn)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+2
      prprevflags(pc)
      LOOP
    }

    IF type = Type_filler DO
    { // A filler byte
      optnl()
      prflags(pc)
      writef("// %i5: %x2", pc+baseaddr, gb(pc))
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+1
      prprevflags(pc)
      LOOP
    }

//IF type>=15 DO
// sawritef("// %i5: type=%n Type_swlen=%n*n", pc, type, Type_swlen)
//IF type>=Type_swlen DO
// abort(1000)
    IF type = Type_swlen DO
    { // The length word of SWL or SWB
      LET len = gw(pc)
      optnl()
      prflags(pc)
      writef("// %i5: %x4        len = %i4", pc+baseaddr, gw(pc), len)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+2
      prprevflags(pc)
//sawritef("length word =%n of a switch just done*n", len)
//abort(1000)
      LOOP
    }

    IF type = Type_swdefault DO
    { // The default label of SWL or SWB
      LET dlab = pc + gsw(pc)
      optnl()
      prflags(pc)
      writef("// %i5: %x4    default => %i5", pc+baseaddr, gw(pc), dlab+baseaddr)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+2
      prprevflags(pc)
//sawritef("length word =%n of a switch just done*n", len)
//abort(1000)
      LOOP
    }

    IF type = Type_swlab DO
    { // The default label of SWL or SWB
      LET lab = pc + gsw(pc)
      optnl()
      prflags(pc)
      writef("// %i5: %x4      label => %i5", pc+baseaddr, gw(pc), lab+baseaddr)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+2
      prprevflags(pc)
//sawritef("length word =%n of a switch just done*n", len)
//abort(1000)
      LOOP
    }

    IF type = Type_swmink DO
    { // The minimum case constant of an SWL
      LET mink = gsw(pc)
      optnl()
      prflags(pc)
      writef("// %i5: %x4       mink = %i5", pc+baseaddr, gw(pc), mink)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+2
      prprevflags(pc)
      LOOP
    }

    IF type = Type_swconst DO
    { // A case constant in SWB
      LET const = gsw(pc)
      optnl()
      prflags(pc)
      writef("// %i5: %x4      const = %i5", pc+baseaddr, gw(pc), const)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+2
      prprevflags(pc)
      LOOP
    }

    IF (flags & M_lll) ~= 0 DO
    { optnl()
      newline()
    }

    IF (flags & (M_ll | M_sl)) ~= 0 DO
    { optnl()
      newline()
      markprog(pc,   M_notf | M_notm)
      markprog(pc+1, M_notf | M_notm)
    }

    IF type = Type_rw DO
    { optnl()
      newline()
      {
        prflags(pc)
        writef("// %i5: %x4          Resolving word => %n",
               pc+baseaddr, gsw(pc), argv!pc+baseaddr)
        IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
        newline()
        pc := pc+2
        prprevflags(pc)
      } REPEATWHILE (flagsv!pc & #x1F) = Type_rw
      LOOP
    }

    IF type = Type_dataword DO
    { optnl()
      prflags(pc)
      writef("// %i5: %x4       DATA %i6", pc+baseaddr, gsw(pc), gsw(pc))
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+2
      prprevflags(pc)
      LOOP
    }

    IF type = Type_string DO
    { LET n = gb(pc)
      LET layout = 0
      optnl()
      prflags(pc)
      writef("// %i5: %x2 String: *"", pc+baseaddr, gb(pc))
      FOR i = 1 TO n DO prchar(gb(pc+i))
      writes("*"")
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }

      UNLESS flagsoption FOR p = pc TO pc+n DO
      { IF layout=0 DO
        { //prprevflags(p)
          newline()
          //prflags(p)
          writef("// %i5: ", p+baseaddr)
        }
        writef("%x2 ", gb(p))
        layout := (layout + 1) MOD 16
      }

      newline()
      pc := pc + n + 1
      UNLESS (pc&1)=0 DO pc := pc+1
      prprevflags(pc)
      LOOP
    }

    IF (flags & M_char) ~= 0 DO
    { LET ch = codev%pc
      optnl()
      prflags(pc)
      writef("// %i5: %x2       ASCII ", pc+baseaddr, ch)
      prchar(ch)
      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()
      pc := pc+1
      prprevflags(pc)
      LOOP
    }


    IF (flags & M_m) ~= 0 DO
    { LET p = pc
      optnl()
//writef("%i5: M_m is set  ", pc+baseaddr)
//IF textv!pc DO writef(" %s", textv!pc)
//writef("*n")
      pc := pr6502instr(pc)
      IF textv!p DO { writef("    %s", textv!p); textv!p := 0 }
      newline()
      LOOP
    }

    IF (flags & M_f) ~= 0 DO
    { LET op = gb(pc)
      LET npc = nextpc(pc)
      LET sz = npc - pc
      IF sz<1 DO sz := 1
      IF sz>3 DO sz := 3 
      IF op=F_swl | op=F_swb DO sz := 1

      optnl()

      prflags(pc)
      writef("// %i5: %x2 ", pc+baseaddr, op)
      TEST sz>1 THEN writef("%x2 ", gb(pc+1)) ELSE writes("   ")
      TEST sz>2 THEN writef("%x2 ", gb(pc+2)) ELSE writes("   ")
      prinstr(pc)
      writef("    ")
      IF (flags & M_gn) ~= 0 DO
      { writef("G%z3", globa)
        IF globnamev!globa DO writef(" %s", globnamev!globa)
      }

      IF (flags & M_lstr) ~= 0 DO
      { LET sa = strv!pc
        writef("*"")
        FOR i = 1 TO gb(sa) DO wrfilterch(gb(sa+i))
        writef("*"")
      }

      IF textv!pc DO { writef("    %s", textv!pc); textv!pc := 0 }
      newline()

     IF (flags & (M_swl | M_swb)) ~= 0 DO
     { // The operand data of switches have been marked
       // by scan and so will be output appropriately by dumpcode.
       pc := pc+1
       LOOP
     }

//writef(" f*n")
//deplete(cos)
//abort(1105)
      pc := npc
      prprevflags(pc)
//writef("%i5: npc=%n*n", pc+baseaddr, npc+baseaddr)
      LOOP
    }
//writef(" g*n")

//writef("%i5: Testing for ll lll sl*n", pc+baseaddr)
    IF (flags & (M_ll | M_lll | M_sl)) ~= 0 &
       (flags & (M_notf | M_notm)) ~= 0 DO
    { // Must be static data
      optnl()
      prflags(pc)
      writef("// %i5: %x4       DATA %i6    Operand of", pc+baseaddr, gw(pc), gsw(pc))
      IF (flags & M_ll)  ~= 0 DO writef(" LL")
      IF (flags & M_lll) ~= 0 DO writef(" LLL")
      IF (flags & M_sl)  ~= 0 DO writef(" SL")

      newline()
      pc := pc + 2
      prprevflags(pc)
      LOOP
    }

    IF (pc & 1) = 1 DO
    { optnl()
      prflags(pc)
      writef("// %i5: %x2", pc+baseaddr, gb(pc))
      IF gb(pc)=F_nop DO writef("       Filler", pc, gb(pc))
      newline()
      pc := pc+1
      prprevflags(pc)
      LOOP
    }

    IF (flagsv!pc & (M_sl | F_ll)) ~= 0 DO
    { optnl()
      prflags(pc)
      writef("// %i5: %x4       DATA %i6   accessed by LL or SL",
              pc+baseaddr, gw(pc), gsw(pc))
      writef("     "); prmarks(flagsv!pc); newline()
      pc := pc+2
      prprevflags(pc)
      LOOP
    }

    // Unknown byte
    UNLESS bytecount DO
    {
      prflags(pc)
      writef("// %i5:", pc+baseaddr)
    }
    bytecount := bytecount+1
    writef(" %x2", gb(pc))
    pc := pc+1
    prprevflags(pc)
    IF bytecount >= 8 DO optnl()
    LOOP
  }
}

AND prchar(ch) BE
{ LET esc = VALOF SWITCHON ch INTO
  { DEFAULT:   RESULTIS 0
    CASE '*n': RESULTIS 'n'
    CASE '*c': RESULTIS 'c'
    CASE '*b': RESULTIS 'b'
    CASE '**': RESULTIS '**'
    CASE '*"': RESULTIS '*"'
  }
  IF esc DO
  { writef("**%c", esc)
    RETURN
  }

  IF 32<=ch<127 DO
  { wrch(ch)
    RETURN
  }

  writef("**%z3", ch)
}

AND optnl() BE IF bytecount DO
{ newline()
  bytecount := 0
}

AND prmarks(flags) BE
{ LET type = flags & #x1F
  writef("Marks:  ")
  IF (flags & M_f)         ~=0 DO writef(" f")        // A Cintcode op code
  IF (flags & M_m)         ~=0 DO writef(" m")        // A 6502 op code
  IF (flags & M_notf)      ~=0 DO writef(" notf")     // Not an instruction
  IF (flags & M_notm)      ~=0 DO writef(" notm")     // Not an instruction
  IF (flags & M_scanned)   ~=0 DO writef(" scanned")
  IF (flags & M_lll)       ~=0 DO writef(" lll")
  IF (flags & M_ll)        ~=0 DO writef(" ll")
  IF (flags & M_sl)        ~=0 DO writef(" sl")
  IF (flags & M_gn)        ~=0 DO writef(" gn")
  IF (flags & M_lstr)      ~=0 DO writef(" lstr")
  IF (flags & M_lab)       ~=0 DO writef(" lab")
  IF (flags & M_swl)       ~=0 DO writef(" swl")
  IF (flags & M_swb)       ~=0 DO writef(" swb")
  IF (flags & M_default)   ~=0 DO writef(" default")
  IF (flags & M_cint)      ~=0 DO writef(" cint")
  IF (flags & M_code)      ~=0 DO writef(" code")
  IF (flags & M_string)    ~=0 DO writef(" string")

  SWITCHON type INTO
  { DEFAULT:  writef(" Bad type: %n*n", type)
              RETURN
    CASE 0:                                      RETURN
    CASE Type_gentry:     writef(" gentry");     RETURN
    CASE Type_ngentry:    writef(" ngentry");    RETURN
    CASE Type_sectname:   writef(" sectname");   RETURN
    CASE Type_needsname:  writef(" needsname");  RETURN
    CASE Type_entryname:  writef(" entryname");  RETURN
    CASE Type_hunkword:   writef(" hunkword");   RETURN
    CASE Type_mhunkword:  writef(" mhunkword");  RETURN
    CASE Type_relocword:  writef(" relocword");  RETURN
    CASE Type_relbdy:     writef(" relbdy");     RETURN
    CASE Type_endword:    writef(" endword");    RETURN
    CASE Type_gdataend:   writef(" gdataend");   RETURN
    CASE Type_gdataentry: writef(" gdataentry"); RETURN
    CASE Type_gdatagn:    writef(" gdatagn");    RETURN
    CASE Type_gdatamaxg:  writef(" gdatamaxg");  RETURN
    CASE Type_string:     writef(" string");     RETURN
    CASE Type_dataword:   writef(" dataword");   RETURN
    CASE Type_rw:         writef(" rw");         RETURN
  }

//abort(5555)
}

AND wrfilterch(ch) = VALOF SWITCHON ch INTO
{ DEFAULT:
             UNLESS 32<=ch<127 DO ch := '?'
             wrch(ch)
             RETURN

  CASE '*"': writes("***""); RETURN
  CASE '*n': writes("**n"); RETURN
  CASE '*c': writes("**c"); RETURN
  CASE '*p': writes("**p"); RETURN
  CASE '*t': writes("**t"); RETURN
  CASE '*b': writes("**b"); RETURN
  CASE '*e': writes("**e"); RETURN
}

AND pr6502instr(pc) = VALOF
{ LET op = gb(pc)
//writef("*n// %i5: pr6502instr: ", pc); prmarks(flagsv!pc); newline()
//abort(4441)

  SWITCHON op INTO
  { DEFAULT:
      prflags(pc)
      writef("// %i5: %x2   Unknown 6502 Instruction*n", pc+baseaddr, op)
      RESULTIS pc+1

    CASE #x00: RESULTIS pr6502imp (pc, "  BRK")
    CASE #x20: RESULTIS pr6502abs (pc, "  JSR")
    CASE #x40: RESULTIS pr6502imp (pc, "  RTI")
    CASE #x60: RESULTIS pr6502imp (pc, "  RTS")
    CASE #x80: RESULTIS pr6502imm (pc, "x NOP")
    CASE #xA0: RESULTIS pr6502imm (pc, "  LDY")
    CASE #xC0: RESULTIS pr6502imm (pc, "  CPY")
    CASE #xE0: RESULTIS pr6502imm (pc, "  CPX")

    CASE #x04: RESULTIS pr6502zpg (pc, "x NOP")
    CASE #x24: RESULTIS pr6502zpg (pc, "  BIT")
    CASE #x44: RESULTIS pr6502zpg (pc, "x NOP")
    CASE #x64: RESULTIS pr6502zpg (pc, "x NOP")
    CASE #x84: RESULTIS pr6502zpg (pc, "  STY")
    CASE #xA4: RESULTIS pr6502zpg (pc, "  LDY")
    CASE #xC4: RESULTIS pr6502zpg (pc, "  CPY")
    CASE #xE4: RESULTIS pr6502zpg (pc, "  CPX")

    CASE #x08: RESULTIS pr6502imp (pc, "  PHP")
    CASE #x28: RESULTIS pr6502imp (pc, "  PLP")
    CASE #x48: RESULTIS pr6502imp (pc, "  PHA")
    CASE #x68: RESULTIS pr6502imp (pc, "  PLA")
    CASE #x88: RESULTIS pr6502imp (pc, "  DEY")
    CASE #xA8: RESULTIS pr6502imp (pc, "  TAY")
    CASE #xC8: RESULTIS pr6502imp (pc, "  INY")
    CASE #xE8: RESULTIS pr6502imp (pc, "  INX")

    CASE #x0C: RESULTIS pr6502abs (pc, "x NOP")
    CASE #x2C: RESULTIS pr6502abs (pc, "  BIT")
    CASE #x4C: RESULTIS pr6502abs (pc, "  JMP")
    CASE #x6C: RESULTIS pr6502ind (pc, "  JMP")
    CASE #x8C: RESULTIS pr6502abs (pc, "  STY")
    CASE #xAC: RESULTIS pr6502abs (pc, "  LDY")
    CASE #xCC: RESULTIS pr6502abs (pc, "  CPY")
    CASE #xEC: RESULTIS pr6502abs (pc, "  CPX")

    CASE #x10: RESULTIS pr6502rel (pc, "  BPL")
    CASE #x30: RESULTIS pr6502rel (pc, "  BMI")
    CASE #x50: RESULTIS pr6502rel (pc, "  BVC")
    CASE #x70: RESULTIS pr6502rel (pc, "  BVS")
    CASE #x90: RESULTIS pr6502rel (pc, "  BCC")
    CASE #xB0: RESULTIS pr6502rel (pc, "  BCS")
    CASE #xD0: RESULTIS pr6502rel (pc, "  BNE")
    CASE #xF0: RESULTIS pr6502rel (pc, "  BEQ")

    CASE #x14: RESULTIS pr6502zpgx(pc, "x NOP")

    CASE #x34: RESULTIS pr6502zpgx(pc, "x NOP")
    CASE #x54: RESULTIS pr6502zpgx(pc, "x NOP")
    CASE #x74: RESULTIS pr6502zpgx(pc, "x NOP")
    CASE #x94: RESULTIS pr6502zpgx(pc, "  STY")
    CASE #xB4: RESULTIS pr6502zpgx(pc, "  LDY")
    CASE #xD4: RESULTIS pr6502zpgx(pc, "x NOP")
    CASE #xF4: RESULTIS pr6502zpgx(pc, "x NOP")

    CASE #x18: RESULTIS pr6502imp (pc, "  CLC")
    CASE #x38: RESULTIS pr6502imp (pc, "  SEC")
    CASE #x58: RESULTIS pr6502imp (pc, "  CLI")
    CASE #x78: RESULTIS pr6502imp (pc, "  SEI")
    CASE #x98: RESULTIS pr6502imp (pc, "  TYA")
    CASE #xB8: RESULTIS pr6502imp (pc, "  CLV")
    CASE #xD8: RESULTIS pr6502imp (pc, "  CLD")
    CASE #xF8: RESULTIS pr6502imp (pc, "  SED")

    CASE #x1C: RESULTIS pr6502absx(pc, "x NOP")
    CASE #x3C: RESULTIS pr6502absx(pc, "x NOP")
    CASE #x5C: RESULTIS pr6502absx(pc, "x NOP")
    CASE #x7C: RESULTIS pr6502absx(pc, "x NOP")
    CASE #x9C: RESULTIS pr6502absx(pc, "x SHY")
    CASE #xBC: RESULTIS pr6502absx(pc, "  LDY")
    CASE #xDC: RESULTIS pr6502absx(pc, "x NOP")
    CASE #xFC: RESULTIS pr6502absx(pc, "x NOP")

    CASE #x01: RESULTIS pr6502indx(pc, "  ORA")
    CASE #x21: RESULTIS pr6502indx(pc, "  AND")
    CASE #x41: RESULTIS pr6502indx(pc, "  EOR")
    CASE #x61: RESULTIS pr6502indx(pc, "  ADC")
    CASE #x81: RESULTIS pr6502indx(pc, "  STA")
    CASE #xA1: RESULTIS pr6502indx(pc, "  LDA")
    CASE #xC1: RESULTIS pr6502indx(pc, "  CMP")
    CASE #xE1: RESULTIS pr6502indx(pc, "  SBC")

    CASE #x05: RESULTIS pr6502zpg (pc, "  ORA")
    CASE #x25: RESULTIS pr6502zpg (pc, "  AND")
    CASE #x45: RESULTIS pr6502zpg (pc, "  EOR")
    CASE #x65: RESULTIS pr6502zpg (pc, "  ADC")
    CASE #x85: RESULTIS pr6502zpg (pc, "  STA")
    CASE #xA5: RESULTIS pr6502zpg (pc, "  LDA")
    CASE #xC5: RESULTIS pr6502zpg (pc, "  CMP")
    CASE #xE5: RESULTIS pr6502zpg (pc, "  SBC")

    CASE #x09: RESULTIS pr6502imm (pc, "  ORA")
    CASE #x29: RESULTIS pr6502imm (pc, "  AND")
    CASE #x49: RESULTIS pr6502imm (pc, "  EOR")
    CASE #x69: RESULTIS pr6502imm (pc, "  ADC")
    CASE #x89: RESULTIS pr6502imm (pc, "x NOP")
    CASE #xA9: RESULTIS pr6502imm (pc, "  LDA")
    CASE #xC9: RESULTIS pr6502imm (pc, "  CMP")
    CASE #xE9: RESULTIS pr6502imm (pc, "  SBC")

    CASE #x0D: RESULTIS pr6502abs (pc, "  ORA")
    CASE #x2D: RESULTIS pr6502abs (pc, "  AND")
    CASE #x4D: RESULTIS pr6502abs (pc, "  EOR")
    CASE #x6D: RESULTIS pr6502abs (pc, "  ADC")
    CASE #x8D: RESULTIS pr6502abs (pc, "  STA")
    CASE #xAD: RESULTIS pr6502abs (pc, "  LDA")
    CASE #xCD: RESULTIS pr6502abs (pc, "  CMP")
    CASE #xED: RESULTIS pr6502abs (pc, "  SBC")

    CASE #x11: RESULTIS pr6502indy(pc, "  ORA")
    CASE #x31: RESULTIS pr6502indy(pc, "  AND")
    CASE #x51: RESULTIS pr6502indy(pc, "  EOR")
    CASE #x71: RESULTIS pr6502indy(pc, "  ADC")
    CASE #x91: RESULTIS pr6502indy(pc, "  STA")
    CASE #xB1: RESULTIS pr6502indy(pc, "  LDA")
    CASE #xD1: RESULTIS pr6502indy(pc, "  CMP")
    CASE #xF1: RESULTIS pr6502indy(pc, "  SBC")

    CASE #x15: RESULTIS pr6502zpgx(pc, "  ORA")
    CASE #x35: RESULTIS pr6502zpgx(pc, "  AND")
    CASE #x55: RESULTIS pr6502zpgx(pc, "  EOR")
    CASE #x75: RESULTIS pr6502zpgx(pc, "  ADC")
    CASE #x95: RESULTIS pr6502zpgx(pc, "  STA")
    CASE #xB5: RESULTIS pr6502zpgx(pc, "  LDA")
    CASE #xD5: RESULTIS pr6502zpgx(pc, "  CMP")
    CASE #xF5: RESULTIS pr6502zpgx(pc, "  SBC")

    CASE #x19: RESULTIS pr6502absy(pc, "  ORA")
    CASE #x39: RESULTIS pr6502absy(pc, "  AND")
    CASE #x59: RESULTIS pr6502absy(pc, "  EOR")
    CASE #x79: RESULTIS pr6502absy(pc, "  ADC")
    CASE #x99: RESULTIS pr6502absy(pc, "  STA")
    CASE #xB9: RESULTIS pr6502absy(pc, "  LDA")
    CASE #xD9: RESULTIS pr6502absy(pc, "  CMP")
    CASE #xF9: RESULTIS pr6502absy(pc, "  SBC")

    CASE #x1D: RESULTIS pr6502absx(pc, "  ORA")
    CASE #x3D: RESULTIS pr6502absx(pc, "  AND")
    CASE #x5D: RESULTIS pr6502absx(pc, "  EOR")
    CASE #x7D: RESULTIS pr6502absx(pc, "  ADC")
    CASE #x9D: RESULTIS pr6502absx(pc, "  STA")
    CASE #xBD: RESULTIS pr6502absx(pc, "  LDA")
    CASE #xDD: RESULTIS pr6502absx(pc, "  CMP")
    CASE #xFD: RESULTIS pr6502absx(pc, "  SBC")

    CASE #x02: RESULTIS pr6502imm (pc, "x STP")
    CASE #x22: RESULTIS pr6502imm (pc, "x STP")
    CASE #x42: RESULTIS pr6502imm (pc, "x STP")
    CASE #x62: RESULTIS pr6502imm (pc, "x STP")
    CASE #x82: RESULTIS pr6502imm (pc, "x NOP")
    CASE #xA2: RESULTIS pr6502imm (pc, "  LDX")
    CASE #xC2: RESULTIS pr6502imm (pc, "x NOP")
    CASE #xE2: RESULTIS pr6502imm (pc, "x NOP")

    CASE #x06: RESULTIS pr6502zpg (pc, "  ASL")
    CASE #x26: RESULTIS pr6502zpg (pc, "  ROL")
    CASE #x46: RESULTIS pr6502zpg (pc, "  LSR")
    CASE #x66: RESULTIS pr6502zpg (pc, "  ROR")
    CASE #x86: RESULTIS pr6502zpg (pc, "  STX")
    CASE #xA6: RESULTIS pr6502zpg (pc, "  LDX")
    CASE #xC6: RESULTIS pr6502zpg (pc, "  DEC")
    CASE #xE6: RESULTIS pr6502zpg (pc, "  INC")

    CASE #x0A: RESULTIS pr6502acc (pc, "  ASL")
    CASE #x2A: RESULTIS pr6502acc (pc, "  ROL")
    CASE #x4A: RESULTIS pr6502acc (pc, "  LSR")
    CASE #x6A: RESULTIS pr6502acc (pc, "  ROR")
    CASE #x8A: RESULTIS pr6502imp (pc, "  TXA")
    CASE #xAA: RESULTIS pr6502imp (pc, "  TAX")
    CASE #xCA: RESULTIS pr6502imp (pc, "  DEX")
    CASE #xEA: RESULTIS pr6502imp (pc, "  NOP")

    CASE #x0E: RESULTIS pr6502abs (pc, "  ASL")
    CASE #x2E: RESULTIS pr6502abs (pc, "  ROL")
    CASE #x4E: RESULTIS pr6502abs (pc, "  LSR")
    CASE #x6E: RESULTIS pr6502abs (pc, "  ROR")
    CASE #x8E: RESULTIS pr6502abs (pc, "  STX")
    CASE #xAE: RESULTIS pr6502abs (pc, "  LDX")
    CASE #xCE: RESULTIS pr6502abs (pc, "  DEC")
    CASE #xEE: RESULTIS pr6502abs (pc, "  INC")

    CASE #x12: RESULTIS pr6502imp (pc, "x STP")
    CASE #x32: RESULTIS pr6502imp (pc, "x STP")
    CASE #x52: RESULTIS pr6502imp (pc, "x STP")
    CASE #x72: RESULTIS pr6502imp (pc, "x STP")
    CASE #x92: RESULTIS pr6502imp (pc, "x STP")
    CASE #xB2: RESULTIS pr6502imp (pc, "x STP")
    CASE #xD2: RESULTIS pr6502imp (pc, "x STP")
    CASE #xF2: RESULTIS pr6502imp (pc, "x STP")

    CASE #x16: RESULTIS pr6502zpgx(pc, "  ASL")
    CASE #x36: RESULTIS pr6502zpgx(pc, "  ROL")
    CASE #x56: RESULTIS pr6502zpgx(pc, "  LSR")
    CASE #x76: RESULTIS pr6502zpgx(pc, "  ROR")
    CASE #x96: RESULTIS pr6502zpgy(pc, "  STX")
    CASE #xB6: RESULTIS pr6502zpgy(pc, "  LDX")
    CASE #xD6: RESULTIS pr6502zpgx(pc, "  DEC")
    CASE #xF6: RESULTIS pr6502zpgx(pc, "  INC")

    CASE #x1A: RESULTIS pr6502imp (pc, "x NOP")
    CASE #x3A: RESULTIS pr6502imp (pc, "x NOP")
    CASE #x5A: RESULTIS pr6502imp (pc, "x NOP")
    CASE #x7A: RESULTIS pr6502imp (pc, "x NOP")
    CASE #x9A: RESULTIS pr6502imp (pc, "  TXS")
    CASE #xBA: RESULTIS pr6502imp (pc, "  TSX")
    CASE #xDA: RESULTIS pr6502imp (pc, "x NOP")
    CASE #xFA: RESULTIS pr6502imp (pc, "x NOP")

    CASE #x1E: RESULTIS pr6502absx(pc, "  ASL")
    CASE #x3E: RESULTIS pr6502absx(pc, "  ROL")
    CASE #x5E: RESULTIS pr6502absx(pc, "  LSR")
    CASE #x7E: RESULTIS pr6502absx(pc, "  ROR")
    CASE #x9E: RESULTIS pr6502absx(pc, "  STX")
    CASE #xBE: RESULTIS pr6502absy(pc, "  LDX")
    CASE #xDE: RESULTIS pr6502absx(pc, "  DEC")
    CASE #xFE: RESULTIS pr6502absx(pc, "  INC")
  }
}

AND pr6502imm(pc, str) = VALOF // #exp
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  prflags(pc)
  writef("// %i5: %x2 %x2     %s   #%i3",
             pc+baseaddr, op, a1, str, a1)
  RESULTIS pc+2
}

AND pr6502abs(pc, str) = VALOF // Absolute
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  LET a2 = gb(pc+2)
  LET w  = gw(pc+1)
  TEST w >= 49152    // Base of the BBC operating system rom
  THEN { prflags(pc)
         writef("// %i5: %x2 %x2 %x2  %s  $%x4",
                    pc+baseaddr, op, a1, a2, str, w)
       }
  ELSE { prflags(pc)
         writef("// %i5: %x2 %x2 %x2  %s  %i5",
                    pc+baseaddr, op, a1, a2, str, w)
       }
  RESULTIS pc+3
}

AND pr6502zpg(pc, str) = VALOF // zero page
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  prflags(pc)
  writef("// %i5: %x2 %x2     %s  %i5",
             pc+baseaddr, op, a1, str, a1)
  RESULTIS pc+2
}

AND pr6502acc(pc, str) = VALOF // Accumulator
{ LET op = gb(pc)
  prflags(pc)
  writef("// %i5: %x2        %s      A",
             pc+baseaddr, op, str)
  RESULTIS pc+1
}

AND pr6502imp(pc, str) = VALOF // Implied
{ LET op = gb(pc)
  prflags(pc)
  writef("// %i5: %x2        %s",
             pc+baseaddr, op, str)
  RESULTIS pc+1
}

AND pr6502indx(pc, str) = VALOF  // (IND,X)
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  prflags(pc)
  writef("// %i5: %x2 %x2     %s  (%i3,X)",
             pc+baseaddr, op, a1, str, a1)
  RESULTIS pc+2
}

AND pr6502indy(pc, str) = VALOF  // (IND),Y
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  prflags(pc)
  writef("// %i5: %x2 %x2     %s  (%i3),Y",
             pc+baseaddr, op, a1, str, a1)
  RESULTIS pc+2
}

AND pr6502zpgx(pc, str) = VALOF  // zpage,X
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  prflags(pc)
  writef("// %i5: %x2 %x2     %s  %i3,X",
             pc+baseaddr, op, a1, str, a1)
  RESULTIS pc+2
}

AND pr6502absx(pc, str) = VALOF  // abs,X
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  LET a2 = gb(pc+2)
  prflags(pc)
  writef("// %i5: %x2 %x2 %x2  %s  %i3,X",
             pc+baseaddr, op, a1, a2, str, a1 | a2<<8)
  RESULTIS pc+3
}

AND pr6502absy(pc, str) = VALOF  // abs,Y
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  LET a2 = gb(pc+2)
  prflags(pc)
  writef("// %i5: %x2 %x2 %x2  %s %i3,Y",
             pc+baseaddr, op, a1, a2, str, a1 | a2<<8)
  RESULTIS pc+3
}

AND pr6502rel(pc, str) = VALOF  // Relative
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  LET sb = gsb(pc+1)
  prflags(pc)
  writef("// %i5: %x2 %x2     %s  %i5",
             pc+baseaddr, op, a1, str, pc+2+sb+baseaddr)
  RESULTIS pc+2
}

AND pr6502ind(pc, str) = VALOF  // Indirect
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  prflags(pc)
  writef("// %i5: %x2 %x2     %s  (%i3)",
             pc+baseaddr, op, a1, str, a1)
  RESULTIS pc+2
}

AND pr6502zpgy(pc, str) = VALOF // zpage,Y
{ LET op = gb(pc)
  LET a1 = gb(pc+1)
  prflags(pc)
  writef("// %i5: %x2 %x2     %s  %i3,Y",
             pc+baseaddr, op, a1, str, a1)
  RESULTIS pc+2
}

