#include <stdio.h>
#include <stdlib.h>
#include <sys/mman.h>
#include "jit_exec.h"
#include "jit_exec_inc.h"

void hex_print(char* data, int length) {
  int ptr = 0;
  printf("  ");
  for(;ptr < length;ptr++) {
    printf("0x%02x ",(unsigned char)*(data+ptr));
    if ((ptr+1) % 8 == 0) { printf("\n  "); }
  }
  printf("\n");
}

char * alloc_executable(size_t len) {
  /* Local variables */
  size_t page_size = getpagesize();
  /* Round up so that len is a multiple of getpagesize() */
  if (len % page_size) {
    len &= ~(page_size-1);
    len += page_size;
  }
  /* Open a file for use as scratch memory */
  int fd = 0, ret = 0;
  void *pa = MAP_FAILED; 
  char template[] = "/tmp/alloc_executable_XXXXXX";
  fd = mkstemp(template);
  if (fd < 0) return pa;
  unlink(template);
  ret = ftruncate(fd, len);
  if (ret < 0) return pa;
  /* Do the allocation */
  return mmap(NULL, len, PROT_READ|PROT_WRITE|PROT_EXEC, MAP_PRIVATE, fd, 0);
}

void print_stack(int * stack, int * end_of_stack) {
  int * ptr = end_of_stack - 1; 
  while (stack <= ptr) {
    printf("  %d:\t%d\n",(int)(stack) + (int)(ptr),*ptr);
    ptr--;
  }
}

main(int argc, char** argv) {
  /* read arguments */
  if (argc < 2) {
    printf("This program expects at least one argument...\n");
    return;
  }
  int verbose = 0;
  int incremental = 0;
  int flags = 1;
  while (flags < argc) {
    if (strcmp(argv[flags],"-v") == 0) { verbose = 1; } 
    else if (strcmp(argv[flags],"-i") == 0) { incremental = 1; }
    else if (strcmp(argv[flags],"-iv") == 0) { incremental = 1; verbose = 1; }
    else if (strcmp(argv[flags],"-vi") == 0) { incremental = 1; verbose = 1; }
    else { break; }
    flags++;
  }
  /* allocate space */
  char * new_code = alloc_executable(100);
  if (new_code == MAP_FAILED) { printf("Allocation failed!\n"); return; }
  char * input = argv[verbose?2:1];
  int stack[256];
  int temp_space[300];
  /* read stack from arguments */
  int i;
  int count = argc - flags - 1;
  int * end_of_stack = &(stack[256]);
  for (i = 0; i < count; i++) {
    stack[255-i] = atoi(argv[i+flags+1]);
  }
  int * stack_ptr = &(stack[255-(i-1)]);
  /* dump stack on screen */
  if (verbose) {
    printf("\nVerbose mode is on.\n");
    if (incremental) { printf("Using incremental version of JIT compiler.\n"); }
    printf("\nInitial state of stack:\n");
    print_stack(stack_ptr, end_of_stack);
    printf("\nExecuting JIT compiler on program: \"%s\"\n", input);
  }
  /* execute JIT compiler */
  if (incremental) {
    stack_ptr = jit_exec_inc(input, stack_ptr, new_code, temp_space);
  } else {
    stack_ptr = jit_exec(input, stack_ptr, new_code);
  }
  /* dump stack on screen */
  if (verbose) {
    printf("\nFinal state of stack:\n", i);
    print_stack(stack_ptr, end_of_stack);
    printf("\nGenerated x86 code:\n");
    hex_print(new_code, 64);
  } else {
    printf("Top of stack on exit: %d\n",*stack_ptr);
  }
}



