/* StegFS - The Steganographic File System */
/* Copyright (c)1999 Andrew McDonald <andrew@mcdonald.org.uk> */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include <string.h>
#include <fcntl.h>
#include <sys/ioctl.h>
#include <errno.h>
#include <linux/stegfs_fs.h>

void printusage(char *);

int main(int argc, char *argv[]) {

	int fd, r, l, l2, errno2;
	struct stegfs_levelset levelset;
	char * pass;
	char * tmpc;

	if (argc < 2 || argc > 3)
		printusage(argv[0]);
	fd = open (argv[1], O_RDONLY|O_NONBLOCK);
	if (fd == -1)
		return -1;

	r = ioctl(fd, STEGFS_IOC_GETLEVEL, &l);
	if (r != 0) {
		perror("Error");
		exit(1);
	}

	if (argc == 2) {
		printf("StegFS at level: %d\n", l);
		exit(0);
	}

	/* change level */

	levelset.level = strtol(argv[2], &tmpc, 0);
	if (*argv[2] == '\0' || *tmpc != '\0' ||
	    levelset.level < 0 || levelset.level > STEGFS_MAX_LEVELS-1) {
		printf("Invalid level number\n");
		exit(1);
	}

	printf("StegFS current level change %d to %d\n", l, levelset.level);

	if (l == levelset.level) {
		printf("No change\n");
		exit(0);
	}
	else if (l < levelset.level) {
		pass = getpass("Enter passphrase: ");

		strncpy(levelset.passw, pass, 256);
		levelset.passw[255] = '\0';  /* force null terminate */
		memset(pass, 0, strlen(pass));
	}
	else /* l > levelset.level */
		levelset.passw[0] = '\0';

	r = ioctl(fd, STEGFS_IOC_SETLEVEL, &levelset);
	memset(levelset.passw, 0, 256);
	if (r != 0) {
		if (errno == EBADPASS) {
			printf("Level open failed: bad passphrase?\n");
			exit(1);
		}
		if (errno != EBUSY) {
			perror("Error");
			exit(1);
		}
	}
	errno2 = errno;

	/* double check */
	r = ioctl(fd, STEGFS_IOC_GETLEVEL, &l2);
	if (r != 0) {
		perror("Error");
		exit(1);
	}

	if (errno2 == EBUSY) {
		errno = errno2;
		perror("Error");
		if (l != l2) {
			/* managed to close some */
			printf("Level changed from %d to %d\n", l, l2);
		}
		exit(1);
	}

	if (l2 != levelset.level) {
		printf("Error: Change failed: Unknown error occurred\n");
		exit(1);
	}

	printf("Level change successful\n");

	close(fd);
	exit(0);
}


void printusage(char *progname) {

  printf("Usage: %s /mnt/mntpoint [newhighestlevel]\n", basename(progname));

  exit(1);
}
