/* Stegfs Helper Functions */
/* Sort of in the vein of libext2, but not as useful or complete */

/* Copyright (c)1999 Andrew McDonald (andrew@mcdonald.org.uk) */
/* $Id: stegfs.c,v 1.17 1999/10/06 17:28:11 adm36 Exp adm36 $ */

#include <stdlib.h>
#include <stdio.h>
#include <string.h>
#include <sys/time.h>
#include <sys/stat.h>
#include <unistd.h>
#include <errno.h>
#include "stegfs.h"
#include "serpent.h"
#include "nullenc.h"
#include "rc6.h"
#include "sha1.h"

static struct stegfs_cipher serpent_ops = {
	"serpent",
	serpent_set_key,
	serpent_encrypt,
	serpent_decrypt
};

static struct stegfs_cipher rc6_ops = {
	"rc6",
	rc6_set_key,
	rc6_encrypt,
	rc6_decrypt
};

static struct stegfs_cipher nullenc_ops = {
	"nullenc",
	nullenc_set_key,
	nullenc_encrypt,
	nullenc_decrypt
};

#define NUMCIPHERS 3
static struct stegfs_cipher *stegfs_ciphers[NUMCIPHERS] = {
	&serpent_ops,
	&rc6_ops,
	&nullenc_ops
};


int open_fs(struct stegfs_fs *fs, char *fsname, char *btabname, char *cipher,
	    int newfs) {

	int i;
	int mount_flags;
	struct stat statbuf;
	errcode_t retval;
	fs->e2fs = 0;

	i = stat(fsname, &statbuf);
	if (i == -1 && errno == ENOENT) {
		printf("Block device %s does not exist\n", fsname);
		return 0;
	}
	if (!S_ISBLK(statbuf.st_mode)) {
		printf("Warning: %s is not a block device\n", fsname);
	}

	retval = ext2fs_check_if_mounted(fsname, &mount_flags);
        if (retval) {
                com_err("ext2fs_check_if_mount", retval,
                        "while determining whether %s is mounted.",
                        fsname);
                return 0;
        }
        if (mount_flags & EXT2_MF_MOUNTED) {
		printf("%s is mounted; will not make a filesystem here!\n",
		       fsname);
		return 0;
	}

	retval = ext2fs_open(fsname, EXT2_FLAG_RW, 0, 0, unix_io_manager,
			     &(fs->e2fs));
        if (retval) {
                com_err(fsname, retval, "while opening file system");
                return 0;
        }
        if (fs->e2fs->super->s_state & EXT2_ERROR_FS) {
		printf("File system on %s contains errors. Run e2fsck.\n",
		       fsname);
		ext2fs_close(fs->e2fs);
		return 0;
	}
	if((fs->e2fs->super->s_state & EXT2_VALID_FS) == 0) {
		printf("File system on %s was not cleanly unmounted.\n",
		       fsname);
		ext2fs_close(fs->e2fs);
		return 0;
	}

	if (newfs) {
		i = stat(btabname, &statbuf);
		if (i == 0) {
			printf("Block table file %s already exists.\n",
			       btabname);
			ext2fs_close(fs->e2fs);
			return 0;
		}
		fs->btabfile = fopen(btabname, "w+");
	}
	else
		fs->btabfile = fopen(btabname, "r+");

	if (!fs->btabfile)
		return 0;

	fs->cipher = NULL;
	for (i=0; i<NUMCIPHERS; i++) {
		if(!strcmp(cipher, stegfs_ciphers[i]->id))
			fs->cipher = stegfs_ciphers[i];
	}
	if (!fs->cipher) {
		printf("Invalid cipher name: %s\n", cipher);
		return 0;
	}

	ext2fs_read_bitmaps(fs->e2fs);
  return 1;
}

int close_fs(struct stegfs_fs *fs) {
	ext2fs_close(fs->e2fs);
	fclose(fs->btabfile);
	return 1;
}

void get_skeys(struct stegfs_fs *fs, int level, char *pass) {

	int i;
	char enckey[16];

	fs->slevel = level;

	fseek(fs->btabfile, fs->e2fs->super->s_blocks_count*
	      sizeof(struct stegfs_btable), SEEK_SET);
	fseek(fs->btabfile, .5*level*(level-1)*sizeof(struct stegfs_btable),
	      SEEK_CUR);

	for (i=0; i<level; i++) {
		fs->slkeys[i] = malloc(16);
		fread(&enckey, 16, 1, fs->btabfile);
		decrypt_skey(fs, pass, enckey, fs->slkeys[i]);
	}

	return;
}

int get_stegfs_inodes(struct stegfs_fs *fs, struct inoblocks **inolist) {

	int i,j,l;
	struct stegfs_btable btab, btab2;
	struct inoblocks *inob, *inobp, *inobtmp;
	int addedblk;

	fseek(fs->btabfile, 0, SEEK_SET);

	for(i=0; i<fs->e2fs->super->s_blocks_count; i++) {

		fread(&btab2, sizeof(struct stegfs_btable), 1,
		      fs->btabfile);
		for(l=0; l<fs->slevel; l++) {
			decrypt_btab2(fs, l+1, i, &btab2, &btab);

			if(btab.magic1 == 0 && btab.magic2 == 1) {
				inob = inolist[l];
				inobp = inob;
				addedblk = 0;
				while(inob != NULL) {
					if(inob->inode == btab.ino) {
						for(j=0; j<STEGFS_MAX_INO_COPIES; j++) {
							if(inob->block[j] == 0) {
								inob->block[j] = i;
								addedblk = 1;
								break;
							}
						}
						if(!addedblk) {
							printf("Too many inode copies: inode %lu block %u\n", inob->inode, i);
							addedblk = 1;
						}
						break;
					}

					inobp = inob;
					inob = inob->next;
				}
				if(!addedblk) {

					inobtmp = (struct inoblocks*)malloc(sizeof(struct inoblocks));
					if(inobtmp == NULL) {
						printf("can't get free memory\n");
						return 0;
					}
					if(inobp == NULL) {

						inolist[l] = inobtmp;
						inobp = inolist[l];
						inobp->inode = btab.ino;
						inobp->block[0] = i;
						inobp->next = NULL;

						for(j=1; j<STEGFS_MAX_INO_COPIES; j++) {
							inobp->block[j] = 0;
						}

					}
					else {

						inobp->next = inobtmp;
						inobp->next->inode = btab.ino;
						inobp->next->block[0] = i;
						inobp->next->next = NULL;
						for(j=1; j<STEGFS_MAX_INO_COPIES; j++)
							inobp->next->block[j] = 0;

					}
				}
			}
		}
	}

  return 1;
}

void randomize_empties(struct stegfs_fs *fs /*, int avoidhidden*/) {
  /* read block group block bitmap then randomize contents of empties */

	unsigned char *buff1;
	errcode_t retval;
	blk_t n;
	FILE *devrand;

#ifdef DEVEL_ZEROFILL
	devrand = fopen("/dev/zero", "r");
#else
	devrand = fopen(RAND_DEV, "r");
#endif
	if(devrand==NULL) {
		printf("Error opening %s\n", RAND_DEV);
		exit(3);
	}

	buff1 = (unsigned char *)malloc(fs->e2fs->blocksize);

	printf("Filling empty blocks: ");

	for(n=fs->e2fs->super->s_first_data_block;
	    n<fs->e2fs->super->s_blocks_count; n++) {
		printf("%3d%%", (100*n/fs->e2fs->super->s_blocks_count));
		if(!ext2fs_test_block_bitmap(fs->e2fs->block_map, n)) {
			fread(buff1, 1, fs->e2fs->blocksize, devrand);
			retval = io_channel_write_blk(fs->e2fs->io,
						      n, 1, buff1);
		}
		printf("\b\b\b\b");
	}

	printf("Done\n");
	fclose(devrand);
	return;
}

u_int8_t ext2_isblockused(struct stegfs_fs *fs, blk_t blocknum) {

	ext2fs_block_bitmap map;
	map = fs->e2fs->block_map;
	/*FIXME: check we got it*/

	return ext2fs_test_block_bitmap(map, blocknum);
}

u_int8_t stegfs_isblockused(struct stegfs_fs *fs, blk_t blocknum) {

	struct stegfs_btable btab, btab2;
	int i;

	get_btab(fs, blocknum, &btab);

	for (i=1; i<(fs->slevel+1); i++) {
		decrypt_btab2(fs, i, blocknum, &btab, &btab2);
		if(btab2.magic1 == 0 &&
		   (btab2.magic2 == 0 || btab2.magic2 == 1))
			return 1;
	}

	return 0;
}

blk_t get_new_block(struct stegfs_fs *fs) {

	FILE *devrand;
	u_int32_t scale, maxblock;
	u_int32_t randbs, randbsx;
	int k;

	devrand = fopen(RAND_DEV, "r");
	if(devrand==NULL) {
		printf("Error opening %s\n", RAND_DEV);
		exit(3);
	}

	maxblock = fs->e2fs->super->s_blocks_count - 1;
	scale = ((unsigned int)-1) / maxblock;

	fread(&randbs, sizeof(u_int32_t), 1, devrand);

	k=0;
	randbsx = randbs / scale;
	while (randbsx > maxblock || randbsx < 1 ||
	       ext2_isblockused(fs,randbsx) ||
	       stegfs_isblockused(fs,randbsx)) {
		k++;
		if(k>10000) {
			return 0;
		}
		fread(&randbs, sizeof(u_int32_t), 1, devrand);
		randbsx = randbs / scale;
	}

	fclose(devrand);

	return randbsx;
}

int get_new_inode_blocks(struct stegfs_fs *fs, int slevel, ino_t ino,
			 int numcopies, unsigned int *blocks) {

	u_int32_t scale, maxblock;
	u_int32_t randbsx;
	int k,i;
	unsigned int digest[5];
	SHA1_CTX sha1ctx;
#ifdef STEGFS_OLD_INO_POS
	unsigned char *keystr;
	int slen = 16+sizeof(int);
#else
	unsigned int keystr[6];
#endif
	int blks = 0;

	maxblock = fs->e2fs->super->s_blocks_count - 1;
	scale = ((unsigned int)-1) / maxblock;

#ifdef STEGFS_OLD_INO_POS
	keystr = malloc(slen);
	memcpy(keystr, fs->slkeys[slevel-1], 16);
	memcpy(keystr+16, &ino, sizeof(int));

	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, keystr, slen);
	SHA1Final((unsigned char *)digest, &sha1ctx);

	free(keystr);
#else
	memcpy((unsigned char*)keystr, fs->slkeys[slevel-1], 16);
	keystr[4] = ino;
	keystr[5] = 0;

        SHA1Init(&sha1ctx);
        SHA1Update(&sha1ctx, (unsigned char *)keystr, 24);
        SHA1Final((unsigned char *)digest, &sha1ctx);
#endif

	k=0;

	while (1) {

		for (i=0; i<5; i++) {
			randbsx = digest[i] / scale;
			if (!(randbsx > maxblock || randbsx < 1 ||
			      ext2_isblockused(fs,randbsx) ||
			      stegfs_isblockused(fs,randbsx))) {
				blocks[blks++] = randbsx;
			}
			if (blks == numcopies)
				return 1;
		}

		SHA1Init(&sha1ctx);
#ifdef STEGFS_OLD_INO_POS
		SHA1Update(&sha1ctx, (unsigned char *)digest,
			   5*sizeof(unsigned int));
#else
		keystr[5]++;
		SHA1Update(&sha1ctx, (unsigned char *)keystr, 24);
#endif
		SHA1Final((unsigned char *)digest, &sha1ctx);

	}


}

void put_btab(struct stegfs_fs *fs, u_int32_t bnum,
	      struct stegfs_btable *btab) {


	fseek(fs->btabfile, bnum*sizeof(struct stegfs_btable), SEEK_SET);

	fwrite(btab, sizeof(struct stegfs_btable), 1, fs->btabfile);

	return;

}

void get_btab(struct stegfs_fs *fs, u_int32_t bnum,
	      struct stegfs_btable *btab) {


	fseek(fs->btabfile, bnum*sizeof(struct stegfs_btable), SEEK_SET);

	fread(btab, sizeof(struct stegfs_btable), 1, fs->btabfile);

	return;

}

unsigned int stegfs_chksum(unsigned char * data, int size) {
#ifdef STEGFS_PARANOID_XSUM
	SHA1_CTX sha1ctx;
	unsigned int digest[5];
  
	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, data, size);
	SHA1Final((unsigned char *)digest, &sha1ctx);

	return digest[0];
#else
        unsigned char *datax;

        datax = data + size - sizeof(unsigned int);

        return *datax;

#endif
}

char *make_key(struct stegfs_fs *fs, int level, unsigned int blocknum) {
#ifdef STEGFS_PARANOID
	char *keystr;
	SHA1_CTX sha1ctx;
	unsigned int digest[5];
	char *key;
	int slen = 16+1+sizeof(unsigned int);

	keystr = malloc(slen);
	strncpy(keystr, fs->slkeys[level-1], slen);
	memcpy(keystr+16+1, &blocknum, sizeof(unsigned int));

	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, keystr, slen);
	SHA1Final((unsigned char *)digest, &sha1ctx);
	free(keystr);

	key = malloc(16);
	memcpy(key, digest, 16);

	return key;
#else
	char *key;
	unsigned int *key2;
	key = malloc(16);
	memcpy(key, fs->slkeys[level-1], 16);
	key2 = (unsigned int *)key;
	key2[3] = key2[3] ^ blocknum;

	return key;
#endif
}

void encrypt_skey(struct stegfs_fs *fs, char *passwd, char *skey,
		  char *skeyout) {
	SHA1_CTX sha1ctx;
	unsigned int digest[5];
	char *keys;

	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, passwd, strlen(passwd));
	SHA1Final((unsigned char *)digest, &sha1ctx);

	keys = fs->cipher->set_key((char *)digest, 128);
	fs->cipher->encrypt(keys, skey, skeyout);
	free(keys);
}

void decrypt_skey(struct stegfs_fs *fs, char *passwd, char *skey,
		  char *skeyout) {
	SHA1_CTX sha1ctx;
	unsigned int digest[5];
	char *keys;

	SHA1Init(&sha1ctx);
	SHA1Update(&sha1ctx, passwd, strlen(passwd));
	SHA1Final((unsigned char *)digest, &sha1ctx);

	keys = fs->cipher->set_key((char *)digest, 128);
	fs->cipher->decrypt(keys, skey, skeyout);
	free(keys);
}

void encrypt_cbc(struct stegfs_fs *fs, int level, blk_t bnum,
		 char *data, int size, unsigned short iv) {

	char *key, *keys;
	char outb[16], inb[16];
	char *p;
	int i;

	key = make_key(fs, level, bnum);
	keys = fs->cipher->set_key(key, 128);
	for(p=data; p<(data+size); p+=16) {
		if (p>data) {
			for(i=0; i<16; i++)
				inb[i] = p[i]^outb[i];
		}
		else {
			memcpy(inb, p, 16);
                        ((unsigned short *)inb)[0] ^= iv;
		}
		fs->cipher->encrypt(keys, inb, outb);
		memcpy(p, outb, 16);
	}
	free(key);
	free(keys);

}

void encrypt_cbc2(struct stegfs_fs *fs, int level, blk_t bnum,
		  char *data, char *data2, int size, unsigned short iv) {

	char *key, *keys;
	char *p, *p2;
	char inb[16];
	int i;

	key = make_key(fs, level, bnum);
	keys = fs->cipher->set_key(key, 128);
	for(p=data, p2=data2; p<(data+size); p+=16, p2+=16) {
		if (p>data) {
			for(i=0; i<16; i++)
				inb[i] = p[i]^(p2-16)[i];
		}
		else {
			memcpy(inb, p, 16);
                        ((unsigned short *)inb)[0] ^= iv;
		}
		fs->cipher->encrypt(keys, inb, p2);
	}
	free(key);
	free(keys);

}

void decrypt_cbc(struct stegfs_fs *fs, int level, blk_t bnum,
		 char *data, int size, unsigned short iv) {

	char *key, *keys;
	char inb[16], inb2[16], outb[16];
	char *p;
	int i;

	key = make_key(fs, level, bnum);
	keys = fs->cipher->set_key(key, 128);
	for(p=data; p<(data+size); p+=16) {
		if (p>data) {
			memcpy(inb2, inb, 16);
			memcpy(inb, p, 16);
		}
		fs->cipher->decrypt(keys, p, outb);
		if (p>data) {
			for(i=0; i<16; i++)
				p[i] = outb[i]^inb2[i];
		}
		else {
                        ((unsigned short *)outb)[0] ^= iv;
			memcpy(inb, p, 16);
			memcpy(p, outb, 16);
		}
	}
	free(key);
	free(keys);

}

void decrypt_cbc2(struct stegfs_fs *fs, int level, blk_t bnum,
		  char *data, char *data2, int size, unsigned short iv) {

	char *key, *keys;
	char *p, *p2;
	int i;

	key = make_key(fs, level, bnum);
	keys = fs->cipher->set_key(key, 128);
	for(p=data, p2=data2; p<(data+size); p+=16, p2+=16) {
		fs->cipher->decrypt(keys, p, p2);
		if (p>data) {
			for(i=0; i<16; i++)
				p2[i] = p2[i]^(p-16)[i];
		}
		else
                        ((unsigned short *)p2)[0] ^= iv;
	}
	free(key);
	free(keys);

}

void encrypt_btab(struct stegfs_fs *fs, int slevel, blk_t bnum,
		  struct stegfs_btable *btab) {

        char *tmpdata;
        char *key, *keys;

        tmpdata = malloc(sizeof(struct stegfs_btable));

        memcpy(tmpdata, btab, sizeof(struct stegfs_btable));

        key = make_key(fs, slevel, bnum);
        keys = fs->cipher->set_key(key, 128);
        free(key);
        fs->cipher->encrypt(keys, tmpdata, (char *)btab);
        free(keys);
        free(tmpdata);

        return;
}

void encrypt_btab2(struct stegfs_fs *fs, int slevel, blk_t bnum,
		   struct stegfs_btable *btab, struct stegfs_btable *btab2) {

        char *key, *keys;

        key = make_key(fs, slevel, bnum);
        keys = fs->cipher->set_key(key, 128);
        free(key);
        fs->cipher->encrypt(keys, (char *)btab, (char *)btab2);
        free(keys);

        return;
}

void decrypt_btab(struct stegfs_fs *fs, int slevel, blk_t bnum,
		  struct stegfs_btable *btab) {

        char *tmpdata;
        char *key, *keys;

        tmpdata = malloc(sizeof(struct stegfs_btable));

        memcpy(tmpdata, btab, sizeof(struct stegfs_btable));

        key = make_key(fs, slevel, bnum);
        keys = fs->cipher->set_key(key, 128);
        free(key);
        fs->cipher->decrypt(keys, tmpdata, (char *)btab);
        free(keys);
        free(tmpdata);

        return;
}


void decrypt_btab2(struct stegfs_fs *fs, int slevel, blk_t bnum,
		   struct stegfs_btable *btab, struct stegfs_btable *btab2) {

        char *key, *keys;

        key = make_key(fs, slevel, bnum);
        keys = fs->cipher->set_key(key, 128);
        free(key);
        fs->cipher->decrypt(keys, (char *)btab, (char *)btab2);
        free(keys);

        return;
}

void mkroot(struct stegfs_fs *fs, int level, char *passwd, int ic, int bc) {

	ino_t inonum;
	int i;
	struct timeval tv;
	struct timezone tz;
	struct stegfs_btable btab, btab2;
	struct stegfs_inode inode, inode2;
	char *datablock, *datablock2;
	struct ext2_dir_entry_2 *dirent;
	blk_t *iblks;
	errcode_t retval;

	inonum = EXT2_ROOT_INO | STEGFS_INO_MASK | (level)<<(STEGFS_LVL_SHIFT);

	btab.magic1 = 0;
	btab.magic2 = 1;
	btab.iv = 0;
	btab.ino = inonum;

	inode.i_mode = S_IFDIR|S_IRWXU|S_IRGRP|S_IXGRP|S_IROTH|S_IXOTH;
	inode.i_uid = fs->e2fs->super->s_def_resuid;
	inode.i_size = fs->e2fs->blocksize;
	gettimeofday(&tv, &tz);
	inode.i_atime = inode.i_ctime = inode.i_mtime = tv.tv_sec;
	inode.i_dtime = 0;
	inode.i_gid = fs->e2fs->super->s_def_resgid;
	inode.i_links_count = 3; /* /, . and .. */
	inode.i_blocks = 1;
	inode.i_flags = 0;
	inode.osd1.linux1.l_i_reserved1 = 0;

	inode.i_version = 0;
	inode.i_file_acl = 0;
	inode.i_dir_acl = 0;
	inode.i_faddr = 0;
	inode.osd2.linux2.l_i_frag = 0;
	inode.osd2.linux2.l_i_fsize = 0;
	inode.osd2.linux2.i_pad1 = 0;
	inode.osd2.linux2.l_i_reserved2[0] = 0;
	inode.osd2.linux2.l_i_reserved2[1] = 0;

	inode.i_icopies = ic;
	inode.i_bcopies = bc;
	inode.i_pad0 = 0;

	for(i=0; i<EXT2_N_BLOCKS*STEGFS_MAX_BLOCK_COPIES+STEGFS_MAX_INO_COPIES;
	    i++)
		inode.i_inode[i] = 0;

	iblks = malloc(inode.i_icopies * sizeof(u_int32_t));
	get_new_inode_blocks(fs, level, inonum, inode.i_icopies, iblks);

	for(i=0; i<inode.i_icopies; i++) {
		inode.i_inode[i] = iblks[i];
		encrypt_btab2(fs, level, inode.i_inode[i], &btab, &btab2);
		put_btab(fs, inode.i_inode[i], &btab2);
		/*
		  printf("i %u\n", inode.i_inode[i]);
		*/
	}
	free(iblks);

	btab.magic2 = 0;
	for(i=0; i<inode.i_bcopies; i++) {
		inode.i_block[i*EXT2_N_BLOCKS] = get_new_block(fs);
		encrypt_btab2(fs, level, inode.i_block[i*EXT2_N_BLOCKS],
			      &btab, &btab2);
		put_btab(fs, inode.i_block[i*EXT2_N_BLOCKS], &btab2);
		/*
		  printf("d %u\n", inode.i_block[i*EXT2_N_BLOCKS]);
		*/
	}

	btab.magic2 = 1;

	for(i=0; i<inode.i_icopies; i++) {
		encrypt_cbc2(fs, level, inode.i_inode[i],
			     (char *)&inode, (char *)&inode2,
			     sizeof(struct stegfs_inode), btab.iv);

		retval = io_channel_write_blk(fs->e2fs->io,
					      inode.i_inode[i], 1, &inode2);
		btab.bchecksum = stegfs_chksum((unsigned char *)&inode2,
					       sizeof(struct stegfs_inode));
		encrypt_btab2(fs, level, inode.i_inode[i], &btab, &btab2);
		put_btab(fs, inode.i_inode[i], &btab2);
	}

	datablock = malloc(fs->e2fs->blocksize);
	datablock2 = malloc(fs->e2fs->blocksize);
	dirent = (struct ext2_dir_entry_2 *)
		malloc(sizeof(struct ext2_dir_entry_2) * 2);
	dirent[0].rec_len = 12;
	dirent[0].name_len = 1;
	dirent[0].file_type = 0;
	strncpy(dirent[0].name, ".", EXT2_NAME_LEN);
	dirent[0].inode = inonum;

	dirent[1].rec_len = fs->e2fs->blocksize - dirent[0].rec_len;
	dirent[1].name_len = 2;
	dirent[1].file_type = 0;
	strncpy(dirent[1].name, "..", EXT2_NAME_LEN);
	dirent[1].inode = inonum;

	memcpy(datablock, &dirent[0], sizeof(struct ext2_dir_entry_2));
	memcpy(datablock+dirent[0].rec_len, &dirent[1],
	       sizeof(struct ext2_dir_entry_2));

	btab.magic2 = 0;
	for(i=0; i<inode.i_bcopies; i++) {
		encrypt_cbc2(fs, level, inode.i_block[i*EXT2_N_BLOCKS],
			     datablock, datablock2,
			     fs->e2fs->blocksize, btab.iv);
		retval = io_channel_write_blk(fs->e2fs->io,
					      inode.i_block[i*EXT2_N_BLOCKS],
					      1, datablock2);
		btab.bchecksum = stegfs_chksum(datablock2,
					       fs->e2fs->blocksize);
		encrypt_btab2(fs, level, inode.i_block[i*EXT2_N_BLOCKS],
			      &btab, &btab2);
		put_btab(fs, inode.i_block[i*EXT2_N_BLOCKS], &btab2);

	}

	free(dirent);
	free(datablock);
	free(datablock2);
}
