/* StegFS - The Steganographic File System */
/* Copyright (c)1999 Andrew McDonald <andrew@mcdonald.org.uk> */

/* Activate additional levels in a StegFS file system. */
/* $Id$ */

#include <stdio.h>
#include <stdlib.h>
#include <unistd.h>
#include "stegfs.h"
#include <sys/stat.h>
#include <sys/sysmacros.h>
#include <linux/major.h>
#include <string.h>
#include "sha1.h"

void printusage(char *);
int verify_skeys(struct stegfs_fs *fs, int level);

int main(int argc, char* argv[]) {

	struct stegfs_fs *fssys;
	int i,j,c;
	char *passwds[STEGFS_MAX_LEVELS-1];
	FILE *devrand;
	struct stat statbuf;

	char cipherstr[35];
	int ic, bc;
	int currmax, ltoadd;
	char promptbuf[45];
	char *pass;
	char enckey[16];

	fssys = (struct stegfs_fs *) malloc(sizeof(struct stegfs_fs));

	printf("StegFS - (c)1999 ADM\n");


	/* defaults */
	ic = 5;
	bc = 5;
	strncpy(cipherstr, "serpent", 35);

	while (1) {

		c = getopt(argc, argv, "n:c:i:b:");

		if (c == -1)
			break;

		switch (c)
		{
		case 'c':
			strncpy(cipherstr, optarg, 35);
			cipherstr[34] = '\0';
			break;
		case 'i':
			ic = atoi(optarg);
			break;
		case 'b':
			bc = atoi(optarg);
			break;

		default:
			printusage(argv[0]);
		}
	}

	if(optind != argc-4) {
		printf("Something not specified\n");
		printusage(argv[0]);
	}

	/* validate args */
	if (ic < 1 || ic > STEGFS_MAX_INO_COPIES) {
		printf("Inode copies: invalid value\n");
		exit(1);
	}
	if (bc < 1 || bc > STEGFS_MAX_BLOCK_COPIES) {
		printf("Block copies: invalid value\n");
		exit(1);
	}

	currmax = atoi(argv[optind+2]);
	ltoadd = atoi(argv[optind+3]);

	if (currmax < 0 || currmax+ltoadd > STEGFS_MAX_LEVELS-1) {
		printf("No. of levels: invalid value\n");
		exit(1);
	}

	/* Check random number source is what we think it is */
	i = stat(RAND_DEV, &statbuf);
	if (i == -1) {
		printf("Error accessing %s\n", RAND_DEV);
		exit(3);
	}
	if (!S_ISCHR(statbuf.st_mode) ||
	    major(statbuf.st_rdev) != MEM_MAJOR ||
	    minor(statbuf.st_rdev) != 9) {
		printf("WARNING!!! %s is not char device major %u minor 9!\n",
		       RAND_DEV, MEM_MAJOR);
	}

	if (!open_fs(fssys, argv[optind], argv[optind+1], cipherstr, 0)) {
		printf("Failed to open device or file\n");
		exit(2);
	}
  

	/* get passphrase for existing levels */
	sprintf(promptbuf, "Enter passphrase for existing level %u: ", currmax);
	pass = getpass(promptbuf);
	if(pass[0] == '\0') {
		printf("Aborted.\n");
		exit(1);
	}
	passwds[currmax-1] = malloc(strlen(pass)+1);
	strncpy(passwds[currmax-1], pass, strlen(pass)+1);
	memset(pass, 0, strlen(pass));

	/* Get keys */
	get_skeys(fssys, currmax, passwds[currmax-1]);
	if(!verify_skeys(fssys, currmax)) {
		printf("Invalid passphrase.\n");
		exit(1);
	}

	j = 0;
	for (i=currmax; i<currmax+ltoadd; i++) {
		sprintf(promptbuf, "Enter passphrase for new level %u: ", i+1);
		pass = getpass(promptbuf);
		if(pass[0] == '\0') {
			printf("Aborted.\n");
			exit(1);
		}
		passwds[i] = malloc(strlen(pass)+1);
		strncpy(passwds[i], pass, strlen(pass)+1);
		memset(pass, 0, strlen(pass));

		sprintf(promptbuf, "Re-enter passphrase for new level %u: ",
			i+1);
		pass = getpass(promptbuf);
		if (strcmp(pass,passwds[i]) != 0) {
			memset(pass, 0, strlen(pass));
			printf("Passphrases differ.\n");
			i--;
			j++;
			if (j > 2) {
				printf("Failed after %d attempts.\n", j);
				exit(1);
			}
		}
		else {
			memset(pass, 0, strlen(pass));
			j = 0;
		}
	}
	passwds[i] = NULL;

	fssys->slevel = currmax+ltoadd;
	if(ltoadd == 0)
		printf("This operation will simply destroy any existing "
		       "levels above level %d.\n", currmax);
	else {
		printf("Adding level");
		if(ltoadd == 1)
			printf(" %d.\n", currmax+1);
		else
			printf("s %d to %d.\n", currmax+1, currmax+ltoadd);
	}
	if (fssys->slevel == 14)
		printf("Level 15");
	else if (fssys->slevel < 14)
		printf("Levels %d to 15", fssys->slevel+1);
	if (fssys->slevel < 15)
		printf(" will not be accessible.\n");

	printf("     **********\n");
	printf("WARNING: This operation cannot be undone.\n");
	if (currmax == 14 && ltoadd == 1)
		printf("If level 15");
	if (currmax < 14)
		printf("If any of levels %d to 15", currmax+1);
	printf(" already exist, their contents will be permanently lost.\n");
	printf("     **********\n");

	printf("Are you sure you want to proceed? (y,n) ");
	if (getchar() != 'y') {
		printf("Aborted.\n");
		exit(1);
	}

	devrand = fopen(RAND_DEV, "r");
	if(devrand==NULL) {
		printf("Error opening %s\n", RAND_DEV);
		exit(3);
	}
	fseek(fssys->btabfile,
	      fssys->e2fs->super->s_blocks_count*sizeof(struct stegfs_btable), SEEK_SET);
	fseek(fssys->btabfile,
	      .5*(currmax+1)*currmax*sizeof(struct stegfs_btable),
	      SEEK_CUR);

	/* Write new encrypted keys. */
	for (i=currmax; i<currmax+ltoadd; i++) {
		fssys->slkeys[i] = malloc(16);
		fread(fssys->slkeys[i], 16, 1, devrand);
		for (j=0; j<(i+1); j++) {
			encrypt_skey(fssys, passwds[i],
				     fssys->slkeys[j], enckey);
			fwrite(enckey, sizeof(enckey), 1, fssys->btabfile);
		}
	}
	/* Overwrite any further old keys. */
	for (i=currmax+ltoadd; i<STEGFS_MAX_LEVELS-1; i++) {
		for (j=0; j<(i+1); j++) {
			fread(enckey, 16, 1, devrand);
			fwrite(enckey, 16, 1, fssys->btabfile);
		}
	}
	fclose(devrand);

	if(ltoadd > 0) {
		printf("Writing root directories: ");
		for (i=currmax; i<currmax+ltoadd; i++) {
			mkroot(fssys, i+1, passwds[i], ic, bc);
		}
		printf("Done\n");
	}

	close_fs(fssys);
	printf("Done.\n");

	return 0;
}




void printusage(char *progname) {
	printf("\nUsage:\n");
	printf("%s /dev/fsdevice btabfile current_max_level levels_to_add\n"
	       "         [-c cipher] [-i inocopies] [-b blkcopies]\n",
	       basename(progname));
	printf("Note: levels_to_add is the number of additional levels to add.\n"
	       "      current_max_level + levels_to_add must be less than or\n"
	       "      equal to %d.\n",
	       STEGFS_MAX_LEVELS-1);
	exit(1);
}

int verify_skeys(struct stegfs_fs *fs, int level) {

	int i,j,l;

	unsigned int maxblock, scale;
	blk_t block;
	int attempts, gotroot;
	struct stegfs_btable ibtab;
        SHA1_CTX sha1ctx;
        unsigned int digest[5];
#ifdef STEGFS_OLD_INO_POS
	unsigned char *keystr;
	int slen = 16+sizeof(int);
#else
	unsigned int keystr[6];
#endif

	/* Verify that they are ok */
	maxblock = fs->e2fs->super->s_blocks_count - 1;
	scale = ((unsigned int)-1) / maxblock;
	for(l=1; l<level+1; l++) {
		j = EXT2_ROOT_INO|STEGFS_INO_MASK|(l<<STEGFS_LVL_SHIFT);

#ifdef STEGFS_OLD_INO_POS
		keystr = malloc(slen);
		memcpy(keystr, fs->slkeys[l-1], 16);
		memcpy(keystr+16, &ino, sizeof(int));

		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, keystr, slen);
		SHA1Final((unsigned char *)digest, &sha1ctx);

		free(keystr);
#else
		memcpy((unsigned char*)keystr, fs->slkeys[l-1], 16);
		keystr[4] = j;
		keystr[5] = 0;

		SHA1Init(&sha1ctx);
		SHA1Update(&sha1ctx, (unsigned char *)keystr, 24);
		SHA1Final((unsigned char *)digest, &sha1ctx);
#endif
		gotroot = 0;
		attempts = 0;
		block = -1;

		while (1) {
			for (i=0; i<5; i++) {
				block = digest[i] / scale;
				if (!(block > maxblock || block < 1)) {
					/* only need to check if block
					   is used by _this_ inode number */
					get_btab(fs, block, &ibtab);
					decrypt_btab(fs, l, block, &ibtab);
					if(ibtab.magic1 == 0 &&
					   ibtab.magic2 == 1 &&
					   ibtab.ino == j) {
						gotroot = 1;
						break;
					}
				}

				attempts++;
			}

			if(gotroot || attempts > STEGFS_INO_BATT)
				break;

			SHA1Init(&sha1ctx);
#ifdef STEGFS_OLD_INO_POS
			SHA1Update(&sha1ctx, (unsigned char *)digest,
				   5*sizeof(unsigned int));
#else
			keystr[5]++;
			SHA1Update(&sha1ctx, (unsigned char *)keystr, 24);
#endif
			SHA1Final((unsigned char *)digest, &sha1ctx);

		}

		if(!gotroot)
			return 0;
	}

	return 1;
}
