{ *************************************************************************
  *                                                                       *
  *                     s t a c k    m a c h i n e                        *
  *                                                                       *
  *************************************************************************}

{To do:
   clean up AlterInt and ApplyInt instructions.
   allow reading input data from some file other than the terminal.
 }

program sm(input, 		{terminal input and input to user program}
	   output,		{terminal output}
	   execLog,		{log file of execution to write}
	   object);		{object file to read and execute}

const
    maxPc = 10000;                     {number of instructions}
    maxStack = 100;                    {stack limit}
    maxDump = 100;                     {dump limit}
    maxLabels = 700;                   {number of labels in code}
    maxRefs = 255;                     {reference count limit}
    maxSmallInt = 127;                {integers saved in table}

type
    chars = packed array [1..10] of char;

    environ = ^envNode;
    struct = ^structNode;
    arrayTree = ^arrayNode;


    envNode = packed
    record refs: 0..maxRefs;  str: struct;  next: environ end;


    {**9**ut/sm common types*******************************************}
    instKind = (haltI,returnI,ldBotI,ldConstI,ldPosI,
                 ldCloseI,ldNcloseI,applyI,applyIntI,pairI,
                 injectI,projectI,isI,alterI,alterIntI,
                 leftI,rightI,notI,plusI,minusI,timesI,divI,modI,
                 eqI,ltI,jumpI,fjumpI,callzI,pushEnvI,popEnvI,
                 labelI,ldIntI);


    instruction = packed
    record
        case kind: instKind of
             ldConstI: (str: struct);
             injectI,projectI,isI,          {arg is selector}
             applyIntI,alterIntI,
             ldPosI,ldIntI: (int: integer); {arg is integer}
             labelI,ldCloseI,ldNcloseI,jumpI,
             fjumpI,callzI:  (pc,level: 0..99999)
    end;
    {***************************************************************}

    structKind = (unusedS,botS,intS,pairS,injectS,arrayS,alterS,closureS);

    pcRange = 0..maxPc;


    {representation of all data in the machine}
    structNode = packed
    record
        refs: 0..maxRefs;
        case kind: structKind of
             unusedS: (next: struct);                  {link for free list}
             intS:    (int: integer);                  {integer constant}
             pairS:   (left,right: struct);            {tuple}
             injectS: (si: integer;  str: struct);     {union domain element}
	     arrayS: (pos,neg: arrayTree;  
	     	      baseFunc: struct);	       {binary tree array}
             alterS:  (func,arg,val: struct);          {function a-list}
             botS,                                     {pc for error location}
             closureS: (env: environ;  pc: pcRange)   {functional closures}
    end;



    {Binary tree implementation of applicative arrays}
    arrayNode = packed
    record  refs: 0..maxRefs;  value: struct;  left,right: arrayTree  end;



    {tables of use statistics for structures and instructions}
    structTable = array [structKind] of integer;

    instTable = array [instKind] of integer;


    fileName = packed array [1..20] of char;
    fileExt = packed array [1..3] of char;


var

    {components of the secd machine:}

    {stack}
    stackTop: integer;
    stack: array [1..maxStack] of struct;


    {environment}
    env: environ;


    {control}
    pc: pcRange;                        {program counter}
    ci: instruction;                    {current instruction}
    contCount: integer;                {number of instructions}
    control: array [1..maxPc] of instruction;


    {dump}
    dumpTop: integer;
    dump: array [1..maxDump] of
    record env: environ;  pc: pcRange end;



    {auxiliary variables for machine}

    condition: (running,halted,aborted);{status of execution}

    i: integer;    envl: environ;
    x,y,z,f,func: struct;               {temps for various instructions}


    {Files}
    execLog: text;			{log of program execution}
    object : file of instruction;         {object code file}
    logFileName: fileName;		{name of execLog}


    intInsts,				{instructions with an integer operand}
    pcInsts: set of instKind;           {instructions with a pc operand}

    lastRtPrinted,                    {last runtime (difference) printed}
    lastRuntime: integer;              {total runtime when last checked}


    {variables for structure management}
    structAlloc,                       {structures ever allocated}
    structCount: structTable;         {each kind of struct in use}
    sCount,                            {structures in use}
    sMaxUsed,                         {max structures ever used}
    sTotalFreed: integer;             {total freed by gc}

    sFree: struct;                     {list of free nodes}

    eFree: environ;                    {list of free environments}
    eCount: integer;                   {count of free environments}

    aFree: arrayTree;		       {free list of binary tree nodes}
    aCount: integer;		       {count of tree nodes in use}


    {predefined 'small integers'}
    siFull: boolean;                   {smallInt table has been initialized}
    smallInt: array [0..maxSmallInt] of struct;




    {instruction mnemonics}
    instChars: array [instKind] of chars;


    dynamicInst: instTable;           {count of instruction executions}



    listLimit: integer;                {print machine state when exceeded}



procedure sDecRefs(s: struct); forward;



procedure markTime;
    var t: integer;
    begin
    t := clock;   lastRtPrinted := t - lastRuntime;
    writeln(execLog, 'runtime: ', lastRtPrinted/1000 :5:1, ' seconds.');
    lastRuntime := t
    end {markTime};



function max(i,j: integer): integer;
    begin
    if i>j then max := i
    else max := j
    end {max};



function min(i,j: integer): integer;
    begin
    if i<j then min := i
    else min := j
    end {min};



procedure abortM;
    {abort the stack machine}
    begin condition := aborted end;



procedure heading;
    {Print a heading onto the log file.}
    var  datetxt,timetxt: packed array[1..10] of char;
    begin
    date(datetxt);
    time(timetxt);
    page(execLog);
    writeln(execLog, 'Stack Machine log file ', logFileName,
	    	     ' on ', datetxt, '  ', timetxt);
    writeln(execLog)
    end {heading};



    {****************************************}
    {*        garbage collection            *}
    {****************************************}


    {this reference counting system does not have a 'usedList', like in the
      universal translator.  a usedList delays the final disposal of garbage,
      allowing obsolete data to obstruct the array compactor.  this garbage
      collector immediately deletes every structure node when its reference 
      count reaches zero.}



procedure eIncRefs(env: environ);
    {increment the reference count of an environment node.}
    begin
    if env <> nil then
        if env^.refs<>maxRefs then env^.refs := env^.refs+1
    end {eIncRefs};



procedure eDecRefs(env: environ);
    {decrement the reference count of an environment;  delete if zero.}
    begin
    if env<>nil then
        if env^.refs<>maxRefs then begin
            env^.refs := env^.refs - 1;
            if env^.refs = 0 then begin
                sDecRefs(env^.str);  eDecRefs(env^.next);
                env^.next := eFree;
                eFree := env;
                eCount := eCount-1
                end
            end
    end {eDecRefs};



procedure ae(var ev: environ;  ee: environ);
    {assign environments, updating reference counts}
    begin
    eIncRefs(ee);  eDecRefs(ev);  ev := ee
    end {ae};



procedure aIncRefs(a: arrayTree);
    {Increment reference count of array node}
    begin
    if a <> nil then 
        with a^ do
	    if refs <> maxRefs then refs := refs+1
    end {aIncRefs};



procedure aDecRefs(a: arrayTree);
    {Decrement reference count of array node}
    begin
    if a<>nil then
        with a^ do
	    if refs <> maxRefs then begin
	        refs := refs - 1;
		if refs = 0 then begin
		    sDecRefs(value);  aDecRefs(left);  aDecRefs(right);
		    aCount := aCount - 1;
		    left := aFree;	{enter node onto free list}
		    aFree := a
		    end
		end
    end {aDecRefs};



procedure sIncRefs(s: struct);
    {increment the reference count of s, unless it is the maximum.
      if count reaches the max then s will never be deallocated.}
    begin
    if s<>nil then
        with s^ do begin
            assert(kind <> unusedS);
            if refs <> maxRefs then refs := refs+1 
	    end
    end {sIncRefs};



procedure sDecRefs{(s: struct)};
    {decrement the reference count of s, unless it is the maximum.
      delete the structure if the count drops to zero.}
    begin
    if s<>nil then
        with s^ do begin
	    assert(kind <> unusedS);
            if refs <> maxRefs then begin
                refs := refs-1;
                if refs = 0 then begin
                    case kind of
                        intS: {nothing else to free};
                        pairS: begin sDecRefs(left); sDecRefs(right) end;
                        alterS:
                            begin
                            sDecRefs(func);  sDecRefs(arg);  sDecRefs(val)
                            end;
                        injectS: sDecRefs(str);
			arrayS:
			    begin  
			    aDecRefs(pos);  
			    aDecRefs(neg);  
			    sDecRefs(baseFunc)  
			    end;
                        botS,closureS: eDecRefs(env)
                        end {case};
                    structCount[kind] := structCount[kind] - 1;
                    kind := unusedS;
                    next := sFree;
                    sFree := s;
                    sCount := sCount - 1;
                    sTotalFreed := sTotalFreed + 1
                    end
                end
	    end
    end {sDecRefs};



function consS(kind: structKind): struct;
    {construct a struct node-- from free list if possible, else from the
      heap.  initialize its top-level fields.}
    var s: struct;
    begin
    if sFree = nil then new(s)
    else begin s := sFree; sFree := sFree^.next end;
    s^.kind := kind; s^.refs := 0;
    structAlloc[kind] := structAlloc[kind] + 1;
    structCount[kind] := structCount[kind] + 1;
    sCount := sCount+1;
    if sCount > sMaxUsed then sMaxUsed := sCount;
    consS := s
    end {consS};



procedure pushS(str: struct);
    {push an operand onto the stack.}
    begin
    sIncRefs(str);
    stackTop := stackTop+1;
    stack[stackTop] := str
    end {pushS};



procedure popS(var str: struct);
    {pop an operand from the stack.}
    begin
    str := stack[stackTop];
    stackTop := stackTop-1
    end {popS};



procedure pushD(env: environ;  pc: pcRange);
    {push a context onto the dump, for a function application.}
    begin
    eIncRefs(env);
    dumpTop := dumpTop+1;
    dump[dumpTop].env := env;
    dump[dumpTop].pc  := pc
    end {pushD};



procedure popD(var env: environ; var pc: pcRange);
    {restore the machine's environment and control from the dump.}
    begin
    eDecRefs(env);
    env := dump[dumpTop].env;
    pc  := dump[dumpTop].pc;
    dumpTop := dumpTop-1
    end {popD};



    {****************************************}
    {*        constructor functions         *}
    {****************************************}


function consEnv(str: struct;  next: environ): environ;
    {construct an environment node.}
    var env: environ;
    begin
    eCount := eCount+1;
    sIncRefs(str);  eIncRefs(next);
    if eFree = nil then new(env)
    else begin  env := eFree;  eFree := eFree^.next end;
    env^.refs := 0;  env^.str := str;  env^.next := next;
    consEnv := env
    end {consEnv};



function consBotStr(pc: integer): struct;
    {construct a 'bottom', recording the current pc}
    var s: struct;
    begin
    s := consS(botS);
    s^.pc := pc;  s^.env := nil;
    consBotStr := s
    end {consBotStr};



function consInt(int: integer): struct;
    var s: struct;
    begin
    if siFull and (0<=int) and (int<=maxSmallInt)
    then consInt := smallInt[int]
    else begin  s := consS(intS);  s^.int := int;  consInt := s end
    end {consInt};



function consPair(left,right: struct): struct;
    var s: struct;
    begin
    sIncRefs(left);  sIncRefs(right);
    s := consS(pairS);  s^.left := left;  s^.right := right;
    consPair := s
    end {consPair};



function consInject(si: integer;  str: struct): struct;
    var s: struct;
    begin
    sIncRefs(str);
    s := consS(injectS);  s^.si := si;  s^.str := str;
    consInject := s
    end {consInject};



function consArray(pos,neg: arrayTree;  baseFunc: struct) : struct;
    var s: struct;
    begin
    aIncRefs(pos);  aIncRefs(neg);  sIncRefs(baseFunc);
    s := consS(arrayS);
    s^.pos := pos;  s^.neg := neg;  s^.baseFunc := baseFunc;
    consArray := s
    end {consArray};



function consAlter(arg,val,func: struct): struct;
    var s: struct;
    begin
    sIncRefs(arg);  sIncRefs(val);  sIncRefs(func);
    s := consS(alterS);
    s^.arg := arg;  s^.val := val;  s^.func := func;
    consAlter := s
    end {consAlter};



function consClosure(env: environ;  pc: pcRange;  level: integer): struct;
    var s: struct;  i: integer;
    begin
    for i := 1 to level do env := env^.next;
    eIncRefs(env);
    s := consS(closureS);  s^.env := env;  s^.pc := pc;
    consClosure := s
    end {consClosure};



function consArrayTree(value: struct;  left,right: arrayTree) : arrayTree;
    var a: arrayTree;
    begin
    if aFree = nil then new(a)
    else begin  a := aFree;  aFree := aFree^.left  end;
    aCount := aCount + 1;
    a^.value := value;  a^.left := left;  a^.right := right;
    a^.refs := 0;  sIncRefs(value);  aIncRefs(left);  aIncRefs(right);
    consArrayTree := a
    end {consArrayTree};   



procedure initStructCounts;
    {clear the counts of structure nodes used.}
    var sk: structKind;
    begin
    eCount := 0;
    sCount := 0;  sMaxUsed := 0;  sTotalFreed := 0;
    for sk := botS to closureS do begin
        structCount[sk] := 0;  structAlloc[sk] := 0
        end
    end {initStructCounts};



procedure printSCounts;
    {print histograms of storage use for each kind of structure.}
    var structChars: array [structKind] of chars;



    procedure printHist(st: structTable);
        var sum: integer;  sk: structKind;
        begin
        writeln(execLog,'kind        used');
        sum := 0;
        for sk := botS to closureS do sum := sum + st[sk];
        if sum <> 0 then
            for sk := botS to closureS do
                if st[sk] <> 0 then
                    writeln(execLog, structChars[sk], st[sk]:6, '     ',
                            ' ********************':
                            trunc(1.5 + 20*st[sk] / sum));
        writeln(execLog);  writeln(execLog, 'total     ', sum:6);
        writeln(execLog)
        end {printHist};

    begin
    structChars[botS]         := 'bot       ';
    structChars[intS]         := 'int       ';
    structChars[pairS]        := 'pair      ';
    structChars[injectS]      := 'inject    ';
    structChars[arrayS]       := 'array     ';
    structChars[alterS]       := 'alter     ';
    structChars[closureS]     := 'closure   ';
    writeln(execLog);
    writeln(execLog, 'structures in use'); printHist(structCount);
    writeln(execLog);
    writeln(execLog, 'structures ever produced'); printHist(structAlloc);
    writeln(execLog, 'max nodes used at any time: ', sMaxUsed:1);
    writeln(execLog, 'total freed by garbage collection: ', sTotalFreed:1);
    writeln(execLog, 'total environments ', eCount:1)
    end {printSCounts};



function equal(x,y: struct): boolean;
    {compare two structures for equality}
    begin
    if x^.kind=y^.kind then
        case x^.kind of
            botS: begin equal := false;  abortM  end;
            intS: equal := x^.int = y^.int;
            pairS: equal := equal(x^.left,y^.left)
                and equal(x^.right,y^.right);
            injectS: equal := (x^.si=y^.si) and equal(x^.str,y^.str)
	    {arrayS and closureS cannot occur due to domain checking}
            end {case}
    else begin
        equal := false;
        if (x^.kind = botS) or (y^.kind = botS) then abortM
        end
    end {equal};



procedure clearInstTable(var tab: instTable);
    {clear a table of instruction counts.}
    var ik: instKind;
    begin
    for ik := haltI to ldIntI do tab[ik] := 0
    end {clearInstTable};



procedure printInstTable(tab: instTable; time: integer);
    {print a histogram of counts for each type of instruction.}
    var sum: integer;   ik: instKind;
    begin
    writeln(execLog, 'instr      count');
    sum := 0;
    for ik := haltI to ldIntI do sum := sum + tab[ik];
    if sum<>0 then
        for ik := haltI to ldIntI do
            if tab[ik] <> 0 then
                writeln(execLog, instChars[ik], tab[ik]:6, '     ',
                        ' ********************':
                        trunc(1.5 + 20*tab[ik] / sum));
    writeln(execLog);  writeln(execLog, 'total     ', sum:6);
    writeln(execLog);
    if time<>0 then 
	writeln(execLog, 1000*sum div time:1, ' instructions/second')
    end {printInstTable};



    {****************************************}
    {*        printing machine state        *}
    {****************************************}



procedure printStruct(var f: text;  str: struct);
    {print the value of a structure.}
    begin
    if str = nil then write(f, 'nil')
    else
        with str^ do
            case kind of
                botS: write(f, 'bot<', pc:1, '>');
                intS: write(f, int:1);
                pairS:
                    begin
                    write(f, '(');   printStruct(f,left);
                    write(f, ',');   printStruct(f,right);
                    write(f, ')')
                    end;
                injectS:
                    begin
                    write(f, si:1, '[');  printStruct(f,str); write(f, ']')
                    end;
		arrayS: write(f, 'array');
                alterS:
		    begin
		    write(f, '[');  printStruct(f,arg);
		    write(f, '->'); printStruct(f,val);
		    write(f, ']');  printStruct(f,func)
		    end;
                closureS: write(f, '\', pc:1)
                end
    end {printStruct};



procedure printStack;
    {print the contents of the stack.}
    var i: integer;
    begin
    writeln(execLog);
    writeln(execLog, 'stack contents');
    for i := stackTop downto 1 do begin
        write(execLog, i:4, ':    ');  printStruct(execLog,stack[i]);
        writeln(execLog)
        end
    end {printStack};



procedure printEnv(env: environ);
    {print an environment.}
    var i: integer;
    begin
    if env<>nil then begin
        writeln(execLog);
        writeln(execLog, 'environment');
        i := 0;
        while env<>nil do begin
            write(execLog, i:2, ':    ');  printStruct(execLog, env^.str);
            writeln(execLog);
            env := env^.next;  i := i+1
            end
        end
    end {printEnv};



procedure printDump;
    {print the contents of the dump.}
    var i: integer;
    begin
    if dumpTop <> 0 then begin
        writeln(execLog);
        writeln(execLog, 'addresses on dump');
        for i := dumpTop downto 1 do
            writeln(execLog, i:4, ':    ', dump[i].pc:1)
        end
    end {printDump};



procedure printState;
    {print the entire machine state.}
    begin
    writeln(execLog, 'pc = ', pc:1);
    printStack;  printEnv(env);  printDump;
    printSCounts
    end {printState};



    {****************************************}
    {*        processing object code        *}
    {****************************************}



procedure readCode;
    {read in instructions, resolve label references}
    var i,next,v,vn: integer;

        lab: array [1..maxLabels] of
        packed record
                   defined: boolean;
                   value:   pcRange
               end;

    begin
    reset(object); contCount := 0;
    for i := 1 to maxLabels do
        begin lab[i].defined := false;  lab[i].value := 0 end;
    while not eof(object) do begin
        if object^.kind = labelI then
            with lab[object^.pc] do begin
                next := contCount+1;   {define label to be next instr}
                assert(not defined);
                v := value;
                while v <> 0 do begin   {fix up prev instrs using label}
                    vn := control[v].pc;
                    control[v].pc := next;
                    v := vn
                    end;
                value := next;  defined := true
                end
        else begin
            contCount := contCount+1;
            control[contCount] := object^;
            with control[contCount] do
                if kind in pcInsts then begin
                    {convert a label number to a code address}
                    {if label undefined then link its references}
                    v := lab[pc].value;
                    if not lab[pc].defined then lab[pc].value := contCount;
                    pc := v     {put label value into instruction}
                    end
            end;
        get(object)
        end;
    for i := 1 to maxLabels do
        assert(lab[i].defined or (lab[i].value = 0))
    end {readCode};



procedure foldConst;
    {'fold' constant instruction streams into ldConst instructions}
    var i: integer;  s: struct;
    begin
    for i := 1 to contCount do
        with control[i] do begin
            if kind = ldIntI then s := consInt(int)
            else if kind = ldBotI then s := consBotStr(i)
            else if kind = ldNcloseI then s := consClosure(nil,pc,0)
            else s := nil;
            if s<>nil then
                begin sIncRefs(s);  kind := ldConstI;  str := s end
            end
    end {foldConst};



procedure elimJchains;
    {eliminate jump chains from the code.  essential for detection of tail-
      recursion because it replaces 'jump x;  x: return' with 'return'}
    var i,j,k,l: integer;  inst: instruction;
    begin
    for i := 1 to contCount do
        if control[i].kind = jumpI then begin
            j := i;
            while control[j].kind = jumpI do j := control[j].pc;
            if control[j].kind = returnI then inst := control[j]
            else begin
                inst.kind := jumpI;
                inst.pc := j;
                inst.level := 0
                end;
            k := i;
            while k<>j do begin
                l := control[k].pc;
                control[k] := inst;
                k := l
                end
            end
    end {elimJchains};



procedure printCode;
    {Print the object code onto the log file.}
    var i: integer;    static: instTable;
    begin
    heading;  clearInstTable(static);
    for i := 1 to contCount do
        with control[i] do begin
            static[kind] := static[kind] + 1;
            write(execLog, i:3, ':   ', instChars[kind]);
	    if kind in intInsts then      write(execLog, int:5)
	    else if kind in pcInsts then  write(execLog, pc:5, level:5)
	    else if kind = ldConstI then    printStruct(execLog,str);
            writeln(execLog)
            end;
    writeln(execLog);  writeln(execLog, 'static instruction counts');
    printInstTable(static,0)
    end {printCode};



procedure initCode;
    begin
    instChars[haltI]          := 'halt      ';
    instChars[returnI]        := 'return    ';
    instChars[ldBotI]         := 'ldBot     ';
    instChars[ldConstI]       := 'ldConst   ';
    instChars[ldPosI]         := 'ldPos     ';
    instChars[ldCloseI]       := 'ldClose   ';
    instChars[ldNcloseI]      := 'ldNclose  ';
    instChars[applyI]         := 'apply     ';
    instChars[applyIntI]      := 'applyInt  ';
    instChars[pairI]          := 'pair      ';
    instChars[injectI]        := 'inject    ';
    instChars[projectI]       := 'project   ';
    instChars[isI]            := 'is        ';
    instChars[alterI]         := 'alter     ';
    instChars[alterIntI]      := 'alterInt  ';
    instChars[leftI]          := 'left      ';
    instChars[rightI]         := 'right     ';
    instChars[notI]           := 'not       ';
    instChars[plusI]          := 'plus      ';
    instChars[minusI]         := 'minus     ';
    instChars[timesI]         := 'times     ';
    instChars[divI]           := 'div       ';
    instChars[modI]           := 'mod       ';
    instChars[eqI]            := 'eq        ';
    instChars[ltI]            := 'lt        ';
    instChars[jumpI]          := 'jump      ';
    instChars[fjumpI]         := 'fjump     ';
    instChars[callzI]         := 'callz     ';
    instChars[pushEnvI]       := 'pushEnv   ';
    instChars[popEnvI]        := 'popEnv    ';
    instChars[labelI]         := 'label     ';
    instChars[ldIntI]         := 'ldInt     '
    end {initCode};



function arrayLookup(idx: integer;  a: arrayTree): struct;
    {Find the array element a[idx] in the array tree}
    begin
    while (a <> nil) and (idx <> 1) do begin
        if odd(idx) then a := a^.right else a := a^.left;
	idx := idx div 2
	end;
    if a = nil then arrayLookup := nil else arrayLookup := a^.value
    end {arrayLookup};



function arrayEnter(value: struct;  idx: integer;  a: arrayTree): arrayTree;
    {Create the altered array [idx->value]a, but as an arrayTree}
    var l,r: arrayTree;  v: struct;
    begin
    if a = nil then begin  v := nil;  l := nil;  r := nil  end
    else begin  v := a^.value;  l := a^.left;  r := a^.right  end;
    if idx = 1 then 
        arrayEnter := consArrayTree(value,l,r)
    else if odd(idx) then
        arrayEnter := consArrayTree(v, l, arrayEnter(value, idx div 2, r))
    else 
        arrayEnter := consArrayTree(v, arrayEnter(value, idx div 2, l), r)
    end {arrayEnter};



function updateArray(idx: integer;  val,func: struct) : struct;
    var pos,neg: arrayTree;  baseFunc: struct;
    begin
    if func^.kind = arrayS then begin
        pos := func^.pos;  neg := func^.neg;  baseFunc := func^.baseFunc
	end
    else begin  pos := nil;  neg := nil;  baseFunc := func  end;
    if idx > 0 then
        updateArray := consArray(arrayEnter(val,idx,pos), neg, baseFunc)
    else
        updateArray := consArray(pos, arrayEnter(val, 1-idx, neg), baseFunc)
    end {updateArray};



procedure searchArray(func: struct;  var f,v: struct;  idx: integer);
    {apply 'f' to 'idx' returning 'v' -- look up 'idx' in 'f'.
      if not found, v=nil and f points past the array tree.}
    begin
    if func^.kind = arrayS then begin
        f := func^.baseFunc;
	if idx>0 then v := arrayLookup(idx, func^.pos)
	else v := arrayLookup(1-idx, func^.neg)
	end
    else begin  f := func;  v := nil  end
    end {searchArray};



procedure call(func,arg: struct);
    {apply the func to its arg}
    begin
    sIncRefs(arg);    {needed because of call in applyInt instruction}
    if func^.kind = closureS then begin
        if control[pc].kind <> returnI then pushD(env,pc);
        ae(env, consEnv(arg, func^.env));
        pc := func^.pc
        end
    else if func^.kind = botS then pushS(func)
    else assert(false);
    sDecRefs(arg)
    end {call};



    {****************************************}
    {*        machine input/output          *}
    {****************************************}



procedure readInput(var s: struct);
    {read a list of integers into s in the form: (mapping, count) }
    var map: struct;  val,len: integer;



    procedure skipBlanks;
        begin while not eof(input) and (input^=' ') do get(input)
        end;

    begin
    write(execLog, 'input = ');
    writeln(output, 'Type input data, followed by <eof>.');
    len := 0;   map := consBotStr(0);
    skipBlanks;
    while not eof(input) do begin
        len := len+1;
        read(input,val);  write(execLog, '  ', val:1);  skipBlanks;
        map := updateArray(len, consInt(val), map)
        end;
    s := consPair(map, consInt(len));
    writeln(execLog)
    end {readInput};



procedure printOutput(var f: text;  s: struct);
    {print the result 's' on the file 'f' as a list of integers}
    var val,map,dummy: struct;   i,len: integer;
    begin
    writeln(f, 'output = ');
    if s^.kind = botS then writeln(f, 'bottom')
    else begin
        assert(s^.kind = pairS);
        map := s^.left;    len := s^.right^.int;
        for i := 1 to len do begin
            searchArray(map,dummy,val,i);
            write(f, '  ');  printStruct(f,val)
            end;
        writeln(f)
        end
    end {printOutput};



procedure initialize;
    var x: struct;



    procedure initS;
        {initialize structure system:  reference counter, small integers, ...}
        var i: integer;
        begin
        sFree := nil;
        siFull := false;           {build table of small integers.}
        for i := 0 to maxSmallInt do
            begin  smallInt[i] := consInt(i);  sIncRefs(smallInt[i]) end;
        siFull := true;
        initStructCounts
        end {initS};


    procedure openFiles;
        var fn: fileName;  len: integer;



        procedure setExtension(var fn: fileName; ext: fileExt);
            begin
            fn[len+1] := '.';
            fn[len+2] := ext[1]; fn[len+3] := ext[2]; fn[len+4] := ext[3]
            end {setExtension};

        begin
        writeln(output, 'Stack Machine');
        write(output, 'Program: '); 
  	len := 0;   fn := '                    ';
	while not eoln(input) do 
	    begin  len := len+1;  read(input, fn[len])  end;
        setExtension(fn,'obj'); reset(object, fn);
        setExtension(fn, 'log'); rewrite(execLog, fn); logFileName := fn;
        end {openFiles};



    begin
    lastRuntime := clock;
    listLimit := 99999999;
    intInsts := [injectI,projectI,isI,applyIntI,alterIntI,ldPosI,ldIntI];
    pcInsts := [ldNcloseI,ldCloseI,jumpI,fjumpI,callzI,labelI];
    initS;  eFree := nil;
    aFree := nil;  aCount := 0;
    openFiles;
    initCode;  readCode;  foldConst;  elimJchains;  printCode;
    markTime;
    initStructCounts;
    clearInstTable(dynamicInst);
    heading;
    stackTop := 0;  env := nil;  pc := contCount;  dumpTop := 0;
    readInput(x);  pushS(x);
    end {initialize};


begin
initialize;
writeln(output, 'Starting execution.');
condition := running;
while condition = running do begin
    if sCount > listLimit then begin
        listLimit := 99999999;  printState
        end;
    ci := control[pc];  pc := pc+1;
    dynamicInst[ci.kind] := dynamicInst[ci.kind] + 1;
    case ci.kind of
        haltI: condition := halted;
        returnI: popD(env,pc);
        ldConstI: pushS(ci.str);
        ldPosI:
            begin
            envl := env;
            for i := 1 to ci.int do envl := envl^.next;
            pushS(envl^.str)
            end;
        ldCloseI:
            if control[pc].kind = applyI then begin
                {combine loadClosure and apply into 'call'}
                if control[pc+1].kind <> returnI then pushD(env,pc+1);
                envl := env;
                for i := 1 to ci.level do envl := envl^.next;
                popS(y);  ae(env, consEnv(y,envl));
                pc := ci.pc;
                sDecRefs(y)
                end
            else pushS(consClosure(env, ci.pc, ci.level));
        applyI:
            begin
            popS(func); popS(y);
            if y^.kind = intS then searchArray(func, f, z, y^.int)
            else begin
                z := nil;  f := func;
                while (z=nil) and (f^.kind = alterS) do begin
                    if equal(y, f^.arg) then z := f^.val;
                    f := f^.func
                    end
                end;
            if z=nil then call(f,y)
            else pushS(z);
            sDecRefs(func); sDecRefs(y)
            end;
        applyIntI:
            begin
            popS(func);
            searchArray(func, f, z, ci.int);
            if z=nil then call(f, consInt(ci.int))
            else pushS(z);
            sDecRefs(func)
            end;
        plusI:
            begin
            popS(y);  popS(x);
            if (x^.kind=intS) and (y^.kind=intS) then
                pushS(consInt(x^.int + y^.int))
            else abortM;
            sDecRefs(x);  sDecRefs(y)
            end;
        minusI:
            begin
            popS(y);  popS(x);
            if (x^.kind=intS) and (y^.kind=intS) then
                pushS(consInt(x^.int - y^.int))
            else abortM;
            sDecRefs(x);  sDecRefs(y)
            end;
        timesI:
            begin
            popS(y);  popS(x);
            if (x^.kind=intS) and (y^.kind=intS) then
                pushS(consInt(x^.int * y^.int))
            else abortM;
            sDecRefs(x);  sDecRefs(y)
            end;
        divI:
            begin
            popS(y);  popS(x);
            if (x^.kind=intS) and (y^.kind=intS) then
                pushS(consInt(x^.int div y^.int))
            else abortM;
            sDecRefs(x);  sDecRefs(y)
            end;
        modI:
            begin
            popS(y);  popS(x);
            if (x^.kind=intS) and (y^.kind=intS) then
                pushS(consInt(x^.int mod y^.int))
            else abortM;
            sDecRefs(x);  sDecRefs(y)
            end;
        eqI:
            begin
            popS(y);  popS(x);
            pushS(consInt(ord(equal(x,y))));
            sDecRefs(x);  sDecRefs(y)
            end;
        ltI:
            begin
            popS(y);  popS(x);
            if (x^.kind=intS) and (y^.kind=intS) then
                pushS(consInt(ord(x^.int < y^.int)))
            else abortM;
            sDecRefs(x);  sDecRefs(y)
            end;
        leftI:
            begin
            popS(x);
            if x^.kind = pairS then pushS(x^.left)
            else if x^.kind = botS then pushS(x)
            else abortM;
            sDecRefs(x)
            end;
        rightI:
            begin
            popS(x);
            if x^.kind = pairS then pushS(x^.right)
            else if x^.kind = botS then pushS(x)
            else abortM;
            sDecRefs(x)
            end;
        notI:
            begin
            popS(x);
            if x^.kind = intS then
                pushS(consInt(ord(x^.int = ord(false) )))
            else abortM;
            sDecRefs(x)
            end;
        pairI:
            begin
            popS(y); popS(x); pushS(consPair(x,y));
            sDecRefs(x);  sDecRefs(y)
            end;
        injectI:
            begin popS(x); pushS(consInject(ci.int, x)); sDecRefs(x) end;
        projectI:
            begin
            popS(x);
            if (x^.kind = injectS) and (ci.int = x^.si) then pushS(x^.str)
            else if x^.kind = botS then pushS(x)
            else abortM;
            sDecRefs(x)
            end;
        isI:
            begin
            popS(x);
            if x^.kind = injectS then
                pushS(consInt(ord(ci.int = x^.si)))
            else abortM;
            sDecRefs(x)
            end;
        alterI:
            begin
            popS(z);  popS(y);  popS(x);
            if x^.kind = botS then abortM
            else if x^.kind = intS then pushS(updateArray(x^.int, y, z))
            else pushS(consAlter(x,y,z));
            sDecRefs(x);  sDecRefs(y);  sDecRefs(z)
            end;
        alterIntI:
            begin
            popS(z);  popS(y);
            pushS(updateArray(ci.int, y, z));
            sDecRefs(y);  sDecRefs(z)
            end;
        jumpI: pc := ci.pc;
        fjumpI:
            begin
            popS(x);
            if x^.kind = intS then
                begin if x^.int = ord(false) then pc := ci.pc
                end
            else abortM;
            sDecRefs(x)
            end;
        callzI:  begin  pushD(env,pc); pc := ci.pc end;
        pushEnvI:
            begin popS(x);  ae(env, consEnv(x,env)); sDecRefs(x) end;
        popEnvI: ae(env, env^.next)
        end {case}
    end;
writeln(execLog);
writeln(output);
if condition = aborted then begin
    writeln(execLog, 'machine aborted');
    writeln(output, 'machine aborted')
    end
else begin
    printOutput(output,  stack[stackTop]);
    printOutput(execLog, stack[stackTop])
    end;
markTime;
printState;
writeln(execLog);  writeln(execLog, 'dynamic instruction counts');
printInstTable(dynamicInst, lastRtPrinted)
end.
