(*  Title: 	tctical
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Tacticals
Derived rules and other operations on theorems and theories

BEST_FIRST should perhaps return the sequence of all solutions, 
	not stop at the first set of solutions
*)

infix RS RSN;
infix RES RESN;
infix THEN ORELSE APPEND THEN' ORELSE' APPEND';

signature TACTICAL =
sig
  structure Thm : THM
  local open Thm
  in
    datatype tactic = Tactic of thm -> thm Sequence.seq
    val all_tac: tactic
    val ALLGOALS: (int -> tactic) -> tactic   
    val APPEND: tactic * tactic -> tactic
    val APPEND': ('a -> tactic) * ('a -> tactic) -> 'a -> tactic
    val assume_ax: theory -> string -> thm
    val BEST_FIRST: (thm -> bool) * (thm -> int) -> tactic -> tactic
    val BREADTH_FIRST: (thm -> bool) -> tactic -> tactic
    val COND: (thm -> bool) -> tactic -> tactic -> tactic   
    val DEPTH_FIRST: (thm -> bool) -> tactic -> tactic
    val DEPTH_SOLVE_1: tactic -> tactic
    val DETERM: tactic -> tactic
    val EVERY: tactic list -> tactic   
    val FIRST: tactic list -> tactic   
    val FIRSTGOAL: (int -> tactic) -> tactic
    val forall_intr_list: Sign.cterm list -> thm -> thm
    val forall_intr_frees: thm -> thm
    val forall_elim_var: int -> thm -> thm
    val forall_elim_vars: int -> thm -> thm
    val has_fewer_prems: int -> thm -> bool   
    val implies_intr_list: Sign.cterm list -> thm -> thm
    val MAP_EVERY: ('a -> tactic) -> 'a list -> tactic
    val MAP_FIRST: ('a -> tactic) -> 'a list -> tactic
    val no_tac: tactic
    val ORELSE: tactic * tactic -> tactic
    val ORELSE': ('a -> tactic) * ('a -> tactic) -> 'a -> tactic
    val read_instantiate: theory -> (string*string*typ)list -> thm -> thm
    val REPEAT1: tactic -> tactic
    val REPEAT: tactic -> tactic
    val REPEAT_FIRST: (int -> tactic) -> tactic
    val REPEAT_SOME: (int -> tactic) -> tactic
    val RES: thm * thm -> thm
    val RESN: thm * (int * thm list) -> thm
    val reslist: thm list * int * thm -> thm list
    val RS: thm * thm -> thm
    val RSN: thm list * (int * thm) -> thm
    val SELECT_GOAL: tactic -> int -> tactic
    val SOMEGOAL: (int -> tactic) -> tactic   
    val standard: thm -> thm
    val STATE: (thm -> tactic) -> tactic
    val SUBGOAL: ((term*int) -> tactic) -> int -> tactic
    val tapply: tactic * thm -> thm Sequence.seq
    val THEN: tactic * tactic -> tactic
    val THEN': ('a -> tactic) * ('a -> tactic) -> 'a -> tactic
    val TRY: tactic -> tactic
    val TRYALL: (int -> tactic) -> tactic   
    val zero_var_indexes: thm -> thm
  end
end;


functor TacticalFun (structure Logic: LOGIC and Thm: THM) : TACTICAL = 
struct
structure Thm = Thm;
structure Sequence = Thm.Sequence;
structure Sign = Thm.Sign;
local open Thm
in


(**** More derived rules and operations on theorems ****)

(*Generalization over a list of variables, IGNORING bad ones*)
fun forall_intr_list [] th = th
  | forall_intr_list (y::ys) th =
	let val gth = forall_intr_list ys th
	in  forall_intr y gth   handle THM _ =>  gth  end;


(*Generalization over all suitable Free variables*)
fun forall_intr_frees th =
    let val {prop,sign,...} = rep_thm th
    in  forall_intr_list
         (map (Sign.cterm_of sign) 
	      (sort Logic.atless (Logic.add_term_frees (prop,[])))) th
    end;


(*Replace outermost quantified variable by Var of given index.
    Could clash with Vars already present.*)
fun forall_elim_var i th = 
    let val {prop,sign,...} = rep_thm th
    in case prop of
	  Const("all",_) $ Abs(a,T,_) =>
	      forall_elim (Sign.cterm_of sign (Var((a,i), T)))  th
	| _ => raise THM("forall_elim_var", i, [th])
    end;


(*Repeat forall_elim_var until all outer quantifiers are removed*)
fun forall_elim_vars i th = 
    forall_elim_vars i (forall_elim_var i th)
	handle THM _ => th;


(* maps [A1,...,An], B  to   A1==>...==>An==>B  *)
fun implies_intr_list [] thB = thB
  | implies_intr_list (cA::cAs) thB =
	implies_intr cA (implies_intr_list cAs thB);      


(*maps  (bs,v)  to   v'::bs    this reverses the identifiers bs*)
fun add_new_id (bs, ((a,_),_)) : string list =  variant bs a  ::  bs;


(*Pair variables with their new names, restoring types*)
fun varpairs sign ([],[]) = []
  | varpairs sign ((var as Var(_,T)) :: vars, b::bs) =
	(Sign.cterm_of sign var, Sign.cterm_of sign (Var((b,0),T))) :: 
        varpairs sign (vars,bs)
  | varpairs sign _ = raise TERM_ERROR("varpairs", []);


(*Reset Var indexes to zero, renaming to preserve distinctness*)
fun zero_var_indexes th = 
    let val {prop,sign,...} = rep_thm th;
        val vars = Logic.add_term_vars(prop,[])
        val bs = itlist_left add_new_id ([], map dest_Var vars)
    in  instantiate (varpairs sign (vars, rev bs)) th  end;


(*Standard form of object-rule: no hypotheses, Frees, or outer quantifiers;
    all generality expressed by Vars having index 0.*)
fun standard th =
    let val {maxidx,...} = rep_thm th
    in  zero_var_indexes (forall_elim_vars(maxidx+1) 
                         (forall_intr_frees(implies_intr_hyps th)))
    end;


(*Assume a new formula, read following the same conventions as axioms. 
  Generalizes over Free variables,
  creates the assumption, and then strips quantifiers.
  Example is [| ALL x:?A. ?P(x) |] ==> [| ?P(?a) |]
	     [ !(A,P,a)[| ALL x:A. P(x) |] ==> [| P(a) |] ]    *)
fun assume_ax thy sP =
    let val sign = sign_of thy;
	val prop = Logic.close_form (Sign.term_of (Sign.read_cterm sign
			 (sP, Aprop)))
    in forall_elim_vars 0 (assume (Sign.cterm_of sign prop))  end;


(*Resolution: exactly one resolvent must be produced.  NEW*) 
fun ths RSN (i,thb) = 
  case Sequence.chop (2, resolution (thb,i,ths)) of
      ([th],_) => th
    |      _   => raise THM("RSN", i, thb::ths);


(*resolution: P==>Q, Q==>R gives P==>R.  (NEW: args reversed from RES!) *)
fun tha RS thb = [tha] RSN (1,thb);


(*Resolution: exactly one resolvent must be produced.  OBSOLETE FORM*) 
fun th2 RESN (i,ths) = 
  case Sequence.chop (2, resolution (th2,i,ths)) of
      ([th],_) => th
    |      _   => raise THM("RESN", i, th2::ths);


(*resolution of first premise with a single theorem.  OBSOLETE FORM*)
fun th2 RES th1 = th2 RESN (1,[th1]);


(*List of all resolvents of ths with thb at i*)
fun reslist (ths, i,thb) = 
  Sequence.list_of_s (resolution (thb,i,ths));


(*Instantiate rule, reading terms from strings under theory "thy".
  Resulting rule will belong to union of the two theories. *)
fun read_instantiate thy sinsts th =
  instantiate (Sign.read_insts (sign_of thy) sinsts) th;


(**** Tactics ****)


(*A tactic maps a proof tree to a sequence of proof trees:
    if length of sequence = 0 then the tactic does not apply;
    if length > 1 then backtracking on the alternatives can occur.*)

datatype tactic = Tactic of thm -> thm Sequence.seq;


fun tapply(Tactic tf, state) = tf (state);


(*** LCF-style tacticals ***)

(*the tactical THEN performs one tactic followed by another*)
fun (Tactic tf1)  THEN  (Tactic tf2) = 
  Tactic (fn state => Sequence.flats (Sequence.maps tf2 (tf1 state)));


(*The tactical ORELSE uses the first tactic that returns a nonempty sequence.
  Like in LCF, ORELSE commits to either tac1 or tac2 immediately.
  Does not backtrack to tac2 if tac1 was initially chosen. *)
fun (Tactic tf1)  ORELSE  (Tactic tf2) = 
  Tactic (fn state =>  
    case Sequence.pull(tf1 state) of
	None       => tf2 state
      | sequencecell => Sequence.seqof(fn()=> sequencecell));


(*The tactical APPEND combines the results of two tactics.
  Like ORELSE, but allows backtracking on both tac1 and tac2.
  The tactic tac2 is not applied until needed.*)
fun (Tactic tf1)  APPEND  (Tactic tf2) = 
  Tactic (fn state =>  Sequence.append(tf1 state,
                          Sequence.seqof(fn()=> Sequence.pull (tf2 state))));


(*Versions for combining tactic-valued functions, as in
     SOMEGOAL (resolve_tac rls THEN' assume_tac) *)
fun tac1 THEN' tac2 = fn x => tac1 x THEN tac2 x;
fun tac1 ORELSE' tac2 = fn x => tac1 x ORELSE tac2 x;
fun tac1 APPEND' tac2 = fn x => tac1 x APPEND tac2 x;

(*passes all proofs through unchanged;  identity of THEN*)
val all_tac = Tactic (fn state => Sequence.single state);

(*passes no proofs through;  identity of ORELSE and APPEND*)
val no_tac  = Tactic (fn state => Sequence.null);


(*Make a tactic deterministic by chopping the tail of the proof sequence*)
fun DETERM (Tactic tf) = Tactic (fn state => 
      case Sequence.pull (tf state) of
	      None => Sequence.null
            | Some(x,_) => Sequence.cons(x, Sequence.null));


(*Conditional tactical: testfun controls which tactic to use next.
  Beware: due to eager evaluation, both thentac and elsetac are evaluated.*)
fun COND testfun (Tactic thenf) (Tactic elsef) = Tactic (fn prf =>
    if testfun prf then  thenf prf   else  elsef prf);

(*Do the tactic or else do nothing*)
fun TRY tac = tac ORELSE all_tac;


(*Performs no backtracking: quits when it gets stuck
  The abstraction over state is essential: else it would always loop!!*)
fun REPEAT tac = Tactic (fn state =>
  tapply((tac THEN REPEAT tac) ORELSE all_tac, state));

(*Repeat 1 or more times*)
fun REPEAT1 tac = Tactic (fn state => 
  tapply(tac THEN (REPEAT1 tac ORELSE all_tac), state));


(* EVERY [tac1,...,tacn]   equals    tac1 THEN ... THEN tacn   *)
fun EVERY tacs = itlist_right (op THEN) (tacs, all_tac);


(* FIRST [tac1,...,tacn]   equals    tac1 ORELSE ... ORELSE tacn   *)
fun FIRST tacs = itlist_right (op ORELSE) (tacs, no_tac);

(*Tacticals that apply tacf to all elements of xs*)
fun MAP_EVERY tacf xs = EVERY (map tacf xs);
fun MAP_FIRST tacf xs = FIRST (map tacf xs);


(*** Tacticals based on subgoal numbering ***)


(*For n subgoals, performs tf(n) THEN ... THEN tf(1) 
  Essential to work backwards since tf(i) may add/delete subgoals at i. *)
fun ALLGOALS tf = 
  let fun tac 0 = all_tac
	| tac n = tf(n) THEN tac(n-1)
  in  Tactic(fn state => tapply(tac(length(prems_of state)), state))  end;


(*For n subgoals, performs tf(n) ORELSE ... ORELSE tf(1)  *)
fun SOMEGOAL tf = 
  let fun tac 0 = no_tac
	| tac n = tf(n) ORELSE tac(n-1)
  in  Tactic(fn state => tapply(tac(length(prems_of state)), state))  end;


(*For n subgoals, performs tf(1) ORELSE ... ORELSE tf(n).
  More appropriate than SOMEGOAL in some cases.*)
fun FIRSTGOAL tf = 
  let fun tac (i,n) = if i>n then no_tac else  tf(i) ORELSE tac (i+1,n)
  in  Tactic(fn state => tapply(tac(1, length(prems_of state)), state))  end;

(*Repeatedly solve some using tf. *)
fun REPEAT_SOME tf = REPEAT1 (SOMEGOAL (REPEAT1 o tf));

(*Repeatedly solve the first possible subgoal using tf. *)
fun REPEAT_FIRST tf = REPEAT1 (FIRSTGOAL (REPEAT1 o tf));

(*For n subgoals, tries to apply tf to n,...1  *)
fun TRYALL tf = ALLGOALS (TRY o tf);


(*Make a tactic for subgoal i, if there is one.  *)
fun SUBGOAL goalfun i = Tactic(fn state =>
  case nth_tail(i-1, prems_of state) of
      [] => Sequence.null
    | prem::_ => tapply(goalfun (prem,i), state));


(*Makes a tactic from one that uses the components of the state.*)
fun STATE tacfun = Tactic (fn state => tapply(tacfun state, state));



(*Tactical for restricting the effect of a tactic to subgoal i.
  Works by making a new state from subgoal i, applying tf to it, and
  composing the resulting metathm with the original state.
  The "main goal" of the new state will not be atomic, some tactics may fail!
  DOES NOT work if tactic affects the main goal other than by instantiation.*)
fun SELECT_GOAL (Tactic tf) i = Tactic (fn state => 
  case nth_tail(i-1, prems_of state) of
      [] => Sequence.null
    | prem::_ => 
	let val st0 = trivial (Sign.cterm_of (#sign(rep_thm state)) prem);
	    val nasms = length(Logic.strip_imp_prems prem);
	    fun next st = bicompose(state, i) (false, st,
				length(prems_of st) - nasms)
	in  Sequence.flats (Sequence.maps next (tf st0))
	end);


(*** Search tacticals ***)

(*Repeats again and again until "satpred" reports proof tree as satisfied*)
fun DEPTH_FIRST satpred tac = STATE (fn state => 
   if satpred state  then  all_tac 
                     else  tac THEN DEPTH_FIRST satpred tac);


(*Predicate: Does the rule have fewer than n premises?*)
fun has_fewer_prems n rule =  (length (prems_of rule) < n);


(*Tactical to reduce the number of premises by 1.
  If no subgoals then it must fail! *)
fun DEPTH_SOLVE_1 tac = STATE
 (fn state => 
    (case length(prems_of state) of
	0 => no_tac
      | n => DEPTH_FIRST (has_fewer_prems n) tac));


(*Two theorems are taken as equal if their propositions are a-convertible*)
fun eq_thm (th1,th2) = #prop(rep_thm th1) aconv #prop(rep_thm th2); 

(*Insertion into priority queue of states *)
fun insert (nth: int*thm, []) = [nth]
  | insert ((m,th), (n,th')::nths) = 
      if  n<m then (n,th') :: insert ((m,th), nths)
      else if  n=m andalso eq_thm(th,th')
              then (n,th')::nths
              else (m,th)::(n,th')::nths;

(*Best-first search for a state that satisfies satpred (incl initial state)
  Function costf estimates cost of problem remaining (smaller means better) *)
fun BEST_FIRST (satpred,costf) (Tactic tf) = 
  let fun paircost th = (costf th, th);
      fun bfs (news,nprfs) =
	   (case  partition satpred news  of
		([],nonsats) => next(itlist_right insert
					(map paircost nonsats, nprfs)) 
	      | (sats,_)  => sats)
      and next [] = []
        | next ((n,prf)::nprfs) =
	    (prs("cost=" ^ string_of_int n ^ "\n");
	       bfs (Sequence.list_of_s (tf prf), nprfs))
  in Tactic (fn state => Sequence.s_of_list (bfs ([state],[]))) end;


(*Breadth-first search to satisfy satpred (including initial state) *)
fun BREADTH_FIRST satpred (Tactic tf) = 
  let val tacf = Sequence.list_of_s o tf;
      fun bfs prfs =
	 (case  partition satpred prfs  of
	      ([],[]) => []
	    | ([],nonsats) => 
		  (prs("breadth=" ^ string_of_int(length nonsats) ^ "\n");
		   bfs (flat (map tacf nonsats)))
	    | (sats,_)  => sats)
  in Tactic (fn state => Sequence.s_of_list (bfs [state])) end;

end;
end;
