(*  Title: 	term.ML
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   Cambridge University 1989
*)


(*Typed lambda-calculus: types, terms, and basic operations*)


infixr 5 -->;
datatype typ = Ground of string
            |  Poly of string
            |  op --> of typ * typ;


(*terms.  Bound variables are indicated by depth number.
  Free variables, (scheme) variables and constants have names.
  An term is "closed" if there every bound variable of level "lev"
  is enclosed by at least "lev" abstractions. 

  It is possible to create meaningless terms containing loose bound vars
  or type mismatches.  But such terms are not allowed in rules. *)


(*Indexnames can be quickly renamed by adding an offset to the integer part,
  for resolution.*)
type indexname = string*int;


infix 9 $;  (*application binds tightly!*)
datatype term = 
    Const of string * typ
  | Free  of string * typ 
  | Var   of indexname * typ
  | Bound of int
  | Abs   of string*typ*term
  | op $  of term*term;


(*For errors involving type mismatches*)
exception TYPE of string * typ list * term list;

(*For system errors involving terms*)
exception TERM_ERROR of string * term list;


(*Note variable naming conventions!
    a,b,c: string
    f,g,h: functions (including terms of function type)
    i,j,m,n: int
    t,u: term
    v,w: indexnames
    x,y: any
    A,B,C: term (denoting formulae)
    T,U: typ
*)


(*Destructors*)

fun dest_Const (Const x) =  x
  | dest_Const t = raise TERM_ERROR("dest_Const", [t]);

fun dest_Free (Free x) =  x
  | dest_Free t = raise TERM_ERROR("dest_Free", [t]);

fun dest_Var (Var x) =  x
  | dest_Var t = raise TERM_ERROR("dest_Var", [t]);


(* maps  [T1,...,Tn]--->T  to the list  [T1,T2,...,Tn]*)
fun binder_types (Ground _)   =  []
  | binder_types (Poly _)   =  []
  | binder_types (T-->U) = T :: binder_types U;

(* maps  [T1,...,Tn]--->T  to the string for the Ground type T*)
fun body_type (Ground a) = a : string
  | body_type (Poly a) = raise TYPE("body_type: type variable "^a, [], [])
  | body_type  (_  -->  T)  = body_type T;

(* maps  [T1,...,Tn]--->T  to   ([T1,T2,...,Tn], T)  *)
fun strip_type T : typ list * string =
  (binder_types T, body_type T);


(*handy for multiple args: [T1,...,Tn]--->T  gives  T1-->(T2--> ... -->T)*)
infixr --->;
val op ---> = itlist_right (op -->);


fun is_var (Var _) = true
  | is_var _ = false;


(*Compute the type of the term, checking that combinations are well-typed
  Ts = [T0,T1,...] holds types of bound variables 0, 1, ...*)
fun type_of1 (Ts, Const (_,T)) = T
  | type_of1 (Ts, Free  (_,T)) = T
  | type_of1 (Ts, Bound i) = (nth_elem (i,Ts)  
  	handle LIST _ => raise TYPE("type_of: bound variable", [], [Bound i]))
  | type_of1 (Ts, Var (_,T)) = T
  | type_of1 (Ts, Abs (_,T,body)) = T --> type_of1(T::Ts, body)
  | type_of1 (Ts, f$u) = 
      let val U = type_of1(Ts,u)
          and T = type_of1(Ts,f)
      in case T of
	    T1-->T2 =>
	      if T1=U then T2  else raise TYPE
	         ("type_of: type mismatch in application", [T1,U], [f$u])
	  | _ => raise TYPE ("type_of: Rator must have function type",
	                        [T,U], [f$u])
      end;


fun type_of t : typ = type_of1 ([],t);



(* maps  (x1,...,xn)t   to   t  *)
fun strip_abs_body (Abs(_,_,t))  =  strip_abs_body t  
  | strip_abs_body u  =  u;


(* maps  (x1,...,xn)t   to   [x1, ..., xn]  *)
fun strip_abs_vars (Abs(a,T,t))  =  (a,T) :: strip_abs_vars t 
  | strip_abs_vars u  =  [] : (string*typ) list;


(* maps   (f, [t1,...,tn])  to  f(t1,...,tn) *)
val list_comb : term * term list -> term = itlist_left (op $);


(* maps   f(t1,...,tn)  to  (f, [t1,...,tn]) *)
fun strip_comb u : term * term list = 
    let fun stripc (f$t, ts) = stripc (f, t::ts)
        |   stripc  x =  x 
    in  stripc(u,[])  end;


(* maps   f(t1,...,tn)  to  f , which is never a combination *)
fun head_of (f$t) = head_of f
  | head_of u = u;


(*Number of atoms and abstractions in a term*)
fun size_of_term (Abs (_,_,body)) = 1 + size_of_term body
  | size_of_term (f$t) = size_of_term f  +  size_of_term t
  | size_of_term _ = 1;

 
(*Connectives of higher order logic*)

val Aprop : typ = Ground "prop";

val implies = Const("==>", Aprop-->Aprop-->Aprop);

fun all T = Const("all", (T-->Aprop)-->Aprop);

fun equals T = Const("==", T-->T-->Aprop);

(*Create a type constraint -- application of typed identity function*)
fun constrain T = Const(".constrain", T-->T);


(* maps  !(x1,...,xn)t   to   t  *)
fun strip_all_body (Const("all",_)$Abs(_,_,t))  =  strip_all_body t  
  | strip_all_body t  =  t;


(* maps  !(x1,...,xn)t   to   [x1, ..., xn]  *)
fun strip_all_vars (Const("all",_)$Abs(a,T,t))  =
		(a,T) :: strip_all_vars t 
  | strip_all_vars t  =  [] : (string*typ) list;



(*increments a term's non-local bound variables
  required when moving a term within abstractions
     inc is  increment for bound variables
     lev is  level at which a bound variable is considered 'loose'*)
fun incr_bv (inc, lev, Bound i) = if i>=lev then Bound(i+inc) else Bound i 
  | incr_bv (inc, lev, Abs(a,T,body)) =
	Abs(a, T, incr_bv(inc,lev+1,body))
  | incr_bv (inc, lev, f$t) = 
      incr_bv(inc,lev,f) $ incr_bv(inc,lev,t)
  | incr_bv (inc, lev, u) = u;

fun incr_boundvars  0  t = t
  | incr_boundvars inc t = incr_bv(inc,0,t);


(*Accumulate all 'loose' bound vars referring to level 'lev' or beyond.
   (Bound 0) is loose at level 0 *)
fun add_loose_bnos (Bound i, lev, js) = 
	if i<lev then js  else  (i-lev) :: js
  | add_loose_bnos (Abs (_,_,t), lev, js) = add_loose_bnos (t, lev+1, js)
  | add_loose_bnos (f$t, lev, js) =
	add_loose_bnos (f, lev, add_loose_bnos (t, lev, js)) 
  | add_loose_bnos (_, _, js) = js;

fun loose_bnos t = add_loose_bnos (t, 0, []);


(*Substitutes arguments for loose bound variables
  Beta-reduction of arg(n-1)...arg0 into t replacing (Bound i) with (argi).
  Note that for ((x,y)c)(a,b), the bound vars in c are x=1 and y=0
	and the appropriate call is  subst_bounds([b,a], c) .
  Loose bound variables >=n are reduced by "n" to
     compensate for the disappearance of lambdas.
  optimization:  DO NOTHING IN CASE OF [B.0]?  [B.0, B.1]???
*)
fun subst_bounds (args: term list, t) : term = 
  let val n = length args;
      fun subst (t as Bound i, lev) =
 	    if i<lev then  t    (*var is locally bound*)
	    else  (case (nth_tail (i-lev,args)) of
		  []     => Bound(i-n)  (*loose: change it*)
	        | arg::_ => incr_boundvars lev arg)
	| subst (Abs(a,T,body), lev) = Abs(a, T,  subst(body,lev+1))
	| subst (f$t, lev) =  subst(f,lev)  $  subst(t,lev)
	| subst (t,lev) = t
  in   case args of [] => t  | _ => subst (t,0)  end;


(*beta-reduce if possible, else form application*)
fun betapply (Abs(_,_,t), u) = subst_bounds([u],t)
  | betapply (f,u) = f$u;


(*Tests whether 2 terms are alpha-convertible and have same type.
  Note that constants and Vars may have more than one type.*)
infix aconv;
fun Const(a,T) aconv Const(b,U) = a=b  andalso  T=U
  | Free(a,T) aconv Free(b,U) = a=b  andalso  T=U
  | Var(v,T) aconv Var(w,U) =   v=w  andalso  T=U
  | Bound i aconv Bound j  =   i=j
  | Abs(_,T,t) aconv Abs(_,U,u) = t aconv u  andalso  T=U
  | f$t aconv g$u = (f aconv g) andalso (t aconv u)
  | _ aconv _  =  false;


(*A fast unification filter: true unless the two terms cannot be unified. 
  Terms must be NORMAL.  Treats all Vars as distinct, ignores bound vars. *)
fun could_unify (t,u) =
  let fun matchrands (f$t, g$u) = could_unify(t,u) andalso  matchrands(f,g)
	| matchrands _ = true
  in case (head_of t , head_of u) of
	(_, Var _) => true
      | (Var _, _) => true
      | (Const(a,_), Const(b,_)) =>  a=b andalso matchrands(t,u)
      | (Free(a,_), Free(b,_)) =>  a=b andalso matchrands(t,u)
      | (Bound i, Bound j) =>  i=j andalso matchrands(t,u)
      | (Abs _, _) =>  true   (*because of possible eta equality*)
      | (_, Abs _) =>  true
      | _ => false
  end;


(*Operations on indexnames and lists of them*)

fun string_of_xname (a,0)   =  a
  | string_of_xname (a,idx) =  a ^ string_of_int idx;

(*a total, irreflexive ordering on index names*)
fun xless ((a,i), (b,j): indexname) = i<j  orelse  (i=j andalso a<b);


(*Operations on term lists*)

fun tassoc ([], key: term) = None
  | tassoc ((keyi,xi)::pairs, key) =
      if  key aconv keyi  then Some xi  else tassoc (pairs,key);


(*membership test for term lists*)
infix tmem;
fun t tmem []  =  false  
  | t tmem (u::us)  =  (t aconv u) orelse (t tmem us);


(*Returns the tail beginning with the first repeated element, or []. *)
fun t_findrep [] = []
  | t_findrep (t::ts) = if  t tmem ts  then  t::ts   else   t_findrep ts;


fun tdistinct1 (seen, []) = rev seen
  | tdistinct1 (seen, t::ts) =
      if t tmem seen then tdistinct1 (seen, ts)
      		      else tdistinct1 (t::seen, ts);

(*return a list of distinct terms*)
fun tdistinct ts = tdistinct1([],ts);



(*Abstraction of the term "body" over its occurrences of v, 
    which must contain no loose bound variables.
  The resulting term is ready to become the body of an Abs.*)
fun abstract_over (v,body) =
  let fun abst (lev,u) = if (v aconv u) then (Bound lev) else
      (case u of
          Abs(a,T,t) => Abs(a, T, abst(lev+1, t))
	| f$rand => abst(lev,f) $ abst(lev,rand)
	| _ => u)
  in  abst(0,body)  end;


(*Form an abstraction over a free variable.*)
fun absfree (a,T,body) = Abs(a, T, abstract_over (Free(a,T), body));


(*Abstraction over a list of free variables*)
fun list_abs_free ([ ] ,     t) = t
  | list_abs_free ((a,T)::vars, t) = 
      absfree(a, T, list_abs_free(vars,t));


(*Quantification over a list of free variables*)
fun list_all_free ([], t: term) = t
  | list_all_free ((a,T)::vars, t) = 
        (all T) $ (absfree(a, T, list_all_free(vars,t)));


(*Replace the ATOMIC term ti by ui;    instl = [(t1,u1), ..., (tn,un)]. 
  A simultaneous substitution:  [ (a,b), (b,a) ] swaps a and b.  *)
fun subst_atomic [] t = t : term
  | subst_atomic (instl: (term*term) list) t =
      let fun subst (Abs(a,T,body)) = Abs(a, T, subst body)
	    | subst (f$t') = subst f $ subst t'
	    | subst t = (case tassoc(instl,t) of
		           Some u => u  |  None => t)
      in  subst t  end;



(*Computing the maximum index of a term*)
fun maxidx_of_term (Const _) = ~1
  | maxidx_of_term (Bound _) = ~1
  | maxidx_of_term (Free _) = ~1 
  | maxidx_of_term (Var ((_,i), _)) = i
  | maxidx_of_term (Abs (_,_,body)) = maxidx_of_term body
  | maxidx_of_term (f$t) =
	max [maxidx_of_term f,  maxidx_of_term t];


exception TERM_MATCH;

(*First-order matching;  see V86/match.ML for higher-order matching
  Calling term_match(var-term list, pattern, object) extends the var-term list.
  The pattern and object may have variables in common without problems:
    instantiation does not affect object, so matching ?a with ?a+1 works.
  A Const does not match a Free of the same name!  Don't be fooled! 
  Does not notice eta-equality, thus f does not match %(x)f(x) *)
fun term_match (vts, v as Var _, t)  =
      (case loose_bnos t of  _::_ =>  raise TERM_MATCH  
	| [] =>
	  (case tassoc(vts,v) of
	      None => (v,t)::vts
	    | Some u => if  t aconv u then vts  else  raise TERM_MATCH))
  | term_match (vts, Free (a,T), Free (b,U)) =
	if  a=b  andalso  T=U  then  vts  else raise TERM_MATCH
  | term_match (vts, Const (a,T), Const (b,U))  =
	if  a=b  andalso  T=U  then  vts  else raise TERM_MATCH
  | term_match (vts, Bound i, Bound j)  =   
	if  i=j  then  vts  else raise TERM_MATCH
  | term_match (vts, Abs(_,_,t), Abs(_,_,u))  =
	term_match (vts,t,u)
  | term_match (vts, f$t, g$u) =
	term_match (term_match (vts,f,g), t, u)
  | term_match _  =  raise TERM_MATCH;



(**** Syntax-related declarations ****)


(*Dummy type for parsing.  Will be replaced during type inference. *)
val Adummy = Ground"dummy";


(*parsing errors*)
exception PARSERR of string;

(*Report lexical error*)
fun lexerr (msg,cs) = raise PARSERR (msg ^ ".  Location: " ^ implode cs);
      


(*A string of letters or ' _ but no digits!
  Identifiers may not contain digits because trailing digits indicate
    an offset to a variable or param name*)
fun scan_ident []  =  lexerr ("end of input; letter expected", [])
  | scan_ident(c::cs) =  
      if  is_letter c  then
        let val (ds,tail) = take_prefix is_quasi_letter cs
	in  (implode(c::ds), tail)  end
      else  lexerr ("letter expected", c::cs);

(*scan a numeral of the given radix, normally 10*)
fun scan_radixint (radix: int, cs) : int * string list =
  let val zero = ord"0";
      val limit = chr (zero+radix);
      fun scan (num,[]) = (num,[])
	| scan (num, c::cs) =
	      if  "0"<=c  andalso  c<limit 
	      then scan(radix*num + ord c - zero, cs)
	      else (num, c::cs)
  in  scan(0,cs)  end;


(*Scan the offset of a Var, if present: default is zero. *)
fun scan_offset cs = case cs of
    (c::_) => if is_digit c then  scan_radixint(10,cs)  else  (0,cs)
  | []     => (0,cs);


(*varname ::=   identifier  |  identifier number*)
fun scan_varname cs : (string*int) * string list =
  let val (a, ds) = scan_ident cs;
      val (i, es) = scan_offset ds
  in  ((a,i), es)  end;


(*characters allowed in symbolic identifiers, reserved are ?()[]{}, *)
val is_special : string -> bool = fn
     "!" => true  |  "#" => true
  |  "$" => true  |  "%" => true
  |  "&" => true  |  "*" => true
  |  "+" => true  |  "-" => true
  |  "<" => true  |  ">" => true
  |  "/" => true  |  "\\"=> true
  |  "^" => true  |  ":" => true
  |  "." => true  |  ";" => true
  |  "~" => true  |  "@" => true
  |  "=" => true  |  "|" => true
  |  "`" => true  |  _  => false;


(*predefined 1-character delimiters: do not stick together: (( is two symbols*)
val is_delimchar : string -> bool = fn
     "[" => true  |  "]" => true
  |  "(" => true  |  ")" => true
  |  "{" => true  |  "}" => true
  |  "," => true  |  _  => false;



(*** Printing ***)

(*suppresses brackets in right nesting of -->*)
fun string_of_type (Ground a) = a 
  | string_of_type (Poly a) = "'" ^ a
  | string_of_type ((T1-->T2)-->U) = 
      "(" ^ string_of_type(T1-->T2) ^ ")-->" ^ string_of_type U
  | string_of_type (T-->U) = string_of_type T ^ "-->" ^ string_of_type U;


(*take variant of the name "a": maps  a  to  a' not present in names "bs"*)
fun variant bs a : string =
  let fun vary a = if (a mem bs) then  vary (bump_string a)  else  a
  in  vary (if a="" then "u" else a)  end;


(*Accumulates the names in the term, suppressing duplicates.
  Includes Frees and Consts.  For choosing unambiguous bound var names.*)
fun add_term_names (Const(a,_), bs) = a ins bs
  | add_term_names (Free(a,_), bs) = a ins bs
  | add_term_names (Abs (_,_,body), bs) = add_term_names(body,bs)
  | add_term_names (f$u, bs) =
	 add_term_names (f, add_term_names(u, bs))
  | add_term_names (_, bs) = bs;


(*Given an abstraction over P, replaces the bound variable by a Free variable
  having a unique name. *)
fun variant_abs (a,T,P) =
  let val b = variant (add_term_names(P,[])) a
  in  (b,  subst_bounds ([Free(b,T)], P))  end;


(*space_implode "..." (explode "hello");  gives  "h...e...l...l...o" *)
fun space_implode a [] = ""
  | space_implode a [b] = b
  | space_implode a (b::bs) = b ^ a ^ space_implode a bs;
