(*  Title: 	goals
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Goal stack package.

An "autotactic" feature (like Nuprl's) was tried but it was just a headache.
To apply a special tactic each time, define your own version of "by":
  fun myby tac = tac THEN my_autotac
At least this way you will remember which autotactic you are using!
*)


signature GOALS =
sig
  structure Tactic : TACTIC
  type gstack
  val back: unit -> unit
  val by: Tactic.tactic -> unit
  val byev: Tactic.tactic list -> unit
  val chop: unit -> unit
  val choplev: int -> unit
  val compat_goal: Tactic.Thm.thm list -> int -> Tactic.Thm.thm list
  val curr_asms: Tactic.Thm.thm list ref
  val filter_goal: (term*term->bool) -> Tactic.Thm.thm list -> int -> Tactic.Thm.thm list
  val get_goal: int -> term
  val get_state: unit -> gstack
  val goal: Tactic.Thm.theory -> string -> Tactic.Thm.thm list
  val goals_limit: int ref
  val pr: unit -> unit
  val prlev: int -> unit
  val result: unit -> Tactic.Thm.thm  
  val uresult: unit -> Tactic.Thm.thm  
  val set_state: gstack -> unit
  val state_goal: Tactic.Thm.thm -> unit
  val top_thm: unit -> Tactic.Thm.thm
  val undo: unit -> unit
  val undo_list: gstack list ref
end;

functor GoalsFun (Tactic: TACTIC) : GOALS = 
struct
structure Tactic = Tactic;
local open Tactic Tactic.Thm
in

(*Each level of goal stack includes a proof state and alternative states,
  the output of the tactic applied to the preceeding level.  *)
type gstack = (thm * thm Sequence.seq) list;


(*** References ***)

(*Max number of goals to print -- set by user*)
val goals_limit = ref 10;

(*Current assumption list -- set by "goal".
  Useful if you forgot to save the value returned by "goal". *)
val curr_asms = ref([] : thm list);

(*Current result maker -- set by "goal", used by "result".  *)
val curr_mkresult = ref((fn _=> error"No current state\n") : bool*thm->thm);

(*List of previous goal stacks, for the undo operation.  Set by set_state. 
  A list of lists!*)
val undo_list = ref([] : gstack list);


(*** Commands etc ***)


(*Return the current goal stack, if any, from undo_list*)
fun get_state() : gstack = case !undo_list of 
      []   => error"No current state\n"
    | x::_ => x;

(*Pops the given goal stack*)
fun pop [] = error"Stack is empty\n"
  | pop (pair::pairs) = (pair,pairs);


(*Print a level of the goal stack.  Ignore Poly/ML spurious I/O exception*)
fun print_top ((th,_), pairs) = 
   (prs("Level " ^ string_of_int(length pairs) ^ "\n"); 
    print_goal_thm (!goals_limit) th)
  handle Io _ => prs"\nio_failure! -- Poly/ML bug!!\n";   


(*Printing can raise exceptions, so the assignment occurs last*)
fun set_state newgoals = 
  (print_top (pop newgoals);  undo_list := newgoals :: !undo_list);

(*Set the top-level goal, initialize *)
fun state_goal th =  (undo_list := [];  set_state [ (th, Sequence.null) ]);

(*Given a proof state transformation, return a command that updates
    the goal stack*)
fun make_command com = set_state (com (pop (get_state())));

(*Apply a function on proof states to the current goal stack*)
fun apply_fun f = f (pop(get_state()));

(*Return the top theorem, representing the proof state*)
fun top_thm () = apply_fun  (fn ((th,_), _) => th);

(*Return the final result.  *)
fun result () = !curr_mkresult (true, top_thm());

(*Return the result UNCHECKED that it equals the goal -- for synthesis,
  answer extraction, or other instantiation of Vars *)
fun uresult () = !curr_mkresult (false, top_thm());


(*Returning some subgoal in the proof state*)
fun get_goal_fun i ((th,_), _) : term =
      (case  nth_tail (i-1, prems_of th)  of
	    [] => error"get_goal: Goal number out of range\n"
	  | Q::_ => Q);

(*Get subgoal i from goal stack*)
fun get_goal i = apply_fun (get_goal_fun i);


(*Which thms could apply to goal i? (to debug tactics involving filter_thms) *)
fun filter_goal could ths i = filter_thms could (999, get_goal i, ths);

(*Which thms are compatible with goal i? *)
fun compat_goal thms i =
  compat_thms(itlist_right insert_thm (thms, Stringtree.null),
                      get_goal i);


fun chop_level n (pair,pairs) = nth_tail (length pairs - n, pair::pairs);

(*Print the given level of the proof*)
fun prlev n = apply_fun (print_top o pop o (chop_level n));
fun pr () = apply_fun print_top;


(*Read main goal.  Set global variables curr_asms, curr_mkresult. *)
fun goal thy agoal = 
  let val (asms, st0, mkresult) = prepare_proof thy agoal
  in  state_goal st0;  
      curr_asms := asms;
      curr_mkresult := mkresult;
      asms
  end;


(*Proof step "by" the given tactic -- apply tactic to the proof state*)
fun by_com tac ((th,ths), pairs) : gstack =
      (case  Sequence.pull(tapply(tac, th))  of
	   None      => error"by: tactic returned no results\n"
	 | Some pair => (pair::(th,ths)::pairs));

fun by tac = make_command (by_com tac);

(* byev[tac1,...,tacn] applies tac1 THEN ... THEN tacn.
   Good for debugging proofs involving prove_goal.*)
val byev = by o EVERY;


(*Backtracking means find an alternative result from a tactic.
  If none at this level, try earlier levels*)
fun backtrack [] = error"backtrack: no alternatives\n"
  | backtrack ((_,thstr) :: pairs) =
      (case Sequence.pull thstr of
	   None      => backtrack pairs
	 | Some pair => pair::pairs);

fun back() = set_state (backtrack (get_state()));

(*Chop back to previous level of the proof*)
fun choplev n = make_command (chop_level n);

(*Chopping back the goal stack*)
fun chop () = make_command (fn (_,pairs) => pairs);

(*Restore the previous proof state;  discard current state. *)
fun undo() = case !undo_list of
      [] => error"No proof state\n"
    | [_] => error"Already at initial state\n"
    | _::newundo =>  (undo_list := newundo;  pr()) ;

end;
end;
