(*  Title: 	NJ/resolve
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Intuitionistic first-order logic tactics
*)


signature NJ_RESOLVE = 
sig
  structure Tactic : TACTIC
  val all_elim: Tactic.Thm.thm
  val all_imp_elim: Tactic.Thm.thm
  val asm_rl: Tactic.Thm.thm
  val conj_elim: Tactic.Thm.thm
  val conj_imp_elim: Tactic.Thm.thm
  val disj_imp_elim: Tactic.Thm.thm
  val exists_imp_elim: Tactic.Thm.thm
  val iff_elim: Tactic.Thm.thm
  val iff_imp_elim: Tactic.Thm.thm
  val iff_intr: Tactic.Thm.thm
  val imp_elim: Tactic.Thm.thm
  val imp_imp_elim: Tactic.Thm.thm
  val mp_tac: int -> Tactic.tactic
  val pc_tac: int -> Tactic.tactic
  val safe_brls: (bool * Tactic.Thm.thm) list
  val safestep_tac: int -> Tactic.tactic
  val safe_tac: int -> Tactic.tactic
  val step_tac: int -> Tactic.tactic
  val subgoals_of_brl: bool * Tactic.Thm.thm -> int
  val unsafe_brls: (bool * Tactic.Thm.thm) list
end;


functor NJ_ResolveFun
    (structure NJ_Syntax: NJ_SYNTAX and NJ_Rule: NJ_RULE and Conv: CONV
	sharing NJ_Syntax.Syntax = NJ_Rule.Thm.Sign.Syntax
	    and Conv.Tactic.Thm = NJ_Rule.Thm)   : NJ_RESOLVE   = 
struct
structure Tactic = Conv.Tactic;
local open  NJ_Syntax  NJ_Rule  Conv  Conv.Tactic  Conv.Tactic.Thm
in


(*The rule P/P, solves assumptions in biresolve_tac*)
val asm_rl = trivial (Sign.read_cterm NJ_Rule.sign ("[|?P|]",Aprop));


val conj_elim = prove_goal NJ_Rule.thy 
    "[| P&Q |] ==> ([| P |] ==> [| Q |] ==> [| R |]) ==> [| R |]"
 (fn asms=>
  [ (REPEAT (resolve_tac asms 1  ORELSE
               (resolve_tac [conjunct1, conjunct2] 1 THEN
                resolve_tac asms 1))) ]);


val imp_elim = prove_goal NJ_Rule.thy 
    "[| P-->Q |] ==> [| P |] ==> ([| Q |] ==> [| R |]) ==> [| R |]"
 (fn asms=>
  [ (REPEAT (resolve_tac (asms@[mp]) 1)) ]);



val all_elim = prove_goal NJ_Rule.thy 
    "[| Forall(P) |] ==> ([| P(a) |] ==> [| R |]) ==> [| R |]"
 (fn asms=>
  [ (REPEAT (resolve_tac (asms@[spec]) 1)) ]);


val iff_intr = prove_goal NJ_Rule.thy 
   "([| P |] ==> [| Q |]) ==> ([| Q |] ==> [| P |]) ==> [| P<->Q |]"
 (fn asms=>
  [ (rewrite_goals_tac [iff_def]),
    (REPEAT (assume_tac 1 ORELSE 
             resolve_tac (asms@[conj_intr, imp_intr]) 1)) ]);


(*Observe use of rewrite_rule to unfold "<->" in meta-assumptions (asms) *)
val iff_elim = prove_goal NJ_Rule.thy 
    "[| P <-> Q |] ==> ([| P-->Q |] ==> [| Q-->P |] ==> [| R |]) ==> [| R |]"
 (fn asms =>
  [ (resolve_tac [conj_elim] 1),
    (REPEAT (assume_tac 1 ORELSE  
	     resolve_tac (map (rewrite_rule [iff_def]) asms) 1)) ]);


(*Simplifications of assumed implications

MOST SHOULD BE DELETED.  INSTEAD TRY IMPLICATION RULE WITH DEPTH BOUND. *)


val conj_imp_elim = prove_goal NJ_Rule.thy 
    "[| (P&Q)-->S |] ==> ([| P-->(Q-->S) |] ==> [| R |]) ==> [| R |]"
 (fn asms=>
  [ (REPEAT (assume_tac 1 ORELSE  
            resolve_tac (conj_intr::imp_intr::asms) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac asms 1))) ]);


val disj_imp_elim = prove_goal NJ_Rule.thy 
    "[| (P|Q)-->S |] ==> ([| P-->S |] ==> [| Q-->S |] ==> [| R |]) ==> [| R |]"
  (fn asms=>
 [ (REPEAT (assume_tac 1 ORELSE  
            biresolve_tac [ (true,disj_intr1), (true,disj_intr2) ] 1  ORELSE 
            resolve_tac (imp_intr::asms) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac asms 1))) ]);


(*Simplifies the implication.  Classical version is stronger. 
  Still UNSAFE since Q must be provable.  *)
val imp_imp_elim = prove_goal NJ_Rule.thy 
     "[| (P-->Q)-->S |] ==> ([| P |] ==> [| Q-->S |] ==> [| Q |]) ==>  \
\                           ([| S |] ==> [| R |]) ==>    [| R |]"
 (fn asms=>
  [ (REPEAT (assume_tac 1 ORELSE  
            resolve_tac (imp_intr::asms) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac asms 1))) ]);


(*Simplifies the implication.   UNSAFE.  *)
val iff_imp_elim = prove_goal NJ_Rule.thy 
    "[| (P<->Q)-->S |] ==> ([| P |] ==> [| Q-->S |] ==> [| Q |]) ==> \
\                          ([| Q |] ==> [| P-->S |] ==> [| P |]) ==>   \
\		([| S |] ==> [| R |]) ==> [| R |]"
 (fn asms=>
  [ (REPEAT (assume_tac 1 ORELSE  
            resolve_tac (iff_intr::imp_intr::asms) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac asms 1))) ]);


(*NOT complete: what if (ALL x.~ ~ P(x))-->(~ ~ ALL x.P(x)) is an assumption? 
  UNSAFE*)
val all_imp_elim = prove_goal NJ_Rule.thy 
    "[| Forall(P)-->S |] ==> (!(x)[| P(x) |]) ==> \
\                            ([| S |] ==> [| R |]) ==>   [| R |]"
 (fn asms=>
  [ (REPEAT (assume_tac 1 ORELSE  
            resolve_tac (all_intr::asms) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac asms 1))) ]);


(*Probably not complete: should not thin the assumption.*)
val exists_imp_elim = prove_goal NJ_Rule.thy 
    "[| Exists(P)-->S |] ==> ([| P(a)-->S |] ==> [| R |]) ==> [| R |]"
 (fn asms=>
  [ (REPEAT (assume_tac 1 ORELSE  
            resolve_tac (imp_intr::exists_intr::asms) 1  ORELSE 
            (resolve_tac [imp_elim] 1  THEN resolve_tac asms 1))) ]);



(*The number of new subgoals produced by the brule*)
fun subgoals_of_brl (true,rule) = length (prems_of rule) - 1
  | subgoals_of_brl (false,rule) = length (prems_of rule);

(*Less-than test: for sorting to minimize number of new subgoals*)
fun lessb (brl1,brl2) = subgoals_of_brl brl1 < subgoals_of_brl brl2;

val safe_brls = sort lessb 
    [ (true,False_elim), (true,asm_rl), 
      (false,imp_intr), (false,all_intr), (true,conj_elim), (true,exists_elim),
      (false,conj_intr), (true,conj_imp_elim),
      (true,disj_imp_elim), (true,exists_imp_elim),
      (true,disj_elim), (false,iff_intr), (true,iff_elim) ];


val unsafe_brls =
    [ (false,disj_intr1), (false,disj_intr2), (false,exists_intr), 
      (true,all_elim), (true,imp_imp_elim), (true,iff_imp_elim),
      (true,all_imp_elim) ];


(*Finds P-->Q and P in the assumptions, replaces implication by Q *)
fun mp_tac i = biresolve_tac [(true,imp_elim)] i  THEN  assume_tac i;


(*0 subgoals vs 1 or more*)
val (safe0_brls, safep_brls) =
    partition (apl(0,op=) o subgoals_of_brl) safe_brls;

val safestep_tac =
    biresolve_tac safe0_brls  ORELSE'  mp_tac  ORELSE'
    (DETERM o biresolve_tac safep_brls);

val safe_tac = DEPTH_SOLVE_1 o safestep_tac;

val step_tac = safestep_tac  ORELSE'  biresolve_tac unsafe_brls;

(*Fails unless it solves the goal!*)
val pc_tac = DEPTH_SOLVE_1 o step_tac;

end;
end;

