(*  Title: 	HOL/types
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Simple types: functions, products, options, unions
*)


(*** Booleans *)

val true_equal_True = prove_goal HOL_Rule.thy
    "[| form(p) |] ==> [| p: bool |] ==> [| [ p = True : bool ] |]"
 (fn asms=>
  [ (resolve_tac [eqterm_intr] 1),
    (REPEAT (resolve_tac (asms@[True_intr, True_type]) 1)) ]);


val false_equal_False = prove_goal HOL_Rule.thy
    "[| ~form(p) |] ==> [| p: bool |] ==> [| [ p = False : bool ] |]"
 (fn asms=>
  [ (resolve_tac [eqterm_intr] 1),
    (REPEAT (ares_tac (asms@[False_type,contr,False_elim]) 1)) ]);


(*Case analysis on P in Q(P), proved by substitution of True and False for P*)
val bool_elim = prove_goal HOL_Rule.thy
    "[| p: bool |] ==> [| Q(True) |] ==> [| Q(False) |] ==> [| Q(p) |]"  
 (fn asms=>
  [ (resolve_tac [ex_middle RS disj_elim] 1),
    (REPEAT (DEPTH_SOLVE_1
	 (ares_tac (asms@[false_equal_False,
			  true_equal_True]) 1  ORELSE  
             res_inst_tac [("P", "Q", Aterm-->Aform)] subst 1))) ]);


(** Conditionals -- thanks to Martin Coen **)

val cond_type = prove_goal HOL_Rule.thy 
     "([| form(p)  |] ==> [| a:A |]) ==> \
\     ([| ~form(p) |] ==> [| b:A |]) ==> \
\     [| cond(A,p,a,b) : A |]"
 (fn asms=>
  [ (rewrite_goals_tac [cond_def]),
    (resolve_tac [Pick_type] 1),
    (resolve_tac [disj_elim] 1),
    (resolve_tac [ex_middle] 1),
    (resolve_tac [exists_intr] 1),
    (resolve_tac [disj_intr2] 1),
    (REPEAT (ares_tac (asms@[conj_intr,refl,exists_intr,disj_intr1]) 1)) ]);

val cond_convT = prove_goal HOL_Rule.thy 
     "[| a:A |] ==> \
\     [| [cond(A,True,a,b)=a:A] |]"
 (fn asms=>
  [ (rewrite_goals_tac [cond_def]),
    (resolve_tac [Pick_iff_equals] 1),
    (Pc.tp_tac asms),
    (REPEAT (ares_tac [True_intr,contr_elim] 1)) ]);

val cond_convF = prove_goal HOL_Rule.thy 
     "[| b:A |] ==> \
\     [| [cond(A,False,a,b)=b:A] |]"
 (fn asms=>
  [ (rewrite_goals_tac [cond_def]),
    (resolve_tac [Pick_iff_equals] 1),
    (Pc.tp_tac asms),
    (REPEAT (ares_tac [not_intr,contr_elim] 1)) ]);

(** Equality on a subtype coincides with equality on the parent type **)

val subtype_abs_equal = prove_goal HOL_Rule.thy
    "[| [ a = b : A ] |] ==> [| P(a) |] ==> [| [ a = b : {x:A.P(x)} ] |]"
 (fn asms=>
  [ (resolve_tac (reslist(asms, 1, sym RS subst)) 1),
    (resolve_tac [subtype_intr RS refl] 1),
    (REPEAT (resolve_tac (asms@[eq_type1]) 1)) ]);

val subtype_rep_equal = prove_goal HOL_Rule.thy
    "[| [ a = b : {x:A.P(x)} ] |] ==> [| [ a = b : A ] |]"
 (fn asms=>
  [ (resolve_tac (reslist(asms, 1, subst)) 1),
    (resolve_tac [refl] 1),
    (resolve_tac [subtype_elim1] 1),
    (resolve_tac [eq_type2] 1),
    (resolve_tac asms 1) ]);



(*Produce conclusions from subtype rules after unfolding definitions --
  for reasoning about subtypes*)
fun subtype_rules defs asms =
    let val rasms = map (rewrite_rule defs) asms;
    in   reslist (rasms, 1, subtype_elim1) 
       @ reslist (rasms, 1, subtype_elim2)
       @ reslist (rasms, 1, subtype_rep_equal)
    end;


(** the types "void" and "unit" *)

val void_elim = prove_goal HOL_Rule.thy 
    "[| a: void |] ==> [| P |]"
 (fn asms=>
  [ (REPEAT (resolve_tac (subtype_rules [void_def] asms
			  @ [False_elim]) 1)) ]);


val True_unit = prove_goal HOL_Rule.thy "[| True : unit |]"  
 (fn asms=>
  [ (rewrite_goals_tac [unit_def]),
    (REPEAT (resolve_tac [True_type, refl, subtype_intr] 1)) ]);


val unit_elim = prove_goal HOL_Rule.thy
    "[| p: unit |] ==> [| Q(True) |] ==> [| Q(p) |]"
 (fn asms=>
  [ (resolve_tac [subst] 1),
    (REPEAT (resolve_tac (subtype_rules [unit_def] asms
			  @ asms) 1)) ]);


(*Used in Cantor's Theorem*)
val equal_contr = prove_goal HOL_Rule.thy 
    "[| [ term(~form(p)) = p : bool ] |] ==> [| Q |]"
 (fn asms=>
  [ (resolve_tac [False_elim] 1),
    (resolve_tac [asms RSN (1,form_iff) RS iff_elim] 1),
    (Pc.fast_tac [] 1) ]);


(*** Functions ***)

val apply_equal = prove_goal HOL_Rule.thy
    "[| [ f = h : A->B ] |] ==> [| [ a = c : A ] |] ==> \
\    ([| [ f`a = h`c : B ] |] ==> [| R |])  ==>  \
\    [| R |]"
 (fn asms =>
  [ (REPEAT (ares_tac (apply_type::asms) 1  ORELSE  subst2_tac "op`" 1)) ]);


(*Given assumption f=g:A->B and a:A, deduces f`a=g`a:B    USED?? *)
fun apply_equal_tac (sa,sA) i =
    (eresolve_tac [apply_equal] i) THEN
    (res_inst_tac [("a",sa, Aterm), ("A",sA, Atype)] refl i);

val beta_equal = prove_goal HOL_Rule.thy
    "[| [ (lam x:A.b(x)) = (lam x:A.b'(x)) : A->B ] |] ==> \
\    [| [ a = a' : A ] |] ==> \
\    (!(x)[| x : A |] ==> [| b(x) : B |]) ==> \
\    (!(x)[| x : A |] ==> [| b'(x) : B |]) ==> \
\    ([| [ b(a) = b'(a') : B ] |] ==> [| R |])  ==>  \
\    [| R |]"
 (fn aseq::asms =>
  [ (resolve_tac [aseq RS apply_equal] 1),  
    (REPEAT (resolve_tac asms 1)),
    (eresolve_tac [box_equals] 1),
    (REPEAT (SOMEGOAL (ares_tac (asms@[beta_conv])))),
    (REPEAT (resolve_tac [eq_type1,eq_type2] 1 
	THEN  resolve_tac asms 1)) ]);

(*Given assumption (lam x:A.b(x)) = (lam x:A.b'(x)):A->B and a:A, 
  deduces f`a=g`a:B  *)
fun beta_equal_tac (sa,sA) i =
    (eresolve_tac [beta_equal] i) THEN
    (res_inst_tac [("a",sa, Aterm), ("A",sA, Atype)] refl i);



(** Standard forms of functions from specific types *)

val void_fun_equal = prove_goal HOL_Rule.thy 
    "[| f : void->B |] ==> [| g : void->B |] ==> [| [ f = g : void->B ] |]"  
 (fn asms=>
  [ (REPEAT (ares_tac (asms@type_rls@[extensionality,void_elim]) 1)) ]);


val unit_fun_equal = prove_goal HOL_Rule.thy 
    "[| f : unit->B |] ==> [| [ (lam x:unit.f`True) = f : unit->B ] |]"  
 (fn asms=>
  [ (REPEAT (ares_tac (asms@type_rls@[True_unit,beta_conv,extensionality]) 1  
     ORELSE  eresolve_tac [unit_elim] 1)) ]);


val bool_fun_equal = prove_goal HOL_Rule.thy 
    "[| f : bool->B |] ==> \
\    [| [ (lam x:bool.cond(x, f`True, f`False)) = f : bool->B ] |]";
 (fn asms=>
  [ (REPEAT (ares_tac (asms@type_rls@
	    [trans RES beta_conv, cond_convT, cond_convF, extensionality]) 1 
	ORELSE  eresolve_tac [bool_elim] 1)) ]);



(**** unions: the type A+B  *)

val plus_defs = [Inl_def, Inr_def, when_def, plus_def];

(** Constructors *)

val Inl_type = prove_goal HOL_Rule.thy "[| a: A |] ==> [| Inl(A,B,a) : A+B |]"  
 (fn asms=>
  [ (rewrite_goals_tac plus_defs),
    (REPEAT (ares_tac (asms@type_rls@
		[subtype_intr, exists_intr, disj_intr1, refl]) 1)) ]);

val Inr_type = prove_goal HOL_Rule.thy "[| b: B |] ==> [| Inr(A,B,b) : A+B |]"  
 (fn asms=>
  [ (rewrite_goals_tac plus_defs),
    (REPEAT (ares_tac (asms@type_rls@
		[subtype_intr, exists_intr, disj_intr2, refl]) 1)) ]);


val plus_elim = prove_goal HOL_Rule.thy
    "[| p : A+B |]  ==> \
\    (!(x)[| x: A |] ==> [| Q(Inl(A,B,x)) |])  ==> \
\    (!(y)[| y: B |] ==> [| Q(Inr(A,B,y)) |])  ==> \
\    [| Q(p) |]"
 (fn asms=>
  [ (resolve_tac [disj_elim] 1),  (*will catch one of the subtype_rules*)
    (REPEAT (assume_tac 1  ORELSE  
             eresolve_tac [subst, exists_elim] 1  	ORELSE  
             resolve_tac (subtype_rules [plus_def] asms
			  @ asms) 1)) ]);


(** Freeness of constructors (follows from their definition) *)

val plus_distinct = prove_goal HOL_Rule.thy
  "[| [ Inl(A,B,a)=Inr(A,B,b): A+B ] |] ==> [| a: A |] ==> [| b: B |] ==> \
\  [| P |]"
 (fn aseq::asms=>
  [ (resolve_tac [rewrite_rule plus_defs aseq  RS subtype_rep_equal
		  RS pair_inject RS False_elim] 1),
    (*Forwards reasoning on equality assumptions. *)
    (beta_equal_tac ("b","B") 1),
    (REPEAT (ares_tac (asms@type_rls@[form_intr,refl]) 1
	ORELSE eresolve_tac [subst] 1)) ]);


val Inl_inject = prove_goal HOL_Rule.thy
  "[| [ Inl(A,B,a) = Inl(A,B,c): A+B ] |] ==>  \
\  [| a: A |] ==> [| c: A |] ==>     [| [a=c:A] |]"
 (fn aseq::asms=>
  [ (resolve_tac [equal_prop] 1),
    (resolve_tac [rewrite_rule plus_defs aseq  RS subtype_rep_equal
		  RS pair_inject] 1),
    (beta_equal_tac ("c","A") 1),
    (REPEAT (ares_tac (asms@type_rls@[refl, form_intr]) 1)) ]);


val Inr_inject = prove_goal HOL_Rule.thy
  "[| [ Inr(A,B,b) = Inr(A,B,d): A+B ] |] ==>  \
\  [| b: B |] ==> [| d: B |] ==>     [| [b=d:B] |]"
 (fn aseq::asms=>
  [ (resolve_tac [equal_prop] 1),
    (resolve_tac [rewrite_rule plus_defs aseq  RS subtype_rep_equal
		  RS pair_inject] 1),
    (beta_equal_tac ("d","B") 1),
    (REPEAT (ares_tac (asms@type_rls@[refl, form_intr]) 1)) ]);


(** The eliminator, when (exists because of freeness) *)

val when_exists = prove_goal HOL_Rule.thy 
  "[| p: A+B |] ==>  \
\  (!(x)[| x: A |] ==> [| c(x) : C |]) ==>    \
\  (!(y)[| y: B |] ==> [| d(y) : C |]) ==>    \
\  [| EXISTS z:C.   \
\	(ALL x:A. [ p = Inl(A,B,x) : A+B ] --> [ z = c(x) : C ]) & \
\	(ALL y:B. [ p = Inr(A,B,y) : A+B ] --> [ z = d(y) : C ]) |]"
 (fn asms=>
  [ (resolve_tac [ asms RSN (1,plus_elim) ] 1),
    (REPEAT_SOME (assume_tac ORELSE'
	    eresolve_tac [Inl_inject RS subst,
			   Inr_inject RS subst,
			   plus_distinct,
			   sym RS plus_distinct] ORELSE'
	    filt_resolve_tac (asms @ type_rls @ [Inl_type,Inr_type,
		all_intr,imp_intr,conj_intr,exists_intr,refl]) 1)) ]);


val when_type = prove_goal HOL_Rule.thy 
  "[| p: A+B |] ==>  \
\  (!(x)[| x: A |] ==> [| c(x) : C |]) ==>    \
\  (!(y)[| y: B |] ==> [| d(y) : C |]) ==>    \
\  [| when(A,B,C,p,c,d) : C |]"
 (fn asms=>
  [ (rewrite_goals_tac [when_def]),
    (REPEAT (ares_tac (when_exists::Pick_type::asms) 1)) ]);


(*Use eq_type against one of the asms to prove some typing subgoals*)
fun eq_type_tac asms =
    resolve_tac [eq_type1,eq_type2]  THEN'
    resolve_tac asms  THEN'
    (REPEAT o assume_tac);


val when_congr = prove_goal HOL_Rule.thy 
  "[| [ p = p' : A+B ] |] ==>  \
\  (!(x)[| x: A |] ==> [| [ c(x) = c'(x) : C ] |]) ==>    \
\  (!(y)[| y: B |] ==> [| [ d(y) = d'(y) : C ] |]) ==>    \
\  [| [ when(A,B,C,p,c,d) = when(A,B,C,p',c',d') : C ] |]"
 (fn asms=>
  [ (rewrite_goals_tac [when_def]),
    (REPEAT (ares_tac ([Pick_congr,refl,Inl_type,Inr_type,when_exists]
		@asms@iff_congr_rls) 1  ORELSE 
	     eq_type_tac asms 1)) ]);



val when_conv_lemma = prove_goal HOL_Rule.thy 
  "[| p: A+B |] ==>  \
\  (!(x)[| x: A |] ==> [| c(x) : C |]) ==>    \
\  (!(y)[| y: B |] ==> [| d(y) : C |]) ==>    \
\  [| (ALL x:A. [ p=Inl(A,B,x): A+B ] --> [ when(A,B,C,p,c,d)=c(x) : C ]) & \
\     (ALL y:B. [ p=Inr(A,B,y): A+B ] --> [ when(A,B,C,p,c,d)=d(y) : C ]) |]"
 (fn asms=>
  [ (rewrite_goals_tac [when_def]),
    (resolve_tac [Pick_intr] 1),
    (REPEAT (ares_tac (when_exists::asms) 1)) ]);


val when_conv_Inl = prove_goal HOL_Rule.thy 
  "[| a: A |] ==>  \
\  (!(x)[| x: A |] ==> [| c(x) : C |]) ==>    \
\  (!(y)[| y: B |] ==> [| d(y) : C |]) ==>    \
\  [| [ when(A, B, C, Inl(A,B,a), c, d) = c(a) : C ] |]"
 (fn asms=>
  [ (metacut_tac (when_conv_lemma RS conjunct1) 1),
    (REPEAT_SOME (eresolve_tac [mp, all_elim])),
    (REPEAT (ares_tac (asms @ type_rls @ [Inl_type,Inr_type,refl]) 1)) ]);

val when_conv_Inr = prove_goal HOL_Rule.thy 
  "[| b: B |] ==>  \
\  (!(x)[| x: A |] ==> [| c(x) : C |]) ==>    \
\  (!(y)[| y: B |] ==> [| d(y) : C |]) ==>    \
\  [| [ when(A, B, C, Inr(A,B,b), c, d) = d(b) : C ] |]"
 (fn asms=>
  [ (metacut_tac (when_conv_lemma RS conjunct2) 1),
    (REPEAT_SOME (eresolve_tac [mp, all_elim])),
    (REPEAT (ares_tac (asms @ type_rls @ [Inl_type,Inr_type,refl]) 1)) ]);


