(*  Title: 	HOL/syntax
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Natural deduction syntax for classical higher-order logic.
*)


signature HOL_SYNTAX =
sig
  structure Syntax : SYNTAX
  val Aform: typ
  val Aterm: typ
  val Atype: typ
  val const_decs: (string list * typ) list
  val parse1: ((string*Syntax.lexsy)list -> term * (string*Syntax.lexsy) list) -> (string*Syntax.lexsy)list -> term * (string*Syntax.lexsy)list
  val parser: (string*Syntax.lexsy) list -> term * (string*Syntax.lexsy) list
  val print1: (Syntax.lexsy Syntax.Symtab.table -> int * int -> term -> unit)-> Syntax.lexsy Syntax.Symtab.table -> int * int -> term -> unit
  val printer: Syntax.lexsy Syntax.Symtab.table -> term -> unit
  val prin: term -> unit
  val read: string -> term
  val syn: Syntax.syntax
end;

functor HOL_SyntaxFun (structure Syntax: SYNTAX and Pretty: PRETTY) 
	: HOL_SYNTAX = 
struct
structure Syntax = Syntax;
structure Symtab = Syntax.Symtab;
local open Syntax
in


(*Meta-types of terms, formulae, types*)
val Aterm  = Ground "term";
val Aform  = Ground "form";
val Atype  = Ground "type";


(*Constants and their types*)
val const_decs = 
  [ (*Types*)
    (["*", "+", "->"],	[Atype,Atype]--->Atype ),

    (*Connectives*)
    (["&", "|", "-->", "<->"],	[Aform,Aform]--->Aform ), 

    (*Propositions*)
    (["Trueprop"],	Aform-->Aprop ),
    (["Elem"],		[Aterm, Atype]--->Aprop ),
    (["Reduce"], 	[Aterm, Aterm]--->Aprop ),

    (*Reflection*)
    (["term"],		Aform-->Aterm ),
    (["form"],		Aterm-->Aform ),

    (["subtype"], 	[Atype, Aterm-->Aform] ---> Atype ),

    (*Functions*)
    (["Lambda"],	[Atype, Aterm-->Aterm]--->Aterm ),
    (["`"],	[Aterm,Aterm]--->Aterm ),

    (*Products*)
    (["Pair"],		[Aterm,Aterm]--->Aterm ),
    (["fst","snd"],	Aterm-->Aterm ),
    (["split"],		[Aterm, [Aterm,Aterm]--->Aterm] --->Aterm  ),

    (["void", "unit"],  Atype ), 

    (*Booleans*)
    (["bool"],  	Atype ), 
    (["True","False"],  Aterm ), 
    (["cond"],          [Atype,Aterm,Aterm,Aterm]--->Aterm),

    (*Unions*)
    (["Inl","Inr"],	[Atype, Atype, Aterm]--->Aterm ),
    (["when"],		[Atype, Atype, Atype,
		 Aterm, Aterm-->Aterm, Aterm-->Aterm] ---> Aterm ),

    (*Classes*)
    (["<:"],		[Aterm,Aterm]--->Aform ),
    (["subset"],	[Atype,Aterm,Aterm]--->Aform ),
    (["un", "int"],	[Atype,Aterm,Aterm]--->Aterm ),
    (["union", "inter", "pow"],	[Atype,Aterm]--->Aterm ),

    (*Natural numbers*)
    (["nat"],  		Atype ), 
    (["0"],		Aterm ),
    (["Succ"],		Aterm-->Aterm ),
    (["rec"],	[Aterm, Aterm, [Aterm,Aterm]--->Aterm] ---> Aterm ),

    (* ~P and [a=b:A] parsed specially*)
    (["Not"],  		Aform-->Aform ), 
    (["Eq"],		[Aterm,Aterm,Atype]--->Aform ),

    (["Pick"],			[Atype, Aterm-->Aform]--->Aterm ), 
    (["Forall","Exists"],	[Atype, Aterm-->Aform]--->Aform ) ];


(*Get constants*)
val get = Sign.get_const const_decs;

val Trueprop	= get "Trueprop";
val Elem	= get "Elem";

val Forall	= get "Forall";
val Exists	= get "Exists";
val Pick	= get "Pick";
val Lambda	= get "Lambda";
val Pair 	= get "Pair";
val subtype 	= get "subtype";
val Not   	= get "Not";
val Eq   	= get "Eq";


(*Parsing of terms and theorems.
  Written as a functional so that set theory parsers can call it:
    recparse handles recursive calls. 
  
  Allows prefix notation: ~P for not(P).  Need a mechanism for prefix ops.
  Treatment of ~ is strange since precedence must be passed to parser. *)
fun parse1 recparse =
  let fun par_binder quant lxs =   (*Quantifiers: FORALL x:A.P *)
            apfst (fn ((s,T), (A,P)) => quant $ A $ absfree(s,T,P))
	      (((parse_ident thenkey":") thenp 
	       (recparse  thenkey"." thenp recparse)) lxs)
      fun parse (("[|",DelimSy) :: lxs) =
	       (parse_thm1 recparse  thenkey"|]") lxs
        | parse (("ALL",DelimSy) :: lxs) = par_binder Forall lxs
        | parse (("EXISTS",DelimSy) :: lxs) = par_binder Exists lxs
        | parse (("PICK",DelimSy) :: lxs) = par_binder Pick lxs
        | parse (("lam",DelimSy) :: lxs) = par_binder Lambda lxs
	| parse (("<",DelimSy) :: lxs) =
	    apfst (fn (a,b) => Pair$a$b)
	      ((recparse  thenkey","  thenp  recparse  thenkey">") lxs)
	| parse (("[",DelimSy) :: lxs) =
	    apfst (fn ((a,b),A) => Eq$a$b$A)
	      (((recparse  thenkey"="  thenp  recparse)
		  thenkey":"  thenp  recparse  thenkey"]") lxs)
	| parse (("{",DelimSy) :: lxs) = (par_binder subtype  thenkey"}") lxs
        | parse (("~",DelimSy) :: lxs) =
	    apfst (fn P => Not$P)  (parse_functl parse 100 lxs)
        | parse lxs = parserr("Illegal expression", lxs)
  in  parse_functl parse 0  end

(*Theorems have the form a:A (typing assertions) or just P (logical truths) *)
and parse_thm1 recparse lxs = 
  let fun parse_judge lxs =
	case  recparse lxs  of
	    (a, (":",DelimSy) :: ls) => apfst (fn A => Elem$a$A) (recparse ls)
	  | (P, ls) => (Trueprop$P, ls)
  in  parse_judge lxs   end;


(*"Tying the knot" -- this parser calls itself*)
fun parser lxs = parse1 parser lxs;



(*** PRINTING *)

(*Special treatment of: not Exists Forall
  Written as a functional so that set theory printers can call it:
    recprint handles recursive calls. *)
fun print1 recprint lextab (arg as (outleft,outright)) =
  let fun printm t = recprint lextab (0,0) t;  (*abstraction prevents loop!*)
      fun pr_binder (pre, A, (s,P)) =
	 (Pretty.bg 3;  Pretty.st(pre ^ " " ^ s ^ ": ");
	  printm A;  Pretty.brks "." 1; 
	  printm P;  Pretty.en())
  in fn
      Const("Trueprop",_) $ P => 
	(Pretty.bgs"[| ";  printm P;  Pretty.ens" |]")
    | Const("Elem",_)   $ a $ A =>
	(Pretty.bgs"[| ";  printm a;  Pretty.brks " :" 1;
	 printm A;  Pretty.ens" |]")
    | Const("Forall",_) $ A $ Abs(s,T,P) => 
  	enclose_if (arg<>(0,0)) pr_binder ("ALL", A, variant_abs(s,T,P))
    | Const("Exists",_) $ A $ Abs(s,T,P) =>
	enclose_if (arg<>(0,0)) pr_binder ("EXISTS", A, variant_abs(s,T,P))
    | Const("Pick",_) $ A $ Abs(s,T,P) =>
	enclose_if (arg<>(0,0)) pr_binder ("PICK", A, variant_abs(s,T,P))
    | Const("Lambda",_) $ A $ Abs(s,T,b) =>
	enclose_if (arg<>(0,0)) pr_binder ("lam", A, variant_abs(s,T,b))
    | Const("subtype",_) $ A $ Abs(s,T,b) =>
	let val (s,P) = variant_abs(s,T,b)
	in  Pretty.bgs"{";  Pretty.st(s ^ ":");
	    printm A;  Pretty.brks "." 1;
	    printm P;  Pretty.ens"}"
	end
    | Const("Pair",_) $ a $ b =>  
	Pretty.list ("<", ">", printm) [a,b]
    | Const("Eq",_) $ a $ b $ A=>
	(Pretty.bgs"[ ";  printm a;  Pretty.brks " =" 1;  printm b;  
	    Pretty.brks ":" 1;  printm A;  Pretty.ens" ]")
    | Const("Not",_)  $ P  =>    
	(Pretty.st"~ ";  recprint lextab (100,outright) P)  
    | t => print_functl (recprint lextab, lextab) arg t
  end;


fun printer lextab =
  let fun pr lextab = print1 pr lextab
  in  pr lextab (0,0)  end;

 

(*Infixes and their precedence*)
val infix_decs = 
    [ (*Terms*)
    (["`"],	"left",   9),		(*function application*)
    (["<:"], "right", 6),		(*membership in a class*)

    (**Types*)
    (["*"],	"right",  5),
    (["+"],	"right",  4),
    (["->"],	"right",  3),

    (*Connectives*)
    (["&"],		"right",  5),
    (["|"],		"right",  4),
    (["-->", "<->"],	"right",  3) ];

val keywords = [".", ":", "<", ">", "[", "]", "=", "{", "}", "lam",
      "~", "ALL", "EXISTS", "PICK"];

val syn = Syntax.extend pure (keywords, infix_decs, parser, printer); 

(*read/print functions*)
fun read a = Syntax.read syn "?" a;
fun prin t = (Pretty.init();  Syntax.prin syn t;  Pretty.nl());

end;
end;
