(*  Title: HOL/ruleshell
    Author: Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1989  University of Cambridge

Rules of Higher-order Logic (Type Theory)

!!!After updating, rebuild  ".rules.ML"  by calling make-rulenames!!! 
*)

signature HOL_RULE =
  sig
  structure Thm : THM
  val sign: Thm.Sign.sg
  val thy: Thm.theory
(*INSERT-RULESIG -- file produced by make-rulenames*)
  end;


functor HOL_RuleFun (structure HOL_Syntax: HOL_SYNTAX and Thm: THM
	sharing HOL_Syntax.Syntax = Thm.Sign.Syntax) : HOL_RULE = 
struct
structure Thm = Thm;

val thy = Thm.enrich_theory Thm.pure_thy "HOL"
    (["term","form","type"], HOL_Syntax.const_decs, HOL_Syntax.syn)
[ 
  (*** Equality ***)

  ("refl",  "[| a: A |] ==> [| [ a = a : A ]  |]"),

  ("sym",  "[| [ a = b : A ]  |] ==> [| [ b = a : A ]  |]"),

  (*Equal terms are well typed -- all rules must enforce this! *)
  ("eq_type1",  "[| [ a = b : A ] |] ==> [| a: A |]"  ),

  ("eq_type2",  "[| [ a = b : A ] |] ==> [| b: A |]"  ),

  ("subst", 
    "[| [ a = c : A ]  |]  ==>  [| P(c) |]  ==>  [| P(a) |]"),


  (*** TYPES ***)

  (** Functions **)

  ("Lambda_type",
    "(!(x)[| x: A |] ==> [| b(x) : B |])  ==>    \ 
\    [| lam x:A. b(x) : A->B |]"  ),

  ("Lambda_congr",
    "(!(x)[| x: A |] ==> [| [ b(x) = c(x) : B ] |])  ==>    \ 
\    [| [ lam x:A. b(x) = lam x:A. c(x) : A->B ] |]"  ),

  ("apply_type",
    "[| f: A->B |] ==> [| a: A |]  ==> [| f`a : B |]"  ),

  ("beta_conv",
    "[| a : A |] ==> (!(x)[| x: A |] ==> [| b(x) : B |])  ==> \ 
\    [| [ (lam x:A.b(x)) ` a = b(a) : B ] |]"  ),

  ("eta_conv", "[| f: A->B |] ==> [| [ lam x:A. f`x = f : A->B ] |]"  ),


  (** Products **)

  ("pair_type", "[| a: A |] ==>  [| b: B |]  ==>  [| <a,b> : A*B |]"  ),

  ("prod_elim", 
    "[| p : A*B |]  ==> \
\    (!(x,y)[| x: A |] ==> [| y: B |] ==> [| Q(<x,y>) |])  ==> \
\    [| Q(p) |]"  ),

  ("pair_inject", 
    "[| [ <a,b> = <c,d> : A*B ] |] ==>  \
\    ([| [ a = c : A ] |] ==> [| [ b = d : B ] |] ==> [| R |])  ==>  \
\    [| R |]"  ),

  (*fst and snd could be defined using descriptions...they are not to avoid
    excessive type labels -- which is the point of defining products here. *)

  ("fst_type",  "[| p: A*B |] ==> [| fst(p) : A |]"  ),
  ("snd_type",  "[| p: A*B |] ==> [| snd(p) : B |]"  ),

  ("fst_conv",  "[| a: A |] ==>  [| b: B |] ==> [| [ fst(<a,b>) = a: A] |]"  ),
  ("snd_conv",  "[| a: A |] ==>  [| b: B |] ==> [| [ snd(<a,b>) = b: B] |]"  ),

  ("split_def", "split(p,f) == f(fst(p), snd(p))"  ),


  (** Subtypes **)

  ("subtype_intr",  "[| a: A |] ==> [| P(a) |] ==> [| a : {x:A.P(x)} |]"  ),

  ("subtype_elim1",   "[| a: {x:A.P(x)} |] ==> [| a:A |]"),
  ("subtype_elim2",   "[| a: {x:A.P(x)} |] ==> [| P(a) |]"),


  (** Natural numbers **)

  ("Zero_type",  "[| 0: nat |]"  ),
  ("Succ_type",  "[| a: nat |] ==> [| Succ(a) : nat |]"  ),

  ("rec_type", 
    "[| a : nat |] ==> \
\    [| b : C |]  ==> \
\    (!(x,y)[| x: nat |] ==> [| y: C |] ==> [| c(x,y): C |])  ==>  \ 
\    [| rec(a,b,c) : C |]"  ),

  ("rec_congr", 
    "[| [ a = a' : nat ] |] ==> \
\    [| [ b = b' : C ] |]  ==> \
\    (!(x,y)[| x: nat |] ==> [| y: C |] ==> \
\           [| [ c(x,y) = c'(x,y): C ] |]) ==>  \ 
\    [| [ rec(a,b,c) = rec(a',b',c') : C ] |]"  ),

  ("rec_conv0", 
    "[| b: C |] ==> \
\    (!(x,y)[| x: nat |] ==> [| y: C |] ==> [| c(x,y): C |])  ==>  \ 
\    [| [ rec(0,b,c) = b : C ] |]"  ),

  ("rec_conv1", 
    "[| a : nat |] ==> \
\    [| b : C |]  ==> \
\    (!(x,y)[| x: nat |] ==> [| y: C |] ==> [| c(x,y): C |]) ==>  \ 
\    [| [ rec(Succ(a),b,c) = c(a, rec(a,b,c)) : C ] |]"  ),

  ("nat_induct", 
    "[| a: nat |] ==> [| Q(0) |] ==>    \
\    (!(x)[| x: nat |] ==> [| Q(x) |] ==> [| Q(Succ(x)) |]) ==>    \
\    [| Q(a) |]"  ),


  (*** Logic ***)

  (** Implication and quantification *)

  ("classical",  "([| ~P |] ==> [| P |])  ==> [| P |]"),

  ("imp_intr",
    "([| P |] ==> [| Q |])  ==>  [| P-->Q |]"),

  ("mp",
    "[| P-->Q |] ==> [| P |]  ==> [| Q |]"),

  ("all_intr",
    "(!(x)[| x: A |] ==> [| P(x) |])  ==>  [| ALL x:A.P(x) |]"),

  ("spec",
    "[| ALL x:A.P(x) |] ==> [| a : A |]  ==> [| P(a) |]"),



  (** Reflection *)

  ("term_type", "[| term(P) : bool |]"  ),

  ("term_conv", "[| p: bool |] ==> [| [ term(form(p)) = p : bool ] |]"  ),

  ("form_intr", "[| P |] ==> [| form(term(P)) |]"),

  ("form_elim", "[| form(term(P)) |] ==> [| P |]"),

  ("term_congr",
    "([| P |] ==> [| Q |]) ==> ([| Q |] ==> [| P |]) ==>   \
\    [| [ term(P) = term(Q) : bool ] |]"),


  (** Reduction predicate for simplification. *)

  (*does not verify a:A!  Sound because only trans_red uses a Reduce premise*)
  ("refl_red", "Reduce(a,a)"  ),

  ("red_if_equal", "[| [ a = b : A ] |] ==> Reduce(a,b)"),

  ("trans_red", "[| [ a = b : A ] |] ==> Reduce(b,c) ==> [| [ a = c : A ] |]"),


  (** Definitions of other connectives*)

  ("False_def", "False == term(ALL p:bool.form(p))"),
  ("True_def", 	"True == term(ALL p:bool.form(p)-->form(p))"),
  ("conj_def", 	"P&Q == ALL r:bool. (P-->Q-->form(r)) --> form(r)"),

  ("disj_def",
   "P|Q == ALL r:bool. (P-->form(r)) --> (Q-->form(r)) --> form(r)"),

  ("exists_def",
   "(EXISTS x:A. P(x)) ==  ALL r:bool. (ALL x:A. P(x)-->form(r)) --> form(r)"),

  ("not_def", 	"~P == (P-->form(False))"),
  ("iff_def", 	"P<->Q == (P-->Q) & (Q-->P)"),


  (** Conditionals *)

  ("cond_def", "cond(A,p,a,b) == PICK x:A.(form(p)  & [x=a:A]) | \
\                                         (~form(p) & [x=b:A])" ),

  (** Descriptions *)

  ("Pick_type", "[| EXISTS x:A.P(x) |] ==> [| (PICK x:A.P(x)) : A |]"),

  ("Pick_congr",
    "(!(x)[| x: A |] ==> [| P(x) <-> Q(x) |])  ==>    \ 
\    [| EXISTS x:A.P(x) |]  ==>  [| [ PICK x:A.P(x) = PICK x:A.Q(x) : A ] |]"),

  ("Pick_intr", "[| EXISTS x:A.P(x) |] ==> [| P(PICK x:A.P(x)) |]"),


  (** Definitions of Classes*)
  ("member_def",	"a<:S == form(S`a)"),
  ("subset_def", 	"subset(A,S,T) == ALL z:A. z<:S --> z<:T"),
  ("un_def",  		"un(A,S,T) == lam z:A. term(z<:S | z<:T)"),

  ("int_def",  		"int(A,S,T) == lam z:A. term(z<:S & z<:T)"),
  ("union_def",
      "union(A,F) == lam z:A. term(EXISTS S:A->bool. S<:F & z<:S)"),
  ("inter_def",
      "inter(A,F) == lam z:A. term(ALL S:A->bool. S<:F --> z<:S)"),
  ("pow_def",
      "pow(A,S) == lam T:A. term(subset(A,T,S))"),


  (** Definitions of types*)

  (*the types "void" and "unit"*)
  ("void_def",	"void == {p: bool. form(False)}"),
  ("unit_def",	"unit == {p: bool. [p=True:bool]}"),

  (*unions: the type A+B *)
  ("plus_def",
    "A+B == {w: (A->bool) * (B->bool). \
\		(EXISTS x:A. [w = Inl(A,B,x) : (A->bool) * (B->bool)]) | \
\		(EXISTS y:B. [w = Inr(A,B,y) : (A->bool) * (B->bool)]) }"),

  ("Inl_def",	"Inl(A,B,a) == <lam x:A.term([ a = x : A ]), lam y:B.False>"),
  ("Inr_def",	"Inr(A,B,b) == <lam x:A.False, lam y:B.term([ b = y : B ])>"),
  ("when_def", 
    "when(A,B,C,p,c,d) == PICK z:C.  \
\	(ALL x:A. [ p = Inl(A,B,x) : A+B ] --> [ z = c(x) : C ]) & \
\	(ALL y:B. [ p = Inr(A,B,y) : A+B ] --> [ z = d(y) : C ])"  )];

val sign = Thm.sign_of thy;
val ax = Thm.get_axiom thy;

(*INSERT-RULENAMES -- file produced by make-rulenames*)
end;

