(*  Title: CTT/ruleshell
    Author: Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Constructive Type Theory rules

!!!After updating, rebuild  ".rules.ML"  by calling make-rulenames!!! 

Equality forms of rules that follow by reflexivity are omitted.
*)

signature CTT_RULE =
  sig
  structure Thm : THM
  val sign: Thm.Sign.sg
  val thy: Thm.theory
(*INSERT-RULESIG -- file produced by make-rulenames*)
  end;


functor CTT_RuleFun (structure CTT_Syntax: CTT_SYNTAX and Thm: THM
	sharing CTT_Syntax.Syntax = Thm.Sign.Syntax) : CTT_RULE = 
struct
structure Thm = Thm;



val thy = Thm.enrich_theory Thm.pure_thy "CTT"
    (["exp","type","prop"], CTT_Syntax.const_decs,  CTT_Syntax.syn)
[ 
  (*Reduction: a weaker notion than equality;  a hack for simplification.
    reduce(a,b) means either that  a=b:A  for some A or else that "a" and "b" 
    are textually identical.*)

  (*does not verify a:A!  Sound because only trans_red uses a reduce premise
    No new theorems can be proved about the standard judgements.*)
  ("refl_red", "[| reduce(a,a) |]"  ),

  ("red_if_equal", "[| a = b : A |] ==> [| reduce(a,b) |]"),

  ("trans_red", "[| a = b : A |] ==> [| reduce(b,c) |] ==> [| a = c : A |]"),



  (*Reflexivity*)

  ("refl_type", "[| A type |] ==> [| A = A |]" ),
 
  ("refl_elem", "[| a : A |] ==> [| a = a : A |]"  ),


  (*Symmetry*)

  ("sym_type",  "[| A = B |] ==> [| B = A |]"  ),

  ("sym_elem",  "[| a = b : A |] ==> [| b = a : A |]"  ),


  (*Transitivity*)

  ("trans_type",
    "[| A = B |] ==> [| B = C |] ==> [| A = C |]"),

  ("trans_elem",
    "[| a = b : A |] ==> [| b = c : A |] ==> [| a = c : A |]"),

  ("equal_types",
    "[| a : A |] ==> [| A = B |] ==> [| a : B |]"),

  ("equal_types_long",
    "[| a = b : A |] ==> [| A = B |] ==> [| a = b : B |]"),


  (*Substitution*)

  ("subst_type", 
    "[| a : A |] ==> (!(z)[| z:A |] ==> [| B(z) type |]) ==> [| B(a) type |]" ),

  ("subst_type_long",
    "[| a = c : A |] ==> (!(z)[| z:A |] ==> [| B(z) = D(z) |])  ==>   \
\    [| B(a) = D(c) |]"   ),

  ("subst_elem",
    "[| a : A |] ==> (!(z)[| z:A |] ==> [| b(z):B(z) |]) ==> [| b(a):B(a) |]"),

  ("subst_elem_long",
    "[| a = c : A |] ==> (!(z)[| z:A |] ==> [| b(z) = d(z) : B(z) |])   ==>   \
\    [| b(a) = d(c) : B(a) |]"   ),


  (*The type N*)

  ("N_form", "[| N type |]"  ),

  ("N_intr0", "[| 0 : N |]"  ),

  ("N_intr_succ", "[| a : N |] ==> [| succ(a) : N |]"  ),


  ("N_intr_succ_long",
    "[| a = b : N |] ==> [| succ(a) = succ(b) : N |]"  ),

  ("N_elim",
    "[| p : N |] ==> [| a : C(0) |]  ==>  \
\    (!(u)[| u: N |] ==> !(v)[| v: C(u) |] ==> [| b(u,v): C(succ(u)) |]) ==> \
\    [| rec(p,a,b) : C(p) |]"  ),

  ("N_elim_long",
    "[| p = q : N |]   ==>  [| a = c : C(0) |]   ==>  \ 
\  (!(u)[| u: N |] ==> !(v)[| v: C(u) |] ==> [| b(u,v)=d(u,v): C(succ(u)) |]) \
\    ==>  [| rec(p,a,b) = rec(q,c,d) : C(p) |]"  ),

  ("N_comp0",
    "[| a : C(0) |]   ==> \
\    (!(u)[| u: N |] ==> !(v)[| v: C(u) |] ==> [| b(u,v) : C(succ(u)) |])   \ 
\     ==> [| rec(0,a,b) = a : C(0) |]"  ),

  ("N_comp_succ",
    "[| p : N |]    ==>  [| a : C(0) |]  ==> \
\    (!(u)[| u: N |] ==> !(v)[| v: C(u) |] ==> [| b(u,v): C(succ(u)) |]) ==> \ 
\    [| rec(succ(p),a,b) = b(p, rec(p,a,b)) : C(succ(p)) |]"  ),


  (*The fourth Peano axiom.  See page 91 of Martin-Lof's book*)
  ("zero_ne_succ",
    "[| a: N |] ==> [| 0 = succ(a) : N |] ==> [| 0: F |]"  ),




  (*The Product of a family of types*)

  ("Prod_form",
    "[| A type |] ==> (!(w)[| w: A |] ==> [| B(w) type |])  ==>    \ 
\    [| Prod(A,B) type |]"  ),


  ("Prod_form_long",
    "[| A = C |] ==> (!(w)[| w: A |] ==> [| B(w) = D(w) |])  ==>    \ 
\    [| Prod(A,B) = Prod(C,D) |]"  ),


  ("Prod_intr",
    "[| A type |] ==>  (!(w)[| w: A |] ==> [| b(w) : B(w) |])  ==>    \ 
\    [| lambda(b) : Prod(A,B) |]"  ),


  ("Prod_intr_long",
    "[| A type |] ==> (!(w)[| w: A |] ==> [| b(w) = c(w) : B(w) |]) \ 
\    ==> [| lambda(b) = lambda(c) : Prod(A,B) |]"  ),


  ("Prod_elim",
    "[| p : Prod(A,B) |] ==> [| a : A |]     \ 
\  ==> [| p ` a : B(a) |]"  ),


  ("Prod_elim_long",
    "[| p = q : Prod(A,B) |] ==> [| a = b : A |]    \ 
\  ==>  [| p ` a = q ` b : B(a) |]"  ),


  ("Prod_comp",
    "[| a : A |] ==> (!(w)[| w: A |] ==> [| b(w) : B(w) |])   \ 
\  ==> [| lambda(b) ` a = b(a) : B(a) |]"  ),

  ("Prod_comp2",
    "[| p : Prod(A,B) |] ==> [| (lam x. p`x) = p : Prod(A,B) |]"  ),


  ("Fun_def", "A-->B == Prod(A, %(z)B)"),
 




  (*The Sum of a family of types*)

  ("Sum_form",
    "[| A type |] ==> (!(w)[| w: A |] ==> [| B(w) type |])  ==>    \ 
\    [| Sum(A,B) type |]"  ),


  ("Sum_form_long", 
    "[| A = C |] ==> (!(w)[| w: A |] ==> [| B(w) = D(w) |])  ==>    \ 
\    [| Sum(A,B) = Sum(C,D) |]"  ),


  ("Sum_intr",
    "[| a : A |] ==>  [| b : B(a) |]  ==>    \ 
\    [| <a,b> : Sum(A,B) |]"  ),


  ("Sum_intr_long",
    "[| a = c : A |] ==>  [| b = d : B(a) |]  ==>    \ 
\    [| <a,b> = <c,d> : Sum(A,B) |]"  ),


  ("Sum_elim",
    "[| p : Sum(A,B) |]  ==>    \ 
\    (!(x)[| x:A |] ==> !(y)[| y:B(x) |] ==> [| c(x,y) : C(<x,y>) |])   ==> \
\    [| split(p,c) : C(p) |]"  ),


  ("Sum_elim_long",
    "[| p = q : Sum(A,B) |]  ==>    \ 
\    (!(x)[| x:A |] ==> !(y)[| y:B(x) |] ==> [| c(x,y)=d(x,y) : C(<x,y>) |]) \ 
\    ==>  [| split(p,c) = split(q,d) : C(p) |]"  ),


  ("Sum_comp",
    "[| a : A |]  ==>  [| b : B(a) |]  ==>    \ 
\    (!(x)[| x:A |] ==> !(y)[| y:B(x) |] ==> [| c(x,y) : C(<x,y>) |]) ==> \
\    [| split(<a,b>, c) = c(a,b) : C(<a,b>) |]"  ),


  ("Times_def",  "A*B == Sum(A, %(z)B)"),
  ("fst_def",  "fst(a) == split(a, %(x,y)x)"),
  ("snd_def",  "snd(a) == split(a, %(x,y)y)"),


  (*The sum of two types*)

  ("Plus_form",
    "[| A type |] ==> [| B type |] ==> [| A+B type |]"  ),


  ("Plus_form_long",
    "[| A = C |] ==> [| B = D |] ==> [| A+B = C+D |]"  ),


  ("Plus_intr_inl",
    "[| a : A |] ==> [| B type |] ==> [| inl(a) : A+B |]"),


  ("Plus_intr_inl_long",
    "[| a = c : A |] ==> [| B type |]  ==>  [| inl(a) = inl(c) : A+B |]"  ),


  ("Plus_intr_inr",
    "[| A type |] ==> [| b : B |] ==> [| inr(b) : A+B |]"  ),


  ("Plus_intr_inr_long",
    "[| A type |] ==> [| b = d : B |]  ==>  [| inr(b) = inr(d) : A+B |]"  ),


  ("Plus_elim",
    "[| p : A+B |]  ==> \
\    (!(x)[| x: A |] ==> [| c(x) : C(inl(x)) |])  ==> \
\    (!(y)[| y: B |] ==> [| d(y) : C(inr(y)) |])  ==> \
\    [| when(p,c,d) : C(p) |]"  ),
 

  ("Plus_elim_long",
    "[| p = q : A+B |]  ==> \
\    (!(x)[| x: A |] ==> [| c(x) = e(x) : C(inl(x)) |])  ==> \
\    (!(y)[| y: B |] ==> [| d(y) = f(y) : C(inr(y)) |])  ==> \
\    [| when(p,c,d) = when(q,e,f) : C(p) |]"  ),
 

  ("Plus_comp_inl",
    "[| a : A |]  ==> \
\    (!(x)[| x: A |] ==> [| c(x) : C(inl(x)) |])  ==> \
\    (!(y)[| y: B |] ==> [| d(y) : C(inr(y)) |])  ==> \
\    [| when(inl(a),c,d) = c(a) : C(inl(a)) |]"  ),
 

  ("Plus_comp_inr",
    "[| b : B |]  ==> \
\    (!(x)[| x: A |] ==> [| c(x) : C(inl(x)) |])  ==> \
\    (!(y)[| y: B |] ==> [| d(y) : C(inr(y)) |])  ==> \
\    [| when(inr(b),c,d) = d(b) : C(inr(b)) |]"  ),
 


  (*The type Eq*)


  ("Eq_form",
    "[| A type |] ==> [| a : A |] ==> [| b : A |] ==> [| Eq(A,a,b) type |]"),


  ("Eq_form_long",
    "[| A = B |] ==> [| a = c: A |] ==> [| b = d : A |] ==>  \
\    [| Eq(A,a,b) = Eq(B,c,d) |]"  ),


  ("Eq_intr", "[| a = b : A |] ==> [| eq : Eq(A,a,b) |]"),


  ("Eq_elim", "[| p : Eq(A,a,b) |] ==> [| a = b : A |]"     ),

  (*By equality of types, can prove C(p) from C(eq), an elimination rule*)
  ("Eq_comp", "[| p : Eq(A,a,b) |] ==> [| p = eq : Eq(A,a,b) |]" ),


  (*The type F*)

  ("F_form", "[| F type |]"  ),

  ("F_elim", "[| p: F |] ==> [| C type |] ==> [| contr(p) : C |]" ),
 
  ("F_elim_long",
    "[| p = q : F |] ==> [| C type |]  ==> \
\    [| contr(p) = contr(q) : C |]"  ),


  (*The type T 
     Martin-Lof's book (page 68) discusses elimination and computation.
     Elimination can be derived by computation and equality of types,
     but with an extra premise C(x) type [| x:T |].
     Also computation can be derived from elimination. *)

  ("T_form", "[| T type |]"  ),

  ("T_intr", "[| tt : T |]"  ),

  ("T_elim",
    "[| p : T |] ==> [| c : C(tt) |]  ==> \
\    [| c : C(p) |]"  ),
 
  ("T_elim_long",
    "[| p = q : T |] ==>  [| c = d : C(tt) |]  ==> \
\    [| c = d : C(p) |]"  ),
 
  ("T_comp",
    "[| p : T |] ==> [| p = tt : T |]" )

];


val sign = Thm.sign_of thy;

val ax = Thm.get_axiom thy;

(*INSERT-RULENAMES -- file produced by make-rulenames*)

end;

