(*  Title: 	CTT/resolve
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   1988  University of Cambridge

Constructive Type Theory resolution tactics*)

signature CTT_RESOLVE = 
sig
  structure Tactic : TACTIC
  val asm_rl: Tactic.Thm.thm
  val ASSUME: (int -> Tactic.tactic) -> int -> Tactic.tactic
  val basic_defs: Tactic.Thm.thm list
  val bi_mp_tac: int -> Tactic.tactic
  val comp_rls: Tactic.Thm.thm list
  val element_rls: Tactic.Thm.thm list
  val elim_long_rls: Tactic.Thm.thm list
  val elim_rls: Tactic.Thm.thm list
  val eqintr_tac: Tactic.tactic
  val equal_tac: Tactic.Thm.thm list -> Tactic.tactic
  val form_long_rls: Tactic.Thm.thm list
  val form_rls: Tactic.Thm.thm list
  val form_tac: Tactic.tactic
  val intr_long2_rls: Tactic.Thm.thm list
  val intr_long_rls: Tactic.Thm.thm list
  val intr_rls: Tactic.Thm.thm list
  val intr_tac: Tactic.Thm.thm list -> Tactic.tactic
  val mp_tac: int -> Tactic.tactic
  val N_elim_tac: string -> int -> Tactic.tactic
  val pc_tac: Tactic.Thm.thm list -> int -> Tactic.tactic
  val Plus_elim_tac: string -> int -> Tactic.tactic
  val reduction_rls: Tactic.Thm.thm list
  val replace_type: Tactic.Thm.thm
  val routine_rls: Tactic.Thm.thm list   
  val routine_tac: Tactic.Thm.thm list -> Tactic.Thm.thm list -> int -> Tactic.tactic   
  val safe_brls: (bool * Tactic.Thm.thm) list
  val safe_tac: Tactic.Thm.thm list -> int -> Tactic.tactic
  val step_tac: Tactic.Thm.thm list -> int -> Tactic.tactic
  val subgoals_of_brl: bool * Tactic.Thm.thm -> int
  val subst_eqtyparg: Tactic.Thm.thm
  val subst_prod_elim: Tactic.Thm.thm
  val Sum_elim_fst: Tactic.Thm.thm
  val Sum_elim_snd: Tactic.Thm.thm
  val Sum_elim_tac: string -> int -> Tactic.tactic
  val Sum_intr_long2: Tactic.Thm.thm
  val test_assume_tac: int -> Tactic.tactic
  val typechk_tac: Tactic.Thm.thm list -> Tactic.tactic
  val unsafe_brls: (bool * Tactic.Thm.thm) list
end;


functor CTT_ResolveFun
    (structure CTT_Syntax: CTT_SYNTAX 
           and CTT_Rule: CTT_RULE
           and Logic: LOGIC 
           and Conv: CONV
	sharing CTT_Syntax.Syntax = CTT_Rule.Thm.Sign.Syntax
	    and Conv.Tactic.Thm = CTT_Rule.Thm) : CTT_RESOLVE = 
struct
structure Tactic = Conv.Tactic;
local open Conv Tactic Thm CTT_Rule CTT_Syntax
in


(*Rule P/P, solves assumptions using biresolve_tac; BELONGS IN TACTIC!!! *)
val asm_rl = trivial (Sign.read_cterm CTT_Rule.sign ("?phi",Aprop));


(*Formation rules*)
val form_rls = [N_form, Prod_form, Sum_form, Plus_form, 
		Eq_form, F_form, T_form];

val form_long_rls = 
    [Prod_form_long, Sum_form_long, Plus_form_long, Eq_form_long];

 
(*Introduction rules
  OMITTED: Eq_intr, because its premise is an eqelem, not an elem*)
val intr_rls = 
    [N_intr0, N_intr_succ, Prod_intr, Sum_intr, 
     Plus_intr_inl, Plus_intr_inr, T_intr];

val intr_long_rls = 
    [N_intr_succ_long, Prod_intr_long, Sum_intr_long, 
     Plus_intr_inl_long, Plus_intr_inr_long];


(*Elimination rules
  OMITTED: Eq_elim, because its conclusion is an eqelem,  not an elem
           T_elim, because it does not involve a constructor *)
val elim_rls = [N_elim, Prod_elim, Sum_elim, Plus_elim, F_elim];

val elim_long_rls =
    [N_elim_long, Prod_elim_long, Sum_elim_long, Plus_elim_long, F_elim_long];

(*eq_comp,  T_comp not included because they cause rewriting to loop:
  p = un = un = un = ... *)
val comp_rls = 
    [N_comp0, N_comp_succ, 
     Prod_comp, Sum_comp, Plus_comp_inl, Plus_comp_inr];

(*rules with conclusion a:A, an elem judgement*)
val element_rls = intr_rls @ elim_rls;


(*Definitions are (meta)equality axioms*)
val basic_defs = [Fun_def,Times_def,fst_def,snd_def];



(*Compare with standard version: B is applied to UNSIMPLIFIED expression! *)
val Sum_intr_long2 = prove_goal CTT_Rule.thy
    "[| c=a : A |] ==> [| d=b : B(a) |] ==> [| <c,d> = <a,b> : Sum(A,B) |]"
 (fn asms=>
  [ (resolve_tac [sym_elem] 1),
    (resolve_tac [Sum_intr_long] 1),
    (ALLGOALS (resolve_tac [sym_elem])),
    (ALLGOALS (resolve_tac asms)) ]);


val intr_long2_rls = 
    [N_intr_succ_long, Prod_intr_long, Sum_intr_long2, 
     Plus_intr_inl_long, Plus_intr_inr_long];


(*Exploit p:Prod(A,B) to create the assumption z:B(a).  
  A more natural form of product elimination. *)
val subst_prod_elim = prove_goal CTT_Rule.thy
    "[| p: Prod(A,B) |] ==> [| a : A |] ==>  \
\    (!(z)([| z : B(a) |] ==> [| c(z) : C(z) |])) ==> [| c(p`a) : C(p`a) |]"
 (fn asms=>
  [ (REPEAT (resolve_tac (asms@[Prod_elim]) 1)) ]);



fun is_rigid_elem (Const("Elem",_) $ a $ _) = not (is_var (head_of a))
  | is_rigid_elem _ = false;

(*Try solving a:A by assumption provided a is rigid!*) 
val test_assume_tac = SUBGOAL(fn (prem,i) =>
    if is_rigid_elem (Logic.strip_assums_concl prem)
    then  assume_tac i  else  no_tac);

fun ASSUME tf i = test_assume_tac i  ORELSE  tf i;


(*Repeatedly tries to solve subgoals using tf. *)

(*For simplification: type formation and checking,
  but no equalities between terms*)
val routine_rls = form_rls @ form_long_rls @ [refl_type] @ element_rls;

fun routine_tac rls asms = ASSUME (compat_resolve_tac (asms @ rls) 4);


(*Solve all subgoals "A type" using formation rules. *)
val form_tac = REPEAT_FIRST (compat_resolve_tac(form_rls) 1);


(*Type checking: solve a:A (a rigid, A flexible) by intro and elim rules. *)
fun typechk_tac asms =
  let val tac = compat_resolve_tac (asms @ form_rls @ element_rls) 3
  in  REPEAT_FIRST (ASSUME tac)  end;


(*Solve a:A (a flexible, A rigid) by introduction rules. 
  Cannot use stringtrees (compat_resolve_tac) since
  goals like ?a:SUM(A,B) have a trivial head-string *)
fun intr_tac asms =
  let val tac = filt_resolve_tac(asms@form_rls@intr_rls) 1
  in  REPEAT_FIRST (ASSUME tac)  end;


(*Equality proving: solve a=b:A (where a is rigid) by long rules. *)
fun equal_tac asms =
  let val rls = asms @ form_rls @ element_rls @ intr_long_rls @
                elim_long_rls @ [refl_elem]
  in  REPEAT_FIRST (ASSUME (compat_resolve_tac rls 3))  end;




(*Simplification.....................*)


(*To simplify the type in a goal*)
val replace_type = prove_goal CTT_Rule.thy
    "[| B = A |] ==> [| a : A |] ==> [| a : B |]"
 (fn asms=>
  [ (resolve_tac [equal_types] 1),
    (resolve_tac [sym_type] 2),
    (ALLGOALS (resolve_tac asms)) ]);


(*Simplify the parameter of a unary type operator.*)
val subst_eqtyparg = prove_goal CTT_Rule.thy
    "[| a=c : A |] ==> (!(z)[| z:A |] ==> [| B(z) type |]) ==> [| B(a)=B(c) |]"
 (fn asms=>
  [ (resolve_tac [subst_type_long] 1),
    (resolve_tac [refl_type] 2),
    (ALLGOALS (resolve_tac asms)),
    (assume_tac 1) ]);


(*Make a reduction rule for simplification.
  A goal a=c becomes b=c, by virtue of a=b *)
fun resolve_trans rl = trans_elem RES rl;

(*Simplification rules for Constructive Type Theory*)
val reduction_rls = map resolve_trans comp_rls;


(*Converts each goal "e : Eq(A,a,b)" into "a=b:A" for simplification.
  Uses other intro rules to avoid changing flexible goals.*)
val eqintr_tac = REPEAT_FIRST (ASSUME (filt_resolve_tac(Eq_intr::intr_rls) 1));



(*Tactics that instantiate CTT-rules.
  Vars in the given terms will be incremented! 
  The (resolve_tac [Eq_elim] i) lets them apply to equality judgements.*)

fun N_elim_tac (sp: string) i = 
  TRY (resolve_tac [Eq_elim] i) THEN 
  res_inst_tac [ ("p",sp,Aexp) ] N_elim i;

fun Sum_elim_tac (sp: string) i = 
  TRY (resolve_tac [Eq_elim] i) THEN 
  res_inst_tac [ ("p",sp,Aexp) ] Sum_elim i;

fun Plus_elim_tac (sp: string) i = 
  TRY (resolve_tac [Eq_elim] i) THEN 
  res_inst_tac [ ("p",sp,Aexp) ] Plus_elim i;



(*Predicate logic reasoning,  WITH THINNING!  Procedures adapted from NJ. *)

(*Finds f:Prod(A,B) and a:A in the assumptions, concludes there is z:B(a) *)
fun mp_tac i = 
    resolve_tac [subst_prod_elim] i  THEN
    assume_tac i  THEN  assume_tac i;

(*Finds P-->Q and P in the assumptions, replaces implication by Q *)
fun bi_mp_tac i = biresolve_tac [(true,subst_prod_elim)] i  THEN  assume_tac i;


(*The number of new subgoals produced by the brule*)
fun subgoals_of_brl (true,rule) = length (prems_of rule) - 1
  | subgoals_of_brl (false,rule) = length (prems_of rule);

(*Less-than test: for sorting to minimize number of new subgoals*)
fun lessb (brl1,brl2) = subgoals_of_brl brl1 < subgoals_of_brl brl2;

(*"safe" when regarded as predicate calculus rules*)
val safe_brls = sort lessb 
    [ (true,F_elim), (true,asm_rl), 
      (false,Prod_intr), (true,Sum_elim), (true,Plus_elim) ];

val unsafe_brls =
    [ (false,Plus_intr_inl), (false,Plus_intr_inr), (false,Sum_intr), 
      (true,subst_prod_elim) ];

(*0 subgoals vs 1 or more*)
val (safe0_brls, safep_brls) =
    partition (apl(0,op=) o subgoals_of_brl) safe_brls;

fun safestep_tac asms i =
    form_tac  ORELSE  
    resolve_tac asms i  ORELSE
    biresolve_tac safe0_brls i  ORELSE  bi_mp_tac i  ORELSE
    DETERM (biresolve_tac safep_brls i);

fun safe_tac asms i = DEPTH_SOLVE_1 (safestep_tac asms i); 

fun step_tac asms = safestep_tac asms  ORELSE'  biresolve_tac unsafe_brls;

(*Fails unless it solves the goal!*)
fun pc_tac asms = DEPTH_SOLVE_1 o (step_tac asms);



(*The elimination rules for fst/snd*)

val Sum_elim_fst = prove_goal CTT_Rule.thy 
    "[| p : Sum(A,B) |] ==> [| fst(p) : A |]"
 (fn asms=>
  [ (rewrite_goals_tac basic_defs),
    (resolve_tac elim_rls 1),
    (REPEAT (pc_tac asms 1)),
    (fold_tac basic_defs) ]);


(*The first premise must be p:Sum(A,B) !!*)
val Sum_elim_snd = prove_goal CTT_Rule.thy 
    "[| p : Sum(A,B) |] ==> [| A type |] ==>  \
\    (!(x)[| x:A |] ==> [| B(x) type |]) ==> \
\    [| snd(p) : B(fst(p)) |]"
 (fn asms=>
  [ (rewrite_goals_tac basic_defs),
    (resolve_tac elim_rls 1),
    (resolve_tac asms 1),
    (resolve_tac [replace_type] 1),
    (resolve_tac [subst_eqtyparg] 1),   (*like B(x) equality formation?*)
    (resolve_tac comp_rls 1),
    (typechk_tac asms),
    (fold_tac basic_defs) ]);

end;
end;
