(*  Title: 	term.ML
    Author: 	Lawrence C Paulson, Cambridge University Computer Laboratory
    Copyright   Cambridge University 1986
*)


(*Arities, terms, and basic operations on them*)


(*Arities are simple types as in typed lambda-calculus
      ground arities  and  function arities
  a function can only be applied to an argument of corresponding arity
  the word "arity" is from Intuitionistic Type Theory.*)

infixr 5 -->;
datatype arity = Ground of string  |  op --> of arity * arity;

(*for [a1,...,an]--->a  returns the list  [a1,a2,...,an]*)
fun binder_arities (Ground _)   =  []
  | binder_arities (ary1-->ary2) = ary1 :: binder_arities ary2;

(*for [a1,...,an]--->a  returns the string for a*)
fun body_arity (Ground aname) = aname : string
  | body_arity  (_  -->  ary)  = body_arity ary;

fun strip_arity ary : arity list * string =
  (binder_arities ary, body_arity ary);


(*handy for multiple args: [a1,...,an]--->a  gives  a1-->(a2--> ... -->a)*)
infixr --->;
val op ---> = itlist_right (op -->);


(*terms.  Bound variables are indicated by depth number.
  Parameters, (scheme) variables and constants have names.
  An term is "closed" if there every bound variable of level "lev"
  is enclosed by at least "lev" abstractions. 

  It is possible to create nonsense terms containing dangling bound vars
  or arity mismatches.  But such terms are not allowed in rules. *)


(*Indexnames can be quickly renamed by adding an offset to the integer part,
  for resolution.*)
type indexname = string*int;


infix 9 $;  (*application binds tightly!*)
datatype term = 
    Const of string * arity
  | Param of indexname * arity 
  | Bound of int
  | Var   of indexname * arity
  | Abs   of string*arity*term
  | op $  of term*term;


(*For system errors involving terms*)
exception term_error : string * term list;

(*For user errors involving arity mismatches*)
exception arity : string * arity list * term list;


(*compute the arity of the term, checking that combinations are well-typed
  bound_arys = [a0,a1,...] holds arities of bound variables 0, 1, ...*)
fun arity_of1 (bound_arys: arity list, tm: term) : arity = case tm of
    Const (_,ary) => ary
  | Param (_,ary) => ary
  | Bound bno => nth_elem (bno,bound_arys)  
  		  handle list=> raise term_error with ("arity_of1", [tm])
  | Var (_,ary) => ary 
  | Abs (_,ary,body) =>  ary -->  arity_of1 (ary::bound_arys, body)
  | rator$rand => 
      let val arand  = arity_of1 (bound_arys, rand)
          and arator = arity_of1 (bound_arys, rator)
      in case arator of
	    Ground _ => 
	      raise arity with ("Rator must have function arity",
	                        [arator,arand], [rator,rand])
	  | arand'-->ary =>
	      if arand'=arand then ary 
	      else raise arity with
	             ("Rator/rand arity mismatch", [arand',arand], [tm])
      end;

fun arity_of tm : arity = arity_of1 ([],tm);


(*If arity of term equals "ary" then returns term, else raises exception.*)
fun check_arity_term ary tm : term =
  let val ary' = arity_of tm
  in  if ary' = ary then tm  
      else  raise arity with ("Unexpected arity", [ary,ary'], [tm]) 
  end;


(* ([x1,...,xn], t)   ======>   (x1,...,xn)t *)
fun list_abs (vars: (string*arity)list, body: term) = case vars of
    [] => body  |  (id,ary)::vars' => Abs(id, ary, list_abs(vars',body));


val list_comb : term * term list -> term = itlist_left (op $);


fun strip_comb tm = 
    let fun stripc (rator $ rand, rands) = stripc (rator, rand::rands)
        |   stripc  x =  x 
    in  stripc(tm,[])  end;


(* for   f(t1,...,tn)  returns  f , which is never a combination *)
fun head_of (rator$rand) = head_of rator
  | head_of tm = tm;


(*  (x1,...,xn) t   ======>   t  *)
fun strip_abs_body (Abs(_,_,tm))  =  strip_abs_body tm  
  | strip_abs_body tm  =  tm;

(*  (x1,...,xn) t   ======>   [x1, ..., xn]  *)
fun strip_abs_vars (Abs(id,ary,tm))  =  (id,ary) :: strip_abs_vars tm 
  | strip_abs_vars tm  =  [];


(* OPERATIONS ON INDEXNAMES *)

fun string_of_xname (id,0)   =  id
  | string_of_xname (id,idx) =  id ^ string_of_int idx;


(*increments a term's non-local bound variables
  required when moving a term within abstractions
     binc is  increment for bound variables
     lev  is  level at which a bound variable is considered 'dangling'*)
fun incr_boundvars1 (binc,lev,tm) : term = case tm of
    Bound bno => if bno>=lev then Bound(bno+binc) else tm 
  | Abs(name,ary,body) => Abs(name, ary, incr_boundvars1(binc,lev+1,body))
  | rator $ rand => 
      incr_boundvars1(binc,lev,rator)  $  incr_boundvars1(binc,lev,rand)
  | _ => tm;

fun incr_boundvars  0   tm = tm
  | incr_boundvars binc tm = incr_boundvars1(binc,0,tm);


(*  increment indexnames in the term (all uvars and params)  *)
fun incr_indexes 0 = I : term->term
  | incr_indexes inc = 
      let fun incr (Param ((s,idx), ary)) = Param((s, idx+inc), ary)
	    | incr (Var ((s,idx), ary)) = Var((s, idx+inc), ary)
	    | incr (Abs (name,ary,body)) = Abs (name, ary, incr body)
	    | incr (rator $ rand) = incr rator $ incr rand
	    | incr tm = tm
      in  incr  end;


(*Predicate: is a bound variable reference to level n present?
  even (Bound 0) is a dangling bound variable at level 0 *)
fun dangling_bound (lev:int, tm: term) : bool = case tm of
    Bound n => (n = lev) 
  | Abs (_,_,body) =>  dangling_bound (lev+1, body)
  | rator $ rand => 
      dangling_bound (lev,rator) orelse dangling_bound (lev,rand)
  | _ => false;


(*Substitutes arguments for dangling bound variables
  DOES IT WORK FOR >1 ARGS?  RECODE!
  Beta-reduction of arg0...arg(n-1) into tm replacing (Bound i) with (argi).
  Dangling bound variables are changed by "binc", to
     compensate for the introduction/disappearance of surrounding lambdas.
*)
fun subst_bounds (binc: int, args: term list, tm) : term = 
  let fun subst (Bound bno) =
	    (case (nth_tail (bno,args)) of
		[]     => Bound(bno+binc)  (*dangling: change it*)
	     | arg::_ => incr_boundvars bno arg)
	| subst (Abs(name,ary,body)) =
	    Abs(name, ary,  subst_bounds (binc, Bound 0 :: args, body))
	| subst (rator $ rand) =  subst rator  $  subst rand
	| subst tm = tm
  in   case args of [] => tm  | _ => subst tm  end;


(*Tests whether 2 terms are alpha-convertible.
  does not check arities: each identifier should have only one arity.*)
fun alphaconv (Const (id1,_), Const (id2,_))  =  id1=id2
  | alphaconv (Param (name1,_), Param (name2,_)) =  name1=name2
  | alphaconv (Var (name1,_), Var (name2,_))  =  name1=name2
  | alphaconv (Bound bno1, Bound bno2)  =   bno1=bno2
  | alphaconv (Abs(_,_,body1), Abs(_,_,body2))  =  alphaconv (body1,body2)
  | alphaconv (rator1 $ rand1, rator2 $ rand2) =
	  (alphaconv (rator1,rator2) andalso alphaconv (rand1,rand2))
  | alphaconv _  =  false;


fun tassoc ([], key: term) = None
  | tassoc ((keyi,xi)::pairs, key) =
      if alphaconv(key,keyi) then Some xi  else tassoc (pairs,key);



(*membership test for term lists*)
infix tmem;
fun t tmem []  =  false  
  | t tmem (u::us)  =  alphaconv(t,u) orelse (t tmem us);

fun tdistinct1 (seen, []) = rev seen
  | tdistinct1 (seen, tm::tms) =
      if tm tmem seen then tdistinct1 (seen, tms)
      		      else tdistinct1 (tm::seen, tms);

(*return a list of distinct terms*)
fun tdistinct tms = tdistinct1([],tms);


(*Substitution of atomic ti for ui,  instl = [(t1,u1), ..., (tn,un)]. 
  NOT USED AT PRESENT!*)
fun subst_term (instl: (term*term) list) : term -> term =
  let fun subst (Abs(name,ary,body)) = Abs(name, ary, subst body)
        | subst (rator$rand) = subst rator $ subst rand
	| subst t = (case tassoc(instl,t) of
		Some u => u  |  None => t)
  in  case instl of [] => I  |  _ => subst  end;



(*a partial ordering (not reflexive) for atomic terms*)
fun atless (Const (id1,_), Const (id2,_))  =  id1<id2
  | atless (Param ((s1,idx1),_), Param ((s2,idx2),_)) =
        s1<s2 orelse (s1=s2 andalso idx1<idx2)
  | atless (Var ((s1,idx1),_), Var ((s2,idx2),_))  =  
        s1<s2 orelse (s1=s2 andalso idx1<idx2)
  | atless (Bound bno1, Bound bno2)  =   bno1<bno2
  | atless _  =  false;


(*insert atomic term into partially sorted list, trying to suppress duplicates*)
fun insert_aterm (t,us) : term list =
  let fun ins [] = [t]
        | ins (us as u::us') = 
	      if atless(t,u) then t::us
	      else if t=u then us (*duplicate*)
	      else u :: ins(us')
  in  ins us  end;


(*Union of sets of atomic terms, represented by sorted lists*)
val union_aterms: term list * term list -> term list = 
  itlist_right insert_aterm;


(*Accumulates the "pvars" (params and uvars) in the term,
  suppressing duplicates*)
fun add_term_pvars (tm, vars: term list) : term list = case tm of
    Var   _ => insert_aterm(tm,vars)
  | Param _ => insert_aterm(tm,vars)
  | Abs (_,_,body) => add_term_pvars(body,vars)
  | rator $ rand =>  add_term_pvars (rator, add_term_pvars(rand, vars))
  | _ => vars;


(*the list of pvars in a list of terms*)
fun pvars_of_terms tms : term list = itlist_right add_term_pvars (tms,[]);
