(*

   Proof of the Jordan curve theorem
   Format: HOL-LIGHT (OCaml version 2003)
   File started April 20, 2004
   Completed January 19, 2005
   Author: Thomas C. Hales

   The proof follows
   Carsten Thomassen
   "The Jordan-Schoenflies theorem and the classification of
    surfaces"
   American Math Monthly 99 (1992) 116 - 130.

   There is one major difference from Thomassen's proof.
   He uses general polygonal jordan curves in the "easy" case of the
   Jordan curve theorem.  This file restricts the "easy" case
   even further to jordan curves that are made of horizontal
   and vertical segments with integer length.

   Thomassen shows finite planar graphs admit polygonal
   embeddings.  This file shows that finite planar graphs such
   that every vertex has degree at most 4 admit
   embeddings with edges that are piecewise horizontal and
   vertical segments of integer length.

   I have apologies:

   1. I'm still a novice and haven't settled on a style.  The
      entire proof is a clumsy experiment.
   2. The lemmas have been ordered by my stream of consciousness.
      The file is long, the dependencies are nontrivial, and reordering
      is best accomplished by an automated tool.

*)


let jordan_def = local_definition "jordan";;
mk_local_interface "jordan";;
prioritize_real();;

let basic_rewrite_bak = basic_rewrites();;
let basic_net_bak = basic_net();;
let PARTIAL_REWRITE_CONV thl =
  GENERAL_REWRITE_CONV true TOP_DEPTH_CONV (basic_net_bak) thl;;
let PARTIAL_REWRITE_TAC thl = CONV_TAC(PARTIAL_REWRITE_CONV thl);;

let reset() = (set_basic_rewrites basic_rewrite_bak);;
extend_basic_rewrites
  (* sets *)
  [(* UNIV *)
   INR IN_UNIV;
   UNIV_NOT_EMPTY;
   EMPTY_NOT_UNIV;
   DIFF_UNIV;
   INSERT_UNIV;
   INTER_UNIV ;
   EQ_UNIV;
   UNIV_SUBSET;
   SUBSET_UNIV;
   (* EMPTY *)
   IN;IN_ELIM_THM';
   (* EMPTY_EXISTS; *)  (* leave EMPTY EXISTS out next time *)
   EMPTY_DELETE;
   INTERS_EMPTY;
   INR NOT_IN_EMPTY;
   EMPTY_SUBSET;
   (* SUBSET_EMPTY; *)  (* leave out *)
   (* INTERS *)
   inters_singleton;
   (* SUBSET_INTER; *)
   (* unions *)
   UNIONS_0;
   UNIONS_1;
  ];;


let DISCH_THEN_REWRITE = (DISCH_THEN (fun t -> REWRITE_TAC[t]));;
let ISUBSET = INR SUBSET;;

(* ------------------------------------------------------------------ *)
(* Logic, Sets, Metric Space Material *)
(* ------------------------------------------------------------------ *)

(* logic *)


(* sets *)
let PAIR_LEMMAv2 = prove_by_refinement(
   `!x (i:A) (j:B). (x = (i,j)) <=> ((FST x = i) /\ (SND x = j))` ,
(* {{{ proof *)
   [
   MESON_TAC[FST;SND;PAIR];
   ]);;
(* }}} *)

let PAIR_SPLIT = prove_by_refinement(
   `!x (y:A#B). (x = y) <=> ((FST x = FST y) /\ (SND x = SND y))` ,
(* {{{ proof *)
   [
   MESON_TAC[FST;SND;PAIR];
   ]);;
(* }}} *)

let single_inter = prove_by_refinement(
  `!(a:A) U. ( ~({a} INTER U = EMPTY) <=> U a)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[INSERT;INTER;EMPTY_EXISTS ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let inters_inter = prove_by_refinement(
  `!(X:A->bool) Y. (X INTER Y) = (INTERS {X,Y})`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `{X,Y} Y` SUBGOAL_TAC;
  REWRITE_TAC[INSERT ];
  DISCH_TAC;
  USE 0 (MATCH_MP delete_inters);
  ASM_REWRITE_TAC[DELETE_INSERT; ];
  COND_CASES_TAC;
  ASM_REWRITE_TAC[INTER;];
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let unions_delete_choice = prove_by_refinement(
  `!(A:(A->bool)->bool). ~(A =EMPTY) ==>
     (UNIONS A = (UNIONS (A DELETE CHOICE A)) UNION (CHOICE A))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[];
  DISCH_ALL_TAC;
  REWRITE_TAC[UNIONS;UNION;DELETE  ];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[];
  TYPE_THEN `A (CHOICE A)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (INR CHOICE_DEF  );
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let image_delete_choice = prove_by_refinement(
  `!(A:(A->bool)) (f:A->B). ~(A= EMPTY) ==>
     (IMAGE f A =
        ((IMAGE f (A DELETE CHOICE A)) UNION {(f (CHOICE A))}))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[];
  DISCH_ALL_TAC;
  REWRITE_TAC[IMAGE;UNION;DELETE];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INSERT ];
  TYPE_THEN `A (CHOICE A)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (INR CHOICE_DEF  );
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let UNIONS_UNION = prove_by_refinement(
  `!(A:(A->bool)->bool) B.
    UNIONS (A UNION B) = (UNIONS A) UNION (UNIONS B)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[UNIONS;UNION];
  IMATCH_MP_TAC EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[];
  MESON_TAC[];
  ]);;
  (* }}} *)

(* reals *)

let half_pos = prove_by_refinement(
  `!x. (&.0 < x) ==> (&.0 < x/(&.2)) /\ (x/(&.2)) < x`,
  (* {{{ proof *)
  [
  MESON_TAC[REAL_LT_HALF2;REAL_LT_HALF1];
  ]);;
  (* }}} *)

(* topology *)
let convex_inter = prove_by_refinement(
  `!S T. (convex S) /\ (convex T) ==> (convex (S INTER T))`,
  (* {{{ proof *)

  [
  REWRITE_TAC[convex;mk_segment;INTER;SUBSET_INTER  ];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  TYPEL_THEN [`x`;`y`] (USE 0 o ISPECL);
  REWR 0;
  TYPEL_THEN [`x`;`y`] (USE 1 o ISPECL);
  REWR 1;
  ]);;

  (* }}} *)

let closed_inter2 = prove_by_refinement(
  `!U (A:A->bool) B. (topology_ U) /\ (closed_ U A) /\ (closed_ U B) ==>
   (closed_ U (A INTER B))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[inters_inter];
  IMATCH_MP_TAC  closed_inter ;
  ASM_REWRITE_TAC[INR INSERT;EMPTY_EXISTS ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let closure_univ = prove_by_refinement(
  `!U (X:A->bool). ~(X SUBSET UNIONS U) ==> (closure U X = UNIV)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  REWRITE_TAC[closure;closed];
  TYPE_THEN `{B | (B SUBSET UNIONS U /\ open_ U (UNIONS U DIFF B)) /\ X SUBSET B} = EMPTY ` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  USE 1 (REWRITE_RULE[EMPTY_EXISTS ]);
  CHO 1;
  ASM_MESON_TAC[SUBSET_TRANS];
  DISCH_THEN_REWRITE;
  ]);;

  (* }}} *)

let closure_inter = prove_by_refinement(
  `!(X:A->bool) Y U.
   (topology_ U)
    ==> ((closure U (X INTER Y) SUBSET
   (closure U X) INTER closure U Y))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  TYPE_THEN `X SUBSET UNIONS  U` ASM_CASES_TAC THEN (TYPE_THEN `Y SUBSET UNIONS  U` ASM_CASES_TAC) THEN TRY(IMP_RES_THEN (fun t -> REWRITE_TAC[t]) closure_univ)  THEN (  IMATCH_MP_TAC  closure_subset );
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  closed_inter2;
  ASM_SIMP_TAC[closure_closed ];
  REWRITE_TAC[INTER;ISUBSET ];
  ASM_MESON_TAC[subset_closure;ISUBSET];
  ASM_MESON_TAC[closure_closed;INTER_SUBSET; SUBSET_TRANS ;subset_closure ];
  ASM_MESON_TAC[closure_closed;INTER_SUBSET; SUBSET_TRANS ;subset_closure ];
  ]);;

  (* }}} *)

let closure_open_ball = prove_by_refinement(
  `!(X:A->bool) d Z .
    ((metric_space(X,d)) /\ (Z SUBSET X)) ==>
     (({a | !r. (&.0 < r) ==> (?z. (Z z /\ open_ball(X,d) a r z))}
         = closure (top_of_metric(X,d)) Z))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `topology_ (top_of_metric(X,d)) /\ (Z SUBSET (UNIONS (top_of_metric (X,d))))` SUBGOAL_TAC;
  ASM_SIMP_TAC[top_of_metric_top;GSYM top_of_metric_unions];
  DISCH_TAC;
  USE 2 (MATCH_MP closure_open);
  TYPE_THEN `{a | !r. (&.0 < r) ==> (?z. (Z z /\ open_ball(X,d) a r z))}` (USE 2 o SPEC);
  ASM_REWRITE_TAC[];
  CONJ_TAC; (* 1st prong *)
  REWRITE_TAC[ISUBSET;];
  GEN_TAC;
  DISCH_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_MESON_TAC[SUBSET;IN;INR open_ball_nonempty];
  CONJ_TAC;
  REWRITE_TAC[closed;open_DEF ];
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  CONJ_TAC;
  REWRITE_TAC[SUBSET;IN;IN_ELIM_THM';open_ball ;];
  DISCH_ALL_TAC;
  TYPE_THEN `&.1` (USE 3 o SPEC);
  UND 3;
  REDUCE_TAC;
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  MESON_TAC[];
  ASM_SIMP_TAC[top_of_metric_nbd];
  REWRITE_TAC[IN;DIFF; ISUBSET ];
  CONJ_TAC;
  MESON_TAC[];
  DISCH_ALL_TAC;
  LEFT 4 "r";
  CHO 4;
  USE 4 (REWRITE_RULE[NOT_IMP]);
  TYPE_THEN `r` EXISTS_TAC;
  NAME_CONFLICT_TAC;
  ASM_REWRITE_TAC[NOT_IMP];
  DISCH_ALL_TAC;
  AND 4;
  SUBCONJ_TAC;
  UND 5;
  REWRITE_TAC[open_ball;  ];
  MESON_TAC[];
  DISCH_TAC;
  LEFT_TAC "r'";
  JOIN 0 5;
  USE 0 (MATCH_MP (INR open_ball_center));
  CHO 0;
  TYPE_THEN `r'` EXISTS_TAC;
  UND 0;
  UND 4;
  MESON_TAC[SUBSET;IN];
  (* final prong *)
  (* fp  *)
  ONCE_REWRITE_TAC[TAUT (`a /\ b ==> e <=> (a /\ ~e ==> ~b)`)];
  REWRITE_TAC[open_DEF;EMPTY_EXISTS ];
  DISCH_ALL_TAC;
  CHO 4;
  USE 4 (REWRITE_RULE[INTER ]);
  AND 4;
  UND 3;
  ASM_SIMP_TAC[top_of_metric_nbd;];
  DISCH_ALL_TAC;
  TSPEC `u` 6;
  REWR 6;
  CHO 6;
  TSPEC `r` 4;
  REWR 4;
  CHO 4;
  TYPE_THEN `z` EXISTS_TAC;
  REWRITE_TAC[INTER];
  ASM_MESON_TAC[ISUBSET];
  ]);;
  (* }}} *)

let closed_union = prove_by_refinement(
  `!U (A:A->bool) B. (topology_ U) /\ (closed_ U A) /\ (closed_ U B) ==>
     (closed_ U (A UNION B))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[closed;open_DEF;union_subset  ];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `UNIONS U DIFF (A UNION B) = (UNIONS U DIFF A) INTER  (UNIONS U DIFF B)` SUBGOAL_TAC;
  REWRITE_TAC[DIFF;UNION;IN;INTER;IN_ELIM_THM'];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC THEN REWRITE_TAC[IN_ELIM_THM'];
  ASM_MESON_TAC[SUBSET;IN];
  DISCH_THEN (fun t->REWRITE_TAC[t]);
  ASM_MESON_TAC[top_inter];
  ]);;
  (* }}} *)

(* euclid *)
let euclid_scale0 = prove_by_refinement(
  `!x. (&.0 *# x) = (euclid0)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[euclid_scale;euclid0];
  REDUCE_TAC;
  ]);;
  (* }}} *)

let euclid_minus0 = prove_by_refinement(
  `!x. (x - euclid0) = x`,
  (* {{{ proof *)
  [
  REWRITE_TAC[euclid0;euclid_minus];
  REDUCE_TAC;
(*** Changed by JRH since MESON no longer automatically applies extensionality
  MESON_TAC[];
 ***)
  REWRITE_TAC[FUN_EQ_THM]
  ]);;
  (* }}} *)

let norm_scale2 = prove_by_refinement(
  `!t x. (euclidean x) ==> (norm (t *# x) = abs(t) * norm x)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP norm_scale);
  TYPEL_THEN [`t`;`&.0`] (USE 0 o ISPECL);
  USE 0 (REWRITE_RULE[euclid_scale0;d_euclid;euclid_minus0]);
  UND 0;
  REDUCE_TAC;
  ]);;
  (* }}} *)


(* ------------------------------------------------------------------ *)
(* half-spaces  *)
(* ------------------------------------------------------------------ *)

let closed_half_space = jordan_def `closed_half_space n v b =
  {z | (euclid n z) /\ (dot v z <=. b) }`;;

let open_half_space = jordan_def `open_half_space n v b =
  {z | (euclid n z) /\ (dot v z <. b) }`;;

let hyperplane = jordan_def `hyperplane n v b =
  {z | (euclid n z) /\ (dot v z = b) }`;;

let closed_half_space_euclid = prove_by_refinement(
  `!n v b. (closed_half_space n v b SUBSET euclid n)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[closed_half_space;SUBSET;IN;IN_ELIM_THM'  ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let open_half_space_euclid = prove_by_refinement(
  `!n v b. (open_half_space n v b SUBSET euclid n)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[open_half_space;SUBSET;IN;IN_ELIM_THM'  ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let hyperplane_euclid = prove_by_refinement(
  `!n v b. (hyperplane n v b SUBSET euclid n)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[hyperplane;SUBSET;IN;IN_ELIM_THM'  ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let closed_half_space_scale = prove_by_refinement(
  `!n v b r. ( &.0 < r) /\ (euclid n v) ==>
   (closed_half_space n (r *# v) (r * b) = closed_half_space n v b)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[closed_half_space];
  IMATCH_MP_TAC  EQ_EXT ;
  GEN_TAC THEN REWRITE_TAC[IN_ELIM_THM'];
  IMATCH_MP_TAC  (TAUT `(C ==> (a <=> b)) ==> ((C /\ a) <=> (C /\ b))`);
  DISCH_ALL_TAC;
  JOIN 1 2;
  USE 1 (MATCH_MP dot_scale);
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[dot_scale];
  IMATCH_MP_TAC  REAL_LE_LMUL_EQ;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let open_half_space_scale = prove_by_refinement(
  `!n v b r. ( &.0 < r) /\ (euclid n v) ==>
   (open_half_space n (r *# v) (r * b) = open_half_space n v b)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[open_half_space];
  IMATCH_MP_TAC  EQ_EXT ;
  GEN_TAC THEN REWRITE_TAC[IN_ELIM_THM'];
  IMATCH_MP_TAC  (TAUT `(C ==> (a <=> b)) ==> ((C /\ a) <=> (C /\ b))`);
  DISCH_ALL_TAC;
  JOIN 1 2;
  USE 1 (MATCH_MP dot_scale);
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[dot_scale];
  IMATCH_MP_TAC  REAL_LT_LMUL_EQ;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let hyperplane_scale = prove_by_refinement(
  `!n v b r. ~( r = &.0) /\ (euclid n v) ==>
   (hyperplane n (r *# v) (r * b)= hyperplane n v  b)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[hyperplane];
  IMATCH_MP_TAC  EQ_EXT ;
  GEN_TAC THEN REWRITE_TAC[IN_ELIM_THM'];
  IMATCH_MP_TAC  (TAUT `(C ==> (a <=> b)) ==> ((C /\ a) <=> (C /\ b))`);
  DISCH_ALL_TAC;
  JOIN 1 2;
  USE 1 (MATCH_MP dot_scale);
  ASM_REWRITE_TAC[REAL_EQ_MUL_LCANCEL ];
  ]);;
  (* }}} *)

let open_half_space_diff = prove_by_refinement(
  `!n v b. (euclid n v) ==>
     ((euclid n) DIFF (open_half_space n v b) =
       (closed_half_space n (-- v) (--. b)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[open_half_space;closed_half_space;DIFF ];
  REWRITE_TAC[IN; IN_ELIM_THM'];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[IN_ELIM_THM;dot_neg ];
  GEN_TAC;
  IMATCH_MP_TAC  (TAUT `(C ==> (a <=> b)) ==> ((C /\ a) <=> (C /\ b))`);
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let closed_half_space_diff = prove_by_refinement(
  `!n v b. (euclid n v) ==>
     ((euclid n) DIFF (closed_half_space n v b) =
       (open_half_space n (-- v) (--. b)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[open_half_space;closed_half_space;DIFF ];
  REWRITE_TAC[IN; IN_ELIM_THM'];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[IN_ELIM_THM;dot_neg ];
  GEN_TAC;
  IMATCH_MP_TAC  (TAUT `(C ==> (a <=> b)) ==> ((C /\ a) <=> (C /\ b))`);
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let closed_half_space_inter = prove_by_refinement(
  `!n v b. (euclid n v) ==>
    (closed_half_space n v b INTER closed_half_space n (-- v) (--b) =
    hyperplane n v b)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[closed_half_space;INTER;IN;hyperplane;IN_ELIM_THM' ];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[IN_ELIM_THM'];
  REWRITE_TAC[GSYM CONJ_ASSOC ];
  IMATCH_MP_TAC  (TAUT `(C ==> (a <=> b)) ==> ((C /\ a) <=> (C /\ b))`);
  DISCH_TAC;
  ASM_REWRITE_TAC[dot_neg ];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let open_half_space_convex = prove_by_refinement(
  `!n v b. (euclid n v) ==> (convex (open_half_space n v b))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[convex;open_half_space;mk_segment;IN_ELIM_THM';SUBSET;IN  ];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  CHO 5;
  UND 5;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  KILL 7;
  ASM_SIMP_TAC[euclid_add_closure;euclid_scale_closure;];
  TYPE_THEN `dot v (euclid_plus (a *# x) ((&1 - a) *# y)) = a * (dot v x) + (&1 - a)* (dot v y)` SUBGOAL_TAC;
  ASM_MESON_TAC[euclid_add_closure;euclid_scale_closure;dot_linear2;dot_scale2 ];
  DISCH_THEN (fun t -> REWRITE_TAC[t]);
  ASM_CASES_TAC `&.0 = a`;
  EXPAND_TAC "a";
  REDUCE_TAC;
  ASM_REWRITE_TAC[];
  GEN_REWRITE_TAC (RAND_CONV)[REAL_ARITH `b = a * b + ((&.1)* b - a* b)`];
  IMATCH_MP_TAC  REAL_LTE_ADD2;
  CONJ_TAC;
  MP_TAC (REAL_ARITH `~(&.0 = a) /\ (&.0 <= a) ==> (&.0 < a)`);
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[REAL_LT_LMUL_EQ];
  REWRITE_TAC[GSYM REAL_SUB_RDISTRIB];
  IMATCH_MP_TAC  REAL_LE_LMUL;
  UND 6;
  UND 4;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let closed_half_space_convex = prove_by_refinement(
  `!n v b. (euclid n v) ==> (convex (closed_half_space n v b))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[convex;closed_half_space;mk_segment;IN_ELIM_THM';SUBSET;IN];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  CHO 5;
  UND 5;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  KILL 7;
  ASM_SIMP_TAC[euclid_add_closure;euclid_scale_closure;];
  TYPE_THEN `dot v (euclid_plus (a *# x) ((&1 - a) *# y)) = a * (dot v x) + (&1 - a)* (dot v y)` SUBGOAL_TAC;
  ASM_MESON_TAC[euclid_add_closure;euclid_scale_closure;dot_linear2;dot_scale2 ];
  DISCH_THEN (fun t -> REWRITE_TAC[t]);
  GEN_REWRITE_TAC (RAND_CONV)[REAL_ARITH `b = a * b + ((&.1)* b - a* b)`];
  IMATCH_MP_TAC  REAL_LE_ADD2;
  REWRITE_TAC[GSYM REAL_SUB_RDISTRIB];
  USE 6 (MATCH_MP (REAL_ARITH `(a <= &.1) ==> (&.0 <= (&1-a))`));
  CONJ_TAC THEN (IMATCH_MP_TAC REAL_LE_LMUL) THEN ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let hyperplane_convex = prove_by_refinement(
  `!n v b. (euclid n v) ==> convex(hyperplane n v b)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  ASM_SIMP_TAC[GSYM closed_half_space_inter];
  IMATCH_MP_TAC  convex_inter;
  ASM_MESON_TAC[closed_half_space_convex;neg_dim ];
  ]);;

  (* }}} *)

let open_half_space_open = prove_by_refinement(
  `!n v b. (euclid n v) ==>
    (top_of_metric(euclid n,d_euclid)) (open_half_space n v b)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  ASM_SIMP_TAC[top_of_metric_nbd;metric_euclid;SUBSET;IN;IN_ELIM_THM' ];
  REWRITE_TAC[open_half_space;open_ball;IN_ELIM_THM' ];
  CONJ_TAC ;
  MESON_TAC[];
  DISCH_ALL_TAC;
  ASM_CASES_TAC `v = euclid0`;
  UND 2;
  ASM_REWRITE_TAC[dot_lzero];
  MESON_TAC[];
  TYPE_THEN `(b - (dot v a))/(norm v)` EXISTS_TAC;
  TYPE_THEN `&.0 < (norm v)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `&0 <= x /\ (~(x = &.0)) ==> (&.0 < x)`);
  ASM_MESON_TAC[norm;norm_nonneg;dot_nonneg;SQRT_EQ_0;dot_zero];
  DISCH_ALL_TAC;
  SUBCONJ_TAC;
  ASM_SIMP_TAC[REAL_LT_RDIV_0];
  UND 2;
  REAL_ARITH_TAC;
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(x:num->real) = a + (x - a)` SUBGOAL_TAC;
  REWRITE_TAC[euclid_plus;euclid_minus];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC THEN BETA_TAC;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t -> ONCE_REWRITE_TAC[t]);
  TYPE_THEN `dot v (a + (x - a)) = (dot v a) + (dot v (x-a))` SUBGOAL_TAC;
  IMATCH_MP_TAC  dot_linear2;
  TYPE_THEN `n` EXISTS_TAC;
  ASM_SIMP_TAC[euclid_sub_closure];
  DISCH_THEN (fun t -> REWRITE_TAC[t]);
  IMATCH_MP_TAC  (REAL_ARITH `(?d. (b<=d) /\ d < C - a) ==> a +b < C`);
  TYPE_THEN `(norm v)*. (d_euclid a x)` EXISTS_TAC;
  CONJ_TAC;
  ASSUME_TAC metric_euclid;
  TYPE_THEN `n` (USE 9 o SPEC);
  COPY 7;
  JOIN  6 7;
  JOIN 9 6;
  USE 6 (MATCH_MP metric_space_symm);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[d_euclid];
  IMATCH_MP_TAC  (REAL_ARITH `||. u <=. C ==> (u <=. C)`);
  IMATCH_MP_TAC  cauchy_schwartz;
  ASM_MESON_TAC[euclidean;euclid_sub_closure];
  UND 8;
  ASM_SIMP_TAC[REAL_LT_RDIV_EQ];
  REAL_ARITH_TAC;
  ]);;

  (* }}} *)

let closed_half_space_closed = prove_by_refinement(
  `!n v b. (euclid n v) ==>
     closed_ (top_of_metric(euclid n,d_euclid))
      (closed_half_space n v b)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[closed;open_DEF ];
  DISCH_ALL_TAC;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_euclid;closed_half_space_diff;open_half_space_open;euclid_neg_closure ];
  REWRITE_TAC[closed_half_space;SUBSET;IN;IN_ELIM_THM' ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let hyperplane_closed = prove_by_refinement(
  `!n v b. (euclid n v) ==>
     closed_ (top_of_metric(euclid n,d_euclid))
     (hyperplane n v b)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  ASM_SIMP_TAC[GSYM closed_half_space_inter];
  IMATCH_MP_TAC  closed_inter2;
  ASM_MESON_TAC[euclid_neg_closure;top_of_metric_top ;metric_euclid ;closed_half_space_closed;];
  ]);;
  (* }}} *)

let closure_half_space = prove_by_refinement(
  `!n v b. (euclid n v) /\ (~(v = euclid0)) ==>
   ((closure (top_of_metric(euclid n,d_euclid))
    (open_half_space n v b)) = (closed_half_space n v b))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC  closure_subset;
  ASM_SIMP_TAC [top_of_metric_top;metric_euclid];
  ASM_SIMP_TAC[GSYM closure_open_ball;metric_euclid;closed_half_space_closed];
  REWRITE_TAC[SUBSET;IN;closed_half_space;open_half_space;IN_ELIM_THM' ];
  MESON_TAC[REAL_ARITH `a < b ==> a <=. b`];
  ASM_SIMP_TAC[GSYM closure_open_ball;metric_euclid;open_half_space_euclid];
  REWRITE_TAC[open_half_space;closed_half_space;SUBSET;IN;IN_ELIM_THM'];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `t = ((r/(&.2))/(norm v ))` ABBREV_TAC;
  TYPE_THEN `u = x - (t)*# v` ABBREV_TAC;
  TYPE_THEN `u` EXISTS_TAC;
  TYPE_THEN `&.0 < (dot v v)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `~(x = &.0) /\ (&.0 <=. x) ==> (&.0 < x)`);
  REWRITE_TAC[dot_nonneg];
  ASM_MESON_TAC[euclidean;dot_zero_euclidean ];
  DISCH_TAC;
  TYPE_THEN `&.0 < t` SUBGOAL_TAC;
  EXPAND_TAC "t";
  IMATCH_MP_TAC  REAL_LT_DIV;
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  REWRITE_TAC[norm];
  IMATCH_MP_TAC  SQRT_POS_LT;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  SUBCONJ_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[euclid_sub_closure ;euclid_scale_closure ];
  TYPE_THEN `dot v u = (dot v x - t* (dot v v))` SUBGOAL_TAC;
  EXPAND_TAC "u";
  ASM_MESON_TAC[dot_minus_linear2;dot_scale2;euclid_sub_closure;euclid_scale_closure];
  DISCH_THEN (fun t->REWRITE_TAC[t]);
  IMATCH_MP_TAC  (REAL_ARITH `(a <= b) /\ (&.0 < C) ==> (a - C < b)`);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  REAL_LT_MUL;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[open_ball;IN_ELIM_THM' ];
  EXPAND_TAC "u";
  REWRITE_TAC[d_euclid];
  TYPE_THEN `euclid_minus x (euclid_minus x (t *# v)) = ( t) *# v` SUBGOAL_TAC;
  REWRITE_TAC[euclid_minus;euclid_scale];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC THEN BETA_TAC;
  REAL_ARITH_TAC ;
  DISCH_THEN (fun t-> REWRITE_TAC[t]);
  TYPE_THEN `norm (t *# v) = t * norm v` SUBGOAL_TAC;
  ASM_MESON_TAC[euclidean;norm_scale2;ABS_REFL;REAL_ARITH `&.0 < t ==> &.0 <= t`];
  DISCH_THEN (fun t -> REWRITE_TAC[t]);
  EXPAND_TAC "t";
  TYPE_THEN `((r / &2) / norm v) * norm v = r/(&.2)` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_DIV_RMUL;
  REWRITE_TAC[norm];
  ASM_MESON_TAC[SQRT_POS_LT;REAL_ARITH `&.0 < x ==> ~(x = &.0)`];
  DISCH_THEN (fun t-> REWRITE_TAC[t]);
  ASM_MESON_TAC[half_pos];
  ]);;

  (* }}} *)


let subset_of_closure = prove_by_refinement(
  `!(A:A->bool) B U. (topology_ U) /\ (A SUBSET B) ==>
    (closure U A SUBSET closure U B)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `(A SUBSET (UNIONS U))` ASM_CASES_TAC;
  TYPE_THEN `(B SUBSET (UNIONS U))` ASM_CASES_TAC;
  IMATCH_MP_TAC  closure_subset;
  ASM_REWRITE_TAC[];
  WITH 0 (MATCH_MP subset_closure);
  USE 4 (ISPEC `B:A->bool`);
  JOIN 1 4;
  USE 1 (MATCH_MP SUBSET_TRANS);
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC [closure_closed;];
  USE 3 (MATCH_MP closure_univ);
  ASM_REWRITE_TAC[];
  TYPE_THEN `~(B SUBSET UNIONS U)` SUBGOAL_TAC;
  UND 2;
  UND 1;
  REWRITE_TAC[ISUBSET];
  MESON_TAC[];
  DISCH_TAC;
  USE 2 (MATCH_MP closure_univ);
  USE 3 (MATCH_MP closure_univ);
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let closure_union = prove_by_refinement(
  `!(A:A->bool)  B U. (topology_ U) ==>
    (closure U (A UNION B) = (closure U A) UNION (closure U B))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN  `A SUBSET UNIONS U` ASM_CASES_TAC THEN (TYPE_THEN `B SUBSET UNIONS U` ASM_CASES_TAC ) THEN TRY(IMP_RES_THEN (fun t -> REWRITE_TAC[t;UNION_UNIV;SUBSET_UNIV;INTER_UNIV]) closure_univ)  THEN TRY (IMATCH_MP_TAC  closure_univ) THEN TRY (UNDISCH_FIND_TAC `(~)`);
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC closure_subset;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  ASM_MESON_TAC[closed_union; closure_closed];
  REWRITE_TAC[union_subset];
  TYPE_THEN `(A SUBSET closure U A) /\ (B SUBSET closure U B)` SUBGOAL_TAC;
  ASM_SIMP_TAC[subset_closure];
  REWRITE_TAC[UNION;ISUBSET ];
  ASM_MESON_TAC[];
  REWRITE_TAC[union_subset];
  CONJ_TAC THEN IMATCH_MP_TAC  subset_of_closure THEN ASM_REWRITE_TAC[ISUBSET;UNION ] THEN (MESON_TAC []);
  REWRITE_TAC [UNION;SUBSET; ];
  MESON_TAC[];
  REWRITE_TAC[UNION;SUBSET];
  MESON_TAC[];
  REWRITE_TAC[UNION;SUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let closure_empty = prove_by_refinement(
  `!U. (topology_ U) ==> (closure U (EMPTY:A->bool) = EMPTY)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  ASM_MESON_TAC[SUBSET_EMPTY;closure_subset;empty_closed];
  ]);;
  (* }}} *)

let closure_unions = prove_by_refinement(
  `!(A:(A->bool)->bool) U. (topology_ U) /\ (FINITE A) ==>
    (closure U (UNIONS A) = UNIONS (IMAGE (closure U) A))`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  TYPE_THEN `n = CARD A` ABBREV_TAC;
  UND 0;
  TYPE_THEN `A` (fun t-> SPEC_TAC (t,t));
  TYPE_THEN `n` (fun t-> SPEC_TAC (t,t));
  INDUCT_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `A HAS_SIZE 0` SUBGOAL_TAC;
  ASM_REWRITE_TAC[HAS_SIZE];
  ASM_REWRITE_TAC[HAS_SIZE_0];
  DISCH_THEN_REWRITE;
  ASM_SIMP_TAC [closure_empty;IMAGE_CLAUSES];
  DISCH_ALL_TAC;
  TYPE_THEN `~(A HAS_SIZE 0)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[HAS_SIZE];
  ARITH_TAC;
  TYPE_THEN `A` (MP_TAC o ((C ISPEC)  CARD_DELETE_CHOICE));
  REWRITE_TAC[HAS_SIZE_0];
  DISCH_ALL_TAC;
  REWR 5;
  USE 5 (CONV_RULE REDUCE_CONV );
  TYPE_THEN `(A DELETE CHOICE A)` (USE 0 o ISPEC);
  USE 0 (REWRITE_RULE[FINITE_DELETE]);
  REWR 0;
  TYPE_THEN `A (CHOICE A)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (INR CHOICE_DEF);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `UNIONS A = (UNIONS (A DELETE CHOICE A)) UNION (CHOICE A)` SUBGOAL_TAC;
  IMATCH_MP_TAC  unions_delete_choice;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `(IMAGE  (closure U) A) = (IMAGE (closure U) (A DELETE CHOICE A) UNION {(closure U (CHOICE A))})` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_delete_choice ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  ASM_SIMP_TAC[closure_union];
  REWRITE_TAC[UNIONS_UNION];
  ]);;
  (* }}} *)

let metric_space_zero2 = prove_by_refinement(
  `!X d (x:A) y. (metric_space(X,d) /\ (X x) /\ (X y)) ==>
   ((d x y = &.0) <=> (x = y))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  USE 0 (REWRITE_RULE[metric_space]);
  TYPEL_THEN [`x`;`y`;`x`] (USE 0 o ISPECL);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let d_euclid_zero = prove_by_refinement(
  `!n x y. (euclid n x) /\ (euclid n y)  ==>
    ((d_euclid x y = &.0) <=> (x = y))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPEL_THEN [`euclid n`;`d_euclid`;`x`;`y`] (ASSUME_TAC o (C ISPECL) metric_space_zero2);
  ASM_MESON_TAC[metric_euclid];
  ]);;
  (* }}} *)

let d_euclid_pos2 = prove_by_refinement(
  `!x y n. ~(x = y) /\ euclid n x /\ euclid n y ==> &0 <. d_euclid x y`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `&.0 <= x /\ ~(x = &.0) ==> (&.0 < x)`);
  ASM_MESON_TAC[d_euclid_pos;d_euclid_zero];
  ]);;
  (* }}} *)

let euclid_segment = prove_by_refinement(
  `!n x y. (euclid n x) /\
   (!t. (&.0 <. t) /\ (t <=. &.1) ==>
         (euclid n (t *# x + (&.1 - t)*# y)))
     ==>
   (euclid n y)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `t = &.1/(&.2)` ABBREV_TAC;
  TYPE_THEN `y = ((&.2) *# ((t *# x) + (&.1 - t)*# y)) - x` SUBGOAL_TAC;
  REWRITE_TAC[euclid_minus;euclid_scale;euclid_plus];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC THEN BETA_TAC ;
  REWRITE_TAC[REAL_ADD_LDISTRIB];
  REWRITE_TAC[REAL_MUL_ASSOC;REAL_SUB_LDISTRIB ];
  EXPAND_TAC "t";
  SIMP_TAC[REAL_DIV_LMUL;REAL_ARITH `~(&.2 = &.0)`];
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]);
  TYPE_THEN `t` (USE 1 o SPEC);
  TYPE_THEN `v = (euclid_plus (t *# x) ((&1 - t) *# y))` ABBREV_TAC;
  KILL 3;
  TYPE_THEN `&0 < t /\ t <= &1` SUBGOAL_TAC;
  EXPAND_TAC "t";
  CONJ_TAC ;
  IMATCH_MP_TAC  REAL_LT_DIV;
  REAL_ARITH_TAC;
  IMATCH_MP_TAC  REAL_LE_LDIV;
  REAL_ARITH_TAC;
  DISCH_TAC;
  REWR 1;
  ASM_SIMP_TAC[euclid_sub_closure;euclid_scale_closure];
  ]);;
  (* }}} *)

let euclid_xy = prove_by_refinement(
  `!n x y. (!t . (&.0 < t) /\ (t < &.1) ==>
    (euclid n (t *# x + (&.1-t)*# y))) ==> (euclid n x) /\ (euclid n y)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  TYPE_THEN `u = (&.1/(&.3))*# x + (&.1 - (&.1/(&.3))) *# y` ABBREV_TAC;
  TYPE_THEN `v = (&.2/(&.3))*# x + (&.1 - (&.2/(&.3))) *# y` ABBREV_TAC;
  TYPE_THEN `euclid n u` SUBGOAL_TAC;
  EXPAND_TAC "u";
  UND 0;
  DISCH_THEN IMATCH_MP_TAC ;
  CONV_TAC REAL_RAT_REDUCE_CONV;
  DISCH_TAC;
  TYPE_THEN `euclid n v` SUBGOAL_TAC;
  EXPAND_TAC "v";
  UND 0;
  DISCH_THEN IMATCH_MP_TAC ;
  CONV_TAC REAL_RAT_REDUCE_CONV;
  DISCH_TAC;
  TYPE_THEN `x = (&.2)*# v - (&.1) *# u` SUBGOAL_TAC;
  EXPAND_TAC "u";
  EXPAND_TAC "v";
  REWRITE_TAC[euclid_minus;euclid_plus;euclid_scale];
  IMATCH_MP_TAC  EQ_EXT;
  DISCH_ALL_TAC;
  BETA_TAC;
  TYPE_THEN `a = x x'`  ABBREV_TAC ;
  TYPE_THEN `b= y x'`  ABBREV_TAC ;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  ASM_SIMP_TAC[euclid_scale_closure;euclid_sub_closure];
  TYPE_THEN `y = (&.2)*# u - (&.1) *# v` SUBGOAL_TAC;
  EXPAND_TAC "u";
  EXPAND_TAC "v";
  REWRITE_TAC[euclid_minus;euclid_plus;euclid_scale];
  IMATCH_MP_TAC  EQ_EXT;
  DISCH_ALL_TAC;
  BETA_TAC;
  TYPE_THEN `a = x x'`  ABBREV_TAC ;
  TYPE_THEN `b= y x'`  ABBREV_TAC ;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  ASM_SIMP_TAC[euclid_scale_closure;euclid_sub_closure];
  ]);;
  (* }}} *)


let closure_segment = prove_by_refinement(
  `!C n x y. (C SUBSET (euclid n)) /\
      (!t. (&.0 < t) /\ (t < &.1) ==> (C (t *# x + (&.1-t)*# y))) ==>
      (closure (top_of_metric(euclid n,d_euclid)) C y)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  TYPE_THEN `euclid n x /\ (euclid n y)` SUBGOAL_TAC;
  IMATCH_MP_TAC  euclid_xy;
  ASM_MESON_TAC[ISUBSET];
  DISCH_ALL_TAC;
  (* case x=y *)
  TYPE_THEN `x = y` ASM_CASES_TAC ;
  TYPE_THEN `C SUBSET (closure (top_of_metric (euclid n,d_euclid)) C)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  subset_closure;
  ASM_SIMP_TAC [top_of_metric_top;metric_euclid];
  REWRITE_TAC[ISUBSET];
  TYPE_THEN `C x` SUBGOAL_TAC;
  REWR 1;
  USE 1 (REWRITE_RULE[trivial_lin_combo]);
  TSPEC `&.1/(&.2)` 1;
  USE 1 (CONV_RULE (REAL_RAT_REDUCE_CONV));
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  (* now ~(x=y) *)
  TYPE_THEN `&.0 < d_euclid x y` SUBGOAL_TAC;
  ASM_MESON_TAC[d_euclid_pos2];
  DISCH_TAC;
  ASM_SIMP_TAC[GSYM closure_open_ball;metric_euclid];
  DISCH_ALL_TAC;
  REWRITE_TAC[open_ball];
  (* ## *)
  TYPE_THEN `?t. (&.0 <. t) /\ (t <. &.1) /\ (t *. (d_euclid x y) <. r)` SUBGOAL_TAC;
  TYPE_THEN  `(&.1/(&.2))*. d_euclid x y < r` ASM_CASES_TAC;
  TYPE_THEN `(&.1/(&.2))` EXISTS_TAC;
  CONV_TAC (REAL_RAT_REDUCE_CONV);
  ASM_REWRITE_TAC[];
  TYPE_THEN `(r/(&.2))/(d_euclid x y)` EXISTS_TAC;
  ASM_SIMP_TAC[REAL_LT_DIV;REAL_LT_HALF1];
  CONJ_TAC;
  ASM_SIMP_TAC[REAL_LT_LDIV_EQ];
  REDUCE_TAC;
  TYPE_THEN `s = d_euclid x y ` ABBREV_TAC;
  ineq_lt_tac `r/(&.2) + ( (&1/(&2))*s - r)*(&1/(&2)) + (s)*(&3/(&4)) = s`;
  ASM_SIMP_TAC[GSYM REAL_LT_RDIV_EQ;REAL_LT_RDIV;half_pos];
  DISCH_TAC;
  CHO 7;
  TYPE_THEN `t` (USE 1 o SPEC);
  REWR 1;
  TYPE_THEN `z = (euclid_plus (t *# x) ((&1 - t) *# y))` ABBREV_TAC ;
  TYPE_THEN `z` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  EXPAND_TAC "z";
  ASM_SIMP_TAC[euclid_add_closure;euclid_scale_closure];
  DISCH_TAC;
  TYPE_THEN `y = (t *# y) + ((&.1 - t)*# y)` SUBGOAL_TAC;
  ASM_MESON_TAC[trivial_lin_combo];
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC [t]);
  EXPAND_TAC "z";
  TYPE_THEN `euclid n (t*# y) /\  (euclid n (t *# x)) /\ (euclid n ((&.1-t)*# y))` SUBGOAL_TAC;
  ASM_SIMP_TAC[euclid_add_closure;euclid_scale_closure];
  DISCH_TAC;
  USE 10 (MATCH_MP metric_translate);
  KILL 8;
  ASM_REWRITE_TAC[];
  TYPE_THEN `d_euclid (t *# y) (t *# x) = d_euclid (t *# x) (t *# y)` SUBGOAL_TAC;
  ASM_MESON_TAC [ISPEC `euclid n` metric_space_symm; euclid_scale_closure;metric_euclid];
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC [t]);
  JOIN 2 3;
  USE 2 (MATCH_MP norm_scale_vec);
  TSPEC `t` 2;
  ASM_REWRITE_TAC[];
  AND 7;
  USE 7 (MATCH_MP (REAL_ARITH `&.0 < t ==> (&.0 <=. t)`));
  USE 7 (REWRITE_RULE[GSYM ABS_REFL]);
  ASM_REWRITE_TAC [];
  ]);;

  (* }}} *)



(* ------------------------------------------------------------------ *)
(* POINTS *)
(* ------------------------------------------------------------------ *)


let point = jordan_def `point z =
   (FST z) *# (dirac_delta 0) + (SND z) *# (dirac_delta 1)`;;

let dest_pt = jordan_def `dest_pt p =
   @u.  p = point u`;;

let point_xy = prove_by_refinement(
  `!x y. point(x,y) = x *# (dirac_delta 0) + y *# (dirac_delta 1)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[point;];
  ]);;
  (* }}} *)

let coord01 = prove_by_refinement(
  `!p. (point p 0 = FST p) /\ (point p 1 = SND p)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[point;euclid_plus;euclid_scale ];
  REWRITE_TAC[dirac_delta;ARITH_RULE   `~(1=0) /\ ~(0=1)`];
  REDUCE_TAC ;
  ]);;
  (* }}} *)

let euclid_point = prove_by_refinement(
  `!p. euclid 2 (point p)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[point;euclid];
  REWRITE_TAC[point;euclid_plus;euclid_scale;dirac_delta ];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP (ARITH_RULE `(2 <=| m) ==> (~(0=m) /\ (~(1=m)))`));
  ASM_REWRITE_TAC[];
  REDUCE_TAC ;
  ]);;
  (* }}} *)

let point_inj = prove_by_refinement(
  `!p q. (point p = point q) <=> (p = q)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  EQ_TAC ;
  DISCH_TAC ;
  WITH  0 (fun t -> AP_THM t `0`);
  USE 0 (fun t-> AP_THM t `1`);
  UND 0;
  UND 1;
  REWRITE_TAC[coord01;];
  ASM_MESON_TAC[PAIR];
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let point_onto = prove_by_refinement(
  `!v. (euclid 2 v) ==> ?p. (v = point p)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `(v 0 ,v 1)` EXISTS_TAC;
  IMATCH_MP_TAC  EQ_EXT ;
  GEN_TAC ;
  REWRITE_TAC[point;euclid_plus;euclid_scale;dirac_delta];
  MP_TAC (ARITH_RULE `(0 = x) \/ ( 1 = x) \/ (2 <= x)`);
  REP_CASES_TAC;
  WITH 1 (MATCH_MP (ARITH_RULE  `(0=x) ==> ~(1=x)`));
  ASM_REWRITE_TAC[];
  EXPAND_TAC "x";
  REDUCE_TAC;
  WITH 1 (MATCH_MP (ARITH_RULE  `(1=x) ==> ~(0=x)`));
  ASM_REWRITE_TAC[];
  EXPAND_TAC "x";
  REDUCE_TAC;
  WITH 1 (MATCH_MP (ARITH_RULE  `(2 <=| x) ==> (~(0=x)/\ ~(1=x))`));
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  ASM_MESON_TAC[euclid];
  ]);;
  (* }}} *)

let dest_pt_point = prove_by_refinement(
  `!p. dest_pt(point p) = p`,
  (* {{{ proof *)
  [
  REWRITE_TAC[dest_pt];
  DISCH_ALL_TAC;
  SELECT_TAC;
  ASM_MESON_TAC[point_inj];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let point_dest_pt = prove_by_refinement(
  `!v. (euclid 2 v) <=> (point (dest_pt v) = v)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  EQ_TAC;
  REWRITE_TAC[dest_pt];
  DISCH_ALL_TAC;
  SELECT_TAC;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[point_onto];
  ASM_MESON_TAC[euclid_point];
  ]);;
  (* }}} *)

let Q_POINT = prove_by_refinement(
  `!Q z. (?u v. (point z = point (u,v)) /\ (Q z u v)) <=> (Q z (FST z) (SND z))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[point_inj];
  EQ_TAC;
  DISCH_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `FST z` EXISTS_TAC;
  TYPE_THEN `SND z` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let pointI = jordan_def `pointI p =
   point(real_of_int (FST p),real_of_int (SND p))`;;

let convex_pointI = prove_by_refinement(
  `!p. (convex {(pointI p)})`,
  (* {{{ proof *)

  [
  REWRITE_TAC[convex;mk_segment;INSERT;IN_ELIM_THM';SUBSET; ];
  REWRITE_TAC[IN;EMPTY];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[trivial_lin_combo];
  DISCH_ALL_TAC;
  CHO 2;
  ASM_REWRITE_TAC[];
  ]);;

  (* }}} *)

let point_closure = prove_by_refinement(
  `!p q a b. (?r. (a *# (point p) + (b *# (point q)) = (point r)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `euclid 2 (a *# (point p) + (b *# (point q)))` SUBGOAL_TAC;
  IMATCH_MP_TAC euclid_add_closure;
  CONJ_TAC THEN (IMATCH_MP_TAC  euclid_scale_closure) THEN REWRITE_TAC [euclid_point];
  MESON_TAC[point_onto];
  ]);;
  (* }}} *)

let point_scale = prove_by_refinement(
  `!a u v. a *# (point (u,v)) = point(a* u,a* v)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[point;euclid_scale;euclid_plus ];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC THEN BETA_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let point_add = prove_by_refinement(
  `!u v u' v'. (point(u,v))+(point(u',v')) = (point(u+u',v+v'))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[point;euclid_plus;euclid_scale];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC THEN BETA_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)



(* ------------------------------------------------------------------ *)
(* the FLOOR function *)
(* ------------------------------------------------------------------ *)


let floor = jordan_def `floor x =
   @m. (real_of_int m <=. x /\ (x < (real_of_int (m + &:1))))`;;

let int_suc = prove_by_refinement(
  `!m. (real_of_int (m + &:1) = real_of_int m + &.1)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[int_add_th;INT_NUM_REAL ];
  ]);;
  (* }}} *)

let floor_ineq = prove_by_refinement(
  `!x. (real_of_int (floor x) <=. x) /\ (x <. (real_of_int (floor x)) + &.1)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[floor];
  SELECT_TAC;
  REWRITE_TAC[int_suc];
  MP_TAC (SPEC `&.1` REAL_ARCH_LEAST);
  REDUCE_TAC;
  DISCH_TAC;
  ASM_CASES_TAC `&.0 <= x`;
  TSPEC `x` 1;
  REWR 1;
  CHO 1;
  LEFT 0 "y";
  TSPEC `&:n` 0;
  USE 0  (REWRITE_RULE[INT_NUM_REAL;int_add_th;REAL_OF_NUM_ADD ]);
  ASM_MESON_TAC[];
  TSPEC `--. x` 1;
    COPY 2;
  IMP_REAL `~(&.0 <=. x) ==> (&.0 <=. (-- x))` 2;
  REWR 1;
  CHO 1;
  LEFT 0 "y";
  ASM_CASES_TAC `&.n = --x`;
  TSPEC `-- (&:n)` 0;
  USE 0 (REWRITE_RULE[int_neg_th;int_add_th ;INT_NUM_REAL;REAL_OF_NUM_ADD]);
  JOIN 0 1;
  USE 0 (REWRITE_RULE[ GSYM REAL_OF_NUM_ADD]);
  PROOF_BY_CONTR_TAC;
  UND 0;
  UND 4;
  REAL_ARITH_TAC ;
  TSPEC `--: (&:(n+| 1))` 0;
  JOIN 1 0;
  USE 0 (REWRITE_RULE[int_neg_th;int_add_th ;INT_NUM_REAL; GSYM REAL_OF_NUM_ADD;]);
  JOIN 4 0;
  PROOF_BY_CONTR_TAC;
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let int_arch = prove_by_refinement(
  `!m n. (n <=: m) /\ (m <: (n +: (&:1))) <=> (n = m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[int_lt;int_le;int_eq ;int_add_th;int_of_num_th   ];
  DISCH_ALL_TAC;
  EQ_TAC;
  MP_TAC (SPEC `m:int` dest_int_rep);
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  MP_TAC (SPEC `n:int` dest_int_rep);
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  REPEAT (DISCH_THEN (REPEAT_TCL DISJ_CASES_THEN ASSUME_TAC)) THEN ((UNDISCH_FIND_TAC  `(/\)`)) THEN (  ASM_REWRITE_TAC[int_add_th;int_of_num_th ]) THEN  REDUCE_TAC THEN   TRY ARITH_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let floor_int = prove_by_refinement(
  `!m. (floor (real_of_int m) = m)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `floor (real_of_int m) <=: m /\ (m <: (floor (real_of_int m)) + (&:1))` SUBGOAL_TAC;
  REWRITE_TAC[int_le;int_lt;int_add_th ;int_of_num_th;floor_ineq  ];
  REWRITE_TAC[int_arch ];
  ]);;
  (* }}} *)

let int_lt_suc_le = prove_by_refinement(
  `!m n. m <: n + &:1 <=> m <=: n`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  EQ_TAC;
  MP_TAC (SPEC `m:int` dest_int_rep);
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  MP_TAC (SPEC `n:int` dest_int_rep);
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  REPEAT (DISCH_THEN (REPEAT_TCL DISJ_CASES_THEN ASSUME_TAC)) THEN ((UNDISCH_FIND_TAC  `(+:)`)) THEN (  ASM_REWRITE_TAC[int_add_th;int_of_num_th ]) THEN  REDUCE_TAC THEN   TRY ARITH_TAC;
  REWRITE_TAC[int_le;int_lt;int_add_th;int_of_num_th];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let floor_le = prove_by_refinement(
  `!m x. (real_of_int m <=. x) <=> (m <=: (floor x))`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  EQ_TAC;
  DISCH_TAC;
  REWRITE_TAC[int_le];
  REWRITE_TAC[GSYM int_le ;GSYM   int_lt_suc_le;];
  REWRITE_TAC[int_lt ;int_add_th;int_of_num_th;];
  ASM_MESON_TAC[floor_ineq; REAL_LET_TRANS];
  REWRITE_TAC[int_le];
  MP_TAC (SPEC `x:real` floor_ineq);
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let floor_lt = prove_by_refinement(
  `!m x. (x < real_of_int m + &.1) <=> (floor x <=: m)`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  EQ_TAC;
  DISCH_TAC;
  REWRITE_TAC[GSYM int_lt_suc_le ;];
  REWRITE_TAC[int_lt;int_add_th;int_of_num_th;];
  UND 0;
  MP_TAC (SPEC `x:real` floor_ineq);
  REAL_ARITH_TAC;
  REWRITE_TAC[int_le;];
  MP_TAC (SPEC `x:real` floor_ineq);
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let floor_mono = prove_by_refinement(
  `!x y. (x <=. y) ==> (floor x <=: floor y)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM floor_le];
  REP_GEN_TAC;
  MP_TAC (SPEC `x:real` floor_ineq);
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let floor_level = prove_by_refinement(
  `!m x. ((&.0 <=. x) /\ (x <. &.1)) ==> (floor (real_of_int(m) + x) = m)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  SUBGOAL_TAC  `!a b. (b <=: a /\ ~(b <: a)) ==> (a = b)`;
  REWRITE_TAC[int_le;int_lt;int_eq];
  REAL_ARITH_TAC;
  DISCH_THEN IMATCH_MP_TAC ;
  SUBCONJ_TAC;
  REWRITE_TAC[GSYM floor_le];
  UND 0;
  REAL_ARITH_TAC;
  DISCH_TAC;
  PROOF_BY_CONTR_TAC;
  USE 3 (REWRITE_RULE[]);
  USE 3 (ONCE_REWRITE_RULE[GSYM INT_LT_RADD ]);
  USE 3 (GEN `z:int`);
  TSPEC `&:1` 3;
  USE 3 (REWRITE_RULE [int_lt_suc_le ;]);
  MP_TAC (SPEC `real_of_int m + x` floor_ineq);
  UND 3;
  UND 1;
  REWRITE_TAC[int_add_th;int_le;int_of_num_th];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)


let floor_range = prove_by_refinement(
  `!x m. (floor x = m) <=> (real_of_int m <=. x /\ x <. real_of_int m +. &.1)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  EQ_TAC;
  DISCH_THEN (fun t -> REWRITE_TAC[GSYM t;floor_ineq]);
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[GSYM INT_LE_ANTISYM;GSYM floor_lt;GSYM floor_le;];
  ]);;
  (* }}} *)


(* ------------------------------------------------------------------ *)
(* edges and squares *)
(* ------------------------------------------------------------------ *)


let h_edge = jordan_def `h_edge p =
   { Z  | ?u v. (Z = point(u,v)) /\
    (real_of_int (FST p) <. u) /\ (u <. (real_of_int ((FST p)+: (&:1)))) /\
       (v = real_of_int (SND p)) }`;;

let v_edge = jordan_def `v_edge p =
   { Z  | ?u v. (Z = point(u,v)) /\
    (real_of_int (SND p) <. v) /\ (v <. (real_of_int ((SND p) +: (&:1)))) /\
       (u = real_of_int (FST p)) }`;;

let squ = jordan_def `squ p =
   {Z | ?u v. (Z = point(u,v)) /\
    (real_of_int (FST p) <. u) /\ (u <. (real_of_int ((FST p) +: (&:1)))) /\
    (real_of_int (SND p) <. v) /\ (v <. (real_of_int ((SND p) +: (&:1)))) }`;;

let row = jordan_def `row k = {Z | ?u . (Z = point(u,real_of_int k))}`;;

let col = jordan_def `col k = {Z | ?v . (Z = point(real_of_int k ,v))}`;;


let pointI_inj = prove_by_refinement(
  `!p q. (pointI p = pointI q) <=> (p = q) `,
  (* {{{ proof *)
  [
  REWRITE_TAC[pointI;point_inj;PAIR_EQ;GSYM int_eq ];
  MESON_TAC[PAIR;PAIR_EQ];
  ]);;
  (* }}} *)

let h_edge_row = prove_by_refinement(
  `!p . h_edge p  SUBSET  row (SND p) `,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;IN;h_edge;row;IN_ELIM_THM';];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  TYPE_THEN `u` EXISTS_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let h_edge_floor = prove_by_refinement(
  `!p. h_edge p SUBSET { z | floor (z 0)  = FST p }`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;IN;h_edge;IN_ELIM_THM';int_of_num_th;int_add_th;];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[coord01;floor_range];
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let row_disj = prove_by_refinement(
  `!a b. ~((row a) INTER (row b) = EMPTY) <=> (a = b)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[EMPTY_EXISTS;IN;INTER;row;IN_ELIM_THM'  ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  AND 0;
  CHO 0;
  CHO 1;
  REWRITE_TAC[int_eq];
  USE 1 (GSYM);
  REWR 1;
  USE 1 (REWRITE_RULE [point_inj;PAIR_EQ ]);
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t-> REWRITE_TAC [t]);
  MESON_TAC[];
   ]);;
  (* }}} *)

let h_edge_disj = prove_by_refinement(
  `!p q. ~(h_edge p INTER h_edge q = EMPTY) <=> (p = q)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[EMPTY_EXISTS;IN;INTER;IN_ELIM_THM'];
  EQ_TAC;
  DISCH_TAC;
  CHO 0;
  ONCE_REWRITE_TAC [GSYM PAIR];
  REWRITE_TAC[PAIR_EQ];
  CONJ_TAC;
  MP_TAC h_edge_floor;
  REWRITE_TAC[SUBSET;IN;IN_ELIM_THM'];
  ASM_MESON_TAC[];
  MP_TAC h_edge_row;
  MP_TAC row_disj;
  REWRITE_TAC[SUBSET;INTER;IN;IN_ELIM_THM';EMPTY_EXISTS;];
  ASM_MESON_TAC[];
  REWRITE_TAC[h_edge;IN_ELIM_THM' ];
  DISCH_THEN (fun t -> REWRITE_TAC[t;int_add_th ;int_of_num_th;]);
  NAME_CONFLICT_TAC;
  LEFT_TAC "u'";
  TYPE_THEN `?x. (&.0 < x ) /\ (x < &.1)` SUBGOAL_TAC;
  TYPE_THEN `&.1/(&.2)` EXISTS_TAC;
  IMATCH_MP_TAC  half_pos;
  ARITH_TAC;
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `real_of_int (FST q) + x` EXISTS_TAC;
  LEFT_TAC "v'";
  TYPE_THEN `real_of_int (SND q)` EXISTS_TAC ;
  TYPE_THEN `point (real_of_int (FST q) + x,real_of_int (SND q))` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let h_edge_pointI = prove_by_refinement(
  `!p q. ~(h_edge p (pointI q))`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[pointI;h_edge;IN_ELIM_THM' ];
  PROOF_BY_CONTR_TAC;
  USE 0 (REWRITE_RULE[]);
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  USE 0 (REWRITE_RULE[point_inj;PAIR_EQ ]);
  USE 0 GSYM ;
  REWR 1;
  REWR 2;
  USE 2 (REWRITE_RULE[GSYM int_lt ;int_lt_suc_le ]);
  USE 2 (REWRITE_RULE[int_le]);
  UND 2;
  UND 1;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let v_edge_col = prove_by_refinement(
  `!p . v_edge p  SUBSET  col (FST p) `,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;IN;v_edge;col;IN_ELIM_THM';];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let v_edge_floor = prove_by_refinement(
  `!p. v_edge p SUBSET { z | floor (z 1)  = SND  p }`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;IN;v_edge;IN_ELIM_THM';int_of_num_th;int_add_th;];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[coord01;floor_range];
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let col_disj = prove_by_refinement(
  `!a b. ~((col a) INTER (col b) = EMPTY) <=> (a = b)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[EMPTY_EXISTS;IN;INTER;col;IN_ELIM_THM'  ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  AND 0;
  CHO 0;
  CHO 1;
  REWRITE_TAC[int_eq];
  USE 1 (GSYM);
  REWR 1;
  USE 1 (REWRITE_RULE [point_inj;PAIR_EQ ]);
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t-> REWRITE_TAC [t]);
  MESON_TAC[];
   ]);;
  (* }}} *)

let v_edge_disj = prove_by_refinement(
  `!p q. ~(v_edge p INTER v_edge q = EMPTY) <=> (p = q)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[EMPTY_EXISTS;IN;INTER;IN_ELIM_THM'];
  EQ_TAC;
  DISCH_TAC;
  CHO 0;
  ONCE_REWRITE_TAC [GSYM PAIR];
  REWRITE_TAC[PAIR_EQ];
  IMATCH_MP_TAC  (TAUT `a /\ b ==> b/\ a`);
  CONJ_TAC;
  MP_TAC v_edge_floor;
  REWRITE_TAC[SUBSET;IN;IN_ELIM_THM'];
  ASM_MESON_TAC[];
  MP_TAC v_edge_col;
  MP_TAC col_disj;
  REWRITE_TAC[SUBSET;INTER;IN;IN_ELIM_THM';EMPTY_EXISTS;];
  ASM_MESON_TAC[];
  REWRITE_TAC[v_edge;IN_ELIM_THM' ];
  DISCH_THEN (fun t -> REWRITE_TAC[t;int_add_th ;int_of_num_th;]);
  NAME_CONFLICT_TAC;
  LEFT_TAC "u'";
  TYPE_THEN `?x. (&.0 < x ) /\ (x < &.1)` SUBGOAL_TAC;
  TYPE_THEN `&.1/(&.2)` EXISTS_TAC;
  IMATCH_MP_TAC  half_pos;
  ARITH_TAC;
  DISCH_THEN CHOOSE_TAC;
  LEFT_TAC "v'";
  LEFT_TAC "v'";
  TYPE_THEN `real_of_int (SND q) + x` EXISTS_TAC;
  TYPE_THEN `real_of_int (FST  q)` EXISTS_TAC ;
  TYPE_THEN `point (real_of_int (FST q),real_of_int (SND q) +x)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let v_edge_pointI = prove_by_refinement(
  `!p q. ~(v_edge p (pointI q))`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[pointI;v_edge;IN_ELIM_THM' ];
  PROOF_BY_CONTR_TAC;
  USE 0 (REWRITE_RULE[]);
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  USE 0 (REWRITE_RULE[point_inj;PAIR_EQ ]);
  USE 0 GSYM ;
  REWR 1;
  REWR 2;
  USE 2 (REWRITE_RULE[GSYM int_lt ;int_lt_suc_le ]);
  USE 2 (REWRITE_RULE[int_le]);
  UND 2;
  UND 1;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let row_col = prove_by_refinement(
  `!a b. (row b INTER col a) = { (pointI(a,b)) }`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[col;row;INTER;IN;IN_ELIM_THM';pointI];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[IN_ELIM_THM';INSERT;NOT_IN_EMPTY ];
  GEN_TAC;
  ASM_MESON_TAC[PAIR_EQ ;point_inj];
  ]);;
  (* }}} *)

let hv_edge = prove_by_refinement(
  `!p q. h_edge p INTER v_edge q = EMPTY`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `h_edge p INTER v_edge q SUBSET (row (SND p)) INTER (col (FST q))` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET_INTER;];
  MESON_TAC[h_edge_row;v_edge_col;SUB_IMP_INTER ];
  REWRITE_TAC[row_col];
  DISCH_TAC;
  PROOF_BY_CONTR_TAC;
  USE 1 (REWRITE_RULE[EMPTY_EXISTS;IN  ]);
  CHO 1;
  USE 0 (REWRITE_RULE[SUBSET;IN;IN_ELIM_THM';INSERT;EMPTY ]);
  TSPEC `u` 0;
  REWR 0;
  REWR 1;
  USE 1 (REWRITE_RULE[INTER;IN;IN_ELIM_THM';h_edge_pointI]);
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let square_col = prove_by_refinement(
  `!p a. (squ p INTER col a) = EMPTY `,
  (* {{{ proof *)

  [
  REWRITE_TAC[squ;col];
  DISCH_ALL_TAC;
  PROOF_BY_CONTR_TAC;
  USE 0 (REWRITE_RULE[EMPTY_EXISTS;IN ]);
  CHO 0;
  USE 0 (REWRITE_RULE[INTER;IN;IN_ELIM_THM']);
  AND 0;
  CHO 0;
  CHO 1;
  CHO 1;
  UND 1;
  DISCH_ALL_TAC;
  REWR 0;
  USE 0 (REWRITE_RULE[point_inj;PAIR_EQ]);
  REWR 3;
  REWR 2;
  USE 3 (REWRITE_RULE[GSYM int_lt; int_lt_suc_le ;]);
  USE 3 (REWRITE_RULE[ int_le;]);
  UND 2;
  UND 3;
  REAL_ARITH_TAC;
  ]);;

  (* }}} *)

let square_row = prove_by_refinement(
  `!p a. (squ p INTER row a) = EMPTY `,
  (* {{{ proof *)
  [
  REWRITE_TAC[squ;row];
  DISCH_ALL_TAC;
  PROOF_BY_CONTR_TAC;
  USE 0 (REWRITE_RULE[EMPTY_EXISTS;IN ]);
  CHO 0;
  USE 0 (REWRITE_RULE[INTER;IN;IN_ELIM_THM']);
  AND 0;
  CHO 0;
  CHO 1;
  CHO 1;
  UND 1;
  DISCH_ALL_TAC;
  REWR 0;
  USE 0 (REWRITE_RULE[point_inj;PAIR_EQ]);
  REWR 5;
  REWR 4;
  USE 5 (REWRITE_RULE[GSYM int_lt; int_lt_suc_le ;]);
  USE 5 (REWRITE_RULE[ int_le;]);
  UND 5;
  UND 4;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let pointI_row = prove_by_refinement(
  `!p.   (row (SND p)) (pointI p)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[row;pointI;IN_ELIM_THM' ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let pointI_col = prove_by_refinement(
  `!p.   (col (FST p)) (pointI p)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[col;pointI;IN_ELIM_THM' ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let square_v_edge = prove_by_refinement(
  `!p q. (squ p INTER v_edge q = EMPTY)`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  TYPE_THEN `squ p INTER v_edge q SUBSET squ p INTER col (FST q)` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET_INTER];
  MESON_TAC[SUB_IMP_INTER;v_edge_col;SUBSET_REFL];
  REWRITE_TAC[square_col;SUBSET_EMPTY ];
  ]);;
  (* }}} *)

let square_h_edge = prove_by_refinement(
  `!p q. (squ p INTER h_edge q = EMPTY)`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  TYPE_THEN `squ p INTER h_edge q SUBSET squ p INTER row (SND  q)` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET_INTER];
  MESON_TAC[SUB_IMP_INTER;h_edge_row;SUBSET_REFL];
  REWRITE_TAC[square_row;SUBSET_EMPTY ];
  ]);;
  (* }}} *)

let square_pointI = prove_by_refinement(
  `!p q. ~(squ p (pointI q))`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  TYPE_THEN `q` (fun t -> ASSUME_TAC (SPEC t pointI_col));
  TYPEL_THEN [`p`;`FST q`] (fun t -> MP_TAC (SPECL t square_col));
  REWRITE_TAC[INTER;IN;];
  IMATCH_MP_TAC  (TAUT `(a ==> ~b) ==> (b ==> ~ a)`);
  DISCH_TAC;
  REWRITE_TAC[EMPTY_EXISTS;IN ];
  TYPE_THEN `pointI q` EXISTS_TAC;
  ASM_REWRITE_TAC[IN_ELIM_THM'];
  ]);;
  (* }}} *)

let square_floor0 = prove_by_refinement(
  `!p. (squ p SUBSET { z | (floor (z 0)) = (FST p) })`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[SUBSET;IN;IN_ELIM_THM';squ];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[coord01;floor_range];
  UND 1;
  UND 2;
  REWRITE_TAC[int_add_th;int_of_num_th];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let square_floor1 = prove_by_refinement(
  `!p. (squ p SUBSET { z | (floor (z 1)) = (SND p) })`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[SUBSET;IN;IN_ELIM_THM';squ];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[coord01;floor_range];
  UND 3;
  UND 4;
  REWRITE_TAC[int_add_th;int_of_num_th];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let square_square = prove_by_refinement(
  `!p q. ~(squ p INTER squ q = {}) ==> (squ p = squ q)`,
  (* {{{ proof *)
  [
  MP_TAC square_floor0;
  MP_TAC square_floor1;
  REWRITE_TAC[SUBSET;IN;IN_ELIM_THM';INTER;EMPTY_EXISTS  ];
  DISCH_ALL_TAC;
  REP_GEN_TAC;
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `p = q` SUBGOAL_TAC;
  ONCE_REWRITE_TAC [GSYM PAIR];
  REWRITE_TAC[PAIR_EQ];
  ASM_MESON_TAC[];
  MESON_TAC[];
  ]);;
  (* }}} *)

let square_disj = prove_by_refinement(
  `!p q. ~(squ p INTER squ q = EMPTY) <=> (p = q)`,
  (* {{{ proof *)
  [
  MP_TAC square_floor0;
  MP_TAC square_floor1;
  REWRITE_TAC[SUBSET;IN;IN_ELIM_THM';INTER;EMPTY_EXISTS  ];
  DISCH_ALL_TAC;
  REP_GEN_TAC;
  EQ_TAC;
  DISCH_THEN CHOOSE_TAC;
  ONCE_REWRITE_TAC [GSYM PAIR];
  REWRITE_TAC[PAIR_EQ];
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[squ];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "u''");
  TYPE_THEN `real_of_int (FST q) + (&.1/(&.2))` EXISTS_TAC;
  TYPE_THEN `real_of_int (SND q) + (&.1/(&.2))` EXISTS_TAC;
  REWRITE_TAC[int_suc];
  TYPE_THEN `a = real_of_int(FST q)` ABBREV_TAC;
(*** Modified by JRH since ABBREV_TAC now forbids existing variables
  TYPE_THEN `a = real_of_int(SND  q)` ABBREV_TAC;
 ****)
  TYPE_THEN `a' = real_of_int(SND  q)` ABBREV_TAC;
  MP_TAC (REAL_RAT_REDUCE_CONV `&.0 < &.1/(&.2) /\ (&.1/(&.2)) < &.1`);
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)


(* ------------------------------------------------------------------ *)
(*  cells *)
(* ------------------------------------------------------------------ *)


let cell = jordan_def `cell =
  {z | (?p. (z = { (pointI p) }) \/ (z = h_edge p) \/
              (z = v_edge p) \/ (z = squ p))}`;;

let cell_rules = prove_by_refinement(
  `!p. cell {(pointI p)} /\ (cell (h_edge p)) /\
      (cell (v_edge p)) /\ (cell (squ p))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[cell;IN_ELIM_THM';];
  MESON_TAC[];
  ]);;
  (* }}} *)

let cell_mem = prove_by_refinement(
  `!C. (cell C) <=> (?p. C = ({(pointI p)})) \/ (?p. C = h_edge p) \/
    (?p. C = v_edge p) \/ (?p. C = squ p)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[cell;IN_ELIM_THM'];
  MESON_TAC[];
  ]);;
  (* }}} *)

let square_domain = prove_by_refinement(
  `!z.  (let (p = (floor(FST z),floor(SND z))) in
       (({(pointI p)} UNION
        (h_edge p) UNION
        (v_edge p) UNION
        (squ p) ))) (point z) `,
  (* {{{ proof *)
  [
  GEN_TAC;
  LET_TAC;
  REWRITE_TAC[UNION;IN;IN_ELIM_THM' ];
  REWRITE_TAC[pointI;h_edge;v_edge;squ;int_add_th;int_of_num_th;IN_ELIM_THM';INSERT;EMPTY;point_inj;Q_POINT ];
  ASSUME_TAC floor_ineq;
  TYPE_THEN `FST z` (WITH 0 o SPEC);
  TSPEC `SND z` 0;
  UND 0;
  UND 1;
  REWRITE_TAC[PAIR_LEMMAv2];
  REWRITE_TAC[REAL_ARITH `(a <= b) <=> ((a = b) \/ (a < b))`];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let square_cell = prove_by_refinement(
  `!z. (let (p = (floor(FST z),floor(SND z))) in
       (({(pointI p)} UNION
        (h_edge p) UNION
        (v_edge p) UNION
        (squ p) ))) SUBSET (UNIONS cell) `,
  (* {{{ proof *)
  [
  GEN_TAC;
  LET_TAC;
  REWRITE_TAC[union_subset];
  REPEAT CONJ_TAC THEN (IMATCH_MP_TAC  sub_union) THEN (REWRITE_TAC[cell_rules]);
  ]);;
  (* }}} *)

let cell_unions = prove_by_refinement(
  `!z. (UNIONS cell (point z))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  ASM_MESON_TAC[square_cell;square_domain;SUBSET;IN];
  ]);;
  (* }}} *)

let cell_partition = prove_by_refinement(
  `!C D. (cell C) /\ (cell D) /\ ~(C INTER D = EMPTY) ==> (C = D)`,
  (* {{{ proof *)
  let revr = PURE_ONCE_REWRITE_RULE [INTER_COMM] in
  [
  PARTIAL_REWRITE_TAC[cell_mem;];
  PARTIAL_REWRITE_TAC[RIGHT_AND_OVER_OR;LEFT_AND_OVER_OR ];
  REP_GEN_TAC;
  PARTIAL_REWRITE_TAC[TAUT `((a \/ b ==> C)) <=> ((a ==> C) /\ (b ==> C))`];
  PARTIAL_REWRITE_TAC[TAUT `((a /\ b) ==> C) <=> (a ==> b ==> C)`];
  REPEAT CONJ_TAC THEN (REPEAT (DISCH_THEN CHOOSE_TAC)) THEN (TRY (UNDISCH_FIND_TAC `(INTER)`))  THEN (ASM PARTIAL_REWRITE_TAC[])  THEN ASM PARTIAL_REWRITE_TAC[square_h_edge;square_v_edge;revr square_h_edge;revr square_v_edge;v_edge_disj;h_edge_disj;hv_edge;revr hv_edge;revr single_inter; single_inter;square_pointI;v_edge_pointI;h_edge_pointI; square_square;INR NOT_IN_EMPTY;INR IN_SING ] THEN (DISCH_THEN (fun t-> REWRITE_TAC[t]));
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* adjacency, closure, convexity, AND strict dominance on cells. *)
(* ------------------------------------------------------------------ *)


let top2 = jordan_def `top2 = top_of_metric (euclid 2,d_euclid)`;;

let adj = jordan_def `adj X Y <=> (~(X = Y) /\
   ~(closure top2 X INTER (closure top2 Y) = EMPTY))`;;

let strict_dom = jordan_def `strict_dom X Y <=> (cell X) /\ (cell Y) /\
  (closure top2 Y PSUBSET (closure top2 X))`;;

let adj_symm = prove_by_refinement(
  `!X Y. (adj X Y) <=> (adj Y X)`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[adj];
  GEN_REWRITE_TAC (LAND_CONV o ONCE_DEPTH_CONV) [INTER_COMM];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let adj_irrefl = prove_by_refinement(
  `!X. (~(adj X X))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[adj;];
  ]);;
  (* }}} *)

let strict_dom_trans = prove_by_refinement(
  `!X Y Z. (strict_dom X Y) /\ (strict_dom Y Z) ==> (strict_dom X Z)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[strict_dom];
  MESON_TAC[PSUBSET_TRANS];
  ]);;
  (* }}} *)

let strict_dom_irrefl = prove_by_refinement(
  `!X. ~(strict_dom X X)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[strict_dom;PSUBSET_IRREFL ];
  ]);;
  (* }}} *)

let dot_point = prove_by_refinement(
  `!p q . (dot (point p) (point q)) = (FST p)*(FST q) + (SND p)*(SND q)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `dot (point p) (point q) = sum (0,2) (\i. (point p i)*(point q i))` SUBGOAL_TAC;
  IMATCH_MP_TAC dot_euclid;
  ASM_SIMP_TAC[euclid_point];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[ARITH_RULE `2 = SUC 1`];
  REWRITE_TAC[sum];
  REWRITE_TAC[ARITH_RULE `1 = SUC 0`];
  REWRITE_TAC[sum];
  REDUCE_TAC;
  REWRITE_TAC[ARITH_RULE `SUC 0 = 1`;coord01];
  ]);;
  (* }}} *)


(* 2d half planes *)
let open_half_plane2D_FLT = prove_by_refinement(
  `!r. { z | ?p. ((z = point p) /\ (FST p <. r))  } =
     open_half_space 2 (point (&.1,&.0)) r `,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[open_half_space ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASM_REWRITE_TAC[dot_point;euclid_point;];
  REDUCE_TAC;
  ASM_REWRITE_TAC [];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP point_onto);
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[dot_point;euclid_point]);
  USE 1 (CONV_RULE REDUCE_CONV);
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let open_half_plane2D_LTF = prove_by_refinement(
  `!r. { z | ?p. ((z = point p) /\ (r <. FST p ))  } =
     open_half_space 2 (point (--. (&.1),&.0)) (--. r) `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[open_half_space ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASM_REWRITE_TAC[dot_point;euclid_point;];
  REDUCE_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP point_onto);
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[dot_point;euclid_point]);
  USE 1 (CONV_RULE REDUCE_CONV);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let open_half_plane2D_SLT = prove_by_refinement(
  `!r. { z | ?p. ((z = point p) /\ (SND p <. r ))  } =
     open_half_space 2 (point (&.0,&.1)) ( r) `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[open_half_space ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASM_REWRITE_TAC[dot_point;euclid_point;];
  REDUCE_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP point_onto);
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[dot_point;euclid_point]);
  USE 1 (CONV_RULE REDUCE_CONV);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let open_half_plane2D_LTS = prove_by_refinement(
  `!r. { z | ?p. ((z = point p) /\ (r <. SND p  ))  } =
     open_half_space 2 (point (&.0,--.(&.1))) (--. r) `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[open_half_space ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASM_REWRITE_TAC[dot_point;euclid_point;];
  REDUCE_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP point_onto);
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[dot_point;euclid_point]);
  USE 1 (CONV_RULE REDUCE_CONV);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let closed_half_plane2D_FLE = prove_by_refinement(
  `!r. { z | ?p. ((z = point p) /\ (FST p <=. r))  } =
     closed_half_space 2 (point (&.1,&.0)) r `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[open_half_space;hyperplane;closed_half_space ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASM_REWRITE_TAC[dot_point;euclid_point;];
  REDUCE_TAC;
  ASM_REWRITE_TAC [];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP point_onto);
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[dot_point;euclid_point]);
  USE 1 (CONV_RULE REDUCE_CONV);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let closed_half_plane2D_LEF = prove_by_refinement(
  `!r. { z | ?p. ((z = point p) /\ (r <=. FST p))  } =
     closed_half_space 2 (point (--.(&.1),&.0)) (--. r) `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[open_half_space;hyperplane;closed_half_space ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASM_REWRITE_TAC[dot_point;euclid_point;];
  REDUCE_TAC;
  ASM_REWRITE_TAC [];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP point_onto);
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[dot_point;euclid_point]);
  USE 1 (CONV_RULE REDUCE_CONV);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let closed_half_plane2D_SLE = prove_by_refinement(
  `!r. { z | ?p. ((z = point p) /\ (SND p <=. r))  } =
     closed_half_space 2 (point (&.0,&.1)) r `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[open_half_space;hyperplane;closed_half_space ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASM_REWRITE_TAC[dot_point;euclid_point;];
  REDUCE_TAC;
  ASM_REWRITE_TAC [];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP point_onto);
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[dot_point;euclid_point]);
  USE 1 (CONV_RULE REDUCE_CONV);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let closed_half_plane2D_LES = prove_by_refinement(
  `!r. { z | ?p. ((z = point p) /\ (r <=. SND p ))  } =
     closed_half_space 2 (point (&.0,(--. (&.1)))) (--. r) `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[open_half_space;hyperplane;closed_half_space ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASM_REWRITE_TAC[dot_point;euclid_point;];
  REDUCE_TAC;
  ASM_REWRITE_TAC [];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP point_onto);
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[dot_point;euclid_point]);
  USE 1 (CONV_RULE REDUCE_CONV);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let line2D_F = prove_by_refinement(
  `!r. { z | ?p. ((z = point p) /\ (FST p = r))  } =
     hyperplane 2 (point (&.1,&.0)) r `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[open_half_space;hyperplane;closed_half_space ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASM_REWRITE_TAC[dot_point;euclid_point;];
  REDUCE_TAC;
  ASM_REWRITE_TAC [];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP point_onto);
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[dot_point;euclid_point]);
  USE 1 (CONV_RULE REDUCE_CONV);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let line2D_S = prove_by_refinement(
  `!r. { z | ?p. ((z = point p) /\ (SND p = r))  } =
     hyperplane 2 (point (&.0,&.1)) r `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[open_half_space;hyperplane;closed_half_space ];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASM_REWRITE_TAC[dot_point;euclid_point;];
  REDUCE_TAC;
  ASM_REWRITE_TAC [];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP point_onto);
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[dot_point;euclid_point]);
  USE 1 (CONV_RULE REDUCE_CONV);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let open_half_plane2D_FLT_open = prove_by_refinement(
  `!r. top2 { z | ?p. ((z = point p) /\ (FST p <. r))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_FLT;top2];
  SIMP_TAC[open_half_space_open;euclid_point];
  ]);;
  (* }}} *)

let open_half_plane2D_LTF_open = prove_by_refinement(
  `!r. top2 { z | ?p. ((z = point p) /\ (r <. FST p ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_LTF;top2];
  SIMP_TAC[open_half_space_open;euclid_point];
  ]);;
  (* }}} *)

let open_half_plane2D_SLT_open = prove_by_refinement(
  `!r. top2 { z | ?p. ((z = point p) /\ (SND p <. r  ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_SLT;top2];
  SIMP_TAC[open_half_space_open;euclid_point];
  ]);;
  (* }}} *)

let open_half_plane2D_LTS_open = prove_by_refinement(
  `!r. top2 { z | ?p. ((z = point p) /\ (r <. SND p   ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_LTS;top2];
  SIMP_TAC[open_half_space_open;euclid_point];
  ]);;
  (* }}} *)

let closed_half_plane2D_FLT_closed = prove_by_refinement(
  `!r. closed_ top2 { z | ?p. ((z = point p) /\ (FST p <=. r))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[closed_half_plane2D_FLE;top2];
  SIMP_TAC[closed_half_space_closed;euclid_point];
  ]);;
  (* }}} *)

let closed_half_plane2D_LTF_closed = prove_by_refinement(
  `!r. closed_ top2 { z | ?p. ((z = point p) /\ (r <=. FST p ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[closed_half_plane2D_LEF;top2];
  SIMP_TAC[closed_half_space_closed;euclid_point];
  ]);;
  (* }}} *)

let closed_half_plane2D_SLT_closed = prove_by_refinement(
  `!r. closed_ top2 { z | ?p. ((z = point p) /\ (SND p <=. r  ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[closed_half_plane2D_SLE;top2];
  SIMP_TAC[closed_half_space_closed;euclid_point];
  ]);;
  (* }}} *)

let closed_half_plane2D_LTS_closed = prove_by_refinement(
  `!r. closed_ top2 { z | ?p. ((z = point p) /\ (r <=. SND p   ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[closed_half_plane2D_LES;top2];
  SIMP_TAC[closed_half_space_closed;euclid_point];
  ]);;
  (* }}} *)

let line2D_F_closed = prove_by_refinement(
  `!r. closed_ top2 { z | ?p. ((z = point p) /\ (FST p = r))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[line2D_F;top2];
  SIMP_TAC[hyperplane_closed;euclid_point];
  ]);;
  (* }}} *)

let line2D_S_closed = prove_by_refinement(
  `!r. closed_ top2 { z | ?p. ((z = point p) /\ (SND p = r))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[line2D_S;top2];
  SIMP_TAC[hyperplane_closed;euclid_point];
  ]);;
  (* }}} *)

let open_half_plane2D_FLT_convex = prove_by_refinement(
  `!r. convex { z | ?p. ((z = point p) /\ (FST p <. r))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_FLT;];
  SIMP_TAC[open_half_space_convex;euclid_point];
  ]);;
  (* }}} *)

let open_half_plane2D_LTF_convex = prove_by_refinement(
  `!r. convex { z | ?p. ((z = point p) /\ (r <. FST p ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_LTF;];
  SIMP_TAC[open_half_space_convex;euclid_point];
  ]);;
  (* }}} *)

let open_half_plane2D_SLT_convex = prove_by_refinement(
  `!r. convex { z | ?p. ((z = point p) /\ (SND p <. r))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_SLT;];
  SIMP_TAC[open_half_space_convex;euclid_point];
  ]);;
  (* }}} *)

let open_half_plane2D_LTS_convex = prove_by_refinement(
  `!r. convex { z | ?p. ((z = point p) /\ (r <. SND p ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_LTS;];
  SIMP_TAC[open_half_space_convex;euclid_point];
  ]);;
  (* }}} *)

let closed_half_plane2D_FLT_convex = prove_by_refinement(
  `!r. convex { z | ?p. ((z = point p) /\ (FST p <=. r))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[closed_half_plane2D_FLE;];
  SIMP_TAC[closed_half_space_convex;euclid_point];
  ]);;
  (* }}} *)

let closed_half_plane2D_LTF_convex = prove_by_refinement(
  `!r. convex { z | ?p. ((z = point p) /\ (r <=. FST p ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[closed_half_plane2D_LEF;];
  SIMP_TAC[closed_half_space_convex;euclid_point];
  ]);;
  (* }}} *)

let closed_half_plane2D_SLT_convex = prove_by_refinement(
  `!r. convex { z | ?p. ((z = point p) /\ (SND p <=. r))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[closed_half_plane2D_SLE;];
  SIMP_TAC[closed_half_space_convex;euclid_point];
  ]);;
  (* }}} *)

let closed_half_plane2D_LTS_convex = prove_by_refinement(
  `!r. convex { z | ?p. ((z = point p) /\ (r <=. SND p ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[closed_half_plane2D_LES;];
  SIMP_TAC[closed_half_space_convex;euclid_point];
  ]);;
  (* }}} *)

let line2D_F_convex = prove_by_refinement(
  `!r. convex { z | ?p. ((z = point p) /\ ( FST p = r ))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[line2D_F;];
  SIMP_TAC[hyperplane_convex;euclid_point];
  ]);;
  (* }}} *)

let line2D_S_convex = prove_by_refinement(
  `!r. convex { z | ?p. ((z = point p) /\ (SND p = r))  }`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[line2D_S;];
  SIMP_TAC[hyperplane_convex;euclid_point];
  ]);;
  (* }}} *)

let closure_FLT = prove_by_refinement(
  `!r. (closure top2 { z | ?p. ((z = point p) /\ (FST p <. r))  } =
       { z | ?p. ((z = point p) /\ (FST p <=. r))  })`,
  (* {{{ proof *)

  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_FLT;closed_half_plane2D_FLE;top2];
  TYPE_THEN `~(point(&.1,&.0) = euclid0)` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  USE 0(REWRITE_RULE[]);
  USE 0  (fun t -> AP_THM t `0`);
  USE 0 (REWRITE_RULE[coord01;euclid0;REAL_ARITH `~(&.1= &.0)`]);
  ASM_REWRITE_TAC[];
  SIMP_TAC[closure_half_space;euclid_point];
  ]);;

  (* }}} *)

let closure_LTF = prove_by_refinement(
  `!r. (closure top2 { z | ?p. ((z = point p) /\ (r <. FST p))  } =
       { z | ?p. ((z = point p) /\ (r <=. FST p ))  })`,
  (* {{{ proof *)

  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_LTF;closed_half_plane2D_LEF;top2];
  TYPE_THEN `~(point(--. (&.1),&.0) = euclid0)` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  USE 0(REWRITE_RULE[]);
  USE 0  (fun t -> AP_THM t `0`);
  USE 0 (REWRITE_RULE[coord01;euclid0;REAL_ARITH `~(--. (&.1)= &.0)`]);
  ASM_REWRITE_TAC[];
  SIMP_TAC[closure_half_space;euclid_point];
  ]);;

  (* }}} *)

let closure_SLT = prove_by_refinement(
  `!r. (closure top2 { z | ?p. ((z = point p) /\ (SND  p <. r))  } =
       { z | ?p. ((z = point p) /\ (SND  p <=. r))  })`,
  (* {{{ proof *)

  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_SLT;closed_half_plane2D_SLE;top2];
  TYPE_THEN `~(point(&.0,&.1) = euclid0)` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  USE 0(REWRITE_RULE[]);
  USE 0  (fun t -> AP_THM t `1`);
  USE 0 (REWRITE_RULE[coord01;euclid0;REAL_ARITH `~(&.1= &.0)`]);
  ASM_REWRITE_TAC[];
  SIMP_TAC[closure_half_space;euclid_point];
  ]);;

  (* }}} *)

let closure_LTS = prove_by_refinement(
  `!r. (closure top2 { z | ?p. ((z = point p) /\ (r <. SND  p))  } =
       { z | ?p. ((z = point p) /\ (r <=. SND  p ))  })`,
  (* {{{ proof *)

  [
  GEN_TAC;
  REWRITE_TAC[open_half_plane2D_LTS;closed_half_plane2D_LES;top2];
  TYPE_THEN `~(point(&.0, --. (&.1)) = euclid0)` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  USE 0(REWRITE_RULE[]);
  USE 0  (fun t -> AP_THM t `1`);
  USE 0 (REWRITE_RULE[coord01;euclid0;REAL_ARITH `~(--. (&.1)= &.0)`]);
  ASM_REWRITE_TAC[];
  SIMP_TAC[closure_half_space;euclid_point];
  ]);;

  (* }}} *)



(* ------------------------------------------------------------------ *)
(* SECTION B *)
(* ------------------------------------------------------------------ *)

(* -> sets *)
let single_subset = prove_by_refinement(
  `!(x:A) A. ({x} SUBSET A) <=> (A x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;INSERT];
  MESON_TAC[];
  ]);;
  (* }}} *)

let top2_top = prove_by_refinement(
  `topology_ top2  `,
  (* {{{ proof *)
  [
  ASM_SIMP_TAC [top2;top_of_metric_top;metric_euclid];
  ]);;
  (* }}} *)


(* ------------------------------------------------------------------ *)
(* H_edge & v_edge, convexity, closure, closed, adj, etc. *)
(* ------------------------------------------------------------------ *)

let e1 = jordan_def `e1 = point(&.1,&.0)`;;
let e2 = jordan_def `e2 = point(&.0,&.1)`;;

let hc_edge = jordan_def `hc_edge m =
   (h_edge m) UNION {(pointI m)} UNION {(pointI m + e1)}`;;

let vc_edge = jordan_def `vc_edge m =
   (v_edge m) UNION {(pointI m)} UNION {(pointI m + e2)}`;;



(* H edge *)
let h_edge_inter = prove_by_refinement(
  `!m. (h_edge m) =
   ({z | ?p. (z = point p) /\ (real_of_int (FST  m) <. FST p)}
      INTER {z | ?p. (z = point p) /\ (FST p <. real_of_int(FST  m +: &:1))}
      INTER {z | ?p. (z = point p) /\ (SND p = real_of_int(SND  m))})`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  REWRITE_TAC[INTER;h_edge];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[point_inj];
  REPEAT CONJ_TAC THEN (TYPE_THEN `(u,real_of_int(SND m))` EXISTS_TAC) THEN ASM_REWRITE_TAC[PAIR_SPLIT];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 1;
  CHO 2;
  TYPE_THEN `FST p` EXISTS_TAC;
  TYPE_THEN `SND  p` EXISTS_TAC;
  REWR 1;
  REWR 2;
  USE 2 (REWRITE_RULE[point_inj]);
  USE 1 (REWRITE_RULE[point_inj]);
  AND 1;
  AND 2;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let h_edge_convex = prove_by_refinement(
  `!m. (convex (h_edge m))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[h_edge_inter;];
  IMATCH_MP_TAC convex_inter;
  CONJ_TAC;
  REWRITE_TAC [open_half_plane2D_LTF_convex;];
  IMATCH_MP_TAC  convex_inter;
  REWRITE_TAC[open_half_plane2D_FLT_convex;line2D_S_convex];
  ]);;
  (* }}} *)

let hc_edge_inter = prove_by_refinement(
  `!m. (hc_edge m) =
   ({z | ?p. (z = point p) /\ (real_of_int (FST  m) <=. FST p)}
      INTER {z | ?p. (z = point p) /\ (FST p <=. real_of_int(FST  m +: &:1))}
      INTER {z | ?p. (z = point p) /\ (SND p = real_of_int(SND  m))})`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[hc_edge;e1];
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[union_subset];
  REPEAT (CONJ_TAC);
  REWRITE_TAC[h_edge_inter];
  REWRITE_TAC[SUBSET;INTER];
  ASM_MESON_TAC[REAL_ARITH `a < b ==> a <=. b`];
  REWRITE_TAC[single_subset;INTER;pointI;point_inj;PAIR_SPLIT ; int_suc];
  REPEAT (CONJ_TAC) THEN (TYPE_THEN `(real_of_int(FST m),real_of_int(SND m))` EXISTS_TAC) THEN REWRITE_TAC[] THEN  ASM_MESON_TAC[REAL_LE_REFL;REAL_ARITH `x <=. x+ &.1`];
  REWRITE_TAC[single_subset;INTER;pointI;point_inj;PAIR_SPLIT ; point_add;int_suc];
  REDUCE_TAC;
  REPEAT (CONJ_TAC) THEN (TYPE_THEN `(real_of_int(FST m) + &.1,real_of_int(SND m))` EXISTS_TAC) THEN REWRITE_TAC[] THEN  ASM_MESON_TAC[REAL_LE_REFL;REAL_ARITH `x <=. x+ &.1`];
  REWRITE_TAC[INTER;SUBSET;UNION;e1;h_edge;pointI;point_add;point_inj;INR IN_SING ;int_suc ];
  GEN_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  REWR 1;
  REWR 2;
  ASM_REWRITE_TAC[point_inj;PAIR_SPLIT ];
  REWRITE_TAC[prove_by_refinement( `!P x y. (?u v. (((x:A) = u) /\ ((y:B) = v)) /\ P u v) <=> (P x y)`,[MESON_TAC[]])];
  UND 2;
  UND 1;
  REWRITE_TAC[point_inj;];
  REWRITE_TAC[prove_by_refinement (`!Q p. (?p'. ((p:A) = p') /\ (Q p')) <=> (Q p)`,[MESON_TAC[]])];
  AND 0;
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let hc_edge_closed = prove_by_refinement(
  `!m. (closed_ top2 (hc_edge m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[hc_edge_inter];
  GEN_TAC;
  IMATCH_MP_TAC  closed_inter2;
  REWRITE_TAC[top2_top;closed_half_plane2D_LTF_closed];
  IMATCH_MP_TAC  closed_inter2;
  REWRITE_TAC[top2_top;closed_half_plane2D_FLT_closed;line2D_S_closed;];
  ]);;
  (* }}} *)

let hc_edge_convex = prove_by_refinement(
  `!m. (convex (hc_edge m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[hc_edge_inter];
  GEN_TAC;
  IMATCH_MP_TAC convex_inter;
  REWRITE_TAC[closed_half_plane2D_LTF_convex];
  IMATCH_MP_TAC  convex_inter;
  REWRITE_TAC[closed_half_plane2D_FLT_convex;line2D_S_convex;];
  ]);;
  (* }}} *)

let h_edge_subset = prove_by_refinement(
  `!m. (h_edge m SUBSET hc_edge m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[hc_edge;SUBSET;UNION;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let h_edge_euclid = prove_by_refinement(
  `!m. (h_edge m) SUBSET (euclid 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;h_edge];
  MESON_TAC[euclid_point];
  ]);;
  (* }}} *)

let h_edge_closure = prove_by_refinement(
  `!m. (closure top2 (h_edge m)) = hc_edge m`,
  (* {{{ proof *)
  [
  GEN_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC  closure_subset;
  REWRITE_TAC[h_edge_subset;top2_top;hc_edge_closed];
  REWRITE_TAC[hc_edge];
  REWRITE_TAC[union_subset;e1;pointI;single_subset;point_add];
  CONJ_TAC;
  IMATCH_MP_TAC  subset_closure;
  REWRITE_TAC[top2_top];
  REWRITE_TAC[top2];
  SUBGOAL_TAC `!t u. t*u +. (&.1- t)*u = u` ;
  REWRITE_TAC[GSYM REAL_RDISTRIB];
  REAL_ARITH_TAC;
  DISCH_TAC;
  CONJ_TAC THEN (IMATCH_MP_TAC  closure_segment) THEN REWRITE_TAC[h_edge_euclid];
  TYPE_THEN `(pointI m)+point(&.1,&.0)` EXISTS_TAC;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[h_edge;pointI;point_add;point_scale;PAIR_SPLIT;point_inj;];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  REDUCE_TAC;
  ASM_REWRITE_TAC[int_suc];
  TYPE_THEN `a = real_of_int(FST m)` ABBREV_TAC;
  UND 1;
  UND 2;
  REAL_ARITH_TAC ;
  TYPE_THEN `pointI m` EXISTS_TAC;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[h_edge;pointI;point_add;point_scale;PAIR_SPLIT;point_inj;];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  REDUCE_TAC;
  ASM_REWRITE_TAC[int_suc];
  TYPE_THEN `a = real_of_int(FST m)` ABBREV_TAC;
  UND 1;
  UND 2;
  REAL_ARITH_TAC ;
  ]);;

  (* }}} *)

(* move up *)
let point_split = prove_by_refinement(
  `!z u v. (z = point(u,v)) <=> (u = z 0) /\ (v = z 1) /\ (euclid 2 z)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  EQ_TAC ;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[coord01;euclid_point];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  DISJ_CASES_TAC (ARITH_RULE  `(x = 0) \/ (x = 1) \/ (2 <= x)`);
  ASM_REWRITE_TAC[coord01];
  UND 3;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[coord01];
  ASM_MESON_TAC[euclid;euclid_point]
  ]);;
  (* }}} *)


(* V edge *)
let v_edge_inter = prove_by_refinement(
  `!m. (v_edge m) =
   ({z | ?p. (z = point p) /\ (real_of_int (SND   m) <. SND  p)}
      INTER {z | ?p. (z = point p) /\ (SND  p <. real_of_int(SND  m +: &:1))}
      INTER {z | ?p. (z = point p) /\ (FST p = real_of_int(FST  m))})`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[INTER;v_edge;int_suc ];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[point_inj];
  CONV_TAC (dropq_conv "p");
  ASM_REWRITE_TAC[];
  CONV_TAC (dropq_conv "p");
  CONV_TAC (dropq_conv "p'");
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "u");
  REWRITE_TAC[point_split;];
  CONV_TAC (dropq_conv "v");
  ASM_MESON_TAC[FST;SND;PAIR;coord01;euclid_point;point_onto];
  ]);;
  (* }}} *)

let v_edge_convex = prove_by_refinement(
  `!m. (convex (v_edge m))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[v_edge_inter;];
  IMATCH_MP_TAC convex_inter;
  CONJ_TAC;
  REWRITE_TAC [open_half_plane2D_LTS_convex;];
  IMATCH_MP_TAC  convex_inter;
  REWRITE_TAC[open_half_plane2D_SLT_convex;line2D_F_convex];
  ]);;
  (* }}} *)

let vc_edge_inter = prove_by_refinement(
  `!m. (vc_edge m) =
   ({z | ?p. (z = point p) /\ (real_of_int (SND   m) <=. SND  p)}
      INTER {z | ?p. (z = point p) /\ (SND p <=. real_of_int(SND  m +: &:1))}
      INTER {z | ?p. (z = point p) /\ (FST  p = real_of_int(FST   m))})`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[vc_edge;e2];
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[union_subset];
  REPEAT (CONJ_TAC);
  REWRITE_TAC[v_edge_inter];
  REWRITE_TAC[SUBSET;INTER];
  ASM_MESON_TAC[REAL_ARITH `a < b ==> a <=. b`];
  REWRITE_TAC[single_subset;INTER;pointI;point_inj;PAIR_SPLIT ; int_suc];
  REPEAT (CONJ_TAC) THEN (TYPE_THEN `(real_of_int(FST m),real_of_int(SND m))` EXISTS_TAC) THEN REWRITE_TAC[] THEN  ASM_MESON_TAC[REAL_LE_REFL;REAL_ARITH `x <=. x+ &.1`];
  REWRITE_TAC[single_subset;INTER;pointI;point_inj;PAIR_SPLIT ; point_add;int_suc];
  REDUCE_TAC;
  REPEAT (CONJ_TAC) THEN (TYPE_THEN `(real_of_int(FST m) ,real_of_int(SND m) + &.1)` EXISTS_TAC) THEN REWRITE_TAC[] THEN  ASM_MESON_TAC[REAL_LE_REFL;REAL_ARITH `x <=. x+ &.1`];
  REWRITE_TAC[INTER;SUBSET;UNION;e2;v_edge;pointI;point_add;point_inj;INR IN_SING ;int_suc ];
  GEN_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  REWR 1;
  REWR 2;
  ASM_REWRITE_TAC[point_inj;PAIR_SPLIT ];
  REWRITE_TAC[prove_by_refinement( `!P x y. (?u v. (((x:A) = u) /\ ((y:B) = v)) /\ P u v) <=> (P x y)`,[MESON_TAC[]])];
  UND 2;
  UND 1;
  REWRITE_TAC[point_inj;];
  REWRITE_TAC[prove_by_refinement (`!Q p. (?p'. ((p:A) = p') /\ (Q p')) <=> (Q p)`,[MESON_TAC[]])];
  AND 0;
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let vc_edge_closed = prove_by_refinement(
  `!m. (closed_ top2 (vc_edge m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[vc_edge_inter];
  GEN_TAC;
  IMATCH_MP_TAC  closed_inter2;
  REWRITE_TAC[top2_top;closed_half_plane2D_LTS_closed];
  IMATCH_MP_TAC  closed_inter2;
  REWRITE_TAC[top2_top;closed_half_plane2D_SLT_closed;line2D_F_closed;];
  ]);;
  (* }}} *)

let vc_edge_convex = prove_by_refinement(
  `!m. (convex (vc_edge m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[vc_edge_inter];
  GEN_TAC;
  IMATCH_MP_TAC convex_inter;
  REWRITE_TAC[closed_half_plane2D_LTS_convex];
  IMATCH_MP_TAC  convex_inter;
  REWRITE_TAC[closed_half_plane2D_SLT_convex;line2D_F_convex;];
  ]);;
  (* }}} *)

let v_edge_subset = prove_by_refinement(
  `!m. (v_edge m SUBSET vc_edge m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[vc_edge;SUBSET;UNION;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let v_edge_euclid = prove_by_refinement(
  `!m. (v_edge m) SUBSET (euclid 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;v_edge];
  MESON_TAC[euclid_point];
  ]);;
  (* }}} *)

let v_edge_closure = prove_by_refinement(
  `!m. (closure top2 (v_edge m)) = vc_edge m`,
  (* {{{ proof *)
  [
  GEN_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC  closure_subset;
  REWRITE_TAC[v_edge_subset;top2_top;vc_edge_closed];
  REWRITE_TAC[vc_edge];
  REWRITE_TAC[union_subset;e2;pointI;single_subset;point_add];
  CONJ_TAC;
  IMATCH_MP_TAC  subset_closure;
  REWRITE_TAC[top2_top];
  REWRITE_TAC[top2];
  SUBGOAL_TAC `!t u. t*u +. (&.1- t)*u = u` ;
  REWRITE_TAC[GSYM REAL_RDISTRIB];
  REAL_ARITH_TAC;
  DISCH_TAC;
  CONJ_TAC THEN (IMATCH_MP_TAC  closure_segment) THEN REWRITE_TAC[v_edge_euclid];
  TYPE_THEN `(pointI m)+point(&.0,&.1)` EXISTS_TAC;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[v_edge;pointI;point_add;point_scale;PAIR_SPLIT;point_inj;];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  REDUCE_TAC;
  ASM_REWRITE_TAC[int_suc];
  TYPE_THEN `a = real_of_int(FST m)` ABBREV_TAC;
  UND 1;
  UND 2;
  REAL_ARITH_TAC ;
  TYPE_THEN `pointI m` EXISTS_TAC;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[v_edge;pointI;point_add;point_scale;PAIR_SPLIT;point_inj;];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  REDUCE_TAC;
  ASM_REWRITE_TAC[int_suc];
  TYPE_THEN `a = real_of_int(FST m)` ABBREV_TAC;
  UND 1;
  UND 2;
  REAL_ARITH_TAC ;
  ]);;

  (* }}} *)

let squ_euclid = prove_by_refinement(
  `!m. (squ m) SUBSET (euclid 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;squ];
  MESON_TAC[euclid_point];
  ]);;
  (* }}} *)

let cell_euclid = prove_by_refinement(
  `!X. (cell X) ==> (X SUBSET euclid 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[cell];
  GEN_TAC;
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  REP_CASES_TAC THEN ASM_REWRITE_TAC[h_edge_euclid;squ_euclid;v_edge_euclid];
  REWRITE_TAC[ISUBSET;INR IN_SING;pointI;euclid_point];
  ASM_MESON_TAC[euclid_point];
  ]);;
  (* }}} *)

let edge = jordan_def `edge C <=> ?m. ((C = v_edge m) \/ (C = h_edge m))`;;

let edge_v = prove_by_refinement(
  `!m. edge (v_edge m)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[edge];
  ]);;
  (* }}} *)

let edge_h = prove_by_refinement(
  `!m. edge (h_edge m)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[edge];
  ]);;
  (* }}} *)

let num_closure = jordan_def `num_closure G x =
      CARD { C | (G C) /\ (closure top2 C x) }`;;

let num_lower = jordan_def `num_lower G n =
   CARD { m | (G (h_edge m)) /\ (FST m = FST n) /\ (SND m <=: SND n) }`;;

let set_lower = jordan_def `set_lower G n =
    { m | (G (h_edge m)) /\ (FST m = FST n) /\ (SND m <=: SND n) }`;;

let num_lower_set = prove_by_refinement(
  `!G n. num_lower G n = CARD (set_lower G n)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[num_lower;set_lower];
  ]);;
  (* }}} *)

let even_cell = jordan_def `even_cell G C <=>
   (?m. (C = {(pointI m)}) /\ (EVEN (num_lower G m))) \/
   (?m. (C = h_edge m) /\ (EVEN (num_lower G m))) \/
   (?m. (C = v_edge m) /\ (EVEN (num_lower G m))) \/
   (?m. (C = squ m) /\ (EVEN (num_lower G m)))`;;

(* set *)
let eq_sing = prove_by_refinement(
(*** Parens added by JRH; parser no longer hacks "=" specially
     so it is really right associative
  `!X (y:A). X = {y} = ((X y) /\ (!u. (X u) ==> (u=y)))`,
 ***)
  `!X (y:A). (X = {y}) <=> ((X y) /\ (!u. (X u) ==> (u=y)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INSERT ;];
  DISCH_ALL_TAC;
  EQ_TAC ;
  DISCH_THEN_REWRITE;
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let h_edge_pointIv2 = prove_by_refinement(
  `!p q. ~(h_edge p = {(pointI q)})`,
  (* {{{ proof *)
  [
  REWRITE_TAC[eq_sing;h_edge_pointI];
  ]);;
  (* }}} *)

let v_edge_pointIv2 = prove_by_refinement(
  `!p q. ~(v_edge p = {(pointI q)})`,
  (* {{{ proof *)
  [
  REWRITE_TAC[eq_sing;v_edge_pointI];
  ]);;
  (* }}} *)

let square_pointIv2 = prove_by_refinement(
  `!p q. ~(squ p = {(pointI q)})`,
  (* {{{ proof *)
  [
  REWRITE_TAC[eq_sing;square_pointI];
  ]);;
  (* }}} *)

let cell_nonempty = prove_by_refinement(
  `!z. (cell z) ==> ~(z = EMPTY)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[cell_mem];
  GEN_TAC;
  REP_CASES_TAC ;
  CHO 1;
  USE 1(  REWRITE_RULE [eq_sing]);
  ASM_MESON_TAC[EMPTY];
  CHO 1;
  ASM_MESON_TAC[h_edge_disj;INTER_EMPTY];
  CHO 1;
  ASM_MESON_TAC[v_edge_disj;INTER_EMPTY];
  CHO 1;
  ASM_MESON_TAC[square_disj;INTER_EMPTY];
  ]);;
  (* }}} *)

let hv_edgeV2 = prove_by_refinement(
  `!p q. ~(h_edge p = v_edge q)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[cell_rules;cell_nonempty;hv_edge;INTER_IDEMPOT];
  ]);;
  (* }}} *)

let square_v_edgeV2 = prove_by_refinement(
  `!p q. ~(squ p = v_edge q)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[cell_rules;cell_nonempty;square_v_edge;INTER_IDEMPOT];
  ]);;
  (* }}} *)

let square_h_edgeV2 = prove_by_refinement(
  `!p q. ~(squ p = h_edge q)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[cell_rules;cell_nonempty;square_h_edge;INTER_IDEMPOT];
  ]);;
  (* }}} *)

let h_edge_inj = prove_by_refinement(
  `!p q . (h_edge p = h_edge q) <=> (p = q)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[cell_rules;cell_nonempty;h_edge_disj;INTER_IDEMPOT];
  ]);;
  (* }}} *)

let v_edge_inj = prove_by_refinement(
  `!p q . (v_edge p = v_edge q) <=> (p = q)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[cell_rules;cell_nonempty;v_edge_disj;INTER_IDEMPOT];
  ]);;
  (* }}} *)

let squ_inj = prove_by_refinement(
  `!p q . (squ p = squ q) <=> (p = q)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[cell_rules;cell_nonempty;square_disj;INTER_IDEMPOT];
  ]);;
  (* }}} *)

let finite_set_lower = prove_by_refinement(
  `!G n. (FINITE G) ==> (FINITE (set_lower G n))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `INJ h_edge (set_lower G n) G` SUBGOAL_TAC;
  REWRITE_TAC[INJ;set_lower;h_edge_inj];
  ASM_MESON_TAC[];
  DISCH_TAC;
  JOIN  0 1;
  USE 0 (MATCH_MP FINITE_INJ);
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let even_cell_point = prove_by_refinement(
  `!G m. even_cell G {(pointI m)} <=> EVEN(num_lower G m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[even_cell;square_pointIv2;v_edge_pointIv2;h_edge_pointIv2];
  REWRITE_TAC[pointI_inj;INSERT;eq_sing];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let even_cell_h_edge = prove_by_refinement(
  `!G m. even_cell G (h_edge m) <=> EVEN(num_lower G m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[even_cell;h_edge_pointIv2];
  REWRITE_TAC[pointI_inj;INSERT;h_edge_inj;GSYM square_h_edgeV2;hv_edgeV2;eq_sing];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let even_cell_v_edge = prove_by_refinement(
  `!G m. even_cell G (v_edge m) <=> EVEN(num_lower G m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[even_cell;v_edge_pointIv2];
  REWRITE_TAC[pointI_inj;INSERT;v_edge_inj;GSYM square_v_edgeV2;hv_edgeV2;eq_sing];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let even_cell_squ = prove_by_refinement(
  `!G m. even_cell G (squ m) <=> EVEN(num_lower G m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[even_cell;v_edge_pointIv2];
  REWRITE_TAC[pointI_inj;INSERT;squ_inj;GSYM square_v_edgeV2;GSYM square_h_edgeV2;square_pointI;eq_sing];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let h_edge_squ_parity = prove_by_refinement(
  `!G m. even_cell G (h_edge m) <=> even_cell G (squ m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[even_cell_squ;even_cell_h_edge;num_lower];
  ]);;
  (* }}} *)

let up = jordan_def `up (m:int#int) = (FST m,SND m +: (&:1))`;;
let down = jordan_def `down (m:int#int) = (FST m,SND m -: (&:1))`;;
let left = jordan_def `left (m:int#int) = (FST m -: (&:1),SND m)`;;
let right = jordan_def `right (m:int#int) = (FST m +: (&:1),SND m)`;;

let set_lower_delete = prove_by_refinement(
  `!G n. set_lower G (down n) = (set_lower G n) DELETE n`,
  (* {{{ proof *)
  [
  REWRITE_TAC[set_lower;down;DELETE ];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[PAIR_SPLIT;INT_LE_SUB_LADD;GSYM INT_LT_DISCRETE;];
  REWRITE_TAC[int_le;int_lt;];
  REWRITE_TAC[ (ARITH_RULE `! x y. (x <. y) <=> ((x <= y) /\ ~(x = y))`)];
  REWRITE_TAC[GSYM int_eq];
  MESON_TAC[];
  ]);;
  (* }}} *)

let set_lower_n = prove_by_refinement(
  `!G n. set_lower G n n = (G (h_edge n))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[set_lower;int_le ; REAL_LE_REFL];
  ]);;
  (* }}} *)

(* set *)
let CARD_SUC_DELETE = prove_by_refinement(
  `!(x:A) s. FINITE s /\ s x ==>
    ((SUC (CARD (s DELETE x))) = CARD s)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `s = (x INSERT (s DELETE x))` SUBGOAL_TAC;
  ASM_MESON_TAC[INR INSERT_DELETE];
  USE 0 (ONCE_REWRITE_RULE[GSYM FINITE_DELETE]);
  TYPE_THEN `b = s DELETE x`  ABBREV_TAC ;
  DISCH_THEN_REWRITE;
  ASM_SIMP_TAC [INR CARD_CLAUSES];
  COND_CASES_TAC;
  ASM_MESON_TAC[INR IN_DELETE];
  REWRITE_TAC[];
  ]);;
  (* }}} *)

let even_delete = prove_by_refinement(
  `!(x:A) s. FINITE s ==>
     ((EVEN (CARD (s DELETE x)) <=> EVEN (CARD s)) <=> ~(s x))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `s x`  ASM_CASES_TAC ;
  ASM_MESON_TAC[CARD_SUC_DELETE;EVEN ];
  ASM_SIMP_TAC[CARD_DELETE];
  ]);;
  (* }}} *)

let num_lower_down = prove_by_refinement(
  `!G m. (FINITE G) ==>
       ((EVEN (num_lower G (down m)) <=> EVEN (num_lower G m)) <=>
           ~(set_lower G m m))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[num_lower_set;set_lower_delete];
  IMATCH_MP_TAC  even_delete;
  REWRITE_TAC[even_cell_squ;even_cell_h_edge;num_lower;down];
  ASM_MESON_TAC[finite_set_lower];
  ]);;
  (* }}} *)

let squ_down = prove_by_refinement(
  `!G m. (FINITE G) ==>
        ((even_cell G (squ (down m)) <=> even_cell G (squ m)) <=>
             ~(set_lower G m m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[even_cell_squ;num_lower_down];
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(*  edge combinatorics *)
(* ------------------------------------------------------------------ *)

let pair_size_2 = prove_by_refinement(
  `!(a:A) b. ~(a= b) ==> ({a, b} HAS_SIZE 2)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[HAS_SIZE];
  ASM_SIMP_TAC[FINITE_SING;CARD_CLAUSES;INR IN_SING ];
  CONJ_TAC;
  REWRITE_TAC[FINITE_INSERT;FINITE_RULES];
  REWRITE_TAC[ARITH_RULE `2 = SUC 1`;SUC_INJ;];
  MESON_TAC[SING;CARD_SING];
  ]);;
  (* }}} *)

let has_size2 = prove_by_refinement(
  `!u. (u HAS_SIZE 2) <=> (?(a:A) b. (u = {a , b}) /\ ~(a=b))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  EQ_TAC;
  REWRITE_TAC[HAS_SIZE];
  DISCH_ALL_TAC;
  TYPE_THEN `~(u = EMPTY)` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  REWR 2;
  REWR 1;
  USE 1 (REWRITE_RULE[CARD_CLAUSES]);
  UND 1;
  ARITH_TAC;
  DISCH_TAC;
  COPY 0;
  COPY 2;
  JOIN 0 2;
  USE 0 (MATCH_MP CARD_DELETE_CHOICE);
  TYPE_THEN `CARD (u DELETE CHOICE u) = 1` SUBGOAL_TAC;
  ONCE_REWRITE_TAC [GSYM SUC_INJ];
  ASM_REWRITE_TAC[];
  ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `u DELETE CHOICE u HAS_SIZE 1` SUBGOAL_TAC;
  REWRITE_TAC[HAS_SIZE];
  ASM_REWRITE_TAC[FINITE_DELETE];
  DISCH_TAC;
  USE 5 (MATCH_MP CARD_SING_CONV);
  USE 5 (REWRITE_RULE [SING]);
  CHO 5;
  TYPE_THEN `CHOICE u` EXISTS_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  USE 5 (SYM);
  ASM_REWRITE_TAC[];
  USE 4 (MATCH_MP CHOICE_DEF);
  ASM_SIMP_TAC[INSERT_DELETE];
  TYPE_THEN `(u DELETE (CHOICE u)) x` SUBGOAL_TAC;
  USE 5 (SYM);
  ASM_REWRITE_TAC[INR IN_SING ];
  DISCH_TAC;
  TYPE_THEN `~((u DELETE CHOICE u) (CHOICE u))` SUBGOAL_TAC;
  REWRITE_TAC[INR IN_DELETE];
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[pair_size_2];
  ]);;
  (* }}} *)

let in_pair = prove_by_refinement(
  `!(a:A) b t. {a , b} t <=> (t = b) \/ (t = a)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INSERT];
  ]);;
  (* }}} *)

let pair_swap_select =
   jordan_def `pair_swap u (x:A) = @y. ~(x = y) /\ (u y)`;;

let pair_swap_pair = prove_by_refinement(
  `!(a:A) b. ~(a = b) ==>
       (pair_swap {a,b} a = b) /\ (pair_swap {a,b} b = a)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[pair_swap_select];
  REWRITE_TAC[in_pair];
  CONJ_TAC THEN SELECT_TAC THEN (ASM_MESON_TAC[]);
  ]);;
  (* }}} *)

let pair_swap = prove_by_refinement(
  `!u (x:A). (u HAS_SIZE 2)/\ (u x) ==>
         (~(pair_swap u x = x)) /\ (u (pair_swap u x))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[has_size2];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[];
  REWR 1;
  USE 1 (REWRITE_RULE[in_pair]);
  CONJ_TAC;
  ASM_MESON_TAC[pair_swap_pair];
  UND 1;
  DISCH_THEN (DISJ_CASES_TAC) THEN ASM_SIMP_TAC [pair_swap_pair] THEN REWRITE_TAC[INSERT];
  ]);;
  (* }}} *)

let pair_swap_invol = prove_by_refinement(
  `!u (x:A). (u HAS_SIZE 2) /\ (u x) ==>
       (pair_swap u (pair_swap u x) = x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[has_size2];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[];
  REWR 1;
  USE 1 (REWRITE_RULE[in_pair]);
  UND 1;
  DISCH_THEN (DISJ_CASES_TAC);
  ASM_SIMP_TAC [pair_swap_pair];
  ASM_SIMP_TAC [pair_swap_pair];
  ]);;
  (* }}} *)



(* ------------------------------------------------------------------ *)
(* SECTION C *)
(* ------------------------------------------------------------------ *)

(* ------------------------------------------------------------------ *)
(* rectagons *)
(* ------------------------------------------------------------------ *)

let rectagon = jordan_def `rectagon G <=>
  (FINITE G) /\ ~(G = EMPTY ) /\ (G SUBSET edge) /\
      (!m . ({0,2} (num_closure G (pointI m)))) /\
      (!S. ((S SUBSET G) /\ ~(S = EMPTY) /\
        (!C C'. (S C) /\ (G C') /\ (adj C C') ==> (S C'))) ==>
        (S = G))`;;

let segment = jordan_def `segment G <=>
  (FINITE G) /\ ~(G = EMPTY ) /\ (G SUBSET edge) /\
      (!m . ({0,1,2} (num_closure G (pointI m)))) /\
      (!S. ((S SUBSET G) /\ ~(S = EMPTY) /\
        (!C C'. (S C) /\ (G C') /\ (adj C C') ==> (S C'))) ==>
        (S = G))`;;

let psegment = jordan_def `psegment G <=>
   segment G /\ ~(rectagon G)`;;

let rectagon_segment = prove_by_refinement(
  `!G. (rectagon G ) ==> (segment G)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[segment;rectagon;INSERT ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let endpoint = jordan_def `endpoint G m <=>
  (num_closure G (pointI m) = 1)`;;

let midpoint = jordan_def `midpoint G m <=>
  (num_closure G (pointI m) = 2)`;;

let psegment_endpoint = prove_by_refinement(
  `!G. (psegment G) ==> (?m. (endpoint G m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[psegment;rectagon;segment;endpoint];
  DISCH_ALL_TAC;
  UND 5;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  LEFT 5 "m";
  CHO 5;
  TSPEC `m` 3;
  USE 3 (REWRITE_RULE[INSERT]);
  USE 5 (REWRITE_RULE[INSERT]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let rectagon_endpoint = prove_by_refinement(
  `!G. (rectagon G) ==> ~(?m. (endpoint G m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[rectagon;endpoint;INSERT ];
  DISCH_ALL_TAC;
  CHO 0;
  ASM_MESON_TAC[ARITH_RULE `(~(1=2)) /\ ~(1=0)` ];
  ]);;
  (* }}} *)

let num_closure_mono = prove_by_refinement(
  `!G G' x. (FINITE G') /\ (G SUBSET G') ==>
       (num_closure G x <= num_closure G' x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[num_closure];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC CARD_SUBSET ;
  REWRITE_TAC[ISUBSET];
  CONJ_TAC;
  ASM_MESON_TAC[ISUBSET];
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `G'` EXISTS_TAC;
  ASM_REWRITE_TAC[ISUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let endpoint_psegment = prove_by_refinement(
  `!G. (?m. (endpoint G m)) /\ (segment G) ==> (psegment G)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC  [psegment;rectagon_endpoint];
  ]);;
  (* }}} *)

let num_closure_size = prove_by_refinement(
  `!G x. FINITE G ==>
     ({C | G C /\ closure top2 C x} HAS_SIZE (num_closure G x) )`,
  (* {{{ proof *)
  [
  REWRITE_TAC[HAS_SIZE;num_closure];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `G` EXISTS_TAC;
  REWRITE_TAC[ISUBSET];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let endpoint_edge = prove_by_refinement(
  `!G m.  (FINITE G) /\ (endpoint G m) ==> (?! e. (G e) /\
     (closure top2 e (pointI m)))`,
  (* {{{ proof *)

  [
  REWRITE_TAC[endpoint;];
  DISCH_ALL_TAC;
  TYPE_THEN `{C | G C /\ closure top2 C (pointI m)} HAS_SIZE 1` SUBGOAL_TAC;
  UND 1;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  IMATCH_MP_TAC  num_closure_size;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  USE 2 (MATCH_MP CARD_SING_CONV);
  USE 2 (REWRITE_RULE[SING]);
  CHO 2;
  USE 2 (REWRITE_RULE[eq_sing]);
  REWRITE_TAC[EXISTS_UNIQUE_ALT];
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let midpoint_edge = prove_by_refinement(
  `!G m. (FINITE G) /\ (midpoint G m) ==>
     {C | G C /\ closure top2 C (pointI m)} HAS_SIZE 2`,
  (* {{{ proof *)
  [
  REWRITE_TAC[midpoint;];
  DISCH_ALL_TAC;
  UND 1;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  IMATCH_MP_TAC  num_closure_size;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let two_endpoint = prove_by_refinement(
  `!e. (edge e) ==> ({ m | (closure top2 e (pointI m)) } HAS_SIZE 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge];
  DISCH_ALL_TAC;
  CHO 0;
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[v_edge_closure;h_edge_closure];
  REWRITE_TAC[vc_edge;UNION;has_size2];
  TYPE_THEN `m` EXISTS_TAC;
  TYPE_THEN `(FST m,SND m +: (&:1))` EXISTS_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INR IN_SING ;];
  TYPE_THEN `euclid_plus (pointI m) e2 = pointI (FST m,SND m +: (&:1))` SUBGOAL_TAC ;
  REWRITE_TAC[pointI;e2;point_add;int_suc ];
  REDUCE_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[v_edge_pointI;pointI_inj;];
  REWRITE_TAC[INSERT];
  MESON_TAC[];
  REWRITE_TAC[PAIR_SPLIT];
  INT_ARITH_TAC;
  (* 2nd case: *)
  ASM_REWRITE_TAC[v_edge_closure;h_edge_closure];
  REWRITE_TAC[hc_edge;UNION;has_size2];
  TYPE_THEN `m` EXISTS_TAC;
  TYPE_THEN `(FST m +: (&:1),SND m )` EXISTS_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INR IN_SING ;];
  TYPE_THEN `euclid_plus (pointI m) e1 = pointI (FST m +: (&:1),SND m )` SUBGOAL_TAC ;
  REWRITE_TAC[pointI;e1;point_add;int_suc ];
  REDUCE_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[h_edge_pointI;pointI_inj;];
  REWRITE_TAC[INSERT];
  MESON_TAC[];
  REWRITE_TAC[PAIR_SPLIT];
  INT_ARITH_TAC;
  ]);;
  (* }}} *)

let edge_midend = prove_by_refinement(
  `!G e m. (segment G) /\ (G e) /\ (closure top2 e (pointI m)) ==>
      (midpoint G m) \/ (endpoint G m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[segment;midpoint;endpoint];
  DISCH_ALL_TAC;
  TSPEC `m` 3;
  USE 3 (REWRITE_RULE[INSERT]);
  TYPE_THEN `~(num_closure G (pointI m) = 0)` SUBGOAL_TAC;
  USE 0 (MATCH_MP num_closure_size);
  TSPEC `pointI m` 0;
  PROOF_BY_CONTR_TAC;
  REWR 7;
  REWR 0;
  USE 0(REWRITE_RULE[HAS_SIZE_0]);
  UND 0;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 3;
  ARITH_TAC;
  ]);;
  (* }}} *)

let plus_e12 = prove_by_refinement(
  `!m. ((pointI m) + e2 = pointI (FST m,SND m +: (&:1))) /\
      ((pointI m) + e1 = pointI (FST m +: (&:1),SND m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[e1;e2];
  REWRITE_TAC[pointI;point_add;int_suc];
  REDUCE_TAC;
  ]);;
  (* }}} *)

let c_edge_euclid = prove_by_refinement(
  `!e. (edge e) ==> (closure top2 e) SUBSET (euclid 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge];
  GEN_TAC;
  DISCH_THEN (CHOOSE_THEN DISJ_CASES_TAC ) THEN ASM_REWRITE_TAC[hc_edge;vc_edge;h_edge_closure;v_edge_closure;union_subset;plus_e12] THEN MESON_TAC[cell_rules; cell_euclid];
  ]);;
  (* }}} *)

(* slow proof... *)
let inter_lattice = prove_by_refinement(
  `!x e e'. (edge e) /\ (edge e') /\ (~(e=e')) /\
    ((closure top2 e INTER closure top2 e') x) ==>
       (?m. x = pointI m)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `euclid 2 x` SUBGOAL_TAC;
  USE 3 (REWRITE_RULE[INTER]);
  AND 3;
  USE 0 (MATCH_MP c_edge_euclid);
  USE 0 (REWRITE_RULE[ISUBSET]);
  ASM_MESON_TAC[];
  DISCH_THEN (MP_TAC o (MATCH_MP point_onto));
  DISCH_TAC;
  CHO 4;
  ASM_REWRITE_TAC[];
  ASSUME_TAC square_domain;
  TSPEC `p` 5;
  USE 5 (CONV_RULE (NAME_CONFLICT_CONV));
  UND 5;
  LET_TAC ;
  REWRITE_TAC[UNION];
  UND 3;
  ASM_REWRITE_TAC[INTER];
  KILL 4;
  UND 2;
  UND 0;
  REWRITE_TAC[edge] ;
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  UND 1;
  REWRITE_TAC[edge] ;
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  REP_CASES_TAC THEN UNDISCH_FIND_TAC `(~)` THEN UNDISCH_FIND_TAC `(closure)` THEN  UNDISCH_FIND_TAC `(point p)` THEN ASM_REWRITE_TAC[] THEN (REWRITE_TAC[INR IN_SING;h_edge_closure;v_edge_closure;UNION;vc_edge;hc_edge;plus_e12 ]) THEN
  (* 1st,2nd,3rd, *)
  (* tx *)
  (let tx = REWRITE_RULE[EQ_EMPTY;INTER ] in  MESON_TAC[tx hv_edge;tx v_edge_disj;tx h_edge_disj;tx square_v_edge;tx square_h_edge;v_edge_inj;h_edge_inj]);
  ]);;
  (* }}} *)

let edgec_convex = prove_by_refinement(
  `!e. (edge e) ==> (convex (closure top2 e))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN DISJ_CASES_TAC ) THEN ASM_REWRITE_TAC[v_edge_closure;h_edge_closure;hc_edge_convex;vc_edge_convex];
  ]);;
  (* }}} *)

let midpoint_h_edge = prove_by_refinement(
  `!m. (h_edge m) (((&.1)/(&.2))*# (pointI m) +
         ((&.1)/(&.2))*# (pointI m + e1))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[plus_e12];
  REWRITE_TAC[h_edge;pointI;point_add;point_scale;point_inj;PAIR_SPLIT;int_suc];
  GEN_TAC;
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  TYPE_THEN `a = real_of_int(SND m)` ABBREV_TAC;
  TYPE_THEN `b = real_of_int(FST  m)` ABBREV_TAC;
  CONJ_TAC;
  real_poly_tac ;
  CONJ_TAC;
  ineq_lt_tac `b + (&.1/(&.2)) = &1 / &2 * b + &1 / &2 * (b + &1)`;
  ineq_lt_tac `((&1 / &2) * b + &1 / &2 * (b + &1)) + (&1 / &2) = b +. &1`
  ]);;
  (* }}} *)

let midpoint_v_edge = prove_by_refinement(
  `!m. (v_edge m) (((&.1)/(&.2))*# (pointI m) +
         ((&.1)/(&.2))*# (pointI m + e2))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[plus_e12];
  REWRITE_TAC[v_edge;pointI;point_add;point_scale;point_inj;PAIR_SPLIT;int_suc];
  GEN_TAC;
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  TYPE_THEN `a = real_of_int(SND m)` ABBREV_TAC;
  TYPE_THEN `b = real_of_int(FST  m)` ABBREV_TAC;
  CONJ_TAC;
  real_poly_tac ;
  CONJ_TAC;
  ineq_lt_tac `a +. (&1/ &2)= &1 / &2 * a + &1 / &2 * (a + &1)`;
  ineq_lt_tac `(&1 / &2 * a + &1 / &2 * (a + &1)) +(&1/ &2) =  a + &1`;
  ]);;
  (* }}} *)

let midpoint_unique = prove_by_refinement(
  `!x y e e'. (edge e) /\ (edge e') /\ (~(e = e')) /\
    ((closure top2 e INTER closure top2 e') x) /\
    ((closure top2 e INTER closure top2 e') y) ==>
    ( x = y)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `convex (closure top2 e INTER closure top2 e')` SUBGOAL_TAC;
  IMATCH_MP_TAC  convex_inter ;
  ASM_MESON_TAC[edgec_convex];
  TYPE_THEN `(?m. x = pointI m) /\ (?n. y = pointI n)` SUBGOAL_TAC;
  ASM_MESON_TAC[inter_lattice];
  DISCH_ALL_TAC;
  CHO 6;
  CHO 7;
  ASM_REWRITE_TAC[];
  REWR 3;
  REWR 4;
  KILL 6;
  KILL 7;
  TYPE_THEN `(closure top2 e (pointI n)) /\ closure top2 e (pointI m)` SUBGOAL_TAC;
  UND 4;
  UND 3;
  REWRITE_TAC[INTER];
  MESON_TAC[];
  DISCH_ALL_TAC;
  WITH 0 (MATCH_MP edgec_convex);
  UND 6;
  USE 0 (REWRITE_RULE[edge]);
  CHO 0;
  UND 0;
  DISCH_THEN DISJ_CASES_TAC THEN ASM_REWRITE_TAC[];
  (* ml -- start of 1st main branch. *)
  DISCH_ALL_TAC;
  TYPE_THEN `((n = m') \/ (n = (FST m',SND m' + &:1))) /\ ((m = m') \/ (m = (FST m',SND m' + &:1)))` SUBGOAL_TAC;
  UND 6;
  UND 7;
  ASM_REWRITE_TAC[h_edge_closure;hc_edge;v_edge_closure;UNION;vc_edge;INR IN_SING;plus_e12;pointI_inj;v_edge_pointI ;h_edge_pointI];
  MESON_TAC[];
  REWRITE_TAC[RIGHT_AND_OVER_OR;LEFT_AND_OVER_OR];
  TYPE_THEN  `X = (closure top2 e INTER closure top2 e')` ABBREV_TAC;
  (* start A*)
  TYPE_THEN `X (pointI m') /\ X (pointI m' + e2) ==> ~(X INTER (v_edge m') = EMPTY)` SUBGOAL_TAC;
  REWRITE_TAC[EMPTY_EXISTS;INTER ];
  USE 5 (REWRITE_RULE[convex;mk_segment]);
  DISCH_TAC ;
  H_MATCH_MP (HYP "5") (HYP "10");
  USE 11 (REWRITE_RULE[ISUBSET]);
  TYPE_THEN `b = (&1 / &2) *# (pointI m') + (&1 / &2) *# (pointI m' + e2)` ABBREV_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  TSPEC `b` 11;
  CONJ_TAC;
  UND 11;
  DISCH_THEN IMATCH_MP_TAC  ;
  TYPE_THEN `&1/ &2` EXISTS_TAC;
  CONV_TAC REAL_RAT_REDUCE_CONV;
  EXPAND_TAC "b";
  MESON_TAC[];
  EXPAND_TAC "b";
  MATCH_ACCEPT_TAC midpoint_v_edge; (* end of goal A *)
  REWRITE_TAC[plus_e12];
  (* start  B*)
  TYPE_THEN `X INTER (v_edge m') = EMPTY ` SUBGOAL_TAC;
  REWRITE_TAC[EQ_EMPTY];
  DISCH_ALL_TAC;
  USE 10 (REWRITE_RULE[INTER]);
  TYPE_THEN `?r. (x = pointI r)` SUBGOAL_TAC;
  ASM_MESON_TAC[inter_lattice;edge];
  DISCH_TAC;
  CHO 11;
  REWR 10;
  ASM_MESON_TAC[v_edge_pointI];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  REP_CASES_TAC THEN ASM_MESON_TAC[];
  (* end of FIRST main branch  -- snd main branch -- fully parallel *)
  DISCH_ALL_TAC;
  TYPE_THEN `((n = m') \/ (n = (FST m' + &:1,SND m'))) /\ ((m = m') \/ (m = (FST m' + &:1,SND m' )))` SUBGOAL_TAC;
  UND 6;
  UND 7;
  ASM_REWRITE_TAC[h_edge_closure;hc_edge;v_edge_closure;UNION;vc_edge;INR IN_SING;plus_e12;pointI_inj;v_edge_pointI ;h_edge_pointI];
  MESON_TAC[];
  REWRITE_TAC[RIGHT_AND_OVER_OR;LEFT_AND_OVER_OR];
  TYPE_THEN  `X = (closure top2 e INTER closure top2 e')` ABBREV_TAC;
  (* start A'  *)
  TYPE_THEN `X (pointI m') /\ X (pointI m' + e1) ==> ~(X INTER (h_edge m') = EMPTY)` SUBGOAL_TAC;
  REWRITE_TAC[EMPTY_EXISTS;INTER ];
  USE 5 (REWRITE_RULE[convex;mk_segment]);
  DISCH_TAC ;
  H_MATCH_MP (HYP "5") (HYP "10");
  USE 11 (REWRITE_RULE[ISUBSET]);
  TYPE_THEN `b = (&1 / &2) *# (pointI m') + (&1 / &2) *# (pointI m' + e1)` ABBREV_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  TSPEC `b` 11;
  CONJ_TAC;
  UND 11;
  DISCH_THEN IMATCH_MP_TAC  ;
  TYPE_THEN `&1/ &2` EXISTS_TAC;
  CONV_TAC REAL_RAT_REDUCE_CONV;
  EXPAND_TAC "b";
  MESON_TAC[];
  EXPAND_TAC "b";
  MATCH_ACCEPT_TAC midpoint_h_edge; (* end of goal A' *)
  REWRITE_TAC[plus_e12];
  (* start  B' *)
  TYPE_THEN `X INTER (h_edge m') = EMPTY ` SUBGOAL_TAC;
  REWRITE_TAC[EQ_EMPTY];
  DISCH_ALL_TAC;
  USE 10 (REWRITE_RULE[INTER]);
  TYPE_THEN `?r. (x = pointI r)` SUBGOAL_TAC;
  ASM_MESON_TAC[inter_lattice;edge];
  DISCH_TAC;
  CHO 11;
  REWR 10;
  ASM_MESON_TAC[h_edge_pointI];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  REP_CASES_TAC  THEN ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let edge_inter = prove_by_refinement(
  `!C C'. (edge C) /\ (edge C') /\ (adj C C')  ==>
      (?m. (closure top2 C) INTER (closure top2 C') = {(pointI m)}) `,
  (* {{{ proof *)

  [
  REWRITE_TAC[adj];
  DISCH_ALL_TAC;
  USE 3 (REWRITE_RULE[EMPTY_EXISTS]);
  CHO 3;
  TYPE_THEN `(?m. u = pointI m)` SUBGOAL_TAC;
  ASM_MESON_TAC[inter_lattice];
  DISCH_THEN (CHOOSE_TAC);
  REWR 3;
  TYPE_THEN `m` EXISTS_TAC;
  ASM_REWRITE_TAC [eq_sing];
  ASM_MESON_TAC[midpoint_unique];
  ]);;

  (* }}} *)

let inter_midpoint = prove_by_refinement(
  `!G C C' m. (segment G) /\ (G C) /\ (G C') /\ (adj C C') /\
      (((closure top2 C) INTER (closure top2 C')) (pointI m)) ==>
    (midpoint G m) `,
  (* {{{ proof *)
  [
  REWRITE_TAC[midpoint;segment];
  DISCH_ALL_TAC;
  TSPEC `m` 3;
  USE 3 (REWRITE_RULE[INSERT]);
  UND 3;
  USE 0 (MATCH_MP num_closure_size);
  TSPEC `pointI m` 0;
  TYPE_THEN `X = {C | G C /\ closure top2 C (pointI m)}` ABBREV_TAC ;
  TYPE_THEN `X C /\ X C'` SUBGOAL_TAC;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  UND 8;
  REWRITE_TAC[INTER]; (* done WITH subgoal *)
  DISCH_TAC;
  TYPE_THEN `~(C = C')` SUBGOAL_TAC;
  ASM_MESON_TAC[adj];
  DISCH_TAC;
  REP_CASES_TAC;
  ASM_REWRITE_TAC[];
  REWR 0;
  USE 0 (MATCH_MP CARD_SING_CONV);
  USE 0 (REWRITE_RULE[SING;eq_sing]);
  ASM_MESON_TAC[];
  REWR 0;
  USE 0 (REWRITE_RULE[HAS_SIZE_0;EQ_EMPTY]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let mid_end_disj = prove_by_refinement(
  `!G m. ~(endpoint G m /\ midpoint G m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[endpoint;midpoint];
  ASM_MESON_TAC[ARITH_RULE `~(1=2)`];
  ]);;
  (* }}} *)

let two_exclusion  = prove_by_refinement(
  `!X p q (r:A). (X HAS_SIZE 2) /\ (X p) /\ (X q) /\ (X r) /\ (~(p = r))
    /\ (~(q = r)) ==> (p = q)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[has_size2;];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  UND 1;
  UND 2;
  UND 3;
  ASM_REWRITE_TAC[INSERT];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let midpoint_exists = prove_by_refinement(
  `!G e. (segment G) /\ (G e) /\ (~(G = {e})) ==>
      (?m. (closure top2 e (pointI m)) /\ (midpoint G m))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `!m. (closure top2 e (pointI m)) ==> (endpoint G m)` SUBGOAL_TAC;
  ASM_MESON_TAC[edge_midend];
  DISCH_TAC;
  UND 2;
  REWRITE_TAC[];
  UND 0;
  REWRITE_TAC[segment];
  DISCH_ALL_TAC;
  TSPEC `{e}` 7;
  UND 7;
  DISCH_THEN (IMATCH_MP_TAC  o GSYM);
  ASM_REWRITE_TAC[ISUBSET;INR IN_SING;];
  CONJ_TAC;
  ASM_MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC [eq_sing];
  DISCH_ALL_TAC;
  TYPE_THEN `(?m. (closure top2 e) INTER (closure top2 C') = {(pointI m)})` SUBGOAL_TAC;
  IMATCH_MP_TAC  edge_inter;
  ASM_MESON_TAC[ISUBSET];
  DISCH_THEN CHOOSE_TAC;
  TSPEC `m` 4;
  TYPE_THEN `endpoint G m` SUBGOAL_TAC;
  UND 4;
  DISCH_THEN IMATCH_MP_TAC ;
  UND 10;
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  REWRITE_TAC[endpoint];
  USE 0 (MATCH_MP num_closure_size);
  TSPEC `(pointI m)` 0;
  DISCH_TAC;
  REWR 0;
  USE 0 (MATCH_MP CARD_SING_CONV);
  USE 0 (REWRITE_RULE[SING]);
  CHO 0;
  USE 0 (REWRITE_RULE[eq_sing]);
  USE 10 (REWRITE_RULE[eq_sing]);
  USE 10 (REWRITE_RULE[INTER]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let pair_swap_unique = prove_by_refinement(
  `!u x (y:A). (u HAS_SIZE 2) /\ (u x) /\ (u y) /\ ~(x = y) ==>
    (y = pair_swap u x)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  two_exclusion ;
  TYPE_THEN `u` EXISTS_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[pair_swap];
  ]);;
  (* }}} *)

let pair_swap_adj = prove_by_refinement(
  `!G e m e'. (segment G) /\ (G e) /\ (midpoint G m) /\
     (closure top2 e (pointI m)) /\
     (e' = pair_swap {C | G C /\ closure top2 C (pointI m)} e) ==>
     ({C | G C /\ closure top2 C (pointI m)} HAS_SIZE 2) /\
             G e' /\ adj e' e /\ (closure top2 e' (pointI m)) `,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  TYPE_THEN `X = {C | G C /\ closure top2 C (pointI m)}` ABBREV_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `X HAS_SIZE 2` SUBGOAL_TAC;
  USE 3 (REWRITE_RULE[midpoint]);
  USE 1 (REWRITE_RULE[segment]);
  UND 1;
  DISCH_ALL_TAC;
  USE 1 (MATCH_MP num_closure_size);
  TSPEC `pointI m` 1;
  REWR 1;
  DISCH_TAC;
  CONJ_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `X e` SUBGOAL_TAC;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (*  SUBCONJ_TAC; *)
  TYPE_THEN `X e'` SUBGOAL_TAC;
  ASM_MESON_TAC[pair_swap];
  DISCH_TAC;
  SUBCONJ_TAC;
  UND 8;
  EXPAND_TAC "X";
  REWRITE_TAC[];
  MESON_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  SUBCONJ_TAC;
  UND 8;
  EXPAND_TAC "X";
  REWRITE_TAC[];
  MESON_TAC[];
  ASM_REWRITE_TAC[adj];
  ASM_SIMP_TAC[pair_swap];
  REWRITE_TAC[EMPTY_EXISTS];
  ASM_REWRITE_TAC[INTER];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)


(*
   A terminal edge is expressed as
   (endpoint G m) /\ (closure top2 e (pointI m))
*)

let terminal_edge_adj = prove_by_refinement(
  `!G e m. (segment G) /\ (G e) /\ (~(G = {e})) /\
     (endpoint G m) /\ (closure top2 e (pointI m))
     ==>
       (?! e'. (G e') /\ (adj e e')) `,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  DISCH_ALL_TAC;
  REWRITE_TAC[EXISTS_UNIQUE_ALT ];
  TYPE_THEN `(?m. (closure top2 e (pointI m)) /\ (midpoint G m))` SUBGOAL_TAC;
  IMATCH_MP_TAC  midpoint_exists;
  ASM_REWRITE_TAC[];
  DISCH_THEN CHOOSE_TAC;
  AND 5;
  COPY 5;
  USE 5 (REWRITE_RULE[midpoint]);
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  USE 8 (MATCH_MP num_closure_size);
  TSPEC `pointI m'` 8;
  REWR 8;
  TYPE_THEN `X = {C | G C /\ closure top2 C (pointI m')}` ABBREV_TAC;
  TYPE_THEN `X e` SUBGOAL_TAC;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `pair_swap X e` EXISTS_TAC;
  GEN_TAC;

  EQ_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `(?m. (closure top2 e) INTER (closure top2 y) = {(pointI m)}) ` SUBGOAL_TAC;
  IMATCH_MP_TAC  edge_inter;
  ASM_MESON_TAC[segment;ISUBSET;];
  DISCH_THEN CHOOSE_TAC;
  (* show m''=m', then X y, then y != e, then it is the PAIR swap *)
  TYPE_THEN `ec = (closure top2 e)` ABBREV_TAC;
  TYPE_THEN `ec (pointI m'')` SUBGOAL_TAC;
  UND 13;
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[INTER];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `m'' = m'` SUBGOAL_TAC;
  TYPE_THEN `Z = {m | ec (pointI m)}` ABBREV_TAC;
  IMATCH_MP_TAC  two_exclusion;
  TYPE_THEN `Z` EXISTS_TAC;
  TYPE_THEN `m` EXISTS_TAC;
  CONJ_TAC;
  EXPAND_TAC "Z";
  EXPAND_TAC "ec";
  IMATCH_MP_TAC  two_endpoint;
  ASM_MESON_TAC[segment;ISUBSET];
  EXPAND_TAC "Z";
  ASM_REWRITE_TAC[];
  TYPE_THEN `midpoint G m''` SUBGOAL_TAC ;
  IMATCH_MP_TAC  inter_midpoint;
  TYPE_THEN `e` EXISTS_TAC;
  TYPE_THEN `y` EXISTS_TAC;
  ASM_REWRITE_TAC[INR IN_SING ];
  ASM_MESON_TAC[mid_end_disj]; (* m'' = m' done *)
  DISCH_TAC;
  TYPE_THEN `X y` SUBGOAL_TAC;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  USE 13 (REWRITE_RULE[INTER;eq_sing]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `~(y = e)` SUBGOAL_TAC;
  UND 12;
  MESON_TAC[adj];
  DISCH_TAC;
  IMATCH_MP_TAC  (GSYM pair_swap_unique);
  ASM_REWRITE_TAC[];
  (* now second direction nsd *)
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  ASSUME_TAC pair_swap_adj;
  TYPEL_THEN [`G`;`e`;`m'`;`pair_swap X e`] (USE 11 o ISPECL);
  UND 11;
  ASM_REWRITE_TAC[];
  TYPE_THEN `X (pair_swap X e)` SUBGOAL_TAC;
  ASM_MESON_TAC[pair_swap];
  DISCH_TAC;
  TYPE_THEN `closure top2 (pair_swap X e) (pointI m')` SUBGOAL_TAC;
  UND 11;
  TYPE_THEN  `e'' = pair_swap X e` ABBREV_TAC ;
  EXPAND_TAC "X";
  REWRITE_TAC[];
  MESON_TAC[];
  ASM_MESON_TAC[adj_symm];
  ]);;
  (* }}} *)

let psegment_edge = prove_by_refinement(
  `!e. (edge e) ==> (psegment {e})`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  endpoint_psegment;
  ASM_REWRITE_TAC[endpoint;segment;EQ_EMPTY ;INR IN_SING;FINITE_SING;ISUBSET;num_closure];
  CONJ_TAC;
  UND 0;
  REWRITE_TAC[edge];
  DISCH_TAC ;
  CHO 0;
  TYPE_THEN `m` EXISTS_TAC;
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  CARD_SING;
  REWRITE_TAC[SING];
  TYPE_THEN `v_edge m` EXISTS_TAC;
  REWRITE_TAC[eq_sing;h_edge_closure;v_edge_closure;vc_edge;hc_edge;UNION;INR IN_SING ];
  MESON_TAC[];
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  CARD_SING;
  REWRITE_TAC[SING];
  TYPE_THEN `h_edge m` EXISTS_TAC;
  REWRITE_TAC[eq_sing;h_edge_closure;v_edge_closure;vc_edge;hc_edge;UNION;INR IN_SING ];
  MESON_TAC[];
  CONJ_TAC;
  MESON_TAC[];
  CONJ_TAC ;
  ASM_MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC[INSERT];
  GEN_TAC;
  TYPE_THEN `closure top2 e (pointI m)`  ASM_CASES_TAC ;
  DISJ1_TAC THEN DISJ2_TAC ;
  IMATCH_MP_TAC  CARD_SING;
  REWRITE_TAC[SING ;eq_sing];
  ASM_MESON_TAC[];
  DISJ2_TAC ;
  TYPE_THEN `{C | (C = e) /\ closure top2 C (pointI m)} = {}` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  USE 2 (REWRITE_RULE[EMPTY_EXISTS]);
  CHO 2;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[CARD_CLAUSES];
  DISCH_ALL_TAC;
  REWRITE_TAC[eq_sing];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let segment_delete = prove_by_refinement(
  `!G e m. (segment G) /\ (endpoint G m) /\
        (closure top2 e (pointI m)) /\ (~(G = {e}))
                ==> (segment (G DELETE e))`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  TYPE_THEN `~G e` ASM_CASES_TAC;
  USE 0 (REWRITE_RULE[INR DELETE_NON_ELEMENT]);
  ASM_MESON_TAC[];
  REWRITE_TAC[segment];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[FINITE_DELETE;delete_empty];
  CONJ_TAC;
  UND 3;
  MESON_TAC[ISUBSET ;INR IN_DELETE];
  CONJ_TAC;
  GEN_TAC;
  REWRITE_TAC[INSERT];
  TYPE_THEN `num_closure (G DELETE e) (pointI m')  <=| (num_closure G (pointI m'))` SUBGOAL_TAC;
  IMATCH_MP_TAC  num_closure_mono;
  ASM_REWRITE_TAC[INR IN_DELETE;ISUBSET];
  MESON_TAC[];
  TSPEC `m'` 4;
  USE 4 (REWRITE_RULE[INSERT]);
  UND 4;
  ARITH_TAC;
  DISCH_ALL_TAC;
  (* tsh1 *)
  TYPE_THEN `(?! e'. (G e') /\ (adj e e'))` SUBGOAL_TAC;
  IMATCH_MP_TAC  terminal_edge_adj;
  REWRITE_TAC[segment];
  TYPE_THEN `m` EXISTS_TAC;
  ASM_MESON_TAC[];
  REWRITE_TAC[EXISTS_UNIQUE_ALT];
  DISCH_THEN CHOOSE_TAC;
  (* tsh2 *)
  TYPE_THEN `(e INSERT S = G) ==> (S = G DELETE e)` SUBGOAL_TAC;
  UND 9;
  IMATCH_MP_TAC  (TAUT `(a ==> b ==> C) ==> (b ==> a ==> C)`);
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  REWRITE_TAC[DELETE_INSERT];
  REWRITE_TAC[DELETE;ISUBSET;];
  DISCH_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  UND 9;
  MESON_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  (* tsh3 *)
  TYPE_THEN `S e'` ASM_CASES_TAC;
  TSPEC `e INSERT S` 5;
  UND 5;
  DISCH_THEN IMATCH_MP_TAC ;
  REWR 0;
  ASM_REWRITE_TAC [INR INSERT_SUBSET;NOT_INSERT_EMPTY];
  CONJ_TAC;
  UND 9;
  MESON_TAC[ISUBSET;INR IN_DELETE];
  DISCH_ALL_TAC;
  TSPEC `C` 11;
  TSPEC `C'` 11;
  REWR 11; (* ok to here *)
  (* oth1 *)
  TYPE_THEN `C' = e` ASM_CASES_TAC;
  ASM_REWRITE_TAC[INSERT];
  ASM_REWRITE_TAC[INSERT]; (* *)
  (* UND 12; *)
  TYPE_THEN `C = e` ASM_CASES_TAC;
  REWR 15;
  TSPEC `C'` 12;
  REWR 12;
  ASM_MESON_TAC[];
  (* start not not -- *)
  UND 11;
  DISCH_THEN IMATCH_MP_TAC ;
  CONJ_TAC;
  UND 5;
  REWRITE_TAC[INSERT];
  ASM_MESON_TAC[];
  UND 14;
  REWRITE_TAC[DELETE];
  ASM_MESON_TAC[];
  (* LAST case *)
  TSPEC `S` 5;
  TYPE_THEN `S = G` SUBGOAL_TAC;
  UND 5;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  UND 9;
  REWRITE_TAC[DELETE;ISUBSET];
  MESON_TAC[];
  DISCH_TAC;
  DISCH_ALL_TAC;
  TYPEL_THEN [`C`;`C'`] (USE 11 o ISPECL);
  UND 11;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[DELETE];
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  TSPEC `C` 12;
  TYPE_THEN `G C /\ adj e C` SUBGOAL_TAC;
  ASM_MESON_TAC[adj_symm;ISUBSET];
  DISCH_TAC;
  REWR 12;
  ASM_MESON_TAC[];
  TSPEC `e'` 12;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let other_end = jordan_def `other_end e m =
  pair_swap {m | closure top2 e (pointI m)} m`;;

let other_end_prop = prove_by_refinement(
  `!e m. (edge e) /\ (closure top2 e (pointI m))==>
   (closure top2 e (pointI (other_end e m))) /\
      (~(other_end e m = m)) /\
      (other_end e (other_end e m) = m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[other_end];
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP two_endpoint);
  TYPE_THEN `X = {m | closure top2 e (pointI m)}` ABBREV_TAC;
  TYPE_THEN `X m` SUBGOAL_TAC;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC [];
  DISCH_TAC;
  ASM_SIMP_TAC[pair_swap_invol;pair_swap];
  TYPE_THEN `X (pair_swap X m)` SUBGOAL_TAC ;
  ASM_SIMP_TAC[pair_swap];
  EXPAND_TAC "X";
  REWRITE_TAC[];
  ]);;
  (* }}} *)

let num_closure_delete = prove_by_refinement(
  `!G e p. (FINITE G) ==> ((num_closure (G DELETE e) p) =
    (if ((G e) /\ (closure top2 e p)) then ((num_closure G p) -| 1)
       else (num_closure G p)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  COND_CASES_TAC;
  REWRITE_TAC[num_closure];
  TYPE_THEN `{C | (G DELETE e) C /\ closure top2 C p} = {C | G C /\ closure top2 C p} DELETE e` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[DELETE ];
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `FINITE {C | G C /\ closure top2 C p}` SUBGOAL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[ISUBSET;];
  MESON_TAC[];
  DISCH_TAC;
  USE 2 (MATCH_MP CARD_DELETE);
  TSPEC `e` 2;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[num_closure;DELETE ];
  AP_TERM_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  GEN_TAC;
  TYPE_THEN `x = e` ASM_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let psegment_delete_end = prove_by_refinement(
  `!G m e. (psegment G) /\ (endpoint G m) /\ (G e) /\
        (closure top2 e (pointI m)) /\ (~(G = {e})) ==>
     (endpoint (G DELETE e) =
       (((other_end e m) INSERT (endpoint G)) DELETE m))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[psegment;segment];
  DISCH_TAC;
  TYPE_THEN `edge e` SUBGOAL_TAC;
  ASM_MESON_TAC[psegment;segment;ISUBSET];
  DISCH_TAC;
  TYPE_THEN `X = {m | closure top2 e (pointI m)}` ABBREV_TAC;
  TYPE_THEN `X HAS_SIZE 2` SUBGOAL_TAC;
  EXPAND_TAC "X";
  IMATCH_MP_TAC  two_endpoint;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[endpoint;ISUBSET;INSERT;];
  GEN_TAC;
  ASM_SIMP_TAC[num_closure_delete];
  REWRITE_TAC[DELETE];
  TYPE_THEN `x = m` ASM_CASES_TAC;
  ASM_REWRITE_TAC[];
  USE 1 (REWRITE_RULE[endpoint]);
  ASM_REWRITE_TAC[];
  ARITH_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `x = other_end e m` ASM_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  COND_CASES_TAC;
  DISCH_TAC;
  TYPE_THEN `X x /\ X m /\ X (other_end e m) /\ (~(m= other_end e m))` SUBGOAL_TAC ;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[other_end_prop];
  DISCH_ALL_TAC;
  ASM_MESON_TAC[two_exclusion];
  MESON_TAC[];
  (* snd half *)
  REWRITE_TAC[SUBSET;endpoint;DELETE_INSERT];
  ASM_SIMP_TAC[other_end_prop];
  ASM_SIMP_TAC[num_closure_delete];
  REWRITE_TAC[INSERT;DELETE ];
  GEN_TAC;
  TYPE_THEN `(?m. (closure top2 e (pointI m)) /\ (midpoint G m))` SUBGOAL_TAC;
  ASM_MESON_TAC[psegment;midpoint_exists];
  DISCH_THEN CHOOSE_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  (* ---m *)
  COND_CASES_TAC;
  TYPE_THEN `X m /\ X m' /\ X x /\ (~(x = m)) /\ (~(m' = m)) /\ (~(x = m'))` SUBGOAL_TAC;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[mid_end_disj];
  ASM_MESON_TAC[two_exclusion];
  USE 10 (REWRITE_RULE[endpoint]);
  ASM_MESON_TAC[];
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[other_end_prop];
  TYPE_THEN `X m /\ X m' /\ X x /\ (~(x = m)) /\ (~(m = m'))` SUBGOAL_TAC;
   EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[other_end_prop];
  ASM_MESON_TAC[mid_end_disj];
  DISCH_TAC;
  TYPE_THEN `x = m'` SUBGOAL_TAC;
  ASM_MESON_TAC[two_exclusion];
  USE 9 (REWRITE_RULE[midpoint]);
  ASM_MESON_TAC[ARITH_RULE `(x = 2) ==> (x -| 1 = 1)`];
  ]);;
  (* }}} *)

let endpoint_size2 = prove_by_refinement(
  `!G. (psegment G) ==> (endpoint G HAS_SIZE 2)`,
  (* {{{ proof *)
  [
  TYPE_THEN `(!n G. (psegment G) /\ (G HAS_SIZE n) ==> (endpoint G HAS_SIZE 2)) ==> (!G. (psegment G) ==> endpoint G HAS_SIZE 2)` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `?n. G HAS_SIZE n` SUBGOAL_TAC;
  REWRITE_TAC[HAS_SIZE];
  CONV_TAC (dropq_conv "n");
  ASM_MESON_TAC[psegment;segment];
  DISCH_THEN CHOOSE_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  INDUCT_TAC;
  REWRITE_TAC[psegment;segment];
  ASM_MESON_TAC[HAS_SIZE_0];
  DISCH_ALL_TAC;
  TYPE_THEN `(?m. (endpoint G m))` SUBGOAL_TAC;
  ASM_SIMP_TAC[psegment_endpoint];
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `FINITE G` SUBGOAL_TAC ;
  ASM_MESON_TAC[psegment;segment];
  DISCH_TAC;
  TYPE_THEN `?e. (G e /\ closure top2 e (pointI m))` SUBGOAL_TAC;
  USE 3 (REWRITE_RULE[endpoint]);
  USE 4 (MATCH_MP num_closure_size);
  TSPEC `(pointI m)` 4;
  REWR 4;
  USE 4 (MATCH_MP CARD_SING_CONV);
  USE 4(REWRITE_RULE[SING]);
  CHO 4;
  USE 4 (REWRITE_RULE[eq_sing]);
  ASM_MESON_TAC[];
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `G = {e}` ASM_CASES_TAC;
  TYPE_THEN `endpoint G = { m | closure top2 e (pointI m)}` SUBGOAL_TAC;
  MATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[endpoint];
  USE 4 (MATCH_MP num_closure_size );
  GEN_TAC;
  TSPEC `pointI x` 4;
  REWR 4;
  USE 4 (REWRITE_RULE[INR IN_SING]);
  EQ_TAC;
  DISCH_TAC;
  REWR 4;
  USE 4 (MATCH_MP CARD_SING_CONV);
  USE 4(REWRITE_RULE[SING;eq_sing]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `{C | (C = e) /\ closure top2 C (pointI x)} ={e}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING ];
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 4;
  USE 4 (REWRITE_RULE[HAS_SIZE]);
  ASM_MESON_TAC[CARD_SING;SING];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  two_endpoint;
  ASM_MESON_TAC[psegment;segment;ISUBSET];
  (*pm*)
  (* main case *)
  TYPE_THEN `edge e` SUBGOAL_TAC;
  ASM_MESON_TAC[psegment;segment;ISUBSET];
  DISCH_TAC;
  TSPEC `G DELETE e` 0;
  TYPE_THEN `psegment (G DELETE e) /\ G DELETE e HAS_SIZE n` SUBGOAL_TAC;
  CONJ_TAC;
  REWRITE_TAC[psegment];
  CONJ_TAC;
  IMATCH_MP_TAC  segment_delete;
  TYPE_THEN `m` EXISTS_TAC;
  ASM_REWRITE_TAC[psegment];
  ASM_MESON_TAC[psegment];
  (* it isn't a rectagon if it has an endpoint *)
  TYPE_THEN `(endpoint (G DELETE e) (other_end e m)) ` SUBGOAL_TAC;
  ASM_SIMP_TAC[psegment_delete_end];
  REWRITE_TAC[DELETE_INSERT];
  COND_CASES_TAC;
  ASM_MESON_TAC[other_end_prop];
  REWRITE_TAC[INSERT];
  ASM_MESON_TAC[rectagon_endpoint];
  UND 2;
  REWRITE_TAC[HAS_SIZE];
  ASM_MESON_TAC[SUC_INJ;FINITE_DELETE_IMP;CARD_SUC_DELETE];
  DISCH_TAC;
  REWR 0;
  UND 0;
  ASM_SIMP_TAC[psegment_delete_end];
  DISCH_TAC;
  TYPE_THEN `G' = (other_end e m INSERT endpoint G)` ABBREV_TAC;
  TYPE_THEN `G' HAS_SIZE 3` SUBGOAL_TAC;
  UND 0;
  REWRITE_TAC[HAS_SIZE;ARITH_RULE `3 = SUC 2`;FINITE_DELETE];
  TYPE_THEN `G' m` SUBGOAL_TAC;
  EXPAND_TAC "G'";
  KILL 9;
  ASM_REWRITE_TAC [INSERT];
  ASM_MESON_TAC[CARD_SUC_DELETE];
  (* nearly there! *)
  EXPAND_TAC "G'";
  REWRITE_TAC[HAS_SIZE;FINITE_INSERT];
  DISCH_ALL_TAC;
  UND 11;
  ASM_SIMP_TAC [CARD_CLAUSES];
  COND_CASES_TAC;
  TYPE_THEN `(?m. (closure top2 e (pointI m)) /\ (midpoint G m))` SUBGOAL_TAC;
  IMATCH_MP_TAC  midpoint_exists;
  ASM_MESON_TAC[psegment];
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `X = { m | closure top2 e (pointI m) }` ABBREV_TAC;
  TYPE_THEN `X HAS_SIZE 2` SUBGOAL_TAC;
  USE 7 (MATCH_MP two_endpoint);
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `X m /\ X m' /\ X (other_end e m) /\ (~(m=m')) /\ (~(m= other_end e m)) /\ (~(m'=other_end e m))` SUBGOAL_TAC;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[other_end_prop];
  ASM_MESON_TAC [mid_end_disj];
  ASM_MESON_TAC[two_exclusion];
  ARITH_TAC;
  ]);;
  (* }}} *)

let sing_has_size1 = prove_by_refinement(
  `!(x:A). {x} HAS_SIZE 1`,
  (* {{{ proof *)
  [
  REWRITE_TAC[HAS_SIZE];
  DISCH_ALL_TAC;
  CONJ_TAC;
  REWRITE_TAC[FINITE_SING ];
  ASM_MESON_TAC[CARD_SING;SING];
  ]);;
  (* }}} *)

let num_closure1 = prove_by_refinement(
  `!G x. (FINITE G) ==>
       ((num_closure G (x) = 1) <=>
          (?e. (!e'. (G e' /\ (closure top2 e' (x))) <=> (e = e'))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  COPY 0;
  USE 0 (MATCH_MP (num_closure_size));
  TSPEC `x` 0;
  TYPE_THEN `t = num_closure G x` ABBREV_TAC;
  EQ_TAC;
  DISCH_TAC;
  REWR 0;
  USE 0 (MATCH_MP CARD_SING_CONV);
  USE 0 (REWRITE_RULE[SING;eq_sing]);
  CHO 0;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  CHO 3;
  TYPE_THEN `{C | G C /\ closure top2 C x} = {e}` SUBGOAL_TAC;
  REWRITE_TAC[eq_sing];
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 0;
  TYPE_THEN `e` (fun t -> ASSUME_TAC (ISPEC t sing_has_size1));
  UND 5;
  UND 0;
  REWRITE_TAC [HAS_SIZE];
  MESON_TAC[];
  ]);;
  (* }}} *)


(* ------------------------------------------------------------------ *)
(* SECTION D *)
(* ------------------------------------------------------------------ *)



let inductive_set = jordan_def `inductive_set G S <=>
   S SUBSET G /\
              ~(S = {}) /\
              (!C C'. S C /\ G C' /\ adj C C' ==> S C')`;;

let inductive_univ = prove_by_refinement(
  `!G. (~(G = EMPTY )) ==> (inductive_set G G)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[inductive_set];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[SUBSET_REFL];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let inductive_inter = prove_by_refinement(
  `!T G. (T SUBSET G) /\ (~(T = EMPTY )) ==>
        (inductive_set G
            (INTERS {S | (T SUBSET S) /\ (inductive_set G S)}))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  ONCE_REWRITE_TAC[inductive_set];
  CONJ_TAC;
  IMATCH_MP_TAC  INTERS_SUBSET2;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET_REFL];
  IMATCH_MP_TAC  inductive_univ;
  UND 1;
  REWRITE_TAC[EMPTY_EXISTS];
  ASM_MESON_TAC[ISUBSET];
  CONJ_TAC;
  USE 1 (REWRITE_RULE[EMPTY_EXISTS]);
  CHO 1;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `u` EXISTS_TAC;
  REWRITE_TAC[INTERS];
  DISCH_ALL_TAC;
  ASM_MESON_TAC[ISUBSET];
  DISCH_ALL_TAC;
  USE  2 (REWRITE_RULE[INTERS]);
  REWRITE_TAC[INTERS];
  DISCH_ALL_TAC;
  TSPEC `u` 2;
  REWR 2;
  ASM_MESON_TAC[inductive_set];
  ]);;
  (* }}} *)

let segment_of = jordan_def `segment_of G e =
   INTERS { S | S e /\ inductive_set G S }`;;

let inductive_segment = prove_by_refinement(
  `!G e. (G e) ==> (inductive_set G (segment_of G e))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[segment_of];
  ASSUME_TAC inductive_inter;
  TYPEL_THEN [`{e}`;`G`] (USE 1 o ISPECL);
  USE 1 (REWRITE_RULE[single_subset;EMPTY_EXISTS;INR IN_SING ]);
  UND 1;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let segment_of_G = prove_by_refinement(
  `!G e. (G e) ==> (segment_of G e ) SUBSET G`,
  (* {{{ proof *)
  [
  REWRITE_TAC[segment_of];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  (INR INTERS_SUBSET2 );
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET_REFL];
  IMATCH_MP_TAC  inductive_univ;
  REWRITE_TAC [EMPTY_EXISTS];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let segment_not_in = prove_by_refinement(
  `!G e. ~(G e) ==> (segment_of G e = UNIV)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[segment_of;];
  DISCH_ALL_TAC;
  TYPE_THEN `{S | S e /\ inductive_set G S} = EMPTY ` SUBGOAL_TAC ;
  REWRITE_TAC[EQ_EMPTY];
  GEN_TAC;
  REWRITE_TAC[inductive_set];
  ASM_MESON_TAC[ISUBSET];
  DISCH_THEN_REWRITE;
  ]);;
  (* }}} *)

let segment_of_finite = prove_by_refinement(
  `!G e. (FINITE G) /\ (G e) ==> (FINITE (segment_of G e))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  ASM_MESON_TAC[segment_of_G];
  ]);;
  (* }}} *)

let segment_of_in = prove_by_refinement(
  `!G e.  (segment_of G e e)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `G e` ASM_CASES_TAC;
  REWRITE_TAC[segment_of;INTERS;inductive_set ];
  MESON_TAC[];
  ASM_SIMP_TAC[segment_not_in];
  ]);;
  (* }}} *)

let segment_of_subset = prove_by_refinement(
  `!G e f. (G e) /\ (segment_of G e f) ==>
      (segment_of G f) SUBSET (segment_of G e)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[ISUBSET;segment_of;INTERS ];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let inductive_diff = prove_by_refinement(
  `!G S S'. (inductive_set G S) /\
        (inductive_set G S') /\ ~(S DIFF S' = {}) ==>
        (inductive_set G (S DIFF S'))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[inductive_set;DIFF;SUBSET  ];
  ASM_MESON_TAC[adj_symm];
  ]);;
  (* }}} *)

(* sets *)
let subset_imp_eq = prove_by_refinement(
  `!A (B:A->bool). (A SUBSET B) /\ (B DIFF A = EMPTY) ==> (A = B)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;DIFF;EQ_EMPTY];
  MESON_TAC[EQ_EXT];
  ]);;
  (* }}} *)

let segment_of_eq = prove_by_refinement(
  `!G e f. (G e) /\ (segment_of G e f) ==>
      ((segment_of G e) = (segment_of G f))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  (GSYM subset_imp_eq);
  CONJ_TAC;
  ASM_MESON_TAC[segment_of_subset];
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `G f` SUBGOAL_TAC;
  USE 0 (MATCH_MP segment_of_G);
  USE 0 (REWRITE_RULE[SUBSET]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `X = (segment_of G e DIFF segment_of G f)` ABBREV_TAC;
  TYPE_THEN `X e` SUBGOAL_TAC;
  EXPAND_TAC "X";
  REWRITE_TAC[DIFF];
  ASM_SIMP_TAC [segment_of_in];
  DISCH_ALL_TAC;
  USE 2 (GSYM);
  USE 2 (REWRITE_RULE[EMPTY_EXISTS]);
  CHO 2;
  UND 2;
  EXPAND_TAC "X";
  REWRITE_TAC[DIFF];
  JOIN 3 5;
  USE 2 (MATCH_MP segment_of_subset);
  ASM_MESON_TAC[ISUBSET]; (* done WITH X e *)
  DISCH_TAC;
  TYPE_THEN `inductive_set G (segment_of G e DIFF segment_of G f)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  inductive_diff;
  ASM_SIMP_TAC[inductive_segment];
  DISCH_TAC;
  TYPE_THEN `segment_of G e SUBSET X` SUBGOAL_TAC;
  REWRITE_TAC[segment_of];
  IMATCH_MP_TAC  INTERS_SUBSET;
  REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  REWRITE_TAC[SUBSET];
  LEFT_TAC "x";
  TYPE_THEN `f` EXISTS_TAC;
  EXPAND_TAC "X";
  REWRITE_TAC[DIFF];
  ASM_MESON_TAC[segment_of_in];
  ]);;
  (* }}} *)

let segment_of_segment = prove_by_refinement(
  `!G P e. (segment G) /\ (P SUBSET G) /\ (P e) ==>
      (segment (segment_of P e))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  TYPE_THEN `FINITE P` SUBGOAL_TAC;
  ASM_MESON_TAC[FINITE_SUBSET];
  DISCH_TAC;
  REWRITE_TAC[segment];
  ASM_SIMP_TAC[segment_of_finite;EMPTY_EXISTS];
  CONJ_TAC;
  ASM_MESON_TAC[segment_of_in];
  SUBCONJ_TAC;
  UND 1;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  MP_TAC  segment_of_G;
  REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASSUME_TAC segment_of_G;
  (* ok to here *)
  CONJ_TAC;
  GEN_TAC;
  REWRITE_TAC[INSERT];
  TYPEL_THEN [`P`;`e`] (USE 6 o ISPECL);
  REWR 6;
  JOIN 4 6;
  USE 4 (MATCH_MP num_closure_mono);
  TSPEC `pointI m` 4;
  UND 4;
  JOIN 3 1;
  USE 1 (MATCH_MP num_closure_mono);
  TSPEC `(pointI m)` 1;
  UND 1;
  UND 0;
  REWRITE_TAC[segment];
  REWRITE_TAC[INSERT];
  DISCH_ALL_TAC;
  TSPEC `m` 7;
  UND 7;
  UND 0;
  UND 1;
  ARITH_TAC;
  (* ok2 *)
  DISCH_ALL_TAC;
  CHO 8;
  (* IMATCH_MP_TAC  subset_imp_eq; *)
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  ASM_REWRITE_TAC[];
  (*   PROOF_BY_CONTR_TAC; *)
  TYPE_THEN `! C C'. S C /\ P C' /\ adj C C' ==> S C'` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `segment_of P C C'` SUBGOAL_TAC;
  REWRITE_TAC[segment_of;INTERS;];
  X_GEN_TAC `R:((num->real)->bool)->bool`;
  REWRITE_TAC[inductive_set];
  DISCH_ALL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `segment_of P e = segment_of P C` SUBGOAL_TAC ;
  IMATCH_MP_TAC  segment_of_eq;
  ASM_MESON_TAC[ISUBSET];
  DISCH_THEN (fun t-> REWRITE_TAC [GSYM t]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `inductive_set P S` SUBGOAL_TAC;
  REWRITE_TAC[inductive_set];
  ASM_REWRITE_TAC[EMPTY_EXISTS];
  ASM_MESON_TAC[ISUBSET;segment_of_G];
  TYPE_THEN `segment_of P e = segment_of P u` SUBGOAL_TAC;
  IMATCH_MP_TAC  segment_of_eq;
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[segment_of];
  DISCH_TAC;
  IMATCH_MP_TAC  (INR INTERS_SUBSET);
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

(* move up *)
let rectagon_subset = prove_by_refinement(
  `!G S. (rectagon G) /\ (segment S) /\ (G SUBSET S) ==> (G = S)`,
  (* {{{ proof *)

  [
  REWRITE_TAC[rectagon;segment];
  DISCH_ALL_TAC;
  TSPEC `G` 9;
  UND 9 ;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  TYPE_THEN `edge C /\ edge C'` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  TYPE_THEN `(?m. closure top2 C INTER closure top2 C' = {(pointI m)})` SUBGOAL_TAC;
  ASM_MESON_TAC[edge_inter];
  DISCH_TAC;
  CHO 14;
  (*loss*)
  COPY 10;
  COPY 5;
  JOIN 5 10;
  USE 5 (MATCH_MP num_closure_mono);
  TSPEC `pointI m` 5;
  TYPE_THEN `num_closure G (pointI m) = 2` SUBGOAL_TAC;
  TSPEC `m` 3;
  USE 3 (REWRITE_RULE[INSERT]);
  UND 3;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  UND 3;
  USE 0 (MATCH_MP num_closure_size);
  TSPEC  `(pointI m)` 0;
  DISCH_ALL_TAC;
  REWR 0;
  USE 0 (REWRITE_RULE[HAS_SIZE_0]);
  UND 0;
  REWRITE_TAC[EMPTY_EXISTS ];
  UND 14;
  REWRITE_TAC[INTER;eq_sing; ];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `num_closure S (pointI m) = 2` SUBGOAL_TAC;
  TSPEC `m` 8;
  USE 8(REWRITE_RULE[INSERT]);
  UND 8;
  TSPEC `m` 3;
  USE 3 (REWRITE_RULE[INSERT]);
  UND 3;
  UND 5;
  UND 10;
  ARITH_TAC;
  DISCH_TAC;
  (* ok  *)
  (* num_closure G = num_closure S, C' in latter, so in former *)
  TYPE_THEN `{C | G C /\ closure top2 C (pointI m)} = {C | S C /\ closure top2 C (pointI m)}`  SUBGOAL_TAC;
  IMATCH_MP_TAC  CARD_SUBSET_LE;
  CONJ_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `S` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET];
  MESON_TAC[];
  CONJ_TAC;
  UND 15;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  USE 0 (MATCH_MP num_closure_size);
  TSPEC `pointI m` 0;
  USE 16 (MATCH_MP num_closure_size);
  TSPEC `pointI m` 16;
  UND 16;
  UND 0;
  ASM_REWRITE_TAC [HAS_SIZE];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  ARITH_TAC;
  DISCH_TAC;
  TAPP `C'` 18;
  UND 18;
  ASM_REWRITE_TAC[];
  UND 14;
  REWRITE_TAC[INTER;eq_sing];
  MESON_TAC[];
  ]);;

  (* }}} *)

let rectagon_h_edge = prove_by_refinement(
  `!G. (rectagon G) ==> (?m. (G (h_edge m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `!e. G e ==> (?m. (e= (v_edge m))) ` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `edge e` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon;ISUBSET];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  DISCH_THEN DISJ_CASES_TAC;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `X = {m | (G (v_edge m)) }` ABBREV_TAC;
  TYPE_THEN `FINITE X /\ ~(X = {})` SUBGOAL_TAC;
  CONJ_TAC;
  TYPE_THEN `?C. C SUBSET X /\ FINITE C /\ (G = IMAGE (v_edge) C)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  finite_subset;
  REWRITE_TAC[IMAGE;SUBSET];
  EXPAND_TAC "X";
  REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  CONJ_TAC;
  DISCH_ALL_TAC;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[rectagon];
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  DISCH_ALL_TAC;
  TYPE_THEN `C = X` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_ALL_TAC;
  UND 7;
  EXPAND_TAC "X";
  REWRITE_TAC[];
  UND 6;
  REWRITE_TAC[IMAGE];
  DISCH_THEN_REWRITE ;
  DISCH_THEN CHOOSE_TAC;
  USE 6 (REWRITE_RULE[v_edge_inj;h_edge_inj]);
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  USE 0 (REWRITE_RULE[rectagon]);
  UND 0;
  DISCH_ALL_TAC;
  USE 5(REWRITE_RULE[EMPTY_EXISTS]);
  CHO 5;
  TSPEC `u` 2;
  REWR 2;
  CHO 2;
  UND 0;
  EXPAND_TAC "X";
  REWRITE_TAC[EMPTY_EXISTS];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* dwf done finite X ...  Messed up. X must have type real->bool. *)
  TYPE_THEN `Y = IMAGE (real_of_int o SND ) X` ABBREV_TAC;
  TYPE_THEN ` FINITE Y /\ ~(Y = EMPTY)` SUBGOAL_TAC;
  CONJ_TAC;
  EXPAND_TAC "Y";
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "Y";
  REWRITE_TAC[IMAGE;EMPTY_EXISTS ];
  CONV_TAC (dropq_conv "u");
  AND 4;
  USE 4 (REWRITE_RULE[EMPTY_EXISTS]);
  CHO 4;
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 6 (MATCH_MP min_finite);
  CHO 6;
  TYPE_THEN `?m. (G (v_edge m)) /\ (real_of_int (SND m) = delta)` SUBGOAL_TAC;
  USE 5 (REWRITE_RULE[IMAGE;o_DEF]);
  TAPP `delta` 5;
  REWR 5;
  CHO 5;
  TAPP `x` 3;
  REWR 3;
  ASM_MESON_TAC[];
  DISCH_TAC;
  CHO 7;
  (* now show that m is an endpoint *)
  TYPE_THEN `endpoint G m` SUBGOAL_TAC;
  REWRITE_TAC[endpoint];
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon];
  DISCH_TAC;
  ASM_SIMP_TAC[num_closure1];
  TYPE_THEN `v_edge m` EXISTS_TAC;
  DISCH_ALL_TAC;
  EQ_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `edge e'` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon;ISUBSET];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[v_edge_inj];
  REWR 10;
  USE 10 (REWRITE_RULE[v_edge_closure;vc_edge ;UNION;INR IN_SING ;plus_e12 ; pointI_inj; v_edge_pointI]);
  UND 10;
  DISCH_THEN   DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `  Y (real_of_int (SND m'))` SUBGOAL_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[IMAGE];
  TYPE_THEN `m'` EXISTS_TAC;
  REWRITE_TAC[o_DEF];
  EXPAND_TAC "X";
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  AND 6;
  TSPEC `(real_of_int(SND m'))` 6;
  REWR 6;
  USE 7 GSYM;
  REWR 6;
  USE 6 (REWRITE_RULE[int_suc ]);
  ASM_MESON_TAC[REAL_ARITH `~(x + &.1 <= x)`];
  ASM_MESON_TAC[hv_edgeV2];
  DISCH_TAC;
  EXPAND_TAC "e'";
  ASM_REWRITE_TAC[];
  EXPAND_TAC "e'";
  REWRITE_TAC[v_edge_closure;vc_edge;UNION ;INR IN_SING ;];
  ASM_MESON_TAC[rectagon_endpoint];
  ]);;
  (* }}} *)

let rectagon_v_edge = prove_by_refinement(
  `!G. (rectagon G) ==> (?m. (G (v_edge m)))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `!e. G e ==> (?m. (e= (h_edge m))) ` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `edge e` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon;ISUBSET];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  DISCH_THEN DISJ_CASES_TAC;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `X = {m | (G (h_edge m)) }` ABBREV_TAC;
  TYPE_THEN `FINITE X /\ ~(X = {})` SUBGOAL_TAC;
  CONJ_TAC;
  TYPE_THEN `?C. C SUBSET X /\ FINITE C /\ (G = IMAGE (h_edge) C)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  finite_subset;
  REWRITE_TAC[IMAGE;SUBSET];
  EXPAND_TAC "X";
  REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  CONJ_TAC;
  DISCH_ALL_TAC;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[rectagon];
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  DISCH_ALL_TAC;
  TYPE_THEN `C = X` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_ALL_TAC;
  UND 7;
  EXPAND_TAC "X";
  REWRITE_TAC[];
  UND 6;
  REWRITE_TAC[IMAGE];
  DISCH_THEN_REWRITE ;
  DISCH_THEN CHOOSE_TAC;
  USE 6 (REWRITE_RULE[h_edge_inj;v_edge_inj]);
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  USE 0 (REWRITE_RULE[rectagon]);
  UND 0;
  DISCH_ALL_TAC;
  USE 5(REWRITE_RULE[EMPTY_EXISTS]);
  CHO 5;
  TSPEC `u` 2;
  REWR 2;
  CHO 2;
  UND 0;
  EXPAND_TAC "X";
  REWRITE_TAC[EMPTY_EXISTS];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* dwfx done finite X ...  Messed up. X must have type real->bool. *)
  TYPE_THEN `Y = IMAGE (real_of_int o FST ) X` ABBREV_TAC;
  TYPE_THEN ` FINITE Y /\ ~(Y = EMPTY)` SUBGOAL_TAC;
  CONJ_TAC;
  EXPAND_TAC "Y";
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "Y";
  REWRITE_TAC[IMAGE;EMPTY_EXISTS ];
  CONV_TAC (dropq_conv "u");
  AND 4;
  USE 4 (REWRITE_RULE[EMPTY_EXISTS]);
  CHO 4;
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 6 (MATCH_MP min_finite);
  CHO 6;
  TYPE_THEN `?m. (G (h_edge m)) /\ (real_of_int (FST  m) = delta)` SUBGOAL_TAC;
  USE 5 (REWRITE_RULE[IMAGE;o_DEF]);
  TAPP `delta` 5;
  REWR 5;
  CHO 5;
  TAPP `x` 3;
  REWR 3;
  ASM_MESON_TAC[];
  DISCH_TAC;
  CHO 7;
  (* now show that m is an endpoint *)
  TYPE_THEN `endpoint G m` SUBGOAL_TAC;
  REWRITE_TAC[endpoint];
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon];
  DISCH_TAC;
  ASM_SIMP_TAC[num_closure1];
  TYPE_THEN `h_edge m` EXISTS_TAC;
  DISCH_ALL_TAC;
  EQ_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `edge e'` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon;ISUBSET];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  IMATCH_MP_TAC  (TAUT `((A \/ B) ==> C) ==> ((B \/ A) ==> C)`);
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[h_edge_inj];
  REWR 10;
  USE 10 (REWRITE_RULE[h_edge_closure;hc_edge ;UNION;INR IN_SING ;plus_e12 ; pointI_inj; h_edge_pointI]);
  UND 10;
  DISCH_THEN   DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `  Y (real_of_int (FST  m'))` SUBGOAL_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[IMAGE];
  TYPE_THEN `m'` EXISTS_TAC;
  REWRITE_TAC[o_DEF];
  EXPAND_TAC "X";
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  AND 6;
  TSPEC `(real_of_int(FST  m'))` 6;
  REWR 6;
  USE 7 GSYM;
  REWR 6;
  USE 6 (REWRITE_RULE[int_suc ]);
  ASM_MESON_TAC[REAL_ARITH `~(x + &.1 <= x)`];
  ASM_MESON_TAC[hv_edgeV2];
  DISCH_TAC;
  EXPAND_TAC "e'";
  ASM_REWRITE_TAC[];
  EXPAND_TAC "e'";
  REWRITE_TAC[h_edge_closure;hc_edge;UNION ;INR IN_SING ;];
  ASM_MESON_TAC[rectagon_endpoint];
  ]);;

  (* }}} *)

(* move down *)
let part_below = jordan_def `part_below G m =
   {C | G C /\
          ((?n. (C = v_edge n) /\ (SND n <=: SND m) /\ (FST n = FST m)) \/
           (?n. (C = h_edge n) /\ (SND n <=: SND m) /\
                 (closure top2 C (pointI (FST m,SND n))))) }`;;

let part_below_h = prove_by_refinement(
  `!G m n. part_below G m (h_edge n) <=>
         (set_lower G m n) \/ (set_lower G (left m) n)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  REWRITE_TAC[part_below;set_lower;left ];
  REWRITE_TAC[h_edge_closure;hc_edge;UNION ;h_edge_pointI];
  REWRITE_TAC[hv_edgeV2;plus_e12;INR IN_SING ;pointI_inj ;PAIR_SPLIT ];
  REWRITE_TAC[h_edge_inj];
  CONV_TAC (dropq_conv "n'");
  REWRITE_TAC[INT_ARITH `(x = y+: &:1) <=> (x -: (&:1) = y)`];
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let part_below_v = prove_by_refinement(
  `!G m n. part_below G m (v_edge n) <=>
         (G (v_edge n)) /\ (FST n = FST m) /\ (SND n <=: SND m)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[part_below;v_edge_closure;vc_edge;UNION;plus_e12; INR IN_SING; pointI_inj ; PAIR_SPLIT; v_edge_inj; hv_edgeV2];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

(* sets *)
let has_size_bij = prove_by_refinement(
  `!(A:A->bool) n. (A HAS_SIZE n) <=> (?f. BIJ f {m | m < n} A)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  EQ_TAC;
  DISCH_TAC;
  USE 0 (MATCH_MP (INR HAS_SIZE_INDEX));
  CHO 0;
  REWRITE_TAC[BIJ;INJ ;SURJ ;];
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  USE 0 (REWRITE_RULE[EXISTS_UNIQUE_ALT]);
  ASM_MESON_TAC[];
  DISCH_THEN CHOOSE_TAC;
  REWRITE_TAC[HAS_SIZE];
  ASSUME_TAC CARD_NUMSEG_LT;
  TSPEC `n` 1;
  EXPAND_TAC "n";
  SUBCONJ_TAC;
  ASSUME_TAC FINITE_NUMSEG_LT;
  TSPEC `n` 2;
  JOIN 2 0;
  USE 0 (MATCH_MP FINITE_BIJ);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  (GSYM BIJ_CARD);
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[FINITE_NUMSEG_LT];
  ]);;
  (* }}} *)

let has_size_bij2 = prove_by_refinement(
  `!(A:A->bool) n. (A HAS_SIZE n) <=> (?f. BIJ f A {m | m < n})`,
  (* {{{ proof *)
  [
  REWRITE_TAC[has_size_bij];
  DISCH_ALL_TAC;
  EQ_TAC;
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `INV f {m | m <| n} A` EXISTS_TAC;
  IMATCH_MP_TAC  INVERSE_BIJ;
  ASM_REWRITE_TAC[];
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `INV f A {m | m <| n}` EXISTS_TAC;
  IMATCH_MP_TAC  INVERSE_BIJ;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let fibre_card = prove_by_refinement(
  `!(f:A->B) A B m n.  (B HAS_SIZE n) /\ (IMAGE f A SUBSET B) /\
        (!b. (B b) ==> ({u | (A u) /\ (f u = b)} HAS_SIZE m)) ==>
           (A HAS_SIZE m*n)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `!b. ?g. (B b) ==> (BIJ g {u | (A u) /\ (f u = b)} {j | j <| m})` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  RIGHT_TAC "g";
  DISCH_TAC;
  REWRITE_TAC[GSYM has_size_bij2];
  TSPEC `b` 2;
  REWR 2;
  DISCH_TAC;
  LEFT 3 "g";
  CHO 3;
  (* case m=0 *)
  DISJ_CASES_TAC (ARITH_RULE `(m=0) \/ 0 < m`);
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  REWRITE_TAC[HAS_SIZE_0];
  REWR 2;
  USE 2 (REWRITE_RULE[HAS_SIZE_0]);
  USE 1 (REWRITE_RULE[IMAGE;ISUBSET ]);
  PROOF_BY_CONTR_TAC;
  USE 5 (REWRITE_RULE[EMPTY_EXISTS]);
  CHO 5;
  USE 1 (CONV_RULE NAME_CONFLICT_CONV);
  USE 1 (CONV_RULE (dropq_conv "x''"));
  TSPEC `u` 1;
  REWR 1;
  TSPEC `f u` 2;
  REWR 2;
  USE 2 (REWRITE_RULE[EQ_EMPTY]);
  ASM_MESON_TAC[];
  TYPE_THEN `BIJ (\x. (f x, g (f x) x)) A {(x,y) | B x /\ {j|j <|m} y}` SUBGOAL_TAC;
  REWRITE_TAC[BIJ;INJ;SURJ];
  SUBCONJ_TAC;
  SUBCONJ_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `f x` EXISTS_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  CONV_TAC (dropq_conv "y");
  SUBCONJ_TAC;
  UND 1;
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TSPEC `f x` 3;
  REWR 3;
  UND 3;
  REWRITE_TAC[BIJ;SURJ];
  DISCH_ALL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  DISCH_ALL_TAC;
  USE 8(REWRITE_RULE[PAIR_SPLIT]);
  AND 8;
  REWR 8;
  (* r8 *)
  TYPE_THEN `B (f y)` SUBGOAL_TAC;
  UND 1;
  REWRITE_TAC [IMAGE;SUBSET];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TSPEC `f y` 3;
  REWR 3;
  USE 3 (REWRITE_RULE[BIJ;INJ]);
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  GEN_TAC;
  NAME_CONFLICT_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  CONV_TAC (dropq_conv "x'");
  NAME_CONFLICT_TAC;
  GEN_TAC;
  LEFT_TAC  "x''";
  GEN_TAC;
  RIGHT_TAC "y''";
  DISCH_THEN_REWRITE ;
  RIGHT_TAC "y''";
  DISCH_ALL_TAC;
  USE 9 GSYM;
  REWR 8;
  ASM_REWRITE_TAC[];
  KILL 9;
  TSPEC `FST x` 2;
  REWR 2;
  TSPEC `FST x` 3;
  REWR 3;
  USE 3 (REWRITE_RULE[BIJ;SURJ]);
  ASM_MESON_TAC[];
  REWRITE_TAC[HAS_SIZE];
  DISCH_TAC;
  (* r9 *)
  TYPE_THEN `FINITE B /\ FINITE {j | j <| m}` SUBGOAL_TAC;
  ASM_REWRITE_TAC[FINITE_NUMSEG_LT];
  ASM_MESON_TAC[HAS_SIZE];
  DISCH_TAC;
  COPY 6;
  USE 6 (MATCH_MP   (INR FINITE_PRODUCT));
  REWR 6;
  COPY 7;
  USE 7 (MATCH_MP (INR CARD_PRODUCT));
  SUBCONJ_TAC;
  JOIN  6 5;
  USE 5 (MATCH_MP FINITE_BIJ2);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  JOIN 9 5;
  USE 5 (MATCH_MP BIJ_CARD);
  REWR 7;
  ASM_REWRITE_TAC[CARD_NUMSEG_LT];
  USE 0 (REWRITE_RULE[HAS_SIZE]);
  ASM_REWRITE_TAC[];
  ARITH_TAC;
  ]);;
  (* }}} *)

(* sets *)
let even_card_even = prove_by_refinement(
  `!X (Y:A->bool). (FINITE X) /\ (FINITE Y) /\ (X INTER Y = EMPTY) ==>
    ((EVEN (CARD X) <=> EVEN (CARD Y)) <=> (EVEN (CARD (X UNION Y))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  ASM_SIMP_TAC [CARD_UNION];
  REWRITE_TAC[EVEN_ADD];
  ]);;
  (* }}} *)


(*
  terminal edge: (endpoint G m) /\ (closure top2 e (pointI m))
  produce bij-MAP from terminal edges to endpoints (of P SUBSET G)
  2-1 MAP from  terminal edges to segments.
  Hence an EVEN number of endpoints.

*)



let terminal_edge = jordan_def `terminal_edge G m =
    @e. (G e) /\ (closure top2 e (pointI m))`;;

let terminal_endpoint = prove_by_refinement(
  `!G m. (FINITE G) /\ (endpoint G m)  ==> ((G (terminal_edge G m)) /\
          (closure top2 (terminal_edge G m) (pointI m)) ) `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[terminal_edge];
  SELECT_TAC;
  MESON_TAC[];
  ASM_MESON_TAC[endpoint_edge;EXISTS_UNIQUE_ALT];
  ]);;
  (* }}} *)

let terminal_unique = prove_by_refinement(
  `!G m e. (FINITE G) /\ (endpoint G m) ==>
       ( (G e) /\ (closure top2 e (pointI m)) <=> (e = terminal_edge G m))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  EQ_TAC;
  REWRITE_TAC[terminal_edge];
  SELECT_TAC;
  USE 1(REWRITE_RULE[endpoint]);
  ASM_MESON_TAC[num_closure1];
  ASM_MESON_TAC[terminal_endpoint];
  ASM_MESON_TAC[terminal_endpoint];
  ]);;
  (* }}} *)


let segment_of_endpoint = prove_by_refinement(
  `!P e m. (P e) /\ (FINITE P) ==>
     (endpoint P m /\
         (segment_of P (terminal_edge P m) = segment_of P e)
        <=>
        endpoint (segment_of P e) m)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE (segment_of P e)` SUBGOAL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  ASM_MESON_TAC[segment_of_G];
  DISCH_TAC;
  EQ_TAC;
  DISCH_ALL_TAC;
  COPY 3;
  UND 5;
  REWRITE_TAC[endpoint];
  ASM_SIMP_TAC[num_closure1];
  DISCH_ALL_TAC;
  CHO 5;
  TYPE_THEN `e'` EXISTS_TAC;
  DISCH_ALL_TAC;
  EQ_TAC;
  USE 0 (MATCH_MP segment_of_G);
  ASM_MESON_TAC[ISUBSET];
  DISCH_THEN (fun t->REWRITE_TAC[GSYM t]);
  COPY 5;
  TSPEC `e'` 5;
  USE 5 (REWRITE_RULE[]);
  ASM_REWRITE_TAC[];
  UND 4;
  DISCH_THEN (fun t->REWRITE_TAC[GSYM t]);
  TSPEC `terminal_edge P m` 6;
  UND 4;
  ASM_SIMP_TAC[terminal_endpoint];
  REWRITE_TAC[segment_of_in];
  DISCH_TAC;
  (* se *)
  SUBCONJ_TAC;
  UND 3;
  REWRITE_TAC[endpoint];
  ASM_SIMP_TAC[num_closure1];
  DISCH_ALL_TAC;
  CHO 3;
  TYPE_THEN `e'` EXISTS_TAC;
  DISCH_ALL_TAC;
  EQ_TAC;
  TYPE_THEN `P e'' /\ closure top2 e'' (pointI m) ==> segment_of P e e''` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  COPY 3;
  TSPEC `e'` 3;
  USE 3 (REWRITE_RULE []);
  TYPE_THEN `e'' = e'` ASM_CASES_TAC;
  ASM_MESON_TAC[];
  USE 0 (MATCH_MP inductive_segment);
  USE 0 (REWRITE_RULE[inductive_set]);
  UND 0;
  DISCH_ALL_TAC;
  TYPEL_THEN [`e'`;`e''`] (USE 9 o ISPECL);
  UND 9;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[adj;EMPTY_EXISTS;];
  TYPE_THEN `pointI m` EXISTS_TAC;
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_THEN (fun t->REWRITE_TAC[GSYM t]);
  ASM_MESON_TAC[segment_of_G;ISUBSET ];
  (* I'm getting lost in the thickets *)
  (* se2 *)
  DISCH_TAC;
  IMATCH_MP_TAC  (GSYM segment_of_eq);
  ASM_REWRITE_TAC[];
  COPY 4;
  COPY 3;
  UND 3;
  UND 4;
  REWRITE_TAC[endpoint];
  ASM_SIMP_TAC[num_closure1];
  DISCH_THEN CHOOSE_TAC;
  DISCH_THEN CHOOSE_TAC;
  (* *)
  COPY 3;
  TSPEC `e''` 3;
  TYPE_THEN `e' = e''` SUBGOAL_TAC;
  TSPEC `e''` 4;
  USE 4 (REWRITE_RULE[]);
  ASM_MESON_TAC[segment_of_G;ISUBSET ];
  DISCH_TAC;
  TSPEC `terminal_edge P m` 7;
  TYPE_THEN `e' = terminal_edge P m` SUBGOAL_TAC;
  ASM_MESON_TAC[terminal_endpoint];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let fibre2 = prove_by_refinement(
  `!P G. (segment G) /\ (P SUBSET G) /\ (~(rectagon P)) ==>
    (!S. ({ S | (?e. (P e) /\ (S = segment_of P e)) }  S) ==>
      ({m | (endpoint P m) /\ (segment_of P (terminal_edge P m) = S)}
              HAS_SIZE 2))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[];
  DISCH_ALL_TAC;
  CHO 3;
  ASM_REWRITE_TAC[];
  USE 3 (CONJUNCT1 );
  TYPE_THEN `psegment (segment_of P e)` SUBGOAL_TAC;
  REWRITE_TAC[psegment];
  CONJ_TAC;
  ASM_MESON_TAC[rectagon_subset;segment_of_G;segment_of_segment];
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `segment_of P e = G` SUBGOAL_TAC;
  IMATCH_MP_TAC  rectagon_subset;
  REWR 4;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[SUBSET_TRANS;segment_of_G];
  USE 3 (MATCH_MP segment_of_G);
  DISCH_TAC;
  REWR 3;
  JOIN 1 3;
  USE 1 (MATCH_MP SUBSET_ANTISYM);
  REWR 4;
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 4 (MATCH_MP endpoint_size2);
  TYPE_THEN `{m | endpoint P m /\ (segment_of P (terminal_edge P m) = segment_of P e)} = endpoint (segment_of P e)` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC ;
  REWRITE_TAC[];
  (* f2 *)
  IMATCH_MP_TAC  segment_of_endpoint;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  FINITE_SUBSET;
  ASM_MESON_TAC[segment];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let endpoint_even = prove_by_refinement(
  `!P G. (segment G) /\ (P SUBSET G) /\ (~(rectagon P)) ==>
        (endpoint P HAS_SIZE 2 *|
           (CARD {S | (?e. (P e) /\ (S = segment_of P e))})  )`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN  `f =  (segment_of P) o (terminal_edge P)` ABBREV_TAC;
  TYPE_THEN `B = { S | (?e. (P e) /\ (S = segment_of P e)) }` ABBREV_TAC;
  TYPE_THEN `f` (fun t-> IMATCH_MP_TAC   (ISPEC t fibre_card));
  TYPE_THEN `B` EXISTS_TAC;
  ASM_REWRITE_TAC[HAS_SIZE;IMAGE;SUBSET ; ];
  EXPAND_TAC "B";
  EXPAND_TAC "f";
  REWRITE_TAC[o_DEF ];
  SUBCONJ_TAC;
  TYPE_THEN `{S | ?e. P e /\ (S = segment_of P e)} = IMAGE (\x. (segment_of P x)) P` SUBGOAL_TAC;
  REWRITE_TAC[IMAGE];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  FINITE_IMAGE;
  IMATCH_MP_TAC  FINITE_SUBSET ;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  CONJ_TAC;
  NAME_CONFLICT_TAC;
  GEN_TAC;
  DISCH_THEN CHOOSE_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `terminal_edge P x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `FINITE P` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;FINITE_SUBSET];
  ASM_MESON_TAC[terminal_endpoint];
  (* ee *)
  REWRITE_TAC[GSYM HAS_SIZE];
  ASSUME_TAC fibre2;
  USE 6 (REWRITE_RULE[]);
  UND 6;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let num_closure0 = prove_by_refinement(
  `! G x.
     FINITE G ==> ((num_closure G x = 0) <=>
             (!e. (G e) ==> (~(closure top2 e x))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP num_closure_size);
  TSPEC `x` 0;
  EQ_TAC;
  DISCH_TAC;
  REWR 0;
  USE 0 (REWRITE_RULE[HAS_SIZE_0;EQ_EMPTY ]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `{C | G C /\ closure top2 C x} = {}` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  USE 2 (REWRITE_RULE[EMPTY_EXISTS]);
  CHO 2;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 0;
  USE 0 (REWRITE_RULE[HAS_SIZE]);
  ASM_MESON_TAC[CARD_CLAUSES];
  ]);;
  (* }}} *)

let num_closure2 = prove_by_refinement(
  `!G x.
    FINITE G ==> ((num_closure G x = 2) <=>
           (?a b. (~(a = b)) /\
              ((!e. (G e /\ closure top2 e x) <=> (( e= a)\/ (e =b))))))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  USE 0 (MATCH_MP num_closure_size);
  TSPEC `x` 0;
  EQ_TAC;
  DISCH_TAC;
  REWR 0;
  USE 0 (REWRITE_RULE[has_size2 ; ]);
  CHO 0;
  CHO 0;
  TYPE_THEN `a` EXISTS_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  AND 0;
  TAPP `e` 2;
  USE 2(REWRITE_RULE[INSERT]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  CHO 1;
  CHO 1;
  TYPE_THEN `X = {C | G C /\ closure top2 C x} ` ABBREV_TAC;
  TYPE_THEN `(?a b. (X = {a, b}) /\ ~(a = b))` SUBGOAL_TAC;
  TYPE_THEN `a` EXISTS_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INSERT];
  EXPAND_TAC "X";
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 3 (REWRITE_RULE[GSYM has_size2]);
  RULE_ASSUM_TAC (REWRITE_RULE[HAS_SIZE]);
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let endpoint_subrectagon = prove_by_refinement(
  `!G P m. (rectagon G) /\ (P SUBSET G) ==>
        ((endpoint P m) <=>
        (?C C'. (P C) /\ (G C') /\ (~(P C')) /\ (~(C = C')) /\
           (closure top2 C (pointI m)) /\ (closure top2 C' (pointI m))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon];
  DISCH_TAC;
  TYPE_THEN `FINITE P` SUBGOAL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  ASM_MESON_TAC[];
  DISCH_TAC;
  EQ_TAC;
  DISCH_TAC;
  TYPE_THEN `midpoint G m` SUBGOAL_TAC;
  REWRITE_TAC[midpoint];
  USE 0 (REWRITE_RULE[rectagon;INSERT]);
  UND 0;
  DISCH_ALL_TAC;
  TSPEC `m` 7;
  UND 7;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  USE 4 (REWRITE_RULE[endpoint]);
  JOIN 0 1;
  USE 0 (MATCH_MP num_closure_mono);
  ASM_MESON_TAC[ARITH_RULE `~(1 <=| 0)`];
  REWRITE_TAC[midpoint];
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon];
  DISCH_THEN (MP_TAC o (MATCH_MP num_closure_size));
  DISCH_ALL_TAC;
  TSPEC `pointI m` 6;
  REWR 6;
  USE 4 (REWRITE_RULE[endpoint]);
  UND 4;
  ASM_SIMP_TAC[num_closure1];
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `X = {C | G C /\ closure top2 C (pointI m)}` ABBREV_TAC;
  COPY 6;
  UND 8;
  REWRITE_TAC[has_size2];
  DISCH_THEN CHOOSE_TAC;
  CHO 8;
  TYPE_THEN `X a /\ X b /\ X e` SUBGOAL_TAC;
  CONJ_TAC;
  ASM_REWRITE_TAC[INSERT ];
  CONJ_TAC;
  ASM_REWRITE_TAC[INSERT];
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  TSPEC `e` 4;
  USE 4(REWRITE_RULE[]);
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  TYPE_THEN `P e /\ (closure top2 e (pointI m))` SUBGOAL_TAC;
  TSPEC `e` 4;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `G a /\ closure top2 a (pointI m) /\ G b /\ closure top2 b (pointI m)` SUBGOAL_TAC;
  UND 9;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  MESON_TAC[];
  DISCH_ALL_TAC;
  TYPE_THEN `(e =a) \/ (e = b)` SUBGOAL_TAC;
  ASM_MESON_TAC[two_exclusion];
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `a` EXISTS_TAC;
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  CHO 4;
  CHO 4;
  UND 4;
  DISCH_ALL_TAC;
  REWRITE_TAC[endpoint];
  UND 0;
  REWRITE_TAC[rectagon;INSERT ];
  DISCH_ALL_TAC;
  TSPEC `m` 12;
  UND 12;
  (* rg *)
  DISCH_THEN DISJ_CASES_TAC;
  USE 3 (MATCH_MP num_closure1);
  ASM_REWRITE_TAC[];
  USE 0 (MATCH_MP num_closure2);
  REWR 12;
  CHO 12;
  CHO 12;
  AND 12;
  TYPE_THEN `(C = a) \/ (C = b)` SUBGOAL_TAC;
  UND 12;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  TYPE_THEN `(C' = a) \/ (C' = b)` SUBGOAL_TAC;
  UND 12;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  TYPE_THEN `C` EXISTS_TAC;
  DISCH_ALL_TAC;
  EQ_TAC;
  DISCH_ALL_TAC;
  TSPEC `e'` 12;
  REWR 12;
  TYPE_THEN `G e'` SUBGOAL_TAC;
  UND 17;
  UND 1;
  MESON_TAC[ISUBSET];
  DISCH_TAC;
  KILL 0;
  KILL 3;
  KILL 18;
  KILL 13;
  ASM_MESON_TAC[];
  KILL 0;
  KILL 3;
  KILL 13;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  ASM_REWRITE_TAC[];
  (* rg2 *)
  USE 0(MATCH_MP num_closure0);
  REWR 12;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let part_below_finite = prove_by_refinement(
  `!G m. (FINITE G) ==> FINITE(part_below G m)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[part_below;ISUBSET ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let part_below_subset = prove_by_refinement(
  `!G m. (part_below G m) SUBSET G`,
  (* {{{ proof *)
  [
  REWRITE_TAC[part_below;ISUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let v_edge_cpoint = prove_by_refinement(
  `!m n. (closure top2 (v_edge m) (pointI n) <=>
          ((n = m) \/ (n = (FST m,SND m +: (&:1)))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[v_edge_closure;vc_edge;UNION];
  REWRITE_TAC[v_edge_pointI;INR IN_SING ;plus_e12;pointI_inj];
  ]);;
  (* }}} *)

let h_edge_cpoint = prove_by_refinement(
  `!m n. (closure top2 (h_edge m) (pointI n) <=>
          ((n = m) \/ (n = (FST m +: (&:1),SND m ))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[h_edge_closure;hc_edge;UNION];
  REWRITE_TAC[h_edge_pointI;INR IN_SING ;plus_e12;pointI_inj];
  ]);;
  (* }}} *)

let endpoint_lemma = prove_by_refinement(
  `!G m x.  (rectagon G) /\
      (endpoint (part_below G m) x)
       ==>
      (? C C' m'.
          ((C = v_edge m') \/ (C = h_edge m')) /\
          (edge C') /\
          (!e. G e /\ closure top2 e (pointI x) <=> (e = C) \/ (e = C')) /\
          (~(G = {})) /\
          (G SUBSET edge) /\
          (part_below G m C) /\
          (G C') /\
          (~part_below G m C') /\
          (~(C = C')) /\
          (closure top2 C (pointI x)) /\
          (closure top2 C' (pointI x)) /\
         (part_below G m SUBSET G) /\
         (endpoint (part_below G m) x))
          `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `part_below G m SUBSET G` SUBGOAL_TAC;
  ASM_MESON_TAC[part_below_subset];
  DISCH_TAC ;
  COPY 2;
  COPY 1;
  UND 1;
  UND 3;
  UND 0;
  SIMP_TAC[endpoint_subrectagon];
  DISCH_TAC;
  DISCH_TAC;
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  DISCH_ALL_TAC;
  USE 0 (REWRITE_RULE[rectagon;INSERT ]);
  UND 0;
  DISCH_ALL_TAC;
  TSPEC `x` 12;
  UND 12;
  DISCH_THEN DISJ_CASES_TAC;
  USE 0 (MATCH_MP num_closure2);
  REWR 12;
  CHO 12;
  CHO 12;
  KILL 0;
  AND 12;
  TYPE_THEN `(C = a) \/ (C = b)`  SUBGOAL_TAC;
 TSPEC `C` 0;
  UND 0;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  ASM_MESON_TAC[ISUBSET];
  TYPE_THEN `(C' = a) \/ (C' = b)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  DISCH_TAC;
  TYPE_THEN `!e. G e /\ closure top2 e (pointI x) <=> ((e = C) \/ (e = C'))` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  TSPEC `e` 0;
  ASM_REWRITE_TAC[];
  UND 15;
  UND 14;
  UND 12;
  UND 7;
  MESON_TAC[];
  DISCH_TAC;
  KILL 15;
  KILL 14;
  KILL 0;
  KILL 12;
  KILL 13;
  TYPE_THEN `edge C /\ edge C'` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET;];
  DISCH_ALL_TAC;
  USE 0 (REWRITE_RULE[edge]);
  UND 0;
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `C` EXISTS_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  TYPE_THEN `m'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* snd case *)
  USE 0 (MATCH_MP num_closure0);
  REWR 12;
  PROOF_BY_CONTR_TAC;
  UND 12;
  UND 5;
  UND 9;
  MESON_TAC[];
  ]);;
  (* }}} *)

let endpoint_lemma_small_fst = prove_by_refinement(
  `!G m x. (rectagon G) /\ (endpoint (part_below G m) x) ==>
       (FST m <=: FST x +: &:1) `,
  (* {{{ proof *)

  [
  REP_GEN_TAC;
  DISCH_TAC;
  COPY 0;
  USE 0 (MATCH_MP endpoint_lemma);
  CHO 0;
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  REWRITE_TAC[INT_ARITH `x <=: z +: y <=> ~(z +: y <: x)`];
  DISCH_ALL_TAC;
  (* setup complete *)
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_v]);
  REWR 10;
  USE 10 (REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `FST x = FST m'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 14;
  AND 6;
  AND 6;
  REWR 14;
  UND 14;
  INT_ARITH_TAC;
  (* 2nd case *)
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_h ;set_lower  ;left  ;]);
  REWR 10;
  USE 10 (REWRITE_RULE[h_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `(FST x = FST m') \/ (FST x = FST m' +: (&:1))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `(FST m' = FST m) \/ (FST m' = FST m -: &:1)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  UND 14;
  INT_ARITH_TAC;
  ]);;

  (* }}} *)

(* identical proof to endpoint_lemma_small_fst *)
let endpoint_lemma_big_fst = prove_by_refinement(
  `!G m x. (rectagon G) /\ (endpoint (part_below G m) x) ==>
       (FST x <=: FST m +: &:1) `,
  (* {{{ proof *)

  [
  REP_GEN_TAC;
  DISCH_TAC;
  COPY 0;
  USE 0 (MATCH_MP endpoint_lemma);
  CHO 0;
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  REWRITE_TAC[INT_ARITH `x <=: z +: y <=> ~(z +: y <: x)`];
  DISCH_ALL_TAC;
  (* setup complete *)
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_v]);
  REWR 10;
  USE 10 (REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `FST x = FST m'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 14;
  AND 6;
  AND 6;
  REWR 14;
  UND 14;
  INT_ARITH_TAC;
  (* 2nd case *)
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_h ;set_lower  ;left  ;]);
  REWR 10;
  USE 10 (REWRITE_RULE[h_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `(FST x = FST m') \/ (FST x = FST m' +: (&:1))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `(FST m' = FST m) \/ (FST m' = FST m -: &:1)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  UND 14;
  INT_ARITH_TAC;
  ]);;

  (* }}} *)

let endpoint_lemma_big_snd = prove_by_refinement(
  `!G m x. (rectagon G) /\ (endpoint (part_below G m) x) ==>
       (SND  x <=: SND  m +: &:1) `,
  (* {{{ proof *)

  [
  REP_GEN_TAC;
  DISCH_TAC;
  COPY 0;
  USE 0 (MATCH_MP endpoint_lemma);
  CHO 0;
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  REWRITE_TAC[INT_ARITH `x <=: z +: y <=> ~(z +: y <: x)`];
  DISCH_ALL_TAC;
  (* setup complete *)
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_v]);
  REWR 10;
  USE 10 (REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `(SND x = SND m') \/ (SND x = SND m' +: &:1)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  UND 14;
  AND 6;
  AND 6;
  UND 6;
  INT_ARITH_TAC;
  (* 2nd case *)
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_h ;set_lower  ;left  ;]);
  REWR 10;
  USE 10 (REWRITE_RULE[h_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `SND x = SND m'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `(SND m' <=: SND m)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  UND 14;
  INT_ARITH_TAC;
  ]);;

  (* }}} *)

let endpoint_lemma_mid_fst = prove_by_refinement(
  `!G m x. (rectagon G) /\ (endpoint (part_below G m) x) ==>
       (FST x = FST m) ==> (SND  x = SND  m +: &:1) `,
  (* {{{ proof *)

  [
  REP_GEN_TAC;
  DISCH_TAC;
  COPY 0;
  USE 0 (MATCH_MP endpoint_lemma);
  CHO 0;
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  (* setup complete *)
  UND 2;
  DISCH_THEN DISJ_CASES_TAC;
  REWR 7;
  USE 7 (REWRITE_RULE[part_below_v]);
  REWR 11;
  USE 11 (REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `(SND x = SND m') \/ (SND x = SND m' +: &:1)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  AND 7;
  AND 7;
  UND 7;
  USE 3 (REWRITE_RULE[edge]);
  CHO 3;
  UND 3;
  DISCH_THEN DISJ_CASES_TAC;
  REWR 9;
  USE 7 (REWRITE_RULE[part_below_v]);
  REWR 8;
  REWR 7;
  REWR 12;
  USE 9 (REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `(FST m'' = FST m) /\ (FST x = FST m'')` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 9;
  REWR 7;
  UND 7;
  UND 9;
  INT_ARITH_TAC;
  (* 2nd case *)
  REWR 12;
  USE 7 (REWRITE_RULE[h_edge_cpoint;PAIR_SPLIT]);
  REWR 8;
  REWR 9;
  USE 9 (REWRITE_RULE[left ;set_lower;part_below_h]);
  REWR 9;
  TYPE_THEN `(FST x = FST m') ` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 7;
  DISCH_ALL_TAC;
  REWR 7;
  KILL 12;
  REWR 7;
  KILL  11;
  (* try *)
  UND 7;
  UND 17;
  UND 18;
  UND 9;
  INT_ARITH_TAC;
  (* 3rd case *)
  (* 3c *)
  REWR 11;
  USE 11(REWRITE_RULE[h_edge_cpoint;PAIR_SPLIT]);
  USE 3(REWRITE_RULE[edge]);
  CHO 3;
  UND 3;
  DISCH_THEN DISJ_CASES_TAC;
  REWR 9;
  USE 9(REWRITE_RULE[left ;set_lower;part_below_v;part_below_h]);
  REWR 8;
  REWR 9;
  UND 9;
  UND 11;
  UND 0;
  REWR 12;
  USE 0(REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT]);
  UND 0;
  USE 1 (MATCH_MP endpoint_lemma_big_snd );
  UND 0;
  INT_ARITH_TAC;
  (* LAST case ,3d *)
  TYPE_THEN `G (h_edge m')` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  REWR 12;
  USE 12 (REWRITE_RULE[h_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `SND x = SND m''` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 12;
  REWR 7;
   USE 7(REWRITE_RULE[left ;set_lower;part_below_v;part_below_h]);
  REWR 7;
  TYPE_THEN `SND m' <=: SND m` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  UND 7;
  COPY 17;
  UND 7;
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  REWR 9;
   USE 9(REWRITE_RULE[left ;set_lower;part_below_v;part_below_h]);
  REWR 8;
  REWR 9;
  TYPE_THEN `SND x = SND m'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  UND 11;
  COPY 18;
  UND 11;
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  TYPE_THEN `(FST m'' = FST m) \/ (FST m'' = FST m -: &:1)` SUBGOAL_TAC;
  UND 11;
  UND 7;
  UND 12;
  INT_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `~(SND m'' <=: SND m)` SUBGOAL_TAC;
  UND 19;
  UND 9;
  INT_ARITH_TAC;
  UND 16;
  UND 18;
  UND 17;
  INT_ARITH_TAC;
  ]);;

  (* }}} *)

let endpoint_lemma_upper_left = prove_by_refinement(
  `!G m . (rectagon G) ==>
       ~(endpoint (part_below G m) (FST m -: &:1, SND m +: &:1))`,
  (* {{{ proof *)

  [
  (* needs to be rewritten, template only *)
  REP_GEN_TAC;
  TYPE_THEN  `(! x. (rectagon G) /\ (endpoint (part_below G m) x) ==> ~(x = (FST m -: &:1, SND m +: &:1))) ==> (rectagon G ==> ~( endpoint (part_below G m) (FST m -: &:1,SND m +: &:1)))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  GEN_TAC;
  DISCH_TAC;
  USE 0 (MATCH_MP endpoint_lemma);
  CHO 0;
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  UND 1;
  DISCH_THEN DISJ_CASES_TAC;
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_v]);
  REWR 10;
  USE 10 (REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `FST m -: &:1 = FST m'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `FST m' = FST m` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  INT_ARITH_TAC;
  (* 2nd case *)
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_h ;set_lower  ;left  ;]);
  REWR 10;
  USE 10 (REWRITE_RULE[h_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `(SND m +: &:1 = SND m') /\ (SND m' <=: SND m)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  INT_ARITH_TAC;
  ]);;

  (* }}} *)

let endpoint_lemma_upper_left = prove_by_refinement(
  `!G m . (rectagon G) ==>
       ~(endpoint (part_below G m) (FST m -: &:1, SND m +: &:1))`,
  (* {{{ proof *)

  [
  (* needs to be rewritten, template only *)
  REP_GEN_TAC;
  TYPE_THEN  `(! x. (rectagon G) /\ (endpoint (part_below G m) x) ==> ~(x = (FST m -: &:1, SND m +: &:1))) ==> (rectagon G ==> ~( endpoint (part_below G m) (FST m -: &:1,SND m +: &:1)))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  GEN_TAC;
  DISCH_TAC;
  USE 0 (MATCH_MP endpoint_lemma);
  CHO 0;
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  UND 1;
  DISCH_THEN DISJ_CASES_TAC;
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_v]);
  REWR 10;
  USE 10 (REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `FST m -: &:1 = FST m'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `FST m' = FST m` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  INT_ARITH_TAC;
  (* 2nd case *)
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_h ;set_lower  ;left  ;]);
  REWR 10;
  USE 10 (REWRITE_RULE[h_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `(SND m +: &:1 = SND m') /\ (SND m' <=: SND m)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  INT_ARITH_TAC;
  ]);;

  (* }}} *)

let endpoint_lemma_upper_right = prove_by_refinement(
  `!G m . (rectagon G) ==>
       ~(endpoint (part_below G m) (FST m +: &:1, SND m +: &:1))`,
  (* {{{ proof *)

  [
  (* needs to be rewritten, template only *)
  REP_GEN_TAC;
  TYPE_THEN  `(! x. (rectagon G) /\ (endpoint (part_below G m) x) ==> ~(x = (FST m +: &:1, SND m +: &:1))) ==> (rectagon G ==> ~( endpoint (part_below G m) (FST m +: &:1,SND m +: &:1)))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  GEN_TAC;
  DISCH_TAC;
  USE 0 (MATCH_MP endpoint_lemma);
  CHO 0;
  CHO 0;
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  UND 1;
  DISCH_THEN DISJ_CASES_TAC;
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_v]);
  REWR 10;
  USE 10 (REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `FST m +: &:1 = FST m'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `FST m' = FST m` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  INT_ARITH_TAC;
  (* 2nd case *)
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_h ;set_lower  ;left  ;]);
  REWR 10;
  USE 10 (REWRITE_RULE[h_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `(SND m +: &:1 = SND m') /\ (SND m' <=: SND m)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  INT_ARITH_TAC;
  ]);;

  (* }}} *)

let endpoint_lemma_summary = prove_by_refinement(
  `!G m x. (rectagon G) /\ (endpoint (part_below G m) x) ==>
    ((FST x = FST m -: &:1) /\ (SND x <=: SND  m)) \/
    ((FST x = FST m +: &:1) /\ (SND x <=: SND m)) \/
    ((FST x = FST m) /\ (SND x = SND m +: &:1 )) `,
  (* {{{ proof *)
  [
  (* USE int -arith to show cases of fst x, then for each give *)
  REP_GEN_TAC;
  DISCH_TAC;
  TYPE_THEN `(FST x < FST m -: &:1) \/ (FST x = FST m -: &:1) \/ (FST x = FST m ) \/ (FST x = FST m +: &:1) \/ (FST m +: &:1 <: FST x  )` SUBGOAL_TAC;
  INT_ARITH_TAC;
  REP_CASES_TAC ;
  USE 0 (MATCH_MP endpoint_lemma_small_fst);
  PROOF_BY_CONTR_TAC;
  UND 0;
  UND 1;
  INT_ARITH_TAC;
  DISJ1_TAC;
  ASM_REWRITE_TAC[];
  COPY 0;
  USE 0 (MATCH_MP endpoint_lemma_big_snd);
  IMATCH_MP_TAC  (INT_ARITH `x <=: m+ &:1 /\ ~(x = m + &:1) ==> ( x <=: m)`);
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  REWR 3;
  TYPE_THEN `x = (FST m -: &:1, SND m + &:1)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[PAIR_SPLIT];
  DISCH_TAC;
  REWR 2;
  ASM_MESON_TAC[endpoint_lemma_upper_left];
  USE 0 (MATCH_MP endpoint_lemma_mid_fst);
  ASM_MESON_TAC[];
  DISJ2_TAC;
  DISJ1_TAC ;
  ASM_REWRITE_TAC[];
  COPY 0;
  USE 0 (MATCH_MP endpoint_lemma_big_snd);
  IMATCH_MP_TAC  (INT_ARITH `x <=: m+ &:1 /\ ~(x = m + &:1) ==> ( x <=: m)`);
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  REWR 3;
  TYPE_THEN `x = (FST m +: &:1, SND m + &:1)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[PAIR_SPLIT];
  DISCH_TAC;
  REWR 2;
  ASM_MESON_TAC[endpoint_lemma_upper_right];
  USE 0 (MATCH_MP endpoint_lemma_big_fst);
  PROOF_BY_CONTR_TAC;
  UND 0;
  UND 1;
  INT_ARITH_TAC;
  ]);;
  (* }}} *)

let terminal_case1 = prove_by_refinement(
  `!G m n x. (rectagon G) /\ (endpoint (part_below G m) x) /\
      (closure top2 (h_edge n) (pointI x)) /\ (set_lower G m n ) ==>
      (x = right  n)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[h_edge_cpoint; set_lower];
  DISCH_ALL_TAC;
  USE 2 (REWRITE_RULE[PAIR_SPLIT]);
  UND 2;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `FST x = FST m` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  JOIN 0 1;
  USE 0 (MATCH_MP endpoint_lemma_mid_fst);
  REWR 0;
  UND 0;
  UND 2;
  UND 5;
  INT_ARITH_TAC;
  TYPE_THEN `FST x = FST m +: &:1` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REWRITE_TAC[PAIR_SPLIT;right  ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let terminal_case2 = prove_by_refinement(
  `!G m n x. (rectagon G) /\ (endpoint (part_below G m) x) /\
      (closure top2 (h_edge n) (pointI x)) /\
          (set_lower G (left  m) n ) ==>
      (x =  n)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[h_edge_cpoint; set_lower ];
  DISCH_ALL_TAC;
  USE 2 (REWRITE_RULE[PAIR_SPLIT]);
  UND 2;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[PAIR_SPLIT];
  TYPE_THEN `FST x = FST m` SUBGOAL_TAC;
  UND 2;
  UND 4;
  REWRITE_TAC[left ];
  INT_ARITH_TAC ;
  DISCH_TAC;
  JOIN 0 1;
  USE 0 (MATCH_MP endpoint_lemma_mid_fst);
  AND 2;
  UND 2;
  REWR 0;
  DISCH_TAC;
  UND 5;
  UND 0;
  REWRITE_TAC[left  ];
  INT_ARITH_TAC;
  ]);;
  (* }}} *)

let terminal_case_v = prove_by_refinement(
  `!G m n x. (rectagon G) /\ (endpoint (part_below G m) x) /\
      (closure top2 (v_edge n) (pointI x)) /\
          (part_below G m (v_edge n)) ==>
      (x = up m) /\ (m =n)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[part_below_v; v_edge_cpoint;];
  DISCH_ALL_TAC;
  JOIN 0 1;
  USE 2 (REWRITE_RULE[PAIR_SPLIT]);
  REWR 1;
  TYPE_THEN `FST x = FST m` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 1;
  REWRITE_TAC[PAIR_SPLIT; up ;];
  ASM_REWRITE_TAC[];
  USE 0 (MATCH_MP endpoint_lemma_mid_fst);
  REWR 0;
  ASM_REWRITE_TAC[];
  UND 0;
  UND 1;
  UND 5;
  INT_ARITH_TAC;
  ]);;
  (* }}} *)

let inj_terminal = prove_by_refinement(
  `!G m. (rectagon G) ==>
     (INJ (terminal_edge (part_below G m))
         (endpoint (part_below G m)) UNIV)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE (part_below G m)` SUBGOAL_TAC ;
  ASM_MESON_TAC[part_below_finite;rectagon];
  DISCH_TAC;
  REWRITE_TAC[INJ];
  DISCH_ALL_TAC;
  TYPE_THEN `e = terminal_edge (part_below G m) x` ABBREV_TAC;
  TYPE_THEN `closure top2 e (pointI x) /\ closure top2 e (pointI y)` SUBGOAL_TAC;
  ASM_MESON_TAC[terminal_endpoint];
  DISCH_ALL_TAC;
  TYPE_THEN `(part_below G m) e` SUBGOAL_TAC;
  ASM_MESON_TAC[terminal_endpoint];
  DISCH_TAC;
  TYPE_THEN `part_below G m SUBSET G` SUBGOAL_TAC;
  REWRITE_TAC[part_below;ISUBSET];
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `edge e` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET;rectagon];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN DISJ_CASES_TAC);
  TYPE_THEN `(x = up m) /\ (y = up m)` SUBGOAL_TAC;
  ASM_MESON_TAC[terminal_case_v];
  MESON_TAC[];
  (* h-case *)
  UND 4;
  REWR 8;
  USE 4 (REWRITE_RULE[part_below_h ;]);
  DISCH_TAC;
  UND 4;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `(x = right  m') /\ (y = right m')` SUBGOAL_TAC  ;
  ASM_MESON_TAC[terminal_case1];
  MESON_TAC[];
  TYPE_THEN `( x= m' ) /\ (y = m') ` SUBGOAL_TAC;
  ASM_MESON_TAC[terminal_case2];
  MESON_TAC[];
  ]);;
  (* }}} *)

(* now start on surjectivity results *)

let endpoint_criterion = prove_by_refinement(
  `!G m e. (FINITE G) /\
       (!e'. (G e' /\ (closure top2 e' (pointI m))) = (e = e')) ==>
     (endpoint G m) /\ (e = terminal_edge G m)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  SUBCONJ_TAC;
  REWRITE_TAC[endpoint;];
  ASM_SIMP_TAC[num_closure1];
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_MESON_TAC[terminal_unique];
  ]);;
  (* }}} *)

let target_set = jordan_def `target_set G m =
    { e | (?n. (e = h_edge n) /\ (set_lower G m n)) \/
          (?n. (e = h_edge n) /\ (set_lower G (left m) n)) \/
          ((e = v_edge m) /\ G e)}`;;

let target_set_subset = prove_by_refinement(
  `!G m. target_set G m SUBSET G`,
  (* {{{ proof *)
  [
  REWRITE_TAC[ISUBSET;target_set;set_lower];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let target_edge = prove_by_refinement(
  `!G m. target_set G m SUBSET edge`,
  (* {{{ proof *)
  [
  REWRITE_TAC[target_set;edge;ISUBSET ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let target_h = prove_by_refinement(
  `!G m n. target_set G m (h_edge n) <=>
         (set_lower G m n) \/ (set_lower G (left  m) n)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[target_set;h_edge_inj; hv_edgeV2;];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let target_v = prove_by_refinement(
  `!G m n. target_set G m (v_edge n) <=>
        (n = m) /\ G (v_edge n)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[target_set;hv_edgeV2;v_edge_inj;];
  ]);;
  (* }}} *)

let part_below_subset = prove_by_refinement(
  `!G m. (part_below G m SUBSET G)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[part_below;ISUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let part_below_finite = prove_by_refinement(
  `!G m. (FINITE G ==> FINITE (part_below G m))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[part_below_subset];
  ]);;
  (* }}} *)

let terminal_edge_image = prove_by_refinement(
  `!G m x. (rectagon G) /\ (endpoint (part_below G m) x) ==>
      (target_set G m (terminal_edge (part_below G m) x))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon];
  DISCH_TAC;
  COPY 2;
  USE 2 ( MATCH_MP part_below_finite);
  TSPEC `m` 2;
  REWRITE_TAC[target_set];
  TYPE_THEN `e = terminal_edge (part_below G m) x` ABBREV_TAC;
  TYPE_THEN `(part_below G m e) /\ (closure top2 e (pointI x))` SUBGOAL_TAC;
  ASM_MESON_TAC[terminal_endpoint];
  DISCH_ALL_TAC;
  TYPE_THEN `edge e` SUBGOAL_TAC;
  ASM_MESON_TAC[part_below_subset;ISUBSET;rectagon];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN DISJ_CASES_TAC);
  ASM_REWRITE_TAC[hv_edgeV2;v_edge_inj];
  REWR 5;
  USE 5 (REWRITE_RULE[part_below_v]);
  ASM_REWRITE_TAC[PAIR_SPLIT ];
  REWR 6;
  USE 6 (REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT]);
  TYPE_THEN `FST x = FST m'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 6;
  TYPE_THEN `SND x = SND m +: &:1` SUBGOAL_TAC;
  ASM_MESON_TAC[endpoint_lemma_mid_fst];
  UND 6;
  AND 5;
  AND 5;
  UND 5;
  INT_ARITH_TAC;
  (* H edge *)
  ASM_REWRITE_TAC[hv_edgeV2;h_edge_inj;];
  REWR 5;
  USE 5(REWRITE_RULE[part_below_h ]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let terminal_edge_surj = prove_by_refinement(
  `!G m e. (rectagon G) /\ (target_set G m e) ==>
       (?x. (endpoint (part_below G m) x) /\
          (e = terminal_edge (part_below G m) x))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon];
  DISCH_TAC;
  TYPE_THEN `FINITE (part_below G m)` SUBGOAL_TAC;
  ASM_MESON_TAC[part_below_finite];
  DISCH_TAC;
  TYPE_THEN `(part_below G m) SUBSET G` SUBGOAL_TAC;
  ASM_MESON_TAC[part_below_subset];
  DISCH_TAC;
  TYPE_THEN `edge e` SUBGOAL_TAC;
  ASM_MESON_TAC[target_edge;ISUBSET];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN DISJ_CASES_TAC);
  REWR 1;
  USE 1(REWRITE_RULE[target_v]);
  AND 1;
  REWR 1;
  REWR 5;
  KILL 6;
  TYPE_THEN `up m` EXISTS_TAC;
  IMATCH_MP_TAC  endpoint_criterion;
  ASM_REWRITE_TAC[];
  GEN_TAC;
  EQ_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `edge e'` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET;rectagon];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN DISJ_CASES_TAC);
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_v]);
  ASM_REWRITE_TAC [v_edge_inj;PAIR_SPLIT];
  REWR 7;
  USE 7(REWRITE_RULE[v_edge_cpoint;PAIR_SPLIT;up;]);
  AND 6;
  AND 6;
  UND 6;
  UND 7;
  INT_ARITH_TAC;
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_h;set_lower;left  ;]);
  TYPE_THEN `SND m' <=: SND m` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 7;
  USE 7(REWRITE_RULE[h_edge_cpoint; up; PAIR_SPLIT ]);
  UND 7;
  UND 9;
  INT_ARITH_TAC;
  DISCH_TAC;
  EXPAND_TAC "e'";
  KILL 6;
  ASM_REWRITE_TAC [part_below_v;v_edge_cpoint;up];
  INT_ARITH_TAC;
  (* half-on-proof , hedge *)
  (* hop *)
  REWR 1;
  USE 1(REWRITE_RULE[target_h]);
  UND 1;
  DISCH_THEN (DISJ_CASES_TAC); (* split LEFT and RIGHT H *)
  TYPE_THEN `right  m'` EXISTS_TAC;
  IMATCH_MP_TAC  endpoint_criterion;
  ASM_REWRITE_TAC[];
  GEN_TAC;
  EQ_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `edge e'` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET;rectagon];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN DISJ_CASES_TAC); (* snd H or v *)
  REWR 6;
  USE 6 (REWRITE_RULE[part_below_v]);
  REWR 7;
  USE 7(REWRITE_RULE[v_edge_cpoint;right  ;PAIR_SPLIT; ]);
  REWRITE_TAC[h_edge_inj;hv_edgeV2;];
  USE 1 (REWRITE_RULE[set_lower]);
  ASM_MESON_TAC[INT_ARITH `~(x +: &:1 = x)`];
  ASM_REWRITE_TAC [h_edge_inj;PAIR_SPLIT ];  (* snd H *)
  KILL 5;
  UND 8;
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE [t]));
  RULE_ASSUM_TAC (REWRITE_RULE[part_below_h;h_edge_cpoint;PAIR_SPLIT;right  ]);
  UND 6;
  DISCH_THEN DISJ_CASES_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[set_lower]);
  ASM_MESON_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[set_lower;left  ]);
  AND 5;
  AND 5;
  PROOF_BY_CONTR_TAC;
  UND 8;
  UND 7;
  UND 1;
  INT_ARITH_TAC;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  REWRITE_TAC[part_below_h;h_edge_cpoint;right  ];
  ASM_REWRITE_TAC[];
  KILL 5;
  (* finally LEFT case: now everything needs to have an endpoint *)
  (* hop3*)
  USE 1 (REWRITE_RULE[set_lower;left  ]);
  TYPE_THEN `  m'` EXISTS_TAC ; (* was left  m *)
  IMATCH_MP_TAC  endpoint_criterion;
  ASM_REWRITE_TAC[];
  GEN_TAC;
  EQ_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `edge e'` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon;ISUBSET];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN DISJ_CASES_TAC);
  ASM_REWRITE_TAC[];
  UND 7;
  DISCH_THEN  (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  RULE_ASSUM_TAC  (REWRITE_RULE[part_below_v;v_edge_cpoint;left  ;PAIR_SPLIT ;]);
  UND 5;
  UND 6;
  UND 1;
  INT_ARITH_TAC;
  (* now H *)
  ASM_REWRITE_TAC[];
  UND 7;
  DISCH_THEN  (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  RULE_ASSUM_TAC  (REWRITE_RULE[part_below_h;h_edge_cpoint;left  ;PAIR_SPLIT ;]);
  UND 5;
  DISCH_THEN DISJ_CASES_TAC;
  USE 5(REWRITE_RULE[set_lower]);
  UND 5;
  UND 6;
  UND 1;
  INT_ARITH_TAC;
  (* hop2 *)
  USE 5 (REWRITE_RULE[set_lower]);
  REWRITE_TAC[h_edge_inj;PAIR_SPLIT;];
  UND 5;
  UND 6;
  UND 1;
  INT_ARITH_TAC;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  ASM_REWRITE_TAC[part_below_h;h_edge_cpoint; set_lower; left  ];
  ]);;
  (* }}} *)

(* set *)
let inj_subset = prove_by_refinement(
  `!t t' s (f:A->B). (INJ f s t') /\ (t SUBSET t') /\
         (IMAGE f s SUBSET t) ==> (INJ f s t)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ;IMAGE;SUBSET ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let terminal_edge_bij = prove_by_refinement(
  `!G m. (rectagon G) ==>
     (BIJ (terminal_edge (part_below G m))
         (endpoint (part_below G m)) (target_set G m))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[BIJ];
  SUBCONJ_TAC;
  IMATCH_MP_TAC  inj_subset;
  TYPE_THEN `UNIV:((num->real)->bool)->bool` EXISTS_TAC;
  ASM_SIMP_TAC[inj_terminal];
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_MESON_TAC[terminal_edge_image];
  REWRITE_TAC[INJ;SURJ];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[terminal_edge_surj];
  ]);;
  (* }}} *)

let target_set_finite = prove_by_refinement(
  `!G m. (FINITE  G) ==> (FINITE (target_set G m))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_MESON_TAC[target_set_subset];
  ]);;
  (* }}} *)

let rectagon_endpoint0 = prove_by_refinement(
  `!G. (rectagon G) ==> ((endpoint G) HAS_SIZE 0)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `endpoint G = {}` SUBGOAL_TAC;
  REWRITE_TAC[EQ_EMPTY];
  ASM_MESON_TAC[rectagon_endpoint];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[HAS_SIZE_0];
  ]);;
  (* }}} *)

let target_set_even = prove_by_refinement(
  `!G m. (rectagon G) ==> (EVEN (CARD (target_set G m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `CARD (endpoint(part_below G m)) = CARD (target_set G m)` SUBGOAL_TAC;
  IMATCH_MP_TAC  BIJ_CARD ;
  TYPE_THEN `terminal_edge (part_below G m)` EXISTS_TAC;
  ASM_SIMP_TAC[terminal_edge_bij];
  ASSUME_TAC terminal_edge_bij;
  TYPEL_THEN [`G`;`m`] (USE 1 o ISPECL);
  REWR 1;
  ASSUME_TAC target_set_finite;
  TYPEL_THEN [`G`;`m`] (USE 2 o ISPECL);
  ASM_MESON_TAC[FINITE_BIJ2;rectagon];
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  TYPE_THEN `rectagon (part_below G m)` ASM_CASES_TAC;
  TYPE_THEN `CARD (endpoint (part_below G m)) =0` SUBGOAL_TAC;
  ASM_MESON_TAC[HAS_SIZE;rectagon_endpoint0];
  MESON_TAC[EVEN];
  TYPE_THEN `P = part_below G m` ABBREV_TAC ;
  TYPE_THEN `segment G /\ (P SUBSET G) /\ ~(rectagon P)` SUBGOAL_TAC;
  ASM_SIMP_TAC[rectagon_segment];
  ASM_MESON_TAC[part_below_subset];
  DISCH_TAC;
  USE 3 (MATCH_MP endpoint_even );
  USE 3 (REWRITE_RULE[HAS_SIZE]);
  ASM_REWRITE_TAC[EVEN_DOUBLE];
  ]);;
  (* }}} *)

let bij_target_set = prove_by_refinement(
  `!G m. (rectagon G) /\ ~(G (v_edge m)) ==>
     (BIJ h_edge (set_lower G (left  m) UNION (set_lower G m))
           (target_set G m))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[BIJ];
  SUBCONJ_TAC;
  REWRITE_TAC[INJ];
  CONJ_TAC;
  REWRITE_TAC[target_set;set_lower;UNION;h_edge_inj;hv_edgeV2; ];
  MESON_TAC[];
  REWRITE_TAC[h_edge_inj;];
  MESON_TAC[];
  REWRITE_TAC[INJ;SURJ];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[target_set;set_lower;UNION;];
  GEN_TAC;
  REP_CASES_TAC;
  CHO 4;
  UND 4;
  DISCH_ALL_TAC;
  ASM_MESON_TAC[];
  CHO 4;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let bij_target_set_odd = prove_by_refinement(
  `!G m. (rectagon G) /\ (G (v_edge m)) ==>
     (BIJ h_edge (set_lower G (left  m) UNION
             (set_lower G m) )
           (target_set G m DELETE (v_edge m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[BIJ];
  SUBCONJ_TAC;
  REWRITE_TAC[INJ];
  CONJ_TAC;
  REWRITE_TAC[target_set;set_lower;UNION;h_edge_inj;hv_edgeV2; DELETE ];
  MESON_TAC[];
  REWRITE_TAC[h_edge_inj;];
  MESON_TAC[];
  REWRITE_TAC[INJ;SURJ];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[target_set;set_lower;UNION;DELETE ];
  GEN_TAC;
  DISCH_TAC;
  AND  4;
  REWR 5;
  UND 5;
  REP_CASES_TAC;
  CHO 5;
  UND 5;
  DISCH_ALL_TAC;
  ASM_MESON_TAC[];
  CHO 5;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let target_set_odd = prove_by_refinement(
  `!G m. (rectagon G) /\ (G (v_edge m)) ==>
         ~(EVEN(CARD (target_set G m DELETE (v_edge m))))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM EVEN];
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE (target_set G m)` SUBGOAL_TAC;
  ASM_MESON_TAC[target_set_finite;rectagon];
  DISCH_TAC;
  TYPE_THEN `target_set G m (v_edge m)` SUBGOAL_TAC;
  ASM_REWRITE_TAC [target_v];
  DISCH_TAC;
  TYPE_THEN `SUC (CARD (target_set G m DELETE (v_edge m))) = CARD (target_set G m )` SUBGOAL_TAC;
  IMATCH_MP_TAC  CARD_SUC_DELETE;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[target_set_even];
  ]);;
  (* }}} *)

let squ_left_even = prove_by_refinement(
  `!G m. (rectagon G) /\ ~(G (v_edge m)) ==>
     ((even_cell G (squ (left m)) = even_cell G(squ m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon];
  DISCH_TAC;
  REWRITE_TAC[even_cell_squ;num_lower_set];
  TYPE_THEN `(EVEN (CARD (set_lower G (left m))) <=> EVEN (CARD (set_lower G m))) <=> (EVEN (CARD ((set_lower G (left m)) UNION (set_lower G m))))` SUBGOAL_TAC;
  IMATCH_MP_TAC  even_card_even;
  ASM_SIMP_TAC[finite_set_lower];
  REWRITE_TAC[set_lower;INTER ;left ;EQ_EMPTY ];
  MESON_TAC[INT_ARITH `~(z = z -: &:1)`];
  DISCH_THEN_REWRITE;
  TYPE_THEN `BIJ h_edge (set_lower G (left  m) UNION (set_lower G m)) (target_set G m) ` SUBGOAL_TAC;
  ASM_MESON_TAC[bij_target_set];
  DISCH_TAC;
  TYPE_THEN `CARD (set_lower G (left  m) UNION (set_lower G m)) = CARD (target_set G m)` SUBGOAL_TAC;
  IMATCH_MP_TAC  BIJ_CARD ;
  TYPE_THEN `h_edge` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[FINITE_UNION];
  ASM_MESON_TAC[finite_set_lower];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[target_set_even];
  ]);;
  (* }}} *)

let squ_left_odd = prove_by_refinement(
  `!G m. (rectagon G) /\ (G (v_edge m)) ==>
     (~(even_cell G (squ (left m)) = even_cell G(squ m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon];
  DISCH_TAC;
  UND 0;
  REWRITE_TAC[even_cell_squ;num_lower_set];
  TYPE_THEN `(EVEN (CARD (set_lower G (left m))) <=> EVEN (CARD (set_lower G m))) <=> (EVEN (CARD ((set_lower G (left m)) UNION (set_lower G m))))` SUBGOAL_TAC;
  IMATCH_MP_TAC  even_card_even;
  ASM_SIMP_TAC[finite_set_lower];
  REWRITE_TAC[set_lower;INTER ;left ;EQ_EMPTY ];
  MESON_TAC[INT_ARITH `~(z = z -: &:1)`];
  DISCH_THEN_REWRITE;
  TYPE_THEN `BIJ h_edge (set_lower G (left  m) UNION (set_lower G m)) (target_set G m DELETE (v_edge m)) ` SUBGOAL_TAC;
  ASM_MESON_TAC[bij_target_set_odd];
  DISCH_TAC;
  TYPE_THEN `CARD (set_lower G (left  m) UNION (set_lower G m)) = CARD (target_set G m DELETE (v_edge m))` SUBGOAL_TAC;
  IMATCH_MP_TAC  BIJ_CARD ;
  TYPE_THEN `h_edge` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[FINITE_UNION];
  ASM_MESON_TAC[finite_set_lower];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[target_set_odd];
  ]);;
  (* }}} *)

let squ_left_par = prove_by_refinement(
  `!G m. (rectagon G) ==>
       (((even_cell G (squ (left m)) = even_cell G(squ m))) <=>
            ~(G (v_edge m)))`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[squ_left_even;squ_left_odd];
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* SECTION E *)
(* ------------------------------------------------------------------ *)


let rectangle = jordan_def `rectangle p q =
  {Z | ?u v. (Z = point(u,v)) /\
    (real_of_int (FST p ) <. u) /\ (u <. (real_of_int (FST q ))) /\
    (real_of_int (SND p ) <. v) /\ (v <. (real_of_int (SND q))) }`;;

let rectangle_inter = prove_by_refinement(
  `!p q. rectangle p q =
      {z | ?r. (z = point r) /\ (real_of_int(FST p) <. FST r)} INTER
      {z | ?r. (z = point r) /\ (real_of_int(SND p) <. SND r)} INTER
     {z | ?r. (z = point r) /\ (FST r ) <. real_of_int(FST q)} INTER
    {z | ?r. (z = point r) /\ (SND  r ) <. real_of_int(SND  q)}  `,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[rectangle;INTER];
  GEN_TAC;
  EQ_TAC;
  DISCH_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[point_inj];
  CONV_TAC (dropq_conv "r");
  ASM_REWRITE_TAC[];
  CONV_TAC (dropq_conv "r");
  ASM_REWRITE_TAC[];
  CONV_TAC (dropq_conv "r'");
  CONV_TAC (dropq_conv "r");
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[point_inj]);
  USE 1(CONV_RULE (dropq_conv "r'"));
  REWR 2;
  USE 2(REWRITE_RULE[point_inj]);
  USE 2(CONV_RULE (dropq_conv "r'"));
  REWR 3;
  USE 3(REWRITE_RULE[point_inj]);
  USE 3(CONV_RULE (dropq_conv "r'"));
  REWRITE_TAC[point_inj;PAIR_SPLIT];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let rectangle_open = prove_by_refinement(
  `!p q. top2 (rectangle p q)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[rectangle_inter];
  ASSUME_TAC top2_top;
  DISCH_ALL_TAC;
  REPEAT (IMATCH_MP_TAC  top_inter THEN ASM_REWRITE_TAC[top_inter;open_half_plane2D_FLT_open;open_half_plane2D_LTF_open;open_half_plane2D_SLT_open;open_half_plane2D_LTS_open]);
  ]);;
  (* }}} *)

let rectangle_convex = prove_by_refinement(
  `!p q. convex (rectangle p q)`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[rectangle_inter];
  REPEAT (IMATCH_MP_TAC  convex_inter THEN REWRITE_TAC[open_half_plane2D_FLT_convex;open_half_plane2D_LTF_convex;open_half_plane2D_SLT_convex;open_half_plane2D_LTS_convex]);
  ]);;
  (* }}} *)

let rectangle_squ = prove_by_refinement(
  `!p. squ p = rectangle p (FST p +: &:1,SND p +: &:1)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[squ;rectangle];
  ]);;
  (* }}} *)

let squ_inter = prove_by_refinement(
  `!p. squ p =
   {z | ?r. (z = point r) /\ (real_of_int(FST p) <. FST r)} INTER
      {z | ?r. (z = point r) /\ (real_of_int(SND p) <. SND r)} INTER
     {z | ?r. (z = point r) /\ (FST r ) <. real_of_int(FST p +: &:1) } INTER
    {z | ?r. (z = point r) /\ (SND  r ) <. real_of_int(SND p +: &:1) }`,
  (* {{{ proof *)
  [
  REWRITE_TAC[rectangle_squ;rectangle_inter];
  ]);;
  (* }}} *)

(* set *)
let subset3_absorb = prove_by_refinement(
  `!(A:A->bool) B C. (B SUBSET C) ==> (B INTER A = B INTER C INTER A)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[INTER_ACI];
  AP_TERM_TAC;
  ASM_MESON_TAC[SUBSET_INTER_ABSORPTION];
  ]);;
  (* }}} *)

let rectangle_lemma1 = prove_by_refinement(
  `!p. squ(down p) =
     (rectangle (FST p , SND p -: &:1) (FST p +: &:1 , SND p +: &:1))
    INTER {z | ?r. (z = point r) /\ (SND  r <. real_of_int(SND  p))}`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[squ_inter;rectangle_inter;down];
  REWRITE_TAC[INT_ARITH `x -: &:1 +: &:1 = x`];
  REWRITE_TAC[INTER_ACI];
  AP_TERM_TAC;
  AP_TERM_TAC;
  AP_TERM_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INTER;int_suc ;];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  ASSUME_TAC (REAL_ARITH `!u. u <. u + &.1`);
  CONJ_TAC;
  TYPE_THEN `r` EXISTS_TAC;
  ASM_MESON_TAC[REAL_LT_TRANS ];
  ASM_MESON_TAC[];
  MESON_TAC[];
  ]);;
  (* }}} *)


let rectangle_lemma2 = prove_by_refinement(
  `!p. squ(p) =
     (rectangle (FST p , SND p -: &:1) (FST p +: &:1 , SND p +: &:1))
    INTER {z | ?r. (z = point r) /\ ( real_of_int(SND  p) <. SND  r)}`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[squ_inter;rectangle_inter;down];
  REWRITE_TAC[INTER_ACI];
  AP_TERM_TAC;
  TYPE_THEN `B = {z | ?r. (z = point r) /\ real_of_int (SND p) < SND r}` ABBREV_TAC ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\ real_of_int (SND p -: &:1) < SND r}` ABBREV_TAC ;
  REWRITE_TAC[INTER_ACI];
  IMATCH_MP_TAC  subset3_absorb;
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  REWRITE_TAC[SUBSET;int_sub_th; int_of_num_th];
  ASM_MESON_TAC[REAL_ARITH `a <. b ==> (a - &.1 <. b)`];
  ]);;
  (* }}} *)

let rectangle_lemma3 = prove_by_refinement(
  `!q. h_edge q =
    (rectangle (FST q , SND q -: &:1) (FST q +: &:1 , SND q +: &:1))
    INTER {z | ?r. (z = point r) /\ ( SND  r = real_of_int(SND  q))}`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[h_edge_inter;rectangle_inter;];
  TYPE_THEN `B = {z | ?p. (z = point p) /\ (SND p = real_of_int (SND q))}` ABBREV_TAC ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\ real_of_int (SND q -: &:1) < SND r}` ABBREV_TAC ;
  TYPE_THEN `D = {z | ?r. (z = point r) /\ SND r < real_of_int (SND q +: &:1)}` ABBREV_TAC ;
  REWRITE_TAC[INTER_ACI];
  TYPE_THEN `!A. B INTER C INTER D INTER A = B INTER (C INTER D) INTER A` SUBGOAL_TAC;
  REWRITE_TAC[INTER_ACI];
  DISCH_THEN (fun t-> PURE_REWRITE_TAC [t]);
  IMATCH_MP_TAC subset3_absorb;
  REWRITE_TAC[SUBSET_INTER];
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  EXPAND_TAC "D";
  REWRITE_TAC[SUBSET;int_sub_th; int_of_num_th;int_add_th;];
  ASM_MESON_TAC[REAL_ARITH `x - &.1 <. x /\ x < x + &.1`];
  ]);;
  (* }}} *)

let rectangle_h = prove_by_refinement(
  `!p. rectangle (FST p , SND p -: &:1) (FST p +: &:1 , SND p +: &:1) =
     ((squ (down p)) UNION (h_edge p) UNION  (squ p) )`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[rectangle_lemma1;rectangle_lemma2;rectangle_lemma3];
  REWRITE_TAC[GSYM UNION_OVER_INTER];
  TYPE_THEN `({z | ?r. (z = point r) /\ SND r < real_of_int (SND p)} UNION  {z | ?r. (z = point r) /\ (SND r = real_of_int (SND p))} UNION  {z | ?r. (z = point r) /\ real_of_int (SND p) < SND r}) = {z | ?r. (z = point r)}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[UNION];
  ASM_MESON_TAC[REAL_ARITH `!x y. (x <. y) \/ (x = y) \/ (y <. x)`];
  DISCH_THEN_REWRITE;
  TYPE_THEN `rectangle (FST p,SND p -: &:1) (FST p +: &:1,SND p +: &:1) SUBSET  {z | ?r. z = point r}` SUBGOAL_TAC;
  REWRITE_TAC[rectangle;SUBSET ];
  ASM_MESON_TAC[];
  REWRITE_TAC [SUBSET_INTER_ABSORPTION;];
  DISCH_THEN_REWRITE;
  ]);;
  (* }}} *)

let rectangle_lemma4 = prove_by_refinement(
  `!p. squ(left   p) =
     (rectangle (FST p -: &:1 , SND p)(FST p +: &:1 , SND p +: &:1))
    INTER {z | ?r. (z = point r) /\ (FST   r <. real_of_int(FST  p))}`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[squ_inter;rectangle_inter;left  ];
  REWRITE_TAC[INT_ARITH `x -: &:1 +: &:1 = x`];
  REWRITE_TAC[INTER_ACI];
  AP_TERM_TAC;
  AP_TERM_TAC;
  TYPE_THEN `B = {z | ?r. (z = point r) /\ FST r < real_of_int (FST p)}` ABBREV_TAC  ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\ FST r < real_of_int (FST p +: &:1)}` ABBREV_TAC ;
  REWRITE_TAC[INTER_ACI];
  IMATCH_MP_TAC  subset3_absorb;
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  REWRITE_TAC[SUBSET;int_suc];
  ASM_MESON_TAC[REAL_ARITH `x <. y ==> x <. y + &.1`];
  ]);;
  (* }}} *)

let rectangle_lemma5 = prove_by_refinement(
  `!p. squ(p) =
     (rectangle (FST p -: &:1 , SND p) (FST p +: &:1 , SND p +: &:1))
    INTER {z | ?r. (z = point r) /\ ( real_of_int(FST   p) <. FST   r)}`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[squ_inter;rectangle_inter;];
TYPE_THEN `B = {z | ?r. (z = point r) /\ real_of_int (FST p) < FST r} ` ABBREV_TAC ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\ real_of_int (FST p -: &:1) < FST r}` ABBREV_TAC ;
  REWRITE_TAC[INTER_ACI];
  IMATCH_MP_TAC  subset3_absorb;
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  REWRITE_TAC[SUBSET;int_sub_th; int_of_num_th];
  ASM_MESON_TAC[REAL_ARITH `a <. b ==> (a - &.1 <. b)`];
  ]);;
  (* }}} *)

let rectangle_lemma6 = prove_by_refinement(
  `!q. v_edge q =
    (rectangle (FST q -: &:1 , SND q) (FST q +: &:1 , SND q +: &:1))
    INTER {z | ?r. (z = point r) /\ ( FST   r = real_of_int(FST   q))}`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[v_edge_inter;rectangle_inter;];
  REWRITE_TAC[INTER_ACI];
  TYPE_THEN `B = {z | ?p. (z = point p) /\ (FST  p = real_of_int (FST  q))}` ABBREV_TAC ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\ real_of_int (FST q -: &:1) < FST r}` ABBREV_TAC ;
  TYPE_THEN `D = {z | ?r. (z = point r) /\ FST r < real_of_int (FST q +: &:1)}` ABBREV_TAC ;
  REWRITE_TAC[INTER_ACI];
  TYPE_THEN `!A. B INTER C INTER D INTER A = B INTER (C INTER D) INTER A` SUBGOAL_TAC;
  REWRITE_TAC[INTER_ACI];
  DISCH_THEN (fun t-> PURE_REWRITE_TAC [t]);
  IMATCH_MP_TAC subset3_absorb;
  REWRITE_TAC[SUBSET_INTER];
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  EXPAND_TAC "D";
  REWRITE_TAC[SUBSET;int_sub_th; int_of_num_th;int_add_th;];
  ASM_MESON_TAC[REAL_ARITH `x - &.1 <. x /\ x < x + &.1`];
  ]);;
  (* }}} *)

let rectangle_v = prove_by_refinement(
  `!p. rectangle (FST p -: &:1 , SND p ) (FST p +: &:1 , SND p +: &:1) =
     ((squ (left p)) UNION (v_edge p) UNION  (squ p) )`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[rectangle_lemma4;rectangle_lemma5;rectangle_lemma6];
  REWRITE_TAC[GSYM UNION_OVER_INTER];
  TYPE_THEN `({z | ?r. (z = point r) /\ FST r < real_of_int (FST p)} UNION  {z | ?r. (z = point r) /\ (FST r = real_of_int (FST p))} UNION  {z | ?r. (z = point r) /\ real_of_int (FST p) < FST r}) = {z | ?r. (z = point r)}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[UNION];
  ASM_MESON_TAC[REAL_ARITH `!x y. (x <. y) \/ (x = y) \/ (y <. x)`];
  DISCH_THEN_REWRITE;
  TYPE_THEN `rectangle (FST p -: &:1 ,SND p) (FST p +: &:1,SND p +: &:1) SUBSET  {z | ?r. z = point r}` SUBGOAL_TAC;
  REWRITE_TAC[rectangle;SUBSET ];
  ASM_MESON_TAC[];
  REWRITE_TAC [SUBSET_INTER_ABSORPTION;];
  DISCH_THEN_REWRITE;
  ]);;
  (* }}} *)

let long_v = jordan_def `long_v p =
  {z | (?r. (z = point r) /\ (FST r = real_of_int (FST p)) /\
       (real_of_int(SND  p) - &1 <. SND r) /\
       (SND r <. real_of_int(SND p) + &1) )}`;;

let long_v_inter = prove_by_refinement(
  `!p. long_v p =
    {z | ?r. (z = point r) /\ (FST r = real_of_int (FST p))} INTER
      {z | ?r. (z = point r) /\ (real_of_int(SND p -: &:1) <. SND r)} INTER
     {z | ?r. (z = point r) /\ (SND  r  <. real_of_int(SND  p +: &:1))} `,
  (* {{{ proof *)

  [
  GEN_TAC;
  IMATCH_MP_TAC  EQ_EXT ;
  REWRITE_TAC[long_v;INTER;int_add_th;int_sub_th;int_of_num_th];
  GEN_TAC;
  EQ_TAC;
  DISCH_THEN CHOOSE_TAC;
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  CHO 0;
  REWR 1;
  REWR 2;
  RULE_ASSUM_TAC  (REWRITE_RULE[point_inj]);
  USE 2(CONV_RULE (dropq_conv "r'"));
  USE 1(CONV_RULE (dropq_conv "r'"));
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let long_v_lemma1 = prove_by_refinement(
  `!q. v_edge (down q) =
     long_v q INTER
         {z | ?r. (z = point r) /\ (SND  r  <. real_of_int(SND  q))}`,
  (* {{{ proof *)
  [
  REWRITE_TAC[v_edge_inter;long_v_inter;down ];
  REWRITE_TAC[INT_ARITH `x -: &:1 +: &:1 = x`];
  GEN_TAC;
  TYPE_THEN `B = {z | ?r. (z = point r) /\ SND r < real_of_int (SND q)}` ABBREV_TAC ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\ SND r < real_of_int (SND q +: &:1)}` ABBREV_TAC ;
  alpha_tac;
  REWRITE_TAC[INTER_ACI];
  IMATCH_MP_TAC  subset3_absorb;
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  REWRITE_TAC[SUBSET;int_add_th;int_of_num_th];
  MESON_TAC[REAL_ARITH `x <. y ==> x <. y + &1`];
  ]);;
  (* }}} *)

let long_v_lemma2 = prove_by_refinement(
  `!q. v_edge q =
     long_v q INTER
         {z | ?r. (z = point r) /\ (real_of_int(SND  q) <. SND  r  )}`,
  (* {{{ proof *)
  [
  REWRITE_TAC[v_edge_inter;long_v_inter;down;int_suc;int_sub_th;int_of_num_th ];
  GEN_TAC;
  TYPE_THEN `B = {z | ?r. (z = point r) /\  real_of_int (SND q) < SND r}` ABBREV_TAC ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\  real_of_int (SND q) - &1 < SND r}` ABBREV_TAC ;
  alpha_tac;
  REWRITE_TAC[INTER_ACI];
  IMATCH_MP_TAC  subset3_absorb;
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  REWRITE_TAC[SUBSET;int_add_th;int_of_num_th];
  MESON_TAC[REAL_ARITH `x <. y ==> x - &1 <. y`];
  ]);;
  (* }}} *)

let pointI_inter = prove_by_refinement(
  `!q. {(pointI q)} =
        {z | ?r. (z = point r) /\ (FST r = real_of_int (FST q))} INTER
        {z | ?r. (z = point r) /\ (real_of_int (SND q) = SND r)}`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;INR IN_SING;pointI ];
  GEN_TAC;
  EQ_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[point_inj];
  CONV_TAC (dropq_conv "r");
  CONV_TAC (dropq_conv "r'");
  DISCH_ALL_TAC;
  CHO 0;
  REWR 1;
  USE 1(REWRITE_RULE[point_inj]);
  USE 1(CONV_RULE (dropq_conv "r'"));
  ASM_REWRITE_TAC[point_inj;PAIR_SPLIT;];
  ]);;
  (* }}} *)

let long_v_lemma3 = prove_by_refinement(
  `!q. {(pointI q)} = long_v q INTER
       { z | ?r. (z = point r) /\ (real_of_int(SND q) = SND r)}`,
  (* {{{ proof *)
  [
  REWRITE_TAC[pointI_inter;long_v_inter];
  GEN_TAC;
  alpha_tac;
  TYPE_THEN `A = {z | ?r. (z = point r) /\ (FST r = real_of_int (FST q))}` ABBREV_TAC ;
  TYPE_THEN `B = {z | ?r. (z = point r) /\ (real_of_int (SND q) = SND r)}` ABBREV_TAC ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\ real_of_int (SND q -: &:1) < SND r}` ABBREV_TAC ;
  TYPE_THEN `D = {z | ?r. (z = point r) /\ SND r < real_of_int (SND q +: &:1)}` ABBREV_TAC ;
  REWRITE_TAC[INTER_ACI];
  AP_TERM_TAC;
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  REWRITE_TAC[GSYM SUBSET_INTER_ABSORPTION];
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  EXPAND_TAC "D";
  REWRITE_TAC[SUBSET;INTER;int_sub_th;int_of_num_th;int_add_th];
  ASM_MESON_TAC[REAL_ARITH `(x = y) ==> (x - &1 <. y /\ x <. y + &1)`];
  ]);;
  (* }}} *)

let long_v_union = prove_by_refinement(
  `!p. long_v p =
      (v_edge (down p)) UNION {(pointI p)} UNION (v_edge p)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[long_v_lemma1;long_v_lemma2;long_v_lemma3];
  REWRITE_TAC[GSYM UNION_OVER_INTER];
  TYPE_THEN `({z | ?r. (z = point r) /\ SND r < real_of_int (SND p)} UNION  {z | ?r. (z = point r) /\ (real_of_int (SND p) = SND r)} UNION  {z | ?r. (z = point r) /\ real_of_int (SND p) < SND r}) = {z | ?r. (z = point r)}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT  ;
  GEN_TAC;
  REWRITE_TAC[UNION;];
  EQ_TAC;
  MESON_TAC[];
  DISCH_THEN CHOOSE_TAC;
  ASM_REWRITE_TAC[point_inj];
  CONV_TAC (dropq_conv "r'");
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ONCE_REWRITE_TAC[EQ_SYM_EQ];
  REWRITE_TAC[GSYM SUBSET_INTER_ABSORPTION;];
  REWRITE_TAC[long_v;SUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let two_two_lemma1 = prove_by_refinement(
  `!p. rectangle(FST p - &:1 , SND p - &:1) (FST p , SND p + &:1) =
  rectangle (FST p -: &:1 , SND p -: &:1) (FST p +: &:1 , SND p + &:1)
     INTER
  {z | (?r. (z = point r) /\ (FST r <. real_of_int(FST p)))}`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[rectangle_inter];
  alpha_tac;
  TYPE_THEN `B  = {z | ?r. (z = point r) /\ FST r < real_of_int (FST p)}` ABBREV_TAC  ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\ FST r < real_of_int (FST p +: &:1)} ` ABBREV_TAC ;
  REWRITE_TAC[INTER_ACI];
  IMATCH_MP_TAC  subset3_absorb;
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  REWRITE_TAC[SUBSET;int_suc;];
  MESON_TAC[REAL_ARITH `x <. y ==> x < y + &1`];
  ]);;
  (* }}} *)

let two_two_lemma2 = prove_by_refinement(
  `!p. rectangle(FST p , SND p - &:1) (FST p + &:1 ,SND p + &:1) =
  rectangle (FST p -: &:1 , SND p -: &:1) (FST p +: &:1 , SND p + &:1)
  INTER
  {z | (?r. (z = point r) /\ ( real_of_int(FST p) <. FST r ))}`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[rectangle_inter];
  alpha_tac;
  TYPE_THEN `B = {z | ?r. (z = point r) /\ real_of_int (FST p) < FST r}` ABBREV_TAC ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\ real_of_int (FST p -: &:1) < FST r}` ABBREV_TAC ;
  REWRITE_TAC[INTER_ACI];
  IMATCH_MP_TAC  subset3_absorb;
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  REWRITE_TAC[SUBSET;int_sub_th;int_add_th;int_of_num_th;];
  ASM_MESON_TAC[REAL_ARITH `x < y ==> (x - &1 <. y)`];
  ]);;
  (* }}} *)

let two_two_lemma3 = prove_by_refinement(
  `!p. long_v p =
  rectangle (FST p -: &:1 , SND p -: &:1) (FST p +: &:1 , SND p + &:1)
  INTER
    {z | (?r. (z = point r) /\ (  FST r =  real_of_int(FST p)  ))}`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[long_v_inter;rectangle_inter];
  alpha_tac;
  TYPE_THEN `B = {z | ?r. (z = point r) /\ (FST r = real_of_int (FST p))} ` ABBREV_TAC ;
  TYPE_THEN `C = {z | ?r. (z = point r) /\ real_of_int (FST p -: &:1) < FST r}` ABBREV_TAC ;
  TYPE_THEN `D = {z | ?r. (z = point r) /\ FST r < real_of_int (FST p +: &:1)} ` ABBREV_TAC ;
  REWRITE_TAC[INTER_ACI];
  TYPE_THEN `!A. (B INTER C INTER D INTER A) = B INTER (C INTER D) INTER A` SUBGOAL_TAC;
  REWRITE_TAC[INTER_ACI];
  DISCH_THEN (fun t-> PURE_REWRITE_TAC[t]);
  IMATCH_MP_TAC  subset3_absorb;
  EXPAND_TAC "B";
  EXPAND_TAC "C";
  EXPAND_TAC "D";
  REWRITE_TAC[SUBSET;INTER;int_sub_th;int_add_th;int_of_num_th];
  GEN_TAC;
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  ASM_MESON_TAC[REAL_ARITH `(x = y) ==> (x - &.1 <. y /\ x <. y+ &1)`];
  ]);;
  (* }}} *)

let two_two_union = prove_by_refinement(
  `!p. rectangle (FST p -: &:1 , SND p -: &:1)
     (FST p +: &:1 , SND p + &:1) =
   rectangle(FST p - &:1 , SND p - &:1) (FST p  , SND p + &:1) UNION
   long_v p UNION
   rectangle(FST p , SND p - &:1) (FST p + &:1 ,SND p + &:1)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[two_two_lemma1;two_two_lemma2;two_two_lemma3];
  REWRITE_TAC[GSYM UNION_OVER_INTER];
  GEN_TAC;
  TYPE_THEN `{z | ?r. (z = point r)} = ({z | ?r. (z = point r) /\ FST r < real_of_int (FST p)} UNION {z | ?r. (z = point r) /\ (FST r = real_of_int (FST p))} UNION {z | ?r. (z = point r) /\ real_of_int (FST p) < FST r})` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[UNION];
  EQ_TAC;
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  DISCH_THEN_REWRITE;
  REWRITE_TAC [point_inj];
  CONV_TAC (dropq_conv "r'");
  REAL_ARITH_TAC;
  MESON_TAC[];
  DISCH_TAC;
  USE 0 SYM;
  ASM_REWRITE_TAC[];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  REWRITE_TAC[GSYM SUBSET_INTER_ABSORPTION];
  REWRITE_TAC[rectangle;SUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let two_two_nine = prove_by_refinement(
  `!p. rectangle (FST p -: &:1 , SND p -: &:1) (FST p +: &:1 , SND p + &:1) =
   squ (FST p -: &:1,SND p -: &:1) UNION squ (FST p -: &:1,SND p ) UNION
   squ (FST p,SND p -: &:1) UNION squ p UNION
   h_edge (left  p) UNION h_edge  p UNION
   v_edge (down p) UNION v_edge p UNION {(pointI p)}`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[two_two_union;rectangle_h;rectangle_v];
  TYPE_THEN `rectangle (FST p -: &:1,SND p -: &:1) (FST p,SND p +: &:1) = rectangle (FST (left  p),SND (left  p) -: &:1) (FST (left  p) +: &:1,SND (left   p) +: &:1)` SUBGOAL_TAC;
  REWRITE_TAC[left ;INT_ARITH `x -: &:1 +: &:1 = x`];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[rectangle_h];
  REWRITE_TAC[left ;down; long_v_union];
  REWRITE_TAC[UNION_ACI];
  ]);;
  (* }}} *)


(* ------------------------------------------------------------------ *)

let curve_cell = jordan_def `curve_cell G = G UNION
   {z | (?n. (z = {(pointI n)}) /\ (closure top2 (UNIONS G) (pointI n)))}`;;

let curve_cell_cell = prove_by_refinement(
  `!G. (G SUBSET edge) ==> (curve_cell G SUBSET cell)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;edge;curve_cell;cell;UNION ];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  UND 1;
  DISCH_THEN DISJ_CASES_TAC;
  TSPEC `x` 0;
  REWR 0;
  CHO 0;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let curve_cell_point = prove_by_refinement(
  `!G n. (FINITE G) /\ (G SUBSET edge) ==> (curve_cell G {(pointI n)} <=>
           (?e. (G e /\ (closure top2 e (pointI n)))))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[curve_cell;UNION ;edge;SUBSET ];
  DISCH_ALL_TAC;
  EQ_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  TSPEC `{(pointI n)}` 1;
  USE 1(GSYM);
  USE 1(REWRITE_RULE[eq_sing;v_edge_pointI;h_edge_pointI;]);
  ASM_MESON_TAC[];
  USE 2 (REWRITE_RULE[eq_sing;INR IN_SING ;pointI_inj]);
  USE 2(CONV_RULE (dropq_conv "n'"));
  ASSUME_TAC top2_top;
  UND 2;
  ASM_SIMP_TAC[closure_unions];
  REWRITE_TAC[IMAGE;INR IN_UNIONS ];
  DISCH_THEN CHOOSE_TAC;
  AND 2;
  CHO 4;
  ASM_MESON_TAC[];
  DISCH_THEN CHOOSE_TAC;
  DISJ2_TAC;
  REWRITE_TAC[eq_sing;INR IN_SING;pointI_inj;];
  CONV_TAC (dropq_conv "n'") ;
  TYPE_THEN `closure top2 e SUBSET closure top2 (UNIONS G)` SUBGOAL_TAC;
  IMATCH_MP_TAC  subset_of_closure;
  REWRITE_TAC[top2_top];
  IMATCH_MP_TAC  sub_union;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let curve_cell_h = prove_by_refinement(
  `!G n. (segment G) ==> (curve_cell G (h_edge n) = G (h_edge n))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[curve_cell;UNION ; eq_sing;INR IN_SING; h_edge_pointI];
  ]);;
  (* }}} *)

let curve_cell_v = prove_by_refinement(
  `!G n. (segment G) ==> (curve_cell G (v_edge n) = G (v_edge n))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[curve_cell;UNION ; eq_sing;INR IN_SING; v_edge_pointI];
  ]);;
  (* }}} *)

let curve_cell_in = prove_by_refinement(
  `!C G . (G SUBSET edge) /\ (curve_cell G C) ==>
    (?n. (C = {(pointI n)}) \/ (C = h_edge n) \/ (C = v_edge n))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[curve_cell;UNION ;SUBSET; edge ];
  DISCH_ALL_TAC;
  UND 1;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let curve_cell_subset = prove_by_refinement(
  `!G. (G SUBSET (curve_cell G))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;curve_cell;UNION ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let curve_closure = prove_by_refinement(
  `!G. (segment G) ==>
    (closure top2 (UNIONS G) = (UNIONS (curve_cell G)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC ;
  ASSUME_TAC top2_top;
  (* ASM_SIMP_TAC[closure_unions]; *)
  TYPE_THEN `G SUBSET edge ` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  ASM_SIMP_TAC[closure_unions];
  REWRITE_TAC[IMAGE;INR IN_UNIONS;SUBSET ];
  DISCH_ALL_TAC;
  CHO 4;
  AND 4;
  CHO 5;
  TYPE_THEN `edge x'` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;ISUBSET];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN DISJ_CASES_TAC);
  REWR 5;
  REWR 4;
  COPY 4;
  USE 4(REWRITE_RULE[v_edge_closure;vc_edge;UNION ;INR IN_SING ]);
  UND 4;
  REP_CASES_TAC;
  TYPE_THEN `v_edge m` EXISTS_TAC;
  ASM_SIMP_TAC [curve_cell_v];
  TYPE_THEN `{(pointI m)}` EXISTS_TAC;

  ASM_SIMP_TAC [curve_cell_point];
  REWRITE_TAC[INR IN_SING];
  ASM_MESON_TAC[];
  USE 4(REWRITE_RULE[plus_e12]);
  TYPE_THEN `{(pointI (FST m,SND m +: &:1))}` EXISTS_TAC;

  ASM_SIMP_TAC [curve_cell_point];
  REWRITE_TAC[INR IN_SING];
  ASM_MESON_TAC[];
  (* dt2 , down to 2 goals *)
  REWR 5;
  REWR 4;
  COPY 4;
  USE 4 (REWRITE_RULE[h_edge_closure;hc_edge;UNION;INR IN_SING]);
  UND 4;
  REP_CASES_TAC;
  TYPE_THEN `h_edge m` EXISTS_TAC;
  ASM_SIMP_TAC[curve_cell_h];
  TYPE_THEN `{(pointI m)}` EXISTS_TAC;
  ASM_SIMP_TAC[curve_cell_point ;INR IN_SING ];
  ASM_MESON_TAC[];
  USE 4(REWRITE_RULE[plus_e12]);
  TYPE_THEN `{x}` EXISTS_TAC;
  ASM_REWRITE_TAC[INR IN_SING];
  ASM_SIMP_TAC[curve_cell_point ;INR IN_SING ];
  ASM_MESON_TAC[];
  (* dt1 *)
  REWRITE_TAC[curve_cell; UNIONS_UNION; union_subset];
  ASM_SIMP_TAC[closure_unions];
  CONJ_TAC;
  REWRITE_TAC[SUBSET;IMAGE;UNIONS];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "u");
  NAME_CONFLICT_TAC;
  CHO 4;
  TYPE_THEN `u` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[subset_closure;ISUBSET ];
  (* // *)
  TYPE_THEN `A = UNIONS (IMAGE (closure top2) G)` ABBREV_TAC ;
  REWRITE_TAC[UNIONS;SUBSET ];
  CONV_TAC (dropq_conv "u");
  REWRITE_TAC[INR IN_SING];
  MESON_TAC[];
  ]);;
  (* }}} *)

(* logic *)
let not_not = prove_by_refinement(
  `!x y. (~x = ~y) <=> (x = y)`,
  (* {{{ proof *)
  [
  MESON_TAC[];
  ]);;
  (* }}} *)

let not_eq = prove_by_refinement(
  `!x y. (~x = y) <=> (x = ~y)`,
  (* {{{ proof *)
  [
  MESON_TAC[];
  ]);;
  (* }}} *)

let cell_inter = prove_by_refinement(
  `!C D. (cell C) /\ (D SUBSET cell) ==>
         ((C INTER (UNIONS D) = EMPTY) <=> ~(D C))`,
  (* {{{ proof *)

  [
  REWRITE_TAC[INTER;IN_UNIONS;SUBSET;EQ_EMPTY  ];
  DISCH_ALL_TAC;
  RIGHT_TAC  "x";
  REWRITE_TAC[not_not ];
  EQ_TAC;
  DISCH_THEN CHOOSE_TAC;
  AND 2;
  CHO 2;
  TYPE_THEN `t = C` SUBGOAL_TAC;
  IMATCH_MP_TAC  cell_partition;
  REWRITE_TAC[EMPTY_EXISTS;INTER ];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 0(MATCH_MP cell_nonempty);
  USE 0(REWRITE_RULE[EMPTY_EXISTS]);
  CHO 0;
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let curve_cell_h_inter = prove_by_refinement(
  `!G m. (segment G) ==>
     (((h_edge m) INTER (UNIONS (curve_cell G)) = {}) <=>
         (~(G (h_edge m))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  ASM_SIMP_TAC[GSYM curve_cell_h];
  IMATCH_MP_TAC  cell_inter;
  ASM_REWRITE_TAC [cell_rules;curve_cell_cell];
  ASM_MESON_TAC[segment;curve_cell_cell];
  ]);;
  (* }}} *)

let curve_cell_v_inter = prove_by_refinement(
  `!G m. (segment G) ==>
     (((v_edge m) INTER (UNIONS (curve_cell G)) = {}) <=>
         (~(G (v_edge m))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  ASM_SIMP_TAC[GSYM curve_cell_v];
  IMATCH_MP_TAC  cell_inter;
  ASM_REWRITE_TAC [cell_rules;curve_cell_cell];
  ASM_MESON_TAC[segment;curve_cell_cell];
  ]);;
  (* }}} *)

let curve_cell_squ = prove_by_refinement(
  `!G m. (segment G) ==> ~curve_cell G (squ m)`,
  (* {{{ proof *)
  [
    REWRITE_TAC[curve_cell;UNION ;eq_sing;square_pointI; segment];
  REWRITE_TAC[SUBSET; edge];
  DISCH_ALL_TAC;
  TSPEC `squ m` 3;
  USE 3(REWRITE_RULE[square_v_edgeV2;square_h_edgeV2;]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let curve_cell_squ_inter = prove_by_refinement(
  `!G m. (segment G) ==>
     (((squ m) INTER (UNIONS (curve_cell G)) = {}))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `cell (squ m)` SUBGOAL_TAC;
  REWRITE_TAC[cell_rules];
  DISCH_TAC;
  TYPE_THEN `(curve_cell G SUBSET cell)` SUBGOAL_TAC;
  ASM_MESON_TAC[curve_cell_cell;segment];
  DISCH_TAC;
  ASM_SIMP_TAC [cell_inter];
  ASM_MESON_TAC [curve_cell_squ];
  ]);;
  (* }}} *)

let curve_point_unions = prove_by_refinement(
  `!G m. (segment G) ==>
     (UNIONS (curve_cell G) (pointI m) = curve_cell G {(pointI m)})`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `UNIONS (curve_cell G) (pointI m) <=> ~({(pointI m)} INTER (UNIONS (curve_cell G)) = EMPTY )` SUBGOAL_TAC;
  REWRITE_TAC[REWRITE_RULE[not_eq] single_inter];
  DISCH_THEN_REWRITE;
  REWRITE_TAC [not_eq];
  IMATCH_MP_TAC  cell_inter;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  ASM_MESON_TAC[cell_rules;curve_cell_cell];
  ]);;
  (* }}} *)

let curve_cell_not_point = prove_by_refinement(
  `!G m. (segment G) ==> ((curve_cell G {(pointI m)} <=>
     ~(num_closure G (pointI m) = 0)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE G /\ (G SUBSET edge)` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  ASM_SIMP_TAC[curve_cell_point;num_closure0];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)

let par_cell = jordan_def `par_cell eps G C <=>
  ((?m. (C = {(pointI m)}) /\ (eps = EVEN (num_lower G m))) \/
         (?m. (C = h_edge m) /\ (eps = EVEN (num_lower G m))) \/
         (?m. (C = v_edge m) /\ (eps = EVEN (num_lower G m))) \/
         (?m. (C = squ m) /\ (eps= EVEN (num_lower G m)))) /\
   (C INTER (UNIONS (curve_cell G)) = EMPTY )`;;

let par_cell_curve_disj = prove_by_refinement(
  `!G C eps. (par_cell eps G C) ==>
          (C INTER (UNIONS (curve_cell G)) = EMPTY )`,
  (* {{{ proof *)
  [
 REWRITE_TAC[par_cell];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let par_cell_cell = prove_by_refinement(
  `!G eps.  (par_cell eps G SUBSET cell)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;par_cell;even_cell];
  DISCH_ALL_TAC;
  ASM_MESON_TAC[cell_rules];
  ]);;
  (* }}} *)

let par_cell_h = prove_by_refinement(
  `!G m eps. (segment G) ==> ((par_cell eps G (h_edge m) <=>
      (~(G (h_edge m))) /\ (eps = EVEN (num_lower G m))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[par_cell;eq_sing;h_edge_inj;hv_edgeV2;h_edge_pointI;];
  REWRITE_TAC[square_h_edgeV2];
  ASM_SIMP_TAC[curve_cell_h_inter];
  CONV_TAC (dropq_conv "m'");
  MESON_TAC[];
  ]);;
  (* }}} *)

let par_cell_v = prove_by_refinement(
  `!G m eps. (segment G) ==> ((par_cell eps G (v_edge m) <=>
      (~(G (v_edge m))) /\ (eps = EVEN (num_lower G m))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[par_cell;eq_sing;v_edge_inj;hv_edgeV2;v_edge_pointI;];
  REWRITE_TAC[square_v_edgeV2];
  ASM_SIMP_TAC[curve_cell_v_inter];
  CONV_TAC (dropq_conv "m'");
  MESON_TAC[];
  ]);;
  (* }}} *)

let par_cell_squ = prove_by_refinement(
  `!G m eps. (segment G) ==> ((par_cell eps G (squ m) <=>
       (eps = EVEN (num_lower G m))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[par_cell;eq_sing;square_h_edgeV2;square_v_edgeV2;squ_inj];
  ASM_SIMP_TAC[curve_cell_squ_inter];
  REWRITE_TAC[square_pointI];
  CONV_TAC (dropq_conv "m'");
  ]);;
  (* }}} *)

let par_cell_point = prove_by_refinement(
  `!G m eps. (segment G) ==> ((par_cell eps G {(pointI m)} <=>
      ((num_closure G (pointI m) = 0) /\
          (eps = EVEN (num_lower G m)))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[par_cell;eq_sing;INR IN_SING;point_inj;];
  SUBGOAL_TAC  `!u x. ({(pointI u)} = x) <=> (x = {(pointI u)})` ;
  ASM_MESON_TAC[];
  DISCH_THEN (fun t-> REWRITE_TAC[t]);
  REWRITE_TAC[eq_sing;INR IN_SING ;h_edge_pointI; v_edge_pointI; square_pointI;];
  REWRITE_TAC[pointI_inj; REWRITE_RULE[not_eq] single_inter];
  CONV_TAC (dropq_conv "m'");
  ASM_SIMP_TAC [curve_point_unions;curve_cell_not_point];
  MESON_TAC[];
  ]);;
  (* }}} *)

let eq_sing_sym = prove_by_refinement(
  `!X (y:A). ({y} = X) <=> X y /\ (!u. X u ==> (u = y))`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[eq_sing];
  ]);;
  (* }}} *)

let par_cell_disjoint = prove_by_refinement(
  `!G eps. (par_cell eps G INTER par_cell (~eps) G = EMPTY)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[EQ_EMPTY;INTER ];
  REP_GEN_TAC;
  REWRITE_TAC[par_cell];
  REPEAT (REPEAT (LEFT_TAC "m") THEN (GEN_TAC));
  REPEAT (LEFT_TAC "m");
  REPEAT (REPEAT (LEFT_TAC "m'") THEN  (GEN_TAC ));
  REPEAT (LEFT_TAC ("m'"));
  REPEAT (REPEAT (LEFT_TAC "m''") THEN  (GEN_TAC ));
  REPEAT (LEFT_TAC ("m''"));
  LEFT_TAC "m'''" THEN GEN_TAC;
  LEFT_TAC "m''''" THEN GEN_TAC;
  LEFT_TAC "m'''''" THEN GEN_TAC;
  REWRITE_TAC[RIGHT_AND_OVER_OR;LEFT_AND_OVER_OR];
  REWRITE_TAC[DE_MORGAN_THM];
  REPEAT (CONJ_TAC) THEN (REWRITE_TAC[GSYM DE_MORGAN_THM;GSYM CONJ_ASSOC]) THEN (REWRITE_TAC[TAUT `~(A /\ B) <=> (A ==> ~B)`]) THEN (DISCH_THEN_REWRITE ) THEN (REWRITE_TAC[eq_sing;eq_sing_sym;pointI_inj;h_edge_pointI;v_edge_pointI;square_pointI; INR IN_SING ; hv_edgeV2; h_edge_inj ; v_edge_inj; square_v_edgeV2;square_h_edgeV2;squ_inj ]) THEN (ASM_MESON_TAC[]);
  ]);;
  (* }}} *)

let par_cell_nonempty = prove_by_refinement(
  `!G eps. (rectagon G) ==> ~(par_cell eps G = EMPTY)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  COPY 1;
  USE 1 (MATCH_MP rectagon_h_edge);
  CHO 1;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon];
  DISCH_TAC ;
  USE 3(MATCH_MP squ_down);
  TSPEC `m` 3;
  USE 3 (REWRITE_RULE[set_lower_n]);
  UND 3;
  ASM_REWRITE_TAC[even_cell_squ;];
  PROOF_BY_CONTR_TAC;
  UND 0;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `segment G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon_segment];
  DISCH_TAC ;
  TYPE_THEN `eps = EVEN (num_lower G m)` ASM_CASES_TAC;
  TYPE_THEN `squ m` EXISTS_TAC;
  ASM_SIMP_TAC [par_cell_squ];
  TYPE_THEN `squ (down m)` EXISTS_TAC;
  ASM_SIMP_TAC[par_cell_squ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let par_cell_unions_nonempty = prove_by_refinement(
  `!G eps. (rectagon G) ==> ~(UNIONS (par_cell eps G) = EMPTY)`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[UNIONS;EMPTY_EXISTS ];
  NAME_CONFLICT_TAC;
  DISCH_TAC ;
  USE 0 (MATCH_MP par_cell_nonempty);
  TSPEC `eps` 0;
  USE 0 (REWRITE_RULE[EMPTY_EXISTS]);
  CHO 0;
 LEFT_TAC "u'";
  TYPE_THEN `u` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `cell u` SUBGOAL_TAC;
  ASM_MESON_TAC[par_cell_cell;ISUBSET ];
  DISCH_THEN (fun t-> MP_TAC (MATCH_MP cell_nonempty t));
  REWRITE_TAC[EMPTY_EXISTS];
  ]);;
  (* }}} *)

let ctop = jordan_def `ctop G =
   induced_top top2 (euclid 2 DIFF (UNIONS (curve_cell G)))`;;

let top2_unions = prove_by_refinement(
  `UNIONS (top2) = (euclid 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC [top2];
  ASM_MESON_TAC[top_of_metric_unions;metric_euclid];
  ]);;
  (* }}} *)

let curve_closed = prove_by_refinement(
  `!G. (segment G) ==> (closed_ top2 (UNIONS (curve_cell G)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  ASM_SIMP_TAC[GSYM curve_closure];
  IMATCH_MP_TAC  closure_closed;
  REWRITE_TAC[top2_top];
  IMATCH_MP_TAC  UNIONS_SUBSET;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  REWRITE_TAC[SUBSET;top2_unions;edge;  ];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  TSPEC `A` 1;
  REWR 1;
  CHO 1;
  ASM_MESON_TAC[REWRITE_RULE[SUBSET] h_edge_euclid;REWRITE_RULE[SUBSET] v_edge_euclid];
  ]);;
  (* }}} *)

let ctop_unions = prove_by_refinement(
  `!G. UNIONS (ctop G) = (euclid 2 DIFF (UNIONS (curve_cell G)))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[ctop];
  REWRITE_TAC[induced_top_support];
  REWRITE_TAC[top2_unions];
  REWRITE_TAC[INTER;DIFF;];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let par_cell_partition = prove_by_refinement(
  `!G eps. (segment G) ==>
  ((UNIONS (par_cell eps G) UNION (UNIONS (par_cell (~eps) G))) =
    (UNIONS (ctop G))) `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM ;
  CONJ_TAC;
  REWRITE_TAC[union_subset];
  TYPE_THEN `eps` (fun t-> SPEC_TAC (t,t));
  RIGHT_TAC "eps";
  SUBCONJ_TAC;
  GEN_TAC;
  IMATCH_MP_TAC  UNIONS_SUBSET;
  REWRITE_TAC[ctop_unions;DIFF_SUBSET ];
  DISCH_ALL_TAC;
  COPY 1;
  USE 2(MATCH_MP par_cell_curve_disj);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  cell_euclid;
  ASM_MESON_TAC[par_cell_cell ;ISUBSET ];
  DISCH_TAC ;
  GEN_TAC;
  TSPEC `~eps` 1;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[ctop_unions;SUBSET ;DIFF ; UNION ; UNIONS ];
  DISCH_ALL_TAC;
  USE 1(MATCH_MP point_onto);
  CHO 1;
  ASSUME_TAC cell_unions;
  TSPEC `p` 3;
  USE 3 (REWRITE_RULE[UNIONS]);
  CHO 3;
  USE 3 (REWRITE_RULE[cell]);
  AND 3;
  CHO 4;
  UND 4;
  REP_CASES_TAC;
  NAME_CONFLICT_TAC;
  ASM_REWRITE_TAC[];
  REWR 3;
  USE 3(REWRITE_RULE[INR IN_SING;pointI;point_inj ;]);
  ASM_REWRITE_TAC[GSYM pointI];
  LEFT_TAC "u'";
  TYPE_THEN `{(pointI p')}` EXISTS_TAC;
  ASM_SIMP_TAC[par_cell_point];
  REWRITE_TAC[INR IN_SING];
  LEFT 2 "u";
  TSPEC `{(pointI p')}` 2;
  REWR 2;
  USE 2(REWRITE_RULE[GSYM pointI;INR IN_SING ]);
  UND 2;
  ASM_SIMP_TAC [curve_cell_not_point];
  MESON_TAC[];
  (* case 2 *)
  LEFT_TAC "u";
  TYPE_THEN `h_edge p'` EXISTS_TAC ;
  ASM_SIMP_TAC [par_cell_h];
  LEFT 2 "u";
  REWR 3;
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `(G (h_edge p'))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC ;
  TSPEC `h_edge p'` 2;
  ASM_MESON_TAC[curve_cell_h];
  (* case 3 *)
  LEFT_TAC "u";
  TYPE_THEN `v_edge p'` EXISTS_TAC ;
  ASM_SIMP_TAC [par_cell_v];
  LEFT 2 "u";
  REWR 3;
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `(G (v_edge p'))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC ;
  TSPEC `v_edge p'` 2;
  ASM_MESON_TAC[curve_cell_v];
  (* case 4 *)
  LEFT_TAC "u";
  TYPE_THEN `squ p'` EXISTS_TAC ;
  ASM_SIMP_TAC [par_cell_squ];
  LEFT 2 "u";
  REWR 3;
  ASM_REWRITE_TAC[];
  MESON_TAC[];
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(*  openness of par_cell *)
(* ------------------------------------------------------------------ *)

let par_cell_h_squ = prove_by_refinement(
  `!G m eps. (segment G) /\ (par_cell eps G (h_edge m)) ==>
     (par_cell eps G (squ m) /\ par_cell eps G (squ (down m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  UND 1;
  ASM_SIMP_TAC [par_cell_h;par_cell_squ];
  DISCH_ALL_TAC;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC ;
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  ASM_SIMP_TAC[num_lower_down];
  ASM_MESON_TAC[set_lower_n];
  ]);;
  (* }}} *)

let par_cell_v_squ = prove_by_refinement(
  `!G m eps. (rectagon G) /\ (par_cell eps G (v_edge m)) ==>
     (par_cell eps G (squ m) /\ par_cell eps G (squ (left m)))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  UND 1;
  TYPE_THEN `segment G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon_segment];
  ASM_SIMP_TAC [par_cell_v;par_cell_squ];
  DISCH_ALL_TAC;
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  ASM_SIMP_TAC[REWRITE_RULE[even_cell_squ] squ_left_par];
  ]);;

  (* }}} *)

(* move up *)
let segment_finite = prove_by_refinement(
  `!G. (segment G) ==> (FINITE G)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[segment];
  ]);;
  (* }}} *)

let num_closure0_edge = prove_by_refinement(
  `!G m. (FINITE G) /\ (num_closure G (pointI m) = 0) ==>
    ~G (v_edge m) /\ ~G (v_edge (down m)) /\
          ~G (h_edge m) /\ ~G(h_edge (left  m))`,
  (* {{{ proof *)

  let rule = REWRITE_RULE[down;left ;h_edge_closure;hc_edge;v_edge_closure;vc_edge;UNION ;plus_e12; INR IN_SING ; INT_ARITH `x -: &:1 +: &:1 = x`] in
  [
  DISCH_ALL_TAC;
  UND 1;
  ASM_SIMP_TAC[num_closure0];
  DISCH_TAC;
  REWRITE_TAC[GSYM DE_MORGAN_THM];
  PURE_REWRITE_TAC [GSYM IMP_CLAUSES];
  REP_CASES_TAC;
  TSPEC `v_edge m` 1;
  JOIN 1 2;
  USE 1(rule);
  ASM_MESON_TAC[];
  TSPEC `v_edge (down m)` 1;
  JOIN 2 1;
  USE 1(rule);
  ASM_MESON_TAC[];
  TSPEC `h_edge ( m)` 1;
  JOIN 1 2;
  USE 1(rule);
  ASM_MESON_TAC[];
  TSPEC `h_edge (left  m)` 1;
  JOIN 1 2;
  USE 1(rule);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let par_cell_point_h = prove_by_refinement(
  `!G m eps. (rectagon G) /\ (par_cell eps G {(pointI m)}) ==>
     (par_cell eps G (h_edge m) /\ par_cell eps G (h_edge (left m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  UND 1;
  TYPE_THEN `segment G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon_segment];
  ASM_SIMP_TAC [par_cell_h;par_cell_point];
  DISCH_ALL_TAC;
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  ASM_SIMP_TAC[REWRITE_RULE[even_cell_squ] squ_left_par];
  UND 1;
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[segment_finite];
  ASM_MESON_TAC[num_closure0_edge];
  ]);;
  (* }}} *)

let par_cell_point_v = prove_by_refinement(
  `!G m eps. (rectagon G) /\ (par_cell eps G {(pointI m)}) ==>
     (par_cell eps G (v_edge m) /\ par_cell eps G (v_edge (down m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  UND 1;
  TYPE_THEN `segment G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon_segment];
  ASM_SIMP_TAC [par_cell_v;par_cell_point];
  DISCH_ALL_TAC;
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_MESON_TAC[segment_finite];
  ASM_SIMP_TAC[num_lower_down];
  REWRITE_TAC [set_lower_n];
  ASM_MESON_TAC[num_closure0_edge];
  ]);;
  (* }}} *)

let par_cell_point_rectangle = prove_by_refinement(
  `!G m eps. (rectagon G) /\ (par_cell eps G {(pointI m)}) ==>
     (rectangle (FST m -: &:1,SND m -: &:1) (FST m +: &:1,SND m +: &:1)
       SUBSET (UNIONS (par_cell eps G)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `segment G` SUBGOAL_TAC;
  ASM_SIMP_TAC[rectagon_segment];
  DISCH_TAC;
  REWRITE_TAC[two_two_union;union_subset];
  CONJ_TAC;
  TYPE_THEN `rectangle (FST m -: &:1,SND m -: &:1) (FST m,SND m +: &:1) = rectangle (FST (left  m),SND (left  m) -: &:1) (FST (left  m) +: &:1,SND (left  m) +: &:1)` SUBGOAL_TAC;
  REWRITE_TAC[left ;INT_ARITH ` x -: &:1 +: &:1 =x`];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[rectangle_h;union_subset ];
  TYPE_THEN `par_cell eps G (h_edge (left  m))` SUBGOAL_TAC;
  ASM_MESON_TAC[par_cell_point_h];
  ASM_MESON_TAC[sub_union;par_cell_h_squ];
  CONJ_TAC;
  REWRITE_TAC[long_v_union;union_subset;];
  ASM_MESON_TAC[sub_union; par_cell_point_v;];
  REWRITE_TAC[rectangle_h;union_subset ];
  TYPE_THEN `par_cell eps G (h_edge (  m))` SUBGOAL_TAC;
  ASM_MESON_TAC[par_cell_point_h];
  ASM_MESON_TAC[sub_union;par_cell_h_squ];
  ]);;
  (* }}} *)

let par_cell_h_rectangle = prove_by_refinement(
  `!G m eps. (rectagon G) /\ (par_cell eps G (h_edge m)) ==>
     (rectangle (FST m ,SND m -: &:1) (FST m +: &:1,SND m +: &:1)
       SUBSET (UNIONS (par_cell eps G)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `segment G` SUBGOAL_TAC;
  ASM_SIMP_TAC[rectagon_segment];
  DISCH_TAC;
  REWRITE_TAC[rectangle_h;union_subset ];
  ASM_MESON_TAC[sub_union;par_cell_h_squ];
  ]);;
  (* }}} *)

let par_cell_v_rectangle = prove_by_refinement(
  `!G m eps. (rectagon G) /\ (par_cell eps G (v_edge m)) ==>
     (rectangle (FST m -: &:1 ,SND m ) (FST m +: &:1,SND m +: &:1)
       SUBSET (UNIONS (par_cell eps G)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `segment G` SUBGOAL_TAC;
  ASM_SIMP_TAC[rectagon_segment];
  DISCH_TAC;
  REWRITE_TAC[rectangle_v;union_subset ];
  ASM_MESON_TAC[sub_union;par_cell_v_squ];
  ]);;
  (* }}} *)

let par_cell_squ_rectangle = prove_by_refinement(
  `!G m eps. (rectagon G) /\ (par_cell eps G (squ m)) ==>
     (rectangle (FST m  ,SND m ) (FST m +: &:1,SND m +: &:1)
       SUBSET (UNIONS (par_cell eps G)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[GSYM rectangle_squ];
  IMATCH_MP_TAC  sub_union;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let par_cell_point_in_rectangle = prove_by_refinement(
  `!m. (rectangle (FST m -: &:1,SND m -: &:1)
            (FST m +: &:1,SND m +: &:1) (pointI m))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[two_two_union;UNION ;long_v_union ; INR IN_SING ;];
  ]);;
  (* }}} *)

let par_cell_h_in_rectangle = prove_by_refinement(
  `!m. (h_edge m SUBSET
     (rectangle (FST m,SND m -: &:1) (FST m +: &:1,SND m +: &:1)))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[rectangle_h; UNION ; ISUBSET; INR IN_SING ;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let par_cell_v_in_rectangle = prove_by_refinement(
  `!m. (v_edge m SUBSET
     (rectangle (FST m -: &:1 ,SND m) (FST m +: &:1,SND m +: &:1)))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[rectangle_v; UNION ; ISUBSET; INR IN_SING ;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let ctop_top = prove_by_refinement(
  `!G. topology_ (ctop G)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[ctop];
  IMATCH_MP_TAC induced_top_top;
  REWRITE_TAC[top2_top];
  ]);;
  (* }}} *)

let ctop_open = prove_by_refinement(
  `!G B eps. (segment G) /\ (B SUBSET UNIONS (par_cell eps G)) /\
      (top2 B) ==> (ctop G B)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[ctop;induced_top;IMAGE];
  TYPE_THEN `B` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  REWRITE_TAC[GSYM SUBSET_INTER_ABSORPTION;GSYM ctop_unions];
  ASM_SIMP_TAC[GSYM par_cell_partition];
  REWRITE_TAC[UNION;ISUBSET ];
  ASM_MESON_TAC[ISUBSET];
  ]);;
  (* }}} *)

let par_cell_open = prove_by_refinement(
  `!G eps. (rectagon G) ==> (ctop G (UNIONS (par_cell eps G )))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `segment G` SUBGOAL_TAC;
  ASM_MESON_TAC[rectagon_segment];
  DISCH_TAC;
  ASSUME_TAC ctop_top;
  TSPEC `G` 2;
  USE 2(MATCH_MP open_nbd);
  UND 2;
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]) ;
  GEN_TAC;
  RIGHT_TAC "B";
  DISCH_TAC;
  USE 2(REWRITE_RULE[UNIONS]);
  CHO 2;
  TYPE_THEN `?p. (u = {(pointI p)}) \/ (u = h_edge p) \/ (u = v_edge p) \/ (u = squ p)` SUBGOAL_TAC;
  AND 2;
  USE 3 (MATCH_MP (REWRITE_RULE[ISUBSET ]par_cell_cell));
  USE 3(REWRITE_RULE[cell]);
  ASM_REWRITE_TAC[];
  DISCH_THEN (CHOOSE_THEN MP_TAC );
  ASSUME_TAC rectangle_open;
  REP_CASES_TAC ;
  (* 1st case *)
  REWR 2;
  USE 2(REWRITE_RULE[INR IN_SING]);
  ASM_REWRITE_TAC[];
  TYPE_THEN `rectangle (FST p -: &:1,SND p -: &:1) (FST p +: &:1,SND p +: &:1)` EXISTS_TAC;
  REWRITE_TAC[par_cell_point_in_rectangle];
  SUBCONJ_TAC;
  ASM_SIMP_TAC[par_cell_point_rectangle];
  ASM_MESON_TAC[ctop_open];
  (* 2nd case *)
  REWR 2;
  TYPE_THEN `rectangle (FST p,SND p -: &:1) (FST p +: &:1,SND p +: &:1)` EXISTS_TAC;
  ASM_SIMP_TAC [REWRITE_RULE[ISUBSET] par_cell_h_in_rectangle];
  SUBCONJ_TAC;
  ASM_SIMP_TAC[par_cell_h_rectangle];
  ASM_MESON_TAC[ctop_open];
  (* 3rd case *)
  REWR 2;
  TYPE_THEN `rectangle (FST p -: &:1,SND p ) (FST p +: &:1,SND p +: &:1)` EXISTS_TAC;
  ASM_SIMP_TAC [REWRITE_RULE[ISUBSET] par_cell_v_in_rectangle];
  SUBCONJ_TAC;
  ASM_SIMP_TAC[par_cell_v_rectangle];
  ASM_MESON_TAC[ctop_open];
  (* 4th case *)
  REWR 2;
  TYPE_THEN `rectangle (FST p,SND p ) (FST p +: &:1,SND p +: &:1)` EXISTS_TAC;
  ASSUME_TAC rectangle_squ;
  TSPEC `p` 5;
  SUBCONJ_TAC;
  ASM_SIMP_TAC[par_cell_squ_rectangle];
  DISCH_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[PAIR];
  ASM_MESON_TAC[ctop_open];
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* start on connected components of ctop G *)
(* ------------------------------------------------------------------ *)

(* move *)
let connected_empty = prove_by_refinement(
  `!(U:(A->bool)->bool). connected U EMPTY `,
  (* {{{ proof *)
  [
  REWRITE_TAC[connected];
  ]);;
  (* }}} *)

let par_cell_union_disjoint = prove_by_refinement(
  `!G eps. (UNIONS (par_cell eps G) INTER (UNIONS (par_cell (~eps) G)) =
              EMPTY )`,
  (* {{{ proof *)

  [
  REWRITE_TAC[INTER;EQ_EMPTY ;UNIONS;];
  DISCH_ALL_TAC;
  AND 0;
  CHO 0;
  CHO 1;
  TYPE_THEN `cell u /\ cell u'` SUBGOAL_TAC;
  ASM_MESON_TAC[par_cell_cell;ISUBSET];
  DISCH_TAC;
  TYPE_THEN `u = u'` SUBGOAL_TAC;
  IMATCH_MP_TAC  cell_partition;
  REWRITE_TAC[EMPTY_EXISTS;INTER ];
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASSUME_TAC par_cell_disjoint;
  USE 4(REWRITE_RULE[INTER;EQ_EMPTY]);
  TYPEL_THEN[`G`;`eps`;`u`] (USE 4 o ISPECL);
  USE 3 (GSYM);
  REWR 1;
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let par_cell_comp = prove_by_refinement(
  `!G eps x. (rectagon G) ==>
         (component  (ctop G) x SUBSET (UNIONS (par_cell eps G))) \/
            (component (ctop G) x SUBSET (UNIONS (par_cell (~eps) G)))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  TYPE_THEN `component  (ctop G) x SUBSET (UNIONS (ctop G))` SUBGOAL_TAC;
  REWRITE_TAC[component_DEF ;SUBSET ;connected ];
  MESON_TAC[];
  TYPE_THEN `segment G` SUBGOAL_TAC;
  ASM_MESON_TAC [rectagon_segment];
  DISCH_TAC;
  ASM_SIMP_TAC[GSYM par_cell_partition];
  DISCH_TAC;
  PROOF_BY_CONTR_TAC;
  USE 3 (REWRITE_RULE[DE_MORGAN_THM;SUBSET ]);
  AND 3;
  LEFT 3 "x'";
  CHO 3;
  LEFT 4 "x'";
  CHO 4;
  TYPE_THEN `component  (ctop G) x x'' /\ component  (ctop G) x x' ` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `component  (ctop G) x' x'' ` SUBGOAL_TAC;
  ASM_MESON_TAC[component_symm;component_trans];
  DISCH_TAC;
  USE 6(REWRITE_RULE[component_DEF]);
  CHO 6;
  USE 6(REWRITE_RULE[connected]);
  AND 6;
  AND 6;
  AND 7;
  TYPE_THEN `A = UNIONS (par_cell eps G)` ABBREV_TAC ;
  TYPE_THEN `B = UNIONS (par_cell (~eps) G)` ABBREV_TAC ;
  TYPEL_THEN [`A`;`B`] (USE 7 o ISPECL);
  UND 7;
  REWRITE_TAC[];
  TYPE_THEN `ctop G A /\ ctop G B` SUBGOAL_TAC;
  ASM_MESON_TAC[par_cell_open];
  DISCH_THEN_REWRITE;
  TYPE_THEN `Z SUBSET (A UNION B)` SUBGOAL_TAC;
  ASM_MESON_TAC[par_cell_partition];
  DISCH_THEN_REWRITE;
  TYPE_THEN `A INTER B = EMPTY` SUBGOAL_TAC;
  EXPAND_TAC "A";
  EXPAND_TAC "B";
  ASM_MESON_TAC[par_cell_union_disjoint;INTER_ACI;];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[ISUBSET];
  ]);;

  (* }}} *)

(* move *)
let connected_component = prove_by_refinement(
  `!U Z (x:A). (connected U Z) /\ (Z x) ==> (Z SUBSET (component U x)) `,
  (* {{{ proof *)
  [
  REWRITE_TAC[component_DEF  ;SUBSET ];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `Z` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let cont_mk_segment = prove_by_refinement(
  `!x y n. (euclid n x) /\ (euclid n y) ==>
    (continuous (joinf (\u. x)
        (joinf (\t. euclid_plus (t *# y) ((&1 - t) *# x)) (\u. y) (&.1))
          (&.0))
   (top_of_metric (UNIV,d_real)) (top_of_metric (euclid n,d_euclid)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  joinf_cont;
  CONJ_TAC;
  IMATCH_MP_TAC  const_continuous;
  IMATCH_MP_TAC  top_of_metric_top;
  REWRITE_TAC[metric_real];
  CONJ_TAC;
  IMATCH_MP_TAC  joinf_cont;
  CONJ_TAC;
  IMATCH_MP_TAC  continuous_lin_combo;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  const_continuous;
  IMATCH_MP_TAC  top_of_metric_top;
  REWRITE_TAC[metric_real];
  BETA_TAC;
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale_one; euclid_scale0; euclid_rzero ];
  REWRITE_TAC[joinf];
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale_one; euclid_scale0; euclid_lzero ];
  ]);;
  (* }}} *)

let mk_segment_image = prove_by_refinement(
  `!x y n. (euclid n x) /\ (euclid n y) ==> (?f.
     (continuous f
        (top_of_metric(UNIV,d_real))
        (top_of_metric (euclid n,d_euclid))) /\
     (IMAGE f {t | &.0 <=. t /\ t <=. &.1}  = mk_segment x y))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `(joinf (\u. x) (joinf (\t. euclid_plus (t *# y) ((&1 - t) *# x)) (\u. y) (&.1)) (&.0))` EXISTS_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  cont_mk_segment;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[joinf;IMAGE ];
  REWRITE_TAC[mk_segment];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  ASM_REWRITE_TAC[];
  EQ_TAC;
  DISCH_TAC;
  CHO 2;
  UND 2;
  COND_CASES_TAC;
  DISCH_ALL_TAC;
  JOIN 3 2;
  ASM_MESON_TAC[REAL_ARITH `~(&0 <=. x /\ x < &.0)`];
  DISCH_ALL_TAC;
  UND 5;
  COND_CASES_TAC;
  DISCH_TAC;
  TYPE_THEN `&1 - x''` EXISTS_TAC;
  SUBCONJ_TAC;
  UND 5;
  REAL_ARITH_TAC ;
  DISCH_TAC;
  CONJ_TAC;
  UND 3;
  REAL_ARITH_TAC ;
  ONCE_REWRITE_TAC [euclid_add_comm];
  REWRITE_TAC[REAL_ARITH `&1 - (&1 - x) = x`];
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `&0` EXISTS_TAC;
  CONJ_TAC;
  REAL_ARITH_TAC ;
  CONJ_TAC;
  REAL_ARITH_TAC ;
  REWRITE_TAC[euclid_scale_one; euclid_scale0; euclid_lzero;REAL_ARITH `&1 - &0 = &1` ];
  (* 2nd half *)
  DISCH_TAC;
  CHO 2;
  TYPE_THEN `&1 - a` EXISTS_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  AND 2;
  AND 2;
  UND 3;
  UND 4;
  REAL_ARITH_TAC ;
  COND_CASES_TAC;
  ASM_MESON_TAC[REAL_ARITH `&1 - a < &0 ==> ~(a <= &1)`];
  COND_CASES_TAC;
  REWRITE_TAC[REAL_ARITH `&1 - (&1 - a) = a`];
  ASM_MESON_TAC [euclid_add_comm];
  TYPE_THEN `a = &.0` SUBGOAL_TAC;
  UND 4;
  UND 3;
  AND 2;
  UND 3;
  REAL_ARITH_TAC ;
  DISCH_TAC;
  REWR 2;
  REWRITE_TAC [euclid_scale_one; euclid_scale0; euclid_lzero;REAL_ARITH `&1 - &0 = &1` ];
  ]);;
  (* }}} *)

let euclid_n_convex = prove_by_refinement(
  `!n. (convex (euclid n))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[convex;mk_segment;SUBSET ];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  CHO 2;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[euclid_add_closure;euclid_scale_closure];
  ]);;
  (* }}} *)

let connected_mk_segment = prove_by_refinement(
  `!x y n. (euclid n x) /\ (euclid n y) ==>
   (connected (top_of_metric(euclid n,d_euclid)) (mk_segment x y))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `?f. (continuous f    (top_of_metric(UNIV,d_real))  (top_of_metric (euclid n,d_euclid))) /\  (IMAGE f {t | &.0 <=. t /\ t <=. &.1}  = mk_segment x y)` SUBGOAL_TAC;
  IMATCH_MP_TAC  mk_segment_image;
  ASM_REWRITE_TAC[];
  DISCH_THEN CHOOSE_TAC;
  USE 2(GSYM);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  connect_image;
  TYPE_THEN `(top_of_metric (UNIV,d_real))` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  USE 2(GSYM);
  ASM_REWRITE_TAC[];
  TYPE_THEN `UNIONS (top_of_metric (euclid n,d_euclid) ) = (euclid n)` SUBGOAL_TAC;
  ASM_MESON_TAC [top_of_metric_unions;metric_euclid];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[convex;euclid_n_convex];
  MATCH_ACCEPT_TAC connect_real;
  ]);;
  (* }}} *)

let ctop_open = prove_by_refinement(
  `!G A. (top2 A /\ (A SUBSET (UNIONS (ctop G))) ==> ctop G A)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[ctop;induced_top;IMAGE ];
  TYPE_THEN `A` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ONCE_REWRITE_TAC[EQ_SYM_EQ];
  REWRITE_TAC[GSYM SUBSET_INTER_ABSORPTION];
  REWRITE_TAC[GSYM ctop_unions];
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let ctop_top2 = prove_by_refinement(
  `!G A. (segment G /\ ctop G A ==> top2 A)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[ctop;induced_top;IMAGE ;];
  DISCH_ALL_TAC;
  TYPE_THEN `U = top_of_metric(euclid 2,d_euclid)` ABBREV_TAC ;
  TYPE_THEN `euclid 2 = UNIONS U` SUBGOAL_TAC;
  EXPAND_TAC "U";
  ASM_MESON_TAC[top_of_metric_unions;metric_euclid];
  CHO 1;
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  top_inter;
  ASM_REWRITE_TAC[top2_top;];
  ASM_SIMP_TAC[GSYM curve_closure;top2];
  IMATCH_MP_TAC  (REWRITE_RULE[open_DEF] closed_open);
  IMATCH_MP_TAC  closure_closed;
  CONJ_TAC;
  EXPAND_TAC "U";
  ASM_MESON_TAC[top_of_metric_top;metric_euclid];
  USE 3(GSYM);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  UNIONS_SUBSET;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  REWRITE_TAC[edge;ISUBSET;];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  TSPEC `A'` 4;
  REWR 4;
  CHO 4;
  UND 4;
  DISCH_THEN DISJ_CASES_TAC THEN ASM_REWRITE_TAC[] ;
  MATCH_ACCEPT_TAC (REWRITE_RULE[ISUBSET;] v_edge_euclid);
  MATCH_ACCEPT_TAC (REWRITE_RULE[ISUBSET;] h_edge_euclid);
  ]);;
  (* }}} *)

let mk_segment_sym_lemma = prove_by_refinement(
  `!x y z. (mk_segment x y z ==> mk_segment y x z)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[mk_segment];
  DISCH_ALL_TAC;
  CHO 0;
  TYPE_THEN `&1 - a` EXISTS_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[REAL_ARITH `a <= &1 ==> &0 <= &1 - a`];
  CONJ_TAC;
  ASM_MESON_TAC[REAL_ARITH `&0 <= a ==> &1 - a <= &1`];
  ONCE_REWRITE_TAC[euclid_add_comm];
  ASM_REWRITE_TAC[REAL_ARITH `&1 - (&1 - a) = a`];
  ]);;
  (* }}} *)

let mk_segment_sym = prove_by_refinement(
  `!x y. (mk_segment x y = mk_segment y x)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  EQ_TAC THEN ASM_MESON_TAC[mk_segment_sym_lemma];
  ]);;
  (* }}} *)

let mk_segment_end = prove_by_refinement(
  `!x y. (mk_segment x y x /\ mk_segment x y y)`,
  (* {{{ proof *)
  [
  RIGHT_TAC "y";
  RIGHT_TAC "x";
  SUBCONJ_TAC;
  DISCH_ALL_TAC;
  REWRITE_TAC[mk_segment];
  TYPE_THEN `&1` EXISTS_TAC;
  REDUCE_TAC;
  CONJ_TAC;
  ARITH_TAC;
  REWRITE_TAC[euclid_scale_one;euclid_scale0;euclid_rzero];
  DISCH_TAC;
  ONCE_REWRITE_TAC[mk_segment_sym];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let convex_connected = prove_by_refinement(
  `!G Z. (segment G /\ convex Z) /\ (Z SUBSET (UNIONS (ctop G))) ==>
            (connected (ctop G) Z)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[connected];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  PROOF_BY_CONTR_TAC;
  USE 7 (REWRITE_RULE[DE_MORGAN_THM;SUBSET ]);
  AND 7;
  LEFT 7 "x";
  CHO 7;
  LEFT 8 "x";
  CHO 8;
  TYPE_THEN `Z x /\ Z x'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `mk_segment x x' SUBSET A UNION B` SUBGOAL_TAC;
  USE 1(REWRITE_RULE[convex]);
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  TYPE_THEN `connected (top_of_metric(euclid 2,d_euclid)) (mk_segment x x')` SUBGOAL_TAC;
  IMATCH_MP_TAC  connected_mk_segment;
  USE 2(REWRITE_RULE[ctop_unions;SUBSET;DIFF;]);
  ASM_MESON_TAC[];
  REWRITE_TAC[connected];
  DISCH_ALL_TAC;
  AND 11;
  TYPEL_THEN [`A`;`B`] (USE 11 o ISPECL);
  REWR 11;
  TYPE_THEN `top_of_metric (euclid 2,d_euclid) A /\ top_of_metric (euclid 2,d_euclid) B` SUBGOAL_TAC;
  REWRITE_TAC[GSYM top2];
  ASM_MESON_TAC[ctop_top2;top2];
  DISCH_TAC;
  UND 11;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[DE_MORGAN_THM;ISUBSET;];
  CONJ_TAC;
  LEFT_TAC "x''";
  TYPE_THEN `x'` EXISTS_TAC;
  REWRITE_TAC[mk_segment_end];
  ASM_MESON_TAC[];
  LEFT_TAC "x''";
  TYPE_THEN `x` EXISTS_TAC;
  REWRITE_TAC[mk_segment_end];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let component_replace = prove_by_refinement(
  `!U (x:A) y. component  U x y ==> (component  U x = component  U y)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  DISCH_ALL_TAC;
  EQ_TAC;
  DISCH_ALL_TAC;
  USE 0(MATCH_MP component_symm);
  ASM_MESON_TAC[component_trans];
  ASM_MESON_TAC[component_trans;component_symm];
  ]);;

  (* }}} *)

let convex_component = prove_by_refinement(
  `!G Z x. (segment G /\ convex Z /\ (Z SUBSET (UNIONS (ctop G))) /\
     (~(Z INTER (component  (ctop G) x ) = EMPTY))  ==>
        (Z SUBSET (component  (ctop G) x)))  `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `connected (ctop G) Z` SUBGOAL_TAC;
  ASM_SIMP_TAC[convex_connected];
  DISCH_TAC;
  USE 3(REWRITE_RULE[EMPTY_EXISTS;INTER ]);
  CHO 3;
  AND 3;
  USE 3(MATCH_MP component_replace);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  connected_component;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let cell_convex = prove_by_refinement(
  `!C.  (cell C) ==> (convex C)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[cell];
  GEN_TAC;
  DISCH_THEN (CHOOSE_THEN MP_TAC ) THEN REP_CASES_TAC THEN ASM_REWRITE_TAC[v_edge_convex;h_edge_convex;convex_pointI;rectangle_squ;rectangle_convex];

  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)

let cell_of = jordan_def `cell_of C = { A | (cell A) /\ (A SUBSET C) }`;;

let unions_cell_of = prove_by_refinement(
  `!G x. (segment G ==>
     (UNIONS (cell_of (component  (ctop G) x)) =
           component  (ctop G) x))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  REWRITE_TAC[UNIONS;SUBSET;cell_of];
  CONJ_TAC;
  DISCH_ALL_TAC;
  CHO 1;
  AND 1;
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  TYPE_THEN `(euclid 2 x')` SUBGOAL_TAC;
  UND 1;
  REWRITE_TAC[component_DEF   ;connected;SUBSET ;ctop_unions;DIFF ];
  DISCH_THEN CHOOSE_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 2 (MATCH_MP point_onto);
  CHO 2;
  REWR 1;
  ASM_REWRITE_TAC[];
  ASSUME_TAC cell_unions;
  TSPEC `p` 3;
  USE 3 (REWRITE_RULE[UNIONS]);
  CHO 3;
  TYPE_THEN `u` EXISTS_TAC;
  TYPE_THEN `u SUBSET (component  (ctop G) x) ==> (!x'. u x' ==> component  (ctop G) x x')` SUBGOAL_TAC;
  REWRITE_TAC[ISUBSET];
  ASM_REWRITE_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  IMATCH_MP_TAC  convex_component ;
  ASM_REWRITE_TAC[EMPTY_EXISTS];
  CONJ_TAC;
  ASM_MESON_TAC[cell_convex];
  CONJ_TAC;
  REWRITE_TAC[ctop_unions];
  REWRITE_TAC[DIFF;SUBSET ];
  DISCH_ALL_TAC;
  CONJ_TAC;
  AND 3;
  UND 5;
  UND 4;
  ASM_MESON_TAC[cell_euclid;ISUBSET];
  REWRITE_TAC[UNIONS];
  LEFT_TAC  "u";
  GEN_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  USE 6 (MATCH_MP   curve_cell_cell);
  USE 6 (REWRITE_RULE[ISUBSET]);
  TSPEC `u'` 6;
  REWR 6;
  TYPE_THEN `u = u'` SUBGOAL_TAC;
  IMATCH_MP_TAC  cell_partition;
  REWRITE_TAC[EMPTY_EXISTS;INTER];
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 1 (REWRITE_RULE[component_DEF;connected;SUBSET ]);
  TYPE_THEN `UNIONS (ctop G) (point p)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REWRITE_TAC[ctop_unions;DIFF ;UNIONS ;DE_MORGAN_THM ];
  DISJ2_TAC ;
  ASM_MESON_TAC[];
  NAME_CONFLICT_TAC;
  TYPE_THEN `point p` EXISTS_TAC;
  ASM_REWRITE_TAC [INTER];
  ]);;
  (* }}} *)




(* ------------------------------------------------------------------ *)
(* SECTION F *)
(* ------------------------------------------------------------------ *)

(* ------------------------------------------------------------------ *)
(* num_abs_of_int *)
(* ------------------------------------------------------------------ *)

let num_abs_of_int_exists = prove_by_refinement(
  `!m. ?i. &i = abs  (real_of_int(m))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[GSYM int_abs_th];
  ASSUME_TAC dest_int_rep;
  TSPEC `||: m` 0;
  CHO 0;
  TYPE_THEN `n` EXISTS_TAC;
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  WITH 0 (REWRITE_RULE[int_abs_th]);
  TYPE_THEN `&0 <= abs  (real_of_int m)` SUBGOAL_TAC;
  REWRITE_TAC[REAL_ABS_POS];
  TYPE_THEN `abs  (real_of_int m) <= &.0` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  REDUCE_TAC ;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC ;
  ]);;
  (* }}} *)

let num_abs_of_int_select = new_definition
     `num_abs_of_int m = @i. (&i = abs  (real_of_int m))`;;

let num_abs_of_int_th = prove_by_refinement(
  `!m. &(num_abs_of_int m) = abs  (real_of_int m)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[num_abs_of_int_select];
  SELECT_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[num_abs_of_int_exists];
  ]);;
  (* }}} *)

let num_abs_of_int_mul = prove_by_refinement(
  `!m n. (num_abs_of_int (m * n) = num_abs_of_int m * num_abs_of_int n)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM REAL_OF_NUM_EQ;GSYM REAL_MUL;num_abs_of_int_th;int_mul_th;ABS_MUL;];
  ]);;
  (* }}} *)

let num_abs_of_int_num = prove_by_refinement(
  `!n. (num_abs_of_int (&: n) = n)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM REAL_OF_NUM_EQ;num_abs_of_int_th;int_of_num_th;REAL_ABS_NUM;];
  ]);;
  (* }}} *)

let num_abs_of_int_triangle = prove_by_refinement(
  `!n m. num_abs_of_int (m + n) <=|
           num_abs_of_int(m) +| num_abs_of_int n`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[GSYM REAL_OF_NUM_LE;num_abs_of_int_th;int_add_th;GSYM REAL_OF_NUM_ADD;ABS_TRIANGLE;];
  ]);;
  (* }}} *)

let num_abs_of_int0 = prove_by_refinement(
  `!m. (num_abs_of_int m = 0) <=> (m = &:0)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[GSYM REAL_OF_NUM_EQ;num_abs_of_int_th;REAL_ABS_ZERO;];
  REWRITE_TAC[int_eq;];
  REWRITE_TAC[int_of_num_th;];
  ]);;
  (* }}} *)

let num_abs_of_int_neg = prove_by_refinement(
  `!m. (num_abs_of_int (--: m) = num_abs_of_int m)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[GSYM REAL_OF_NUM_EQ;num_abs_of_int_th;int_neg_th;REAL_ABS_NEG;];
  ]);;
  (* }}} *)

let num_abs_of_int_suc = prove_by_refinement(
  `!m. (&:0 <=: m) ==>
     (SUC (num_abs_of_int m) = num_abs_of_int (m +: &:1))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[int_le;int_of_num_th;];
  DISCH_ALL_TAC;
  REWRITE_TAC[GSYM REAL_OF_NUM_EQ;num_abs_of_int_th;ADD1;GSYM REAL_ADD;int_suc];
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let num_abs_of_int_pre = prove_by_refinement(
  `!m. (m <=: &:0) ==>
     (SUC (num_abs_of_int m) = num_abs_of_int (m -: &:1))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[int_le;int_of_num_th;];
  DISCH_ALL_TAC;
  REWRITE_TAC[GSYM REAL_OF_NUM_EQ;num_abs_of_int_th;ADD1;GSYM REAL_ADD;int_suc;int_sub_th;int_of_num_th;];
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* closure of squares *)
(* ------------------------------------------------------------------ *)

let right_left = prove_by_refinement(
  `!m. (right  (left  m) = m) /\ (left  (right  m) = m) /\
    (up (down m) = m) /\ (down (up m) = m) /\
    (up (right  m) = right  (up m)) /\ (up (left  m) = left  (up m)) /\
    (down (right  m) = right  (down m)) /\
    (down (left  m) = (left  (down m)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[right ;left ;up;down;PAIR_SPLIT];
  INT_ARITH_TAC;
  ]);;
  (* }}} *)

let squc = jordan_def `squc p = {Z | ?u v.
                  (Z = point (u,v)) /\
                  real_of_int (FST p) <= u /\
                  u <= real_of_int (FST p +: &:1) /\
                  real_of_int (SND p) <= v /\
                  v <= real_of_int (SND p +: &:1)}`;;

let squc_inter = prove_by_refinement(
  `!p. squc p =
   {z | ?r. (z = point r) /\ real_of_int (FST p) <= FST r} INTER
         {z | ?r. (z = point r) /\ real_of_int (SND p) <= SND r} INTER
         {z | ?r. (z = point r) /\ FST r <= real_of_int (FST p +: &:1)} INTER
         {z | ?r. (z = point r) /\ SND r <= real_of_int (SND p +: &:1)}`,
  (* {{{ proof *)

  [
  REWRITE_TAC[squc];
  GEN_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INTER];
  EQ_TAC;
  DISCH_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[point_inj;];
  CONV_TAC (dropq_conv "r");
  ASM_REWRITE_TAC[];
  CONV_TAC (dropq_conv "r");
  ASM_REWRITE_TAC[];
  CONV_TAC (dropq_conv "r'");
  ASM_REWRITE_TAC[];
  CONV_TAC (dropq_conv "r");
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  CHO 0;
  AND 0;
  REWR 1;
  REWRITE_TAC[point_inj;PAIR_SPLIT ;];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  USE 1 (REWRITE_RULE[point_inj;]);
  USE 1 (CONV_RULE (dropq_conv "r'"));
  REWR 2;
  USE 2 (REWRITE_RULE[point_inj;]);
  USE 2 (CONV_RULE (dropq_conv "r'"));
  REWR 3;
  USE 3 (REWRITE_RULE[point_inj;]);
  USE 3 (CONV_RULE (dropq_conv "r'"));
  ASM_REWRITE_TAC[];
  ]);;

  (* }}} *)

let squc_closed = prove_by_refinement(
  `!p. closed_ (top2) (squc p)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  ASSUME_TAC top2_top;
  REWRITE_TAC[squc_inter];
  ASM_SIMP_TAC[closed_inter2;closed_half_plane2D_LTS_closed;closed_half_plane2D_SLT_closed;closed_half_plane2D_LTF_closed;closed_half_plane2D_FLT_closed];
  ]);;
  (* }}} *)

let squ_subset_sqc = prove_by_refinement(
  `!p. (squ p SUBSET (squc p))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[SUBSET;squ;squc];
  GEN_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  TYPE_THEN `u` EXISTS_TAC;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_MESON_TAC[REAL_ARITH `x < y ==> x <=. y`];
  ]);;
  (* }}} *)

let squc_union_lemma1 = prove_by_refinement(
  `!p. squc p INTER
     {z | ?r. (z = point r) /\ (real_of_int(FST p) = FST r)} =
   {(pointI p)} UNION (v_edge p) UNION {(pointI (up p))}`,
  (* {{{ proof *)
  [
  GEN_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[squc;UNION ;INR IN_SING ;INTER ;up; int_of_num_th; int_add_th;];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  REWR 1;
  USE 1(REWRITE_RULE[point_inj]);
  USE 1(CONV_RULE (dropq_conv "r"));
  UND 0;
  DISCH_ALL_TAC;
  UND 4;
  UND 5;
  REWRITE_TAC[REAL_ARITH `(x <=y) <=> (y = x) \/ (x <. y)`];
  KILL 2;
  KILL 3;
  KILL 0;
  USE 1 (GSYM);
  ASM_REWRITE_TAC[];
  KILL 0;
  REP_CASES_TAC;
  ASM_MESON_TAC[REAL_ARITH `~(v = v + &.1)`];
  EXPAND_TAC "v";
  REWRITE_TAC[pointI;int_suc;];
  ASM_REWRITE_TAC[pointI];
  REWRITE_TAC[v_edge];
  DISJ2_TAC ;
  DISJ1_TAC ;
  REWRITE_TAC[point_inj; PAIR_SPLIT];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v'");
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[int_suc];
  REP_CASES_TAC;
  ASM_REWRITE_TAC[pointI;point_inj;];
  CONJ_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  REAL_ARITH_TAC ;
  CONV_TAC (dropq_conv "r");
  USE 0 (REWRITE_RULE[v_edge]);
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[point_inj];
  CONJ_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v'");
  AND  0;
  UND 0;
  REWRITE_TAC[int_suc];
  REAL_ARITH_TAC ;
  CONV_TAC (dropq_conv "r");
  (* LAST *)
  ASM_REWRITE_TAC[pointI;point_inj;];
  CONJ_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  REWRITE_TAC[int_suc];
  REAL_ARITH_TAC ;
  CONV_TAC (dropq_conv "r");
  ]);;
  (* }}} *)

let squc_union_lemma2 = prove_by_refinement(
  `!p. squc p INTER
     {z | ?r. (z = point r) /\ (real_of_int(FST p) + &1= FST r )} =
   {(pointI (right  p))} UNION (v_edge (right  p)) UNION
     {(pointI (up (right  p)))}`,
  (* {{{ proof *)
  [
  GEN_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[squc;right  ;UNION ;INR IN_SING ;INTER ;up; int_of_num_th; int_add_th;];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  REWR 1;
  USE 1(REWRITE_RULE[point_inj]);
  USE 1(CONV_RULE (dropq_conv "r"));
  UND 0;
  DISCH_ALL_TAC;
  UND 4;
  UND 5;
  REWRITE_TAC[REAL_ARITH `(x <=y) <=> (y = x) \/ (x <. y)`];
  KILL 2;
  KILL 3;
  KILL 0;
  USE 1 (GSYM);
  ASM_REWRITE_TAC[];
  KILL 0;
  REP_CASES_TAC;
  ASM_MESON_TAC[REAL_ARITH `~(v = v + &.1)`];
  EXPAND_TAC "v";
  REWRITE_TAC[pointI;int_suc;];
  (* 3 LEFT *)
  ASM_REWRITE_TAC[pointI;int_suc;];
  (* 2 LEFT *)
  REWRITE_TAC[v_edge];
  DISJ2_TAC ;
  DISJ1_TAC ;
  REWRITE_TAC[point_inj; PAIR_SPLIT];
  CONV_TAC (dropq_conv "u");
  REWRITE_TAC[int_suc];
  CONV_TAC (dropq_conv "v'");
  ASM_REWRITE_TAC[];
  (* second half  *)
  ASM_REWRITE_TAC[int_suc];
  REP_CASES_TAC;
  ASM_REWRITE_TAC[pointI;point_inj;];
  CONJ_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  ASM_REWRITE_TAC[int_suc];
  REAL_ARITH_TAC ;
  CONV_TAC (dropq_conv "r");
  REWRITE_TAC[int_suc];
  (* 2 LEFT *)
  USE 0 (REWRITE_RULE[v_edge]);
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[point_inj];
  CONJ_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v'");
  AND  0;
  UND 0;
  REWRITE_TAC[int_suc];
  REAL_ARITH_TAC ;
  CONV_TAC (dropq_conv "r");
  REWRITE_TAC[int_suc];
  (* LAST *)
  ASM_REWRITE_TAC[pointI;point_inj;];
  CONJ_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  REWRITE_TAC[int_suc];
  REAL_ARITH_TAC ;
  CONV_TAC (dropq_conv "r");
  REWRITE_TAC[int_suc];
  ]);;
  (* }}} *)

let squc_union_lemma3 = prove_by_refinement(
  `!p. squc p INTER
    {z | ?r. (z = point r) /\ (FST r <. real_of_int(FST p) + &1 ) /\
       (real_of_int(FST p) <. FST r) } =
    (h_edge p) UNION squ p UNION (h_edge (up p))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INTER;squc;UNION;];
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  REWR 1;
  USE 1 (REWRITE_RULE[point_inj]);
  USE 1 (CONV_RULE (dropq_conv "r"));
  AND 0;
  UND 0;
  DISCH_ALL_TAC;
  KILL  0;
  KILL  3;
  UND 4;
  UND 5;
  REWRITE_TAC[REAL_ARITH `(x <= y) <=> (y = x) \/ (x <. y)`;int_suc];
  REP_CASES_TAC;
  ASM_MESON_TAC[REAL_ARITH `~(v = v + &1)`];
  EXPAND_TAC "v";
  REWRITE_TAC[up;h_edge];
  DISJ2_TAC;
  DISJ2_TAC;
  REWRITE_TAC[point_inj;];
  REWRITE_TAC[PAIR_SPLIT];
  CONV_TAC (dropq_conv "u'");
  CONV_TAC (dropq_conv "v");
  ASM_REWRITE_TAC[int_suc];
  (* 3 to go *)
  ASM_REWRITE_TAC[];
  DISJ1_TAC;
  REWRITE_TAC[h_edge;point_inj;PAIR_SPLIT];
  CONV_TAC (dropq_conv "u'");
  CONV_TAC (dropq_conv "v");
  ASM_REWRITE_TAC[int_suc];
  (* 2 to go *)
  DISJ2_TAC;
  DISJ1_TAC;
  REWRITE_TAC[squ;point_inj;PAIR_SPLIT];
  CONV_TAC (dropq_conv "u'");
  CONV_TAC (dropq_conv "v'");
  ASM_REWRITE_TAC[int_suc];
  (* 2nd half *)
  DISCH_TAC;
  TYPE_THEN `?q. x = point q` ASM_CASES_TAC;
  CHO 1;
  ASM_REWRITE_TAC[point_inj];
  CONJ_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  REWR 0;
  UND 0;
  REWRITE_TAC[h_edge;squ;up;int_suc ;point_inj; PAIR_SPLIT ;];
  REP_CASES_TAC;
  USE 0 (CONV_RULE (dropq_conv "u"));
  USE 0 (CONV_RULE (dropq_conv "v"));
  UND 0;
  REAL_ARITH_TAC ;
  USE 0 (CONV_RULE (dropq_conv "u"));
  USE 0 (CONV_RULE (dropq_conv "v"));
  UND 0;
  REAL_ARITH_TAC ;
  USE 0 (CONV_RULE (dropq_conv "u"));
  USE 0 (CONV_RULE (dropq_conv "v"));
  UND 0;
  REAL_ARITH_TAC ;
  CONV_TAC (dropq_conv "r");
  REWR 0;
  UND 0;
  REWRITE_TAC[h_edge;squ;up;int_suc ;point_inj; PAIR_SPLIT ;];
  REP_CASES_TAC;
  USE 0 (CONV_RULE (dropq_conv "u"));
  USE 0 (CONV_RULE (dropq_conv "v"));
  UND 0;
  REAL_ARITH_TAC ;
  USE 0 (CONV_RULE (dropq_conv "u"));
  USE 0 (CONV_RULE (dropq_conv "v"));
  UND 0;
  REAL_ARITH_TAC ;
  USE 0 (CONV_RULE (dropq_conv "u"));
  USE 0 (CONV_RULE (dropq_conv "v"));
  UND 0;
  REAL_ARITH_TAC ;
  (* 1 goal LEFT *)
  PROOF_BY_CONTR_TAC;
  KILL 2;
  UND 1;
  REWRITE_TAC[];
  IMATCH_MP_TAC  point_onto;
  ASM_MESON_TAC[h_edge_euclid;squ_euclid;v_edge_euclid;ISUBSET ];
  ]);;
  (* }}} *)

let squc_lemma4 = prove_by_refinement(
  `!p. squc p SUBSET
    {z | ?r. (z = point r) /\ (real_of_int(FST p) = FST r)} UNION
     {z | ?r. (z = point r) /\ (real_of_int(FST p) + &1= FST r )} UNION
      {z | ?r. (z = point r) /\ (FST r <. real_of_int(FST p) + &1 ) /\
       (real_of_int(FST p) <. FST r) } `,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;UNION ;squc ];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[point_inj ;];
  LEFT_TAC "r";
  CONV_TAC (dropq_conv "r");
  UND 0;
  DISCH_ALL_TAC;
  UND 1;
  UND 2;
  ASM_REWRITE_TAC[int_suc];
  REAL_ARITH_TAC ;
  ]);;
  (* }}} *)

let squc_union = prove_by_refinement(
  `!p. squc p = {(pointI p)} UNION {(pointI (right  p))} UNION
       {(pointI (up p))} UNION {(pointI (up (right   p)))} UNION
       (h_edge p) UNION (h_edge (up p)) UNION
       (v_edge p) UNION (v_edge (right  p)) UNION
       (squ p)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  TYPE_THEN `squc p = squc p  INTER ({z | ?r. (z = point r) /\ (real_of_int(FST p) = FST r)} UNION   {z | ?r. (z = point r) /\ (real_of_int(FST p) + &1= FST r )} UNION   {z | ?r. (z = point r) /\ (FST r <. real_of_int(FST p) + &1 ) /\  (real_of_int(FST p) <. FST r) } )` SUBGOAL_TAC;
  ONCE_REWRITE_TAC[EQ_SYM_EQ];
  REWRITE_TAC  [GSYM SUBSET_INTER_ABSORPTION];
  MATCH_ACCEPT_TAC squc_lemma4;
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]);
  REWRITE_TAC[UNION_OVER_INTER];
  REWRITE_TAC[squc_union_lemma1;squc_union_lemma2;squc_union_lemma3];
  REWRITE_TAC[UNION_ACI];
  ]);;
  (* }}} *)

let squ_closure_h = prove_by_refinement(
  `!p. (h_edge p) SUBSET (closure top2 (squ p))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[top2];
  IMATCH_MP_TAC  closure_segment;
  ASM_REWRITE_TAC[squ_euclid];
  TYPE_THEN `?q. (x = point q)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  point_onto;
  ASM_MESON_TAC[REWRITE_RULE[ISUBSET] h_edge_euclid];
  DISCH_TAC;
  CHO 1;
  REWR 0;
  KILL 1;
  TYPE_THEN `point (FST q, SND q + &1)` EXISTS_TAC;
  REWRITE_TAC[point_scale;point_add;];
  UND 0;
  TYPE_THEN `point q = point (FST q,SND q)` SUBGOAL_TAC;
  REWRITE_TAC[];
  DISCH_THEN (fun t-> PURE_REWRITE_TAC [t]);
  PURE_REWRITE_TAC[point_add;point_scale];
  REWRITE_TAC[h_edge;squ;point_inj;PAIR_SPLIT;];
  DISCH_ALL_TAC;
  USE 0 (CONV_RULE (dropq_conv "u"));
  USE 0 (CONV_RULE (dropq_conv "v"));
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  UND 0;
  REWRITE_TAC[int_suc];
  ASSUME_TAC (real_poly_conv `t *x + (&1 - t)* x`);
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  ASSUME_TAC (real_poly_conv `t *(y + &1) + (&1- t)* y`);
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  UND 1;
  UND 2;
  REDUCE_TAC ;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let squ_closure_up_h = prove_by_refinement(
  `!p. (h_edge (up   p)) SUBSET (closure top2 (squ p))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;up  ];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[top2];
  IMATCH_MP_TAC  closure_segment;
  ASM_REWRITE_TAC[squ_euclid];
  TYPE_THEN `?q. (x = point q)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  point_onto;
  ASM_MESON_TAC[REWRITE_RULE[ISUBSET] h_edge_euclid];
  DISCH_TAC;
  CHO 1;
  REWR 0;
  KILL 1;
  TYPE_THEN `point (FST q , SND q - &1)` EXISTS_TAC;
  REWRITE_TAC[point_scale;point_add;];
  UND 0;
  TYPE_THEN `point q = point (FST q,SND q)` SUBGOAL_TAC;
  REWRITE_TAC[];
  DISCH_THEN (fun t-> PURE_REWRITE_TAC [t]);
  PURE_REWRITE_TAC[point_add;point_scale];
  REWRITE_TAC[h_edge;squ;point_inj;PAIR_SPLIT;];
  DISCH_ALL_TAC;
  USE 0 (CONV_RULE (dropq_conv "u"));
  USE 0 (CONV_RULE (dropq_conv "v"));
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  UND 0;
  REWRITE_TAC[int_suc];
  ASSUME_TAC (real_poly_conv `t *x + (&1 - t)* x`);
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  ASSUME_TAC (real_poly_conv `t *(y - &1) + (&1- t)* y`);
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  UND 1;
  UND 2;
  REDUCE_TAC ;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let squ_closure_down_h = prove_by_refinement(
  `!p. (h_edge p SUBSET (closure top2 (squ (down p))))`,
  (* {{{ proof *)

  [
  GEN_TAC;
  ASSUME_TAC squ_closure_up_h ;
  TSPEC `down p` 0;
  USE 0 (REWRITE_RULE [right_left]);
  ASM_REWRITE_TAC[];
  ]);;

  (* }}} *)

let squ_closure_v = prove_by_refinement(
  `!p. (v_edge p) SUBSET (closure top2 (squ p))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[top2];
  IMATCH_MP_TAC  closure_segment;
  ASM_REWRITE_TAC[squ_euclid];
  TYPE_THEN `?q. (x = point q)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  point_onto;
  ASM_MESON_TAC[REWRITE_RULE[ISUBSET] v_edge_euclid];
  DISCH_TAC;
  CHO 1;
  REWR 0;
  KILL 1;
  TYPE_THEN `point (FST q + &1, SND q )` EXISTS_TAC;
  REWRITE_TAC[point_scale;point_add;];
  UND 0;
  TYPE_THEN `point q = point (FST q,SND q)` SUBGOAL_TAC;
  REWRITE_TAC[];
  DISCH_THEN (fun t-> PURE_REWRITE_TAC [t]);
  PURE_REWRITE_TAC[point_add;point_scale];
  REWRITE_TAC[v_edge;squ;point_inj;PAIR_SPLIT;];
  DISCH_ALL_TAC;
  USE 0 (CONV_RULE (dropq_conv "u"));
  USE 0 (CONV_RULE (dropq_conv "v"));
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  UND 0;
  REWRITE_TAC[int_suc];
  ASSUME_TAC (real_poly_conv `t *x + (&1 - t)* x`);
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  ASSUME_TAC (real_poly_conv `t *(y + &1) + (&1- t)* y`);
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  UND 1;
  UND 2;
  REDUCE_TAC ;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let squ_closure_right_v = prove_by_refinement(
  `!p. (v_edge (right     p)) SUBSET (closure top2 (squ p))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;right    ];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[top2];
  IMATCH_MP_TAC  closure_segment;
  ASM_REWRITE_TAC[squ_euclid];
  TYPE_THEN `?q. (x = point q)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  point_onto;
  ASM_MESON_TAC[REWRITE_RULE[ISUBSET] v_edge_euclid];
  DISCH_TAC;
  CHO 1;
  REWR 0;
  KILL 1;
  TYPE_THEN `point (FST q - &1 , SND q )` EXISTS_TAC;
  REWRITE_TAC[point_scale;point_add;];
  UND 0;
  TYPE_THEN `point q = point (FST q,SND q)` SUBGOAL_TAC;
  REWRITE_TAC[];
  DISCH_THEN (fun t-> PURE_REWRITE_TAC [t]);
  PURE_REWRITE_TAC[point_add;point_scale];
  REWRITE_TAC[v_edge;squ;point_inj;PAIR_SPLIT;];
  DISCH_ALL_TAC;
  USE 0 (CONV_RULE (dropq_conv "u"));
  USE 0 (CONV_RULE (dropq_conv "v"));
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  UND 0;
  REWRITE_TAC[int_suc];
  ASSUME_TAC (real_poly_conv `t *x + (&1 - t)* x`);
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  ASSUME_TAC (real_poly_conv `t *(y - &1) + (&1- t)* y`);
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  UND 1;
  UND 2;
  REDUCE_TAC ;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let squ_closure_left_v  = prove_by_refinement(
  `!p. (v_edge p SUBSET (closure top2 (squ (left  p))))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  ASSUME_TAC squ_closure_right_v;
  TSPEC `left  p` 0;
  USE 0 (REWRITE_RULE[right_left]);
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let squ_closure_hc = prove_by_refinement(
  `!p. (hc_edge p) SUBSET (closure top2 (squ p))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  REWRITE_TAC[GSYM h_edge_closure];
  IMATCH_MP_TAC  closure_subset;
  ASSUME_TAC top2_top;
  ASM_REWRITE_TAC[squ_closure_h];
  IMATCH_MP_TAC  closure_closed;
  ASM_REWRITE_TAC[top2_unions;squ_euclid];
  ]);;

  (* }}} *)

let squ_closure_up_hc = prove_by_refinement(
  `!p. (hc_edge (up p)) SUBSET (closure top2 (squ p))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[GSYM h_edge_closure];
  IMATCH_MP_TAC  closure_subset;
  ASSUME_TAC top2_top;
  ASM_REWRITE_TAC[squ_closure_up_h];
  IMATCH_MP_TAC  closure_closed;
  ASM_REWRITE_TAC[top2_unions;squ_euclid];
  ]);;
  (* }}} *)

let squ_closure_vc = prove_by_refinement(
  `!p. (vc_edge p) SUBSET (closure top2 (squ p))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[GSYM v_edge_closure];
  IMATCH_MP_TAC  closure_subset;
  ASSUME_TAC top2_top;
  ASM_REWRITE_TAC[squ_closure_v];
  IMATCH_MP_TAC  closure_closed;
  ASM_REWRITE_TAC[top2_unions;squ_euclid];
  ]);;
  (* }}} *)

let squ_closure = prove_by_refinement(
  `!p. (closure top2 (squ p)) = (squc p)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  ASSUME_TAC top2_top;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC  closure_subset;
  ASM_REWRITE_TAC[squc_closed];
  REWRITE_TAC[squc_union];
  REWRITE_TAC[SUBSET;UNION];
  ASM_MESON_TAC[];
  REWRITE_TAC[squc_union];
  REWRITE_TAC[union_subset];
  ASSUME_TAC squ_closure_hc;
  TSPEC `p` 1;
  ASSUME_TAC squ_closure_up_hc;
  TSPEC `p` 2;
  USE 1 (REWRITE_RULE[hc_edge;plus_e12;union_subset]);
  USE 2 (REWRITE_RULE[hc_edge;plus_e12;up;union_subset]);
  ASM_REWRITE_TAC [up;right;squ_closure_v;REWRITE_RULE[right  ] squ_closure_right_v  ];
  ASM_SIMP_TAC[subset_closure];
  ]);;

  (* }}} *)

(* ------------------------------------------------------------------ *)
(* adj_edge *)
(* ------------------------------------------------------------------ *)


let adj_edge = jordan_def `adj_edge x y <=> (~(x = y)) /\
  (?e. (edge e) /\
   (e SUBSET (closure top2 x)) /\ (e SUBSET (closure top2 y)))`;;

let adj_edge_sym = prove_by_refinement(
  `!x y. (adj_edge x y = adj_edge y x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[adj_edge];
  MESON_TAC[];
  ]);;
  (* }}} *)

let adj_edge_left = prove_by_refinement(
  `!m. (adj_edge (squ m) (squ (left  m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[adj_edge];
  REWRITE_TAC[squ_closure;squ_inj;];
  CONJ_TAC;
  REWRITE_TAC[left ;PAIR_SPLIT;];
  INT_ARITH_TAC;
  TYPE_THEN `v_edge m` EXISTS_TAC;
  REWRITE_TAC[edge;v_edge_inj;];
  CONV_TAC (dropq_conv "m'");
  REWRITE_TAC[squc_union; SUBSET;UNION ;];
  REWRITE_TAC[right_left];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let adj_edge_right = prove_by_refinement(
  `!m. (adj_edge (squ m) (squ (right    m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[adj_edge];
  REWRITE_TAC[squ_closure;squ_inj;];
  CONJ_TAC;
  REWRITE_TAC[right   ;PAIR_SPLIT;];
  INT_ARITH_TAC;
  TYPE_THEN `v_edge (right  m)` EXISTS_TAC;
  REWRITE_TAC[edge;v_edge_inj;];
  CONV_TAC (dropq_conv "m'");
  REWRITE_TAC[squc_union; SUBSET;UNION ;];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let adj_edge_down = prove_by_refinement(
  `!m. (adj_edge (squ m) (squ (down  m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[adj_edge];
  REWRITE_TAC[squ_closure;squ_inj;];
  CONJ_TAC;
  REWRITE_TAC[down ;PAIR_SPLIT;];
  INT_ARITH_TAC;
  TYPE_THEN `h_edge m` EXISTS_TAC;
  REWRITE_TAC[edge;h_edge_inj;];
  CONV_TAC (dropq_conv "m'");
  REWRITE_TAC[squc_union; SUBSET;UNION ;];
  REWRITE_TAC[right_left];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let adj_edge_right = prove_by_refinement(
  `!m. (adj_edge (squ m) (squ (up    m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[adj_edge];
  REWRITE_TAC[squ_closure;squ_inj;];
  CONJ_TAC;
  REWRITE_TAC[up   ;PAIR_SPLIT;];
  INT_ARITH_TAC;
  TYPE_THEN `h_edge (up  m)` EXISTS_TAC;
  REWRITE_TAC[edge;h_edge_inj;];
  CONV_TAC (dropq_conv "m'");
  REWRITE_TAC[squc_union; SUBSET;UNION ;];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* components  *)
(* ------------------------------------------------------------------ *)

let rectangle_euclid = prove_by_refinement(
  `!p q. (rectangle p q SUBSET (euclid 2))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[rectangle;SUBSET ;];
  DISCH_ALL_TAC;
  CHO 0;
  CHO 0;
  ASM_REWRITE_TAC[euclid_point];
  ]);;
  (* }}} *)

let component_unions = prove_by_refinement(
  `!U (x:A). (component  U x SUBSET (UNIONS U))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET; component_DEF; connected ;];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let comp_h_rect = prove_by_refinement(
  `!G m x. (segment G /\
     (h_edge m SUBSET component  (ctop G) x)) ==>
   (rectangle (FST m , SND m -: &:1) (FST m +: &:1,SND m +: &:1)
       SUBSET component  (ctop G) x)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC   convex_component;
  ASM_REWRITE_TAC[rectangle_convex; ctop_unions;];
  CONJ_TAC;
  REWRITE_TAC[DIFF_SUBSET;rectangle_euclid];
  REWRITE_TAC[rectangle_h; EQ_EMPTY ;UNION ; INTER;];
  DISCH_ALL_TAC;
  AND 2;
  TYPE_THEN `~(squ (down m) x') /\ ~(squ m x')` SUBGOAL_TAC;
  USE 0(MATCH_MP curve_cell_squ_inter);
  COPY 0;
  TSPEC `m` 0;
  TSPEC `down m` 4;
  UND 4;
  UND 0;
  REWRITE_TAC [EQ_EMPTY; INTER];
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  REWR 3;
  TYPE_THEN `h_edge m SUBSET (UNIONS (ctop G))` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `component  (ctop G) x` EXISTS_TAC;
  ASM_REWRITE_TAC[component_unions];
  REWRITE_TAC[ctop_unions ;DIFF_SUBSET; EQ_EMPTY ; h_edge_euclid; INTER;];
  ASM_MESON_TAC[];
  REWRITE_TAC[rectangle_h; EMPTY_EXISTS; UNION ; INTER;];
  USE 1 (REWRITE_RULE[SUBSET]);
  TYPE_THEN `~(h_edge m = EMPTY)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  cell_nonempty;
  REWRITE_TAC[cell_rules];
  REWRITE_TAC[EMPTY_EXISTS];
  DISCH_TAC;
  CHO 2;
  TYPE_THEN `u` EXISTS_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let comp_v_rect = prove_by_refinement(
  `!G m x. (segment G /\
     (v_edge m SUBSET component  (ctop G) x)) ==>
   (rectangle (FST m -: &:1, SND m ) (FST m +: &:1,SND m +: &:1)
       SUBSET component  (ctop G) x)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC   convex_component;
  ASM_REWRITE_TAC[rectangle_convex; ctop_unions;];
  CONJ_TAC;
  REWRITE_TAC[DIFF_SUBSET;rectangle_euclid];
  REWRITE_TAC[rectangle_v; EQ_EMPTY ;UNION ; INTER;];
  DISCH_ALL_TAC;
  AND 2;
  TYPE_THEN `~(squ (left   m) x') /\ ~(squ m x')` SUBGOAL_TAC;
  USE 0(MATCH_MP curve_cell_squ_inter);
  COPY 0;
  TSPEC `m` 0;
  TSPEC `left   m` 4;
  UND 4;
  UND 0;
  REWRITE_TAC [EQ_EMPTY; INTER];
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  REWR 3;
  TYPE_THEN `v_edge m SUBSET (UNIONS (ctop G))` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `component  (ctop G) x` EXISTS_TAC;
  ASM_REWRITE_TAC[component_unions];
  REWRITE_TAC[ctop_unions ;DIFF_SUBSET; EQ_EMPTY ; v_edge_euclid; INTER;];
  ASM_MESON_TAC[];
  REWRITE_TAC[rectangle_v; EMPTY_EXISTS; UNION ; INTER;];
  USE 1 (REWRITE_RULE[SUBSET]);
  TYPE_THEN `~(v_edge m = EMPTY)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  cell_nonempty;
  REWRITE_TAC[cell_rules];
  REWRITE_TAC[EMPTY_EXISTS];
  DISCH_TAC;
  CHO 2;
  TYPE_THEN `u` EXISTS_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let long_v_convex = prove_by_refinement(
  `!p. (convex (long_v p))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[long_v_inter];
  GEN_TAC;
  IMATCH_MP_TAC  convex_inter;
  REWRITE_TAC[line2D_F_convex];
  IMATCH_MP_TAC  convex_inter;
  REWRITE_TAC[open_half_plane2D_LTS_convex;open_half_plane2D_SLT_convex];
  ]);;
  (* }}} *)

let long_v_euclid = prove_by_refinement(
  `!p. (long_v p SUBSET (euclid 2))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[long_v_union;union_subset;v_edge_euclid;single_subset;pointI;euclid_point];
  ]);;
  (* }}} *)

let comp_pointI_long = prove_by_refinement(
  `!G m x. (segment G /\ component  (ctop G) x (pointI m)) ==>
   (long_v m SUBSET component  (ctop G) x)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  convex_component;
  ASM_REWRITE_TAC[long_v_convex;ctop_unions;DIFF_SUBSET;long_v_euclid];
  CONJ_TAC;
  REWRITE_TAC[long_v_union;EQ_EMPTY;UNION;INTER];
  GEN_TAC;
  TYPE_THEN `UNIONS (ctop G) (pointI m)` SUBGOAL_TAC;
  ASSUME_TAC (ISPEC `(ctop G)` component_unions);
  ASM_MESON_TAC[ISUBSET];
  REWRITE_TAC[ctop_unions;DIFF ;];
  DISCH_ALL_TAC;
  AND 2;
  TYPE_THEN `~(curve_cell G {(pointI m)})` SUBGOAL_TAC;
  USE 4(REWRITE_RULE[UNIONS]);
  LEFT 4 "u";
  TSPEC `{(pointI m)}` 4;
  USE 4(REWRITE_RULE [INR IN_SING;]);
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[curve_cell_not_point;];
  TYPE_THEN `FINITE G` SUBGOAL_TAC;
  ASM_SIMP_TAC[segment_finite];
  ASM_SIMP_TAC[num_closure0];
  DISCH_TAC;
  UND 5;
  REP_CASES_TAC; (* cases *)
  TYPE_THEN `~(v_edge (down m) INTER  UNIONS (curve_cell G) = EMPTY)` SUBGOAL_TAC;
  REWRITE_TAC[EMPTY_EXISTS;INTER ];
  ASM_MESON_TAC[];
  ASM_SIMP_TAC[curve_cell_v_inter];
  DISCH_ALL_TAC;
  TSPEC `v_edge (down m)` 5;
  UND 5;
  ASM_REWRITE_TAC[v_edge_closure;vc_edge;plus_e12;UNION; INR IN_SING; pointI_inj; down; PAIR_SPLIT ; INT_ARITH `x = x -: &:1 +: &:1`;];
  (* next case *)
  USE 7 (REWRITE_RULE[INR IN_SING]);
  ASM_MESON_TAC[];
  TYPE_THEN `~(v_edge (m) INTER  UNIONS (curve_cell G) = EMPTY)` SUBGOAL_TAC;
  REWRITE_TAC[EMPTY_EXISTS;INTER ];
  ASM_MESON_TAC[];
  ASM_SIMP_TAC[curve_cell_v_inter];
  DISCH_ALL_TAC;
  TSPEC `v_edge (m)` 5;
  UND 5;
  ASM_REWRITE_TAC[v_edge_closure;vc_edge;plus_e12;UNION; INR IN_SING; pointI_inj; down; PAIR_SPLIT ; INT_ARITH `x = x -: &:1 +: &:1`;];
  (* LAST *)
  REWRITE_TAC[long_v_union;EMPTY_EXISTS;];
  TYPE_THEN `(pointI m)` EXISTS_TAC;
  ASM_REWRITE_TAC[INTER;UNION;INR IN_SING;];
  ]);;
  (* }}} *)

let comp_h_squ = prove_by_refinement(
  `!G x m. (segment G /\ (h_edge m SUBSET (component  (ctop G) x)) ==>
     (squ m SUBSET (component  (ctop G ) x)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `(rectangle (FST m , SND m -: &:1) (FST m +: &:1,SND m +: &:1) SUBSET component  (ctop G) x)` SUBGOAL_TAC;
  IMATCH_MP_TAC comp_h_rect;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `rectangle (FST m,SND m -: &:1) (FST m +: &:1,SND m +: &:1)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[rectangle_h];
  REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  ]);;
  (* }}} *)

let comp_v_squ = prove_by_refinement(
  `!G x m. (segment G /\ (v_edge m SUBSET (component  (ctop G) x)) ==>
     (squ m SUBSET (component  (ctop G ) x)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `(rectangle (FST m -: &:1 , SND m ) (FST m +: &:1,SND m +: &:1) SUBSET component  (ctop G) x)` SUBGOAL_TAC;
  IMATCH_MP_TAC comp_v_rect;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `rectangle (FST m -: &:1 ,SND m) (FST m +: &:1,SND m +: &:1)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[rectangle_v];
  REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  ]);;
  (* }}} *)

let comp_p_squ = prove_by_refinement(
  `!G x m. (segment G /\ (component  (ctop G) x (pointI m))) ==>
     (squ m SUBSET (component  (ctop G ) x))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `long_v m SUBSET component  (ctop G) x` SUBGOAL_TAC;
  IMATCH_MP_TAC comp_pointI_long;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[long_v_union];
  REWRITE_TAC[union_subset];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  comp_v_squ;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let comp_squ = prove_by_refinement(
  `!G x. (segment G /\ (~(component  (ctop G) x = EMPTY)) ==>
     (?m. (squ m SUBSET (component  (ctop G ) x))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  COPY 0;
  USE 0 (MATCH_MP unions_cell_of);
  TSPEC `x` 0;
  USE 0 (SYM);
  USE 1 (REWRITE_RULE[EMPTY_EXISTS]);
  CHO 1;
  UND 0;
  DISCH_THEN (fun t-> USE 1 (ONCE_REWRITE_RULE[t]));
  USE 0 (REWRITE_RULE[cell_of;UNIONS]);
  CHO 0;
  UND 0;
  DISCH_ALL_TAC;
  USE 0 (REWRITE_RULE[cell]);
  CHO 0;
  UND 0;
  REP_CASES_TAC;
  REWR 1;
  USE 1 (REWRITE_RULE[single_subset]);
  ASM_MESON_TAC[comp_p_squ];
  ASM_MESON_TAC[comp_h_squ];
  ASM_MESON_TAC[comp_v_squ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let comp_squ_left_rect_v = prove_by_refinement(
  `!G m x. (segment G /\ ~(G (v_edge (  m))) /\
    (squ m SUBSET component (ctop G) x) ==>
   (rectangle (FST m -: &:1 ,SND m ) (FST m +: &:1,SND m +: &:1) SUBSET
 component (ctop G) x))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  UND 1;
  ASM_SIMP_TAC[GSYM curve_cell_v];
  DISCH_TAC;
  (*  *)
  IMATCH_MP_TAC   convex_component;
  ASM_REWRITE_TAC[rectangle_convex; ctop_unions;];
  CONJ_TAC;
  REWRITE_TAC[DIFF_SUBSET;rectangle_euclid];
  REWRITE_TAC[rectangle_v; EQ_EMPTY ;UNION ; INTER;];
  DISCH_ALL_TAC;
  AND 3;
  TYPE_THEN `~(squ (left   m) x') /\ ~(squ m x')` SUBGOAL_TAC;
  USE 0(MATCH_MP curve_cell_squ_inter);
  COPY 0;
  TSPEC `m` 0;
  TSPEC `left   m` 5;
  UND 5;
  UND 0;
  REWRITE_TAC [EQ_EMPTY; INTER];
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  REWR 4;
  USE 3 (REWRITE_RULE[UNIONS;]);
  CHO 3;
  TYPE_THEN `cell u` SUBGOAL_TAC;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  ASM_MESON_TAC[ISUBSET; curve_cell_cell];
  DISCH_TAC;
  TYPE_THEN `u = v_edge m ` SUBGOAL_TAC;
  IMATCH_MP_TAC  cell_partition;
  ASM_REWRITE_TAC[EMPTY_EXISTS;INTER;cell_rules];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  REWRITE_TAC[rectangle_v;EMPTY_EXISTS;];
  TYPE_THEN `~(squ m = EMPTY )` SUBGOAL_TAC;
  ASM_MESON_TAC[cell_nonempty;cell_rules];
  REWRITE_TAC[EMPTY_EXISTS;UNION;INTER;];
  USE 2(REWRITE_RULE[ISUBSET]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let comp_squ_left_rect = prove_by_refinement(
  `!G m x. (segment G /\
    (~(?p e. (G e /\ e SUBSET closure top2 (squ p) /\
         (squ p SUBSET (component  (ctop G) x))))) /\
     (squ m SUBSET component  (ctop G) x)) ==>
   (rectangle (FST m -: &:1, SND m ) (FST m +: &:1,SND m +: &:1)
       SUBSET component  (ctop G) x)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  LEFT 1 "p";
  TSPEC `m` 1;
  LEFT 1 "e";
  TSPEC `v_edge m` 1;
  REWR 1;
  USE 1(REWRITE_RULE[squ_closure_v]);
  IMATCH_MP_TAC  comp_squ_left_rect_v;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let comp_squ_right_rect_v = prove_by_refinement(
  `!G m x. (segment G /\ ~(G (v_edge (right  m))) /\
    (squ m SUBSET component (ctop G) x) ==>
   (rectangle (FST m,SND m ) (FST m +: &:2,SND m +: &:1) SUBSET
 component (ctop G) x))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  UND 1;
  ASM_SIMP_TAC[GSYM curve_cell_v];
  DISCH_TAC;
  (*  *)
  IMATCH_MP_TAC   convex_component;
  ASM_REWRITE_TAC[rectangle_convex; ctop_unions;];
  TYPE_THEN `rectangle m (FST m +: &:2,SND m +: &:1) = rectangle (FST (right  m) -: &:1, SND (right  m)) (FST (right  m) +: &:1, SND (right  m) +: &:1)` SUBGOAL_TAC;
  REWRITE_TAC[right ;INT_ARITH `(x +: &:1)-: &:1 = x`;INT_ARITH `(x +: &:1) +: &:1 = x +: &:2` ];
  DISCH_THEN_REWRITE;
  CONJ_TAC;
  REWRITE_TAC[DIFF_SUBSET;rectangle_euclid];
  REWRITE_TAC[rectangle_v; EQ_EMPTY ;UNION ; INTER;];
  DISCH_ALL_TAC;
  AND 3;
  USE 4 (REWRITE_RULE[right_left]);
  TYPE_THEN `~(squ  m x') /\ ~(squ (right  m) x')` SUBGOAL_TAC;
  USE 0(MATCH_MP curve_cell_squ_inter);
  COPY 0;
  TSPEC `m` 0;
  TSPEC `right   m` 5;
  UND 5;
  UND 0;
  REWRITE_TAC [EQ_EMPTY; INTER];
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  REWR 4;
  USE 3 (REWRITE_RULE[UNIONS;]);
  CHO 3;
  TYPE_THEN `cell u` SUBGOAL_TAC;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  ASM_MESON_TAC[ISUBSET; curve_cell_cell];
  DISCH_TAC;
  TYPE_THEN `u = v_edge (right  m) ` SUBGOAL_TAC;
  IMATCH_MP_TAC  cell_partition;
  ASM_REWRITE_TAC[EMPTY_EXISTS;INTER;cell_rules];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  REWRITE_TAC[rectangle_v;EMPTY_EXISTS;];
  REWRITE_TAC[right_left];
  TYPE_THEN `~(squ m = EMPTY )` SUBGOAL_TAC;
  ASM_MESON_TAC[cell_nonempty;cell_rules];
  REWRITE_TAC[EMPTY_EXISTS;UNION;INTER;];
  USE 2(REWRITE_RULE[ISUBSET]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let comp_squ_right_rect = prove_by_refinement(
  `!G m x. (segment G /\
    (~(?p e. (G e /\ e SUBSET closure top2 (squ p) /\
         (squ p SUBSET (component  (ctop G) x))))) /\
     (squ m SUBSET component  (ctop G) x)) ==>
   (rectangle (FST m , SND m ) (FST m +: &:2,SND m +: &:1)
       SUBSET component  (ctop G) x)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  LEFT 1 "p";
  TSPEC `m` 1;
  LEFT 1 "e";
  TSPEC `v_edge (right  m)` 1;
  REWR 1;
  USE 1(REWRITE_RULE[squ_closure_right_v]);
  IMATCH_MP_TAC  comp_squ_right_rect_v;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let comp_squ_down_rect_h = prove_by_refinement(
  `!G m x. (segment G /\ ~(G (h_edge m)) /\
    (squ m SUBSET component (ctop G) x) ==>
   (rectangle (FST m,SND m -: &:1) (FST m +: &:1,SND m +: &:1) SUBSET
 component (ctop G) x))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  UND 1;
  ASM_SIMP_TAC[GSYM curve_cell_h];
  DISCH_TAC;
  (*  *)
  IMATCH_MP_TAC   convex_component;
  ASM_REWRITE_TAC[rectangle_convex; ctop_unions;];
  CONJ_TAC;
  REWRITE_TAC[DIFF_SUBSET;rectangle_euclid];
  REWRITE_TAC[rectangle_h; EQ_EMPTY ;UNION ; INTER;];
  DISCH_ALL_TAC;
  AND 3;
  TYPE_THEN `~(squ (down   m) x') /\ ~(squ m x')` SUBGOAL_TAC;
  USE 0(MATCH_MP curve_cell_squ_inter);
  COPY 0;
  TSPEC `m` 0;
  TSPEC `down   m` 5;
  UND 5;
  UND 0;
  REWRITE_TAC [EQ_EMPTY; INTER];
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  REWR 4;
  USE 3 (REWRITE_RULE[UNIONS;]);
  CHO 3;
  TYPE_THEN `cell u` SUBGOAL_TAC;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  ASM_MESON_TAC[ISUBSET; curve_cell_cell];
  DISCH_TAC;
  TYPE_THEN `u = h_edge m ` SUBGOAL_TAC;
  IMATCH_MP_TAC  cell_partition;
  ASM_REWRITE_TAC[EMPTY_EXISTS;INTER;cell_rules];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  REWRITE_TAC[rectangle_h;EMPTY_EXISTS;];
  TYPE_THEN `~(squ m = EMPTY )` SUBGOAL_TAC;
  ASM_MESON_TAC[cell_nonempty;cell_rules];
  REWRITE_TAC[EMPTY_EXISTS;UNION;INTER;];
  USE 2(REWRITE_RULE[ISUBSET]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let comp_squ_down_rect = prove_by_refinement(
  `!G m x. (segment G /\
    (~(?p e. (G e /\ e SUBSET closure top2 (squ p) /\
         (squ p SUBSET (component  (ctop G) x))))) /\
     (squ m SUBSET component  (ctop G) x)) ==>
   (rectangle (FST m , SND m -: &:1) (FST m +: &:1,SND m +: &:1)
       SUBSET component  (ctop G) x)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  LEFT 1 "p";
  TSPEC `m` 1;
  LEFT 1 "e";
  TSPEC `h_edge m` 1;
  REWR 1;
  USE 1(REWRITE_RULE[squ_closure_h]);
  ASM_MESON_TAC[comp_squ_down_rect_h];
  ]);;

  (* }}} *)

let comp_squ_up_rect_h = prove_by_refinement(
  `!G m x. (segment G /\ ~(G (h_edge (up m))) /\
    (squ m SUBSET component (ctop G) x) ==>
   (rectangle (FST m,SND m ) (FST m +: &:1,SND m +: &:2) SUBSET
 component (ctop G) x))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  UND 1;
  ASM_SIMP_TAC[GSYM curve_cell_h];
  DISCH_TAC;
  (*  *)
  IMATCH_MP_TAC   convex_component;
  ASM_REWRITE_TAC[rectangle_convex; ctop_unions;];
  TYPE_THEN `rectangle m (FST m +: &:1,SND m +: &:2) = rectangle (FST (up  m) , SND (up  m) -: &:1) (FST (up  m) +: &:1, SND (up  m) +: &:1)` SUBGOAL_TAC;
  REWRITE_TAC[up ;INT_ARITH `(x +: &:1)-: &:1 = x`;INT_ARITH `(x +: &:1) +: &:1 = x +: &:2` ];
  DISCH_THEN_REWRITE;
  CONJ_TAC;
  REWRITE_TAC[DIFF_SUBSET;rectangle_euclid];
  REWRITE_TAC[rectangle_h; EQ_EMPTY ;UNION ; INTER;];
  DISCH_ALL_TAC;
  AND 3;
  USE 4 (REWRITE_RULE[right_left]);
  TYPE_THEN `~(squ  m x') /\ ~(squ (up  m) x')` SUBGOAL_TAC;
  USE 0(MATCH_MP curve_cell_squ_inter);
  COPY 0;
  TSPEC `m` 0;
  TSPEC `up   m` 5;
  UND 5;
  UND 0;
  REWRITE_TAC [EQ_EMPTY; INTER];
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  REWR 4;
  USE 3 (REWRITE_RULE[UNIONS;]);
  CHO 3;
  TYPE_THEN `cell u` SUBGOAL_TAC;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  ASM_MESON_TAC[ISUBSET; curve_cell_cell];
  DISCH_TAC;
  TYPE_THEN `u = h_edge (up  m) ` SUBGOAL_TAC;
  IMATCH_MP_TAC  cell_partition;
  ASM_REWRITE_TAC[EMPTY_EXISTS;INTER;cell_rules];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  REWRITE_TAC[rectangle_h;EMPTY_EXISTS;];
  REWRITE_TAC[right_left];
  TYPE_THEN `~(squ m = EMPTY )` SUBGOAL_TAC;
  ASM_MESON_TAC[cell_nonempty;cell_rules];
  REWRITE_TAC[EMPTY_EXISTS;UNION;INTER;];
  USE 2(REWRITE_RULE[ISUBSET]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let comp_squ_up_rect = prove_by_refinement(
  `!G m x. (segment G /\
    (~(?p e. (G e /\ e SUBSET closure top2 (squ p) /\
         (squ p SUBSET (component  (ctop G) x))))) /\
     (squ m SUBSET component  (ctop G) x)) ==>
   (rectangle (FST m , SND m ) (FST m +: &:1,SND m +: &:2)
       SUBSET component  (ctop G) x)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  LEFT 1 "p";
  TSPEC `m` 1;
  LEFT 1 "e";
  TSPEC `h_edge (up  m)` 1;
  REWR 1;
  USE 1(REWRITE_RULE[squ_closure_up_h]);
  IMATCH_MP_TAC  comp_squ_up_rect_h;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let comp_squ_right_left = prove_by_refinement(
  `!G x m. (segment G /\ (squ m SUBSET (component  (ctop G) x))  /\
    (~(?p e. (G e /\ e SUBSET closure top2 (squ p) /\
         (squ p SUBSET (component  (ctop G) x)))))) ==>
     (squ (left    m) SUBSET (component  (ctop G) x))  /\
    (squ (right      m) SUBSET (component  (ctop G) x))  /\
    (squ (up  m) SUBSET (component  (ctop G) x))  /\
   (squ (down  m) SUBSET (component  (ctop G) x))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  JOIN 2 1;
  JOIN 0 1;
  WITH 0 (MATCH_MP comp_squ_up_rect);
  WITH 0 (MATCH_MP comp_squ_down_rect);
  WITH 0 (MATCH_MP comp_squ_left_rect);
  WITH 0 (MATCH_MP comp_squ_right_rect);
  TYPE_THEN `rectangle m (FST m +: &:1,SND m +: &:2) = rectangle (FST (up  m) , SND (up  m) -: &:1) (FST (up  m) +: &:1, SND (up  m) +: &:1)` SUBGOAL_TAC;
  REWRITE_TAC[up ;INT_ARITH `(x +: &:1)-: &:1 = x`;INT_ARITH `(x +: &:1) +: &:1 = x +: &:2` ];
  DISCH_THEN (fun t-> USE 1 (REWRITE_RULE[t]));
  TYPE_THEN `rectangle m (FST m +: &:2,SND m +: &:1) = rectangle (FST (right  m) -: &:1, SND (right  m)) (FST (right  m) +: &:1, SND (right  m) +: &:1)` SUBGOAL_TAC;
  REWRITE_TAC[right ;INT_ARITH `(x +: &:1)-: &:1 = x`;INT_ARITH `(x +: &:1) +: &:1 = x +: &:2` ];
  DISCH_THEN (fun t-> USE 4 (REWRITE_RULE[t]));
  RULE_ASSUM_TAC (REWRITE_RULE[rectangle_h;rectangle_v;union_subset;right_left ]);
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

(* move *)
let suc_sum = prove_by_refinement(
  `!j a b. (SUC j = a+ b) ==> (?k. (SUC k = a) \/ (SUC k = b))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  PROOF_BY_CONTR_TAC;
  LEFT 1 "k";
  USE 1(REWRITE_RULE[DE_MORGAN_THM]);
  TYPE_THEN `a = 0 ` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  ASM_MESON_TAC[num_CASES];
  TYPE_THEN `b = 0` SUBGOAL_TAC;
  ASM_MESON_TAC[num_CASES];
  UND 0;
  ARITH_TAC;
  ]);;
  (* }}} *)

let squ_induct = prove_by_refinement(
  `!j m n. ?p.
    ((SUC j) = (num_abs_of_int (FST m -: FST n) +
             num_abs_of_int (SND  m -: SND  n))) ==>
    ((j = (num_abs_of_int (FST p -: FST n) +
             num_abs_of_int (SND  p -: SND  n))) /\
     ((p = left  m) \/ (p = right  m) \/ (p = up m) \/ (p = down m))) `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  RIGHT_TAC "p";
  DISCH_TAC;
  WITH  0 (MATCH_MP suc_sum);
  CHO 1;
  UND 1;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `~(num_abs_of_int (FST m -: FST n) = 0)` SUBGOAL_TAC;
  UND 1;
  ARITH_TAC;
  REWRITE_TAC[num_abs_of_int0];
  DISCH_TAC;
  TYPE_THEN `FST m <: FST n \/ FST n <: FST m` SUBGOAL_TAC;
  UND 2;
  INT_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `right  m` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[right ];
  ONCE_REWRITE_TAC[GSYM EQ_SUC];
  REWRITE_TAC[GSYM ADD];
  TYPE_THEN `(FST m +: &:1) -: FST n <=: &:0` SUBGOAL_TAC;
  UND 3;
  INT_ARITH_TAC;
  ASM_SIMP_TAC[num_abs_of_int_pre];
  TYPE_THEN `(FST m +: &:1) -: FST n -: &:1 = FST m -: FST n` SUBGOAL_TAC;
  INT_ARITH_TAC;
  DISCH_THEN_REWRITE;
  (* next *)
  TYPE_THEN `left    m` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[left   ];
  ONCE_REWRITE_TAC[GSYM EQ_SUC];
  REWRITE_TAC[GSYM ADD];
  TYPE_THEN `&:0 <=: (FST m -: &:1) -: FST n ` SUBGOAL_TAC;
  UND 3;
  INT_ARITH_TAC;
  ASM_SIMP_TAC[num_abs_of_int_suc];
  TYPE_THEN `(FST m -: &:1 -: FST n +: &:1) = FST m -: FST n` SUBGOAL_TAC;
  INT_ARITH_TAC;
  DISCH_THEN_REWRITE;
  (* next *)
  TYPE_THEN `~(num_abs_of_int (SND  m -: SND  n) = 0)` SUBGOAL_TAC;
  UND 1;
  ARITH_TAC;
  REWRITE_TAC[num_abs_of_int0];
  DISCH_TAC;
  TYPE_THEN `SND  m <: SND  n \/ SND  n <: SND  m` SUBGOAL_TAC;
  UND 2;
  INT_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  (* next *)
  TYPE_THEN `up    m` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[up  ];
  ONCE_REWRITE_TAC[GSYM EQ_SUC];
  REWRITE_TAC[GSYM ADD_SUC];
  TYPE_THEN `(SND  m +: &:1) -: SND  n <=: &:0` SUBGOAL_TAC;
  UND 3;
  INT_ARITH_TAC;
  ASM_SIMP_TAC[num_abs_of_int_pre];
  TYPE_THEN `((SND  m +: &:1) -: SND  n -: &:1) = SND  m -: SND  n` SUBGOAL_TAC;
  INT_ARITH_TAC;
  DISCH_THEN_REWRITE;
  (* final *)
  TYPE_THEN `down    m` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[down   ];
  ONCE_REWRITE_TAC[GSYM EQ_SUC];
  REWRITE_TAC[GSYM ADD_SUC];
  TYPE_THEN `&:0 <=: (SND  m -: &:1) -: SND  n ` SUBGOAL_TAC;
  UND 3;
  INT_ARITH_TAC;
  ASM_SIMP_TAC[num_abs_of_int_suc];
  TYPE_THEN `(SND  m -: &:1 -: SND  n +: &:1) = SND  m -: SND  n` SUBGOAL_TAC;
  INT_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ]);;
  (* }}} *)

let comp_squ_fill = prove_by_refinement(
  `!G x m. (segment G /\ (squ m SUBSET (component  (ctop G ) x)) /\
  (~(?p e. (G e /\ e SUBSET closure top2 (squ p) /\
         (squ p SUBSET (component  (ctop G) x)))))) ==>
  (!n. (squ n SUBSET (component  (ctop G) x)))
  `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  GEN_TAC;
  TYPE_THEN `(!j n. (j = (num_abs_of_int (FST n -: FST m) + num_abs_of_int (SND  n -: SND  m))) ==> (squ n SUBSET component (ctop G) x)) ==> (squ n SUBSET component (ctop G) x)` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  INDUCT_TAC;
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  REWRITE_TAC[ADD_EQ_0;num_abs_of_int0];
  GEN_TAC;
  DISCH_TAC;
  TYPE_THEN `n = m` SUBGOAL_TAC;
  UND 3;
  REWRITE_TAC[PAIR_SPLIT];
  INT_ARITH_TAC;
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  USE 4 (MATCH_MP (CONV_RULE (quant_right_CONV "p") squ_induct));
  CHO 4;
  TSPEC `p` 3;
  REWR 3;
  AND 4;
  TYPE_THEN `(n = left p) \/ (n = right p) \/ (n = up p) \/ (n = down p)` SUBGOAL_TAC;
  UND 4;
  REP_CASES_TAC THEN (ASM_REWRITE_TAC[right_left]);
  KILL 4;
  KILL 5;
  KILL 1;
  JOIN  3 2;
  JOIN 0 1;
  USE 0 (MATCH_MP comp_squ_right_left);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let comp_squ_adj = prove_by_refinement(
  `!G x m. (segment G /\ (squ m SUBSET (component  (ctop G ) x))) ==>
     (?p e. (G e /\ e SUBSET closure top2 (squ p) /\
         (squ p SUBSET (component  (ctop G) x))))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `(!n. (squ n SUBSET (component  (ctop G) x)))` SUBGOAL_TAC;
  ASM_MESON_TAC[comp_squ_fill];
  DISCH_TAC;
  TYPE_THEN `?e. (G e /\ (edge e))` SUBGOAL_TAC;
  USE 0 (REWRITE_RULE [segment;EMPTY_EXISTS;SUBSET;]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  UND 2;
  REWRITE_TAC[];
  LEFT_TAC "e";
  CHO 4;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  AND 2;
  USE 2(REWRITE_RULE[edge]);
  CHO 2;
  UND 2;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `m'` EXISTS_TAC;
  ASM_REWRITE_TAC[squ_closure_v;squ_closure_h];
  ASM_MESON_TAC[squ_closure_v;squ_closure_h];
  ]);;

  (* }}} *)

(* ------------------------------------------------------------------ *)


let along_seg = jordan_def `along_seg G e x <=> G e /\
     (?p. (e SUBSET closure top2 (squ p) /\
          squ p SUBSET (component  (ctop G) x) ))`;;

let along_lemma1 = prove_by_refinement(
  `!G m x.  (segment G /\ (squ m SUBSET component  (ctop G) x) /\
     (G (v_edge m)) /\ (G (h_edge m))) ==>
   (?p. (h_edge m) SUBSET closure top2 (squ p) /\
       (squ p SUBSET (component  (ctop G) x)))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  TYPE_THEN `m` EXISTS_TAC;
  ASM_MESON_TAC[squ_closure_h];
  ]);;

  (* }}} *)

let midpoint_exclusion = prove_by_refinement(
  `!G m e e' e''. (segment G /\ G e /\ G e' /\ G e'' /\ (~(e = e')) /\
    (closure top2 e (pointI m)) /\ (closure top2 e' (pointI m)) /\
    (closure top2 e'' (pointI m))   ==> ((e'' = e) \/ (e'' = e')))
    `,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  USE 0 (REWRITE_RULE[segment;INSERT; ]);
  UND 0;
  DISCH_ALL_TAC;
  TYPE_THEN `num_closure G (pointI m) = 2` SUBGOAL_TAC;
  TSPEC `m` 10;
  UND 10;
  REP_CASES_TAC;
  ASM_REWRITE_TAC[];
  UND 10;
  USE 0 (MATCH_MP num_closure1);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  CHO 10;
  COPY 10;
  TSPEC `e` 12;
  TSPEC `e'` 10;
  ASM_MESON_TAC[];
  USE 0 (MATCH_MP num_closure0);
  TSPEC `pointI m` 0;
  REWR 0;
  TSPEC `e` 0;
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 0 (MATCH_MP num_closure_size);
  TSPEC `pointI m` 0;
  REWR 0;
  TYPE_THEN `X = {C | G C /\ closure top2 C (pointI m)}` ABBREV_TAC ;
  TYPE_THEN `X e /\ X e' /\ X e''` SUBGOAL_TAC;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  UND 0;
  UND 4;
  MESON_TAC[two_exclusion];
  ]);;
  (* }}} *)

(* indexed to here *)
let along_lemma2 = prove_by_refinement(
  `!G m. (segment G /\ G (v_edge m) /\ G (v_edge (down m)) ==>
     ~(G (h_edge m)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `(h_edge m = v_edge m) \/ (h_edge m = v_edge (down m))` SUBGOAL_TAC;
  IMATCH_MP_TAC  midpoint_exclusion;
  TYPE_THEN `G` EXISTS_TAC;
  TYPE_THEN `m` EXISTS_TAC;
  ASM_REWRITE_TAC[v_edge_inj;down;v_edge_cpoint;h_edge_cpoint;PAIR_SPLIT;];
  INT_ARITH_TAC ;
  REWRITE_TAC[hv_edgeV2;GSYM hv_edgeV2];
  ]);;
  (* }}} *)

let along_lemma3 = prove_by_refinement(
  `!G m. (segment G /\ G (v_edge m) /\ G(h_edge (left  m)) ==>
     ~(G (h_edge m)) /\ ~(G (v_edge (down m))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  CONJ_TAC;
  PROOF_BY_CONTR_TAC;
  USE 3(REWRITE_RULE[]);
  TYPE_THEN `(h_edge m = v_edge m) \/ (h_edge m = h_edge (left  m))` SUBGOAL_TAC;
  IMATCH_MP_TAC  midpoint_exclusion;
  TYPE_THEN `G` EXISTS_TAC;
  TYPE_THEN `m` EXISTS_TAC;
  ASM_REWRITE_TAC[v_edge_inj;left;v_edge_cpoint;GSYM hv_edgeV2;h_edge_cpoint;PAIR_SPLIT;];
  INT_ARITH_TAC ;
  REWRITE_TAC[hv_edgeV2;GSYM hv_edgeV2;left ;h_edge_inj;PAIR_SPLIT;];
  INT_ARITH_TAC;
  PROOF_BY_CONTR_TAC;
  USE 3(REWRITE_RULE[]);
  TYPE_THEN `(h_edge (left  m) = v_edge m) \/ (h_edge (left  m) = v_edge (down m))` SUBGOAL_TAC;
  IMATCH_MP_TAC  midpoint_exclusion;
  TYPE_THEN `G` EXISTS_TAC;
  TYPE_THEN `m` EXISTS_TAC;
  ASM_REWRITE_TAC[v_edge_inj;down;left ;v_edge_cpoint;h_edge_cpoint;PAIR_SPLIT;];
  INT_ARITH_TAC ;
  REWRITE_TAC[hv_edgeV2;GSYM hv_edgeV2];
  ]);;
  (* }}} *)

let along_lemma4 = prove_by_refinement(
  `!G m x.  (segment G /\ (squ m SUBSET component  (ctop G) x) /\
     (G (v_edge m)) /\ (G (v_edge (down m)))) ==>
   (?p. (v_edge (down m)) SUBSET closure top2 (squ p) /\
       (squ p SUBSET (component  (ctop G) x)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `down m` EXISTS_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[squ_closure_v];
  TYPE_THEN `~(G (h_edge m))` SUBGOAL_TAC;
  ASM_MESON_TAC[along_lemma2];
  DISCH_TAC;
  TYPE_THEN `(rectangle (FST m , SND m -: &:1) (FST m +: &:1,SND m +: &:1) SUBSET component  (ctop G) x)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  comp_squ_down_rect_h;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[rectangle_h; union_subset];
  MESON_TAC [];
  ]);;
  (* }}} *)

let along_lemma5 = prove_by_refinement(
  `!G m x. (segment G /\ (squ m SUBSET component  (ctop G) x) /\
     (G (v_edge m)) /\ (G (h_edge (left   m)))) ==>
   (?p. (h_edge (left   m)) SUBSET closure top2 (squ p) /\
       (squ p SUBSET (component  (ctop G) x)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `left  (down m)` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[GSYM right_left];
  ASM_MESON_TAC[squ_closure_down_h];
  TYPE_THEN ` ~(G (h_edge m)) /\ ~(G (v_edge (down m)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  along_lemma3;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  TYPE_THEN `(rectangle (FST m , SND m -: &:1) (FST m +: &:1,SND m +: &:1) SUBSET component  (ctop G) x)` SUBGOAL_TAC ;
  IMATCH_MP_TAC  comp_squ_down_rect_h;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[rectangle_h; union_subset];
  DISCH_ALL_TAC;
  TYPE_THEN `(rectangle (FST (down m) -: &:1,SND (down m)) (FST (down m) +: &:1,SND (down m) +: &:1) SUBSET component  (ctop G) x)` SUBGOAL_TAC;
  IMATCH_MP_TAC  comp_squ_left_rect_v;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[rectangle_v;union_subset;];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let along_lemma6 = prove_by_refinement(
  `!G m x e. (segment G /\ (squ m SUBSET component  (ctop G) x) /\
     (G (v_edge m)) /\ G e /\ (closure top2 e (pointI m)) ==>
   (?p. e SUBSET closure top2 (squ p) /\
       (squ p SUBSET (component  (ctop G) x))))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC ;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  TYPE_THEN `edge e` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET;];
  REWRITE_TAC[edge];
  DISCH_THEN (CHOOSE_THEN DISJ_CASES_TAC);
  REWR 4;
  USE 4 (REWRITE_RULE[v_edge_cpoint]);
  UND 4;
  DISCH_TAC;
  TYPE_THEN `(m' = m) \/ (m' = (down m))` SUBGOAL_TAC;
  UND 4;
  REWRITE_TAC[down;PAIR_SPLIT];
  INT_ARITH_TAC ;
  KILL 4;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `m` EXISTS_TAC;
  ASM_REWRITE_TAC[squ_closure_v];
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  along_lemma4;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ASM_REWRITE_TAC[];
  REWR 4;
  USE 4(REWRITE_RULE[h_edge_cpoint]);
  TYPE_THEN `(m' = m) \/ (m' = (left  m))` SUBGOAL_TAC;
  UND 4;
  REWRITE_TAC[left;PAIR_SPLIT];
  INT_ARITH_TAC ;
  KILL 4;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  along_lemma1;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  along_lemma5;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)

let reflAf = jordan_def
   `reflAf r (x:num->real) = point(&2 * (real_of_int r) - x 0, x 1)`;;

let reflAi = jordan_def
   `reflAi r (x:int#int) = ((&:2 *: r) -: FST x,SND x)`;;

let reflBf = jordan_def
   `reflBf r (x:num->real) = point( x 0 , &2 * (real_of_int r) - x 1)`;;

let reflBi = jordan_def
   `reflBi r (x:int#int) = (FST x, (&:2 *: r) -: SND x)`;;

let reflCf = jordan_def
   `reflCf  (x:num->real) = point (x 1, x 0)`;;

let reflCi = jordan_def
   `reflCi  (x:int#int) = (SND  x, FST  x)`;;

let reflAf_inv = prove_by_refinement(
  `!r m.  (reflAf r (reflAf r (point m)) = (point m))`,
  (* {{{ proof *)

  [
  REP_GEN_TAC;
  REWRITE_TAC[reflAf;coord01;PAIR_SPLIT ;point_inj ;];
  REAL_ARITH_TAC ;
  ]);;

  (* }}} *)

let reflBf_inv = prove_by_refinement(
  `!r m.  (reflBf r (reflBf r (point m)) = (point m))`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[reflBf;coord01;PAIR_SPLIT ;point_inj ;];
  REAL_ARITH_TAC ;
  ]);;
  (* }}} *)

let reflCf_inv = prove_by_refinement(
  `!m.  (reflCf  (reflCf  (point m)) = (point m))`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[reflCf;coord01;PAIR_SPLIT ;point_inj ;];
  ]);;
  (* }}} *)

let reflAi_inv = prove_by_refinement(
  `!r x.  (reflAi r (reflAi r x) = x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[reflAi;PAIR_SPLIT;];
  INT_ARITH_TAC;
  ]);;
  (* }}} *)

let reflBi_inv = prove_by_refinement(
  `!r x.  (reflBi r (reflBi r x) = x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[reflBi;PAIR_SPLIT;];
  INT_ARITH_TAC;
  ]);;
  (* }}} *)

let reflCi_inv = prove_by_refinement(
  `!x.  (reflCi  (reflCi  x) = x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[reflCi;PAIR_SPLIT;];
  ]);;
  (* }}} *)

let invo_BIJ = prove_by_refinement(
  `!f. (!m . (f (f (point m)) = (point m))) /\
        (!x. (euclid 2 (f x))) ==>
             (BIJ f (euclid 2) (euclid 2))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[BIJ;INJ;SURJ;];
  SUBCONJ_TAC;
  CONJ_TAC;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 2 (MATCH_MP (point_onto));
  USE 3 (MATCH_MP (point_onto));
  CHO 2;
  CHO 3;
  REWR 4;
  TYPE_THEN `f` (USE 4 o AP_TERM );
  REWR 4;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 4(MATCH_MP point_onto);
  CHO 4;
  ASM_REWRITE_TAC[];
  TYPE_THEN ` f (point p)` EXISTS_TAC ;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let reflA_BIJ = prove_by_refinement(
  `!r. (BIJ (reflAf r) (euclid 2) (euclid 2))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  IMATCH_MP_TAC  invo_BIJ;
  REWRITE_TAC[reflAf_inv];
  REWRITE_TAC[reflAf;euclid_point;];
  ]);;
  (* }}} *)

let reflB_BIJ = prove_by_refinement(
  `!r. (BIJ (reflBf r) (euclid 2) (euclid 2))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  IMATCH_MP_TAC  invo_BIJ;
  REWRITE_TAC[reflBf_inv];
  REWRITE_TAC[reflBf;euclid_point;];
  ]);;
  (* }}} *)

let reflC_BIJ = prove_by_refinement(
  `(BIJ (reflCf ) (euclid 2) (euclid 2))`,
  (* {{{ proof *)
  [
  IMATCH_MP_TAC  invo_BIJ;
  REWRITE_TAC[reflCf_inv];
  REWRITE_TAC[reflCf;euclid_point;];
  ]);;
  (* }}} *)

let invo_homeo = prove_by_refinement(
  `!U (f:A->A). (continuous f U U) /\ (BIJ f (UNIONS U) (UNIONS U)) /\
    (!x. (UNIONS U x ==> (f (f x ) = x))) ==> (homeomorphism f U U)`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  bicont_homeomorphism;
  ASM_REWRITE_TAC[];
  TYPE_THEN `!x. (UNIONS U x) ==> (INV f (UNIONS U) (UNIONS U) x = f x)` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  TYPE_THEN `UNIONS U (f x)` SUBGOAL_TAC;
  UND 1;
  REWRITE_TAC[BIJ;SURJ];
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_SIMP_TAC [(INR INVERSE_XY)];
  DISCH_ALL_TAC;
  UND 0;
  REWRITE_TAC[continuous];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  TSPEC `v` 0;
  REWR 0;
  UND 0;
  REWRITE_TAC[preimage];
  TYPE_THEN `{x | UNIONS U x /\ v (INV f (UNIONS U) (UNIONS U) x)} = {x | UNIONS U x /\ v (f x)}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[];
  IMATCH_MP_TAC  (TAUT `(C ==> (A <=> B)) ==> ( C /\ A <=> C /\ B)`);
  DISCH_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  ]);;

  (* }}} *)

let d_euclid_point = prove_by_refinement(
  `!r s. (d_euclid (point r) (point s) =
       sqrt ((FST r - FST s) pow 2 + ((SND r - SND s) pow 2)))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `euclid 2 (point r) /\ euclid 2 (point s)` SUBGOAL_TAC;
  REWRITE_TAC[euclid_point];
  DISCH_TAC ;
  USE 0(MATCH_MP d_euclid_n);
  ASM_REWRITE_TAC[];
  AP_TERM_TAC;
  REWRITE_TAC[ARITH_RULE `2 = SUC 1`];
  REWRITE_TAC[sum_DEF];
  REDUCE_TAC;
  REWRITE_TAC[ARITH_RULE `1 = SUC 0`];
  REWRITE_TAC[sum_DEF];
  REDUCE_TAC;
  REWRITE_TAC[ARITH_RULE `(SUC 0  =1) /\ (SUC (SUC 0) = 2)`];
  REWRITE_TAC[coord01];
  REWRITE_TAC[POW_2];
  ]);;
  (* }}} *)

let reflA_cont = prove_by_refinement(
  `!r. continuous (reflAf r) top2 top2`,
  (* {{{ proof *)
  [
  REWRITE_TAC[top2];
  GEN_TAC;
  TYPE_THEN `(IMAGE (reflAf r) (euclid 2)) SUBSET (euclid 2) /\ (metric_space (euclid 2,d_euclid))` SUBGOAL_TAC;
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_SIMP_TAC[metric_euclid];
  CONV_TAC (dropq_conv "x");
  REWRITE_TAC[reflAf;euclid_point];
  DISCH_TAC;
  ASM_SIMP_TAC[metric_continuous_continuous;metric_continuous;metric_continuous_pt;];
  DISCH_ALL_TAC;
  TYPE_THEN `epsilon` EXISTS_TAC;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 2(MATCH_MP point_onto);
  CHO 2;
  USE 3(MATCH_MP point_onto);
  CHO 3;
  UND 4;
  ASM_REWRITE_TAC[reflAf;d_euclid_point;coord01;];
  TYPE_THEN `(&2 * real_of_int r - FST p - (&2 * real_of_int r - FST p'))  = --. (FST p - FST p') ` SUBGOAL_TAC;
  REAL_ARITH_TAC ;
  DISCH_THEN_REWRITE;
  ONCE_REWRITE_TAC[GSYM REAL_POW2_ABS];
  REWRITE_TAC[ABS_NEG];
  ]);;
  (* }}} *)

let reflB_cont = prove_by_refinement(
  `!r. continuous (reflBf r) top2 top2`,
  (* {{{ proof *)
  [
  REWRITE_TAC[top2];
  GEN_TAC;
  TYPE_THEN `(IMAGE (reflBf r) (euclid 2)) SUBSET (euclid 2) /\ (metric_space (euclid 2,d_euclid))` SUBGOAL_TAC;
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_SIMP_TAC[metric_euclid];
  CONV_TAC (dropq_conv "x");
  REWRITE_TAC[reflBf;euclid_point];
  DISCH_TAC;
  ASM_SIMP_TAC[metric_continuous_continuous;metric_continuous;metric_continuous_pt;];
  DISCH_ALL_TAC;
  TYPE_THEN `epsilon` EXISTS_TAC;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 2(MATCH_MP point_onto);
  CHO 2;
  USE 3(MATCH_MP point_onto);
  CHO 3;
  UND 4;
  ASM_REWRITE_TAC[reflBf;d_euclid_point;coord01;];
  TYPE_THEN `(&2 * real_of_int r - SND  p - (&2 * real_of_int r - SND  p'))  = --. (SND  p - SND  p') ` SUBGOAL_TAC;
  REAL_ARITH_TAC ;
  DISCH_THEN_REWRITE;
  ONCE_REWRITE_TAC[GSYM REAL_POW2_ABS];
  REWRITE_TAC[ABS_NEG];
  ]);;
  (* }}} *)

let reflC_cont = prove_by_refinement(
  ` continuous (reflCf) top2 top2`,
  (* {{{ proof *)
  [
  REWRITE_TAC[top2];
  TYPE_THEN `(IMAGE (reflCf) (euclid 2)) SUBSET (euclid 2) /\ (metric_space (euclid 2,d_euclid))` SUBGOAL_TAC;
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_SIMP_TAC[metric_euclid];
  CONV_TAC (dropq_conv "x");
  REWRITE_TAC[reflCf;euclid_point];
  DISCH_TAC;
  ASM_SIMP_TAC[metric_continuous_continuous;metric_continuous;metric_continuous_pt;];
  DISCH_ALL_TAC;
  TYPE_THEN `epsilon` EXISTS_TAC;
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 2(MATCH_MP point_onto);
  CHO 2;
  USE 3(MATCH_MP point_onto);
  CHO 3;
  UND 4;
  ASM_REWRITE_TAC[reflCf;d_euclid_point;coord01;];
  REWRITE_TAC[REAL_ADD_AC];
  ]);;
  (* }}} *)

let reflA_homeo = prove_by_refinement(
  `!r. (homeomorphism (reflAf r) top2 top2)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  ASSUME_TAC reflA_BIJ;
  ASSUME_TAC top2_unions;
  IMATCH_MP_TAC  invo_homeo;
  REWRITE_TAC[reflA_cont];
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 2(MATCH_MP   point_onto);
  CHO 2;
  ASM_REWRITE_TAC[reflAf_inv];
  ]);;
  (* }}} *)

let reflB_homeo = prove_by_refinement(
  `!r. (homeomorphism (reflBf r) top2 top2)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  ASSUME_TAC reflB_BIJ;
  ASSUME_TAC top2_unions;
  IMATCH_MP_TAC  invo_homeo;
  REWRITE_TAC[reflB_cont];
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 2(MATCH_MP   point_onto);
  CHO 2;
  ASM_REWRITE_TAC[reflBf_inv];
  ]);;
  (* }}} *)

let reflC_homeo = prove_by_refinement(
  ` (homeomorphism (reflCf ) top2 top2)`,
  (* {{{ proof *)
  [
  ASSUME_TAC reflC_BIJ;
  ASSUME_TAC top2_unions;
  IMATCH_MP_TAC  invo_homeo;
  REWRITE_TAC[reflC_cont];
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 2(MATCH_MP   point_onto);
  CHO 2;
  ASM_REWRITE_TAC[reflCf_inv];
  ]);;
  (* }}} *)

let IMAGE2 = new_definition
   `IMAGE2 (f:A->B) U = IMAGE (IMAGE (f:A->B)) U`;;

let reflA_h_edge = prove_by_refinement(
  `!m r.  IMAGE (reflAf r) (h_edge m) = h_edge (left  (reflAi r m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge;reflAf;reflAi;IMAGE ;left  ;];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[h_edge];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "x'");
  CONV_TAC (dropq_conv "v");
  REWRITE_TAC[coord01];
  EQ_TAC;
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `&2 * real_of_int r - u` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 0;
  ASM_REWRITE_TAC[int_sub_th;int_of_num_th;int_add_th;int_mul_th;];
  DISCH_ALL_TAC;
  UND 0;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `&2 * real_of_int r - u` EXISTS_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `y - (y - x) = x`];
  UND 0;
  ASM_REWRITE_TAC[int_sub_th;int_of_num_th;int_add_th;int_mul_th;];
  DISCH_ALL_TAC;
  UND 2;
  UND 1;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let reflA_v_edge = prove_by_refinement(
  `!m r.  IMAGE (reflAf r) (v_edge m) = v_edge (  (reflAi r m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge;reflAf;reflAi;IMAGE ;left  ;];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[v_edge];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "x'");
  CONV_TAC (dropq_conv "u");
  REWRITE_TAC[coord01];
  REWRITE_TAC[int_sub_th;int_mul_th;int_of_num_th;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let reflA_edge = prove_by_refinement(
  `!r e. (edge e ==> edge (IMAGE (reflAf r) e))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge];
  DISCH_ALL_TAC;
  CHO 0;
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  MESON_TAC[reflA_v_edge];
  ASM_REWRITE_TAC[];
  MESON_TAC[reflA_h_edge];
  ]);;
  (* }}} *)

let reflB_v_edge = prove_by_refinement(
  `!m r.  IMAGE (reflBf r) (v_edge m) = v_edge (down  (reflBi r m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge;reflBf;reflBi;IMAGE ;down  ;];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[v_edge];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "x'");
  CONV_TAC (dropq_conv "u");
  REWRITE_TAC[coord01];
  EQ_TAC;
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `&2 * real_of_int r - v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 0;
  ASM_REWRITE_TAC[int_sub_th;int_of_num_th;int_add_th;int_mul_th;];
  DISCH_ALL_TAC;
  UND 0;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN CHOOSE_TAC;
  TYPE_THEN `&2 * real_of_int r - v` EXISTS_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `y - (y - x) = x`];
  UND 0;
  ASM_REWRITE_TAC[int_sub_th;int_of_num_th;int_add_th;int_mul_th;];
  DISCH_ALL_TAC;
  UND 2;
  UND 1;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let reflB_h_edge = prove_by_refinement(
  `!m r.  IMAGE (reflBf r) (h_edge m) = h_edge (  (reflBi r m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge;reflBf;reflBi;IMAGE ;down  ;];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[h_edge];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "x'");
  CONV_TAC (dropq_conv "v");
  REWRITE_TAC[coord01];
  REWRITE_TAC[int_sub_th;int_mul_th;int_of_num_th;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let reflB_edge = prove_by_refinement(
  `!r e. (edge e ==> edge (IMAGE (reflBf r) e))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge];
  DISCH_ALL_TAC;
  CHO 0;
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  MESON_TAC[reflB_v_edge];
  ASM_REWRITE_TAC[];
  MESON_TAC[reflB_h_edge];
  ]);;
  (* }}} *)

let reflC_vh_edge = prove_by_refinement(
  `!m .  IMAGE (reflCf) (v_edge m) = h_edge ( (reflCi m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge;reflCf;reflCi;IMAGE ;down  ;];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[v_edge;h_edge];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "x'");
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  REWRITE_TAC[coord01];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let reflC_hv_edge = prove_by_refinement(
  `!m .  IMAGE (reflCf) (h_edge m) = v_edge ( (reflCi m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge;reflCf;reflCi;IMAGE ;down  ;];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[v_edge;h_edge];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "x'");
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  REWRITE_TAC[coord01];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let reflC_edge = prove_by_refinement(
  `!e. (edge e ==> edge (IMAGE (reflCf ) e))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[edge];
  DISCH_ALL_TAC;
  CHO 0;
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  MESON_TAC[reflC_vh_edge];
  ASM_REWRITE_TAC[];
  MESON_TAC[reflC_hv_edge];
  ]);;
  (* }}} *)

let homeo_bij = prove_by_refinement(
  `!(f:A->B) U V. (homeomorphism f U V) ==> (BIJ (IMAGE f) U V)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[BIJ;homeomorphism;continuous;preimage;];
  DISCH_ALL_TAC;
  SUBCONJ_TAC;
  REWRITE_TAC[INJ];
  ASM_REWRITE_TAC[IMAGE;];
  DISCH_ALL_TAC;
  TAPP `u:B` 6;
  USE 6 (REWRITE_RULE[]);
  USE 6(CONV_RULE NAME_CONFLICT_CONV);
  IMATCH_MP_TAC  EQ_EXT;
  USE 6 (GEN `u:B`);
  GEN_TAC;
  COPY 6;
  EQ_TAC;
  DISCH_TAC;
  TSPEC `f x'` 7;
  TYPE_THEN `(?x''. x x'' /\ (f x' = f x''))` SUBGOAL_TAC;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  UND 7;
  KILL 6;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  CHO 6;
  CHO 9;
  TYPE_THEN `(UNIONS U) x'' /\ (UNIONS U) x'''` SUBGOAL_TAC;
  REWRITE_TAC[UNIONS;];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `(UNIONS U x')` SUBGOAL_TAC;
  REWRITE_TAC[UNIONS;];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `x' = x'''` SUBGOAL_TAC;
  USE 0(REWRITE_RULE[INJ]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `x' = x''` SUBGOAL_TAC;
  USE 0(REWRITE_RULE[INJ]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_MESON_TAC[];
  (* mm *)
  DISCH_TAC;
  TSPEC `f x'` 7;
  TYPE_THEN `(?x''. y x'' /\ (f x' = f x''))` SUBGOAL_TAC;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  UND 7;
  KILL 6;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  CHO 6;
  CHO 9;
  TYPE_THEN `(UNIONS U) x'' /\ (UNIONS U) x'''` SUBGOAL_TAC;
  REWRITE_TAC[UNIONS;];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `(UNIONS U x')` SUBGOAL_TAC;
  REWRITE_TAC[UNIONS;];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `x' = x'''` SUBGOAL_TAC;
  USE 0(REWRITE_RULE[INJ]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `x' = x''` SUBGOAL_TAC;
  USE 0(REWRITE_RULE[INJ]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_MESON_TAC[];
  REWRITE_TAC[INJ;SURJ];
  DISCH_ALL_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  TYPE_THEN `{z | UNIONS U z /\ x (f z)}` EXISTS_TAC;
  CONJ_TAC;
  UND 2;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[IMAGE;SUBSET ;];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x''");
  MESON_TAC[];
  REWRITE_TAC[SUBSET;IMAGE];
  DISCH_ALL_TAC;
  NAME_CONFLICT_TAC;
  UND 1;
  REWRITE_TAC[SURJ];
  DISCH_ALL_TAC;
  TSPEC `x'` 8;
  TYPE_THEN `UNIONS V x'` SUBGOAL_TAC;
  REWRITE_TAC[UNIONS;];
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 8;
  CHO 8;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let homeo_unions = prove_by_refinement(
  `!(f:A->B) U V. (homeomorphism f U V) ==>
      (IMAGE f (UNIONS U) = (UNIONS V))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[homeomorphism;BIJ;SURJ;IMAGE;];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  GEN_TAC;
  NAME_CONFLICT_TAC;
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 5;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TSPEC `x` 2;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let homeo_closed = prove_by_refinement(
  `!(f:A->B) U V A. (homeomorphism f U V /\ (A SUBSET (UNIONS U)) ==>
    (closed_ V (IMAGE f A) = closed_ U A))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
   TYPE_THEN `BIJ f (UNIONS U) (UNIONS V)` SUBGOAL_TAC;
  ASM_MESON_TAC[homeomorphism];
  DISCH_TAC;
  USE 2(MATCH_MP DIFF_SURJ);
  TSPEC `A` 2;
  REWR 2;
  ASM_REWRITE_TAC[closed;open_DEF];
  EQ_TAC;
  DISCH_ALL_TAC;
  USE 0(REWRITE_RULE[homeomorphism;continuous]);
  UND 0;
  DISCH_ALL_TAC;
  USE 2 SYM;
  REWR 4;
  TSPEC `IMAGE f (UNIONS U DIFF A)` 5;
  REWR 5;
  TYPE_THEN `preimage (UNIONS U) f (IMAGE f (UNIONS U DIFF A)) = UNIONS U DIFF A` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT ;
  GEN_TAC;
  REWRITE_TAC[INR in_preimage;IMAGE;DIFF;];
  USE 0(REWRITE_RULE[BIJ;INJ]);
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 8;
  ASM_MESON_TAC[];
  MESON_TAC[];
  DISCH_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  USE 0 (REWRITE_RULE[homeomorphism;BIJ;SURJ]);
  REWRITE_TAC[IMAGE;SUBSET];
  GEN_TAC;
  NAME_CONFLICT_TAC;
  UND 1;
  REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[];
  USE 0(REWRITE_RULE[homeomorphism]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)


(* ------------------------------------------------------------------ *)
(* SECTION G *)
(* ------------------------------------------------------------------ *)


let IMAGE_INTERS = prove_by_refinement(
  `!(f:A->B) A X . (INJ f X UNIV) /\ (UNIONS A SUBSET X) /\
     ~(A = EMPTY) ==>
   ((IMAGE f) (INTERS A) = (INTERS (IMAGE2 f A)))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  REWRITE_TAC[IMAGE2;INTERS;IMAGE;];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  EQ_TAC;
  DISCH_ALL_TAC;
  CHO 3;
  AND 3;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  CHO 5;
  AND 5;
  ASM_REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_MESON_TAC[];
  DISCH_ALL_TAC;
  USE 3 (CONV_RULE (dropq_conv "u'"));
  USE 3 (CONV_RULE (dropq_conv "y'"));
  USE 2(REWRITE_RULE[EMPTY_EXISTS]);
  CHO 2;
  COPY 3;
  TSPEC `u` 3;
  CHO 3;
  REWR 3;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  USE 0(REWRITE_RULE[INJ]);
  TSPEC `u'` 4;
  CHO 4;
  REWR 4;
  TYPEL_THEN [`x'`;`x''`] (USE 0 o ISPECL);
  USE 1(REWRITE_RULE[UNIONS;ISUBSET]);
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let homeo_closure = prove_by_refinement(
  `!(f:A->B) U V A. (homeomorphism f U V) /\ (A SUBSET (UNIONS U)) /\
     (topology_ U)  ==>
     (IMAGE f (closure U A) = closure V (IMAGE f A))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  REWRITE_TAC[closure];
  TYPE_THEN `INJ f (UNIONS U) (UNIV)` SUBGOAL_TAC;
  USE 0(REWRITE_RULE[homeomorphism;BIJ;INJ;]);
  ASM_REWRITE_TAC[INJ];
  DISCH_TAC;
  TYPE_THEN `C = {B | closed_ U B /\ A SUBSET B}` ABBREV_TAC ;
  TYPE_THEN `(UNIONS C SUBSET UNIONS U)` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET;];
  EXPAND_TAC "C";
  REWRITE_TAC[closed];
  TYPE_THEN `X = UNIONS U` ABBREV_TAC ;
  REWRITE_TAC[UNIONS];
  MESON_TAC[ISUBSET];
  DISCH_TAC;
  TYPE_THEN `~(C = EMPTY)` SUBGOAL_TAC;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `UNIONS U` EXISTS_TAC;
  EXPAND_TAC "C";
  ASM_REWRITE_TAC[closed; ISUBSET; DIFF_EQ_EMPTY;];
  ASM_SIMP_TAC[INR open_EMPTY];
  DISCH_TAC;
  JOIN 5 6;
  JOIN 3 5;
  USE 3 (MATCH_MP IMAGE_INTERS);
  ASM_REWRITE_TAC[];
  AP_TERM_TAC;
  REWRITE_TAC[IMAGE2];
  EXPAND_TAC "C";
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  TYPE_THEN `g = IMAGE f` ABBREV_TAC ;
  REWRITE_TAC[IMAGE];
  NAME_CONFLICT_TAC;
  EQ_TAC;
  DISCH_THEN CHOOSE_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "g";
  KILL 5;
  TYPE_THEN `x' SUBSET (UNIONS U)` SUBGOAL_TAC;
  USE 6(REWRITE_RULE[closed]);
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[homeo_closed];
  DISCH_TAC;
  REWRITE_TAC[ISUBSET;IMAGE];
  NAME_CONFLICT_TAC;
  ASM_MESON_TAC[ISUBSET];
  DISCH_ALL_TAC;
  TYPE_THEN `preimage (UNIONS U) f x` EXISTS_TAC;
  TYPE_THEN `x = g (preimage (UNIONS U) f x)` SUBGOAL_TAC;
  REWRITE_TAC[preimage];
  EXPAND_TAC "g";
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  EQ_TAC;
  DISCH_TAC;
  REWRITE_TAC[IMAGE];
  NAME_CONFLICT_TAC;
  USE 0 (REWRITE_RULE[homeomorphism;BIJ;SURJ]);
  UND 0;
  DISCH_ALL_TAC;
  TSPEC `x'` 10;
  TYPE_THEN `UNIONS V x'` SUBGOAL_TAC;
  USE 6(REWRITE_RULE[closed]);
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  REWR 10;
  ASM_MESON_TAC[];
  REWRITE_TAC[IMAGE];
  DISCH_THEN CHOOSE_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  USE 8 (SYM);
  ONCE_ASM_REWRITE_TAC[];
  REWRITE_TAC[];
  CONJ_TAC;
  TYPE_THEN `preimage (UNIONS U) f x SUBSET (UNIONS U)` SUBGOAL_TAC;
  REWRITE_TAC[preimage;SUBSET;];
  MESON_TAC[];
  ASM_SIMP_TAC[GSYM homeo_closed];
  REWRITE_TAC[preimage;SUBSET];
  DISCH_ALL_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[ISUBSET];
  UND 7;
  EXPAND_TAC "g";
  REWRITE_TAC[IMAGE;ISUBSET;];
  UND 9;
  MESON_TAC[];
  ]);;

  (* }}} *)

let INJ_IMAGE = prove_by_refinement(
  `!(f :A->B) A B X . (A SUBSET X) /\ (B SUBSET X) /\
     (INJ f X UNIV) ==> ((IMAGE f A = IMAGE f B) <=> (A = B))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  EQ_TAC;
  DISCH_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  RULE_ASSUM_TAC (REWRITE_RULE[IMAGE]);
  TAPP `y:B` 3;
  RULE_ASSUM_TAC  (REWRITE_RULE[]);
  USE 3(GEN `y:B`);
  REWRITE_TAC[SUBSET];
  PROOF_BY_CONTR_TAC;
  USE 4(REWRITE_RULE [DE_MORGAN_THM]);
  FIRST_ASSUM (DISJ_CASES_TAC);

  LEFT  5 "x";
  REP_BASIC_TAC;
  TSPEC `f x ` 3;
  TYPE_THEN `A x` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `(?x'. A x' /\ (f x = f x'))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `(?x'. B x' /\ (f x = f x'))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REP_BASIC_TAC;
  USE 0(REWRITE_RULE[BIJ;INJ]);
  TYPE_THEN `x = x'` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[];

  LEFT  5 "x";
  REP_BASIC_TAC;
  TSPEC `f x ` 3;
  TYPE_THEN `B x` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `(?x'. B x' /\ (f x = f x'))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `(?x'. A x' /\ (f x = f x'))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REP_BASIC_TAC;
  USE 0(REWRITE_RULE[BIJ;INJ]);
  TYPE_THEN `x = x'` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  ]);;
  (* }}} *)

let INJ_UNIV = prove_by_refinement(
  `!(f: A->B) X Y. (INJ f X Y) ==> (INJ f X UNIV)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ];
  REP_BASIC_TAC;
  ASM_MESON_TAC [];
  ]);;
  (* }}} *)

let homeo_adj = prove_by_refinement(
  `!f X Y.  (homeomorphism f top2 top2) /\ (X SUBSET euclid 2) /\
       (Y SUBSET euclid 2)
       ==> (adj X Y ==> (adj (IMAGE f X) (IMAGE f Y)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[adj;INTER;EMPTY_EXISTS];
  REP_BASIC_TAC;
  ASSUME_TAC top2_top;
  ASSUME_TAC top2_unions;
  TYPE_THEN `X SUBSET (UNIONS top2) /\ Y SUBSET (UNIONS (top2))` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `closure top2 (IMAGE f X) = IMAGE f (closure top2 X)` SUBGOAL_TAC;
  ASM_MESON_TAC[GSYM homeo_closure];
  DISCH_THEN_REWRITE;
  TYPE_THEN `closure top2 (IMAGE f Y) = IMAGE f (closure top2 Y)` SUBGOAL_TAC;
  ASM_MESON_TAC[GSYM homeo_closure];
  DISCH_THEN_REWRITE;
  REP_BASIC_TAC;
  CONJ_TAC;
  PROOF_BY_CONTR_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[]);
  UND 2;
  REWRITE_TAC[];
  UND 10;
  TYPE_THEN `INJ f (euclid 2) UNIV` SUBGOAL_TAC;
  IMATCH_MP_TAC  INJ_UNIV;
  RULE_ASSUM_TAC  (REWRITE_RULE[homeomorphism;BIJ]);
  REP_BASIC_TAC;
  REWR 11;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  ASM_MESON_TAC[INJ_IMAGE];
  (* done WITH both *)
  TYPE_THEN `f u` EXISTS_TAC;
  REWRITE_TAC[IMAGE];
  ASM_MESON_TAC[];
  (* converse *)
  ]);;
  (* }}} *)

let homeomorphism_inv = prove_by_refinement(
  `!(f:A->B) U V. homeomorphism f U V ==>
    (homeomorphism (INV f (UNIONS U) (UNIONS V)) V U)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[homeomorphism];
  ASM_SIMP_TAC[INV_homeomorphism];
  USE 0(REWRITE_RULE [homeomorphism;continuous;]);
  REP_BASIC_TAC;
  ASM_SIMP_TAC[INVERSE_BIJ];
  REP_BASIC_TAC;
  TSPEC `A` 1;
  REWR 1;
  TYPE_THEN `g = INV f (UNIONS U) (UNIONS V)` ABBREV_TAC ;
  TYPE_THEN `BIJ g (UNIONS V) (UNIONS U)` SUBGOAL_TAC;
  EXPAND_TAC "g";
  IMATCH_MP_TAC  INVERSE_BIJ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `!x'. (A x' ==> (f (g x') = x'))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPEL_THEN  [`f`;`UNIONS U`;`UNIONS V`] (fun t->  ASSUME_TAC (ISPECL  t (INR INVERSE_DEF)));
  RULE_ASSUM_TAC  (REWRITE_RULE[BIJ]);
  REWR 6;
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC  ;
  REWRITE_TAC[UNIONS];
  ASM_MESON_TAC[];
  DISCH_TAC;
  DISCH_TAC;
  (* branch *)
  TYPE_THEN `(IMAGE g A) = preimage (UNIONS U) f A` SUBGOAL_TAC;
  REWRITE_TAC[IMAGE;preimage];
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  EQ_TAC;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[];
  EXPAND_TAC "g";
  USE 2(MATCH_MP   INVERSE_BIJ);
  RULE_ASSUM_TAC  (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC [UNIONS];
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `f x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `f x = f (g (f x))` SUBGOAL_TAC;
  ASM_SIMP_TAC[];
  DISCH_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[BIJ;INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  USE 9 SYM;
  ASM_REWRITE_TAC[];
  TYPE_THEN `UNIONS V (f x)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let inv_comp_left = prove_by_refinement(
  `!(f:A->B) X Y x.  (BIJ f X Y /\ X x) ==> (INV f X Y (f x) = x)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `Y (f x)` SUBGOAL_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[BIJ;SURJ]);
  ASM_MESON_TAC[];
  ASM_MESON_TAC[INR INVERSE_XY];
  ]);;
  (* }}} *)

let inv_comp_right = prove_by_refinement(
  `!(f:A->B) X Y y. (BIJ f X Y /\ Y y) ==> (f (INV f X Y y) = y)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[BIJ]);
  ASM_MESON_TAC[INR INVERSE_DEF;];
  ]);;
  (* }}} *)

let image_inv_image = prove_by_refinement(
  `!(f:A->B) A X Y. (BIJ f X Y) /\ (A SUBSET X) ==>
    (IMAGE (INV f X Y) (IMAGE f A) = A)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE];
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x''");
  EQ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `x = x'` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC [inv_comp_left;ISUBSET;];
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ONCE_REWRITE_TAC[EQ_SYM_EQ];
  IMATCH_MP_TAC  inv_comp_left;
  ASM_MESON_TAC[ISUBSET];
  ]);;
  (* }}} *)

let homeo_adj_eq = prove_by_refinement(
  `!f X Y. (homeomorphism f top2 top2) /\ (X SUBSET euclid 2) /\
       (Y SUBSET euclid 2)
       ==> (adj X Y = (adj (IMAGE f X) (IMAGE f Y)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  EQ_TAC;
  ASM_MESON_TAC[homeo_adj];
  TYPEL_THEN  [`INV f (euclid 2) (euclid 2)`;`IMAGE f X`;`IMAGE f Y`] (fun t-> MP_TAC (ISPECL t homeo_adj));
  ASSUME_TAC top2_unions;
  TYPE_THEN `homeomorphism (INV f (euclid 2) (euclid 2)) top2 top2` SUBGOAL_TAC;
  ASM_MESON_TAC[homeomorphism_inv];
  DISCH_THEN_REWRITE;
  TYPE_THEN `BIJ f (euclid 2) (euclid 2)` SUBGOAL_TAC;
  ASM_MESON_TAC[homeomorphism];
  DISCH_TAC;
  ASM_SIMP_TAC[image_inv_image];
  REP_BASIC_TAC;
  TYPE_THEN `IMAGE f X SUBSET euclid 2 /\ IMAGE f Y SUBSET euclid 2` SUBGOAL_TAC;
  REWRITE_TAC[IMAGE;SUBSET];
  NAME_CONFLICT_TAC;
  CONJ_TAC THEN (CONV_TAC (dropq_conv "x''")) THEN (RULE_ASSUM_TAC  (REWRITE_RULE[BIJ;SURJ]));
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let finite_num_closure = prove_by_refinement(
  `!G top (x:A). FINITE G ==> (FINITE {C | G C /\ closure top C x})`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  IMATCH_MP_TAC FINITE_SUBSET;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let image_powerset = prove_by_refinement(
  `!(f:A->B) X Y. (BIJ f X Y ==>
     (BIJ (IMAGE f) {z | z SUBSET X} { z | z SUBSET Y}))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[BIJ];
  SUBCONJ_TAC;
  REWRITE_TAC[INJ];
  REP_BASIC_TAC;
  CONJ_TAC;
  REP_BASIC_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC ;
  REWRITE_TAC[IMAGE;SUBSET;];
  ASM_MESON_TAC[ISUBSET ;];
  REWRITE_TAC[IMAGE;SUBSET;];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[BIJ;INJ]);
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;

  TAPP `z:B` 1;
  USE 1(REWRITE_RULE[]);
  USE 1(GEN `z:B`);
  EQ_TAC;
  TSPEC `f x'` 1;
  REP_BASIC_TAC;
  UND 1;
  NAME_CONFLICT_TAC;
  TYPE_THEN `(?x''. x x'' /\ (f x' = f x''))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  REP_BASIC_TAC;
  TYPE_THEN `x' = x''` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  (* 2 *)
  TSPEC `f x'` 1;
  REP_BASIC_TAC;
  UND 1;
  NAME_CONFLICT_TAC;
  TYPE_THEN `(?x''. y x'' /\ (f x' = f x''))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  REP_BASIC_TAC;
  TYPE_THEN `x' = x''` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  REWRITE_TAC[INJ;SURJ];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `{z | X z /\ x (f z) }` EXISTS_TAC;
  SUBCONJ_TAC;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  REWRITE_TAC[IMAGE];
  IMATCH_MP_TAC  EQ_EXT ;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  EQ_TAC;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC;
  TSPEC `x'` 0;
  USE 3(REWRITE_RULE[SUBSET]);
  TSPEC  `x'` 3;
  REWR 3;
  REWR 0;
  REP_BASIC_TAC;
  TYPE_THEN `y` EXISTS_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let image_power_inj = prove_by_refinement(
  `!(f:A->B) X Y A B. (BIJ f X Y /\ A SUBSET X /\ B SUBSET X ==>
     ((IMAGE f A = IMAGE f B) <=> (A = B)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPEL_THEN [`f`;`X`;`Y`]  (fun t -> ASSUME_TAC (ISPECL t image_powerset ));
  REWR 3;
  USE 3(REWRITE_RULE[BIJ;INJ;]);
  REP_BASIC_TAC;
  EQ_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  ]);;
  (* }}} *)

let image_power_surj = prove_by_refinement(
  `!(f:A->B) X Y B. (BIJ f X Y /\ B SUBSET Y ==>
    (?A. (A SUBSET X /\ (IMAGE f A = B))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPEL_THEN [`f`;`X`;`Y`]  (fun t -> ASSUME_TAC (ISPECL t image_powerset ));
  REWR 2;
  USE 2(REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let segment_euclid = prove_by_refinement(
  `!G e. (segment G /\ G e) ==> (e SUBSET (euclid 2))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  REP_BASIC_TAC;
  USE 3(REWRITE_RULE[SUBSET]);
  TSPEC `e` 3;
  REWR 3;
  USE 3(REWRITE_RULE[edge]);
  REP_BASIC_TAC;
  ASM_MESON_TAC[h_edge_euclid;v_edge_euclid];
  ]);;
  (* }}} *)

let image_app = prove_by_refinement(
  `!(f:A->B) X Y x t. INJ f X Y /\ x SUBSET X /\ (X t) ==>
   (IMAGE f x (f t) = x t)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ;IMAGE;SUBSET ;];
  REP_BASIC_TAC;
  EQ_TAC;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let homeo_num_closure = prove_by_refinement(
  `!G f m. (homeomorphism f top2 top2 /\ segment G) ==>
   (num_closure G (pointI m) =
           (num_closure (IMAGE2 f G) (f (pointI m))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  ASSUME_TAC top2_unions;
  ASSUME_TAC top2_top;
  TYPE_THEN `BIJ f (euclid 2) (euclid 2)` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism]);
  ASM_MESON_TAC [];
  DISCH_TAC;
  TYPE_THEN `G` (fun t-> ASSUME_TAC (ISPEC t segment_euclid));
  REWRITE_TAC[num_closure];
  IMATCH_MP_TAC  BIJ_CARD;
  TYPE_THEN `IMAGE f` EXISTS_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  finite_num_closure;
  ASM_MESON_TAC[segment_finite];
  REWRITE_TAC[BIJ];
  SUBCONJ_TAC;
  REWRITE_TAC[INJ];
  REP_BASIC_TAC;
  CONJ_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE2];
  CONJ_TAC;
  REWRITE_TAC[IMAGE];
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `x SUBSET (UNIONS top2)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `IMAGE f (closure top2 x) = closure top2 (IMAGE f x)` SUBGOAL_TAC;
  ASM_MESON_TAC [homeo_closure];
  DISCH_THEN (fun t->REWRITE_TAC[GSYM t]);
  REWRITE_TAC[IMAGE];
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `x SUBSET (euclid 2) /\ y SUBSET (euclid 2)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_MESON_TAC[image_power_inj];
  REWRITE_TAC[INJ;SURJ];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[IMAGE2]);
  UND 9;
  TYPE_THEN `g = IMAGE f` ABBREV_TAC ;
  REWRITE_TAC[IMAGE];
  EXPAND_TAC "g";
  REP_BASIC_TAC;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWR 8;
  UND 8;
  TYPE_THEN `x' SUBSET (UNIONS top2)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `closure top2 (g x') = IMAGE f (closure top2 x')` SUBGOAL_TAC;
  ASM_MESON_TAC [GSYM homeo_closure];
  DISCH_THEN_REWRITE;
  (* m3 *)
  TYPE_THEN `INJ f (euclid 2) (euclid 2) /\ (closure top2 x' SUBSET (euclid 2)) /\ (euclid 2 (pointI m))` SUBGOAL_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[BIJ]);
   ASM_REWRITE_TAC[pointI;euclid_point];
  IMATCH_MP_TAC  c_edge_euclid;
  ASM_MESON_TAC[segment;ISUBSET];
  DISCH_TAC;
  USE 12 (MATCH_MP image_app);
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* SECTION H *)
(* ------------------------------------------------------------------ *)

let reflA_pointI = prove_by_refinement(
  `!r m. (reflAf r (pointI m) = pointI (reflAi r m))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[reflAi;reflAf;pointI];
  REWRITE_TAC[point_inj;PAIR_SPLIT;];
  REWRITE_TAC[int_of_num_th;int_add_th;int_mul_th;int_sub_th;coord01];
  ]);;
  (* }}} *)

let reflB_pointI = prove_by_refinement(
  `!r m. (reflBf r (pointI m) = pointI (reflBi r m))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[reflBi;reflBf;pointI];
  REWRITE_TAC[point_inj;PAIR_SPLIT;];
  REWRITE_TAC[int_of_num_th;int_add_th;int_mul_th;int_sub_th;coord01];
  ]);;
  (* }}} *)

let reflC_pointI = prove_by_refinement(
  `!m. (reflCf  (pointI m) = pointI (reflCi m))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[reflCi;reflCf;pointI];
  REWRITE_TAC[point_inj;PAIR_SPLIT;];
  REWRITE_TAC[int_of_num_th;int_add_th;int_mul_th;int_sub_th;coord01];
  ]);;
  (* }}} *)

let edge_euclid2 = prove_by_refinement(
  `!e. (edge e ==> e SUBSET (euclid 2))`,
  (* {{{ proof *)
  [
  MESON_TAC [edge;h_edge_euclid;v_edge_euclid;];
  ]);;
  (* }}} *)

let reflA_segment = prove_by_refinement(
  `!G r. (segment G ==> (segment (IMAGE2 (reflAf r) G)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[segment];
  COPY 0;
  USE 0(REWRITE_RULE[segment]);
  REP_BASIC_TAC;
  TYPE_THEN `homeomorphism (reflAf r) top2 top2` SUBGOAL_TAC;
  REWRITE_TAC[reflA_homeo];
  DISCH_TAC;
  ASSUME_TAC top2_top;
  ASSUME_TAC top2_unions;
  TYPE_THEN `BIJ (reflAf r) (euclid 2) (euclid 2)` SUBGOAL_TAC;
  ASM_MESON_TAC[homeomorphism];
  DISCH_TAC;
  TYPE_THEN `INJ (IMAGE (reflAf r)) edge edge` SUBGOAL_TAC;
  REWRITE_TAC[INJ;reflA_edge;];
  REP_BASIC_TAC;
  TYPE_THEN `y SUBSET (euclid 2) /\ x SUBSET (euclid 2)` SUBGOAL_TAC;
  ASM_MESON_TAC[edge_euclid2];
  DISCH_TAC;
  ASM_MESON_TAC[image_power_inj];
  DISCH_TAC;
  (* start cases *)
  SUBCONJ_TAC;
  REWRITE_TAC[IMAGE2];
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  SUBCONJ_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[IMAGE2; EQ_EMPTY]);
  TSPEC `IMAGE (reflAf r) u` 4;
  UND 4;
  REWRITE_TAC[];
  TYPE_THEN `IMAGE (IMAGE (reflAf r)) G (IMAGE (reflAf r) u) = G u` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_app;
  EXISTS_TAC `edge`;
  EXISTS_TAC `edge`;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (*
  ASM_MESON_TAC[image_power_inj];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[];
  DISCH_TAC;
  *)
  SUBCONJ_TAC;
  REWRITE_TAC[IMAGE2;SUBSET];
  GEN_TAC;
  GEN_REWRITE_TAC (LAND_CONV o ONCE_DEPTH_CONV )  [IMAGE];
  REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  reflA_edge;
  ASM_MESON_TAC[ISUBSET;];
  DISCH_TAC;
  (* num closure clause *)
  CONJ_TAC;
  GEN_TAC;
  TYPE_THEN `pointI m = reflAf r (pointI (reflAi r m))` SUBGOAL_TAC;
  REWRITE_TAC[reflA_pointI;reflAi_inv];
  DISCH_THEN_REWRITE;
  TYPE_THEN `num_closure (IMAGE2 (reflAf r) G) (reflAf r (pointI (reflAi r m))) = num_closure G (pointI (reflAi r m))` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_num_closure);
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[];
  (* inductive_set clause *)
  REP_BASIC_TAC;
  (* isc *)
  USE 16(REWRITE_RULE[IMAGE2]);
  USE 16 (MATCH_MP SUBSET_PREIMAGE);
  REP_BASIC_TAC;
  TSPEC `Z` 0;
  TYPE_THEN `Z SUBSET G /\ ~(Z = {}) /\ (!C C'. Z C /\ G C' /\ adj C C' ==> Z C')` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  PROOF_BY_CONTR_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[]);
  REWR 16;
  RULE_ASSUM_TAC  (REWRITE_RULE[IMAGE_CLAUSES]);
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `D = IMAGE (reflAf r) C` ABBREV_TAC ;
  TYPE_THEN `D' = IMAGE (reflAf r) C'` ABBREV_TAC ;
  TSPEC `D` 14; (* *)
  TSPEC `D'` 14;
  TYPE_THEN `S D /\ IMAGE2 (reflAf r) G D' /\ adj D D'` SUBGOAL_TAC;
  SUBCONJ_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "D";
  TYPE_THEN `IMAGE (IMAGE (reflAf r)) Z (IMAGE (reflAf r) C) = Z C` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_app;
  TYPE_THEN `edge` EXISTS_TAC;
  TYPE_THEN `edge` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* fh1 *)
  SUBCONJ_TAC;
  EXPAND_TAC "D'";
  REWRITE_TAC[IMAGE2;IMAGE];
  NAME_CONFLICT_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  EXPAND_TAC "D";
  EXPAND_TAC "D'";
  TYPE_THEN `C SUBSET (euclid 2) /\ (C' SUBSET (euclid 2))` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET;edge_euclid2];
  DISCH_TAC;
  TYPE_THEN `(adj C C' ==> adj (IMAGE (reflAf r) C) (IMAGE (reflAf r) C'))` SUBGOAL_TAC;
  IMATCH_MP_TAC  homeo_adj;
  ASM_REWRITE_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWR 14;
  UND 14;
  EXPAND_TAC "D'";
  TYPE_THEN `IMAGE (IMAGE (reflAf r)) Z (IMAGE (reflAf r) C') = Z C'` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_app;
  TYPE_THEN `edge` EXISTS_TAC;
  TYPE_THEN `edge` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  UND 3;
  UND 19;
  ASM_MESON_TAC[ISUBSET];
  MESON_TAC[];
  DISCH_TAC;
  REWR 0;
  ASM_REWRITE_TAC[IMAGE2];
  ]);;
  (* }}} *)

let reflB_segment = prove_by_refinement(
  `!G r. (segment G ==> (segment (IMAGE2 (reflBf r) G)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[segment];
  COPY 0;
  USE 0(REWRITE_RULE[segment]);
  REP_BASIC_TAC;
  TYPE_THEN `homeomorphism (reflBf r) top2 top2` SUBGOAL_TAC;
  REWRITE_TAC[reflB_homeo];
  DISCH_TAC;
  ASSUME_TAC top2_top;
  ASSUME_TAC top2_unions;
  TYPE_THEN `BIJ (reflBf r) (euclid 2) (euclid 2)` SUBGOAL_TAC;
  ASM_MESON_TAC[homeomorphism];
  DISCH_TAC;
  TYPE_THEN `INJ (IMAGE (reflBf r)) edge edge` SUBGOAL_TAC;
  REWRITE_TAC[INJ;reflB_edge;];
  REP_BASIC_TAC;
  TYPE_THEN `y SUBSET (euclid 2) /\ x SUBSET (euclid 2)` SUBGOAL_TAC;
  ASM_MESON_TAC[edge_euclid2];
  DISCH_TAC;
  ASM_MESON_TAC[image_power_inj];
  DISCH_TAC;
  (* start cases *)
  SUBCONJ_TAC;
  REWRITE_TAC[IMAGE2];
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  SUBCONJ_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[IMAGE2; EQ_EMPTY]);
  TSPEC `IMAGE (reflBf r) u` 4;
  UND 4;
  REWRITE_TAC[];
  TYPE_THEN `IMAGE (IMAGE (reflBf r)) G (IMAGE (reflBf r) u) = G u` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_app;
  EXISTS_TAC `edge`;
  EXISTS_TAC `edge`;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (*
  ASM_MESON_TAC[image_power_inj];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[];
  DISCH_TAC;
  *)
  SUBCONJ_TAC;
  REWRITE_TAC[IMAGE2;SUBSET];
  GEN_TAC;
  GEN_REWRITE_TAC (LAND_CONV o ONCE_DEPTH_CONV )  [IMAGE];
  REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  reflB_edge;
  ASM_MESON_TAC[ISUBSET;];
  DISCH_TAC;
  (* num closure clause *)
  CONJ_TAC;
  GEN_TAC;
  TYPE_THEN `pointI m = reflBf r (pointI (reflBi r m))` SUBGOAL_TAC;
  REWRITE_TAC[reflB_pointI;reflBi_inv];
  DISCH_THEN_REWRITE;
  TYPE_THEN `num_closure (IMAGE2 (reflBf r) G) (reflBf r (pointI (reflBi r m))) = num_closure G (pointI (reflBi r m))` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_num_closure);
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[];
  (* inductive_set clause *)
  REP_BASIC_TAC;
  (* isc *)
  USE 16(REWRITE_RULE[IMAGE2]);
  USE 16 (MATCH_MP SUBSET_PREIMAGE);
  REP_BASIC_TAC;
  TSPEC `Z` 0;
  TYPE_THEN `Z SUBSET G /\ ~(Z = {}) /\ (!C C'. Z C /\ G C' /\ adj C C' ==> Z C')` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  PROOF_BY_CONTR_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[]);
  REWR 16;
  RULE_ASSUM_TAC  (REWRITE_RULE[IMAGE_CLAUSES]);
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `D = IMAGE (reflBf r) C` ABBREV_TAC ;
  TYPE_THEN `D' = IMAGE (reflBf r) C'` ABBREV_TAC ;
  TSPEC `D` 14; (* *)
  TSPEC `D'` 14;
  TYPE_THEN `S D /\ IMAGE2 (reflBf r) G D' /\ adj D D'` SUBGOAL_TAC;
  SUBCONJ_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "D";
  TYPE_THEN `IMAGE (IMAGE (reflBf r)) Z (IMAGE (reflBf r) C) = Z C` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_app;
  TYPE_THEN `edge` EXISTS_TAC;
  TYPE_THEN `edge` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* fh1 *)
  SUBCONJ_TAC;
  EXPAND_TAC "D'";
  REWRITE_TAC[IMAGE2;IMAGE];
  NAME_CONFLICT_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  EXPAND_TAC "D";
  EXPAND_TAC "D'";
  TYPE_THEN `C SUBSET (euclid 2) /\ (C' SUBSET (euclid 2))` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET;edge_euclid2];
  DISCH_TAC;
  TYPE_THEN `(adj C C' ==> adj (IMAGE (reflBf r) C) (IMAGE (reflBf r) C'))` SUBGOAL_TAC;
  IMATCH_MP_TAC  homeo_adj;
  ASM_REWRITE_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWR 14;
  UND 14;
  EXPAND_TAC "D'";
  TYPE_THEN `IMAGE (IMAGE (reflBf r)) Z (IMAGE (reflBf r) C') = Z C'` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_app;
  TYPE_THEN `edge` EXISTS_TAC;
  TYPE_THEN `edge` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  UND 3;
  UND 19;
  ASM_MESON_TAC[ISUBSET];
  MESON_TAC[];
  DISCH_TAC;
  REWR 0;
  ASM_REWRITE_TAC[IMAGE2];
  ]);;
  (* }}} *)

let reflC_segment = prove_by_refinement(
  `!G . (segment G ==> (segment (IMAGE2 (reflCf) G)))`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  REWRITE_TAC[segment];
  COPY 0;
  USE 0(REWRITE_RULE[segment]);
  REP_BASIC_TAC;
  TYPE_THEN `homeomorphism (reflCf) top2 top2` SUBGOAL_TAC;
  REWRITE_TAC[reflC_homeo];
  DISCH_TAC;
  ASSUME_TAC top2_top;
  ASSUME_TAC top2_unions;
  TYPE_THEN `BIJ (reflCf) (euclid 2) (euclid 2)` SUBGOAL_TAC;
  ASM_MESON_TAC[homeomorphism];
  DISCH_TAC;
  TYPE_THEN `INJ (IMAGE (reflCf)) edge edge` SUBGOAL_TAC;
  REWRITE_TAC[INJ;reflC_edge;];
  REP_BASIC_TAC;
  TYPE_THEN `y SUBSET (euclid 2) /\ x SUBSET (euclid 2)` SUBGOAL_TAC;
  ASM_MESON_TAC[edge_euclid2];
  DISCH_TAC;
  ASM_MESON_TAC[image_power_inj];
  DISCH_TAC;
  (* start cases *)
  SUBCONJ_TAC;
  REWRITE_TAC[IMAGE2];
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  SUBCONJ_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[IMAGE2; EQ_EMPTY]);
  TSPEC `IMAGE (reflCf) u` 4;
  UND 4;
  REWRITE_TAC[];
  TYPE_THEN `IMAGE (IMAGE (reflCf)) G (IMAGE (reflCf) u) = G u` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_app;
  EXISTS_TAC `edge`;
  EXISTS_TAC `edge`;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (*
  ASM_MESON_TAC[image_power_inj];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[];
  DISCH_TAC;
  *)
  SUBCONJ_TAC;
  REWRITE_TAC[IMAGE2;SUBSET];
  GEN_TAC;
  GEN_REWRITE_TAC (LAND_CONV o ONCE_DEPTH_CONV )  [IMAGE];
  REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  reflC_edge;
  ASM_MESON_TAC[ISUBSET;];
  DISCH_TAC;
  (* num closure clause *)
  CONJ_TAC;
  GEN_TAC;
  TYPE_THEN `pointI m = reflCf (pointI (reflCi m))` SUBGOAL_TAC;
  REWRITE_TAC[reflC_pointI;reflCi_inv];
  DISCH_THEN_REWRITE;
  TYPE_THEN `num_closure (IMAGE2 (reflCf) G) (reflCf (pointI (reflCi m))) = num_closure G (pointI (reflCi m))` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_num_closure);
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[];
  (* inductive_set clause *)
  REP_BASIC_TAC;
  (* isc *)
  USE 16(REWRITE_RULE[IMAGE2]);
  USE 16 (MATCH_MP SUBSET_PREIMAGE);
  REP_BASIC_TAC;
  TSPEC `Z` 0;
  TYPE_THEN `Z SUBSET G /\ ~(Z = {}) /\ (!C C'. Z C /\ G C' /\ adj C C' ==> Z C')` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  PROOF_BY_CONTR_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[]);
  REWR 16;
  RULE_ASSUM_TAC  (REWRITE_RULE[IMAGE_CLAUSES]);
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `D = IMAGE (reflCf) C` ABBREV_TAC ;
  TYPE_THEN `D' = IMAGE (reflCf) C'` ABBREV_TAC ;
  TSPEC `D` 14; (* *)
  TSPEC `D'` 14;
  TYPE_THEN `S D /\ IMAGE2 (reflCf) G D' /\ adj D D'` SUBGOAL_TAC;
  SUBCONJ_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "D";
  TYPE_THEN `IMAGE (IMAGE (reflCf)) Z (IMAGE (reflCf) C) = Z C` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_app;
  TYPE_THEN `edge` EXISTS_TAC;
  TYPE_THEN `edge` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* fh1 *)
  SUBCONJ_TAC;
  EXPAND_TAC "D'";
  REWRITE_TAC[IMAGE2;IMAGE];
  NAME_CONFLICT_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  EXPAND_TAC "D";
  EXPAND_TAC "D'";
  TYPE_THEN `C SUBSET (euclid 2) /\ (C' SUBSET (euclid 2))` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET;edge_euclid2];
  DISCH_TAC;
  TYPE_THEN `(adj C C' ==> adj (IMAGE (reflCf) C) (IMAGE (reflCf) C'))` SUBGOAL_TAC;
  IMATCH_MP_TAC  homeo_adj;
  ASM_REWRITE_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWR 14;
  UND 14;
  EXPAND_TAC "D'";
  TYPE_THEN `IMAGE (IMAGE (reflCf)) Z (IMAGE (reflCf) C') = Z C'` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_app;
  TYPE_THEN `edge` EXISTS_TAC;
  TYPE_THEN `edge` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  UND 3;
  UND 19;
  ASM_MESON_TAC[ISUBSET];
  MESON_TAC[];
  DISCH_TAC;
  REWR 0;
  ASM_REWRITE_TAC[IMAGE2];
  ]);;

  (* }}} *)

let point_x = prove_by_refinement(
  `!x m. (x = point m) <=> (euclid 2 x /\ (FST m = x 0) /\ (SND m = x 1))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  EQ_TAC ;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[coord01;euclid_point];
  REP_BASIC_TAC;
  USE 2 (MATCH_MP   point_onto );
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[point_inj];
  REWRITE_TAC[PAIR_SPLIT];
  ASM_REWRITE_TAC[coord01];
  ]);;
  (* }}} *)

(* next IMAGE of square *)

let reflA_squ = prove_by_refinement(
  `!m r.  IMAGE (reflAf r) (squ m) = squ (left  (reflAi r m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[squ;reflAf;reflAi;IMAGE ;left  ;];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "x'");
  REWRITE_TAC[coord01;];
  REWRITE_TAC[point_x];
  CONV_TAC (dropq_conv "v");
  EQ_TAC ;
  REP_BASIC_TAC;
  TYPE_THEN `&2 * real_of_int r - u` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 4;
  UND 5;
  USE 0 (GSYM );
  ASM_REWRITE_TAC[int_sub_th;int_of_num_th;int_add_th;int_mul_th;];
  REAL_ARITH_TAC;
  (* 2 *)
  REP_BASIC_TAC;
  TYPE_THEN `&2 * real_of_int r - u` EXISTS_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `y - (y - x) = x`];
  UND 2;
  UND 3;
  USE 4 (GSYM);
  ASM_REWRITE_TAC[int_sub_th;int_of_num_th;int_add_th;int_mul_th;];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let reflB_squ = prove_by_refinement(
  `!m r.  IMAGE (reflBf r) (squ m) = squ (down  (reflBi r m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[squ;reflBf;reflBi;IMAGE ;down  ;];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "x'");
  REWRITE_TAC[coord01;];
  REWRITE_TAC[point_x];
  CONV_TAC (dropq_conv "u");
  EQ_TAC ;
  REP_BASIC_TAC;
  TYPE_THEN `&2 * real_of_int r - v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 2;
  UND 3;
  USE 0 (GSYM );
  ASM_REWRITE_TAC[int_sub_th;int_of_num_th;int_add_th;int_mul_th;];
  REAL_ARITH_TAC;
  (* 2 *)
  REP_BASIC_TAC;
  TYPE_THEN `&2 * real_of_int r - v` EXISTS_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `y - (y - x) = x`];
  UND 0;
  UND 1;
  USE 4 (GSYM);
  ASM_REWRITE_TAC[int_sub_th;int_of_num_th;int_add_th;int_mul_th;];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let reflC_squ = prove_by_refinement(
  `!m.  IMAGE (reflCf) (squ m) = squ (  (reflCi m))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[squ;reflCf;reflCi;IMAGE ; ];
  DISCH_ALL_TAC;
  IMATCH_MP_TAC EQ_EXT;
  REWRITE_TAC[];
  DISCH_ALL_TAC;
  CONV_TAC (dropq_conv "x'");
  REWRITE_TAC[coord01;];
  REWRITE_TAC[point_x];
  CONV_TAC (dropq_conv "u");
  CONV_TAC (dropq_conv "v");
  MESON_TAC[];
  ]);;
  (* }}} *)

(* move to sets *)
let powerset = jordan_def `powerset (X:A->bool) = { z | z SUBSET X }`;;

let image_sing = prove_by_refinement(
  `!(f:A -> B) x. (IMAGE f {x} = {(f x)})`,
  (* {{{ proof *)
  [
  REWRITE_TAC[IMAGE;INSERT];
  CONV_TAC (dropq_conv "x'");
  ]);;
  (* }}} *)

let image_unions = prove_by_refinement(
  `!(f:A->B)  U.
     (IMAGE f (UNIONS U) = UNIONS (IMAGE (IMAGE f) U))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[IMAGE;UNIONS;];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[];
  EQ_TAC;
  REP_BASIC_TAC;
  CONV_TAC (dropq_conv "u");
  ASM_REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  NAME_CONFLICT_TAC;
  REWR 0;
  KILL 1;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

(* move *)
let segment_euclid = prove_by_refinement(
  `!G. (segment G) ==> (closure top2 (UNIONS G) SUBSET euclid 2)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  closure_subset;
  ASM_REWRITE_TAC[top2_top;GSYM top2_unions];
  CONJ_TAC;
  IMATCH_MP_TAC  closed_UNIV;
  REWRITE_TAC[top2_top];
  REWRITE_TAC[top2_unions;SUBSET;UNIONS;];
  REP_BASIC_TAC;
  TYPE_THEN `edge u` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;ISUBSET];
  ASM_MESON_TAC[edge_euclid2;ISUBSET];
  ]);;
  (* }}} *)

let image_curve_cell_reflA  = prove_by_refinement(
  `!G r. (segment G) ==>
    (curve_cell (IMAGE2 (reflAf r) G) =
           IMAGE2 (reflAf r) (curve_cell G))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[curve_cell];
  REWRITE_TAC[IMAGE2;IMAGE_UNION;];
  AP_TERM_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  TYPE_THEN `UNIONS G SUBSET (euclid 2)` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET;UNIONS;];
  REP_BASIC_TAC;
  TYPE_THEN `edge u` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;ISUBSET;];
  ASM_MESON_TAC[edge_euclid2;ISUBSET];
  DISCH_TAC;
  ASSUME_TAC top2_top;
  ASSUME_TAC top2_unions;
  (*  *)
  TYPE_THEN `UNIONS (IMAGE (IMAGE (reflAf r)) G) = IMAGE (reflAf r) (UNIONS G)` SUBGOAL_TAC;
  REWRITE_TAC[GSYM image_unions];
  DISCH_THEN_REWRITE ;
  (*  *)
  TYPE_THEN `closure top2 (IMAGE (reflAf r) (UNIONS G)) = IMAGE (reflAf r) (closure top2 (UNIONS G))` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_closure);
  ASM_REWRITE_TAC[top2_top;reflA_homeo;top2_unions;];
  DISCH_THEN_REWRITE;
  (*  *)
  TYPE_THEN `!n. IMAGE (reflAf r) (closure top2 (UNIONS G)) (pointI n) = closure top2 (UNIONS G) (pointI (reflAi r n))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN  `n' = reflAi r n` ABBREV_TAC ;
  TYPE_THEN `pointI n = reflAf r (pointI n')` SUBGOAL_TAC;
  EXPAND_TAC "n'";
  KILL 4;
  ASM_REWRITE_TAC[reflA_pointI;reflAi_inv];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  image_app;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
 TYPE_THEN `(euclid 2)` EXISTS_TAC;
  REWRITE_TAC[pointI;euclid_point];
  ASSUME_TAC reflA_homeo;
  RULE_ASSUM_TAC  (REWRITE_RULE[homeomorphism;BIJ;top2_unions;]);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  segment_euclid;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  (*  *)
  REWRITE_TAC[IMAGE;];
  CONV_TAC (dropq_conv "x'");
(**** Modified by JRH to avoid GSPEC
  REWRITE_TAC[INR IN_SING;GSPEC;];
 ****)
  REWRITE_TAC[INR IN_SING; UNWIND_THM2];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x'");
  CONV_TAC (dropq_conv "y'");
(**** Removed by JRH
  REWRITE_TAC[GSPEC];
 ****)
  (*  *)
  EQ_TAC ;
  REP_BASIC_TAC;
  TYPE_THEN `reflAi r n'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING; reflA_pointI; reflAi_inv;];
(*** Removed by JRH
  MESON_TAC[];
 ****)
  (*   *)
  REP_BASIC_TAC;
  TYPE_THEN `reflAi r n'` EXISTS_TAC;
  ASM_REWRITE_TAC[reflAi_inv;];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING;reflA_pointI;];
(*** Removed by JRH
  MESON_TAC[];
 ****)
  ]);;
  (* }}} *)

let image_curve_cell_reflB  = prove_by_refinement(
  `!G r. (segment G) ==>
    (curve_cell (IMAGE2 (reflBf r) G) =
           IMAGE2 (reflBf r) (curve_cell G))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[curve_cell];
  REWRITE_TAC[IMAGE2;IMAGE_UNION;];
  AP_TERM_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  TYPE_THEN `UNIONS G SUBSET (euclid 2)` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET;UNIONS;];
  REP_BASIC_TAC;
  TYPE_THEN `edge u` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;ISUBSET;];
  ASM_MESON_TAC[edge_euclid2;ISUBSET];
  DISCH_TAC;
  ASSUME_TAC top2_top;
  ASSUME_TAC top2_unions;
  (*  *)
  TYPE_THEN `UNIONS (IMAGE (IMAGE (reflBf r)) G) = IMAGE (reflBf r) (UNIONS G)` SUBGOAL_TAC;
  REWRITE_TAC[GSYM image_unions];
  DISCH_THEN_REWRITE ;
  (*  *)
  TYPE_THEN `closure top2 (IMAGE (reflBf r) (UNIONS G)) = IMAGE (reflBf r) (closure top2 (UNIONS G))` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_closure);
  ASM_REWRITE_TAC[top2_top;reflB_homeo;top2_unions;];
  DISCH_THEN_REWRITE;
  (*  *)
  TYPE_THEN `!n. IMAGE (reflBf r) (closure top2 (UNIONS G)) (pointI n) = closure top2 (UNIONS G) (pointI (reflBi r n))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN  `n' = reflBi r n` ABBREV_TAC ;
  TYPE_THEN `pointI n = reflBf r (pointI n')` SUBGOAL_TAC;
  EXPAND_TAC "n'";
  KILL 4;
  ASM_REWRITE_TAC[reflB_pointI;reflBi_inv];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  image_app;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
 TYPE_THEN `(euclid 2)` EXISTS_TAC;
  REWRITE_TAC[pointI;euclid_point];
  ASSUME_TAC reflB_homeo;
  RULE_ASSUM_TAC  (REWRITE_RULE[homeomorphism;BIJ;top2_unions;]);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  segment_euclid;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  (*  *)
  REWRITE_TAC[IMAGE;];
  CONV_TAC (dropq_conv "x'");

(*** JRH changed this line to avoid GSPEC
  REWRITE_TAC[INR IN_SING;GSPEC;];
 ***)
  REWRITE_TAC[INR IN_SING; UNWIND_THM2];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x'");
  CONV_TAC (dropq_conv "y'");
(*** JRH removed this to avoid GSPEC
  REWRITE_TAC[GSPEC];
 ***)
  (*  *)
  EQ_TAC ;
  REP_BASIC_TAC;
  TYPE_THEN `reflBi r n'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING; reflB_pointI; reflBi_inv;];
(*** Removed by JRH
  MESON_TAC[];
 ****)
  (*   *)
  REP_BASIC_TAC;
  TYPE_THEN `reflBi r n'` EXISTS_TAC;
  ASM_REWRITE_TAC[reflBi_inv;];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING;reflB_pointI;];
(*** Removed by JRH
  MESON_TAC[];
 ****)
  ]);;
  (* }}} *)

let image_curve_cell_reflC  = prove_by_refinement(
  `!G . (segment G) ==>
    (curve_cell (IMAGE2 (reflCf ) G) =
           IMAGE2 (reflCf) (curve_cell G))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[curve_cell];
  REWRITE_TAC[IMAGE2;IMAGE_UNION;];
  AP_TERM_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  TYPE_THEN `UNIONS G SUBSET (euclid 2)` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET;UNIONS;];
  REP_BASIC_TAC;
  TYPE_THEN `edge u` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;ISUBSET;];
  ASM_MESON_TAC[edge_euclid2;ISUBSET];
  DISCH_TAC;
  ASSUME_TAC top2_top;
  ASSUME_TAC top2_unions;
  (*  *)
  TYPE_THEN `UNIONS (IMAGE (IMAGE (reflCf)) G) = IMAGE (reflCf) (UNIONS G)` SUBGOAL_TAC;
  REWRITE_TAC[GSYM image_unions];
  DISCH_THEN_REWRITE ;
  (*  *)
  TYPE_THEN `closure top2 (IMAGE (reflCf) (UNIONS G)) = IMAGE (reflCf) (closure top2 (UNIONS G))` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_closure);
  ASM_REWRITE_TAC[top2_top;reflC_homeo;top2_unions;];
  DISCH_THEN_REWRITE;
  (*  *)
  TYPE_THEN `!n. IMAGE (reflCf) (closure top2 (UNIONS G)) (pointI n) = closure top2 (UNIONS G) (pointI (reflCi n))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN  `n' = reflCi n` ABBREV_TAC ;
  TYPE_THEN `pointI n = reflCf (pointI n')` SUBGOAL_TAC;
  EXPAND_TAC "n'";
  KILL 4;
  ASM_REWRITE_TAC[reflC_pointI;reflCi_inv];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  image_app;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
 TYPE_THEN `(euclid 2)` EXISTS_TAC;
  REWRITE_TAC[pointI;euclid_point];
  ASSUME_TAC reflC_homeo;
  RULE_ASSUM_TAC  (REWRITE_RULE[homeomorphism;BIJ;top2_unions;]);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  segment_euclid;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  (*  *)
  REWRITE_TAC[IMAGE;];
  CONV_TAC (dropq_conv "x'");
(*** This line changed by JRH to avoid GSPEC
  REWRITE_TAC[INR IN_SING;GSPEC;];
 ***)
  REWRITE_TAC[INR IN_SING; UNWIND_THM2];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x'");
  CONV_TAC (dropq_conv "y'");
 (*** Removed by JRH to avoid GSPEC
  REWRITE_TAC[GSPEC];
 ***)
  (*  *)
  EQ_TAC ;
  REP_BASIC_TAC;
  TYPE_THEN `reflCi n'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING; reflC_pointI; reflCi_inv;];
(*** Removed by JRH
  MESON_TAC[];
 ****)
  (*   *)
  REP_BASIC_TAC;
  TYPE_THEN `reflCi n'` EXISTS_TAC;
  ASM_REWRITE_TAC[reflCi_inv;];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING;reflC_pointI;];
(*** Removed by JRH
  MESON_TAC[];
 ****)
  ]);;
  (* }}} *)

let inj_inter = prove_by_refinement(
  `!(f:A->B) X Y A B. (INJ f X Y) /\ (A SUBSET X) /\ (B SUBSET X) ==>
     (IMAGE f (A INTER B) = (IMAGE f A) INTER (IMAGE f B))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE;INTER ];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  EQ_TAC;
  REP_BASIC_TAC;
  ASM_MESON_TAC[ISUBSET;];
  REP_BASIC_TAC;
  TYPE_THEN `x' = x''` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[ISUBSET;];
  REP_BASIC_TAC;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let homeomorphism_induced_top = prove_by_refinement(
  `!(f:A->B) U V A.  (homeomorphism f U V) /\ (A SUBSET (UNIONS U)) ==>
      (IMAGE2 f (induced_top U A) = induced_top V (IMAGE f A))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[induced_top;];
  COPY 1;
  USE 1 (MATCH_MP homeo_bij);
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[IMAGE2];
  TYPE_THEN `g = IMAGE f` ABBREV_TAC ;
  REWRITE_TAC[IMAGE];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x''");
  (*  *)
  TYPE_THEN `!t. U t ==> (g (t INTER A)  = g t INTER g A)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC "g";
  IMATCH_MP_TAC  inj_inter;
  TYPE_THEN `(UNIONS U)` EXISTS_TAC;
  TYPE_THEN `(UNIONS V)` EXISTS_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism;BIJ]);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  sub_union;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (*   *)
  EQ_TAC;
  REP_BASIC_TAC;
  TSPEC `x'` 4;
  REWR 4;
  ASM_REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  TYPE_THEN `g x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  (*  *)
  REP_BASIC_TAC;
  TYPE_THEN `?t. U t /\ (g t = x')` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TSPEC `t` 4;
  REWR 4;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let ctop_reflA = prove_by_refinement(
  `!G r. (segment G) ==>
      (IMAGE2 (reflAf r) (ctop G) = ctop (IMAGE2 (reflAf r) G))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[ctop];
  ASSUME_TAC reflA_homeo;
  TYPE_THEN `euclid 2 DIFF UNIONS (curve_cell G) SUBSET (UNIONS top2)` SUBGOAL_TAC;
  REWRITE_TAC[top2_unions;DIFF;SUBSET;];
  MESON_TAC[];
  DISCH_TAC ;
  (*   *)
  TYPE_THEN `IMAGE2 (reflAf r) (induced_top top2 (euclid 2 DIFF UNIONS (curve_cell G))) = induced_top top2 (IMAGE (reflAf r) (euclid 2 DIFF  (UNIONS (curve_cell G))))` SUBGOAL_TAC;
  IMATCH_MP_TAC  homeomorphism_induced_top;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  AP_TERM_TAC;
  TSPEC `r` 1;
  (*  *)
  TYPE_THEN `IMAGE (reflAf r) (euclid 2 DIFF UNIONS (curve_cell G)) = euclid 2 DIFF (IMAGE (reflAf r) (UNIONS (curve_cell G)))` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism;top2_unions;]);
  REP_BASIC_TAC;
  USE 4 (MATCH_MP DIFF_SURJ);
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[UNIONS;SUBSET;];
  REP_BASIC_TAC;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  TYPE_THEN `cell u` SUBGOAL_TAC;
  USE 7 (MATCH_MP curve_cell_cell);
  ASM_MESON_TAC[ISUBSET;];
  ASM_MESON_TAC[ISUBSET;cell_euclid];
  DISCH_THEN_REWRITE;
  AP_TERM_TAC;
  REWRITE_TAC[image_unions];
  AP_TERM_TAC;
  ASM_SIMP_TAC[image_curve_cell_reflA];
  REWRITE_TAC[IMAGE2];
  ]);;
  (* }}} *)

let ctop_reflB = prove_by_refinement(
  `!G r. (segment G) ==>
      (IMAGE2 (reflBf r) (ctop G) = ctop (IMAGE2 (reflBf r) G))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[ctop];
  ASSUME_TAC reflB_homeo;
  TYPE_THEN `euclid 2 DIFF UNIONS (curve_cell G) SUBSET (UNIONS top2)` SUBGOAL_TAC;
  REWRITE_TAC[top2_unions;DIFF;SUBSET;];
  MESON_TAC[];
  DISCH_TAC ;
  (*   *)
  TYPE_THEN `IMAGE2 (reflBf r) (induced_top top2 (euclid 2 DIFF UNIONS (curve_cell G))) = induced_top top2 (IMAGE (reflBf r) (euclid 2 DIFF  (UNIONS (curve_cell G))))` SUBGOAL_TAC;
  IMATCH_MP_TAC  homeomorphism_induced_top;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  AP_TERM_TAC;
  TSPEC `r` 1;
  (*  *)
  TYPE_THEN `IMAGE (reflBf r) (euclid 2 DIFF UNIONS (curve_cell G)) = euclid 2 DIFF (IMAGE (reflBf r) (UNIONS (curve_cell G)))` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism;top2_unions;]);
  REP_BASIC_TAC;
  USE 4 (MATCH_MP DIFF_SURJ);
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[UNIONS;SUBSET;];
  REP_BASIC_TAC;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  TYPE_THEN `cell u` SUBGOAL_TAC;
  USE 7 (MATCH_MP curve_cell_cell);
  ASM_MESON_TAC[ISUBSET;];
  ASM_MESON_TAC[ISUBSET;cell_euclid];
  DISCH_THEN_REWRITE;
  AP_TERM_TAC;
  REWRITE_TAC[image_unions];
  AP_TERM_TAC;
  ASM_SIMP_TAC[image_curve_cell_reflB];
  REWRITE_TAC[IMAGE2];
  ]);;
  (* }}} *)

let ctop_reflC = prove_by_refinement(
  `!G . (segment G) ==>
      (IMAGE2 (reflCf) (ctop G) = ctop (IMAGE2 (reflCf) G))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[ctop];
  ASSUME_TAC reflC_homeo;
  TYPE_THEN `euclid 2 DIFF UNIONS (curve_cell G) SUBSET (UNIONS top2)` SUBGOAL_TAC;
  REWRITE_TAC[top2_unions;DIFF;SUBSET;];
  MESON_TAC[];
  DISCH_TAC ;
  (*   *)
  TYPE_THEN `IMAGE2 (reflCf) (induced_top top2 (euclid 2 DIFF UNIONS (curve_cell G))) = induced_top top2 (IMAGE (reflCf) (euclid 2 DIFF  (UNIONS (curve_cell G))))` SUBGOAL_TAC;
  IMATCH_MP_TAC  homeomorphism_induced_top;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  AP_TERM_TAC;
  (*  *)
  TYPE_THEN `IMAGE (reflCf) (euclid 2 DIFF UNIONS (curve_cell G)) = euclid 2 DIFF (IMAGE (reflCf) (UNIONS (curve_cell G)))` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism;top2_unions;]);
  REP_BASIC_TAC;
  USE 4 (MATCH_MP DIFF_SURJ);
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[UNIONS;SUBSET;];
  REP_BASIC_TAC;
  TYPE_THEN `G SUBSET edge` SUBGOAL_TAC;
  ASM_MESON_TAC[segment];
  DISCH_TAC;
  TYPE_THEN `cell u` SUBGOAL_TAC;
  USE 7 (MATCH_MP curve_cell_cell);
  ASM_MESON_TAC[ISUBSET;];
  ASM_MESON_TAC[ISUBSET;cell_euclid];
  DISCH_THEN_REWRITE;
  AP_TERM_TAC;
  REWRITE_TAC[image_unions];
  AP_TERM_TAC;
  ASM_SIMP_TAC[image_curve_cell_reflC];
  REWRITE_TAC[IMAGE2];
  ]);;
  (* }}} *)

let connected_homeo = prove_by_refinement(
  `!(f:A->B) U V Z. (homeomorphism f U V /\ (Z SUBSET UNIONS U) ==>
       (connected V (IMAGE f Z) = connected U Z))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `g = INV f (UNIONS U) (UNIONS V)` ABBREV_TAC ;
  TYPE_THEN `Z = IMAGE g (IMAGE f Z)` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[IMAGE];
  EXPAND_TAC "g";
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x''");
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism]);
  REP_BASIC_TAC;
  TYPE_THEN `!x'. (UNIONS U x') ==> (INV f (UNIONS U) (UNIONS V) (f x') = x')` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  inv_comp_left;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (*  *)
  EQ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN ` x` EXISTS_TAC;
  KILL 2;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET;];
  REP_BASIC_TAC;
  TSPEC `x'` 5;
  TYPE_THEN `UNIONS U x'` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  REWR 5;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  EQ_TAC;
  REP_BASIC_TAC;
  UND 3;
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]);
  IMATCH_MP_TAC  connect_image;
  TYPE_THEN `V` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  EXPAND_TAC "g";
  IMATCH_MP_TAC  INV_homeomorphism;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE;SUBSET;];
  REP_BASIC_TAC;
  UND 3;
  EXPAND_TAC "g";
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `UNIONS U x''` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism]);
  TYPE_THEN `x = x''` SUBGOAL_TAC;
  ASM_MESON_TAC[inv_comp_left];
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  connect_image;
  TYPE_THEN `U` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE [homeomorphism;BIJ;INJ]);
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[SUBSET;IMAGE;];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x''");
  ASM_MESON_TAC[ISUBSET;];
  ]);;
  (* }}} *)

(* start here , Tues Jun 8 , 2004 *)

let component = prove_by_refinement(
  `!U (x:A) . (component  U x = {y | ?Z. connected U Z /\ Z x /\ Z y})`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[component_DEF ;];
  ]);;
  (* }}} *)

let component_homeo = prove_by_refinement(
  `!(f:A->B) U V x. (homeomorphism f U V) /\ (UNIONS U x) ==>
     (IMAGE f (component U x) = (component  V (f x)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[component ;IMAGE ; ];
  IMATCH_MP_TAC  EQ_EXT ;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  CONV_TAC (dropq_conv "x'");
  EQ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `IMAGE f Z` EXISTS_TAC;
  CONJ_TAC;
  TYPE_THEN `Z SUBSET UNIONS U` SUBGOAL_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[connected]);
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[connected_homeo];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE];
  ASM_MESON_TAC[];
  (*  *)
  REP_BASIC_TAC;
  (* *)
  TYPE_THEN `?A. A SUBSET (UNIONS U) /\ (IMAGE f A = Z)` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_power_surj;
  TYPE_THEN `UNIONS V` EXISTS_TAC;
  ASM_MESON_TAC[connected;homeomorphism];
  REP_BASIC_TAC;
  TYPE_THEN `A` EXISTS_TAC;
  NAME_CONFLICT_TAC;
  WITH 5 (REWRITE_RULE[IMAGE]);
  USE 7 (GSYM);
  REWR 2;
  REP_BASIC_TAC;
  TYPE_THEN `x''` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWR 3;
  REP_BASIC_TAC;
  TYPE_THEN ` x = x'''` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE [homeomorphism;BIJ;INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC  ;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  KILL 7;
  ASM_SIMP_TAC[GSYM connected_homeo];
  ]);;
  (* }}} *)

let bij_homeo = prove_by_refinement(
  `!(f:A->B) U V. (BIJ f (UNIONS U) (UNIONS V)) /\
    (BIJ (IMAGE f) U V) ==> (homeomorphism f U V)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[homeomorphism;continuous;];
  ASM_REWRITE_TAC[preimage;];
  CONJ_TAC;
  REP_BASIC_TAC;
  COPY 1;
  UND 3;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;INJ;SURJ]);
  REP_BASIC_TAC;
  TSPEC  `v` 1;
  REWR 1;
  REP_BASIC_TAC;
  EXPAND_TAC "v";
  TYPE_THEN `{x | UNIONS U x /\ IMAGE f y (f x)} = y` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  GEN_TAC;
  EQ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `IMAGE f y (f x) = y x` SUBGOAL_TAC;
  IMATCH_MP_TAC image_app ;
  TYPE_THEN `(UNIONS U)` EXISTS_TAC;
  TYPE_THEN `(UNIONS V)` EXISTS_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ]);
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[sub_union];
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[sub_union;ISUBSET];
  REWRITE_TAC[IMAGE];
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  (* *)
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let homeomorphism_subset = prove_by_refinement(
  `!(f:A->B) U V C. (homeomorphism f U V) /\ (C SUBSET U) ==>
   (homeomorphism f C (IMAGE2 f C))`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  bij_homeo;
  SUBCONJ_TAC;
  TYPE_THEN `UNIONS C SUBSET UNIONS U` SUBGOAL_TAC;
  IMATCH_MP_TAC  UNIONS_UNIONS ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWRITE_TAC[IMAGE2 ;GSYM  image_unions;];
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism;BIJ]);
  REP_BASIC_TAC;
  REWRITE_TAC[BIJ];
  SUBCONJ_TAC;
  REWRITE_TAC[INJ];
    SUBCONJ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `IMAGE f (UNIONS C) (f x) = (UNIONS C) x` SUBGOAL_TAC;
  IMATCH_MP_TAC  (image_app);
  TYPE_THEN `(UNIONS U)` EXISTS_TAC;
  TYPE_THEN `(UNIONS V)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC [ISUBSET];
  REWRITE_TAC[INJ];
  REP_BASIC_TAC;
  REWRITE_TAC[SURJ];
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[IMAGE]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWRITE_TAC[BIJ];
  WITH_FIRST (MATCH_MP homeo_bij);
  SUBCONJ_TAC;
  REWRITE_TAC[INJ];
  CONJ_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE2;];
  TYPE_THEN `g = IMAGE f` ABBREV_TAC ;
  REWRITE_TAC[IMAGE];
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  REWRITE_TAC[INJ;SURJ];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[IMAGE2]);
  TYPE_THEN `g = IMAGE f` ABBREV_TAC ;
  UND 6;
  REWRITE_TAC[IMAGE];
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let component_reflA = prove_by_refinement(
  `!(f:A->B) G r x. (segment G) /\ (UNIONS (ctop G) x) ==>
    (IMAGE (reflAf r) (component  (ctop G) x) =
         (component  (ctop (IMAGE2 (reflAf r) G)) (reflAf r x)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  component_homeo;
  ASM_REWRITE_TAC[];
  TYPE_THEN `ctop (IMAGE2 (reflAf r) G) = IMAGE2 (reflAf r) (ctop G)` SUBGOAL_TAC ;
  ASM_MESON_TAC[ctop_reflA];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  homeomorphism_subset;
  TYPE_THEN `top2` EXISTS_TAC;
  TYPE_THEN `top2` EXISTS_TAC;
  REWRITE_TAC[reflA_homeo];
  REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[ctop_top2];
  ]);;
  (* }}} *)

let component_reflB = prove_by_refinement(
  `!(f:A->B) G r x. (segment G) /\ (UNIONS (ctop G) x) ==>
    (IMAGE (reflBf r) (component  (ctop G) x) =
         (component  (ctop (IMAGE2 (reflBf r) G)) (reflBf r x)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  component_homeo;
  ASM_REWRITE_TAC[];
  TYPE_THEN `ctop (IMAGE2 (reflBf r) G) = IMAGE2 (reflBf r) (ctop G)` SUBGOAL_TAC ;
  ASM_MESON_TAC[ctop_reflB];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  homeomorphism_subset;
  TYPE_THEN `top2` EXISTS_TAC;
  TYPE_THEN `top2` EXISTS_TAC;
  REWRITE_TAC[reflB_homeo];
  REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[ctop_top2];
  ]);;
  (* }}} *)

let component_reflC = prove_by_refinement(
  `!(f:A->B) G x. (segment G) /\ (UNIONS (ctop G) x) ==>
    (IMAGE (reflCf) (component  (ctop G) x) =
         (component  (ctop (IMAGE2 (reflCf) G)) (reflCf x)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  component_homeo;
  ASM_REWRITE_TAC[];
  TYPE_THEN `ctop (IMAGE2 (reflCf) G) = IMAGE2 (reflCf) (ctop G)` SUBGOAL_TAC ;
  ASM_MESON_TAC[ctop_reflC];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  homeomorphism_subset;
  TYPE_THEN `top2` EXISTS_TAC;
  TYPE_THEN `top2` EXISTS_TAC;
  REWRITE_TAC[reflC_homeo];
  REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[ctop_top2];
  ]);;
  (* }}} *)

let subset_union_inter = prove_by_refinement(
  `!(X:A->bool) A B. (X SUBSET (A UNION B)   ==>
      (~(X INTER A = EMPTY )) \/ (~(X INTER B = EMPTY)) \/ (X = EMPTY ))`,
  (* {{{ proof *)
  [
  (REWRITE_TAC [EMPTY_EXISTS;SUBSET;UNION;INTER;EQ_EMPTY ; ]);
  MESON_TAC[];
  ]);;
  (* }}} *)

let squ_disj = prove_by_refinement(
  `!m n. ((squ m INTER squ n = {}) <=> ~(m = n))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
    EQ_TAC;
  DISCH_ALL_TAC;
  REWR 1;
  RULE_ASSUM_TAC (REWRITE_RULE[INTER_IDEMPOT;]);
  ASM_MESON_TAC[cell_nonempty;cell_rules];
  DISCH_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `squ m = squ n` SUBGOAL_TAC;
  IMATCH_MP_TAC  cell_partition;
  ASM_MESON_TAC[cell_rules];
  ASM_REWRITE_TAC[squ_inj];
  ]);;
  (* }}} *)

(* move way up *)
let cell_clauses = prove_by_refinement(
  `(!m. (~(v_edge m = EMPTY ) /\ ~(h_edge m = EMPTY )
       /\ ~(squ m = EMPTY ) /\ ~({(pointI m)} = EMPTY ))) /\
   (!m n. (v_edge m INTER {(pointI n)} = EMPTY ) /\
         ({(pointI n)} INTER v_edge m = EMPTY ) /\
  (h_edge m INTER {(pointI n)} = EMPTY ) /\
         ({(pointI n)} INTER h_edge m = EMPTY ) /\
  (squ m INTER {(pointI n)} = EMPTY ) /\
         ({(pointI n)} INTER squ m = EMPTY ) /\
       ((v_edge m INTER v_edge n  = EMPTY ) <=> ~(m = n) ) /\
   ((h_edge m INTER h_edge n  = EMPTY ) <=> ~(m = n) ) /\
  ((squ m INTER squ n  = EMPTY ) <=> ~(m = n) ) /\
  (squ m INTER h_edge n = EMPTY ) /\
         (h_edge n INTER squ m = EMPTY ) /\
  (squ m INTER v_edge n = EMPTY ) /\
        ( v_edge n INTER squ m = EMPTY ) /\
   (h_edge m INTER v_edge n = EMPTY ) /\
        ( v_edge n INTER h_edge m = EMPTY ) /\
   (({(pointI n)} INTER {(pointI m)} = EMPTY ) <=> ~(n = m)) /\
   (({(pointI n)} = {(pointI m)}  ) <=> (n = m)) /\
   ~(h_edge n = {(pointI m)}) /\
   ~(v_edge n = {(pointI m)}) /\
   ~(squ n = {(pointI m)}) /\
   ~( {(pointI m)} = h_edge n) /\
~( {(pointI m)} = v_edge n) /\
~( {(pointI m)} = squ n) /\
~(h_edge m = v_edge n) /\
((h_edge m = h_edge n) <=> (m = n)) /\
~(h_edge m = squ n) /\
~(v_edge m = h_edge n) /\
((v_edge m = v_edge n) <=> (m = n)) /\
~(v_edge m = squ n) /\
~(squ m = h_edge n) /\
((squ m = squ n) <=> (m = n)) /\
~(squ m = v_edge n) /\
~(squ m (pointI n)) /\
~(v_edge m (pointI n)) /\
~(h_edge m (pointI n)) /\
((pointI n = pointI m) <=> (n = m)))  `,

  (* {{{ proof *)
  (let notrr = REWRITE_RULE[not_eq] in
  let interc = ONCE_REWRITE_RULE[INTER_COMM] in
  ([
  CONJ_TAC ;
  ASM_MESON_TAC[cell_nonempty;cell_rules];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[INTER_ACI;notrr v_edge_disj;notrr h_edge_disj;interc square_h_edge;square_h_edge;interc square_v_edge;square_v_edge;square_disj;single_inter;h_edge_inj;v_edge_inj;notrr squ_inj;INR IN_SING;hv_edgeV2; square_h_edgeV2; square_v_edgeV2;hv_edge;square_pointIv2;v_edge_pointIv2;h_edge_pointIv2;notrr single_inter;v_edge_pointI;h_edge_pointI;square_pointI;pointI_inj;squ_disj];
  REWRITE_TAC[eq_sing;INR IN_SING;pointI_inj;];
  CONV_TAC (dropq_conv "u");
  ASM_MESON_TAC[pointI_inj];
  ])));;
  (* }}} *)

let inter_union = prove_by_refinement(
  `!X A (B:A->bool). ~(X INTER (A UNION B) = EMPTY) ==>
    ~(X INTER A = EMPTY) \/ ~(X INTER B = EMPTY)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INTER;UNION;EMPTY_EXISTS;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let squc_v = prove_by_refinement(
  `!m n. (v_edge m SUBSET squc n) ==> (n = m) \/ (n = left  m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[squc_union;];
  REP_BASIC_TAC;
  USE_FIRST (MATCH_MP subset_union_inter) THEN (RULE_ASSUM_TAC (REWRITE_RULE[cell_clauses])) ;
  REPEAT (USE_FIRST (MATCH_MP inter_union) THEN (RULE_ASSUM_TAC (REWRITE_RULE[cell_clauses]))) ;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  KILL 0;
  USE_FIRST (MATCH_MP inter_union) THEN (RULE_ASSUM_TAC (REWRITE_RULE[cell_clauses])) ;
  ASM_REWRITE_TAC[right_left];
  (*   *)
  ]);;
  (* }}} *)

let squc_h = prove_by_refinement(
  `!m n. (h_edge m SUBSET squc n) ==> (n = m) \/ (n = down  m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[squc_union;];
  REP_BASIC_TAC;
  USE_FIRST (MATCH_MP subset_union_inter) THEN (RULE_ASSUM_TAC (REWRITE_RULE[cell_clauses])) ;
  REPEAT (USE_FIRST (MATCH_MP inter_union) THEN (RULE_ASSUM_TAC (REWRITE_RULE[cell_clauses]))) ;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  KILL 0;
  USE_FIRST (MATCH_MP inter_union) THEN (RULE_ASSUM_TAC (REWRITE_RULE[cell_clauses])) ;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[right_left];
  KILL 0;
  REPEAT (USE_FIRST (MATCH_MP inter_union) THEN (RULE_ASSUM_TAC (REWRITE_RULE[cell_clauses]))) ;
  ASM_MESON_TAC [];
  (*   *)
  ]);;
  (* }}} *)

let component_empty = prove_by_refinement(
  `!U (x:A). (topology_ U) ==> ((component  U x = EMPTY) = ~(UNIONS U x))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[component ;EQ_EMPTY;];
  EQ_TAC;
  REP_BASIC_TAC;
  TSPEC `x` 2;
  ASM_MESON_TAC[connected_sing;INR IN_SING;];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[connected]);
  REP_BASIC_TAC;
  ASM_MESON_TAC[ISUBSET];
  ]);;
  (* }}} *)

let image_imp = prove_by_refinement(
  `!(f:A->B) X t. X t ==> (IMAGE f X) (f t)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let image_inj = prove_by_refinement(
  `!(f:A->B) X A B. (INJ f X UNIV) /\ (A SUBSET X ) /\ (B SUBSET X) /\
     (IMAGE f A SUBSET IMAGE f B) ==> (A SUBSET B)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ;IMAGE;SUBSET;];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let closure_euclid = prove_by_refinement(
  `closure (top2) (euclid 2) = euclid 2`,
  (* {{{ proof *)
  [
  REWRITE_TAC[closure;top2];
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC  INTERS_SUBSET;
  REWRITE_TAC[SUBSET_REFL;];
  ASM_MESON_TAC[closed_UNIV;top_of_metric_top;metric_euclid;top_of_metric_unions;];
  REWRITE_TAC[INTERS;SUBSET];
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let closure_euclid = prove_by_refinement(
  `!A. (A SUBSET (euclid 2) ==> (closure top2 A SUBSET (euclid 2)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  ONCE_REWRITE_TAC [GSYM closure_euclid];
  IMATCH_MP_TAC  subset_of_closure;
  ASM_REWRITE_TAC[top2_top];
  ]);;
  (* }}} *)

let along_lemma7 = prove_by_refinement(
  `!G m n x e. (segment G /\ (squ n SUBSET component  (ctop G) x) /\
     (v_edge m SUBSET squc n) /\
     (G (v_edge m)) /\ G e /\ (closure top2 e (pointI m)) ==>
   (?p. e SUBSET closure top2 (squ p) /\
       (squ p SUBSET (component  (ctop G) x))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  WITH_FIRST (MATCH_MP squc_v);
  FIRST_ASSUM (DISJ_CASES_TAC);
  REWR 3;
  IMATCH_MP_TAC  along_lemma6;
  TYPE_THEN `m` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWR 4;
  (* 2nd side *)
  REWR 4;
  REWR 3;
  KILL 6;
  KILL 7;
  TYPE_THEN `e' = IMAGE (reflAf (&:0)) e ` ABBREV_TAC ;
  TYPE_THEN `G' = IMAGE2 (reflAf (&:0)) G` ABBREV_TAC ;
  TYPE_THEN `x' = reflAf (&:0) x` ABBREV_TAC ;
  TYPE_THEN `UNIONS (ctop G) x` SUBGOAL_TAC;
  TYPE_THEN `~(component  (ctop G) x = EMPTY)` SUBGOAL_TAC;
  USE 4(REWRITE_RULE[SUBSET]);
  TYPE_THEN `~(squ (left  m) = EMPTY)` SUBGOAL_TAC;
  ASM_MESON_TAC[cell_nonempty;cell_rules];
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  TSPEC `u` 4;
  REWR 4;
  ASM_MESON_TAC[];
  TYPE_THEN `topology_ (ctop G)` SUBGOAL_TAC;
  ASM_MESON_TAC[ctop_top];
  ASM_SIMP_TAC [component_empty];
  DISCH_TAC;
  TYPE_THEN `component  (ctop G') x' = IMAGE (reflAf (&:0)) (component  (ctop G) x)` SUBGOAL_TAC;
  ASM_MESON_TAC[component_reflA;];
  DISCH_TAC;
  (*  *)
  TYPE_THEN `?p'. e' SUBSET closure top2 (squ p') /\ squ p' SUBSET component (ctop G') x'` SUBGOAL_TAC;
  IMATCH_MP_TAC  along_lemma6;
  TYPE_THEN `reflAi (&:0) m` EXISTS_TAC;
  (SUBCONJ_TAC);
  (* 1st claus *)
  EXPAND_TAC "G'";
  IMATCH_MP_TAC reflA_segment;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  (* 2nd clause *)
  ASM_REWRITE_TAC[];
  (* goal 2c *)
  USE 4(MATCH_MP (ISPEC `reflAf (&:0)` IMAGE_SUBSET ));
  TYPE_THEN `squ(reflAi (&:0) m) = IMAGE (reflAf (&:0)) (squ (left  m))` SUBGOAL_TAC;
  REWRITE_TAC[reflA_squ];
  AP_TERM_TAC;
  REWRITE_TAC[reflAi;left ;PAIR_SPLIT; ];
  INT_ARITH_TAC;
  ASM_MESON_TAC[];
  (* 3 *)
  CONJ_TAC;
  REWRITE_TAC[GSYM reflA_v_edge];
  EXPAND_TAC "G'";
  REWRITE_TAC[IMAGE2];
  UND 2;
  (* goal 3c *)
  MESON_TAC[image_imp];
  (* <2> *)
  CONJ_TAC;
  EXPAND_TAC "G'";
  EXPAND_TAC "e'";
  REWRITE_TAC[IMAGE2];
  ASM_MESON_TAC[image_imp];
  EXPAND_TAC "e'";
  TYPE_THEN `closure top2 (IMAGE (reflAf (&:0)) e) = IMAGE (reflAf (&:0)) (closure top2 e)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_closure);
  ASM_REWRITE_TAC[top2_top;reflA_homeo;top2_unions;];
  TYPE_THEN `edge e ` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;ISUBSET];
  MESON_TAC[ISUBSET;edge_euclid2;];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[GSYM reflA_pointI];
  IMATCH_MP_TAC  image_imp;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* <1> *)
  TYPE_THEN `p = left  (reflAi (&:0) p')` ABBREV_TAC ;
  TYPE_THEN `squ p' = IMAGE (reflAf (&:0) ) (squ p)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[reflA_squ;];
  AP_TERM_TAC;
  EXPAND_TAC "p";
  REWRITE_TAC[left ;reflAi;PAIR_SPLIT;];
  INT_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `p` EXISTS_TAC;
  (* LAST *)
  ASSUME_TAC top2_top;
  TYPE_THEN `homeomorphism (reflAf (&:0)) top2 top2` SUBGOAL_TAC;
  ASM_MESON_TAC[reflA_homeo];
  DISCH_TAC;
  ASSUME_TAC top2_unions;
  TYPE_THEN `squ p SUBSET (UNIONS (top2))` SUBGOAL_TAC;
  MESON_TAC[squ_euclid;top2_unions];
  DISCH_TAC;
  CONJ_TAC; (* split *)
  UND 12;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "e'";
  TYPE_THEN `closure top2 (IMAGE (reflAf (&:0)) (squ p)) = IMAGE (reflAf (&:0)) (closure top2 (squ p))` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_closure);
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  (* x *)
  DISCH_TAC;
  IMATCH_MP_TAC  (ISPEC `reflAf (&:0)` image_inj);
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  INJ_UNIV;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
  REWRITE_TAC [REWRITE_RULE[homeomorphism;BIJ;top2_unions] reflA_homeo;];
  CONJ_TAC;
    TYPE_THEN `edge e ` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;ISUBSET];
  MESON_TAC[ISUBSET;edge_euclid2;];
  IMATCH_MP_TAC  closure_euclid;
  REWRITE_TAC[squ_euclid];
  (* last'' *)
  IMATCH_MP_TAC  (ISPEC `reflAf (&:0)` image_inj);
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  INJ_UNIV;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
  REWRITE_TAC [REWRITE_RULE[homeomorphism;BIJ;top2_unions] reflA_homeo;];
  CONJ_TAC;
  REWRITE_TAC[squ_euclid];
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `UNIONS (ctop G)` EXISTS_TAC;
  ASM_REWRITE_TAC[component_unions;ctop_unions];
  REWRITE_TAC[DIFF;SUBSET];
  MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let v_edge_cases = prove_by_refinement(
  `!j m. closure top2 (v_edge j) (pointI m) ==> (j = m) \/ (j = down m)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[v_edge_closure;vc_edge];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[UNION;cell_clauses;INR IN_SING;plus_e12]);
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_MESON_TAC[];
  DISJ2_TAC;
  ASM_REWRITE_TAC[down;PAIR_SPLIT;];
  INT_ARITH_TAC;
  ]);;
  (* }}} *)

let squ_squc = prove_by_refinement(
  `!r n m. (IMAGE (reflBf r) (squ n) = squ m) ==>
    (IMAGE (reflBf r) (squc n) = squc m)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[GSYM squ_closure];
  TYPE_THEN `IMAGE (reflBf r) (closure top2 (squ n)) = closure top2 (IMAGE (reflBf r) (squ n))` SUBGOAL_TAC;
  IMATCH_MP_TAC  homeo_closure;
  ASM_REWRITE_TAC[top2_top;top2_unions;reflB_homeo;squ_euclid;];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let squ_squc_C = prove_by_refinement(
  `!n m. (IMAGE (reflCf) (squ n) = squ m) ==>
    (IMAGE (reflCf) (squc n) = squc m)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[GSYM squ_closure];
  TYPE_THEN `IMAGE (reflCf) (closure top2 (squ n)) = closure top2 (IMAGE (reflCf) (squ n))` SUBGOAL_TAC;
  IMATCH_MP_TAC  homeo_closure;
  ASM_REWRITE_TAC[top2_top;top2_unions;reflC_homeo;squ_euclid;];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let along_lemma8 = prove_by_refinement(
  `!G m n j x e. (segment G /\ (squ n SUBSET component  (ctop G) x) /\
     (v_edge j SUBSET squc n) /\ (closure top2 (v_edge j) (pointI m)) /\
    (G (v_edge j)) /\ G e /\ (closure top2 e (pointI m)) ==>
   (?p. e SUBSET closure top2 (squ p) /\
       (squ p SUBSET (component  (ctop G) x))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  USE_FIRST (MATCH_MP v_edge_cases);
  FIRST_ASSUM (DISJ_CASES_TAC);
  IMATCH_MP_TAC  along_lemma7;
  ASM_MESON_TAC[];
  KILL 3;
  REWR 4;
  REWR 2;
  KILL 7;
  (* INSERT lemmas here *)
  TYPE_THEN `e' = IMAGE (reflBf (&:0)) e ` ABBREV_TAC ;
  TYPE_THEN `G' = IMAGE2 (reflBf (&:0)) G` ABBREV_TAC ;
  TYPE_THEN `x' = reflBf (&:0) x` ABBREV_TAC ;
  TYPE_THEN `UNIONS (ctop G) x` SUBGOAL_TAC;
  TYPE_THEN `~(component  (ctop G) x = EMPTY)` SUBGOAL_TAC;
  USE 5(REWRITE_RULE[SUBSET]);
  TYPE_THEN `~(squ (n) = EMPTY)` SUBGOAL_TAC;
  ASM_MESON_TAC[cell_nonempty;cell_rules];
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `topology_ (ctop G)` SUBGOAL_TAC;
  ASM_MESON_TAC[ctop_top];
  ASM_SIMP_TAC [component_empty];
  DISCH_TAC;
  TYPE_THEN `component  (ctop G') x' = IMAGE (reflBf (&:0)) (component  (ctop G) x)` SUBGOAL_TAC;
  ASM_MESON_TAC[component_reflB;];
  DISCH_TAC;
  (*  gok to here *)
  TYPE_THEN `?p'. e' SUBSET closure top2 (squ p') /\ squ p' SUBSET component (ctop G') x'` SUBGOAL_TAC;
  IMATCH_MP_TAC  along_lemma7;
  TYPE_THEN `(reflBi (&:0))  m` EXISTS_TAC;
  TYPE_THEN `down (reflBi (&:0) n)` EXISTS_TAC;
  (SUBCONJ_TAC);
  (* 1st claus *)
  EXPAND_TAC "G'";
  IMATCH_MP_TAC reflB_segment;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  (* 2nd clause *)
  ASM_REWRITE_TAC[GSYM reflB_squ];
  (* goal 2c *)
  IMATCH_MP_TAC   (ISPEC `reflBf (&:0)` IMAGE_SUBSET );
  ASM_REWRITE_TAC[];
  (* 3 *)
  TYPE_THEN `squc (down (reflBi (&:0) n)) = IMAGE (reflBf (&:0)) (squc n)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM squ_squc);
  REWRITE_TAC[reflB_squ];
  DISCH_THEN_REWRITE;  (* end *)
  TYPE_THEN `v_edge (reflBi (&:0) m) = IMAGE (reflBf (&:0)) (v_edge (down m))` SUBGOAL_TAC;
  REWRITE_TAC[reflB_v_edge];
  AP_TERM_TAC ;
  REWRITE_TAC[reflBi;down;PAIR_SPLIT ];
  INT_ARITH_TAC;
  DISCH_THEN_REWRITE;
  CONJ_TAC;
  IMATCH_MP_TAC  IMAGE_SUBSET;
  ASM_REWRITE_TAC[];
  (* gok2 *)
  CONJ_TAC;
  EXPAND_TAC "G'";
  REWRITE_TAC[IMAGE2];
  UND 2;
  (* goal 3c *)
  MESON_TAC[image_imp];
  (* <2> gok1 *)
  CONJ_TAC;
  EXPAND_TAC "G'";
  EXPAND_TAC "e'";
  REWRITE_TAC[IMAGE2];
  ASM_MESON_TAC[image_imp];
  EXPAND_TAC "e'";
  (* 2 total *)
  TYPE_THEN `closure top2 (IMAGE (reflBf (&:0)) e) = IMAGE (reflBf (&:0)) (closure top2 e)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_closure);
  ASM_REWRITE_TAC[top2_top;reflB_homeo;top2_unions;];
  TYPE_THEN `edge e ` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;ISUBSET];
  MESON_TAC[ISUBSET;edge_euclid2;];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[GSYM reflB_pointI];
  IMATCH_MP_TAC  image_imp;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* <1> *)
  TYPE_THEN `p = down  (reflBi (&:0) p')` ABBREV_TAC ;
  TYPE_THEN `squ p' = IMAGE (reflBf (&:0) ) (squ p)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[reflB_squ;];
  AP_TERM_TAC;
  EXPAND_TAC "p";
  REWRITE_TAC[down ;reflBi;PAIR_SPLIT;];
  INT_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `p` EXISTS_TAC;
  (* LAST *)
  ASSUME_TAC top2_top;
  TYPE_THEN `homeomorphism (reflBf (&:0)) top2 top2` SUBGOAL_TAC;
  ASM_MESON_TAC[reflB_homeo];
  DISCH_TAC;
  ASSUME_TAC top2_unions;
  TYPE_THEN `squ p SUBSET (UNIONS (top2))` SUBGOAL_TAC;
  MESON_TAC[squ_euclid;top2_unions];
  DISCH_TAC;
  CONJ_TAC; (* split *)
  UND 12;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "e'";
  TYPE_THEN `closure top2 (IMAGE (reflBf (&:0)) (squ p)) = IMAGE (reflBf (&:0)) (closure top2 (squ p))` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_closure);
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  (* x *)
  DISCH_TAC;
  IMATCH_MP_TAC  (ISPEC `reflBf (&:0)` image_inj);
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  INJ_UNIV;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
  REWRITE_TAC [REWRITE_RULE[homeomorphism;BIJ;top2_unions] reflB_homeo;];
  CONJ_TAC;
    TYPE_THEN `edge e ` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;ISUBSET];
  MESON_TAC[ISUBSET;edge_euclid2;];
  IMATCH_MP_TAC  closure_euclid;
  REWRITE_TAC[squ_euclid];
  (* last'' *)
  IMATCH_MP_TAC  (ISPEC `reflBf (&:0)` image_inj);
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  INJ_UNIV;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
  REWRITE_TAC [REWRITE_RULE[homeomorphism;BIJ;top2_unions] reflB_homeo;];
  CONJ_TAC;
  REWRITE_TAC[squ_euclid];
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `UNIONS (ctop G)` EXISTS_TAC;
  ASM_REWRITE_TAC[component_unions;ctop_unions];
  REWRITE_TAC[DIFF;SUBSET];
  MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let along_lemma9 = prove_by_refinement(
  `!G m n e' x e. (segment G /\ (squ n SUBSET component  (ctop G) x) /\
     (e' SUBSET squc n) /\ (closure top2 e' (pointI m)) /\ (edge e') /\
    (G e') /\ G e /\ (closure top2 e (pointI m)) ==>
   (?p. e SUBSET closure top2 (squ p) /\
       (squ p SUBSET (component  (ctop G) x))))`,
  (* {{{ proof *)
  [
    REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[edge]);
  REP_BASIC_TAC;
  FIRST_ASSUM (DISJ_CASES_TAC);
  IMATCH_MP_TAC  along_lemma8;
  ASM_MESON_TAC[];
  TYPE_THEN `edge e` SUBGOAL_TAC;
  ASM_MESON_TAC[segment;ISUBSET];
  ASM_SIMP_TAC[];
  DISCH_TAC;
  KILL 3;
  REWR 4;
  REWR 2;
  REWR 5;
  KILL 8;
  (* INSERT lemmas here *)
  TYPE_THEN `e' = IMAGE (reflCf) e ` ABBREV_TAC ;
  TYPE_THEN `G' = IMAGE2 (reflCf) G` ABBREV_TAC ;
  TYPE_THEN `x' = reflCf x` ABBREV_TAC ;
  TYPE_THEN `UNIONS (ctop G) x` SUBGOAL_TAC;
  TYPE_THEN `~(component  (ctop G) x = EMPTY)` SUBGOAL_TAC;
  USE 6(REWRITE_RULE[SUBSET]);
  TYPE_THEN `~(squ (n) = EMPTY)` SUBGOAL_TAC;
  ASM_MESON_TAC[cell_nonempty;cell_rules];
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `topology_ (ctop G)` SUBGOAL_TAC;
  ASM_MESON_TAC[ctop_top];
  ASM_SIMP_TAC [component_empty];
  DISCH_TAC;
  TYPE_THEN `component  (ctop G') x' = IMAGE (reflCf) (component  (ctop G) x)` SUBGOAL_TAC;
  ASM_MESON_TAC[component_reflC;];
  DISCH_TAC;
  (*  gok to here *)
  TYPE_THEN `?p'. e' SUBSET closure top2 (squ p') /\ squ p' SUBSET component (ctop G') x'` SUBGOAL_TAC;
  IMATCH_MP_TAC  along_lemma8;
  TYPE_THEN `(reflCi)  m` EXISTS_TAC;
  TYPE_THEN `(reflCi n)` EXISTS_TAC;
  TYPE_THEN `reflCi m'` EXISTS_TAC;
  (SUBCONJ_TAC);
  (* 1st claus *)
  EXPAND_TAC "G'";
  IMATCH_MP_TAC reflC_segment;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  (* 2nd clause *)
  ASM_REWRITE_TAC[GSYM reflC_squ];
  (* goal 2c *)
  IMATCH_MP_TAC   (ISPEC `reflCf` IMAGE_SUBSET );
  ASM_REWRITE_TAC[];
  (* 3 *)
  TYPE_THEN `squc ( (reflCi n)) = IMAGE (reflCf) (squc n)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM squ_squc_C);
  REWRITE_TAC[reflC_squ];
  DISCH_THEN_REWRITE;  (* end *)
  TYPE_THEN `v_edge (reflCi  m') = IMAGE (reflCf ) (h_edge ( m'))` SUBGOAL_TAC;
  REWRITE_TAC[reflC_hv_edge];
  DISCH_THEN_REWRITE;
  CONJ_TAC;
  IMATCH_MP_TAC  IMAGE_SUBSET;
  ASM_REWRITE_TAC[];
  (* gok2 *)
  (* INSERT *)
  TYPE_THEN `!e. (edge e) ==> (closure top2 (IMAGE (reflCf ) e) = IMAGE (reflCf) (closure top2 e))` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_closure);
  ASM_REWRITE_TAC[top2_top;reflC_homeo;top2_unions;];
  IMATCH_MP_TAC  edge_euclid2;
  ASM_REWRITE_TAC[];
  DISCH_TAC ;
  TYPE_THEN `edge (h_edge m')` SUBGOAL_TAC;
  ASM_MESON_TAC[edge];
  DISCH_TAC;
  ASM_SIMP_TAC[];
  REWRITE_TAC[GSYM reflC_pointI];
  CONJ_TAC;
  ASM_MESON_TAC[image_imp];
  (* to here *)
  CONJ_TAC;
  EXPAND_TAC "G'";
  REWRITE_TAC[IMAGE2];
  UND 2;
  (* goal 3c *)
  MESON_TAC[image_imp];
  (* <2> gok1 *)
  CONJ_TAC;
  EXPAND_TAC "G'";
  EXPAND_TAC "e'";
  REWRITE_TAC[IMAGE2];
  ASM_MESON_TAC[image_imp];
  EXPAND_TAC "e'";
  (* 2 total *)
  ASM_SIMP_TAC[];
  IMATCH_MP_TAC  image_imp;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* <1> *)
  TYPE_THEN `p = reflCi p'` ABBREV_TAC ;
  TYPE_THEN `squ p' = IMAGE (reflCf ) (squ p)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[reflC_squ;];
  AP_TERM_TAC;
  EXPAND_TAC "p";
  REWRITE_TAC[reflCi_inv;PAIR_SPLIT;];
  DISCH_TAC;
  TYPE_THEN `p` EXISTS_TAC;
  (* LAST *)
  ASSUME_TAC top2_top;
  TYPE_THEN `homeomorphism (reflCf) top2 top2` SUBGOAL_TAC;
  ASM_MESON_TAC[reflC_homeo];
  DISCH_TAC;
  ASSUME_TAC top2_unions;
  TYPE_THEN `squ p SUBSET (UNIONS (top2))` SUBGOAL_TAC;
  MESON_TAC[squ_euclid;top2_unions];
  DISCH_TAC;
  TYPE_THEN `closure top2 (IMAGE (reflCf) (squ p)) = IMAGE (reflCf) (closure top2 (squ p))` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM homeo_closure);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  CONJ_TAC; (* split *)
  IMATCH_MP_TAC  (ISPEC `reflCf` image_inj);
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  INJ_UNIV;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
  REWRITE_TAC [REWRITE_RULE[homeomorphism;BIJ;top2_unions] reflC_homeo;];
  CONJ_TAC;
  ASM_MESON_TAC[edge_euclid2];
  CONJ_TAC;
  IMATCH_MP_TAC  closure_euclid;
  REWRITE_TAC[squ_euclid];
  UND 21;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  REWRITE_TAC[reflC_squ];
  TYPE_THEN `reflCi p = p'` SUBGOAL_TAC;
  EXPAND_TAC "p";
  REWRITE_TAC[reflCi_inv];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  (* last'' *)
  UND 13;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  (ISPEC `reflCf` image_inj);
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  INJ_UNIV;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
  REWRITE_TAC [REWRITE_RULE[homeomorphism;BIJ;top2_unions] reflC_homeo;];
  CONJ_TAC;
  REWRITE_TAC[squ_euclid];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `UNIONS (ctop G)` EXISTS_TAC;
  ASM_REWRITE_TAC[component_unions;ctop_unions];
  REWRITE_TAC[DIFF;SUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let along_lemma10 = prove_by_refinement(
  `!G x. (segment G /\ ~(component  (ctop G) x  = EMPTY) ) ==>
    inductive_set G
        { e | (G e /\ (?p. (e SUBSET squc p) /\
              (squ p SUBSET component  (ctop G) x)) ) } `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `S = { e | (G e /\ (?p. (e SUBSET squc p) /\ (squ p SUBSET component  (ctop G) x)) ) } ` ABBREV_TAC ;
  REWRITE_TAC[inductive_set];
  CONJ_TAC;
  EXPAND_TAC "S";
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  CONJ_TAC;
  TYPE_THEN `(?m. squ m SUBSET (component  (ctop G) x))` SUBGOAL_TAC;
  IMATCH_MP_TAC  comp_squ;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `(?p e. G e /\ e SUBSET closure top2 (squ p) /\ squ p SUBSET component (ctop G) x)` SUBGOAL_TAC;
  IMATCH_MP_TAC  comp_squ_adj;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  UND 3;
  REWRITE_TAC[EMPTY_EXISTS ];
  EXPAND_TAC "S";
  REWRITE_TAC[];
  REWRITE_TAC [squ_closure];
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `p` EXISTS_TAC;
  ASM_REWRITE_TAC[GSYM squ_closure];
  REP_BASIC_TAC;
  UND 5;
  EXPAND_TAC "S";
  REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `edge C /\ edge C'` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SUBSET]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `(?m. closure top2 C INTER closure top2 C' = {(pointI m)})` SUBGOAL_TAC;
  IMATCH_MP_TAC  edge_inter;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  REWRITE_TAC[GSYM squ_closure];
  IMATCH_MP_TAC  along_lemma9;
  RULE_ASSUM_TAC (REWRITE_RULE[INTER;eq_sing;]);
  TYPE_THEN `m` EXISTS_TAC;
  TYPE_THEN `p` EXISTS_TAC;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let along_lemma11 = prove_by_refinement(
  `!G  x e .  (segment G /\ ~(component  (ctop G) x  = EMPTY)  /\
     (G e)) ==>
   (?p. (e SUBSET squc p) /\ (squ p SUBSET component  (ctop G) x))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `S = {e | (G e /\ (?p. (e SUBSET squc p) /\ (squ p SUBSET component  (ctop G) x)) ) }` ABBREV_TAC ;
  TYPE_THEN ` S = G` SUBGOAL_TAC;
  COPY  2;
  UND 4;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `inductive_set G S` SUBGOAL_TAC;
  EXPAND_TAC "S";
  IMATCH_MP_TAC  along_lemma10;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[inductive_set];
  EXPAND_TAC "S";
  DISCH_TAC;
  USE 4 GSYM;
  PROOF_BY_CONTR_TAC;
  UND 0;
  REWRITE_TAC[];
  ONCE_ASM_REWRITE_TAC[];
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)


(* along_lemma11
   is essentially the proof that there are only two connected
   components (because there are only two possible instantiations of p
   Come back and finish the proof  of the Jordan curve.  *)


(* ------------------------------------------------------------------ *)
(* SECTION I *)
(* ------------------------------------------------------------------ *)

(* ALL about graphs *)

(*** JRH systematically changed (Y,X)graph to (X,Y)graph for all X and Y,
     and made corresponding changes to other type annotations.
     The core now alphabetically sorts the type variables in a definition.
 ***)

let (mk_graph_t,dest_graph_t) = abbrev_type
   `:(A->bool)#(B->bool)#(B->(A->bool))` "graph_t";;

let graph_vertex = jordan_def
   `graph_vertex (G:(A,B)graph_t) = FST (dest_graph_t G)`;;

let graph_edge = jordan_def
   `graph_edge (G:(A,B)graph_t) = part1 (dest_graph_t G)`;;

let graph_inc = jordan_def
   `graph_inc (G:(A,B)graph_t) = drop1 (dest_graph_t G)`;;

let graph = jordan_def `graph (G:(A,B)graph_t) <=>
   (IMAGE (graph_inc G) (graph_edge G)) SUBSET
   { s | (s SUBSET (graph_vertex G)) /\ (s HAS_SIZE 2) }`;;

let graph_incident = jordan_def `graph_incident
   (G:(A,B)graph_t) e x <=>
   (graph_edge G e) /\ (graph_inc G e x)`;;

let graph_iso = jordan_def
   `graph_iso f (G:(A,B)graph_t) (H:(A',B')graph_t) <=>
   (?u v. (f = (u,v)) /\ (BIJ u (graph_vertex G) (graph_vertex H)) /\
   (BIJ v (graph_edge G) (graph_edge H)) /\
   (!e. (graph_edge G e) ==>
      (graph_inc H (v e) = IMAGE u (graph_inc G e))))`;;

(* specify a graph by
   { {a,b}, .... } of endpoints of edges.  *)

let mk_simple_graph = jordan_def `mk_simple_graph (E:(A->bool)->bool) =
  mk_graph_t
  (UNIONS E, (E:(A->bool)->bool),
   (\ (x:A->bool) (y:A). (x y)))`;;

let K33 = jordan_def `K33 = mk_simple_graph
   { {1,10}, {2,10}, {3,10},
     {1,20}, {2,20}, {3,20},
     {1,30}, {2,30}, {3,30} }`;;

let graph_del = jordan_def `graph_del (G:(A,B)graph_t) V E =
  mk_graph_t
   ((graph_vertex G DIFF V),
    (graph_edge G DIFF
        (E UNION { (e:B) | ?(v:A). (V v /\ graph_incident G e v ) })),
    (graph_inc G))`;;

let graph_path = jordan_def `graph_path (G:(A,B)graph_t) f n <=>
   (?v e . (f = (v,e)) /\ (INJ v { m | m <=| n } (graph_vertex G)) /\
   (INJ e { m | m <| n } (graph_edge G)) /\
   (!i. (i <| n )  ==>
         (graph_inc G (e i) = {(v  i), (v (SUC i))})))`;;

let graph_cycle = jordan_def `graph_cycle (G:(A,B)graph_t) f n <=>
   (?v e . (f = (v,e)) /\ (INJ v { m | m <| n } (graph_vertex G)) /\
   (INJ e { m | m <| n } (graph_edge G)) /\
   (!i. (i <| n )  ==>
         (graph_inc G (e i) = {(v  i), (v ((SUC i) %| (n)))})))`;;

let graph_connected = jordan_def `graph_connected (G:(A,B)graph_t) <=>
  !v v'. (graph_vertex G v) /\ (graph_vertex G v') /\ ~(v = v') ==>
   (?f n. (graph_path G f n) /\ (FST f 0 = v) /\ (FST f n = v'))`;;

let graph_2_connected = jordan_def `graph_2_connected (G:(A,B)graph_t) <=>
  (graph_connected G) /\
  (!v. (graph_vertex G v) ==> (graph_connected
     (graph_del G {v} EMPTY)))`;;

let simple_arc = jordan_def `simple_arc (U:(A->bool)->bool) C <=>
   (?f. (C = IMAGE f { x | &.0 <= x /\ x <= &.1}) /\
   (continuous f (top_of_metric(UNIV,d_real)) U) /\
   (INJ f { x | &.0 <= x /\ x <= &.1} (UNIONS U)))`;;

let simple_closed_curve = jordan_def
   `simple_closed_curve (U:(A->bool)->bool) C <=>
   (?f. (C = IMAGE f { x | &.0 <= x /\ x <= &.1}) /\
   (continuous f (top_of_metric(UNIV,d_real)) U) /\
   (INJ f { x | &.0 <= x /\ x < &.1} (UNIONS U)) /\
   (f (&.0) = f (&.1)))`;;

let simple_polygonal_arc = jordan_def
   `simple_polygonal_arc PE C <=>
    (simple_arc (top_of_metric(euclid 2,d_euclid)) C) /\
    (?E. (C SUBSET UNIONS E) /\ (FINITE E) /\ (PE E))`;;

let simple_polygonal_curve = jordan_def
   `simple_polygonal_curve PE C <=>
    (simple_closed_curve (top_of_metric(euclid 2,d_euclid)) C) /\
    (?E. (C SUBSET UNIONS E) /\ (FINITE E) /\ (PE E))`;;

let hv_line = jordan_def
   `hv_line E <=> (!e. (E e) ==> (?x y. (e = mk_line (point x) (point y)) /\
      ((FST x = FST y) \/ (SND x = SND y))))`;;

let p_conn = jordan_def
   `p_conn A x y <=> (?C. (simple_polygonal_arc hv_line C) /\
     (C SUBSET A) /\ (C x) /\ (C y))`;;

let subf = jordan_def
   `subf A (f:A->B) g x = if (A x) then (f x) else (g x)`;;

let min_real_le = prove_by_refinement(
  `!x y. (min_real x y <= x) /\ (min_real x y <= y)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[min_real];
  COND_CASES_TAC;
  UND 0;
  REAL_ARITH_TAC;
  UND 0;
  REAL_ARITH_TAC ;
  ]);;
  (* }}} *)

let subf_lemma = prove_by_refinement(
  `!X dX B (x:A).
     (metric_space (X,dX)) /\ (closed_ (top_of_metric(X,dX)) B) /\
     (~(B x)) /\ (X x) ==>
     (?delta. (&0 < delta) /\ (!y. (dX x y < delta) ==> (~(B y))))`,
  (* {{{ proof *)

  [
  REWRITE_TAC[closed;open_DEF ];
  REP_BASIC_TAC;
  UND 2;
  UND 3;
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  REP_BASIC_TAC;
  TYPE_THEN `(X DIFF B) x` SUBGOAL_TAC;
  REWRITE_TAC[DIFF];
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPEL_THEN [`X`;`dX`;`(X DIFF B)`;`x`] (fun t-> ASSUME_TAC (ISPECL t open_ball_nbd)); (* // *)
  REP_BASIC_TAC;
  REWR 6;
  TYPE_THEN `e` EXISTS_TAC;
  UND 6;
  REWRITE_TAC[open_ball;SUBSET;DIFF;];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_MESON_TAC[ISUBSET ;];
  ]);;

  (* }}} *)

let subf_cont = prove_by_refinement(
  `!X dX Y dY A B (f:A->B) g.
     ((metric_space (X,dX)) /\ (metric_space (Y,dY)) /\
     (closed_ (top_of_metric(X,dX)) A ) /\
     (closed_ (top_of_metric(X,dX)) B ) /\
     (metric_continuous f (A,dX) (Y,dY)) /\
     (metric_continuous g (B,dX) (Y,dY)) /\
     (!x. (A x /\ B x) ==> (f x = g x))) ==>
     (metric_continuous (subf A f g) (A UNION B,dX) (Y,dY))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  DISCH_ALL_TAC;
  DISCH_ALL_TAC;
  RIGHT_TAC "delta";
  DISCH_TAC;
  REWRITE_TAC[UNION];
  TYPE_THEN `(A x \/ ~(A x)) /\ (B x \/ (~(B x)))` (fun t-> MP_TAC (TAUT  t ));
  DISCH_THEN (fun t -> MP_TAC (REWRITE_RULE[GSYM DISJ_ASSOC;RIGHT_AND_OVER_OR;LEFT_AND_OVER_OR] t));
  REP_CASES_TAC;
  TYPEL_THEN [`x`;`epsilon`] (USE 4 o ISPECL);
  TYPEL_THEN [`x`;`epsilon`] (USE 5 o ISPECL);
  REP_BASIC_TAC;
  REWR 8;
  REWR 9;
  TYPE_THEN `min_real delta delta'` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[min_real];
  COND_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `A y \/ (~(A y) /\ B y)` SUBGOAL_TAC;
  UND 9;
  MESON_TAC[];
  DISCH_THEN DISJ_CASES_TAC;
  REWRITE_TAC[subf];
  ASM_REWRITE_TAC[];
  UND 12;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 8;
  (* save_goal "ss" *)
  TYPE_THEN `min_real delta delta' <= delta'` SUBGOAL_TAC;
  REWRITE_TAC[min_real_le];
  REAL_ARITH_TAC;
  (* 1b case *)
  REWRITE_TAC[subf];
  ASM_REWRITE_TAC[];
  TYPE_THEN `f x = g x` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  UND 10;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 8;
  TYPE_THEN `min_real delta delta' <= delta` SUBGOAL_TAC;
  REWRITE_TAC[min_real_le];
  REAL_ARITH_TAC ;
  (* 2nd case *)
  TYPE_THEN `X x` SUBGOAL_TAC;
  UND 2;
  REWRITE_TAC[closed;open_DEF;SUBSET ;];
  REP_BASIC_TAC;
  TSPEC  `x` 8;
  UND 8;
  ASM_REWRITE_TAC[];
  UND 0;
  SIMP_TAC[GSYM top_of_metric_unions];
  DISCH_TAC;
  TYPE_THEN `(?delta. (&0 < delta) /\ (!y. (dX x y < delta) ==> (~(B y))))` SUBGOAL_TAC;
  IMATCH_MP_TAC  subf_lemma;
  TYPE_THEN `X` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPEL_THEN [`x`;`epsilon`] (USE 4 o ISPECL);
  REP_BASIC_TAC;
  REWR 4;
  TYPE_THEN `min_real delta delta'` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[min_real];
  COND_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `A y` SUBGOAL_TAC;
  TYPE_THEN `~(B y) ==> A y` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 4;
  TYPE_THEN `min_real delta delta' <= delta` SUBGOAL_TAC;
  REWRITE_TAC[min_real_le];
  REAL_ARITH_TAC;
  REWRITE_TAC[subf];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 4;
  TYPE_THEN `min_real delta delta' <= delta'` SUBGOAL_TAC;
  REWRITE_TAC[min_real_le];
  REAL_ARITH_TAC;
  (* 2 LEFT *)
  TYPE_THEN `X x` SUBGOAL_TAC;
  UND 3;
  REWRITE_TAC[closed;open_DEF;SUBSET ;];
  REP_BASIC_TAC;
  TSPEC  `x` 8;
  UND 8;
  ASM_REWRITE_TAC[];
  UND 0;
  SIMP_TAC[GSYM top_of_metric_unions];
  DISCH_TAC;
  TYPE_THEN `(?delta. (&0 < delta) /\ (!y. (dX x y < delta) ==> (~(A y))))` SUBGOAL_TAC;
  IMATCH_MP_TAC  subf_lemma;
  TYPE_THEN `X` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPEL_THEN [`x`;`epsilon`] (USE 5 o ISPECL);
  REP_BASIC_TAC;
  REWR 5;
  TYPE_THEN `min_real delta delta'` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[min_real];
  COND_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `~(A y)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 5;
  TYPE_THEN `min_real delta delta' <= delta` SUBGOAL_TAC;
  REWRITE_TAC[min_real_le];
  REAL_ARITH_TAC;
  REWRITE_TAC[subf];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `B y` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  UND 5;
  TYPE_THEN `min_real delta delta' <= delta'` SUBGOAL_TAC;
  REWRITE_TAC[min_real_le];
  REAL_ARITH_TAC;
  (* 1 LEFT *)
  TYPE_THEN `&1` EXISTS_TAC;
  ASM_MESON_TAC [REAL_ARITH `&0 < &1`];
  ]);;
  (* }}} *)

let p_conn_subset = prove_by_refinement(
  `!A B x y. (A SUBSET B) /\ (p_conn A x y) ==> (p_conn B x y)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[p_conn];
  REP_BASIC_TAC;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  ]);;
  (* }}} *)

let mk_line_symm = prove_by_refinement(
  `!x y. mk_line x y = mk_line y x`,
  (* {{{ proof *)
  [
  REWRITE_TAC[mk_line];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  EQ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `(&1 - t)` EXISTS_TAC;
  ONCE_REWRITE_TAC [euclid_add_comm];
  ASM_REWRITE_TAC[REAL_ARITH `(&1 - (&1 - t)) = t`];
  REP_BASIC_TAC;
  TYPE_THEN `(&1 - t)` EXISTS_TAC;
  ONCE_REWRITE_TAC [euclid_add_comm];
  ASM_REWRITE_TAC[REAL_ARITH `(&1 - (&1 - t)) = t`];
  ]);;
  (* }}} *)

let mk_line_sub = prove_by_refinement(
  `!x y z. ( ~(x = z) /\ (mk_line x y z)) ==>
        (mk_line x y = mk_line x z)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[mk_line];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  EQ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `~(t = &1)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWR 0;
  UND 0;
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale0;euclid_scale_one;euclid_rzero];
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `s = (&1 /(&1 - t))` ABBREV_TAC;
  TYPE_THEN `(t' - t)*s` EXISTS_TAC;
  ASM_REWRITE_TAC[euclid_ldistrib;GSYM euclid_add_assoc;euclid_scale_act;GSYM euclid_rdistrib;];
  TYPE_THEN `(&1 - t) * s = &1` SUBGOAL_TAC;
  EXPAND_TAC "s";
  IMATCH_MP_TAC  REAL_DIV_LMUL;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `(t' - t) * s + (&1 - (t' - t) * s) * t = (t' - t) *((&1- t)* s) + t ` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(&1 - (t' - t) * s)*(&1 - t) = (&1 - t) - (t' - t)*(&1-t)*s` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[REAL_ARITH  `((t' - t)* &1 + t = t') /\ (&1 - t - (t' - t)* &1 = (&1 - t'))`];
  (* 2nd half *)
  REP_BASIC_TAC;
  UND 2;
  ASM_REWRITE_TAC[euclid_ldistrib;GSYM euclid_add_assoc;euclid_scale_act;GSYM euclid_rdistrib;];
  DISCH_THEN_REWRITE;
  TYPE_THEN `t' + (&1 - t')*t` EXISTS_TAC;
  TYPE_THEN `(&1 - (t' + (&1 - t') * t)) = ((&1 - t') * (&1 - t))` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  ]);;
  (* }}} *)

let mk_line_2 = prove_by_refinement(
  `!x y p q. (mk_line x y p) /\ (mk_line x y q) /\ (~(p = q)) ==>
    (mk_line x y = mk_line p q)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `x = p`  ASM_CASES_TAC ;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  mk_line_sub;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[mk_line_sub;mk_line_symm];
  ]);;
  (* }}} *)

let mk_line_inter = prove_by_refinement(
  `!x y p q. ~(mk_line x y = mk_line p q) ==>
    (?z. (mk_line x y INTER mk_line p q) SUBSET {z} )`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(?z. (mk_line x y INTER mk_line p q) z)` ASM_CASES_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `z` EXISTS_TAC;
  REWRITE_TAC[INTER;SUBSET;INR IN_SING;];
  REP_BASIC_TAC;
  UND 1;
  REWRITE_TAC[INTER];
  REP_BASIC_TAC;
  ASM_MESON_TAC[mk_line_2];
  REWRITE_TAC[SUBSET;INR IN_SING];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let mk_line_fin_inter = prove_by_refinement(
  `!E. (FINITE E) /\ (!e. (E e) ==> (?x y. e = mk_line x y)) ==>
    (?X. (FINITE X) /\
    (!e f z. (E e) /\ (E f) /\ ~(e = f) /\ e z /\ f z ==> (X z)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `E2 = { (e,f) | (E e) /\ (E f) /\ (~(e = f)) }` ABBREV_TAC;
  TYPE_THEN `EE = { (e,f) | (E e) /\ (E f) }` ABBREV_TAC;
  (*   *)
  TYPE_THEN `FINITE EE` SUBGOAL_TAC;
  EXPAND_TAC "EE";
  IMATCH_MP_TAC  (INR FINITE_PRODUCT);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (*   *)
  TYPE_THEN `FINITE E2` SUBGOAL_TAC;
  EXPAND_TAC "E2";
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `EE` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "EE";
  EXPAND_TAC "E2";
  REWRITE_TAC[SUBSET;];
  MESON_TAC[];
  DISCH_TAC;
  (*  *)
  TYPE_THEN `E3 = IMAGE (\u. (FST u INTER SND u)) E2` ABBREV_TAC;
  TYPE_THEN `FINITE E3` SUBGOAL_TAC;
  EXPAND_TAC "E3";
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (*  *)
  TYPE_THEN `UNIONS E3` EXISTS_TAC;
  CONJ_TAC;
  ASM_SIMP_TAC[FINITE_UNIONS];
  GEN_TAC;
  EXPAND_TAC "E3";
  EXPAND_TAC "E2";
  REWRITE_TAC[IMAGE];
  CONV_TAC (dropq_conv "x");
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `e` (WITH 0 o ISPEC);
  TYPE_THEN `f` (USE 0 o ISPEC);
  UND 0;
  UND 12;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  (*  *)
  TYPE_THEN `(?z. (mk_line x y INTER mk_line x' y') SUBSET {z} )` SUBGOAL_TAC;
  IMATCH_MP_TAC mk_line_inter;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `{z}` EXISTS_TAC;
  ASM_REWRITE_TAC[FINITE_SING ];
  REP_BASIC_TAC;
  EXPAND_TAC "E3";
  EXPAND_TAC "E2";
  REWRITE_TAC[IMAGE];
  REWRITE_TAC[UNIONS];
  CONV_TAC (dropq_conv "x");
  CONV_TAC (dropq_conv "u");
  REWRITE_TAC[INTER];
  TYPE_THEN `e` EXISTS_TAC;
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let euclid_euclid0 = prove_by_refinement(
  `!n. (euclid n (euclid0))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[euclid0;euclid];
  ]);;
  (* }}} *)

let euclid0_point = prove_by_refinement(
  `euclid0 = point(&0,&0)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[point_split;euclid_euclid0];
  REWRITE_TAC[euclid0];
  ]);;
  (* }}} *)

let EVEN2 = prove_by_refinement(
  `EVEN 0 /\ ~(EVEN 1) /\ (EVEN 2) /\ ~(EVEN 3) /\
  (EVEN 4) /\ ~(EVEN 5)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[EVEN; ARITH_RULE `(1 = SUC 0) /\ (2 = SUC 1) /\ (3 = SUC 2) /\ (4 = SUC 3) /\ (5 = SUC 4)`];
  ]);;
  (* }}} *)

let h_seg_openball = prove_by_refinement(
  `!x e e'. (&0 < e) /\ (&0 <= e') /\ (e' < e) /\ (euclid 2 x) ==>
     (mk_segment x (x + e' *# e1) SUBSET
              (open_ball(euclid 2,d_euclid)) x e)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[open_ball;mk_segment;SUBSET;];
  REP_BASIC_TAC;
  USE 4 (SYM);
  UND 4;
  REWRITE_TAC[GSYM euclid_add_assoc;euclid_ldistrib;GSYM euclid_rdistrib];
  REWRITE_TAC[REAL_ARITH `a + &1 - a = &1`;euclid_scale_one;euclid_scale_act];
  TYPE_THEN  `x'' = (((&1 - a) * e') *# e1)` ABBREV_TAC ;
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `euclid 2 x''` SUBGOAL_TAC;
  EXPAND_TAC "x''";
  IMATCH_MP_TAC  euclid_scale_closure;
  REWRITE_TAC[e1;euclid_point];
  DISCH_TAC;
  SUBCONJ_TAC;
  EXPAND_TAC "x'";
  IMATCH_MP_TAC  euclid_add_closure;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `!x y.  d_euclid x y = d_euclid (x+euclid0) y ` SUBGOAL_TAC;
  REWRITE_TAC[euclid_rzero];
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]);
  EXPAND_TAC "x'";
  ASSUME_TAC euclid_euclid0;
  KILL 7;
  TYPE_THEN `d_euclid (euclid_plus x euclid0) (euclid_plus x x'') = d_euclid euclid0 x''` SUBGOAL_TAC;
  ASM_MESON_TAC[metric_translate_LEFT];
  DISCH_THEN_REWRITE;
  EXPAND_TAC "x''";
  REWRITE_TAC[e1;point_scale];
  REDUCE_TAC;
  REWRITE_TAC[euclid0_point;d_euclid_point;];
  REDUCE_TAC;
  REWRITE_TAC[EXP_2;ARITH_RULE `0 *| 0 = 0`];
  REDUCE_TAC;
  REWRITE_TAC[REAL_ARITH `&0 - x = --x`;REAL_POW_NEG;EVEN2];
  TYPE_THEN `&0 <= (&1 - a) * e'` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_LE_MUL;
  ASM_REWRITE_TAC[];
  UND 5;
  REAL_ARITH_TAC;
  ASM_SIMP_TAC[POW_2_SQRT;];
  DISCH_TAC;
  ASM_CASES_TAC `a = &0`;
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(&1 - a) * e' < &1 * e ==> (&1 - a) * e' <  e` SUBGOAL_TAC;
  REAL_ARITH_TAC;
  DISCH_THEN IMATCH_MP_TAC ;
  IMATCH_MP_TAC  REAL_LT_MUL2;
  ASM_REWRITE_TAC[];
  UND 5;
  UND 6;
  UND 11;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let openball_convex = prove_by_refinement(
  `!x e n. (convex (open_ball (euclid n,d_euclid) x e))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[convex;open_ball;SUBSET;mk_segment;];
  REP_BASIC_TAC;
  USE 0 SYM;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  EXPAND_TAC "x''";
  IMATCH_MP_TAC  (euclid_add_closure);
  CONJ_TAC THEN (IMATCH_MP_TAC  euclid_scale_closure) THEN (ASM_REWRITE_TAC[]);
  DISCH_TAC;
  TYPE_THEN `d_euclid x x'' = d_euclid (a *# x + (&1 - a) *# x) x''` SUBGOAL_TAC;
  REWRITE_TAC[trivial_lin_combo];
  DISCH_THEN_REWRITE;
  EXPAND_TAC "x''";
  (* special case *)
  ASM_CASES_TAC `a = &0` ;
  UND 10;
  DISCH_THEN_REWRITE;
  REDUCE_TAC;
  ASM_REWRITE_TAC [euclid_scale0;euclid_scale_one;euclid_lzero;];
  TYPE_THEN `(!d. (?u v. (d <= u + v) /\ (u < a*e) /\ (v <= (&1- a)*e))  ==> (d < e))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `u + v < (a*e) + (&1 - a)*e` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_LTE_ADD2;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[GSYM REAL_RDISTRIB;REAL_ARITH `(a + &1 - a = &1) /\ (&1 * C = C )`];
  UND 13;
  REAL_ARITH_TAC ;
  DISCH_THEN IMATCH_MP_TAC ;
  TYPE_THEN `z = a *# x' + (&1 - a) *# x` ABBREV_TAC;
  TYPE_THEN `d_euclid (a *# x + (&1 - a)*# x) z` EXISTS_TAC;
  TYPE_THEN `d_euclid z x''` EXISTS_TAC;
  TYPE_THEN `euclid n z` SUBGOAL_TAC;
  EXPAND_TAC "z";
  IMATCH_MP_TAC  (euclid_add_closure);
  CONJ_TAC THEN (IMATCH_MP_TAC  euclid_scale_closure) THEN (ASM_REWRITE_TAC[]);
  DISCH_TAC;
  CONJ_TAC;
  EXPAND_TAC "x''";
  IMATCH_MP_TAC  metric_space_triangle;
  TYPE_THEN `euclid n` EXISTS_TAC;
  REWRITE_TAC[metric_euclid];
  ASM_REWRITE_TAC[trivial_lin_combo];
  CONJ_TAC;
  EXPAND_TAC "z";
  TYPE_THEN `(d_euclid (euclid_plus (a *# x) ((&1 - a) *# x)) (euclid_plus (a *# x') ((&1 - a) *# x))) = d_euclid  (a *# x) (a *# x') ` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_translate;
  TYPE_THEN `n` EXISTS_TAC;
  REPEAT (CONJ_TAC THEN TRY (IMATCH_MP_TAC  euclid_scale_closure) THEN ASM_REWRITE_TAC[]);
  DISCH_THEN_REWRITE;
  TYPE_THEN `d_euclid (a *# x) (a *# x')  = abs  (a) * d_euclid x x'` SUBGOAL_TAC;
  IMATCH_MP_TAC  norm_scale_vec;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `abs  a = a` SUBGOAL_TAC;
  ASM_MESON_TAC[REAL_ABS_REFL];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  REAL_PROP_LT_LMUL;
  ASM_REWRITE_TAC[];
  UND 10;
  UND 2;
  REAL_ARITH_TAC;
  (* LAST case *)
  EXPAND_TAC "z";
  EXPAND_TAC "x''";
  TYPE_THEN `d_euclid (euclid_plus (a *# x') ((&1 - a) *# x)) (euclid_plus (a *# x') ((&1 - a) *# y)) = d_euclid ((&1 - a) *# x) ((&1 - a) *# y)` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_translate_LEFT;
  TYPE_THEN `n` EXISTS_TAC;
  REPEAT (CONJ_TAC THEN TRY (IMATCH_MP_TAC  euclid_scale_closure) THEN ASM_REWRITE_TAC[]);
  DISCH_THEN_REWRITE;
  TYPE_THEN `!b. d_euclid (b *# x) (b *# y)  = abs  (b) * d_euclid x y` SUBGOAL_TAC;
  GEN_TAC;
  IMATCH_MP_TAC  norm_scale_vec;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `abs  (&1 - a) = (&1 - a)` SUBGOAL_TAC;
  REWRITE_TAC [REAL_ABS_REFL];
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  REAL_PROP_LE_LMUL;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  UND 1;
  REAL_ARITH_TAC;
  UND 3;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let openball_mk_segment_end = prove_by_refinement(
  `!x e n u v.
     (open_ball(euclid n,d_euclid) x e u) /\
     (open_ball(euclid n,d_euclid) x e v) ==>
     (mk_segment u v SUBSET (open_ball(euclid n,d_euclid) x e))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  ASSUME_TAC openball_convex;
  TYPEL_THEN [`x`;`e`;`n`] (USE 2 o ISPECL);
  USE 2 (REWRITE_RULE[convex]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let euclid_eq_minus = prove_by_refinement(
  `!x y. (x = y) <=> (euclid_minus x y = euclid0)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[euclid_minus;euclid0];
  REP_BASIC_TAC;
  EQ_TAC ;
  DISCH_THEN_REWRITE;
  REDUCE_TAC;
  DISCH_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  ONCE_REWRITE_TAC [REAL_ARITH `(a = b) <=> (a - b = &0)`];
  GEN_TAC;
  FIRST_ASSUM  (fun t-> MP_TAC (AP_THM t `x':num`));
  BETA_TAC ;
  MESON_TAC[];
  ]);;
  (* }}} *)

let euclid_plus_pair = prove_by_refinement(
  `!x y u v. (euclid_plus (x + y) (u + v) = (x + u) + (y + v))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[euclid_plus];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  BETA_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let euclid_minus_scale = prove_by_refinement(
  `!x y. (euclid_minus x y = euclid_plus x ((-- &.1) *# y))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[euclid_minus;euclid_plus;euclid_scale];
  IMATCH_MP_TAC  EQ_EXT;
  BETA_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let euclid_scale_cancel = prove_by_refinement(
  `!t x y . (~(t = &0)) /\ (t *# x = t *# y) ==> (x = y)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  FIRST_ASSUM  (fun t -> MP_TAC (AP_THM t `x':num`));
  REWRITE_TAC[euclid_scale;];
  ASM_MESON_TAC[REAL_MUL_LTIMES];
  ]);;
  (* }}} *)

let mk_segment_inj_image = prove_by_refinement(
  `!x y n. (euclid n x) /\ (euclid n y) /\ ~(x = y) ==> (?f.
     (continuous f
        (top_of_metric(UNIV,d_real))
        (top_of_metric (euclid n,d_euclid))) /\
      (INJ f {x | &0 <= x /\ x <= &1} (euclid n)) /\
     (IMAGE f {t | &.0 <=. t /\ t <=. &.1}  = mk_segment x y))`,
  (* {{{ proof *)

  [
  DISCH_ALL_TAC;
  TYPE_THEN `(joinf (\u. x) (joinf (\t. euclid_plus (t *# y) ((&1 - t) *# x)) (\u. y) (&.1)) (&.0))` EXISTS_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  cont_mk_segment;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[joinf;IMAGE ];
  REWRITE_TAC[mk_segment];
  CONJ_TAC;
  (* new stuff *)
  REWRITE_TAC[INJ];
  CONJ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `~(x' < &0)` SUBGOAL_TAC;
  UND 4;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_CASES_TAC `x' < &1`;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  euclid_add_closure;
  CONJ_TAC THEN (IMATCH_MP_TAC  euclid_scale_closure) THEN (ASM_REWRITE_TAC[]);
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  UND 3;
  TYPE_THEN `~(x' < &0)` SUBGOAL_TAC;
  UND 7;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `~(y' < &0)` SUBGOAL_TAC;
  UND 5;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `(if (x' < &1) then (euclid_plus (x' *# y) ((&1 - x') *# x)) else y) = ( euclid_plus (x' *# y) ((&1 - x') *# x))` SUBGOAL_TAC;
 TYPE_THEN `(x' < &1) \/ (x' = &1)` SUBGOAL_TAC;
  UND 6;
  REAL_ARITH_TAC;
  DISCH_THEN   DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `~(x' < &1)` SUBGOAL_TAC;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale_one;euclid_scale0;euclid_rzero];
  DISCH_THEN_REWRITE;

  TYPE_THEN `(if (y' < &1) then (euclid_plus (y' *# y) ((&1 - y') *# x)) else y) = ( euclid_plus (y' *# y) ((&1 - y') *# x))` SUBGOAL_TAC;
 TYPE_THEN `(y' < &1) \/ (y' = &1)` SUBGOAL_TAC;
  UND 4;
  REAL_ARITH_TAC;
  DISCH_THEN   DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `~(y' < &1)` SUBGOAL_TAC;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale_one;euclid_scale0;euclid_rzero];
  DISCH_THEN_REWRITE;
  (* th *)
  ONCE_REWRITE_TAC [euclid_eq_minus];
  REWRITE_TAC[euclid_minus_scale;euclid_ldistrib;euclid_scale_act];
  ONCE_REWRITE_TAC [euclid_plus_pair];
  REWRITE_TAC[GSYM euclid_rdistrib];
  REDUCE_TAC;
  REWRITE_TAC[REAL_ARITH  `x' + -- &1 * y' = x' - y'`];
  REWRITE_TAC[REAL_ARITH `&1 - x' - (&1 - y') = -- &1 *(x' - y')`];
  REWRITE_TAC[GSYM euclid_scale_act;GSYM euclid_minus_scale;ONCE_REWRITE_RULE[EQ_SYM_EQ] euclid_eq_minus];
  (* th1 *)
  DISCH_TAC;
  PROOF_BY_CONTR_TAC;
  UND 2;
  REWRITE_TAC[];
  IMATCH_MP_TAC  euclid_scale_cancel;
  TYPE_THEN `(x' - y')` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 8;
  REAL_ARITH_TAC;
  KILL 2;
  (* old stuff *)
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  ASM_REWRITE_TAC[];
  EQ_TAC;
  DISCH_TAC;
  CHO 2;
  UND 2;
  COND_CASES_TAC;
  DISCH_ALL_TAC;
  JOIN 3 2;
  ASM_MESON_TAC[REAL_ARITH `~(&0 <=. x /\ x < &.0)`];
  DISCH_ALL_TAC;
  UND 5;
  COND_CASES_TAC;
  DISCH_TAC;
  TYPE_THEN `&1 - x''` EXISTS_TAC;
  SUBCONJ_TAC;
  UND 5;
  REAL_ARITH_TAC ;
  DISCH_TAC;
  CONJ_TAC;
  UND 3;
  REAL_ARITH_TAC ;
  ONCE_REWRITE_TAC [euclid_add_comm];
  REWRITE_TAC[REAL_ARITH `&1 - (&1 - x) = x`];
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `&0` EXISTS_TAC;
  CONJ_TAC;
  REAL_ARITH_TAC ;
  CONJ_TAC;
  REAL_ARITH_TAC ;
  REWRITE_TAC[euclid_scale_one; euclid_scale0; euclid_lzero;REAL_ARITH `&1 - &0 = &1` ];
  (* 2nd half *)
  DISCH_TAC;
  CHO 2;
  TYPE_THEN `&1 - a` EXISTS_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  AND 2;
  AND 2;
  UND 3;
  UND 4;
  REAL_ARITH_TAC ;
  COND_CASES_TAC;
  ASM_MESON_TAC[REAL_ARITH `&1 - a < &0 ==> ~(a <= &1)`];
  COND_CASES_TAC;
  REWRITE_TAC[REAL_ARITH `&1 - (&1 - a) = a`];
  ASM_MESON_TAC [euclid_add_comm];
  TYPE_THEN `a = &.0` SUBGOAL_TAC;
  UND 4;
  UND 3;
  AND 2;
  UND 3;
  REAL_ARITH_TAC ;
  DISCH_TAC;
  REWR 2;
  REWRITE_TAC [euclid_scale_one; euclid_scale0; euclid_lzero;REAL_ARITH `&1 - &0 = &1` ];
  ]);;

  (* }}} *)

let h_simple_polygonal = prove_by_refinement(
  `!x e. (euclid 2 x) /\ (~(e = &0)) ==>
    (simple_polygonal_arc hv_line (mk_segment x (x + e *# e1)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_polygonal_arc;hv_line;simple_arc ];
  REP_BASIC_TAC;
  CONJ_TAC;
  ASSUME_TAC mk_segment_inj_image;
  TYPEL_THEN [`x`;`x + (e *# e1)`;`2`] (USE 2 o ISPECL);
  TYPE_THEN `euclid 2 x /\ euclid 2 (euclid_plus x (e *# e1)) /\ ~(x = euclid_plus x (e *# e1))` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  euclid_add_closure;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  euclid_scale_closure;
  REWRITE_TAC [e1;euclid_point];
  REP_BASIC_TAC;
  FIRST_ASSUM  (fun t-> MP_TAC (AP_THM t `0`));
  REWRITE_TAC[euclid_plus;euclid_scale;e1;coord01];
  UND 0;
  REAL_ARITH_TAC;
  DISCH_TAC;
  REWR 2;
  REP_BASIC_TAC;
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  SIMP_TAC  [GSYM top_of_metric_unions;metric_euclid];
  ASM_REWRITE_TAC[];
  (* E *)
  USE 1 (MATCH_MP point_onto);
  REP_BASIC_TAC;
  TYPE_THEN `{(mk_line (point p) (point p + (e *# e1)))}` EXISTS_TAC;
  REWRITE_TAC[INR IN_SING];
  CONJ_TAC;
  REWRITE_TAC[e1;ISUBSET;mk_segment;mk_line];
  REP_BASIC_TAC;
  TYPE_THEN `a` EXISTS_TAC;
  ASM_MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC[FINITE_SING];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `p` EXISTS_TAC;
  TYPE_THEN `(FST p + e, SND p)` EXISTS_TAC;
  REWRITE_TAC[];
  AP_TERM_TAC;
  REWRITE_TAC[e1;point_scale];
  REDUCE_TAC;
  TYPE_THEN `euclid_plus (point p) (point (e,&0)) = euclid_plus (point (FST p,SND p)) (point (e,&0))` SUBGOAL_TAC;
  REWRITE_TAC[];
  DISCH_THEN (fun t-> PURE_ONCE_REWRITE_TAC[t]);
  REWRITE_TAC[point_add];
  REDUCE_TAC;
  ]);;
  (* }}} *)

let pconn_refl = prove_by_refinement(
  `!A x. (top2 A) /\ (A x) ==> (p_conn A x x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[p_conn;top2];
  REP_BASIC_TAC;
  TYPE_THEN `?e. (&0 < e) /\ (open_ball(euclid 2,d_euclid) x e SUBSET A)` SUBGOAL_TAC;
  ASM_MESON_TAC[open_ball_nbd;metric_euclid];
  REP_BASIC_TAC;
  TYPE_THEN `mk_segment x (x + (e/(&2))*# e1)` EXISTS_TAC;
  TYPE_THEN `euclid 2 x` SUBGOAL_TAC;
  USE 1(MATCH_MP sub_union);
  UND 1;
  ASM_MESON_TAC [top_of_metric_unions;metric_euclid;ISUBSET];
  DISCH_TAC;
  TYPE_THEN `~(e/(&2) = &0)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `(&0 < x) ==> (~(x = &0))` );
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  DISCH_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  h_simple_polygonal;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `open_ball (euclid 2,d_euclid) x e ` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  h_seg_openball;
  ASM_REWRITE_TAC[];
  UND 3;
  MESON_TAC[half_pos;REAL_ARITH `&0 < x ==> &0 <= x`];
  REWRITE_TAC[mk_segment];
  TYPE_THEN `&1` EXISTS_TAC;
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale_one ;euclid_scale0;euclid_rzero;];
  ARITH_TAC;
  ]);;
  (* }}} *)

let pconn_symm = prove_by_refinement(
  `!A x y. (p_conn A x y ==> p_conn A y x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[p_conn;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let compose_cont = prove_by_refinement(
  `!(f:A->B) (g:B->C) X dX Y dY Z dZ.
    (metric_continuous f (X,dX) (Y,dY)) /\
    (metric_continuous g (Y,dY) (Z,dZ)) /\
    (IMAGE f X SUBSET Y) ==>
    (metric_continuous (compose g f) (X,dX) (Z,dZ))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  REP_BASIC_TAC;
  RIGHT_TAC "delta";
  DISCH_TAC;
  REWRITE_TAC[compose];
  TYPEL_THEN [`f x`;`epsilon`] (USE 1 o ISPECL);
  REP_BASIC_TAC;
  REWR 1;
  REP_BASIC_TAC;
  TYPEL_THEN [`x`;`delta`] (USE 2 o ISPECL);
  REP_BASIC_TAC;
  REWR 2;
  REP_BASIC_TAC;
  TYPE_THEN `delta'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  USE 0 (REWRITE_RULE[IMAGE;SUBSET]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let compose_image = prove_by_refinement(
  `!(f:A->B) (g:B->C) X.
   (IMAGE (compose g f) X) =
    (IMAGE g (IMAGE f X))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[IMAGE];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  GEN_TAC;
  NAME_CONFLICT_TAC;
  REWRITE_TAC[compose];
  CONV_TAC (dropq_conv "x''");
  ]);;
  (* }}} *)

let linear_cont = prove_by_refinement(
  `!a b. metric_continuous (\t. t * a + (&1 - t)* b)
     (UNIV,d_real) (UNIV,d_real)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  REP_BASIC_TAC;
  RIGHT_TAC "delta";
  DISCH_TAC;
  TYPE_THEN `a = b` ASM_CASES_TAC;
  ASM_REWRITE_TAC[GSYM REAL_RDISTRIB;REAL_ARITH `!u. u + &1 - u = &1`];
  REDUCE_TAC;
  ASM_REWRITE_TAC[d_real;REAL_ARITH `b - b = &0`;ABS_0;];
  TYPE_THEN `epsilon` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* snd *)
  TYPE_THEN `delta = epsilon/(abs  (a-b))` ABBREV_TAC;
  TYPE_THEN `delta` EXISTS_TAC;
  SUBCONJ_TAC;
  EXPAND_TAC "delta";
  IMATCH_MP_TAC  REAL_LT_DIV;
  ASM_REWRITE_TAC[];
  UND 1;
  REAL_ARITH_TAC;
  DISCH_TAC;
  REWRITE_TAC[d_real];
  REP_BASIC_TAC;
  TYPE_THEN `((x * a + (&1 - x) * b) - (y * a + (&1 - y) * b))  = (x - y)*(a - b)` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  TYPE_THEN `epsilon = delta * (abs  (a - b))` SUBGOAL_TAC;
  EXPAND_TAC "delta";
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  IMATCH_MP_TAC  REAL_DIV_RMUL;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[ABS_MUL];
  IMATCH_MP_TAC  REAL_PROP_LT_RMUL;
  ASM_REWRITE_TAC[];
  UND 1;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let linear_image_gen = prove_by_refinement(
  `!a b c d. (a < b) /\ (c < d) ==>
     (IMAGE (\t. (t - c)/(d-c) * a + (d - t)/(d - c) *b )
         {x | c <= x /\ x <= d } =
            {y | a <= y /\ y <= b})`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE];
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  TYPE_THEN `&0 < (b - a)` SUBGOAL_TAC;
  UND 1;
  REAL_ARITH_TAC;
  TYPE_THEN `&0 < (d - c)` SUBGOAL_TAC;
  UND 0;
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  ABBREV_TAC   `e = &1/(d-c)`;
  TYPE_THEN `!u. u/(d - c) = u*e` SUBGOAL_TAC;
  GEN_TAC;
  EXPAND_TAC "e";
  REWRITE_TAC[real_div];
  REDUCE_TAC;
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(d - c)*e = &1` SUBGOAL_TAC;
  EXPAND_TAC "e";
  REWRITE_TAC[real_div];
  REDUCE_TAC;
  REWRITE_TAC[GSYM real_div];
  IMATCH_MP_TAC  REAL_DIV_REFL;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `&0 < e` SUBGOAL_TAC;
  EXPAND_TAC "e";
  IMATCH_MP_TAC  REAL_LT_DIV;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  DISCH_TAC;
  (*   *)
  EQ_TAC;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  TYPE_THEN `((d-c)*e*a <= ((x' - c) * e) * a + ((d - x') * e) * b) ==> (a <= ((x' - c) * e) * a + ((d - x') * e) * b)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_MUL_ASSOC];
  REDUCE_TAC;
  DISCH_THEN IMATCH_MP_TAC ;
  ineq_le_tac `(d-c)*e*a + (d - x')*(b - a)*e = ((x' - c) * e) * a + ((d - x') * e) * b`;
  TYPE_THEN `(((x' - c) * e) * a + ((d - x') * e) * b <= b*((d- c)*e)) ==> (((x' - c) * e) * a + ((d - x') * e) * b <= b)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `x* &1 = x`];
  DISCH_THEN IMATCH_MP_TAC ;
  ineq_le_tac `(((x' - c) * e) * a + ((d - x') * e) * b) + (x'-c )*(b-a)*e = b * (d - c) * e`;
  (* 2nd direction *)
  REP_BASIC_TAC;
  TYPE_THEN `x' = ((d*b - a*c) - (d -c)*x)/(b - a)` ABBREV_TAC ;
  TYPE_THEN `x'` EXISTS_TAC;
  TYPE_THEN `x'*(b - a) = ((d*b - a*c) - (d -c)*x)` SUBGOAL_TAC;
  EXPAND_TAC "x'";
  IMATCH_MP_TAC  REAL_DIV_RMUL;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* sv *)
  SUBGOAL_TAC `!x a b. (a * x <= b * x /\ &0 < x) ==> (a <= b)`;
  MESON_TAC[REAL_PROP_LE_RCANCEL];
  DISCH_TAC;
  CONJ_TAC;
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `(b - a)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ineq_le_tac `c * (b - a) + (d-c)*(b-x) = d * b - a * c - (d - c) * x`;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `(b - a)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ineq_le_tac `(d * b - a * c - (d - c) * x) + (d-c)*(x-a) = d * (b - a)`;
  TYPE_THEN `((x' - c) * e) * a + ((d - x') * e) * b = (d*b - c*a - x'*(b-a))*e` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(d * b - c * a - (d * b - a * c - (d - c) * x)) = x*(d-c)` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[GSYM REAL_MUL_ASSOC];
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  ]);;
  (* }}} *)

let linear_image_rev = prove_by_refinement(
  `!a b c d. (a < b) /\ (c < d) ==>
     (IMAGE (\t. (t - c)/(d-c) * b + (d - t)/(d - c) *a )
         {x | c <= x /\ x <= d } =
            {y | a <= y /\ y <= b})`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE];
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  TYPE_THEN `&0 < (b - a)` SUBGOAL_TAC;
  UND 1;
  REAL_ARITH_TAC;
  TYPE_THEN `&0 < (d - c)` SUBGOAL_TAC;
  UND 0;
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  ABBREV_TAC   `e = &1/(d-c)`;
  TYPE_THEN `!u. u/(d - c) = u*e` SUBGOAL_TAC;
  GEN_TAC;
  EXPAND_TAC "e";
  REWRITE_TAC[real_div];
  REDUCE_TAC;
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(d - c)*e = &1` SUBGOAL_TAC;
  EXPAND_TAC "e";
  REWRITE_TAC[real_div];
  REDUCE_TAC;
  REWRITE_TAC[GSYM real_div];
  IMATCH_MP_TAC  REAL_DIV_REFL;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `&0 < e` SUBGOAL_TAC;
  EXPAND_TAC "e";
  IMATCH_MP_TAC  REAL_LT_DIV;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  DISCH_TAC;
  (*   *)
  EQ_TAC;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  TYPE_THEN `((d-c)*e*a <= ((x' - c) * e) * b + ((d - x') * e) * a) ==> (a <= ((x' - c) * e) * b + ((d - x') * e) * a)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_MUL_ASSOC];
  REDUCE_TAC;
  DISCH_THEN IMATCH_MP_TAC ;
  ineq_le_tac `(d-c)*e*a + (x' - c)*(b - a)*e = ((x' - c) * e) * b + ((d - x') * e) * a`;
  TYPE_THEN `(((x' - c) * e) * b + ((d - x') * e) * a <= b*((d- c)*e)) ==> (((x' - c) * e) * b + ((d - x') * e) * a <= b)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `x* &1 = x`];
  DISCH_THEN IMATCH_MP_TAC ;
  ineq_le_tac `(((x' - c) * e) * b + ((d - x') * e) * a) + (d - x' )*(b-a)*e = b * (d - c) * e`;
  (* 2nd direction *)
  REP_BASIC_TAC;
  TYPE_THEN `x' = ((b*c  - a*d) + (d -c)*x)/(b - a)` ABBREV_TAC ;
  TYPE_THEN `x'` EXISTS_TAC;
  TYPE_THEN `x'*(b - a) = ((b*c - a*d ) + (d -c)*x)` SUBGOAL_TAC;
  EXPAND_TAC "x'";
  IMATCH_MP_TAC  REAL_DIV_RMUL;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* sv *)
  SUBGOAL_TAC `!x a b. (a * x <= b * x /\ &0 < x) ==> (a <= b)`;
  MESON_TAC[REAL_PROP_LE_RCANCEL];
  DISCH_TAC;
  CONJ_TAC;
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `(b - a)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ineq_le_tac `c * (b - a) + (d-c)*(x-a) = b*c  - a*d + (d - c) * x`;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `(b - a)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ineq_le_tac `(b*c - a*d + (d - c) * x) + (d-c)*(b - x) = d * (b - a)`;
  TYPE_THEN `((x' - c) * e) * b + ((d - x') * e) * a = (d*a - c*b + x'*(b-a))*e` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(d * a - c * b + b * c - a * d + (d - c) * x) = x*(d-c)` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[GSYM REAL_MUL_ASSOC];
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  ]);;
  (* }}} *)

let linear_inj = prove_by_refinement(
  `!a b c d. (a < b) /\ (c < d) ==>
     (INJ (\t. (t - c)/(d-c) * a + (d - t)/(d - c) *b )
         {x | c <= x /\ x <= d }
            {y | a <= y /\ y <= b})`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[INJ];
  CONJ_TAC;
  REP_BASIC_TAC;
  ASSUME_TAC linear_image_gen;
  TYPEL_THEN [`a`;`b`;`c`;`d`] (USE 4 o ISPECL);
  REWR 4;
  UND 4;
  REWRITE_TAC[IMAGE];
  DISCH_TAC;
  FIRST_ASSUM (fun t-> ASSUME_TAC (AP_THM t `(x - c) / (d - c) * a + (d - x) / (d - c) * b`));
  UND 5;
  REWRITE_TAC[];
  DISCH_THEN (fun t->REWRITE_TAC[GSYM t]);
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* INJ proper *)
  REP_BASIC_TAC;
  UND 2;
  TYPE_THEN `&0 < (d - c)` SUBGOAL_TAC;
  UND 0;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `e = &1/(d-c)` ABBREV_TAC ;
  TYPE_THEN `!u. (u/(d-c) = u*e)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC"e";
  REWRITE_TAC[real_div];
  REDUCE_TAC;
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  USE 8(ONCE_REWRITE_RULE [REAL_ARITH `(x = y) <=> (x - y = &0)`]);
  UND 8;
  TYPE_THEN `(((x - c) * e) * a + ((d - x) * e) * b) - (((y - c) * e) * a + ((d - y) * e) * b) = e*(b-a)*(y - x)` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[REAL_ENTIRE];
  TYPE_THEN `~(b - a = &0)` SUBGOAL_TAC;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `~(e = &0)` SUBGOAL_TAC;
  EXPAND_TAC"e";
  REWRITE_TAC[real_div];
  REDUCE_TAC;
  REWRITE_TAC[REAL_INV_EQ_0];
  UND 0;
  REAL_ARITH_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let linear_inj_rev = prove_by_refinement(
  `!a b c d. (a < b) /\ (c < d) ==>
     (INJ (\t. (t - c)/(d-c) * b + (d - t)/(d - c) *a )
         {x | c <= x /\ x <= d }
            {y | a <= y /\ y <= b})`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[INJ];
  CONJ_TAC;
  REP_BASIC_TAC;
  ASSUME_TAC linear_image_rev;
  TYPEL_THEN [`a`;`b`;`c`;`d`] (USE 4 o ISPECL);
  REWR 4;
  UND 4;
  REWRITE_TAC[IMAGE];
  DISCH_TAC;
  FIRST_ASSUM (fun t-> ASSUME_TAC (AP_THM t `(x - c) / (d - c) * b + (d - x) / (d - c) * a`));
  UND 5;
  REWRITE_TAC[];
  DISCH_THEN (fun t->REWRITE_TAC[GSYM t]);
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* INJ proper *)
  REP_BASIC_TAC;
  UND 2;
  TYPE_THEN `&0 < (d - c)` SUBGOAL_TAC;
  UND 0;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `e = &1/(d-c)` ABBREV_TAC ;
  TYPE_THEN `!u. (u/(d-c) = u*e)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC"e";
  REWRITE_TAC[real_div];
  REDUCE_TAC;
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  USE 8(ONCE_REWRITE_RULE [REAL_ARITH `(x = y) <=> (x - y = &0)`]);
  UND 8;
  TYPE_THEN `(((x - c) * e) * b + ((d - x) * e) * a) - (((y - c) * e) * b + ((d - y) * e) * a) = e*(a-b)*(y - x)` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[REAL_ENTIRE];
  TYPE_THEN `~(a-b = &0)` SUBGOAL_TAC;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `~(e = &0)` SUBGOAL_TAC;
  EXPAND_TAC"e";
  REWRITE_TAC[real_div];
  REDUCE_TAC;
  REWRITE_TAC[REAL_INV_EQ_0];
  UND 0;
  REAL_ARITH_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let comp_comp = prove_by_refinement(
  `(o) = (compose:(B->C) -> ((A->B)-> (A->C))) `,
  (* {{{ proof *)
  [
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[o_DEF;compose];
  ]);;
  (* }}} *)

let arc_reparameter_rev = prove_by_refinement(
  `!f a b c d. ( continuous f (top_of_metric (UNIV,d_real)) (top2) /\
           INJ f {x | c <= x /\ x <= d} (euclid 2) /\
         (a < b) /\ (c < d)  ==>
           (?g. continuous g (top_of_metric (UNIV,d_real)) (top2) /\
           INJ g {x | a <= x /\ x <= b} (euclid 2) /\
         (f d  = g a) /\ (f c = g b) /\
      (!x y x' y'. (f x = g x') /\ (f y = g y') /\
         (c <= x /\ x <= d) /\ (c <= y /\ y <= d) /\
         (a <= x' /\ x' <= b) /\ (a <= y' /\ y' <= b) ==>
           ((x < y) = (y' < x'))) /\
      (IMAGE f { x | c <= x /\ x <= d } =
         IMAGE g { x | a <= x /\ x <= b } )))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `f2 = (\t. (t - a)/(b - a) * (c) + (b - t)/(b - a) *(d) )` ABBREV_TAC ;
  TYPE_THEN `g = (f o f2)` ABBREV_TAC ;
  TYPE_THEN `g` EXISTS_TAC;
  (* general facts *)
  TYPE_THEN `UNIONS(top_of_metric(UNIV,d_real)) = UNIV` SUBGOAL_TAC;
  MESON_TAC[metric_real;top_of_metric_unions];
  DISCH_TAC;
  (* continuity *)
  CONJ_TAC;
  EXPAND_TAC "g";
  IMATCH_MP_TAC  continuous_comp;
  TYPE_THEN `(top_of_metric(UNIV,d_real))` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[top2];
  ASM_SIMP_TAC[metric_continuous_continuous;metric_real;SUBSET_UNIV];
  TYPE_THEN `f2 = (\t. t* (c - d + d*b - c*a)/(b - a) + (&1 - t)*(d*b-c*a)/(b - a))` SUBGOAL_TAC;
  EXPAND_TAC "f2";
  IMATCH_MP_TAC  EQ_EXT;
  BETA_TAC;
  GEN_TAC;
  REWRITE_TAC[real_div;GSYM REAL_MUL_ASSOC;REAL_ARITH `(inv x)*y = y*(inv x)`];
  REWRITE_TAC[REAL_MUL_ASSOC;GSYM REAL_RDISTRIB;REAL_EQ_MUL_RCANCEL];
  DISJ1_TAC ;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[linear_cont];
  (* IMAGE *)
  TYPE_THEN `{x | c <= x /\ x <= d} = IMAGE f2 {x | a <= x /\ x <= b}` SUBGOAL_TAC;
  REWRITE_TAC[];
  EXPAND_TAC "f2";
  ASM_SIMP_TAC[linear_image_gen];
  DISCH_TAC;
  TYPE_THEN `(IMAGE f {x | c <= x /\ x <= d} = IMAGE g {x | a <= x /\ x <= b})` SUBGOAL_TAC;
  EXPAND_TAC "g";
  REWRITE_TAC[comp_comp;compose_image;];
  AP_TERM_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  (* INJ *)
  EXPAND_TAC "g";
  REWRITE_TAC[comp_comp];
  (* XXX *)
  CONJ_TAC;
  IMATCH_MP_TAC  (COMP_INJ);
  TYPE_THEN `{x | c <= x /\ x <= d}` EXISTS_TAC;
  UND 2;
  DISCH_THEN_REWRITE;
  KILL 7;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "f2";
  IMATCH_MP_TAC  linear_inj;
  ASM_REWRITE_TAC[];
  (* ends   *)
  IMATCH_MP_TAC  (TAUT `(A /\ B) /\ C ==> A /\ B /\ C`);
  CONJ_TAC;
  EXPAND_TAC "f2";
  REWRITE_TAC[compose];
  REDUCE_TAC;
  REWRITE_TAC[real_div;REAL_MUL_ASSOC;];
  REDUCE_TAC;
  TYPE_THEN `(b-a)*inv(b-a) = &1` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_MUL_RINV;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REDUCE_TAC;
  (* monotone *)
  REWRITE_TAC[compose];
  REP_BASIC_TAC;
  TYPE_THEN `c <= f2 y' /\ f2 y' <= d` SUBGOAL_TAC;
  USE 7 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `f2 y'` (fun s -> FIRST_ASSUM (fun t-> MP_TAC (AP_THM t s)));
  REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `y'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `c <= f2 x' /\ f2 x' <= d` SUBGOAL_TAC;
  USE 7 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `f2 x'` (fun s -> FIRST_ASSUM (fun t-> MP_TAC (AP_THM t s)));
  REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `x = f2 x'` SUBGOAL_TAC;
  USE 2 (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `y = f2 y'` SUBGOAL_TAC;
  USE 2 (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "f2";
  ONCE_REWRITE_TAC [REAL_ARITH `x < y <=> ( &0 < y - x)`];
  REWRITE_TAC[real_div];
  TYPE_THEN `e = inv(b-a)` ABBREV_TAC ;
  TYPE_THEN `(((y' - a) * e) * c + ((b - y') * e) * d) - (((x' - a) * e) * c + ((b - x') * e) * d) = (x' - y')*e*(d-c)` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  TYPE_THEN `&0 < e` SUBGOAL_TAC;
  EXPAND_TAC"e";
  IMATCH_MP_TAC  REAL_PROP_POS_INV;
  UND 1;
  REAL_ARITH_TAC;
  TYPE_THEN `&0 < (d - c)` SUBGOAL_TAC;
  UND 0;
  REAL_ARITH_TAC;
  REWRITE_TAC[REAL_MUL_ASSOC];
  ASM_SIMP_TAC[REAL_PROP_POS_RMUL];
  ]);;
  (* }}} *)

let arc_reparameter_gen = prove_by_refinement(
  `!f a b c d. ( continuous f (top_of_metric (UNIV,d_real)) (top2) /\
           INJ f {x | c <= x /\ x <= d} (euclid 2) /\
         (a < b) /\ (c < d)  ==>
           (?g. continuous g (top_of_metric (UNIV,d_real)) (top2) /\
           INJ g {x | a <= x /\ x <= b} (euclid 2) /\
         (f c  = g a) /\ (f d = g b) /\
      (!x y x' y'. (f x = g x') /\ (f y = g y') /\
         (c <= x /\ x <= d) /\ (c <= y /\ y <= d) /\
         (a <= x' /\ x' <= b) /\ (a <= y' /\ y' <= b) ==>
           ((x < y) = (x' < y'))) /\
      (IMAGE f { x | c <= x /\ x <= d } =
         IMAGE g { x | a <= x /\ x <= b } )))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `f2 = (\t. (t - a)/(b - a) * (d) + (b - t)/(b - a) *(c) )` ABBREV_TAC ;
  TYPE_THEN `g = (f o f2)` ABBREV_TAC ;
  TYPE_THEN `g` EXISTS_TAC;
  (* general facts *)
  TYPE_THEN `UNIONS(top_of_metric(UNIV,d_real)) = UNIV` SUBGOAL_TAC;
  MESON_TAC[metric_real;top_of_metric_unions];
  DISCH_TAC;
  (* continuity *)
  CONJ_TAC;
  EXPAND_TAC "g";
  IMATCH_MP_TAC  continuous_comp;
  TYPE_THEN `(top_of_metric(UNIV,d_real))` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[top2];
  ASM_SIMP_TAC[metric_continuous_continuous;metric_real;SUBSET_UNIV];
  TYPE_THEN `f2 = (\t. t* (d - c + c*b - d*a)/(b - a) + (&1 - t)*(c*b-d*a)/(b - a))` SUBGOAL_TAC;
  EXPAND_TAC "f2";
  IMATCH_MP_TAC  EQ_EXT;
  BETA_TAC;
  GEN_TAC;
  REWRITE_TAC[real_div;GSYM REAL_MUL_ASSOC;REAL_ARITH `(inv x)*y = y*(inv x)`];
  REWRITE_TAC[REAL_MUL_ASSOC;GSYM REAL_RDISTRIB;REAL_EQ_MUL_RCANCEL];
  DISJ1_TAC ;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[linear_cont];
  (* IMAGE *)
  TYPE_THEN `{x | c <= x /\ x <= d} = IMAGE f2 {x | a <= x /\ x <= b}` SUBGOAL_TAC;
  REWRITE_TAC[];
  EXPAND_TAC "f2";
  ASM_SIMP_TAC[linear_image_rev];
  DISCH_TAC;
  TYPE_THEN `(IMAGE f {x | c <= x /\ x <= d} = IMAGE g {x | a <= x /\ x <= b})` SUBGOAL_TAC;
  EXPAND_TAC "g";
  REWRITE_TAC[comp_comp;compose_image;];
  AP_TERM_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  (* INJ *)
  EXPAND_TAC "g";
  REWRITE_TAC[comp_comp];
  (* XXX *)
  CONJ_TAC;
  IMATCH_MP_TAC  (COMP_INJ);
  TYPE_THEN `{x | c <= x /\ x <= d}` EXISTS_TAC;
  UND 2;
  DISCH_THEN_REWRITE;
  KILL 7;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "f2";
  IMATCH_MP_TAC  linear_inj_rev;
  ASM_REWRITE_TAC[];
  (* ends   *)
  IMATCH_MP_TAC  (TAUT `(A /\ B) /\ C ==> A /\ B /\ C`);
  CONJ_TAC;
  EXPAND_TAC "f2";
  REWRITE_TAC[compose];
  REDUCE_TAC;
  REWRITE_TAC[real_div;REAL_MUL_ASSOC;];
  REDUCE_TAC;
  TYPE_THEN `(b-a)*inv(b-a) = &1` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_MUL_RINV;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REDUCE_TAC;
  (* monotone *)
  REWRITE_TAC[compose];
  REP_BASIC_TAC;
  TYPE_THEN `c <= f2 y' /\ f2 y' <= d` SUBGOAL_TAC;
  USE 7 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `f2 y'` (fun s -> FIRST_ASSUM (fun t-> MP_TAC (AP_THM t s)));
  REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `y'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `c <= f2 x' /\ f2 x' <= d` SUBGOAL_TAC;
  USE 7 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `f2 x'` (fun s -> FIRST_ASSUM (fun t-> MP_TAC (AP_THM t s)));
  REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `x = f2 x'` SUBGOAL_TAC;
  USE 2 (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `y = f2 y'` SUBGOAL_TAC;
  USE 2 (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "f2";
  ONCE_REWRITE_TAC [REAL_ARITH `x < y <=> ( &0 < y - x)`];
  REWRITE_TAC[real_div];
  TYPE_THEN `e = inv(b-a)` ABBREV_TAC ;
  TYPE_THEN `(((y' - a) * e) * d + ((b - y') * e) * c) - (((x' - a) * e) * d + ((b - x') * e) * c) = (y' - x')*e*(d-c)` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  TYPE_THEN `&0 < e` SUBGOAL_TAC;
  EXPAND_TAC"e";
  IMATCH_MP_TAC  REAL_PROP_POS_INV;
  UND 1;
  REAL_ARITH_TAC;
  TYPE_THEN `&0 < (d - c)` SUBGOAL_TAC;
  UND 0;
  REAL_ARITH_TAC;
  REWRITE_TAC[REAL_MUL_ASSOC];
  ASM_SIMP_TAC[REAL_PROP_POS_RMUL];
  ]);;
  (* }}} *)

let image_preimage = prove_by_refinement(
  `!(f:A->B) X Y. IMAGE f (preimage X f Y) SUBSET Y`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE;SUBSET;INR in_preimage ;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let preimage_union2 = prove_by_refinement(
  `!(f:A->B) A B X. (preimage X f (A UNION B)) =
    (preimage X f A UNION preimage X f B)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[preimage_union;image_preimage;];
  REWRITE_TAC[preimage;SUBSET;];
  MESON_TAC[];
  REWRITE_TAC[union_subset];
  REWRITE_TAC[preimage;SUBSET;UNION];
  MESON_TAC[];
  ]);;
  (* }}} *)

let union_diff  = prove_by_refinement(
  `!(X:A->bool) A B. (X = A UNION B) /\ (A INTER B = EMPTY) ==>
     (X DIFF B = A)`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  SET_TAC[];
  ]);;
  (* }}} *)

let preimage_closed = prove_by_refinement(
  `!U V C (f:A->B). (continuous f U V) /\ (closed_ V C) /\
       (IMAGE f (UNIONS U) SUBSET (UNIONS V)) ==>
           (closed_ U (preimage (UNIONS U) f C))`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  REWRITE_TAC[closed;open_DEF;];
  TYPE_THEN `(UNIONS U DIFF (preimage (UNIONS U) f C)) = preimage (UNIONS U) f (UNIONS V DIFF C)` SUBGOAL_TAC;
  IMATCH_MP_TAC  union_diff;
  REWRITE_TAC[GSYM preimage_union2];
  CONJ_TAC;
  TYPE_THEN `UNIONS V DIFF C UNION C = UNIONS V` SUBGOAL_TAC;
  TYPE_THEN `!P. C SUBSET P ==> (P DIFF C UNION C = P)` SUBGOAL_TAC;
  SET_TAC[];
  TYPE_THEN `C SUBSET UNIONS V` SUBGOAL_TAC;
  UND 1;
  REWRITE_TAC[closed;open_DEF;];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  DISCH_THEN (fun t-> ASM_SIMP_TAC[t]);
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  ASM_REWRITE_TAC [  subset_preimage;];
  REWRITE_TAC[preimage;SUBSET];
  MESON_TAC[];
  IMATCH_MP_TAC  preimage_disjoint;
  SET_TAC[];
  DISCH_THEN_REWRITE;
  CONJ_TAC;
  REWRITE_TAC[SUBSET;preimage];
  MESON_TAC[];
  UND 2;
  REWRITE_TAC[continuous];
  DISCH_THEN IMATCH_MP_TAC ;
  UND 1;
  REWRITE_TAC[closed;open_DEF;];
  MESON_TAC[];
  ]);;

  (* }}} *)

let preimage_restrict = prove_by_refinement(
  `!(f:A->B) Z A B.  (A SUBSET B) ==>
      (preimage A f Z = A INTER preimage B f Z)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[preimage;INTER;];
  TYPE_THEN `!y. (A SUBSET B ==> (A y /\ B y <=> A y))` SUBGOAL_TAC;
  MESON_TAC[ISUBSET];
  ASM_SIMP_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let continuous_delta = prove_by_refinement(
  `continuous (\x. (x *# dirac_delta 0)) (top_of_metric(UNIV,d_real))
     (top_of_metric(euclid 1,d_euclid)) `,
  (* {{{ proof *)
  [
  TYPE_THEN `IMAGE (\x. (x *# dirac_delta 0)) (UNIV) SUBSET (euclid 1)` SUBGOAL_TAC;
  REWRITE_TAC[IMAGE;SUBSET;];
  MESON_TAC[euclid_dirac];
  ASM_SIMP_TAC[metric_continuous_continuous;metric_euclid;metric_real];
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  REP_BASIC_TAC;
  RIGHT_TAC "delta";
  REP_BASIC_TAC;
  TYPE_THEN `epsilon` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_SIMP_TAC[euclid_dirac;euclid1_abs];
  REWRITE_TAC[dirac_0];
  USE 2 (REWRITE_RULE [d_real]);
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let continuous_neg_delta = prove_by_refinement(
  `continuous (\x. ((-- x) *# dirac_delta 0))
   (top_of_metric(UNIV,d_real))
     (top_of_metric(euclid 1,d_euclid)) `,
  (* {{{ proof *)
  [
  TYPE_THEN `IMAGE (\x. (-- x *# dirac_delta 0)) (UNIV) SUBSET (euclid 1)` SUBGOAL_TAC;
  REWRITE_TAC[IMAGE;SUBSET;];
  MESON_TAC[euclid_dirac];
  ASM_SIMP_TAC[metric_continuous_continuous;metric_euclid;metric_real];
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  REP_BASIC_TAC;
  RIGHT_TAC "delta";
  REP_BASIC_TAC;
  TYPE_THEN `epsilon` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_SIMP_TAC[euclid_dirac;euclid1_abs];
  REWRITE_TAC[dirac_0];
  USE 2 (REWRITE_RULE [d_real]);
  UND 2;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let compact_max_real = prove_by_refinement(
  `!(f:A->real) U K.
    continuous f U (top_of_metric (UNIV,d_real)) /\
          compact U K /\
          ~(K = {})
          ==> (?x. K x /\ (!y. K y ==> f y  <= f x ))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `g = (\x. (x *# dirac_delta 0)) o f` ABBREV_TAC ;
  TYPE_THEN `(?x. K x /\ (!y. K y ==> g y 0 <= g x 0 ))` SUBGOAL_TAC;
  IMATCH_MP_TAC  compact_max;
  TYPE_THEN `U` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "g";
  REWRITE_TAC[IMAGE_o];
  TYPE_THEN `X = IMAGE f K` ABBREV_TAC ;
  REWRITE_TAC[IMAGE ;SUBSET];
  CONJ_TAC;
  IMATCH_MP_TAC  continuous_comp;
  TYPE_THEN `top_of_metric(UNIV,d_real)` EXISTS_TAC;
  ASM_REWRITE_TAC[continuous_delta];
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_real];
  MESON_TAC[euclid_dirac];
  REP_BASIC_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  UND 4;
  EXPAND_TAC "g";
  REWRITE_TAC[o_DEF;dirac_0];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let compact_min_real = prove_by_refinement(
  `!(f:A->real) U K.
    continuous f U (top_of_metric (UNIV,d_real)) /\
          compact U K /\
          ~(K = {})
          ==> (?x. K x /\ (!y. K y ==> f x  <= f y ))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `g = (\x. (-- x *# dirac_delta 0)) o f` ABBREV_TAC ;
  TYPE_THEN `(?x. K x /\ (!y. K y ==> g y 0 <= g x 0 ))` SUBGOAL_TAC;
  IMATCH_MP_TAC  compact_max;
  TYPE_THEN `U` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "g";
  REWRITE_TAC[IMAGE_o];
  TYPE_THEN `X = IMAGE f K` ABBREV_TAC ;
  REWRITE_TAC[IMAGE ;SUBSET];
  CONJ_TAC;
  IMATCH_MP_TAC  continuous_comp;
  TYPE_THEN `top_of_metric(UNIV,d_real)` EXISTS_TAC;
  ASM_REWRITE_TAC[continuous_neg_delta];
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_real];
  MESON_TAC[euclid_dirac];
  REP_BASIC_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  UND 4;
  EXPAND_TAC "g";
  REWRITE_TAC[o_DEF;dirac_0];
  ASM_MESON_TAC[REAL_ARITH `!u v. (-- u <= --v) <=> (v <= u)`];
  ]);;
  (* }}} *)

let continuous_I = prove_by_refinement(
  `continuous I (top_of_metric(UNIV,d_real))
     (top_of_metric(UNIV,d_real))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[continuous];
  REP_BASIC_TAC;
  REWRITE_TAC[preimage];
  SIMP_TAC [GSYM top_of_metric_unions;metric_real];
  REWRITE_TAC[I_DEF];
  TYPE_THEN `{x | v x} = v` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let compact_sup = prove_by_refinement(
  `!X. (compact (top_of_metric(UNIV,d_real)) X) /\ ~(X=EMPTY ) ==>
    (?x. (X x) /\ (!y. (X y) ==> (y <= x)))`,
  (* {{{ proof *)
  [
  TYPE_THEN `!(u:real). I u = u` SUBGOAL_TAC;
  REWRITE_TAC[I_DEF];
  DISCH_TAC;
  TYPE_THEN `!x y. y <= x <=> (I y <= I x)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t ->  ONCE_REWRITE_TAC [t]);
  REP_BASIC_TAC;
  IMATCH_MP_TAC  compact_max_real;
  TYPE_THEN `(top_of_metric(UNIV,d_real))` EXISTS_TAC;
  ASM_REWRITE_TAC[continuous_I];
  ]);;
  (* }}} *)

let compact_inf = prove_by_refinement(
  `!X. (compact (top_of_metric(UNIV,d_real)) X) /\ ~(X=EMPTY ) ==>
    (?x. (X x) /\ (!y. (X y) ==> (x <= y)))`,
  (* {{{ proof *)
  [
  TYPE_THEN `!(u:real). I u = u` SUBGOAL_TAC;
  REWRITE_TAC[I_DEF];
  DISCH_TAC;
  TYPE_THEN `!x y. y <= x <=> (I y <= I x)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t ->  ONCE_REWRITE_TAC [t]);
  REP_BASIC_TAC;
  IMATCH_MP_TAC  compact_min_real;
  TYPE_THEN `(top_of_metric(UNIV,d_real))` EXISTS_TAC;
  ASM_REWRITE_TAC[continuous_I];
  ]);;
  (* }}} *)

let preimage_compact = prove_by_refinement(
  `!C (f:A->B) Y dY Z dZ Y0.
   metric_space (Y,dY) /\ metric_space (Z,dZ) /\
  (compact (top_of_metric(Y,dY)) Y0) /\
  (continuous f (top_of_metric(Y0,dY))
            (top_of_metric(Z,dZ))) /\
  (IMAGE f Y0 SUBSET Z) /\
  (closed_ (top_of_metric(Z,dZ)) C) /\
  ~(IMAGE f Y0 INTER C = EMPTY) ==>
  (compact (top_of_metric(Y,dY)) (preimage Y0 f C))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `X = preimage Y0 f C` ABBREV_TAC ;
  TYPE_THEN `(UNIONS (top_of_metric(Y,dY)) = Y) /\ (UNIONS(top_of_metric(Z,dZ)) = Z)` SUBGOAL_TAC;
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  REP_BASIC_TAC;
  TYPE_THEN `Y0 SUBSET Y` SUBGOAL_TAC;
  ASM_MESON_TAC [compact;];
  DISCH_TAC;
  WITH 10 (MATCH_MP preimage_restrict);
  TYPEL_THEN [`f`;`C`] (USE 11 o ISPECL);
  TYPE_THEN `metric_space (Y0,dY)` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_subspace;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `closed_ (top_of_metric(Y0,dY)) X` SUBGOAL_TAC;
  EXPAND_TAC "X";
  TYPE_THEN `preimage Y0 f C = preimage (UNIONS (top_of_metric(Y0,dY))) f C` SUBGOAL_TAC;
  AP_THM_TAC;
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  preimage_closed;
  TYPE_THEN `(top_of_metric (Z,dZ))` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  DISCH_TAC;
  TYPE_THEN `~(X = EMPTY)` SUBGOAL_TAC;
  REWRITE_TAC[EMPTY_EXISTS;];
  UND 0;
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  UND 0;
  REWRITE_TAC[IMAGE;INTER];
  REP_BASIC_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  EXPAND_TAC "X";
  REWRITE_TAC[preimage];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* next X compact in the reals , take inf X, *)
  TYPE_THEN `U = top_of_metric(Y,dY)` ABBREV_TAC ;
  TYPE_THEN `U0 = top_of_metric(Y0,dY)` ABBREV_TAC ;
  TYPE_THEN `U00 = top_of_metric (X,dY)` ABBREV_TAC ;
  TYPE_THEN `X SUBSET Y0` SUBGOAL_TAC;
  EXPAND_TAC "X";
  KILL 7;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INTER;SUBSET;];
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `induced_top U Y0 = U0` SUBGOAL_TAC;
  EXPAND_TAC "U";
  EXPAND_TAC "U0";
  IMATCH_MP_TAC  top_of_metric_induced;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `UNIONS U = Y` SUBGOAL_TAC;
  EXPAND_TAC "U";
  ASM_SIMP_TAC [GSYM top_of_metric_unions];
  DISCH_TAC;
  TYPE_THEN `compact U0 Y0` SUBGOAL_TAC;
  KILL 16;
  EXPAND_TAC "U0";
  ASM_SIMP_TAC[GSYM induced_compact;];
  REP_BASIC_TAC;
  (* ok to here *)
  TYPE_THEN `compact U0 X` SUBGOAL_TAC;
  IMATCH_MP_TAC  closed_compact;
  TYPE_THEN `Y0` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  KILL 19;
  EXPAND_TAC "U0";
  IMATCH_MP_TAC  top_of_metric_top;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* done WITH compac U0 X *)
  TYPE_THEN `induced_top U0 X = U00` SUBGOAL_TAC;
  KILL 19;
  EXPAND_TAC "U0";
  EXPAND_TAC "U00";
  IMATCH_MP_TAC  top_of_metric_induced;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `compact U00 X` SUBGOAL_TAC;
  EXPAND_TAC "U00";
  TYPE_THEN `X SUBSET UNIONS U0` SUBGOAL_TAC;
  KILL 19;
  EXPAND_TAC "U0";
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  ASM_SIMP_TAC[GSYM induced_compact];
  DISCH_TAC;
  TYPE_THEN `induced_top U X = U00` SUBGOAL_TAC;
  KILL 19;
  EXPAND_TAC "U";
  KILL 23;
  EXPAND_TAC "U00";
  IMATCH_MP_TAC  top_of_metric_induced;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  ASM_MESON_TAC[];
  DISCH_TAC;
  UND 24;
  EXPAND_TAC "U00";
  TYPE_THEN `compact (induced_top U X) X = compact U X` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM induced_compact);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  ASM_MESON_TAC[];
  MESON_TAC[];
  ]);;
  (* }}} *)

let preimage_compact_interval = prove_by_refinement(
  `!C n f a b.
  (continuous f (top_of_metric({x | a <= x /\ x <= b},d_real))
            (top_of_metric(euclid n,d_euclid)) /\
  (IMAGE f {x | a <= x /\ x <= b} SUBSET (euclid n)) /\
  (closed_ (top_of_metric(euclid n,d_euclid)) C) /\
  ~(IMAGE f {x | a <= x /\ x <= b} INTER C = EMPTY)) ==>
  (compact (top_of_metric(UNIV,d_real))
         (preimage {x | a <= x /\ x <= b} f C))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  preimage_compact;
  TYPE_THEN `(euclid n)` EXISTS_TAC;
  TYPE_THEN `d_euclid` EXISTS_TAC;
  ASM_REWRITE_TAC[metric_real;metric_euclid;interval_compact;];
  ]);;
  (* }}} *)

let preimage_first = prove_by_refinement(
  `!C n f a b.
  (continuous f (top_of_metric({x | a <= x /\ x <= b},d_real))
            (top_of_metric(euclid n,d_euclid)) /\
  (IMAGE f {x | a <= x /\ x <= b} SUBSET (euclid n)) /\
  (closed_ (top_of_metric(euclid n,d_euclid)) C) /\
  ~(IMAGE f {x | a <= x /\ x <= b} INTER C = EMPTY)) ==>
  (?t. (a <= t /\ t <= b) /\ (C (f t)) /\
    (!s. (a <=s /\ s < t) ==> ~(C (f s))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(compact (top_of_metric(UNIV,d_real)) (preimage {x | a <= x /\ x <= b} f C))` SUBGOAL_TAC;
  IMATCH_MP_TAC preimage_compact_interval;
  TYPE_THEN `n` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `~(preimage {x | a <= x /\ x <= b} f C = EMPTY)` SUBGOAL_TAC;
  UND 0;
  REWRITE_TAC[EMPTY_EXISTS];
  REWRITE_TAC[IMAGE ;INTER;preimage];
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `X = preimage {x | a <= x /\ x <= b } f C` ABBREV_TAC ;
  TYPE_THEN `(?x. (X x) /\ (!y. (X y) ==> (x <= y)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  compact_inf;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  UND 8;
  UND 7;
  EXPAND_TAC "X";
  REWRITE_TAC[preimage];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TSPEC `s` 10;
  REWR 10;
  UND 10;
  UND 12;
  UND 8;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let inj_subset_domain = prove_by_refinement(
  `!s s' t (f:A->B). INJ f s t /\ (s' SUBSET s) ==> INJ f s' t`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ;SUBSET;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let arc_restrict = prove_by_refinement(
  `!a b c d C f t t'. (c <= t /\ t < t' /\ t' <= d) /\ (a < b) /\
     (C = IMAGE f { x | c <= x /\ x <= d }) /\
     INJ f {x | c <= x /\ x <= d} (euclid 2) /\
     continuous f (top_of_metric(UNIV,d_real))
            (top_of_metric(euclid 2,d_euclid)) ==>
    (?g.
  (IMAGE g {x | a <= x /\ x <= b} = IMAGE f {x | t <= x /\ x <= t'})  /\
     (g a = f t) /\ (g b = f t') /\
       INJ g { x | a <= x /\ x <= b} (euclid 2) /\
       continuous g (top_of_metric(UNIV,d_real))
            (top_of_metric(euclid 2,d_euclid)))`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  TYPE_THEN ` continuous f (top_of_metric (UNIV,d_real)) (top2) /\ INJ f {x | t <= x /\ x <= t'} (euclid 2) /\ (a < b) /\ (t < t')` SUBGOAL_TAC;
  ASM_REWRITE_TAC[top2];
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | c <= x /\ x <= d}` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET;];
  UND 4;
  UND 5;
  UND 6;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t -> ASSUME_TAC (MATCH_MP arc_reparameter_gen t));
  REP_BASIC_TAC;
  TYPE_THEN `g` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[top2];
  ]);;

  (* }}} *)

let continuous_induced_domain = prove_by_refinement(
  `!(f:A->B) U V K. (continuous f U V) /\ (K SUBSET (UNIONS U)) ==>
    (continuous f (induced_top U K) V)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[continuous;induced_top_support;];
  REWRITE_TAC[preimage;induced_top];
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE];
  TYPE_THEN `{x | UNIONS U x /\ v (f x)}` EXISTS_TAC;
  ASM_SIMP_TAC[];
  REWRITE_TAC[INTER];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  MESON_TAC[];
  ]);;
  (* }}} *)

let inj_split = prove_by_refinement(
  `!A B Z (f:A->B). (INJ f A Z) /\ (INJ f B Z) /\
     (IMAGE f A INTER IMAGE f B = EMPTY) ==> (INJ f (A UNION B) Z)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ;INTER;IMAGE;UNION;];
  REP_BASIC_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[];
  REP_GEN_TAC;
  REP_BASIC_TAC;
  UND 7;
  UND 6;
  REP_CASES_TAC;
  KILL 1;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 0;
  REWRITE_TAC[EQ_EMPTY];
  NAME_CONFLICT_TAC;
  DISCH_TAC;
  TSPEC `f y` 0;
  USE 0 (REWRITE_RULE[DE_MORGAN_THM]);
  ASM_MESON_TAC[];
  USE 0 (REWRITE_RULE[EQ_EMPTY]);
  TSPEC `f x` 0;
  ASM_MESON_TAC[];
  KILL 3;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let joinf_inj_below = prove_by_refinement(
  `!(f:real->B) g a A.
    (A SUBSET {x | x < a}) ==> (INJ (joinf f g a) A = INJ f A)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  IMATCH_MP_TAC EQ_EXT;
  REWRITE_TAC[INJ];
  REP_BASIC_TAC;
  TYPE_THEN `!z. A z ==> (joinf f g a z = f z)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[joinf];
  TSPEC `z` 0;
  REWR 0;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let joinf_inj_above = prove_by_refinement(
  `!(f:real->B) g a A.
    (A SUBSET {x | a <= x}) ==> (INJ (joinf f g a) A = INJ g A)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  IMATCH_MP_TAC EQ_EXT;
  REWRITE_TAC[INJ];
  REP_BASIC_TAC;
  TYPE_THEN `!z. A z ==> (joinf f g a z = g z)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[joinf];
  TSPEC `z` 0;
  REWR 0;
  ASM_REWRITE_TAC[REAL_ARITH ` (z < a) <=> ~(a <= z) `];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let joinf_image_below = prove_by_refinement(
  `!(f:real->B) g a A.
    (A SUBSET {x | x < a}) ==> (IMAGE (joinf f g a) A = IMAGE f A)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  IMATCH_MP_TAC EQ_EXT;
  REWRITE_TAC[IMAGE];
  REP_BASIC_TAC;
  TYPE_THEN `!z. A z ==> (joinf f g a z = f z)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[joinf];
  TSPEC `z` 0;
  REWR 0;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let joinf_image_above = prove_by_refinement(
  `!(f:real->B) g a A.
    (A SUBSET {x | a <= x}) ==> (IMAGE (joinf f g a) A = IMAGE g A)`,
  (* {{{ proof *)

  [
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  IMATCH_MP_TAC EQ_EXT;
  REWRITE_TAC[IMAGE];
  REP_BASIC_TAC;
  TYPE_THEN `!z. A z ==> (joinf f g a z = g z)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[joinf];
  TSPEC `z` 0;
  REWR 0;
  ASM_REWRITE_TAC[REAL_ARITH ` (z < a) <=> ~(a <= z) `];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let pconn_trans = prove_by_refinement(
  `!A x y z. (p_conn A x y /\ p_conn A y z ==> p_conn A x z)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[p_conn;simple_polygonal_arc;simple_arc;];
  REP_BASIC_TAC;
  TYPE_THEN `C' x`  ASM_CASES_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  TYPE_THEN `f'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  TYPE_THEN `~(x = y)` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* now ~( x= y) *)
  TYPE_THEN `C z` ASM_CASES_TAC;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  TYPE_THEN `~(z = y)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* now ~( z = y) *)
  TYPE_THEN `?tx. (&0 <= tx) /\ (tx <= &1) /\ (f tx = x)` SUBGOAL_TAC;
  UND 10;
  ASM_REWRITE_TAC[IMAGE;];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `?ty. (&0 <= ty) /\ (ty <= &1) /\ (f ty = y)` SUBGOAL_TAC;
  UND 9;
  ASM_REWRITE_TAC[IMAGE;];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `~(tx = ty)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* reparameter C *)
  TYPE_THEN `?g. (g (&0) = x) /\ (g (&1) = y) /\ INJ g { x | &0 <= x /\ x <= &1 } (euclid 2) /\ continuous g (top_of_metric(UNIV,d_real)) (top_of_metric(euclid 2,d_euclid)) /\ IMAGE g { x | &0 <= x /\ x <= &1 } SUBSET C` SUBGOAL_TAC;
  TYPE_THEN `(tx < ty) \/ (ty < tx)` SUBGOAL_TAC;
  UND 28;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `(?g.   (IMAGE g {x | &0 <= x /\ x <= &1} = IMAGE f {x | tx <= x /\ x <= ty})  /\     (g (&0) = f tx) /\ (g (&1) = f ty) /\       INJ g { x | &0 <= x /\ x <= &1 } (euclid 2) /\       continuous g (top_of_metric(UNIV,d_real))             (top_of_metric(euclid 2,d_euclid)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  arc_restrict;
  TYPE_THEN `&0` EXISTS_TAC;
  TYPE_THEN `&1` EXISTS_TAC;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `&0 < &1`;];
  UND 15;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_euclid];
  REP_BASIC_TAC;
  TYPE_THEN `g` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  IMAGE_SUBSET;
  REWRITE_TAC[SUBSET];
  GEN_TAC;
  UND 24;
  UND 26;
  REAL_ARITH_TAC;
  TYPE_THEN `(?g.   (IMAGE g {x | &0 <= x /\ x <= &1} = IMAGE f {x | ty <= x /\ x <= tx})  /\     (g (&0) = f ty) /\ (g (&1) = f tx) /\       INJ g { x | &0 <= x /\ x <= &1 } (euclid 2) /\       continuous g (top_of_metric(UNIV,d_real))             (top_of_metric(euclid 2,d_euclid)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  arc_restrict;
  TYPE_THEN `&0` EXISTS_TAC;
  TYPE_THEN `&1` EXISTS_TAC;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `&0 < &1`;];
  UND 15;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_euclid];
  REP_BASIC_TAC;
  (* REVERSE reparameter on C XX0 *)
  TYPE_THEN `(?g'. continuous g' (top_of_metric (UNIV,d_real)) (top2) /\           INJ g' {x | (&0) <= x /\ x <= (&1)} (euclid 2) /\         (g (&1)  = g' (&0)) /\ (g (&0) = g' (&1)) /\      (!x y x' y'. (g x = g' x') /\ (g y = g' y') /\         ((&0) <= x /\ x <= (&1)) /\ ((&0) <= y /\ y <= (&1)) /\         ((&0) <= x' /\ x' <= (&1)) /\ ((&0) <= y' /\ y' <= (&1)) ==>           ((x < y) <=> (y' < x'))) /\      (IMAGE g { x | (&0) <= x /\ x <= (&1) } =          IMAGE g' { x | (&0) <= x /\ x <= (&1) } ))` SUBGOAL_TAC;
  IMATCH_MP_TAC  arc_reparameter_rev;
  ASM_REWRITE_TAC[REAL_ARITH `&0 < &1`;top2;];
  REP_BASIC_TAC;
  TYPE_THEN `g'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  ASM_MESON_TAC[];  (* L80 *)
  CONJ_TAC;
  ASM_MESON_TAC[];
  CONJ_TAC;
  ASM_MESON_TAC[top2];
  TYPE_THEN `IMAGE g' {x | &0 <= x /\ x <= &1} = IMAGE f {x | ty <= x /\ x <= tx }` SUBGOAL_TAC;
  UND 34;
  UND 35;
  alpha_tac;
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  IMAGE_SUBSET;
  REWRITE_TAC[SUBSET];
  UND 23;
  UND 27;
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  (* now restrict C to [x,y'] *)
  (* rC *)
  TYPE_THEN `Cg = IMAGE g {x | &0 <= x /\ x <= &1 }` ABBREV_TAC ;
  TYPE_THEN `Z = Cg INTER C'` ABBREV_TAC ;
  TYPE_THEN `?t'. (&0 <= t' /\ t' <= &1) /\ (Z (g t')) /\ (!s. (&0 <=s /\ s < t') ==> ~(Z (g s)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  preimage_first;
  EXISTS_TAC `2`;
  (* restriction conditions *)
  CONJ_TAC;
  TYPE_THEN `induced_top(top_of_metric(UNIV,d_real)) {x | &0 <= x /\ x <= &1 } = top_of_metric ({x | &0 <= x /\ x <= &1 },d_real)` SUBGOAL_TAC;
  ASM_SIMP_TAC[SUBSET_UNIV;metric_real;top_of_metric_induced];
  DISCH_THEN (fun t -> REWRITE_TAC[GSYM t]);
  IMATCH_MP_TAC  continuous_induced_domain;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_real];
  SUBCONJ_TAC;
  UND 31;
  REWRITE_TAC[INJ;IMAGE;SUBSET;];
  MESON_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  (* rC2 *)
  TYPE_THEN `!C. (?f a b. (continuous f (top_of_metric(UNIV,d_real)) (top2)) /\ (INJ f {x | a <= x /\ x <= b} (euclid 2)) /\ (IMAGE f {x | a <= x /\ x <= b} = C)) ==> (closed_ top2 C)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  compact_closed;
  ASM_SIMP_TAC[top2;metric_hausdorff;metric_euclid];
  ASM_SIMP_TAC[top_of_metric_top;metric_euclid];
  EXPAND_TAC "C''";
  IMATCH_MP_TAC  image_compact;
  TYPE_THEN `(top_of_metric(UNIV,d_real))` EXISTS_TAC;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_euclid;interval_compact];
  ASM_SIMP_TAC[GSYM top2];
  EXPAND_TAC "C''";
  UND 38;
  REWRITE_TAC[INJ;IMAGE;SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  REWRITE_TAC[GSYM top2];
  EXPAND_TAC "Z";
  IMATCH_MP_TAC  closed_inter2;
  REWRITE_TAC[top2_top];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `g` EXISTS_TAC;
  TYPE_THEN `&0` EXISTS_TAC;
  TYPE_THEN `&1` EXISTS_TAC;
  ASM_REWRITE_TAC[];  (* XX2 *)
  ASM_SIMP_TAC[top2];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `f'` EXISTS_TAC;
  TYPE_THEN `&0` EXISTS_TAC;
  TYPE_THEN `&1` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[top2];
  UND 6;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_euclid];
  EXPAND_TAC "Z";
  REWRITE_TAC[EMPTY_EXISTS;INTER;IMAGE];
  CONV_TAC (dropq_conv "u");
  TYPE_THEN `&1` EXISTS_TAC;
  EXPAND_TAC "Cg";
  ASM_REWRITE_TAC[IMAGE;];
  REPEAT (CONJ_TAC THEN (TRY (REAL_ARITH_TAC)));
  EXPAND_TAC "Cg";  (* L160 *)
  (remark "LINE 160"; ALL_TAC);
  REWRITE_TAC[IMAGE];
  TYPE_THEN `&1` EXISTS_TAC;
  REPEAT (CONJ_TAC THEN (TRY (REAL_ARITH_TAC)));
  ASM_REWRITE_TAC[];
  UND 1;
  ASM_REWRITE_TAC[IMAGE];
  REP_BASIC_TAC;
  TYPE_THEN `(t' = &0) \/ (&0 < t')` SUBGOAL_TAC;
  UND 39;
  REAL_ARITH_TAC;
  (* elim t' =0 *)
  DISCH_THEN DISJ_CASES_TAC;
  UND 37;
  EXPAND_TAC "Z";
  REWRITE_TAC[INTER];
  ASM_MESON_TAC[];
  (*  **  START ON 2nd BRANCH  ** *** ** *)
  (* 2b*)
  TYPE_THEN `?tz. (&0 <= tz) /\ (tz <= &1) /\ (f' tz = z)` SUBGOAL_TAC;
  UND 0;
  ASM_REWRITE_TAC[IMAGE;];
  DISCH_THEN (CHOOSE_THEN MP_TAC);
  LEFT_TAC "tz";
  TYPE_THEN `x'` EXISTS_TAC;
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `?t''. (&0 <= t'') /\ (t'' <= &1) /\ (f' t'' = g t')` SUBGOAL_TAC;
  UND 37;
  EXPAND_TAC "Z";
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[IMAGE;];
  DISCH_THEN (fun t-> MP_TAC (CONJUNCT2 t));
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `~(tz = t'')` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `C (g t')` SUBGOAL_TAC;
  UND 37;
  EXPAND_TAC "Z";
  REWRITE_TAC[INTER];
  UND 29;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* reparam on C' *)
  TYPE_THEN `?h. (h (&1/(&2)) = g t') /\ (h (&1) = z) /\ INJ h { x | &1/(&2) <= x /\ x <= &1 } (euclid 2) /\ continuous h (top_of_metric(UNIV,d_real)) (top_of_metric(euclid 2,d_euclid)) /\ IMAGE h { x | &1/(&2) <= x /\ x <= &1 } SUBSET C'` SUBGOAL_TAC;
  TYPE_THEN `(t'' < tz) \/ (tz < t'')` SUBGOAL_TAC;
  UND 47;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `(?h.   (IMAGE h {x | &1/(&2) <= x /\ x <= &1} = IMAGE f' {x | t'' <= x /\ x <= tz})  /\     (h (&1/(&2)) = f' t'') /\ (h (&1) = f' tz) /\       INJ h { x | &1/(&2) <= x /\ x <= &1 } (euclid 2) /\       continuous h (top_of_metric(UNIV,d_real))             (top_of_metric(euclid 2,d_euclid)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  arc_restrict;
  TYPE_THEN `&0` EXISTS_TAC;
  TYPE_THEN `&1` EXISTS_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 6;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_euclid];
  DISCH_TAC;
  REWRITE_TAC[REAL_LT_HALF2];
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `h` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  IMAGE_SUBSET;
  REWRITE_TAC[SUBSET];
  GEN_TAC;
  UND 42;
  UND 46;
  REAL_ARITH_TAC;
  TYPE_THEN `(?h.   (IMAGE h {x | &1/(&2) <= x /\ x <= &1} = IMAGE f' {x | tz <= x /\ x <= t'' })  /\     (h (&1/(&2)) = f' tz) /\ (h (&1) = f' t'') /\       INJ h { x | &1/(&2) <= x /\ x <= &1 } (euclid 2) /\       continuous h (top_of_metric(UNIV,d_real))  (top_of_metric(euclid 2,d_euclid)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  arc_restrict;
  TYPE_THEN `&0` EXISTS_TAC;
  TYPE_THEN `&1` EXISTS_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[REAL_LT_HALF2;REAL_ARITH `&0 < &1`];
  UND 6;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_euclid];
  REP_BASIC_TAC;  (* L240 *)
  (remark "LINE 240"; ALL_TAC);
  (* REVERSE reparameter on C *)
  TYPE_THEN `(?h'. continuous h' (top_of_metric (UNIV,d_real)) (top2) /\           INJ h' {x | (&1/(&2)) <= x /\ x <= (&1)} (euclid 2) /\         (h (&1)  = h' (&1/(&2))) /\ (h (&1/(&2)) = h' (&1)) /\      (!x y x' y'. (h x = h' x') /\ (h y = h' y') /\         ((&1/(&2)) <= x /\ x <= (&1)) /\ ((&1/(&2)) <= y /\ y <= (&1)) /\         ((&1/(&2)) <= x' /\ x' <= (&1)) /\ ((&1/(&2)) <= y' /\ y' <= (&1)) ==>           ((x < y) <=> (y' < x'))) /\      (IMAGE h { x | (&1/(&2)) <= x /\ x <= (&1) } =          IMAGE h' { x | (&1/(&2)) <= x /\ x <= (&1) } ))` SUBGOAL_TAC;
  IMATCH_MP_TAC  arc_reparameter_rev;
  ASM_REWRITE_TAC[REAL_LT_HALF2;REAL_ARITH `&0 < &1`;top2;];
  REP_BASIC_TAC;
  TYPE_THEN `h'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  ASM_MESON_TAC[];
  CONJ_TAC;
  ASM_MESON_TAC[];
  CONJ_TAC;
  ASM_MESON_TAC[top2];
  TYPE_THEN `IMAGE h' {x | &1/(&2) <= x /\ x <= &1} = IMAGE f' {x | tz <= x /\ x <= t'' }` SUBGOAL_TAC;
  UND 53;  (* ZZZ *)
  UND 54;
  alpha_tac;
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  IMAGE_SUBSET;
  REWRITE_TAC[SUBSET];
  UND 43;
  UND 45;
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  (* reparam g [0,1/2] *)
  (* rg *)
  TYPE_THEN `?g'. ((g' (&0)) = x) /\ (g' (&1/(&2)) = g t') /\ INJ g' { x | &0 <= x /\ x <= &1/(&2) } (euclid 2) /\ continuous g' (top_of_metric(UNIV,d_real)) (top_of_metric(euclid 2,d_euclid)) /\ (IMAGE g' { x | &0 <= x /\ x <= &1/(&2) } = IMAGE g {x | &0 <= x /\ x <= t'}) ` SUBGOAL_TAC; (* was SUBSET Cg *)
  ASSUME_TAC arc_reparameter_gen;
  TYPEL_THEN [`g`;`&0`;`&1/(&2)`;`&0`;`t'`] (fun t-> FIRST_ASSUM (fun s-> (MP_TAC (ISPECL t s))));
  KILL 53;   (* ZZZ *)
  ASM_REWRITE_TAC[REAL_ARITH `&0 < &1`;REAL_LT_HALF1;];
  UND 30;
  REWRITE_TAC[top2];
  DISCH_THEN_REWRITE;
  TYPE_THEN `INJ g {x | &0 <= x /\ x <= t'} (euclid 2)` SUBGOAL_TAC;
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x <= &1 }` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  UND 38;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REP_BASIC_TAC;
  TYPE_THEN `g'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* deleted lines here *)
  REP_BASIC_TAC;
  TYPE_THEN `fm = joinf g' h (&1/(&2))` ABBREV_TAC ;
  TYPE_THEN `Cm = IMAGE fm {x | &0 <= x /\ x <= &1}` ABBREV_TAC ;
  TYPE_THEN `Cm` EXISTS_TAC;
  (* final instantiation *)
  (* fi *)
  REPEAT (IMATCH_MP_TAC  (TAUT `A /\ B/\ C ==> (A /\ B) /\C`));
  CONJ_TAC;
  TYPE_THEN `fm` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  EXPAND_TAC "fm";
  IMATCH_MP_TAC  joinf_cont;
  ASM_REWRITE_TAC[];
  TYPE_THEN `{x | &0 <= x /\ x <= &1} = {x | &0 <= x /\ x < &1/(&2)} UNION {x | &1/(&2) <= x /\ x <= &1 }` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION];
  GEN_TAC;
  TYPE_THEN `&0 < &1/(&2) /\ (&1/(&2) < &1)` SUBGOAL_TAC;
  REWRITE_TAC[REAL_LT_HALF1;REAL_LT_HALF2;REAL_ARITH `&0 < &1`];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  inj_split;
  EXPAND_TAC "fm";
  TYPE_THEN `{x | &0 <= x /\ x < &1/(&2)} SUBSET {x | x < &1/(&2)} /\ {x | &1/(&2) <= x /\ x <= &1} SUBSET {x | &1/(&2) <= x}` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  KILL 58;
  ASM_SIMP_TAC[joinf_inj_above;joinf_inj_below;joinf_image_above;joinf_image_below];
  DISCH_TAC;
  (* cases *)
  CONJ_TAC;
  IMATCH_MP_TAC  inj_subset_domain;  (* L320 *)
  (remark "LINE 320"; ALL_TAC);
  TYPE_THEN `{x | &0 <= x /\ x <= &1/(&2) }` EXISTS_TAC;
  ASM_SIMP_TAC[GSYM  top_of_metric_unions;metric_euclid];
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  CONJ_TAC;
  ASM_SIMP_TAC[GSYM  top_of_metric_unions;metric_euclid];
  ASM_REWRITE_TAC[];
  TYPE_THEN `IMAGE g' { x | &0 <= x /\ x <= &1/(&2)} INTER IMAGE h {x | &1/(&2) <= x /\ x <= &1} SUBSET {(g t')}` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `IMAGE g { x | &0 <= x /\ x <= t' } SUBSET Cg` SUBGOAL_TAC;
  EXPAND_TAC "Cg";
  IMATCH_MP_TAC  IMAGE_SUBSET;
  REWRITE_TAC[SUBSET];
  UND 38;
  REAL_ARITH_TAC;
  DISCH_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `IMAGE g {x | &0 <= x /\ x <= t'} INTER Z` EXISTS_TAC;
  CONJ_TAC;
  EXPAND_TAC "Z";
  UND 48;
  UND 60;
  REWRITE_TAC[SUBSET;INTER];
  (* MESON_TAC[]; *)
  POP_ASSUM_LIST (fun t-> ALL_TAC);
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  (* LINE 350 *)
  CONJ_TAC THEN (FIRST_ASSUM IMATCH_MP_TAC ) THEN ASM_REWRITE_TAC[];
  UND 36;
  REWRITE_TAC[INTER;SUBSET;IMAGE];
  UND 37;
  POP_ASSUM_LIST (fun t-> ALL_TAC);
  REP_BASIC_TAC;
  REWRITE_TAC[INR IN_SING];
  UND 0;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `(x' = t') \/ (x' < t')` SUBGOAL_TAC;
  UND 2;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  PROOF_BY_CONTR_TAC;
  USE 61 (REWRITE_RULE[EMPTY_EXISTS ]);
  REP_BASIC_TAC;
  TYPE_THEN `!B' B (u:num->real). (B' u /\ B' SUBSET B) ==> (B u)` SUBGOAL_TAC;
  MESON_TAC[ISUBSET];
  DISCH_TAC;
  TYPE_THEN `{(g t')} u` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `(IMAGE g' {x | &0 <= x /\ x < &1 / &2} INTER IMAGE h {x | &1 / &2 <= x /\ x <= &1})` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `(IMAGE g' {x | &0 <= x /\ x <= &1 / &2} INTER IMAGE h {x | &1 / &2 <= x /\ x <= &1})` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[INTER;SUBSET;IMAGE];
  MESON_TAC[REAL_ARITH `x < t ==> x <= t`];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INR IN_SING];
  REP_BASIC_TAC;
  UND 62;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INTER;IMAGE;DE_MORGAN_THM;];
  DISJ1_TAC;
  USE 56 SYM;
  ASM_REWRITE_TAC[];
  UND 55;
  POP_ASSUM_LIST (fun t-> ALL_TAC);
  REWRITE_TAC[INJ];
  REP_BASIC_TAC;
  USE 1(REWRITE_RULE [REAL_ARITH `(x < &1/(&2)) <=> (x <= &1/(&2) /\ ~(x = &1/(&2)))`]);
  TYPEL_THEN [`x`;`&1/(&2)`] (USE 3 o ISPECL);
  TYPE_THEN `&0 <= &1/ &2 /\ &1/ &2 <= &1/ (&2)` SUBGOAL_TAC;
  REWRITE_TAC[REAL_ARITH `x <= x`];
  IMATCH_MP_TAC  REAL_LE_DIV;
  REAL_ARITH_TAC;
  ASM_MESON_TAC[];
  (* Now E *)   (* L400 *)
  (remark "LINE 400"; ALL_TAC);
  (* ne *)
  TYPE_THEN ` {x | &0 <= x /\ x <= &1} = {x | &0 <= x /\ x < &1/(&2)} UNION {x | &1/(&2) <= x /\ x <= &1 }` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION];
  GEN_TAC;
  TYPE_THEN `&0 < &1/(&2) /\ (&1/(&2) < &1)` SUBGOAL_TAC;
  REWRITE_TAC[REAL_LT_HALF1;REAL_LT_HALF2;REAL_ARITH `&0 < &1`];
  REAL_ARITH_TAC;
  EXPAND_TAC "Cm";
  DISCH_THEN_REWRITE;
  REWRITE_TAC[IMAGE_UNION];
  TYPE_THEN `{x | &0 <= x /\ x < &1/(&2)} SUBSET {x | x < &1/(&2)} /\ {x | &1/(&2) <= x /\ x <= &1} SUBSET {x | &1/(&2) <= x}` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  EXPAND_TAC "fm";
  KILL 58;
  ASM_SIMP_TAC[joinf_image_above;joinf_image_below];
  DISCH_TAC;
  TYPE_THEN `(IMAGE g' {x | &0 <= x /\ x < &1 / &2} UNION  IMAGE h {x | &1 / &2 <= x /\ x <= &1}) z` SUBGOAL_TAC;
  UND 51;
  REWRITE_TAC[UNION;IMAGE];
  POP_ASSUM_LIST (fun t->ALL_TAC);
  REP_BASIC_TAC;
  DISJ2_TAC;
  TYPE_THEN `&1` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[REAL_ARITH `&1 <= &1`];
  IMATCH_MP_TAC  REAL_LE_LDIV;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `(IMAGE g' {x | &0 <= x /\ x < &1 / &2} UNION  IMAGE h {x | &1 / &2 <= x /\ x <= &1}) x` SUBGOAL_TAC;
  UND 57;
  REWRITE_TAC[UNION;IMAGE];
  POP_ASSUM_LIST (fun t->ALL_TAC);
  REP_BASIC_TAC;
  DISJ1_TAC;
  TYPE_THEN `&0` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[REAL_ARITH `&0 <= &0`];
  REWRITE_TAC[REAL_LT_HALF1];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  (* gh *)
  UND 48;
  TYPE_THEN `IMAGE g' {x | &0 <= x /\ x < &1/ &2} SUBSET C` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `Cg ` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  EXPAND_TAC "Cg";
  TYPE_THEN `IMAGE g {x | &0 <= x /\ x <= t'}` EXISTS_TAC;
  CONJ_TAC;
  USE 53 SYM;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE;SUBSET];
  MESON_TAC[REAL_ARITH `x < t ==> x <= t`];
  REWRITE_TAC[IMAGE;SUBSET];
  UND 38;
  MESON_TAC[REAL_ARITH `t' <= &1 ==> (x <= t' ==> x<= &1)`];
  TYPE_THEN `GCG = IMAGE g' {x | &0 <= x /\ x < &1 / &2}` ABBREV_TAC ;
  TYPE_THEN `HCH = IMAGE h {x | &1 / &2 <= x /\ x <= &1}` ABBREV_TAC ;
  UND 11;
  UND 2;
  UND 4;
  UND 5;
  UND 13;
  UND 14;
  UND 12;
  UND 3;
  POP_ASSUM_LIST (fun t->ALL_TAC);
  REP_BASIC_TAC;
  CONJ_TAC;
  TYPE_THEN `E UNION E'` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[UNIONS_UNION];
  REWRITE_TAC[union_subset];
  CONJ_TAC;
  UND 1;
  UND 7;
  REWRITE_TAC[UNION;SUBSET];  (* L480 *)
  (remark "LINE 480"; ALL_TAC);
  MESON_TAC[];
  UND 0;
  UND 5;
  REWRITE_TAC[UNION;SUBSET];
  MESON_TAC[];
  CONJ_TAC;
  ASM_REWRITE_TAC[FINITE_UNION];
  UND 8;
  UND 9;
  REWRITE_TAC[hv_line;UNION;];
  MESON_TAC[];
  UND 1;
  UND 0;
  UND 2;
  UND 3;
  REWRITE_TAC[SUBSET;UNION;];
  MESON_TAC[];
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* SECTION J *)
(* ------------------------------------------------------------------ *)


(* Conclusion of Jordan Curve, page 1 *)

let v_simple_polygonal = prove_by_refinement(
  `!x e. (euclid 2 x) /\ (~(e = &0)) ==>
    (simple_polygonal_arc hv_line (mk_segment x (x + e *# e2)))`,
  (* {{{ proof *)

  [
  REWRITE_TAC[simple_polygonal_arc;hv_line;simple_arc ];
  REP_BASIC_TAC;
  CONJ_TAC;
  ASSUME_TAC mk_segment_inj_image;
  TYPEL_THEN [`x`;`x + (e *# e2)`;`2`] (USE 2 o ISPECL);
  TYPE_THEN `euclid 2 x /\ euclid 2 (euclid_plus x (e *# e2)) /\ ~(x = euclid_plus x (e *# e2))` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  euclid_add_closure;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  euclid_scale_closure;
  REWRITE_TAC [e2;euclid_point];
  REP_BASIC_TAC;
  FIRST_ASSUM  (fun t-> MP_TAC (AP_THM t `1`));
  REWRITE_TAC[euclid_plus;euclid_scale;e2;coord01];
  UND 0;
  REAL_ARITH_TAC;
  DISCH_TAC;
  REWR 2;
  REP_BASIC_TAC;
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  SIMP_TAC  [GSYM top_of_metric_unions;metric_euclid];
  ASM_REWRITE_TAC[];
  (* E *)
  USE 1 (MATCH_MP point_onto);
  REP_BASIC_TAC;
  TYPE_THEN `{(mk_line (point p) (point p + (e *# e2)))}` EXISTS_TAC;
  REWRITE_TAC[INR IN_SING];
  CONJ_TAC;
  REWRITE_TAC[e2;ISUBSET;mk_segment;mk_line];
  REP_BASIC_TAC;
  TYPE_THEN `a` EXISTS_TAC;
  ASM_MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC[FINITE_SING];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `p` EXISTS_TAC;
  TYPE_THEN `(FST p , SND p + e)` EXISTS_TAC;
  REWRITE_TAC[];
  AP_TERM_TAC;
  REWRITE_TAC[e2;point_scale];
  REDUCE_TAC;
  TYPE_THEN `euclid_plus (point p) (point (&0,e)) = euclid_plus (point (FST p,SND p)) (point (&0,e))` SUBGOAL_TAC;
  REWRITE_TAC[];
  DISCH_THEN (fun t-> PURE_ONCE_REWRITE_TAC[t]);
  REWRITE_TAC[point_add];
  REDUCE_TAC;
  ]);;

  (* }}} *)

let p_conn_ball = prove_by_refinement(
  `! x y r. (open_ball(euclid 2,d_euclid) x r y) ==>
      (p_conn (open_ball(euclid 2,d_euclid) x r) x y)`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  TYPE_THEN `open_ball (euclid 2,d_euclid) x r x` SUBGOAL_TAC;
  SIMP_TAC [metric_euclid;INR open_ball_nonempty_center];
  REWRITE_TAC[EMPTY_EXISTS];
  ASM_MESON_TAC[];
  DISCH_TAC;

  TYPE_THEN `euclid 2 x /\ euclid 2 y` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[open_ball]);
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  RULE_ASSUM_TAC  (fun t -> try (MATCH_MP point_onto t) with  Failure _ -> t);
  REP_BASIC_TAC;

  TYPE_THEN `y' = point(FST p,SND p')` ABBREV_TAC ;
  TYPE_THEN `A = open_ball(euclid 2,d_euclid) x r` ABBREV_TAC ;

  TYPE_THEN `y' = euclid_plus x ((SND  p' - SND  p) *# e2)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "y'";
  REWRITE_TAC[e2];
  REWRITE_TAC[point_add;point_scale;];
  REDUCE_TAC;
  PURE_ONCE_REWRITE_TAC [GSYM PAIR];
  PURE_REWRITE_TAC [point_add];
  REWRITE_TAC[];
  REDUCE_TAC;
  AP_TERM_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  REAL_ARITH_TAC;
  DISCH_TAC;

  TYPE_THEN `A y'` SUBGOAL_TAC;
  UND 0;
  EXPAND_TAC "y'";
  KILL 4;
  EXPAND_TAC "A";
  KILL 5;
  ASM_REWRITE_TAC[open_ball;euclid_point;d_euclid_point;];
  REWRITE_TAC[REAL_ARITH `(x - x = &0)`;POW_0;ARITH_RULE  `2 = SUC 1`];
  IMATCH_MP_TAC  (REAL_ARITH `(x <= y) ==> (y < r ==> x < r)`);
  IMATCH_MP_TAC  SQRT_MONO_LE;
  REWRITE_TAC[REAL_ARITH `&0 + x = x`;ARITH_RULE `SUC 1 = 2`;REAL_PROP_NN_SQUARE];
  IMATCH_MP_TAC  (REAL_ARITH `&0 <= x ==> (y <= x + y)`);
  REWRITE_TAC[REAL_PROP_NN_SQUARE];
  DISCH_TAC;

  TYPE_THEN `p_conn A x y'` SUBGOAL_TAC;
  TYPE_THEN `x = y'` ASM_CASES_TAC;
  EXPAND_TAC "y'";
  IMATCH_MP_TAC  pconn_refl;
  REWRITE_TAC[p_conn];
  CONJ_TAC;
  EXPAND_TAC "A";
  REWRITE_TAC[top2];
  IMATCH_MP_TAC  open_ball_open;
  MESON_TAC[metric_euclid];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[p_conn];
  TYPE_THEN `mk_segment x y'` EXISTS_TAC;
  CONJ_TAC;
  UND 6;
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  v_simple_polygonal;
  ASM_REWRITE_TAC[euclid_point];
  REWRITE_TAC[REAL_SUB_0];
  DISCH_ALL_TAC;
  UND 8;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "y'";
  AP_TERM_TAC;
  ASM_MESON_TAC[PAIR];
  CONJ_TAC;
  EXPAND_TAC "A";
  IMATCH_MP_TAC  openball_mk_segment_end;
  ASM_MESON_TAC[];
  REWRITE_TAC[mk_segment_end];
  DISCH_TAC;

  TYPE_THEN `y' = euclid_plus y ((FST   p - FST   p') *# e1)` SUBGOAL_TAC;
  KILL 6;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "y'";
  REWRITE_TAC[e1];
  REWRITE_TAC[point_add;point_scale;];
  REDUCE_TAC;
  PURE_ONCE_REWRITE_TAC [GSYM PAIR];
  PURE_REWRITE_TAC [point_add];
  REWRITE_TAC[];
  REDUCE_TAC;
  AP_TERM_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  REAL_ARITH_TAC;
  DISCH_TAC;

  TYPE_THEN `p_conn A y y'` SUBGOAL_TAC;
  TYPE_THEN `y = y'` ASM_CASES_TAC;
  EXPAND_TAC "y'";
  IMATCH_MP_TAC  pconn_refl;
  CONJ_TAC;
  EXPAND_TAC "A";
  REWRITE_TAC[top2];
  IMATCH_MP_TAC  open_ball_open;
  MESON_TAC[metric_euclid];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[p_conn];
  TYPE_THEN `mk_segment y y'` EXISTS_TAC;
  CONJ_TAC;
  UND 9;
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  h_simple_polygonal;
  ASM_REWRITE_TAC[euclid_point];
  REWRITE_TAC[REAL_SUB_0];
  DISCH_ALL_TAC;
  UND 10;
  KILL 6;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "y'";
  AP_TERM_TAC;
  ASM_MESON_TAC[PAIR];
  CONJ_TAC;
  EXPAND_TAC "A";
  IMATCH_MP_TAC  openball_mk_segment_end;
  ASM_MESON_TAC[];
  REWRITE_TAC[mk_segment_end];
  DISCH_TAC;
  IMATCH_MP_TAC  pconn_trans;
  TYPE_THEN `y'` EXISTS_TAC;
  UND 8;
  DISCH_THEN_REWRITE;
  UND 10;
  MESON_TAC[pconn_symm];
  (* Wed Aug  4 10:40:05 EDT 2004 *)

  ]);;

  (* }}} *)

let p_conn_euclid = prove_by_refinement(
  `!A x. p_conn A x SUBSET (euclid 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;p_conn;simple_polygonal_arc;simple_arc;];
  REP_BASIC_TAC;
  UND 0;
  ASM_REWRITE_TAC[];
  UND 6;
  SIMP_TAC[GSYM top_of_metric_unions;metric_euclid];
  REWRITE_TAC[INJ;IMAGE];
  MESON_TAC[];
  (* Wed Aug  4 10:55:53 EDT 2004 *)
  ]);;
  (* }}} *)

let p_connA = prove_by_refinement(
  `!A x. p_conn A x SUBSET A`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[p_conn;SUBSET;];
  ASM_MESON_TAC[];
  (* Wed Aug  4 11:11:21 EDT 2004 *)
  ]);;
  (* }}} *)

let p_conn_open = prove_by_refinement(
  `!A x. top2 A ==> (top2 (p_conn A x))`,
  (* {{{ proof *)
  [
  (* Wed Aug  4 10:43:29 EDT 2004 *)
  REP_BASIC_TAC;
  ASM_SIMP_TAC[top2;top_of_metric_nbd;metric_euclid;p_conn_euclid];
  REP_BASIC_TAC;

  TYPE_THEN `A a` SUBGOAL_TAC;
  ASM_MESON_TAC[p_connA;ISUBSET];
  DISCH_TAC;

  TYPE_THEN `?r. (&0 < r) /\ open_ball (euclid 2,d_euclid) a r SUBSET A` SUBGOAL_TAC;
  ASM_MESON_TAC[metric_euclid;top2;open_ball_nbd;];
  REP_BASIC_TAC;
  TYPE_THEN `r` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET;];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  pconn_trans;
  TYPE_THEN `a` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  p_conn_subset;
  TYPE_THEN `open_ball (euclid 2,d_euclid) a r` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  p_conn_ball;
  ASM_REWRITE_TAC[];
  (* Wed Aug  4 11:21:18 EDT 2004 *)
  ]);;
  (* }}} *)

let p_conn_diff = prove_by_refinement(
  `!A x.  top2 A ==> (top2 (A DIFF (p_conn A x)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  SIMP_TAC[top2;metric_euclid;top_of_metric_nbd];
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `A` EXISTS_TAC;
  REWRITE_TAC[SUBSET_DIFF];
  UND 0;
  REWRITE_TAC[top2;];
  DISCH_TAC;
  FIRST_ASSUM (fun t-> ASSUME_TAC (MATCH_MP sub_union t));
  UND 1;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_euclid];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[DIFF]);
  REP_BASIC_TAC;

  TYPE_THEN `?r. (&0 < r) /\ open_ball (euclid 2,d_euclid) a r SUBSET A` SUBGOAL_TAC;
  ASM_MESON_TAC[metric_euclid;top2;open_ball_nbd;];
  REP_BASIC_TAC;

  TYPE_THEN `r` EXISTS_TAC;
  ASM_REWRITE_TAC[DIFF_SUBSET];
  PROOF_BY_CONTR_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[EMPTY_EXISTS;INTER]);
  REP_BASIC_TAC;
  FIRST_ASSUM (fun t -> ASSUME_TAC (MATCH_MP  p_conn_ball t));
  TYPE_THEN `p_conn A a u` SUBGOAL_TAC;
  IMATCH_MP_TAC  p_conn_subset;
  ASM_MESON_TAC[];
  DISCH_TAC;
  UND 1;
  REWRITE_TAC[];
  IMATCH_MP_TAC  pconn_trans;
  TYPE_THEN `u` EXISTS_TAC;
  ASM_MESON_TAC[pconn_symm];
  (* Wed Aug  4 12:00:13 EDT 2004 *)
  ]);;
  (* }}} *)

let p_conn_conn = prove_by_refinement(
  `!A x y. (top2 A /\ connected top2 A /\ A x /\ A y) ==>
     (p_conn A x y)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[connected];
  REP_BASIC_TAC;
  TYPEL_THEN [`p_conn A x`;`A DIFF (p_conn A x)`] (USE 2 o ISPECL);
  UND 2;
  ASM_SIMP_TAC[p_conn_open;p_conn_diff];

  TYPE_THEN `!(w:(num->real)->bool) z. (w INTER (z DIFF w) = EMPTY)` SUBGOAL_TAC;
  SET_TAC[INTER;DIFF];
  DISCH_THEN_REWRITE;

  TYPE_THEN `!(x:(num->real)->bool) y. (x SUBSET (y UNION (x DIFF y)))` SUBGOAL_TAC;
  SET_TAC[SUBSET;UNION;DIFF];
  DISCH_THEN_REWRITE;

  DISCH_THEN (DISJ_CASES_TAC);
  ASM_MESON_TAC[ISUBSET];
  UND 2;
  REWRITE_TAC[SUBSET;DIFF];
  ASM_MESON_TAC[pconn_refl];
  (* Wed Aug  4 12:42:12 EDT 2004 *)
  ]);;
  (* }}} *)

let plane_graph = jordan_def
  `plane_graph G <=>
     graph_vertex G SUBSET (euclid 2) /\
     graph G /\
     graph_edge G SUBSET (simple_arc top2) /\
     (!e. (graph_edge G e ==>
        (graph_inc G e = e INTER (graph_vertex G)))) /\
     (!e e'. (graph_edge G e /\ graph_edge G e' /\ ~(e = e')) ==>
        (e INTER e' SUBSET (graph_vertex G)))`;;

let graph_isomorphic = jordan_def
  `graph_isomorphic (G:(A,B)graph_t) (H:(A',B')graph_t) <=>
     ?f. (graph_iso f G H)`;;

let I_BIJ = prove_by_refinement(
  `!(x:A->bool). BIJ I x x`,
  (* {{{ proof *)
  [
  REWRITE_TAC[BIJ;INJ;SURJ;I_THM;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let graph_isomorphic_refl = prove_by_refinement(
  `!(G:(A,B)graph_t). graph_isomorphic G G`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_isomorphic;graph_iso;];
  REP_BASIC_TAC;
  RIGHT_TAC  "f";
  RIGHT_TAC  "f";
  TYPE_THEN `I:A->A` EXISTS_TAC;
  TYPE_THEN `I:B->B` EXISTS_TAC;
  TYPE_THEN `(I:A->A,I:B->B)` EXISTS_TAC;
  ASM_REWRITE_TAC[I_THM;IMAGE_I;I_BIJ];
  (* Wed Aug  4 13:08:32 EDT 2004 *)

  ]);;
  (* }}} *)

let graph_inc_subset = prove_by_refinement(
  `!(G:(A,B)graph_t) e. (graph G /\ graph_edge G e) ==>
       (graph_inc G e SUBSET graph_vertex G)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph;IMAGE;SUBSET;];
  NAME_CONFLICT_TAC;
  REP_BASIC_TAC;
  USE 2 (CONV_RULE (dropq_conv "x''"));
  TSPEC  `e'` 2;
  REWR 2;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let graph_isomorphic_symm = prove_by_refinement(
  `!(G:(A,B)graph_t) (H:(A',B')graph_t).
     graph G /\ graph_isomorphic G H ==> graph_isomorphic H G`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_isomorphic;graph_iso];
  REP_BASIC_TAC;
  RIGHT_TAC "f";
  RIGHT_TAC "f";
  TYPE_THEN `u' = INV u (graph_vertex G) (graph_vertex H)` ABBREV_TAC  ;
  TYPE_THEN `v' = INV v (graph_edge G) (graph_edge H)` ABBREV_TAC ;
  TYPE_THEN `u'` EXISTS_TAC;
  TYPE_THEN `v'` EXISTS_TAC;
  TYPE_THEN `(u',v')` EXISTS_TAC;
  REWRITE_TAC[];
  CONJ_TAC;
  EXPAND_TAC "u'";
  IMATCH_MP_TAC  INVERSE_BIJ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  EXPAND_TAC "v'";
  IMATCH_MP_TAC  INVERSE_BIJ;
  ASM_REWRITE_TAC[];
  (* LAST step *)
  REP_BASIC_TAC;
  TYPE_THEN `e' = v' e` ABBREV_TAC ;

  TYPE_THEN `e = v e'` SUBGOAL_TAC;
  ASM_MESON_TAC [inv_comp_right];
  DISCH_TAC;
  ASM_REWRITE_TAC[];

  TYPE_THEN `BIJ v' (graph_edge H) (graph_edge G)` SUBGOAL_TAC;
  ASM_MESON_TAC[INVERSE_BIJ];
  DISCH_TAC;

  TYPE_THEN `graph_edge G e'` SUBGOAL_TAC;
  EXPAND_TAC "e'";
  UND 10;
  REWRITE_TAC[BIJ;SURJ;];
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_SIMP_TAC[];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  EXPAND_TAC "u'";
  IMATCH_MP_TAC  image_inv_image;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  graph_inc_subset;
  ASM_MESON_TAC[];
  (* Wed Aug  4 13:53:24 EDT 2004 *)

  ]);;
  (* }}} *)

let graph_isomorphic_trans = prove_by_refinement(
  `!(G:(A,B)graph_t) (H:(A',B')graph_t) (J:(A'',B'')graph_t).
    graph_isomorphic G H /\ graph_isomorphic H J ==>
     graph_isomorphic G J`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_isomorphic;graph_iso;];
  REP_BASIC_TAC;
  KILL 3;
  KILL 7;
  RIGHT_TAC "f";
  RIGHT_TAC "f";
  TYPE_THEN `u' o u` EXISTS_TAC;
  TYPE_THEN `v' o v` EXISTS_TAC;
  TYPE_THEN `(u' o u, v' o v)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  REWRITE_TAC[comp_comp];
  IMATCH_MP_TAC  COMP_BIJ;
  ASM_MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC[comp_comp];
  IMATCH_MP_TAC  COMP_BIJ;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE_o];
  REWRITE_TAC[o_DEF];

  TYPE_THEN `graph_edge H (v e)` SUBGOAL_TAC;
  UND 5;
  REWRITE_TAC[BIJ;SURJ];
  UND 3;
  MESON_TAC[];
  ASM_SIMP_TAC[];
  (* Wed Aug  4 14:13:25 EDT 2004 *)
  ]);;
  (* }}} *)

let graph_isomorphic_graph = prove_by_refinement(
  `!(G:(A,B)graph_t) H.
     graph G /\ graph_isomorphic G (H:(A',B')graph_t) ==> graph H`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  TYPE_THEN `!z. (graph_edge G z ==> graph_inc G z SUBSET graph_vertex G)` SUBGOAL_TAC;
  ASM_MESON_TAC[graph_inc_subset];
  DISCH_TAC;
  UND 0;
  UND 1;
  REWRITE_TAC[graph;graph_isomorphic;graph_iso];
  REP_BASIC_TAC;
  REWRITE_TAC[SUBSET;IMAGE;];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x''");
  REP_BASIC_TAC;
  TYPE_THEN `?y'. (graph_edge G y' /\ (v y' = x'))` SUBGOAL_TAC;
  UND 1;
  REWRITE_TAC[BIJ;SURJ];
  UND 6;
  MESON_TAC[];
  REP_BASIC_TAC;

  TYPE_THEN `graph_inc H x' = IMAGE u (graph_inc G y')` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;

  TYPE_THEN `graph_inc G y' SUBSET graph_vertex G` SUBGOAL_TAC;
  ASM_SIMP_TAC[];
  DISCH_TAC;
  KILL 2;

  SUBCONJ_TAC;
  ASM_REWRITE_TAC[IMAGE];
  UND 10;
  UND 3;
  REWRITE_TAC[BIJ;SURJ];
  MESON_TAC[ISUBSET];
  DISCH_TAC;

  (* has size *)
  TYPE_THEN `(graph_inc G y') HAS_SIZE 2` SUBGOAL_TAC;
  UND 5;
  REWRITE_TAC[SUBSET;IMAGE];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x''");
  UND 8;
  MESON_TAC[];
  DISCH_TAC;


  ASM_REWRITE_TAC[];
  REWRITE_TAC[HAS_SIZE];
  SUBCONJ_TAC;
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_MESON_TAC[HAS_SIZE];
  DISCH_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[HAS_SIZE]);
  REP_BASIC_TAC;
  UND 11;
  DISCH_THEN (fun t -> REWRITE_TAC[GSYM t]);
  IMATCH_MP_TAC  CARD_IMAGE_INJ;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  UND 3;
  REWRITE_TAC[BIJ;INJ];
  REP_BASIC_TAC;
  ASM_MESON_TAC[ISUBSET];
  (* Wed Aug  4 15:18:06 EDT 2004 *)
  ]);;

  (* }}} *)

let planar_graph = jordan_def
  `planar_graph (G:(A,B)graph_t) <=>
      (?H. (plane_graph H) /\ (graph_isomorphic H G))`;;

let plane_planar = prove_by_refinement(
  `!G. (plane_graph G) ==> (planar_graph G)`,
  (* {{{ proof *)

  [
  REWRITE_TAC[planar_graph];
  REP_BASIC_TAC;
  ASM_MESON_TAC[graph_isomorphic_refl];
  ]);;

  (* }}} *)

let planar_is_graph = prove_by_refinement(
  `!(G:(A,B)graph_t). (planar_graph G ==> graph G)`,
  (* {{{ proof *)

  [
  REWRITE_TAC[planar_graph;plane_graph];
  REP_BASIC_TAC;
  ASM_MESON_TAC[graph_isomorphic_graph];
  ]);;

  (* }}} *)

let planar_iso = prove_by_refinement(
  `!G H. (planar_graph (G:(A,B)graph_t)) /\ (graph_isomorphic G H) ==>
    (planar_graph (H:(A',B')graph_t))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[planar_graph];
  REP_BASIC_TAC;
  TYPE_THEN `H'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  JOIN 1 0;
  USE 0 (MATCH_MP graph_isomorphic_trans);
  ASM_REWRITE_TAC[];
  (* Wed Aug  4 15:41:05 EDT 2004 *)

  ]);;
  (* }}} *)

(* almost the same ans num_MAX .  The minimization is num_WOP. *)
let select_num_max = prove_by_refinement(
  `!Y. FINITE Y /\ (~(Y= EMPTY)) ==>
        (?z. (Y z /\ (!y. Y y ==> y <=| z)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `f = \ (t:num). --. (&. t)` ABBREV_TAC ;
  TYPE_THEN `Z = IMAGE f Y` ABBREV_TAC ;
  TYPE_THEN `FINITE Z /\ ~(Z = {})` SUBGOAL_TAC;
  EXPAND_TAC "Z";
  CONJ_TAC;
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC  (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  UND 0;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `f u` EXISTS_TAC;
  REWRITE_TAC[IMAGE];
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 4 (MATCH_MP   min_finite);
  REP_BASIC_TAC;
  TYPE_THEN `?z. Y z /\ (f z = delta)` SUBGOAL_TAC;
  UND 5;
  EXPAND_TAC "Z";
  REWRITE_TAC[IMAGE;SUBSET];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `z` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `(f z <= f y) ==> (y <=| z)` SUBGOAL_TAC;
  EXPAND_TAC "f";
  REDUCE_TAC;
  DISCH_THEN IMATCH_MP_TAC ;
  TYPE_THEN `Z (f y)` SUBGOAL_TAC;
  EXPAND_TAC "Z";
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let select_image_num_max = prove_by_refinement(
  `!(X:A->bool) f.  (?N. (!x. (X x ==> f x <| N))) /\ ~(X = EMPTY)  ==>
      (?z. (X z /\ (!x. (X x ==> f x <=| f z))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `Y = IMAGE f X` ABBREV_TAC ;
  TYPE_THEN `Y SUBSET {n | n <| N}` SUBGOAL_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[IMAGE;SUBSET;];
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `FINITE Y /\ (~(Y= EMPTY))` SUBGOAL_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `{n | n <| N}` EXISTS_TAC;
  ASM_REWRITE_TAC[FINITE_NUMSEG_LT];
  RULE_ASSUM_TAC  (REWRITE_RULE[EMPTY_EXISTS]);
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  TYPE_THEN `f u` EXISTS_TAC;
  UND 2;
  UND 0;
  REWRITE_TAC[IMAGE;SUBSET];
  DISCH_TAC;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 4 (MATCH_MP   select_num_max);
  REP_BASIC_TAC;
  TYPE_THEN `?r. X r /\ (f r = z)` SUBGOAL_TAC;
  UND 5;
  EXPAND_TAC "Y";
  REWRITE_TAC[IMAGE;SUBSET];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `r` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TSPEC `f x` 4;
  TYPE_THEN `Y (f x)` SUBGOAL_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  (* Wed Aug  4 16:41:51 EDT 2004 *)

  ]);;
  (* }}} *)

let select_image_num_min = prove_by_refinement(
  `!(X:A->bool) f. (~(X = EMPTY)) ==>
     (?z. (X z  /\ (!x. (X x ==> f z <=| f x))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `Y = IMAGE f X` ABBREV_TAC ;
  RULE_ASSUM_TAC  (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  TYPE_THEN `(?n. Y n)` SUBGOAL_TAC;
  TYPE_THEN `f u` EXISTS_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_MESON_TAC[];
  DISCH_TAC;
  RULE_ASSUM_TAC (ONCE_REWRITE_RULE[num_WOP]);
  REP_BASIC_TAC;
  TYPE_THEN `?z. (X z) /\ (f z = n)` SUBGOAL_TAC;
  UND 3;
  EXPAND_TAC "Y";
  REWRITE_TAC[IMAGE;SUBSET];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `z` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TSPEC `f x` 2;
  IMATCH_MP_TAC  (ARITH_RULE `~(f x <| n) ==> (n <=| f x)`);
  DISCH_ALL_TAC;
  UND 2;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "Y";
  KILL 1;
  ASM_REWRITE_TAC[IMAGE;SUBSET];
   ASM_MESON_TAC[];
  (* Wed Aug  4 19:37:29 EDT 2004 *)

  ]);;
  (* }}} *)

let select_card_max = prove_by_refinement(
  `!(X:(A->bool)->bool).  (~(X = EMPTY) /\ (FINITE (UNIONS X))) ==>
    (?z. (X z /\ (!x. (X x ==> (CARD x <= CARD z)))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  select_image_num_max;
  ASM_REWRITE_TAC[];
  TYPE_THEN `SUC (CARD (UNIONS X))` EXISTS_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `x SUBSET (UNIONS X)` SUBGOAL_TAC;
  IMATCH_MP_TAC  sub_union;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
   REWRITE_TAC[ARITH_RULE `(a <| SUC b) <=> (a <=| b)`];
  IMATCH_MP_TAC  CARD_SUBSET;
  ASM_REWRITE_TAC[];
  (* Thu Aug  5 10:50:37 EDT 2004 *)

  ]);;
  (* }}} *)

let select_card_min = prove_by_refinement(
  `!(X:(A->bool)->bool).  ~(X = EMPTY) ==>
    (?z. (X z /\ (!x. (X x ==> (CARD z <= CARD x)))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  select_image_num_min;
  ASM_REWRITE_TAC[];
  (* Thu Aug  5 10:52:02 EDT 2004 *)
  ]);;
  (* }}} *)

(* D embeddings of planar graphs *)

let induced_top_interval = prove_by_refinement(
  `!a b. induced_top (top_of_metric(UNIV,d_real))
       {x | a <= x /\ x <= b } =
     top_of_metric ({x | a <= x /\ x <= b}, d_real)
      `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  top_of_metric_induced;
  ASM_REWRITE_TAC[SUBSET_UNIV;metric_real];
  ]);;
  (* }}} *)

let continuous_interval = prove_by_refinement(
  `!f a b. (continuous f (top_of_metric(UNIV,d_real)) top2) ==>
     (continuous f (top_of_metric({x | a <= x /\ x <= b},d_real)) top2)`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  REWRITE_TAC[GSYM induced_top_interval];
  IMATCH_MP_TAC  continuous_induced_domain;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_real;SUBSET_UNIV ];
  ]);;
  (* }}} *)

let inj_image_subset  = prove_by_refinement(
  `!(f:A->B) X Y. (INJ f X Y ==> IMAGE f X SUBSET Y)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ;IMAGE;SUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let subset_contain = prove_by_refinement(
  `!a b c d. (c <= a) /\ (b <= d) ==>
        {x | a <= x /\ x <= b} SUBSET {x | c <= x /\ x <= d}`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let curve_restriction = prove_by_refinement(
  `!C K K' a b.
       simple_arc top2 C /\
       closed_ top2 K /\ closed_ top2 K' /\
       (C INTER K INTER K' = EMPTY) /\
       ~(C INTER K = EMPTY) /\
       ~(C INTER K' = EMPTY) /\
        (a <. b) ==>
       (?C' f. (C' = IMAGE f {x | a <= x /\ x <= b}) /\ (C' SUBSET C) /\
            continuous f (top_of_metric(UNIV,d_real)) top2 /\
            INJ f {x | a <= x /\ x <= b} (euclid 2) /\
            (C' INTER K = {(f a)}) /\
            (C' INTER K' = {(f b)})
       )
       `,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc];
  REP_BASIC_TAC;
  ASSUME_TAC top2_unions;
  (* K parameter *)
  TYPE_THEN `?t. (&0 <= t /\ t <= &1) /\ (K (f t)) /\ (!s. (&0 <=s /\ s < t) ==> ~(K (f s)))` SUBGOAL_TAC;
  ASSUME_TAC preimage_first ;
  TYPEL_THEN [`K`;`2`] (USE 10 o ISPECL);
  FIRST_ASSUM (fun t -> IMATCH_MP_TAC  t);
  KILL 10;
  ASM_REWRITE_TAC[GSYM top2;];
  ASM_SIMP_TAC[continuous_interval];
  UND 2;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  REWR 6;
  IMATCH_MP_TAC  inj_image_subset;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* K' parameter *)
  TYPE_THEN `?t. (&0 <= t /\ t <= &1) /\ (K' (f t)) /\ (!s. (&0 <=s /\ s < t) ==> ~(K' (f s)))` SUBGOAL_TAC;
  ASSUME_TAC preimage_first ;
  TYPEL_THEN [`K'`;`2`] (USE 14 o ISPECL);
  FIRST_ASSUM (fun t -> IMATCH_MP_TAC  t);
  KILL 14;
  ASM_REWRITE_TAC[GSYM top2;];
  ASM_SIMP_TAC[continuous_interval];
  UND 1;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  REWR 6;
  IMATCH_MP_TAC  inj_image_subset;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `(t < t' \/ t' < t)` SUBGOAL_TAC;
  REWRITE_TAC[(REAL_ARITH `(t < t' \/ t' < t) <=> ~( t = t')`)];
  DISCH_ALL_TAC;
  UND 3;
  REWRITE_TAC[EMPTY_EXISTS;INTER;];
  TYPE_THEN `(f t)` EXISTS_TAC;
  REWR 11;
  REWRITE_TAC[IMAGE;SUBSET];
  CONJ_TAC;
  TYPE_THEN `t'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  (* main cases split [main] *)
  ASSUME_TAC (REAL_ARITH `&0 < &1`);
  DISCH_THEN (DISJ_CASES_TAC);
  TYPE_THEN `continuous f (top_of_metric (UNIV,d_real)) (top2) /\  INJ f {x | t <= x /\ x <= t'} (euclid 2) /\ (&0 < &1) /\ (t < t')  ` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x <= &1}` EXISTS_TAC;
  REWR 6;
  ASM_REWRITE_TAC[SUBSET ];
   UND 19;
  UND 16;
  UND 13;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t -> ASSUME_TAC (MATCH_MP arc_reparameter_rev t));
  REP_BASIC_TAC;
  TYPE_THEN `Ca = IMAGE g {x | &0 <= x /\ x <= &1}` ABBREV_TAC ;
  TYPE_THEN `Ca INTER K' = {(g (&0))}` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[INTER;SUBSET;INR IN_SING;];
  KILL 26;
  EXPAND_TAC "Ca";
  REWRITE_TAC[IMAGE;SUBSET];
  REP_BASIC_TAC;
  TYPE_THEN `x' < t' \/ (x' = t')` SUBGOAL_TAC;
  UND 28;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  PROOF_BY_CONTR_TAC;
  UND 26;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 29;
  UND 13;
  REAL_ARITH_TAC;
  ASM_MESON_TAC[];
  REWRITE_TAC[SUBSET;INTER;INR IN_SING;];
  KILL 26;
  EXPAND_TAC "Ca";
  REWRITE_TAC[IMAGE;SUBSET];
  NAME_CONFLICT_TAC;
  REP_BASIC_TAC;
  CONJ_TAC;
  TYPE_THEN `t'` EXISTS_TAC;
  ASM_MESON_TAC[REAL_ARITH `(t < t' ==> t<= t') /\ (t' <= t')`];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `~(Ca INTER K = EMPTY)` SUBGOAL_TAC;
  REWRITE_TAC[INTER;EMPTY_EXISTS];
  TYPE_THEN `f t` EXISTS_TAC;
  KILL 26;
  EXPAND_TAC "Ca";
  REWRITE_TAC[IMAGE;SUBSET;];
  ASM_REWRITE_TAC[];
  TYPE_THEN `t` EXISTS_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `t <= t`];
  ASM_SIMP_TAC[REAL_ARITH `(t < t') ==> (t <= t')`];
  DISCH_TAC;
  KILL 21;
  (* ADD Ca SUBSET C *)
  TYPE_THEN `Ca SUBSET C` SUBGOAL_TAC;
  KILL 26;
  EXPAND_TAC "Ca";
  KILL 20;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE;SUBSET];
  NAME_CONFLICT_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 21;
  UND 26;
  UND 13;
  UND 19;
  UND 16;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* t'' parameter for g and K *)
  TYPE_THEN `?t''. (&0 <= t'' /\ t'' <= &1) /\ (K (g t'')) /\ (!s. (&0 <=s /\ s < t'') ==> ~(K (g s)))` SUBGOAL_TAC;
  ASSUME_TAC preimage_first ;
  TYPEL_THEN [`K`;`2`] (USE 29 o ISPECL);
  FIRST_ASSUM (fun t -> IMATCH_MP_TAC  t);
  KILL 29;
  ASM_REWRITE_TAC[GSYM top2;];
  ASM_SIMP_TAC[continuous_interval];
  EXPAND_TAC "Ca";
  IMATCH_MP_TAC  inj_image_subset;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* set up for arc_reparameter_rev *)
  TYPE_THEN `continuous g (top_of_metric (UNIV,d_real)) (top2) /\  INJ g {x | &0 <= x /\ x <= t''} (euclid 2) /\ (a < b) /\ (&0 < t'')  ` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `&0 < t'' \/ (t'' = &0)` SUBGOAL_TAC;
  UND 32;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x <= &1}` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET ];
  UND 31;
  REAL_ARITH_TAC;
  PROOF_BY_CONTR_TAC;
  UND 3;
  REWRITE_TAC[EMPTY_EXISTS;INTER;];
  TYPE_THEN `g (&0)` EXISTS_TAC;
  TYPE_THEN `Ca (g (&0))` SUBGOAL_TAC;
  TYPE_THEN `{(g (&0))} SUBSET Ca` SUBGOAL_TAC;
  ASM_MESON_TAC[INTER_SUBSET];
  REWRITE_TAC[SUBSET;INR IN_SING];
  MESON_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  UND 3;
  UND 21;
  MESON_TAC[ISUBSET];
  REWR 30;
  ASM_REWRITE_TAC[];
  UND 15;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  FIRST_ASSUM (fun t -> ASSUME_TAC (MATCH_MP arc_reparameter_rev t));
  REP_BASIC_TAC;
  TYPE_THEN `C' =IMAGE g' {x | a <= x /\ x <= b}` ABBREV_TAC ;
  (* now finally go after the goal in the FIRST case *)
  TYPE_THEN `C'` EXISTS_TAC;
  TYPE_THEN `g'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* now finish off the three conditions *)
  KILL 34;
  TYPE_THEN `C' SUBSET Ca` SUBGOAL_TAC;
  KILL 43;
  EXPAND_TAC "C'";
  EXPAND_TAC "Ca";
  IMATCH_MP_TAC  IMAGE_SUBSET;
  IMATCH_MP_TAC subset_contain;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  DISCH_TAC;
  CONJ_TAC; (* 1*)
  ASM_REWRITE_TAC[];
  USE 8 GSYM;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `Ca` EXISTS_TAC ;
  ASM_MESON_TAC[];
  CONJ_TAC; (* 2 *)
  KILL 43;
  EXPAND_TAC "C'";
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[INTER;IMAGE;SUBSET];
  NAME_CONFLICT_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[INR IN_SING];
  TYPE_THEN `(x' < t'') \/ (x' = t'')` SUBGOAL_TAC;
  UND 45;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  TSPEC `x'` 14;
  UND 43;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  REWRITE_TAC[SUBSET;IMAGE;INTER;IN_SING];
  NAME_CONFLICT_TAC;
  REP_BASIC_TAC;
  CONJ_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `t''` EXISTS_TAC;
  ASM_MESON_TAC[REAL_ARITH `t'' <= t''`];
  ASM_MESON_TAC[];
  (* 3 *)
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `Ca INTER K'` EXISTS_TAC;
  CONJ_TAC;
  UND 34;
  REWRITE_TAC[SUBSET;INTER];
  MESON_TAC[];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET;INR IN_SING];
  REWRITE_TAC[SUBSET;INTER;INR IN_SING ];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  EXPAND_TAC "C'";
  REWRITE_TAC[IMAGE;SUBSET];
  TYPE_THEN `b` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 40;
  REAL_ARITH_TAC;
  ASM_MESON_TAC[];
  (* sh *)
  (*  *******************  START THE SECOND HALF ************  *)

  TYPE_THEN `continuous f (top_of_metric (UNIV,d_real)) (top2) /\  INJ f {x | t' <= x /\ x <= t} (euclid 2) /\ (&0 < &1) /\ (t' < t)  ` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x <= &1}` EXISTS_TAC;
  REWR 6;
  ASM_REWRITE_TAC[SUBSET ];
   UND 19;
  UND 12;
  UND 17;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t -> ASSUME_TAC (MATCH_MP arc_reparameter_rev t));
  REP_BASIC_TAC;
  TYPE_THEN `Ca = IMAGE g {x | &0 <= x /\ x <= &1}` ABBREV_TAC ;
  TYPE_THEN `Ca INTER K = {(g (&0))}` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[INTER;SUBSET;INR IN_SING;];
  KILL 26;
  EXPAND_TAC "Ca";
  REWRITE_TAC[IMAGE;SUBSET];
  REP_BASIC_TAC;
  TYPE_THEN `x' < t \/ (x' = t)` SUBGOAL_TAC;
  UND 28;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  PROOF_BY_CONTR_TAC;
  UND 26;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 29;
  UND 17;
  REAL_ARITH_TAC;
  ASM_MESON_TAC[];
  REWRITE_TAC[SUBSET;INTER;INR IN_SING;];
  KILL 26;
  EXPAND_TAC "Ca";
  REWRITE_TAC[IMAGE;SUBSET];
  NAME_CONFLICT_TAC;
  REP_BASIC_TAC;
  CONJ_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  ASM_MESON_TAC[REAL_ARITH `(t' < t ==> t'<= t) /\ (t <= t)`];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `~(Ca INTER K' = EMPTY)` SUBGOAL_TAC;
  REWRITE_TAC[INTER;EMPTY_EXISTS];
  TYPE_THEN `f t'` EXISTS_TAC;
  KILL 26;
  EXPAND_TAC "Ca";
  REWRITE_TAC[IMAGE;SUBSET;];
  ASM_REWRITE_TAC[];
  TYPE_THEN `t'` EXISTS_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `t' <= t'`];
  ASM_SIMP_TAC[REAL_ARITH `(t' < t) ==> (t' <= t)`];
  DISCH_TAC;
  KILL 21;
  (* ADD Ca SUBSET C *)
  TYPE_THEN `Ca SUBSET C` SUBGOAL_TAC;
  KILL 26;
  EXPAND_TAC "Ca";
  KILL 20;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE;SUBSET];
  NAME_CONFLICT_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 21;
  UND 26;
  UND 17;
  UND 19;
  UND 12;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* gK *)
  (* t'' parameter for g and K *)
  TYPE_THEN `?t''. (&0 <= t'' /\ t'' <= &1) /\ (K' (g t'')) /\ (!s. (&0 <=s /\ s < t'') ==> ~(K' (g s)))` SUBGOAL_TAC;
  ASSUME_TAC preimage_first ;
  TYPEL_THEN [`K'`;`2`] (USE 29 o ISPECL);
  FIRST_ASSUM (fun t -> IMATCH_MP_TAC  t);
  KILL 29;
  ASM_REWRITE_TAC[GSYM top2;];
  ASM_SIMP_TAC[continuous_interval];
  EXPAND_TAC "Ca";
  IMATCH_MP_TAC  inj_image_subset;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* set up for arc_reparameter_gen *)
  TYPE_THEN `continuous g (top_of_metric (UNIV,d_real)) (top2) /\  INJ g {x | &0 <= x /\ x <= t''} (euclid 2) /\ (a < b) /\ (&0 < t'')  ` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `&0 < t'' \/ (t'' = &0)` SUBGOAL_TAC;
  UND 32;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x <= &1}` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET ];
  UND 31;
  REAL_ARITH_TAC;
  PROOF_BY_CONTR_TAC;
  UND 3;
  REWRITE_TAC[EMPTY_EXISTS;INTER;];
  TYPE_THEN `g (&0)` EXISTS_TAC;
  TYPE_THEN `Ca (g (&0))` SUBGOAL_TAC;
  TYPE_THEN `{(g (&0))} SUBSET Ca` SUBGOAL_TAC;
  ASM_MESON_TAC[INTER_SUBSET];
  REWRITE_TAC[SUBSET;INR IN_SING];
  MESON_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  UND 3;
  UND 21;
  MESON_TAC[ISUBSET];
  REWR 30;
  ASM_REWRITE_TAC[];
  UND 11;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  FIRST_ASSUM (fun t -> ASSUME_TAC (MATCH_MP arc_reparameter_gen t));
  REP_BASIC_TAC;
  TYPE_THEN `C' =IMAGE g' {x | a <= x /\ x <= b}` ABBREV_TAC ;
  (* now finally go after the goal in the FIRST case *)
  TYPE_THEN `C'` EXISTS_TAC;
  TYPE_THEN `g'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* nfo *)
  (* now finish off the three conditions *)
  KILL 34;
  TYPE_THEN `C' SUBSET Ca` SUBGOAL_TAC;
  KILL 43;
  EXPAND_TAC "C'";
  EXPAND_TAC "Ca";
  IMATCH_MP_TAC  IMAGE_SUBSET;
  IMATCH_MP_TAC subset_contain;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  DISCH_TAC;
  CONJ_TAC; (* 1*)
  ASM_REWRITE_TAC[];
  USE 8 GSYM;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `Ca` EXISTS_TAC ;
  ASM_MESON_TAC[];
  (* s2 *)
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  CONJ_TAC ; (* 2 *)
  KILL 43;
  EXPAND_TAC "C'";
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[INTER;IMAGE;SUBSET];
  NAME_CONFLICT_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[INR IN_SING];
  TYPE_THEN `(x' < t'') \/ (x' = t'')` SUBGOAL_TAC;
  UND 45;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  TSPEC `x'` 14;
  UND 43;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  REWRITE_TAC[SUBSET;IMAGE;INTER;IN_SING];
  NAME_CONFLICT_TAC;
  REP_BASIC_TAC;
  CONJ_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `t''` EXISTS_TAC;
  ASM_MESON_TAC[REAL_ARITH `t'' <= t''`];
  ASM_MESON_TAC[];
  (* s3 *)
  (* 3 *)
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `Ca INTER K` EXISTS_TAC;
  CONJ_TAC;
  UND 34;
  REWRITE_TAC[SUBSET;INTER];
  MESON_TAC[];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET;INR IN_SING];
  REWRITE_TAC[SUBSET;INTER;INR IN_SING ];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  EXPAND_TAC "C'";
  REWRITE_TAC[IMAGE;SUBSET];
  TYPE_THEN `a` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 40;
  REAL_ARITH_TAC;
  ASM_MESON_TAC[];
  (* Thu Aug  5 08:09:38 EDT 2004  *)

  ]);;
  (* }}} *)

let simple_arc_end = jordan_def
  `simple_arc_end C v v' <=>
    (?f. (C = IMAGE f {x | &0 <= x /\ x <= &1 }) /\
       continuous f (top_of_metric(UNIV,d_real)) top2 /\
       INJ f {x | &0 <= x /\ x <= &1} (euclid 2) /\
       (f (&0) = v) /\ (f(&1) = v'))`;;

let good_plane_graph = jordan_def
   `good_plane_graph G <=> plane_graph G /\
      (!e v v'. (graph_edge G e /\ ~(v = v') /\
           (graph_inc G e v) /\ (graph_inc G e v') ==>
           (simple_arc_end e v v')))`;;

let graph_edge_mod  = jordan_def
  `graph_edge_mod (G:(A,B)graph_t) (f:B->B') =
     mk_graph_t (graph_vertex G,IMAGE f (graph_edge G),
       (\ e' v. (?e. graph_edge G e /\ graph_inc G e v /\ (f e = e'))))`;;

let graph_edge_mod_v = prove_by_refinement(
  `!(G:(A,B)graph_t) (f:B->B').
     graph_vertex (graph_edge_mod G f) = graph_vertex G `,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_edge_mod;graph_vertex;dest_graph_t;];
  ]);;
  (* }}} *)

let graph_edge_mod_e = prove_by_refinement(
  `!(G:(A,B)graph_t) (f:B->B').
     graph_edge (graph_edge_mod G f) = IMAGE f (graph_edge G )`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_edge_mod;graph_edge;dest_graph_t;part1;drop0];
  ]);;
  (* }}} *)

let graph_edge_mod_i = prove_by_refinement(
  `!(G:(A,B)graph_t) (f:B->B') e v.
     graph_inc (graph_edge_mod G f) e v <=>
         (?e'. (graph_edge G e' /\ graph_inc G e' v /\ (f e' = e)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_edge_mod;graph_inc;dest_graph_t;part1;drop1];
  ]);;
  (* }}} *)

let inj_bij = prove_by_refinement(
  `!(f:A->B) X. (INJ f X UNIV) ==> (BIJ f X (IMAGE f X))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[BIJ];
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE_SURJ];
  UND 0;
  REWRITE_TAC[INJ;IMAGE;SUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let graph_edge_iso = prove_by_refinement(
  `! f (G:(A,B)graph_t). (INJ (f:B->B') (graph_edge G) (UNIV)) ==>
    (graph_isomorphic G (graph_edge_mod G f))`,
  (* {{{ proof *)

  [
  REWRITE_TAC[graph_isomorphic;graph_iso];
  REP_BASIC_TAC;
  RIGHT_TAC "f";
  RIGHT_TAC "f";
  TYPE_THEN `I:A->A` EXISTS_TAC ;
  TYPE_THEN `f` EXISTS_TAC;
  NAME_CONFLICT_TAC;
  EXISTS_TAC `(I:A->A,f:B->B')` ;
  REWRITE_TAC[graph_edge_mod_v;graph_edge_mod_e];
  CONJ_TAC;
  REWRITE_TAC[I_DEF;BIJ;INJ;SURJ;];
  MESON_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  inj_bij;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[graph_edge_mod_i;IMAGE_I;];
  EQ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `e'' = e'` SUBGOAL_TAC;
  RULE_ASSUM_TAC(REWRITE_RULE  [INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let graph_edge_graph = prove_by_refinement(
  `!f (G:(A,B)graph_t). (graph G) /\
      (INJ (f:B->B') (graph_edge G) (UNIV)) ==>
    (graph (graph_edge_mod G f)) `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC    graph_isomorphic_graph;
  TYPE_THEN `G` EXISTS_TAC;
  ASM_MESON_TAC[graph_edge_iso];
  ]);;
  (* }}} *)

let plane_graph_mod = prove_by_refinement(
  `!G f. (plane_graph G) /\ (INJ f (graph_edge G) UNIV) /\
      (!e e'. (graph_edge G e /\ graph_edge G e' /\ ~(e = e') ==>
        (f e INTER f e' SUBSET e INTER e') )) /\
      (!e. (graph_edge G e ==> (simple_arc top2 (f e)))) /\
      (!e. (graph_edge G e) ==>
         (e INTER graph_vertex G = (f e) INTER graph_vertex G)) ==>
      (plane_graph (graph_edge_mod G f))
  `,
  (* {{{ proof *)

  [
  REWRITE_TAC[plane_graph];
  REP_BASIC_TAC;
  REWRITE_TAC[graph_edge_mod_v;graph_edge_mod_e;];
  CONJ_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  ASM_MESON_TAC[graph_edge_graph];
  CONJ_TAC;
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC[IMAGE;SUBSET];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER];
  REP_BASIC_TAC;
  REWRITE_TAC[graph_edge_mod_i];
  EQ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `e' = x` SUBGOAL_TAC;
   RULE_ASSUM_TAC  (REWRITE_RULE[INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TSPEC `e'` 5;
  TSPEC `e'` 0;
  UND 0;
  UND 5;
  ASM_REWRITE_TAC[];
  DISCH_ALL_TAC;
  TYPE_THEN `(f x INTER graph_vertex G) x'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REWRITE_TAC[INTER;SUBSET];
  REP_BASIC_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TSPEC `x` 5;
  TSPEC `x` 0;
  UND 0;
  REWR 5;
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]);
  ASM_SIMP_TAC[];
  REWRITE_TAC[INTER;SUBSET];
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  UND 10;
  REWRITE_TAC[IMAGE];
  REP_BASIC_TAC;
  UND 11;
  REWRITE_TAC[IMAGE];
  REP_BASIC_TAC;
  TYPE_THEN `~(x = x')` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `x' INTER x` EXISTS_TAC;
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* Thu Aug  5 10:17:38 EDT 2004 *)

  ]);;

  (* }}} *)

let compact_point = prove_by_refinement(
  `!U (x:A). (UNIONS U x) ==> (compact U {x})`,
  (* {{{ proof *)
  [
  REWRITE_TAC[compact];
  REP_BASIC_TAC;
  CONJ_TAC;
  ASM_REWRITE_TAC [single_subset];
  REP_BASIC_TAC;
  TYPE_THEN `?u. V u /\ u x` SUBGOAL_TAC;
  UND 2;
  REWRITE_TAC[SUBSET;UNIONS;INR IN_SING];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `{u}` EXISTS_TAC;
  ASM_REWRITE_TAC [single_subset;FINITE_SING];
  (* Thu Aug  5 12:02:40 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_arc_end_select = prove_by_refinement(
  `!C v v'. (simple_arc top2 C) /\ (C v) /\ (C v') /\ ~(v = v') ==>
    (?C'. (C' SUBSET C) /\ (simple_arc_end C' v v'))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc_end];
  REP_BASIC_TAC;
  (* A *)
  TYPE_THEN `!v. (C v) ==> (closed_ top2 {v})` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  compact_closed;
  ASM_SIMP_TAC[top2_top;metric_hausdorff;top2;metric_euclid;compact_point];
  IMATCH_MP_TAC  compact_point;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_euclid];
  UND 3;
  REWRITE_TAC[simple_arc];
  REP_BASIC_TAC;
  TYPE_THEN `C SUBSET euclid 2` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  inj_image_subset;
  RULE_ASSUM_TAC (REWRITE_RULE [top2_unions]);
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  (* B hypotheses of curve_restriction *)
  TYPE_THEN `simple_arc top2 C /\ closed_ top2 {v} /\ closed_ top2 {v'} /\      (C INTER {v} INTER { v' } = EMPTY) /\ ~(C INTER {v} = EMPTY) /\       ~(C INTER {v'} = EMPTY) /\        (&0 < &1)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC ;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC ;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[REAL_ARITH `&0 < &1`];
  REWRITE_TAC[INTER;INR IN_SING;EMPTY_EXISTS ];
  REWRITE_TAC[EQ_EMPTY];
  ASM_MESON_TAC[];
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP curve_restriction t));
  REP_BASIC_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `!A u v. (A INTER {u} = {v}) ==> ( (v:num->real)=u)` SUBGOAL_TAC;
  REWRITE_TAC[eq_sing;INTER;INR IN_SING;];
  MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let graph_edge2 = prove_by_refinement(
  `!(G:(A,B)graph_t) e.
      (graph G /\ graph_edge G e) ==> (graph_inc G e HAS_SIZE 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph];
  REWRITE_TAC[IMAGE;SUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let simple_arc_end_symm = prove_by_refinement(
  `!C' v v'. (simple_arc_end C' v v' ==> simple_arc_end C' v' v)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc_end];
  REP_BASIC_TAC;
  TYPE_THEN `( continuous f (top_of_metric (UNIV,d_real)) (top2) /\ INJ f {x | &0 <= x /\ x <= &1} (euclid 2) /\ (&0 < &1) /\ (&0 < &1))` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_ARITH `&0 < &1`];
  DISCH_THEN (fun t -> ASSUME_TAC (MATCH_MP arc_reparameter_rev t));
  REP_BASIC_TAC;
  TYPE_THEN `g` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let simple_arc_end_plane_select = prove_by_refinement(
  `!G e. (plane_graph G /\ graph_edge G e) ==> (?e'.
     (e' SUBSET e /\
     (!v v'. graph_inc G e v /\ graph_inc G e v' /\ ~(v = v') ==>
        simple_arc_end e' v v')))`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G e HAS_SIZE 2` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE [plane_graph]);
  IMATCH_MP_TAC graph_edge2;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[has_size2];
  REP_BASIC_TAC;
  TYPE_THEN `(?e'. (e' SUBSET e) /\ (simple_arc_end e' a b))` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_select;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC  (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  CONJ_TAC;
  UND 5;
  ASM_MESON_TAC [ISUBSET];
  TYPE_THEN `graph_inc G e a /\ graph_inc G e b` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[in_pair];
  KILL 3;
  ASM_SIMP_TAC[];
  REWRITE_TAC[INTER;SUBSET];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `e'` EXISTS_TAC;
  ASM_REWRITE_TAC[in_pair];
  REP_BASIC_TAC;
  TYPE_THEN `((v = a) /\ (v' = b)) \/ ((v = b) /\ (v' =a ))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  (* Thu Aug  5 14:10:17 EDT 2004 *)

  ]);;

  (* }}} *)

let plane_graph_contain = prove_by_refinement(
  `!G e e'. (plane_graph G /\ graph_edge G e /\ graph_edge G e' /\
      (e SUBSET e') ==> (e = e'))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `e INTER e' SUBSET graph_vertex G` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `e INTER e' SUBSET e' INTER graph_vertex G` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET_INTER];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INTER;SUBSET];
  MESON_TAC[];
  TYPE_THEN `e' INTER graph_vertex G = graph_inc G e'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `graph_inc G e' HAS_SIZE 2` SUBGOAL_TAC;
  ASM_MESON_TAC[graph_edge2];
  TYPE_THEN `e INTER e' = e` SUBGOAL_TAC;
  UND 0;
  REWRITE_TAC[SUBSET_INTER_ABSORPTION];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[has_size2];
  REP_BASIC_TAC;
  REWR 10;
  TYPE_THEN `simple_arc top2 e` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  REWRITE_TAC[simple_arc];
  REP_BASIC_TAC;
  TYPE_THEN `!x. (&0 <= x /\ x <= &1) ==> {a,b} (f x)` SUBGOAL_TAC;
  REWR 10;
  UND 10;
  REWRITE_TAC[IMAGE;SUBSET];
  MESON_TAC[];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  TYPE_THEN `(f (&0) = f(&1))` SUBGOAL_TAC;
  IMATCH_MP_TAC  two_exclusion;
  TYPE_THEN `{a,b}` EXISTS_TAC;
  TYPE_THEN `?t. (&0 < t /\ t < &1)` SUBGOAL_TAC;
  TYPE_THEN `&1/ (&2)` EXISTS_TAC;
  IMATCH_MP_TAC  half_pos;
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `f t` EXISTS_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[pair_size_2];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  REAL_ARITH_TAC;
  CONJ_TAC;
  FIRST_ASSUM  IMATCH_MP_TAC ;
  REAL_ARITH_TAC;
  CONJ_TAC;
  FIRST_ASSUM  IMATCH_MP_TAC ;
  UND 18;
  UND 19;
  REAL_ARITH_TAC;
  CONJ_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `~(&0 = t)` SUBGOAL_TAC;
  UND 19;
  REAL_ARITH_TAC;
  REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWR 20;
  ASM_REWRITE_TAC[];
  UND 18;
  UND 19;
  REAL_ARITH_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `~(&1 = t)` SUBGOAL_TAC;
  UND 18;
  REAL_ARITH_TAC;
  REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWR 20;
  ASM_REWRITE_TAC[];
  UND 18;
  UND 19;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `~(&0 = &1)` SUBGOAL_TAC;
  REAL_ARITH_TAC;
  REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  (* Thu Aug  5 15:11:20 EDT 2004 *)

  ]);;
  (* }}} *)

let graph_edge_end_select = prove_by_refinement(
  `!(G:(A,B)graph_t) e. (graph G /\ graph_edge G e ==>
     (?v v'. graph_inc G e v /\ graph_inc G e v' /\ ~(v = v')))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G e HAS_SIZE 2` SUBGOAL_TAC;
  IMATCH_MP_TAC  graph_edge2;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[has_size2];
  REP_BASIC_TAC;
  TYPE_THEN `a` EXISTS_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  ASM_REWRITE_TAC[in_pair];
  (* Thu Aug  5 19:26:02 EDT 2004 *)

  ]);;
  (* }}} *)


(* ------------------------------------------------------------------ *)
(* SECTION K *)
(* ------------------------------------------------------------------ *)

(* Thu Aug  5 21:17:36 EDT 2004 *)

let inf = jordan_def `inf (X:real->bool) =
   @s. ((!x. X x ==> s <= x) /\ (!y. (!x. X x ==> y <= x) ==> (y <= s)))`;;
let interval_closed = prove_by_refinement(
  `!a b. closed_ (top_of_metric(UNIV,d_real)) {x | a <= x /\ x <= b}`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  compact_closed;
  ASM_SIMP_TAC[interval_compact;top_of_metric_top;metric_real];
  ASM_SIMP_TAC[metric_hausdorff;metric_real;];
  ]);;
  (* }}} *)

let half_closed = prove_by_refinement(
  `!a. closed_ (top_of_metric(UNIV,d_real)) {x | x <= a}`,
  (* {{{ proof *)
  [
  REWRITE_TAC[closed];
  REP_BASIC_TAC;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_real];
  TYPE_THEN `UNIV DIFF {x | x <= a } = {x | a < x}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[DIFF;UNIV];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC [open_DEF;half_open_above];
  ]);;
  (* }}} *)

let half_closed_above = prove_by_refinement(
  `!a. closed_ (top_of_metric(UNIV,d_real)) {x | a <= x}`,
  (* {{{ proof *)
  [
  REWRITE_TAC[closed];
  REP_BASIC_TAC;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_real];
  TYPE_THEN `UNIV DIFF {x | a <= x } = {x | x < a}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[DIFF;UNIV];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC [open_DEF;half_open];
  ]);;
  (* }}} *)

let inf_LB = prove_by_refinement(
  `!X. (~(X = EMPTY) /\ (?t. !x. (X x ==> t <= x))) ==>
     (!x. X x ==> inf X <= x) /\
          (!y. (!x. X x ==> y <= x) ==> (y <= inf X))`,
  (* {{{ proof *)
  [
  GEN_TAC;
  TYPE_THEN `topology_ (top_of_metric(UNIV,d_real))` SUBGOAL_TAC;
  ASM_SIMP_TAC[top_of_metric_top;metric_real];
  DISCH_TAC;
  (*  *)
  TYPE_THEN `X SUBSET closure (top_of_metric(UNIV,d_real)) X` SUBGOAL_TAC;
  ASM_SIMP_TAC[subset_closure];
  DISCH_TAC;
  (*  *)
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  REWRITE_TAC[inf];
  SELECT_TAC;
  ASM_MESON_TAC[];
  PROOF_BY_CONTR_TAC;
  UND 4;
  KILL 5;
  REWRITE_TAC[];
  TYPE_THEN `XC = closure (top_of_metric(UNIV,d_real)) X INTER {x | t <= x /\ x <= u}` ABBREV_TAC ;
  TYPE_THEN `compact (top_of_metric(UNIV,d_real)) XC` SUBGOAL_TAC;
  IMATCH_MP_TAC  closed_compact;
  TYPE_THEN `{x | t <= x /\ x <= u}` EXISTS_TAC;
  ASM_SIMP_TAC[interval_compact;top_of_metric_top;metric_real];
  EXPAND_TAC "XC";
  CONJ_TAC;
  IMATCH_MP_TAC  closed_inter2;
  ASM_SIMP_TAC[interval_closed;top_of_metric_top;metric_real];
  IMATCH_MP_TAC  closure_closed;
  ASM_SIMP_TAC[top_of_metric_top;metric_real;GSYM top_of_metric_unions;];
  ASM_REWRITE_TAC[INTER_SUBSET];
  DISCH_TAC;
  (*   *)
  TYPE_THEN `(?z. (XC z /\ (!y. XC y ==> z <= y)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  compact_inf;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `u` EXISTS_TAC;
  EXPAND_TAC "XC";
  REWRITE_TAC[INTER;SUBSET];
  CONJ_TAC;
  UND 1;
  REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[REAL_ARITH `u <= u`];
  REP_BASIC_TAC;
  TYPE_THEN `z` EXISTS_TAC;
  CONJ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `(x <= u) \/ (u < x)` SUBGOAL_TAC;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `XC x` SUBGOAL_TAC;
  EXPAND_TAC "XC";
  REWRITE_TAC[INTER;SUBSET];
  CONJ_TAC;
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  UND 7;
  EXPAND_TAC "XC";
  REWRITE_TAC[INTER;SUBSET];
  REP_BASIC_TAC;
  ASM_MESON_TAC[REAL_ARITH `z <= u /\ u < x ==> z <= x`];
  REP_BASIC_TAC;
  TYPE_THEN `closed_ (top_of_metric (UNIV,d_real)) {x | y' <= x }` SUBGOAL_TAC;
  REWRITE_TAC[half_closed_above];
  DISCH_TAC;
  TYPE_THEN `closure (top_of_metric (UNIV,d_real)) X SUBSET {x | y' <= x }` SUBGOAL_TAC;
  IMATCH_MP_TAC  closure_subset;
  ASM_REWRITE_TAC[SUBSET ];
  DISCH_TAC;
  TYPE_THEN `XC SUBSET {x | y' <= x}` SUBGOAL_TAC;
  EXPAND_TAC "XC";
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `closure (top_of_metric (UNIV,d_real)) X ` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "XC";
  REWRITE_TAC[INTER_SUBSET];
  REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[];
  (* Fri Aug  6 05:51:24 EDT 2004 *)

  ]);;
  (* }}} *)

let inf_eps = prove_by_refinement(
  `!X. (~(X = EMPTY) /\ (?t. !x. (X x ==> t <= x))) ==>
       (!epsilon. (&0 < epsilon) ==> (?x. X x /\ (x < inf X + epsilon)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(!y. (!x. X x ==> y <= x) ==> (y <= inf X))` SUBGOAL_TAC;
  ASM_MESON_TAC[inf_LB];
  DISCH_TAC;
  TSPEC `inf X + epsilon` 3;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `(!x. X x ==> inf X + epsilon <= x)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `~(v < u)  ==> u <= v`);
  ASM_MESON_TAC[];
  ASM_MESON_TAC[REAL_ARITH `(x + y <= x ==> ~(&0 < y))`];
  ]);;
  (* }}} *)

let supm = jordan_def `supm (X:real->bool) =
   --. (inf ({x | ?z. X z /\ (x = --. z)}))`;;

let supm_UB = prove_by_refinement(
  `!X. (~(X = EMPTY) /\ (?t. !x. (X x ==> x <= t))) ==>
     (!x. X x ==> x <= supm X ) /\
          (!y. (!x. X x ==> x <= y) ==> (supm X <= y))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[supm];
  TYPE_THEN `Y = {x | ?z. X z /\ (x = --z)}` ABBREV_TAC ;
  TYPE_THEN `!u. (Y u = X (-- u)) /\ (Y (--u ) = X u)` SUBGOAL_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  MESON_TAC[REAL_ARITH `(-- (-- u) = u)`];
  DISCH_TAC;
  TYPE_THEN `(~(Y = EMPTY) /\ (?t. !x. (Y x ==> t <= x)))` SUBGOAL_TAC;
  UND 1;
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  CONJ_TAC;
  TYPE_THEN `-- u` EXISTS_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `-- t` EXISTS_TAC;
  REP_BASIC_TAC;
  ASM_MESON_TAC[REAL_ARITH `--t <= x <=> (-- x <= t)`];
  DISCH_THEN ( ASSUME_TAC o (MATCH_MP inf_LB));
  CONJ_TAC;
  REP_BASIC_TAC;
  ASM_MESON_TAC[REAL_ARITH `y <= --x <=> x <= --y`];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `--y <= inf Y ==> -- inf Y <= y`);
  FIRST_ASSUM IMATCH_MP_TAC ;
  REP_BASIC_TAC;
  ASM_MESON_TAC[ REAL_ARITH `--x <= y <=> --y <= x`];
  (* Fri Aug  6 06:42:14 EDT 2004 *)

  ]);;
  (* }}} *)

let supm_eps = prove_by_refinement(
  `!X. (~(X = EMPTY) /\ (?t. !x. (X x ==> x <= t))) ==>
       (!epsilon.(&0 < epsilon) ==> (?x. X x /\ (supm X - epsilon < x)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(!y. (!x. X x ==> x <= y) ==> (supm X <= y))` SUBGOAL_TAC;
  ASM_MESON_TAC[supm_UB];
  DISCH_TAC;
  TSPEC `supm X - epsilon` 3;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `(!x. X x ==> x <= supm X - epsilon)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `~(v < u)  ==> u <= v`);
  ASM_MESON_TAC[];
  ASM_MESON_TAC[REAL_ARITH `(x <= x - y  ==> ~(&0 < y))`];
  (* Fri Aug  6 06:47:22 EDT 2004 *)

  ]);;
  (* }}} *)

let compact_subset = prove_by_refinement(
  `!(X:A->bool) K d. (K SUBSET X /\ metric_space(X,d)) ==>
        (compact(top_of_metric(X,d)) K = compact(top_of_metric(K,d))K) `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  ASM_SIMP_TAC[GSYM top_of_metric_induced];
  ASM_MESON_TAC[induced_compact;top_of_metric_unions];
  ]);;
  (* }}} *)

let exp_gt1 = prove_by_refinement(
  `!n. (0 < n) ==> (1 < 2 **| n)`,
  (* {{{ proof *)
  [
  TYPE_THEN `1 = 2 **| 0` SUBGOAL_TAC;
  REWRITE_TAC[EXP];
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]);
  REP_BASIC_TAC;
  REWRITE_TAC[LT_EXP];
  UND 0;
  ARITH_TAC;
  ]);;
  (* }}} *)

let twopow_lt = prove_by_refinement(
  `!a b. (a < b) ==> (twopow a < twopow b)`,
  (* {{{ proof *)
  [
  ONCE_REWRITE_TAC [INT_ARITH `(a <: b) <=> (&:0 <: b -: a)`];
  ASSUME_TAC twopow_pos;
  ONCE_REWRITE_TAC [REAL_ARITH `x < y <=> &1*x < y`];
  ASM_SIMP_TAC[GSYM REAL_LT_RDIV_EQ];
  REWRITE_TAC[real_div];
  REWRITE_TAC[GSYM TWOPOW_INV;GSYM TWOPOW_ADD_INT;GSYM INT_SUB];
  REP_GEN_TAC;
  TYPE_THEN `C = b -: a` ABBREV_TAC ;
  ASSUME_TAC INT_REP2 ;
  TSPEC `C` 2;
  REP_BASIC_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  UND 2;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[TWOPOW_POS];
  REDUCE_TAC;
  REWRITE_TAC[INT_OF_NUM_LT;exp_gt1];
  PROOF_BY_CONTR_TAC;
  UND 2;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INT_ARITH `(~(&:0 <: --: y) <=> (&:0 <=: y))`];
  REWRITE_TAC[INT_OF_NUM_LE];
  ARITH_TAC;
  ]);;
  (* }}} *)

let compact_distance = prove_by_refinement(
  `!(X:A->bool) d K K'. (metric_space(X,d) /\
   ~(K=EMPTY) /\ ~(K' = EMPTY) /\
   (compact (top_of_metric(X,d)) K) /\ (compact(top_of_metric(X,d))K'))
   ==> (?p p'. (K p /\ K' p' /\ (!q q'. (K q /\ K' q') ==>
              (d p p' <= d q q'))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `UNIONS (top_of_metric(X,d)) = X` SUBGOAL_TAC;
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  DISCH_TAC;
  TYPE_THEN `K SUBSET X /\ K' SUBSET X` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[compact]);
  REWR 0;
  REWR 1;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `Y = { z | ?q q'. (K q /\ K' q' /\ (z = d q q'))}` ABBREV_TAC ;
  TYPE_THEN `!y. (Y y) ==> (&0 <= y)` SUBGOAL_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[metric_space]);
  TYPEL_THEN [`q`;`q'`;`q'`] (USE 4 o ISPECL);
  ASM_MESON_TAC[metric_space;ISUBSET];
  REP_BASIC_TAC;
  (*  *)
  TYPE_THEN `~(Y= EMPTY)` SUBGOAL_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  UND 2;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `d u' u` EXISTS_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* inf Y *)
  TYPE_THEN `(!epsilon. (&0 < epsilon) ==> (?x. Y x /\ (x < inf Y + epsilon)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  inf_eps;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  ASSUME_TAC twopow_pos;
  TYPE_THEN `(!n. ?p. ?p'. K p /\ K' p' /\ (d p p' < inf Y + twopow( -- (&:n))))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `(?x. Y x /\ x < inf Y + twopow (--: (&:n)))` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  UND 14;
  EXPAND_TAC "Y";
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  RIGHT 13 "n";
  REP_BASIC_TAC;
  (* compact,complete,totally bounded *)
  TYPE_THEN `metric_space (K,d) /\ metric_space(K',d)` SUBGOAL_TAC;
  ASM_MESON_TAC[metric_subspace];
  REP_BASIC_TAC;
  TYPE_THEN `compact (top_of_metric(K,d)) K /\ compact (top_of_metric(K',d)) K'` SUBGOAL_TAC;
  ASM_MESON_TAC[compact_subset];
  REP_BASIC_TAC;
  TYPE_THEN `complete (K,d)  /\ complete (K',d) ` SUBGOAL_TAC;
  ASM_MESON_TAC[compact_complete];
  REP_BASIC_TAC;
  TYPE_THEN `totally_bounded(K,d) /\ totally_bounded(K',d)` SUBGOAL_TAC;
  ASM_MESON_TAC[compact_totally_bounded;];
  REP_BASIC_TAC;
  (* construct subseq of p *)
  TYPE_THEN `(?ss. subseq ss /\ converge (K,d) (p o ss))` SUBGOAL_TAC;
  IMATCH_MP_TAC  convergent_subseq;
  ASM_REWRITE_TAC[sequence;SUBSET;UNIV;IMAGE];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x''");
  RIGHT 13 "p'";
  ASM_MESON_TAC[];
  REWRITE_TAC[converge];
  REP_BASIC_TAC;
  (* construct q *)
  TYPE_THEN `!n. ?p'. K' p' /\ d x p' < inf Y + twopow(--: (&:n))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TSPEC `twopow (--: (&:(SUC(n))))` 22;
  REP_BASIC_TAC;
  REWR 22;
  TSPEC  `SUC(n') + SUC (n)` 22;
  RULE_ASSUM_TAC (REWRITE_RULE[ARITH_RULE `x <=| SUC x +| y`]);
  TSPEC `ss (SUC n' +| SUC n)` 13;
  REP_BASIC_TAC;
  TYPE_THEN `twopow (--: (&:(ss(SUC n'+SUC n)))) < twopow(--: (&:(SUC n)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  twopow_lt;
  REWRITE_TAC[INT_LT_NEG;INT_OF_NUM_LT;];
  IMATCH_MP_TAC (ARITH_RULE `(?t. (a <= t /\ t <| b)) ==> (a <| b)`);
  TYPE_THEN `ss (SUC n)` EXISTS_TAC;
  ASM_SIMP_TAC[SEQ_SUBLE;subseq];
  RULE_ASSUM_TAC (REWRITE_RULE[subseq]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `p'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC  (REWRITE_RULE[metric_space]);
  REP_BASIC_TAC;
  TYPEL_THEN [`x`;`p (ss (SUC n' +| SUC n))`;`p'`] (USE 4 o ISPECL);
  REP_BASIC_TAC;
  TYPE_THEN `X x /\ X (p (ss (SUC n' +| SUC n))) /\ X p'` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  REWR 4;
  REP_BASIC_TAC;
  TYPE_THEN `twopow( --: (&:(SUC n))) + twopow (--: (&:(SUC n))) = twopow (--: (&:n))` SUBGOAL_TAC;
  REWRITE_TAC[GSYM REAL_MUL_2;ADD1;twopow_double];
  UND 4;
  UND 13;
  UND 27;
  UND 22;
  REWRITE_TAC[o_DEF];
  REAL_ARITH_TAC;
  DISCH_TAC;
  RIGHT 25 "n" ;
  REP_BASIC_TAC;
  (* take subseq of p' *)
  TYPE_THEN `(?ss'. subseq ss' /\ converge (K',d) (p' o ss'))` SUBGOAL_TAC;
  IMATCH_MP_TAC  convergent_subseq;
  ASM_REWRITE_TAC[sequence;SUBSET;UNIV;IMAGE];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x''");
  ASM_MESON_TAC[];
  REWRITE_TAC[converge];
  REP_BASIC_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* now go in for the KILL.  *)
  (*   Show d x x' <= inf Y because d x x' < inf Y + eps *)
  (* [K] *)
  IMATCH_MP_TAC  (REAL_ARITH `(?t. (t <= y) /\ (x <= t)) ==> (x <= y)`);
  TYPE_THEN `inf Y` EXISTS_TAC;
  CONJ_TAC;
  TYPE_THEN `(!y. Y y ==> inf Y <= y)` SUBGOAL_TAC;
  ASM_MESON_TAC[inf_LB];
  DISCH_THEN IMATCH_MP_TAC ;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  TYPE_THEN `q` EXISTS_TAC;
  TYPE_THEN `q'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  SUBGOAL_TAC  `!x y. (!e. (&0 <e) ==> (x < y + e)) ==> (x <= y)`;
  REP_GEN_TAC;
  DISCH_THEN (fun t -> MP_TAC (SPEC `x'' - y` t));
  REAL_ARITH_TAC;
  DISCH_THEN IMATCH_MP_TAC ;
  REP_BASIC_TAC;
  KILL 15;
  KILL 14;
  KILL 17;
  KILL 16;
  KILL 18;
  KILL 19;
  KILL 20;
  KILL 21;
  KILL 2;
  KILL 3;
  KILL 0;
  KILL 1;
  KILL 8;
  KILL 29;
  KILL 30;
  (* GEN needed inequalities *)
  (* [L] *)
  TYPE_THEN `?n. (&1)* twopow(--: (&:n)) < e` SUBGOAL_TAC;
  ASM_MESON_TAC[twopow_eps;REAL_ARITH `&0 < &1`];
  REDUCE_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `twopow( --: (&:(SUC n))) + twopow (--: (&:(SUC n))) = twopow (--: (&:n))` SUBGOAL_TAC;
  REWRITE_TAC[GSYM REAL_MUL_2;ADD1;twopow_double];
  REP_BASIC_TAC;
  TSPEC `twopow(--: (&:(SUC n)))` 26;
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[twopow_pos]);

  TSPEC `SUC (n) + SUC n'` 2;
  USE 2(REWRITE_RULE[ARITH_RULE `a <=| b + SUC a`]);
  TSPEC `ss' (SUC n + SUC n')` 25;
  TYPE_THEN `twopow (--: (&:(ss' (SUC  n +| SUC n')))) < twopow (--: (&:(SUC n)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  twopow_lt;
  REWRITE_TAC[INT_LT_NEG;INT_OF_NUM_LT ];
  IMATCH_MP_TAC  (ARITH_RULE  `(?t. (a <=| t /\ (t <| b)))    ==> (a <| b)`);
  TYPE_THEN `(ss' (SUC n) )` EXISTS_TAC;
  ASM_SIMP_TAC[SEQ_SUBLE];
  RULE_ASSUM_TAC (REWRITE_RULE[subseq]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ARITH_TAC;
  DISCH_TAC;
  REP_BASIC_TAC;
  (* metric space ineq *)
  TYPE_THEN `X x /\ X x' /\ X (p' (ss' (SUC n +| SUC n')))` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[o_DEF]);
  TYPE_THEN `r = p' (ss' (SUC n +| SUC n'))` ABBREV_TAC ;
  TYPE_THEN `d x' r = d r x'` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_space_symm;
  ASM_MESON_TAC[];
  TYPE_THEN `d x x' <= d x r + d r x'` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_space_triangle;
  ASM_MESON_TAC[];
  UND 0;
  UND 1;
  UND 2;
  UND 3;
  UND 8;
  REAL_ARITH_TAC;
  (* Fri Aug  6 11:54:33 EDT 2004 *)
  ]);;
  (* }}} *)

let max_real_le = prove_by_refinement(
  `!x y. x <= max_real x y  /\ y <= max_real x y `,
  (* {{{ proof *)
  [
  REWRITE_TAC[max_real];
  REP_GEN_TAC;
  COND_CASES_TAC;
  UND 0;
  REAL_ARITH_TAC;
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let min_real_le = prove_by_refinement(
  `!x y.  min_real x y <= x /\ min_real x y <= y`,
  (* {{{ proof *)
  [
  REWRITE_TAC[min_real];
  REP_GEN_TAC;
  COND_CASES_TAC;
  UND 0;
  REAL_ARITH_TAC;
  UND 0;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let finite_UB = prove_by_refinement(
  `!X. (FINITE X) ==> (?t. (!x. X x ==> x <=. t))`,
  (* {{{ proof *)
  [
  TYPE_THEN `!n X. (X HAS_SIZE n) ==> (?t. (!x. X x ==> x <= t))` SUBGOAL_TAC;
  INDUCT_TAC ;
  REWRITE_TAC[HAS_SIZE_0;EQ_EMPTY;];
  MESON_TAC[];
  REWRITE_TAC[HAS_SIZE_SUC];
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  TSPEC `X DELETE u` 0;
  TYPE_THEN `(?t. !x. (X DELETE u) x ==> x <= t)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `max_real t u` EXISTS_TAC;
  GEN_TAC;
  DISCH_TAC;
  TYPE_THEN `x = u` ASM_CASES_TAC;
  ASM_MESON_TAC[max_real_le];
  TSPEC `x` 3;
  RULE_ASSUM_TAC (REWRITE_RULE[DELETE]);
  ASM_MESON_TAC[max_real_le;REAL_LE_TRANS];
  REWRITE_TAC[HAS_SIZE];
  ASM_MESON_TAC[];
  (* Fri Aug  6 12:50:04 EDT 2004 *)
  ]);;
  (* }}} *)

let finite_LB = prove_by_refinement(
  `!X. (FINITE X) ==> (?t. (!x. X x ==> t <=. x))`,
  (* {{{ proof *)
  [
  TYPE_THEN `!n X. (X HAS_SIZE n) ==> (?t. (!x. X x ==> t <= x))` SUBGOAL_TAC;
  INDUCT_TAC ;
  REWRITE_TAC[HAS_SIZE_0;EQ_EMPTY;];
  MESON_TAC[];
  REWRITE_TAC[HAS_SIZE_SUC];
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  TSPEC `X DELETE u` 0;
  TYPE_THEN `(?t. !x. (X DELETE u) x ==> t <= x)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `min_real t u` EXISTS_TAC;
  GEN_TAC;
  DISCH_TAC;
  TYPE_THEN `x = u` ASM_CASES_TAC;
  ASM_MESON_TAC[min_real_le];
  TSPEC `x` 3;
  RULE_ASSUM_TAC (REWRITE_RULE[DELETE]);
  ASM_MESON_TAC[min_real_le;REAL_LE_TRANS];
  REWRITE_TAC[HAS_SIZE];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let finite_compact = prove_by_refinement(
  `!(X:A->bool) U. (FINITE X) /\ (X SUBSET UNIONS U) ==> (compact U X)`,
  (* {{{ proof *)
  [
  TYPE_THEN `!n (X:A->bool) U. (X HAS_SIZE n) /\ (X SUBSET UNIONS U) ==> (compact U X)` SUBGOAL_TAC;
  INDUCT_TAC;
  REWRITE_TAC[HAS_SIZE_0];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[compact];
  REP_BASIC_TAC;
  TYPE_THEN `EMPTY:(A->bool)->bool` EXISTS_TAC;
  REWRITE_TAC[FINITE_RULES];
  REWRITE_TAC[HAS_SIZE_SUC;EMPTY_EXISTS;compact ;];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `X DELETE u HAS_SIZE n` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPEL_THEN [`X DELETE u`;`U`] (USE 0 o ISPECL);
  REP_BASIC_TAC;
  REWR 0;
  TYPE_THEN `X DELETE u SUBSET UNIONS U` SUBGOAL_TAC;
  UND 1;
  REWRITE_TAC[SUBSET;DELETE];
  MESON_TAC[];
  DISCH_TAC;
  REWR 0;
  RULE_ASSUM_TAC (REWRITE_RULE[compact]);
  REP_BASIC_TAC;
  TSPEC `V` 0;
  REWR 0;
  TYPE_THEN `X DELETE u SUBSET UNIONS V` SUBGOAL_TAC;
  UND 6;
  REWRITE_TAC[SUBSET;DELETE];
  MESON_TAC[];
  DISCH_TAC;
  REWR 0;
  REP_BASIC_TAC;
  USE 6 (REWRITE_RULE[SUBSET;UNIONS]);
  TSPEC `u` 6;
  REWR 6;
  REP_BASIC_TAC;
  TYPE_THEN `u' INSERT W` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[INSERT_SUBSET];
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[FINITE_INSERT];
  REWRITE_TAC[UNIONS_INSERT];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `u' UNION (X DELETE u)` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[SUBSET;DELETE;UNION];
  ASM_MESON_TAC[];
  UND 0;
  REWRITE_TAC[UNION;SUBSET];
  MESON_TAC[];
  REWRITE_TAC[HAS_SIZE];
  MESON_TAC[];
  ]);;
  (* }}} *)

let compact_supm = prove_by_refinement(
  `!X. (compact(top_of_metric(UNIV,d_real)) X) /\ ~(X = EMPTY) ==>
          X (supm X)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(?x. X x /\ (!y. X y ==> y <= x))` SUBGOAL_TAC;
  IMATCH_MP_TAC  compact_sup;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `(!x. X x ==> x <= supm X ) /\ (!y. (!x. X x ==> x <= y) ==> (supm X <= y))` SUBGOAL_TAC;
  IMATCH_MP_TAC  supm_UB;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `x = supm X` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `x <= supm X /\ supm X <= x ==> (x = supm X)`);
  TSPEC `x` 4;
  REWR 4;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];

  ]);;
  (* }}} *)

let compact_infm = prove_by_refinement(
  `!X. (compact(top_of_metric(UNIV,d_real)) X) /\ ~(X = EMPTY) ==>
          X (inf X)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(?x. X x /\ (!y. X y ==> x <= y))` SUBGOAL_TAC;
  IMATCH_MP_TAC  compact_inf;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `(!x. X x ==> inf X <= x ) /\ (!y. (!x. X x ==> y <= x) ==> ( y <= inf X))` SUBGOAL_TAC;
  IMATCH_MP_TAC  inf_LB;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `x = inf X` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `x <= inf X /\ inf X <= x ==> (x = inf X)`);
  TSPEC `x` 4;
  REWR 4;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  (* Fri Aug  6 13:45:50 EDT 2004 *)

  ]);;
  (* }}} *)

let finite_supm = prove_by_refinement(
  `!X. (FINITE X) /\ ~(X = EMPTY) ==> X (supm X)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  compact_supm;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  finite_compact;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_real;SUBSET_UNIV;];
  ]);;
  (* }}} *)

let finite_inf = prove_by_refinement(
  `!X. (FINITE X) /\ ~(X = EMPTY) ==> X (inf X)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  compact_infm;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  finite_compact;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_real;SUBSET_UNIV;];
  (* Fri Aug  6 13:49:38 EDT 2004 *)
  ]);;
  (* }}} *)

let finite_supm_max = prove_by_refinement(
  `!X. (FINITE X) /\ ~(X = EMPTY) ==> (!x. X x ==> x <= supm X)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(?t. !x. (X x ==> x <= t))` SUBGOAL_TAC;
  ASM_MESON_TAC[finite_UB];
  ASM_MESON_TAC[supm_UB];
  ]);;
  (* }}} *)

let finite_inf_min = prove_by_refinement(
  `!X. (FINITE X) /\ ~(X = EMPTY) ==> (!x. X x ==> inf X <= x)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(?t. !x. (X x ==> t <= x))` SUBGOAL_TAC;
  ASM_MESON_TAC[finite_LB];
  ASM_MESON_TAC[inf_LB];
  ]);;
  (* }}} *)

let bij_inj_image = prove_by_refinement(
  `!(f:A->B) X Y. (INJ f X Y /\ Y SUBSET IMAGE f X) ==>
      (BIJ f X Y)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ;BIJ;SURJ;SUBSET;IMAGE];
  MESON_TAC[];
  ]);;
  (* }}} *)

let suc_interval = prove_by_refinement(
  `!n. {x | x <| SUC n} = {x | x <| n} UNION {n}`,
  (* {{{ proof *)
  [
  GEN_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[UNION;INR IN_SING;];
  ARITH_TAC;
  ]);;
  (* }}} *)

let inj_domain_sub = prove_by_refinement(
  `!(f:A->B) g X Y. (!x. (X x ==> (f x = g x))) ==> (INJ f X Y = INJ g X Y)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let image_domain_sub = prove_by_refinement(
  `!(f:A->B) g X . (!x. (X x ==> (f x = g x))) ==> (IMAGE f X  = IMAGE g X)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[IMAGE];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let real_finite_increase = prove_by_refinement(
  `!X. ( (FINITE X) ==>
     (? u. (BIJ u {x | x <| CARD X} X) /\
        (!i j. (i <| CARD X /\ (j <| CARD X) /\ (i <| j) ==>
         (u i <. u j)))))`,
  (* {{{ proof *)
  [
  TYPE_THEN `!n X. ( (X HAS_SIZE  n) ==> (? u. (BIJ u {x | x <| CARD X} X) /\  (!i j. (i <| CARD X /\ (j <| CARD X) /\ (i <| j) ==> (u i <. u j)))))` SUBGOAL_TAC;
  INDUCT_TAC;
  REWRITE_TAC[HAS_SIZE_0];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[CARD_CLAUSES;BIJ;INJ;SURJ];
  REWRITE_TAC[ARITH_RULE `~(j <| 0)`];
  REP_BASIC_TAC;
  COPY 1;
  UND 1;
  REWRITE_TAC[HAS_SIZE_SUC;];
  REP_BASIC_TAC;
  TYPE_THEN `X (supm X)` SUBGOAL_TAC;
  IMATCH_MP_TAC  finite_supm;
  ASM_REWRITE_TAC[];
  KILL 0;
  USE 3(REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  TSPEC `u` 1;
  ASM_MESON_TAC[FINITE_DELETE;HAS_SIZE;];
  DISCH_TAC;
  TSPEC `supm X` 1;
  REWR 1;
  TSPEC `X DELETE supm X` 0;
  REWR 0;
  REP_BASIC_TAC;
  TYPE_THEN `v = (\j. if (j = n) then supm X else u j)` ABBREV_TAC ;
  TYPE_THEN `v` EXISTS_TAC;
  TYPE_THEN `CARD (X DELETE supm X) = n` SUBGOAL_TAC;
  ASM_MESON_TAC[HAS_SIZE];
  DISCH_TAC;
  (* [th] *)
  TYPE_THEN `!x. ({x | x <| n} x ==> (v x = u x))` SUBGOAL_TAC;
  REWRITE_TAC[];
  EXPAND_TAC "v";
  GEN_TAC;
  COND_CASES_TAC;
  ASM_REWRITE_TAC[ARITH_RULE `~(n <| n)`];
  REWRITE_TAC[];
  DISCH_TAC;
    TYPE_THEN `INJ v {x | x <| n} X = INJ u {x | x <| n} X` SUBGOAL_TAC;
  IMATCH_MP_TAC  inj_domain_sub;
  UND 8;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `v n = supm X` SUBGOAL_TAC;
  EXPAND_TAC "v";
  ASM_REWRITE_TAC[];
  DISCH_TAC;
    TYPE_THEN `IMAGE v {x | x <| n} = IMAGE u {x | x <| n}` SUBGOAL_TAC;
  IMATCH_MP_TAC  image_domain_sub;
  UND 8;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `IMAGE v {x | x <| n} = X DELETE supm X` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  UND 5;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[BIJ];
  alpha_tac;
  MESON_TAC[SURJ_IMAGE];
  DISCH_TAC;
  (* obligations *)
  CONJ_TAC;
  IMATCH_MP_TAC  bij_inj_image;
  CONJ_TAC;
  TYPE_THEN `{x | x <| CARD X} = {x | x <| n} UNION {n}` SUBGOAL_TAC;
  USE 2(REWRITE_RULE[HAS_SIZE]);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[suc_interval];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  inj_split;
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;DELETE]);
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  UND 13;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INJ;SUBSET];
  MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC[INJ;SUBSET;INR IN_SING];
  ASM_MESON_TAC[];
  REWRITE_TAC[EQ_EMPTY;INTER;image_sing;INR IN_SING;];
  KILL 11;
  ASM_REWRITE_TAC[DELETE;SUBSET;];
  MESON_TAC[];
  TYPE_THEN `X = supm X INSERT (X DELETE supm X)` SUBGOAL_TAC;
  ASM_SIMP_TAC[INR INSERT_DELETE];
  USE 2 (REWRITE_RULE[HAS_SIZE]);
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]);
  REWRITE_TAC[INSERT_SUBSET];
  KILL 11;
  CONJ_TAC;
  REWRITE_TAC[IMAGE];
  TYPE_THEN `n` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ARITH_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `IMAGE v {x| x <| n}` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET_REFL];
  USE 12 GSYM;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  IMAGE_SUBSET;
  REWRITE_TAC[SUBSET];
  ARITH_TAC;
  REP_GEN_TAC;
  (* monotonicity [m] *)
  USE 2 (REWRITE_RULE[HAS_SIZE]);
  ASM_REWRITE_TAC[];
  TYPE_THEN `(!x. X x ==> x <= supm X)` SUBGOAL_TAC;
  ASM_MESON_TAC[finite_supm_max];
  DISCH_TAC;
  TYPE_THEN `j = n` ASM_CASES_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM  (fun t-> MP_TAC (AP_THM t `(v:num->real) i`));
  REWRITE_TAC[IMAGE;DELETE;];
  TSPEC  `(v i)` 13;
  UND 13;
  MESON_TAC[REAL_ARITH `a < b <=> (a<= b /\ ~(a = b))`];
  KILL 3;
  KILL 4;
  KILL 5;
  REP_BASIC_TAC;
  TYPE_THEN `~(i = n)` SUBGOAL_TAC;
  UND 2;
  UND 3;
  ARITH_TAC;
  REWR 0;
  DISCH_TAC;
  TYPE_THEN `i <| n /\ j <| n` SUBGOAL_TAC;
  UND 3;
  UND 4;
  UND 14;
  UND 16;
  ARITH_TAC;
  REP_BASIC_TAC;
  REWR 8;
  ASM_SIMP_TAC[];
  (* end *)
  REWRITE_TAC[HAS_SIZE];
  REP_BASIC_TAC;
  RIGHT 1 "n" ;
  TSPEC `X` 1;
  TSPEC `CARD X` 1;
  alpha_tac;
  ASM_MESON_TAC[];
  (* Fri Aug  6 19:51:16 EDT 2004 *)
  ]);;
  (* }}} *)

let connected_nogap = prove_by_refinement(
  `!A a b. connected (top_of_metric(UNIV,d_real)) A /\
          A a /\ A b ==>
       {x | a <= x /\ x <= b } SUBSET A`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(a = b) \/ (b < a) \/ (a < b)` SUBGOAL_TAC;
  REAL_ARITH_TAC;
  REP_CASES_TAC;
  ASM_REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[REAL_ARITH `b <= x /\ x <= b ==> (x = b)`];
  REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[REAL_ARITH `a <=x /\ x <= b ==> ~(b < a)`];
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `a < x` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `(a <= x /\ ~(a = x)) ==> a < x`);
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `x < b` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `(x <= b /\ ~(b = x)) ==> x < b`);
  ASM_MESON_TAC[];
  DISCH_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[connected]);
  REP_BASIC_TAC;
  TYPEL_THEN [` {t | t < x}`;` {t | x < t}`] (USE 2 o SPECL);
  UND 2;
  REWRITE_TAC[half_open;half_open_above];
  TYPE_THEN `({t | t < x} INTER {t | x < t} = {}) /\ A SUBSET {t | t < x} UNION {t | x < t}` SUBGOAL_TAC;
  REWRITE_TAC[INTER;EQ_EMPTY;UNION;SUBSET;];
  REWRITE_TAC[REAL_ARITH `x' < x \/ x < x' <=> ~(x' = x)`];
  CONJ_TAC;
  REAL_ARITH_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[SUBSET;];
  ASM_MESON_TAC[REAL_ARITH `x < b ==> ~(b < x)`];
  (* Fri Aug  6 20:24:45 EDT 2004 *)

  ]);;
  (* }}} *)

let connected_open = prove_by_refinement(
  `!A a b. (connected (top_of_metric(UNIV,d_real)) A /\
       (top_of_metric(UNIV,d_real) A) /\
       (~(A = EMPTY)) /\
       A SUBSET {x | a <= x /\ x <= b}) ==>
         ( A = {x | inf A < x /\ x < supm A})`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  TYPE_THEN `(!epsilon. &0 < epsilon ==> (?x. A x /\ supm A - epsilon < x))` SUBGOAL_TAC;
  IMATCH_MP_TAC  supm_eps;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `(!epsilon. &0 < epsilon ==> (?x. A x /\ x < inf A + epsilon))` SUBGOAL_TAC;
  IMATCH_MP_TAC  inf_eps;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `(!x. A x ==> x <= supm A)` SUBGOAL_TAC;
  ASM_MESON_TAC[supm_UB];
  DISCH_TAC;
  TYPE_THEN `(!x. A x ==> inf A <= x)` SUBGOAL_TAC;
  ASM_MESON_TAC[inf_LB];
  DISCH_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  TYPE_THEN `!x. (A x  ==> ?e. &0 < e /\ open_ball(UNIV,d_real) x e SUBSET A)` SUBGOAL_TAC;
  UND 2;
  MP_TAC metric_real;
  MESON_TAC[open_ball_nbd];
  REWRITE_TAC[open_ball;d_real];
  DISCH_TAC;
  (*  *)
  TYPE_THEN `!x. A x ==> (?y. A y /\ ~(x <= y))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TSPEC  `x` 8;
  REWR 8;
  REP_BASIC_TAC;
  USE 8(REWRITE_RULE[SUBSET]);
  TYPE_THEN `x - e/(&2)` EXISTS_TAC;
  REWRITE_TAC[REAL_ARITH `~(x <= x - e/(&2)) <=> (&0 < e/(&2))`];
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[REAL_ARITH `(x - (x - t)) = t`];
  TYPE_THEN `abs  (e/(&2)) = (e/(&2))` SUBGOAL_TAC;
  REWRITE_TAC[REAL_ABS_REFL];
  IMATCH_MP_TAC  (REAL_ARITH `(a < b) ==> (a <= b)`);
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[REAL_LT_HALF2];
  DISCH_TAC;
  (*  *)
  TYPE_THEN `!x. A x ==> (?y. A y /\ ~(y <= x))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TSPEC  `x` 8;
  REWR 8;
  REP_BASIC_TAC;
  USE 8(REWRITE_RULE[SUBSET]);
  TYPE_THEN `x + e/(&2)` EXISTS_TAC;
  REWRITE_TAC[REAL_ARITH `~( x + e/(&2) <= x) <=> (&0 < e/(&2))`];
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[REAL_ARITH `(x - (x + t)) = --. t`];
  TYPE_THEN `abs (--. (e/(&2))) = (e/(&2))` SUBGOAL_TAC;
  REWRITE_TAC[REAL_ABS_REFL;ABS_NEG;];
  IMATCH_MP_TAC  (REAL_ARITH `(a < b) ==> (a <= b)`);
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[REAL_LT_HALF2];
  DISCH_TAC;
  (* FIRST direction *)
  CONJ_TAC;
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  REWRITE_TAC[REAL_ARITH `u < v  <=> (u <= v /\ ~(u = v))`];
  CONJ_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  CONJ_TAC;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  (* 2 *)
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  TYPE_THEN `?a'. A a' /\ (a' < x)` SUBGOAL_TAC;
  TSPEC `x - inf A` 5;
  USE 5 (REWRITE_RULE[REAL_ARITH `&0 < x - y <=> (y < x)`;REAL_ARITH `t + x - t = x`]);
  REWR 5;
  DISCH_TAC;
  TSPEC `supm A - x` 4;
  USE 4(REWRITE_RULE[REAL_ARITH `&0 < y - x <=> (x < y)`;REAL_ARITH `t - (t -x) = x`]);
  REWR 4;
  REP_BASIC_TAC;
  TYPE_THEN `{t | a' <= t /\ t <= x'} SUBSET A` SUBGOAL_TAC;
  IMATCH_MP_TAC  connected_nogap;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_TAC;
  TSPEC `x` 16;
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 4;
  UND 14;
  REAL_ARITH_TAC;
  (* Fri Aug  6 21:34:56 EDT 2004 *)

  ]);;
  (* }}} *)

let closure_real_set = prove_by_refinement(
  `!Z a.
     (closure(top_of_metric(UNIV,d_real)) Z a <=>
       (!e. (&0 < e) ==> (?z. Z z /\ (abs  (a - z) <= e))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `metric_space (UNIV,d_real) /\ Z SUBSET UNIV` SUBGOAL_TAC;
  REWRITE_TAC[metric_real;SUBSET_UNIV];
  DISCH_THEN (fun t -> MP_TAC (MATCH_MP closure_open_ball t));
  DISCH_THEN (fun t -> MP_TAC (AP_THM t `a:real`));
  REWRITE_TAC[];
  DISCH_THEN (fun t ->  REWRITE_TAC[GSYM t]);
  REWRITE_TAC[open_ball;d_real;];
  EQ_TAC;
  ASM_MESON_TAC[REAL_ARITH `a < b ==> a <= b`];
  REP_BASIC_TAC;
  TSPEC `r/(&2)` 1;
  RULE_ASSUM_TAC (REWRITE_RULE[REAL_LT_HALF1]);
  REWR 1;
  REP_BASIC_TAC;
  TYPE_THEN `z` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  (REAL_ARITH `(a <= b/(&2)) /\ (b/(&2) < b)   ==> (a < b)`);
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[half_pos];
  (* Sat Aug  7 08:14:28 EDT 2004 *)

  ]);;
  (* }}} *)

let real_div_assoc = prove_by_refinement(
  `!a b c. (a*b)/c = a*(b/c)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[real_div;REAL_MUL_AC;];
  ]);;
  (* }}} *)

let real_middle1_lt = prove_by_refinement(
  `!a b. (a < b) ==> a < (a + b)/(&2) `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(&2*a)/(&2) < (a+b)/(&2)` SUBGOAL_TAC;
  ASM_SIMP_TAC[REAL_LT_DIV2_EQ;REAL_ARITH `&0 < &2`];
  REWRITE_TAC[REAL_MUL_2];
  UND 0;
  REAL_ARITH_TAC;
  REWRITE_TAC[real_div_assoc];
  ASM_SIMP_TAC[REAL_DIV_LMUL;REAL_ARITH `~(&2 = &0)`];
  ]);;
  (* }}} *)

let real_middle2_lt = prove_by_refinement(
  `!a b. (a < b) ==>  (a + b)/(&2) < b `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN ` (a+b)/(&2) < (&2*b)/(&2)` SUBGOAL_TAC;
  ASM_SIMP_TAC[REAL_LT_DIV2_EQ;REAL_ARITH `&0 < &2`];
  REWRITE_TAC[REAL_MUL_2];
  UND 0;
  REAL_ARITH_TAC;
  REWRITE_TAC[real_div_assoc];
  ASM_SIMP_TAC[REAL_DIV_LMUL;REAL_ARITH `~(&2 = &0)`];
  ]);;
  (* }}} *)

let real_sub_half = prove_by_refinement(
  `!a b. (a - (a + b)/(&2) = (a - b)/(&2))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `((&2*a)/(&2) - (a+b)/(&2) = (a - b)/(&2))` SUBGOAL_TAC;
  REWRITE_TAC[real_div;GSYM REAL_SUB_RDISTRIB];
  REWRITE_TAC[REAL_EQ_RMUL_IMP];
  AP_THM_TAC;
  AP_TERM_TAC;
  REWRITE_TAC[REAL_MUL_2];
  REAL_ARITH_TAC;
  ASM_SIMP_TAC[REAL_ARITH `~(&2 = &0)`;REAL_DIV_LMUL;real_div_assoc];
  ]);;
  (* }}} *)

let closure_open_interval = prove_by_refinement(
  `!a b. (a < b) ==>
      (closure (top_of_metric(UNIV,d_real)) {x | a < x /\ x < b} =
       {x | a <= x /\ x <= b}) `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC  closure_subset;
  ASM_SIMP_TAC[interval_closed;top_of_metric_top;metric_real];
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  (* 2 *)
  TYPE_THEN `{x | a <= x /\ x <= b} = a INSERT (b INSERT {x | a < x /\ x < b})` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INSERT];
  GEN_TAC;
  UND 0;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[INSERT_SUBSET];
  ASM_SIMP_TAC[top_of_metric_top;metric_real;subset_closure;];
  (* USE closure_real_set *)
  REWRITE_TAC[closure_real_set];
  TYPE_THEN `!e. (&0 < e) ==> (a + e < b) \/ ((b - a)/(&2) < e)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  ASM_CASES_TAC `(a + e < b)`;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  (REAL_ARITH `(x <= y/(&2) /\ y/(&2) < y)  ==> (x < y)`);
  ASM_SIMP_TAC [half_pos];
  ASM_SIMP_TAC[REAL_LE_DIV2_EQ;REAL_ARITH `&0 < &2`];
  UND 2;
  REAL_ARITH_TAC;
  DISCH_ALL_TAC;
  (* 1 *)
  CONJ_TAC;
  REP_BASIC_TAC;
  TSPEC `e` 1;
  REWR 1;
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `a + e` EXISTS_TAC;
  REWRITE_TAC[REAL_ARITH `(a < a + e <=> &0 < e) /\ (a - (a + e) = --. e)`];
  ASM_REWRITE_TAC[ABS_NEG;];
  IMATCH_MP_TAC  (REAL_ARITH `(x = y) ==> (x <= y)`);
  REWRITE_TAC[REAL_ABS_REFL];
  UND 2;
  REAL_ARITH_TAC;
  (* 2 *)
  REP_BASIC_TAC;
  TYPE_THEN `(a + b)/(&2)` EXISTS_TAC;
  ASM_SIMP_TAC[real_middle1_lt;real_middle2_lt;real_sub_half];
  UND 3;
  UND 0;
  REWRITE_TAC[real_div;ABS_MUL];
  ASM_SIMP_TAC[REAL_ARITH `(a < b) ==> (abs(a - b) = (b-a))`];
  TYPE_THEN `abs (inv(&2)) = inv(&2)` SUBGOAL_TAC;
  REWRITE_TAC[ABS_REFL;REAL_LE_INV_EQ];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REAL_ARITH_TAC;
  (* 3 *)
  REP_BASIC_TAC;
  TSPEC `e` 1;
  REWR 1;
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `b - e` EXISTS_TAC;
  REWRITE_TAC[REAL_ARITH `(b - e < b <=> &0 < e) /\ (b - (b - e) =  e)`];
  REWRITE_TAC[REAL_ARITH `(a < b - e) <=> (a + e < b)`];
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  (REAL_ARITH `(x = y) ==> (x <= y)`);
  REWRITE_TAC[REAL_ABS_REFL];
  UND 2;
  REAL_ARITH_TAC;
  (* 4 *)
  REP_BASIC_TAC;
  TYPE_THEN `(b + a)/(&2)` EXISTS_TAC;
  ASM_SIMP_TAC[real_middle1_lt;real_middle2_lt;real_sub_half];
  ONCE_REWRITE_TAC [REAL_ARITH `(a + b) = (b + a)`];
  ASM_SIMP_TAC[real_middle1_lt;real_middle2_lt;real_sub_half];
  UND 3;
  UND 0;
  REWRITE_TAC[real_div;ABS_MUL];
  ASM_SIMP_TAC[REAL_ARITH `(a < b) ==> (abs(b - a) = (b-a))`];
  TYPE_THEN `abs (inv(&2)) = inv(&2)` SUBGOAL_TAC;
  REWRITE_TAC[ABS_REFL;REAL_LE_INV_EQ];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REAL_ARITH_TAC;
  (* Sat Aug  7 09:45:29 EDT 2004 *)
  ]);;

  (* }}} *)

let interval_subset  = prove_by_refinement(
  `!a b c d. {x | a <= x /\ x <= b} SUBSET  {x | c <= x /\ x <= d} <=>
      (b < a) \/ ((c <= a ) /\ (b <= d))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET ];
  REP_BASIC_TAC;
  ASM_CASES_TAC `b < a` ;
  ASM_REWRITE_TAC[];
  UND 0;
  REAL_ARITH_TAC;
  ASM_REWRITE_TAC[];
  EQ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `a` (WITH 1 o SPEC);
  TYPE_THEN `b` (USE 1 o SPEC);
  UND 0;
  UND 1;
  UND 2;
  REAL_ARITH_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let subset_antisym_eq = prove_by_refinement(
  `!(A:A->bool) B. (A = B) <=> (A SUBSET B /\ B SUBSET A) `,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;FUN_EQ_THM ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let interval_eq = prove_by_refinement(
(**** Parens added by JRH for real right associativity of =
  `!a b c d. {x | a <= x /\ x <= b} =  {x | c <= x /\ x <= d} =
      ((b < a) /\ (d < c)) \/ ((c = a ) /\ (b = d))`,
 ****)
  `!a b c d. ({x | a <= x /\ x <= b} =  {x | c <= x /\ x <= d}) <=>
      ((b < a) /\ (d < c)) \/ ((c = a ) /\ (b = d))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[subset_antisym_eq;interval_subset;];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let connected_open_closure = prove_by_refinement(
  `!A a b. (connected (top_of_metric(UNIV,d_real)) A /\
       (top_of_metric(UNIV,d_real) A) /\
    (closure (top_of_metric(UNIV,d_real)) A = {x | a <= x /\ x <= b}) ==>
    (A = { x | a < x /\ x < b }))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  (* deal WITH emptyset *)
  TYPE_THEN `A = EMPTY` ASM_CASES_TAC;
  REWR 0;
  UND 0;
  ASM_SIMP_TAC[top_of_metric_top;metric_real;closure_empty;];
  DISCH_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  GEN_TAC;
  FIRST_ASSUM (fun t -> MP_TAC (AP_THM t `x:real`));
  REWRITE_TAC[];
  REAL_ARITH_TAC;
  (* deal WITH containment *)
  TYPE_THEN `A SUBSET {x | a <= x /\ x <= b}` SUBGOAL_TAC;
  USE 0 SYM;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  subset_closure;
  ASM_SIMP_TAC[top_of_metric_top;metric_real];
  DISCH_TAC;
  (* quote previous result *)
  TYPE_THEN `( A = {x | inf A < x /\ x < supm A})` SUBGOAL_TAC;
  IMATCH_MP_TAC  connected_open;
  TYPE_THEN `a` EXISTS_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* now USE the closure of an open interval is the closed interval *)

  PROOF_BY_CONTR_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  UND 3;
  REWRITE_TAC[];
  ASM ONCE_REWRITE_TAC [];
  REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `inf A < supm A` SUBGOAL_TAC;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_TAC;
  USE 7(MATCH_MP closure_open_interval);
  UND 6;
  UND 0;
  REWRITE_TAC[];
  ASM ONCE_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  USE 0(REWRITE_RULE[interval_eq]);
  FIRST_ASSUM DISJ_CASES_TAC;
  UND 8;
  UND 3;
  UND 6;
  REAL_ARITH_TAC;
  ASM_REWRITE_TAC[];
  (* Sat Aug  7 10:38:12 EDT 2004 *)

  ]);;
  (* }}} *)

(* Sat Aug  7 11:01:27 EDT 2004 *)

let closed_ball_empty = prove_by_refinement(
  `!n a r. (r < &0) ==> (closed_ball(euclid n,d_euclid) a r = EMPTY)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[closed_ball;EQ_EMPTY;];
  ASM_MESON_TAC[d_euclid_pos;REAL_ARITH `&0 <= d /\ d <= r ==> ~(r < &0)`];
  ]);;
  (* }}} *)

let closed_ball_pt = prove_by_refinement(
  `!n a. (closed_ball(euclid n,d_euclid) a (&0) SUBSET {a})`,
  (* {{{ proof *)
  [
  REWRITE_TAC[closed_ball;SUBSET;INR IN_SING;];
  ASM_MESON_TAC [d_euclid_pos;d_euclid_zero;REAL_ARITH `(x <= &0 /\ &0 <= x) ==> (x = &0)`];
  ]);;
  (* }}} *)

let closed_ball_subset_open = prove_by_refinement(
  `!n a r. ?r'. closed_ball(euclid n,d_euclid) a r SUBSET
      open_ball(euclid n,d_euclid) a r'`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[closed_ball;open_ball;SUBSET ];
  TYPE_THEN `r + &1` EXISTS_TAC;
  MESON_TAC[ REAL_ARITH `(u <= r) ==> (u < r + &1)`];
  ]);;
  (* }}} *)

let closed_ball_compact = prove_by_refinement(
  `!n a r.  (compact (top_of_metric(euclid n,d_euclid))
        (closed_ball(euclid n,d_euclid) a r)) `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `closed_ball(euclid n,d_euclid) a r SUBSET (euclid n)` SUBGOAL_TAC;
  REWRITE_TAC[closed_ball;SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `open_ball(euclid n,d_euclid) a r SUBSET (euclid n)` SUBGOAL_TAC;
  REWRITE_TAC[open_ball;SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  ASM_SIMP_TAC[compact_euclid;closed_ball_closed;metric_euclid;];
  REWRITE_TAC[metric_bounded];
  TYPE_THEN `a` EXISTS_TAC;
  TYPE_THEN `r + &1`EXISTS_TAC;
  REWRITE_TAC[open_ball;SUBSET;];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  UND 2;
  REWRITE_TAC[closed_ball];
  REP_BASIC_TAC;
  TYPE_THEN `d_euclid a a = &0` SUBGOAL_TAC;
  ASM_MESON_TAC[d_euclid_zero];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[d_euclid_pos;REAL_ARITH `&0 <= d /\ d <= r ==> &0 <= r`;REAL_ARITH `u <= r ==> (u < r + &1)`];
  (* Sat Aug  7 12:15:05 EDT 2004 *)

  ]);;
  (* }}} *)

let set_dist = jordan_def
  `set_dist d (K:A->bool) (K':B->bool) =
       inf { z | (?p p'. (K p /\ K' p' /\ (z = d p p')))}`;;

let set_dist_inf = prove_by_refinement(
  `!(X:A->bool) d K K'. metric_space(X,d) /\ (K SUBSET X) /\
      (K' SUBSET X) ==>
    (!p p'. (K p /\ K' p' ==> (set_dist d K K' <= d p p')))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[set_dist];
  REP_BASIC_TAC;
  TYPE_THEN `Y = {z | ?p p'. K p /\ K' p' /\ (z = d p p')}` ABBREV_TAC ;
  TYPE_THEN `!x. Y x ==> &0 <= x` SUBGOAL_TAC;
  GEN_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[metric_space]);
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  TYPE_THEN `Y (d p p')` SUBGOAL_TAC;

  EXPAND_TAC "Y";
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;

  TYPE_THEN `~(Y = {}) /\ (?t. !x. Y x ==> t <= x)` SUBGOAL_TAC;
  CONJ_TAC;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `d p p'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_THEN (ASSUME_TAC o (MATCH_MP   inf_LB));
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let set_dist_nn = prove_by_refinement(
  `!(X:A->bool) d K K'. metric_space(X,d) /\ (K SUBSET X) /\
     ~(K = EMPTY) /\      ~(K' = EMPTY) /\
      (K' SUBSET X) ==> (&0 <= set_dist d K K')`,
  (* {{{ proof *)
  [
  REWRITE_TAC[set_dist];
  REP_BASIC_TAC;
  TYPE_THEN `Y = {z | ?p p'. K p /\ K' p' /\ (z = d p p')}` ABBREV_TAC ;
  TYPE_THEN `!x. Y x ==> &0 <= x` SUBGOAL_TAC;
  REP_BASIC_TAC;
  UND 6;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[metric_space]);
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  TYPE_THEN `~(Y = {})` SUBGOAL_TAC;
  REWRITE_TAC[EMPTY_EXISTS];
  RULE_ASSUM_TAC (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  TYPE_THEN `d u' u` EXISTS_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `~(Y = {}) /\ (?t. !x. Y x ==> t <= x)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN (ASSUME_TAC o (MATCH_MP   inf_LB));
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let set_dist_eq = prove_by_refinement(
  `!(X:A->bool) d K K'. metric_space(X,d) /\ (K SUBSET X) /\
     ~(K = EMPTY) /\      ~(K' = EMPTY) /\
    (compact (top_of_metric(X,d)) K) /\
    (compact (top_of_metric (X,d)) K') /\
      (K' SUBSET X) ==>
    (?p p'. K p /\ K' p' /\ (set_dist d K K' = d p p'))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[set_dist];
  REP_BASIC_TAC;
  TYPE_THEN `Y = {z | ?p p'. K p /\ K' p' /\ (z = d p p')}` ABBREV_TAC ;
  TYPE_THEN `!x. Y x ==> &0 <= x` SUBGOAL_TAC;
  REP_BASIC_TAC;
  UND 8;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[metric_space]);
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  TYPE_THEN `~(Y = {})` SUBGOAL_TAC;
  REWRITE_TAC[EMPTY_EXISTS];
  RULE_ASSUM_TAC (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  TYPE_THEN `d u' u` EXISTS_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `~(Y = {}) /\ (?t. !x. Y x ==> t <= x)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN (ASSUME_TAC o (MATCH_MP   inf_LB));
  TYPE_THEN `(?p p'. K p /\ K' p' /\ (!q q'. K q /\ K' q' ==> d p p' <= d q q'))` SUBGOAL_TAC;
  IMATCH_MP_TAC  compact_distance;
  TYPE_THEN `X` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `p` EXISTS_TAC;
  TYPE_THEN `p'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* 1 *)
  TYPE_THEN `Y (d p p')` SUBGOAL_TAC;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `a <= b /\ b <= a ==> (a = b)`);
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  EXPAND_TAC "Y";
  REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* Sat Aug  7 13:19:01 EDT 2004 *)

  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* SECTION L *)
(* ------------------------------------------------------------------ *)


let simple_arc_compact = prove_by_refinement(
  `!C. simple_arc top2 C ==> compact top2 C`,
  (* {{{ proof *)

  [
  REWRITE_TAC[simple_arc];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  image_compact;
  TYPE_THEN `top_of_metric(UNIV,d_real)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[inj_image_subset;interval_compact;];
  (* Sat Aug  7 12:24:22 EDT 2004 *)

  ]);;

  (* }}} *)

let simple_arc_nonempty = prove_by_refinement(
  `!C. simple_arc top2 C ==> ~(C = EMPTY)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc;EMPTY_EXISTS;];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[IMAGE;];
  TYPE_THEN `f (&0)` EXISTS_TAC;
  TYPE_THEN `&0` EXISTS_TAC;
  REWRITE_TAC[];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let graph_edge_compact = prove_by_refinement(
  `!G e. (plane_graph G) /\ (graph_edge G e) ==>
      (compact top2 e)`,
  (* {{{ proof *)
  [
  REWRITE_TAC [plane_graph];
  REP_BASIC_TAC;
  USE 3 (REWRITE_RULE[SUBSET]);
  ASM_MESON_TAC[simple_arc_compact];
  ]);;
  (* }}} *)

let graph_vertex_exist = prove_by_refinement(
  `!G. graph (G:(A,B)graph_t) /\ ~(graph_edge G = EMPTY) ==>
   (?v. graph_vertex G v)`,
  (* {{{ proof *)

  [
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G u SUBSET graph_vertex G` SUBGOAL_TAC;
  ASM_SIMP_TAC[graph_inc_subset];
  DISCH_TAC;
  TYPE_THEN `graph_inc G u HAS_SIZE 2` SUBGOAL_TAC;
  ASM_SIMP_TAC[graph_edge2;];
  REWRITE_TAC[has_size2];
  REP_BASIC_TAC;
  REWR 2;
  UND 2;
  REWRITE_TAC[SUBSET ;INR in_pair ];
  MESON_TAC[];
  ]);;

  (* }}} *)

let graph_vertex_2 = prove_by_refinement(
  `!G. graph (G:(A,B)graph_t) /\ ~(graph_edge G = EMPTY) ==>
   (?v v'. graph_vertex G v /\ graph_vertex G v' /\ ~(v = v'))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[EMPTY_EXISTS];
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G u SUBSET graph_vertex G` SUBGOAL_TAC;
  ASM_SIMP_TAC[graph_inc_subset];
  DISCH_TAC;
  TYPE_THEN `graph_inc G u HAS_SIZE 2` SUBGOAL_TAC;
  ASM_SIMP_TAC[graph_edge2;];
  REWRITE_TAC[has_size2];
  REP_BASIC_TAC;
  REWR 2;
  TYPE_THEN `a` EXISTS_TAC;
  TYPE_THEN `b` EXISTS_TAC ;
  UND 2;
  REWRITE_TAC[SUBSET ;INR in_pair ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let graph_disk_lemma1 = prove_by_refinement(
  `!G. plane_graph G /\ FINITE (graph_vertex G) /\ FINITE (graph_edge G)
       ==>
    FINITE {z | (?e v. graph_edge G e /\ graph_vertex G v /\
              ~(graph_inc G e v) /\ (z = (e,v)))}`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `Y = {z | (?e v. graph_edge G e /\ graph_vertex G v /\ ~(graph_inc G e v) /\ (z = (e,v)))}` ABBREV_TAC ;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `{(e,v) | graph_edge G e /\ graph_vertex G v}` EXISTS_TAC;
  TYPEL_THEN [`graph_edge G `;`graph_vertex G `] (fun t -> ASSUME_TAC (ISPECL t FINITE_PRODUCT));
  REWR 4;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "Y";
  REWRITE_TAC[SUBSET];
 MESON_TAC[];
  (* Sat Aug  7 14:21:19 EDT 2004 *)

    ]);;
  (* }}} *)

let image_empty = prove_by_refinement(
  `!(A:A->bool) (f:A->B). (IMAGE f A = EMPTY) <=> (A = EMPTY)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[IMAGE;FUN_EQ_THM;];
  MESON_TAC[];
  ]);;
  (* }}} *)

(* not used *)
let pair_apply = prove_by_refinement(
  `!P. (!x. P x) <=> ! (u:A) (v:B) . P (u,v)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  EQ_TAC;
  REP_BASIC_TAC;
  TSPEC `(u,v)` 0;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPEL_THEN [`FST x`;`SND x`] (USE 0 o ISPECL);
  USE 0(REWRITE_RULE[]);
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let set_dist_pos = prove_by_refinement(
  `!(X:A->bool) d K K'. metric_space(X,d) /\ (K SUBSET X) /\
     ~(K = EMPTY) /\      ~(K' = EMPTY) /\
    (compact (top_of_metric(X,d)) K) /\
    (compact (top_of_metric (X,d)) K') /\ (K INTER K' = EMPTY) /\
      (K' SUBSET X) ==>
    (&0 < (set_dist d K K' ))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  (REAL_ARITH  `~(x = &0) /\ (&0 <= x) ==> (&0 < x)`);
  CONJ_TAC;
  TYPE_THEN `(?p p'. K p /\ K' p' /\ (set_dist d K K' = d p p'))` SUBGOAL_TAC;
  IMATCH_MP_TAC  set_dist_eq;
  TYPE_THEN `X` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `p = p'` SUBGOAL_TAC;
  REWR 9;
  TYPE_THEN `X p /\ X p'` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  USE 9 SYM;
  REP_BASIC_TAC;
  UND 9;
  ASM_MESON_TAC  [metric_space_zero2];
  UND 1;
  UND 10;
  UND 11;
  REWRITE_TAC[EQ_EMPTY;INTER;];
  MESON_TAC[];
  IMATCH_MP_TAC  set_dist_nn;
  TYPE_THEN `X` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let closed_ball_inter = prove_by_refinement(
  `!(x:A) y r r' X d. (metric_space(X,d) /\
    ~(closed_ball(X,d) x r INTER closed_ball(X,d) y r' = EMPTY) ==>
   (d x y <= r + r'))`,
  (* {{{ proof *)

  [
  REWRITE_TAC[closed_ball;EMPTY_EXISTS;INTER];
  REP_BASIC_TAC;
  TYPE_THEN `d x y <= d x u + d u y` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_space_triangle;
  ASM_MESON_TAC[];
  TYPE_THEN `d u y = d y u` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_space_symm;
  ASM_MESON_TAC[];
  UND 0;
  UND 3;
  REAL_ARITH_TAC;
  ]);;

  (* }}} *)

let graph_disk = prove_by_refinement(
  `!G. plane_graph G /\
       FINITE (graph_edge G) /\ FINITE (graph_vertex G) /\
     ~(graph_edge G = EMPTY)
      ==> (?r. (&0 < r ) /\
     (!v v'. graph_vertex G v /\ graph_vertex G v' /\ ~(v = v') ==>
        (closed_ball (euclid 2,d_euclid) v r INTER
            closed_ball (euclid 2,d_euclid) v' r = EMPTY)) /\
     (!e v. (graph_edge G e /\ graph_vertex G v /\ ~(graph_inc G e v) ==>
           (e INTER closed_ball (euclid 2,d_euclid) v r = EMPTY) )))`,
  (* {{{ proof *)

  [
    REP_BASIC_TAC;
  (* A' *)
  TYPE_THEN `A = { (v,v') |  (graph_vertex G v) /\ graph_vertex G v' /\ ~(v = v') }` ABBREV_TAC ;
  TYPE_THEN `FINITE A` SUBGOAL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `{ (v,v') | (graph_vertex G v) /\ graph_vertex G v'}` EXISTS_TAC;
  TYPEL_THEN  [`graph_vertex G`;`graph_vertex G`] (fun t-> ASSUME_TAC (ISPECL   t FINITE_PRODUCT));
  REWR 5;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "A";
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `A' = IMAGE  (\ (v,v'). (d_euclid v v')/(&2)) A` ABBREV_TAC ;
  TYPE_THEN `FINITE A'` SUBGOAL_TAC;
  EXPAND_TAC "A'";
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* [B] *)
  TYPE_THEN `B = { (e,v) | graph_edge G e /\ graph_vertex G v /\ ~(graph_inc G e v) }` ABBREV_TAC ;
  TYPE_THEN `B' = IMAGE (\ (e,v). (set_dist d_euclid {v} e)) B`  ABBREV_TAC ;
  TYPE_THEN `FINITE B'` SUBGOAL_TAC;
  EXPAND_TAC "B'";
  IMATCH_MP_TAC  FINITE_IMAGE;
  TYPE_THEN `B = {z | (?e v. graph_edge G e /\ graph_vertex G v /\ ~( graph_inc G e v) /\ (z = (e,v)))}` SUBGOAL_TAC;
  EXPAND_TAC "B";
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  graph_disk_lemma1;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* [C] : A' B' C nonempty *)
  TYPE_THEN `C' = A' UNION B'` ABBREV_TAC ;
  TYPE_THEN `FINITE C' /\ ~(C' = EMPTY)` SUBGOAL_TAC;
  EXPAND_TAC "C'";
  ASM_REWRITE_TAC[FINITE_UNION];
  EXPAND_TAC "C'";
  REWRITE_TAC[EMPTY_EXISTS;UNION;];
  TYPE_THEN `~(A' = EMPTY)` SUBGOAL_TAC;
  EXPAND_TAC "A'";
  REWRITE_TAC[image_empty; ];
  TYPE_THEN `(?v v'. graph_vertex G v /\ graph_vertex G v' /\ ~(v = v'))` SUBGOAL_TAC;
  IMATCH_MP_TAC graph_vertex_2;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[plane_graph];
  REP_BASIC_TAC;
  UND 12;
  REWRITE_TAC[];
  EXPAND_TAC "A";
  REWRITE_TAC[EMPTY_EXISTS];
  CONV_TAC (dropq_conv "u");
  TYPE_THEN `v` EXISTS_TAC;
  TYPE_THEN `v'` EXISTS_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  MESON_TAC[];
  DISCH_TAC;
  (* [D]:  C(inf C) *)
  TYPE_THEN `graph_vertex G SUBSET (euclid 2)` SUBGOAL_TAC;
  UND 3;
  REWRITE_TAC[plane_graph];
  MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `C'(inf C')` SUBGOAL_TAC;
  IMATCH_MP_TAC  finite_inf;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!x. C' x ==> (inf C' <= x)` SUBGOAL_TAC;
  IMATCH_MP_TAC  finite_inf_min;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!v. (graph_vertex G v ==> compact top2 {v})` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  compact_point;
  UND 13;
  REWRITE_TAC[SUBSET;top2_unions];
  UND 12;
  MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!e. (graph_edge G e ==> compact top2 e)` SUBGOAL_TAC;
  ASM_MESON_TAC[graph_edge_compact];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!x. A' x <=> (?v' v''. graph_vertex G v' /\ graph_vertex G v'' /\  ~(v' = v'') /\ (x = d_euclid v' v'' / &2))` SUBGOAL_TAC;
  EXPAND_TAC "A'";
  EXPAND_TAC "A";
  REWRITE_TAC[IMAGE];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x'");
(*** Next steps removed by JRH: now paired beta-conversion automatic ***)
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!x. B' x <=> (?e' v'. graph_edge G e' /\ graph_vertex G v' /\  ~(graph_inc G e' v') /\ (x = set_dist d_euclid {  v' } e'))`
  SUBGOAL_TAC;
  EXPAND_TAC "B'";
  EXPAND_TAC "B";
  REWRITE_TAC[IMAGE];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x'");
(*** Next steps removed by JRH: now paired beta-conversion automatic ***)
  DISCH_TAC;
  (* -- [temp] *)
  TYPE_THEN `!x. C' x ==> (&0 < x)` SUBGOAL_TAC;
  EXPAND_TAC "C'";
  REWRITE_TAC[UNION];
  GEN_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  UND 20;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  IMATCH_MP_TAC  (REAL_ARITH `(&0 <= y /\ ~(y = &0) ) ==> &0 < y `);
  TYPE_THEN `euclid 2 v' /\ euclid 2 v''` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  UND 20;
  ASM_MESON_TAC [d_euclid_pos;d_euclid_zero;];
  (* -2-  *)
  UND 20;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  set_dist_pos;
  TYPE_THEN `euclid 2` EXISTS_TAC ;
  REWRITE_TAC[metric_euclid;single_subset];
  CONJ_TAC;
  UND 13;
  REWRITE_TAC[SUBSET];
  UND 21;
  MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC[EMPTY_EXISTS;INR IN_SING;];
  MESON_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_nonempty;
  UND 3;
  UND 22;
  REWRITE_TAC[plane_graph;SUBSET;];
  MESON_TAC[];
  REWRITE_TAC[GSYM top2];
  ASM_SIMP_TAC[];
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  TSPEC `e'` 25;
  REWR 25;
  TYPE_THEN `v'` (fun u -> FIRST_ASSUM (fun t-> (MP_TAC (AP_THM t u))));
  ASM_REWRITE_TAC[EQ_EMPTY;];
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[INR IN_SING;];
  MESON_TAC[];
  UND 22;
  UND 17;
  REWRITE_TAC[compact;top2_unions];
  MESON_TAC[];
  DISCH_TAC;
  (* [E] r good for A' *)
  TYPE_THEN `?r. (&0 < r /\ r < inf C')` SUBGOAL_TAC;
  TYPE_THEN `inf C' /(&2)` EXISTS_TAC;
  IMATCH_MP_TAC  half_pos;
  UND 20;
  UND 14;
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `r` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `A' ((d_euclid v v')/(&2))` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* -2- *)
  TYPE_THEN `r < ((d_euclid v v')/(&2))` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `(?t . (r < t /\ t <= u)) ==> (r < u)`);
  TYPE_THEN `inf C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  EXPAND_TAC "C'";
  REWRITE_TAC[UNION];
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWRITE_TAC[EQ_EMPTY ;INTER;];
  REP_BASIC_TAC;
  (* -2- triangle ineq *)
  UND 29;
  UND 30;
  UND 28;
  UND 21;
  POP_ASSUM_LIST (fun t-> ALL_TAC);
  REP_BASIC_TAC;
  (* [* temp] *)
  TYPE_THEN `d_euclid v v' <= r + r` SUBGOAL_TAC;
  IMATCH_MP_TAC  closed_ball_inter;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  REWRITE_TAC[INTER;EMPTY_EXISTS ;metric_euclid;];
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `d_euclid v v' < d_euclid v v'/(&2) + d_euclid v v'/(&2)` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `(?t. (d <= t + t /\ t < u)) ==> (d < u + u)`);
  TYPE_THEN `r` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[REAL_HALF_DOUBLE];
  REAL_ARITH_TAC;
  (* [F] good for B' *)
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  USE 27(REWRITE_RULE[EMPTY_EXISTS;INTER;]);
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `B' (set_dist d_euclid {v} e)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `e` EXISTS_TAC;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `r < set_dist d_euclid {v} e` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `(?t. (r < t /\ t <= q)) ==> (r < q)`);
  TYPE_THEN `inf C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  EXPAND_TAC "C'";
  REWRITE_TAC[UNION];
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `(!p p'. ({v} p /\ e p' ==> (set_dist d_euclid {v} e <= d_euclid p p')))` SUBGOAL_TAC;
  IMATCH_MP_TAC  set_dist_inf;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[metric_euclid;single_subset;];
  CONJ_TAC;
  UND 13;
  UND 25;
  MESON_TAC[ISUBSET];
  UND 17;
  UND 26;
  REWRITE_TAC[compact;top2_unions;];
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `set_dist d_euclid {v} e <= d_euclid v u` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[INR IN_SING];
  TYPE_THEN `d_euclid v u <= r` SUBGOAL_TAC;
  UND 27;
  REWRITE_TAC[closed_ball];
  MESON_TAC[];
  UND 30;
  REAL_ARITH_TAC;
  (* Sat Aug  7 21:33:13 EDT 2004 *)

  ]);;

  (* }}} *)

let norm2 = jordan_def `norm2 x = d_euclid x euclid0`;;

let cis = jordan_def `cis x = point(cos(x),sin(x))`;;

let norm2_cis = prove_by_refinement(
  `!x. norm2(cis(x)) = &1`,
  (* {{{ proof *)
  [
  REWRITE_TAC[norm2;cis;euclid0_point;d_euclid_point];
  REDUCE_TAC;
  ONCE_REWRITE_TAC [REAL_ARITH `(x + y) = (y + x)`];
  REWRITE_TAC[SIN_CIRCLE;SQRT_1];
  (* Sat Aug  7 21:47:16 EDT 2004 *)
  ]);;
  (* }}} *)

let norm2_nn = prove_by_refinement(
  `!x . (euclid 2 x) ==> &0 <= norm2 x`,
  (* {{{ proof *)
  [
  REWRITE_TAC[norm2;euclid0_point];
  ASM_MESON_TAC[d_euclid_pos;euclid_point];
  (* Sat Aug  7 21:52:31 EDT 2004 *)

  ]);;
  (* }}} *)

let norm2_0 = prove_by_refinement(
  `!x. (euclid 2 x) /\ (norm2 x = &0) <=> (x = euclid0)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  EQ_TAC;
  REWRITE_TAC[norm2;euclid0_point;];
  MESON_TAC[d_euclid_zero;euclid_point];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[euclid0_point;euclid_point;norm2;];
  ASM_MESON_TAC[d_euclid_zero;euclid_point];
  (* Sat Aug  7 21:59:11 EDT 2004 *)
  ]);;
  (* }}} *)

let cis_inj = prove_by_refinement(
  `!t t'. (&0 <= t /\ t < &2*pi) /\ (&0 <= t' /\ t' < &2*pi) ==>
      ((cis t = cis t') <=> (t = t'))`,
  (* {{{ proof *)
  [
  (* A trivial direction *)
  REP_BASIC_TAC;
  REWRITE_TAC[cis;point_inj;PAIR_SPLIT ];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  EQ_TAC;
  DISCH_THEN_REWRITE;
  (* B  range of s *)
  REP_BASIC_TAC;
  TYPE_THEN `s = (\t. (if (t < pi) then t else ((&2)*pi - t)))` ABBREV_TAC ;
  TYPE_THEN `!t. (&0 <= t /\ t < (&2 * pi)) ==> (&0 <= s t /\ s t <= pi)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC "s";
  COND_CASES_TAC;
  UND 9;
  UND 8;
  REAL_ARITH_TAC;
  CONJ_TAC;
  UND 7;
  REAL_ARITH_TAC;
  REWRITE_TAC[REAL_MUL_2;];
  UND 9;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* [C] : cos (s t) *)
  TYPE_THEN `!t. cos (s t) = cos t` SUBGOAL_TAC;
  EXPAND_TAC "s";
  GEN_TAC;
  COND_CASES_TAC;
  REWRITE_TAC[];
  REWRITE_TAC  [REAL_ARITH `x - t = (--. t + x)`;COS_PERIODIC;COS_NEG];
  DISCH_TAC;
  (* D : (s t) = (s t') *)
  TYPE_THEN `(s t= s t') ==> ((t = t') \/ (t' = (&2 * pi - t)))` SUBGOAL_TAC;
  EXPAND_TAC "s";
  COND_CASES_TAC;
  COND_CASES_TAC;
  MESON_TAC[];
  REAL_ARITH_TAC;
  COND_CASES_TAC;
  REAL_ARITH_TAC;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* E : show s t = s t' *)
  USE 8 GSYM;
  UND 5;
  (ASM ONCE_REWRITE_TAC []);
  DISCH_THEN (fun t -> MP_TAC (AP_TERM `acs` t));
  DISCH_TAC;
  TYPE_THEN `s t = s t'` SUBGOAL_TAC;
  TYPE_THEN `acs (cos (s t)) = s t` SUBGOAL_TAC;
  IMATCH_MP_TAC  COS_ACS;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `acs (cos (s t')) = s t'` SUBGOAL_TAC;
  IMATCH_MP_TAC  COS_ACS;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 9;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  UND 4;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[(REAL_ARITH `x - y = -- y + x`);SIN_PERIODIC ;SIN_NEG ;];
  REWRITE_TAC [(REAL_ARITH `(x = --x) <=> (x = &0)`)];
  REWRITE_TAC[SIN_ZERO_PI];
  PROOF_BY_CONTR_TAC;
  USE 4 (REWRITE_RULE[]);
  (* now t is a MULT of pi, finish *)
  FIRST_ASSUM DISJ_CASES_TAC;
  REP_BASIC_TAC;
  UND 2;
  ASM_REWRITE_TAC[];
  ASSUME_TAC PI_POS;
  ASM_SIMP_TAC[REAL_LT_RMUL_EQ];
  REWRITE_TAC  [REAL_LT];
  REWRITE_TAC[ARITH_RULE  `n <| 2 <=> (n = 0) \/ (n =1)`];
  DISCH_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  REWR 13;
  REWR 11;
  UND 0;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  UND 12;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  UND 3;
  ASM_REWRITE_TAC[];
  ASSUME_TAC PI_POS;
  REWRITE_TAC[REAL_ARITH (` ~(&0 <= -- x) <=> (&0 <. x) `)];
  IMATCH_MP_TAC  REAL_LT_MUL;
  ASM_REWRITE_TAC[REAL_LT ];
  REWRITE_TAC[ARITH_RULE  `0 <| n <=> ~(n = 0)`];
  DISCH_TAC;
  UND 0;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  (* Sun Aug  8 08:42:13 EDT 2004 *)

  ]);;
  (* }}} *)

let norm2_scale_cis = prove_by_refinement(
  `!x r. norm2(r *# cis(x)) = abs (r)`,
  (* {{{ proof *)

  [
  REWRITE_TAC[norm2;cis;euclid0_point;d_euclid_point;point_scale;];
  REDUCE_TAC;
  REWRITE_TAC[POW_MUL;GSYM REAL_LDISTRIB];
  ONCE_REWRITE_TAC [REAL_ARITH `(x + y) = (y + x)`];
  REWRITE_TAC[SIN_CIRCLE;REAL_MUL_RID;POW_2_SQRT_ABS];
  (* Sun Aug  8 08:46:56 EDT 2004 *)

  ]);;

  (* }}} *)

let norm2_scale = prove_by_refinement(
  `!x r. (euclid 2 x) ==> (norm2(r *# x) = abs (r)*norm2(x))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `?u v. (x = point(u,v))` SUBGOAL_TAC;
  USE 0 (MATCH_MP point_onto);
  REP_BASIC_TAC;
  TYPE_THEN `FST p` EXISTS_TAC;
  TYPE_THEN `SND p` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[norm2;cis;euclid0_point;d_euclid_point;point_scale;];
  REDUCE_TAC;
  REWRITE_TAC[POW_MUL;GSYM REAL_LDISTRIB];
  REWRITE_TAC[GSYM POW_2_SQRT_ABS];
  IMATCH_MP_TAC  SQRT_MUL;
  REWRITE_TAC[REAL_LE_SQUARE_POW];
  IMATCH_MP_TAC  (REAL_ARITH `&0 <= x /\ &0 <= y ==> &0 <= x + y`);
  REWRITE_TAC[REAL_LE_SQUARE_POW];

  ]);;
  (* }}} *)

let polar_inj = prove_by_refinement(
  `!x x' r r'. (&0 <= r) /\ (&0 <= r') /\ (&0 <= x) /\ (&0 <= x') /\
     (x < &2 *pi) /\ (x' < &2 * pi) /\ (r *# cis(x) = r' *# cis(x')) ==>
     ((r = &0) /\ (r' = &0)) \/ ((r = r') /\ (x = x'))`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  TYPE_THEN `abs  r = abs  r'` SUBGOAL_TAC;
  FIRST_ASSUM (fun t -> MP_TAC (AP_TERM `norm2` t));
  REWRITE_TAC[norm2_scale_cis];
  DISCH_TAC;
  TYPE_THEN `r' = r` SUBGOAL_TAC;
  ASM_MESON_TAC[ABS_REFL];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  ASM_CASES_TAC `(r = &0)` ;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  REWR 0;
  TYPE_THEN `cis x = cis x'` SUBGOAL_TAC;
  IMATCH_MP_TAC  euclid_scale_cancel;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[cis_inj];
  ]);;

  (* }}} *)

let norm2_bounds = prove_by_refinement(
  `!a b s t. (&0 < a) /\ (a < b) /\ (&0 <= t) /\ (t <= &1) ==>
    (a <= norm2((a + t*(b-a))*# cis(s))) /\
    ( norm2((a + t*(b-a))*# cis(s)) <= b) `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[norm2_scale_cis];
  TYPE_THEN `a <= a + t*(b - a)` SUBGOAL_TAC;
  REWRITE_TAC[REAL_ARITH `x <= x + y <=> (&0 <= y)`];
  IMATCH_MP_TAC  REAL_LE_MUL;
  ASM_REWRITE_TAC[];
  UND 2;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `&0 <= a + t*(b-a)` SUBGOAL_TAC;
  UND 4;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `abs  (a + t*(b-a)) = a + t*(b-a)` SUBGOAL_TAC;
  REWRITE_TAC[ABS_REFL];
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  ineq_le_tac `(a + t*(b-a)) + (&1 - t)*(b - a) = b`;
  (* Sun Aug  8 09:12:18 EDT 2004  *)

  ]);;
  (* }}} *)

let norm2_point = prove_by_refinement(
  `!u v. norm2(point(u,v)) = sqrt(u pow 2 + v pow 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[norm2;euclid0_point;d_euclid_point;];
  REDUCE_TAC;
  ]);;
  (* }}} *)

let cis_exist_lemma = prove_by_refinement(
  `!x. (euclid 2 x) /\ (norm2 x = &1) ==>
    (? t. x =  cis(t))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `? u v. x = point (u,v)` SUBGOAL_TAC;
  USE 1 (MATCH_MP point_onto);
  REP_BASIC_TAC;
  TYPE_THEN `FST p` EXISTS_TAC;
  TYPE_THEN `SND p` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REWR 0;
  UND 0;
  REWRITE_TAC[norm2_point];
  DISCH_TAC;
  USE 0 (fun t -> AP_TERM `\t. t pow 2` t);
  UND 0;
  BETA_TAC;
  REDUCE_TAC;
  TYPE_THEN `(sqrt (u pow 2 + v pow 2) pow 2 = u pow 2 + v pow 2)` SUBGOAL_TAC;
  IMATCH_MP_TAC  SQRT_POW_2;
  IMATCH_MP_TAC  (REAL_ARITH `&0 <= x /\ &0 <= y ==> &0 <= x + y`);
  ASM_REWRITE_TAC[REAL_LE_POW_2];
  DISCH_THEN_REWRITE;
  DISCH_THEN (fun t -> MP_TAC (MATCH_MP CIRCLE_SINCOS t));
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[cis];
  MESON_TAC[];

  ]);;
  (* }}} *)

let cos_period = prove_by_refinement(
  `! j t. (cos (t + &j * &2 *pi) = cos(t))`,
  (* {{{ proof *)
  [
  INDUCT_TAC;
  REDUCE_TAC;
  REWRITE_TAC[ADD1;GSYM REAL_ADD;REAL_ADD_RDISTRIB;REAL_ADD_ASSOC;];
  REDUCE_TAC;
  REWRITE_TAC[COS_PERIODIC];
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let sin_period = prove_by_refinement(
  `! j t. (sin (t + &j * &2 *pi) = sin(t))`,
  (* {{{ proof *)
  [
  INDUCT_TAC;
  REDUCE_TAC;
  REWRITE_TAC[ADD1;GSYM REAL_ADD;REAL_ADD_RDISTRIB;REAL_ADD_ASSOC;];
  REDUCE_TAC;
  REWRITE_TAC[SIN_PERIODIC];
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let cos_period_neg = prove_by_refinement(
  `! j t. (cos (t - &j * &2 *pi) = cos(t))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  ASSUME_TAC cos_period;
  TYPEL_THEN [`j`;`t - &j * &2 * pi`] (USE 0 o ISPECL);
  RULE_ASSUM_TAC (REWRITE_RULE [REAL_ARITH `t - x + x = t`]);
  USE 0 SYM;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let sin_period_neg = prove_by_refinement(
  `! j t. (sin (t - &j * &2 *pi) = sin(t))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  ASSUME_TAC sin_period;
  TYPEL_THEN [`j`;`t - &j * &2 * pi`] (USE 0 o ISPECL);
  RULE_ASSUM_TAC (REWRITE_RULE [REAL_ARITH `t - x + x = t`]);
  USE 0 SYM;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let cos_period_int = prove_by_refinement(
  `!m t. (cos (t + real_of_int m * &2 *pi) = cos (t))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  ASSUME_TAC INT_REP2 ;
  TSPEC `m` 0;
  REP_BASIC_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[int_of_num_th;cos_period];
  ASM_REWRITE_TAC[int_of_num_th;int_neg_th;cos_period_neg;GSYM real_sub;REAL_MUL_LNEG];
  ]);;
  (* }}} *)

let sin_period_int = prove_by_refinement(
  `!m t. (sin (t + real_of_int m * &2 *pi) = sin (t))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  ASSUME_TAC INT_REP2 ;
  TSPEC `m` 0;
  REP_BASIC_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[int_of_num_th;sin_period];
  ASM_REWRITE_TAC[int_of_num_th;int_neg_th;sin_period_neg;GSYM real_sub;REAL_MUL_LNEG];
  ]);;
  (* }}} *)

let cos_sin_reduce = prove_by_refinement(
  `!t. ?t'. (cos t = cos t') /\
      (sin t = sin t') /\ (&0 <= t') /\ (t' < &2 * pi)`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
    ASSUME_TAC floor_ineq;
  TSPEC `t/(&2 *pi)` 0;
  TYPE_THEN `f = floor (t/(&2 * pi))` ABBREV_TAC ;
  REP_BASIC_TAC;
  TYPE_THEN `t' = t - real_of_int(f)*(&2)*pi` ABBREV_TAC  ;
  TYPE_THEN `t'` EXISTS_TAC;
  TYPE_THEN `t' = t + real_of_int (--: f) *(&2)*pi` SUBGOAL_TAC;
  EXPAND_TAC "t'";
  REWRITE_TAC[REAL_ARITH `x -y = x + (-- y)`;REAL_ARITH `-- (x * y) = (-- x)*y`;GSYM int_neg_th];
  DISCH_TAC;
  CONJ_TAC;
  ASM_REWRITE_TAC[cos_period_int];
  CONJ_TAC;
  ASM_REWRITE_TAC[sin_period_int];
  EXPAND_TAC "t'";
  TYPE_THEN `&0 < (&2 *pi)` SUBGOAL_TAC;
  REWRITE_TAC[REAL_MUL_2];
  MP_TAC PI_POS;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `~(&0 = &2* pi)` SUBGOAL_TAC;
  UND 5;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `t = (t/(&2 *pi))*(&2 *pi)` SUBGOAL_TAC;
  ASM_SIMP_TAC[REAL_DIV_RMUL];
  DISCH_TAC;
  USE 7 SYM ;
  TYPE_THEN `&0 <= (t/(&2*pi))*(&2*pi) - real_of_int f * (&2*pi)` SUBGOAL_TAC;
  REWRITE_TAC[GSYM REAL_SUB_RDISTRIB];
  IMATCH_MP_TAC  REAL_LE_MUL;
  UND 2;
  UND 5;
  REAL_ARITH_TAC;
    KILL 4;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  EXPAND_TAC "t'";
  TYPE_THEN ` (t/(&2*pi))*(&2*pi) - real_of_int f * (&2*pi) < &1* &2*pi` SUBGOAL_TAC;
  REWRITE_TAC[GSYM REAL_SUB_RDISTRIB];
  IMATCH_MP_TAC  REAL_LT_RMUL;
  UND 0;
  UND 5;
  REAL_ARITH_TAC;
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  (* Tue Aug 10 09:57:36 EDT 2004 *)

  ]);;

  (* }}} *)

let cis_lemma = prove_by_refinement(
  `!x. (euclid 2 x) /\ (norm2 x = &1) ==>
    (?t. &0 <= t /\ t < &2 * pi /\ (x = cis t))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(?t. x = cis t)` SUBGOAL_TAC;
  IMATCH_MP_TAC  cis_exist_lemma;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASSUME_TAC cos_sin_reduce;
  TSPEC `t` 3;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[cis;point_inj;PAIR_SPLIT];
  ASM_MESON_TAC[];
  (* Tue Aug 10 10:01:55 EDT 2004 *)
  ]);;
  (* }}} *)

let polar_exist = prove_by_refinement(
  `!x. (euclid 2 x) ==>
    (?r t. (&0 <= t /\ t < &2 * pi /\ &0 <= r /\ (x = r *# cis(t))))`,
  (* {{{ proof *)
  [
  (* A: trivial case of norm 0 *)
  REP_BASIC_TAC;
  ASM_CASES_TAC `norm2 x = &0` ;
  TYPE_THEN `x = euclid0` SUBGOAL_TAC;
  ASM_MESON_TAC[norm2_0];
  DISCH_THEN_REWRITE;
  TYPE_THEN `&0` EXISTS_TAC;
  TYPE_THEN `&0` EXISTS_TAC;
  REWRITE_TAC[euclid_scale0;REAL_MUL_2 ];
  MP_TAC PI_POS;
  REAL_ARITH_TAC;
  (* B: rescale to 1 *)
  TYPE_THEN `&0 < norm2 x` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `~(x = &0) /\ (&0 <= x) ==> (&0 < x)`);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  norm2_nn;
  ASM_REWRITE_TAC[];
  TYPE_THEN `r = norm2 x ` ABBREV_TAC ;
  DISCH_TAC;
  TYPE_THEN `r` EXISTS_TAC;
  TYPE_THEN `y = (&1/r)*# x` ABBREV_TAC ;
  TYPE_THEN `x = r*# y` SUBGOAL_TAC;
  EXPAND_TAC "y";
  REWRITE_TAC[euclid_scale_act;GSYM real_div_assoc];
  REDUCE_TAC;
  ASM_SIMP_TAC[REAL_DIV_REFL; euclid_scale_one;];
  DISCH_TAC;
  REWR 2;
  ASM_REWRITE_TAC[];
  TYPE_THEN `euclid 2 y` SUBGOAL_TAC;
  EXPAND_TAC "y";
  IMATCH_MP_TAC  euclid_scale_closure;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  UND 2;
  ASM_SIMP_TAC[norm2_scale];
  TYPE_THEN `abs  r = r` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_ABS_REFL];
  UND 3;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  TYPE_THEN `norm2 y = &1` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_EQ_LCANCEL_IMP;
  TYPE_THEN `r` EXISTS_TAC;
  REDUCE_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* C: invoke norm2=1 case *)
  TYPE_THEN `(?t. &0 <= t /\ t < &2 * pi /\ (y = cis t))` SUBGOAL_TAC;
  IMATCH_MP_TAC  cis_lemma;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 3;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

(*
vert r = hyperplane 2 e1 r
horz r = hyperplane 2 e2 r
cf. line2D_F..., line2D_S....
*)

let subset_union_pair = prove_by_refinement(
  `!(A:A->bool) B A' B'. (A SUBSET A') /\ (B SUBSET B') ==>
       (A UNION B) SUBSET (A' UNION B')`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  ]);;
  (* }}} *)

let subset_inter_pair = prove_by_refinement(
  `!(A:A->bool) B A' B'. (A SUBSET A') /\ (B SUBSET B') ==>
       (A INTER B) SUBSET (A' INTER B')`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;INTER];
  MESON_TAC[];
  ]);;
  (* }}} *)

let simple_arc_end_simple = prove_by_refinement(
  `!C v v'. simple_arc_end C v v' ==> simple_arc top2 C`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc_end;simple_arc];
  REP_BASIC_TAC;
  REWRITE_TAC[top2_unions];
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* Tue Aug 10 10:33:30 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_arc_end_restriction = prove_by_refinement(
  `!C K K' . simple_arc top2 C /\ closed_ top2 K /\
      closed_ top2 K' /\ (C INTER K INTER K' = EMPTY ) /\
     ~(C INTER K = EMPTY ) /\ ~(C INTER K' = EMPTY) ==>
    (?C' v v'.   C' SUBSET C /\ simple_arc_end C' v v' /\
         (C' INTER K = {v}) /\ (C' INTER K' = {v'})) `,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  TYPE_THEN `(?C' f. (C' = IMAGE f {x | &0 <= x /\ x <= &1 }) /\ C' SUBSET C /\  continuous f (top_of_metric (UNIV,d_real)) top2 /\  INJ f {x | &0 <= x /\ x <= (&1)} (euclid 2) /\  (C' INTER K = {(f (&0))}) /\  (C' INTER K' = {(f (&1))}))` SUBGOAL_TAC;
  IMATCH_MP_TAC  curve_restriction;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  TYPE_THEN `f(&0)` EXISTS_TAC;
  TYPE_THEN `f(&1)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[simple_arc_end];
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ]);;

  (* }}} *)

let simple_arc_end_trans  = prove_by_refinement(
  `!C C' v v' v'' . simple_arc_end C v v' /\ simple_arc_end C' v' v'' /\
   ( C INTER C' = {v'}) ==>
    simple_arc_end (C UNION C') v v''`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc_end];
  REP_BASIC_TAC;
  TYPE_THEN `continuous f (top_of_metric (UNIV,d_real)) top2 /\ INJ f {x | &0 <= x /\ x <= &1} (euclid 2) /\  &0 < &1/(&2) /\  &0 < &1` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  REAL_ARITH_TAC;
  DISCH_THEN (fun t -> ASSUME_TAC (MATCH_MP arc_reparameter_gen t));
  REP_BASIC_TAC;
  KILL 12;
  TYPE_THEN `continuous f' (top_of_metric (UNIV,d_real)) top2 /\ INJ f' {x | &0 <= x /\ x <= &1} (euclid 2) /\  &1/(&2) < &1 /\  &0 < &1` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_LT_HALF2];
  REAL_ARITH_TAC;
  DISCH_THEN (fun t -> ASSUME_TAC (MATCH_MP arc_reparameter_gen t));
  REP_BASIC_TAC;
  KILL 17;
  TYPE_THEN `joinf g g' (&1/(&2))` EXISTS_TAC;
  (* A: prelims *)
  TYPE_THEN `&0 < &1/(&2) /\ &1/(&2) < &1` SUBGOAL_TAC;
  REWRITE_TAC[REAL_LT_HALF1;REAL_LT_HALF2];
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `{x | &0 <= x /\ x <= &1} = {x | &0 <= x /\ x < &1/(&2)} UNION {x | &1/(&2) <= x /\ x <= &1}` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM union_closed_interval);
  UND 17;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `{x | &0 <= x /\ x < &1} SUBSET {x | &0 <= x /\ x <= &1}` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `{x | &0 <= x /\ x < &1 / &2} SUBSET {x | x < &1/(&2)}` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `{x | &1 / &2 <= x /\ x <= &1} SUBSET {x | &1/ (&2) <= x}` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `{x | &0 <= x /\ x <= &1/(&2)} = {x | &0 <= x /\ x < &1/(&2)} UNION {(&1 /(&2))}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION;INR IN_SING ];
  GEN_TAC;
  UND 17;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `g (&1/(&2)) = g' (&1/(&2))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  (* [B]: IMAGE *)
  SUBCONJ_TAC;
  ASM_REWRITE_TAC[IMAGE_UNION];
  ASM_SIMP_TAC[joinf_image_above;joinf_image_below];
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[union_subset];
  CONJ_TAC;
  CONJ_TAC;
  REWRITE_TAC[SUBSET_UNION];
   REWRITE_TAC[SUBSET;UNION];
  REWRITE_TAC[IMAGE;INR IN_SING;];
  NAME_CONFLICT_TAC;
  ASM_REWRITE_TAC[];
  CONV_TAC (dropq_conv "x''");
  GEN_TAC;
  DISCH_THEN_REWRITE;
  UND 27;
  DISCH_THEN_REWRITE;
  DISJ2_TAC ;
  TYPE_THEN `&1/(&2)` EXISTS_TAC;
  REWRITE_TAC[];
  UND 17;
  REAL_ARITH_TAC;
  REWRITE_TAC[SUBSET_UNION];
  (* --2-- *)
  USE 26 SYM;
  ASM_REWRITE_TAC[GSYM IMAGE_UNION];
  REWRITE_TAC[union_subset];
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `IMAGE g {x | &0 <= x /\ x <= &1/(&2)}` EXISTS_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  IMAGE_SUBSET;
  ASM_REWRITE_TAC[SUBSET;];
  REAL_ARITH_TAC;
  REWRITE_TAC[SUBSET_UNION];
  REWRITE_TAC[SUBSET_UNION];
  DISCH_TAC;
  (* [C]: cont,INJ *)
  CONJ_TAC;
  IMATCH_MP_TAC  joinf_cont;
  ASM_REWRITE_TAC[];
  (* -- *)
  CONJ_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  inj_split;
  ASM_SIMP_TAC[joinf_inj_above;joinf_inj_below];
  CONJ_TAC;
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x <= &1/(&2)}` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET_UNION];
  (* --2-- *)
  TYPE_THEN `IMAGE (joinf g g' (&1 / &2)) {x | &0 <= x /\ x < &1 / &2} = IMAGE g {x | &0 <= x /\ x < &1 / &2}` SUBGOAL_TAC;
  ASM_SIMP_TAC[joinf_image_below];
  DISCH_THEN_REWRITE;
  TYPE_THEN `IMAGE (joinf g g' (&1 / &2)) {x | &1 / &2 <= x /\ x <= &1} = IMAGE g' {x | &1 / &2 <= x /\ x <= &1}` SUBGOAL_TAC;
  ASM_SIMP_TAC[joinf_image_above];
  DISCH_THEN_REWRITE;
  TYPE_THEN `IMAGE g {x | &0 <= x /\ x < &1 / &2} INTER IMAGE g' {x | &1 / &2 <= x /\ x <= &1} SUBSET {v'}` SUBGOAL_TAC;
  UND 0;
  DISCH_THEN (fun t -> REWRITE_TAC[SYM t]);
  USE 26 GSYM;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  subset_inter_pair;
  REWRITE_TAC[SUBSET_REFL];
  IMATCH_MP_TAC  IMAGE_SUBSET;
  ASM_REWRITE_TAC[SUBSET ];
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `IMAGE g {x | &0 <= x /\ x < &1 /(&2)} INTER {v'} = EMPTY` SUBGOAL_TAC;
  REWRITE_TAC[EQ_EMPTY];
  GEN_TAC;
  REWRITE_TAC[IMAGE;INTER;INR IN_SING;DE_MORGAN_THM;];
  NAME_CONFLICT_TAC;
  LEFT_TAC  "x'";
  IMATCH_MP_TAC  (TAUT `(B ==> A)    ==> A \/ ~B`);
  DISCH_THEN_REWRITE;
  GEN_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `x' = &1/(&2)` SUBGOAL_TAC;
  USE 15 (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  USE 27 GSYM;
  ASM_REWRITE_TAC[];
  TYPE_THEN `g x' = g(&1/(&2))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  UND 30;
  UND 33;
  REAL_ARITH_TAC;
  UND 30;
  REAL_ARITH_TAC;
  UND 29;
  REWRITE_TAC[SUBSET;EQ_EMPTY ;INTER;INR IN_SING;];
  POP_ASSUM_LIST (fun t -> ALL_TAC);
  REP_BASIC_TAC;
  TSPEC  `x` 3;
  REWR 3;
  TSPEC `x` 2;
  REWR 2;
  (* [D] final touches *)
  CONJ_TAC;
  REWRITE_TAC[joinf];
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  REWRITE_TAC[joinf];
  ASM_SIMP_TAC [REAL_ARITH `&1/(&2) < &1 ==> (&1 < &1/ &2 <=> F)`];
  ASM_MESON_TAC[];
  (* Tue Aug 10 13:15:07 EDT 2004 *)

  ]);;
  (* }}} *)

let continuous_uninduced = prove_by_refinement(
  `!(f:A->B) U V Y.
     continuous f U (induced_top V Y) /\ IMAGE f (UNIONS U) SUBSET Y
     ==> continuous f U V`,
  (* {{{ proof *)
  [
  REWRITE_TAC[continuous;];
  REP_BASIC_TAC;
  TSPEC `v INTER Y` 2;
  TYPE_THEN `induced_top V Y (v INTER Y)` SUBGOAL_TAC;
  REWRITE_TAC[induced_top;IMAGE;];
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWR 2;
  UND 2;
  REWRITE_TAC [preimage;INTER];
  TYPE_THEN `{x | UNIONS U x /\ v (f x) /\ Y (f x)} = {x | UNIONS U x /\ v (f x)}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  GEN_TAC;
  TYPE_THEN `UNIONS U x ==> Y (f x)` SUBGOAL_TAC;
  UND 1;
  REWRITE_TAC[IMAGE;SUBSET];
  MESON_TAC[];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  (* Tue Aug 10 19:11:27 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_arc_homeo = prove_by_refinement(
  `!X d (C:A->bool). (simple_arc (top_of_metric(X,d)) C) /\
        (metric_space(X,d)) ==>
    (?f. homeomorphism f
   (top_of_metric({x | &0 <= x /\ x <= &1},d_real))
            (top_of_metric(C,d)))`,
  (* {{{ proof *)

  [
  REWRITE_TAC[simple_arc];
  REP_BASIC_TAC;
  TYPE_THEN `(UNIONS (top_of_metric(X,d)) = X) ` SUBGOAL_TAC;
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  DISCH_TAC;
  REWR 1;
  (* -- *)
  TYPE_THEN `C SUBSET X` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  inj_image_subset;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN ` (UNIONS (top_of_metric(C,d)) = C)` SUBGOAL_TAC;
  KILL 3;
  ASM_MESON_TAC [GSYM top_of_metric_unions;metric_subspace];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `{x | &0 <= x /\ x <= &1} SUBSET UNIV` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET_UNIV];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `metric_space ({x | &0 <= x /\ x <= &1},d_real)` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_subspace;
  TYPE_THEN `UNIV:real->bool` EXISTS_TAC;
  ASM_REWRITE_TAC[metric_real];
  DISCH_TAC;
  (* -- *)
  ASSUME_TAC metric_real;
  (* -- *)
  TYPE_THEN `compact (top_of_metric({x | &0 <= x /\ x <= &1},d_real)) {x | &0 <= x /\ x <= &1}` SUBGOAL_TAC;
  TYPEL_THEN [`UNIV:real->bool`;`{x| &0 <= x /\ x <= &1}`;`d_real`] (fun t-> ASSUME_TAC (ISPECL t compact_subset));
  REWR 10;
  USE 10 SYM;
  ASM_REWRITE_TAC[interval_compact];
  DISCH_TAC;
  (* -- *)
  USE 3 GSYM ;
  (* -- *)
  (* A: show homeomorphism *)
  TYPE_THEN `f` EXISTS_TAC;
    IMATCH_MP_TAC  hausdorff_homeomorphsim;
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  ASM_SIMP_TAC[top_of_metric_top;metric_subspace];
  (* -- *)
    TYPE_THEN `metric_space (C,d)` SUBGOAL_TAC;
  ASM_MESON_TAC [metric_subspace];
  DISCH_TAC;
  TYPE_THEN `IMAGE f {x| &0 <= x /\ x <= &1} SUBSET C` SUBGOAL_TAC;
  ASM_REWRITE_TAC[SUBSET_REFL ];
  DISCH_TAC;
  TYPE_THEN `IMAGE f {x| &0 <= x /\ x <= &1} SUBSET X` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* B: final obligations *)
  CONJ_TAC;
  EXPAND_TAC "C";
  IMATCH_MP_TAC  inj_bij;
  UND 1;
  REWRITE_TAC[INJ];
  MESON_TAC[];
  (* -- *)
  TYPE_THEN `induced_top (top_of_metric (UNIV,d_real)) {x| &0 <= x /\ x <= &1} {x | &0 <= x /\ x <= &1}` SUBGOAL_TAC;
  ASM_SIMP_TAC[top_of_metric_induced];
  TYPE_THEN `topology_ (top_of_metric ({x | &0 <= x /\ x <= &1},d_real))` SUBGOAL_TAC;
  ASM_SIMP_TAC[top_of_metric_top];
  DISCH_THEN (fun t-> MP_TAC (MATCH_MP top_univ t));
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  DISCH_TAC;
  TYPE_THEN `continuous f (induced_top (top_of_metric (UNIV,d_real)) {x | &0 <= x /\ x <= &1}) (top_of_metric(X,d))` SUBGOAL_TAC;
  IMATCH_MP_TAC  continuous_induced_domain;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_real];
  ASM_SIMP_TAC[metric_real;top_of_metric_induced];
  ASM_SIMP_TAC[metric_continuous_continuous;metric_subspace];
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  DISCH_THEN_REWRITE;
  ASM_SIMP_TAC[top_of_metric_top];
  IMATCH_MP_TAC  metric_hausdorff;
  ASM_REWRITE_TAC[];
  (* Tue Aug 10 20:34:30 EDT 2004 *)

  ]);;

  (* }}} *)

let continuous_metric_extend = prove_by_refinement(
  `!(f:A->B) U C X d. (metric_space(X,d) /\
      continuous f U (top_of_metric (C,d)) /\
          IMAGE f (UNIONS U) SUBSET C /\ C SUBSET X ==>
    continuous f U (top_of_metric(X,d)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `metric_space(C,d)` SUBGOAL_TAC;
  IMATCH_MP_TAC metric_subspace;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `top_of_metric(C,d) = induced_top(top_of_metric(X,d)) C` SUBGOAL_TAC;
  ASM_SIMP_TAC[top_of_metric_induced];
  DISCH_TAC;
  REWR 2;
  IMATCH_MP_TAC  continuous_uninduced;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* Tue Aug 10 20:47:53 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_arc_end_distinct = prove_by_refinement(
  `!C v v'. simple_arc_end C v v' ==> ~(v = v')`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc_end;INJ];
  REP_BASIC_TAC;
  TYPE_THEN `&0 = &1` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `f (&0)  = f(&1)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  REAL_ARITH_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let bij_imp_image = prove_by_refinement(
  `!(f:A->B) X Y. BIJ f X Y ==> (IMAGE f X = Y)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[BIJ;SURJ];
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let homeo_inj = prove_by_refinement(
  `!(f:A->B) U C X d. (homeomorphism f U (top_of_metric(C,d))) /\
     (C SUBSET X) /\ (metric_space (X,d)) ==>
    ( continuous f U (top_of_metric(X,d)) /\ INJ f (UNIONS U) C /\
      (IMAGE f (UNIONS U) = C))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[homeomorphism];
  REP_BASIC_TAC;
  TYPE_THEN`metric_space(C,d)` SUBGOAL_TAC;
  ASM_MESON_TAC [metric_subspace];
  DISCH_TAC;
  (* -- *)
  UND 4;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `IMAGE f (UNIONS U)= C` SUBGOAL_TAC;
  IMATCH_MP_TAC  bij_imp_image ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ]);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  continuous_metric_extend;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET_REFL ];
  (* Tue Aug 10 20:58:37 EDT 2004 *)


  ]);;
  (* }}} *)

let simple_arc_coord = prove_by_refinement(
  `!X d (C:A->bool). (simple_arc (top_of_metric(X,d)) C) /\
        (metric_space(X,d)) ==>
    (?f.
  (continuous f (top_of_metric(C,d)) (top_of_metric(UNIV,d_real))) /\
  (INJ f C UNIV) /\
  (IMAGE f C = {x | &0 <= x /\ x <= &1}))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `(UNIONS (top_of_metric(X,d)) = X) ` SUBGOAL_TAC;
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `C SUBSET X` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[simple_arc]);
  REP_BASIC_TAC;
  USE 4 GSYM;
  REWR 1;
  EXPAND_TAC "C";
  IMATCH_MP_TAC  inj_image_subset;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN ` (UNIONS (top_of_metric(C,d)) = C)` SUBGOAL_TAC;
  ASM_MESON_TAC [GSYM top_of_metric_unions;metric_subspace];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `{x | &0 <= x /\ x <= &1} SUBSET UNIV` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET_UNIV];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `metric_space ({x | &0 <= x /\ x <= &1},d_real)` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_subspace;
  TYPE_THEN `UNIV:real->bool` EXISTS_TAC;
  ASM_REWRITE_TAC[metric_real];
  DISCH_TAC;
  (* -- *)
  ASSUME_TAC metric_real;
  (* -- *)
  TYPE_THEN `(?f. homeomorphism f (top_of_metric({x | &0 <= x /\ x <= &1},d_real)) (top_of_metric(C,d)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_homeo;
  TYPE_THEN `X` EXISTS_TAC; (* // *)
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN ` g = (INV f  ({x | &0 <= x /\ x <= &1}) (C:A->bool))` ABBREV_TAC ;
  TYPE_THEN `g = INV f  (UNIONS((top_of_metric({x | &0 <= x /\ x <= &1},d_real)))) (UNIONS((top_of_metric(C,d))))` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[GSYM  top_of_metric_unions;metric_subspace;];
  DISCH_TAC;
  (* A: *)
  TYPE_THEN `g` EXISTS_TAC;
  (* -- *)
  (* TYPE_THEN `U = top_of_metric({x | &0 <= x /\ x <= &1},d_real)` ABBREV_TAC ; *)
  TYPE_THEN `(homeomorphism g (top_of_metric(C,d)) (top_of_metric({x | &0 <= x /\ x <= &1},d_real))) /\ ({x | &0 <= x /\ x <= &1} SUBSET UNIV) /\ (metric_space (UNIV,d_real))` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  TYPEL_THEN [`f`;`(top_of_metric({x | &0 <= x /\ x <= &1},d_real))`;`top_of_metric(C,d)`] (fun t-> ASSUME_TAC (ISPECL t homeomorphism_inv));
  REWR 11;
  DISCH_TAC;
    USE 11 (MATCH_MP homeo_inj);
  REP_BASIC_TAC;
  KILL 9;
  KILL 10;
  ASM_REWRITE_TAC[];
  UND 11;
  UND 12;
  ASM_REWRITE_TAC[];
  UND 5;
  POP_ASSUM_LIST (fun t-> ALL_TAC);
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[INJ_UNIV];
  (* Tue Aug 10 21:49:22 EDT 2004 *)

  ]);;
  (* }}} *)

(* slow! *)
let image_interval = prove_by_refinement(
  `!a b f. (a < b) /\
   (continuous f (top_of_metric(UNIV,d_real))
        (top_of_metric( UNIV,d_real)))  /\
    (INJ f {x | a <= x /\ x <= b} UNIV) ==>
   (?c d. (c < d) /\ ({ c , d} = {(f a),(f b)}) /\
    (IMAGE f {x | a <= x /\ x <= b} =
       {x | c <= x /\ x <= d})
     ) `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  (* -- *)
  ASSUME_TAC connect_real;
  TYPE_THEN `!a b. connected (top_of_metric(UNIV,d_real)) (IMAGE f {x |  a<= x /\ x <= b})` SUBGOAL_TAC;
  REP_GEN_TAC;
  IMATCH_MP_TAC  connect_image;
  TYPE_THEN `top_of_metric(UNIV,d_real)` EXISTS_TAC ;
  ASM_SIMP_TAC[GSYM top_of_metric_unions;metric_real];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `c = min_real (f a) (f b)` ABBREV_TAC ;
  TYPE_THEN `d = max_real (f a) (f b)` ABBREV_TAC ;
  TYPE_THEN `c`EXISTS_TAC;
  TYPE_THEN `d` EXISTS_TAC;
  TYPE_THEN `~(f a = f b)` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  TYPE_THEN `a = b` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 2;
  REAL_ARITH_TAC;
  UND 2;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  SUBCONJ_TAC;
  EXPAND_TAC "d";
  EXPAND_TAC "c";
  REWRITE_TAC[min_real;max_real];
  TYPE_THEN `f a < f b \/ f b < f a` SUBGOAL_TAC;
  UND 7;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `~(f b < f a)` SUBGOAL_TAC;
  UND 8;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  TYPE_THEN `~(f a < f b)` SUBGOAL_TAC;
  UND 8;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  SUBCONJ_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[in_pair];
  EXPAND_TAC "d";
  EXPAND_TAC "c";
  REWRITE_TAC[max_real;min_real];
  TYPE_THEN `f a < f b \/ f b < f a` SUBGOAL_TAC;
  UND 7;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `~(f b < f a)` SUBGOAL_TAC;
  UND 9;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  TYPE_THEN `~(f a < f b)` SUBGOAL_TAC;
  UND 9;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  MESON_TAC[];
  DISCH_TAC;
  (* B *)
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  SUBCONJ_TAC;
  IMATCH_MP_TAC  connected_nogap;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "c";
  EXPAND_TAC "d";
  REWRITE_TAC[max_real;min_real];
  TYPE_THEN `f a < f b \/ f b < f a` SUBGOAL_TAC;
  UND 7;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `~(f b < f a)` SUBGOAL_TAC;
  UND 10;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_MESON_TAC[REAL_ARITH `a<= a`;REAL_ARITH `a < b ==> a <= b`];
  TYPE_THEN `~(f a < f b)` SUBGOAL_TAC;
  UND 10;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_MESON_TAC[REAL_ARITH `a<= a`;REAL_ARITH `a < b ==> a <= b`];
  DISCH_TAC;
  (* C set up cases *)
  REWRITE_TAC[IMAGE;SUBSET;];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  USE 14 (REWRITE_RULE[DE_MORGAN_THM]);
  USE 9 (REWRITE_RULE[FUN_EQ_THM;in_pair ]);
  TYPE_THEN `((c = f a) /\ (d = f b)) \/ ((c = f b) /\ (d = f a))` SUBGOAL_TAC;
  UND 9;
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `f x' < c \/ d < f x'` SUBGOAL_TAC;
  UND 14;
  ARITH_TAC;
  DISCH_TAC;
  KILL 9;
  KILL 14;
  KILL 11;
  (* D generic case *)
  TYPE_THEN `!r s t. (a <= r /\ r <= b /\ a <= s /\ s <= b /\ a <= t /\ t <= b /\ (r < t) /\ (f r < f s) /\ (f s < f t) ==> (r < s /\ s < t))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  TYPEL_THEN [`r`;`t`] (USE 4 o ISPECL);
  USE 4(REWRITE_RULE[connected]);
  REP_BASIC_TAC;
  TYPE_THEN `IMAGE f {x | r <= x /\ x <= t} SUBSET {x | x < f s} \/ IMAGE f {x | r <= x /\ x <= t} SUBSET {x | f s < x}` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[half_open;half_open_above;EQ_EMPTY;INTER;];
  CONJ_TAC;
  REAL_ARITH_TAC;
  REWRITE_TAC[IMAGE;SUBSET;UNION;];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC (REAL_ARITH  `~(f x'' = f s) ==> (f x'' < f s \/ f s < f x'')` );
  DISCH_TAC;
  TYPE_THEN `x'' = s` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 26;
  UND 27;
  UND 22;
  UND 17;
  REAL_ARITH_TAC;
  UND 9;
  UND 11;
  UND 23;
  UND 26;
  UND 27;
  POP_ASSUM_LIST (fun t-> ALL_TAC);
  REP_BASIC_TAC;
  TYPE_THEN `~(r = s)` SUBGOAL_TAC;
  ASM_MESON_TAC[REAL_ARITH `a < b ==> ~(a=b)`];
  TYPE_THEN `~(s = t)` SUBGOAL_TAC;
  ASM_MESON_TAC[REAL_ARITH `a < b ==> ~(a=b)`];
  KILL 1;
  KILL 2;
  UND 0;
  UND 3;
  UND 4;
  UND 5;
  REAL_ARITH_TAC;
  REWRITE_TAC[DE_MORGAN_THM ];
  CONJ_TAC;
  REWRITE_TAC[IMAGE;SUBSET;];
  LEFT_TAC "x";
  TYPE_THEN `f t` EXISTS_TAC;
  LEFT_TAC "x'";
  REP_BASIC_TAC;
  TSPEC `t` 25;
  UND 25;
  UND 9;
  UND 14;
  REAL_ARITH_TAC;
  REWRITE_TAC[IMAGE;SUBSET;];
  LEFT_TAC "x";
  TYPE_THEN `f r` EXISTS_TAC;
  REP_BASIC_TAC;
  LEFT 25 "x'" ;
  TSPEC `r` 25;
  UND 25;
  UND 14;
  UND 11;
  REAL_ARITH_TAC;
  (* D' generic case *)
  TYPE_THEN `!r s t. (a <= r /\ r <= b /\ a <= s /\ s <= b /\ a <= t /\ t <= b /\ (t < r) /\ (f r < f s) /\ (f s < f t) ==> (t < s /\ s < r))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  TYPEL_THEN [`t`;`r`] (USE 4 o ISPECL);
  USE 4(REWRITE_RULE[connected]);
  REP_BASIC_TAC;
  TYPE_THEN `IMAGE f {x | t <= x /\ x <= r} SUBSET {x | x < f s} \/ IMAGE f {x | t <= x /\ x <= r} SUBSET {x | f s < x}` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[half_open;half_open_above;EQ_EMPTY;INTER;];
  CONJ_TAC;
  REAL_ARITH_TAC;
  REWRITE_TAC[IMAGE;SUBSET;UNION;];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC (REAL_ARITH  `~(f x'' = f s) ==> (f x'' < f s \/ f s < f x'')` );
  DISCH_TAC;
  TYPE_THEN `x'' = s` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 26;
  UND 27;
  UND 18;
  UND 21;
  REAL_ARITH_TAC;
  UND 9;
  UND 11;
  UND 23;
  UND 26;
  UND 27;
  POP_ASSUM_LIST (fun t-> ALL_TAC);
  REP_BASIC_TAC;
  TYPE_THEN `~(r = s)` SUBGOAL_TAC;
  ASM_MESON_TAC[REAL_ARITH `a < b ==> ~(a=b)`];
  TYPE_THEN `~(s = t)` SUBGOAL_TAC;
  ASM_MESON_TAC[REAL_ARITH `a < b ==> ~(a=b)`];
  KILL 1;
  KILL 2;
  UND 0;
  UND 3;
  UND 4;
  UND 5;
  REAL_ARITH_TAC;
  REWRITE_TAC[DE_MORGAN_THM ];
  CONJ_TAC;
  REWRITE_TAC[IMAGE;SUBSET;];
  LEFT_TAC "x";
  TYPE_THEN `f t` EXISTS_TAC;
  LEFT_TAC "x'";
  REP_BASIC_TAC;
  TSPEC `t` 25;
  UND 25;
  UND 9;
  UND 14;
  REAL_ARITH_TAC;
  REWRITE_TAC[IMAGE;SUBSET;];
  LEFT_TAC "x";
  TYPE_THEN `f r` EXISTS_TAC;
  REP_BASIC_TAC;
  LEFT 25 "x'" ;
  TSPEC `r` 25;
  UND 25;
  UND 14;
  UND 11;
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  (* end generic  *)
  KILL 4;
  KILL 3;
  KILL 0;
  KILL 1;
  KILL 10;
  KILL 6;
  KILL 5;
  (* E: actual cases *)
  UND 16;
  UND 15;
  REP_CASES_TAC;
  (* --2a-- *)
  KILL 11;
  TYPEL_THEN[`x'`;`a`;`b`] (USE 9 o ISPECL);
  TYPE_THEN `~(f x' = f b)` SUBGOAL_TAC;
  REPEAT (POP_ASSUM MP_TAC);
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `~(x' = b)` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REPEAT (POP_ASSUM MP_TAC);
  REAL_ARITH_TAC;
  (* --2b-- *)
  KILL 11;
  TYPEL_THEN [`a`;`b`;`x'`] (USE 9 o ISPECL);
  TYPE_THEN `~(f a = f x')` SUBGOAL_TAC;
  REPEAT (POP_ASSUM MP_TAC);
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `~(a = x')` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REPEAT (POP_ASSUM MP_TAC);
  REAL_ARITH_TAC;
  (* --2c-- *)
  KILL 9;
  TYPEL_THEN [`x'`;`b`;`a`] (USE 11 o ISPECL);
  TYPE_THEN `~(f x' = f a)` SUBGOAL_TAC;
  REPEAT (POP_ASSUM MP_TAC);
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `~(a = x')` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REPEAT (POP_ASSUM MP_TAC);
  REAL_ARITH_TAC;
  (* --2d-- *)
  KILL 9;
  TYPEL_THEN [`b`;`a`;`x'`] (USE 11 o ISPECL);
  TYPE_THEN `~(f x' = f b)` SUBGOAL_TAC;
  REPEAT (POP_ASSUM MP_TAC);
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `~(b = x')` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REPEAT (POP_ASSUM MP_TAC);
  REAL_ARITH_TAC;
  (* Wed Aug 11 09:36:14 EDT 2004 *)
  ]);;
  (* }}} *)

let metric_continuous_range = prove_by_refinement(
  `!(f:A->B) X dX Y dY Y'.
   metric_continuous f (X,dX) (Y,dY) <=>
   metric_continuous f (X,dX) (Y',dY)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  ]);;
  (* }}} *)

let continuous_range = prove_by_refinement(
  `!(f:A->B) X dX Y dY Y'.
   metric_space(X,dX) /\ metric_space(Y,dY) /\ metric_space(Y',dY) /\
   continuous f (top_of_metric(X,dX)) (top_of_metric(Y,dY)) /\
   IMAGE f X SUBSET Y /\ IMAGE f X SUBSET Y' ==>
   continuous f (top_of_metric(X,dX)) (top_of_metric(Y',dY))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `continuous f (top_of_metric (X,dX)) (top_of_metric (Y',dY)) = metric_continuous f (X,dX) (Y',dY)`  SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_continuous_continuous;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `continuous f (top_of_metric (X,dX)) (top_of_metric (Y,dY)) = metric_continuous f (X,dX) (Y,dY)`  SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_continuous_continuous;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  REWR 2;
  ASM_MESON_TAC[metric_continuous_range];
  ]);;
  (* }}} *)

let metric_continuous_domain = prove_by_refinement(
  `!(f:A->B) X dX Y dY Y' A.
   metric_continuous f (X,dX) (Y,dY) /\ A SUBSET X ==>
  metric_continuous f (A,dX) (Y',dY)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[metric_continuous;metric_continuous_pt;SUBSET];
  MESON_TAC[];
  ]);;
  (* }}} *)

let pair_order_endpoint = prove_by_refinement(
  `!a b c d . (c < d) /\ ({c , d} = {a ,b}) ==>
    (c = min_real a b) /\ (d = max_real a b)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  USE 0 (REWRITE_RULE[FUN_EQ_THM;in_pair]);
  TYPE_THEN `((c = a) /\ (d = b)) \/ ((c = b) /\ (d = a))` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  REWR 1;
  ASM_REWRITE_TAC[min_real;max_real];
  ASM_SIMP_TAC[REAL_ARITH `a < b ==> ~(b < a)`];
  ASM_REWRITE_TAC[];
  REWR 1;
  ASM_REWRITE_TAC[min_real;max_real];
  ASM_SIMP_TAC[REAL_ARITH `a < b ==> ~(b < a)`];
  ]);;
  (* }}} *)

let cont_extend_real_lemma = prove_by_refinement(
  `!a b (f:real->A) Y dY. (a < b) /\
   (continuous f (top_of_metric({x | a <= x /\ x <= b},d_real))
     (top_of_metric(Y,dY))) /\ (metric_space(Y,dY)) /\
   IMAGE f {x | a <= x /\ x <= b} SUBSET Y ==>
  (
   ?g. (continuous g (top_of_metric(UNIV,d_real))
   (top_of_metric(Y,dY))) /\
     (!x. (a <= x /\ x <= b) ==> (f x = g x)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `?t. (a < t /\ t < b)` SUBGOAL_TAC;
  TYPE_THEN `(a+b)/(&2)` EXISTS_TAC;
  ASM_MESON_TAC[real_middle1_lt;real_middle2_lt];
  REP_BASIC_TAC;
  ASSUME_TAC metric_real;
  TYPE_THEN `{x | a <= x /\ x <= b} SUBSET UNIV` SUBGOAL_TAC;
  ASM_REWRITE_TAC[SUBSET_UNIV];
  DISCH_TAC;
  TYPE_THEN `metric_space ({x | a <= x /\ x <= b},d_real)` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_subspace;
  TYPE_THEN `UNIV:real->bool` EXISTS_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `metric_continuous f ({x | a <= x /\ x <= b},d_real) (Y,dY)` SUBGOAL_TAC;
  UND 2;
  ASM_SIMP_TAC [metric_continuous_continuous];
  DISCH_TAC;
  TYPE_THEN `A = {x | x <= a}` ABBREV_TAC ;
  TYPE_THEN `B = {x | b <= x}` ABBREV_TAC ;
  TYPE_THEN `fA  = (\(t:real). f a)` ABBREV_TAC ;
  TYPE_THEN `fB = (\(t:real). f b)` ABBREV_TAC ;
  ASSUME_TAC half_closed;
  ASSUME_TAC half_closed_above;
  (* -- *)
  TYPE_THEN `!r A. (Y r) ==> (metric_continuous (\t. r) (A,d_real) (Y,dY))` SUBGOAL_TAC;
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  REP_BASIC_TAC;
  RIGHT_TAC "delta";
  REP_BASIC_TAC;
  TYPE_THEN `epsilon` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_MESON_TAC[metric_space_zero];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `metric_continuous (subf A fA fB) (A UNION B,d_real) (Y,dY)` SUBGOAL_TAC;
  IMATCH_MP_TAC  subf_cont;
  TYPE_THEN `UNIV:real->bool` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "A";
  EXPAND_TAC "B";
  ASM_REWRITE_TAC[];
  EXPAND_TAC "fA";
  EXPAND_TAC "fB";
  TYPE_THEN `!x. x <= a /\ b <= x <=> F` SUBGOAL_TAC;
  UND 3;
  REAL_ARITH_TAC ;
  DISCH_THEN_REWRITE;
  TYPE_THEN `Y (f a) /\ Y(f b)` SUBGOAL_TAC;
  UND 0;
  REWRITE_TAC[IMAGE;SUBSET];
  TYPE_THEN `a <= a /\ a <= b /\ b <= b` SUBGOAL_TAC;
  UND 3;
  REAL_ARITH_TAC;
  MESON_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `A' = A UNION B` ABBREV_TAC ;
  TYPE_THEN `B' = {x | a <= x /\ x <= b}` ABBREV_TAC ;
  TYPE_THEN `fA' = subf A fA fB` ABBREV_TAC ;
  TYPE_THEN `metric_continuous (subf A' fA' f) (A' UNION B',d_real) (Y,dY)` SUBGOAL_TAC;
  IMATCH_MP_TAC  subf_cont;
  TYPE_THEN `UNIV:real->bool` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "A'";
  EXPAND_TAC "B'";
  CONJ_TAC;
  IMATCH_MP_TAC  closed_union;
  EXPAND_TAC "A";
  EXPAND_TAC "B";
  ASM_SIMP_TAC[top_of_metric_top];
  ASM_REWRITE_TAC[interval_closed];
  EXPAND_TAC "fA'";
  EXPAND_TAC "A'";
  EXPAND_TAC "A";
  EXPAND_TAC "B";
  REWRITE_TAC[UNION];
  GEN_TAC ;
  DISCH_TAC;
  TYPE_THEN `(x = a) \/ (x = b)` SUBGOAL_TAC;
  UND 21;
  REAL_ARITH_TAC;
  EXPAND_TAC "fA";
  EXPAND_TAC "fB";
  DISCH_THEN DISJ_CASES_TAC;
  UND 22;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[subf;REAL_ARITH `a <= a`];
  UND 22;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[subf];
  TYPE_THEN `~(b <= a)` SUBGOAL_TAC;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `A' UNION B' = UNIV` SUBGOAL_TAC;
  EXPAND_TAC "A'";
  EXPAND_TAC "A";
  EXPAND_TAC "B";
  EXPAND_TAC "B'";
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION];
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `g = subf A' fA' f` ABBREV_TAC  ;
  TYPE_THEN `!x. A x ==> (g x = f a)` SUBGOAL_TAC;
  EXPAND_TAC "g";
  REWRITE_TAC[subf];
  EXPAND_TAC "A'";
  REWRITE_TAC[UNION];
  GEN_TAC;
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "fA'";
  REWRITE_TAC[subf];
  ASM_REWRITE_TAC[];
  EXPAND_TAC "fA";
  REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!x. B x ==> (g x = f b)` SUBGOAL_TAC;
  EXPAND_TAC "g";
  REWRITE_TAC[subf];
  EXPAND_TAC "A'";
  REWRITE_TAC[UNION];
  GEN_TAC;
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "fA'";
  REWRITE_TAC[subf];
  TYPE_THEN `~(A x)` SUBGOAL_TAC;
  UND 25;
  EXPAND_TAC "B";
  EXPAND_TAC "A";
  REWRITE_TAC[];
  UND 3;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  EXPAND_TAC "fB";
  REWRITE_TAC[];
  DISCH_TAC;
  (* A  *)
  TYPE_THEN `!x. B' x ==> (g x = f x)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `A x` ASM_CASES_TAC;
  TYPE_THEN `A x /\ B' x ==> (x = a)` SUBGOAL_TAC;
  EXPAND_TAC "A";
  EXPAND_TAC "B'";
  REWRITE_TAC[];
  REAL_ARITH_TAC;
  DISCH_TAC;
  ASM_MESON_TAC[];
  (* --2-- *)
  TYPE_THEN `B x` ASM_CASES_TAC;
  TYPE_THEN `B x /\ B' x ==> (x = b)` SUBGOAL_TAC;
  EXPAND_TAC "B";
  EXPAND_TAC "B'";
  REWRITE_TAC[];
  REAL_ARITH_TAC;
  DISCH_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `~(A' x)` SUBGOAL_TAC;
  UND 27;
  UND 28;
  EXPAND_TAC "A'";
  REWRITE_TAC[UNION];
  MESON_TAC[];
  EXPAND_TAC "g";
  REWRITE_TAC[subf];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* B start on goal *)
  TYPE_THEN `g` EXISTS_TAC;
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  CONJ_TAC;
  UND 26;
  EXPAND_TAC "B'";
  REWRITE_TAC[];
  MESON_TAC[];
  TYPE_THEN `IMAGE g UNIV SUBSET Y /\ metric_space (UNIV,d_real) /\ metric_space (Y,dY)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  UND 22;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  REWRITE_TAC[IMAGE_UNION;union_subset];
  CONJ_TAC;
  EXPAND_TAC "A'";
  REWRITE_TAC[IMAGE_UNION;union_subset];
  UND 24;
  UND 25;
  REWRITE_TAC[IMAGE;SUBSET];
    TYPE_THEN `Y (f a) /\ Y(f b)` SUBGOAL_TAC;
  UND 0;
  EXPAND_TAC "B'";
  REWRITE_TAC[IMAGE;SUBSET];
  TYPE_THEN `a <= a /\ a <= b /\ b <= b` SUBGOAL_TAC;
  UND 3;
  REAL_ARITH_TAC;
  MESON_TAC[];
  MESON_TAC[];
  UND 26;
  UND 0;
  EXPAND_TAC "B'";
  REWRITE_TAC[IMAGE;SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  COPY 27;
  (* C final KILL *)
  USE 28 (MATCH_MP metric_continuous_continuous);
  ASM_REWRITE_TAC[];
  REWR 21;
  (* Wed Aug 11 12:37:40 EDT 2004 *)

  ]);;
  (* }}} *)

let image_interval2 = prove_by_refinement(
  `!a b f. (a < b) /\
   (continuous f (top_of_metric({x | a <= x /\ x <= b},d_real))
        (top_of_metric( UNIV,d_real)))  /\
    (INJ f {x | a <= x /\ x <= b} UNIV) ==>
   (?c d. (c < d) /\ ({ c , d} = {(f a),(f b)}) /\
    (IMAGE f {x | a <= x /\ x <= b} =
       {x | c <= x /\ x <= d})
     )`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `?g. (continuous g (top_of_metric(UNIV,d_real))  (top_of_metric(UNIV,d_real))) /\ (!x. (a <= x /\ x <= b) ==> (f x = g x))` SUBGOAL_TAC;
  IMATCH_MP_TAC  cont_extend_real_lemma;
  ASM_REWRITE_TAC[metric_real];
  REP_BASIC_TAC;
  TYPE_THEN `(a < b) /\ (continuous g (top_of_metric(UNIV,d_real))  (top_of_metric( UNIV,d_real)))  /\ (INJ g {x | a <= x /\ x <= b} UNIV)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `INJ g {x | a <= x /\ x <= b} UNIV= INJ f {x | a <= x /\ x <= b} UNIV` SUBGOAL_TAC;
  IMATCH_MP_TAC  inj_domain_sub;
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP image_interval t));
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `c` EXISTS_TAC;
  TYPE_THEN `d` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  TYPE_THEN `(f a = g a) /\ (f b = g b)` SUBGOAL_TAC;
  UND 3;
  UND 2;
  MESON_TAC[REAL_ARITH `(a < b) ==> (a<= a /\ a <= b /\ b <= b)`];
  DISCH_THEN_REWRITE;
  USE 5 SYM;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  image_domain_sub;
  ASM_REWRITE_TAC[];
  (* Wed Aug 11 12:51:52 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_arc_euclid = prove_by_refinement(
  `!C. (simple_arc top2 C ==> (C SUBSET (euclid 2)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  USE 0 (MATCH_MP simple_arc_compact);
  RULE_ASSUM_TAC (REWRITE_RULE[compact;top2_unions]);
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let simple_arc_end_inj = prove_by_refinement(
  `!A B C v v'. (simple_arc_end A v v' /\ simple_arc_end B v v') /\
     (simple_arc top2 C) /\ (A SUBSET C) /\ (B SUBSET C) ==>
     (A = B)`,
  (* {{{ proof *)
  [
  (* A: *)
  REWRITE_TAC[simple_arc_end];
  REP_BASIC_TAC;
  TYPE_THEN `simple_arc (top_of_metric(euclid 2,d_euclid)) C /\ (metric_space(euclid 2,d_euclid))` SUBGOAL_TAC;
  ASM_REWRITE_TAC[GSYM top2;metric_euclid];
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP   simple_arc_coord t));
  REP_BASIC_TAC;
  (* push to reals *)
  TYPE_THEN `(IMAGE f'' A = IMAGE f'' B) <=> (A = B)` SUBGOAL_TAC;
  IMATCH_MP_TAC  INJ_IMAGE ;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  (* -- *)
  TYPE_THEN `C SUBSET (euclid 2)` SUBGOAL_TAC;
  IMATCH_MP_TAC simple_arc_euclid;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `metric_space (C,d_euclid )` SUBGOAL_TAC;
  ASM_MESON_TAC[metric_subspace;metric_euclid];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `{x | &0 <= x /\ x <= &1} SUBSET UNIV` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET_UNIV];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `metric_space ({x | &0 <= x /\ x <= &1},d_real)` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_subspace;
  TYPE_THEN `UNIV:real->bool` EXISTS_TAC ;
  ASM_REWRITE_TAC[metric_real];
  DISCH_TAC;
  (* -- *)
  (* -- *)
  TYPE_THEN `g = f'' o f` ABBREV_TAC ;
  TYPE_THEN `g'= f'' o f'` ABBREV_TAC ;
  TYPE_THEN `top_of_metric({x| &0 <= x /\ x <= &1},d_real) = induced_top(top_of_metric(UNIV,d_real)) {x | &0 <= x /\ x <= &1}` SUBGOAL_TAC;
  IMATCH_MP_TAC  (GSYM top_of_metric_induced);
  ASM_REWRITE_TAC[metric_real];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `continuous f (top_of_metric({x| &0 <= x /\ x<= &1},d_real)) top2` SUBGOAL_TAC;
  ASM_REWRITE_TAC[top2 ];
  IMATCH_MP_TAC  continuous_induced_domain;
  ASM_SIMP_TAC [GSYM top2; GSYM top_of_metric_unions; metric_real];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `continuous f' (top_of_metric({x| &0 <= x /\ x<= &1},d_real)) top2` SUBGOAL_TAC;
  ASM_REWRITE_TAC[top2 ];
  IMATCH_MP_TAC  continuous_induced_domain;
  ASM_SIMP_TAC [GSYM top2; GSYM top_of_metric_unions; metric_real];
  DISCH_TAC;
  KILL 11;
  KILL 6;
  (* A *)
  TYPE_THEN `(&0 < &1) /\ (continuous g (top_of_metric({x | &0 <= x /\ x <= &1},d_real))  (top_of_metric( UNIV,d_real)))  /\ (INJ g {x | &0 <= x /\ x <= &1} UNIV)` SUBGOAL_TAC;
  CONJ_TAC;
  REAL_ARITH_TAC;
  CONJ_TAC;
  EXPAND_TAC "g";
  IMATCH_MP_TAC  continuous_comp;
  TYPE_THEN `top_of_metric(C,d_euclid)` EXISTS_TAC;
  USE 22 GSYM;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  CONJ_TAC;
  UND 1;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  continuous_range;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[GSYM top2];
  ASM_SIMP_TAC[metric_euclid];
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  SUBCONJ_TAC;
  UND 1;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* --2-- *)
  EXPAND_TAC "g";
  IMATCH_MP_TAC  (REWRITE_RULE[GSYM comp_comp] COMP_INJ);
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  inj_subset;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 1;
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t -> ASSUME_TAC (MATCH_MP image_interval2 t));
  REP_BASIC_TAC;
  (* -- *)
  ASM_REWRITE_TAC[];
  REWRITE_TAC[GSYM IMAGE_o];
  ASM_REWRITE_TAC[];
  (* B *)
    TYPE_THEN `(&0 < &1) /\ (continuous g' (top_of_metric({x | &0 <= x /\ x <= &1},d_real))  (top_of_metric( UNIV,d_real)))  /\ (INJ g' {x | &0 <= x /\ x <= &1} UNIV)` SUBGOAL_TAC;
  CONJ_TAC;
  REAL_ARITH_TAC;
  CONJ_TAC;
  EXPAND_TAC "g'";
  IMATCH_MP_TAC  continuous_comp;
  TYPE_THEN `top_of_metric(C,d_euclid)` EXISTS_TAC;
  USE 22 GSYM;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  CONJ_TAC;
  UND 0;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  continuous_range;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[GSYM top2];
  ASM_SIMP_TAC[metric_euclid];
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  SUBCONJ_TAC;
  UND 0;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* --2-- *)
  EXPAND_TAC "g'";
  IMATCH_MP_TAC  (REWRITE_RULE[GSYM comp_comp] COMP_INJ);
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  inj_subset;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 0;
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t -> ASSUME_TAC (MATCH_MP image_interval2 t));
  REP_BASIC_TAC;
  (* C final steps *)
  TYPE_THEN `(g (&0) = g'(&0)) /\ (g(&1) = g'(&1))` SUBGOAL_TAC;
  EXPAND_TAC "g";
  EXPAND_TAC "g'";
  REWRITE_TAC[o_DEF ];
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  UND 11;
  ASM_REWRITE_TAC[];
  (* temp *)
  DISCH_TAC;
  TYPE_THEN `(c = min_real (g'(&0)) (g'(&1))) /\ (d = max_real(g'(&0)) (g'(&1)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  pair_order_endpoint;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `(c' = min_real (g'(&0)) (g'(&1))) /\ (d' = max_real(g'(&0)) (g'(&1)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  pair_order_endpoint;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  (* Wed Aug 11 15:10:02 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_arc_end_cut = prove_by_refinement(
  `!C v v' v''. simple_arc_end C v v' /\ (C v'') /\ ~(v'' = v) /\
    ~(v'' = v') ==>
    (?C' C''. (simple_arc_end C' v v'') /\ (simple_arc_end C'' v'' v') /\
     (C' INTER C'' = {v''}) /\ (C' UNION C'' = C))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc_end];
  REP_BASIC_TAC;
  (* -- INTER *)
  TYPE_THEN `?t. (&0 <= t /\ t <= &1 /\ (f t = v''))` SUBGOAL_TAC;
  UND 2;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE];
   MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `IMAGE f {x | &0 <= x /\ x <= t}` EXISTS_TAC;
  TYPE_THEN `IMAGE f {x | t <= x /\ x <= &1}` EXISTS_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `IMAGE f {x | &0 <= x /\ x <= t} INTER IMAGE f {x | t <= x /\ x <= &1} = IMAGE f ({x | &0 <= x /\ x <= t} INTER  {x | t <= x /\ x <= &1})` SUBGOAL_TAC;
  IMATCH_MP_TAC (GSYM inj_inter );
  TYPE_THEN `{x | &0 <= x /\ x <= &1}` EXISTS_TAC;
  TYPE_THEN `(euclid 2)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  UND 9;
  UND 10;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `{x | &0 <= x /\ x <= t} INTER {x | t <= x /\ x <= &1} = {t}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;INR IN_SING];
  UND 9;
  UND 10;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[image_sing];
  ASM_REWRITE_TAC[];
  (* A UNION *)
  REWRITE_TAC[GSYM IMAGE_UNION];
  TYPE_THEN `{x | &0 <= x /\ x <= t} UNION {x | t <= x /\ x <= &1} = {x | &0 <= x /\ x <= &1}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION;];
  UND 9;
  UND 10;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  (* B FIRST piece *)
  CONJ_TAC;
  TYPE_THEN `continuous f (top_of_metric (UNIV,d_real)) top2 /\ INJ f {x | &0 <= x /\ x <= t} (euclid 2) /\ &0 < &1 /\ &0 < t` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x <= &1}` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  UND 9;
  REAL_ARITH_TAC;
  TYPE_THEN `~(&0 = t)` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  REWR 11;
  REWR 4;
  UND 10;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP arc_reparameter_gen t));
  REP_BASIC_TAC;
  TYPE_THEN `g` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  (* C LAST piece  *)
  TYPE_THEN `continuous f (top_of_metric (UNIV,d_real)) top2 /\ INJ f {x | t <= x /\ x <= &1} (euclid 2) /\ &0 < &1 /\ t < &1` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x <= &1}` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  UND 10;
  REAL_ARITH_TAC;
  TYPE_THEN `~( &1 = t)` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  REWR 11;
  REWR 3;
  UND 9;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP arc_reparameter_gen t));
  REP_BASIC_TAC;
  TYPE_THEN `g` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  (* Wed Aug 11 15:54:37 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_closed_curve_pt = prove_by_refinement(
  `!C  v. (simple_closed_curve top2 C /\ C v) ==>
    (?f. (C = IMAGE f {x | &0 <= x /\ x <= &1}) /\
               continuous f (top_of_metric (UNIV,d_real)) top2 /\
               INJ f {x | &0 <= x /\ x < &1} (UNIONS top2) /\
               (f (&0) = v) /\
               (f (&0) = f (&1)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_closed_curve];
  REP_BASIC_TAC;
  TYPE_THEN `f(&0) = v` ASM_CASES_TAC;
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  (* -- *)
  TYPE_THEN `?t. (&0 <= t /\ t <= &1 /\ (f t = v))` SUBGOAL_TAC;
  UND 0;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `~(t = &0)` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  REWR 9;
  REWR 6;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `~(t = &1)` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `{x | t <= x /\ x <= &1} = {x | t <= x /\ x < &1} UNION {(&1)}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION;INR IN_SING];
  UND 7;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `INJ f {x | t <= x /\ x <= &1} (euclid 2)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  inj_split;
  CONJ_TAC;
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x < &1}` EXISTS_TAC ;
  ASM_REWRITE_TAC[GSYM top2_unions];
  REWRITE_TAC[SUBSET];
  UND 8;
  REAL_ARITH_TAC;
  CONJ_TAC;
  REWRITE_TAC[INJ;INR IN_SING;];
  USE 2 (REWRITE_RULE[top2_unions]);
  TYPE_THEN `euclid 2 (f (&0))` SUBGOAL_TAC;
  USE 2 (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  REAL_ARITH_TAC;
  ASM_REWRITE_TAC[];
  MESON_TAC[];
  REWRITE_TAC[EQ_EMPTY;IMAGE;INTER;image_sing;INR IN_SING;];
  NAME_CONFLICT_TAC;
  CONV_TAC (dropq_conv "x''");
  REP_GEN_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `x' = &0` SUBGOAL_TAC;
  USE 2(REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 14;
  UND 8;
  REAL_ARITH_TAC;
  UND 14;
  UND 8;
  UND 9;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* [A] reparameter 1st part *)
  TYPE_THEN `(continuous f (top_of_metric (UNIV,d_real)) top2) /\   (INJ f {x | t <= x /\ x <= &1} (euclid 2)) /\   (&0 < &1/(&2)) /\  (t < &1)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  UND 7;
  UND 10;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP  arc_reparameter_gen t));
  REP_BASIC_TAC;
  KILL 14;
  (* B 2nd part *)
  TYPE_THEN `(continuous f (top_of_metric (UNIV,d_real)) top2) /\   (INJ f {x | &0 <= x /\ x <= t} (euclid 2)) /\   (&1/(&2) < &1) /\  (&0 < t)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_LT_HALF2];
  CONJ_TAC;
  USE 2(REWRITE_RULE[top2_unions]);
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x < &1} ` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET];
  UND 7;
  UND 10;
  REAL_ARITH_TAC;
  UND 8;
  UND 9;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP  arc_reparameter_gen t));
  REP_BASIC_TAC;
  KILL 19;
  (* [C] JOIN functions *)
  TYPE_THEN `joinf g g' (&1/(&2))` EXISTS_TAC;
  TYPE_THEN `&0 < &1/(&2)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `&1/(&2) < &1` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_LT_HALF2];
  REAL_ARITH_TAC ;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `joinf g g' (&1/(&2)) (&0) = v` SUBGOAL_TAC;
  ASM_REWRITE_TAC[joinf];
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `joinf g g' (&1/(&2)) (&1) = v` SUBGOAL_TAC;
  ASM_REWRITE_TAC[joinf];
  ASM_SIMP_TAC[REAL_ARITH `(&1/ &2 < &1) ==> ~(&1 < (&1/(&2)))`];
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  (* -- *)
  TYPE_THEN `continuous (joinf g g' (&1 / &2)) (top_of_metric (UNIV,d_real)) top2` SUBGOAL_TAC;
  REWRITE_TAC[top2];
  IMATCH_MP_TAC  joinf_cont;
  ASM_REWRITE_TAC[GSYM top2];
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  (* [D] INJ *)
  TYPE_THEN `{x | &0 <= x /\ x < &1} = {x | &0 <= x /\ x < (&1/(&2))} UNION {x | (&1/(&2)) <= x /\ x < &1}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  ASM_REWRITE_TAC[UNION];
  UND 24;
  UND 19;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  (* -- *)
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  REWRITE_TAC[top2_unions];
  RULE_ASSUM_TAC (REWRITE_RULE[top2_unions]);
  CONJ_TAC;
  IMATCH_MP_TAC  inj_split;
  TYPE_THEN `INJ (joinf g g' (&1 / &2)) {x | &0 <= x /\ x < &1 / &2} = INJ g {x | &0 <= x /\ x < &1 / &2}` SUBGOAL_TAC;
  IMATCH_MP_TAC  joinf_inj_below;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `INJ (joinf g g' (&1 / &2)) {x | &1 / &2 <= x /\ x < &1} = INJ g' {x | &1 / &2 <= x /\ x < &1}` SUBGOAL_TAC;
  IMATCH_MP_TAC  joinf_inj_above;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE ;
  CONJ_TAC;
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x <= &1/(&2)}` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &1/(&2) <= x /\ x <= &1}` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  (* --2-- E IMAGE *)
  REWRITE_TAC[EQ_EMPTY];
  TYPE_THEN `IMAGE (joinf g g' (&1 / &2)) {x | &0 <= x /\ x < &1 / &2} = IMAGE g {x | &0 <= x /\ x < &1 / &2}` SUBGOAL_TAC;
  IMATCH_MP_TAC  joinf_image_below;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `IMAGE (joinf g g' (&1 / &2)) {x | &1 / &2 <= x /\ x < &1} = IMAGE g' {x | &1 / &2 <= x /\ x < &1}` SUBGOAL_TAC;
  IMATCH_MP_TAC  joinf_image_above;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[INTER];
  GEN_TAC;
  REWRITE_TAC[IMAGE;];
  DISCH_TAC;
  REP_BASIC_TAC;
  REWR 27;
  KILL 30;
  USE 13 (REWRITE_RULE[FUN_EQ_THM ]);
  TSPEC `g x'` 13;
  USE 13 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `(?x. (&0 <= x /\ x <= &1 / &2) /\ (g x' = g x))` SUBGOAL_TAC;
  ASM_MESON_TAC[REAL_ARITH `x' < u ==> x' <= u`];
  DISCH_TAC;
  REWR 13;
  KILL 30;
  REP_BASIC_TAC;
  USE 14 (REWRITE_RULE[FUN_EQ_THM;]);
  TSPEC `g' x''` 14;
  USE 14 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `(?x. (&1 / &2 <= x /\ x <= &1) /\ (g' x'' = g' x))` SUBGOAL_TAC;
  ASM_MESON_TAC[REAL_ARITH `x' < u ==> x' <= u`];
  DISCH_TAC;
  REWR 14;
  KILL 34;
  REP_BASIC_TAC;
  TYPE_THEN `(x = x''')` SUBGOAL_TAC;
  USE 2 (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `~(x = &0)` SUBGOAL_TAC;
  DISCH_TAC;
  TYPE_THEN `g (&1/(&2)) = g (x')` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `&1/(&2) = x'` SUBGOAL_TAC;
  USE 17(REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 31;
  UND 24;
  UND 19;
  REAL_ARITH_TAC;
  UND 31;
  REAL_ARITH_TAC;
  TYPE_THEN `~(x = &1)` SUBGOAL_TAC;
  DISCH_TAC;
  TYPE_THEN `g (&1/(&2)) = g (x')` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `&1/(&2) = x'` SUBGOAL_TAC;
  USE 17(REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 31;
  UND 24;
  UND 19;
  REAL_ARITH_TAC;
  UND 31;
  REAL_ARITH_TAC;
  UND 34;
  UND 7;
  UND 10;
  UND 33;
  UND 8;
  UND 9;
  UND 30;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* --2-- *)
  TYPE_THEN `x = t` SUBGOAL_TAC;
  UND 36;
  UND 35;
  UND 34;
  UND 33;
  UND 30;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `g' (&1) = g'(x'')` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `&1 = x''` SUBGOAL_TAC;
  USE 22(REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 28;
  UND 24;
  UND 19;
  REAL_ARITH_TAC;
  UND 28;
  REAL_ARITH_TAC;
  (* F IMAGE *)
  TYPE_THEN `{x | &0 <= x /\ x <= &1} = {x | &0 <= x /\ x < &1/(&2)} UNION {x | &1/(&2) <= x /\ x <= &1}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION ];
  UND  24;
  UND 19;
  REAL_ARITH_TAC;
  DISCH_TAC;
  TYPEL_THEN [`joinf g g' (&1/(&2))`;`{x | &0 <= x /\ x < &1/(&2)}`;`{x | &1/(&2) <= x /\ x <= &1}`] (fun t-> ASSUME_TAC (ISPECL t IMAGE_UNION ));
  ASM_REWRITE_TAC[];
  USE 27 SYM;
  ASM_REWRITE_TAC[];
  TYPE_THEN `IMAGE (joinf g g' (&1 / &2)) {x | &0 <= x /\ x < &1 / &2} = IMAGE g {x | &0 <= x /\ x < &1 / &2}` SUBGOAL_TAC;
  IMATCH_MP_TAC  joinf_image_below;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `IMAGE (joinf g g' (&1 / &2)) {x | &1 / &2 <= x /\ x <= &1} = IMAGE g' {x | &1 / &2 <= x /\ x <= &1}` SUBGOAL_TAC;
  IMATCH_MP_TAC  joinf_image_above;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  USE 14 GSYM ;
  ASM_REWRITE_TAC[];
  (* F final  *)
  TYPE_THEN `{x | &0 <= x /\ x <= &1} = {x | &0 <= x /\ x < &1} UNION {(&1)}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION;INR IN_SING];
  REAL_ARITH_TAC;
  DISCH_TAC ;
  (* -- *)
  TYPE_THEN `IMAGE f {x | &0 <= x /\ x <= &1} = IMAGE f {x | &0 <= x /\ x < &1}` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE_UNION;image_sing; ];
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[union_subset;SUBSET_REFL];
  REWRITE_TAC[SUBSET;INR IN_SING;];
  GEN_TAC;
  DISCH_THEN_REWRITE;
  UND 1;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  REWRITE_TAC[IMAGE];
  TYPE_THEN `&0` EXISTS_TAC;
  REWRITE_TAC[];
  REAL_ARITH_TAC;
  REWRITE_TAC[SUBSET_UNION];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  (* -- *)
  TYPE_THEN `IMAGE g {x | &0 <= x /\ x < &1/(&2)} = IMAGE f {x | t <= x /\ x < &1}` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `IMAGE f {x | t <= x /\ x <= &1} DELETE (f (&1))` EXISTS_TAC;
  CONJ_TAC;
  ASM_REWRITE_TAC[SUBSET_DELETE];
  CONJ_TAC;
  REWRITE_TAC[IMAGE;];
  REP_BASIC_TAC;
  TYPE_THEN `x = (&1/(&2))` SUBGOAL_TAC;
  USE 17(REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 32;
  UND 19;
  REAL_ARITH_TAC;
  UND 32;
  REAL_ARITH_TAC;
  IMATCH_MP_TAC  IMAGE_SUBSET;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  REWRITE_TAC[DELETE;IMAGE;SUBSET;];
  REWRITE_TAC[REAL_ARITH `x <= &1 <=> (x < &1 \/ (x = &1))`];
  MESON_TAC[];
  (* --2--*)
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `IMAGE g {x | &0 <= x /\ x <= &1/(&2)} DELETE (g (&1/(&2)))` EXISTS_TAC;
  CONJ_TAC;
  USE 13 GSYM;
  USE 15 GSYM;
  ASM_REWRITE_TAC[SUBSET_DELETE];
  CONJ_TAC;
  REWRITE_TAC[IMAGE;];
  REP_BASIC_TAC;
  TYPE_THEN `&1 = x` SUBGOAL_TAC;
  USE 12(REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 32;
  REAL_ARITH_TAC;
  UND 32;
  REAL_ARITH_TAC;
  USE 11 SYM;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  IMAGE_SUBSET;
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  REWRITE_TAC[DELETE;IMAGE;SUBSET;];
  REWRITE_TAC[REAL_ARITH `x <= &1/(&2) <=> (x < &1/(&2) \/ (x = &1/(&2)))`];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  (* G *)
  REWRITE_TAC[GSYM IMAGE_UNION];
  AP_TERM_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION];
  UND 8;
  UND 7;
  UND 10;
  REAL_ARITH_TAC;
  (* -- World's worst proof *)
  (* Thu Aug 12 07:44:29 EDT 2004 *)

  ]);;


  (* }}} *)

let shift_inj = prove_by_refinement(
  `!(f:real->A) X t. (INJ f {x | &0 <= x /\ x < &1} X) /\
          (f (&0) = f(&1)) /\ (&0 < t) ==>
     INJ f {x | t <= x /\ x <= &1} X`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ];
  REP_BASIC_TAC;
  CONJ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `x < &1` ASM_CASES_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 5;
  UND 0;
  REAL_ARITH_TAC;
  TYPE_THEN `x = &1` SUBGOAL_TAC;
  UND 4;
  UND 6;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  USE 1 GSYM;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `((x = &1) /\ (y = &1)) \/ ((x < &1) /\ (y = &1)) \/ ((x = &1) /\ (y < &1)) \/ ((x < &1) /\ (y < &1))` SUBGOAL_TAC;
  UND 5;
  UND 7;
  REAL_ARITH_TAC;
  REP_CASES_TAC;
  ASM_REWRITE_TAC[];
  USE 1 SYM ;
  REWR 4;
  TYPE_THEN `x = &0` SUBGOAL_TAC;
  FIRST_ASSUM  IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 8;
  UND 0;
  REAL_ARITH_TAC;
  UND 8;
  UND 0;
  REAL_ARITH_TAC;
  USE 1 SYM;
  REWR 4;
  TYPE_THEN `y = &0` SUBGOAL_TAC;
  FIRST_ASSUM  IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 6;
  UND 0;
  REAL_ARITH_TAC;
  UND 6;
  UND 0;
  REAL_ARITH_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 6;
  UND 8;
  UND 0;
  REAL_ARITH_TAC;
  (* Thu Aug 12 08:33:16 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_arc_segment = prove_by_refinement(
  `!f u v.
          continuous f (top_of_metric (UNIV,d_real)) top2 /\
              INJ f {x | &0 <= x /\ x < &1} (euclid 2) /\
              (f (&0) = f (&1)) /\
       (&0 <= u /\ u < v /\ v <= &1 /\ (&0 < u \/ v < &1)) ==>
     simple_arc_end (IMAGE f {x | u <= x /\ x <= v}) (f u) (f v)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[simple_arc_end];
  (* -- *)
  TYPE_THEN `(&0 < u) ==> INJ f { x | u <= x /\ x <= &1} (euclid 2)` SUBGOAL_TAC ;
  DISCH_TAC;
  IMATCH_MP_TAC  shift_inj;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `INJ f { x | u <= x /\ x <= v } (euclid 2)`  SUBGOAL_TAC;
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | u <= x /\ x <= &1}` EXISTS_TAC;
  REWR 7;
  ASM_REWRITE_TAC[SUBSET ];
  UND 1;
  REAL_ARITH_TAC;
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `{x | &0 <= x /\ x < &1}` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  UND 0;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `continuous f (top_of_metric (UNIV,d_real)) top2 /\  INJ f {x | u <= x /\ x <= v} (euclid 2) /\  &0 < &1 /\  u < v` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP arc_reparameter_gen t));
  REP_BASIC_TAC;
  TYPE_THEN `g` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* Thu Aug 12 08:55:11 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_closed_cut = prove_by_refinement(
  `!C v v'. (simple_closed_curve top2 C /\ C v /\ C v' /\ ~(v = v')
   ==> (?C' C''. simple_arc_end C' v v' /\ simple_arc_end C'' v v'
      /\ (  C' UNION C'' = C) /\ (C' INTER C'' = {v,v'})))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `simple_closed_curve top2 C /\ C v` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP simple_closed_curve_pt t));
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `?t. (&0 <= t /\ t <= &1 /\ (f(t) = v'))` SUBGOAL_TAC;
  UND 1;
  ASM_REWRITE_TAC[IMAGE];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `t < &1` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `~( t= &1) /\ (t <= &1) ==> (t  < &1)`);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWR 9;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `&0 < t` SUBGOAL_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `~(t = &0) /\ (&0 <= t) ==> (&0 < t)`);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWR 9;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `C' = IMAGE f {x | &0 <= x /\ x <= t}` ABBREV_TAC ;
  TYPE_THEN `C'' = IMAGE f {x | t <= x /\ x <= &1}` ABBREV_TAC ;
  TYPE_THEN `C'` EXISTS_TAC;
  TYPE_THEN `C''` EXISTS_TAC;
  CONJ_TAC;
  EXPAND_TAC "C'";
  EXPAND_TAC "v";
  EXPAND_TAC "v'";
  IMATCH_MP_TAC simple_arc_segment;
  RULE_ASSUM_TAC (REWRITE_RULE[top2_unions]);
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[REAL_ARITH `x <= x`];
  (* -- *)
  CONJ_TAC;
  USE 5 SYM;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "C''";
  EXPAND_TAC "v'";
  IMATCH_MP_TAC  simple_arc_end_symm;
  IMATCH_MP_TAC  simple_arc_segment;
  RULE_ASSUM_TAC (REWRITE_RULE[top2_unions]);
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  (* -- *)
  CONJ_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "C'";
  EXPAND_TAC "C''";
  REWRITE_TAC[GSYM IMAGE_UNION];
  AP_TERM_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION];
  UND 13;
  UND 12;
  REAL_ARITH_TAC;
  (* -- *)
  TYPE_THEN `C'' = IMAGE f {x | t <= x /\ x < &1} UNION IMAGE f {(&1)}` SUBGOAL_TAC;
  REWRITE_TAC[GSYM IMAGE_UNION];
  EXPAND_TAC "C''";
  AP_TERM_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION;INR IN_SING ];
  UND 12;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  (* -- *)
  REWRITE_TAC[UNION_OVER_INTER;image_sing];
  EXPAND_TAC "C'";
  TYPE_THEN `(IMAGE f ({x | &0 <= x /\ x <= t} INTER  {x | t <= x /\ x < &1})) = (IMAGE f {x | &0 <= x /\ x <= t} INTER IMAGE f {x | t <= x /\ x < &1})` SUBGOAL_TAC;
  IMATCH_MP_TAC  inj_inter;
  TYPE_THEN `{x | &0 <= x /\ x < &1}` EXISTS_TAC;
  TYPE_THEN `(UNIONS top2)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  UND 12;
  UND 13;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  (* -- *)
  TYPE_THEN `({x | &0 <= x /\ x <= t} INTER {x | t <= x /\ x < &1}) = {t}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;INR IN_SING];
  UND 13;
  UND 12;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `{(f (&1))} = IMAGE f {(&0)}` SUBGOAL_TAC;
  REWRITE_TAC[image_sing];
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `(IMAGE f ({x | &0 <= x /\ x <= t} INTER  {(&0)})  ) = (IMAGE f {x | &0 <= x /\ x <= t} INTER IMAGE f {(&0)} )` SUBGOAL_TAC;
  IMATCH_MP_TAC  inj_inter;
  TYPE_THEN `{x | &0 <= x /\ x < &1}` EXISTS_TAC;
  TYPE_THEN `UNIONS top2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET;INR IN_SING];
  UND 12;
  UND 13;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> REWRITE_TAC[GSYM t]);
  (* -- *)
  TYPE_THEN `({x | &0 <= x /\ x <= t} INTER {(&0)}) = {(&0)}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;INR IN_SING ];
  UND 11;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[image_sing];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[in_pair];
  REWRITE_TAC[UNION;INR IN_SING];
  ASM_MESON_TAC[];
  (* Thu Aug 12 09:35:48 EDT 2004 *)

  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* SECTION M *)
(* ------------------------------------------------------------------ *)


let closed_point = prove_by_refinement(
  `!x. (euclid 2 x) ==> (closed_ top2 {x})`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  compact_closed;
  REWRITE_TAC[top2_top];
  ASM_SIMP_TAC[top2;metric_hausdorff;metric_euclid];
  IMATCH_MP_TAC  compact_point;
  ASM_REWRITE_TAC[GSYM top2;top2_unions];
  (* Fri Aug 13 08:42:22 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_arc_end_closed = prove_by_refinement(
  `!C v v'. (simple_arc_end C v v' ==> closed_ top2 C) `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  compact_closed;
  REWRITE_TAC[top2_top];
  ASM_SIMP_TAC[top2;metric_hausdorff;metric_euclid];
  REWRITE_TAC [GSYM top2];
  IMATCH_MP_TAC  simple_arc_compact;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  (* Fri Aug 13 09:33:35 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_arc_end_end = prove_by_refinement(
  `!C v v'. (simple_arc_end C v v' ==> C v)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc_end];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "v";
  REWRITE_TAC[IMAGE;];
  TYPE_THEN `&0` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  (* Fri Aug 13 09:40:59 EDT 2004 *)

  ]);;
  (* }}} *)

let simple_arc_end_end2 = prove_by_refinement(
  `!C v v'. (simple_arc_end C v v' ==> C v')`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc_end];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "v'";
  REWRITE_TAC[IMAGE;];
  TYPE_THEN `&1` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  (* Fri Aug 13 09:42:07 EDT 2004 *)
  ]);;
  (* }}} *)

let simple_arc_end_end_closed = prove_by_refinement(
  `!C v v'. simple_arc_end C v v' ==> closed_ top2 {v}`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  closed_point;
  TYPE_THEN `C SUBSET euclid 2` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_euclid;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  TYPE_THEN `C v` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_end;
  ASM_MESON_TAC[];
  MESON_TAC[ISUBSET];
  ]);;
  (* }}} *)

let simple_arc_end_end_closed2 = prove_by_refinement(
  `!C v v'. simple_arc_end C v v' ==> closed_ top2 {v'}`,
  (* {{{ proof *)

  [
  ASM_MESON_TAC[simple_arc_end_end_closed;simple_arc_end_symm;];
  ]);;

  (* }}} *)

let simple_arc_sep3 = prove_by_refinement(
  `!A C1 C2 C3 x p1 p2 p3.
     (C1 UNION C2 UNION C3 SUBSET A) /\
     (simple_arc_end C1 x p1) /\ ~(C1 p2) /\ ~(C1 p3) /\
     (simple_arc_end C2 x p2) /\ ~(C2 p1) /\ ~(C2 p3) /\
     (simple_arc_end C3 x p3) /\ ~(C3 p1) /\ ~(C3 p2) ==>
     (?x' C1' C2' C3'.
     (C1' UNION C2' UNION C3' SUBSET A) /\
     (simple_arc_end C1' x' p1) /\
     (simple_arc_end C2' x' p2) /\
     (simple_arc_end C3' x' p3) /\
     ~(C2' p3) /\ ~(C3' p2) /\
     (C1' INTER C2' = {x'} ) /\
     (C1' INTER C3' = {x'} ))
     `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `K = C2 UNION C3` ABBREV_TAC ;
  TYPE_THEN `~((C1 INTER K) = EMPTY)` SUBGOAL_TAC;
  EXPAND_TAC "K";
  REWRITE_TAC[EMPTY_EXISTS;INTER ];
  REWRITE_TAC[UNION];
  TYPE_THEN `x` EXISTS_TAC;
  ASM_MESON_TAC[simple_arc_end_end;simple_arc_end_end2];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `closed_ top2 K` SUBGOAL_TAC;
  EXPAND_TAC "K";
  IMATCH_MP_TAC  closed_union;
  ASM_MESON_TAC[simple_arc_end_closed;top2_top];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `~((C1 INTER {p1}) = EMPTY)` SUBGOAL_TAC;
  REWRITE_TAC[INTER;EMPTY_EXISTS;INR IN_SING];
  ASM_MESON_TAC[simple_arc_end_end2];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `(?C1' x' v'. C1' SUBSET C1 /\ simple_arc_end C1' x' v' /\ (C1' INTER K = {x'}) /\ (C1' INTER {p1} = {v'}))` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_restriction;
  ASM_REWRITE_TAC[EQ_EMPTY;INTER;INR IN_SING ];
  CONJ_TAC;
  ASM_MESON_TAC[simple_arc_end_simple];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_end_closed2;
  ASM_MESON_TAC[];
  CONV_TAC (dropq_conv "x");
  REWRITE_TAC[DE_MORGAN_THM];
  DISJ2_TAC;
  EXPAND_TAC "K";
  REWRITE_TAC[UNION];
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `v' = p1` SUBGOAL_TAC;
  USE 14 (REWRITE_RULE[FUN_EQ_THM]);
  USE 14 (REWRITE_RULE[INTER;INR IN_SING]);
  ASM_MESON_TAC[];
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  KILL 14;
  (* -- *)
  (* [A] case x' = x *)
  TYPE_THEN `x' = x` ASM_CASES_TAC;
  UND 14;
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  TYPE_THEN `x` EXISTS_TAC;
  TYPE_THEN `C1` EXISTS_TAC;
  TYPE_THEN `C2` EXISTS_TAC;
  TYPE_THEN `C3` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `C1' = C1` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_inj;
  TYPE_THEN `C1` EXISTS_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  TYPE_THEN `p1` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET_REFL ];
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  (* --2-- *)
  CONJ_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INTER;INR IN_SING];
  EQ_TAC;
  USE 15 (REWRITE_RULE[FUN_EQ_THM;]);
  USE 14 (REWRITE_RULE[INTER;INR IN_SING]);
  UND 14;
  EXPAND_TAC "K";
  REWRITE_TAC[UNION];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[simple_arc_end_end];
  (* --2'-- *)
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INTER;INR IN_SING];
  EQ_TAC;
  USE 15 (REWRITE_RULE[FUN_EQ_THM;]);
  USE 14 (REWRITE_RULE[INTER;INR IN_SING]);
  UND 14;
  EXPAND_TAC "K";
  REWRITE_TAC[UNION];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[simple_arc_end_end];
  (* B cut C1 at- x'  *)
  TYPE_THEN `~(x' = p1)` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_distinct];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `C1' x'` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_end;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `simple_arc_end C1 x p1 /\ C1 x' /\ ~(x' = x) /\ ~(x' = p1)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  UND 17;
  UND 19;
  MESON_TAC[ISUBSET];
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP simple_arc_end_cut t));
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `C'' = C1'` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_inj;
  TYPE_THEN `C1` EXISTS_TAC;
  TYPE_THEN `x'` EXISTS_TAC;
  TYPE_THEN `p1` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  UND 20;
  SET_TAC[UNION;SUBSET];
  DISCH_THEN (fun t -> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  (* -- *)
  TYPE_THEN `C1 x'` SUBGOAL_TAC;
  UND 19;
  UND 17;
  MESON_TAC[ISUBSET];
  DISCH_TAC;
  (* -- *)
    TYPE_THEN `x'` EXISTS_TAC;
  TYPE_THEN `C1'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ONCE_REWRITE_TAC[union_subset];
  TYPE_THEN `C1' SUBSET A` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1 UNION K ` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET_UNION];
  DISCH_THEN_REWRITE;
  (* [C] C2 x'  *)
  (* ------- *)
  TYPE_THEN `C2 x'` ASM_CASES_TAC;
  TYPE_THEN `simple_arc_end C2 x p2 /\ C2 x' /\ ~(x' = x) /\ ~(x' = p2)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
    ASM_MESON_TAC[];
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP simple_arc_end_cut t));
  REP_BASIC_TAC;
  TYPE_THEN `C2' = C''''` ABBREV_TAC ;
  KILL 30;
  (*---- *)
  TYPE_THEN `C2'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `C2' SUBSET C2` SUBGOAL_TAC;
  USE 26 ( (REWRITE_RULE[FUN_EQ_THM]));
  USE 26 (REWRITE_RULE[UNION]);
  UND 26;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `~C2' p3` SUBGOAL_TAC;
  UND 30;
  UND 3;
  MESON_TAC[ISUBSET];
  DISCH_THEN_REWRITE;
  ONCE_REWRITE_TAC [union_subset];
  TYPE_THEN `C2' SUBSET A` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1 UNION K` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "K";
  REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `C1' INTER C2' = {x'}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;INR IN_SING];
  GEN_TAC;
  EQ_TAC;
  UND 15;
  UND 30;
  EXPAND_TAC "K";
  REWRITE_TAC [eq_sing];
  REWRITE_TAC[INTER;UNION;SUBSET];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[simple_arc_end_end];
  DISCH_THEN_REWRITE;
  (* --[C2]-- branch again for C3 x' -- *)
  TYPE_THEN `C3 x'` ASM_CASES_TAC;
  TYPE_THEN `simple_arc_end C3 x p3 /\ C3 x' /\ ~(x' = x) /\ ~(x' = p3)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP simple_arc_end_cut t));
  REP_BASIC_TAC;
  TYPE_THEN `C3' = C''''''` ABBREV_TAC ;
  KILL 36;
  TYPE_THEN `C3'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `C3' SUBSET C3` SUBGOAL_TAC;
  UND 32;
  SET_TAC[UNION;SUBSET];
  DISCH_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1 UNION K` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "K";
  UND 36;
  REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  CONJ_TAC;
  UND 36;
  UND 0;
  MESON_TAC[ISUBSET];
  TYPE_THEN `C3' x'` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_end;
  ASM_MESON_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING];
  GEN_TAC;
  EQ_TAC;
  UND 15;
  UND 36;
  EXPAND_TAC "K";
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[UNION;SUBSET;INTER];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[];
  (* --[C2']-- now C3 doesn't meet x'. This will be repeated for C2 *)
  (* -- cut C' from {x'} to FIRST point on C3 -- *)
  TYPEL_THEN [`C'`;`{x'}`;`C3`] (fun t->  MP_TAC  (ISPECL t simple_arc_end_restriction));
  DISCH_THEN ANT_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_end_closed;
  ASM_MESON_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_closed;
  ASM_MESON_TAC[];
  CONJ_TAC;
  UND 31;
  REWRITE_TAC[EQ_EMPTY;INTER;INR IN_SING];
  MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC[EMPTY_EXISTS];
  REWRITE_TAC[INTER;INR IN_SING];
  USE 23 (MATCH_MP simple_arc_end_end2);
  UND 23;
  MESON_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  REWRITE_TAC[INTER;INR IN_SING];
  USE 23 (MATCH_MP simple_arc_end_end);
  UND 23;
  USE 2 (MATCH_MP simple_arc_end_end);
  UND 2;
  MESON_TAC[];
  REP_BASIC_TAC;
  (* ---[a] *)
  TYPE_THEN `C3a = C'''''` ABBREV_TAC ;
  KILL 36;
  TYPE_THEN `v = x'` SUBGOAL_TAC;
  USE 33(REWRITE_RULE[FUN_EQ_THM]);
  USE 33(REWRITE_RULE[INTER;INR IN_SING]);
  UND 33;
  MESON_TAC[];
  DISCH_THEN (fun t -> (RULE_ASSUM_TAC  (REWRITE_RULE[t])));
  KILL 33;
  TYPE_THEN `C3a SUBSET C1` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 20;
  SET_TAC[UNION;SUBSET];
  DISCH_TAC;
  TYPE_THEN `C3a SUBSET A /\ simple_arc_end C3a x' v'' /\ ~(C3a p2) /\ (C1' INTER C3a = {(x')}) /\ (C3 INTER C3a = {(v'')}) /\ (~C3a p3)` SUBGOAL_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1 UNION K` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1` EXISTS_TAC;
  REWRITE_TAC[SUBSET_UNION];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET_REFL ];
  CONJ_TAC;
  UND 7;
  UND 33;
  MESON_TAC[ISUBSET];
  CONJ_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INR IN_SING];
  EQ_TAC;
  UND 21;
  UND 35;
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[SUBSET;INTER];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[simple_arc_end_end];
  (* --- *)
  CONJ_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INR IN_SING];
  EQ_TAC;
  UND 32;
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[SUBSET;INTER];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[];
  UND 32;
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  UND 35;
  USE 20 (REWRITE_RULE[FUN_EQ_THM]);
  USE 20 (REWRITE_RULE[UNION]);
  UND 20;
  UND 6;
  MESON_TAC  [ISUBSET];
  KILL 32;
  KILL 33;
  KILL 34;
  KILL 31;
  REP_BASIC_TAC;
  (* --[b] *)
  TYPE_THEN `(v'' = x)` ASM_CASES_TAC;
  FIRST_ASSUM (fun t -> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  TYPE_THEN `C3 UNION C3a` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  ONCE_REWRITE_TAC[union_subset];
  ASM_REWRITE_TAC[];
  UND 9;
  EXPAND_TAC "K";
  REWRITE_TAC[union_subset];
  MESON_TAC[];
  (* --- *)
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  IMATCH_MP_TAC  simple_arc_end_trans;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  REWRITE_TAC[UNION;DE_MORGAN_THM];
  ASM_REWRITE_TAC[];
  (* --- *)
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;UNION;INR IN_SING];
  GEN_TAC;
  EQ_TAC ;
  REWRITE_TAC[LEFT_AND_OVER_OR];
  DISCH_THEN DISJ_CASES_TAC;
  UND 39;
  UND 15;
  EXPAND_TAC "K";
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[INTER;UNION];
  MESON_TAC[];
  UND 39;
  UND 33;
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  UND 33;
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  (* -- *)
  (* --[c] cut off C3b at- v'' *)
  TYPEL_THEN [`C3`;`x`;`p3`;`v''`] (fun t -> MP_TAC (ISPECL t simple_arc_end_cut));
  DISCH_THEN ANT_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  UND 32;
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  PROOF_BY_CONTR_TAC;
  USE 39 (REWRITE_RULE[]);
  FIRST_ASSUM (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  UND 31;
  REWRITE_TAC[];
  UND 32;
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `C3b = C'''''''` ABBREV_TAC ;
  KILL 43;
  TYPE_THEN `C3b SUBSET C3` SUBGOAL_TAC;
  UND 39;
  SET_TAC[UNION;SUBSET];
  DISCH_TAC;
  (* -- [d] EXISTS_TAC *)
  TYPE_THEN `C3a UNION C3b` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS ;
  TYPE_THEN `C1 UNION K` EXISTS_TAC ;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  subset_union_pair;
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 20;
  SET_TAC[UNION;SUBSET];
  EXPAND_TAC "K";
  UND 43;
  REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  (* -- *)
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_trans;
  (* IMATCH_MP_TAC  SUBSET_TRANS;    *)
  TYPE_THEN `v''` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 43;
  UND 32;
  UND 40;
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[INTER;SUBSET];
  MESON_TAC[];
  (* -- *)
  CONJ_TAC;
  REWRITE_TAC[UNION;DE_MORGAN_THM];
  ASM_REWRITE_TAC[];
  UND 43;
  UND 0;
  MESON_TAC[ISUBSET];
  IMATCH_MP_TAC  EQ_EXT ;
  REWRITE_TAC[INTER;UNION;INR IN_SING;LEFT_AND_OVER_OR];
  GEN_TAC;
  EQ_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  FIRST_ASSUM MP_TAC;
  UND 21;
  UND 33;
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  FIRST_ASSUM MP_TAC;
  UND 43;
  UND 15;
  EXPAND_TAC "K";
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[INTER;UNION;SUBSET];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  DISJ1_TAC;
  UND 36;
  MESON_TAC[simple_arc_end_end];
  (* D *)
  TYPE_THEN `C3 x'` SUBGOAL_TAC;
  UND 25;
  UND 15;
  REWRITE_TAC[eq_sing];
  EXPAND_TAC "K";
  REWRITE_TAC[INTER;UNION];
  MESON_TAC[];
  DISCH_TAC;
  (* [E]  back to ONE goal *)
  (* TYPE_THEN `C3 x'` ASM_CASES_TAC; *)
  TYPE_THEN `simple_arc_end C3 x p3 /\ C3 x' /\ ~(x' = x) /\ ~(x' = p3)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
    ASM_MESON_TAC[];
  DISCH_THEN (fun t-> ASSUME_TAC (MATCH_MP simple_arc_end_cut t));
  REP_BASIC_TAC;
  TYPE_THEN `C3' = C''''` ABBREV_TAC ;
  KILL 31;
  (*---- *)
  LEFT_TAC "C3'";
  USE 10 (ONCE_REWRITE_RULE[UNION_COMM]);
  TYPE_THEN `C3'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `C3' SUBSET C3` SUBGOAL_TAC;
  USE 27 ( (REWRITE_RULE[FUN_EQ_THM]));
  USE 27 (REWRITE_RULE[UNION]);
  UND 27;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `~C3' p2` SUBGOAL_TAC;
  UND 31;
  UND 0;
  MESON_TAC[ISUBSET];
  DISCH_THEN_REWRITE;
  ONCE_REWRITE_TAC [union_subset];
  TYPE_THEN `C3' SUBSET A` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1 UNION K` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C3` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "K";
  REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `C1' INTER C3' = {x'}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;INR IN_SING];
  GEN_TAC;
  EQ_TAC;
  UND 15;
  UND 31;
  EXPAND_TAC "K";
  REWRITE_TAC [eq_sing];
  REWRITE_TAC[INTER;UNION;SUBSET];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[simple_arc_end_end];
  DISCH_THEN_REWRITE;
  (* --[XC2]-- now C2 doesn't meet x'. This is repeat. *)
  (* -- cut C' from {x'} to FIRST point on C2 -- *)
  TYPEL_THEN [`C'`;`{x'}`;`C2`] (fun t->  MP_TAC  (ISPECL t simple_arc_end_restriction));
  DISCH_THEN ANT_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_end_closed;
  ASM_MESON_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_closed;
  ASM_MESON_TAC[];
  CONJ_TAC;
  UND 25;
  REWRITE_TAC[EQ_EMPTY;INTER;INR IN_SING];
  MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC[EMPTY_EXISTS];
  REWRITE_TAC[INTER;INR IN_SING];
  USE 23 (MATCH_MP simple_arc_end_end2);
  UND 23;
  MESON_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  REWRITE_TAC[INTER;INR IN_SING];
  USE 23 (MATCH_MP simple_arc_end_end);
  UND 23;
  USE 5 (MATCH_MP simple_arc_end_end);
  UND 5;
  MESON_TAC[];
  REP_BASIC_TAC;
  (* ---[Xa] *)
  TYPE_THEN `C2a = C'''''` ABBREV_TAC ;
  KILL 36;
  TYPE_THEN `v = x'` SUBGOAL_TAC;
  USE 33(REWRITE_RULE[FUN_EQ_THM]);
  USE 33(REWRITE_RULE[INTER;INR IN_SING]);
  UND 33;
  MESON_TAC[];
  DISCH_THEN (fun t -> (RULE_ASSUM_TAC  (REWRITE_RULE[t])));
  KILL 33;
  TYPE_THEN `C2a SUBSET C1` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 20;
  SET_TAC[UNION;SUBSET];
  DISCH_TAC;
  TYPE_THEN `C2a SUBSET A /\ simple_arc_end C2a x' v'' /\ ~(C2a p3) /\ (C1' INTER C2a = {(x')}) /\ (C2 INTER C2a = {(v'')}) /\ (~C2a p2)` SUBGOAL_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1 UNION K` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1` EXISTS_TAC;
  REWRITE_TAC[SUBSET_UNION];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET_REFL ];
  CONJ_TAC;
  UND 6;
  UND 33;
  MESON_TAC[ISUBSET];
  CONJ_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INR IN_SING];
  EQ_TAC;
  UND 21;
  UND 35;
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[SUBSET;INTER];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[simple_arc_end_end];
  (* --- *)
  CONJ_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INR IN_SING];
  EQ_TAC;
  UND 32;
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[SUBSET;INTER];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[];
  UND 32;
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  UND 35;
  USE 20 (REWRITE_RULE[FUN_EQ_THM]);
  USE 20 (REWRITE_RULE[UNION]);
  UND 20;
  UND 7;
  MESON_TAC  [ISUBSET];
  KILL 32;
  KILL 33;
  KILL 34;
  KILL 35;  (*  attention *)
  REP_BASIC_TAC;
  (* --[Xb] *)
  TYPE_THEN `(v'' = x)` ASM_CASES_TAC;
  FIRST_ASSUM (fun t -> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  TYPE_THEN `C2 UNION C2a` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  ONCE_REWRITE_TAC[union_subset];
  ASM_REWRITE_TAC[];
  UND 9;
  EXPAND_TAC "K";
  REWRITE_TAC[union_subset];
  MESON_TAC[];
  (* --- *)
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  IMATCH_MP_TAC  simple_arc_end_trans;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  REWRITE_TAC[UNION;DE_MORGAN_THM];
  ASM_REWRITE_TAC[];
  (* --- *)
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;UNION;INR IN_SING];
  GEN_TAC;
  EQ_TAC ;
  REWRITE_TAC[LEFT_AND_OVER_OR];
  DISCH_THEN DISJ_CASES_TAC;
  UND 39;
  UND 15;
  EXPAND_TAC "K";
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[INTER;UNION];
  MESON_TAC[];
  UND 39;
  UND 34;
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  UND 34;
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  (* -- *)
  (* --[Xc] cut off C3b at- v'' *)
  TYPEL_THEN [`C2`;`x`;`p2`;`v''`] (fun t -> MP_TAC (ISPECL t simple_arc_end_cut));
  DISCH_THEN ANT_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  UND 33;
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  PROOF_BY_CONTR_TAC;
  USE 39 (REWRITE_RULE[]);
  FIRST_ASSUM (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  UND 32;
  REWRITE_TAC[];
  UND 33;
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `C2b = C''''''` ABBREV_TAC ;
  KILL 43;
  TYPE_THEN `C2b SUBSET C2` SUBGOAL_TAC;
  UND 39;
  SET_TAC[UNION;SUBSET];
  DISCH_TAC;
  (* -- [Xd] EXISTS_TAC *)
  TYPE_THEN `C2a UNION C2b` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  REWRITE_TAC[union_subset ];
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS ;
  TYPE_THEN `C1 UNION K` EXISTS_TAC ;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "K";
  REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  (* -- *)
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_trans;
  TYPE_THEN `v''` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 43;
  UND 33;
  UND 40;
  REWRITE_TAC[eq_sing ];
  REWRITE_TAC[INTER;SUBSET];
  MESON_TAC[];
  (* -- *)
  CONJ_TAC;
  REWRITE_TAC[UNION;DE_MORGAN_THM];
  ASM_REWRITE_TAC[];
  UND 43;
  UND 3;
  MESON_TAC[ISUBSET];
  IMATCH_MP_TAC  EQ_EXT ;
  REWRITE_TAC[INTER;UNION;INR IN_SING;LEFT_AND_OVER_OR];
  GEN_TAC;
  EQ_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  FIRST_ASSUM MP_TAC;
  UND 21;
  UND 34;
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[INTER];
  MESON_TAC[];
  FIRST_ASSUM MP_TAC;
  UND 43;
  UND 15;
  EXPAND_TAC "K";
  REWRITE_TAC[eq_sing];
  REWRITE_TAC[INTER;UNION;SUBSET];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  DISJ1_TAC;
  UND 36;
  MESON_TAC[simple_arc_end_end];
  (* Fri Aug 13 17:43:15 EDT 2004 *)

  ]);;

  (* }}} *)


let simple_arc_sep2 = prove_by_refinement(
  `!A C1 C2 C3 x p1 p2 p3.
     (
     C1 UNION C2 UNION C3 SUBSET A /\
     (simple_arc_end C1 x p1) /\
     (simple_arc_end C2 x p2) /\
     (simple_arc_end C3 x p3) /\
     (C1 INTER C2 = {x}) /\
     (C1 INTER C3 = {x}) /\
     ~(C2 p3) /\ ~(C3 p2)) ==>
     (?x' C1' C2' C3'.
     (C1' UNION C2' UNION C3' SUBSET A) /\
     (simple_arc_end C1' x' p1) /\
     (simple_arc_end C2' x' p2) /\
     (simple_arc_end C3' x' p3) /\
     (C1' INTER C2' = {x'}) /\
     (C2' INTER C3' = {x'}) /\
     (C3' INTER C1' = {x'})
     )`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPEL_THEN[`C2`;`C3`;`{p2}`] (fun t -> ANT_TAC (ISPECL t simple_arc_end_restriction));
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_closed;
  ASM_MESON_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_end_closed;
  TYPE_THEN `C2` EXISTS_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_MESON_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  REWRITE_TAC[EQ_EMPTY;INTER;INR IN_SING];
  TYPE_THEN `C2 p2` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_end2];
  TYPE_THEN `C2 x` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_end];
  TYPE_THEN `C3 x` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_end];
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `v' = p2` SUBGOAL_TAC;
  UND 8;
  REWRITE_TAC[eq_sing; INR IN_SING;];
  REWRITE_TAC[INTER;INR IN_SING ];
  MESON_TAC[];
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  KILL 8;
  TYPE_THEN `v` EXISTS_TAC;
  LEFT_TAC "C2'";
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* A easy case *)
  TYPE_THEN `v = x` ASM_CASES_TAC;
  FIRST_ASSUM (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]) THEN REWRITE_TAC[t]);
  TYPE_THEN `C' = C2` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_inj;
  TYPE_THEN `C2` EXISTS_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  TYPE_THEN `p2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  REWRITE_TAC[SUBSET_REFL];
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]) THEN REWRITE_TAC[t]);
  TYPE_THEN `C1` EXISTS_TAC;
  TYPE_THEN `C3` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ONCE_REWRITE_TAC [INTER_COMM];
  ASM_REWRITE_TAC[];
  (* [B] general case *)
  TYPEL_THEN [`C3`;`x`;`p3`;`v`] (fun t-> ANT_TAC (ISPECL t simple_arc_end_cut));
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  UND 9;
  REWRITE_TAC[eq_sing;INTER];
  MESON_TAC[];
  DISCH_TAC;
  FIRST_ASSUM (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  TYPE_THEN `C' p3` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_end];
  UND 1;
  UND 11;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `C1 UNION C''` EXISTS_TAC;
  TYPE_THEN `C'''` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(C1 UNION C'') UNION C' UNION C''' = C1 UNION C' UNION (C'' UNION C''')` SUBGOAL_TAC;
  SET_TAC[UNION];
  DISCH_THEN_REWRITE;
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C1 UNION C2 UNION C3` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC subset_union_pair ;
  REWRITE_TAC[SUBSET_REFL];
  IMATCH_MP_TAC  subset_union_pair ;
  ASM_REWRITE_TAC[SUBSET_REFL];
  (* -- *)
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  IMATCH_MP_TAC  simple_arc_end_trans;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;INR IN_SING ];
  GEN_TAC;
  EQ_TAC ;
  UND 2;
  TYPE_THEN `C'' SUBSET C3` SUBGOAL_TAC;
  UND 12;
  SET_TAC [SUBSET;UNION];
  REWRITE_TAC[eq_sing;INTER;SUBSET];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[simple_arc_end_end;simple_arc_end_end2];
  (* --[a] *)
  TYPE_THEN `(C1 UNION C'') v /\ (C' v) /\ (C''' v)` SUBGOAL_TAC;
  REWRITE_TAC[UNION];
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  DISJ2_TAC;
  ASM_MESON_TAC[simple_arc_end_end2];
  ASM_MESON_TAC[simple_arc_end_end;];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `C''' SUBSET C3` SUBGOAL_TAC;
  UND 12;
  SET_TAC[UNION;SUBSET];
  DISCH_TAC;
  TYPE_THEN `C' INTER C''' = {v}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING];
  GEN_TAC;
  EQ_TAC;
  UND 17;
  UND 9;
  REWRITE_TAC[eq_sing;SUBSET;INTER];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[INTER;];
  DISCH_THEN_REWRITE;
  (* -- *)
  TYPEL_THEN [`C2`;`p2`;`x`;`v`] (fun t-> ANT_TAC(ISPECL t simple_arc_end_cut));
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  UND 11;
  REP_BASIC_TAC;
  UND 11;
  UND 18;
  MESON_TAC[ISUBSET];
  IMATCH_MP_TAC  simple_arc_end_distinct;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `C'''' = C'` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_inj;
  TYPE_THEN `C2` EXISTS_TAC;
  TYPE_THEN `p2` EXISTS_TAC;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  UND 16;
  SET_TAC[UNION;SUBSET];
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  (* -- *)
  TYPE_THEN `~C' x` SUBGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  REWR 24;
  TYPE_THEN `C''''' x` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_end2];
  UND 8;
  UND 18;
  UND 24;
  REWRITE_TAC[eq_sing;INTER;];
  MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  KILL 7;
  KILL 6;
  KILL 5;
  KILL 4;
  TYPE_THEN `C'' x` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_end];
  DISCH_TAC;
  KILL 15;
  KILL 14;
  KILL 20;
  KILL 19;
  (* --[b] *)
  CONJ_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION;INTER;INR IN_SING];
  GEN_TAC;
  EQ_TAC;
  TYPE_THEN `C'' SUBSET C3` SUBGOAL_TAC;
  UND 12;
  SET_TAC[UNION;SUBSET];
  UND 2;
  UND 3;
  UND 11;
  UND 24;
  UND 9;
  REWRITE_TAC[SUBSET;INTER;eq_sing];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  UND 13;
  REWRITE_TAC[eq_sing;INTER];
  MESON_TAC[];
  (* -- *)
  TYPE_THEN `~ (C''' x)` SUBGOAL_TAC;
  DISCH_TAC;
  UND 13;
  UND 5;
  UND 4;
  UND 8;
  REWRITE_TAC[eq_sing;INTER;];
  MESON_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;UNION;INR IN_SING];
  GEN_TAC;
  EQ_TAC ;
  UND 13;
  UND 2;
  UND 17;
  UND 5;
  REWRITE_TAC[SUBSET;INTER;eq_sing];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  UND 23;
  REWRITE_TAC[UNION];
  (* Fri Aug 13 20:36:09 EDT 2004 *)

  ]);;

  (* }}} *)

let simple_arc_sep = prove_by_refinement(
  `!A C1 C2 C3 x p1 p2 p3.
     (C1 UNION C2 UNION C3 SUBSET A) /\
     (simple_arc_end C1 x p1) /\ ~(C1 p2) /\ ~(C1 p3) /\
     (simple_arc_end C2 x p2) /\ ~(C2 p1) /\ ~(C2 p3) /\
     (simple_arc_end C3 x p3) /\ ~(C3 p1) /\ ~(C3 p2) ==>
  (?x' C1' C2' C3'.
     (C1' UNION C2' UNION C3' SUBSET A) /\
     (simple_arc_end C1' x' p1) /\
     (simple_arc_end C2' x' p2) /\
     (simple_arc_end C3' x' p3) /\
     (C1' INTER C2' = {x'}) /\
     (C2' INTER C3' = {x'}) /\
     (C3' INTER C1' = {x'})
     )`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  DISCH_TAC;
  IMATCH_MP_TAC  simple_arc_sep2;
  USE 0 (MATCH_MP simple_arc_sep3);
  REP_BASIC_TAC;
  TYPE_THEN `C1'` EXISTS_TAC;
  TYPE_THEN `C2'` EXISTS_TAC;
  TYPE_THEN `C3'` EXISTS_TAC;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* SECTION N *)
(* ------------------------------------------------------------------ *)

(*  K33 stuff *)

let isthree = prove_by_refinement(
  `?x. (\t. (t < 3)) x`,
  (* {{{ proof *)

  [
  TYPE_THEN `0` EXISTS_TAC;
  BETA_TAC;
  ARITH_TAC;
  (* Sat Aug 14 11:56:32 EDT 2004 *)
  ]);;

  (* }}} *)

let three_t = new_type_definition "three_t" ("ABS3","REP3")
  isthree;;

let type_bij = prove_by_refinement(
  `!X (fXY:A->B) gYX.
     (!a. fXY (gYX a) = a)  /\ (!r. X r = (gYX (fXY r) = r)) ==>
    (BIJ fXY X UNIV) /\ (BIJ gYX UNIV X)`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  bij_inj_image;
  REWRITE_TAC[INJ;SUBSET;IMAGE ;];
  CONJ_TAC;
  REP_BASIC_TAC;
  USE 2 (AP_TERM `gYX:B->A` );
  REWR 3;
  REWR 4;
  REWR 2;
  (* -- *)
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  NAME_CONFLICT_TAC;
  GEN_TAC;
  TYPE_THEN `gYX x''` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* -- *)
  IMATCH_MP_TAC  bij_inj_image;
  REWRITE_TAC[INJ;SUBSET;IMAGE];
  CONJ_TAC;
  REP_BASIC_TAC;
  CONJ_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  USE 2(AP_TERM `fXY:A->B`);
  REWR 2;
  REP_BASIC_TAC;
  TYPE_THEN `fXY x` EXISTS_TAC;
  REWR 2;
  ASM_REWRITE_TAC[];
  ]);;

  (* }}} *)

let thr_bij  = prove_by_refinement(
  `(BIJ ABS3 {x | x < 3} UNIV) /\ (BIJ REP3 UNIV {x | x < 3})`,
  (* {{{ proof *)
  [
  IMATCH_MP_TAC  type_bij ;
  ASSUME_TAC three_t;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[three_t];
  REP_BASIC_TAC;
  UND 0;
  BETA_TAC;
  DISCH_THEN_REWRITE;
  ]);;
  (* }}} *)

let thr_finite = prove_by_refinement(
  `(UNIV:three_t->bool) HAS_SIZE 3`,
  (* {{{ proof *)
  [
  REWRITE_TAC [has_size_bij2];
  TYPE_THEN `REP3` EXISTS_TAC;
  ASM_REWRITE_TAC[thr_bij];
  (* Sat Aug 14 12:28:58 EDT 2004 *)
  ]);;
  (* }}} *)

let has_size3_bij = prove_by_refinement(
  `!(A:A->bool).  A HAS_SIZE 3 <=> (?f. BIJ f (UNIV:three_t->bool) A)`,
  (* {{{ proof *)

  [
  REWRITE_TAC[has_size_bij];
  REP_BASIC_TAC;
  EQ_TAC;
  REP_BASIC_TAC;
  ASSUME_TAC thr_bij;
  TYPE_THEN `compose f REP3` EXISTS_TAC;
  IMATCH_MP_TAC  COMP_BIJ;
  TYPE_THEN `{m | m < 3}` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* -- *)
  REP_BASIC_TAC;
  TYPE_THEN `compose f ABS3` EXISTS_TAC;
  IMATCH_MP_TAC  COMP_BIJ;
  TYPE_THEN `UNIV:three_t->bool` EXISTS_TAC;
  ASM_REWRITE_TAC[thr_bij];
  (* Sat Aug 14 12:36:22 EDT 2004 *)

  ]);;

  (* }}} *)

let has_size3_bij2 = prove_by_refinement(
  `!(A:A->bool). A HAS_SIZE 3 <=> (?f. BIJ f A (UNIV:three_t->bool) )`,
  (* {{{ proof *)
  [
  REWRITE_TAC[has_size_bij2];
  GEN_TAC;
  EQ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `compose ABS3 f` EXISTS_TAC;
  IMATCH_MP_TAC  COMP_BIJ;
  TYPE_THEN `{m | m < 3}` EXISTS_TAC;
  ASM_REWRITE_TAC[thr_bij];
  (* -- *)
  REP_BASIC_TAC;
  TYPE_THEN `compose REP3 f` EXISTS_TAC;
  IMATCH_MP_TAC  COMP_BIJ;
  TYPE_THEN `UNIV:three_t ->bool` EXISTS_TAC;
  ASM_REWRITE_TAC[thr_bij];
  (* Sat Aug 14 12:40:48 EDT 2004 *)

  ]);;
  (* }}} *)

let cartesian = jordan_def
  `cartesian (X:A->bool) (Y:B->bool) =
       { (x,y) | X x /\ Y y}`;;

let cartesian_pair = prove_by_refinement(
  `!X Y (x:A) (y:B).  cartesian X Y (x,y) <=> (X x) /\ (Y y)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[cartesian;PAIR_SPLIT ;];
  MESON_TAC[];
  ]);;
  (* }}} *)

let cartesian_el = prove_by_refinement(
`!X Y (x:(A#B)).  cartesian X Y x  <=> (X (FST x)) /\ (Y (SND x))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[cartesian];
  EQ_TAC;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN`FST x` EXISTS_TAC;
  TYPE_THEN `SND x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

(* ignore earlier K33 def *)

let k33_graph = jordan_def
  `k33_graph = mk_graph_t (
           cartesian (UNIV:three_t ->bool) UNIV,
           cartesian UNIV UNIV,
           (\e. { (FST e,T),  (SND e,F)} ) )`;;

let graph_edge_mk_graph = prove_by_refinement(
  `!(V:A->bool) (E:B->bool) C. graph_edge(mk_graph_t (V,E,C)) = E`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_edge;dest_graph_t;part1;drop0];
  ]);;
  (* }}} *)

let graph_vertex_mk_graph = prove_by_refinement(
 `!(V:A->bool) (E:B->bool) C. graph_vertex(mk_graph_t (V,E,C)) = V`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_vertex;dest_graph_t;];
  ]);;
  (* }}} *)

let graph_inc_mk_graph = prove_by_refinement(
 `!(V:A->bool) (E:B->bool) C. graph_inc(mk_graph_t (V,E,C)) = C`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_inc;dest_graph_t;drop1];
  ]);;
  (* }}} *)

let k33_isgraph = prove_by_refinement(
  `graph (k33_graph)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph;has_size2];
  REWRITE_TAC[IMAGE;SUBSET;];
  NAME_CONFLICT_TAC;
  REWRITE_TAC[k33_graph;graph_inc_mk_graph;graph_edge_mk_graph;graph_vertex_mk_graph;in_pair;cartesian];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[in_pair];
  CONJ_TAC;
  GEN_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  TYPE_THEN `(x,T)` EXISTS_TAC;
  TYPE_THEN `(y,F)` EXISTS_TAC;
  REWRITE_TAC[];
  REWRITE_TAC[PAIR_SPLIT];
  (* Sat Aug 14 13:18:16 EDT 2004 *)

  ]);;
  (* }}} *)

let k33_iso = prove_by_refinement(
  `!(A:A->bool) B (E:B->bool) f.
      A HAS_SIZE 3 /\ B HAS_SIZE 3 /\ (A INTER B = EMPTY) /\
      BIJ f E (cartesian A B) ==>
    (graph_isomorphic k33_graph
         (mk_graph_t
             (A UNION B, E,( \ e. { (FST (f e)), (SND (f e)) }))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[graph_isomorphic;graph_iso;k33_graph;graph_edge_mk_graph;graph_vertex_mk_graph;graph_inc_mk_graph;];
  RULE_ASSUM_TAC (REWRITE_RULE[has_size3_bij]);
  REP_BASIC_TAC;
  TYPE_THEN `u = ( \ t. (if (SND t) then (f'' (FST t)) else (f'(FST t))))` ABBREV_TAC ;
  LEFT_TAC "u";
  TYPE_THEN `u` EXISTS_TAC;
  TYPE_THEN `g = INV f E (cartesian A B)` ABBREV_TAC ;
  TYPE_THEN `v = ( \t . (g (f'' (FST t), f' (SND t))))` ABBREV_TAC ;
  LEFT_TAC "v";
  TYPE_THEN `v` EXISTS_TAC;
  TYPE_THEN `(u,v)` EXISTS_TAC;
  REWRITE_TAC[];
  (* A  u *)
  CONJ_TAC;
  REWRITE_TAC[BIJ;SURJ;INJ];
  SUBCONJ_TAC ;
  CONJ_TAC;
  EXPAND_TAC "u";
  REWRITE_TAC[cartesian_el];
  REWRITE_TAC[UNION;];
  GEN_TAC;
  COND_CASES_TAC;
  UND 2;
  REWRITE_TAC[BIJ;SURJ];
  MESON_TAC[];
  UND 3;
  REWRITE_TAC[BIJ;SURJ];
  MESON_TAC[];
  REWRITE_TAC[cartesian_el;];
  EXPAND_TAC "u";
  REP_GEN_TAC ;
  COND_CASES_TAC;
  COND_CASES_TAC;
  UND 2;
  REWRITE_TAC[BIJ;INJ];
  REP_BASIC_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  UND 1;
  REWRITE_TAC[EMPTY_EXISTS ];
  TYPE_THEN `f'' (FST x)` EXISTS_TAC;
  REWRITE_TAC[INTER];
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  ASM_MESON_TAC[];
  COND_CASES_TAC;
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  UND 1;
  REWRITE_TAC[EMPTY_EXISTS ];
  TYPE_THEN `f' (FST x)` EXISTS_TAC;
  REWRITE_TAC[INTER];
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  ASM_MESON_TAC[];
  REWRITE_TAC[PAIR_SPLIT];
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  USE 3(REWRITE_RULE[BIJ;INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE ;
  REWRITE_TAC[UNION];
  GEN_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  TYPE_THEN `( ((INV f'' UNIV A) x ), T )` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[cartesian_el];
  EXPAND_TAC "u";
  REWRITE_TAC[SND ];
  IMATCH_MP_TAC  inv_comp_right;
  ASM_REWRITE_TAC[];
  TYPE_THEN `( ((INV f' UNIV B) x ), F )` EXISTS_TAC;
  REWRITE_TAC[cartesian_el];
  EXPAND_TAC "u";
  REWRITE_TAC[SND ];
  IMATCH_MP_TAC  inv_comp_right;
  ASM_REWRITE_TAC[];
  (* B graph_inc  *)
  REWRITE_TAC[cartesian_el];
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  CONJ_TAC;
  GEN_TAC;
  EXPAND_TAC "u";
  REWRITE_TAC[IMAGE_CLAUSES];
  EXPAND_TAC "v";
  EXPAND_TAC "g";
  TYPE_THEN `cartesian A B (f'' (FST e), f' (SND e))` SUBGOAL_TAC;
  REWRITE_TAC[cartesian_el];
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  ASM_MESON_TAC[];
  ASM_SIMP_TAC[inv_comp_right];
  (* C  BIJ v *)
  TYPE_THEN `BIJ g (cartesian A B) E` SUBGOAL_TAC;
  EXPAND_TAC "g";
  IMATCH_MP_TAC  INVERSE_BIJ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWRITE_TAC[BIJ];
  SUBCONJ_TAC;
  REWRITE_TAC[INJ];
  REWRITE_TAC[cartesian_el];
  EXPAND_TAC "v";
  CONJ_TAC;
  (* --- *)
  USE 7(REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[cartesian_el];
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;INJ]);
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `(f'' (FST x),f' (SND x)) = (f''(FST y),f' (SND y))` SUBGOAL_TAC;
  USE 7(REWRITE_RULE[BIJ;INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC [cartesian_el];
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  ASM_MESON_TAC[];
  REWRITE_TAC[PAIR_SPLIT];
  REP_BASIC_TAC;
  CONJ_TAC;
  USE 2 (REWRITE_RULE[BIJ;INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  USE 3 (REWRITE_RULE[BIJ;INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* -- *)
  REWRITE_TAC[INJ;SURJ];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[cartesian_el];
  EXPAND_TAC "v";
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `?u0. (f'' u0 = FST (f x))` SUBGOAL_TAC ;
  USE 2 (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC ;
  FIRST_ASSUM IMATCH_MP_TAC ;
  USE 0 (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC;
  TSPEC `x` 11;
  REWR 11;
  USE 11(REWRITE_RULE[cartesian_el]);
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `?u1. (f' u1 = SND (f x))` SUBGOAL_TAC ;
  USE 3 (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC ;
  FIRST_ASSUM IMATCH_MP_TAC ;
  USE 0 (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC;
  TSPEC `x` 12;
  REWR 12;
  USE 12(REWRITE_RULE[cartesian_el]);
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `(u0,u1)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "g";
  IMATCH_MP_TAC  inv_comp_left;
  ASM_REWRITE_TAC[];
  (* Sat Aug 14 14:58:11 EDT 2004 *)

  ]);;
  (* }}} *)


(* ********************************************************* *)

let mk_segment_inj_image2 = prove_by_refinement(
  `!x y n.
    euclid n x /\ euclid n y /\ ~(x = y)
          ==> (?f. continuous f (top_of_metric (UNIV,d_real))
                   (top_of_metric (euclid n,d_euclid)) /\
                   INJ f {x | &0 <= x /\ x <= &1} (euclid n) /\
                   (f (&0) = x) /\ (f (&1) = y) /\
                   (IMAGE f {t | &0 <= t /\ t <= &1} = mk_segment x y))`,
  (* {{{ proof *)
  [
  DISCH_ALL_TAC;
  TYPE_THEN `(joinf (\u. x) (joinf (\t. euclid_plus (t *# y) ((&1 - t) *# x)) (\u. y) (&.1)) (&.0))` EXISTS_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  cont_mk_segment;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[joinf;IMAGE ];
  REWRITE_TAC[mk_segment];
  (* new new *)
  TYPE_THEN `((if &0 < &0   then x   else if &0 < &1 then euclid_plus (&0 *# y) ((&1 - &0) *# x) else y) =  x) /\ ((if &1 < &0   then x   else if &1 < &1 then euclid_plus (&1 *# y) ((&1 - &1) *# x) else y) =  y)` SUBGOAL_TAC;
  REWRITE_TAC[REAL_ARITH `~(&0 < &0) /\ ~(&1 < &0) /\ (&0 < &1) /\ ~(&1 < &1)`];
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale0; euclid_scale_one ; euclid_lzero];
  DISCH_THEN_REWRITE;
  (* end new new *)
  CONJ_TAC;
  (* new stuff *)
  REWRITE_TAC[INJ];
  CONJ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `~(x' < &0)` SUBGOAL_TAC;
  UND 4;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_CASES_TAC `x' < &1`;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  euclid_add_closure;
  CONJ_TAC THEN (IMATCH_MP_TAC  euclid_scale_closure) THEN (ASM_REWRITE_TAC[]);
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  UND 3;
  TYPE_THEN `~(x' < &0)` SUBGOAL_TAC;
  UND 7;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `~(y' < &0)` SUBGOAL_TAC;
  UND 5;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `(if (x' < &1) then (euclid_plus (x' *# y) ((&1 - x') *# x)) else y) = ( euclid_plus (x' *# y) ((&1 - x') *# x))` SUBGOAL_TAC;
 TYPE_THEN `(x' < &1) \/ (x' = &1)` SUBGOAL_TAC;
  UND 6;
  REAL_ARITH_TAC;
  DISCH_THEN   DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `~(x' < &1)` SUBGOAL_TAC;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale_one;euclid_scale0;euclid_rzero];
  DISCH_THEN_REWRITE;

  TYPE_THEN `(if (y' < &1) then (euclid_plus (y' *# y) ((&1 - y') *# x)) else y) = ( euclid_plus (y' *# y) ((&1 - y') *# x))` SUBGOAL_TAC;
 TYPE_THEN `(y' < &1) \/ (y' = &1)` SUBGOAL_TAC;
  UND 4;
  REAL_ARITH_TAC;
  DISCH_THEN   DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `~(y' < &1)` SUBGOAL_TAC;
  UND 3;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale_one;euclid_scale0;euclid_rzero];
  DISCH_THEN_REWRITE;
  (* th *)
  ONCE_REWRITE_TAC [euclid_eq_minus];
  REWRITE_TAC[euclid_minus_scale;euclid_ldistrib;euclid_scale_act];
  ONCE_REWRITE_TAC [euclid_plus_pair];
  REWRITE_TAC[GSYM euclid_rdistrib];
  REDUCE_TAC;
  REWRITE_TAC[REAL_ARITH  `x' + -- &1 * y' = x' - y'`];
  REWRITE_TAC[REAL_ARITH `&1 - x' - (&1 - y') = -- &1 *(x' - y')`];
  REWRITE_TAC[GSYM euclid_scale_act;GSYM euclid_minus_scale;ONCE_REWRITE_RULE[EQ_SYM_EQ] euclid_eq_minus];
  (* th1 *)
  DISCH_TAC;
  PROOF_BY_CONTR_TAC;
  UND 2;
  REWRITE_TAC[];
  IMATCH_MP_TAC  euclid_scale_cancel;
  TYPE_THEN `(x' - y')` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 8;
  REAL_ARITH_TAC;
  KILL 2;
  (* old stuff *)
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  ASM_REWRITE_TAC[];
  EQ_TAC;
  DISCH_TAC;
  CHO 2;
  UND 2;
  COND_CASES_TAC;
  DISCH_ALL_TAC;
  JOIN 3 2;
  ASM_MESON_TAC[REAL_ARITH `~(&0 <=. x /\ x < &.0)`];
  DISCH_ALL_TAC;
  UND 5;
  COND_CASES_TAC;
  DISCH_TAC;
  TYPE_THEN `&1 - x''` EXISTS_TAC;
  SUBCONJ_TAC;
  UND 5;
  REAL_ARITH_TAC ;
  DISCH_TAC;
  CONJ_TAC;
  UND 3;
  REAL_ARITH_TAC ;
  ONCE_REWRITE_TAC [euclid_add_comm];
  REWRITE_TAC[REAL_ARITH `&1 - (&1 - x) = x`];
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `&0` EXISTS_TAC;
  CONJ_TAC;
  REAL_ARITH_TAC ;
  CONJ_TAC;
  REAL_ARITH_TAC ;
  REWRITE_TAC[euclid_scale_one; euclid_scale0; euclid_lzero;REAL_ARITH `&1 - &0 = &1` ];
  (* 2nd half *)
  DISCH_TAC;
  CHO 2;
  TYPE_THEN `&1 - a` EXISTS_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  AND 2;
  AND 2;
  UND 3;
  UND 4;
  REAL_ARITH_TAC ;
  COND_CASES_TAC;
  ASM_MESON_TAC[REAL_ARITH `&1 - a < &0 ==> ~(a <= &1)`];
  COND_CASES_TAC;
  REWRITE_TAC[REAL_ARITH `&1 - (&1 - a) = a`];
  ASM_MESON_TAC [euclid_add_comm];
  TYPE_THEN `a = &.0` SUBGOAL_TAC;
  UND 4;
  UND 3;
  AND 2;
  UND 3;
  REAL_ARITH_TAC ;
  DISCH_TAC;
  REWR 2;
  REWRITE_TAC [euclid_scale_one; euclid_scale0; euclid_lzero;REAL_ARITH `&1 - &0 = &1` ];
   ]);;
  (* }}} *)

let mk_segment_simple_arc_end = prove_by_refinement(
  `!x y.
     (euclid 2 x) /\ (euclid 2 y) /\ ~(x = y) ==>
       simple_arc_end (mk_segment x y) x y`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  REWRITE_TAC[simple_arc_end];
  TYPEL_THEN [`x`;`y`;`2`] (fun t-> ANT_TAC (ISPECL t mk_segment_inj_image2));
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `f` EXISTS_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[GSYM top2 ]);
  ASM_REWRITE_TAC[];
  (* Tue Aug 17 10:10:00 EDT 2004 *)

  ]);;

  (* }}} *)

let cis0 = prove_by_refinement(
  `cis (&0) = e1`,
  (* {{{ proof *)
  [
  REWRITE_TAC[cis;COS_0;SIN_0;e1;];
  ]);;
  (* }}} *)

let cispi2 = prove_by_refinement(
  `cis (pi/(&2)) = e2`,
  (* {{{ proof *)
  [
  REWRITE_TAC [cis;COS_PI2;SIN_PI2;e2];
  ]);;
  (* }}} *)

let neg_point = prove_by_refinement(
  `!x y. -- (point (x,y)) = point (--x, --y)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[euclid_neg];
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  BETA_TAC;
  MP_TAC (ARITH_RULE  `(x' = 0) \/ (x' = 1) \/ (2 <=| x')`);
  REP_CASES_TAC ;
  ASM_REWRITE_TAC[coord01];
  ASM_REWRITE_TAC[coord01];
  TYPE_THEN `euclid 2(point(x,y)) /\ euclid 2(point(--x,--y))` SUBGOAL_TAC;
  ASM_MESON_TAC[euclid_point];
  REWRITE_TAC[euclid];
  REP_BASIC_TAC;
  TSPEC `x'` 1;
  TSPEC `x'` 2;
  ASM_MESON_TAC[REAL_ARITH `-- &0 = &0`];
  (* Tue Aug 17 10:27:14 EDT 2004 *)

  ]);;
  (* }}} *)

let cispi = prove_by_refinement(
  `cis(pi) = -- e1`,
  (* {{{ proof *)
  [
  REWRITE_TAC[cis;COS_PI ;SIN_PI;e1];
  REWRITE_TAC[neg_point];
  AP_TERM_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  REAL_ARITH_TAC;
  (* Tue Aug 17 10:28:55 EDT 2004 *)

  ]);;
  (* }}} *)

let cis3pi2 = prove_by_refinement(
  `cis(&3 *pi/(&2)) = -- e2`,
  (* {{{ proof *)
  [
  TYPE_THEN `&3 *pi/(&2) = pi/(&2) + pi` SUBGOAL_TAC;
  REWRITE_TAC[REAL_ARITH `&3 = &1 + &1 + &1`];
  REWRITE_TAC[REAL_ARITH `(x + y)*z = x*z + y*z`];
  REDUCE_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[cis;COS_PERIODIC_PI;SIN_PERIODIC_PI;GSYM neg_point;];
  AP_TERM_TAC;
  REWRITE_TAC[GSYM cis;cispi2];
  (* Tue Aug 17 10:34:32 EDT 2004 *)

  ]);;
  (* }}} *)

let closedball_convex = prove_by_refinement(
  `!x e n. (convex (closed_ball (euclid n,d_euclid) x e))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[convex;closed_ball;SUBSET;mk_segment;];
  REP_BASIC_TAC;
  USE 0 SYM;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  EXPAND_TAC "x''";
  IMATCH_MP_TAC  (euclid_add_closure);
  CONJ_TAC THEN (IMATCH_MP_TAC  euclid_scale_closure) THEN (ASM_REWRITE_TAC[]);
  DISCH_TAC;
  TYPE_THEN `d_euclid x x'' = d_euclid (a *# x + (&1 - a) *# x) x''` SUBGOAL_TAC;
  REWRITE_TAC[trivial_lin_combo];
  DISCH_THEN_REWRITE;
  EXPAND_TAC "x''";
  (* special case *)
  ASM_CASES_TAC `a = &0` ;
  UND 10;
  DISCH_THEN_REWRITE;
  REDUCE_TAC;
  ASM_REWRITE_TAC [euclid_scale0;euclid_scale_one;euclid_lzero;];
  TYPE_THEN `(!d. (?u v. (d <= u + v) /\ (u <= a*e) /\ (v <= (&1- a)*e))  ==> (d <= e))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `u + v <= (a*e) + (&1 - a)*e` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_LE_ADD2;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[GSYM REAL_RDISTRIB;REAL_ARITH `(a + &1 - a = &1) /\ (&1 * C = C )`];
  UND 13;
  REAL_ARITH_TAC ;
  DISCH_THEN IMATCH_MP_TAC ;
  TYPE_THEN `z = a *# x' + (&1 - a) *# x` ABBREV_TAC;
  TYPE_THEN `d_euclid (a *# x + (&1 - a)*# x) z` EXISTS_TAC;
  TYPE_THEN `d_euclid z x''` EXISTS_TAC;
  TYPE_THEN `euclid n z` SUBGOAL_TAC;
  EXPAND_TAC "z";
  IMATCH_MP_TAC  (euclid_add_closure);
  CONJ_TAC THEN (IMATCH_MP_TAC  euclid_scale_closure) THEN (ASM_REWRITE_TAC[]);
  DISCH_TAC;
  CONJ_TAC;
  EXPAND_TAC "x''";
  IMATCH_MP_TAC  metric_space_triangle;
  TYPE_THEN `euclid n` EXISTS_TAC;
  REWRITE_TAC[metric_euclid];
  ASM_REWRITE_TAC[trivial_lin_combo];
  CONJ_TAC;
  EXPAND_TAC "z";
  TYPE_THEN `(d_euclid (euclid_plus (a *# x) ((&1 - a) *# x)) (euclid_plus (a *# x') ((&1 - a) *# x))) = d_euclid  (a *# x) (a *# x') ` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_translate;
  TYPE_THEN `n` EXISTS_TAC;
  REPEAT (CONJ_TAC THEN TRY (IMATCH_MP_TAC  euclid_scale_closure) THEN ASM_REWRITE_TAC[]);
  DISCH_THEN_REWRITE;
  TYPE_THEN `d_euclid (a *# x) (a *# x')  = abs  (a) * d_euclid x x'` SUBGOAL_TAC;
  IMATCH_MP_TAC  norm_scale_vec;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `abs  a = a` SUBGOAL_TAC;
  ASM_MESON_TAC[REAL_ABS_REFL];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  REAL_PROP_LE_LMUL;
  ASM_REWRITE_TAC[];

  (* LAST case *)
  EXPAND_TAC "z";
  EXPAND_TAC "x''";
  TYPE_THEN `d_euclid (euclid_plus (a *# x') ((&1 - a) *# x)) (euclid_plus (a *# x') ((&1 - a) *# y)) = d_euclid ((&1 - a) *# x) ((&1 - a) *# y)` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_translate_LEFT;
  TYPE_THEN `n` EXISTS_TAC;
  REPEAT (CONJ_TAC THEN TRY (IMATCH_MP_TAC  euclid_scale_closure) THEN ASM_REWRITE_TAC[]);
  DISCH_THEN_REWRITE;
  TYPE_THEN `!b. d_euclid (b *# x) (b *# y)  = abs  (b) * d_euclid x y` SUBGOAL_TAC;
  GEN_TAC;
  IMATCH_MP_TAC  norm_scale_vec;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `abs  (&1 - a) = (&1 - a)` SUBGOAL_TAC;
  REWRITE_TAC [REAL_ABS_REFL];
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  REAL_PROP_LE_LMUL;
  ASM_REWRITE_TAC[];
  UND 1;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let closedball_mk_segment_end = prove_by_refinement(
  `!x e n u v.
     (closed_ball(euclid n,d_euclid) x e u) /\
     (closed_ball(euclid n,d_euclid) x e v) ==>
     (mk_segment u v SUBSET (closed_ball(euclid n,d_euclid) x e))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  ASSUME_TAC closedball_convex;
  TYPEL_THEN [`x`;`e`;`n`] (USE 2 o ISPECL);
  USE 2 (REWRITE_RULE[convex]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let euclid2_e12 = prove_by_refinement(
  `euclid 2 e1 /\ euclid 2 e2`,
  (* {{{ proof *)
  [
  REWRITE_TAC[e1;e2;euclid_point];
  ]);;
  (* }}} *)

let in_union = prove_by_refinement(
  `!X Y Z. (X:A->bool) SUBSET Y \/ (X SUBSET Z) ==> (X SUBSET Y UNION Z)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[SUBSET;UNION ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let mk_segment_hyperplane = prove_by_refinement(
  `!p r i. (i < 4) /\ (&0 <r) /\ (euclid 2 p) ==>
    (mk_segment p (p + r *# (cis(&i * pi/(&2))))) SUBSET
     (hyperplane 2 e2 (p 1) UNION
                     hyperplane 2 e1 (p 0))  `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `?x y. p = point (x,y)` SUBGOAL_TAC;
  USE 0 (MATCH_MP point_onto);
  REP_BASIC_TAC;
  TYPE_THEN `FST p'` EXISTS_TAC;
  TYPE_THEN `SND p'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  UND 3;
  DISCH_THEN (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  REWRITE_TAC[coord01];
  (* -- *)
  TYPE_THEN `convex(hyperplane 2 e2 y)` SUBGOAL_TAC;
  IMATCH_MP_TAC  hyperplane_convex;
  REWRITE_TAC[euclid2_e12];
  TYPE_THEN `convex(hyperplane 2 e1 x)` SUBGOAL_TAC;
  IMATCH_MP_TAC  hyperplane_convex;
  REWRITE_TAC[euclid2_e12];
  REWRITE_TAC[convex];
  REP_BASIC_TAC;
  TYPE_THEN `hyperplane 2 e1 x (point(x,y)) /\ hyperplane 2 e2 y (point(x,y))` SUBGOAL_TAC;
  REWRITE_TAC[e1;e2;GSYM line2D_S;GSYM  line2D_F];
  CONJ_TAC;
  TYPE_THEN `(x,y)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(x,y)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  USE 2 (MATCH_MP (ARITH_RULE (`(i < 4) ==> (i = 0) \/ (i = 1) \/ (i = 2) \/ (i = 3)`)));
  (* -- *)
  IMATCH_MP_TAC  in_union;
  TYPE_THEN `z = (euclid_plus (point (x,y)) (r *# cis (&i * pi / &2)))` ABBREV_TAC ;
  TYPE_THEN `hyperplane 2 e2 y z \/ hyperplane 2 e1 x z ==> mk_segment (point (x,y)) z SUBSET hyperplane 2 e2 y \/  mk_segment (point (x,y)) z SUBSET hyperplane 2 e1 x` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  (* -- *)
  TYPE_THEN `( (cis (&i *pi/(&2))) 0 = &0) ==> (hyperplane 2 e1 x z)` SUBGOAL_TAC;
  REWRITE_TAC[e1;GSYM line2D_F];
  EXPAND_TAC "z";
  REWRITE_TAC[cis;coord01];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[point_scale;point_add];
  REDUCE_TAC;
  TYPE_THEN `(x, y+ r*sin (&i *pi/(&2)))` EXISTS_TAC;
  REWRITE_TAC[];
  (* -- *)
  TYPE_THEN `( (cis (&i *pi/(&2))) 1 = &0) ==> (hyperplane 2 e2 y z)` SUBGOAL_TAC;
  REWRITE_TAC[e2;GSYM line2D_S];
  EXPAND_TAC "z";
  REWRITE_TAC[cis;coord01];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[point_scale;point_add];
  REDUCE_TAC;
  TYPE_THEN `(x + r*cos(&i *pi/(&2)) , y)` EXISTS_TAC;
  REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `(cis (&i * pi / &2) 0 = &0) \/ (cis (&i * pi / &2) 1 = &0) ==> hyperplane 2 e2 y z \/ hyperplane 2 e1 x z` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN IMATCH_MP_TAC ;
  UND 2;
  POP_ASSUM_LIST (fun t-> ALL_TAC);
  (* A -- *)
  REP_CASES_TAC;
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  ASM_REWRITE_TAC[cis0;e1;coord01];
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  ASM_REWRITE_TAC[cispi2;e2;coord01];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[REAL_MUL_2];
  REDUCE_TAC;
  ASM_REWRITE_TAC[cispi;e1;coord01;neg_point];
  REDUCE_TAC;
  ASM_REWRITE_TAC[cis3pi2;e2;coord01;neg_point];
  REDUCE_TAC;
  (* Tue Aug 17 11:46:56 EDT 2004 *)

  ]);;
  (* }}} *)

let d_euclid_mk_segment = prove_by_refinement(
  `!n a p q . (&0 <= a) /\ (a <= &1) /\ (euclid n p) /\ (euclid n q) ==>
      (d_euclid p (a*#p + (&1 - a)*#q) = (&1 - a)*(d_euclid p q))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `!z. d_euclid (a*# p + (&1 - a)*# p) z = d_euclid p z` SUBGOAL_TAC;
  REWRITE_TAC[trivial_lin_combo];
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC[GSYM t]);
  TYPE_THEN `d_euclid (euclid_plus (a *# p) ((&1 - a) *# p)) (euclid_plus (a *# p) ((&1 - a) *# q)) = d_euclid ( ((&1 - a) *# p)) ( ((&1 - a) *# q))` SUBGOAL_TAC;
  ASM_MESON_TAC [metric_translate_LEFT;euclid_scale_closure];
  DISCH_THEN_REWRITE;
  TYPE_THEN `d_euclid ((&1 - a) *# p) ((&1 - a) *# q) = abs  (&1- a) * d_euclid p q` SUBGOAL_TAC;
  ASM_MESON_TAC[euclid_scale_closure;norm_scale_vec];
  DISCH_THEN_REWRITE;
  TYPE_THEN `abs  (&1 - a) = (&1 - a)` SUBGOAL_TAC;
  UND 2;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[trivial_lin_combo];
  (* Tue Aug 17 12:24:07 EDT 2004 *)

  ]);;
  (* }}} *)

let mk_segment_eq = prove_by_refinement(
  `! a p x y. ((a*# p + (&1 - a)*# x) = (a *# p + (&1 - a)*# y)) ==>
      (a = &1) \/ (x = y)`,
  (* {{{ proof *)
  [
  ONCE_REWRITE_TAC[euclid_eq_minus];
  REWRITE_TAC[euclid_minus;euclid_plus;euclid0;euclid_scale];
  REP_BASIC_TAC;
  USE 0 (REWRITE_RULE[FUN_EQ_THM]);
  IMATCH_MP_TAC  (TAUT `(~A ==>B) ==> (A \/ B)`);
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  BETA_TAC;
  USE 0 (SPEC `x':num` );
  UND 0;
  REWRITE_TAC[REAL_ARITH  `(a*b + r*c ) - (a*b + r*d) = r*c - r*d`];
  REWRITE_TAC[REAL_ARITH `a*y - a*z = a*(y-z)`];
  REWRITE_TAC[REAL_ENTIRE];
  UND 1;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let mk_segment_endpoint = prove_by_refinement(
  `!p x y n . (d_euclid p x = d_euclid p y) /\ ~(x = y) /\
       (euclid n x) /\ (euclid n y) /\ (euclid n p) ==>
    (mk_segment p x INTER mk_segment p y = {p})`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER;INR IN_SING];
  GEN_TAC;
  (* A -- *)
  EQ_TAC;
  REWRITE_TAC[mk_segment];
  REP_BASIC_TAC;
  UND 5;
  DISCH_THEN (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `~(a' = &1)` SUBGOAL_TAC;
  DISCH_TAC;
  UND 11;
  DISCH_THEN (fun t -> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  UND 5;
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale0;euclid_scale_one;euclid_rzero];
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `(&1- a')*d_euclid p y = (&1- a)*d_euclid p x` SUBGOAL_TAC;
  KILL 4;
  ASM_MESON_TAC[d_euclid_mk_segment];
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  REWR 12;
  (* -- *)
  TYPE_THEN `d_euclid p y = &0` ASM_CASES_TAC;
  TYPE_THEN `p = y` SUBGOAL_TAC;
  ASM_MESON_TAC [d_euclid_zero];
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  ASM_MESON_TAC[d_euclid_zero];
  USE 12 (REWRITE_RULE[REAL_EQ_MUL_RCANCEL]);
  REWR 12;
  TYPE_THEN `a' = a` SUBGOAL_TAC;
  UND 12;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  USE 8 (MATCH_MP mk_segment_eq);
  REWR 8;
  (* -- *)
  DISCH_THEN_REWRITE;
  REWRITE_TAC[mk_segment_end];
  (* Tue Aug 17 14:04:19 EDT 2004 *)

  ]);;
  (* }}} *)

let cases4 = prove_by_refinement(
  `!i j.  (i < j) /\ (j < 4) ==> ((i=0) /\ (j=1))\/ ((i=0) /\ (j=2)) \/
           ((i=0) /\ (j=3)) \/ ((i=1) /\ (j=2)) \/ ((i=1) /\ (j=3)) \/
         ((i=2)/\ (j=3))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `!k. (k < 4) ==> (k = 0) \/ (k =1)\/ (k=2) \/ (k=3)` SUBGOAL_TAC;
  ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `(j = 0) \/ (j = 1) \/ (j = 2) \/ (j = 3)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `~(j=0)` SUBGOAL_TAC;
  UND 1;
  ARITH_TAC;
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  TYPE_THEN `(i < 3)` SUBGOAL_TAC;
  UND 0;
  UND 1;
  ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `(i=0) \/ (i = 1) \/ (i=2)` SUBGOAL_TAC;
  UND 4;
  ARITH_TAC;
  DISCH_TAC;
  JOIN 5 3;
  USE 3 (REWRITE_RULE [RIGHT_AND_OVER_OR;LEFT_AND_OVER_OR]);
  TYPE_THEN `!k. ~((i = k) /\ (j = k))` SUBGOAL_TAC;
  GEN_TAC;
  UND 1;
  ARITH_TAC;
  DISCH_THEN (fun t-> USE 3 (REWRITE_RULE[t]));
  TYPE_THEN `~((i=2) /\ (j = 1))` SUBGOAL_TAC;
  UND 1;
  ARITH_TAC ;
  DISCH_THEN (fun t-> USE 3(REWRITE_RULE[t]));
  ASM_REWRITE_TAC[];
  UND 3;
  REP_CASES_TAC THEN (ASM_REWRITE_TAC[]);
  ]);;
  (* }}} *)

let cis_distinct = prove_by_refinement(
  `!i j r p. (i < 4) /\ (j < 4) /\ ~(i = j) /\ (&0 < r) ==>
        ~((p + r*# (cis(&i * pi/(&2)))) = (p + r*# (cis(&j * pi/(&2)))))`,
  (* {{{ proof *)

  [
  TYPE_THEN `!i j r p. (i < 4) /\ (j < 4) /\ ~(i = j) /\ (i < j) /\ (&0 < r) ==> ~((p + r*# (cis(&i * pi/(&2)))) = (p + r*# (cis(&j * pi/(&2)))))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `!p x y. (euclid_plus p x = euclid_plus p y) ==> (x = y)` SUBGOAL_TAC;
  REWRITE_TAC[euclid_plus];
  REP_BASIC_TAC;
  USE 6 (REWRITE_RULE[FUN_EQ_THM]);
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  TSPEC `x'` 6;
  UND 6;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> USE 0 (MATCH_MP t));
  USE 0 (AP_TERM `( *# ) (&1/r)`);
  USE 0 (REWRITE_RULE [euclid_scale_act]);
  TYPE_THEN `&1/r * r = &1` SUBGOAL_TAC;
  ONCE_REWRITE_TAC [REAL_ARITH `x*y = y*x`];
  ASM_MESON_TAC[REAL_DIV_LMUL;REAL_ARITH `&0 < r ==> ~(r = &0)`];
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  USE 0(REWRITE_RULE[euclid_scale_one]);
  TYPE_THEN `((i=0) /\ (j=1))\/ ((i=0) /\ (j=2)) \/ ((i=0) /\ (j=3)) \/ ((i=1) /\ (j=2)) \/ ((i=1) /\ (j=3)) \/ ((i=2)/\ (j=3))` SUBGOAL_TAC;
  IMATCH_MP_TAC  cases4;
  ASM_REWRITE_TAC[];
  REP_CASES_TAC THEN (FIRST_ASSUM MP_TAC) THEN (DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t;REAL_ARITH `(&1*x=x) /\ (&0*x= &0)`;e1;e2;cis0;cispi;cispi2;cis3pi2;neg_point;point_inj; PAIR_SPLIT; REAL_ARITH `~(&1 = &0) /\ ~(&0 = &1) /\ (-- &0 = &0) /\ ~(&1 = -- &1) /\ ~(-- &1 = &0) /\ ~(&0 = -- &1)`;REAL_MUL_2; REAL_HALF_DOUBLE ]))) THEN (ASM_REWRITE_TAC[]);
  REP_BASIC_TAC;
  TYPE_THEN `( i <| j) \/ (j <| i)` SUBGOAL_TAC;
  UND 2;
  ARITH_TAC;
  REP_CASES_TAC;
  TYPEL_THEN [`i`;`j`;`r`] (USE 5 o ISPECL);
  ASM_MESON_TAC[];
  TYPEL_THEN [`j`;`i`;`r`] (USE 5 o ISPECL);
  ASM_MESON_TAC[];
  (* Tue Aug 17 15:01:38 EDT 2004 *)




  ]);;

  (* }}} *)

let cis_nz = prove_by_refinement(
  `!t. ~(cis(t) = euclid0)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  USE 0 (AP_TERM `norm2`);
  RULE_ASSUM_TAC (REWRITE_RULE[norm2_cis]);
  ASM_MESON_TAC[REAL_ARITH `~(&1= &0)`;norm2_0;];
  ]);;
  (* }}} *)

let polar_nz = prove_by_refinement(
  `!r t. ~(r = &0) ==> ~(r *# cis(t) =euclid0)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  USE 0 (AP_TERM `norm2`);
  RULE_ASSUM_TAC (REWRITE_RULE[norm2_scale_cis]);
  ASM_MESON_TAC[REAL_ARITH `(abs  r = &0) ==> (r = &0)`;norm2_0];
  ]);;
  (* }}} *)

let polar_euclid = prove_by_refinement(
  `!r t. euclid 2 (r *# (cis t))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[cis;point_scale;euclid_point];
  ]);;
  (* }}} *)

let d_euclidpq = prove_by_refinement(
  `!n p q . (euclid n p) /\ (euclid n q) ==> (d_euclid p (p+q) =
      d_euclid q euclid0)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `!z. d_euclid p z = d_euclid (p + euclid0) z` SUBGOAL_TAC;
  REWRITE_TAC[euclid_rzero];
  DISCH_THEN (fun t->ONCE_REWRITE_TAC[t]);
  TYPE_THEN `d_euclid (euclid_plus p euclid0) (euclid_plus p q) = d_euclid euclid0 q` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_translate_LEFT;
  TYPE_THEN `n` EXISTS_TAC;
  ASM_REWRITE_TAC[euclid_euclid0;polar_euclid;];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC metric_space_symm;
  TYPE_THEN `euclid n` EXISTS_TAC ;
  ASM_REWRITE_TAC[metric_euclid;euclid_euclid0;polar_euclid];
  ]);;
  (* }}} *)

let degree4_vertex_hv = prove_by_refinement(
  `!r p. (&0 < r) /\ (euclid 2 p) ==>
    (?C.
        (!i. (i< 4) ==>
           simple_arc_end (C i) p (p + r*# (cis(&i * pi/(&2))))) /\
        (!i. (i < 4) ==>
           (C i = mk_segment p (p + r*# (cis(&i * pi/(&2)))))) /\
        (!i j. (i < 4) /\ (j < 4) /\ (~(i=j)) ==>
           (C i INTER C j = {p})) /\
        (!i. (i < 4) ==>
          (C i INTER {x | r <= d_euclid p x } =
               { (p + r *# (cis(&i* pi/(&2)))) })) /\
        (!i. (i< 4) ==>
           C i SUBSET (closed_ball (euclid 2,d_euclid) p r)) /\
        (!i. (i< 4) ==>
           C i SUBSET (hyperplane 2 e2 (p 1) UNION
                     hyperplane 2 e1 (p 0))))   `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(\i. mk_segment p (euclid_plus p (r *# cis (&i * pi / &2))))` EXISTS_TAC;
  BETA_TAC;
  ASM_REWRITE_TAC[];
  (* -- *)
  TYPE_THEN `!i. ~(r *# cis (&i * pi/(&2)) = euclid0)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  ASM_MESON_TAC[polar_nz;REAL_ARITH `&0 < r ==> ~( r= &0)`];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!i . euclid 2 (r *# cis (&i * pi/(&2)))` SUBGOAL_TAC;
  GEN_TAC;
  REWRITE_TAC[polar_euclid];
  DISCH_TAC;
  (* -- *)
  CONJ_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC   mk_segment_simple_arc_end;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  ASM_SIMP_TAC[euclid_add_closure];
  DISCH_TAC;
  TSPEC `i` 2;
  UND 2;
  TYPE_THEN `z =r *# cis(&i *pi/(&2))` ABBREV_TAC ;
  REWRITE_TAC[euclid0];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  USE 5 (REWRITE_RULE[FUN_EQ_THM ]);
  TSPEC `x` 5;
  UND 5;
  REWRITE_TAC[euclid_plus];
  REAL_ARITH_TAC;
  (* -- *)
  CONJ_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  mk_segment_endpoint;
  TYPE_THEN `2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  TYPE_THEN `!i. d_euclid p (euclid_plus p (r *# cis (&i * pi / &2))) = d_euclid (r *# cis (&i * pi / &2)) euclid0` SUBGOAL_TAC;
  GEN_TAC;
  IMATCH_MP_TAC  d_euclidpq;
  TYPE_THEN `2` EXISTS_TAC;
  ASM_REWRITE_TAC[polar_euclid];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[GSYM norm2];
  REWRITE_TAC[norm2_scale_cis];
  CONJ_TAC;
  IMATCH_MP_TAC  cis_distinct;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[polar_euclid;euclid_add_closure];
  (* [B] *)
  TYPE_THEN `!a q. (euclid 2 q) /\ (&0 <= a) /\ (a <= &1) ==> (d_euclid p (a*#p + (&1 - a)*#(p + q)) = (&1 - a)*(d_euclid p (p + q)))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  d_euclid_mk_segment;
  TYPE_THEN `2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[euclid_add_closure];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!a i. (&0 <= a) /\ (a <= &1) ==> (d_euclid p (a*#p + (&1 - a)*#(p + r *# (cis (&i * pi/(&2))))) = (&1 - a)*r)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `d_euclid p (p + r *# (cis (&i * pi/(&2)))) = norm2 ( r *# (cis (&i * pi/(&2))))` SUBGOAL_TAC;
  REWRITE_TAC[norm2];
  IMATCH_MP_TAC  d_euclidpq;
  TYPE_THEN `2` EXISTS_TAC;
  ASM_REWRITE_TAC[polar_euclid];
  REWRITE_TAC[norm2_scale_cis];
  TYPE_THEN `abs  r = r` SUBGOAL_TAC;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPEL_THEN [`2`;`a`;`p`;`p + (r *# cis (&i * pi / &2))`] (fun t-> ANT_TAC (ISPECL t d_euclid_mk_segment));
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[euclid_add_closure;polar_euclid];
  DISCH_THEN_REWRITE;
  DISCH_THEN_REWRITE;
  REP_BASIC_TAC;
  (* -- *)
  CONJ_TAC;
  REP_BASIC_TAC ;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[mk_segment;INTER;INR IN_SING];
  EQ_TAC;
  REP_BASIC_TAC;
  UND 8;
  DISCH_THEN (fun t->(REWRITE_TAC[t]) THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  TYPEL_THEN [`a`;`i`] (USE 5 o ISPECL);
  REWR 5;
  ASM_REWRITE_TAC[];
  REWR 7;
  TYPE_THEN `&1 * r <= (&1 - a) * r` SUBGOAL_TAC;
  REDUCE_TAC;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[REAL_LE_RMUL_EQ];
  DISCH_TAC;
  TYPE_THEN `a = &0` SUBGOAL_TAC;
  UND 10;
  UND 8;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t->(REWRITE_TAC[t]) THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale0;euclid_scale_one;euclid_lzero];
  DISCH_THEN_REWRITE;
  CONJ_TAC;
  TYPE_THEN `&0` EXISTS_TAC;
  REWRITE_TAC [REAL_ARITH `&0 <= &0 /\ &0 <= &1`];
  REDUCE_TAC;
  REWRITE_TAC[euclid_scale0;euclid_scale_one;euclid_lzero];
  TYPE_THEN `d_euclid p (euclid_plus p (r *# cis (&i * pi / &2))) = d_euclid (r *# cis (&i * pi/(&2))) euclid0` SUBGOAL_TAC;
  IMATCH_MP_TAC  d_euclidpq;
  TYPE_THEN `2` EXISTS_TAC;
  ASM_REWRITE_TAC[polar_euclid];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[GSYM norm2;norm2_scale_cis];
  UND 1;
  REAL_ARITH_TAC;
  (* C-- *)
  CONJ_TAC;
  REP_BASIC_TAC ;
  REWRITE_TAC[SUBSET];
  GEN_TAC;
  REWRITE_TAC[mk_segment;closed_ball];
  REP_BASIC_TAC;
  UND 7;
  DISCH_THEN (fun t->(REWRITE_TAC[t]) THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  TYPEL_THEN [`a`;`i`] (USE 5 o ISPECL);
  REWR 5;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[euclid_add_closure;polar_euclid;euclid_scale_closure];
  ONCE_REWRITE_TAC[REAL_ARITH `x <= y <=> x <= &1*y`];
  IMATCH_MP_TAC  REAL_PROP_LE_RMUL;
  UND 1;
  UND 9;
  REAL_ARITH_TAC;
  (* D-- *)
  REP_BASIC_TAC;
  IMATCH_MP_TAC  mk_segment_hyperplane;
  ASM_REWRITE_TAC[];
  (* Tue Aug 17 17:02:28 EDT 2004 *)

  ]);;
  (* }}} *)

let diff_pow1 = prove_by_refinement(
  `!t x. (( \ x. (t*x)) diffl t) x`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(\ x. (t * x)) = (\x. (t * (\u. (u pow 1)) x))` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  BETA_TAC;
  REWRITE_TAC[POW_1];
  DISCH_THEN_REWRITE;
  TYPE_THEN `((\x. (t * (\u. (u pow 1)) x)) diffl (t* &1)) x ` SUBGOAL_TAC;
  IMATCH_MP_TAC  DIFF_CMUL;
  TYPEL_THEN[`1`;`x`] (fun t-> ASSUME_TAC  (ISPECL t DIFF_POW));
  UND 0;
  REWRITE_TAC[ARITH_RULE `1-1 = 0`;pow];
  REDUCE_TAC;
  BETA_TAC;
  REDUCE_TAC;
  ]);;
  (* }}} *)

let pi_bounds = prove_by_refinement(
  `&3 < pi /\ pi < &22/ (&7)`,
  (* {{{ proof *)
  let tpi = recompute_pi 12 in
  let t3 = INTERVAL_OF_TERM 12 `&3` in
  let t227 = INTERVAL_OF_TERM 12 `&22/(&7)` in
  let th1 = INTERVAL_TO_LESS_CONV t3 tpi in
  let th2 = INTERVAL_TO_LESS_CONV tpi t227 in
  (
  [
  REP_BASIC_TAC;
  ASSUME_TAC th2;
  ASSUME_TAC th1;
  ASM_REWRITE_TAC[];
  ]));;
  (* }}} *)

let sinx_le_x = prove_by_refinement(
  `!x. (&0 <=x) ==> (sin x <= x)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `x = &0` ASM_CASES_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SIN_0;];
  REAL_ARITH_TAC;
  TYPE_THEN `&0 < x` SUBGOAL_TAC;
  UND 0;
  UND 1;
  REAL_ARITH_TAC;
  POP_ASSUM_LIST (fun t-> ALL_TAC);
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `f = ( \ t x. t * x - sin(x))` ABBREV_TAC ;
  TYPE_THEN `!t. (&1 < t) ==> (!x. (&0 < x) ==> (&0 < f t x))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  (* --- *)
  TYPE_THEN `!x. (f t diffl (t - cos x)) x` SUBGOAL_TAC;
  EXPAND_TAC "f";
  GEN_TAC;
  IMATCH_MP_TAC  DIFF_SUB;
  REWRITE_TAC[DIFF_SIN;diff_pow1;];
  DISCH_TAC;
  TYPEL_THEN [`f t`;`&0`;`x'`] (fun t-> ANT_TAC (ISPECL t MVT));
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  REP_BASIC_TAC;
  ASM_MESON_TAC[DIFF_CONT];
  REWRITE_TAC[differentiable];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  UND 6;
  TYPE_THEN `f t (&0) = &0` SUBGOAL_TAC;
  EXPAND_TAC "f";
  REWRITE_TAC[SIN_0];
  REDUCE_TAC;
  DISCH_THEN_REWRITE;
  REDUCE_TAC;
  DISCH_TAC;
  UND 4;
  REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  REAL_LT_MUL;
  ASM_REWRITE_TAC[];
  TSPEC `z` 5;
  TYPE_THEN `l = t - cos z` SUBGOAL_TAC;
  IMATCH_MP_TAC  DIFF_UNIQ;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  UND 3;
  MP_TAC COS_BOUNDS;
  DISCH_TAC;
  TSPEC `z` 3;
  REP_BASIC_TAC;
  UND 5;
  UND 3;
  REAL_ARITH_TAC;
  (* -- *)
  DISCH_TAC;
  IMATCH_MP_TAC  (REAL_ARITH  `~(x < sin x) ==> (sin x <= x)`) ;
  DISCH_TAC;
  TYPE_THEN `&1 < sin x/x` SUBGOAL_TAC;
  ASM_SIMP_TAC[REAL_LT_RDIV_EQ];
  REDUCE_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TSPEC  `(sin x)/x` 2;
  REWR 2;
  TSPEC `x` 2;
  REWR 2;
  UND 2;
  EXPAND_TAC "f";
  (* -- *)
  ASM_SIMP_TAC[REAL_DIV_RMUL;REAL_ARITH `&0 < x ==> ~(x = &0)`];
  REDUCE_TAC;
  (* Tue Aug 17 19:35:13 EDT 2004 *)

  ]);;
  (* }}} *)

let abssinx_lemma = prove_by_refinement(
  `!x. (&0 <= x) ==> ((abs  (sin x)) <= abs  x)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `abs  x = x` SUBGOAL_TAC;
  UND 0;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  TYPE_THEN `x <= pi` ASM_CASES_TAC;
  TYPE_THEN `&0 <= sin x` SUBGOAL_TAC;
  IMATCH_MP_TAC  SIN_POS_PI_LE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `abs  (sin x) = sin x` SUBGOAL_TAC;
  UND 2;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[sinx_le_x];
  IMATCH_MP_TAC  REAL_LE_TRANS;
  TYPE_THEN `&1` EXISTS_TAC;
  CONJ_TAC;
  ASSUME_TAC SIN_BOUNDS;
  TSPEC `x` 2;
  UND 2;
  REAL_ARITH_TAC;
  UND 1;
  TYPE_THEN `&3 < pi` SUBGOAL_TAC;
  REWRITE_TAC[pi_bounds];
  REAL_ARITH_TAC;
  (* Tue Aug 17 22:54:49 EDT 2004 *)

  ]);;
  (* }}} *)

let abssinx_le = prove_by_refinement(
  `!x. abs  (sin x) <= abs  x`,
  (* {{{ proof *)
  [
  GEN_TAC;
  TYPE_THEN `(&0 <= x) \/ (&0 <= -- x)` SUBGOAL_TAC;
  REAL_ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  ASM_MESON_TAC[abssinx_lemma];
  TYPE_THEN `y = --x` ABBREV_TAC ;
  TYPE_THEN `x = --y` SUBGOAL_TAC;
  UND 1;
  REAL_ARITH_TAC;
  DISCH_THEN (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  REWRITE_TAC[SIN_NEG;REAL_ABS_NEG];
  ASM_MESON_TAC[abssinx_lemma];
  (* Tue Aug 17 22:59:20 EDT 2004 *)

  ]);;
  (* }}} *)

let cos_double2 = prove_by_refinement(
  `!x. cos (&2 * x) = &1 - &2 * (sin x pow 2)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  REWRITE_TAC[COS_DOUBLE;GSYM SIN_CIRCLE ];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let sin_half = prove_by_refinement(
  `!x. &2 * (sin (x/(&2)) pow 2) = &1 - cos (x)`,
  (* {{{ proof *)
  [
  GEN_TAC;
  ASSUME_TAC cos_double2;
  TSPEC `x/ &2` 0;
  TYPE_THEN `&2 *(x/(&2)) = x` SUBGOAL_TAC;
  REWRITE_TAC[REAL_MUL_2;];
  REDUCE_TAC;
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let x_diff_y2 = prove_by_refinement(
  `!x y. (x - y) pow 2 = x*x - &2*x*y + y*y`,
  (* {{{ proof *)
  [
  REWRITE_TAC[REAL_POW_2];
  real_poly_tac;
  ]);;
  (* }}} *)

let cosdiff2 = prove_by_refinement(
  `!x y. (cos x - cos y) pow 2 + (sin x - sin y) pow 2 =
         (&2 * sin ((x - y)/(&2))) pow 2`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[POW_MUL];
  TYPE_THEN  `!z. &2 pow 2 * z = &2 *(&2 *z)` SUBGOAL_TAC ;
  REWRITE_TAC[POW_2];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[sin_half];

  TYPE_THEN `cos (x - y) = cos (x + (--y))` SUBGOAL_TAC;
  AP_TERM_TAC;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[COS_ADD ];
  REWRITE_TAC[SIN_NEG;COS_NEG;REAL_ARITH `x - u*(-- v) = x + u*v`];
  REWRITE_TAC[x_diff_y2];
  REWRITE_TAC[POW_2];
  TYPE_THEN `a = cos x` ABBREV_TAC ;
  TYPE_THEN `b = sin x` ABBREV_TAC ;
  TYPE_THEN `a' = cos y` ABBREV_TAC ;
  TYPE_THEN `b' = sin y` ABBREV_TAC ;
  REWRITE_TAC[REAL_ARITH `x*(y-z) = x*y - x*z`];
  TYPE_THEN `&2 * &1 = ((b pow 2) + (a pow 2)) + ((b' pow 2) + (a' pow 2))` SUBGOAL_TAC;
  EXPAND_TAC "a";
  EXPAND_TAC "b";
  EXPAND_TAC "a'";
  EXPAND_TAC "b'";
  REWRITE_TAC[SIN_CIRCLE];
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[POW_2];
  real_poly_tac;
  (* Tue Aug 17 23:38:27 EDT 2004 *)

  ]);;
  (* }}} *)

let d_euclid_cis = prove_by_refinement(
  `!x y. d_euclid (cis x) (cis y) = &2 * (abs  (sin ((x-y)/(&2))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[cis;d_euclid_point;cosdiff2;POW_2_SQRT_ABS;ABS_MUL;];
  REWRITE_TAC[REAL_ARITH `abs  (&2) = &2`];
  (* Tue Aug 17 23:41:30 EDT 2004 *)
  ]);;
  (* }}} *)

let d_euclid_cis_ineq = prove_by_refinement(
  `!x y. d_euclid (cis x) (cis y) <= abs  (x - y)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[d_euclid_cis];
  REP_GEN_TAC;
  IMATCH_MP_TAC  REAL_LE_TRANS;
  TYPE_THEN `&2 * (abs  ((x-y)/(&2)))` EXISTS_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  REAL_PROP_LE_LMUL;
  ASM_REWRITE_TAC[REAL_ARITH `&0 <= &2`;abssinx_le];
  REWRITE_TAC[REAL_ARITH `!z. &2*(abs  z) = abs  (&2 *z)`];
  TYPE_THEN `&2 * ((x - y)/(&2)) = (x - y)` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_DIV_LMUL;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REAL_ARITH_TAC;
  (* Wed Aug 18 06:42:28 EDT 2004 *)

  ]);;
  (* }}} *)

let polar_fg_inj = prove_by_refinement(
  `!f g p. (INJ f {x | &0 <= x /\ x <= &1} UNIV) /\
    (!x. (&0 <= x /\ x <= &1) ==> (&0 <= f x)) /\ (euclid 2 p) ==>
   INJ (\t. p + (f t)*# (cis (g t))) {x | &0 <= x /\ x <= &1} (euclid 2)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[INJ;polar_euclid];
  ASM_SIMP_TAC[euclid_add_closure;polar_euclid];
  REP_BASIC_TAC;
  (* INSERT *)
  TYPE_THEN `(f x *# cis (g x)) = (f y *# cis (g y))` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  USE 3 (REWRITE_RULE[FUN_EQ_THM]);
  TSPEC `x'` 3;
  USE 3(REWRITE_RULE[euclid_plus]);
  UND 3;
  REAL_ARITH_TAC;
  KILL 3;
  DISCH_TAC;
  (* end ins *)
  USE 3 (AP_TERM `norm2`);
  USE 3 (REWRITE_RULE[norm2_scale_cis]);
  TYPE_THEN `&0 <= f x /\ &0 <= f y` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[GSYM REAL_ABS_REFL]);
  REWR 3;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];

  ]);;
  (* }}} *)

let polar_distinct = prove_by_refinement(
  `!f g g'. (INJ f {x | &0 <= x /\ x <= &1} UNIV) /\
    (!x. (&0 <= x /\ x <= &1) ==> (&0 < f x)) /\
    (!x. (&0 <= x /\ x <= &1) ==> (&0 <= g x /\ g x < &2 * pi)) /\
    (!x. (&0 <= x /\ x <= &1) ==> (&0 <= g' x /\ g' x < &2 * pi))
    ==>
    (!x y. (&0 <= x /\ x <= &1 /\ &0 <= y /\ y <= &1 /\
      ((f x)*# (cis (g x)) = (f y)*# (cis (g' y)))) ==>
      (x = y) /\ (g x = g' y)) `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  COPY 0;
  USE 0 (AP_TERM `norm2`);
  USE 0 (REWRITE_RULE[norm2_scale_cis]);
  TYPE_THEN `&0 < f x /\ &0 < f y` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `f x = f y` SUBGOAL_TAC;
  UND 0;
  UND 10;
  UND 11;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  SUBCONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE [INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN  (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  TYPEL_THEN [`g y`;`g' y`;`f y`;`f y`] (fun t-> ANT_TAC (ISPECL t polar_inj));
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[REAL_ARITH `&0 < t ==> &0 <= t`];
  DISCH_THEN DISJ_CASES_TAC;
  PROOF_BY_CONTR_TAC;
  REP_BASIC_TAC;
  UND 13;
  UND 10;
  REAL_ARITH_TAC;
  ASM_REWRITE_TAC[];
  (* Wed Aug 18 07:42:54 EDT 2004 *)

  ]);;
  (* }}} *)

let d_euclid_eq_arg = prove_by_refinement(
  `!r r' x. (d_euclid (r *# (cis x)) (r' *# (cis x)) = abs  (r - r'))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[cis;point_scale;d_euclid_point];
  REWRITE_TAC[GSYM REAL_SUB_RDISTRIB;POW_MUL;GSYM REAL_ADD_LDISTRIB];
  ONCE_REWRITE_TAC [REAL_ARITH `x + y = y + x`];
  REWRITE_TAC[SIN_CIRCLE];
  REDUCE_TAC;
  REWRITE_TAC[POW_2_SQRT_ABS];
  (* Wed Aug 18 08:15:39 EDT 2004 *)
  ]);;
  (* }}} *)

(* not used *)
let one_over_plus1 = prove_by_refinement(
  `!t. (&0 <= t) ==> (t / (&1 + t) <= &1)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  REAL_LE_LDIV;
  UND 0;
  REAL_ARITH_TAC;
  (* Wed Aug 18 08:17:46 EDT 2004 *)

  ]);;
  (* }}} *)

let polar_cont = prove_by_refinement(
  `!p f g. continuous f (top_of_metric(UNIV,d_real))
        (top_of_metric(UNIV,d_real)) /\
     continuous g (top_of_metric(UNIV,d_real))
        (top_of_metric(UNIV,d_real)) /\ (euclid 2 p)  ==>
     continuous (\t. p + (f t) *# cis(g t)) (top_of_metric(UNIV,d_real))
        (top2)`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  DISCH_TAC;
  TYPE_THEN `IMAGE (\t. p + (f t) *# cis(g t)) UNIV SUBSET (euclid 2)` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET;IMAGE ];
  ASM_MESON_TAC[euclid_add_closure;polar_euclid];
  REWRITE_TAC[top2];
  UND 0;
  ASM_SIMP_TAC[SUBSET_UNIV;metric_continuous_continuous;metric_euclid;metric_real];
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  REP_BASIC_TAC;
  RIGHT_TAC "delta";
  DISCH_TAC;
  TYPEL_THEN [`x`;`epsilon/(&2)`] (USE 3 o ISPECL);
  TYPEL_THEN [`x`;`(&1/(&1 + abs  (f x)))*(epsilon/(&2))`] (USE 2 o ISPECL);
  REP_BASIC_TAC;
  TYPE_THEN `&0 < epsilon/(&2)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  DISCH_TAC;
  TYPE_THEN `&0 < &1 / (&1 + abs (f x)) * epsilon / &2` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_PROP_POS_MUL2;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  REAL_LT_DIV;
  REAL_ARITH_TAC;
  DISCH_TAC;
  REWR 3;
  REWR 2;
  REP_BASIC_TAC;
  TYPE_THEN `min_real delta delta'` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[min_real];
  UND 3;
  UND 8;
  COND_CASES_TAC;
  REAL_ARITH_TAC;
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `d_real x y < delta /\ d_real x y < delta'` SUBGOAL_TAC ;
  UND 9;
  REWRITE_TAC[min_real];
  COND_CASES_TAC;
  UND 9;
  REAL_ARITH_TAC;
  UND 9;
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  TSPEC `y` 2;
  TSPEC `y` 7;
  REWR 2;
  REWR 7;
  (* A-- *)
  IMATCH_MP_TAC  REAL_LET_TRANS;
  TYPE_THEN `d_euclid (p + f x *# cis(g x)) (p + f x *# cis(g y)) + d_euclid (p + f x *# cis(g y)) (p + f y *# cis(g y))` EXISTS_TAC;
  TYPE_THEN `!z r x r' x'. d_euclid (p + r *# (cis x)) (p + r' *# (cis x')) = d_euclid (r*# (cis x)) (r' *# (cis x'))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  metric_translate_LEFT;
  TYPE_THEN `2` EXISTS_TAC;
  ASM_REWRITE_TAC[polar_euclid];
  DISCH_THEN_REWRITE;
  (* end of add-on *)
  CONJ_TAC;
  IMATCH_MP_TAC  metric_space_triangle;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_SIMP_TAC[polar_euclid;metric_euclid];
  REWRITE_TAC[d_euclid_eq_arg];
  TYPEL_THEN[`2`;`f x`;`cis (g x)`;`cis (g y)`] (fun t-> ANT_TAC (ISPECL t norm_scale_vec));
  REWRITE_TAC[cis;euclid_point];
  DISCH_THEN_REWRITE;
  TYPE_THEN `!x y z. (x <= z/ &2 /\ y < z/ &2 ==> x + y < z/ &2 + z/ &2)` SUBGOAL_TAC;
  REAL_ARITH_TAC;
  REWRITE_TAC[REAL_HALF_DOUBLE];
  DISCH_THEN IMATCH_MP_TAC ;
  USE 2 (REWRITE_RULE[d_real]);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  REAL_LE_TRANS;
  TYPE_THEN `abs  (f x) * (&1 / (&1 + abs (f x)) * epsilon / &2)` EXISTS_TAC;
  (* B-- *)
  CONJ_TAC;
  IMATCH_MP_TAC  REAL_PROP_LE_LMUL;
  REWRITE_TAC[REAL_MK_NN_ABS];
  IMATCH_MP_TAC (REAL_ARITH `!y. (x <= y /\ y < z) ==> (x <= z)`);
  TYPE_THEN `abs  (g x - g y)` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[d_euclid_cis_ineq];
  USE 7 (REWRITE_RULE[d_real]);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[REAL_ARITH `(x*y*z <= z) <=> ((x*y)*(z) <= &1 * (z))`];
  IMATCH_MP_TAC  REAL_PROP_LE_RMUL;
  CONJ_TAC;
  REWRITE_TAC[real_div];
  REDUCE_TAC;
  REWRITE_TAC[GSYM real_div];
  IMATCH_MP_TAC  REAL_LE_LDIV;
  REAL_ARITH_TAC;
  UND 5;
  REAL_ARITH_TAC;

  ]);;
  (* }}} *)

let lc_bounds = prove_by_refinement(
  `!a b x. (&0 <= x /\ x <= &1) ==> (min_real a b <= x*a + (&1- x)*b) /\
       (x*a + (&1 - x)*b <= max_real a b)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  CONJ_TAC;
  REWRITE_TAC[min_real];
  COND_CASES_TAC;
  ineq_le_tac `a + (&1 - x)*(b - a) = (x*a + (&1- x)*b)`;
  ineq_le_tac `b + x*(a - b) = x*a + (&1- x)*b`;
  REWRITE_TAC[max_real];
  COND_CASES_TAC;
  ineq_le_tac `(x*a + (&1 - x)*b) + (&1 - x)*(a - b) = a`;
  ineq_le_tac `(x*a + (&1 - x)*b) + (x*(b - a)) = b`;
  (* Wed Aug 18 11:52:54 EDT 2004 *)

  ]);;
  (* }}} *)

let min_real_symm = prove_by_refinement(
  `!a b. min_real a b = min_real b a`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[min_real];
  COND_CASES_TAC;
  USE 0 (MATCH_MP (REAL_ARITH `a < b ==> ~(b < a)`));
  ASM_REWRITE_TAC[];
  COND_CASES_TAC;
  ASM_REWRITE_TAC[];
  UND 0;
  UND 1;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let max_real_symm = prove_by_refinement(
  `!a b. max_real a b = max_real b a`,
  (* {{{ proof *)
  [
  REP_GEN_TAC;
  REWRITE_TAC[max_real];
  COND_CASES_TAC;
  USE 0 (MATCH_MP (REAL_ARITH `a < b ==> ~(b < a)`));
  ASM_REWRITE_TAC[];
  COND_CASES_TAC;
  ASM_REWRITE_TAC[];
  UND 0;
  UND 1;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let curve_annulus_lemma = prove_by_refinement(
  `!r g p. (&0 < r) /\ (euclid 2 p) ==>
      (IMAGE (\t. p + (t*r + (&1 - t)*(r/(&2)))*# (cis (g t)))
           {x | &0 <= x /\ x <= &1})
         SUBSET ({ x | (r/(&2) <= d_euclid p x /\
                             d_euclid p x <= r)} )`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE;SUBSET];
  REP_BASIC_TAC;
  UND 2;
  DISCH_THEN (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC  (REWRITE_RULE[t])));
  TYPE_THEN `d_euclid p (euclid_plus p ((x' * r + (&1 - x') * r / &2) *# cis (g x'))) = d_euclid ((x' * r + (&1 - x') * r / &2) *# cis (g x')) euclid0` SUBGOAL_TAC;
  IMATCH_MP_TAC  d_euclidpq;
  TYPE_THEN `2` EXISTS_TAC;
  ASM_REWRITE_TAC[polar_euclid];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[GSYM norm2;norm2_scale_cis];
  TYPE_THEN `r/(&2) < r` SUBGOAL_TAC;
  ASM_MESON_TAC[half_pos];
  DISCH_TAC;
  TYPE_THEN `(min_real (r/(&2)) r = (r/(&2))) /\ (max_real (r/(&2)) r = r)` SUBGOAL_TAC;
  REWRITE_TAC[min_real;max_real];
  ASM_REWRITE_TAC[];
  COND_CASES_TAC;
  UND 2;
  UND 5;
  REAL_ARITH_TAC;
  REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `&0 <= (x' *r + (&1 - x')*(r/(&2)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_LE_TRANS;
  TYPE_THEN `min_real (r/ &2) r` EXISTS_TAC ;
  CONJ_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  (REAL_ARITH `&0 < x ==> &0 <= x`);
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  ONCE_REWRITE_TAC [min_real_symm];
  ASM_MESON_TAC[lc_bounds];
  REWRITE_TAC[GSYM ABS_REFL];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[lc_bounds;min_real_symm;max_real_symm];
  (* Wed Aug 18 12:13:50 EDT 2004 *)

  ]);;

  (* }}} *)

let curve_circle_lemma = prove_by_refinement(
  `!r g p. (&0 < r) /\ (euclid 2 p) ==>
      (((IMAGE (\t. p + (t*r + (&1 - t)*(r/(&2)))*# (cis (g t)))
           {x | &0 <= x /\ x <= &1})
     INTER ({ x |  d_euclid p x <= (r/(&2))})) =
                          { ( p + (r/(&2)) *# (cis (g (&0) ))) })
     `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE;SUBSET;INTER;];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  GEN_TAC;
  (* A *)
  EQ_TAC;
  DISCH_THEN (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  REP_BASIC_TAC;
  CONJ_TAC;
  TYPE_THEN `&0` EXISTS_TAC;
  REDUCE_TAC;
  TYPEL_THEN [`2`;`p`;`(r / &2 *# cis (g (&0)))`] (fun t-> ANT_TAC (ISPECL t d_euclidpq));
  ASM_REWRITE_TAC[polar_euclid];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[GSYM norm2;norm2_scale_cis;];
  IMATCH_MP_TAC  (REAL_ARITH `(x = y) ==> (x <= y)`);
  REWRITE_TAC[ABS_REFL];
  IMATCH_MP_TAC  (REAL_ARITH `(&0 < x) ==> (&0 <= x)`);
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  REP_BASIC_TAC;
  (* B other direction *)
  UND 3;
  DISCH_THEN (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  PROOF_BY_CONTR_TAC;
  UND 2;
  TYPE_THEN `d_euclid p (euclid_plus p ((x' * r + (&1 - x') * r / &2) *# cis (g x'))) = d_euclid ((x' * r + (&1 - x') * r / &2) *# cis (g x')) euclid0` SUBGOAL_TAC;
  IMATCH_MP_TAC  d_euclidpq;
  TYPE_THEN `2` EXISTS_TAC;
  ASM_REWRITE_TAC[polar_euclid];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[GSYM norm2;norm2_scale_cis];
  TYPE_THEN `r/(&2) < r` SUBGOAL_TAC;
  ASM_MESON_TAC[half_pos];
  DISCH_TAC;
  TYPE_THEN `(min_real (r/(&2)) r = (r/(&2))) /\ (max_real (r/(&2)) r = r)` SUBGOAL_TAC;
  REWRITE_TAC[min_real;max_real];
  ASM_REWRITE_TAC[];
  COND_CASES_TAC;
  UND 2;
  UND 6;
  REAL_ARITH_TAC;
  REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `&0 <= (x' *r + (&1 - x')*(r/(&2)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  REAL_LE_TRANS;
  TYPE_THEN `min_real (r/ &2) r` EXISTS_TAC ;
  CONJ_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  (REAL_ARITH `&0 < x ==> &0 <= x`);
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  ONCE_REWRITE_TAC [min_real_symm];
  ASM_MESON_TAC[lc_bounds];
  REWRITE_TAC[GSYM ABS_REFL];
  DISCH_THEN_REWRITE;
  TYPE_THEN `~(x'  = &0)` SUBGOAL_TAC;
  DISCH_TAC;
  UND 7;
  DISCH_THEN (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  UND 3;
  REDUCE_TAC;
  DISCH_TAC;
  TYPE_THEN `&0 < x'` SUBGOAL_TAC;
  UND 7;
  UND 5;
  REAL_ARITH_TAC;
  DISCH_TAC;
  IMATCH_MP_TAC  (REAL_ARITH `a < b ==> ~(b <= a)`);
  ineq_lt_tac `(r/ &2) + x'* (r - (r/(&2))) = (x' * r + (&1 - x') * r / &2)`;
  (* Wed Aug 18 12:41:16 EDT 2004 *)

  ]);;
  (* }}} *)

let curve_simple_lemma = prove_by_refinement(
  `!r g p. (&0 < r) /\ (euclid 2 p) /\
    (continuous g (top_of_metric(UNIV,d_real))
       (top_of_metric(UNIV,d_real))) ==>
   (simple_arc_end
      (IMAGE (\t. p + (t*r + (&1 - t)*(r/(&2)))*# (cis (g t)))
           {x | &0 <= x /\ x <= &1}) (p + (r/(&2))*# (cis (g (&0))))
             (p + (r)*# (cis (g (&1)))))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc_end];
  REP_BASIC_TAC;
  TYPE_THEN `(\t. p + (t*r + (&1 - t)*(r/(&2)))*# (cis (g t)))` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  polar_cont;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[metric_continuous_continuous;metric_real;SUBSET_UNIV];
  REWRITE_TAC[linear_cont];
  IMATCH_MP_TAC  polar_fg_inj;
  ASM_REWRITE_TAC[INJ;SUBSET_UNIV ];
  (* -- *)
  CONJ_TAC;
  REP_BASIC_TAC;
  USE 3 (ONCE_REWRITE_RULE[REAL_ARITH `( x = y) <=> (x - y = &0)`]);
  TYPE_THEN `(x * r + (&1 - x) * r / &2) - (y * r + (&1 - y) * r / &2) = (x - y)*(r - r/(&2)) ` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_TAC;
  REWR 3;
  USE 3(REWRITE_RULE[REAL_ENTIRE]);
  UND 3;
  DISCH_THEN DISJ_CASES_TAC;
  UND 3;
  REAL_ARITH_TAC;
  PROOF_BY_CONTR_TAC;
  UND 3;
  TYPE_THEN `r - r/(&2) = (r/ &2 + r/ &2) - r/ &2` SUBGOAL_TAC;
  REWRITE_TAC[REAL_HALF_DOUBLE];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[REAL_ARITH `(x + x) - x = x`];
  USE 2 (ONCE_REWRITE_RULE  [GSYM REAL_HALF_DOUBLE]);
  USE 2 (REWRITE_RULE[REAL_DIV_LZERO]);
  UND 2;
  REAL_ARITH_TAC;
  (* -- *)
  GEN_TAC;
  DISCH_TAC;
  WITH 3 (MATCH_MP lc_bounds);
  TYPEL_THEN [`r`;`r/ &2`] (USE 4 o ISPECL);
  IMATCH_MP_TAC  REAL_LE_TRANS;
  TYPE_THEN `min_real r (r/ &2)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `r / &2 < r` SUBGOAL_TAC;
  UND 2;
  MESON_TAC [half_pos];
  TYPE_THEN `&0 < r/ (&2)` SUBGOAL_TAC;
  ASM_MESON_TAC[half_pos];
  TYPE_THEN `a = r/ &2` ABBREV_TAC ;
  REWRITE_TAC[min_real];
  COND_CASES_TAC;
  REAL_ARITH_TAC;
  REAL_ARITH_TAC;
  (* Wed Aug 18 14:02:54 EDT 2004 *)

  ]);;
  (* }}} *)

let segpath = jordan_def
  `segpath x y t = t* x + (&1 - t)*y` ;;

let segpathxy = prove_by_refinement(
  `!x y. segpath x y = (\ t. t*x + (&1 - t)*y)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[segpath];
  ]);;
  (* }}} *)

let segpath_lemma = prove_by_refinement(
  `(!x y . (continuous (segpath x y) (top_of_metric(UNIV,d_real))
       (top_of_metric(UNIV,d_real)))) /\
   (!x y b. (&0 <= x /\ x < b /\ &0 <= y /\ y < b ==>
     (!t. &0 <= t /\ t <= &1 ==> &0 <= segpath x y t /\
       segpath x y t < b))) /\
   (!x y x' y' t. (x < x' /\ y < y' /\ &0 <= t /\ t <= &1)
        ==> ~(segpath x y t = segpath x' y' t))`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  CONJ_TAC;
  REP_BASIC_TAC;
  ASM_SIMP_TAC[SUBSET_UNIV;metric_continuous_continuous;metric_real];
  REWRITE_TAC[segpathxy;linear_cont];
  (* -- *)
  CONJ_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[segpath];
  CONJ_TAC;
  IMATCH_MP_TAC  REAL_LE_TRANS;
  TYPE_THEN `min_real x y` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[min_real];
  COND_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[lc_bounds];
  IMATCH_MP_TAC  REAL_LET_TRANS;
  TYPE_THEN `max_real x y` EXISTS_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[lc_bounds];
  REWRITE_TAC[max_real];
  COND_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  (* -- *)
  REWRITE_TAC[segpath];
  REP_BASIC_TAC;
  UND 0;
  REWRITE_TAC[REAL_ARITH `(u + v = u' + v') <=> ((u' - u) + (v' - v) = &0)`];
  REWRITE_TAC[GSYM REAL_SUB_LDISTRIB];
  TYPE_THEN `t = &0` ASM_CASES_TAC;
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  UND 3;
  REAL_ARITH_TAC;
  TYPE_THEN `t = &1` ASM_CASES_TAC;
  ASM_REWRITE_TAC[];
  REDUCE_TAC;
  UND 4;
  REAL_ARITH_TAC;
  (* -- *)
  TYPE_THEN `&0 < t * (x' - x) + (&1 - t)*(y' - y)` SUBGOAL_TAC;
  ineq_lt_tac `&0 + t * (x' - x) + (&1 - t)*(y' - y) = (t*(x' - x) + (&1- t)*(y' - y))` ;
  UND 5;
  UND 1;
  REAL_ARITH_TAC;
  REAL_ARITH_TAC;
  (* Wed Aug 18 14:48:37 EDT 2004 *)

  ]);;

  (* }}} *)

let segpath_end = prove_by_refinement(
  `!x y. ( segpath x y (&0) = y) /\ (segpath x y (&1) = x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[segpath];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let segpath_inj = prove_by_refinement(
  `!x y. ~(x = y) ==> INJ (segpath x y) {t | &0 <= t /\ t <= &1} UNIV`,
  (* {{{ proof *)

  [
  REWRITE_TAC[segpath;INJ;SUBSET_UNIV];
  REP_BASIC_TAC;
  USE 0 (ONCE_REWRITE_RULE[REAL_ARITH `( x = y) <=> (x - y = &0)`]);
  TYPE_THEN `(x' * x + (&1 - x') * y) - (y' * x + (&1 - y') * y) = (x' - y')*(x - y) ` SUBGOAL_TAC;
  real_poly_tac;
  DISCH_TAC;
  REWR 0;
  USE 0(REWRITE_RULE[REAL_ENTIRE]);
  UND 0;
  DISCH_THEN DISJ_CASES_TAC;
  UND 0;
  REAL_ARITH_TAC;
  PROOF_BY_CONTR_TAC;
  UND 0;
  UND 5;
  REAL_ARITH_TAC;
  (* Wed Aug 18 15:15:11 EDT 2004 *)

  ]);;

  (* }}} *)

let degree_vertex_annulus = prove_by_refinement(
  `!n r p xx zz. (&0 < r) /\ (euclid 2 p) /\
    (!j. j < n ==> (&0 <= xx j /\ xx j < &2 * pi)) /\
   (!j. j < n ==> (&0 <= zz j /\ zz j < &2 * pi)) /\
    (!i j. (i < j) /\ (j <| n) ==> (xx i < xx j)) /\
       (!i j. (i < j) /\ (j < n) ==> (zz i < zz j))  ==>
    (?C.
       (!i. (i < n) ==>
          simple_arc_end (C i ) (p + (r/ &2)*# (cis(zz i)))
                                (p + r*# (cis(xx i)))) /\
       (!i j. (i < n) /\ (j < n) /\ (~(i=j)) ==>
           (C i INTER C j = EMPTY )) /\
       (!i. (i< n) ==>
           C i SUBSET ({ x | (r/(&2) <= d_euclid p x /\
                             d_euclid p x <= r)} )) /\
       (!i. (i< n) ==>
           (C i INTER  ({ x |  d_euclid p x <= (r/(&2))}) =
                          { ( p + (r/(&2)) *# (cis (zz i ))) }))
       )
    `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `C = ( \ i. IMAGE ( \ t. p + (t*r + (&1 - t)*(r/(&2)))*# (cis (segpath (xx i) (zz i)  t))) {t | &0 <= t /\ t <= &1})` ABBREV_TAC ;
  TYPE_THEN `C` EXISTS_TAC;
  (* -- *)
  CONJ_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC "C";
  TYPEL_THEN [`r`;`segpath (xx i) (zz i)`;`p`] (fun t-> (ANT_TAC(ISPECL t curve_simple_lemma)));
  ASM_REWRITE_TAC[segpath_lemma];
  REWRITE_TAC[segpath_end];
  (* -- *)
  TYPE_THEN `&0 < r/ &2 /\ r / &2 < r` SUBGOAL_TAC;
  IMATCH_MP_TAC  half_pos;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  REP_BASIC_TAC;
  TYPEL_THEN [`( \ t. t * r + (&1 - t) * r / &2)`;`segpath (xx i) (zz i)`;`segpath (xx j) (zz j)`] (fun t-> ANT_TAC (ISPECL t polar_distinct));
  ASM_REWRITE_TAC[];
  (* --- *)
  CONJ_TAC;
  TYPEL_THEN [`r`;`r / &2`] (fun t-> ANT_TAC(ISPECL t segpath_inj));
  UND 10;
  REAL_ARITH_TAC;
  REWRITE_TAC[segpathxy];
  (* --- *)
  CONJ_TAC;
  REP_BASIC_TAC;
  ineq_lt_tac `&0 + (x* (r - r/(&2))) + (r/ &2) = x*r + (&1 - x)*(r/ &2)`;
  (* --- *)
  ASM_MESON_TAC[segpath_lemma];
  (* -- *)
  DISCH_TAC;
  EXPAND_TAC "C";
  REWRITE_TAC[EQ_EMPTY];
  GEN_TAC;
  REWRITE_TAC[IMAGE;INTER];
  REP_BASIC_TAC;
  UND 13;
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]) THEN (REWRITE_TAC [t]));
  TYPEL_THEN[`x'`;`x''`] (USE 12 o ISPECL);
  REWR 12;
  TYPE_THEN `((x'' * r + (&1 - x'') * r / &2) *# cis (segpath (xx j) (zz j) x'')) = ((x' * r + (&1 - x') * r / &2) *# cis (segpath (xx i) (zz i) x'))` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  USE 16 ( (REWRITE_RULE[FUN_EQ_THM]));
  TSPEC `x'''` 13;
  UND 13;
  REWRITE_TAC[euclid_plus];
  REAL_ARITH_TAC;
  DISCH_TAC;
  KILL 16;
  USE 13 (ONCE_REWRITE_RULE [EQ_SYM_EQ]);
  REWR 12;
  REP_BASIC_TAC;
  USE 16 GSYM;
  UND 16;
    DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]) THEN (REWRITE_TAC [t]));
  TYPE_THEN `(i <| j) \/ (j < i)` SUBGOAL_TAC;
  UND 7;
  ARITH_TAC;
  (* ---- *)
  DISCH_THEN DISJ_CASES_TAC;
  TYPEL_THEN [`i`;`j`] (USE 0 o ISPECL);
  TYPEL_THEN [`i`;`j`] (USE 1 o ISPECL);
  KILL  2;
  KILL  3;
  KILL 6;
  KILL 13;
  ASM_MESON_TAC[CONJUNCT2 (CONJUNCT2 segpath_lemma)];
  TYPEL_THEN [`j`;`i`] (USE 0 o ISPECL);
  TYPEL_THEN [`j`;`i`] (USE 1 o ISPECL);
  KILL  2;
  KILL  3;
  KILL 6;
  KILL 13;
  ASM_MESON_TAC[CONJUNCT2 (CONJUNCT2 segpath_lemma)];
  (* B-- *)
  CONJ_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC "C";
  IMATCH_MP_TAC  curve_annulus_lemma;
  ASM_REWRITE_TAC[];
  (* -- *)
  REP_BASIC_TAC;
  EXPAND_TAC "C";
  TYPEL_THEN[`r`;`segpath (xx i) (zz i)`;`p`] (fun t-> ANT_TAC(ISPECL t curve_circle_lemma));
  ASM_REWRITE_TAC[];
  REWRITE_TAC[segpath_end];
  (* Wed Aug 18 15:57:53 EDT 2004 *)
  ]);;
  (* }}} *)

let closed_ball2_center = prove_by_refinement(
  `!p r. closed_ball (euclid 2,d_euclid) p r p <=> (euclid 2 p) /\ (&0 <= r)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[closed_ball];
  TYPE_THEN `!p. (euclid 2 p) ==> (d_euclid p p = &0)` SUBGOAL_TAC;
  DISCH_ALL_TAC;
  IMATCH_MP_TAC  metric_space_zero;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[metric_euclid];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let degree_vertex_disk = prove_by_refinement(
  `!r p xx . (&0 < r) /\ (euclid 2 p) /\
  (!j. j < 4 ==> (&0 <= xx j /\ xx j < &2 * pi)) /\
    (!i j. (i < j) /\ (j < 4) ==> (xx i < xx j))
  ==>
      (?C.
       (!i. (i< 4) ==> (?C' C'' v.
           simple_arc_end C' p v /\
           simple_arc_end C'' v (p + r*# (cis(xx i )))  /\
           C' SUBSET closed_ball(euclid 2,d_euclid) p (r/ &2) /\
           (C' INTER C'' = {v}) /\
           (C' UNION C'' = C i )) /\
          simple_arc_end (C i ) p  (p + r*# (cis(xx i))) /\
           C i SUBSET (closed_ball(euclid 2,d_euclid) p r) /\
           C i  INTER (closed_ball(euclid 2,d_euclid) p (r / &2))
           SUBSET (hyperplane 2 e2 (p 1) UNION
                     hyperplane 2 e1 (p 0))) /\
       (!i j. (i < 4) /\ (j < 4) /\ (~(i=j)) ==>
           (C i INTER C j = {p} )))
       `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `(&0 < (r /(&2))) /\ (euclid 2 p)` SUBGOAL_TAC;
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  DISCH_THEN (fun t-> MP_TAC (MATCH_MP   degree4_vertex_hv t));
  REP_BASIC_TAC;
  TYPE_THEN `C' = C` ABBREV_TAC ;
  KILL 10;
  TYPE_THEN `zz = (\j. (&j) * pi/(&2))` ABBREV_TAC ;
  TYPE_THEN `(&0 < r) /\ (euclid 2 p) /\  (!j. j < 4 ==> (&0 <= xx j /\ xx j < &2 * pi)) /\  (!j. j < 4 ==> (&0 <= zz j /\ zz j < &2 * pi)) /\  (!i j. (i < j) /\ (j < 4) ==> (xx i < xx j)) /\ (!i j. (i < j) /\ (j < 4) ==> (zz i < zz j))` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  EXPAND_TAC "zz";
  REP_BASIC_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  REAL_LE_MUL;
  CONJ_TAC;
  REDUCE_TAC;
  IMATCH_MP_TAC  REAL_LE_DIV;
  MP_TAC PI_POS;
  REAL_ARITH_TAC;
  REWRITE_TAC[real_div;REAL_ARITH `pi*x = x*pi`];
  REWRITE_TAC[REAL_ARITH `x*y*z = (x*y)*z`];
  IMATCH_MP_TAC  REAL_PROP_LT_RMUL;
  ASM_REWRITE_TAC[PI_POS;GSYM real_div;];
  ASM_SIMP_TAC[REAL_LT_LDIV_EQ;REAL_ARITH `&0 < &2`];
  REDUCE_TAC;
  UND 11;
  ARITH_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC "zz";
  ONCE_REWRITE_TAC [REAL_ARITH `x < y <=> (&0 < y - x)`];
  REWRITE_TAC[REAL_ARITH `x*y - z*y = (x - z)*y`];
  IMATCH_MP_TAC  REAL_PROP_POS_MUL2;
  REWRITE_TAC[PI2_BOUNDS];
  REDUCE_TAC;
  UND 12;
  REWRITE_TAC[REAL_ARITH `&0 < &j - &i <=> &i < &j`];
  REDUCE_TAC;
  DISCH_THEN (fun t-> MP_TAC (MATCH_MP degree_vertex_annulus t));
  REP_BASIC_TAC;
  (* A *)
  TYPE_THEN `(\j. C' j UNION C'' j)` EXISTS_TAC;
  BETA_TAC;
  (* B 1st conjunct *)
  TYPE_THEN `!i. (i<| 4) ==> (simple_arc_end (C' i ) p (p + ((r/ &2) *# (cis (&i * pi/(&2))))) /\   simple_arc_end (C'' i) (p + ((r/ &2) *# (cis (&i * pi/(&2))))) (euclid_plus p (r *# cis (xx i))) /\ (C' i) SUBSET closed_ball (euclid 2,d_euclid) p (r / &2) /\  ((C' i) INTER (C'' i) = {(p + ((r/ &2) *# (cis (&i * pi/(&2)))))})) ` SUBGOAL_TAC;
  REP_BASIC_TAC;
  SUBCONJ_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  SUBCONJ_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  SUBCONJ_TAC;
  ASM_MESON_TAC[];
  DISCH_TAC;
  REWRITE_TAC[];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INR IN_SING;INTER ];
  EQ_TAC;
  DISCH_TAC;
  TYPE_THEN `closed_ball (euclid 2,d_euclid) p (r / &2) x` SUBGOAL_TAC;
  UND 18;
  REWRITE_TAC[SUBSET];
  UND 19;
  MESON_TAC[];
  TSPEC `i` 11;
  REWR 11;
  REWRITE_TAC[closed_ball];
  FIRST_ASSUM (fun t-> MP_TAC (AP_THM t `x:num->real`));
  UND 19;
  REWRITE_TAC[INTER;INR IN_SING;];
  DISCH_THEN_REWRITE;
  DISCH_THEN_REWRITE;
  EXPAND_TAC "zz";
  DISCH_THEN_REWRITE;
  DISCH_THEN_REWRITE;
  UND 17;
  UND 16;
  MESON_TAC[simple_arc_end_end;simple_arc_end_end2];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  (* [C] 1nd conjunct. simple-arc-end; *)
  TYPE_THEN `D = closed_ball (euclid 2,d_euclid) p (r /(&2))` ABBREV_TAC ;
  TYPE_THEN `!i x. (i <| 4) /\ (D x) ==> ((C' i UNION C'' i) x = C' i x)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[UNION];
  IMATCH_MP_TAC  (TAUT `(b ==> a) ==> (a \/ b <=> a)`);
  TSPEC `i` 11;
  REWR 11;
  FIRST_ASSUM (fun t-> MP_TAC (AP_THM t `x:num->real`));
  UND 17;
  EXPAND_TAC"D";
  REWRITE_TAC[closed_ball];
  REWRITE_TAC[INTER;INR IN_SING];
  DISCH_THEN_REWRITE;
  DISCH_THEN_REWRITE;
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[simple_arc_end_end2];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!i x. (i <| 4) /\ ~(D x) ==> ((C' i UNION C'' i) x = C'' i x)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[UNION];
  IMATCH_MP_TAC  (TAUT `(a ==> b) ==> (a \/ b <=> b)`);
  TSPEC `i` 5;
  REWR 5;
  USE 5 (REWRITE_RULE[SUBSET]);
  TSPEC `x` 5;
  UND 5;
  UND 18;
  MESON_TAC[];
  DISCH_TAC;
  ONCE_REWRITE_TAC [TAUT `(x /\ y) <=> (y /\ x)`];
  (* D-- *)
  CONJ_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INTER;INR IN_SING];
  TYPE_THEN `D x` ASM_CASES_TAC;
  TYPEL_THEN [`i`;`x`] (WITH 17 o ISPECL);
  TYPEL_THEN [`j`;`x`] (WITH 17 o ISPECL);
  UND 23;
  UND 24;
  KILL 17;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  DISCH_THEN_REWRITE;
  TYPEL_THEN [`i`;`j`;] (USE 7 o ISPECL);
  REWR 7;
  FIRST_ASSUM (fun t-> MP_TAC (AP_THM t `x:num->real`));
  REWRITE_TAC[INTER;INR IN_SING];
  (* --2-- *)
  TYPEL_THEN [`i`;`x`] (WITH 18 o ISPECL);
  TYPEL_THEN [`j`;`x`] (WITH 18 o ISPECL);
  UND 23;
  UND 24;
  KILL 18;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  DISCH_THEN_REWRITE;
  TYPEL_THEN [`i`;`j`;] (USE 13 o ISPECL);
  REWR 13;
  USE 13 (REWRITE_RULE[EQ_EMPTY;INTER ]);
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  USE 18(REWRITE_RULE[]);
  UND 18;
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[t]));
  UND 22;
  REWRITE_TAC[];
  EXPAND_TAC "D";
  REWRITE_TAC[closed_ball2_center];
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  (REAL_ARITH `&0 <x ==> &0 <= x`);
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  (* E *)
  REP_BASIC_TAC;
  CONJ_TAC;
  TYPE_THEN `C' i` EXISTS_TAC;
  TYPE_THEN `C'' i` EXISTS_TAC;
  TYPE_THEN `p + (r / &2 *# cis (&i * pi / &2))` EXISTS_TAC;
  ASM_MESON_TAC[];
  (* -- *)
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_trans;
  ASM_MESON_TAC[];
  (* -- *)
  CONJ_TAC;
  REWRITE_TAC[union_subset];
  CONJ_TAC;
  TSPEC `i` 5;
  UND 5;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "D";
  REWRITE_TAC[SUBSET;closed_ball;];
  TYPE_THEN `r / &2 < r` SUBGOAL_TAC;
  UND 3;
  MESON_TAC[half_pos];
  MESON_TAC[REAL_ARITH `(x <= y) /\ (y < z) ==> (x <= z)`];
  TSPEC `i` 12;
  UND 12;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET;closed_ball];
  ASM_REWRITE_TAC[];
  TSPEC `i` 14;
  REWR 12;
  TYPE_THEN `C'' i SUBSET (euclid 2)` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_euclid;
  IMATCH_MP_TAC  simple_arc_end_simple;
  UND 12;
  MESON_TAC[];
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  (* -- *)
  KILL 15;
  KILL 9;
  KILL 8;
  KILL 11;
  KILL 12;
  TYPE_THEN `(C' i UNION C'' i) INTER D = (C' i INTER D)` SUBGOAL_TAC;
  REWRITE_TAC[INTER];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  UND 17;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TSPEC `i` 4;
  REWR 4;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C' i` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INTER;SUBSET];
  MESON_TAC[];
  (* Thu Aug 19 07:36:47 EDT 2004 *)

   ]);;
  (* }}} *)

let euclid_cancel1 = prove_by_refinement(
  `!x y z. (x = euclid_plus y z) <=> (x - y = z)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  EQ_TAC;
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[euclid_plus;euclid_minus];
  REAL_ARITH_TAC;
  DISCH_TAC;
  USE 0 SYM;
  ASM_REWRITE_TAC[];
    IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[euclid_plus;euclid_minus];
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let infinite_subset = prove_by_refinement(
  `!(X:A->bool) Y. INFINITE X /\ X SUBSET Y ==> INFINITE Y`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INFINITE];
  MESON_TAC[FINITE_SUBSET];
  ]);;
  (* }}} *)

let EXPinj = prove_by_refinement(
  `!x y n. (1 < n) /\ (n **| x = n **| y) ==> (x = y)`,
  (* {{{ proof *)
  [
  TYPE_THEN `! x y n. (x <| y) /\ (n **| x = n **| y) ==> ~(1 <| n)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `n **| y <= n **| x` SUBGOAL_TAC;
  UND 1;
  ARITH_TAC;
  REWRITE_TAC[LE_EXP];
  TYPE_THEN `~(n = 0)` SUBGOAL_TAC;
  UND 0;
  ARITH_TAC;
  DISCH_THEN_REWRITE;
  REWRITE_TAC[DE_MORGAN_THM];
  CONJ_TAC;
  UND 0;
  ARITH_TAC;
  UND 2;
  ARITH_TAC;
  DISCH_TAC;
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `x < y \/ y <| x` SUBGOAL_TAC;
  UND 3;
  ARITH_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  TYPEL_THEN[`x`;`y`;`n`] (USE 0 o ISPECL);
  ASM_MESON_TAC[];
  TYPEL_THEN[`y`;`x`;`n`] (USE 0 o ISPECL);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let infinite_interval = prove_by_refinement(
  `!a b. a < b ==> (INFINITE {x | a < x /\ x < b})`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  infinite_subset;
  TYPE_THEN `f = (\ n. a + (b-a)/((&2) pow (SUC n)))` ABBREV_TAC ;
  TYPE_THEN `IMAGE f  UNIV` EXISTS_TAC ;
  CONJ_TAC;
  TYPE_THEN `(! x y. (f x = f y) ==> (x = y))` SUBGOAL_TAC;
  EXPAND_TAC "f";
  REP_BASIC_TAC;
  USE 2 (REWRITE_RULE[REAL_ARITH `(a + d = a + d') <=> (d = d')`;real_div;REAL_PROP_EQ_RMUL_';]);
  TYPE_THEN `~(b - a = &0)` SUBGOAL_TAC;
  UND 0;
  REAL_ARITH_TAC;
  DISCH_TAC;
  REWR 2;
  USE 2 (REWRITE_RULE[GSYM REAL_EQ_INV]);
  UND 2;
  REDUCE_TAC;
  DISCH_TAC;
  ONCE_REWRITE_TAC[GSYM EQ_SUC];
  IMATCH_MP_TAC  EXPinj;
  TYPE_THEN `2` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ARITH_TAC;
  DISCH_TAC;
  TYPE_THEN `INFINITE (UNIV:num->bool) ==> INFINITE (IMAGE f UNIV)` SUBGOAL_TAC;
  ASM_MESON_TAC[INFINITE_IMAGE_INJ];
  REWRITE_TAC[num_INFINITE];
  (* -- *)
  REWRITE_TAC[IMAGE;SUBSET];
  GEN_TAC;
  REP_BASIC_TAC;
  UND 2;
  DISCH_THEN_REWRITE;
  EXPAND_TAC "f";
  CONJ_TAC;
  ONCE_REWRITE_TAC[REAL_ARITH `a < a + x <=> &0 < x`];
  REWRITE_TAC[real_div];
  IMATCH_MP_TAC  REAL_PROP_POS_MUL2;
  CONJ_TAC;
  UND 0;
  REAL_ARITH_TAC;
  IMATCH_MP_TAC  REAL_PROP_POS_INV;
  REDUCE_TAC;
  ARITH_TAC;
  ONCE_REWRITE_TAC [REAL_ARITH `a + x < b <=> x < (b - a)*(&1)`];
  REWRITE_TAC[real_div];
  IMATCH_MP_TAC  REAL_PROP_LT_LMUL;
  CONJ_TAC;
  UND 0;
  REAL_ARITH_TAC;
  ONCE_REWRITE_TAC[GSYM REAL_INV_1];
  IMATCH_MP_TAC  REAL_LT_INV2;
  REDUCE_TAC;
  IMATCH_MP_TAC  exp_gt1;
  ARITH_TAC;
  (* Thu Aug 19 14:59:58 EDT 2004 *)
  ]);;
  (* }}} *)

let finite_augment1 = prove_by_refinement(
  `!n (X:A->bool) . (INFINITE X) ==> (?Z. Z SUBSET X /\ Z HAS_SIZE n)`,
  (* {{{ proof *)
  [
  INDUCT_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `EMPTY:A->bool` EXISTS_TAC  ;
  REWRITE_TAC[HAS_SIZE_0];
  REP_BASIC_TAC;
  TSPEC `X` 0;
  REWR 0;
  REP_BASIC_TAC;
  TYPE_THEN `INFINITE (X DIFF Z)` SUBGOAL_TAC;
  IMATCH_MP_TAC  INFINITE_DIFF_FINITE;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[HAS_SIZE];
  DISCH_TAC;
  USE 3 (MATCH_MP INFINITE_NONEMPTY);
  USE 3 (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  TYPE_THEN `u INSERT Z` EXISTS_TAC;
  CONJ_TAC;
  UND 2;
  UND 3;
  REWRITE_TAC[DIFF;SUBSET;INSERT];
  ASM_MESON_TAC[];
  (* -- *)
  USE 0 (REWRITE_RULE[HAS_SIZE]);
  ASM_SIMP_TAC [HAS_SIZE;FINITE_INSERT;CARD_CLAUSES;];
  UND 3;
  REWRITE_TAC[DIFF];
  DISCH_THEN_REWRITE;
  ]);;
  (* }}} *)

let finite_augment = prove_by_refinement(
  `!(X:A->bool) Y n m . (n <= m) /\ (X HAS_SIZE n) /\ (INFINITE Y) /\
   (X SUBSET Y) ==> (?Z. (X SUBSET Z /\ Z SUBSET Y /\ Z HAS_SIZE m))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `INFINITE (Y DIFF X)` SUBGOAL_TAC;
  IMATCH_MP_TAC  INFINITE_DIFF_FINITE;
  ASM_MESON_TAC[HAS_SIZE];
  DISCH_TAC;
  USE 4(MATCH_MP finite_augment1);
  USE 3(REWRITE_RULE[LE_EXISTS]);
  REP_BASIC_TAC;
  TSPEC `d` 4;
  REP_BASIC_TAC;
  TYPE_THEN `X UNION Z` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  REWRITE_TAC[union_subset];
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  UND 5;
  SET_TAC[SUBSET;DIFF];
  REWRITE_TAC[HAS_SIZE];
  CONJ_TAC;
  ASM_REWRITE_TAC[FINITE_UNION];
  ASM_MESON_TAC[HAS_SIZE];
  RULE_ASSUM_TAC (REWRITE_RULE[HAS_SIZE]);
  REP_BASIC_TAC;
  EXPAND_TAC "d";
  EXPAND_TAC "n";
  IMATCH_MP_TAC  CARD_UNION;
  ASM_REWRITE_TAC[];
  UND 5;
  REWRITE_TAC[SUBSET;DIFF;INTER;EQ_EMPTY ];
  MESON_TAC[];
  (* Thu Aug 19 15:29:05 EDT 2004 *)

  ]);;
  (* }}} *)

let euclid_add_cancel = prove_by_refinement(
  `!p q q'. (euclid_plus p q = euclid_plus p q') <=> (q = q')`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[FUN_EQ_THM];
  REWRITE_TAC [euclid_plus;];
  REWRITE_TAC[REAL_ARITH `(x + a = x + b) <=> (a = b)`];
  ]);;
  (* }}} *)


let degree_vertex_disk_ver2 = prove_by_refinement(
  `!r p X. (&0 < r) /\ (euclid 2 p) /\ (FINITE X) /\ (CARD X <= 4) /\
     (X SUBSET {x | (euclid 2 x) /\ (d_euclid p x = r)}) ==>
    (?C. (!i. (X i) ==> (?C' C'' v.
           simple_arc_end C' p v /\
           simple_arc_end C'' v i  /\
           C' SUBSET closed_ball(euclid 2,d_euclid) p (r/ &2) /\
           (C' INTER C'' = {v}) /\
           (C' UNION C'' = C i )) /\
          simple_arc_end (C i ) p  i /\
           C i SUBSET (closed_ball(euclid 2,d_euclid) p r) /\
           C i  INTER (closed_ball(euclid 2,d_euclid) p (r / &2))
           SUBSET (hyperplane 2 e2 (p 1) UNION
                     hyperplane 2 e1 (p 0))) /\
       (!i j. (X i ) /\ (X j) /\ (~(i=j)) ==>
           (C i INTER C j = {p} )))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `!x. (X x) ==> (?r t. &0 <= t /\ t < &2 * pi /\ &0 <= r /\ (x = p + r *# cis t))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[euclid_cancel1];
  IMATCH_MP_TAC  polar_exist;
  USE 0(REWRITE_RULE[SUBSET]);
  ASM_MESON_TAC[euclid_sub_closure];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!x. (X x) ==> (?t. &0 <= t /\ t < &2 * pi /\ &0 <= r /\ (x = p + r *# cis t))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TSPEC `x` 5;
  REWR 5;
  REP_BASIC_TAC;
  UND 5;
  DISCH_THEN (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])));
  TYPE_THEN `t` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  UND 4;
  REAL_ARITH_TAC;
  USE 0 (REWRITE_RULE[SUBSET]);
  TSPEC `euclid_plus p (r' *# cis t)` 0;
  REWR 0;
  REP_BASIC_TAC;
  UND 0;
  TYPEL_THEN[`2`;`p`;`r' *# cis t`] (fun t-> ANT_TAC (ISPECL t d_euclidpq));
  ASM_REWRITE_TAC[polar_euclid];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[GSYM norm2;norm2_scale_cis];
  DISCH_TAC;
  TYPE_THEN `abs  r' = r'` SUBGOAL_TAC;
  UND 7;
  REAL_ARITH_TAC;
  DISCH_TAC;
  REWR 0;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  KILL 5;
  (* -- *)
  TYPE_THEN `TX = {t | (&0 <= t /\ t < &2 *pi /\ (X( p + (r *# (cis t))))) }` ABBREV_TAC ;
  TYPE_THEN `BIJ ( \ t. p + r *# cis t) TX X` SUBGOAL_TAC;
  REWRITE_TAC[BIJ;INJ;SURJ];
  SUBCONJ_TAC;
  CONJ_TAC;
  EXPAND_TAC "TX";
  REWRITE_TAC[];
  MESON_TAC[];
  EXPAND_TAC "TX";
  REWRITE_TAC[];
  REP_BASIC_TAC;
  USE 7 (REWRITE_RULE[euclid_add_cancel]);
  PROOF_BY_CONTR_TAC;
  TYPEL_THEN[`x`;`y`;`r`;`r`] (fun t-> ANT_TAC(ISPECL t polar_inj));
  ASM_REWRITE_TAC[];
  UND 4;
  REAL_ARITH_TAC;
  ASM_REWRITE_TAC[];
  UND 4;
  REAL_ARITH_TAC;
  DISCH_THEN_REWRITE;
  REP_BASIC_TAC;
  EXPAND_TAC "TX";
  REWRITE_TAC[];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `INFINITE {x | &0 <= x /\ x < &2* pi}` SUBGOAL_TAC;
  IMATCH_MP_TAC  infinite_subset;
  TYPE_THEN `{x | &0 < x /\ x < &2 * pi}` EXISTS_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  infinite_interval;
  IMATCH_MP_TAC  REAL_PROP_POS_MUL2;
  REWRITE_TAC[PI_POS];
  REAL_ARITH_TAC;
  REWRITE_TAC[SUBSET];
  MESON_TAC[REAL_ARITH `&0 < x ==> &0 <= x`];
  DISCH_TAC;
  (* A -- *)
  TYPE_THEN `TX HAS_SIZE CARD X` SUBGOAL_TAC;
  REWRITE_TAC[HAS_SIZE];
  SUBCONJ_TAC;
  COPY 7;
  JOIN 2 7;
  USE 2 (MATCH_MP FINITE_BIJ2);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  IMATCH_MP_TAC BIJ_CARD;
  ASM_REWRITE_TAC [];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `(?Z. (TX SUBSET Z /\ Z SUBSET {x | &0 <= x /\ x < &2 *pi}  /\ Z HAS_SIZE 4))` SUBGOAL_TAC;
  IMATCH_MP_TAC  finite_augment;
  TYPE_THEN `CARD X` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC"TX";
  REWRITE_TAC[SUBSET];
  REAL_ARITH_TAC;
  REP_BASIC_TAC;
  (* B -- order points *)
  TYPE_THEN `FINITE Z` SUBGOAL_TAC;
  ASM_MESON_TAC[HAS_SIZE];
  DISCH_TAC;
  USE 13 (MATCH_MP real_finite_increase);
  REP_BASIC_TAC;
  USE 10(REWRITE_RULE[HAS_SIZE]);
  REP_BASIC_TAC;
  REWR 13;
  REWR 14;
  (* -- *)
  TYPEL_THEN [`r`;`p`;`u`] (fun t-> ANT_TAC (ISPECL t degree_vertex_disk));
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  UND 14;
  REWRITE_TAC[BIJ;SURJ];
  REP_BASIC_TAC;
  USE 11(REWRITE_RULE[SUBSET]);
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 16;
  UND 17;
  ARITH_TAC;
  REP_BASIC_TAC;
  (* [C] -- create C *)
  TYPE_THEN `f = (\t. euclid_plus p (r *# cis t))` ABBREV_TAC ;
  TYPE_THEN `g = INV f TX X` ABBREV_TAC ;
  TYPE_THEN `u' = INV u {x | x <| 4} Z` ABBREV_TAC ;
  TYPE_THEN `BIJ g X TX` SUBGOAL_TAC;
  EXPAND_TAC "g";
  IMATCH_MP_TAC  INVERSE_BIJ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `BIJ u' Z {x | x <| 4}` SUBGOAL_TAC;
  EXPAND_TAC "u'";
  IMATCH_MP_TAC  INVERSE_BIJ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `INJ (compose u'  g) X { x | x <| 4}` SUBGOAL_TAC;
  IMATCH_MP_TAC  COMP_INJ;
  TYPE_THEN `TX` EXISTS_TAC;
  CONJ_TAC;
  UND 21;
  REWRITE_TAC[BIJ];
  MESON_TAC[];
  IMATCH_MP_TAC  inj_subset_domain;
  TYPE_THEN `Z` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 22;
  REWRITE_TAC [BIJ];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  TYPE_THEN `(\ j. C ((compose u' g) j))` EXISTS_TAC;
  REWRITE_TAC[];
  (* D -- check properties *)
  CONJ_TAC;
  REP_BASIC_TAC;
  TYPE_THEN   `j = compose u' g i` ABBREV_TAC ;
  TSPEC `j` 17;
  TYPE_THEN `j <| 4` SUBGOAL_TAC;
  USE 23 (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  EXPAND_TAC "j";
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWR 17;
  ASM_REWRITE_TAC[];
  (* --2-- *)
  TYPE_THEN `i = f (u j)` SUBGOAL_TAC;
  EXPAND_TAC "j";
  EXPAND_TAC "f";
  EXPAND_TAC "u'";
  REWRITE_TAC[compose];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  TYPE_THEN `u (INV u {x | x <| 4} Z (g i)) = (g i)` SUBGOAL_TAC;
  IMATCH_MP_TAC  inv_comp_right;
  ASM_REWRITE_TAC[];
  UND 21;
  UND 12;
  REWRITE_TAC[SUBSET;BIJ;SURJ;];
  UND 24;
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `f (g i) = i` SUBGOAL_TAC;
  EXPAND_TAC "g";
  IMATCH_MP_TAC  inv_comp_right;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "f";
  DISCH_THEN_REWRITE;
  EXPAND_TAC "f";
  DISCH_THEN (fun t-> RULE_ASSUM_TAC (REWRITE_RULE[GSYM t]));
  ASM_REWRITE_TAC[];
  (* E *)
  REP_BASIC_TAC;
  TYPE_THEN `i' = compose u' g i` ABBREV_TAC ;
  TYPE_THEN `j' = compose u' g j` ABBREV_TAC ;
  KILL 17;
  TYPE_THEN `~(i' = j')` SUBGOAL_TAC;
  DISCH_TAC;
  UND 24;
  REWRITE_TAC[];
  USE 23 (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `(i' <| 4) /\ (j' <| 4) ` SUBGOAL_TAC;
  EXPAND_TAC "i'";
  EXPAND_TAC "j'";
  USE 23 (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPEL_THEN [`i'`;`j'`] (USE 16 o ISPECL);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* Thu Aug 19 18:06:33 EDT 2004 *)

  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* SECTION O *)
(* ------------------------------------------------------------------ *)


let simple_arc_connected = prove_by_refinement(
  `!C. simple_arc top2 C ==> connected top2 C`,
  (* {{{ proof *)

  [
  REWRITE_TAC[simple_arc;];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  connect_image;
  TYPE_THEN `(top_of_metric(UNIV,d_real))` EXISTS_TAC;
  ASM_REWRITE_TAC[connect_real];
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REWRITE_TAC[IMAGE;SUBSET];
  REP_BASIC_TAC;
  ASM_SIMP_TAC[];
  (* Fri Aug 20 08:32:31 EDT 2004 *)
  ]);;

  (* }}} *)

let disk_endpoint = prove_by_refinement(
  `!C r p v v'. simple_arc_end C v v' /\ (&0 < r) /\ (euclid 2 p) /\
       (C INTER (closed_ball(euclid 2,d_euclid) p r) = {v}) ==>
      (d_euclid p v = r)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `connected top2 C` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_connected;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `A = euclid 2 DIFF (closed_ball (euclid 2, d_euclid) p r)` ABBREV_TAC ;
  TYPE_THEN `B = closed_ball(euclid 2, d_euclid) p r` ABBREV_TAC ;
  TYPE_THEN `closed_ top2 B` SUBGOAL_TAC;
  EXPAND_TAC "B";
  REWRITE_TAC[top2];
  IMATCH_MP_TAC  closed_ball_closed;
  REWRITE_TAC[metric_euclid];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `top2 A` SUBGOAL_TAC;
  UND 8;
  EXPAND_TAC "A";
  EXPAND_TAC "B";
  REWRITE_TAC[closed;top2_unions;open_DEF ;];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* - *)
  TYPE_THEN `C SUBSET euclid 2` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_euclid;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `B' = open_ball(euclid 2,d_euclid) p r` ABBREV_TAC ;
  TYPE_THEN `C SUBSET B' UNION A` SUBGOAL_TAC;
  EXPAND_TAC "A";
  EXPAND_TAC "B'";
  EXPAND_TAC "B";
  REWRITE_TAC[open_ball;SUBSET;DIFF;closed_ball;UNION];
  USE 10 (REWRITE_RULE[SUBSET]);
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TSPEC `x` 10;
  REWR 10;
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  USE 13 (REWRITE_RULE[DE_MORGAN_THM]);
  REP_BASIC_TAC;
  TYPE_THEN `B x` SUBGOAL_TAC;
  EXPAND_TAC "B";
  REWRITE_TAC[closed_ball];
  ASM_REWRITE_TAC[];
  USE 0 (REWRITE_RULE[FUN_EQ_THM]);
  USE 0 (REWRITE_RULE[INTER;INR IN_SING]);
  ASM_MESON_TAC[REAL_ARITH `u <= v /\ ~(u = v) ==> (u < v)`];
  (* - *)
  USE 5 (REWRITE_RULE[connected;top2_unions]);
  REP_BASIC_TAC;
  TYPEL_THEN[`B'`;`A`] (USE 12 o ISPECL);
  REWR 12;
  TYPE_THEN `top2 B'` SUBGOAL_TAC;
  EXPAND_TAC "B'";
  REWRITE_TAC[top2];
  IMATCH_MP_TAC  open_ball_open;
  REWRITE_TAC[metric_euclid];
  DISCH_THEN_FULL_REWRITE;
  (* - *)
  TYPE_THEN `B' INTER A = EMPTY` SUBGOAL_TAC;
  EXPAND_TAC "A";
  EXPAND_TAC "B'";
  EXPAND_TAC "B";
  REWRITE_TAC[open_ball;closed_ball;DIFF;EQ_EMPTY;INTER;];
  REP_BASIC_TAC;
  UND 14;
  ASM_REWRITE_TAC[];
  UND 16;
  REAL_ARITH_TAC;
  DISCH_THEN_FULL_REWRITE;
  (* - *)
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `C SUBSET B` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `B'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "B";
  EXPAND_TAC "B'";
  REWRITE_TAC[SUBSET;open_ball;closed_ball];
  MESON_TAC[REAL_ARITH `x < y ==> x <= y`];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `~(v = v')` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_distinct;
  ASM_MESON_TAC[];
  REWRITE_TAC[];
  TYPE_THEN `C v'` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_end2];
  DISCH_TAC;
  TYPE_THEN `B v'` SUBGOAL_TAC;
  UND 15;
  UND 16;
  MESON_TAC[ISUBSET];
  UND 16;
  UND 0;
  REWRITE_TAC[INTER;eq_sing];
  MESON_TAC[];
  (* - *)
  TYPE_THEN `C v` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_end];
  DISCH_TAC;
  TYPE_THEN `A v` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  TYPE_THEN `B v` SUBGOAL_TAC;
  UND 0;
  REWRITE_TAC[INTER;eq_sing];
  DISCH_THEN_REWRITE;
  EXPAND_TAC "A";
  REWRITE_TAC[DIFF];
  DISCH_THEN_REWRITE;
  (* Fri Aug 20 09:12:44 EDT 2004 *)

  ]);;
  (* }}} *)

let disk_endpoint_gen = prove_by_refinement(
  `!C B' B v v'. simple_arc_end C v v'  /\
      (top2 B') /\ (closed_ top2 B) /\ (B' SUBSET B) /\
       (C INTER B = {v}) ==>
      (~(B' v))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `connected top2 C` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_connected;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `A = euclid 2 DIFF B` ABBREV_TAC ;
  (* - *)
  TYPE_THEN `top2 A` SUBGOAL_TAC;
  EXPAND_TAC "A";
  UND 3;
  REWRITE_TAC[closed;top2_unions;open_DEF ;];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* - *)
  TYPE_THEN `C SUBSET euclid 2` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_euclid;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `C SUBSET B' UNION A` SUBGOAL_TAC;
  EXPAND_TAC "A";
  REWRITE_TAC[open_ball;SUBSET;DIFF;closed_ball;UNION];
  USE 9 (REWRITE_RULE[SUBSET]);
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `B x` ASM_CASES_TAC;
  ASM_REWRITE_TAC[];
  USE 1(REWRITE_RULE[INTER;eq_sing]);
  REP_BASIC_TAC;
  TYPE_THEN `(x = v)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  DISJ2_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* - *)
  DISCH_TAC;
  USE 6 (REWRITE_RULE[connected;top2_unions]);
  REP_BASIC_TAC;
  TYPEL_THEN[`B'`;`A`] (USE 6 o ISPECL);
  REWR 6;
  (* - *)
  TYPE_THEN `B' INTER A = EMPTY` SUBGOAL_TAC;
  EXPAND_TAC "A";
  REWRITE_TAC[open_ball;closed_ball;DIFF;EQ_EMPTY;INTER;];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SUBSET]);
  ASM_MESON_TAC[];
  DISCH_THEN_FULL_REWRITE;
  (* - *)
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `C SUBSET B` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `B'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `~(v = v')` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_distinct;
  ASM_MESON_TAC[];
  REWRITE_TAC[];
  TYPE_THEN `C v'` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_end2];
  DISCH_TAC;
  TYPE_THEN `B v'` SUBGOAL_TAC;
  UND 13;
  UND 14;
  MESON_TAC[ISUBSET];
  UND 14;
  UND 1;
  REWRITE_TAC[INTER;eq_sing];
  MESON_TAC[];
  (* - *)
  TYPE_THEN `C v` SUBGOAL_TAC;
  ASM_MESON_TAC[simple_arc_end_end];
  DISCH_TAC;
  TYPE_THEN `A v` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  TYPE_THEN `B v` SUBGOAL_TAC;
  UND 1;
  REWRITE_TAC[INTER;eq_sing];
  DISCH_THEN_REWRITE;
  EXPAND_TAC "A";
  REWRITE_TAC[DIFF];
  DISCH_THEN_REWRITE;
  ]);;
  (* }}} *)

let disk_endpoint_outer = prove_by_refinement(
  `!C r p v v'. simple_arc_end C v v'  /\ (&0 < r) /\ (euclid 2 p) /\
      (C INTER (euclid 2 DIFF (open_ball(euclid 2,d_euclid) p r)) = {v})
     ==>
      (d_euclid p v = r)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `B = (euclid 2 DIFF (open_ball(euclid 2,d_euclid) p r))` ABBREV_TAC ;
  TYPE_THEN `B' = (euclid 2 DIFF (closed_ball(euclid 2,d_euclid) p r))` ABBREV_TAC ;
  (* - *)
  TYPE_THEN `B' SUBSET B` SUBGOAL_TAC;
  EXPAND_TAC "B'";
  EXPAND_TAC "B";
  REWRITE_TAC[closed_ball;open_ball;SUBSET;DIFF];
  MESON_TAC[REAL_ARITH `x < u ==> x <= u`];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `closed_ top2 B` SUBGOAL_TAC;
  EXPAND_TAC "B";
  REWRITE_TAC[closed;top2_unions;open_DEF ;SUBSET_DIFF];
  TYPE_THEN `open_ball (euclid 2,d_euclid) p r SUBSET (euclid 2)` SUBGOAL_TAC;
  REWRITE_TAC[open_ball;SUBSET];
  MESON_TAC[];
  ASM_SIMP_TAC[DIFF_DIFF2];
  ASM_SIMP_TAC [open_ball_open;top2;metric_euclid];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `top2 B'` SUBGOAL_TAC;
  EXPAND_TAC "B'";
  TH_INTRO_TAC [`top2`;`closed_ball (euclid 2,d_euclid) p r`] closed_open;
  REWRITE_TAC[metric_euclid;top2];
  IMATCH_MP_TAC  closed_ball_closed;
  REWRITE_TAC[metric_euclid];
  REWRITE_TAC[open_DEF;top2_unions;];
  DISCH_TAC;
  (* - *)
  TH_INTRO_TAC [`C`;`B'`;`B`;`v`;`v'`] disk_endpoint_gen;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `B v` SUBGOAL_TAC;
  UND 0;
  REWRITE_TAC[INTER;eq_sing];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* - *)
  TYPE_THEN `B v /\ ~B' v ==> (d_euclid p v = r)` SUBGOAL_TAC;
  EXPAND_TAC "B";
  EXPAND_TAC "B'";
  REWRITE_TAC[DIFF;open_ball;closed_ball;];
  MESON_TAC[REAL_ARITH `x <= y /\ ~(x < y) ==> (x = y)`];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let graph_edge_around = jordan_def
  `graph_edge_around (G:(A,B)graph_t) v =
   { e | graph_edge G e /\ graph_inc G e v}`;;

let graph_edge_around_empty = prove_by_refinement(
  `!(G:(A,B)graph_t) v. (graph G) /\ ~(graph_vertex G v) ==>
      (graph_edge_around G v = EMPTY)`,
  (* {{{ proof *)

  [
  REWRITE_TAC[graph_edge_around;EQ_EMPTY;];
  REP_BASIC_TAC;
  TH_INTRO_TAC [`G`;`x`] graph_inc_subset;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  ASM_MESON_TAC[];
  (* Fri Aug 20 09:25:57 EDT 2004 *)

  ]);;

  (* }}} *)

let graph_disk_hv_preliminaries = prove_by_refinement(
  `!G. plane_graph G /\
      FINITE (graph_edge G) /\ FINITE (graph_vertex G) /\
      ~(graph_edge G = EMPTY) /\
     (!v. (CARD (graph_edge_around G v) <=| 4))
   ==>
  (?NC D short_end hyper r d f. ((!e p. graph_edge G e /\ (!v. ~D v p) ==> (f e p = d e p)) /\
  (!e v p.
           graph_edge G e /\ graph_vertex G v /\ ~graph_inc G e v /\ D v p
           ==> ~f e p) /\
  (!e v p.
           (graph_edge G e /\ graph_inc G e v) /\ D v p
           ==> (f e p = NC e v p)) /\
  (!e. f e = {x | d e x \/ (?v. graph_inc G e v /\ NC e v x)}) /\
  (!v e e'.
           graph_edge G e /\
           graph_edge G e' /\
           graph_inc G e v /\
           graph_inc G e' v /\
           ~(e = e')
           ==> (NC e v INTER NC e' v = {v})) /\
  (!e v. graph_edge G e /\ graph_inc G e v ==> d e (short_end e v)) /\
  (!e e'.
           graph_edge G e /\ graph_edge G e' /\ ~(e = e')
           ==> (d e INTER d e' = {})) /\
  (!e v.
           graph_edge G e /\ graph_inc G e v
           ==> ~graph_vertex G (short_end e v)) /\
  (!v v'.
           graph_vertex G v /\ graph_vertex G v' /\ ~(v = v')
           ==> (D v INTER D v' = {})) /\
  (!e v.
           graph_edge G e /\ graph_inc G e v
           ==> simple_arc_end (NC e v) v (short_end e v) /\
               NC e v SUBSET D v /\
               hyper (NC e v) v) /\
  ((\ B v.
            B INTER closed_ball (euclid 2,d_euclid) v (r / &2) SUBSET
            hyperplane 2 e2 (v 1) UNION hyperplane 2 e1 (v 0)) =
       hyper) /\
  (!e v. graph_edge G e /\ graph_inc G e v ==> graph_vertex G v) /\
  (!e v.
           graph_edge G e /\ graph_vertex G v /\ ~graph_inc G e v
           ==> (d e INTER D v = {})) /\
  (!e. graph_edge G e ==> d e SUBSET e) /\
  (!e v.
           graph_edge G e /\ graph_inc G e v
           ==> (d e INTER D v = {(short_end e v)}) /\
               (d_euclid v (short_end e v) = r) /\
               (!v'. graph_inc G e v' /\ ~(v = v')
                     ==> simple_arc_end (d e) (short_end e v)
                         (short_end e v'))) /\
  (!v. euclid 2 v ==> D v v) /\
  (!u. closed_ top2 (D u)) /\
  (( \ u. closed_ball (euclid 2,d_euclid) u r) = D) /\
  (&0 < r) /\
  (plane_graph G)))
     `,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  TH_INTRO_TAC [`G`] graph_disk;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* TYPE_THEN `r /(&2)` EXISTS_TAC; *)
  (* - *)
  TYPE_THEN `D = (\u. (closed_ball (euclid 2,d_euclid ) u r))` ABBREV_TAC ;
  TYPE_THEN `!u. closed_ top2 (D u)` SUBGOAL_TAC;
  EXPAND_TAC "D";
  GEN_TAC;
  REWRITE_TAC[top2];
  IMATCH_MP_TAC  closed_ball_closed;
  REWRITE_TAC[metric_euclid];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `!v. (euclid 2 v) ==> D v v` SUBGOAL_TAC;
  EXPAND_TAC "D";
  REWRITE_TAC[closed_ball2_center];
  GEN_TAC;
  DISCH_THEN_REWRITE;
  UND 7;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* - *)
  (* [A]- Pick middle arcs *)
  (* {{{ *)

  TYPE_THEN `!e. ?d. (graph_edge G e) ==> (?u u' v v'.  simple_arc_end d u u' /\ graph_inc G e v /\ graph_inc G e v' /\ ~(v = v') /\  (d INTER (D v) = {u}) /\ (d INTER (D v') = {u'}) /\ (d SUBSET e) /\ (d_euclid v u = r) /\ (d_euclid v' u' = r))` SUBGOAL_TAC ;
  GEN_TAC;
  RIGHT_TAC "d";
  DISCH_TAC;
  TH_INTRO_TAC [`G`;`e`] graph_edge_end_select;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);  (* -xx- *)
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TH_INTRO_TAC [`e`;`D v`;`D v'`] simple_arc_end_restriction;
  ASM_REWRITE_TAC[GSYM top2];
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  USE 16 (REWRITE_RULE[SUBSET ]);
  ASM_MESON_TAC[];
  UND 6;
  DISCH_THEN (TH_INTRO_TAC [`v`;`v'`] );
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE [plane_graph;]);
  ASM_MESON_TAC[REWRITE_RULE[SUBSET] graph_inc_subset];
  DISCH_TAC;
  CONJ_TAC;
  EXPAND_TAC "D";
  UND 6;
  REWRITE_TAC[INTER;EQ_EMPTY];
  MESON_TAC[];
  REWRITE_TAC[EMPTY_EXISTS ];
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  TSPEC `e` 15;
  REWR 15;
  REWR 13;
  REWR 14;
  UND 18;
  REWRITE_TAC[SUBSET];
  UND 13;
  UND 14;
  REWRITE_TAC[INTER];
  UND 10;
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  TYPE_THEN `v''` EXISTS_TAC;
  TYPE_THEN `v'''` EXISTS_TAC;
  TYPE_THEN `v` EXISTS_TAC;
  TYPE_THEN `v'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* -- *)
  CONJ_TAC;
  IMATCH_MP_TAC  disk_endpoint;
  TYPE_THEN `C'` EXISTS_TAC;
  TYPE_THEN `v'''` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 16;
  EXPAND_TAC "D";
  DISCH_THEN_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  USE 21 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[REWRITE_RULE[ISUBSET] graph_inc_subset];
  (* -- *)
  IMATCH_MP_TAC  disk_endpoint;
  TYPE_THEN `C'` EXISTS_TAC;
  TYPE_THEN `v''` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 15;
  EXPAND_TAC "D";
  DISCH_THEN_REWRITE;
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  USE 21 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[REWRITE_RULE[ISUBSET] graph_inc_subset];
  DISCH_TAC;
  RIGHT  11 "e";
  REP_BASIC_TAC;
  (* B-  short_end *)
  TYPE_THEN `short_end = ( \ e v. @s. (d e INTER (D v)) s)` ABBREV_TAC ;
  TYPE_THEN `!e v. (graph_edge G e /\ graph_inc G e v) ==> (d e INTER (D v) = {(short_end e v)}) /\ (d_euclid v (short_end e v) = r) /\ (!v'. (graph_inc G e v' /\ ~(v = v') ==> (simple_arc_end (d e) (short_end e v) (short_end e v'))))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TSPEC `e` 11;
  REWR 11;
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G e HAS_SIZE 2` SUBGOAL_TAC;
  IMATCH_MP_TAC graph_edge2;
  UND 4;
  REWRITE_TAC[plane_graph];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `!u. graph_inc G e u ==> (u = v') \/ (u = v'')` SUBGOAL_TAC;
  ASM_MESON_TAC[two_exclusion];
  DISCH_TAC;
  TYPE_THEN `?s. (d e INTER D v) s` SUBGOAL_TAC;
  TSPEC `v` 24;
  REWR 24;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[INR IN_SING ];
  MESON_TAC[];
  ASM_REWRITE_TAC[INR IN_SING ];
  MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `(d e INTER D v) (short_end e v)` SUBGOAL_TAC;
  EXPAND_TAC "short_end";
  SELECT_TAC;
  DISCH_THEN_REWRITE ;
  ASM_MESON_TAC[];
  DISCH_TAC;
  LEFT_TAC "v'";
  LEFT_TAC "v'";
  GEN_TAC;
  TYPE_THEN `(v = v') \/ (v = v'')` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(graph_inc G e v''') ==> (v''' = v') \/ (v''' = v'')` SUBGOAL_TAC;
  DISCH_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* --- *)
  DISCH_THEN DISJ_CASES_TAC;
  FIRST_ASSUM MP_TAC;
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  TYPE_THEN `short_end e v' = u` SUBGOAL_TAC;
  REWR 26;
  USE 26 (REWRITE_RULE[INR IN_SING]);
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  KILL 24;
  REWR 27;
  UND 24;
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `short_end e v'' = u'` SUBGOAL_TAC;
  TYPE_THEN `?s. (d e INTER D v'') s` SUBGOAL_TAC;
  ASM_REWRITE_TAC[INR IN_SING ];
  MESON_TAC[];
  EXPAND_TAC "short_end";
  SELECT_TAC;
  ASM_REWRITE_TAC[INR IN_SING ];
  DISCH_THEN_REWRITE;
  UND 24;
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  (* -- *)
  FIRST_ASSUM MP_TAC;
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  TYPE_THEN `short_end e v'' = u'` SUBGOAL_TAC;
  REWR 26;
  USE 26 (REWRITE_RULE[INR IN_SING]);
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  KILL 24;
  REWR 27;
  UND 24;
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `short_end e v' = u` SUBGOAL_TAC;
  TYPE_THEN `?s. (d e INTER D v') s` SUBGOAL_TAC;
  ASM_REWRITE_TAC[INR IN_SING ];
  MESON_TAC[];
  EXPAND_TAC "short_end";
  SELECT_TAC;
  ASM_REWRITE_TAC[INR IN_SING ];
  DISCH_THEN_REWRITE;
  UND 24;
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  DISCH_TAC;

  (* }}} *)
  (* [C]- *)
  TYPE_THEN `X = (\ v. (IMAGE (\ e. short_end e v) (graph_edge_around G v)))` ABBREV_TAC ;
  TYPE_THEN `!v. FINITE (graph_edge_around G v)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[graph_edge_around];
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `graph_edge G ` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `!v. graph_vertex G v ==> (FINITE (X v) /\ (CARD (X v) <=| 4) /\ ((X v) SUBSET {x | euclid 2 x /\ (d_euclid v x = r)}))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC "X";
  SUBCONJ_TAC;
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  LE_TRANS;
  TYPE_THEN `CARD (graph_edge_around G v)` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  CARD_IMAGE_LE;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET;IMAGE];
  REP_BASIC_TAC;
  UND 18;
  DISCH_THEN_FULL_REWRITE;
  USE 19 (REWRITE_RULE[graph_edge_around]);
  TSPEC `x'` 13;
  TSPEC `v` 13;
  REWR 13;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  UND 19;
  EXPAND_TAC "D";
  REWRITE_TAC[INTER;eq_sing;closed_ball];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* -D now generate curves C in disk.  *)
  TYPE_THEN `!v. (graph_vertex G v) ==> (?C. (!i. X v i                       ==> (?C' C'' v'.                                simple_arc_end C' v v' /\                                simple_arc_end C'' v' i /\                                C' SUBSET                                closed_ball (euclid 2,d_euclid) v (r / &2) /\                                (C' INTER C'' = {v'}) /\                                (C' UNION C'' = C i)) /\                           simple_arc_end (C i) v i /\                           C i SUBSET closed_ball (euclid 2,d_euclid) v r /\                           C i INTER                           closed_ball (euclid 2,d_euclid) v (r / &2) SUBSET                           hyperplane 2 e2 (v 1) UNION hyperplane 2 e1 (v 0)) /\                  (!i j. X v i /\ X v j /\ ~(i = j) ==> (C i INTER C j = {v})))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  degree_vertex_disk_ver2;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(\j. X v j) = X v` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  BETA_TAC;
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  TSPEC `v` 16;
  REWR 16;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  LEFT 17 "C";
  LEFT 17 "C";
  REP_BASIC_TAC;
  TYPE_THEN `f = (\ e. { x | d e x \/ (?v. graph_inc G e v /\ C v (short_end e v) x)})` ABBREV_TAC ;
  (* -[E] lets try to flatten some hypotheses *)
  TYPE_THEN `NC  = (\ e v. (C v (short_end e v)))` ABBREV_TAC ;
  KILL 1;
  KILL 2;
  KILL 3;
  KILL 0;
  (* rework 5 *)
  TYPE_THEN `!e . graph_edge G e ==> (d e SUBSET e)` SUBGOAL_TAC;
  UND 11;
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `!e v. graph_edge G e /\ graph_vertex G v /\ ~graph_inc G e v ==> (d e INTER (D v) = EMPTY)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPEL_THEN [`e`;`v`] (USE 5 o ISPECL);
  REWR 5;
  UND 5;
  UND 0;
  REWRITE_TAC[SUBSET;EQ_EMPTY];
  UND 3;
  EXPAND_TAC "D";
  REWRITE_TAC[INTER];
  MESON_TAC[];
  DISCH_TAC;
  KILL 5;
  KILL 11;
  KILL 12;
  (* rework 16 *)
  TYPE_THEN `!e v. graph_edge G e /\ graph_inc G e v ==> graph_vertex G v` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  TH_INTRO_TAC  [`G`;`e`] graph_inc_subset;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `!e v. (graph_edge G e /\ graph_inc G e v ==> X v (short_end e v))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC "X";
  REWRITE_TAC[IMAGE];
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[graph_edge_around];
  DISCH_TAC;
  KILL 16;
  KILL 14;
  (* rework 17 *)
  TYPE_THEN `hyper = (\ B v. (B INTER closed_ball (euclid 2,d_euclid) v (r / &2) SUBSET hyperplane 2 e2 (v 1) UNION hyperplane 2 e1 (v 0)))` ABBREV_TAC ;
  TYPE_THEN `!e v. graph_edge G e /\ graph_inc G e v ==> (simple_arc_end (NC e v) v (short_end e v)) /\ (NC e v SUBSET D v) /\ (hyper (NC e v) v)` SUBGOAL_TAC;
  EXPAND_TAC "hyper";
  EXPAND_TAC "NC";
  REP_BASIC_TAC;
  TSPEC `v` 17;
  TYPE_THEN `graph_vertex G v` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  REP_BASIC_TAC;
  TSPEC `short_end e v` 16;
  TYPE_THEN `X v (short_end e v)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "D";
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* F- continue simplification *)
  TYPE_THEN `!v v'. graph_vertex G v /\ graph_vertex G v' /\ ~(v = v') ==> (D v INTER D v' = EMPTY)` SUBGOAL_TAC;
  EXPAND_TAC "D";
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  KILL 6;
  (* - *)
  TYPE_THEN `!e v. (graph_edge G e /\ graph_inc G e v ==> ~(graph_vertex G (short_end e v)))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPEL_THEN [`e`;`v`] (USE 13 o ISPECL);
  REWR 13;
  REP_BASIC_TAC;
  USE 21 (REWRITE_RULE[eq_sing;INTER]);
  REP_BASIC_TAC;
  TYPE_THEN `D (short_end e v) (short_end e v)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  RULE_ASSUM_TAC (REWRITE_RULE [plane_graph]);
  REP_BASIC_TAC;
  USE 27 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `~(D (short_end e v) INTER D v = EMPTY)` SUBGOAL_TAC;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `short_end e v` EXISTS_TAC;
  ASM_REWRITE_TAC[INTER];
  REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  USE 25 (REWRITE_RULE[]);
  UND 25;
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `d_euclid v v = &0` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_space_zero;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[metric_euclid];
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  USE 28 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  UND 20;
  UND 7;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* - *)
  TYPE_THEN `!e e'. graph_edge G e /\ graph_edge G e' /\ ~(e = e') ==> (d e INTER d e' = EMPTY)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  USE 21 (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  TYPEL_THEN [`e`;`e'`] (USE 4 o ISPECL);
  REWR 4;
  TYPE_THEN `d e INTER d e' SUBSET graph_vertex G` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `e INTER e'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  subset_inter_pair;
  UND 0;
  UND 20;
  UND 16;
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `graph_vertex G u` SUBGOAL_TAC;
  USE 26 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  USE 21(REWRITE_RULE[INTER]);
  TYPE_THEN `graph_inc G e u` ASM_CASES_TAC;
  TYPEL_THEN [`e`;`u`] (USE 13 o ISPECL);
  REWR 13;
  TYPE_THEN `(d e INTER D u) u` SUBGOAL_TAC;
  REP_BASIC_TAC;
  USE 28 GSYM;
  ASM_REWRITE_TAC[INTER];
  FIRST_ASSUM IMATCH_MP_TAC ;
  USE 25 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  USE 28 GSYM;
  REWR 28;
  USE 28 (REWRITE_RULE[INR IN_SING]);
  UND 28;
  DISCH_THEN (fun t-> ONCE_REWRITE_TAC [t]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `d e INTER D u = EMPTY ` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC [];
  USE 26 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[INTER];
  DISCH_TAC;
  USE 28(REWRITE_RULE[EQ_EMPTY]);
  TSPEC `u` 28;
  DISCH_TAC;
  USE 28(REWRITE_RULE[INTER]);
  UND 28;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  USE 25 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -G continue to simplify *)
  TYPE_THEN `!e v. graph_edge G e /\ graph_inc G e v ==> d e (short_end e v)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPEL_THEN [`e`;`v`] (USE 13 o ISPECL);
  REWR 13;
  REP_BASIC_TAC;
  USE 22(REWRITE_RULE[eq_sing;INTER]);
  ASM_REWRITE_TAC[];
 DISCH_TAC;
  (* - *)
  TYPE_THEN `! v e e'. graph_edge G e /\ graph_edge G e' /\ graph_inc G e v /\ graph_inc G e' v /\ ~(e = e') ==> (NC e v INTER NC e' v = {v})` SUBGOAL_TAC;
  EXPAND_TAC "NC";
  REP_BASIC_TAC;
  TSPEC `v` 17;
  TYPE_THEN `graph_vertex G v` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  REP_BASIC_TAC;
  TYPEL_THEN  [`short_end e v`;`short_end e' v`](USE 17 o ISPECL);
  KILL 25;
  FIRST_ASSUM IMATCH_MP_TAC ;
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  KILL 17;
  DISCH_TAC;
  TYPE_THEN `d e (short_end e v)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `d e' (short_end e' v)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `d e INTER d e' = EMPTY ` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EQ_EMPTY;INTER];
  UND 17;
  MESON_TAC[];
  DISCH_TAC;
  KILL 17;
  KILL 3;
  KILL 15;
  (* H- *)
  TYPE_THEN `!e. f e = {x | d e x \/ (?v. graph_inc G e v /\ NC e v x)}` SUBGOAL_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC "f";
  EXPAND_TAC "NC";
  REWRITE_TAC[];
  DISCH_TAC;
  KILL 18;
  KILL 19;
  TYPE_THEN `!e v p. (graph_edge G e /\ graph_inc G e v) /\ (D v p) ==> (f e p = NC e v  p)` SUBGOAL_TAC  ;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  EQ_TAC;
  UND 17;
  MESON_TAC[];
  DISCH_THEN DISJ_CASES_TAC;
  TYPEL_THEN [`e`;`v`] (USE 13 o ISPECL);
  REWR 13;
  REP_BASIC_TAC;
  USE 22 (REWRITE_RULE[eq_sing;INTER ]);
  REP_BASIC_TAC;
  TSPEC `p` 22;
  REWR 22;
  UND 22;
  DISCH_THEN_FULL_REWRITE;
  TYPEL_THEN [`e`;`v`] (USE 11 o ISPECL);
  REWR 11;
  REP_BASIC_TAC;
  UND 25;
  MESON_TAC[simple_arc_end_end2];
  REP_BASIC_TAC;
  TYPE_THEN `v' = v` ASM_CASES_TAC;
  UND 19;
  ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `D v INTER D v' = {}` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `p` EXISTS_TAC;
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[];
  TYPEL_THEN[`e`;`v'`] (USE 11 o ISPECL);
  REWR 11;
  REP_BASIC_TAC;
  USE 24 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `!e v p. (graph_edge G e /\ (graph_vertex G v) /\ ~(graph_inc G e v) /\ (D v p)  ==> ~(f e p))` SUBGOAL_TAC;
  ASM_REWRITE_TAC[DE_MORGAN_THM ];
  REP_BASIC_TAC;
  CONJ_TAC;
  DISCH_TAC;
  TYPE_THEN `d e INTER D v = EMPTY` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS;INTER  ];
  TYPE_THEN `p` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  LEFT_TAC "v";
  GEN_TAC;
  DISCH_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `~(v = v')` SUBGOAL_TAC;
  DISCH_TAC;
  UND 23;
  UND 18;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `D v INTER D v' = {}` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS;INTER];
  TYPE_THEN `p` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPEL_THEN [`e`;`v'`] (USE 11 o ISPECL);
  REP_BASIC_TAC;
  REWR 11;
  REP_BASIC_TAC;
  USE 25 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `!e p.  graph_edge G e /\ (!v. ~(D v p)) ==> (f e p = d e p)` SUBGOAL_TAC ;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  (TAUT `~B ==> (A \/ B <=> A)`);
  DISCH_TAC;
  REP_BASIC_TAC;
  TSPEC `v` 18;
  UND 18;
  REWRITE_TAC[];
  TYPEL_THEN [`e`;`v`] (USE 11 o ISPECL);
  REWR 11;
  REP_BASIC_TAC;
  USE 18(REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* I- *)
  TYPE_THEN `NC` EXISTS_TAC;
  TYPE_THEN `D` EXISTS_TAC;
  TYPE_THEN `short_end` EXISTS_TAC;
  TYPE_THEN `hyper` EXISTS_TAC;
  TYPE_THEN `r` EXISTS_TAC;
  TYPE_THEN `d` EXISTS_TAC;
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* Sat Aug 21 08:06:22 EDT 2004 *)

  ]);;

  (* }}} *)


let graph_vertex_exhaust = prove_by_refinement(
  `!(G:(A,B)graph_t) e v v'.
  (graph G /\ (graph_edge G e) /\ (graph_inc G e v) /\
     (graph_inc G e v') /\ ~(v = v') ==> (graph_inc G e = {v,v'}))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G e HAS_SIZE 2` SUBGOAL_TAC;
  IMATCH_MP_TAC  graph_edge2;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[has_size2];
  REP_BASIC_TAC;
  UND 6;
  DISCH_THEN_FULL_REWRITE;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[in_pair];
  KILL 3;
  KILL 4;
  RULE_ASSUM_TAC (REWRITE_RULE[in_pair]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)


let graph_disk_hv = prove_by_refinement(
  `!G. plane_graph G /\
      FINITE (graph_edge G) /\ FINITE (graph_vertex G) /\
      ~(graph_edge G = EMPTY) /\
     (!v. (CARD (graph_edge_around G v) <=| 4))
   ==>
    (?r H . graph_isomorphic G H /\ good_plane_graph H /\
      (&0 < r) /\
      (!v v'.
         graph_vertex H v /\ graph_vertex H v' /\ ~(v = v')
         ==> (closed_ball (euclid 2,d_euclid) v r INTER
                closed_ball (euclid 2,d_euclid) v' r =
                {})) /\
      (!e v.
         graph_edge H e /\ graph_vertex H v /\ ~graph_inc H e v
         ==> (e INTER closed_ball (euclid 2,d_euclid) v r = {})) /\
      (!e v.
         graph_edge H e /\  graph_inc H e v
         ==> (e INTER closed_ball (euclid 2, d_euclid) v r SUBSET
            (hyperplane 2 e2 (v 1) UNION hyperplane 2 e1 (v 0))))
    )`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  TH_INTRO_TAC [`G`] graph_disk_hv_preliminaries;
  ASM_REWRITE_TAC[];
  POP_ASSUM_LIST (fun t-> ALL_TAC);
  REP_BASIC_TAC;
  (* - *) (* redo 19 *)
  TYPE_THEN `!e p. graph_edge G e /\ (!v. graph_inc G e v ==> ~(D v p)) ==> (f e p = d e p)` SUBGOAL_TAC;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  (TAUT  `~B ==> (A \/ B <=> A)`);
  DISCH_TAC;
  REP_BASIC_TAC;
  TSPEC `v` 20;
  UND 20;
  ASM_REWRITE_TAC[];
  TYPEL_THEN[`e`;`v`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  USE 20 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  KILL 19;
  (* - *)
  TYPE_THEN `!e e'. graph_edge G e /\ graph_edge G e' /\ ~(e = e') ==> (f e INTER f e' SUBSET e INTER e')` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[SUBSET;INTER ];
  REP_BASIC_TAC;
  TYPE_THEN `?v. (graph_inc G e v /\ D v x)` ASM_CASES_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `f e x = NC e v x` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `graph_inc G e' v` ASM_CASES_TAC;
  TYPE_THEN `f e' x = NC e' v x` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `(NC e v INTER NC e' v = {v})` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[FUN_EQ_THM];
  REWRITE_TAC[INR IN_SING;INTER];
  DISCH_TAC;
  TSPEC `x` 28;
  REWR 28;
  UND 28;
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  TYPE_THEN `e` (WITH 28 o ISPEC);
  TSPEC `e'` 28;
  UND 28;
  UND 32;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  DISCH_THEN_FULL_REWRITE;
  UND 26;
  UND 27;
  REWRITE_TAC[INTER];
  DISCH_THEN_REWRITE;
  PROOF_BY_CONTR_TAC;
  UND 23;
  REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* -- *)
  TYPE_THEN `(f e x = d e x)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  GEN_TAC;
  UND 25;
  MESON_TAC[];
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `(?v. graph_inc G e' v /\ D v x)` ASM_CASES_TAC;
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `d e INTER D v = {}` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  LEFT 25 "v";
  TSPEC `v` 25;
  UND 25;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS;INTER ];
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `f e' x = d e' x` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  GEN_TAC;
  UND 26;
  MESON_TAC[];
  DISCH_THEN_FULL_REWRITE;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `d e INTER d e' = EMPTY` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS ;INTER];
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* A injective *)
  TYPE_THEN `INJ f (graph_edge G) UNIV` SUBGOAL_TAC;
  REWRITE_TAC[INJ];
  REP_BASIC_TAC;
  TYPE_THEN ` (graph_inc G x ) HAS_SIZE 2` SUBGOAL_TAC;
  IMATCH_MP_TAC  graph_edge2;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[has_size2];
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G x a` SUBGOAL_TAC;
  ASM_REWRITE_TAC[in_pair];
  DISCH_TAC;
  TYPE_THEN `d x SUBSET f x` SUBGOAL_TAC;
  KILL 21;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `d x (short_end x a)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `f x (short_end x a)` SUBGOAL_TAC;
  UND 28;
  UND 27;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `f x INTER f y SUBSET  x INTER y` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `(x INTER y) (short_end x a)` SUBGOAL_TAC;
  USE 31 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  USE 21 GSYM;
  KILL 16;
  ASM_REWRITE_TAC[INTER_IDEMPOT];
  TYPE_THEN `(x INTER y) SUBSET (graph_vertex G)` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `(graph_vertex G (short_end x a))` SUBGOAL_TAC;
  USE 33(REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* B now simple arc -- ugh *)
  TYPE_THEN `(!e v v'. (graph_edge G e /\ graph_inc G e v /\ graph_inc G e v' /\ ~(v = v') ==> (simple_arc_end (f e) v v')))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `f e = (NC e v UNION d e) UNION NC e v'` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[UNION];
  ONCE_REWRITE_TAC [EQ_SYM_EQ;];
  REWRITE_TAC[GSYM DISJ_ASSOC];
  EQ_TAC;
  REP_CASES_TAC;
  DISJ2_TAC;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  DISJ2_TAC;
  TYPE_THEN `v'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_CASES_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G e = {v , v'}` SUBGOAL_TAC;
  IMATCH_MP_TAC  graph_vertex_exhaust;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWR 27;
  USE 27 (REWRITE_RULE[in_pair]);
  UND 27;
  REP_CASES_TAC;
  UND 27;
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  UND 27;
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  DISCH_THEN (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])) THEN ASSUME_TAC t);
  (* -- *)
  TYPE_THEN `simple_arc_end (NC e v UNION d e) v (short_end e v')` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_end_trans;
  TYPE_THEN `short_end e v` EXISTS_TAC;
  CONJ_TAC;
  TYPEL_THEN [`e`;`v`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  TYPEL_THEN [`e`;`v`] (USE 5 o ISPECL);
  REWR 5;
  REP_BASIC_TAC;
  TSPEC `v'` 5;
  REWR 5;
  (* --- *)
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING;INTER ];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  GEN_TAC;
  EQ_TAC;
  DISCH_THEN_FULL_REWRITE;
  CONJ_TAC;
  TYPE_THEN `simple_arc_end (NC e v) v (short_end e v)` SUBGOAL_TAC;
  TYPEL_THEN [`e`;`v`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  MESON_TAC[simple_arc_end_end2];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* --- *)
  DISCH_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `D v x` SUBGOAL_TAC;
  TYPEL_THEN [`e`;`v`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  USE 29 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `d e INTER D v = {(short_end e v)}` SUBGOAL_TAC;
  TYPEL_THEN [`e`;`v`] (USE 5 o ISPECL);
  REWR 5;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[eq_sing];
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  IMATCH_MP_TAC  simple_arc_end_trans;
  TYPE_THEN `(short_end e v')` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  TYPEL_THEN [`e`;`v'`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INTER];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING];
  GEN_TAC;
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  EQ_TAC;
  DISCH_THEN_FULL_REWRITE;
  CONJ_TAC;
  UND 27;
  MESON_TAC[simple_arc_end_end2];
  TYPEL_THEN[`e`;`v'`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  UND 29;
  MESON_TAC[simple_arc_end_end2];
  REP_BASIC_TAC;
  UND 29;
  REWRITE_TAC[UNION];
  REP_CASES_TAC ;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `D v INTER D v' = {}` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS;INTER];
  TYPE_THEN `x` EXISTS_TAC;
  CONJ_TAC;
  TYPEL_THEN [`e`;`v`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  USE 31 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPEL_THEN [`e`;`v'`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  USE 31 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `D v' x` SUBGOAL_TAC;
  TYPEL_THEN [`e`;`v'`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  USE 30 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `d e INTER D v' = {(short_end e v')}` SUBGOAL_TAC;
  TYPEL_THEN [`e`;`v'`] (USE 5 o ISPECL);
  REWR 5;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INTER;eq_sing];
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* C - *)
  TYPE_THEN `!e v. (graph_edge G e) ==> ( e INTER graph_vertex G = (f e) INTER (graph_vertex G))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INTER];
  GEN_TAC;
  IMATCH_MP_TAC  (TAUT `(A ==> (B <=> C)) ==> (B /\ A <=> C /\ A)`);
  DISCH_TAC;
  TYPE_THEN `D x x` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph;SUBSET ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `graph_inc G e x` ASM_CASES_TAC;
  TYPE_THEN `f e x = NC e x x` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `NC e x x` SUBGOAL_TAC;
  TYPEL_THEN[`e`;`x`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  UND 28;
  MESON_TAC[simple_arc_end_end];
  DISCH_THEN_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  TSPEC `e` 27;
  REWR 27;
  REWR 26;
  UND 26;
  REWRITE_TAC[INTER];
  DISCH_THEN_REWRITE;
  TYPE_THEN `~f e x` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  UND 26;
  REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  TSPEC `e` 26;
  REWR 26;
  ASM_REWRITE_TAC[INTER];
  DISCH_TAC;
  (* D start on graph and goal *)
  TYPE_THEN `r /(&2)` EXISTS_TAC;
  TYPE_THEN `graph_edge_mod G f` EXISTS_TAC;
  REWRITE_TAC[good_plane_graph];
  ASM_REWRITE_TAC[REAL_LT_HALF1];
  CONJ_TAC;
  IMATCH_MP_TAC  graph_edge_iso;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[TAUT `(A /\ B) /\ C <=> (A /\ (B /\ C))`];
  (* - *)
  CONJ_TAC;
  IMATCH_MP_TAC  plane_graph_mod;
  USE 16 GSYM;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  simple_arc_end_simple;
  TH_INTRO_TAC [`G`;`e`] graph_edge_end_select;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);  (* --x-- *)
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `v` EXISTS_TAC;
  TYPE_THEN `v'` EXISTS_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* - *)
  CONJ_TAC;
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_edge_mod_e;graph_edge_mod_i]);
  REP_BASIC_TAC;
  USE 29 GSYM;
  UND 29;
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `e'' =e'` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* - *)
  TYPE_THEN `!v. closed_ball (euclid 2, d_euclid) v (r/(&2)) SUBSET D v` SUBGOAL_TAC;
  GEN_TAC;
  EXPAND_TAC "D";
  REWRITE_TAC[closed_ball;SUBSET];
  TYPE_THEN `r /(&2) < r` SUBGOAL_TAC;
  UND 1;
  MESON_TAC[  half_pos];
  MESON_TAC[REAL_ARITH `x <= u /\ u < v ==> x <= v`];
  DISCH_TAC;
  (* - *)
  CONJ_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `D v INTER D v'` EXISTS_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  subset_inter_pair;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[graph_edge_mod_v]);
  TYPE_THEN `(D v INTER D v' = EMPTY)` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[];
  (* E - down to 2 *)
  CONJ_TAC;
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_edge_mod_v;graph_edge_mod_i;graph_edge_mod_e]);
  USE 27 (REWRITE_RULE[IMAGE]);
  REP_BASIC_TAC;
  UND 27;
  DISCH_THEN_FULL_REWRITE;
  LEFT 25 "e'";
  TSPEC `x` 25;
  PROOF_BY_CONTR_TAC;
  USE 27(REWRITE_RULE[EMPTY_EXISTS;INTER]);
  REP_BASIC_TAC;
  TYPE_THEN `D v u` SUBGOAL_TAC;
  USE 24 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `~f x u` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 25;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  (* - final *)
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_edge_mod_i;graph_edge_mod_e]);
  USE 26 (REWRITE_RULE[IMAGE]);
  REP_BASIC_TAC;
  UND 28;
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `e' = x` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  (* - *)
  TYPE_THEN `f x INTER D v = NC x v INTER D v` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INTER];
  IMATCH_MP_TAC  (TAUT `(A ==> (B <=> C)) ==> (B /\ A <=> C /\ A)`);
  DISCH_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `f x INTER (closed_ball (euclid 2,d_euclid) v (r/(&2))) = NC x v INTER (closed_ball(euclid 2, d_euclid) v (r/(&2)))` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INTER];
  USE 28 (REWRITE_RULE[FUN_EQ_THM]);
  TSPEC `x'` 28;
  UND 28;
  UND 24;
  REWRITE_TAC[SUBSET;INTER];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPEL_THEN[`x`;`v`] (USE 10 o ISPECL);
  REWR 10;
  REP_BASIC_TAC;
  UND 10;
  EXPAND_TAC "hyper";
  DISCH_THEN_REWRITE;
  (* Sat Aug 21 14:12:41 EDT 2004 *)

  ]);;

  (* }}} *)

let hv_finite = jordan_def `hv_finite C <=>
   (?E. C SUBSET UNIONS E /\ FINITE E /\ hv_line E)`;;

let hv_finite_subset = prove_by_refinement(
  `!A B. hv_finite B /\ A SUBSET B ==> hv_finite A`,
  (* {{{ proof *)
  [
  REWRITE_TAC[hv_finite];
  REP_BASIC_TAC;
  TYPE_THEN `E` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `B` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let mk_line_hyper2_e1 = prove_by_refinement(
  `!z. mk_line (point (z, &0)) (point(z, &1)) = hyperplane 2 e1 z`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM line2D_F;e1;mk_line;];
  GEN_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[point_scale;point_add];
  GEN_TAC;
  REDUCE_TAC;
  TYPE_THEN `!t. t * z + (&1 - t) * z = z` SUBGOAL_TAC;
  GEN_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  EQ_TAC;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[point_inj;PAIR_SPLIT];
  TYPE_THEN `(z, &1 - t)` EXISTS_TAC;
  REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[point_inj;PAIR_SPLIT];
  TYPE_THEN `&1 - (SND p)` EXISTS_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let mk_line_hyper2_e2 = prove_by_refinement(
  `!z. mk_line (point (&0, z)) (point(&1, z)) = hyperplane 2 e2 z`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM line2D_S;e2;mk_line;];
  GEN_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[point_scale;point_add];
  GEN_TAC;
  REDUCE_TAC;
  TYPE_THEN `!t. t * z + (&1 - t) * z = z` SUBGOAL_TAC;
  GEN_TAC;
  real_poly_tac;
  DISCH_THEN_REWRITE;
  EQ_TAC;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[point_inj;PAIR_SPLIT];
  TYPE_THEN `( &1 - t, z)` EXISTS_TAC;
  REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[point_inj;PAIR_SPLIT];
  TYPE_THEN `&1 - (FST  p)` EXISTS_TAC;
  REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let hv_finite_hyper = prove_by_refinement(
  `!C.
  (?v. C SUBSET (hyperplane 2 e2 (v 1) UNION hyperplane 2 e1 (v 0))) ==>
   (hv_finite C)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[hv_finite];
  TYPE_THEN `{(hyperplane 2 e2 (v 1)), (hyperplane 2 e1 (v 0))}` EXISTS_TAC ;
  ASM_REWRITE_TAC[UNIONS_2;FINITE_INSERT;FINITE_SING;FINITE_RULES; ];
  REWRITE_TAC[hv_line;in_pair;GSYM mk_line_hyper2_e2;GSYM mk_line_hyper2_e1];
  GEN_TAC;
  REP_CASES_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `(v 0, &0)` EXISTS_TAC;
  TYPE_THEN `(v 0, &1)` EXISTS_TAC;
  REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  TYPE_THEN `(&0, v 1)` EXISTS_TAC;
  TYPE_THEN `(&1, v 1)` EXISTS_TAC;
  REWRITE_TAC[];
  ]);;

   (* }}} *)

let graph_hv_finite_radius = jordan_def
  `graph_hv_finite_radius G r <=> (good_plane_graph G /\
      (&0 < r) /\
      (!v v'.
         graph_vertex G v /\ graph_vertex G v' /\ ~(v = v')
         ==> (closed_ball (euclid 2,d_euclid) v r INTER
                closed_ball (euclid 2,d_euclid) v' r =
                {})) /\
      (!e v.
         graph_edge G e /\ graph_vertex G v /\ ~graph_inc G e v
         ==> (e INTER closed_ball (euclid 2,d_euclid) v r = {})) /\
      (!e v.
         graph_edge G e /\  graph_inc G e v
         ==> (hv_finite (e INTER closed_ball (euclid 2, d_euclid) v r))))
    `;;

let p_conn_hv_finite = prove_by_refinement(
  `!A x y. ~(x = y) ==>
     (p_conn A x y <=> (?C. (hv_finite C) /\ (C SUBSET A) /\
    (simple_arc_end C x y)))`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  REWRITE_TAC[p_conn;simple_polygonal_arc];
  (* - *)
  EQ_TAC;
  REP_BASIC_TAC;
  TH_INTRO_TAC [`C`;`x`;`y`] simple_arc_end_select;
  ASM_REWRITE_TAC[top2];
  REP_BASIC_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  REWRITE_TAC[hv_finite];
  CONJ_TAC;
  TYPE_THEN `E` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[hv_finite]);
  REP_BASIC_TAC;
  TYPE_THEN `C` EXISTS_TAC;
  CONJ_TAC;
  CONJ_TAC;
  REWRITE_TAC[GSYM top2];
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  ASM_MESON_TAC[simple_arc_end_end;simple_arc_end_end2];
  ]);;

  (* }}} *)


let graph_iso_around = prove_by_refinement(
  `!(G:(A,B)graph_t) (H:(A',B')graph_t) f v. (graph G) /\
     graph_iso f G H /\ (graph_vertex G v) ==>
        (graph_edge_around H (FST  f v) =
            (IMAGE (SND f) (graph_edge_around G v)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_iso;graph_edge_around];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REP_BASIC_TAC;
  REWRITE_TAC[];
  EQ_TAC ;
  REP_BASIC_TAC;
  TYPE_THEN `(?y. graph_edge G y /\ (v' y = x))` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  USE 8 GSYM;
  UND 8;
  DISCH_THEN_FULL_REWRITE;
  TSPEC `y` 1;
  REWR 1;
  REWRITE_TAC[IMAGE];
  TYPE_THEN `y` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWR 6;
  USE 6 (REWRITE_RULE[IMAGE]);
  REP_BASIC_TAC;
  TYPE_THEN `v = x'` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TH_INTRO_TAC [`G`;`y`] graph_inc_subset;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_THEN IMATCH_MP_TAC  ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE];
  REP_BASIC_TAC;
  REWR 6;
  UND 6;
  DISCH_THEN_FULL_REWRITE;
  SUBCONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  ASM_SIMP_TAC[];
  REWRITE_TAC[IMAGE];
  REP_BASIC_TAC;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* Sat Aug 21 16:49:58 EDT 2004 *)

  ]);;
  (* }}} *)

let graph_radius_exists = prove_by_refinement(
  `!G. planar_graph (G:(A,B) graph_t) /\
      FINITE (graph_edge G) /\ FINITE (graph_vertex G) /\
      ~(graph_edge G = EMPTY) /\
     (!v. (CARD (graph_edge_around G v) <=| 4)) ==>
   (?r H.
       (graph_isomorphic G H /\ graph_hv_finite_radius H r))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[planar_graph]);
  REP_BASIC_TAC;
  TYPE_THEN `FINITE (graph_edge H) /\ FINITE (graph_vertex H) /\ ~(graph_edge H = EMPTY) /\  (!v. (CARD (graph_edge_around H v) <=| 4))` SUBGOAL_TAC;
  WITH 4 (REWRITE_RULE[graph_isomorphic]);
  REP_BASIC_TAC;
  SUBCONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_iso]);
  REP_BASIC_TAC;
  TH_INTRO_TAC [`graph_edge H`;`graph_edge G`;`v`] FINITE_BIJ2;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* -- *)
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_iso]);
  REP_BASIC_TAC;
  TH_INTRO_TAC [`graph_vertex H`;`graph_vertex G`;`u`] FINITE_BIJ2;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  CONJ_TAC;
  REWRITE_TAC[EMPTY_EXISTS];
  RULE_ASSUM_TAC (REWRITE_RULE[EMPTY_EXISTS]);
  REP_BASIC_TAC;
   RULE_ASSUM_TAC (REWRITE_RULE[graph_iso;BIJ;SURJ]);
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  GEN_TAC;
  (* -- *)
  TYPE_THEN `graph_vertex H v` ASM_CASES_TAC;
  TH_INTRO_TAC [`H`;`G`;`f`;`v`] graph_iso_around;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_iso]);
  REP_BASIC_TAC;
  UND 12;
  DISCH_THEN_FULL_REWRITE;
  TSPEC `u v` 0;
  REWR 0;
  TH_INTRO_TAC [`v'`;`graph_edge_around H v`] CARD_IMAGE_INJ;
  REWRITE_TAC[];
  CONJ_TAC;
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ;BIJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[graph_edge_around]);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `graph_edge H` EXISTS_TAC ;
  ASM_REWRITE_TAC[SUBSET;graph_edge_around];
  MESON_TAC[];
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  TH_INTRO_TAC [`H`;`v`] graph_edge_around_empty;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[CARD_CLAUSES];
  ARITH_TAC;
  REP_BASIC_TAC;
  (* - *)
  TH_INTRO_TAC [`H`] graph_disk_hv;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `r` EXISTS_TAC;
  TYPE_THEN `H'` EXISTS_TAC;
  REWRITE_TAC[graph_hv_finite_radius];
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  TH_INTRO_TAC [`G`;`H`;`H'`] graph_isomorphic_trans;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  graph_isomorphic_symm;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  (* - *)
  REP_BASIC_TAC;
  TYPEL_THEN [`e`;`v`] (USE 10 o ISPECL);
  REWR 10;
  IMATCH_MP_TAC  hv_finite_hyper;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* Sat Aug 21 17:28:09 EDT 2004 *)

  ]);;
  (* }}} *)

let replace = jordan_def `replace (x:A) y =
    (\ z. (if (z  = x) then y else z))`;;

let replace_x = prove_by_refinement(
  `!(x:A) y. replace x y x = y`,
  (* {{{ proof *)
  [
  REWRITE_TAC[replace];
  (* Sun Aug 22 09:01:27 EDT 2004 *)

  ]);;
  (* }}} *)

let graph_replace = jordan_def
   `graph_replace (G:(A,B)graph_t) e e' =
     graph_edge_mod G (replace e e')`;;

let replace_inj = prove_by_refinement(
  `!(x:A) y Z. ~(Z y) ==> INJ (replace x y) Z UNIV`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ;replace];
  REP_BASIC_TAC;
  MP_TAC (TAUT  `((x' = (x:A)) \/ ~(x' = x)) /\ ((y' = x) \/ ~(y' = x))`);
  REWRITE_TAC[RIGHT_AND_OVER_OR;LEFT_AND_OVER_OR];
  REP_CASES_TAC THEN (REWR 0);
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let graph_replace_iso = prove_by_refinement(
  `!(G:(A,B)graph_t) e e'.
      ~(graph_edge G e') ==> graph_isomorphic G (graph_replace G e e')`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_replace];
  REP_BASIC_TAC;
  IMATCH_MP_TAC  graph_edge_iso;
  IMATCH_MP_TAC  replace_inj;
  ASM_REWRITE_TAC[];
  (* Sun Aug 22 09:30:14 EDT 2004 *)

  ]);;
  (* }}} *)

let graph_replace_plane = prove_by_refinement(
  `!G e e'. plane_graph G /\ ~(graph_edge G e') /\
      (graph_edge G e) /\
      (!e''. graph_edge G e'' /\ ~(e'' = e) ==>
           (e' INTER e'' SUBSET  e INTER e'')) /\
      (simple_arc top2 e') /\
      (e INTER graph_vertex G = e' INTER graph_vertex G) ==>
      plane_graph (graph_replace G e e')`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[graph_replace];
  IMATCH_MP_TAC  plane_graph_mod;
  ASM_REWRITE_TAC[];
  (* - *)
  CONJ_TAC;
  IMATCH_MP_TAC  replace_inj;
  ASM_REWRITE_TAC[];
  (* - *)
  CONJ_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[replace];
  TYPE_THEN `((e'' = e) \/ ~(e'' = e)) /\ ((e''' = e) \/ ~(e''' = e))` (fun t-> MP_TAC (TAUT t));
  REWRITE_TAC[RIGHT_AND_OVER_OR;LEFT_AND_OVER_OR];
  REP_CASES_TAC THEN (FIRST_ASSUM (fun t-> REWRITE_TAC[t] THEN (RULE_ASSUM_TAC (REWRITE_RULE[t])) THEN (ASSUME_TAC t)));
  ASM_MESON_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  ONCE_REWRITE_TAC [INTER_COMM];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET_REFL];
  (* - *)
  CONJ_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[replace];
  COND_CASES_TAC;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph;SUBSET ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* - *)
  REP_BASIC_TAC;
  REWRITE_TAC[replace];
  COND_CASES_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[];
  (* Sun Aug 22 10:28:15 EDT 2004 *)

  ]);;
  (* }}} *)

let good_replace = prove_by_refinement(
  `!G e e'. (good_plane_graph G) /\ plane_graph (graph_replace G e e') /\
      ~(graph_edge G e') /\
   ( e INTER (graph_vertex G) = e' INTER (graph_vertex G)) /\
      (!v v'. (graph_vertex G v) /\ (graph_vertex G v') /\
            ~(v = v') /\ e' v /\  e' v' ==> simple_arc_end e' v v')
    ==> (good_plane_graph (graph_replace G e e'))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[good_plane_graph;graph_replace];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_edge_mod_e;graph_edge_mod_i ;IMAGE ]);
  REP_BASIC_TAC;
  UND 6;
  DISCH_THEN_FULL_REWRITE;
  TH_INTRO_TAC [`e`;`e'`;`graph_edge G`] replace_inj;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `e'''' = x` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `e''' = x` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  (* - *)
  REWRITE_TAC[replace];
  COND_CASES_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UNDF `x`;
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G e = e INTER graph_vertex G` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  UNDF `e INTER u = e' INTER u`;
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[INTER;]);
  ASM_REWRITE_TAC[];
  (* - *)
  KILL 0;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* Sun Aug 22 10:59:34 EDT 2004 *)

  ]);;
  (* }}} *)

let graph_replace_hv_finite_radius = prove_by_refinement(
  `!G r e e'. graph_hv_finite_radius G r /\ ~(graph_edge G e') /\
     good_plane_graph (graph_replace G e e') /\
    (e INTER (graph_vertex G) = e' INTER (graph_vertex G)) /\
    (!v. graph_vertex G v /\ ~(e' v) ==>
        ((e' INTER closed_ball (euclid 2,d_euclid) v r = {}))) /\
    (hv_finite e')
    ==> graph_hv_finite_radius (graph_replace G e e') r`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_hv_finite_radius];
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  (* - *)
  CONJ_TAC;
  REP_BASIC_TAC;
  UND 7;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[graph_replace ;graph_edge_mod_v]);
  ASM_REWRITE_TAC[];
  (* - *)
  CONJ_TAC;
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_replace;graph_edge_mod_v;IMAGE;graph_edge_mod_i;graph_edge_mod_e]);
  REP_BASIC_TAC;
  UNDF `e''`;
  DISCH_THEN_FULL_REWRITE;
  REWRITE_TAC[replace];
  COND_CASES_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWR 13;
  DISCH_TAC;
  LEFT 10 "e'''";
  TSPEC `e` 10;
  UND 10;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph;plane_graph]);
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G e = e INTER graph_vertex G` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[INTER];
  KILL 1;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  LEFT 10 "e'''";
  TSPEC `x` 1;
  REWR 1;
  (* - *)
  REP_BASIC_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[graph_replace;graph_edge_mod_v;IMAGE;graph_edge_mod_i;graph_edge_mod_e]);
  REP_BASIC_TAC;
  UNDF `e''`;
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `e''' = x` SUBGOAL_TAC;
  TH_INTRO_TAC [`e`;`e'`;`graph_edge G`] replace_inj;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INJ];
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  (* - *)
  REWRITE_TAC[replace];
  COND_CASES_TAC ;
  UNDF `x`;
  DISCH_THEN_FULL_REWRITE;
  IMATCH_MP_TAC  hv_finite_subset;
  TYPE_THEN `e'` EXISTS_TAC;
  ASM_REWRITE_TAC[INTER;SUBSET;];
  MESON_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* Sun Aug 22 12:09:03 EDT 2004 *)

  ]);;
  (* }}} *)

let card_suc_insert = prove_by_refinement(
  `!(x:A) s. FINITE s /\ (~(s x)) ==> (SUC (CARD s) = CARD(x INSERT s))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  ASM_SIMP_TAC [CARD_CLAUSES];
  ]);;
  (* }}} *)

let graph_replace_card = prove_by_refinement(
  `!G e e'.
    (FINITE (graph_edge (G:(A,(num->real)->bool)graph_t))) /\
      (graph_edge G e) /\ ~(graph_edge G e') /\
     ~(hv_finite e) /\ (hv_finite e') ==>
   (CARD {x | graph_edge (graph_replace G e e') x /\ ~(hv_finite x)} <
      CARD{ x | graph_edge G x /\ ~hv_finite x})
                                                `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  (ARITH_RULE `(SUC x = y) ==> (x <| y)`);
  (* - *)
  TYPE_THEN `FINITE (graph_edge (graph_replace G e e'))` SUBGOAL_TAC;
  REWRITE_TAC[graph_edge_mod_e;graph_replace];
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `A = {x | graph_edge (graph_replace G e e') x /\ ~hv_finite x}` ABBREV_TAC ;
  TYPE_THEN `FINITE A` SUBGOAL_TAC;
  EXPAND_TAC "A";
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `graph_edge (graph_replace G e e')` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "A";
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `~A e` SUBGOAL_TAC;
  EXPAND_TAC"A";
  REWRITE_TAC[];
  ASM_REWRITE_TAC[graph_replace;graph_edge_mod_e;IMAGE];
  DISCH_TAC;
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[replace]);
  UND 8;
  COND_CASES_TAC;
  ASM_MESON_TAC[];
  UND 8;
  REWRITE_TAC[];
  MESON_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `SUC (CARD A) = CARD(e INSERT A)` SUBGOAL_TAC;
  IMATCH_MP_TAC  card_suc_insert;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  (* - *)
  AP_TERM_TAC;
  EXPAND_TAC "A";
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[INSERT;graph_replace;graph_edge_mod_e;IMAGE;replace; ];
  EQ_TAC;
  REP_BASIC_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  REP_BASIC_TAC;
  UNDF `x = u`;
  DISCH_THEN_FULL_REWRITE;
  COND_CASES_TAC;
  UNDF `x' = e`;
  DISCH_THEN_FULL_REWRITE;
  ASM_MESON_TAC[];
  REWR 10;
  UNDF `x = e`;
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  (* - *)
  REP_BASIC_TAC;
  TYPE_THEN `x = e` ASM_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let graph_edge_end_select_other = prove_by_refinement(
  `!(G:(A,B)graph_t) e v. (graph G /\ graph_edge G e /\
         (graph_inc G e v) ==>
    (?v'. (graph_inc G e v' /\ ~(v = v'))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TH_INTRO_TAC [`G`;`e`] graph_edge_end_select;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc G e HAS_SIZE 2` SUBGOAL_TAC;
  IMATCH_MP_TAC  graph_edge2;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[has_size2];
  REP_BASIC_TAC;
  UND 7;
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[in_pair]);
  REWRITE_TAC[in_pair];
  TYPE_THEN `(v'' = b)` ASM_CASES_TAC;
  UNDF `v''`;
  DISCH_THEN_FULL_REWRITE;
  REWR 5;
  UNDF`v'`;
  DISCH_THEN_FULL_REWRITE;
  ASM_MESON_TAC[];
  REWR 4;
  UNDF`v''`;
  DISCH_THEN_FULL_REWRITE;
  REWR 5;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let graph_rad_pt_select = prove_by_refinement(
  `!G r e v. graph_hv_finite_radius G r /\ graph_inc G e v  /\
     graph_edge G e ==>
     (?C u. (hv_finite C) /\ (simple_arc_end C v u) /\ (euclid 2 u) /\
        (d_euclid v u = r) /\ (C SUBSET e) /\ (C SUBSET (closed_ball(euclid 2,d_euclid) v r)))   `,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_hv_finite_radius];
  REP_BASIC_TAC;
  (* - *)
  TH_INTRO_TAC [`e`;`{v}`;`(euclid 2 DIFF (open_ball(euclid 2,d_euclid) v r))`] simple_arc_end_restriction;
  (* -- *)
    CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE [good_plane_graph;plane_graph;SUBSET ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* -- *)
  TH_INTRO_TAC[`G`;`e`;`v`] graph_edge_end_select_other;
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph;plane_graph]);
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* -- *)
  CONJ_TAC;
  RULE_ASSUM_TAC  (REWRITE_RULE[good_plane_graph;plane_graph]);
  REP_BASIC_TAC;
  IMATCH_MP_TAC simple_arc_end_end_closed;
  TYPE_THEN `e` EXISTS_TAC;
  TYPE_THEN `v'` EXISTS_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* -- *)
  CONJ_TAC;
  TH_INTRO_TAC [`top2`;`open_ball(euclid 2,d_euclid) v r`] open_closed;
  REWRITE_TAC[top2_top];
  ASM_SIMP_TAC [top2;open_ball_open;metric_euclid;open_DEF ];
  REWRITE_TAC[top2_unions];
  (* -- *)
  CONJ_TAC;
  REWRITE_TAC[INTER;DIFF;EQ_EMPTY;open_ball;INR IN_SING ];
  REP_BASIC_TAC;
  UNDF  `x = v`;
  DISCH_THEN_FULL_REWRITE;
  UNDF `x < r`;
  ASM_REWRITE_TAC[];
  TYPE_THEN `d_euclid v v = &0` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_space_zero;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[metric_euclid];
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  (* -- *)
  CONJ_TAC;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `v` EXISTS_TAC;
  REWRITE_TAC[INTER;INR IN_SING];
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph;plane_graph]);
  REP_BASIC_TAC;
  UNDF `graph_inc G e = y`;
  DISCH_THEN (TH_INTRO_TAC [`e`]);
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[INTER]);
  ASM_REWRITE_TAC[];
  (* -- *)
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `v'` EXISTS_TAC;
  REWRITE_TAC[INTER];
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph;plane_graph]);
  REP_BASIC_TAC;
  UNDF `graph_inc G e = y`;
  DISCH_THEN (TH_INTRO_TAC [`e`]);
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[INTER]);
  ASM_REWRITE_TAC[];
  (* -- *)
  REWRITE_TAC[DIFF];
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph;plane_graph;SUBSET]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TH_INTRO_TAC [`G`;`e`] graph_inc_subset;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[open_ball;DE_MORGAN_THM ];
  DISJ2_TAC;
  DISJ2_TAC;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!v. graph_inc G e v ==> graph_vertex G v` SUBGOAL_TAC;
  TH_INTRO_TAC [`G`;`e`] graph_inc_subset;
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph;plane_graph;SUBSET]);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `!v. graph_inc G e v ==> euclid 2 v` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph;plane_graph;SUBSET]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  UND 4;
  DISCH_THEN (  TH_INTRO_TAC [`v`;`v'`] );
  ASM_MESON_TAC [];
  REWRITE_TAC[INTER;EMPTY_EXISTS];
  TYPE_THEN `v` EXISTS_TAC;
  REWRITE_TAC[closed_ball];
  TYPE_THEN `euclid 2 v` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `euclid 2 v'` SUBGOAL_TAC;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `d_euclid v v = &0` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_space_zero;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_MESON_TAC[metric_euclid];
  DISCH_THEN_REWRITE;
  UND 5;
  UND 9;
  TYPE_THEN `d_euclid v v' = d_euclid v' v` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_space_symm;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_MESON_TAC[metric_euclid];
  DISCH_THEN_REWRITE;
  REAL_ARITH_TAC;
  (* A- *)
  REP_BASIC_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  TYPE_THEN `v''` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `v' = v` SUBGOAL_TAC;
  UND 8;
  REWRITE_TAC[INTER;eq_sing;INR IN_SING ];
  MESON_TAC[];
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  (* - *)
  TYPE_THEN `euclid 2 v''` SUBGOAL_TAC;
  FIRST_ASSUM MP_TAC;
  REWRITE_TAC[INTER;DIFF;eq_sing;];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  (* - *)
  TYPE_THEN `d_euclid v v'' = r` SUBGOAL_TAC;
  IMATCH_MP_TAC  disk_endpoint_outer;
  TYPE_THEN `C'` EXISTS_TAC;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  TH_INTRO_TAC [`C'`] simple_arc_euclid;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_THEN IMATCH_MP_TAC ;
  UND 9;
  MESON_TAC[simple_arc_end_end];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  (* B- *)
  TYPE_THEN `C' SUBSET closed_ball(euclid 2,d_euclid) v r` SUBGOAL_TAC;
  UND 7;
  REWRITE_TAC[SUBSET;closed_ball;INTER;open_ball;DIFF;eq_sing;INR IN_SING];
  REP_BASIC_TAC;
  TYPE_THEN `!x. C' x ==> euclid 2 x` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TH_INTRO_TAC[`C'`] simple_arc_euclid;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `C' v` SUBGOAL_TAC;
  UND 8;
  REWRITE_TAC[INTER;INR IN_SING;eq_sing;];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `x = v''` ASM_CASES_TAC;
  UNDF `x = v''`;
  DISCH_THEN_FULL_REWRITE;
  UND 12;
  REAL_ARITH_TAC;
  TSPEC `x` 13;
  PROOF_BY_CONTR_TAC;
  UND 19;
  REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  SUBCONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[DE_MORGAN_THM];
  DISJ2_TAC;
  UND 20;
  REAL_ARITH_TAC;
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  hv_finite_subset;
  TYPE_THEN `e INTER (closed_ball(euclid 2,d_euclid) v r)` EXISTS_TAC;
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET_INTER];
  ASM_REWRITE_TAC[];
  (* Sun Aug 22 15:50:58 EDT 2004 *)

  ]);;

  (* }}} *)

(* not needed here *)
let top_union = prove_by_refinement(
  `!A B U. topology_ U /\ U A /\ U (B:A->bool) ==> U(A UNION B)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[GSYM UNIONS_2];
  IMATCH_MP_TAC  top_unions;
  ASM_REWRITE_TAC[in_pair; SUBSET;];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let top_closed_unions = prove_by_refinement(
  `!(B:(A->bool)->bool) U.
     topology_ U /\ FINITE B /\ B SUBSET (closed_ U) ==>
            closed_ U(UNIONS B)`,
  (* {{{ proof *)
  [
  TYPE_THEN `!n (B:(A->bool)->bool) U. (CARD B = n) /\  topology_ U /\ FINITE B /\ B SUBSET (closed_ U) ==> closed_ U(UNIONS B)` SUBGOAL_TAC;
  INDUCT_TAC;
  REP_BASIC_TAC;
  TYPE_THEN `B HAS_SIZE 0` SUBGOAL_TAC;
  ASM_REWRITE_TAC[HAS_SIZE];
  REWRITE_TAC[HAS_SIZE_0];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  empty_closed;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* -- *)
  TYPE_THEN `~(B = EMPTY)` SUBGOAL_TAC;
  DISCH_TAC;
  UNDF `EMPTY`;
  DISCH_THEN_FULL_REWRITE;
  UNDF `SUC`;
  REWRITE_TAC[CARD_CLAUSES];
  ARITH_TAC;
  DISCH_TAC;
  (* -- *)
  TH_INTRO_TAC [`B`] CARD_DELETE_CHOICE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  USEF `SUC` SYM;
  REWR 4;
  RULE_ASSUM_TAC (REWRITE_RULE[SUC_INJ]);
  TYPEL_THEN [`(B DELETE CHOICE B)`;`U`] (USE 0 o ISPECL);
  UNDF `n`;
  DISCH_THEN (TH_INTRO_TAC []);
  ASM_REWRITE_TAC[FINITE_DELETE];
  UNDF `(SUBSET)`;
  REWRITE_TAC[SUBSET;DELETE];
  MESON_TAC[];
  (* -- *)
  DISCH_TAC;
  TYPE_THEN `closed_ U( UNIONS (B DELETE CHOICE B) UNION (CHOICE B))` SUBGOAL_TAC;
  IMATCH_MP_TAC  closed_union;
  ASM_REWRITE_TAC[];
  UND 1;
  REWRITE_TAC[SUBSET];
  USEF `(~)` (MATCH_MP CHOICE_DEF);
  UNDF  `(IN)`;
  REWRITE_TAC[];
  MESON_TAC[];
  ASM_MESON_TAC[unions_delete_choice];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let euclid2_d0 = prove_by_refinement(
  `!x. (euclid 2 x) ==> (d_euclid x x = &0)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  metric_space_zero;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[metric_euclid];
  ]);;
  (* }}} *)

let union_imp_subset = prove_by_refinement(
  `!(Z1:A->bool) Z2 A. (Z1 UNION Z2 = A) ==>
         (Z1 SUBSET A /\ Z2 SUBSET A)`,
  (* {{{ proof *)
  [
  SET_TAC[UNION;SUBSET];
  ]);;
  (* }}} *)

let loc_path_conn_top2 = prove_by_refinement(
  `loc_path_conn top2`,
  (* {{{ proof *)
  [
  REWRITE_TAC[top2];
  IMATCH_MP_TAC  loc_path_conn_euclid;
  TYPE_THEN `2` EXISTS_TAC;
  MESON_TAC[metric_euclid;top_of_metric_top;top_of_metric_unions;top_univ];
  ]);;
  (* }}} *)

let connected_empty = prove_by_refinement(
  `!U. connected (U:(A->bool)->bool) EMPTY `,
  (* {{{ proof *)
  [
  REWRITE_TAC[connected];
  ]);;
  (* }}} *)

let component_imp_connected = prove_by_refinement(
  `!U (x:A). (topology_ U) ==> (connected U (component U x))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `~(UNIONS U x)` ASM_CASES_TAC;
  UND 1;
  ASM_SIMP_TAC[GSYM component_empty];
  REWRITE_TAC[connected_empty];
  REWR 1;
  (* - *)
  REWRITE_TAC[connected];
  CONJ_TAC;
  REWRITE_TAC[SUBSET;connected;component];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `component U x x` SUBGOAL_TAC;
  ASM_MESON_TAC[component_refl];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `A x \/ B x` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SUBSET;UNION]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* - *)
  TYPE_THEN `!A B. component U x SUBSET A UNION B /\ (A INTER B = EMPTY) /\ U B /\ U A /\ A x ==> component U x SUBSET A` SUBGOAL_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `B' x'` SUBGOAL_TAC;
  USE 11 (REWRITE_RULE[SUBSET;UNION]);
  TSPEC `x'` 11;
  ASM_MESON_TAC[];
  DISCH_TAC;
  USE 12 (REWRITE_RULE[component]);
  REP_BASIC_TAC;
  TYPE_THEN `Z SUBSET (component U x)` SUBGOAL_TAC;
  IMATCH_MP_TAC  connected_component;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  USE 16 (REWRITE_RULE[connected]);
  REP_BASIC_TAC;
  TYPEL_THEN[`A'`;`B'`] (USE 16 o ISPECL);
  UND 16;
  ASM_REWRITE_TAC[];
  TYPE_THEN `Z SUBSET A' UNION B'` SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `component U x` EXISTS_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  REWRITE_TAC[DE_MORGAN_THM];
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[];
  USE 10 (REWRITE_RULE[INTER;EQ_EMPTY]);
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* - *)
  DISCH_THEN DISJ_CASES_TAC;
  TYPEL_THEN[`A`;`B`] (USE 7 o ISPECL);
  ASM_MESON_TAC[];
  TYPEL_THEN [`B`;`A`] (USE 7 o ISPECL);
  REWR 7;
  DISJ2_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ONCE_REWRITE_TAC[INTER_COMM];
  ASM_REWRITE_TAC[];
  ONCE_REWRITE_TAC[UNION_COMM];
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let open_induced = prove_by_refinement(
  `!U (A:A->bool). (topology_ U) /\ U A ==>
          (induced_top U A = { B | U B /\ B SUBSET A })`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[induced_top;IMAGE;];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[];
  GEN_TAC;
  EQ_TAC;
  REP_BASIC_TAC;
  FIRST_ASSUM MP_TAC ;
  DISCH_THEN_FULL_REWRITE;
  CONJ_TAC;
  IMATCH_MP_TAC  top_inter;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INTER;SUBSET];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  UND 2;
  SET_TAC [INTER;SUBSET];
  ]);;
  (* }}} *)

let connected_induced = prove_by_refinement(
  `!U (C:A->bool) . (topology_ U /\ U C ) ==>
           (connected U C = connected (induced_top U C) C)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[connected];
  ASM_SIMP_TAC[open_induced];
  EQ_TAC;
  REP_BASIC_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  sub_union;
  ASM_REWRITE_TAC[SUBSET_REFL ];
  REP_BASIC_TAC;
  TYPEL_THEN [`A`;`B`] (USE 2 o ISPECL);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* - *)
  REP_BASIC_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `UNIONS {B | U B /\ B SUBSET C}` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  UNIONS_UNIONS;
  ONCE_REWRITE_TAC[SUBSET];
  REWRITE_TAC[];
  MESON_TAC[];
  (* - *)
  REP_BASIC_TAC;
  TYPEL_THEN[`A INTER C`;`B INTER C`] (USE 2 o ISPECL);
  REWR 2;
  UND 2;
  DISCH_THEN  (TH_INTRO_TAC []);
  TYPE_THEN `!A'. (U A' ==> U (A' INTER C))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC top_inter;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWRITE_TAC[GSYM CONJ_ASSOC];
  CONJ_TAC;
  ASM_MESON_TAC[];
  REWRITE_TAC[INTER_SUBSET];
  CONJ_TAC;
  ASM_MESON_TAC[];
  CONJ_TAC;
  UND 5;
  SET_TAC[INTER];
  UND 4;
  SET_TAC[SUBSET;UNION;INTER];
  SET_TAC[INTER;SUBSET];
  ]);;
  (* }}} *)

let connected_induced2 = prove_by_refinement(
  `!U (C:A->bool) Z. (topology_ U /\ U C /\ Z SUBSET (UNIONS U))  ==>
        (connected (induced_top U C) Z <=> (Z SUBSET C) /\ (connected U Z))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[connected];
  ASM_SIMP_TAC[open_induced];
  EQ_TAC;
  REP_BASIC_TAC;
  SUBCONJ_TAC;
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  USE 4(REWRITE_RULE[SUBSET;UNIONS]);
  TSPEC `x` 4;
  REWR 4;
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REP_BASIC_TAC;
  TYPEL_THEN [`A INTER C`;`B INTER C`] (USE 3 o ISPECL);
  REWR 3;
  UND 3;
  DISCH_THEN  (TH_INTRO_TAC []);
  TYPE_THEN `!A'. (U A' ==> U (A' INTER C))` SUBGOAL_TAC;
  REP_BASIC_TAC;
  IMATCH_MP_TAC top_inter;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWRITE_TAC[GSYM CONJ_ASSOC];
  CONJ_TAC;
  ASM_MESON_TAC[];
  REWRITE_TAC[INTER_SUBSET];
  CONJ_TAC;
  ASM_MESON_TAC[];
  CONJ_TAC;
  UND 7;
  SET_TAC[INTER];
  UND 6;
  UND 5;
  SET_TAC[INTER;SUBSET;UNION];
  UND 5;
  SET_TAC[INTER;SUBSET;UNION];
  REP_BASIC_TAC;
  (* - *)
  CONJ_TAC;
  UND 0;
  REWRITE_TAC[SUBSET;UNIONS];
  REP_BASIC_TAC;
  TSPEC `x` 5;
  REWR 5;
  REP_BASIC_TAC;
  TYPE_THEN `u INTER C` EXISTS_TAC;
  REWRITE_TAC[GSYM CONJ_ASSOC];
  CONJ_TAC;
  IMATCH_MP_TAC  top_inter;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[INTER];
  ASM_MESON_TAC[ISUBSET ];
  (* - *)
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  ]);;
  (* }}} *)

let connected_metric = prove_by_refinement(
  `!X d (C:A->bool). metric_space (X,d) /\ C SUBSET X /\
    (top_of_metric(X,d)C) ==>
     (connected(top_of_metric(X,d))C <=> connected(top_of_metric(C,d))C)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `top_of_metric(C,d) = induced_top(top_of_metric(X,d))C` SUBGOAL_TAC;
  ASM_MESON_TAC[top_of_metric_induced];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  connected_induced;
  ASM_MESON_TAC[top_of_metric_top];
  ]);;
  (* }}} *)

let connected_metric_pair = prove_by_refinement(
  `!(X:A->bool) Y Z d. metric_space (X,d) /\
     top_of_metric(X,d) Y /\ top_of_metric(X,d) Z /\
       Z SUBSET Y  ==>
   (connected (top_of_metric(X,d)) Z = connected (top_of_metric(Y,d)) Z)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  (* - *)
  TYPE_THEN `Y SUBSET X` SUBGOAL_TAC;
  USE 2(MATCH_MP sub_union);
  UND 2;
  ASM_SIMP_TAC[GSYM top_of_metric_unions];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `Z SUBSET X` SUBGOAL_TAC ;
  ASM_MESON_TAC[SUBSET_TRANS];
  DISCH_TAC;
  ASM_SIMP_TAC[connected_metric];
  (* - *)
  TYPE_THEN `metric_space (Y,d)` SUBGOAL_TAC;
  ASM_MESON_TAC[metric_subspace];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `top_of_metric(Y,d)  = induced_top(top_of_metric(X,d)) Y` SUBGOAL_TAC;
  ASM_MESON_TAC[top_of_metric_induced];
  DISCH_TAC;
  TYPE_THEN `top_of_metric(Y,d) Z` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  ASM_SIMP_TAC[open_induced;top_of_metric_top];
  DISCH_TAC;
  ASM_SIMP_TAC[connected_metric];
  ]);;
  (* }}} *)

let construct_hv_finite = prove_by_refinement(
  `!A C v v'. (top2 A) /\ (C SUBSET A) /\ (simple_arc_end C v v') ==>
    (?C'. C' SUBSET A /\ simple_arc_end C' v v' /\ hv_finite C')`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `A' = path_component(top_of_metric(A,d_euclid)) v` ABBREV_TAC ;
  TYPE_THEN `A' = component (top_of_metric(A,d_euclid)) v` SUBGOAL_TAC;
  EXPAND_TAC "A'";
  AP_THM_TAC;
  IMATCH_MP_TAC  loc_path_euclid_cor ;
  TYPE_THEN `2` EXISTS_TAC;
  ASM_REWRITE_TAC[GSYM top2];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `A SUBSET (euclid 2)` SUBGOAL_TAC;
  USEF `top2`  (MATCH_MP sub_union );
  RULE_ASSUM_TAC (REWRITE_RULE[top2_unions]);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN`UNIONS (top_of_metric(A,d_euclid)) = A` SUBGOAL_TAC;
  ASM_MESON_TAC [GSYM top_of_metric_unions;metric_euclid;metric_subspace];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `A' SUBSET (UNIONS (top_of_metric(A,d_euclid)))` SUBGOAL_TAC;
  ASM_MESON_TAC[component_unions];
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `A' SUBSET (euclid 2)`  SUBGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `A` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  ASSUME_TAC  loc_path_conn_top2 ;
  (* - *)
  TYPE_THEN `A v` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SUBSET]);
  UND 1;
  DISCH_THEN IMATCH_MP_TAC ;
  UND 0;
  MESON_TAC[simple_arc_end_end];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `top_of_metric(A,d_euclid) = induced_top top2 A` SUBGOAL_TAC;
  REWRITE_TAC[top2];
  UND 5;
  SIMP_TAC [metric_euclid;top_of_metric_induced ];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `top2 A'` SUBGOAL_TAC;
  EXPAND_TAC "A'";
  UND 11;
  DISCH_THEN_REWRITE;
  USE 9 (REWRITE_RULE[ loc_path_conn]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `~(v  = v')` SUBGOAL_TAC;
  UND 0;
  ASM_MESON_TAC[simple_arc_end_distinct];
  DISCH_TAC;
  (* A' - *)
  TYPE_THEN `connected (top_of_metric(A,d_euclid)) A'` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  component_imp_connected;
  ASM_MESON_TAC[top_of_metric_top;metric_subspace;metric_euclid];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `connected (top_of_metric(euclid 2,d_euclid)) A'` SUBGOAL_TAC;
  TH_INTRO_TAC [`euclid 2`;`A`;`A'`;`d_euclid`] connected_metric_pair;
  ASM_MESON_TAC [metric_euclid;GSYM top2];
  DISCH_THEN_REWRITE;
  ASM_MESON_TAC[];
  REWRITE_TAC[GSYM top2];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `connected top2 C` SUBGOAL_TAC;
  IMATCH_MP_TAC  simple_arc_connected;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `C SUBSET A'` SUBGOAL_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  connected_component;
  IMATCH_MP_TAC  (TAUT `a /\ b ==> b /\a`);
  CONJ_TAC;
  UND 0;
  MESON_TAC[simple_arc_end_end];
  TH_INTRO_TAC[`top2`;`A`;`C`] connected_induced2;
  REWRITE_TAC[top2_top;top2_unions];
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[SUBSET_TRANS];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `C v /\ C v'` SUBGOAL_TAC;
  UND 0;
  MESON_TAC[simple_arc_end_end;simple_arc_end_end2];
  DISCH_TAC;
  TYPE_THEN `A' v /\ A' v'` SUBGOAL_TAC;
  ASM_MESON_TAC[ISUBSET];
  DISCH_TAC;
  (* - *)
  TH_INTRO_TAC[`A'`;`v`;`v'`] p_conn_conn;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TH_INTRO_TAC[`A'`;`v`;`v'`] p_conn_hv_finite;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  REP_BASIC_TAC;
  TYPE_THEN `C'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `A'` EXISTS_TAC;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let graph_rad_pt_center_piece = prove_by_refinement(
  `!G r e v v'.
     graph_hv_finite_radius G r /\ graph_inc G e v /\
     FINITE(graph_edge G) /\ FINITE(graph_vertex G) /\
    graph_edge G e /\ graph_inc G e v' /\ ~(v = v') ==>
   (? Cv u Cv' u' C''.
        (hv_finite Cv /\ hv_finite Cv' /\  (hv_finite C'') /\
        ~(graph_vertex G u) /\
        ~(graph_vertex G u') /\
        simple_arc_end Cv v u /\
        simple_arc_end Cv' v' u' /\
        simple_arc_end C'' u u' /\
         ~C'' v /\ ~C'' v' /\
        (euclid 2 u)  /\ (euclid 2 u') /\
        (d_euclid v u = r) /\ (d_euclid v' u' = r) /\
        (Cv SUBSET e) /\ (Cv' SUBSET e) /\
        (Cv SUBSET  (closed_ball(euclid 2,d_euclid) v r)) /\
        (Cv' SUBSET (closed_ball(euclid 2,d_euclid) v' r)) /\
   (!e'. (graph_edge G e') /\ ~(e = e') ==> (C'' INTER e' = EMPTY)) /\
   (!v''. graph_vertex G v'' /\ ~(graph_inc G e v'') ==>
        (C'' INTER (closed_ball(euclid 2,d_euclid) v'' r) = EMPTY))
     ))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TH_INTRO_TAC [`G`;`r`;`e`;`v`] graph_rad_pt_select;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `Cv = C` ABBREV_TAC ;
  KILL 13;
  TYPE_THEN `Cv` EXISTS_TAC;
  TYPE_THEN `u` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TH_INTRO_TAC [`G`;`r`;`e`;`v'`] graph_rad_pt_select;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `Cv' = C'` ABBREV_TAC ;
  KILL 19;
  TYPE_THEN `Cv'` EXISTS_TAC;
  TYPE_THEN `u'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* A' *)
  TYPE_THEN `!v''. graph_vertex G v'' ==> (euclid 2 v'')` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph;SUBSET ]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `!v''. graph_inc G e v'' ==> graph_vertex G v''`  SUBGOAL_TAC;
  REP_BASIC_TAC;
  TH_INTRO_TAC [`G`;`e`] graph_inc_subset;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph;]);
  ASM_REWRITE_TAC[SUBSET ];
  FIRST_ASSUM MP_TAC;
  MESON_TAC[ISUBSET];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `D = (\ v. (closed_ball(euclid 2,d_euclid) v r))` ABBREV_TAC ;
  TYPE_THEN `B  = (UNIONS { e' | graph_edge G e' /\ ~(e' = e)})` ABBREV_TAC ;
  TYPE_THEN `B' = (UNIONS { DD | ?v''. (graph_vertex G v'' /\ (DD = D v'') /\ ~(graph_inc G e v''))})` ABBREV_TAC ;
  TYPE_THEN `B'' = {v, v'}` ABBREV_TAC ;
  TYPE_THEN `A = (euclid 2 DIFF (B UNION B' UNION B''))` ABBREV_TAC ;
  TYPE_THEN `top2 A` SUBGOAL_TAC;
  TH_INTRO_TAC [`top2`;`B UNION B' UNION B''`] closed_open;
  IMATCH_MP_TAC  closed_union;
  REWRITE_TAC[top2_top];
  EXPAND_TAC "B";
  EXPAND_TAC "B'";
  EXPAND_TAC "B''";
  CONJ_TAC;
  IMATCH_MP_TAC  top_closed_unions;
  REWRITE_TAC[top2_top;SUBSET;];
  CONJ_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `graph_edge G` EXISTS_TAC ;
  ASM_REWRITE_TAC[SUBSET];
  MESON_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph;]);
  REP_BASIC_TAC;
  IMATCH_MP_TAC  simple_arc_end_closed;
  TH_INTRO_TAC [`G`;`x`] graph_edge_end_select;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  ASM_MESON_TAC[];
  (* --- *)
  IMATCH_MP_TAC  closed_union;
  REWRITE_TAC[top2_top];
  CONJ_TAC;
  IMATCH_MP_TAC  top_closed_unions;
  REWRITE_TAC[top2_top];
  CONJ_TAC;
  TYPE_THEN `{DD | ?v''. graph_vertex G v'' /\ (DD = D v'') /\ ~graph_inc G e v''} = IMAGE D { v'' | graph_vertex G v'' /\ ~graph_inc G e v''}` SUBGOAL_TAC;
  REWRITE_TAC[IMAGE];
  IMATCH_MP_TAC  EQ_EXT;
  GEN_TAC;
  REWRITE_TAC[];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  FINITE_IMAGE;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `graph_vertex G` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET ];
  MESON_TAC[];
  REWRITE_TAC[SUBSET];
  REP_BASIC_TAC;
  UNDF `x = D v''`;
  DISCH_THEN_FULL_REWRITE;
  EXPAND_TAC "D";
  REWRITE_TAC[top2];
  IMATCH_MP_TAC  closed_ball_closed;
  REWRITE_TAC[metric_euclid];
  (* --- *)
  TYPE_THEN `{v,v'} = {v} UNION {v'}` SUBGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[in_pair;UNION;INR IN_SING];
  MESON_TAC[];
  DISCH_THEN_REWRITE;
  IMATCH_MP_TAC  closed_union;
  REWRITE_TAC[top2_top];
  TYPE_THEN `graph_inc G e v` (FIND_ASSUM MP_TAC);
  TYPE_THEN `graph_inc G e v'` (FIND_ASSUM MP_TAC);
  ASM_MESON_TAC[closed_point];
  REWRITE_TAC[open_DEF;top2_unions];
  EXPAND_TAC "A";
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* B' *)
  TYPE_THEN `!u'' v''. graph_vertex G v'' /\ (d_euclid v'' u'' = r) ==> ~(graph_vertex G u'')` SUBGOAL_TAC;
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  TYPEL_THEN [`u''`;`v''`] (USE 31 o ISPECL);
  TYPE_THEN `~(u'' = v'')` SUBGOAL_TAC;
  DISCH_TAC;
  POP_ASSUM MP_TAC;
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `d_euclid v'' v'' = &0` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_space_zero;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[metric_euclid];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  UNDF `&0 = r`;
  UNDF   `&0 < r`;
  REAL_ARITH_TAC;
  DISCH_TAC;
  UNDF `(graph_vertex)`;
  ASM_REWRITE_TAC[EMPTY_EXISTS ;INTER ;closed_ball ;];
  TYPE_THEN `u''` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `d_euclid u'' u'' = &0` SUBGOAL_TAC;
  IMATCH_MP_TAC  metric_space_zero;
  TYPE_THEN `euclid 2` EXISTS_TAC;
  ASM_REWRITE_TAC[metric_euclid];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `euclid 2 u'' ` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `euclid 2 v'' ` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  UNDF `&0 < r`;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* B1'- *)
  TYPE_THEN `~graph_vertex G u` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `v` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  (* - *)
  TYPE_THEN `~graph_vertex G u'` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `v'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  ASM_REWRITE_TAC[];
  (* C' *)
  TYPE_THEN `!(X:A->bool) Y Z. (X UNION Y = Z) ==> (X SUBSET Z)` SUBGOAL_TAC;
  SET_TAC[UNION;SUBSET];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `simple_arc_end e v v'` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE [graph_hv_finite_radius;good_plane_graph]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `graph_vertex G v` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `graph_vertex G v'` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `~D v u'` SUBGOAL_TAC;
  EXPAND_TAC "D";
  PROOF_BY_CONTR_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;]);
  REP_BASIC_TAC;
  GRABF `~(v = v')` (TH_INTRO_TAC [`v`;`v'`]);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `u'` EXISTS_TAC;
  ASM_REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[closed_ball];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* C1'- *)
  TYPE_THEN `~(v = u) /\ ~(v = u')` SUBGOAL_TAC;
  CONJ_TAC;
  DISCH_TAC;
  POP_ASSUM MP_TAC;
  DISCH_THEN_FULL_REWRITE;
  TH_INTRO_TAC[`u`] euclid2_d0;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  UNDF `&0 < r`;
  UNDF `&0 = r`;
  REAL_ARITH_TAC;
  DISCH_TAC;
  POP_ASSUM MP_TAC;
  DISCH_THEN_FULL_REWRITE;
  POP_ASSUM MP_TAC;
  EXPAND_TAC "D";
  REWRITE_TAC[closed_ball];
  ASM_REWRITE_TAC[];
  TH_INTRO_TAC [`u'`] euclid2_d0;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  UNDF `&0 < r`;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* - *)
  TYPE_THEN `~(v' = u') ` SUBGOAL_TAC;
  DISCH_TAC;
  POP_ASSUM MP_TAC;
  DISCH_THEN_FULL_REWRITE;
  TH_INTRO_TAC[`u'`] euclid2_d0;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  UNDF `&0 < r`;
  UNDF `&0 = r`;
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* - *)
  TH_INTRO_TAC [`e`;`v`;`v'`;`u'`] simple_arc_end_cut;
  ASM_REWRITE_TAC[];
  TYPE_THEN `Cv' u'` SUBGOAL_TAC;
  TYPE_THEN `simple_arc_end Cv' v' u'` (FIND_ASSUM  MP_TAC );
  MESON_TAC[simple_arc_end_end2];
  RULE_ASSUM_TAC (REWRITE_RULE[SUBSET]);
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `Cvu' = C''` ABBREV_TAC ;
  POP_ASSUM (fun t-> ALL_TAC);
  TYPE_THEN `Cu'v' = C'''` ABBREV_TAC ;
  POP_ASSUM (fun t -> ALL_TAC);
  TYPE_THEN `Cu'v' v'` SUBGOAL_TAC;
  TYPE_THEN `simple_arc_end Cu'v' u' v'` (FIND_ASSUM  MP_TAC );
  MESON_TAC[simple_arc_end_end2];
  DISCH_TAC;
  TYPE_THEN `~Cvu' v'` SUBGOAL_TAC;
  DISCH_TAC;
  USEF `(INTER)` (REWRITE_RULE[FUN_EQ_THM]);
  TSPEC `v'` 37;
  RULE_ASSUM_TAC (REWRITE_RULE[INTER;eq_sing ;INR IN_SING]);
  UND 37;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `~D v' u` SUBGOAL_TAC;
  EXPAND_TAC "D";
  DISCH_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;]);
  REP_BASIC_TAC;
  GRABF `~(v' = v)` (TH_INTRO_TAC [`v'`;`v`]);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `u` EXISTS_TAC;
  ASM_REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[closed_ball];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  REAL_ARITH_TAC;
  DISCH_TAC;
  (* D'- *)
  TYPE_THEN `Cvu' u \/ Cu'v' u` SUBGOAL_TAC;
  USE 35 (REWRITE_RULE[FUN_EQ_THM;]);
  TSPEC  `u` 35 ;
  USE 35 (REWRITE_RULE[UNION]);
  ASM_REWRITE_TAC[];
  USE 8(REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 11;
  MESON_TAC[simple_arc_end_end2];
  DISCH_TAC;
  (* - *)
  USE 35 (MATCH_MP   union_imp_subset);
  TYPE_THEN `Cu'v' = Cv'` SUBGOAL_TAC;
  TH_INTRO_TAC [`Cu'v'`;`Cv'`;`e`;`v'`;`u'`] simple_arc_end_inj;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  DISCH_THEN_FULL_REWRITE;
  (* - *)
  TYPE_THEN `~Cv' u` SUBGOAL_TAC;
  DISCH_TAC;
  UNDF `~D v' u` ;
  REWRITE_TAC[];
  EXPAND_TAC "D";
  RULE_ASSUM_TAC (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  REWR 45;
  (* - *)
  TYPE_THEN `~(u = u')` SUBGOAL_TAC;
  DISCH_TAC;
  UND 47;
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  GRABF `~(v=v')` (TH_INTRO_TAC[`v`;`v'`]);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `u'` EXISTS_TAC;
  REWRITE_TAC[INTER;closed_ball];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[REAL_ARITH `r <= r`];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TH_INTRO_TAC[`Cvu'`;`v`;`u'`;`u`] simple_arc_end_cut;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `CC = C'''''` ABBREV_TAC ;
  POP_ASSUM (fun t->ALL_TAC);
  (* E' *)
  TYPE_THEN `~CC v` SUBGOAL_TAC;
  DISCH_TAC;
  TYPE_THEN `C'''' v` SUBGOAL_TAC;
  UND 50;
  MESON_TAC[simple_arc_end_end];
  DISCH_TAC;
  TYPE_THEN `v = u` SUBGOAL_TAC;
  UND 48;
   REWRITE_TAC[INTER;eq_sing;INR IN_SING];
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_THEN_FULL_REWRITE;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `~CC v'` SUBGOAL_TAC;
  DISCH_TAC;
  USE 35 (MATCH_MP union_imp_subset);
  UND 43;
  REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[SUBSET]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `CC SUBSET A` SUBGOAL_TAC;
  EXPAND_TAC "A";
  REWRITE_TAC[DIFF_SUBSET];
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_euclid;
  IMATCH_MP_TAC  simple_arc_end_simple;
  UND 49;
  MESON_TAC[];
  PROOF_BY_CONTR_TAC;
  USE 55 (MATCH_MP inter_union);
  FIRST_ASSUM MP_TAC;
  REWRITE_TAC[];
  REWRITE_TAC[DE_MORGAN_THM];
  TYPE_THEN `CC SUBSET e` SUBGOAL_TAC;
  USE 35 (MATCH_MP union_imp_subset);
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `Cvu'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_TAC;
  (* -- *)
  CONJ_TAC;
  EXPAND_TAC"B";
  REWRITE_TAC[INTER;UNIONS;EQ_EMPTY ];
  REP_BASIC_TAC;
  TYPE_THEN `e x` SUBGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph]);
  REP_BASIC_TAC  ; (* we are up to 69 in the hypothesis stack *)
  TYPEL_THEN  [`e`;`u''`] (USE 66 o ISPECL);
  REWR 66;
  TYPE_THEN `graph_vertex G x` SUBGOAL_TAC;
  USE 66 (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[INTER];
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* --- *)
  TYPE_THEN `graph_inc G e HAS_SIZE 2` SUBGOAL_TAC;
  IMATCH_MP_TAC  graph_edge2;
  ASM_REWRITE_TAC[];
  TYPE_THEN `graph_inc G e x` SUBGOAL_TAC;
  ASM_SIMP_TAC[];
  ASM_REWRITE_TAC[INTER];
  REP_BASIC_TAC;
  TH_INTRO_TAC [`graph_inc G e`;`v`;`x`;`v'`] two_exclusion;
  ASM_REWRITE_TAC[];
   UND 60;
  UND 54;
  MESON_TAC[];
  UND 60;
  UND 53;
  MESON_TAC[];
  (* -- *)
  PROOF_BY_CONTR_TAC;
  USE 57 (MATCH_MP inter_union);
  UND 57;
  REWRITE_TAC[DE_MORGAN_THM];
  CONJ_TAC;
  EXPAND_TAC "B'";
  REWRITE_TAC[INTER;UNIONS;];
  REWRITE_TAC [EQ_EMPTY];
  REP_BASIC_TAC;
  UNDF `u''' = D v''` ;
  DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  TYPEL_THEN [`e`;`v''`] (USE 59 o ISPECL);
  REWR 59;
  UND 59;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `x` EXISTS_TAC;
  REWRITE_TAC[INTER];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UND 57;
  EXPAND_TAC "D";
  DISCH_THEN_REWRITE;
  (* -- *)
  EXPAND_TAC "B''";
  REWRITE_TAC[INTER;EQ_EMPTY;in_pair];
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* F' *)
  TH_INTRO_TAC[`A`;`CC`;`u`;`u'`] construct_hv_finite;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `Chv = C''''''` ABBREV_TAC ;
  KILL 59;
  TYPE_THEN `Chv` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* - *)
  TYPE_THEN `~(A v) /\ ~(A v')` SUBGOAL_TAC;
  EXPAND_TAC "A";
  EXPAND_TAC "B''";
  REWRITE_TAC[DIFF;UNION;in_pair];
  DISCH_TAC;
  TYPE_THEN `~(Chv v) /\ ~(Chv v')` SUBGOAL_TAC;
  UND 59;
  UND 58;
  MESON_TAC[ISUBSET];
  DISCH_THEN_REWRITE;
  (* - *)
  TYPE_THEN `(!e'. ~(e = e') /\ (graph_edge G e') ==> (A INTER e' = {}))` SUBGOAL_TAC;
  EXPAND_TAC "A";
  EXPAND_TAC "B";
  REP_BASIC_TAC;
  REWRITE_TAC[EQ_EMPTY;INTER;DIFF;UNION;UNIONS ];
  REP_BASIC_TAC;
  LEFT 64 "u";
  LEFT 64 "u";
  TSPEC `e'` 64;
  UND 64;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  CONJ_TAC;
  REP_BASIC_TAC;
  TSPEC `e'` 60;
  REWR 60;
  UND 60;
  UND 58;
  REWRITE_TAC[EQ_EMPTY;INTER;SUBSET;];
  MESON_TAC[];
  (* - *)
  TYPE_THEN `!v''. graph_vertex G v'' /\ ~graph_inc G e v'' ==> (A INTER closed_ball (euclid 2,d_euclid) v'' r = {})` SUBGOAL_TAC;
  REP_BASIC_TAC;
  EXPAND_TAC "A";
  EXPAND_TAC "B'";
  REP_BASIC_TAC;
  REWRITE_TAC[EQ_EMPTY;INTER;DIFF;UNION;UNIONS;];
  EXPAND_TAC "D";
  REP_BASIC_TAC;
  UND 65;
  REWRITE_TAC[];
  DISJ2_TAC;
  DISJ1_TAC;
  CONV_TAC (dropq_conv "u");
  TYPE_THEN `v''` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TSPEC `v''` 62;
  REWR 62;
  UND 62;
  UND 58;
  REWRITE_TAC[EQ_EMPTY;INTER;SUBSET;];
  MESON_TAC[];
  (* Wed Aug 25 14:58:37 EDT 2004 *)


  ]);;
  (* }}} *)

let planar_graph_hv = prove_by_refinement(
  `!(G:(A,B)graph_t). (planar_graph G) /\
         FINITE (graph_edge G) /\
         FINITE (graph_vertex G) /\
         ~(graph_edge G = {}) /\
         (!v. CARD (graph_edge_around G v) <=| 4)
         ==> (?H. graph_isomorphic G H /\
              good_plane_graph H /\ (!e. graph_edge H e ==>
           hv_finite e))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TH_INTRO_TAC[`G`] graph_radius_exists;
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  (* - *)
  TYPE_THEN `X = { K | graph_isomorphic H K /\ graph_hv_finite_radius K r}` ABBREV_TAC  ;
  TYPE_THEN `c = (\ (K:(num->real,(num->real)->bool)graph_t). CARD {x | graph_edge K x /\ ~hv_finite x})` ABBREV_TAC ;
  TYPE_THEN `D = (\ v. (closed_ball(euclid 2,d_euclid) v r))` ABBREV_TAC ;
  TH_INTRO_TAC[`X`;`c`] select_image_num_min;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `H` EXISTS_TAC;
  EXPAND_TAC "X";
  REWRITE_TAC[];
  ASM_REWRITE_TAC[graph_isomorphic_refl];
  REP_BASIC_TAC;
  TYPE_THEN `K = z` ABBREV_TAC ;
  KILL 12;
  TYPE_THEN `K` EXISTS_TAC;
  CONJ_TAC;
  UND 11;
  EXPAND_TAC "X";
  REWRITE_TAC[];
  ASM_MESON_TAC[graph_isomorphic_trans];
  (* - *)
  TYPE_THEN `graph_hv_finite_radius K r` SUBGOAL_TAC;
  UND 11;
  EXPAND_TAC "X";
  REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* - *)
  CONJ_TAC;
  UND 12;
  REWRITE_TAC[graph_hv_finite_radius];
  DISCH_THEN_REWRITE;
  REP_BASIC_TAC;
  PROOF_BY_CONTR_TAC;
  (* - *)
  TH_INTRO_TAC[`K`;`e`] graph_edge_end_select;
  ASM_REWRITE_TAC[];
  UND 12;
  REWRITE_TAC[graph_hv_finite_radius;good_plane_graph;plane_graph];
  DISCH_THEN_REWRITE;
  REP_BASIC_TAC;
  (* A *)
  TYPE_THEN `graph_isomorphic G K` SUBGOAL_TAC;
  TH_INTRO_TAC[`G`;`H`;`K`] graph_isomorphic_trans;
  ASM_REWRITE_TAC[];
  UND 11;
  EXPAND_TAC "X";
  REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* - *)
  TYPE_THEN `FINITE (graph_edge K)` SUBGOAL_TAC;
  USE 18(REWRITE_RULE[graph_isomorphic;graph_iso]);
  REP_BASIC_TAC;
  UND 19;
  UND 3;
  MESON_TAC[FINITE_BIJ];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `~(? e' . (~graph_edge K e') /\ hv_finite e' /\ simple_arc_end e' v v' /\ (e INTER (graph_vertex K) = (e' INTER (graph_vertex K))) /\ (!v. graph_vertex K v /\ ~e' v  ==> (e' INTER closed_ball (euclid 2,d_euclid) v r = {})) /\ (!e''. graph_edge K e'' /\ ~(e'' = e)  ==> e' INTER e'' SUBSET e INTER e''))` SUBGOAL_TAC;
  DISCH_TAC;
  REP_BASIC_TAC;
  (* -- *)
  TH_INTRO_TAC[`K`;`e`;`e'`] graph_replace_card;
  ASM_REWRITE_TAC[];
  TYPE_THEN `K' = graph_replace K e e'` ABBREV_TAC ;
  DISCH_TAC;
  TYPE_THEN `graph_isomorphic H K'` SUBGOAL_TAC;
  EXPAND_TAC "X";
  EXPAND_TAC "K'";
  REWRITE_TAC[];
  TH_INTRO_TAC[`H`;`K`;`K'`] graph_isomorphic_trans;
  ASM_REWRITE_TAC[];
  UND 11;
  EXPAND_TAC "X";
  REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  EXPAND_TAC "K'";
  IMATCH_MP_TAC  graph_replace_iso;
  ASM_REWRITE_TAC[];
  EXPAND_TAC "K'";
  DISCH_THEN_REWRITE;
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `plane_graph K'` SUBGOAL_TAC;
  EXPAND_TAC "K'";
  IMATCH_MP_TAC  graph_replace_plane;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph]);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `good_plane_graph K'` SUBGOAL_TAC;
  EXPAND_TAC "K'";
  IMATCH_MP_TAC  good_replace;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `e v'' /\ e v'''` SUBGOAL_TAC;
  USE 22 (REWRITE_RULE[FUN_EQ_THM]);
  TYPE_THEN  `v''` (WITH 22 o ISPEC);
  TYPE_THEN `v'''` (USE 22 o ISPEC);
  RULE_ASSUM_TAC (REWRITE_RULE[INTER]);
  UND 22;
  UND 35;
  UND 33;
  UND 34;
  DISCH_THEN_REWRITE;
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  MESON_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc K e = {v,v'}` SUBGOAL_TAC;
  IMATCH_MP_TAC  graph_vertex_exhaust;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph]);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `graph_inc K e = {v'',v'''}` SUBGOAL_TAC;
  IMATCH_MP_TAC  graph_vertex_exhaust;
  USE 37 (SYM);
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph]);
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  TSPEC `e` 46;
  REWR 46;
  ASM_REWRITE_TAC[INTER];
  DISCH_THEN_FULL_REWRITE;
  TYPE_THEN `((v'' = v) /\ (v''' = v')) \/ ((v'' = v') /\ (v''' = v))` SUBGOAL_TAC;
  USE 37 (REWRITE_RULE[FUN_EQ_THM]);
  TYPE_THEN `v''` (WITH 37 o ISPEC);
  TYPE_THEN `v'''` (USE 37 o ISPEC);
  UND 37;
  UND 38;
  REWRITE_TAC[in_pair];
  UND 32;
  UND 15;
  MESON_TAC[];
  DISCH_THEN DISJ_CASES_TAC;
  REP_BASIC_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* -- *)
  TYPE_THEN `graph_hv_finite_radius K' r` SUBGOAL_TAC;
  EXPAND_TAC "K'";
  IMATCH_MP_TAC  graph_replace_hv_finite_radius;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TYPE_THEN `X K'` SUBGOAL_TAC;
  EXPAND_TAC "X";
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  TSPEC `K'` 10;
  REWR 10;
  UND 10;
  EXPAND_TAC "c";
  UND 27;
(**** Changed by JRH; the new ARITH_TAC doesn't accept alpha-equivs (maybe)
  ARITH_TAC;
 ****)
  REWRITE_TAC[NOT_IMP; NOT_LE];
  REWRITE_TAC[];
  (* B *)
  TH_INTRO_TAC [`K`;`r`;`e`;`v`;`v'`] graph_rad_pt_center_piece;
  ASM_REWRITE_TAC[];
  USE 18 (REWRITE_RULE[graph_isomorphic;graph_iso]);
  REP_BASIC_TAC;
  UND 21;
  UND 2;
  MESON_TAC[FINITE_BIJ];
  REP_BASIC_TAC;
  KILL 4;
  KILL 3;
  KILL 2;
  KILL 1;
  KILL 0;
  KILL 6;
  KILL 5;
  KILL 7;
  KILL 8;
  KILL 11;
  KILL 10;
  KILL 18;
  KILL 19;
  TYPE_THEN `graph_inc K e  = {v,v'}` SUBGOAL_TAC;
  IMATCH_MP_TAC  graph_vertex_exhaust;
  ASM_REWRITE_TAC[];
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph]);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `e INTER graph_vertex K = {v,v'}` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph]);
  REP_BASIC_TAC;
  TSPEC `e` 7;
  REWR 7;
  ASM_REWRITE_TAC[];
  DISCH_THEN_REWRITE;
  (* C- *)
  TYPE_THEN `!e v. graph_edge K e /\ graph_inc K e v ==> graph_vertex K v` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TH_INTRO_TAC[`K`;`e'`] graph_inc_subset;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph]);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `p_conn (Cv UNION Cv' UNION C'') v v'` SUBGOAL_TAC;
  IMATCH_MP_TAC  pconn_trans;
  TYPE_THEN `u` EXISTS_TAC;
  CONJ_TAC;
  TH_INTRO_TAC[`Cv UNION Cv' UNION C''`;`v`;`u`] p_conn_hv_finite;
  IMATCH_MP_TAC  simple_arc_end_distinct;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `Cv` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  IMATCH_MP_TAC  pconn_trans;
  TYPE_THEN `u'` EXISTS_TAC;
  CONJ_TAC;
  TH_INTRO_TAC[`Cv UNION Cv' UNION C''`;`u`;`u'`] p_conn_hv_finite;
  IMATCH_MP_TAC  simple_arc_end_distinct;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `C''` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  TH_INTRO_TAC[`Cv UNION Cv' UNION C''`;`u'`;`v'`] p_conn_hv_finite;
  IMATCH_MP_TAC  simple_arc_end_distinct;
  TYPE_THEN `Cv'` EXISTS_TAC;
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_MESON_TAC[];
  DISCH_THEN_REWRITE;
  TYPE_THEN `Cv'` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET;UNION];
  CONJ_TAC;
  MESON_TAC[];
  IMATCH_MP_TAC  simple_arc_end_symm;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TH_INTRO_TAC[`Cv UNION Cv' UNION C''`;`v`;`v'`] p_conn_hv_finite;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  REP_BASIC_TAC;
  TYPE_THEN `C` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* D final constraints *)
  TYPE_THEN`graph K` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph]);
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  TYPE_THEN `!e v. graph_edge K e /\ graph_inc K e v ==> graph_vertex K v` SUBGOAL_TAC;
  REP_BASIC_TAC;
  TH_INTRO_TAC[`K`;`e'`]graph_inc_subset;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  (* - *)
  CONJ_TAC;
  DISCH_TAC;
  TYPE_THEN `C = e` ASM_CASES_TAC;
  ASM_MESON_TAC[];
  TSPEC `C` 21;
  REWR 11;
  TYPE_THEN `C SUBSET Cv UNION Cv'` SUBGOAL_TAC;
  UND 11;
  UND 4;
  REWRITE_TAC[SUBSET;UNION;EQ_EMPTY;INTER ];
  MESON_TAC[];
  DISCH_TAC;
  TYPE_THEN `D v INTER D v' = EMPTY ` SUBGOAL_TAC;
  EXPAND_TAC "D";
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  UND 21;
  DISCH_THEN IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  DISCH_TAC;
  (* -- *)
  UND 10;
  REWRITE_TAC[];
  IMATCH_MP_TAC  simple_arc_end_inj;
  TYPE_THEN `e` EXISTS_TAC;
  TYPE_THEN `v` EXISTS_TAC;
  TYPE_THEN `v'` EXISTS_TAC;
  ASM_REWRITE_TAC[SUBSET_REFL];
  SUBCONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph]);
  REP_BASIC_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  DISCH_TAC;
  CONJ_TAC;
  IMATCH_MP_TAC  simple_arc_end_simple;
  ASM_MESON_TAC[];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `Cv UNION Cv'` EXISTS_TAC;
  ASM_REWRITE_TAC[union_subset ];
  (* E *)
  CONJ_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[in_pair;INTER ];
  GEN_TAC;
  EQ_TAC;
  DISCH_THEN DISJ_CASES_TAC;
  UND 8;
  DISCH_THEN_FULL_REWRITE;
  CONJ_TAC;
  UND 3;
  MESON_TAC[simple_arc_end_end2];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  UND 8;
  DISCH_THEN_FULL_REWRITE;
  CONJ_TAC;
  UND 3;
  MESON_TAC[simple_arc_end_end];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  (* -- *)
  TYPE_THEN `graph_inc K e x` ASM_CASES_TAC;
  REWR 8;
  RULE_ASSUM_TAC (REWRITE_RULE[in_pair]);
  ASM_REWRITE_TAC[];
  USE 4 (REWRITE_RULE[SUBSET ]);
  REP_BASIC_TAC;
  TSPEC `x` 4;
  REWR 4;
  USE 4(REWRITE_RULE[UNION]);
  UND 4;
  REP_CASES_TAC;
  DISJ2_TAC;
  PROOF_BY_CONTR_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  UND 40;
  DISCH_THEN (TH_INTRO_TAC[`v`;`x`]);
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `x` EXISTS_TAC;
  REWRITE_TAC[INTER];
  CONJ_TAC;
  UND 4;
  UND 23;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  REWRITE_TAC[closed_ball2_center];
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph;plane_graph]);
  REP_BASIC_TAC;
  CONJ_TAC;
  USEF `X SUBSET euclid 2` (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UNDF `&0 < r`;
  REAL_ARITH_TAC;
  (* --- *)
  DISJ1_TAC;
  PROOF_BY_CONTR_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  UNDF `~(v = v')`;
  DISCH_THEN (TH_INTRO_TAC[`v'`;`x`]);
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `x` EXISTS_TAC;
  REWRITE_TAC[INTER];
  CONJ_TAC;
  UND 4;
  UND 22;
  REWRITE_TAC[SUBSET];
  MESON_TAC[];
  REWRITE_TAC[closed_ball2_center];
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph;plane_graph]);
  REP_BASIC_TAC;
  CONJ_TAC;
  USEF `X SUBSET euclid 2` (REWRITE_RULE[SUBSET]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UNDF `&0 < r`;
  REAL_ARITH_TAC;
 (* -- *)
  TYPE_THEN `graph_inc K e x` ASM_CASES_TAC;
  REWR 18;
  TSPEC `x` 20;
  REWR 19;
  PROOF_BY_CONTR_TAC;
  UND 19;
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[INTER;closed_ball2_center];
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius;good_plane_graph;plane_graph]);
  REP_BASIC_TAC;
  USEF `X SUBSET euclid 2` (REWRITE_RULE[SUBSET]);
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  UNDF `&0 < r`;
  REAL_ARITH_TAC;
  (* F *)
  KILL 14;
  KILL 39;
  KILL 38;
  KILL 37;
  KILL 36;
  KILL 35;
  KILL 34;
  KILL 33;
  KILL 32;
  KILL 29;
  KILL 28;
  KILL 27;
  KILL 26;
  KILL 5;
  KILL 2;
  IMATCH_MP_TAC  (TAUT `a /\ b ==> b /\ a`);
  CONJ_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[SUBSET;INTER];
  REP_BASIC_TAC;
  USEF `(SUBSET)` (REWRITE_RULE[SUBSET]);
  TSPEC `x` 4;
  REWR 4;
  UND 4;
  REWRITE_TAC[UNION];
  REP_CASES_TAC;
  ASM_MESON_TAC[ISUBSET];
  ASM_MESON_TAC[ISUBSET];
  PROOF_BY_CONTR_TAC;
  UND 21;
  DISCH_THEN (TH_INTRO_TAC[`e''`]);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[INTER];
  (* G *)
  REP_BASIC_TAC;
  TYPE_THEN `graph_inc K e v''` ASM_CASES_TAC;
  REWR 8;
  UND 8;
  REWRITE_TAC[in_pair];
  REP_CASES_TAC;
  UND 8;
  DISCH_THEN_FULL_REWRITE;
  PROOF_BY_CONTR_TAC;
  UND 2;
  UND 3;
  MESON_TAC[simple_arc_end_end2];
  UND 8;
  DISCH_THEN_FULL_REWRITE;
  PROOF_BY_CONTR_TAC;
  UND 2;
  UND 3;
  MESON_TAC[simple_arc_end_end];
  (* - *)
  TYPE_THEN `C SUBSET D v UNION D v' UNION C''` SUBGOAL_TAC;
  EXPAND_TAC "D";
  UND 4;
  UND 22;
  UND 23;
  REWRITE_TAC[SUBSET;UNION];
  MESON_TAC[];
  REWRITE_TAC[SUBSET];
  DISCH_TAC;
  PROOF_BY_CONTR_TAC;
  USE 11 (REWRITE_RULE[EMPTY_EXISTS;INTER]);
  REP_BASIC_TAC;
  TSPEC `u` 10;
  REWR 10;
  USE 10 (REWRITE_RULE[UNION]);
  UND 10;
  REP_CASES_TAC ;
  (* -- *)
  UND 8;
  ASM_REWRITE_TAC[in_pair];
  PROOF_BY_CONTR_TAC;
  USE 8 (REWRITE_RULE[DE_MORGAN_THM]);
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  UND 26;
  DISCH_THEN (TH_INTRO_TAC[`v`;`v''`]);
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  REWRITE_TAC[INTER;EMPTY_EXISTS];
  TYPE_THEN `u` EXISTS_TAC;
  UND 10;
  EXPAND_TAC "D";
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  (* -- *)
  UND 8;
  ASM_REWRITE_TAC[in_pair];
  PROOF_BY_CONTR_TAC;
  USE 8 (REWRITE_RULE[DE_MORGAN_THM]);
  REP_BASIC_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[graph_hv_finite_radius]);
  REP_BASIC_TAC;
  UND 26;
  DISCH_THEN (TH_INTRO_TAC[`v'`;`v''`]);
  ASM_REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[];
  REWRITE_TAC[INTER;EMPTY_EXISTS];
  TYPE_THEN `u` EXISTS_TAC;
  UND 10;
  EXPAND_TAC "D";
  DISCH_THEN_REWRITE;
  ASM_REWRITE_TAC[];
  (* - *)
  UND 20;
  DISCH_THEN (TH_INTRO_TAC[`v''`]);
  ASM_REWRITE_TAC[];
  REWRITE_TAC[EMPTY_EXISTS;INTER];
  ASM_MESON_TAC[];
  (* Thu Aug 26 08:46:13 EDT 2004 *)

  ]);;
  (* }}} *)

(* ------------------------------------------------------------------ *)
(* SECTION P *)
(* ------------------------------------------------------------------ *)


let (UNDISCHQ_TAC:(term->bool) -> tactic) =
  fun cond (asl,w) ->
  let cond' x = try (cond x) with failure -> false in
  let asl' = (fst(partition cond' (map (concl o snd) asl))) in
  EVERY (map (TRY o UNDISCH_TAC ) asl') (asl,w);;

let UNABBREV_TAC tm  =
  FIRST[ UNDISCHQ_TAC ( ((=) tm o rhs))
      THEN (DISCH_THEN (MP_TAC o SYM))  ;
      UNDISCHQ_TAC ( ((=) tm o lhs)) ]
  THEN DISCH_THEN_FULL_REWRITE;;

let set_simp_rewrites,extend_simp_rewrites,simp_rewrites,simp_net =
  let rewrites = ref (basic_rewrites())
  and conv_net = ref (basic_net()) in
  let set_simp_rewrites thl =
    let canon_thl = itlist (mk_rewrites false) thl ([]:thm list) in
    (rewrites := canon_thl;
     conv_net := itlist (net_of_thm true) canon_thl empty_net) in
  let extend_simp_rewrites thl =
    (* is false in simp.ml .  Important change.  *)
    let canon_thl = itlist (mk_rewrites true) thl ([]:thm list) in
     (rewrites := canon_thl @ !rewrites;
      conv_net := itlist (net_of_thm true) canon_thl (!conv_net)) in
  let simp_rewrites() = !rewrites in
  let simp_net() = !conv_net in
  set_simp_rewrites,extend_simp_rewrites,simp_rewrites,simp_net;;

let simp_ss =
  let rewmaker = mk_rewrites true in
  fun thl ->
    let cthms = itlist rewmaker thl ([]:thm list) in
    let net' = itlist (net_of_thm true) cthms (simp_net()) in
    let net'' = itlist net_of_cong (basic_congs()) net' in
  Simpset(net'',basic_prover,([]:prover list),rewmaker);;

let RSIMP_CONV thl = ONCE_SIMPLIFY_CONV (simp_ss ([]:thm list)) thl;;

let (RSIMP_TAC:thm list -> tactic) = fun (thl:thm list) -> CONV_TAC(RSIMP_CONV thl);;

let ASM_RSIMP_TAC = ASM RSIMP_TAC;;

EVERY_STEP_TAC :=
     (RSIMP_TAC[]) THEN
     REP_BASIC_TAC THEN (DROP_ALL_ANT_TAC) THEN
     (ASM_RSIMP_TAC[]) THEN
     (REWRITE_TAC[]) ;;

let SUBAGOAL_TAC t = SUBGOAL_THEN t ASSUME_TAC;;

(* EVERY_STEP_TAC := ALL_TAC *)

let subset_imp = prove_by_refinement(
  `!A B (x:A). A x /\ A SUBSET B ==> B x`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[ISUBSET];
  ]);;
  (* }}} *)

(*
extend_simp_rewrites[subset_imp]
*)

(* ------------------------------------------------------------------ *)
(* ------------------------------------------------------------------ *)


let plane_graph_image = jordan_def
  `plane_graph_image (f:(num->real)->(num->real)) G =
     mk_graph_t
       (IMAGE f (graph_vertex G),
        IMAGE2 f (graph_edge G),
        ( \ e v. (?e' v'. (graph_edge G e') /\
             (IMAGE f e' = e) /\ (f v' = v) /\
            (graph_inc G e' v'))))`;;

let plane_graph_image_e = prove_by_refinement(
  `!f G. (graph_edge (plane_graph_image f G)) =
         IMAGE2 f (graph_edge G)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[plane_graph_image;graph_edge;part1;drop0;dest_graph_t];
  (* Thu Aug 26 10:16:26 EDT 2004 *)

  ]);;
  (* }}} *)

let plane_graph_image_v = prove_by_refinement(
  `!f G. (graph_vertex (plane_graph_image f G)) =
          IMAGE f (graph_vertex G)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[plane_graph_image;dest_graph_t;graph_vertex;];
  (*     Thu Aug 26 10:17:56 EDT 2004 *)

  ]);;
  (* }}} *)

let plane_graph_image_i = prove_by_refinement(
  `!f G. (graph_inc (plane_graph_image f G)) =
     ( \ e v. (?e' v'. (graph_edge G e') /\
             (IMAGE f e' = e) /\ (f v' = v) /\
            (graph_inc G e' v')))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[plane_graph_image ;graph_inc;dest_graph_t;drop1];
  (* Thu Aug 26 10:20:07 EDT 2004 *)

  ]);;
  (* }}} *)

let plane_graph_image_bij = prove_by_refinement(
  `!f G. homeomorphism f top2 top2 /\ plane_graph G ==>
   BIJ f (graph_vertex G) (IMAGE f (graph_vertex G)) /\
   BIJ (IMAGE f) (graph_edge G) (IMAGE2 f (graph_edge G))`,
  (* {{{ proof *)
  [
  ALL_TAC ;
  (* - *)
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism;top2_unions]);
  TYPE_THEN `graph_vertex G SUBSET (euclid 2)` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  ASM_REWRITE_TAC[];
  (* - *)
  TYPE_THEN `!e. graph_edge G e ==> (e SUBSET (euclid 2))` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  IMATCH_MP_TAC  simple_arc_euclid;
  IMATCH_MP_TAC  subset_imp;
  UNIFY_EXISTS_TAC;
  (* - *)
  CONJ_TAC;
  IMATCH_MP_TAC  inj_bij;
  REWRITE_TAC[INJ];
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[subset_imp];
  (* - *)
  USE 3 (MATCH_MP image_powerset);
  REWRITE_TAC[IMAGE2];
  IMATCH_MP_TAC  inj_bij;
  REWRITE_TAC[INJ];
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  (* ASM_MESON_TAC[ISUBSET]; *)
  ]);;
  (* }}} *)

let plane_graph_image_iso = prove_by_refinement(
  `!f G. (homeomorphism f top2 top2 /\ plane_graph G ==>
      graph_isomorphic G (plane_graph_image f G))`,
  (* {{{ proof *)
  [
  ALL_TAC;
  REWRITE_TAC[graph_isomorphic;graph_iso;];
  LEFT_TAC "u";
  TYPE_THEN `f` EXISTS_TAC;
  LEFT_TAC "v";
  TYPE_THEN `IMAGE f` EXISTS_TAC;
  TYPE_THEN `f,IMAGE f` EXISTS_TAC;
  REWRITE_TAC[plane_graph_image_e;plane_graph_image_v;plane_graph_image_i];
  (* - *)
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism;top2_unions]);
  TYPE_THEN `graph_vertex G SUBSET (euclid 2)` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  (* - *)
  TYPE_THEN `!e. graph_edge G e ==> (e SUBSET (euclid 2))` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  IMATCH_MP_TAC  simple_arc_euclid;
  IMATCH_MP_TAC  subset_imp;
  UNIFY_EXISTS_TAC;
  (* - *)
  CONJ_TAC;
  IMATCH_MP_TAC  inj_bij;
  REWRITE_TAC[INJ];
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_MESON_TAC[ISUBSET];
  (* - *)
  SUBCONJ_TAC;
  USE 3 (MATCH_MP image_powerset);
  REWRITE_TAC[IMAGE2];
  IMATCH_MP_TAC  inj_bij;
  REWRITE_TAC[INJ];
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  (* A- *)
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  EQ_TAC;
  TYPE_THEN `x` UNABBREV_TAC;
  TYPE_THEN `e' = e` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[IMAGE2;BIJ;INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  (* ---- *)
  TYPE_THEN `e'`  UNABBREV_TAC ;
  REWRITE_TAC[IMAGE];
  USE 5 GSYM;
  UNIFY_EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* - *)
  USE 8(REWRITE_RULE[IMAGE]);
  UNIFY_EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* Thu Aug 26 10:49:22 EDT 2004 *)
  ]);;
  (* }}} *)

extend_simp_rewrites [(REAL_ARITH `&0 < &1`)];;

extend_simp_rewrites [prove_by_refinement(
  `metric_space(euclid 2,d_euclid)`,
  (* {{{ proof *)
  [
  ASM_MESON_TAC[metric_euclid];
  ])];;
  (* }}} *)

extend_simp_rewrites [prove_by_refinement(
  `!G. plane_graph G ==> graph_vertex G SUBSET (euclid 2)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[plane_graph];
  ])];;
  (* }}} *)

let simple_arc_end_cont = prove_by_refinement(
  `!C v v'. simple_arc_end C v v' <=>
       (?f. (C = IMAGE f {x | &0 <= x /\ x <= &1}) /\
        continuous f
           (top_of_metric ({x | &0 <= x /\ x <= &1},d_real)) top2 /\
              INJ f {x | &0 <= x /\ x <= &1} (euclid 2) /\
              (f (&0) = v) /\
              (f (&1) = v'))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[simple_arc_end];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  EQ_TAC;
  TH_INTRO_TAC [`&0`;`&1`;`f`;`euclid 2`;`d_euclid`] cont_extend_real_lemma;
  CONJ_TAC;
  ASM_REWRITE_TAC[GSYM top2];
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REWRITE_TAC[IMAGE;SUBSET];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `g` EXISTS_TAC;
  CONJ_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE];
  IMATCH_MP_TAC  EQ_EXT;
  EQ_TAC;
  UNIFY_EXISTS_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  UNIFY_EXISTS_TAC;
  ASM_MESON_TAC[];
  (* -- *)
  ASM_REWRITE_TAC[top2];
  CONJ_TAC;
  REWRITE_TAC[INJ];
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  ASM_MESON_TAC[];
  ASM_MESON_TAC[REAL_ARITH `x <=. x `;REAL_ARITH `&0 <=. &1`];
  (* - *)
  UNIFY_EXISTS_TAC;
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  continuous_interval;
  (* Thu Aug 26 12:57:09 EDT 2004 *)
  ]);;
  (* }}} *)

let graph_edge_euclid =  prove_by_refinement(
  `!G e. (plane_graph G /\ graph_edge G e) ==> (e SUBSET (euclid 2))`,
  (* {{{ proof *)
  [
  ALL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  IMATCH_MP_TAC  simple_arc_euclid;
  IMATCH_MP_TAC  subset_imp;
  UNIFY_EXISTS_TAC;
  ]);;
  (* }}} *)

let plane_graph_image_plane = prove_by_refinement(
  `!f G. (homeomorphism f top2 top2 /\ good_plane_graph G ==>
     good_plane_graph(plane_graph_image f G))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[good_plane_graph];
  TH_INTRO_TAC[`G`;`plane_graph_image f G`] graph_isomorphic_graph;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  plane_graph_image_iso;
  ASM_REWRITE_TAC[plane_graph];
  (* - *)
  TYPE_THEN `graph_vertex G SUBSET (euclid 2)` SUBGOAL_TAC;
  (* - *)
  TYPE_THEN `!e. graph_edge G e ==> (e SUBSET (euclid 2))` SUBGOAL_TAC;
  IMATCH_MP_TAC  graph_edge_euclid;
  UNIFY_EXISTS_TAC;
  (* - *)
  TH_INTRO_TAC[`f`;`G`] plane_graph_image_bij;
  (* A- *)
  ASM_REWRITE_TAC[plane_graph;GSYM CONJ_ASSOC;];
  TYPE_THEN `(!e v v'.  graph_edge (plane_graph_image f G) e /\  ~(v = v') /\  graph_inc (plane_graph_image f G) e v /\  graph_inc (plane_graph_image f G) e v' ==> simple_arc_end e v v')` SUBGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph_image_e;plane_graph_image_v;plane_graph_image_i]);
  TYPE_THEN `v` UNABBREV_TAC;
  TYPE_THEN `v'` UNABBREV_TAC;
  TYPE_THEN `e` UNABBREV_TAC;
  TYPE_THEN `e' = e''` SUBGOAL_TAC ;
  USE 6 (REWRITE_RULE[BIJ;INJ;IMAGE2]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `e''` UNABBREV_TAC;
  UND 0 THEN (DISCH_THEN (TH_INTRO_TAC [`e'`;`v'''`;`v''`]));
  DISCH_TAC;
  TYPE_THEN `v'''` UNABBREV_TAC;
  USE 0 (REWRITE_RULE[simple_arc_end_cont]);
  REWRITE_TAC[simple_arc_end_cont];
  TYPE_THEN `f o f'` EXISTS_TAC;
  REWRITE_TAC[IMAGE_o];
  (* -- *)
  CONJ_TAC;
  IMATCH_MP_TAC  continuous_comp;
  TYPE_THEN `top2` EXISTS_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism]);
  ASM_REWRITE_TAC[top2_unions];
  TYPE_THEN `UNIONS (top_of_metric ({x | &0 <= x /\ x <= &1},d_real)) = {x | &0 <= x /\ x <= &1}` SUBGOAL_TAC;
  TH_INTRO_TAC[`{x | &0 <= x /\ x <= &1}`;`d_real`] top_of_metric_unions;
  TYPE_THEN `{x | &0 <= x /\ x <= &1} SUBSET UNIV ` SUBAGOAL_TAC;
  alpha_tac;
  IMATCH_MP_TAC  metric_subspace;
  UNIFY_EXISTS_TAC;
  REWRITE_TAC [metric_real;];
  UND 21 THEN   DISCH_THEN (fun t->ONCE_REWRITE_TAC[GSYM t]);
  REWRITE_TAC[];
  USE 15 (REWRITE_RULE[INJ]);
  REWRITE_TAC[IMAGE;SUBSET];
  FIRST_ASSUM IMATCH_MP_TAC ;
  (* -- *)
  CONJ_TAC;
  REWRITE_TAC[comp_comp];
  IMATCH_MP_TAC  COMP_INJ;
  UNIFY_EXISTS_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism;BIJ;top2_unions]);
  REWRITE_TAC[o_DEF];
  (* B- *)
  ASM_REWRITE_TAC[];
  TYPE_THEN `graph_edge (plane_graph_image f G) SUBSET simple_arc top2` SUBGOAL_TAC;
  REWRITE_TAC[SUBSET];
  TH_INTRO_TAC[`plane_graph_image f G`;`x`] graph_edge_end_select;
  UND 8 THEN DISCH_THEN (TH_INTRO_TAC[`x`;`v`;`v'`]);
  IMATCH_MP_TAC  simple_arc_end_simple;
  UNIFY_EXISTS_TAC;
  KILL 8;
  (* - *)
  CONJ_TAC;
  MP_TAC plane_graph_image_v THEN DISCH_THEN_FULL_REWRITE;
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism;BIJ;INJ;]);
  USE 16 (REWRITE_RULE[top2_unions]);
  REWRITE_TAC[IMAGE;SUBSET];
  FIRST_ASSUM IMATCH_MP_TAC ;
  IMATCH_MP_TAC  subset_imp;
  UNIFY_EXISTS_TAC;
  (* - *)
  CONJ_TAC;
  (fun t-> (RULE_ASSUM_TAC (REWRITE_RULE t) THEN REWRITE_TAC t ))  [plane_graph_image_e;plane_graph_image_v;plane_graph_image_i];
  IMATCH_MP_TAC  EQ_EXT;
  EQ_TAC;
  TYPE_THEN `x`  UNABBREV_TAC ;
  TYPE_THEN `e` UNABBREV_TAC;
  REWRITE_TAC[INTER];
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  TSPEC `e'` 11;
  REWR 10;
  USE 10 (REWRITE_RULE[INTER]);
  REWRITE_TAC[IMAGE];
  UNIFY_EXISTS_TAC;
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE];
  TYPE_THEN `v'` EXISTS_TAC;
  TH_INTRO_TAC [`G`;`e'`] graph_inc_subset;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  IMATCH_MP_TAC  subset_imp;
  UNIFY_EXISTS_TAC;
  USE 8 (REWRITE_RULE[IMAGE2]);
  TYPE_THEN `FF = IMAGE f` ABBREV_TAC ;
  USE 8 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `x'` EXISTS_TAC;
  USE 10 (REWRITE_RULE[INTER]);
  TYPE_THEN `FF`  UNABBREV_TAC;
  USE 10 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `x` UNABBREV_TAC;
  TYPE_THEN `x''` EXISTS_TAC;
  TYPE_THEN `e` UNABBREV_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  REWRITE_TAC[INTER];
  USE 13 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `x''  =x` SUBAGOAL_TAC;
  USE 2(REWRITE_RULE[homeomorphism;BIJ;INJ;top2_unions]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  CONJ_TAC;
  IMATCH_MP_TAC  subset_imp;
  UNIFY_EXISTS_TAC;
  TSPEC `x'` 5;
  IMATCH_MP_TAC  subset_imp;
  TYPE_THEN `x'` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* C- *)
  (fun t-> (RULE_ASSUM_TAC (REWRITE_RULE t) THEN REWRITE_TAC t ))  [plane_graph_image_e;plane_graph_image_v;plane_graph_image_i];
  USE 10 (REWRITE_RULE[IMAGE2]);
  USE 11 (REWRITE_RULE[IMAGE2]);
  TYPE_THEN `FF = IMAGE f` ABBREV_TAC ;
  USE 10 (REWRITE_RULE[IMAGE]);
  USE 11 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `e` UNABBREV_TAC;
  TYPE_THEN `e'` UNABBREV_TAC;
  TH_INTRO_TAC [`f`;`euclid 2`;`euclid 2`;`x'`;`x`] (GSYM inj_inter);
  RULE_ASSUM_TAC (REWRITE_RULE[homeomorphism;BIJ;top2_unions]);
  TYPE_THEN `FF` UNABBREV_TAC;
  IMATCH_MP_TAC  IMAGE_SUBSET;
  RULE_ASSUM_TAC (REWRITE_RULE[plane_graph]);
  TYPEL_THEN [`x'`;`x`] (fun t-> UND 1 THEN DISCH_THEN (TH_INTRO_TAC t));
  DISCH_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  ]);;
  (* }}} *)

(* state MP *)

let h_compat = jordan_def `h_compat f <=> !x y. (SND x = SND y) ==>
   (IMAGE f (mk_line (point x) (point y)) =
          mk_line (f (point x)) (f (point y)))`;;

let v_compat = jordan_def `v_compat f <=> !x y. (FST x = FST y) ==>
   (IMAGE f (mk_line (point x) (point y)) =
          mk_line (f (point x)) (f (point y)))`;;

let h_translate = jordan_def `h_translate r p = p + r *# e1`;;

let v_translate = jordan_def `v_translate r p = p + r *# e2`;;

let r_scale = jordan_def `r_scale r p =
        if ( &.0 < p 0) then (point (r * p 0, p 1)) else p`;;

let u_scale = jordan_def `u_scale r p =
        if ( &.0 < p 1) then (point ( p 0, r * p 1)) else p`;;

let cont_domain = prove_by_refinement(
  `!(f:A->B) g U V. (continuous f U V) /\ (!x. UNIONS U x ==> (f x = g x))
    ==> (continuous g U V)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[preimage;continuous;];
  TYPE_THEN `{x | UNIONS U x /\ v (g x)} = {x | UNIONS U x /\ v (f x)}` SUBAGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  IMATCH_MP_TAC  (TAUT `(A ==> (B <=> C)) ==> (A /\ B <=> A /\ C)`);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ]);;
  (* }}} *)

let h_translate_bij = prove_by_refinement(
  `!r. BIJ (h_translate r) (euclid 2) (euclid 2)`,
  (* {{{ proof *)

  [
  REWRITE_TAC[BIJ;INJ;h_translate];
  SUBCONJ_TAC;
  CONJ_TAC;
  ASM_SIMP_TAC[euclid_add_closure;e1;point_scale;euclid_point];
  RULE_ASSUM_TAC (REWRITE_RULE[euclid_plus;euclid_scale;e1]);
  IMATCH_MP_TAC  EQ_EXT;
  USE 0 (REWRITE_RULE[FUN_EQ_THM]);
  TSPEC `x'` 0;
  UND 0 THEN REAL_ARITH_TAC;
  REWRITE_TAC[SURJ;h_translate];
  REP_BASIC_TAC;
  TYPE_THEN `x - (r *# e1)` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[point_scale;e1];
  ASM_SIMP_TAC[euclid_sub_closure;euclid_point];
  REWRITE_TAC[euclid_plus;euclid_minus;euclid_scale];
  IMATCH_MP_TAC  EQ_EXT;
  REAL_ARITH_TAC;
  (* Tue Sep  7 10:15:46 EDT 2004 *)

  ]);;

  (* }}} *)

let v_translate_bij = prove_by_refinement(
  `!r. BIJ (v_translate r) (euclid 2) (euclid 2)`,
  (* {{{ proof *)

  [
  REWRITE_TAC[BIJ;INJ;v_translate];
  SUBCONJ_TAC;
  CONJ_TAC;
  ASM_SIMP_TAC[euclid_add_closure;e2;point_scale;euclid_point];
  RULE_ASSUM_TAC (REWRITE_RULE[euclid_plus;euclid_scale;e2]);
  IMATCH_MP_TAC  EQ_EXT;
  USE 0 (REWRITE_RULE[FUN_EQ_THM]);
  TSPEC `x'` 0;
  UND 0 THEN REAL_ARITH_TAC;
  REWRITE_TAC[SURJ;v_translate];
  REP_BASIC_TAC;
  TYPE_THEN `x - (r *# e2)` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[point_scale;e2];
  ASM_SIMP_TAC[euclid_sub_closure;euclid_point];
  REWRITE_TAC[euclid_plus;euclid_minus;euclid_scale];
  IMATCH_MP_TAC  EQ_EXT;
  REAL_ARITH_TAC;
  (* Tue Sep  7 10:16:38 EDT 2004 *)

  ]);;

  (* }}} *)

extend_simp_rewrites [euclid_point];;
extend_simp_rewrites [coord01];;

let r_scale_bij = prove_by_refinement(
  `!r. (&0 < r) ==> BIJ (r_scale r) (euclid 2) (euclid 2)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[BIJ;INJ;r_scale;];
  SUBCONJ_TAC;
  CONJ_TAC;
  COND_CASES_TAC;
  REWRITE_TAC[euclid_point];
  USE 2 (MATCH_MP   point_onto);
  USE 3 (MATCH_MP   point_onto);
  REWRITE_TAC[point_inj];
  TYPE_THEN `x` UNABBREV_TAC;
  TYPE_THEN `y` UNABBREV_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  RULE_ASSUM_TAC (REWRITE_RULE[coord01]);
  UND 1 THEN COND_CASES_TAC;
  UND 1 THEN COND_CASES_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[point_inj;PAIR_SPLIT]);
  RULE_ASSUM_TAC (REWRITE_RULE[REAL_EQ_LMUL]);
  UND 4 THEN UND 0 THEN REAL_ARITH_TAC ;
  RULE_ASSUM_TAC (REWRITE_RULE[point_inj;PAIR_SPLIT]);
  TYPE_THEN `FST p` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  UND 3 THEN REWRITE_TAC[];
  REWRITE_TAC[real_gt];
  IMATCH_MP_TAC  REAL_LT_MUL;
  UND 1 THEN COND_CASES_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[point_inj;PAIR_SPLIT ]);
  TYPE_THEN `FST p'` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  UND 2 THEN REWRITE_TAC[];
  IMATCH_MP_TAC  REAL_LT_MUL;
  RULE_ASSUM_TAC (REWRITE_RULE[point_inj]);
  KILL 1;
  REWRITE_TAC[SURJ;r_scale];
  KILL 2;
  USE 1 (MATCH_MP point_onto);
  TYPE_THEN `x` UNABBREV_TAC;
  TYPE_THEN `&0 < FST p` ASM_CASES_TAC;
  TYPE_THEN `point ((&1/r)* FST p, SND p)` EXISTS_TAC;
  TYPE_THEN `&0 < &1/ r  * FST p` SUBAGOAL_TAC;
  IMATCH_MP_TAC  REAL_LT_MUL;
  IMATCH_MP_TAC  REAL_LT_DIV;
  ASM_REWRITE_TAC[];
  AP_TERM_TAC;
  REWRITE_TAC[PAIR_SPLIT;REAL_MUL_ASSOC];
  TYPE_THEN `(r * &1/r) * FST p = &1 * FST p` SUBAGOAL_TAC;
  AP_THM_TAC;
  AP_TERM_TAC;
  IMATCH_MP_TAC  REAL_DIV_LMUL;
  UND 3 THEN UND 0 THEN REAL_ARITH_TAC;
  REDUCE_TAC;
  TYPE_THEN `point p` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* Tue Sep  7 10:55:54 EDT 2004 *)

  ]);;
  (* }}} *)

let u_scale_bij = prove_by_refinement(
  `!r. (&0 < r) ==> BIJ (u_scale r) (euclid 2) (euclid 2)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[BIJ;INJ;u_scale;];
  SUBCONJ_TAC;
  CONJ_TAC;
  COND_CASES_TAC;
  USE 2 (MATCH_MP   point_onto);
  USE 3 (MATCH_MP   point_onto);
  REWRITE_TAC[point_inj];
  TYPE_THEN `x` UNABBREV_TAC;
  TYPE_THEN `y` UNABBREV_TAC;
  REWRITE_TAC[PAIR_SPLIT];
  RULE_ASSUM_TAC (REWRITE_RULE[coord01]);
  UND 1 THEN COND_CASES_TAC;
  UND 1 THEN COND_CASES_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[point_inj;PAIR_SPLIT]);
  RULE_ASSUM_TAC (REWRITE_RULE[REAL_EQ_LMUL]);
  UND 1 THEN UND 0 THEN REAL_ARITH_TAC ;
  RULE_ASSUM_TAC (REWRITE_RULE[point_inj;PAIR_SPLIT]);
  TYPE_THEN `SND p` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  UND 3 THEN REWRITE_TAC[];
  IMATCH_MP_TAC  REAL_LT_MUL;
  UND 1 THEN COND_CASES_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[point_inj;PAIR_SPLIT ]);
  TYPE_THEN `SND p'` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  UND 2 THEN REWRITE_TAC[];
  IMATCH_MP_TAC  REAL_LT_MUL;
  RULE_ASSUM_TAC (REWRITE_RULE[point_inj]);
  KILL 1;
  REWRITE_TAC[SURJ;u_scale];
  KILL 2;
  USE 1 (MATCH_MP point_onto);
  TYPE_THEN `x` UNABBREV_TAC;
  TYPE_THEN `&0 < SND  p` ASM_CASES_TAC;
  TYPE_THEN `point (FST p, (&1/r)* SND  p)` EXISTS_TAC;
  TYPE_THEN `&0 < &1/ r  * SND  p` SUBAGOAL_TAC;
  IMATCH_MP_TAC  REAL_LT_MUL;
  IMATCH_MP_TAC  REAL_LT_DIV;
  ASM_REWRITE_TAC[];
  AP_TERM_TAC;
  REWRITE_TAC[PAIR_SPLIT;REAL_MUL_ASSOC];
  TYPE_THEN `(r * &1/r) * SND  p = &1 * SND  p` SUBAGOAL_TAC;
  AP_THM_TAC;
  AP_TERM_TAC;
  IMATCH_MP_TAC  REAL_DIV_LMUL;
  UND 3 THEN UND 0 THEN REAL_ARITH_TAC;
  REDUCE_TAC;
  TYPE_THEN `point p` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* Tue Sep  7 11:01:53 EDT 2004 *)

  ]);;
  (* }}} *)

let h_translate_inv = prove_by_refinement(
  `!r x. (euclid 2 x) ==>
   (h_translate (--. r) x = INV (h_translate r) (euclid 2) (euclid 2) x)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_SYM;
  TH_INTRO_TAC[`h_translate r`;`euclid 2`;`euclid 2`;`h_translate (--. r) x`;`x`] INVERSE_XY;
  ASM_REWRITE_TAC[h_translate_bij;h_translate;e1;point_scale];
  ASM_SIMP_TAC[euclid_add_closure;euclid_point];
  REWRITE_TAC[h_translate;euclid_plus;e1;euclid_scale];
  IMATCH_MP_TAC  EQ_EXT;
  REAL_ARITH_TAC;
  (* Tue Sep  7 11:11:17 EDT 2004 *)
  ]);;
  (* }}} *)

let v_translate_inv = prove_by_refinement(
  `!r x. (euclid 2 x) ==>
   (v_translate (--. r) x = INV (v_translate r) (euclid 2) (euclid 2) x)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_SYM;
  TH_INTRO_TAC[`v_translate r`;`euclid 2`;`euclid 2`;`v_translate (--. r) x`;`x`] INVERSE_XY;
  ASM_REWRITE_TAC[v_translate_bij;v_translate;e2;point_scale];
  ASM_SIMP_TAC[euclid_add_closure;euclid_point];
  REWRITE_TAC[v_translate;euclid_plus;e2;euclid_scale];
  IMATCH_MP_TAC  EQ_EXT;
  REAL_ARITH_TAC;
  (* Tue Sep  7 11:12:42 EDT 2004 *)
  ]);;
  (* }}} *)

extend_simp_rewrites[prove_by_refinement(
  `!x r. (&0 < r) ==> (r * (&1/r) * x = x)`,
  (* {{{ proof *)
  [
  REWRITE_TAC [REAL_MUL_ASSOC];
  TYPE_THEN `(r * &1/r) * x = &1 * x` SUBAGOAL_TAC;
  AP_THM_TAC;
  AP_TERM_TAC;
  IMATCH_MP_TAC  REAL_DIV_LMUL;
  UND 1 THEN UND 0 THEN REAL_ARITH_TAC;
  REDUCE_TAC;
  ])];;
  (* }}} *)

extend_simp_rewrites[ prove_by_refinement(
  `!r. (&0 < r) ==> (&0 < &1 / r)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  REAL_LT_DIV;
  ])];;
  (* }}} *)

extend_simp_rewrites[ REAL_LE_POW_2];;

extend_simp_rewrites[ prove_by_refinement(
  `!x y. &0 <= x pow 2 + y pow 2`,
  (* {{{ proof *)
  [
  ALL_TAC;
  IMATCH_MP_TAC  REAL_LE_ADD;
  ])];;
  (* }}} *)

let r_scale_inv = prove_by_refinement(
  `!r x. (&0 < r) /\ (euclid 2 x) ==>
   (r_scale (&1/r) x = INV (r_scale r) (euclid 2) (euclid 2) x)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_SYM;
  TH_INTRO_TAC[`r_scale r`;`euclid 2`;`euclid 2`;`r_scale (&1/r) x`;`x`] INVERSE_XY;
  ASM_SIMP_TAC [r_scale_bij];
  TH_INTRO_TAC[`&1/r`] r_scale_bij;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REWRITE_TAC[r_scale];
  USE 0 (MATCH_MP point_onto);
  TYPE_THEN `x` UNABBREV_TAC;
  TYPE_THEN `&0 < FST p` ASM_CASES_TAC;
  REWRITE_TAC[coord01];
  TYPE_THEN `&0 < (&1 / r) * FST p` SUBAGOAL_TAC;
  IMATCH_MP_TAC  REAL_LT_MUL;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  (* Tue Sep  7 11:40:41 EDT 2004 *)

  ]);;
  (* }}} *)

let u_scale_inv = prove_by_refinement(
  `!r x. (&0 < r) /\ (euclid 2 x) ==>
   (u_scale (&1/r) x = INV (u_scale r) (euclid 2) (euclid 2) x)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_SYM;
  TH_INTRO_TAC[`u_scale r`;`euclid 2`;`euclid 2`;`u_scale (&1/r) x`;`x`] INVERSE_XY;
  ASM_SIMP_TAC [u_scale_bij];
  TH_INTRO_TAC[`&1/r`] u_scale_bij;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REWRITE_TAC[u_scale];
  USE 0 (MATCH_MP point_onto);
  TYPE_THEN `x` UNABBREV_TAC;
  TYPE_THEN `&0 < SND p` ASM_CASES_TAC;
  REWRITE_TAC[coord01];
  TYPE_THEN `&0 < (&1 / r) * SND  p` SUBAGOAL_TAC;
  IMATCH_MP_TAC  REAL_LT_MUL;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  (* Tue Sep  7 11:56:05 EDT 2004 *)


  ]);;
  (* }}} *)

let metric_continuous_continuous_top2 = prove_by_refinement(
  `!f. (IMAGE f (euclid 2) SUBSET (euclid 2) ==>
     (continuous f top2 top2 =
         metric_continuous f (euclid 2,d_euclid) (euclid 2,d_euclid)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[top2];
  IMATCH_MP_TAC  metric_continuous_continuous;
  ]);;
  (* }}} *)

let h_translate_cont = prove_by_refinement(
  `!r. continuous (h_translate r) (top2) (top2)`,
  (* {{{ proof *)
  [
  ALL_TAC;
  TH_INTRO_TAC [`h_translate r`] metric_continuous_continuous_top2;
  ASSUME_TAC h_translate_bij;
  TSPEC `r` 0;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REWRITE_TAC[IMAGE;SUBSET];
  FIRST_ASSUM IMATCH_MP_TAC ;
  (* - *)
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  TYPE_THEN `epsilon` EXISTS_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[h_translate];
  TH_INTRO_TAC[`2`;`x`;`y`;`r *# e1`] metric_translate;
  REWRITE_TAC[e1;point_scale];
  (* Tue Sep  7 12:09:30 EDT 2004 *)

  ]);;
  (* }}} *)

let v_translate_cont = prove_by_refinement(
  `!r. continuous (v_translate r) (top2) (top2)`,
  (* {{{ proof *)
  [
  ALL_TAC;
  TH_INTRO_TAC [`v_translate r`] metric_continuous_continuous_top2;
  ASSUME_TAC v_translate_bij;
  TSPEC `r` 0;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REWRITE_TAC[IMAGE;SUBSET];
  FIRST_ASSUM IMATCH_MP_TAC ;
  (* - *)
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  TYPE_THEN `epsilon` EXISTS_TAC;
  REP_BASIC_TAC;
  REWRITE_TAC[v_translate];
  TH_INTRO_TAC[`2`;`x`;`y`;`r *# e2`] metric_translate;
  REWRITE_TAC[e2;point_scale];
  (* Tue Sep  7 12:10:54 EDT 2004 *)
  ]);;
  (* }}} *)

let r_scale_cont = prove_by_refinement(
  `!r. (&0 < r) ==> (continuous (r_scale r) top2 top2)`,
  (* {{{ proof *)
  [
  ALL_TAC;
  TYPE_THEN `&0 < (&1 + r)` SUBAGOAL_TAC;
  UND 0 THEN REAL_ARITH_TAC;
  TH_INTRO_TAC[`r_scale r`] metric_continuous_continuous_top2;
  ASSUME_TAC r_scale_bij;
  TSPEC `r` 2;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REWRITE_TAC[IMAGE;SUBSET];
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  TYPE_THEN `&1/(&1 + r)*epsilon` EXISTS_TAC;
  TYPE_THEN `epsilon' = &1/(&1+r)*epsilon` ABBREV_TAC ;
  TYPE_THEN `epsilon = (&1 + r)*epsilon'` SUBAGOAL_TAC;
  TYPE_THEN `epsilon'` UNABBREV_TAC;
  TYPE_THEN `epsilon` UNABBREV_TAC;
  KILL 4;
  SUBCONJ_TAC;
  ASM_MESON_TAC[REAL_PROP_POS_LMUL];
  USE 5(MATCH_MP point_onto);
  TYPE_THEN `y` UNABBREV_TAC;
  USE 6(MATCH_MP point_onto);
  TYPE_THEN `x` UNABBREV_TAC;
  (* - *)
  TYPE_THEN `!x y. (r*x - r*y) pow 2 <= ((&1 + r) pow 2 ) * ((x - y) pow 2)` SUBAGOAL_TAC;
  REWRITE_TAC[GSYM REAL_SUB_LDISTRIB;REAL_POW_MUL ];
  IMATCH_MP_TAC  REAL_LE_RMUL;
  REWRITE_TAC[REAL_POW_2];
  IMATCH_MP_TAC  ABS_SQUARE_LE;
  UND 0 THEN REAL_ARITH_TAC;
  REWRITE_TAC[GSYM REAL_POW_MUL];
  (* - *)
  TYPE_THEN `!x y. (&1 pow 2) *((x - y) pow 2) <= ((&1 + r) pow 2 ) * ((x - y) pow 2)` SUBAGOAL_TAC;
  IMATCH_MP_TAC  REAL_LE_RMUL;
  REWRITE_TAC[REAL_POW_2];
  IMATCH_MP_TAC  ABS_SQUARE_LE;
  UND 0 THEN  REAL_ARITH_TAC;
  UND 6 THEN REDUCE_TAC;
  (* - *)
  TYPE_THEN `!x y. (&0 <= x) /\ (&0 <= y) ==> ((r*x + y) pow 2 <= ((&1 + r) pow 2) * ((x + y) pow 2))` SUBAGOAL_TAC;
  REWRITE_TAC[GSYM REAL_POW_MUL];
  REWRITE_TAC[REAL_POW_2];
  IMATCH_MP_TAC  ABS_SQUARE_LE;
  TYPE_THEN `abs  (r*x' + y') = r*x' + y'` SUBAGOAL_TAC;
  REWRITE_TAC[ABS_REFL];
  IMATCH_MP_TAC  REAL_LE_ADD;
  ASM_MESON_TAC[REAL_LE_MUL;REAL_ARITH `&0 < x==> &0 <= x`];
  ineq_le_tac `(r*x' + y') + x' + r*y'  = (&1 + r)*(x' + y')` ;
  (* A - *)
  TYPE_THEN `sqrt ((&1 + r) pow 2 * ((FST p' - FST p) pow 2 + (SND p' - SND p) pow 2)) < (&1 + r) * epsilon'` SUBAGOAL_TAC;
  TYPE_THEN `sqrt (((&1 + r)*epsilon') pow 2) = (&1 + r)*epsilon'` SUBAGOAL_TAC;
  IMATCH_MP_TAC  POW_2_SQRT;
  IMATCH_MP_TAC  REAL_LE_MUL;
  UND 7 THEN UND 1 THEN REAL_ARITH_TAC;
  UND 9 THEN DISCH_THEN (fun t-> ONCE_REWRITE_TAC [GSYM t]);
  IMATCH_MP_TAC SQRT_MONO_LT;
  REWRITE_TAC[GSYM REAL_POW_MUL;REAL_ADD_LDISTRIB ];
  REWRITE_TAC[REAL_POW_MUL;GSYM REAL_ADD_LDISTRIB ];
  IMATCH_MP_TAC  REAL_LT_LMUL;
  CONJ_TAC;
  IMATCH_MP_TAC  REAL_PROP_POS_POW;
  TH_INTRO_TAC [`(FST p' - FST p) pow 2 + (SND p' - SND p) pow 2`;`epsilon' pow 2`] (GSYM REAL_PROP_LT_SQRT);
  TYPE_THEN `sqrt(epsilon' pow 2) = epsilon'` SUBAGOAL_TAC;
  IMATCH_MP_TAC  POW_2_SQRT;
  UND 7 THEN REAL_ARITH_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[d_euclid_point]);
  (* - *)
  IMATCH_MP_TAC  REAL_LET_TRANS;
  TYPE_THEN `sqrt ((&1 + r) pow 2 * ((FST p' - FST p) pow 2 + (SND p' - SND p) pow 2))` EXISTS_TAC;
  (* B- *)
  REWRITE_TAC[r_scale];
  COND_CASES_TAC THEN COND_CASES_TAC;
  UND 4 THEN  REWRITE_TAC[d_euclid_point];
  IMATCH_MP_TAC  SQRT_MONO_LE;
  (*  IMATCH_MP_TAC  REAL_LET_TRANS; *)
  REWRITE_TAC[REAL_LDISTRIB];
  IMATCH_MP_TAC  REAL_LE_ADD2;
  (* 3 LEFT *)
  UND 4 THEN (REWRITE_TAC [d_euclid_point]);
  TYPE_THEN `u = --. (FST p)` ABBREV_TAC ;
  TYPE_THEN `FST p = -- u` SUBAGOAL_TAC;
  UND 12 THEN REAL_ARITH_TAC;
  REWRITE_TAC[REAL_ARITH `x - --. y = x + y`];
  IMATCH_MP_TAC  SQRT_MONO_LE;
  REWRITE_TAC[REAL_LDISTRIB];
  IMATCH_MP_TAC  REAL_LE_ADD2;
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 10 THEN UND 13 THEN UND 11 THEN REAL_ARITH_TAC;
  (* 2 LEFT *)
  UND 4 THEN (REWRITE_TAC [d_euclid_point]);
  TYPE_THEN `u = --. (FST p')` ABBREV_TAC ;
  TYPE_THEN `FST p' = -- u` SUBAGOAL_TAC;
  UND 12 THEN REAL_ARITH_TAC;
  REWRITE_TAC[REAL_ARITH `-- x -  v = -- (v + x)`;REAL_POW_NEG;EVEN2 ];
  IMATCH_MP_TAC  SQRT_MONO_LE;
  REWRITE_TAC[REAL_LDISTRIB];
  IMATCH_MP_TAC  REAL_LE_ADD2;
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 10 THEN UND 13 THEN UND 11 THEN REAL_ARITH_TAC;
  (* 1 LEFT *)
  UND 4 THEN (REWRITE_TAC [d_euclid_point]);
  IMATCH_MP_TAC  SQRT_MONO_LE;
  REWRITE_TAC[REAL_LDISTRIB];
  IMATCH_MP_TAC  REAL_LE_ADD2;
  (* Tue Sep  7 15:33:59 EDT 2004 *)

  ]);;
  (* }}} *)

let u_scale_cont = prove_by_refinement(
  `!r. (&0 < r) ==> (continuous (u_scale r) top2 top2)`,
  (* {{{ proof *)
  [
  ALL_TAC;
  TYPE_THEN `&0 < (&1 + r)` SUBAGOAL_TAC;
  UND 0 THEN REAL_ARITH_TAC;
  TH_INTRO_TAC[`u_scale r`] metric_continuous_continuous_top2;
  ASSUME_TAC u_scale_bij;
  TSPEC `r` 2;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REWRITE_TAC[IMAGE;SUBSET];
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWRITE_TAC[metric_continuous;metric_continuous_pt];
  TYPE_THEN `&1/(&1 + r)*epsilon` EXISTS_TAC;
  TYPE_THEN `epsilon' = &1/(&1+r)*epsilon` ABBREV_TAC ;
  TYPE_THEN `epsilon = (&1 + r)*epsilon'` SUBAGOAL_TAC;
  TYPE_THEN `epsilon'` UNABBREV_TAC;
  TYPE_THEN `epsilon` UNABBREV_TAC;
  KILL 4;
  SUBCONJ_TAC;
  ASM_MESON_TAC[REAL_PROP_POS_LMUL];
  USE 5(MATCH_MP point_onto);
  TYPE_THEN `y` UNABBREV_TAC;
  USE 6(MATCH_MP point_onto);
  TYPE_THEN `x` UNABBREV_TAC;
  (* - *)
  TYPE_THEN `!x y. (r*x - r*y) pow 2 <= ((&1 + r) pow 2 ) * ((x - y) pow 2)` SUBAGOAL_TAC;
  REWRITE_TAC[GSYM REAL_SUB_LDISTRIB;REAL_POW_MUL ];
  IMATCH_MP_TAC  REAL_LE_RMUL;
  REWRITE_TAC[REAL_POW_2];
  IMATCH_MP_TAC  ABS_SQUARE_LE;
  UND 0 THEN REAL_ARITH_TAC;
  REWRITE_TAC[GSYM REAL_POW_MUL];
  (* - *)
  TYPE_THEN `!x y. (&1 pow 2) *((x - y) pow 2) <= ((&1 + r) pow 2 ) * ((x - y) pow 2)` SUBAGOAL_TAC;
  IMATCH_MP_TAC  REAL_LE_RMUL;
  REWRITE_TAC[REAL_POW_2];
  IMATCH_MP_TAC  ABS_SQUARE_LE;
  UND 0 THEN  REAL_ARITH_TAC;
  UND 6 THEN REDUCE_TAC;
  (* - *)
  TYPE_THEN `!x y. (&0 <= x) /\ (&0 <= y) ==> ((r*x + y) pow 2 <= ((&1 + r) pow 2) * ((x + y) pow 2))` SUBAGOAL_TAC;
  REWRITE_TAC[GSYM REAL_POW_MUL];
  REWRITE_TAC[REAL_POW_2];
  IMATCH_MP_TAC  ABS_SQUARE_LE;
  TYPE_THEN `abs  (r*x' + y') = r*x' + y'` SUBAGOAL_TAC;
  REWRITE_TAC[ABS_REFL];
  IMATCH_MP_TAC  REAL_LE_ADD;
  ASM_MESON_TAC[REAL_LE_MUL;REAL_ARITH `&0 < x==> &0 <= x`];
  ineq_le_tac `(r*x' + y') + x' + r*y'  = (&1 + r)*(x' + y')` ;
  (* A - *)
  TYPE_THEN `sqrt ((&1 + r) pow 2 * ((FST p' - FST p) pow 2 + (SND p' - SND p) pow 2)) < (&1 + r) * epsilon'` SUBAGOAL_TAC;
  TYPE_THEN `sqrt (((&1 + r)*epsilon') pow 2) = (&1 + r)*epsilon'` SUBAGOAL_TAC;
  IMATCH_MP_TAC  POW_2_SQRT;
  IMATCH_MP_TAC  REAL_LE_MUL;
  UND 7 THEN UND 1 THEN REAL_ARITH_TAC;
  UND 9 THEN DISCH_THEN (fun t-> ONCE_REWRITE_TAC [GSYM t]);
  IMATCH_MP_TAC SQRT_MONO_LT;
  REWRITE_TAC[GSYM REAL_POW_MUL;REAL_ADD_LDISTRIB ];
  REWRITE_TAC[REAL_POW_MUL;GSYM REAL_ADD_LDISTRIB ];
  IMATCH_MP_TAC  REAL_LT_LMUL;
  CONJ_TAC;
  IMATCH_MP_TAC  REAL_PROP_POS_POW;
  TH_INTRO_TAC [`(FST p' - FST p) pow 2 + (SND p' - SND p) pow 2`;`epsilon' pow 2`] (GSYM REAL_PROP_LT_SQRT);
  TYPE_THEN `sqrt(epsilon' pow 2) = epsilon'` SUBAGOAL_TAC;
  IMATCH_MP_TAC  POW_2_SQRT;
  UND 7 THEN REAL_ARITH_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[d_euclid_point]);
  (* - *)
  IMATCH_MP_TAC  REAL_LET_TRANS;
  TYPE_THEN `sqrt ((&1 + r) pow 2 * ((FST p' - FST p) pow 2 + (SND p' - SND p) pow 2))` EXISTS_TAC;
  (* B- *)
  REWRITE_TAC[u_scale];
  COND_CASES_TAC THEN COND_CASES_TAC;
  UND 4 THEN  REWRITE_TAC[d_euclid_point];
  IMATCH_MP_TAC  SQRT_MONO_LE;
  (*  IMATCH_MP_TAC  REAL_LET_TRANS; *)
  REWRITE_TAC[REAL_LDISTRIB];
  IMATCH_MP_TAC  REAL_LE_ADD2;
  (* 3 LEFT *)
  UND 4 THEN (REWRITE_TAC [d_euclid_point]);
  TYPE_THEN `u = --. (SND p)` ABBREV_TAC ;
  TYPE_THEN `SND p = -- u` SUBAGOAL_TAC;
  UND 12 THEN REAL_ARITH_TAC;
  REWRITE_TAC[REAL_ARITH `x - --. y = x + y`];
  IMATCH_MP_TAC  SQRT_MONO_LE;
  REWRITE_TAC[REAL_LDISTRIB];
  IMATCH_MP_TAC  REAL_LE_ADD2;
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 10 THEN UND 13 THEN UND 11 THEN REAL_ARITH_TAC;
  (* 2 LEFT *)
  UND 4 THEN (REWRITE_TAC [d_euclid_point]);
  TYPE_THEN `u = --. (SND p')` ABBREV_TAC ;
  TYPE_THEN `SND p' = -- u` SUBAGOAL_TAC;
  UND 12 THEN REAL_ARITH_TAC;
  REWRITE_TAC[REAL_ARITH `-- x -  v = -- (v + x)`;REAL_POW_NEG;EVEN2 ];
  IMATCH_MP_TAC  SQRT_MONO_LE;
  REWRITE_TAC[REAL_LDISTRIB];
  IMATCH_MP_TAC  REAL_LE_ADD2;
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 10 THEN UND 13 THEN UND 11 THEN REAL_ARITH_TAC;
  (* 1 LEFT *)
  UND 4 THEN (REWRITE_TAC [d_euclid_point]);
  IMATCH_MP_TAC  SQRT_MONO_LE;
  REWRITE_TAC[REAL_LDISTRIB];
  IMATCH_MP_TAC  REAL_LE_ADD2;
  (* Tue Sep  7 15:40:34 EDT 2004 *)
  ]);;
  (* }}} *)

let h_translate_hom = prove_by_refinement(
  `!r. (homeomorphism (h_translate r) top2 top2)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  bicont_homeomorphism;
  REWRITE_TAC[top2_unions;h_translate_bij;h_translate_cont];
  IMATCH_MP_TAC  cont_domain;
  REWRITE_TAC[top2_unions];
  TYPE_THEN `h_translate (-- r)` EXISTS_TAC;
  REWRITE_TAC[h_translate_inv;h_translate_cont];
  (* Tue Sep  7 15:56:20 EDT 2004 *)

  ]);;
  (* }}} *)

let v_translate_hom = prove_by_refinement(
  `!r. (homeomorphism (v_translate r) top2 top2)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  bicont_homeomorphism;
  REWRITE_TAC[top2_unions;v_translate_bij;v_translate_cont];
  IMATCH_MP_TAC  cont_domain;
  REWRITE_TAC[top2_unions];
  TYPE_THEN `v_translate (-- r)` EXISTS_TAC;
  REWRITE_TAC[v_translate_inv;v_translate_cont];
  (* Tue Sep  7 15:57:06 EDT 2004 *)
  ]);;
  (* }}} *)

let r_scale_hom = prove_by_refinement(
  `!r. (&0 < r) ==> (homeomorphism (r_scale r) top2 top2)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  bicont_homeomorphism;
  ASM_SIMP_TAC [top2_unions;r_scale_bij;r_scale_cont];
  IMATCH_MP_TAC  cont_domain;
  REWRITE_TAC[top2_unions];
  TYPE_THEN `r_scale (&1/r)` EXISTS_TAC;
  TYPE_THEN `&0 < &1/r` SUBAGOAL_TAC;
  ASM_SIMP_TAC [r_scale_inv;r_scale_cont];
  (* Tue Sep  7 16:00:14 EDT 2004 *)

  ]);;
  (* }}} *)

let u_scale_hom = prove_by_refinement(
  `!r. (&0 < r) ==> (homeomorphism (u_scale r) top2 top2)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  bicont_homeomorphism;
  ASM_SIMP_TAC [top2_unions;u_scale_bij;u_scale_cont];
  IMATCH_MP_TAC  cont_domain;
  REWRITE_TAC[top2_unions];
  TYPE_THEN `u_scale (&1/r)` EXISTS_TAC;
  TYPE_THEN `&0 < &1/r` SUBAGOAL_TAC;
  ASM_SIMP_TAC [u_scale_inv;u_scale_cont];
  (* Tue Sep  7 16:01:04 EDT 2004 *)


  ]);;
  (* }}} *)

let h_translate_h = prove_by_refinement(
  `!r. (h_compat (h_translate r))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[h_compat;h_translate;e1;point_scale;mk_line;IMAGE];
  IMATCH_MP_TAC  EQ_EXT;
  EQ_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  TYPE_THEN `x''` UNABBREV_TAC;
  REDUCE_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  REWRITE_TAC[point_inj;PAIR_SPLIT ];
  REAL_ARITH_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  CONV_TAC (dropq_conv "x");
  CONV_TAC (dropq_conv "x''");
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  REWRITE_TAC[point_inj;PAIR_SPLIT ];
  REAL_ARITH_TAC;
  (* Tue Sep  7 16:13:50 EDT 2004 *)

  ]);;
  (* }}} *)

let v_translate_v = prove_by_refinement(
  `!r. (v_compat (v_translate r))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[v_compat;v_translate;e2;point_scale;mk_line;IMAGE];
  IMATCH_MP_TAC  EQ_EXT;
  EQ_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  TYPE_THEN `x''` UNABBREV_TAC;
  REDUCE_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  REWRITE_TAC[point_inj;PAIR_SPLIT ];
  REAL_ARITH_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  CONV_TAC (dropq_conv "x");
  CONV_TAC (dropq_conv "x''");
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  REWRITE_TAC[point_inj;PAIR_SPLIT ];
  REAL_ARITH_TAC;
  (* Tue Sep  7 16:15:33 EDT 2004 *)


  ]);;
  (* }}} *)

let h_translate_v = prove_by_refinement(
  `!r. (v_compat (h_translate r))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[v_compat;h_translate;e1;point_scale;mk_line;IMAGE];
  IMATCH_MP_TAC  EQ_EXT;
  EQ_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  TYPE_THEN `x''` UNABBREV_TAC;
  REDUCE_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  REWRITE_TAC[point_inj;PAIR_SPLIT ];
  REAL_ARITH_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  CONV_TAC (dropq_conv "x");
  CONV_TAC (dropq_conv "x''");
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  REWRITE_TAC[point_inj;PAIR_SPLIT ];
  REAL_ARITH_TAC;
  (* Tue Sep  7 16:17:13 EDT 2004 *)
  ]);;
  (* }}} *)

let v_translate_h = prove_by_refinement(
  `!r. (h_compat (v_translate r))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[h_compat;v_translate;e2;point_scale;mk_line;IMAGE];
  IMATCH_MP_TAC  EQ_EXT;
  EQ_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  TYPE_THEN `x''` UNABBREV_TAC;
  REDUCE_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  REWRITE_TAC[point_inj;PAIR_SPLIT ];
  REAL_ARITH_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  CONV_TAC (dropq_conv "x");
  CONV_TAC (dropq_conv "x''");
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  TYPE_THEN `t` EXISTS_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  REWRITE_TAC[point_inj;PAIR_SPLIT ];
  REAL_ARITH_TAC;
  (* Tue Sep  7 16:18:12 EDT 2004 *)

  ]);;
  (* }}} *)

let lin_solve_x = prove_by_refinement(
  `!a  c. ~(c = &0) ==> (?t. c*t = a)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `a/c` EXISTS_TAC;
  IMATCH_MP_TAC  REAL_DIV_LMUL;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let mk_line_pt = prove_by_refinement(
  `!x. mk_line x x = {x}`,
  (* {{{ proof *)
  [
  REWRITE_TAC[mk_line;trivial_lin_combo];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING];
  ]);;
  (* }}} *)

let h_compat_bij = prove_by_refinement(
  `!f t. (BIJ f (euclid 2) (euclid 2) /\
          (!x. f (point x) 1 = t + SND x) ==>
    h_compat f)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[BIJ;h_compat];
  TYPE_THEN `x = y` ASM_CASES_TAC;
  REWRITE_TAC[mk_line_pt];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[IMAGE;INR IN_SING];
  EQ_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN`point y` EXISTS_TAC;
  (* - *)
  TYPE_THEN `!x. f (point x) = (point ( (f (point x)) 0, t + SND x ))` SUBAGOAL_TAC;
  TYPE_THEN `euclid 2 (f (point x'))` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SURJ]);
  USE 5 (MATCH_MP point_onto);
  REWRITE_TAC[point_inj ;PAIR_SPLIT;];
  TSPEC `x'` 1;
  REWR 1;
  UND 1 THEN REWRITE_TAC[coord01];
  (* A- *)
  UND 5 THEN DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t] THEN (ASSUME_TAC t));
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[IMAGE;SUBSET;];
  TYPE_THEN `x'` UNABBREV_TAC;
  UND 7 THEN REWRITE_TAC[mk_line];
  TYPE_THEN `x''` UNABBREV_TAC;
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  TYPE_THEN `x' = (t' * FST x + (&1 - t') * FST y,t' * SND y + (&1 - t') * SND y)` ABBREV_TAC ;
  TYPE_THEN `SND x' = SND y` SUBAGOAL_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  REAL_ARITH_TAC;
  KILL 8;
  COPY 5;
  TSPEC `x'` 5;
  UND 5 THEN (DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]));
  REWRITE_TAC[point_inj ;PAIR_SPLIT;];
  TH_INTRO_TAC[`f (point x') 0 - f(point y) 0`;`f (point x) 0 - f (point y) 0`] lin_solve_x;
  TYPE_THEN `f (point x) = f (point y)` SUBAGOAL_TAC;
  UND 8 THEN (DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]));
  REWRITE_TAC[point_inj ;PAIR_SPLIT ];
  UND 5 THEN REAL_ARITH_TAC;
  UND 4 THEN REWRITE_TAC[];
  ONCE_REWRITE_TAC[GSYM point_inj];
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `t'` EXISTS_TAC;
  CONJ_TAC;
  UND 5 THEN REAL_ARITH_TAC;
  REAL_ARITH_TAC;
  (* - *)
  REWRITE_TAC[mk_line;SUBSET;IMAGE];
  CONV_TAC (dropq_conv "x''");
  TYPE_THEN `x'` UNABBREV_TAC;
  TYPE_THEN `?u. (euclid_plus (t' *# point (f (point x) 0,t + SND y))  ((&1 - t') *# point (f (point y) 0,t + SND y))) = point (u , t + SND y)` SUBAGOAL_TAC;
  REWRITE_TAC[point_scale;point_add ;point_inj ; PAIR_SPLIT ;];
  CONV_TAC (dropq_conv "u");
  REAL_ARITH_TAC;
  KILL 6;
  (* - *)
  TYPE_THEN `?x'. point(u, t + SND y) = f (point x')` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SURJ]);
  TSPEC `point (u,t + SND y)` 2;
  RULE_ASSUM_TAC (REWRITE_RULE[euclid_point]);
  USE 7 (MATCH_MP point_onto);
  TYPE_THEN `y'` UNABBREV_TAC;
  TYPE_THEN `p` EXISTS_TAC;
  (* - *)
  TH_INTRO_TAC[`FST x' - FST y`;`FST x - FST y`] lin_solve_x;
  UND 4 THEN REWRITE_TAC[PAIR_SPLIT ];
  UND 7 THEN REAL_ARITH_TAC;
  TYPE_THEN `t'` EXISTS_TAC;
  AP_TERM_TAC;
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  TYPE_THEN `x' = FST x',SND x'` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add;point_inj;PAIR_SPLIT;];
  CONJ_TAC;
  UND 7 THEN REAL_ARITH_TAC;
  (* - *)
  TSPEC `x'` 5;
  TYPE_THEN `f (point x')` UNABBREV_TAC;
  USE 5 (REWRITE_RULE[point_inj;PAIR_SPLIT;]);
  UND 5 THEN REAL_ARITH_TAC;
  (* Tue Sep  7 22:08:48 EDT 2004 *)

  ]);;
  (* }}} *)

let r_scale_h = prove_by_refinement(
  `!r. (&0 < r) ==> (h_compat (r_scale r))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  h_compat_bij;
  TYPE_THEN `&0` EXISTS_TAC;
  REDUCE_TAC;
  ASM_SIMP_TAC [r_scale_bij];
  REWRITE_TAC[r_scale];
  COND_CASES_TAC;
  (* Tue Sep  7 22:11:42 EDT 2004 *)

  ]);;
  (* }}} *)

let h_compat_bij2 = prove_by_refinement(
  `!f s. (BIJ f (euclid 2) (euclid 2) /\
          (!x. f (point x) 1 = s(SND x)) /\ (INJ s UNIV UNIV) ==>
    h_compat f)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[BIJ;h_compat];
  TYPE_THEN `x = y` ASM_CASES_TAC;
  REWRITE_TAC[mk_line_pt];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[IMAGE;INR IN_SING];
  EQ_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN`point y` EXISTS_TAC;
  (* - *)
  TYPE_THEN `!x. f (point x) = (point ( (f (point x)) 0, s(SND x) ))` SUBAGOAL_TAC;
  TYPE_THEN `euclid 2 (f (point x'))` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SURJ]);
  USE 6 (MATCH_MP point_onto);
  REWRITE_TAC[point_inj ;PAIR_SPLIT;];
  TSPEC `x'` 2;
  REWR 2;
  UND 2 THEN REWRITE_TAC[coord01];
  (* A- *)
  UND 5 THEN DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t] THEN (ASSUME_TAC t));
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[IMAGE;SUBSET;];
  TYPE_THEN `x'` UNABBREV_TAC;
  UND 8 THEN REWRITE_TAC[mk_line];
  TYPE_THEN `x''` UNABBREV_TAC;
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  TYPE_THEN `x' = (t * FST x + (&1 - t) * FST y,t * SND y + (&1 - t) * SND y)` ABBREV_TAC ;
  TYPE_THEN `SND x' = SND y` SUBAGOAL_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  REAL_ARITH_TAC;
  KILL 9;
  COPY 6;
  TSPEC `x'` 6;
  UND 6 THEN (DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]));
  REWRITE_TAC[point_inj ;PAIR_SPLIT;];
  TH_INTRO_TAC[`f (point x') 0 - f(point y) 0`;`f (point x) 0 - f (point y) 0`] lin_solve_x;
  TYPE_THEN `f (point x) = f (point y)` SUBAGOAL_TAC;
  UND 9 THEN (DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]));
  REWRITE_TAC[point_inj ;PAIR_SPLIT ];
  UND 6 THEN REAL_ARITH_TAC;
  UND 5 THEN REWRITE_TAC[];
  ONCE_REWRITE_TAC[GSYM point_inj];
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `t` EXISTS_TAC;
  CONJ_TAC;
  UND 6 THEN REAL_ARITH_TAC;
  REAL_ARITH_TAC;
  (* - *)
  REWRITE_TAC[mk_line;SUBSET;IMAGE];
  CONV_TAC (dropq_conv "x''");
  TYPE_THEN `x'` UNABBREV_TAC;
  TYPE_THEN `?u. (euclid_plus (t *# point (f (point x) 0,s(SND y)))  ((&1 - t) *# point (f (point y) 0,s(SND y)))) = point (u , s(SND y))` SUBAGOAL_TAC;
  REWRITE_TAC[point_scale;point_add ;point_inj ; PAIR_SPLIT ;];
  CONV_TAC (dropq_conv "u");
  REAL_ARITH_TAC;
  ONCE_ASM_REWRITE_TAC [];
  UND 7 THEN DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]);
  (* - *)
  TYPE_THEN `?x'. point(u, s(SND y)) = f (point x')` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SURJ]);
  TSPEC `point (u,s(SND y))` 3;
  RULE_ASSUM_TAC (REWRITE_RULE[euclid_point]);
  USE 8 (MATCH_MP point_onto);
  TYPE_THEN `y'` UNABBREV_TAC;
  TYPE_THEN `p` EXISTS_TAC;
  (* B- *)
  TH_INTRO_TAC[`FST x' - FST y`;`FST x - FST y`] lin_solve_x;
  UND 5 THEN REWRITE_TAC[PAIR_SPLIT ];
  UND 8 THEN REAL_ARITH_TAC;

  TYPE_THEN `t` EXISTS_TAC;
  AP_TERM_TAC;
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  TYPE_THEN `x' = FST x',SND x'` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add;point_inj;PAIR_SPLIT;];
  CONJ_TAC;
  UND 8 THEN REAL_ARITH_TAC;
  (* - *)
  TSPEC `x'` 6;
  TYPE_THEN `f (point x')` UNABBREV_TAC;
  USE 6 (REWRITE_RULE[point_inj;PAIR_SPLIT;]);
  TYPE_THEN `SND y = SND x'` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 12 THEN REAL_ARITH_TAC;
  (* Wed Sep  8 20:04:34 EDT 2004 *)

  ]);;
  (* }}} *)

let u_scale_h = prove_by_refinement(
  `!r. (&0 < r) ==> (h_compat (u_scale r))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  h_compat_bij2;
  TYPE_THEN `(\ z. if (&0 < z) then (r*z) else z)` EXISTS_TAC;
  ASM_SIMP_TAC[u_scale_bij];
  CONJ_TAC;
  REWRITE_TAC[u_scale];
  TYPE_THEN `&0 < SND x` ASM_CASES_TAC;
  REWRITE_TAC[coord01];
  TYPE_THEN `x = FST x, SND x` SUBAGOAL_TAC;
  REWRITE_TAC[INJ];
  UND 1 THEN COND_CASES_TAC THEN COND_CASES_TAC;
  IMATCH_MP_TAC  REAL_EQ_LMUL_IMP;
  UNIFY_EXISTS_TAC;
  UND 0 THEN REAL_ARITH_TAC;
  TYPE_THEN `y` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  UND 2 THEN REWRITE_TAC[];
  IMATCH_MP_TAC REAL_PROP_POS_MUL2;
  TYPE_THEN `x` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  UND 3 THEN REWRITE_TAC[];
  IMATCH_MP_TAC REAL_PROP_POS_MUL2;
  ]);;
  (* }}} *)

let v_compat_bij2 = prove_by_refinement(
  `!f s. (BIJ f (euclid 2) (euclid 2) /\
          (!x. f (point x) 0 = s(FST  x)) /\ (INJ s UNIV UNIV) ==>
    v_compat f)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[BIJ;v_compat];
  TYPE_THEN `x = y` ASM_CASES_TAC;
  REWRITE_TAC[mk_line_pt];
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[IMAGE;INR IN_SING];
  EQ_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN`point y` EXISTS_TAC;
  (* - *)
  TYPE_THEN `!x. f (point x) = point(s(FST x),  (f (point x)) 1 )` SUBAGOAL_TAC;
  TYPE_THEN `euclid 2 (f (point x'))` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SURJ]);
  USE 6 (MATCH_MP point_onto);
  REWRITE_TAC[point_inj ;PAIR_SPLIT;];
  TSPEC `x'` 2;
  REWR 2;
  UND 2 THEN REWRITE_TAC[coord01];
  (* A- *)
  UND 5 THEN DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t] THEN (ASSUME_TAC t));
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  REWRITE_TAC[IMAGE;SUBSET;];
  TYPE_THEN `x'` UNABBREV_TAC;
  UND 8 THEN REWRITE_TAC[mk_line];
  TYPE_THEN `x''` UNABBREV_TAC;
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add];
  TYPE_THEN `x' = (t * FST y + (&1 - t) * FST y,t * SND x + (&1 - t) * SND y)` ABBREV_TAC ;
  TYPE_THEN `FST  x' = FST  y` SUBAGOAL_TAC;
  TYPE_THEN `x'` UNABBREV_TAC;
  REAL_ARITH_TAC;
  KILL 9;
  COPY 6;
  TSPEC `x'` 6;
  UND 6 THEN (DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]));
  REWRITE_TAC[point_inj ;PAIR_SPLIT;];
  TH_INTRO_TAC[`f (point x') 1 - f(point y) 1`;`f (point x) 1 - f (point y) 1`] lin_solve_x;
  TYPE_THEN `f (point x) = f (point y)` SUBAGOAL_TAC;
  UND 9 THEN (DISCH_THEN (fun t-> ONCE_REWRITE_TAC[t]));
  REWRITE_TAC[point_inj ;PAIR_SPLIT ];
  UND 6 THEN REAL_ARITH_TAC;
  UND 5 THEN REWRITE_TAC[];
  ONCE_REWRITE_TAC[GSYM point_inj];
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `t` EXISTS_TAC;
  CONJ_TAC;
  UND 6 THEN REAL_ARITH_TAC;
  UND 6 THEN REAL_ARITH_TAC;
  (* - *)
  REWRITE_TAC[mk_line;SUBSET;IMAGE];
  CONV_TAC (dropq_conv "x''");
  TYPE_THEN `x'` UNABBREV_TAC;
  TYPE_THEN `?u. (euclid_plus (t *# (f (point x)))  ((&1 - t) *# (f (point y)))) = point ( s(FST  y), u)` SUBAGOAL_TAC;
  ONCE_ASM_REWRITE_TAC[];
  REWRITE_TAC[point_scale;point_add ;point_inj ; PAIR_SPLIT ;];
  CONV_TAC (dropq_conv "u");
    REAL_ARITH_TAC;
  (* - *)
  TYPE_THEN `?x'. point( s(FST  y),u) = f (point x')` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SURJ]);
  TSPEC `point (s(FST  y),u)` 3;
  RULE_ASSUM_TAC (REWRITE_RULE[euclid_point]);
  USE 9 (MATCH_MP point_onto);
  TYPE_THEN `y'` UNABBREV_TAC;
  TYPE_THEN `p` EXISTS_TAC;
  (* B- *)
  TH_INTRO_TAC[`SND  x' - SND  y`;`SND  x - SND  y`] lin_solve_x;
  UND 5 THEN REWRITE_TAC[PAIR_SPLIT ];
  UND 9 THEN REAL_ARITH_TAC;
  TYPE_THEN `t'` EXISTS_TAC;
  AP_TERM_TAC;
  TYPE_THEN `x = FST x,SND x` SUBAGOAL_TAC;
  TYPE_THEN `y = FST y,SND y` SUBAGOAL_TAC;
  TYPE_THEN `x' = FST x',SND x'` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN PURE_REWRITE_TAC[point_scale;point_add;point_inj;PAIR_SPLIT;];
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  CONJ_TAC;
  UND 9 THEN REAL_ARITH_TAC;
  (* - *)
  TSPEC `x'` 6;
  TYPE_THEN `f (point x')` UNABBREV_TAC;
  USE 6 (REWRITE_RULE[point_inj;PAIR_SPLIT;]);
  TYPE_THEN `FST  y = FST  x'` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 13 THEN REAL_ARITH_TAC;
  (* Wed Sep  8 21:10:34 EDT 2004 *)


  ]);;
  (* }}} *)

let r_scale_v = prove_by_refinement(
  `!r. (&0 < r) ==> (v_compat (r_scale r))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  v_compat_bij2;
  TYPE_THEN `(\ z. if (&0 < z) then (r*z) else z)` EXISTS_TAC;
  ASM_SIMP_TAC[r_scale_bij];
  CONJ_TAC;
  REWRITE_TAC[r_scale];
  TYPE_THEN `&0 < FST  x` ASM_CASES_TAC;
  REWRITE_TAC[coord01];
  TYPE_THEN `x = FST x, SND x` SUBAGOAL_TAC;
  REWRITE_TAC[INJ];
  UND 1 THEN COND_CASES_TAC THEN COND_CASES_TAC;
  IMATCH_MP_TAC  REAL_EQ_LMUL_IMP;
  UNIFY_EXISTS_TAC;
  UND 0 THEN REAL_ARITH_TAC;
  TYPE_THEN `y` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  UND 2 THEN REWRITE_TAC[];
  IMATCH_MP_TAC REAL_PROP_POS_MUL2;
  TYPE_THEN `x` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  UND 3 THEN REWRITE_TAC[];
  IMATCH_MP_TAC REAL_PROP_POS_MUL2;
  ]);;
  (* }}} *)

let u_scale_v = prove_by_refinement(
  `!r. (&0 < r) ==> (v_compat (u_scale r))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  v_compat_bij2;
  TYPE_THEN `(\ z.  &0 + z)` EXISTS_TAC;
  ASM_SIMP_TAC[u_scale_bij];
  REDUCE_TAC;
  CONJ_TAC;
  REWRITE_TAC[u_scale];
  COND_CASES_TAC;
  REWRITE_TAC[INJ];
  ]);;
  (* }}} *)


(* ------------------------------------------------------------------ *)
(* SECTION Q *)
(* ------------------------------------------------------------------ *)

let mk_line_hyper2_fst = prove_by_refinement(
  `!x y. (FST x = FST y) ==> (mk_line (point x) (point y) SUBSET
    hyperplane 2 e1 (FST x))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[];
  TYPE_THEN `x = y` ASM_CASES_TAC;
  REWRITE_TAC[mk_line_pt;SUBSET;INR IN_SING ];
  REWRITE_TAC[e1;GSYM line2D_F;SUBSET;mk_line;];
  TYPE_THEN `y` EXISTS_TAC;
  (* - *)
  IMATCH_MP_TAC  (prove_by_refinement( `!A B. (A = B) ==> (A SUBSET (B:A->bool))`,[MESON_TAC[SUBSET_REFL]]));
  REWRITE_TAC[GSYM mk_line_hyper2_e1];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  IMATCH_MP_TAC  mk_line_2;
  REWRITE_TAC[mk_line_hyper2_e1;];
  REWRITE_TAC[e1;GSYM line2D_F;point_inj;PAIR_SPLIT];
  CONJ_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  TYPE_THEN `y` EXISTS_TAC;
  UND 1 THEN ASM_REWRITE_TAC[PAIR_SPLIT];
  (* Thu Sep  9 10:13:23 EDT 2004 *)

  ]);;
  (* }}} *)

let mk_line_hyper2_snd = prove_by_refinement(
  `!x y. (SND x = SND y) ==> (mk_line (point x) (point y) SUBSET
    hyperplane 2 e2 (SND x))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[];
  TYPE_THEN `x = y` ASM_CASES_TAC;
  REWRITE_TAC[mk_line_pt;SUBSET;INR IN_SING ];
  REWRITE_TAC[e2;GSYM line2D_S;SUBSET;mk_line;];
  TYPE_THEN `y` EXISTS_TAC;
  (* - *)
  IMATCH_MP_TAC  (prove_by_refinement( `!A B. (A = B) ==> (A SUBSET (B:A->bool))`,[MESON_TAC[SUBSET_REFL]]));
  REWRITE_TAC[GSYM mk_line_hyper2_e2];
  ONCE_REWRITE_TAC [EQ_SYM_EQ];
  IMATCH_MP_TAC  mk_line_2;
  REWRITE_TAC[mk_line_hyper2_e2;];
  REWRITE_TAC[e2;GSYM line2D_S;point_inj;PAIR_SPLIT];
  CONJ_TAC;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  CONJ_TAC;
  TYPE_THEN `y` EXISTS_TAC;
  UND 1 THEN ASM_REWRITE_TAC[PAIR_SPLIT];
  (* Thu Sep  9 10:16:19 EDT 2004 *)
  ]);;
  (* }}} *)

let hv_line_hyper = prove_by_refinement(
  `!E e. hv_line E /\ E e ==> (?z.
     (e SUBSET hyperplane 2 e1 z) \/ (e SUBSET  hyperplane 2 e2 z))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[hv_line];
  TSPEC `e` 1;
  REP_BASIC_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `FST y` EXISTS_TAC;
  DISJ1_TAC;
  USE 3 SYM;
  IMATCH_MP_TAC  mk_line_hyper2_fst;
  TYPE_THEN `SND x` EXISTS_TAC;
  USE 3 SYM;
  DISJ2_TAC;
  IMATCH_MP_TAC  mk_line_hyper2_snd;
  (* Thu Sep  9 10:20:05 EDT 2004 *)

  ]);;
  (* }}} *)

let hv_line_hyper2 = prove_by_refinement(
  `!E. hv_line E /\ FINITE E ==> (?E'.
   (UNIONS E SUBSET UNIONS E') /\ (FINITE E') /\
   (!e. E' e ==>
     (?z. (e = hyperplane 2 e1 z) \/ (e = hyperplane 2 e2 z))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `!e. ?h. (E e ==> (e SUBSET h /\ (?z. (h = hyperplane 2 e1 z) \/ (h =  hyperplane 2 e2 z))))` SUBAGOAL_TAC;
  RIGHT_TAC "h";
  TH_INTRO_TAC[`E`;`e`] hv_line_hyper;
  FIRST_ASSUM DISJ_CASES_TAC;
  UNIFY_EXISTS_TAC;
  TYPE_THEN `z` EXISTS_TAC;
  UNIFY_EXISTS_TAC;
  TYPE_THEN `z` EXISTS_TAC;
  LEFT 2 "h";
  TYPE_THEN `IMAGE h E` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[UNIONS;SUBSET;IMAGE];
  CONV_TAC (dropq_conv "u");
  NAME_CONFLICT_TAC;
  TYPE_THEN `u` EXISTS_TAC;
  ASM_MESON_TAC[ISUBSET];
  (* - *)
  CONJ_TAC;
  IMATCH_MP_TAC  FINITE_IMAGE;
  RULE_ASSUM_TAC (REWRITE_RULE[IMAGE]);
  ASM_MESON_TAC[];
  (* Thu Sep  9 10:32:28 EDT 2004 *)

  ]);;
  (* }}} *)

let finite_graph_edge = prove_by_refinement(
  `!(G:(A,B)graph_t) (H:(A',B')graph_t). FINITE(graph_edge G) /\
    graph_isomorphic G H ==> FINITE (graph_edge H)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_isomorphic;graph_iso];
  ASM_MESON_TAC[FINITE_BIJ];
  ]);;
  (* }}} *)

let finite_graph_vertex = prove_by_refinement(
  `!(G:(A,B)graph_t) (H:(A',B')graph_t). FINITE(graph_vertex G) /\
    graph_isomorphic G H ==> FINITE (graph_vertex H)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_isomorphic;graph_iso];
  ASM_MESON_TAC[FINITE_BIJ];
  ]);;
  (* }}} *)

let graph_edge_nonempty = prove_by_refinement(
  `!(G:(A,B)graph_t) (H:(A',B')graph_t). ~(graph_edge G = EMPTY ) /\
    graph_isomorphic G H ==> ~(graph_edge H  = EMPTY )`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_isomorphic;graph_iso];
  USE 5 (REWRITE_RULE[EMPTY_EXISTS]);
  UND 0 THEN (REWRITE_TAC [EMPTY_EXISTS]);
  TYPE_THEN `v u'` EXISTS_TAC ;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  ]);;
  (* }}} *)

let graph_edge_around_finite = prove_by_refinement(
  `!(G:(A,B)graph_t) v.
        (FINITE (graph_edge G)) ==> (FINITE (graph_edge_around G v))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[graph_edge_around];
  IMATCH_MP_TAC  FINITE_SUBSET;
  UNIFY_EXISTS_TAC;
  REWRITE_TAC[SUBSET];
  ]);;
  (* }}} *)

let graph_edge_around4 = prove_by_refinement(
  `!(G:(A,B)graph_t) (H:(A',B')graph_t). (graph G) /\
        (FINITE (graph_edge G)) /\
        (!v. CARD (graph_edge_around G v) <=| 4)  /\
    graph_isomorphic G H ==> (!v. CARD (graph_edge_around H v) <=| 4)`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  TYPE_THEN `graph_vertex H v` ASM_CASES_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE [graph_isomorphic]);
  TYPE_THEN `?v'. (graph_vertex G v' /\ ((FST f) v' = v))` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[PAIR_SPLIT ;graph_iso]);
  USE 6 (REWRITE_RULE[BIJ;SURJ]);
  TYPE_THEN `v` UNABBREV_TAC;
  TH_INTRO_TAC[`G`;`H`;`f`;`v'`] graph_iso_around;
  TH_INTRO_TAC[`SND f`; `(graph_edge_around G v')`] CARD_IMAGE_LE;
  IMATCH_MP_TAC  graph_edge_around_finite;
  IMATCH_MP_TAC  LE_TRANS;
  UNIFY_EXISTS_TAC;
  ASM_MESON_TAC [ARITH_RULE `0 <=| 4`; CARD_CLAUSES;graph_isomorphic_graph;graph_edge_around_empty];
  (* Thu Sep  9 11:49:01 EDT 2004 *)

  ]);;

  (* }}} *)

let graph_near_support = prove_by_refinement(
  `!(G:(A,B)graph_t). (planar_graph G) /\
         FINITE (graph_edge G) /\
         FINITE (graph_vertex G) /\
         ~(graph_edge G = {}) /\
         (!v. CARD (graph_edge_around G v) <=| 4)
         ==> (?H E. graph_isomorphic G H /\
           (FINITE E) /\ (good_plane_graph H) /\
        (!e. (graph_edge H e ==> e SUBSET UNIONS E)) /\
        (!v. (graph_vertex H v ==>
         E (hyperplane 2 e1 (v 0)) /\ E (hyperplane 2 e2 (v 1)))) /\
         (!e. (E e ==>
            (?z. (e = hyperplane 2 e1 z) \/ (e = hyperplane 2 e2 z)))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TH_INTRO_TAC[`G`] planar_graph_hv;
  TYPE_THEN `H` EXISTS_TAC;
  TYPE_THEN `A = IMAGE (\ v. hyperplane 2 e1 (v 0)) (graph_vertex H)` ABBREV_TAC ;
  TYPE_THEN `B = IMAGE (\ v. hyperplane 2 e2 (v 1)) (graph_vertex H)` ABBREV_TAC ;
  RULE_ASSUM_TAC (REWRITE_RULE[hv_finite]);
  LEFT 5 "E";
  LEFT 5 "E";
  TYPE_THEN `?E'. !e. (graph_edge H e ==> (e SUBSET UNIONS (E' e)) /\ (FINITE (E' e)) /\ (!e'. E' e e' ==> (?z. (e' = hyperplane 2 e1 z) \/ (e' = hyperplane 2 e2 z))))` SUBAGOAL_TAC;
  LEFT_TAC "e";
  RIGHT_TAC "E'";
  TSPEC `e` 5;
  TH_INTRO_TAC[`E e`] hv_line_hyper2;
  TYPE_THEN `E'` EXISTS_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  UNIFY_EXISTS_TAC;
  (* - *)
  TYPE_THEN `C = UNIONS (IMAGE E' (graph_edge H))` ABBREV_TAC ;
  TYPE_THEN `A UNION B UNION C` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[FINITE_UNION];
  CONJ_TAC;
  TYPE_THEN `A` UNABBREV_TAC;
  IMATCH_MP_TAC  FINITE_IMAGE;
  IMATCH_MP_TAC  finite_graph_vertex;
  UNIFY_EXISTS_TAC;
  CONJ_TAC;
  TYPE_THEN `B` UNABBREV_TAC;
  IMATCH_MP_TAC  FINITE_IMAGE;
  IMATCH_MP_TAC  finite_graph_vertex;
  UNIFY_EXISTS_TAC;
  TYPE_THEN `C` UNABBREV_TAC;
  TH_INTRO_TAC[`IMAGE E' (graph_edge H)`] FINITE_UNIONS;
  IMATCH_MP_TAC  FINITE_IMAGE;
  IMATCH_MP_TAC  finite_graph_edge;
  UNIFY_EXISTS_TAC;
  USE 11 (REWRITE_RULE[IMAGE]);
  ASM_MESON_TAC[];
  (* - *)
  CONJ_TAC;
  REWRITE_TAC[UNIONS_UNION];
  IMATCH_MP_TAC  in_union;
  DISJ2_TAC;
  IMATCH_MP_TAC  in_union;
  DISJ2_TAC;
  TYPE_THEN `C` UNABBREV_TAC;
  TSPEC `e` 10;
  REP_BASIC_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  UNIFY_EXISTS_TAC;
  IMATCH_MP_TAC  UNIONS_UNIONS;
  REWRITE_TAC[SUBSET;UNIONS;IMAGE;];
  CONV_TAC (dropq_conv "u");
  UNIFY_EXISTS_TAC;
  (* - *)
  CONJ_TAC;
  REWRITE_TAC[UNION];
  TYPE_THEN  `A` UNABBREV_TAC;
  TYPE_THEN `B` UNABBREV_TAC;
  REWRITE_TAC[IMAGE];
  CONJ_TAC;
  DISJ1_TAC;
  UNIFY_EXISTS_TAC;
  ASM_REWRITE_TAC[];
  DISJ2_TAC;
  DISJ1_TAC;
  UNIFY_EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* - *)
  USE 12 (REWRITE_RULE[UNION]);
  UND 12 THEN REP_CASES_TAC;
  TYPE_THEN `A` UNABBREV_TAC;
  USE 12 (REWRITE_RULE[IMAGE]);
  MESON_TAC[];
  TYPE_THEN `B` UNABBREV_TAC;
  USE 12 (REWRITE_RULE[IMAGE]);
  MESON_TAC[];
  TYPE_THEN `C` UNABBREV_TAC;
  USE 12 (REWRITE_RULE[IMAGE;UNIONS]);
  TYPE_THEN `u` UNABBREV_TAC;
  TSPEC `x` 10;
  (* Thu Sep  9 12:12:51 EDT 2004 *)

  ]);;
  (* }}} *)

let h_translate_point = prove_by_refinement(
  `!u v r. (h_translate r (point (u,v)) = point (u+r,v))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[h_translate;e1;point_scale;point_add];
  REDUCE_TAC;
  ]);;
  (* }}} *)

let v_translate_point = prove_by_refinement(
  `!u v r. (v_translate r (point (u,v)) = point (u,v + r))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[v_translate;e2;point_scale;point_add];
  REDUCE_TAC;
  ]);;
  (* }}} *)

let hyperplane1_h_translate = prove_by_refinement(
  `!z r. (IMAGE (h_translate r) (hyperplane 2 e1 z) =
            (hyperplane 2 e1 (z + r)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM mk_line_hyper2_e1];
  ASSUME_TAC v_compat;
  TSPEC `(h_translate r)` 0;
  RULE_ASSUM_TAC (REWRITE_RULE[h_translate_v]);
  UND 0 THEN (DISCH_THEN (TH_INTRO_TAC[`z, &0`;`z, &1`]));
  REWRITE_TAC[h_translate_point];
  ]);;
  (* }}} *)

let hyperplane2_h_translate = prove_by_refinement(
  `!z r. (IMAGE (h_translate r) (hyperplane 2 e2 z) =
            (hyperplane 2 e2 z))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM mk_line_hyper2_e2];
  ASSUME_TAC h_compat;
  TSPEC `(h_translate r)` 0;
  RULE_ASSUM_TAC (REWRITE_RULE[h_translate_h]);
  UND 0 THEN (DISCH_THEN (TH_INTRO_TAC[` &0,z`;` &1,z`]));
  REWRITE_TAC[h_translate_point];
  ONCE_REWRITE_TAC[EQ_SYM_EQ];
  IMATCH_MP_TAC  mk_line_2;
  REWRITE_TAC[mk_line_hyper2_e2;];
  REWRITE_TAC[GSYM line2D_S;e2;point_inj ];
  CONJ_TAC;
  CONV_TAC (dropq_conv "p");
  CONJ_TAC;
  CONV_TAC (dropq_conv "p");
   RULE_ASSUM_TAC (REWRITE_RULE[PAIR_SPLIT]);
  UND 1 THEN REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let hyperplane2_v_translate = prove_by_refinement(
  `!z r. (IMAGE (v_translate r) (hyperplane 2 e2 z) =
            (hyperplane 2 e2 (z + r)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM mk_line_hyper2_e2];
  ASSUME_TAC h_compat;
  TSPEC `(v_translate r)` 0;
  RULE_ASSUM_TAC (REWRITE_RULE[v_translate_h]);
  UND 0 THEN (DISCH_THEN (TH_INTRO_TAC[`&0,z`;`&1,z`]));
  REWRITE_TAC[v_translate_point];
  ]);;
  (* }}} *)

let hyperplane1_v_translate = prove_by_refinement(
  `!z r. (IMAGE (v_translate r) (hyperplane 2 e1 z) =
            (hyperplane 2 e1 z))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM mk_line_hyper2_e1];
  ASSUME_TAC v_compat;
  TSPEC `(v_translate r)` 0;
  RULE_ASSUM_TAC (REWRITE_RULE[v_translate_v]);
  UND 0 THEN (DISCH_THEN (TH_INTRO_TAC[` z,&0`;`z,&1`]));
  REWRITE_TAC[v_translate_point];
  ONCE_REWRITE_TAC[EQ_SYM_EQ];
  IMATCH_MP_TAC  mk_line_2;
  REWRITE_TAC[mk_line_hyper2_e1;];
  REWRITE_TAC[GSYM line2D_F;e1;point_inj ];
  CONJ_TAC;
  CONV_TAC (dropq_conv "p");
  CONJ_TAC;
  CONV_TAC (dropq_conv "p");
   RULE_ASSUM_TAC (REWRITE_RULE[PAIR_SPLIT]);
  UND 1 THEN REAL_ARITH_TAC;
  (* Thu Sep  9 13:43:45 EDT 2004 *)

  ]);;
  (* }}} *)

let r_scale_point = prove_by_refinement(
  `!r u v. (r_scale r (point (u,v))) =
  point ((if (&0 < u) then r*u else u),v)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[r_scale];
  TYPE_THEN `&0  < u` ASM_CASES_TAC;
  ]);;
  (* }}} *)

let u_scale_point = prove_by_refinement(
  `!r u v. (u_scale r (point (u,v))) =
  point (u,(if (&0 < v) then r*v else v))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[u_scale];
  TYPE_THEN `&0  < v` ASM_CASES_TAC;
  ]);;
  (* }}} *)

let hyperplane2_r_scale = prove_by_refinement(
  `!z r. (&0 < r) ==> (IMAGE (r_scale r) (hyperplane 2 e2 z) =
             (hyperplane 2 e2 z))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM mk_line_hyper2_e2];
  ASSUME_TAC h_compat;
  TSPEC `(r_scale r)` 1;
  TYPE_THEN `h_compat(r_scale r)` SUBAGOAL_TAC THENL [ASM_MESON_TAC[r_scale_h];ALL_TAC];
  REWR 1;
  UND 1 THEN (DISCH_THEN (TH_INTRO_TAC[` &0,z`;`&1,z`]));
  REWRITE_TAC[r_scale_point];
  ONCE_REWRITE_TAC[EQ_SYM_EQ];
  IMATCH_MP_TAC  mk_line_2;
  REWRITE_TAC[REAL_ARITH `~(&0 < &0)`];
  REWRITE_TAC[mk_line_hyper2_e2;];
  REWRITE_TAC[GSYM line2D_S;e2;point_inj ];
  CONJ_TAC;
  CONV_TAC (dropq_conv "p");
  CONJ_TAC;
  CONV_TAC (dropq_conv "p");
  RULE_ASSUM_TAC (REWRITE_RULE[PAIR_SPLIT;REAL_ARITH `r * &1 = r`]);
  UND 3 THEN UND 0 THEN REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let hyperplane1_r_scale = prove_by_refinement(
  `!z r. (&0 < r) ==> (IMAGE (r_scale r) (hyperplane 2 e1 z) =
             (hyperplane 2 e1 (if &0 < z then r*z else z)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM mk_line_hyper2_e1];
  ASSUME_TAC v_compat;
  TSPEC `(r_scale r)` 1;
  TYPE_THEN `v_compat(r_scale r)` SUBAGOAL_TAC THENL [ASM_MESON_TAC[r_scale_v];ALL_TAC];
  REWR 1;
  UND 1 THEN (DISCH_THEN (TH_INTRO_TAC[`z,&0`;`z,&1`]));
  REWRITE_TAC[r_scale_point];
  ]);;
  (* }}} *)

let hyperplane1_u_scale = prove_by_refinement(
  `!z r. (&0 < r) ==> (IMAGE (u_scale r) (hyperplane 2 e1 z) =
             (hyperplane 2 e1 z))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM mk_line_hyper2_e1];
  ASSUME_TAC v_compat;
  TSPEC `(u_scale r)` 1;
  TYPE_THEN `v_compat(u_scale r)` SUBAGOAL_TAC THENL [ASM_MESON_TAC[u_scale_v];ALL_TAC];
  REWR 1;
  UND 1 THEN (DISCH_THEN (TH_INTRO_TAC[` z,&0`;`z,&1`]));
  REWRITE_TAC[u_scale_point];
  ONCE_REWRITE_TAC[EQ_SYM_EQ];
  IMATCH_MP_TAC  mk_line_2;
  REWRITE_TAC[REAL_ARITH `~(&0 < &0)`];
  REWRITE_TAC[mk_line_hyper2_e1;];
  REWRITE_TAC[GSYM line2D_F;e1;point_inj ];
  CONJ_TAC;
  CONV_TAC (dropq_conv "p");
  CONJ_TAC;
  CONV_TAC (dropq_conv "p");
  RULE_ASSUM_TAC (REWRITE_RULE[PAIR_SPLIT;REAL_ARITH `r * &1 = r`]);
  UND 3 THEN UND 0 THEN REAL_ARITH_TAC;
  ]);;
  (* }}} *)

let hyperplane2_u_scale = prove_by_refinement(
  `!z r. (&0 < r) ==> (IMAGE (u_scale r) (hyperplane 2 e2 z) =
             (hyperplane 2 e2 (if &0 < z then r*z else z)))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[GSYM mk_line_hyper2_e2];
  ASSUME_TAC h_compat;
  TSPEC `(u_scale r)` 1;
  TYPE_THEN `h_compat(u_scale r)` SUBAGOAL_TAC THENL [ASM_MESON_TAC[u_scale_h];ALL_TAC];
  REWR 1;
  UND 1 THEN (DISCH_THEN (TH_INTRO_TAC[`&0,z`;`&1,z`]));
  REWRITE_TAC[u_scale_point];
  (* Thu Sep  9 14:04:58 EDT 2004 *)

  ]);;
  (* }}} *)

let homeomorphism_compose = prove_by_refinement(
  `!U V W (f:A->B) (g:B->C). homeomorphism f U V /\ homeomorphism g V W
   ==>
   homeomorphism (g o f) U W`,
  (* {{{ proof *)
  [
  REWRITE_TAC[homeomorphism];
  SUBCONJ_TAC;
  REWRITE_TAC[comp_comp];
  IMATCH_MP_TAC  COMP_BIJ;
  UNIFY_EXISTS_TAC;
  (* - *)
  CONJ_TAC;
  IMATCH_MP_TAC  continuous_comp;
  UNIFY_EXISTS_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ;SURJ]);
  REWRITE_TAC[IMAGE;SUBSET];
  FIRST_ASSUM IMATCH_MP_TAC ;
  (* - *)
  REWRITE_TAC[IMAGE_o];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ]);;
  (* }}} *)

let hyperplane1_inj = prove_by_refinement(
  `!z w. (hyperplane 2 e1 z = hyperplane 2 e1 w) ==> (z = w)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[e1; GSYM line2D_F];
  USE 0 (ONCE_REWRITE_RULE[FUN_EQ_THM]);
  USE 0 (REWRITE_RULE[]);
  TSPEC `point(z,&0)` 0;
  RULE_ASSUM_TAC (REWRITE_RULE[point_inj]);
  USE 0 SYM;
  TYPE_THEN `(?p. (z,&0 = p) /\ (FST p = z))` SUBAGOAL_TAC;
  CONV_TAC (dropq_conv "p");
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let hyperplane2_inj = prove_by_refinement(
  `!z w. (hyperplane 2 e2 z = hyperplane 2 e2 w) ==> (z = w)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[e2; GSYM line2D_S];
  USE 0 (ONCE_REWRITE_RULE[FUN_EQ_THM]);
  USE 0 (REWRITE_RULE[]);
  TSPEC `point(z,z)` 0;
  RULE_ASSUM_TAC (REWRITE_RULE[point_inj]);
  USE 0 SYM;
  TYPE_THEN `(?p. (z,z = p) /\ (SND p = z))` SUBAGOAL_TAC;
  CONV_TAC (dropq_conv "p");
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let graph_support_init = prove_by_refinement(
  `!(G:(A,B)graph_t). (planar_graph G) /\
         FINITE (graph_edge G) /\
         FINITE (graph_vertex G) /\
         ~(graph_edge G = {}) /\
         (!v. CARD (graph_edge_around G v) <=| 4)
         ==> (?H E. graph_isomorphic G H /\
           (FINITE E) /\ (good_plane_graph H) /\
        (!e. (graph_edge H e ==> e SUBSET UNIONS E)) /\
        (!v. (graph_vertex H v ==>
         E (hyperplane 2 e1 (v 0)) /\ E (hyperplane 2 e2 (v 1)))) /\
         (!e. (E e ==>
            (?z. (&0 < z) /\
               ((e = hyperplane 2 e1 z) \/ (e = hyperplane 2 e2 z))))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TH_INTRO_TAC[`G`] graph_near_support;
  TYPE_THEN `EH = E INTER { h | ?z. (h = hyperplane 2 e1 z) }` ABBREV_TAC ;
  TYPE_THEN `EV = E INTER {h | ?z. (h = hyperplane 2 e2 z) }` ABBREV_TAC ;
  TYPE_THEN `E = EH UNION EV` SUBAGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  CONJ_TAC;
  TYPE_THEN `EH` UNABBREV_TAC;
  TYPE_THEN `EV` UNABBREV_TAC;
  REWRITE_TAC[SUBSET;INTER;UNION];
  ASM_MESON_TAC[];
  REWRITE_TAC[UNION;SUBSET];
  TYPE_THEN `EH` UNABBREV_TAC;
  TYPE_THEN `EV` UNABBREV_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INTER;GSYM LEFT_AND_OVER_OR]);
  (* - *)
  TYPE_THEN `FINITE EH /\ FINITE EV` SUBAGOAL_TAC;
  USE 13 SYM;
  USE 13 (MATCH_MP union_imp_subset);
  ASM_MESON_TAC[FINITE_SUBSET];
(*** Modified by JRH for new theorem name
  TH_INTRO_TAC[`(\ z. (hyperplane 2 e1 z))`;`UNIV:real->bool`;`EH`] FINITE_SUBSET_IMAGE;
 ***)
  TH_INTRO_TAC[`(\ z. (hyperplane 2 e1 z))`;`UNIV:real->bool`;`EH`] FINITE_SUBSET_IMAGE_IMP;
  TYPE_THEN `EH` UNABBREV_TAC;
  REWRITE_TAC[INTER;SUBSET;IMAGE;UNIV];
(*** Modified by JRH for new theorem name
  TH_INTRO_TAC[`(\ z. (hyperplane 2 e2 z))`;`UNIV:real->bool`;`EV`] FINITE_SUBSET_IMAGE;
 ***)
  TH_INTRO_TAC[`(\ z. (hyperplane 2 e2 z))`;`UNIV:real->bool`;`EV`] FINITE_SUBSET_IMAGE_IMP;
  TYPE_THEN `EV` UNABBREV_TAC;
  REWRITE_TAC[INTER;SUBSET;IMAGE;UNIV];
  (* - *)
  WITH 21 (MATCH_MP finite_LB);
  WITH 18 (MATCH_MP finite_LB);
  TYPE_THEN `f = (h_translate (&1 - t')) o (v_translate (&1 - t))` ABBREV_TAC ;
  TYPE_THEN `plane_graph_image f H` EXISTS_TAC;
  TYPE_THEN `IMAGE2 f E` EXISTS_TAC;
  (* A- *)
  TYPE_THEN `homeomorphism f top2 top2` SUBAGOAL_TAC;
  TYPE_THEN `f` UNABBREV_TAC;
  IMATCH_MP_TAC  homeomorphism_compose;
  TYPE_THEN `top2` EXISTS_TAC;
  REWRITE_TAC[v_translate_hom;h_translate_hom];
  (* - *)
  TYPE_THEN `graph_isomorphic H (plane_graph_image f H)` SUBAGOAL_TAC;
  IMATCH_MP_TAC  plane_graph_image_iso;
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph]);
  (* - *)
  CONJ_TAC;
  TH_INTRO_TAC[`G`;`H`;`plane_graph_image f H`] graph_isomorphic_trans;
  (* - *)
  CONJ_TAC;
  REWRITE_TAC[IMAGE2];
  IMATCH_MP_TAC  FINITE_IMAGE;
  ASM_REWRITE_TAC[FINITE_UNION];
  (* - *)
  CONJ_TAC;
  IMATCH_MP_TAC  plane_graph_image_plane;
  (* B- *)
  TYPE_THEN `!z. IMAGE  f (hyperplane 2 e1 z) = hyperplane 2 e1 (z - t' + &1)` SUBAGOAL_TAC;
  TYPE_THEN `f` UNABBREV_TAC;
  REWRITE_TAC[IMAGE_o;hyperplane1_v_translate;hyperplane1_h_translate];
  AP_TERM_TAC;
  REAL_ARITH_TAC;
  TYPE_THEN `!z. IMAGE f (hyperplane 2 e2 z) = hyperplane 2 e2 (z - t + &1)` SUBAGOAL_TAC;
  TYPE_THEN `f` UNABBREV_TAC;
  REWRITE_TAC[IMAGE_o;hyperplane2_v_translate;hyperplane2_h_translate];
  AP_TERM_TAC;
  REAL_ARITH_TAC;
  REWRITE_TAC[IMAGE2;GSYM image_unions;];
  REWRITE_TAC[plane_graph_image_e;plane_graph_image_v;IMAGE2];
  (* - *)
  CONJ_TAC;
  TYPE_THEN `g = IMAGE f` ABBREV_TAC ;
  USE 29 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `g` UNABBREV_TAC;
  IMATCH_MP_TAC  IMAGE_SUBSET;
  USE 13 GSYM;
  FIRST_ASSUM IMATCH_MP_TAC ;
  (* C- *)
  USE 13 GSYM;
  CONJ_TAC;
  USE 29 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `euclid 2 x` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[good_plane_graph;plane_graph;SUBSET]);
  USE 31 (MATCH_MP point_onto);
  TYPE_THEN `x` UNABBREV_TAC;
  TYPE_THEN `v` UNABBREV_TAC;
  TYPE_THEN `f (point p) = point(FST p - t' + &1 , SND p  - t + &1)` SUBAGOAL_TAC;
  TYPE_THEN `f` UNABBREV_TAC;
  TYPE_THEN `p = FST p,SND p` SUBAGOAL_TAC;
  PURE_ONCE_ASM_REWRITE_TAC[] THEN  PURE_REWRITE_TAC[h_translate_point;v_translate_point;o_DEF ;];
  PURE_ONCE_ASM_REWRITE_TAC[] THEN  PURE_REWRITE_TAC[h_translate_point;v_translate_point;o_DEF ;];
  REWRITE_TAC[point_inj ;PAIR_SPLIT];
  REAL_ARITH_TAC;
  USE 28 GSYM ;
  USE 27 GSYM;
  TSPEC `point p` 6;
  CONJ_TAC;
  IMATCH_MP_TAC  image_imp;
  RULE_ASSUM_TAC (REWRITE_RULE[coord01]);
  IMATCH_MP_TAC  image_imp;
  RULE_ASSUM_TAC (REWRITE_RULE[coord01]);
  (* D- *)
  TYPE_THEN `g = IMAGE f` ABBREV_TAC ;
  USE 29 (REWRITE_RULE[IMAGE]);
  TYPE_THEN `EH x \/ EV x` SUBAGOAL_TAC;
  TYPE_THEN `E` UNABBREV_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[UNION]);
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `EH` UNABBREV_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INTER]);
  ASM_REWRITE_TAC[];
  TYPE_THEN `z - t' + &1` EXISTS_TAC;
  TYPE_THEN `s' z` SUBAGOAL_TAC;
  USE 16 (REWRITE_RULE[SUBSET;IMAGE]);
  TSPEC `x` 16;
  REWR 16;
  LEFT 16 "z'";
  TSPEC `z` 16;
  REWR 16;
  TYPE_THEN `z = x'` SUBAGOAL_TAC;
  IMATCH_MP_TAC  hyperplane1_inj;
  ASM_REWRITE_TAC[];
  TSPEC `z` 23;
  UND 23 THEN REAL_ARITH_TAC;
  TYPE_THEN `EV` UNABBREV_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INTER]);
  ASM_REWRITE_TAC[];
  TYPE_THEN `z - t + &1` EXISTS_TAC;
  TYPE_THEN `s'' z` SUBAGOAL_TAC;
  USE 19 (REWRITE_RULE[SUBSET;IMAGE]);
  TSPEC `x` 19;
  REWR 19;
  LEFT 19 "z'";
  TSPEC `z` 19;
  REWR 19;
  TYPE_THEN `z = x'` SUBAGOAL_TAC;
  IMATCH_MP_TAC  hyperplane2_inj;
  ASM_REWRITE_TAC[];
  TSPEC `z` 22;
  UND 22 THEN REAL_ARITH_TAC;
  (* Thu Sep  9 17:00:37 EDT 2004 *)

  ]);;
  (* }}} *)

let hyperplane_ne = prove_by_refinement(
  `!z z'. ~(hyperplane 2 e1 z = hyperplane 2 e2 z')`,
  (* {{{ proof *)
  [
  REWRITE_TAC[e1;e2;GSYM line2D_S;GSYM line2D_F];
  RULE_ASSUM_TAC (ONCE_REWRITE_RULE[FUN_EQ_THM]);
  TSPEC `point(z, z'+ &1)` 0;
  REWR 0;
  RULE_ASSUM_TAC (REWRITE_RULE[PAIR_SPLIT;point_inj]);
  USE 0 SYM;
  TYPE_THEN `(?p. ((z = FST p) /\ (z' + &1 = SND p)) /\ (FST p = z))` SUBAGOAL_TAC;
  TYPE_THEN `(z,z' + &1)` EXISTS_TAC;
  ASSUME_TAC (REAL_ARITH `~(z' + &1 = z')`);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)


(* ------------------------------------------------------------------ *)
(* SECTION R *)
(* ------------------------------------------------------------------ *)


extend_simp_rewrites[UNION_EMPTY ];;

let inductive_set_restrict = prove_by_refinement(
  `!G A S. inductive_set G S /\
     ~(S INTER A = EMPTY) /\
     segment A /\ A SUBSET G ==> inductive_set A (S INTER A)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[inductive_set];
  CONJ_TAC;
  REWRITE_TAC[INTER;SUBSET];
  REWRITE_TAC[INTER];
  FIRST_ASSUM IMATCH_MP_TAC ;
  RULE_ASSUM_TAC (REWRITE_RULE[INTER]);
  UNIFY_EXISTS_TAC;
  ASM_MESON_TAC[ISUBSET];
  ]);;
  (* }}} *)

let inductive_set_adj = prove_by_refinement(
  `!A B S m. inductive_set (A UNION B) S /\ (endpoint B m) /\
   (FINITE A) /\ (FINITE B) /\
   (endpoint A m) /\ (A SUBSET S) ==> (~(S INTER B = EMPTY)) `,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `?e. A e /\ closure top2 e (pointI m)` SUBAGOAL_TAC;
  TYPE_THEN `terminal_edge A m` EXISTS_TAC;
  IMATCH_MP_TAC  terminal_endpoint;
  TYPE_THEN `?e'. B e' /\ closure top2 e' (pointI m)` SUBAGOAL_TAC;
  TYPE_THEN `terminal_edge B m` EXISTS_TAC;
  IMATCH_MP_TAC  terminal_endpoint;
  RULE_ASSUM_TAC (REWRITE_RULE[inductive_set]);
  TSPEC `e` 6;
  TSPEC `e'` 6;
  (* - *)
  TYPE_THEN `e = e'` ASM_CASES_TAC;
  TYPE_THEN `e'` UNABBREV_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[SUBSET ;EQ_EMPTY;INTER; ]);
  ASM_MESON_TAC[];
  (* - *)
  TYPE_THEN `S e /\ (A UNION B) e' /\ adj e e'` SUBAGOAL_TAC;
  CONJ_TAC;
  ASM_MESON_TAC[ISUBSET];
  CONJ_TAC;
  REWRITE_TAC[UNION];
  REWRITE_TAC[adj];
  REWRITE_TAC[EMPTY_EXISTS;INTER;];
  UNIFY_EXISTS_TAC;
  REWR 6;
  RULE_ASSUM_TAC (REWRITE_RULE[EQ_EMPTY ;INTER]);
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let inductive_set_join = prove_by_refinement(
  `!A B S . ~(S INTER A = EMPTY) /\ (segment B) /\ (segment A) /\
      (?m. endpoint A m /\ endpoint B m) /\
      (inductive_set (A UNION B) S)  ==>
    (S = (A UNION B))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TH_INTRO_TAC[`A UNION B`;`A`;`S`] inductive_set_restrict;
  REWRITE_TAC[SUBSET;UNION];
  (* - *)
  TYPE_THEN `(S INTER A) = A` SUBAGOAL_TAC;
  USE 6 (REWRITE_RULE[inductive_set]);
  USE 3 (REWRITE_RULE[segment]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `A SUBSET S` SUBAGOAL_TAC;
  UND 7 THEN DISCH_THEN (fun t-> ONCE_REWRITE_TAC[GSYM t]);
  REWRITE_TAC[INTER;SUBSET];
  (* - *)
  TH_INTRO_TAC [`A`;`B`;`S`;`m`] inductive_set_adj;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  (* - *)
  TH_INTRO_TAC[`A UNION B`;`B`;`S`] inductive_set_restrict;
  REWRITE_TAC[SUBSET;UNION];
  TYPE_THEN `(S INTER B) = B` SUBAGOAL_TAC;
  USE 10 (REWRITE_RULE[inductive_set]);
  USE 4 (REWRITE_RULE[segment]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];
  TYPE_THEN `B SUBSET S` SUBAGOAL_TAC;
  UND 11 THEN DISCH_THEN (fun t-> ONCE_REWRITE_TAC[GSYM t]);
  REWRITE_TAC[INTER;SUBSET];
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  USE 0 (REWRITE_RULE[inductive_set]);
  REWRITE_TAC[union_subset];
  ]);;
  (* }}} *)

let segment_union = prove_by_refinement(
  `!A B m. segment A /\ segment B /\
     endpoint A m /\ endpoint B m /\
     (A INTER B = EMPTY) /\
  (!n. (0 < num_closure A (pointI n)) /\
          (0 < num_closure B (pointI n)) ==> (n = m) )
    ==>
    segment (A UNION B)` ,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  (* - *)
  TYPE_THEN `FINITE A /\ FINITE B` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  (* - *)
  REWRITE_TAC[segment];
  ASM_REWRITE_TAC[FINITE_UNION];
  (* - *)
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  RULE_ASSUM_TAC (REWRITE_RULE[EMPTY_EXISTS]);
  UND 8 THEN REWRITE_TAC[EMPTY_EXISTS;UNION];
  TYPE_THEN `u` EXISTS_TAC;
  (* - *)
  CONJ_TAC;
  REWRITE_TAC[union_subset];
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  (* - *)
  TYPE_THEN `!m'. { C | (A UNION B) C /\ closure top2 C (pointI m')} = {C | A C /\ closure top2 C (pointI m')} UNION {C | B C /\ closure top2 C (pointI m')}` SUBAGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION];
  TYPE_THEN `A x` ASM_CASES_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[EQ_EMPTY;INTER]);
  TSPEC `x` 1;
  REWR 1;
  TYPE_THEN `!m. num_closure(A UNION B) (pointI m) =  num_closure A (pointI m) + num_closure B (pointI m)` SUBAGOAL_TAC;
  REWRITE_TAC[num_closure];
  IMATCH_MP_TAC  (CARD_UNION);
  CONJ_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `A` EXISTS_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  REWRITE_TAC[SUBSET];
  CONJ_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `B` EXISTS_TAC;
  REWRITE_TAC[SUBSET];
  REWRITE_TAC[EQ_EMPTY ];
  RULE_ASSUM_TAC (REWRITE_RULE[EQ_EMPTY;INTER ]);
  ASM_MESON_TAC[];
  (* - *)
  CONJ_TAC;
  TYPE_THEN `num_closure A (pointI m') = 0` ASM_CASES_TAC;
  REDUCE_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  TYPE_THEN `num_closure B (pointI m') = 0` ASM_CASES_TAC;
  REDUCE_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  UND 10 THEN UND 11 THEN REWRITE_TAC [ARITH_RULE  `~(x = 0) <=> (0 < x)`];
  TYPE_THEN `m' = m` SUBAGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  RULE_ASSUM_TAC (REWRITE_RULE[endpoint]);
  REWRITE_TAC[ARITH_RULE `1+ 1 = 2`;INR IN_INSERT];
  (* -A *)
  TYPE_THEN `inductive_set (A UNION B) S` SUBAGOAL_TAC;
  REWRITE_TAC[inductive_set];
  ASM_REWRITE_TAC[];
  (* - *)
  TYPE_THEN `~(S INTER A = EMPTY)` ASM_CASES_TAC;
  (* -- cut here *)
  IMATCH_MP_TAC  inductive_set_join;
  UNIFY_EXISTS_TAC;
  REWR 14;
  TYPE_THEN `~(S INTER B = EMPTY)` SUBAGOAL_TAC;
  UND 15 THEN UND 14 THEN UND 11 THEN UND 12 THEN REWRITE_TAC[INTER;EQ_EMPTY;SUBSET;UNION] THEN MESON_TAC[];
  (* - *)
  ONCE_REWRITE_TAC [UNION_COMM];
  IMATCH_MP_TAC  inductive_set_join;
  ONCE_REWRITE_TAC [UNION_COMM];
  UNIFY_EXISTS_TAC;
  ]);;
  (* }}} *)

let two_endpoint_segment = prove_by_refinement(
  `!C p q m. segment C /\ endpoint C q /\ endpoint C p /\ endpoint C m /\
     ~(m = p) ==>
      (q = m) \/ (q = p)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `psegment C` SUBAGOAL_TAC;
  IMATCH_MP_TAC  endpoint_psegment;
  UNIFY_EXISTS_TAC;
  (* - *)
  TH_INTRO_TAC[`C`] endpoint_size2;
  IMATCH_MP_TAC  (TAUT `(~A ==> B) ==> (A \/ B)`);
  IMATCH_MP_TAC  two_exclusion;
  UNIFY_EXISTS_TAC;
  ]);;
  (* }}} *)

let EQ_ANTISYM = prove_by_refinement(
  `!A B. (A ==>B) /\ (B ==> A) ==> (A = B)`,
  (* {{{ proof *)
  [
  MESON_TAC[];
  ]);;
  (* }}} *)

let segment_union2 = prove_by_refinement(
  `!A B m p. segment A /\ segment B /\ ~(m = p) /\
     endpoint A m /\ endpoint B m /\
     endpoint A p /\ endpoint B p /\
     (A INTER B = EMPTY) /\
  (!n. (0 < num_closure A (pointI n)) /\ (0 < num_closure B (pointI n)) <=>
          (((n = m ) \/ (n = p) )))
    ==>
    rectagon (A UNION B)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `FINITE A /\ FINITE B` SUBAGOAL_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  (* - *)
  REWRITE_TAC[rectagon];
  ASM_REWRITE_TAC[FINITE_UNION];
  (* - *)
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  RULE_ASSUM_TAC (REWRITE_RULE[EMPTY_EXISTS]);
  UND 11 THEN REWRITE_TAC[EMPTY_EXISTS;UNION];
  TYPE_THEN `u` EXISTS_TAC;
  (* - *)
  CONJ_TAC;
  REWRITE_TAC[union_subset];
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  (* - *)
  TYPE_THEN `!m'. { C | (A UNION B) C /\ closure top2 C (pointI m')} = {C | A C /\ closure top2 C (pointI m')} UNION {C | B C /\ closure top2 C (pointI m')}` SUBAGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION];
  TYPE_THEN `A x` ASM_CASES_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[EQ_EMPTY;INTER]);
  TSPEC `x` 1;
  REWR 1;
  (* - *)
  TYPE_THEN `!m. num_closure(A UNION B) (pointI m) =  num_closure A (pointI m) + num_closure B (pointI m)` SUBAGOAL_TAC;
  REWRITE_TAC[num_closure];
  IMATCH_MP_TAC  (CARD_UNION);
  CONJ_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `A` EXISTS_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  REWRITE_TAC[SUBSET];
  CONJ_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `B` EXISTS_TAC;
  REWRITE_TAC[SUBSET];
  REWRITE_TAC[EQ_EMPTY ];
  RULE_ASSUM_TAC (REWRITE_RULE[EQ_EMPTY;INTER ]);
  ASM_MESON_TAC[];
  (* - *)
  TYPE_THEN `!q. endpoint A q ==> (q = m) \/ (q = p)` SUBAGOAL_TAC;
  IMATCH_MP_TAC two_endpoint_segment;
  UNIFY_EXISTS_TAC;
  TYPE_THEN `!q. endpoint B q ==> (q = m) \/ (q = p)` SUBAGOAL_TAC;
  IMATCH_MP_TAC two_endpoint_segment;
  TYPE_THEN  `B` EXISTS_TAC;
  UNIFY_EXISTS_TAC;
  (* -A *)
  TYPE_THEN `!m. (num_closure A (pointI m) = 1) <=> (num_closure B (pointI m) = 1)` SUBAGOAL_TAC;
  IMATCH_MP_TAC  EQ_ANTISYM;
  RULE_ASSUM_TAC (REWRITE_RULE[endpoint]);
  CONJ_TAC;
  TSPEC `m'` 13;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  TSPEC `m'` 14;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  (* - *)
  CONJ_TAC;
  FULL_REWRITE_TAC[endpoint];
  TYPE_THEN `!x. {0, 2} x <=> {0, 1, 2} x /\ ~(x = 1)` SUBAGOAL_TAC;
  REWRITE_TAC[INSERT];
  ARITH_TAC;
  KILL 16;
  TYPE_THEN `num_closure A (pointI m') = 0` ASM_CASES_TAC;
  REDUCE_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  TSPEC `m'` 15;
  REWR 25;
  UND 25 THEN ARITH_TAC;
  (* -- *)
  TYPE_THEN `num_closure B (pointI m') = 0` ASM_CASES_TAC;
  REDUCE_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[segment]);
  ARITH_TAC;
  FULL_REWRITE_TAC [ARITH_RULE  `~(x = 0) <=> (0 < x)`];
  TYPE_THEN `(m' = m) \/ (m' = p)` SUBAGOAL_TAC;
  TSPEC `m'` 0;
  REWR 0;
  TYPE_THEN `num_closure A (pointI m') = 1` SUBAGOAL_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  TYPE_THEN `num_closure B (pointI m') = 1` SUBAGOAL_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[ARITH_RULE `1+ 1 = 2`;INR IN_INSERT;ARITH_RULE `~(2 = 1)`];
  (* - *)
  TYPE_THEN `inductive_set (A UNION B) S` SUBAGOAL_TAC;
  REWRITE_TAC[inductive_set];
  ASM_REWRITE_TAC[];
  (* - *)
  TYPE_THEN `~(S INTER A = EMPTY)` ASM_CASES_TAC;
  (* -- *)
  IMATCH_MP_TAC  inductive_set_join;
  UNIFY_EXISTS_TAC;
  REWR 20;
  TYPE_THEN `~(S INTER B = EMPTY)` SUBAGOAL_TAC;
  UND 20 THEN UND 21 THEN UND 17 THEN UND 18 THEN REWRITE_TAC[INTER;EQ_EMPTY;SUBSET;UNION] THEN MESON_TAC[];
  (* - *)
  ONCE_REWRITE_TAC [UNION_COMM];
  IMATCH_MP_TAC  inductive_set_join;
  ONCE_REWRITE_TAC [UNION_COMM];
  UNIFY_EXISTS_TAC;
  ]);;
  (* }}} *)

let card_inj = prove_by_refinement(
  `!(f:A->B) A B. INJ f A B /\ FINITE B ==> (CARD A <= CARD B)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `CARD (IMAGE f A) = CARD A` SUBAGOAL_TAC;
  IMATCH_MP_TAC  CARD_IMAGE_INJ;
  CONJ_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  FIRST_ASSUM IMATCH_MP_TAC ;
  IMATCH_MP_TAC  FINITE_INJ;
  ASM_MESON_TAC[];
  USE 2 GSYM;
  IMATCH_MP_TAC  CARD_SUBSET;
  RULE_ASSUM_TAC (REWRITE_RULE[INJ]);
  REWRITE_TAC[IMAGE;SUBSET];
  FIRST_ASSUM IMATCH_MP_TAC ;
  ]);;
  (* }}} *)

let inj_bij_size = prove_by_refinement(
  `!A B (f:A->B). INJ f A B /\ B HAS_SIZE (CARD A) ==> BIJ f A B`,
  (* {{{ proof *)
  [
  REWRITE_TAC[HAS_SIZE];
  TH_INTRO_TAC [`f`;`A`] inj_bij;
  FULL_REWRITE_TAC[INJ];
  ASM_MESON_TAC[];
  TYPE_THEN `IMAGE f A = B` SUBAGOAL_TAC;
  IMATCH_MP_TAC  CARD_SUBSET_EQ;
  CONJ_TAC;
  FULL_REWRITE_TAC[INJ];
  REWRITE_TAC[IMAGE;SUBSET];
  ASM_MESON_TAC[];
  IMATCH_MP_TAC  EQ_SYM;
  IMATCH_MP_TAC  BIJ_CARD;
  UNIFY_EXISTS_TAC;
  ASM_MESON_TAC[FINITE_INJ];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let bij_empty = prove_by_refinement(
  `!(f:A->B). BIJ f EMPTY EMPTY `,
  (* {{{ proof *)
  [
  REWRITE_TAC[BIJ;INJ;SURJ];
  ]);;
  (* }}} *)

let bij_sing = prove_by_refinement(
  `!(f:A->B) a b. BIJ f {a} {b} <=> (f a = b)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[BIJ;INJ;SURJ;INR IN_SING ];
  MESON_TAC[];
  ]);;
  (* }}} *)

let card_sing = prove_by_refinement(
  `!(a:A). (CARD {a} = 1)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  THM_INTRO_TAC[`a`;`EMPTY:A->bool`] card_suc_insert;
  REWRITE_TAC[FINITE_RULES];
  FULL_REWRITE_TAC[CARD_CLAUSES];
  TYPE_THEN `CARD {a}` UNABBREV_TAC;
  ARITH_TAC;
  ]);;
  (* }}} *)

let pair_indistinct = prove_by_refinement(
  `!(a:A). {a,a} = {a}`,
  (* {{{ proof *)
  [
  MESON_TAC[INR ABSORPTION;INR COMPONENT];
  ]);;
  (* }}} *)

let has_size2_distinct = prove_by_refinement(
  `!(a:A) b. {a,b} HAS_SIZE 2 ==> ~(a = b)`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `b` UNABBREV_TAC;
  FULL_REWRITE_TAC [pair_indistinct];
  THM_INTRO_TAC[`a`] sing_has_size1;
  FULL_REWRITE_TAC[HAS_SIZE];
  UND 0 THEN UND 2 THEN ARITH_TAC;
  ]);;
  (* }}} *)

let has_size2_subset = prove_by_refinement(
  `!X (a:A) b. X HAS_SIZE 2 /\ X SUBSET {a,b} ==> (X = {a,b})`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  FULL_REWRITE_TAC [has_size2];
  TYPE_THEN `X` UNABBREV_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  FULL_REWRITE_TAC[SUBSET;in_pair];
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `x` UNABBREV_TAC;
  COPY 0;
  TSPEC `b'` 0;
  TSPEC `a'` 3;
  ASM_MESON_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let inj_subset2 = prove_by_refinement(
  `!t t' s (f:A->B). INJ f s t /\ t SUBSET t' ==> INJ f s t'`,
  (* {{{ proof *)
  [
  REWRITE_TAC[INJ;SUBSET;];
  CONJ_TAC;
  FIRST_ASSUM IMATCH_MP_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ]);;
  (* }}} *)

let terminal_adj = prove_by_refinement(
  `!E b. segment E /\ endpoint E b /\ ~(SING E) ==>
    (?!e.  E e /\ adj (terminal_edge E b) e )`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[EXISTS_UNIQUE_ALT];
  THM_INTRO_TAC[`E`;`b`] terminal_endpoint;
  FULL_REWRITE_TAC[segment];
  (* - *)
  THM_INTRO_TAC[`terminal_edge E b`] two_endpoint;
  FULL_REWRITE_TAC[segment;ISUBSET];
  (* - *)
  FULL_REWRITE_TAC[has_size2];
  USE 6 (REWRITE_RULE[FUN_EQ_THM]);
  TYPE_THEN `?x. !y. (closure top2 (terminal_edge E b) (pointI y) <=> ((y = x) \/ (y = b)))` SUBAGOAL_TAC;
  USE 6 (REWRITE_RULE[in_pair]);
  REWRITE_TAC[in_pair];
  TYPE_THEN `(b = b') \/ (b = a)` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  FIRST_ASSUM DISJ_CASES_TAC  ;
  TYPE_THEN  `a` EXISTS_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `b'` EXISTS_TAC;
  (* - *)
  TYPE_THEN `!e. (adj (terminal_edge E b) e /\ (E e) ==> (closure top2 e (pointI x)))` SUBAGOAL_TAC;
  THM_INTRO_TAC[`terminal_edge E b`;`e`] edge_inter;
  ASM_MESON_TAC[segment;ISUBSET];
  FULL_REWRITE_TAC[INTER;eq_sing];
  TSPEC `m` 7;
  REWR 7;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_MESON_TAC[];
  FULL_REWRITE_TAC[endpoint];
  THM_INTRO_TAC[`E`;`(pointI b)`] num_closure1;
  FULL_REWRITE_TAC[segment];
  REWR 14;
  COPY 14;
  TSPEC `terminal_edge E b` 15;
  TSPEC `e` 14;
  TYPE_THEN `e' = terminal_edge E b` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `e' = e` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  FULL_REWRITE_TAC[adj];
  UND 18 THEN UND 17 THEN UND 16 THEN MESON_TAC[];
  (* - *)
  THM_INTRO_TAC[`E`;`terminal_edge E b`] midpoint_exists;
  FULL_REWRITE_TAC[SING];
  LEFT 0 "x" ;
  TSPEC `terminal_edge E b` 0;
  ASM_MESON_TAC[];
  (* - *)
  FULL_REWRITE_TAC[midpoint];
  THM_INTRO_TAC[`E`;`(pointI m)`] num_closure2;
  FULL_REWRITE_TAC[segment];
  REWR 11;
  (* -DD *)
  TYPE_THEN `?c. ~(terminal_edge E b = c) /\ (E c) /\ (closure top2 c (pointI m))` SUBAGOAL_TAC;
  COPY 12;
  TSPEC `terminal_edge E b` 11;
  REWR 11;
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `b''` EXISTS_TAC;
  TYPE_THEN `a'` EXISTS_TAC;
  (* - *)
  TYPE_THEN `c` EXISTS_TAC;
  COPY 7;
  TSPEC `m` 16;
  REWR 16;
  TYPE_THEN `adj (terminal_edge E b) c` SUBAGOAL_TAC;
  REWRITE_TAC[adj];
  REWRITE_TAC[EMPTY_EXISTS;INTER;];
  TYPE_THEN `pointI m` EXISTS_TAC;
  (* - *)
  IMATCH_MP_TAC  EQ_ANTISYM ;
  CONJ_TAC;
  TYPE_THEN `closure top2 y (pointI x)` SUBAGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `closure top2 c (pointI x)` SUBAGOAL_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  KILL 6;
  TYPE_THEN `closure top2 (terminal_edge E b) (pointI x)` SUBAGOAL_TAC;
  TYPE_THEN `({0,1,2} (num_closure E (pointI x)))` SUBAGOAL_TAC;
  UND 2 THEN MESON_TAC[segment];
  FULL_REWRITE_TAC[INSERT;];
  TYPE_THEN `FINITE E` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[segment];
  THM_INTRO_TAC[`E`;`(pointI x)`] num_closure0;
  REWR 22;
  THM_INTRO_TAC[`E`;`(pointI x)`] num_closure1;
  THM_INTRO_TAC[`E`;`(pointI x)`] num_closure2;
  REWR 22;
  UND 22 THEN REP_CASES_TAC ;
  TYPE_THEN `(terminal_edge E b = a'') \/ (terminal_edge E b = b''')` SUBAGOAL_TAC;
  TSPEC `terminal_edge E b` 22;
  REWR 22;
  TYPE_THEN `(c = a'') \/ (c = b''')` SUBAGOAL_TAC;
  TSPEC `c` 22;
  REWR 22;
  TYPE_THEN `(y = a'') \/ (y = b''')` SUBAGOAL_TAC;
  TSPEC `y` 22;
  REWR 22;
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `a''` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  REWR 29;
  TYPE_THEN `b'''` UNABBREV_TAC;
  USE 18(REWRITE_RULE[adj]);
  UND 29 THEN UND 15 THEN UND 28 THEN MESON_TAC[];
  TYPE_THEN `b'''` UNABBREV_TAC;
  USE 18 (REWRITE_RULE[adj]);
  UND 31 THEN UND 15 THEN UND 29 THEN UND 28 THEN MESON_TAC[];
  (* --- *)
  UND 20 THEN UND 21 THEN UND 14 THEN UND 19 THEN UND 22 THEN MESON_TAC[];
  UND 22 THEN UND 19 THEN UND 20 THEN MESON_TAC[];
  (* - *)
  TYPE_THEN `y` UNABBREV_TAC;
  ]);;
  (* }}} *)

let psegment_order_induct_lemma = prove_by_refinement(
  `!n. !E a b. psegment E /\ (CARD E = n) /\ (endpoint E a) /\
    (endpoint E b) /\ ~(a = b) ==>
    (?f. (BIJ f { p | p < n} E) /\ (f 0 = terminal_edge E a) /\
      ((0 < n) ==> (f (n - 1) = terminal_edge E b)) /\
      (!i j. (i < CARD E /\ j < CARD E) ==>
             (adj (f i) (f j) = ((SUC i = j) \/ (SUC j = i)   ))))`,
  (* {{{ proof *)
  [
  INDUCT_TAC;
  (* -- 0 case *)
  TYPE_THEN `f = (\ (x:num). terminal_edge E a)` ABBREV_TAC ;
  TYPE_THEN `f` EXISTS_TAC;
  TYPE_THEN `{ p | p < 0} = EMPTY` SUBAGOAL_TAC;
  REWRITE_TAC[EQ_EMPTY];
  UND 6 THEN ARITH_TAC;
  TYPE_THEN `E HAS_SIZE 0` SUBAGOAL_TAC;
  REWRITE_TAC[HAS_SIZE];
  FULL_REWRITE_TAC[psegment;segment];
  FULL_REWRITE_TAC[HAS_SIZE_0];
  REWRITE_TAC[ARITH_RULE `~(k <| 0)`;bij_empty];
  EXPAND_TAC "f";
  (* - 1 case *)
  REWRITE_TAC[ARITH_RULE `0 <| SUC n /\ (SUC n - 1 = n)`];
  TYPE_THEN `n = 0` ASM_CASES_TAC;
  KILL 5;
  REWRITE_TAC[ARITH_RULE `i <| SUC 0 <=> (i = 0)`;];
  REWRITE_TAC[ARITH_RULE `~(SUC 0 = 0)`;adj];
  TYPE_THEN `n` UNABBREV_TAC;
  FULL_REWRITE_TAC[ARITH_RULE `SUC 0 = 1`];
  TYPE_THEN `E HAS_SIZE 1` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[HAS_SIZE;psegment;segment];
  USE 5(MATCH_MP   CARD_SING_CONV);
  FULL_REWRITE_TAC[SING];
  TYPE_THEN `E` UNABBREV_TAC;
  TYPE_THEN `f = (\ (y:num). x )` ABBREV_TAC ;
  TYPE_THEN `f` EXISTS_TAC;
  TYPE_THEN `FINITE {x}` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[psegment;segment];
  TYPE_THEN `{p | p = 0} = {0}` SUBAGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[INR IN_SING];
  KILL 7;
  TYPE_THEN `f 0 = x` SUBAGOAL_TAC;
  EXPAND_TAC "f";
  REWRITE_TAC[bij_sing];
  TH_INTRO_TAC[`{x}`;`a`] terminal_endpoint;
  TH_INTRO_TAC[`{x}`;`b`] terminal_endpoint;
  FULL_REWRITE_TAC[INR IN_SING];
  (* - A2 and above *)
  TYPE_THEN `e = terminal_edge E b` ABBREV_TAC ;
  TYPE_THEN `b' = other_end e b` ABBREV_TAC ;
  TYPE_THEN `E' = E DELETE e` ABBREV_TAC ;
  (* - *)
  TYPE_THEN `E e /\ closure top2 e (pointI b)` SUBAGOAL_TAC;
  TYPE_THEN `e` UNABBREV_TAC;
  IMATCH_MP_TAC  terminal_endpoint;
  RULE_ASSUM_TAC (REWRITE_RULE[psegment;segment]);
  (* - *)
  TYPE_THEN `psegment E'` SUBAGOAL_TAC;
  REWRITE_TAC[psegment];
  CONJ_TAC;
  TYPE_THEN `E'` UNABBREV_TAC;
  IMATCH_MP_TAC  segment_delete;
  TYPE_THEN `b` EXISTS_TAC;
  RULE_ASSUM_TAC (REWRITE_RULE[psegment]);
  REWRITE_TAC[];
  TYPE_THEN `E` UNABBREV_TAC;
  THM_INTRO_TAC [`e`] sing_has_size1;
  RULE_ASSUM_TAC (REWRITE_RULE[HAS_SIZE]);
  UND 12 THEN UND 3 THEN UND 6 THEN ARITH_TAC;
  THM_INTRO_TAC [`E'`;`E`] rectagon_subset;
  RULE_ASSUM_TAC (REWRITE_RULE[psegment]);
  TYPE_THEN `E'` UNABBREV_TAC;
  REWRITE_TAC[DELETE;SUBSET];
  TYPE_THEN `E'` UNABBREV_TAC;
  UND 13 THEN UND 11 THEN MESON_TAC[INR DELETE_NON_ELEMENT];
  (* - *)
  TYPE_THEN `SUC (CARD E') = SUC n` SUBAGOAL_TAC;
  TYPE_THEN `E'` UNABBREV_TAC;
  TYPE_THEN `SUC n` UNABBREV_TAC;
  IMATCH_MP_TAC  CARD_SUC_DELETE;
  FULL_REWRITE_TAC[psegment;segment];
  FULL_REWRITE_TAC[EQ_SUC];
  (* -B *)
  THM_INTRO_TAC [`E`;`b`;`e`] psegment_delete_end;
  REWRITE_TAC[];
  TYPE_THEN `E` UNABBREV_TAC;
  FULL_REWRITE_TAC[card_sing];
  UND 3 THEN UND 6 THEN ARITH_TAC;
  (* - *)
  TYPE_THEN `endpoint E' = {a,b'}` SUBAGOAL_TAC;
  IMATCH_MP_TAC  has_size2_subset;
  CONJ_TAC;
  IMATCH_MP_TAC  endpoint_size2;
  TYPE_THEN `E'` UNABBREV_TAC;
  REWRITE_TAC[SUBSET;INSERT;DELETE];
  FIRST_ASSUM DISJ_CASES_TAC;
  THM_INTRO_TAC [`E`;`x`;`a`;`b`] two_endpoint_segment;
  FULL_REWRITE_TAC[psegment];
  ASM_MESON_TAC[];
  THM_INTRO_TAC[`e`;`b`] other_end_prop;
  UND 4 THEN REWRITE_TAC[psegment;segment;SUBSET;];
  (* - *)
  TYPE_THEN `{a,b'} HAS_SIZE 2` SUBAGOAL_TAC;
  TYPE_THEN `{a,b'}` UNABBREV_TAC;
  IMATCH_MP_TAC  endpoint_size2;
  USE 16 (MATCH_MP has_size2_distinct);
  UND 5 THEN DISCH_THEN (THM_INTRO_TAC[`E'`;`a`;`b'`]);
  REWRITE_TAC[in_pair];
  (* - *)
  TYPE_THEN `g = (\ i.  if (i <| n) then f i else e)` ABBREV_TAC ;
  TYPE_THEN `!i. (i <| n) ==> (g i = f i)` SUBAGOAL_TAC;
  TYPE_THEN `g` UNABBREV_TAC;
  TYPE_THEN `g n = e` SUBAGOAL_TAC;
  TYPE_THEN `g` UNABBREV_TAC;
  REWRITE_TAC[ARITH_RULE `~(n <| n)`];
  TYPE_THEN `g` EXISTS_TAC;
  (* - FINAL PUSH *)
  SUBCONJ_TAC;
  IMATCH_MP_TAC  inj_bij_size;
  REWRITE_TAC[CARD_NUMSEG_LT];
  CONJ_TAC;
  TYPE_THEN `{p | p <| SUC n} = {p | p <| n} UNION {n}` SUBAGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION;INR IN_SING];
  ARITH_TAC;
  IMATCH_MP_TAC  inj_split;
  CONJ_TAC;
  TYPE_THEN `INJ g {p | p <| n} E = INJ f {p | p <| n} E` SUBAGOAL_TAC;
  IMATCH_MP_TAC  inj_domain_sub;
  USE 24 (REWRITE_RULE[]);
  RULE_ASSUM_TAC (REWRITE_RULE[BIJ]);
  (* --- temp *)
  IMATCH_MP_TAC  inj_subset2;
  UNIFY_EXISTS_TAC;
  UND 9 THEN REWRITE_TAC[SUBSET;DELETE];
  TYPE_THEN `E'` UNABBREV_TAC;
  CONJ_TAC;
  REWRITE_TAC[INJ;INR IN_SING;];
  REP_BASIC_TAC;
  REWRITE_TAC[IMAGE;INTER;EQ_EMPTY;INR IN_SING ];
  TYPE_THEN `x''` UNABBREV_TAC;
  TYPE_THEN `x` UNABBREV_TAC;
  TYPE_THEN `g n` UNABBREV_TAC;
  TSPEC `x'` 21;
  TYPE_THEN `g x'` UNABBREV_TAC;
  FULL_REWRITE_TAC[BIJ;SURJ];
  TSPEC `x'` 22;
  TYPE_THEN `E'` UNABBREV_TAC;
  FULL_REWRITE_TAC[DELETE];
  ASM_MESON_TAC[];
  UND 4 THEN ASM_REWRITE_TAC[HAS_SIZE;psegment;segment;rectagon];
  (* - C*)
  TYPE_THEN `E' SUBSET E` SUBAGOAL_TAC;
  TYPE_THEN `E'` UNABBREV_TAC;
  REWRITE_TAC[DELETE;SUBSET];
  (* - *)
  TSPEC `0` 21;
  TYPE_THEN `0 <| n` SUBAGOAL_TAC;
  UND 6 THEN ARITH_TAC;
  TYPE_THEN `f 0` UNABBREV_TAC;
  CONJ_TAC;
  TYPE_THEN `e' = terminal_edge E' a` ABBREV_TAC ;
  THM_INTRO_TAC[`E'`;`a`;`e'`] terminal_unique;
  REWRITE_TAC[INR in_pair];
  UND 12 THEN REWRITE_TAC[psegment;segment];
  TYPE_THEN `e'` UNABBREV_TAC;
  TYPE_THEN `g 0 ` UNABBREV_TAC;
  THM_INTRO_TAC[`E`;`a`;`terminal_edge E' a`] terminal_unique;
  UND 4 THEN (REWRITE_TAC[psegment;segment]);
  REWR 26;
  ASM_MESON_TAC[ISUBSET];
  (* -D *)
  TYPE_THEN `E' (terminal_edge E' b')` SUBAGOAL_TAC;
  THM_INTRO_TAC[`E'`;`b'`] terminal_endpoint;
  FULL_REWRITE_TAC[psegment;segment;INR in_pair ];
  (* - *)
  TYPE_THEN `~(E' (terminal_edge E b))` SUBAGOAL_TAC;
  TYPE_THEN `E'` UNABBREV_TAC;
  FULL_REWRITE_TAC[DELETE];
  TYPE_THEN `terminal_edge E b` UNABBREV_TAC;
  (* - *)
  TYPE_THEN `adj e (g (n - 1))` SUBAGOAL_TAC;
  TYPE_THEN `g (n - 1) = f (n-1 )` SUBAGOAL_TAC;
  TYPE_THEN `g` UNABBREV_TAC;
  TYPE_THEN `n - 1 < n` SUBAGOAL_TAC;
  UND 21 THEN ARITH_TAC;
  TYPE_THEN `f (n - 1)` UNABBREV_TAC;
  TYPE_THEN `e` UNABBREV_TAC;
  REWRITE_TAC[adj];
  REWRITE_TAC[INTER;EMPTY_EXISTS];
  CONJ_TAC;
   TYPE_THEN `g n` UNABBREV_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `pointI b'` EXISTS_TAC;
  CONJ_TAC;
  TYPE_THEN `b'` UNABBREV_TAC;
  THM_INTRO_TAC[`terminal_edge E b`;`b`]other_end_prop;
  FULL_REWRITE_TAC[psegment;segment;ISUBSET];
  THM_INTRO_TAC  [`E'`;`b'`] terminal_endpoint;
  FULL_REWRITE_TAC[psegment;segment;in_pair];
  (* - *)
  TYPE_THEN `!i. (i <| SUC n) ==> (adj (g n) (g i) = (SUC i = n))` SUBAGOAL_TAC;
  TYPE_THEN `( i' = n) \/ (i' <| n)` SUBAGOAL_TAC;
  UND 30 THEN ARITH_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  REWRITE_TAC[adj];
  ARITH_TAC;
  (* -- *)
  THM_INTRO_TAC[`E`;`b`] terminal_adj;
  FULL_REWRITE_TAC[psegment];
  REWRITE_TAC[];
  USE 35 (MATCH_MP CARD_SING);
  TYPE_THEN `CARD E` UNABBREV_TAC;
  UND 3 THEN UND 21 THEN ARITH_TAC;
  FULL_REWRITE_TAC[EXISTS_UNIQUE_ALT];
  TYPE_THEN `!i'. (i' <| n) ==> (adj e (g i') = (e' = (g i')))` SUBAGOAL_TAC;
  TSPEC  `g (i'')`33;
  TYPE_THEN `E (g i'')` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[BIJ;SURJ];
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 34 THEN ARITH_TAC;
  REWR 33;
  IMATCH_MP_TAC  EQ_ANTISYM;
  CONJ_TAC;
  TYPE_THEN `e'` UNABBREV_TAC;
  TSPEC `n - 1` 34;
  TYPE_THEN `n - 1 < n` SUBAGOAL_TAC;
  UND 21 THEN ARITH_TAC;
  TYPE_THEN `(g i' = g (n - 1)) ==> (SUC i' = n)` SUBAGOAL_TAC;
  FULL_REWRITE_TAC [BIJ;INJ];
  IMATCH_MP_TAC  (ARITH_RULE  `((i' = n - 1) /\ (0 < n)) ==> (SUC i' = n)` );
  FIRST_ASSUM IMATCH_MP_TAC ;
  ARITH_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  REWR 34;
  (* -- *)
  TYPE_THEN `i' = n - 1` SUBAGOAL_TAC;
  UND 35 THEN UND 21 THEN ARITH_TAC;
  TSPEC `i'` 34;
  TYPE_THEN `i'` UNABBREV_TAC;
  REWR 32;
  (* -E *)
  TYPE_THEN `(i = n) \/ (i <| n)` SUBAGOAL_TAC;
  UND 26 THEN ARITH_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  TSPEC `j` 30;
  UND 30 THEN ARITH_TAC;
  (* - *)
  TYPE_THEN `(j = n) \/ (j <| n)` SUBAGOAL_TAC;
  UND 25 THEN ARITH_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  ONCE_REWRITE_TAC [adj_symm];
  UND 26 THEN ARITH_TAC;
  (* - *)
  TYPE_THEN `g` UNABBREV_TAC;
  FIRST_ASSUM IMATCH_MP_TAC ;
  ASM_REWRITE_TAC[];

  ]);;
  (* }}} *)

(* a couple of variants *)
let psegment_order = prove_by_refinement(
  `!E a b. psegment E /\ (endpoint E a) /\
    (endpoint E b) /\ ~(a = b) ==>
    (?f. (BIJ f { p | p < CARD E} E) /\ (f 0 = terminal_edge E a) /\
      ((0 < CARD E) ==> (f (CARD E - 1) = terminal_edge E b)) /\
      (!i j. (i < CARD E /\ j < CARD E) ==>
             (adj (f i) (f j) = ((SUC i = j) \/ (SUC j = i)   ))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  THM_INTRO_TAC[`CARD E`;`E`;`a`;`b`] psegment_order_induct_lemma;
  REWRITE_TAC[];
  ]);;
  (* }}} *)

let psegment_order' = prove_by_refinement(
  `!A m. psegment A /\ endpoint A m  ==>
    (?f. BIJ f {p | p < CARD A} A /\
        (f 0 = terminal_edge A m) /\
        (!i j. (i < CARD A /\ j < CARD A) ==>
             (adj (f i) (f j) = ((SUC i = j) \/ (SUC j = i)   ))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  THM_INTRO_TAC[`A`] endpoint_size2;
  FULL_REWRITE_TAC[has_size2];
  TYPE_THEN `?n. (endpoint A n) /\ ~(m = n)` SUBAGOAL_TAC;
  REWR 0;
  FULL_REWRITE_TAC[in_pair];
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `a` EXISTS_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  THM_INTRO_TAC[`A`;`m`;`n`] psegment_order;
  TYPE_THEN `f` EXISTS_TAC;
  ASM_REWRITE_TAC[];
    ]);;
  (* }}} *)

let order_imp_psegment = prove_by_refinement(
  `!f n. (INJ f { p | p < n} (edge)) /\ (0 < n) /\
     (!i j. (i < n /\ j < n) ==>
             (adj (f i) (f j) = ((SUC i = j) \/ (SUC j = i)   ))) ==>
    (psegment (IMAGE f { p | p < n}))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `E = IMAGE f {p | p <| n}` ABBREV_TAC ;
  IMATCH_MP_TAC  endpoint_psegment;
  REWRITE_TAC[segment;];
  TYPE_THEN `FINITE E` SUBAGOAL_TAC;
  TYPE_THEN `E` UNABBREV_TAC;
  IMATCH_MP_TAC  FINITE_IMAGE;
  REWRITE_TAC[FINITE_NUMSEG_LT];
  (* - *)
  TYPE_THEN `~(E = {})` SUBAGOAL_TAC;
  TYPE_THEN `E` UNABBREV_TAC;
  FULL_REWRITE_TAC[image_empty];
  FULL_REWRITE_TAC[EQ_EMPTY];
  ASM_MESON_TAC[];
  (* - *)
  TYPE_THEN `E SUBSET edge` SUBAGOAL_TAC;
  TYPE_THEN `E` UNABBREV_TAC;
  FULL_REWRITE_TAC[IMAGE;INJ;SUBSET];
  FIRST_ASSUM IMATCH_MP_TAC ;
  (* - *)
  TYPE_THEN `E (f 0)` SUBAGOAL_TAC;
  TYPE_THEN `E` UNABBREV_TAC ;
  REWRITE_TAC[IMAGE];
  TYPE_THEN `0` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  (* - *)
  TYPE_THEN `edge (f 0)` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[SUBSET];
  (* -A *)
  TYPE_THEN `?m. endpoint E m` SUBAGOAL_TAC;
  REWRITE_TAC[endpoint];
  ASM_SIMP_TAC[num_closure1];
  LEFT_TAC "e";
  TYPE_THEN `f 0 ` EXISTS_TAC;
  THM_INTRO_TAC[`f 0`] two_endpoint;
  FULL_REWRITE_TAC[has_size2];
  ASM_CASES_TAC `n =1`;
  TYPE_THEN `a` EXISTS_TAC;
  IMATCH_MP_TAC  EQ_ANTISYM;
  CONJ_TAC;
  TYPE_THEN `E` UNABBREV_TAC;
  TYPE_THEN `n` UNABBREV_TAC;
  FULL_REWRITE_TAC[IMAGE];
  TYPE_THEN `(x' = 0) /\ (x = 0)` SUBAGOAL_TAC;
  UND 7 THEN UND 13 THEN ARITH_TAC;
  TYPE_THEN `e'` UNABBREV_TAC;
  USE 10 (ONCE_REWRITE_RULE[FUN_EQ_THM]);
  TSPEC `a` 10;
  FULL_REWRITE_TAC[in_pair];
  (* -- *)
  TYPE_THEN `E (f 1)` SUBAGOAL_TAC;
  TYPE_THEN `E` UNABBREV_TAC;
  REWRITE_TAC[IMAGE];
  TYPE_THEN `1` EXISTS_TAC;
  UND 11 THEN UND 1 THEN ARITH_TAC;
  (* -- *)
  TYPE_THEN `edge (f 1)` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[SUBSET];
  (* -- *)
  TYPE_THEN `adj (f 0 ) (f 1)` SUBAGOAL_TAC;
  UND 0 THEN DISCH_THEN (THM_INTRO_TAC[`0`;`1`]);
  UND 11 THEN UND 1 THEN ARITH_TAC;
  ARITH_TAC;
  THM_INTRO_TAC[`f 0`;`f 1`] edge_inter;
  FULL_REWRITE_TAC[INTER;INR eq_sing  ];
  (* -- *)
  TYPE_THEN `?r. closure top2 (f 0) (pointI r) /\ ~(r = m)` SUBAGOAL_TAC;
  USE 10 (ONCE_REWRITE_RULE[FUN_EQ_THM]);
  FULL_REWRITE_TAC[in_pair];
  TYPE_THEN `m = a` ASM_CASES_TAC;
  TYPE_THEN `m` UNABBREV_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  TYPE_THEN `a` EXISTS_TAC;
  TYPE_THEN `r` EXISTS_TAC;
  IMATCH_MP_TAC  EQ_ANTISYM;
  CONJ_TAC;
  TYPE_THEN`?j. (j <| n) /\ (e' = f j)` SUBAGOAL_TAC;
  TYPE_THEN`E` UNABBREV_TAC;
  FULL_REWRITE_TAC[IMAGE];
  TYPE_THEN`x` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `e'` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `adj (f 0) (f j)` SUBAGOAL_TAC;
  REWRITE_TAC[adj;EMPTY_EXISTS;INTER ];
  TYPE_THEN`pointI r` EXISTS_TAC;
  UND 0 THEN DISCH_THEN (THM_INTRO_TAC[` 0`;` j`] );
  REWR 0;
  TYPE_THEN `j = 1` SUBAGOAL_TAC;
  UND 0 THEN ARITH_TAC;
  TYPE_THEN `j` UNABBREV_TAC;
  TSPEC `pointI r` 15;
  REWR 15;
  FULL_REWRITE_TAC[pointI_inj];
  ASM_MESON_TAC[];
  TYPE_THEN `e'` UNABBREV_TAC;
  CONJ_TAC;
  UNIFY_EXISTS_TAC;
  (* -B *)
  TYPE_THEN `!e. (E e ==> ?i. (i <| n) /\ (e = f i))` SUBAGOAL_TAC;
  TYPE_THEN `E` UNABBREV_TAC;
  FULL_REWRITE_TAC[IMAGE];
  ASM_MESON_TAC[];
  (* - *)
  CONJ_TAC;
  REWRITE_TAC[INSERT];
  ASM_SIMP_TAC [num_closure0;num_closure1;num_closure2];
  PROOF_BY_CONTR_TAC;
  FULL_REWRITE_TAC[DE_MORGAN_THM];
  LEFT 11 "e";
  LEFT 12 "e";
  TSPEC `e` 12;
  LEFT 12 "e'";
  FULL_REWRITE_TAC[NOT_IMP];
  TYPE_THEN `E e' /\ closure top2 e' (pointI m') /\ ~(e = e')` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `adj e e'` SUBAGOAL_TAC;
  REWRITE_TAC[adj;EMPTY_EXISTS;INTER;];
  UNIFY_EXISTS_TAC;
  TYPE_THEN `(?i. (i <| n) /\ (e = f i))` SUBAGOAL_TAC;
  TYPE_THEN `(?j. (j <| n) /\ (e' = f j))` SUBAGOAL_TAC;
  TYPE_THEN `e` UNABBREV_TAC;
  TYPE_THEN `e'` UNABBREV_TAC;
  TYPE_THEN `(SUC i = j) \/ (SUC j = i)` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  LEFT 13 "a";
  TSPEC `f i` 13;
  LEFT 13 "b";
  TSPEC `f j` 13;
  UND 13 THEN REWRITE_TAC[];
  REWRITE_TAC[];
  IMATCH_MP_TAC  EQ_ANTISYM;
  CONJ_TAC;
  TYPE_THEN `?k. (k <| n) /\ (e'' = f k)` SUBAGOAL_TAC;
  TYPE_THEN `e''` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  FULL_REWRITE_TAC[DE_MORGAN_THM];
  TYPE_THEN `adj (f i) (f k) /\ adj (f j) (f k)` SUBAGOAL_TAC;
  REWRITE_TAC[adj];
  REWRITE_TAC[INTER;EMPTY_EXISTS];
  LEFT_TAC "u";
  UNIFY_EXISTS_TAC;
  TYPE_THEN `(SUC j = k) \/ (SUC k = j)` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `(SUC i = k) \/ (SUC k = i)` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
   UND 29 THEN UND 28 THEN UND 19 THEN ARITH_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_REWRITE_TAC[];
  ASM_REWRITE_TAC[];
  (* -C *)
  TYPE_THEN `X = {p | p <| n /\ S (f p)}` ABBREV_TAC ;
  TYPE_THEN `~(X = EMPTY)` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[EMPTY_EXISTS;SUBSET];
  TYPE_THEN `E u` SUBAGOAL_TAC;
  TYPE_THEN `(?i. (i <| n) /\ (u = f i))` SUBAGOAL_TAC;
  TYPE_THEN `u` UNABBREV_TAC;
  UNDF `EMPTY` THEN REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `i` EXISTS_TAC;
  TYPE_THEN `X` UNABBREV_TAC;
  (* - *)
  TYPE_THEN `!j k. X j /\ (k <| n) /\ ((SUC j = k) \/ (SUC k = j)) ==> (X k)` SUBAGOAL_TAC;
  TYPE_THEN `j = k` ASM_CASES_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `S (f j)` SUBAGOAL_TAC;
  TYPE_THEN `X` UNABBREV_TAC;
  TYPE_THEN `E (f k)` SUBAGOAL_TAC;
  TYPE_THEN `E` UNABBREV_TAC;
  REWRITE_TAC[IMAGE];
  TYPE_THEN `k` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  TYPE_THEN `adj (f j) (f k)` SUBAGOAL_TAC;
  TYPE_THEN `X` UNABBREV_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `S (f k)` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `X` UNABBREV_TAC;
  (* - *)
  TYPE_THEN `(?i. X i /\ (!m. m <| i ==> ~X m))` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[EMPTY_EXISTS];
  ASM_MESON_TAC[num_WOP];
  TYPE_THEN `i = 0` SUBAGOAL_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `?j. SUC j = i` SUBAGOAL_TAC;
  TYPE_THEN `i - 1` EXISTS_TAC;
  UND 19 THEN ARITH_TAC;
  TSPEC `j` 17;
  UND 17 THEN DISCH_THEN (THM_INTRO_TAC[]);
  UND 20 THEN ARITH_TAC;
  UND 17 THEN REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `i` EXISTS_TAC;
  TYPE_THEN `X` UNABBREV_TAC;
  UND 17 THEN UND 20 THEN ARITH_TAC;
  TYPE_THEN `i` UNABBREV_TAC;
  (* -D *)
  TYPE_THEN `X = { p | p <| n }` SUBAGOAL_TAC;
  IMATCH_MP_TAC  subset_imp_eq;
  CONJ_TAC;
  TYPE_THEN `X` UNABBREV_TAC;
  REWRITE_TAC[SUBSET];
  PROOF_BY_CONTR_TAC;
  FULL_REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `Z = ({p | p <| n} DIFF X)` ABBREV_TAC ;
  TYPE_THEN `?n. Z n /\ (!m. m <| n ==> ~Z m)` SUBAGOAL_TAC;
  UND 19 THEN MESON_TAC[num_WOP];
  TYPE_THEN `Z` UNABBREV_TAC;
  FULL_REWRITE_TAC[DIFF];
  TSPEC `n' - 1` 21;
  TYPE_THEN `~(n' = 0)` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `n' - 1 <| n'` SUBAGOAL_TAC;
  UND 24 THEN ARITH_TAC;
  TYPE_THEN `n' - 1 <| n` SUBAGOAL_TAC;
  UND 20 THEN ARITH_TAC;
  REWR 21;
  UND 19 THEN REWRITE_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  TYPE_THEN `n' - 1` EXISTS_TAC;
  UND 24 THEN ARITH_TAC;
  IMATCH_MP_TAC  SUBSET_ANTISYM;
  REWRITE_TAC[SUBSET];
  TYPE_THEN `E` UNABBREV_TAC;
  TYPE_THEN `X` UNABBREV_TAC;
  USE 20 (REWRITE_RULE[IMAGE]);
  USE 19 (ONCE_REWRITE_RULE[FUN_EQ_THM]);
  TSPEC `x'` 19;
  FULL_REWRITE_TAC[];
  REWR 19;
  ]);;
  (* }}} *)

let rectagon_nonsing = prove_by_refinement(
  `!G. rectagon G ==> ~SING G`,
  (* {{{ proof *)
  [
  REWRITE_TAC[rectagon;SING];
  TYPE_THEN `G` UNABBREV_TAC;
  THM_INTRO_TAC [`x`] two_endpoint;
  FULL_REWRITE_TAC[SUBSET;INR IN_SING;];
  FULL_REWRITE_TAC[has_size2];
  USE 6 (ONCE_REWRITE_RULE [FUN_EQ_THM]);
  FULL_REWRITE_TAC[in_pair];
  TSPEC `b` 6;
  REWR 6;
  TSPEC `b` 2;
  THM_INTRO_TAC[`{x}`;`pointI b`] num_closure0;
  FULL_REWRITE_TAC[INR IN_SING];
  REWR 2;
  LEFT 2 "e" ;
  TSPEC  `x` 2;
  REWR 2;
  THM_INTRO_TAC[`{x}`;`pointI b`] num_closure2;
  REWR 8;
  FULL_REWRITE_TAC[INR IN_SING];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let rectagon_2 = prove_by_refinement(
  `!G S. rectagon G /\ S SUBSET G /\ ~(S = EMPTY) /\
    (!m. {0,2} (num_closure S (pointI m))) ==> (S = G)`,
  (* {{{ proof *)

  [
  REP_BASIC_TAC;
  TYPE_THEN `Tx = { A | ~(A = EMPTY) /\ A SUBSET S /\ (!m. {0,2} (num_closure A (pointI m))) }` ABBREV_TAC ;
  TYPE_THEN `~(Tx = EMPTY)` SUBAGOAL_TAC;
  UND 5 THEN REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `S` EXISTS_TAC;
  TYPE_THEN `Tx` UNABBREV_TAC;
  REWRITE_TAC[SUBSET];
  USE 5 (MATCH_MP select_card_min);
  (* - *)
  TYPE_THEN `z SUBSET G` SUBAGOAL_TAC;
  TYPE_THEN `Tx` UNABBREV_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  UNIFY_EXISTS_TAC;
  (* - *)
  TYPE_THEN `(z = G) ==> (S = G)` SUBAGOAL_TAC;
  TYPE_THEN `Tx` UNABBREV_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  FULL_REWRITE_TAC [ISUBSET];
  ASM_MESON_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  KILL 8;
  (* - *)
  IMATCH_MP_TAC  rectagon_subset;
  TYPE_THEN `segment G` SUBAGOAL_TAC;
  IMATCH_MP_TAC  rectagon_segment;
  (* - *)
  REWRITE_TAC[rectagon];
  TYPE_THEN `Tx` UNABBREV_TAC;
  SUBCONJ_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `G` EXISTS_TAC;
  FULL_REWRITE_TAC[rectagon];
  CONJ_TAC;
  FULL_REWRITE_TAC[rectagon];
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `G` EXISTS_TAC;
  (* -A1 *)
  IMATCH_MP_TAC  CARD_SUBSET_LE;
  FIRST_ASSUM IMATCH_MP_TAC ;
  CONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  UNIFY_EXISTS_TAC;
  KILL 5;
  KILL 0;
  TSPEC `m` 4;
  FULL_REWRITE_TAC[INSERT];
  USE 0 (MATCH_MP (TAUT `a \/ b ==> b \/ a`));
  FIRST_ASSUM DISJ_CASES_TAC;
  THM_INTRO_TAC[`S'`;`z`;`pointI m`] num_closure_mono;
  UND 4 THEN UND 5 THEN ARITH_TAC;
  KILL 0;
  (* - *)
  TYPE_THEN `~(num_closure S' (pointI m) = 1)` ASM_CASES_TAC;
  THM_INTRO_TAC[`S'`;`z`;`pointI m`] num_closure_mono;
  UND 5 THEN UND 0 THEN UND 4 THEN ARITH_TAC;
  REWR 0;
  (* - *)
  THM_INTRO_TAC[`S'`;`(pointI m)`] num_closure1;
  IMATCH_MP_TAC  FINITE_SUBSET;
  UNIFY_EXISTS_TAC;
  REWR 5;
  (* - *)
  THM_INTRO_TAC[`z`;`pointI m`] num_closure2;
  REWR 14;
  COPY 14;
  TSPEC `e` 16;
  COPY 5;
  TSPEC `e` 5;
  USE 5 (REWRITE_RULE[]);
  TYPE_THEN `z e` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[ISUBSET];
  TYPE_THEN `(e = a) \/ (e = b)` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  KILL 16;
  (* -B1 *)
  TYPE_THEN `?e'. (closure top2 e' (pointI m)) /\ z e' /\ ~(e = e')` SUBAGOAL_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `a` EXISTS_TAC;
  ASM_MESON_TAC[];
  (* - *)
  UND 11 THEN DISCH_THEN (THM_INTRO_TAC[`e`;`e'`]);
  REWRITE_TAC[adj;INTER;EMPTY_EXISTS;];
  TYPE_THEN `pointI m` EXISTS_TAC;
  TSPEC  `e'` 17 ;
  ASM_MESON_TAC[];
  ]);;

  (* }}} *)

let closure_imp_adj = prove_by_refinement(
  `!X Y m. (closure top2 X (pointI m) /\ closure top2 Y (pointI m) /\
      ~(X = Y) ==> adj X Y)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[adj];
  REWRITE_TAC[INTER;EMPTY_EXISTS];
  UNIFY_EXISTS_TAC;
  ]);;
  (* }}} *)

let inductive_set_endpoint = prove_by_refinement(
  `!G S. FINITE G /\ inductive_set G S ==>
     (endpoint S SUBSET endpoint G)`,
  (* {{{ proof *)
  [
  REWRITE_TAC[inductive_set];
  REWRITE_TAC[SUBSET;endpoint];
  TYPE_THEN `FINITE S` SUBAGOAL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  UNIFY_EXISTS_TAC;
  THM_INTRO_TAC[`S`;`pointI x`] num_closure1;
  REWR 6;
  ASM_SIMP_TAC[num_closure1];
  TYPE_THEN `e` EXISTS_TAC;
  IMATCH_MP_TAC  EQ_ANTISYM;
  CONJ_TAC;
  COPY 6;
  TSPEC `e'` 6;
  TSPEC `e` 9;
  REWR 6;
  REWR 9;
  PROOF_BY_CONTR_TAC;
  UND 0 THEN DISCH_THEN (  THM_INTRO_TAC[`e`;`e'`]);
  IMATCH_MP_TAC  closure_imp_adj;
  TYPE_THEN `x` EXISTS_TAC;
  ASM_MESON_TAC[];
  (* - *)
  TYPE_THEN `e'` UNABBREV_TAC;
  TSPEC `e` 6;
  ASM_MESON_TAC[ISUBSET];
  ]);;
  (* }}} *)

let endpoint_closure = prove_by_refinement(
  `!e. (edge e) ==> (endpoint {e} = {m | closure top2 e (pointI m)})`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[endpoint];
  THM_INTRO_TAC[`{e}`;`pointI x`] num_closure1;
  REWRITE_TAC[FINITE_SING];
  REWRITE_TAC[INR IN_SING];
  IMATCH_MP_TAC  EQ_ANTISYM;
  CONJ_TAC;
  TYPE_THEN `e = e'` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `e'` UNABBREV_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `e` EXISTS_TAC;
  ASM_REWRITE_TAC[];
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let rectagon_delete = prove_by_refinement(
  `!E e. (rectagon E) /\ (E e) ==> (psegment (E DELETE e))`,
  (* {{{ proof *)
  [
  REWRITE_TAC[psegment];
  IMATCH_MP_TAC  (TAUT `A /\ B ==> B /\ A`);
  CONJ_TAC;
  THM_INTRO_TAC[`E DELETE e`;`E`] rectagon_subset;
  CONJ_TAC;
  IMATCH_MP_TAC  rectagon_segment;
  REWRITE_TAC[DELETE;SUBSET];
  ASM_MESON_TAC[INR DELETE_NON_ELEMENT];
  (* - *)
  REWRITE_TAC[segment];
  CONJ_TAC;
  FULL_REWRITE_TAC[rectagon];
  REWRITE_TAC[FINITE_DELETE];
  (* - *)
  SUBCONJ_TAC;
  FULL_REWRITE_TAC[delete_empty];
  FULL_REWRITE_TAC[EMPTY_EXISTS];
  USE 1 (MATCH_MP rectagon_nonsing);
  FULL_REWRITE_TAC[SING];
  ASM_MESON_TAC[];
  (* - *)
  SUBCONJ_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  TYPE_THEN `E` EXISTS_TAC;
  CONJ_TAC;
  REWRITE_TAC[DELETE;SUBSET];
  FULL_REWRITE_TAC[rectagon];
  (* - *)
  SUBCONJ_TAC;
  THM_INTRO_TAC[`E DELETE e`;`E`;`pointI m`] num_closure_mono;
  FULL_REWRITE_TAC[rectagon;DELETE;SUBSET];
  FULL_REWRITE_TAC[rectagon];
  UND 5 THEN UND 4 THEN (REWRITE_TAC[INSERT]) ;
  TSPEC `m` 4;
  UND 4 THEN UND 5 THEN ARITH_TAC;
  (* -A *)
  TYPE_THEN `~S e` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[SUBSET;DELETE];
  ASM_MESON_TAC[];
  TYPE_THEN `(e INSERT S = E) ==> (S = E DELETE e)` SUBAGOAL_TAC;
  TYPE_THEN `E` UNABBREV_TAC;
  REWRITE_TAC [DELETE_INSERT];
  ASM_MESON_TAC[INR DELETE_NON_ELEMENT];
  FIRST_ASSUM IMATCH_MP_TAC ;
  (* - *)
  TYPE_THEN `FINITE (E DELETE e)` SUBAGOAL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `E` EXISTS_TAC;
  FULL_REWRITE_TAC[rectagon];
  REWRITE_TAC[DELETE;SUBSET];
  (* - *)
  THM_INTRO_TAC[`E DELETE e`;`S`] inductive_set_endpoint;
  REWRITE_TAC[inductive_set];
  ASM_REWRITE_TAC[];
  IMATCH_MP_TAC  rectagon_2;
  CONJ_TAC;
  REWRITE_TAC[INSERT_SUBSET];
  UND 6 THEN REWRITE_TAC[SUBSET;DELETE];
  (* - *)
  CONJ_TAC;
  FULL_REWRITE_TAC[EQ_EMPTY;INSERT;];
  ASM_MESON_TAC[];
  (* -B *)
  TYPE_THEN `e INSERT S SUBSET E` SUBAGOAL_TAC;
  UND 6 THEN REWRITE_TAC[INSERT;DELETE;SUBSET];
  ASM_MESON_TAC[];
  (* - *)
  THM_INTRO_TAC[`e INSERT S`;`E`;`pointI m`] num_closure_mono;
  FULL_REWRITE_TAC[rectagon];
  TYPE_THEN `~(num_closure (e INSERT S) (pointI m) = 1)` ASM_CASES_TAC;
  TYPE_THEN `S' = e INSERT S` ABBREV_TAC ;
  KILL 15;
  FULL_REWRITE_TAC[INSERT;rectagon];
  TSPEC `m` 15;
  UND 15 THEN UND 14 THEN UND 13 THEN ARITH_TAC;
  REWR 14;
  PROOF_BY_CONTR_TAC;
  KILL 13;
  KILL 15;
  KILL 9;
  (* - *)
  TYPE_THEN `!A x. (A SUBSET E) /\ (num_closure A (pointI x) = 1) ==> (num_closure E (pointI x) = 2)` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[rectagon];
  TSPEC `x` 15;
  USE 15 (REWRITE_RULE[INSERT]);
  FIRST_ASSUM DISJ_CASES_TAC;
  THM_INTRO_TAC[`A`;`E`;`pointI x`] num_closure_mono;
  UND 20 THEN UND 19 THEN UND 9 THEN ARITH_TAC;
  (* - *)
  TYPE_THEN `endpoint (E DELETE e) SUBSET  endpoint {e}` SUBAGOAL_TAC;
  REWRITE_TAC[SUBSET;endpoint];
  UND 9 THEN DISCH_THEN (THM_INTRO_TAC[`E DELETE e`;`x`]);
  REWRITE_TAC[SUBSET;DELETE];
  THM_INTRO_TAC[`E`;`pointI x`] num_closure2;
  FULL_REWRITE_TAC[rectagon];
  REWR 15;
  THM_INTRO_TAC[`E DELETE e`;`pointI x`] num_closure1;
  REWR 17;
  USE 17 (REWRITE_RULE[DELETE]);
  THM_INTRO_TAC[`{e}`;`pointI x`] num_closure1;
  REWRITE_TAC[FINITE_SING];
  REWRITE_TAC[INR IN_SING];
  TYPE_THEN `e` EXISTS_TAC;
  IMATCH_MP_TAC  EQ_ANTISYM;
  REWRITE_TAC[];
  TYPE_THEN `e''` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  TYPE_THEN `E a /\ closure top2 a (pointI x)` SUBAGOAL_TAC;
  TYPE_THEN `E b /\ closure top2 b (pointI x)` SUBAGOAL_TAC;
  TSPEC `e` 15;
  UND 15 THEN ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC ;
  USE 15 (REWRITE_RULE[DE_MORGAN_THM]);
  COPY 17;
  TSPEC `a` 17;
  TSPEC `b` 25;
  KILL 18;
  KILL 4;
  KILL 7;
  TYPE_THEN `e' = b` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  KILL 25;
  TYPE_THEN `e' = a` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  UND 7 THEN UND 4 THEN UND 16 THEN MESON_TAC[];
  (* -C *)
  TYPE_THEN `endpoint S SUBSET endpoint {e}` SUBAGOAL_TAC;
  IMATCH_MP_TAC  SUBSET_TRANS;
  UNIFY_EXISTS_TAC;
  KILL 13;
  KILL 11;
  (* - *)
  THM_INTRO_TAC[`S`;`E`] endpoint_even;
  SUBCONJ_TAC;
  ASM_MESON_TAC[rectagon_segment];
  SUBCONJ_TAC;
  UND 12 THEN REWRITE_TAC[INSERT;SUBSET] THEN MESON_TAC[];
  THM_INTRO_TAC[`S`;`E`] rectagon_subset;
  TYPE_THEN `S` UNABBREV_TAC;
  UND 8 THEN REWRITE_TAC[];
  (* - *)
  TYPE_THEN `X = {S' | ?e. S e /\ (S' = segment_of S e)}` ABBREV_TAC ;
  TYPE_THEN `FINITE X` SUBAGOAL_TAC;
  THM_INTRO_TAC[`segment_of S`;`S`] FINITE_IMAGE;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `E DELETE e` EXISTS_TAC;
  TYPE_THEN `X = IMAGE (segment_of S) S` SUBAGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  TYPE_THEN `X` UNABBREV_TAC;
  REWRITE_TAC[IMAGE];
  ASM_REWRITE_TAC[];
  TYPE_THEN `~(X = EMPTY)` SUBAGOAL_TAC;
  USE 5 (REWRITE_RULE[EMPTY_EXISTS]);
  UND 17 THEN REWRITE_TAC[EMPTY_EXISTS];
  TYPE_THEN `segment_of S u` EXISTS_TAC;
  TYPE_THEN `X` UNABBREV_TAC;
  UNIFY_EXISTS_TAC;
  ASM_REWRITE_TAC[];
  FULL_REWRITE_TAC[HAS_SIZE];
  (* -D *)
  TYPE_THEN `edge e` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[rectagon];
  FULL_REWRITE_TAC[ISUBSET];
  THM_INTRO_TAC[`e`] endpoint_closure;
  THM_INTRO_TAC[`e`] two_endpoint;
  FULL_REWRITE_TAC[HAS_SIZE];
  (* - *)
  TYPE_THEN `endpoint S = endpoint {e}` SUBAGOAL_TAC;
  IMATCH_MP_TAC  CARD_SUBSET_LE;
  CONJ_TAC;
  ASM_MESON_TAC[];
  IMATCH_MP_TAC  (ARITH_RULE  `~(CARD X = 0) ==> 2 <= 2 * CARD X`);
  TYPE_THEN `X HAS_SIZE 0` SUBAGOAL_TAC;
  ASM_REWRITE_TAC[HAS_SIZE];
  FULL_REWRITE_TAC[HAS_SIZE_0];
  ASM_MESON_TAC[];
  (* - *)
  THM_INTRO_TAC[`e INSERT S`;`pointI m`] num_closure1;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `E` EXISTS_TAC;
  FULL_REWRITE_TAC[rectagon];
  REWR 24;
  USE 24 (REWRITE_RULE[INSERT]);
  TYPE_THEN `closure top2 e (pointI m)` ASM_CASES_TAC;
  TYPE_THEN `e' = e` SUBAGOAL_TAC;
  TSPEC `e` 24;
  ASM_MESON_TAC[];
  TYPE_THEN `e'` UNABBREV_TAC;
  TYPE_THEN `endpoint S m` SUBAGOAL_TAC;
  ASM_REWRITE_TAC[];
  THM_INTRO_TAC[`S`;`m`]endpoint_edge;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `E DELETE e` EXISTS_TAC ;
  FULL_REWRITE_TAC[EXISTS_UNIQUE_ALT];
  TSPEC  `e''` 27;
  TSPEC  `e''` 24;
  TYPE_THEN `e = e''` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `e''` UNABBREV_TAC;
  KILL 9;
  KILL 20;
  KILL 7;
  ASM_MESON_TAC[];
  (* - *)
  TYPE_THEN `~endpoint S m` SUBAGOAL_TAC;
  UND 26 THEN ASM_REWRITE_TAC[];
  (* - *)
  USE 26 (REWRITE_RULE[endpoint]);
  THM_INTRO_TAC[`S`;`E`;`pointI m`] num_closure_mono;
  FULL_REWRITE_TAC[rectagon];
  UND 6 THEN REWRITE_TAC[DELETE;SUBSET];
  TYPE_THEN `{0,2} (num_closure E (pointI m))` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[rectagon];
  TYPE_THEN `FINITE S` SUBAGOAL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET ;
  TYPE_THEN `E DELETE e` EXISTS_TAC;
  TYPE_THEN `~(num_closure S (pointI m) = 0)` SUBAGOAL_TAC;
  THM_INTRO_TAC[`S`;`pointI m`] num_closure0;
  REWR 30;
  TSPEC `e'` 30;
  COPY 24;
  TSPEC `e` 32;
  TSPEC `e'` 24;
  REWR 24;
  FIRST_ASSUM DISJ_CASES_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `e'` UNABBREV_TAC;
  KILL 4;
  KILL 9;
  ASM_MESON_TAC[];
  (* - *)
  USE 28 (REWRITE_RULE [INSERT]);
  USE 28 (MATCH_MP (TAUT `a \/ b ==> b \/ a`));
  FIRST_ASSUM DISJ_CASES_TAC;
  UND 27 THEN UND 31 THEN UND 30 THEN ARITH_TAC;
  KILL 28;
  TYPE_THEN `num_closure S (pointI m) = 2` SUBAGOAL_TAC;
  UND 31 THEN UND 30 THEN UND 26 THEN UND 27 THEN ARITH_TAC;
  KILL 31;
  KILL 9;
  KILL 4;
  KILL 7;
  KILL 30;
  (* -E *)
  THM_INTRO_TAC[`S`;`pointI m`] num_closure2;
  REWR 4;
  TYPE_THEN `S a /\ closure top2 a (pointI m)` SUBAGOAL_TAC;
  TYPE_THEN `S b /\ closure top2 b (pointI m)` SUBAGOAL_TAC;
  KILL 4;
  TYPE_THEN `e' = a` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `e' =b` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  UND 7 THEN REWRITE_TAC[];
  TYPE_THEN `e'` UNABBREV_TAC;
  ]);;
  (* }}} *)

let rectagon_adj = prove_by_refinement(
  `!E e f. (rectagon E) /\ E e /\ E f ==>
         (adj e f <=>
    (?a. endpoint (E DELETE e) a /\ (f = terminal_edge (E DELETE e) a)))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  TYPE_THEN `FINITE E` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[rectagon];
  TYPE_THEN `FINITE (E DELETE e)` SUBAGOAL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  TYPE_THEN `E` EXISTS_TAC;
  REWRITE_TAC[DELETE;SUBSET];
  (* - *)
  IMATCH_MP_TAC  EQ_ANTISYM;
  IMATCH_MP_TAC  (TAUT `A /\ b ==> b /\ A`);
  CONJ_TAC;
  IMATCH_MP_TAC closure_imp_adj;
  TYPE_THEN `a` EXISTS_TAC;
  TYPE_THEN `f` UNABBREV_TAC;
  FULL_REWRITE_TAC[endpoint];
  THM_INTRO_TAC[`E DELETE e`;`pointI a`] num_closure1;
  REWR 5;
  USE 5 (REWRITE_RULE[DELETE]);
  TYPE_THEN `{0,2} (num_closure E (pointI a))` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[rectagon];
  USE 7 (REWRITE_RULE[INSERT]);
  FIRST_ASSUM DISJ_CASES_TAC;
  THM_INTRO_TAC[`E`;`pointI a`] num_closure2;
  REWR 9;
  TYPE_THEN `E a' /\ closure top2 a' (pointI a)` SUBAGOAL_TAC;
  TYPE_THEN `E b /\ closure top2 b (pointI a)` SUBAGOAL_TAC;
  SUBCONJ_TAC;
  PROOF_BY_CONTR_TAC;
  TSPEC `e` 9;
  UND 9 THEN ASM_REWRITE_TAC[];
  PROOF_BY_CONTR_TAC;
  USE 9(REWRITE_RULE[DE_MORGAN_THM]);
  COPY 5;
  TSPEC `a'` 5;
  TSPEC `b` 17;
  TYPE_THEN `e' = b` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `e'` UNABBREV_TAC;
  ASM_MESON_TAC[];
  THM_INTRO_TAC[`E DELETE e`;`a`]terminal_endpoint;
  REWRITE_TAC[endpoint];
  UND 17 THEN REWRITE_TAC[DELETE] THEN MESON_TAC[];
  (* -- case 0 *)
  THM_INTRO_TAC[`E`;`pointI a`] num_closure0;
  REWR 9;
  ASM_MESON_TAC[];
  (* -A *)
  THM_INTRO_TAC[`e`;`f`] edge_inter;
  FULL_REWRITE_TAC[rectagon;ISUBSET];
  FULL_REWRITE_TAC[INTER;INR eq_sing];
  TYPE_THEN `m` EXISTS_TAC;
  SUBCONJ_TAC;
  REWRITE_TAC[endpoint];
  THM_INTRO_TAC[`E DELETE e`;`pointI m`] num_closure1;
  KILL 9;
  TYPE_THEN `f` EXISTS_TAC;
  REWRITE_TAC[DELETE];
  IMATCH_MP_TAC  EQ_ANTISYM;
  IMATCH_MP_TAC  (TAUT `a /\ b ==> b /\ a`);
  CONJ_TAC;
  TYPE_THEN `e''` UNABBREV_TAC;
  FULL_REWRITE_TAC[adj];
  ASM_MESON_TAC[];
  (* -- *)
  TYPE_THEN `{0, 2} (num_closure E (pointI m))` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[rectagon];
  FULL_REWRITE_TAC[INSERT];
  FIRST_ASSUM DISJ_CASES_TAC;
  THM_INTRO_TAC[`E`;`pointI m`]num_closure2;
  REWR 14;
  PROOF_BY_CONTR_TAC;
  COPY 14;
  COPY 14;
  TSPEC `e` 14;
  TSPEC `f` 18;
  TSPEC `e''` 17;
  KILL 13;
  KILL 12;
  KILL 6;
  TYPE_THEN `e'' = a` ASM_CASES_TAC ;
  TYPE_THEN `e''` UNABBREV_TAC;
  TYPE_THEN `(f = b)` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `f` UNABBREV_TAC;
  TYPE_THEN `e = b` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `e` UNABBREV_TAC;
  FULL_REWRITE_TAC[adj];
  TYPE_THEN `e'' = b` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `e''` UNABBREV_TAC;
  TYPE_THEN `f = a` SUBAGOAL_TAC;
  KILL 14;
  ASM_MESON_TAC[];
  TYPE_THEN `f` UNABBREV_TAC ;
  FULL_REWRITE_TAC[adj];
  ASM_MESON_TAC[];
  (* -- 0 case -- *)
  THM_INTRO_TAC[`E`;`pointI m`] num_closure0;
  REWR 14;
  KILL 6;
  ASM_MESON_TAC[];
  (* -B *)
  THM_INTRO_TAC[`E DELETE e`;`m`;`f`] terminal_unique;
  USE 10 (ONCE_REWRITE_RULE [EQ_SYM_EQ]);
  ASM_REWRITE_TAC[DELETE];
  ASM_MESON_TAC[adj];
  ]);;
  (* }}} *)

let rectagon_delete_end = prove_by_refinement(
  `!E e m. rectagon E /\ E e /\ closure top2 e (pointI m) ==>
       endpoint (E DELETE e ) m`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  REWRITE_TAC[endpoint];
  TYPE_THEN `FINITE E` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[rectagon];
  TYPE_THEN `FINITE (E DELETE e)` SUBAGOAL_TAC;
  IMATCH_MP_TAC  FINITE_SUBSET;
  UNIFY_EXISTS_TAC;
  REWRITE_TAC[DELETE;SUBSET];
  THM_INTRO_TAC[`E DELETE e`;`pointI m`] num_closure1;
  KILL 5;
  REWRITE_TAC[DELETE];
  (* - *)
  TYPE_THEN `{0,2} (num_closure E (pointI m))` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[rectagon];
  FULL_REWRITE_TAC[INSERT];
  (* - *)
  FIRST_ASSUM DISJ_CASES_TAC;
  KILL 5;
  THM_INTRO_TAC[`E`;`pointI m`] num_closure2;
  REWR 5;
  TYPE_THEN `(e = a) \/ (e = b)` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `?c. (E c /\ ~(c = e) /\ closure top2 c (pointI m)) /\ (!e'. E e' /\ closure top2 e' (pointI m) <=> (e' = e) \/ (e' = c))` SUBAGOAL_TAC;
  FIRST_ASSUM DISJ_CASES_TAC;
  TYPE_THEN `b` EXISTS_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `a` EXISTS_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `c` EXISTS_TAC;
  TYPE_THEN `c = e''` ASM_CASES_TAC;
  TYPE_THEN `e''` UNABBREV_TAC;
  PROOF_BY_CONTR_TAC;
  REWR 14;
  KILL 5;
  TSPEC `e''` 9;
  ASM_MESON_TAC[];
  (* - *)
  THM_INTRO_TAC[`E`;`pointI m`] num_closure0;
  REWR 7;
  ASM_MESON_TAC[];
  ]);;
  (* }}} *)

let rectagon_order = prove_by_refinement(
  `!E e m. rectagon E /\ E e /\ closure top2 e (pointI m) ==>
     (?f. BIJ f { p | p < CARD E } E /\
         (f (CARD E - 1) = e) /\ (closure top2 (f 0) (pointI m)) /\
      (!i j. (i < CARD E /\ j < CARD E) ==>
            (adj (f i) (f j) <=> ((SUC i = j) \/ (SUC j = i) \/
   ((i = 0) /\ (j = (CARD E -1))) \/ ((i = CARD E -1) /\ (j = 0))))))`,
  (* {{{ proof *)
  [
  REP_BASIC_TAC;
  THM_INTRO_TAC[`E`;`e`] rectagon_delete;
  TYPE_THEN `FINITE E` SUBAGOAL_TAC;
  FULL_REWRITE_TAC[rectagon];
  TYPE_THEN `FINITE (E DELETE e)` SUBAGOAL_TAC;
  IMATCH_MP_TAC   FINITE_SUBSET;
  UNIFY_EXISTS_TAC;
  REWRITE_TAC[DELETE;SUBSET];
  TYPE_THEN `endpoint (E DELETE e) m` SUBAGOAL_TAC;
  IMATCH_MP_TAC  rectagon_delete_end;
  (* - *)
  TYPE_THEN `?n. (endpoint (E DELETE e) n) /\ ~(n = m)` SUBAGOAL_TAC;
  THM_INTRO_TAC[`E DELETE e`] endpoint_size2;
  FULL_REWRITE_TAC[has_size2];
  TYPE_THEN `m = a` ASM_CASES_TAC ;
  TYPE_THEN `b` EXISTS_TAC;
  REWRITE_TAC[INR in_pair];
  TYPE_THEN `a` EXISTS_TAC;
  REWRITE_TAC[INR in_pair];
  (* - *)
  THM_INTRO_TAC[`E DELETE e`;`m`;`n`] psegment_order;
  THM_INTRO_TAC[`e`;`E`;] CARD_SUC_DELETE;
  TYPE_THEN `~(CARD E = 0)` SUBAGOAL_TAC;
  TYPE_THEN `E HAS_SIZE 0` SUBAGOAL_TAC;
  REWRITE_TAC[HAS_SIZE];
  FULL_REWRITE_TAC[HAS_SIZE_0;EQ_EMPTY];
  ASM_MESON_TAC[];
  TYPE_THEN `CARD (E DELETE e) = CARD (E) - 1` SUBAGOAL_TAC;
  UND 14 THEN UND 13 THEN ARITH_TAC;
  (* - *)
  TYPE_THEN `g = \ (i:num). if (i < CARD E - 1) then f i else e` ABBREV_TAC ;
  TYPE_THEN `(g (CARD E - 1) = e)` SUBAGOAL_TAC;
  TYPE_THEN `g` UNABBREV_TAC;
  REWRITE_TAC[ARITH_RULE `~(x <| x)`];
  TYPE_THEN `(!i. (i < CARD E -| 1) ==> (g i = f i))` SUBAGOAL_TAC;
  TYPE_THEN `g` UNABBREV_TAC;
  KILL 16;
  TYPE_THEN `g` EXISTS_TAC;
  (* -A *)
  TYPE_THEN `{p | p < CARD E - 1} UNION {(CARD E - 1)} = {p | p <| CARD E}` SUBAGOAL_TAC;
  IMATCH_MP_TAC  EQ_EXT;
  REWRITE_TAC[UNION;INR IN_SING ];
  UND 14 THEN ARITH_TAC;
  (* - *)
  SUBCONJ_TAC;
  REWRITE_TAC[BIJ];
  SUBCONJ_TAC;
  USE 16 (SYM);
  IMATCH_MP_TAC  inj_split;
  CONJ_TAC;
  FULL_REWRITE_TAC[BIJ;INJ];
  TYPE_THEN `CARD (E DELETE e)` UNABBREV_TAC;
  CONJ_TAC;
  UND 20 THEN REWRITE_TAC[DELETE] THEN UND 15 THEN MESON_TAC[];
  FIRST_ASSUM IMATCH_MP_TAC ;
  UND 15 THEN UND 21 THEN UND 22 THEN UND 18 THEN MESON_TAC[];
  CONJ_TAC;
  REWRITE_TAC[INJ;INR IN_SING ];
  ASM_REWRITE_TAC[];
  REWRITE_TAC[IMAGE;INTER;EQ_EMPTY;INR IN_SING  ];
  TYPE_THEN `x` UNABBREV_TAC ;
  TYPE_THEN `x''` UNABBREV_TAC;
  REWR 19;
  TYPE_THEN `g x' = f x'` SUBAGOAL_TAC;
  ASM_MESON_TAC[];
  TYPE_THEN `g x'` UNABBREV_TAC;
  FULL_REWRITE_TAC[BIJ;INJ];
  TYPE_THEN `CARD(E DELETE e)` UNABBREV_TAC;
  USE 21(REWRITE_RULE[DELETE]);
  ASM_MESON_TAC[];
  (* -- SURJ -- *)
  REWRITE_TAC[SURJ];
  USE 19 (REWRITE_RULE[INJ]);
  REWRITE_TAC[];
  TYPE_THEN `x = e` ASM_CASES_TAC;
  TYPE_THEN `CARD E - 1` EXISTS_TAC;
  UND 14 THEN ARITH_TAC;
  TYPE_THEN `(E DELETE e) x` SUBAGOAL_TAC;
  ASM_REWRITE_TAC[DELETE];
  FULL_REWRITE_TAC[BIJ;SURJ];
  TSPEC `x` 12;
  REWR 12;
  T