(* ========================================================================= *)
(* Abstract type of theorems and primitive inference rules.                  *)
(*                                                                           *)
(*       John Harrison, University of Cambridge Computer Laboratory          *)
(*                                                                           *)
(*            (c) Copyright, University of Cambridge 1998                    *)
(* ========================================================================= *)

(* ------------------------------------------------------------------------- *)
(* A few bits of general derived syntax.                                     *)
(* ------------------------------------------------------------------------- *)

let rator tm =
  try fst(dest_comb tm)
  with Failure _ -> failwith "rator: Not a combination";;

let rand tm =
  try snd(dest_comb tm)
  with Failure _ -> failwith "rand: Not a combination";;

(* ------------------------------------------------------------------------- *)
(* Syntax operations for equations.                                          *)
(* ------------------------------------------------------------------------- *)

let mk_eq =
  let eq = mk_const("=",[]) in
  fun (l,r) ->
    try let ty = type_of l in
        let eq_tm = inst [ty,aty] eq in
        mk_comb(mk_comb(eq_tm,l),r)
    with Failure _ -> failwith "mk_eq";;

let dest_eq tm =
  match tm with
    Comb(Comb(Const("=",_),l),r) -> l,r
  | _ -> failwith "dest_eq";;

let is_eq tm =
  match tm with
    Comb(Comb(Const("=",_),_),_) -> true
  | _ -> false

(* ------------------------------------------------------------------------- *)
(* Useful to have term union modulo alpha-conversion for assumption lists.   *)
(* ------------------------------------------------------------------------- *)

let rec term_union l1 l2 =
  match l1 with
    [] -> l2
  | (h::t) -> let subun = term_union t l2 in
              if exists (aconv h) subun then subun else h::subun;;

(* ------------------------------------------------------------------------- *)
(* The abstract type of theorems.                                            *)
(* ------------------------------------------------------------------------- *)

module type Hol_thm_primitives =
  sig type thm
  val dest_thm : thm -> term list * term
  val hyp : thm -> term list
  val concl : thm -> term
  val REFL : term -> thm
  val TRANS : thm -> thm -> thm
  val MK_COMB : thm * thm -> thm
  val ABS : term -> thm -> thm
  val BETA : term -> thm
  val ASSUME : term -> thm
  val EQ_MP : thm -> thm -> thm
  val DEDUCT_ANTISYM_RULE : thm -> thm -> thm
  val INST_TYPE : (hol_type * hol_type) list -> thm -> thm
  val INST : (term * term) list -> thm -> thm
  val axioms : unit -> thm list
  val new_axiom : term -> thm
  val new_basic_definition : term -> thm
  val new_basic_type_definition : string -> string * string -> thm -> thm * thm

  val equals_thm : thm -> thm -> bool
  val le_thm : thm -> thm -> bool
  val less_thm : thm -> thm -> bool

  val proof_of : thm -> proof
  val substitute_proof : thm -> proof -> thm
  val save_thm : string -> thm -> thm
end;;

(* ------------------------------------------------------------------------- *)
(* This is the implementation of those primitives.                           *)
(* ------------------------------------------------------------------------- *)

module Hol : Hol_thm_primitives = struct

  type thm = Sequent of (term list * term * proof)

(* ------------------------------------------------------------------------- *)
(* Basic theorem destructors.                                                *)
(* ------------------------------------------------------------------------- *)

  let dest_thm (Sequent(asl,c,_)) = (asl,c)

  let hyp (Sequent(asl,c,_)) = asl

  let concl (Sequent(asl,c,_)) = c

(* ------------------------------------------------------------------------- *)
(* Basic equality properties; TRANS is derivable but included for efficiency *)
(* ------------------------------------------------------------------------- *)

  let REFL tm =
    Sequent([],mk_eq(tm,tm),proof_REFL tm)

  let TRANS (Sequent(asl1,c1,p1)) (Sequent(asl2,c2,p2)) =
    try let l,m1 = dest_eq c1 in
        let m2,r = dest_eq c2 in
        if aconv m1 m2 then Sequent(term_union asl1 asl2,mk_eq(l,r),proof_TRANS (p1,p2)) else fail()
    with Failure _ -> failwith "TRANS"

(* ------------------------------------------------------------------------- *)
(* Congruence properties of equality.                                        *)
(* ------------------------------------------------------------------------- *)

  let MK_COMB(Sequent(asl1,c1,p1),Sequent(asl2,c2,p2)) =
    try let l1,r1 = dest_eq c1
        and l2,r2 = dest_eq c2 in
        Sequent(term_union asl1 asl2,mk_eq(mk_comb(l1,l2),mk_comb(r1,r2)),proof_MK_COMB (p1,p2))
    with Failure _ -> failwith "MK_COMB"


  let ABS v (Sequent(asl,c,p)) =
    let l,r = try dest_eq c with Failure _ -> failwith "ABS: not an equation" in
    if exists (vfree_in v) asl
    then failwith "ABS: variable is free in assumptions"
    else Sequent(asl,mk_eq(mk_abs(v,l),mk_abs(v,r)),proof_ABS v p)

(* ------------------------------------------------------------------------- *)
(* Trivial case of lambda calculus beta-conversion.                          *)
(* ------------------------------------------------------------------------- *)

  let BETA tm =
    try let f,arg = dest_comb tm in
        let v,bod = dest_abs f in
        if arg = v then Sequent([],mk_eq(tm,bod), proof_BETA tm) else fail()
    with Failure _ -> failwith "BETA: not a trivial beta-redex"

(* ------------------------------------------------------------------------- *)
(* Rules connected with deduction.                                           *)
(* ------------------------------------------------------------------------- *)

  let ASSUME tm =
    if type_of tm = bool_ty then Sequent([tm],tm, proof_ASSUME tm)
    else failwith "ASSUME: not a proposition"

  let EQ_MP (Sequent(asl1,eq,p1) as th1) (Sequent(asl2,c,p2) as th2) =
    try let l,r = dest_eq eq in
        if aconv l c then Sequent(term_union asl1 asl2,r, proof_EQ_MP p1 p2) else fail()
    with Failure _ -> failwith "EQ_MP"

  let DEDUCT_ANTISYM_RULE (Sequent(asl1,c1,p1)) (Sequent(asl2,c2,p2)) =
    let asl1' = filter (not o aconv c2) asl1
    and asl2' = filter (not o aconv c1) asl2 in
    Sequent(term_union asl1' asl2',mk_eq(c1,c2),
            proof_DEDUCT_ANTISYM_RULE (p1,c1) (p2,c2))

(* ------------------------------------------------------------------------- *)
(* Type and term instantiation.                                              *)
(* ------------------------------------------------------------------------- *)

  let INST_TYPE theta (Sequent(asl,c,p) as th) =
    let inst_fn = inst theta in
    Sequent(map inst_fn asl,inst_fn c, proof_INST_TYPE theta p);;

  let INST theta (Sequent(asl,c,p)) =
    let inst_fun = vsubst theta in
    Sequent(map inst_fun asl,inst_fun c, proof_INST theta p);;

(* ------------------------------------------------------------------------- *)
(* Handling of axioms.                                                       *)
(* ------------------------------------------------------------------------- *)

  let the_axioms = ref ([]:thm list)

  let axioms() = !the_axioms

  let new_axiom tm =
    if fst(dest_type(type_of tm)) = "bool" then
      let axname = new_axiom_name "" in
      let p = proof_new_axiom (axname) tm in
      let th = Sequent([],tm,p) in
       (the_axioms := th::(!the_axioms);
        save_proof axname p (Some tm);
        th)
    else failwith "new_axiom: Not a proposition";;

(* ------------------------------------------------------------------------- *)
(* Handling of (term) definitions.                                           *)
(* ------------------------------------------------------------------------- *)

  let new_basic_definition tm =
    let l,r = dest_eq tm in
    let cname,ty = dest_var l in
    if not (freesin [] r) then failwith "new_definition: term not closed" else
    if not (subset (type_vars_in_term r) (tyvars ty))
    then failwith "new_definition: Type variables not reflected in constant"
    else
      let c = new_constant(cname,ty); mk_const(cname,[]) in
      let p = proof_new_definition cname r in
      let concl = mk_eq(c,r) in
      let _ = save_proof ("DEF_"^cname) p (Some concl) in
        (Sequent([], concl, p))

(* ------------------------------------------------------------------------- *)
(* Handling of type definitions.                                             *)
(*                                                                           *)
(* This function now involves no logical constants beyond equality.          *)
(*                                                                           *)
(*             |- P t                                                        *)
(*    ---------------------------                                            *)
(*        |- abs(rep a) = a                                                  *)
(*     |- P r = (rep(abs r) = r)                                             *)
(*                                                                           *)
(* Where "abs" and "rep" are new constants with the nominated names.         *)
(* ------------------------------------------------------------------------- *)

  let new_basic_type_definition
        tyname ((absname,repname) as binames) (Sequent (asl,c,p) as th) =
    if exists (can get_const_type) [absname; repname] then
      failwith "new_basic_type_definition: Constant(s) already in use" else
    if not (asl = []) then
      failwith "new_basic_type_definition: Assumptions in theorem" else
    let P,x = try dest_comb c
              with Failure _ ->
                failwith "new_basic_type_definition: Not a combination" in
    if not (freesin [] P) then
      failwith "new_basic_type_definition: Predicate is not closed" else
    let tyvars = sort (<=) (type_vars_in_term P) in
    let _ = try new_type(tyname,length tyvars)
            with Failure _ ->
                failwith "new_basic_type_definition: Type already defined" in
    let aty = mk_type(tyname, tyvars)
    and rty = type_of x in
    let abs = new_constant(absname,mk_fun_ty rty aty); mk_const(absname,[])
    and rep = new_constant(repname,mk_fun_ty aty rty); mk_const(repname,[]) in
    let a = mk_var("a",aty) and r = mk_var("r",rty) in
    let ax1 = mk_eq(mk_comb(abs,mk_comb(rep,a)),a) in
    let ax2 = mk_eq(mk_comb(P,r),mk_eq(mk_comb(rep,mk_comb(abs,r)),r)) in
    let tp = proof_new_basic_type_definition tyname (absname, repname) (P,x) p in
    let tname = "TYDEF_"^tyname in
    let retval =
      Sequent([], ax1, proof_CONJUNCT1 tp),
      Sequent([], ax2, proof_CONJUNCT2 tp)
    in
    save_proof tname tp None;
    retval;;

(* ------------------------------------------------------------------------- *)
(* Dealing with proof objects.                                               *)
(* ------------------------------------------------------------------------- *)

  let substitute_proof =
    if use_extended_proofobjects then
      fun (Sequent (asl, c, p)) pnew -> Sequent (asl, c, pnew)
    else
      fun th p -> th;;

  let equals_thm (Sequent (p1,c1,_)) (Sequent (p2,c2,_)) =
    (p1 = p2) & (c1 = c2)

  let le_thm (Sequent (p1,c1,_)) (Sequent (p2,c2,_)) = (p1, c1) <= (p2, c2)

  let less_thm (Sequent (p1, c1,_)) (Sequent (p2, c2,_)) = (p1, c1) < (p2, c2)

  let proof_of (Sequent(_,_,p)) = p

  let save_thm name th =
    (save_proof name (proof_of th) (Some (concl th)); th)

end;;

include Hol;;
