(* ========================================================================= *)
(* Cross products in real^3.                                                 *)
(* ========================================================================= *)

needs "Multivariate/determinants.ml";;

prioritize_vector();;

(* ------------------------------------------------------------------------- *)
(* The definition.                                                           *)
(* ------------------------------------------------------------------------- *)

parse_as_infix("cross",(20,"right"));;

let cross = new_definition
 `(a:real^3) cross (b:real^3) =
    vector [a$2 * b$3 - a$3 * b$2;
            a$3 * b$1 - a$1 * b$3;
            a$1 * b$2 - a$2 * b$1] :real^3`;;

(* ------------------------------------------------------------------------- *)
(* Some simple automation.                                                   *)
(* ------------------------------------------------------------------------- *)

let VEC3_TAC =
  SIMP_TAC[CART_EQ; LAMBDA_BETA; FORALL_3; SUM_3; DIMINDEX_3; VECTOR_3;
           vector_add; vec; dot; cross; orthogonal; basis; DET_3;
           vector_neg; vector_sub; vector_mul; ARITH] THEN
  CONV_TAC REAL_RING;;

let VEC3_RULE tm = prove(tm,VEC3_TAC);;

(* ------------------------------------------------------------------------- *)
(* Basic lemmas.                                                             *)
(* ------------------------------------------------------------------------- *)

let ORTHOGONAL_CROSS = prove
 (`!x y. orthogonal (x cross y) x /\ orthogonal (x cross y) y /\
         orthogonal x (x cross y) /\ orthogonal y (x cross y)`,
  VEC3_TAC);;

let CROSS_LZERO = prove
 (`!x. (vec 0) cross x = vec 0`,
  VEC3_TAC);;

let CROSS_RZERO = prove
 (`!x. x cross (vec 0) = vec 0`,
  VEC3_TAC);;

let CROSS_SKEW = prove
 (`!x y. (x cross y) = --(y cross x)`,
  VEC3_TAC);;

let CROSS_REFL = prove
 (`!x. x cross x = vec 0`,
  VEC3_TAC);;

let CROSS_LADD = prove
 (`!x y z. (x + y) cross z = (x cross z) + (y cross z)`,
  VEC3_TAC);;

let CROSS_RADD = prove
 (`!x y z. x cross (y + z) = (x cross y) + (x cross z)`,
  VEC3_TAC);;

let CROSS_LMUL = prove
 (`!c x y. (c % x) cross y = c % (x cross y)`,
  VEC3_TAC);;

let CROSS_RMUL = prove
 (`!c x y. x cross (c % y) = c % (x cross y)`,
  VEC3_TAC);;

let CROSS_LNEG = prove
 (`!x y. (--x) cross y = --(x cross y)`,
  VEC3_TAC);;

let CROSS_RNEG = prove
 (`!x y. x cross (--y) = --(x cross y)`,
  VEC3_TAC);;

let CROSS_JACOBI = prove
 (`!x y z.
    x cross (y cross z) + y cross (z cross x) + z cross (x cross y) = vec 0`,
  VEC3_TAC);;

let CROSS_LAGRANGE = prove
 (`!x y z. x cross (y cross z) = (x dot z) % y - (x dot y) % z`,
  VEC3_TAC);;

let CROSS_TRIPLE = prove
 (`!x y z.  (x cross y) dot z = (y cross z) dot x`,
  VEC3_TAC);;

let DOT_CROSS_SELF = prove
 (`(!x y. x dot (x cross y) = &0) /\
   (!x y. x dot (y cross x) = &0) /\
   (!x y. (x cross y) dot y = &0) /\
   (!x y. (y cross x) dot y = &0)`,
  VEC3_TAC);;

let CROSS_COMPONENTS = prove
 (`!x y. (x cross y)$1 = x$2 * y$3 - y$2 * x$3 /\
         (x cross y)$2 = x$3 * y$1 - y$3 * x$1 /\
         (x cross y)$3 = x$1 * y$2 - y$1 * x$2`,
  VEC3_TAC);;

let CROSS_BASIS = prove
 (`(basis 1) cross (basis 2) = basis 3 /\
   (basis 2) cross (basis 1) = --(basis 3) /\
   (basis 2) cross (basis 3) = basis 1 /\
   (basis 3) cross (basis 2) = --(basis 1) /\
   (basis 3) cross (basis 1) = basis 2 /\
   (basis 1) cross (basis 3) = --(basis 2)`,
  VEC3_TAC);;

let CROSS_BASIS_NONZERO = prove
 (`!u. ~(u = vec 0)
       ==> ~(u cross basis 1 = vec 0) \/
           ~(u cross basis 2 = vec 0) \/
           ~(u cross basis 3 = vec 0)`,
  VEC3_TAC);;

let CROSS_DOT_CANCEL = prove
 (`!x y z.
     x dot y = x dot z /\ x cross y = x cross z /\ ~(x = vec 0) ==> y = z`,
  ONCE_REWRITE_TAC[GSYM VECTOR_SUB_EQ] THEN REWRITE_TAC[GSYM DOT_EQ_0] THEN
  VEC3_TAC);;

let NORM_CROSS_DOT = prove
 (`!x y. norm(x cross y) pow 2 + (x dot y) pow 2 = (norm(x) * norm y) pow 2`,
  REWRITE_TAC[REAL_POW_MUL; NORM_POW_2] THEN VEC3_TAC);;

let DOT_CROSS_DET = prove
 (`!x y z. x dot (y cross z) = det(vector[x;y;z]:real^3^3)`,
  VEC3_TAC);;

let CROSS_CROSS_DET = prove
 (`!w x y z. (w cross x) cross (y cross z) =
                det(vector[w;x;z]:real^3^3) % y -
                det(vector[w;x;y]:real^3^3) % z`,
  VEC3_TAC);;
