(* ------------------------------------------------------------------------- *)
(* Translate conjunction of polynmial equations to Maple syntax.             *)
(* ------------------------------------------------------------------------- *)

let rec maple_poly varindex tm =
  try let l,r = dest_add tm in
      (maple_poly varindex l)^" + "^(maple_poly varindex r)
  with Failure _ -> try
      let l,r = dest_mul tm in
      (maple_poly varindex l)^"*"^(maple_poly varindex r)
  with Failure _ -> try
      let l,r = dest_sub tm in
      (maple_poly varindex l)^" - "^(maple_poly varindex r)
  with Failure _ -> try
      let l,r = dest_pow tm in
      (maple_poly varindex l)^"^"^(string_of_int(dest_small_numeral r))
  with Failure _ -> try
      let x = dest_complex_const tm in string_of_num x
  with Failure _ ->
      assoc tm varindex;;

let maple_polys tm =
  let fvs = frees tm in
  let args = make_args "x" [] (map type_of fvs) in
  let varindex = zip fvs (map (fst o dest_var) args) in
  let cjs = conjuncts tm in
  let strs = map (maple_poly varindex o lhs) cjs in
  map snd varindex,end_itlist (fun x y -> x^",\n"^y) strs ^"\n\n";;

(* ------------------------------------------------------------------------- *)
(* Example.                                                                  *)
(* ------------------------------------------------------------------------- *)

let tm =  `((p1 - p1'''''') * (p2'''''' - p2'''') -
   (p2 - p2'''''') * (p1'''''' - p1'''') =
   Cx (&0)) /\
  ((p1''' - p1'''''') * (p2'''''' - p2') -
   (p2''' - p2'''''') * (p1'''''' - p1') =
   Cx (&0)) /\
  ((p1 - p1''''''') * (p2''''''' - p2''''') -
   (p2 - p2''''''') * (p1''''''' - p1''''') =
   Cx (&0)) /\
  ((p1''' - p1''''''') * (p2''''''' - p2'') -
   (p2''' - p2''''''') * (p1''''''' - p1'') =
   Cx (&0)) /\
  ((p1' - p1'''''''') * (p2'''''''' - p2''''') -
   (p2' - p2'''''''') * (p1'''''''' - p1''''') =
   Cx (&0)) /\
  ((p1'''' - p1'''''''') * (p2'''''''' - p2'') -
   (p2'''' - p2'''''''') * (p1'''''''' - p1'') =
   Cx (&0)) /\
  (w * (p1 - p1') + z * (p2 - p2') + Cx (&1) = Cx (&0)) /\
  (w' * (p1''' - p1'''') + z' * (p2''' - p2'''') + Cx (&1) = Cx (&0)) /\
  (w'' * (p1 - p1'') + z'' * (p2 - p2'') + Cx (&1) = Cx (&0)) /\
  (w''' * (p1''' - p1''''') + z''' * (p2''' - p2''''') + Cx (&1) = Cx (&0)) /\
  (w'''' * (p1' - p1'') + z'''' * (p2' - p2'') + Cx (&1) = Cx (&0)) /\
  (w''''' * (p1'''' - p1''''') + z''''' * (p2'''' - p2''''') + Cx (&1) =
   Cx (&0)) /\
  (z'''''' *
   ((p1'''''' - p1''''''') * (p2''''''' - p2'''''''') -
    (p2'''''' - p2''''''') * (p1''''''' - p1'''''''')) +
   Cx (&1) =
   Cx (&0))`;;

let vars,pstr = maple_polys tm;;

print_string pstr;;

do the following in Maple;;

F := [x5 - x13*x16 - x17 - x7 - x16*x13 - x14,
x9 - x19*x22 - x23 - x11 - x22*x19 - x20,
x5 - x25*x27 - x22 - x7 - x27*x25 - x19,
x9 - x25*x27 - x16 - x11 - x27*x25 - x13,
x5 - x29*x28 - x23 - x7 - x28*x29 - x20,
x9 - x29*x28 - x17 - x11 - x28*x29 - x14,
x13 - x30*x26 - x23 - x16 - x26*x30 - x20,
x19 - x30*x26 - x17 - x22 - x26*x30 - x14,
x0*x5 - x13 + x1*x7 - x16 + 1,
x2*x9 - x19 + x3*x11 - x22 + 1,
x4*x5 - x14 + x6*x7 - x17 + 1,
x8*x9 - x20 + x10*x11 - x23 + 1,
x12*x13 - x14 + x15*x16 - x17 + 1,
x18*x19 - x20 + x21*x22 - x23 + 1,
x24*x25 - x29*x28 - x26 - x27 - x28*x29 - x30 + 1];

X :=  [x0, x1, x2, x3, x4, x5, x6, x7, x8, x9, x10, x11,
       x12, x13, x14, x15, x16, x17, x18, x19, x20, x21,
       x22, x23, x24, x25, x26, x27, x28, x29, x30];

grobner['gbasis']( F, X, 'plex' );

(* ------------------------------------------------------------------------- *)
(* Also test the formula simplifications.                                    *)
(* ------------------------------------------------------------------------- *)

simplify
   ((x1^2 + x2^2 + x3^2 + x4^2)^4 -
   ((1 / 840) * ((x1 + x2 + x3 + x4)^8 +
                     (x1 + x2 + x3 - x4)^8 +
                     (x1 + x2 - x3 + x4)^8 +
                     (x1 + x2 - x3 - x4)^8 +
                     (x1 - x2 + x3 + x4)^8 +
                     (x1 - x2 + x3 - x4)^8 +
                     (x1 - x2 - x3 + x4)^8 +
                     (x1 - x2 - x3 - x4)^8) +
    (1 / 5040) * (((2) * x1 + x2 + x3)^8 +
                      ((2) * x1 + x2 - x3)^8 +
                      ((2) * x1 - x2 + x3)^8 +
                      ((2) * x1 - x2 - x3)^8 +
                      ((2) * x1 + x2 + x4)^8 +
                      ((2) * x1 + x2 - x4)^8 +
                      ((2) * x1 - x2 + x4)^8 +
                      ((2) * x1 - x2 - x4)^8 +
                      ((2) * x1 + x3 + x4)^8 +
                      ((2) * x1 + x3 - x4)^8 +
                      ((2) * x1 - x3 + x4)^8 +
                      ((2) * x1 - x3 - x4)^8 +
                      ((2) * x2 + x3 + x4)^8 +
                      ((2) * x2 + x3 - x4)^8 +
                      ((2) * x2 - x3 + x4)^8 +
                      ((2) * x2 - x3 - x4)^8) +
     (1 / 84) * ((x1 + x2)^8 + (x1 - x2)^8 +
                     (x1 + x3)^8 + (x1 - x3)^8 +
                     (x1 + x4)^8 + (x1 - x4)^8 +
                     (x2 + x3)^8 + (x2 - x3)^8 +
                     (x2 + x4)^8 + (x2 - x4)^8 +
                     (x3 + x4)^8 + (x3 - x4)^8) +
     (1 / 840) * (((2) * x1)^8 + ((2) * x2)^8 +
                      ((2) * x3)^8 + ((2) * x4)^8)));
