package uk.ac.cam.cl.gfxintro.gd355.tick2;

import org.joml.Matrix4f;
import org.joml.Vector3f;
import org.joml.Vector4f;
import org.lwjgl.BufferUtils;
import org.lwjgl.glfw.*;
import org.lwjgl.opengl.GL11;
import org.lwjgl.opengl.GL13;

import javax.imageio.ImageIO;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.nio.ByteBuffer;
import java.nio.FloatBuffer;
import java.nio.IntBuffer;
import java.util.Random;

import static org.lwjgl.glfw.GLFW.*;
import static org.lwjgl.opengl.GL.createCapabilities;
import static org.lwjgl.opengl.GL11.*;
import static org.lwjgl.opengl.GL15.*;
import static org.lwjgl.opengl.GL20.*;
import static org.lwjgl.opengl.GL30.*;
import static org.lwjgl.system.MemoryUtil.NULL;


/***
 * Class for an OpenGL Window with rendering loop and meshes to draw
 *
 */
public class OpenGLApplication {

	private static final float FOV_Y = (float) Math.toRadians(50);
	protected static int WIDTH = 800, HEIGHT = 600;
	private Camera camera;
	private long window;
	
	private long currentTime;


	// Callbacks for input handling
	private GLFWCursorPosCallback cursor_cb;
	private GLFWScrollCallback scroll_cb;
	private GLFWKeyCallback key_cb;


	private String heightmapFilename;
	private Mesh terrain;
	
	private Mesh water;
	
	private Cube[] cubes;
	private Vector3f[] cubeVelocities;
	private float[] cubeCols; 

	private RenderTarget reflectionTarget;

	public OpenGLApplication(String heightmapFilename) {
		this.heightmapFilename = heightmapFilename;
	}


	/***
	 * Initialise OpenGL and the scene
	 * @throws Exception
	 */
	public void initialize() throws Exception {

		if (glfwInit() != true)
			throw new RuntimeException("Unable to initialize the graphics runtime.");

		glfwWindowHint(GLFW_RESIZABLE, GLFW_TRUE);

		// Ensure that the right version of OpenGL is used (at least 3.2)
		glfwWindowHint(GLFW_CONTEXT_VERSION_MAJOR, 3);
		glfwWindowHint(GLFW_CONTEXT_VERSION_MINOR, 2);
		glfwWindowHint(GLFW_OPENGL_PROFILE, GLFW_OPENGL_CORE_PROFILE); // Use CORE OpenGL profile without depreciated functions
		glfwWindowHint(GLFW_OPENGL_FORWARD_COMPAT, GL_TRUE); // Make it forward compatible

		window = glfwCreateWindow(WIDTH, HEIGHT, "Tick 3", NULL, NULL);
		if (window == NULL)
			throw new RuntimeException("Failed to create the application window.");

		GLFWVidMode mode = glfwGetVideoMode(glfwGetPrimaryMonitor());
		glfwSetWindowPos(window, (mode.width() - WIDTH) / 2, (mode.height() - HEIGHT) / 2);
		glfwMakeContextCurrent(window);
		createCapabilities();

		// Enable v-sync
		glfwSwapInterval(1);

		// Cull back-faces of polygons
		glEnable(GL_CULL_FACE);
		glCullFace(GL_BACK);

		// Do depth comparisons when rendering
		glEnable(GL_DEPTH_TEST);

		// Create camera, and setup input handlers
		camera = new Camera((double) WIDTH / HEIGHT, FOV_Y);
		initializeInputs();
		
		
		
		// This is where we are creating the meshes
		Texture terrainTexture = new Texture();
		terrainTexture.load( "resources/texture.png");
		terrain = new Terrain(heightmapFilename, terrainTexture);
		terrain.iniatialise();
		terrain.setPosition(new Vector3f(0, -1.2f, 0));

		cubes = new Cube[150];
		cubeVelocities = new Vector3f[cubes.length];
		cubeCols = new float[cubes.length];
		Random ran = new Random(); 
		for (int i = 0; i < cubes.length; i++) {
			
			cubes[i] = new Cube(new Vector3f(1.0f, 0.8f, 0.6f));
			cubes[i].iniatialise();
			cubes[i].setScale(new Vector3f(0.1f, 0.1f, 0.1f));
			cubes[i].setPosition(new Vector3f(ran.nextFloat(), 3 + ran.nextFloat(), ran.nextFloat()));
			cubeVelocities[i] = new Vector3f(ran.nextFloat() * 0.1f - 0.05f, 0.1f + ran.nextFloat() * 0.1f, ran.nextFloat() * 0.1f - 0.05f);
			cubeCols[i] = ran.nextFloat() * 0.1f;
		}
	
		
		// water with reflection
		reflectionTarget = new RenderTarget(WIDTH, HEIGHT); // this is the target where we draw the reflection
		water = new Water(reflectionTarget.getTargetColourTexture());
		water.iniatialise();
		water.setPosition(new Vector3f(0, 0, 0));
		water.setScale(new Vector3f(50, 50, 50));

		currentTime = System.currentTimeMillis();
	}

	private void initializeInputs() {

		// Callback for: when dragging the mouse, rotate the camera
		cursor_cb = new GLFWCursorPosCallback() {
			private double prevMouseX, prevMouseY;

			public void invoke(long window, double mouseX, double mouseY) {
				boolean dragging = glfwGetMouseButton(window, GLFW_MOUSE_BUTTON_LEFT) == GLFW_PRESS;
				if (dragging) {
					camera.rotate(mouseX - prevMouseX, mouseY - prevMouseY);
				}
				prevMouseX = mouseX;
				prevMouseY = mouseY;
			}
		};

		// Callback for: when scrolling, zoom the camera
		scroll_cb = new GLFWScrollCallback() {
			public void invoke(long window, double dx, double dy) {
				camera.zoom(dy > 0);
			}
		};

		// Callback for keyboard controls: "W" - wireframe, "P" - points, "S" - take screenshot
		key_cb = new GLFWKeyCallback() {
			public void invoke(long window, int key, int scancode, int action, int mods) {
				if (key == GLFW_KEY_W && action == GLFW_PRESS) {
					glPolygonMode(GL_FRONT_AND_BACK, GL_LINE);
					glDisable(GL_CULL_FACE);
				} else if (key == GLFW_KEY_P && action == GLFW_PRESS) {
					glPolygonMode(GL_FRONT_AND_BACK, GL_POINT);
				} else if (key == GLFW_KEY_S && action == GLFW_RELEASE) {
					takeScreenshot("screenshot.png");
				} else if (action == GLFW_RELEASE) {
					glEnable(GL_CULL_FACE);
					glPolygonMode(GL_FRONT_AND_BACK, GL_FILL);
				}
			}
		};

		GLFWFramebufferSizeCallback fbs_cb = new GLFWFramebufferSizeCallback() {
			@Override
			public void invoke(long window, int width, int height) {
				glViewport( 0, 0, width, height );
				camera.setAspectRatio( width * 1.f/height );
			}
		};

		// Set callbacks on the window
		glfwSetCursorPosCallback(window, cursor_cb);
		glfwSetScrollCallback(window, scroll_cb);
		glfwSetKeyCallback(window, key_cb);
		glfwSetFramebufferSizeCallback(window, fbs_cb);
	}

	/***
	 * Run loop
	 * @throws Exception
	 */
	public void run() throws Exception {

		initialize();

		while (glfwWindowShouldClose(window) != true) {
			
			long newTime = System.currentTimeMillis();
			updateScene((newTime - currentTime) / 1000.f);
			currentTime = newTime;
			render();
		}
	}
	
	/***
	 * Update animation
	 * @param dt
	 */
	public void updateScene(float dt) {
		Random ran = new Random(); 
		for (int i = 0; i < cubes.length; i++) {
			cubes[i].setPosition(cubes[i].getPosition().add(cubeVelocities[i]));
			cubeVelocities[i] = cubeVelocities[i].add(new Vector3f(0, -dt * 0.0981f, 0));
			
			if (cubes[i].getPosition().y < -10) {
				cubes[i].setPosition(new Vector3f(ran.nextFloat(), 3 + ran.nextFloat(), ran.nextFloat()));
				cubeVelocities[i] = new Vector3f(ran.nextFloat() * 0.1f - 0.05f, 0.1f + ran.nextFloat() * 0.1f, ran.nextFloat() * 0.1f - 0.05f);
				cubeCols[i] = ran.nextFloat() * 0.1f;
			}
			
			cubes[i].setColour(new Vector3f(0.9f, cubeCols[i], cubeCols[i]));
			cubeCols[i] = Float.min(cubeCols[i] + dt * 0.2f, 1.f);
		}
	}

	/***
	 * Draw the scene
	 */
	public void render() {
		
		// render the reflection first
		reflectionTarget.startUsing();
		camera.setReflected();
		camera.setClipPlane(new Vector4f(0, 1, 0, 0));
		glEnable(GL_CLIP_DISTANCE0);
		glClearColor(1.0f, 1.0f, 1.0f, 1.0f); // Set the background colour to dark gray
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		terrain.render(camera);   
		for (Mesh cube : cubes) {
			cube.render(camera);
		}
		reflectionTarget.stopUsing();
		glDisable(GL_CLIP_DISTANCE0);
		camera.setNotReflected();

		// now render the final scene
		glClearColor(1.0f, 1.0f, 1.0f, 1.0f); // Set the background colour to dark gray
		glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
		terrain.render(camera);   
		for (Mesh cube : cubes) {
			cube.render(camera);
		}
		water.render(camera);
		
		

		checkError();
		glfwSwapBuffers(window);
		glfwPollEvents();
		checkError();
	}

	
	
	public void takeScreenshot(String output_path) {
		int bpp = 4;

		// Take screenshot of the fixed size irrespective of the window resolution
		int screenshot_width = 800;
		int screenshot_height = 600;

		int fbo = glGenFramebuffers();
		glBindFramebuffer( GL_FRAMEBUFFER, fbo );

		int rgb_rb = glGenRenderbuffers();
		glBindRenderbuffer( GL_RENDERBUFFER, rgb_rb );
		glRenderbufferStorage( GL_RENDERBUFFER, GL_RGBA, screenshot_width, screenshot_height );
		glFramebufferRenderbuffer( GL_FRAMEBUFFER, GL_COLOR_ATTACHMENT0, GL_RENDERBUFFER, rgb_rb );

		int depth_rb = glGenRenderbuffers();
		glBindRenderbuffer( GL_RENDERBUFFER, depth_rb );
		glRenderbufferStorage( GL_RENDERBUFFER, GL_DEPTH_COMPONENT, screenshot_width, screenshot_height );
		glFramebufferRenderbuffer( GL_FRAMEBUFFER, GL_DEPTH_ATTACHMENT, GL_RENDERBUFFER, depth_rb );
		checkError();

		float old_ar = camera.getAspectRatio();
		camera.setAspectRatio( (float)screenshot_width  / screenshot_height );
		glViewport(0,0, screenshot_width, screenshot_height );
		render();
		camera.setAspectRatio( old_ar );

		glReadBuffer(GL_COLOR_ATTACHMENT0);
		ByteBuffer buffer = BufferUtils.createByteBuffer(screenshot_width * screenshot_height * bpp);
		glReadPixels(0, 0, screenshot_width, screenshot_height, GL_RGBA, GL_UNSIGNED_BYTE, buffer);
		checkError();

		glBindFramebuffer( GL_FRAMEBUFFER, 0 );
		glDeleteRenderbuffers( rgb_rb );
		glDeleteRenderbuffers( depth_rb );
		glDeleteFramebuffers( fbo );
		checkError();

		BufferedImage image = new BufferedImage(screenshot_width, screenshot_height, BufferedImage.TYPE_INT_ARGB);
		for (int i = 0; i < screenshot_width; ++i) {
			for (int j = 0; j < screenshot_height; ++j) {
				int index = (i + screenshot_width * (screenshot_height - j - 1)) * bpp;
				int r = buffer.get(index + 0) & 0xFF;
				int g = buffer.get(index + 1) & 0xFF;
				int b = buffer.get(index + 2) & 0xFF;
				image.setRGB(i, j, 0xFF << 24 | r << 16 | g << 8 | b);
			}
		}
		try {
			ImageIO.write(image, "png", new File(output_path));
		} catch (IOException e) {
			throw new RuntimeException("failed to write output file - ask for a demonstrator");
		}
	}

	public void stop() {
		glfwDestroyWindow(window);
		glfwTerminate();
	}

	private void checkError() {
		int error = glGetError();
		if (error != GL_NO_ERROR)
			throw new RuntimeException("OpenGL produced an error (code " + error + ") - ask for a demonstrator");
	}
}
