/**
 * Copyright 2010 Andrew Rice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package uk.ac.cam.acr31.oggdecoder;

import java.util.concurrent.ArrayBlockingQueue;
import java.util.concurrent.BlockingQueue;

public class QueuedOggStream extends OggStream implements BufferListenerable {

	private final int BUFSIZE = 1024;
	private final int QUEUELEN = 512;
	private final int QUEUEBYTES = BUFSIZE * QUEUELEN;
	private BlockingQueue<byte[]> q = new ArrayBlockingQueue<byte[]>(QUEUELEN);
	private BlockingQueue<byte[]> q2 = new ArrayBlockingQueue<byte[]>(
			QUEUELEN + 2);
	private BufferStatusListener bufferStatusListener;
	private OggStream mSource;
	private byte[] current = new byte[BUFSIZE];
	private int readPointer = BUFSIZE;

	public QueuedOggStream(OggStream source) {
		this.mSource = source;
		int cap = q2.remainingCapacity();
		for (int i = 0; i < cap; ++i) {
			try {
				q2.put(new byte[BUFSIZE]);
			} catch (InterruptedException e) {
			}
		}

		Thread t = new Thread() {
			@Override
			public void run() {
				try {
					while (true) {
						fill();
					}
				} catch (EndOfStreamException e) {
					if (bufferStatusListener != null) {
						bufferStatusListener.endOfStream();
					}
				} catch (InvalidStreamException e) {
				}
			}

		};
		t.start();
	}

	private void fill() throws EndOfStreamException, InvalidStreamException {
		byte[] buf;
		try {
			buf = q2.take();
			mSource.read(buf, 0, buf.length);
			q.put(buf);
			if (bufferStatusListener != null) {
				bufferStatusListener.updateBytesReadable(q.size() * BUFSIZE,
						QUEUEBYTES);
			}
		} catch (InterruptedException e1) {
		}
	}

	@Override
	public void close() {
		mSource.close();
	}

	@Override
	public void read(byte[] buf, int off, int len) throws EndOfStreamException,
			InvalidStreamException {
		try {
			int remaining = Math.min(len, current.length - readPointer);
			for (int i = 0; i < remaining; ++i) {
				buf[i + off] = current[i + readPointer];
			}
			len -= remaining;
			off += remaining;
			readPointer += remaining;
			if (len == 0) {
				if (bufferStatusListener != null) {
					bufferStatusListener.updateBytesReadable(
							q.size() * BUFSIZE, QUEUEBYTES);
				}
				return;
			}
			readPointer = 0;
			q2.put(current);
			current = q.take();
			read(buf, off, len);
		} catch (InterruptedException e) {
		}
	}

	@Override
	public void setBufferStatusListener(BufferStatusListener l) {
		this.bufferStatusListener = l;
	}

}
