/**
 * Copyright 2010 Andrew Rice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package uk.ac.cam.acr31.oggdecoder;

public class BlockingRingBuffer {

	private RingBuffer wrapped;
	private boolean endOfStream;
	
	public BlockingRingBuffer(int bufSize) {
		this.wrapped = new RingBuffer(bufSize);
	}

	public synchronized int getBytesRemaining() {
		return wrapped.getBytesRemaining();
	}
	
	public synchronized void setEndOfStream() {
		endOfStream = true;
		notifyAll();
	}

	
	public synchronized void blockingWrite(byte[] data, int off, int len) {
		while (wrapped.getBytesRemaining() < len) {
			try {
				wait();
			} catch (InterruptedException e) {
			}
		}
		try {
			wrapped.write(data, off, len);
		} catch (BufferOverflowException e) {
			throw new ImpossibleExceptionError(e);
		}
		notifyAll();
	}

	public synchronized int blockingRead(byte[] data, int off, int len) {
		while (!endOfStream && wrapped.getBytesAvailable() == 0) {
			try {
				wait();
			} catch (InterruptedException e) {
			}
		}
		if (endOfStream && wrapped.getBytesAvailable() == 0) return -1;
		int result;
		try {
			result = wrapped.read(data, off, len);
		} catch (BufferUnderflowException e) {
			throw new ImpossibleExceptionError(e);
		}
		notifyAll();
		return result;
	}
}
