/**
 * Copyright 2010 Andrew Rice
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package uk.ac.cam.acr31.oggdecoder;

import java.util.HashMap;
import java.util.LinkedList;
import java.util.List;
import java.util.Map;

public class OggContainer {

	private Map<Integer, OggLogicalStream> logicalStreams = null;
	private OggStream source;
	private List<OggPacketReceiverFactory> factories;

	public OggContainer(OggStream source) {
		this.source = source;
		factories = new LinkedList<OggPacketReceiverFactory>();
	}

	private void nextPage() throws OggFormatException, EndOfStreamException,
			InvalidStreamException {
		OggPage page = new OggPage(source);

		OggLogicalStream l = null;
		if (page.isBeginningOfStream()) {
			l = new OggLogicalStream(page.getBitstreamSerialNumber());
			l.setPacketReceiver(new StreamRecognisingPacketReceiver(l,
					factories));
			logicalStreams.put(page.getBitstreamSerialNumber(), l);
		} else {
			l = logicalStreams.get(page.getBitstreamSerialNumber());
		}
		if (page.isEndOfStream()) {
			logicalStreams.remove(l);
		}

		int[] segs = page.getSegmentLengths();
		long g = page.getGranulePosition();
		int length = 0;
		for (int i = 0; i < segs.length; ++i) {
			length += segs[i];
			if (segs[i] != 255 || i == segs.length - 1) {
				l.writeSegment(g, source, length, segs[i] != 255);
				length = 0;
			}
		}
	}

	public void advance() throws OggFormatException, EndOfStreamException,
			InvalidStreamException {
		if (logicalStreams == null) {
			this.logicalStreams = new HashMap<Integer, OggLogicalStream>();
			nextPage();
		}
		for (OggLogicalStream s : logicalStreams.values()) {
			while (s.needsData()) {
				nextPage();
			}
		}
	}

	public void registerPacketReceiver(OggPacketReceiverFactory prf) {
		factories.add(prf);
	}
}
