(**************************************************************************)
(*         x86 Multiprocessor Machine Code Semantics: HOL sources         *)
(*                                                                        *)
(*                                                                        *)
(*  Susmit Sarkar (1), Peter Sewell (1), Francesco Zappa Nardelli (2),    *)
(*  Scott Owens (1), Tom Ridge (1), Thomas Braibant (2),                  *)
(*  Magnus Myreen (1), Jade Alglave (2)                                   *)
(*                                                                        *)
(*   (1) Computer Laboratory, University of Cambridge                     *)
(*   (2) Moscova project, INRIA Paris-Rocquencourt                        *)
(*                                                                        *)
(*    Copyright 2007-2008                                                 *)
(*                                                                        *)
(*  Redistribution and use in source and binary forms, with or without    *)
(*  modification, are permitted provided that the following conditions    *)
(*  are met:                                                              *)
(*                                                                        *)
(*  1. Redistributions of source code must retain the above copyright     *)
(*     notice, this list of conditions and the following disclaimer.      *)
(*  2. Redistributions in binary form must reproduce the above copyright  *)
(*     notice, this list of conditions and the following disclaimer in    *)
(*     the documentation and/or other materials provided with the         *)
(*     distribution.                                                      *)
(*  3. The names of the authors may not be used to endorse or promote     *)
(*     products derived from this software without specific prior         *)
(*     written permission.                                                *)
(*                                                                        *)
(*  THIS SOFTWARE IS PROVIDED BY THE AUTHORS ``AS IS'' AND ANY EXPRESS    *)
(*  OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED     *)
(*  WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE    *)
(*  ARE DISCLAIMED. IN NO EVENT SHALL THE AUTHORS BE LIABLE FOR ANY       *)
(*  DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL    *)
(*  DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE     *)
(*  GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS         *)
(*  INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,          *)
(*  WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING             *)
(*  NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS    *)
(*  SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.          *)
(*                                                                        *)
(**************************************************************************)

(******************************************************************************)
(* begin niceness *)

load "x86_niceness_statementTheory";
load "tactic";
load "arithmeticTheory";

Globals.linewidth:=100;

(******************************************************************************)

open HolKernel boolLib Parse bossLib 
local open x86_niceness_statementTheory in end;

open pred_setTheory utilTheory x86_axiomatic_modelTheory x86_niceness_statementTheory tactic 

val _ = new_theory "x86_niceness_proof";

(* use "tactic.sml"; 
open tactic; *)
val set_tac = ssimp[SUBSET_DEF,DIFF_DEF,GSPECIFICATION,IN_DISJOINT,EXTENSION] THEN simp[SPECIFICATION];

(******************************************************************************)
(* util *)

g `
(! x. x IN X ==> (f x = Z)) ==> (IMAGE f (X:'a set) = { Z:'b } ) \/ (IMAGE f (X:'a set) = {} )
`;
e(intros);
e(ssimp[EXTENSION]);
e(tac[]);
val IN_IMAGE_SINGLETON_EMPTY = top_thm();
 
g `
FINITE { m | m <= (n:num) }
`;
have `{ m | m <= n } = { m | m < SUC n }`;
 e(ssimp[EXTENSION]);
 e(numLib.ARITH_TAC);
e(tac[FINITE_COUNT,count_def]);
val FINITE_leq = top_thm();

g `
(BIGUNION o IMAGE f) X = { y | ? x. x IN X /\ y IN f x }
`;
e(ssimp[EXTENSION]);
e(tac[]);
val BIGUNION_o_IMAGE_def = top_thm();

(* FIXME do we need the following? *)
g `
(X = (BIGUNION o IMAGE f) Y) = (! x. x IN X = ? y. y IN Y /\ x IN f y)
`;
e(ssimp[EXTENSION]);
e(tac[]);
val BIGUNION_o_IMAGE_eq = top_thm();

g `
FINITE P /\ (! p. p IN P ==> FINITE (f p)) ==> FINITE ((BIGUNION o IMAGE f) P)
`;
e(intros);
have `FINITE (IMAGE f P)`; e(tac[IMAGE_FINITE]);
e(ssimp[FINITE_BIGUNION]);
e(intros);
e(ssimp[]);
e(tac[]);
val FINITE_BIGUNION_o_IMAGE = top_thm();

g `
(! n. r n SUBSET r (SUC n))
==> m <= n ==> r m SUBSET r n
`;
ir();
have `! m n. r m SUBSET r (m+n)`;
 e(intros);
 e(Induct_on `n`);
  e(ssimp[]);

  e(ssimp[]);
  have `m+SUC n = SUC (m+n)`; e(numLib.ARITH_TAC);
  e(ssimp[]);
  e(tac[SUBSET_DEF]);
have `m <= n ==> ? n':num. n = m+n'`; 
 have `m <= n ==> (n = m+(n:num-m))`; e(numLib.ARITH_TAC);
 e(tac[]);
e(tac[]);
val r_mono = top_thm();


g `
(! x y:num. x < y \/ (x=y) \/ y<x)
/\ (! x y:num. x <= y = (x < y \/ (x = y)))
/\ (! x y z:num. x < y /\ y < z ==> x < z)
/\ (! x y:num. x<= SUC y = ((x = SUC y) \/ x <= y))
/\ (! x y:num. x <=y /\ y<x ==> F)
/\ (! x y:num. ~ (x < y) = (y <=x))
/\ (! x y:num. x <= (x+y))
`;
e(numLib.ARITH_TAC);
val less_thms = top_thm();

g `
x IN {} ==> F
`;
e(ssimp[]);
val IN_EMPTY = top_thm();

g `
!f v. GSPEC f v = ?x. (v,T) = f x
`;
e(tac[SPECIFICATION,GSPECIFICATION]);
val GSPEC_DEF_2 = top_thm();

val linear_order_append_def = Define `
linear_order_append R1 R2 = 
  R1 
  UNION (FIELD R1 CROSS FIELD R2) 
  UNION R2
`;

g `
(A DIFF B) INTER C = (A INTER C) DIFF (B INTER C)
`;
e(ssimp[EXTENSION]);
e(set_tac);
e(tac[]);
val DIFF_INTER_IFF = top_thm();

val RTC_def = Define `
RTC r (x,y) = (x,y) IN sTC r \/ (x = y)
`;

g `
A INTER (COMPL B) = A DIFF B
`;
e(set_tac);
val INTER_COMPL = top_thm();

g `
(x,y) IN RTC r = ((x=y) \/ (x,y) IN sTC r)
`;
e(ssimp[RTC_def,SPECIFICATION]);
e(tac[]);
val IN_RTC = top_thm();


g `
FINITE E ==> FINITE (DOM E)
`;
e(intros);
e(QCUT_TAC `DOM E = IMAGE FST E`); 
 e(ssimp[DOM_def,IMAGE_DEF]);
 e(ssimp[EXTENSION]);
 e(intros);
 e(EQR_TAC);
  e(elims);
  er ``(x,y)``;
  e(ssimp[]);

  e(elims);
  e(Cases_on `x'`);
  e(ssimp[]);
  e(tac[]);
e(tac[IMAGE_FINITE]);
val FINITE_DOM = top_thm();

g `
acyclic r ==> ~ ((e:'a,e) IN r)
`;
e(intros);
e(ssimp[acyclic_def]);
xal ``e:'a``;
e(tac[sTC_rules,IN_UNION]);
val acyclic_not_reflexive = top_thm();


(******************************************************************************)
(* linear orders *)

g `
strict_linear_order r xs ==> (x,y) IN r ==> x IN xs /\ y IN xs /\ ~ (x=y)
`;
e(ssimp[strict_linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def,SUBSET_DEF]);
e(tac[]);
val strict_linear_order_typing = top_thm();


g `
linear_order r xs ==> (x,y) IN r ==> x IN xs /\ y IN xs
`;
e(ssimp[linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def,SUBSET_DEF]);
e(tac[]);
val linear_order_typing = top_thm();

g `
linear_order r xs ==> e IN xs ==> (e,e) IN r
`;
e(ssimp[linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def,SUBSET_DEF]);
e(tac[]);
val linear_order_reflexive = top_thm();

g `
linear_order r xs ==> (FIELD r = xs)
`;
e(ssimp[linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def,SUBSET_DEF,FIELD_def]);
e(set_tac);
e(tac[]);
val linear_order_FIELD = top_thm();

g `
linear_order {} {}
`;
e(ssimp[linear_order_def]);
e(ssimp[DOM_def,RANGE_def,RES_FORALL_THM]);
val linear_order_empty = top_thm();

g `
R1 SUBSET linear_order_append R1 R2
`;
e(ssimp[linear_order_append_def]);
e(set_tac THEN NO_TAC);
val SUBSET_linear_order_append = top_thm();

g `
linear_order R1 xs1
/\ linear_order R2 xs2
/\ (DISJOINT xs1 xs2)
==> linear_order (linear_order_append R1 R2) (xs1 UNION xs2)
`;
e(intros);
e(elims);
e(ssimp[linear_order_def,linear_order_append_def]);
e(ssimp[DOM_def,FIELD_def,RANGE_def]);
e(ssimp[IN_DISJOINT]);
e(ssimp[RES_FORALL_THM]);
e(set_tac);
e(tac[]);
val linear_order_linear_order_append = top_thm();

g `
linear_order_append {} R2 = R2
`;
e(ssimp[linear_order_append_def]);
e(ssimp[EXTENSION]);
e(ssimp[pairTheory.FORALL_PROD]);
e(ssimp[FIELD_def,DOM_def,RANGE_def]);
val linear_order_append_empty = top_thm();

g `
linear_order R2 xs2
==> linear_order (linear_order_append {} R2) xs2
`;
e(ssimp[linear_order_append_empty]);
val linear_order_linear_order_append_empty = top_thm();

g `
(x,y) IN linear_order_append R1 R2 
==> (x,y) IN R1 \/ (x IN FIELD R1 /\ y IN FIELD R2) \/ (x,y) IN R2
`;
e(ssimp[linear_order_append_def]);
e(tac[]);
val IN_linear_order_append = top_thm();

g `
linear_order r1 xs1
==> linear_order r2 xs2
==> DISJOINT xs1 xs2
==> (e1,e2) IN r1
==> (e1,x) IN linear_order_append r1 r2
==> (x,e2) IN linear_order_append r1 r2
==> (e1,x) IN r1 /\ (x,e2) IN r1
`;
e(intros);
e(elims);
e(ssimp[linear_order_def,linear_order_append_def]);
e(ssimp[DOM_def,FIELD_def,RANGE_def]);
e(ssimp[IN_DISJOINT]);
e(ssimp[RES_FORALL_THM]);
e(set_tac);
e(tac[]);
val linear_order_append_lower = top_thm();

g `
linear_order r1 xs1
==> linear_order r2 xs2
==> DISJOINT xs1 xs2
==> (e1,e2) IN r2
==> (e1,x) IN linear_order_append r1 r2
==> (x,e2) IN linear_order_append r1 r2
==> (e1,x) IN r2 /\ (x,e2) IN r2
`;
e(intros);
e(elims);
e(ssimp[linear_order_def,linear_order_append_def]);
e(ssimp[DOM_def,FIELD_def,RANGE_def]);
e(ssimp[IN_DISJOINT]);
e(ssimp[RES_FORALL_THM]);
e(set_tac);
e(tac[]);
val linear_order_append_upper = top_thm();

g `
linear_order r xs ==> linear_order (RRESTRICT r ys) (xs INTER ys)
`;
e(ssimp[linear_order_def,RRESTRICT_def,INTER_DEF]);
e(ssimp[DOM_def,RANGE_def]);
e(ssimp[RES_FORALL_THM]);
e(set_tac);
e(tac[]);
val linear_order_RRESTRICT = top_thm();

g `
strict_linear_order L ls 
==> FINITE A /\ A SUBSET ls
==> ~ (A = {})
==> (? l. l IN maximal_elements A L)
`;
e(intros);
e(elims);
e(ASSUME_TAC finite_acyclic_has_maximal_thm);
xal ``A:'a set``;
e(ssimp[]);
xal ``L:('a#'a)set``;
il 0; e(tac[strict_linear_order_acyclic_thm]);
e(tac[]);
val strict_linear_order_maximal_elements = top_thm();

(* the union of a monotonically increasing sequence of linear orders
is a linear order *)

g `
(! x:num. linear_order ((f x):'a reln) (g x))
==> (! x y. x <= y ==> f x SUBSET f y /\ g x SUBSET g y)
==> linear_order ((BIGUNION o IMAGE f) UNIV) ((BIGUNION o IMAGE g) UNIV)
`;
e(intros);
e(ssimp[linear_order_def]);
e(QCUT_TAC `! x y e1 e2. (e1,e2) IN f x ==> (e1,e2) IN f(x+y)`); 
 e(ssimp[SUBSET_DEF]);
 e(QCUT_TAC `! x:num y. x <= x+y`); e(numLib.ARITH_TAC);
 e(tac[]);
e(QCUT_TAC `! x y e. e IN g x ==> e IN g(x+y)`);
 e(ssimp[SUBSET_DEF]);
 e(QCUT_TAC `! x:num y. x <= x+y`); e(numLib.ARITH_TAC);
 e(tac[]);
e(cintros);
 (* DOM *)
 e(ssimp[DOM_def]);
 e(set_tac);
 e(tac[]);

 (* RANGE *)
 e(ssimp[RANGE_def]);
 e(set_tac);
 e(tac[]);

 e(elims);
 e(RENAME_TAC [``x:num``|->``x':num``,``x':num``|->``x:num``]);
 e(ssimp[]);
 er ``(f:num->'a reln) (x+x')``;
 cr(); defer(); er ``x+x':num``; e(tac[]);
 e(QCUT_TAC `(x1,x2) IN f(x+x')`); e(tac[]);
 e(QCUT_TAC `(x2,x3) IN f(x+x')`); e(tac[arithmeticTheory.ADD_COMM]);
 e(tac[]);

 e(elims);
 e(RENAME_TAC [``x:num``|->``x':num``,``x':num``|->``x:num``]);
 e(ssimp[]);
 e(QCUT_TAC `(x1,x2) IN f(x+x')`); e(tac[]);
 e(QCUT_TAC `(x2,x1) IN f(x+x')`); e(tac[arithmeticTheory.ADD_COMM]);
 e(set_tac);
 e(tac[]);

 e(ssimp[RES_FORALL_THM]);
 e(intros);
 e(elims);
 e(RENAME_TAC [``x:num``|->``x':num``,``x':num``|->``x:num``]);
 e(QCUT_TAC `x1 IN g(x+x') /\ x2 IN g(x+x')`); e(tac[arithmeticTheory.ADD_COMM]);
 e(QCUT_TAC `(x1,x2) IN f(x+x') \/ (x2,x1) IN f(x+x')`); e(tac[]);
 e(tac[]);
val linear_order_BIGUNION = top_thm();



(******************************************************************************)
(* sTC *)

dropn 100;
g `
! sTC'.
(sTC' = \ x y.
  ? E. FINITE E /\ E SUBSET A /\ (x,y) IN sTC E)
==> (x:'a,y) IN sTC A ==> sTC' x y
`;
ar();
ir();
e(ASSUME_TAC (ISPEC ``A:'a reln`` sTC_ind));
xal ``sTC':'a->'a->bool``;
il 0; defer(); e(tac[]);
cr();
 e(ssimp[]);
 e(intros);
 er ``{ (x:'a,y:'a) }``;
 e(ssimp[]);
 e(QCUT_TAC `(x,y) IN {(x,y)}`); e(ssimp[] THEN NO_TAC);
 e(tac[sTC_rules]);

 e(intros); 
 e(elims);
 e(simp[]);
 er ``E:('a#'a)set UNION E'``;
 e(ssimp[]);
 e(tac[sTC_UNION_lem,UNION_COMM,sTC_rules]);
val path_FINITE' = top_thm();

g `
! A x y:'a. (x,y) IN sTC A ==> ? E. FINITE E /\ E SUBSET A /\ (x,y) IN sTC E
`;
e(REPEAT FORALLR_TAC);
e(ASSUME_TAC path_FINITE');
e(ssimp[]);
val path_FINITE = top_thm();

g `
! A B x y:'a. (x,y) IN sTC A ==> A SUBSET B ==> (x,y) IN sTC B
`;
e(tac[SUBSET_DEF,sTC_implication_lem]);
val sTC_mono = top_thm();


(**************************************)
(* union of two relations *)

(* a cycle in a UNION must include an edge from both *)
dropn 100;
g `
! sTC'.
(sTC' = \ x y.
(x,y) IN sTC (A UNION B)
/\ ((x,y) IN sTC A \/ ? z z'. (x,z) IN RTC A /\ (z,z') IN B /\ (z',y) IN RTC (A UNION B)))
==> 
(x:'a,y) IN sTC (A UNION B) ==> sTC' x y 
`;
ar();
ir();
e(ASSUME_TAC (ISPEC ``A UNION B:'a reln`` sTC_ind));
xal ``sTC':'a->'a->bool``;
il 0; 
 cr();
  e(ssimp[]);
  e(intros);
  cr(); e(tac[sTC_rules,IN_UNION,SPECIFICATION]);
  e(ssimp[SPECIFICATION,RTC_def]);
  e(tac[sTC_rules,IN_UNION,SPECIFICATION]);

  e(intros); 
  e(elims);
  e(simp[]);
  cr(); e(tac[sTC_rules,IN_UNION,SPECIFICATION]);
  e(ASSUME_TAC sTC_rules);
  e(ssimp[SPECIFICATION,RTC_def]);
  e(tac[IN_UNION,SPECIFICATION]);
 e(tac[]);  
val sTC_UNION' = top_thm();

g `
! A B x y.
(x:'a,y) IN sTC (A UNION B)
==> ((x,y) IN sTC A \/ ? z z'. (x,z) IN RTC A /\ (z,z') IN B /\ (z',y) IN RTC (A UNION B))
`;
e(REPEAT FORALLR_TAC);
e(ASSUME_TAC sTC_UNION');
e(ssimp[]);
val sTC_UNION = top_thm();

(**************************************)
(* stuff more specific to our proof *)

(* FIXME this needs an assumption about no back P edges in L *)
(* FIXME also can rule out the l2=l1 case- acyclic P *)
dropn 100;
g `
acyclic P
/\ strict_linear_order L ls
/\ ~ (? x y. (x,y) IN L /\ (y,x) IN P)
/\ ~ (acyclic (P UNION L))
==> 
? l1 l1' l2' l2:'a. l1 IN ls /\ ~(l1' IN ls) /\ ~ (l2' IN ls) /\ l2 IN ls
/\ (l1,l1') IN P
/\ (l1',l2') IN RTC P
/\ (l2',l2) IN P 
/\ (l2,l1) IN L
`;
e(intros);
e(elims);
e(QCUT_TAC `? x. (x,x) IN sTC (P UNION L)`); e(ssimp[acyclic_def] THEN NO_TAC);
e(elims);
e(QCUT_TAC `? z z':'a. (x,z) IN RTC P /\ (z,z') IN L /\ (z',x) IN RTC (P UNION L)`);
 e(ASSUME_TAC sTC_UNION);
 xal ``P:'a reln``;
 xal ``L:'a reln``;
 xal ``x:'a``;
 xal ``x:'a``;
 e(ssimp[]);
 e(ssimp[acyclic_def]);
e(elims);
e(QCUT_TAC `z IN ls`); 
 e(ssimp[strict_linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def]);
 e(set_tac);
 e(tac[]);
e(QCUT_TAC `(z,z) IN sTC (P UNION L)`);
 e(QCUT_TAC `(x,z) IN RTC (P UNION L)`); 
  e(ssimp[IN_RTC]);
  e(tac[sTC_UNION_lem]);
 e(QCUT_TAC `(z,z') IN (P UNION L)`); e(set_tac);
 e(simp[IN_RTC]);
 e(tac[sTC_rules]);
(* at this point, we can get rid of all the other vars... *)
e(REMOVE_VAR_TAC ``z':'a``);
e(REMOVE_VAR_TAC ``x:'a``);
(* now we have a point in ls which is part of a cycle on a finite set
of edges... so can find the maximal such *)
val here = p();

dropn 100; add(here);
e(QCUT_TAC `? E. FINITE E /\ E SUBSET (P UNION L) /\ (z,z) IN sTC E`);
 e(tac[path_FINITE]);
e(elims);
e(QCUT_TAC `? A. A = { l | l IN ls /\ (l,l) IN sTC E }`); e(tac[]);
e(elims);
e(QCUT_TAC `FINITE A`);
(*  e(QCUT_TAC `FINITE (E INTER (P UNION L))`); e(ssimp[] THEN NO_TAC); *)
 e(QCUT_TAC `! l. l IN A ==> l IN DOM E`); 
  e(intros);
  e(ssimp[]);
  e(QCUT_TAC `l IN DOM E`); e(tac[sTC_DOM_RNG_thm]);  
  e(INIT_TAC);
 e(QCUT_TAC `A SUBSET DOM E`); e(set_tac THEN NO_TAC);
 e(QCUT_TAC `FINITE (DOM E)`); e(tac[FINITE_DOM]);
 e(tac[SUBSET_FINITE]);
e(QCUT_TAC `~ (A = {})`); 
 e(QCUT_TAC `z IN A`); e(ssimp[]);
 e(ssimp[]);
 ir();
 e(ssimp[EXTENSION]);
 e(set_tac);
 e(tac[]);
(* OK, don't need z anymore *)
e(REMOVE_VAR_TAC ``z:'a``);
e(QCUT_TAC `A SUBSET ls`); e(set_tac THEN NO_TAC);
e(QCUT_TAC `? l1. l1 IN maximal_elements A L`); e(tac[strict_linear_order_maximal_elements]);
e(elims);
e(QCUT_TAC `l1 IN ls /\ (l1,l1) IN sTC E`); e(ssimp[maximal_elements_def] THEN NO_TAC);
(* work on this fact *)
val here2 = p();

dropn 100; add(here2);
e(QCUT_TAC `? l1'. ~(l1'=l1) /\ (l1,l1') IN E /\ (l1',l1) IN sTC E`); 
 e(QCUT_TAC `~ ((l1,l1) IN E)`); 
  e(CCONTR_TAC);
  e(ssimp[]);
  e(QCUT_TAC `(l1,l1) IN P \/ (l1,l1) IN L`); e(set_tac THEN NO_TAC);
  dl();
   e(ssimp[acyclic_def]);
   xal ``l1:'a``;
   e(tac[sTC_rules]);

   e(ssimp[strict_linear_order_def] THEN NO_TAC);

 e(tac[sTC_cases_left]);
e(elims); 
e(QCUT_TAC `~ (l1' IN ls)`); 
 ir();
 e(QCUT_TAC `(l1',l1) IN L \/ (l1,l1') IN L`); 
  e(ssimp[strict_linear_order_def,RES_FORALL_THM] THEN NO_TAC);
 dl();
  (* this is the case that L contains a P back-edge *)
  e(QCUT_TAC `~ (? x y. (x,y) IN L /\ (y,x) IN P)`); e(INIT_TAC);
  il 0;
   er ``l1':'a``;
   er ``l1:'a``;
   e(ssimp[]);
   e(QCUT_TAC `(l1,l1') IN P \/ (l1,l1') IN L`); e(set_tac THEN NO_TAC); 
   e(QCUT_TAC `~ ((l1,l1') IN L)`); 
    e(ssimp[strict_linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def]);
    e(ssimp[EXTENSION]);
    e(set_tac);
    e(tac[]);   
   e(tac[]);

   e(tac[]);

  (* (l1,l1') IN L *)
  (* but then l1 is not maximal *)
  e(QCUT_TAC `(l1',l1') IN sTC E`); e(tac[sTC_rules]);
  e(QCUT_TAC `l1' IN ls`); e(INIT_TAC);
  e(QCUT_TAC `l1' IN A`); e(set_tac THEN NO_TAC); 
  e(ssimp[maximal_elements_def] THEN NO_TAC);
(*
  e(QCUT_TAC `x=l1`); e(tac[]);
  e(ssimp[]);
  e(ssimp[strict_linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def]);
  e(ssimp[EXTENSION]);
  e(set_tac);
  e(tac[]);   *)
e(QCUT_TAC `(l1,l1') IN P \/ (l1,l1') IN L`); e(set_tac THEN NO_TAC);
e(QCUT_TAC `~ ((l1,l1') IN L)`); 
 e(ssimp[strict_linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def]);
 e(ssimp[EXTENSION]);
 e(set_tac);
 e(tac[]);   
e(QCUT_TAC `(l1,l1') IN P`); e(tac[]);
val here3 = p();

dropn 100; add(here3);
(* work on getting l2 now- recall previous facts *)
e(QCUT_TAC `(l1',l1) IN sTC E /\ ~ (l1' IN ls) /\ l1 IN ls`); e(ssimp[maximal_elements_def] THEN NO_TAC);
(* so there is l2 st. (x,l2) IN sTC E INTER P /\ l2 IN ls /\ (l2,l2) IN sTC E *)
e(QCUT_TAC `? P' L'. ((E INTER P) DIFF (ls CROSS ls) = P') /\ (E INTER L = L') /\ (P' UNION L' = E)`); 
 e(ssimp[]);
 e(QCUT_TAC `! x. ~ ((x,x) IN P)`); e(tac[acyclic_not_reflexive]);
 e(ssimp[strict_linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def]);
 e(set_tac);
 e(ssimp[pairTheory.FORALL_PROD]);
 e(tac[]);
e(elims);
e(QCUT_TAC `P' SUBSET P`); e(ssimp[EXTENSION,SUBSET_DEF]); e(tac[]);
e(QCUT_TAC `(l1,l1') IN P'`); 
 e(set_tac);
 e(tac[]);
e(QCUT_TAC `(l1',l1) IN sTC (P' UNION L')`); e(tac[]);
(* sigh... have to repeat the same trick and get a cycle from l1' which uses an edge in L *)
e(QCUT_TAC `(l1',l1') IN sTC (P' UNION L')`); e(tac[sTC_rules]);
e(QCUT_TAC `? l2 z'. (l1',l2) IN RTC P' /\ (l2,z') IN L' /\ (z',l1') IN RTC (P' UNION L')`);
 e(ASSUME_TAC sTC_UNION);
 xal ``P':'a reln``;
 xal ``L':'a reln``;
 xal ``l1'``;
 xal ``l1'``;
 il 0; e(INIT_TAC);
 e(QCUT_TAC `~((l1',l1') IN sTC P')`); 
  e(ssimp[acyclic_def]);
  e(set_tac);
  e(tac[sTC_implication_lem,SPECIFICATION]);
 e(ssimp[]);
e(elims);
e(QCUT_TAC `(l2,l2) IN sTC E`); 
 e(QCUT_TAC `(l1',l2) IN RTC E`); 
  e(ssimp[IN_RTC]);
  e(set_tac);
  e(tac[sTC_implication_lem,SPECIFICATION]);
 e(QCUT_TAC `(l2,z') IN E`); 
  e(ssimp[IN_RTC]);
  e(set_tac);
  e(tac[sTC_implication_lem,SPECIFICATION]);
 e(ssimp[IN_RTC]);
 e(tac[sTC_rules]);
e(QCUT_TAC `l2 IN ls`); 
 e(QCUT_TAC `(l2,z') IN L'`); e(INIT_TAC);
 e(QCUT_TAC `(l2,z') IN L`); e(set_tac); e(tac[]);
 e(tac[strict_linear_order_typing]);
(* can remove z' *)
e(REMOVE_VAR_TAC ``z'``);
e(QCUT_TAC `~ (l2 = l1)`); 
val here = p(); 
dropn 1; add(here);
 ir();
 e(ssimp[IN_RTC]);
 e(QCUT_TAC `(l1,l1) IN sTC P'`);e(tac[sTC_rules]);
 e(QCUT_TAC `(l1,l1) IN sTC P`); e(tac[sTC_mono]);
 e(ssimp[acyclic_def]);
e(QCUT_TAC `(l2,l1) IN L`);
 e(ssimp[maximal_elements_def]);
 e(ssimp[strict_linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def]);
 e(ssimp[EXTENSION]);
 e(set_tac);
 e(tac[]);
val here4 = p();

dropn 100; add(here4);
(* now need l2' *)
e(QCUT_TAC `~ (l2 = l1')`); e(tac[]);
e(QCUT_TAC `(l1',l2) IN sTC P'`); e(ssimp[IN_RTC] THEN NO_TAC);
e(QCUT_TAC `(l1',l2) IN sTC P`); 
 e(set_tac);
 e(tac[sTC_implication_lem,SPECIFICATION]);
e(QCUT_TAC `(l1',l2) IN P' \/ (? z. (l1',z) IN sTC P' /\ (z,l2) IN P')`); e(tac[sTC_cases_right]);
e(QCUT_TAC `? l2'. (l1',l2') IN RTC P' /\ (l2',l2) IN P'`);  
 e(ssimp[IN_RTC]);
 e(tac[]);
e(elims);
e(QCUT_TAC `~ (l2' IN ls)`);
 ir();
 e(QCUT_TAC `l2 IN ls /\ l2' IN ls`); e(tac[]);
 e(SYM_TAC ``E INTER P DIFF ls CROSS ls = P'``);
 e(ssimp[EXTENSION] THEN NO_TAC);
e(QCUT_TAC `(l1',l2') IN RTC P /\ (l2',l2) IN P`);
 e(ssimp[IN_RTC]);
 cr();
  e(tac[sTC_mono]);

  e(tac[SUBSET_DEF]);
er ``l1:'a``;
er ``l1':'a``;
er ``l2':'a``;
er ``l2:'a``;
e(ssimp[]);
val nasty = top_thm(); 


(******************************************************************************)
(* some definitions that mirror those in the axiomatic model *)

(******************************************************************************)
(* basic operations on datatypes *)

val reg_events_def = Define `
reg_events = { e | case e.action of Access _1 (Location_reg p r) _3 -> T || _ -> F }
`;

val mem_events_def = Define `
mem_events = {e | case e.action of Access _1 (Location_mem a) _3 -> T || _ -> F }
`;

val writes'_def = Define `
writes' = { e | case e.action of Access W l v -> T || _ -> F }
`;

val reads'_def = Define `
reads' = {e | case e.action of Access R l v -> T || _ -> F }
`;


(******************************************************************************)
(* basic structure: E *)

val wf_events_def = Define `
wf_events procs events =  
  (!e1 e2 :: events. (e1.eiid = e2.eiid) /\ (e1.iiid = e2.iiid) ==> (e1 = e2)) /\
  (!e :: events. !p r. (loc e = SOME (Location_reg p r)) ==> (p = proc e)) /\
  (!iiid. FINITE {eiid | ?e :: (events). (e.iiid = iiid) /\ (e.eiid = eiid)}) /\
  (FINITE procs) /\
  (!e :: (events). proc e IN procs)
`;

(* intra_causality is a (transitively closed?) strict partial order, really per iiid *)
val wf_intra_causality_def = Define `
wf_intra_causality events intra_causality = 
  (FIELD intra_causality) SUBSET events /\
  (!(e1, e2) :: intra_causality. (e1.iiid = e2.iiid))  /\

  acyclic intra_causality /\ (* FIXME should this be acyclic (sTC intra_causality)? Yes, but acyclic does the sTC for you :( *)

  (* if two ii events are to the same register, and one at least is a write, then they must be related by intra_causality *)
  (* so register writes are linearly ordered with imtermediate partially ordered reads *)
  (!e1 e2 :: (events INTER reg_events). 
      (e1.iiid = e2.iiid) /\ 
      e1 IN writes' /\
      (loc e1 = loc e2) 
      ==>
      (e1 = e2) \/
      (e1, e2) IN sTC intra_causality \/
      (e2, e1) IN sTC intra_causality) /\

  (!(e1, e2) :: (intra_causality). ~(mem_store e1))


`;

val wf_atomicity_def = Define `
wf_atomicity events atomicity = 
  PER events atomicity /\  
  (! es :: atomicity. ? iiid. es = events INTER { e | e.iiid = iiid }) /\
  (!es :: (atomicity). ?e :: es. mem_load e) 
`;

val wf_E_def = Define `
wf_E E = 
  wf_events E.procs E.events /\
  wf_intra_causality E.events E.intra_causality /\
  wf_atomicity E.events E.atomicity 
`;


(******************************************************************************)
(* basic structure: X *)


(* FIXME we need finiteness in the following *)

val check_atomicity'_def = Define `
  check_atomicity' atomicity vop =
       !es :: atomicity. 
          !e1 e2 :: es. (e1, e2) IN (vop) ==> 
            { e | (e1, e) IN vop /\ (e, e2) IN vop } SUBSET es
`;


(******************************************************************************)
(* rfmap *)

val reads_from_map_candidates'_def = Define `
reads_from_map_candidates' events rfmap = 
  rfmap SUBSET (events CROSS events)
  /\ rfmap SUBSET (writes' CROSS reads')
  /\ ! (ew, er) :: rfmap. (loc ew, value_of ew) = (loc er,value_of er)
`;

(* want to make this only check some reads ... *)
val check_rfmap_initial'_def = Define `
check_rfmap_initial' viewed_events_ vop rfmap initial_state =
  !er :: viewed_events_ INTER (reads' DIFF (RANGE rfmap)).
  (initial_state (THE (loc er)) = SOME(THE(value_of er))) /\
  (! ew' :: writes'. (ew', er) IN vop /\ (loc er = loc ew') ==> F)
`;

(* no vo intervening writes to the same location *)
val check_rfmap_written'_def = Define `
check_rfmap_written' vop rfmap =
  !(ew, er) :: rfmap.
  ! ew' :: writes'. 
  (ew, ew') IN vop /\ (ew', er) IN vop /\ (loc er = loc ew') ==> (ew = ew')
`;


(******************************************************************************)
(* valid execution *)

val check_causality'_def = Define `
check_causality' vop hb = acyclic ((strict (vop)) UNION hb)
`;

val check_finite_vop_def = Define `
check_finite_vop vop = ! e'. FINITE { e | (e,e') IN vop }
`;

val valid_vop_def = Define `
valid_vop E X (p,vop) = 
  (FIELD vop = (* SUBSET for partial structures *) (viewed_events E p)) /\
  linear_order vop (FIELD vop) /\
  check_atomicity' E.atomicity vop /\
  check_causality' vop (happens_before E X) /\ (* don't mind if E is a bit more *)
  check_rfmap_initial' (viewed_events E p) vop X.rfmap X.initial_state /\
  check_rfmap_written' vop X.rfmap /\ 
  check_finite_vop vop
`;

val valid_execution'_def = Define `
valid_execution' E X =
  wf_E E /\ 
  (* write_serialization is a linear order per loc on E.events *)
  X.write_serialization IN write_serialization_candidates E /\
  (* lock_serialization is a linear order on E.atomicity *)
  X.lock_serialization IN lock_serialization_candidates E /\
  X.rfmap IN reads_from_map_candidates E /\

  (! p :: (E.procs). valid_vop E X (p,X.vo p)) /\
  (!p. ~(p IN E.procs) ==> (X.vo p = {}))
`; 

val assumption1_def = Define `
assumption1 E = 
  (! e1 e2. (e1,e2) IN E.intra_causality /\ e1 IN mem_store ==> F)
`;

val assumption2_def = Define `
assumption2 E = ! es e. es IN E.atomicity /\ e IN es ==> ? e'. e' IN es /\ e' IN mem_load
`;

(******************************************************************************)
(* x86_axiomatic_model, basic stuff *)

g `
valid_execution' E X
==> p IN E.procs
==> valid_vop E X (p,X.vo p)
`;
e(ssimp[valid_execution'_def]);
val valid_execution'_valid_vop = top_thm();

g `
x IN write_serialization_candidates E = write_serialization_candidates E x
`;
e(ssimp[SPECIFICATION]);
val IN_write_serialization_candidates = top_thm();

g `
x IN reads_from_map_candidates E = reads_from_map_candidates E x
`;
e(ssimp[SPECIFICATION]);
val IN_reads_from_map_candidates = top_thm();

g `
(mem_store x = x IN mem_store)
/\ (mem_load x = x IN mem_load)
/\ (locked E e = e IN locked E)
`;
e(ssimp[SPECIFICATION]);
val reverse_SPECIFICATION = top_thm();

g `
e IN mem_load =     case e.action of
       Access R (Location_mem a) v -> T
    || _ -> F`;
e(ssimp[SPECIFICATION,mem_load_def]);
val IN_mem_load = top_thm();

g `
e IN mem_store =     case e.action of
       Access W (Location_mem a) v -> T
      || _ -> F`;
e(ssimp[SPECIFICATION,mem_store_def]);
val IN_mem_store = top_thm();


(**************************************)
(* disjointness of events *)

g `
! e. e IN reg_events \/ e IN mem_events
`;
e(intros);
e(Cases_on `e`);
e(ssimp[reg_events_def, mem_events_def]);
e(Cases_on `a`);
e(ssimp[]);
e(Cases_on `l` THEN ssimp[]);
val reg_events_or_mem_events = top_thm();

g `
DISJOINT reg_events mem_events
`;
e(ssimp[reg_events_def,mem_events_def]);
e(ssimp[IN_DISJOINT]);
e(intros);
e(Cases_on `x.action` THEN ssimp[]);
e(Cases_on `l` THEN ssimp[]);
val DISJOINT_reg_events_mem_events = top_thm();

g `
mem_events = mem_load UNION mem_store
`;
e(ssimp[EXTENSION]);
e(ssimp[mem_events_def,IN_mem_load,IN_mem_store]);
e(intros);
e(Cases_on `x.action` THEN ssimp[]);
e(Cases_on `l` THEN ssimp[]);
 e(Cases_on `d` THEN ssimp[]);

 e(Cases_on `d` THEN ssimp[]);
val mem_events_eq_mem_load_UNION_mem_store = top_thm();

g `
DISJOINT mem_load mem_store
`;
e(ssimp[IN_DISJOINT]);
e(intros);
e(ssimp[IN_mem_load,IN_mem_store]);
e(Cases_on `x.action` THEN ssimp[]);
e(Cases_on `d` THEN ssimp[]);
val DISJOINT_mem_load_mem_store = top_thm();

g `
(x IN mem_store \/ x IN mem_load \/ x IN reg_events)
/\ ~ (x IN mem_store /\ x IN mem_load)
/\ ~ (x IN mem_store /\ x IN reg_events)
/\ ~ (x IN mem_load /\ x IN reg_events)
/\ (x IN mem_events = (x IN mem_load \/ x IN mem_store))
`;
cr();
 e(ASSUME_TAC reg_events_or_mem_events);
 e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
 e(tac[]);
  
 e(ASSUME_TAC DISJOINT_mem_load_mem_store);
 e(ASSUME_TAC DISJOINT_reg_events_mem_events);
 e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
 e(ssimp[IN_DISJOINT]);
 e(tac[]);
val DISJOINT_events = top_thm();

g `
(loc e2 = loc e1) ==> (e2 IN mem_events = e1 IN mem_events) /\ (e2 IN reg_events = e1 IN reg_events)
`;
e(intros);
e(ssimp[loc_def,mem_events_def,reg_events_def]);
e(Cases_on `e1.action` THEN ssimp[]);
e(Cases_on `e2.action` THEN ssimp[]);
val loc_eq_loc_mem_events_reg_events = top_thm();

g `
e IN locked E = ? es. es IN E.atomicity /\ e IN es
`;
e(ssimp[SPECIFICATION,locked_def]);
e(tac[]);
val IN_locked = top_thm();

g `
viewed_events E p SUBSET E.events
`;
e(ssimp[viewed_events_def]);
e(ssimp[SUBSET_DEF]);
val viewed_events_SUBSET_E_events = top_thm();



(******************************************************************************)
(* equivalence of valid_execution' and valid_execution *)

g `
valid_execution' E X = (well_formed_event_structure E /\ valid_execution E X)
`;
e(EQR_TAC);
 (* g: well_formed_event_structure E /\ valid_execution E X *)
 cr();
  e(ssimp[well_formed_event_structure_def,valid_execution'_def,wf_E_def,RES_FORALL_THM]);
  e(cintros);  
   (* FINITE {eiid | ?e::(E.events). (e.iiid = iiid) /\ (e.eiid = eiid)} *)
   e(ssimp[wf_events_def] THEN NO_TAC);

   e(ssimp[wf_events_def] THEN NO_TAC);

   e(ssimp[wf_events_def] THEN NO_TAC);

   e(ssimp[wf_events_def] THEN NO_TAC);

   e(ssimp[wf_intra_causality_def,DOM_def,FIELD_def] THEN NO_TAC);

   e(ssimp[wf_intra_causality_def,RANGE_def,FIELD_def] THEN NO_TAC);

   e(ssimp[wf_intra_causality_def] THEN NO_TAC);

   e(ssimp[wf_intra_causality_def] THEN NO_TAC);

   e(ssimp[wf_intra_causality_def] THEN NO_TAC);   

   e(ssimp[wf_intra_causality_def,RES_FORALL_THM]);
   e(ssimp[]);
   e(SYM_TAC ``SOME (Location_reg p r) = loc e2``);
   e(ssimp[]);
   xal ``e1:event``;   
   il 0;
    e(ssimp[reg_events_def]);
    e(ssimp[loc_def]);
    e(Cases_on `e1.action`);
    e(ssimp[]);
    e(ssimp[writes_def]);
   xal ``e2:event``;   
   il 0;
    e(ssimp[reg_events_def]);
    e(ssimp[loc_def]);
    e(Cases_on `e2.action`);
    e(ssimp[]);
    e(ssimp[writes_def,reads_def]);
    e(tac[]);
   il 0;
    e(ssimp[writes'_def,writes_def]);
    e(Cases_on `e1.action`);
    e(ssimp[]);
   e(tac[]);
 
   e(ssimp[wf_atomicity_def] THEN NO_TAC);

   e(ssimp[wf_atomicity_def] THEN NO_TAC);

   e(ssimp[wf_atomicity_def,RES_FORALL_THM]);
   xal ``es:event set``;
   e(ssimp[]);     
   e(ssimp[]);

   e(ssimp[wf_atomicity_def,RES_FORALL_THM]);
   xal ``es:event set``;
   e(ssimp[]);     
   e(ssimp[]);

   e(ssimp[wf_events_def,RES_FORALL_THM]);
val here = p();

dropn 1; add(here);
  (* valid_execution E X *)   
  e(ssimp[valid_execution_def]);
  e(cintros);
   (* view_orders_well_formed E X.vo *)
   e(ssimp[view_orders_well_formed_def,RES_FORALL_THM]);
   cr();
    e(intros);  
    e(ASSUME_TAC valid_execution'_valid_vop);
    e(ssimp[valid_vop_def]); 
    e(ssimp[]);
    e(ssimp[check_finite_vop_def]);

    e(intros); 
    e(ssimp[valid_execution'_def]);

   e(ssimp[valid_execution'_def]);

   e(ssimp[valid_execution'_def]);

   e(ssimp[valid_execution'_def]);
  
   (* check_causality E X.vo (happens_before E X) *)
   e(ssimp[check_causality_def,RES_FORALL_THM]);
   e(intros);
   e(ASSUME_TAC valid_execution'_valid_vop);
   e(ssimp[valid_vop_def]);
   e(ssimp[check_causality'_def]);
 
   (* check_rfmap_written E X.vo X.rfmap *)
   e(ssimp[check_rfmap_written_def,RES_FORALL_THM]);
   e(intros);
   e(Cases_on `x`);
   e(RENAME_TAC [``q:event``|->``ew:event``,``r:event``|->``er:event``]);
   e(ssimp[]);
   e(intros);
   have `(ew,er) IN X.rfmap`; e(ssimp[RRESTRICT_def]);
   e(ASSUME_TAC valid_execution'_valid_vop);
   e(ssimp[valid_vop_def]);
   e(ssimp[check_rfmap_written'_def,RES_FORALL_THM]);
   xal ``(ew:event,er:event)``;
   e(ssimp[]);
   xal ``ew':event``;
   il 0; e(ssimp[writes_def,writes'_def]); e(ssimp[]);
   e(ssimp[]);
   e(ssimp[valid_execution'_def]);
   e(ssimp[IN_reads_from_map_candidates,reads_from_map_candidates_def,RES_FORALL_THM]);
   xal ``(ew:event,er:event)``;  
   e(ssimp[]);
   e(ssimp[loc_def]);  
 
   (* check_rfmap_initial E X.vo X.rfmap X.initial_state *)
   e(ssimp[check_rfmap_initial_def,RES_FORALL_THM]);
   e(intros);
   e(Cases_on `er.action`);
   e(RENAME_TAC [``c:value``|->``v:value``]);
   e(ssimp[]);
   cr(); e(ssimp[reads_def]);  
   e(ASSUME_TAC valid_execution'_valid_vop);
   e(ssimp[valid_vop_def]);
   e(ssimp[check_rfmap_initial'_def,RES_FORALL_THM]);
   xal ``er:event``;
   il 0;
    e(ssimp[reads'_def,reads_def]);
   e(elims);
   cr();
    e(ssimp[loc_def]);
    e(ssimp[value_of_def]);
   e(intros);
   xal ``ew':event``;
   il 0;
    e(ssimp[writes'_def,writes_def]);
    e(ssimp[]);
   e(ssimp[]);
 
   (* check_atomicity E X.vo *)
   e(ssimp[check_atomicity_def,RES_FORALL_THM]);
   e(intros);
   e(ASSUME_TAC valid_execution'_valid_vop);
   e(ssimp[valid_vop_def]);
   e(ssimp[check_atomicity'_def,RES_FORALL_THM]);
   e(ssimp[SUBSET_DEF]); 
   e(tac[]);
val here = p();

dropn 1; add(here); 
 (* g: valid_execution' E X a: valid_execution E X *)
 e(ssimp[valid_execution'_def,valid_execution_def]);
 e(cintros);
  (* wf_E E *)
  e(ssimp[wf_E_def,well_formed_event_structure_def]);
  e(cintros);  
   e(ssimp[wf_events_def]);

   e(ssimp[wf_intra_causality_def]);
   cr(); e(ssimp[FIELD_def]);
   e(ssimp[RES_FORALL_THM]);
   e(intros);
   e(TM_THIN_TAC ``!e. e IN E.events ==> !p r. (loc e = SOME (Location_reg p r)) ==> (p = proc e)``);
   e(TM_THIN_TAC ``!e1.
            e1 IN E.events ==>
            !e2. e2 IN E.events ==> (e1.eiid = e2.eiid) /\ (e1.iiid = e2.iiid) ==> (e1 = e2)``);
   e(TM_THIN_TAC ``!e. e IN E.events ==> proc e IN E.procs``);
   xal ``e1:event``;
   e(ssimp[]);
   e(Cases_on `e1 = e2`); e(ssimp[]);
   e(ssimp[]);
   il 7; e(ssimp[writes_def,writes'_def]); e(Cases_on `e1.action` THEN ssimp[]); e(Cases_on `d` THEN ssimp[]);
   xal ``e2:event``;
   e(ssimp[]);
   il 0;
    e(ssimp[writes_def,reads_def,writes'_def,reads'_def]);
    e(Cases_on `e2.action` THEN ssimp[]);
    e(ssimp[loc_def]);
    e(Cases_on `l` THEN ssimp[] THEN Cases_on `e1.action` THEN ssimp[]);
     e(Cases_on `d` THEN ssimp[]);

     e(Cases_on `d'` THEN ssimp[]);
     e(ssimp[reg_events_def]);  
   e(INIT_TAC);
val here = p();

dropn 1; add(here);
   e(ssimp[wf_atomicity_def,RES_FORALL_THM]);
   e(intros);
   e(ssimp[EXTENSION]);
   have `~(es = {})`; e(ssimp[PER_def]); e(tac[]);
   have `? e. e IN es`; e(set_tac);
   e(elims);
   er ``e.iiid``;    
   e(intros);
   e(ssimp[RES_FORALL_THM]);
   e(ssimp[PER_def]);
   e(ssimp[SUBSET_DEF]);
   e(tac[]);

  (* !p :: E.procs. valid_vop E X (p,X.vo p) *)
  e(ssimp[RES_FORALL_THM]);
  e(intros);
  e(ssimp[valid_vop_def,view_orders_well_formed_def]);
  e(ssimp[RES_FORALL_THM]);
  xal ``p:proc``;
  xal ``p:proc``;
  have `FIELD (X.vo p) = viewed_events E p`;
   e(ssimp[FIELD_def,linear_order_def]);
   e(ssimp[EXTENSION]);
   e(ssimp[RES_FORALL_THM]);
   e(ssimp[SUBSET_DEF,RANGE_def,DOM_def]);
   e(tac[]);
  e(cintros);
   e(INIT_TAC);

   e(tac[]);   

   e(ssimp[check_atomicity'_def,check_atomicity_def,RES_FORALL_THM]);
   e(intros);
   e(ssimp[SUBSET_DEF]);
   e(tac[]);

   e(ssimp[check_causality'_def,check_causality_def,RES_FORALL_THM]);   

   e(ssimp[check_rfmap_initial'_def,check_rfmap_initial_def,RES_FORALL_THM]);
   e(intros);
   xal ``p:proc``;
   e(ssimp[]);
   e(TM_THIN_TAC ``!e. e IN viewed_events E p ==> FINITE {e' | (e',e) IN X.vo p}``);
   xal ``er:event``;
   il 0; 
    e(ssimp[reads_def,reads'_def]); cr(); e(tac[viewed_events_SUBSET_E_events,SUBSET_DEF]);
    e(Cases_on `er.action` THEN ssimp[]);
    e(Cases_on `d` THEN ssimp[]);
   cr();
    (* X.initial_state (THE (loc er)) = SOME (THE (value_of er)) *)
    e(ssimp[]);
    e(ssimp[loc_def,value_of_def]);

    (* !ew'. ew' IN writes' ==> ~((ew',er) IN X.vo p) \/ ~(loc er = loc ew') *)
    e(ssimp[]);
    e(intros);
    e(CCONTR_TAC);
    e(ssimp[writes_def,writes'_def]);
    e(Cases_on `ew'.action` THEN ssimp[]);    
    e(Cases_on `d` THEN ssimp[]);
    xal ``ew':event``;
    il 0;
     e(ssimp[]);
     e(ssimp[linear_order_def,DOM_def,RANGE_def,SUBSET_DEF]);
     e(tac[viewed_events_SUBSET_E_events,SUBSET_DEF]);
    e(tac[]);

    e(ssimp[check_rfmap_written_def,check_rfmap_written'_def,RES_FORALL_THM,pairTheory.FORALL_PROD]);
    e(intros);       
    e(RENAME_TAC [``p_1:event``|->``e1:event``,``p_2:event``|->``e2:event``]);
    e(TM_THIN_TAC ``!e. e IN viewed_events E p ==> FINITE {e' | (e',e) IN X.vo p}``);     
    xal ``p:proc``;
    il 0; e(ssimp[]);
    xal ``e1:event``;
    xal ``e2:event``;
    il 0;
     e(ssimp[RRESTRICT_def,linear_order_def,SUBSET_DEF,RANGE_def,DOM_def,RES_FORALL_THM]);
     e(tac[]);
    xal ``ew':event``;
    il 0; 
     e(ssimp[writes'_def,writes_def]);
     e(Cases_on `ew'.action` THEN ssimp[]);
     e(Cases_on `d` THEN ssimp[]);
     e(ssimp[RRESTRICT_def,linear_order_def,SUBSET_DEF,RANGE_def,DOM_def,RES_FORALL_THM]);
     e(tac[viewed_events_SUBSET_E_events,SUBSET_DEF]);
    have `loc e2 = loc e1`; 
     e(ssimp[IN_reads_from_map_candidates,reads_from_map_candidates_def,pairTheory.FORALL_PROD,RES_FORALL_THM]);
     e(ssimp[loc_def]);
     e(Cases_on `ew'.action` THEN ssimp[]);
     e(Cases_on `e1.action` THEN ssimp[]);
     e(Cases_on `e2.action` THEN ssimp[]);
     xal ``e1:event``;
     xal ``e2:event``;
     e(ssimp[]);
    e(tac[]); 

   (* check_finite_vop (X.vo p) *)
   e(ssimp[check_finite_vop_def]);
   e(intros);
   e(Cases_on `e' IN viewed_events E p`); defer();
    have `{ e | (e,e') IN X.vo p} = {}`;
     e(ssimp[EXTENSION]);
     e(ssimp[linear_order_def,DOM_def,RANGE_def]);
     e(ssimp[SUBSET_DEF]); 
     e(tac[]);    
    e(ssimp[]);   

    (* e' IN viewed_events E p *)
    e(ssimp[RES_FORALL_THM]);    

  (* X.vo p = {} *)
  e(ssimp[view_orders_well_formed_def]);
val valid_execution'_eq_valid_execution = top_thm();


(******************************************************************************)
(* trivial consequences of valid_execution', typing *)

g `
valid_execution' E X
==> (e1,e2) IN X.write_serialization
==> ? l. get_l_stores E l e1 /\ get_l_stores E l e2 
`;
e(intros);
e(ssimp[valid_execution'_def]);
e(ssimp[SPECIFICATION,write_serialization_candidates_def,RES_FORALL_THM]);
xal ``(e1:event,e2:event)``;
e(ssimp[]);
val IN_X_write_serialization = top_thm();

g `
valid_execution' E X 
==> (e1,e2) IN E.intra_causality
==> e1 IN E.events /\ e2 IN E.events /\ (e2.iiid = e1.iiid)
`;
e(ssimp[valid_execution'_def]);
e(intros);
e(ssimp[wf_E_def,wf_intra_causality_def,RES_FORALL_THM,pairTheory.FORALL_PROD]);
e(ssimp[FIELD_def,DOM_def,RANGE_def,SUBSET_DEF]);
e(tac[]);
val intra_causality_typing = top_thm();

g `
valid_execution' E X 
==> (e1,e2) IN preserved_program_order E
==> e1 IN E.events /\ e2 IN E.events /\ (e2.iiid.proc = e1.iiid.proc)
`;
e(intros);
e(ssimp[preserved_program_order_def]);
e(ssimp[po_strict_def]);
val preserved_program_order_typing = top_thm();

g `
valid_execution' E X
==> (e1,e2) IN X.write_serialization
==> e1 IN E.events /\ e1 IN mem_store /\ e2 IN E.events /\ e2 IN mem_store
`;
e(intros);
e(ASSUME_TAC IN_X_write_serialization);
e(ssimp[]);
e(ssimp[get_l_stores_def]);
e(ssimp[GSPEC_DEF_2]);
e(set_tac);
val write_serialization_typing = top_thm();

(* FIXME this can replace the above *)
g `
valid_execution' E X
==> es IN E.atomicity
==> e1 IN es
==> e2 IN es
==> e1 IN E.events /\ e2 IN E.events /\ (e1.iiid = e2.iiid)
`;
e(intros);
e(ssimp[valid_execution'_def]);
e(ssimp[wf_E_def,wf_atomicity_def,RES_FORALL_THM]);
xal ``es:event set``;
e(ssimp[]);
e(ssimp[]);
val atomicity_typing = top_thm();

g `
valid_execution' E X
==> (e1,e2) IN X.lock_serialization
==> e1 IN E.events /\ e1 IN locked E /\ e2 IN E.events /\ e2 IN locked E
`;
e(intros);
have `valid_execution' E X`; e(INIT_TAC);
e(ASM_CONV_TAC (fn ths => SIMP_CONV (srw_ss()) [valid_execution'_def]) 0);
e(ssimp[locked_def]);
e(ssimp[lock_serialization_candidates_def,LET_THM,RES_FORALL_THM,RES_EXISTS_THM]);
e(ssimp[]);
e(Cases_on `x`);
e(RENAME_TAC [``q:event set``|->``es1:event set``,``r:event set``|->``es2:event set``]);
e(ssimp[]);
e(ssimp[strict_linearisations_def]);
have `e1 IN E.events /\ e2 IN E.events`; 
 have `es1 IN E.atomicity /\ es2 IN E.atomicity`; e(tac[strict_linear_order_typing]);
 e(tac[atomicity_typing]);
e(ssimp[IN_locked]);
e(tac[strict_linear_order_typing]);
val IN_lock_serialization_typing = top_thm();

g `
valid_execution' E X
==> (e1,e2) IN X.rfmap 
==> e1 IN E.events /\ e2 IN E.events
 /\ (e2 IN mem_events = e1 IN mem_events) 
 /\ (e2 IN reg_events = e1 IN reg_events)
 /\ (e1 IN writes' /\ e2 IN reads') 
 /\ (loc e2 = loc e1)
 /\ (e1 IN reg_events ==> (e2.iiid.proc = e1.iiid.proc))
`;
e(intros);
e(ssimp[valid_execution'_def]);
e(ssimp[IN_reads_from_map_candidates]);
e(ssimp[reads_from_map_candidates_def,RES_FORALL_THM,pairTheory.FORALL_PROD]);
xal ``e1:event``;
xal ``e2:event``;
e(ssimp[]);
e(QCUT_TAC `loc e2 = loc e1`); e(ssimp[loc_def]);
e(ssimp[loc_eq_loc_mem_events_reg_events]);
e(ssimp[writes'_def,reads'_def]);
e(intros);
e(ssimp[wf_E_def]);
e(ssimp[wf_events_def,RES_FORALL_THM]);
have `? p r. loc e1 = SOME(Location_reg p r)`; 
 e(ssimp[reg_events_def,loc_def]);
 e(Cases_on `l` THEN ssimp[]);
e(ssimp[proc_def]);
e(Cases_on `l`);
 e(ssimp[]);
 
 e(ssimp[]);
val rfmap_typing = top_thm();

g `
valid_execution' E X
==> p IN E.procs
==> (e1,e2) IN (X.vo p)
==> e1 IN viewed_events E p /\ e2 IN viewed_events E p /\ e1 IN E.events /\ e2 IN E.events
`;
e(intros);
e(QCUT_TAC `e1 IN viewed_events E p /\ e2 IN viewed_events E p`);
 e(ssimp[valid_execution'_def,RES_FORALL_THM]);
 xal ``p:proc``;
 e(ssimp[valid_vop_def]);
 e(tac[linear_order_typing]);
e(QCUT_TAC `e1 IN E.events /\ e2 IN E.events`); e(ssimp[viewed_events_def]);
e(tac[]);
val X_vop_typing = top_thm();

g `
valid_execution' E X
==> p IN E.procs
==> e IN viewed_events E p
==> (e,e) IN X.vo p
`;
e(intros);
e(ssimp[valid_execution'_def,RES_FORALL_THM]);
xal ``p:proc``;
e(ssimp[valid_vop_def]);
e(ssimp[linear_order_def,RES_FORALL_THM]);
e(tac[]);
val X_vop_reflexive = top_thm();

g `
valid_execution' E X
==> happens_before E X SUBSET E.events CROSS E.events
`;
e(ssimp[happens_before_def,SUBSET_DEF,pairTheory.FORALL_PROD]);
e(intros);
e(tac[intra_causality_typing,preserved_program_order_typing,write_serialization_typing,IN_lock_serialization_typing,rfmap_typing]);
val happens_before_typing = top_thm();

(* finiteness *)

g `
valid_execution' E X
==> FINITE {e | e IN E.events /\ (e.iiid = i)}
`;
e(intros);
have `FINITE {eiid | ?e::(E.events). (e.iiid = i) /\ (e.eiid = eiid)}`; e(ssimp[valid_execution'_def,wf_E_def,wf_events_def]);
have `!e1 e2::(E.events). (e1.eiid = e2.eiid) /\ (e1.iiid = e2.iiid) ==> (e1 = e2)`; e(ssimp[valid_execution'_def,wf_E_def,wf_events_def]);
have `{eiid | ?e::(E.events). (e.iiid = i) /\ (e.eiid = eiid)} = IMAGE (\ e. e.eiid) {e | e IN E.events /\ (e.iiid = i)}`;
 e(ssimp[EXTENSION,RES_EXISTS_THM]);
 e(tac[]);
have `INJ (\ e. e.eiid) { e | e IN E.events /\ (e.iiid = i)} {eiid | ?e::(E.events). (e.iiid = i) /\ (e.eiid = eiid)}`;
 e(ssimp[INJ_DEF]);
 e(ssimp[RES_EXISTS_THM]);
 e(tac[]);
e(tac[FINITE_INJ]);
val FINITE_events_iiid = top_thm();

g `
valid_execution' E X
==> FINITE { e | e IN E.events /\ (e.iiid.program_order_index = n) }
`;
e(intros);
have `? f. f = \ p. {e | e IN E.events /\ (e.iiid.program_order_index = n) /\ (e.iiid.proc = p) }`; e(tac[]);
e(elims);
have `FINITE (((BIGUNION o IMAGE f)) (E.procs))`; 
 have `FINITE (E.procs)`; e(ssimp[valid_execution'_def,wf_E_def,wf_events_def] THEN NO_TAC);
 have `! p. FINITE { e | e IN E.events /\ (e.iiid = <| program_order_index:=n; proc:=p |>) }`; e(tac[FINITE_events_iiid]);
 have `! p.
  { e | e IN E.events /\ (e.iiid = <| program_order_index:=n; proc:=p |>) } = 
  {e | e IN E.events /\ (e.iiid.program_order_index = n) /\ (e.iiid.proc = p) }
 `;
  ar();
  e(ssimp[EXTENSION]);
  e(intros);
  e(EQR_TAC);
   e(ssimp[]);

   e(ASSUME_TAC x86_coretypesTheory.iiid_literal_nchotomy);
   xal ``x.iiid``;
   e(ssimp[]);
   e(ssimp[]);
 have `! p. p IN E.procs ==> FINITE (f p)`; e(ssimp[]);
 e(tac[FINITE_BIGUNION_o_IMAGE]);
have `FINITE (((BIGUNION o IMAGE f)) UNIV)`; 
 have `UNIV = (E.procs) UNION (COMPL (E.procs))`; e(ssimp[EXTENSION]);
 have `(((BIGUNION o IMAGE f)) UNIV) = BIGUNION (IMAGE f (E.procs UNION (COMPL (E.procs))))`; e(ssimp[]);
 have `IMAGE f ((E.procs) UNION (COMPL (E.procs))) = (IMAGE f (E.procs)) UNION (IMAGE f (COMPL (E.procs)))`; e(ssimp[]);
 have `(IMAGE f (COMPL (E.procs)) = {{}}) \/ (IMAGE f (COMPL (E.procs)) = {})`; 
  e(MATCH_MP_TAC IN_IMAGE_SINGLETON_EMPTY);
  e(ssimp[proc_def]);
  e(ssimp[proc_def,valid_execution'_def,wf_E_def,wf_events_def,RES_FORALL_THM]);
  e(ssimp[EXTENSION]);
  e(tac[]);
 dl(); e(ssimp[]); e(ssimp[]);
val here = p(); 

dropn 1; add(here);
have `(((BIGUNION o IMAGE f)) UNIV) = { e | e IN E.events /\ (e.iiid.program_order_index = n) }`;
 e(simp[BIGUNION_o_IMAGE_def]);
 e(ssimp[EXTENSION]);
e(tac[]);
val FINITE_program_order_index_eq = top_thm(); 

g `
valid_execution' E X
==> FINITE { e | e IN E.events /\ e.iiid.program_order_index <= n }
`;
e(intros);
have `? f. f = \ n. { e | e IN E.events /\ (e.iiid.program_order_index = n) }`; e(tac[]);
e(elims);
have `{ e | e IN  E.events /\ e.iiid.program_order_index <= n } = (BIGUNION o IMAGE f) { m | m <= n}`;
 e(simp[BIGUNION_o_IMAGE_def]);
 e(ssimp[EXTENSION]);
 e(tac[]);
have `FINITE { m | m <= n}`; e(tac[FINITE_leq]);
have `! m. FINITE (f m)`; e(tac[FINITE_program_order_index_eq]);
e(tac[FINITE_BIGUNION_o_IMAGE]);
val FINITE_program_order_index_leq = top_thm();



(******************************************************************************)
(* happens_before *)

g `
valid_execution' E X 
==> p IN E.procs
==> acyclic (happens_before E X)
`;
e(intros);
e(ssimp[valid_execution'_def,RES_FORALL_THM]);
xal ``p:proc``;
e(ssimp[valid_vop_def]);
e(ssimp[check_causality'_def]);
e(ssimp[acyclic_def]);
e(intros);
xal ``x:event``;
il 0;
 e(tac[sTC_implication_lem,IN_UNION]);
 
 e(tac[]);
val acyclic_happens_before = top_thm();

g `
valid_execution' E X
==> p IN E.procs
==> e1 IN viewed_events E p /\ e2 IN viewed_events E p
==> (e1,e2) IN happens_before E X
==> (e1,e2) IN (X.vo p)
`;
e(intros);
e(QCUT_TAC `(e1,e2) IN (X.vo p) \/ (e2,e1) IN (X.vo p)`);
 e(ssimp[valid_execution'_def,valid_vop_def,RES_FORALL_THM]);
 xal ``p:proc``;
 e(elims);
 e(ssimp[linear_order_def]);
dl(); e(tac[]);
e(QCUT_TAC `~(e1 = e2)`); e(tac[acyclic_happens_before,acyclic_not_reflexive]);
e(ssimp[valid_execution'_def,RES_FORALL_THM]);
xal ``p:proc``;
e(ssimp[valid_vop_def]);
e(ssimp[check_causality'_def]);
e(ssimp[acyclic_def]);
xal ``e1:event``;
il 0; defer(); e(tac[]);
e(QCUT_TAC `(e2,e1) IN strict(X.vo p)`); e(ssimp[strict_def] THEN NO_TAC);
e(tac[sTC_rules,IN_UNION]);
val happens_before_X_vop' = top_thm();

g `
valid_execution' E X
==> p IN E.procs
==> e1 IN viewed_events E p /\ e2 IN viewed_events E p
==> (e1,e2) IN happens_before E X
==> ((e1,e2) IN (X.vo p) /\ ~ ((e2,e1) IN (X.vo p)))
`;
e(intros);
e(ASSUME_TAC happens_before_X_vop');
e(ASSUME_TAC acyclic_happens_before);
e(ssimp[]);
e(CCONTR_TAC);
e(QCUT_TAC `e1 = e2`);
 e(ssimp[valid_execution'_def,RES_FORALL_THM]);
 xal ``p:proc``;
 e(ssimp[valid_vop_def]);
 e(ssimp[linear_order_def] THEN NO_TAC);
e(ssimp[]);
e(ssimp[acyclic_def]);
xal ``e2:event``;
e(tac[sTC_rules]);
val happens_before_X_vop = top_thm();

(* basically there can't be any single hb edges between reg events that goes against po *)

(* some of the main cases- these deal with reg and read events on the same processor *)
g `
valid_execution' E X
==> assumption2 E
==> ~ (e1 IN mem_events /\ e2 IN mem_events)
==> (e1.iiid.proc = e2.iiid.proc) (* for locked *)
==> e1.iiid.program_order_index < e2.iiid.program_order_index
==> (e2,e1) IN happens_before E X
==> F
`;
e(intros);
e(REORDER_TAC ``(e2,e1) IN happens_before E X``);
e(ASM_CONV_TAC (fn ths => SIMP_CONV (srw_ss()) [happens_before_def]) 0);
e(REPEAT DISJL_TAC);
 e(ssimp[valid_execution'_def]);
 e(ssimp[wf_E_def]);
 e(ssimp[wf_intra_causality_def,RES_FORALL_THM]);
 xal ``(e2:event,e1:event)``;
 e(ssimp[] THEN NO_TAC);
 
 e(ssimp[preserved_program_order_def]); 
 e(ssimp[po_strict_def]);
 e(QCUT_TAC `! x y:num. x < y /\ y < x ==> F`); e(numLib.ARITH_TAC);
 e(tac[]);
 
 (* ws *)
 e(QCUT_TAC `e1 IN mem_store /\ e2 IN mem_store`); e(tac[write_serialization_typing]);
 e(ASSUME_TAC mem_events_eq_mem_load_UNION_mem_store);
 e(ssimp[IN_DISJOINT,EXTENSION] THEN NO_TAC);
val here = p();

dropn 1; add(here);
 (* ls *)
 e(QCUT_TAC `valid_execution' E X`); e(INIT_TAC);
 e(ASM_CONV_TAC (fn ths => SIMP_CONV (srw_ss()) ([valid_execution'_def])) 0);
 e(ssimp[lock_serialization_candidates_def]);
 e(ssimp[LET_THM]);
 e(ssimp[]);
 e(ssimp[RES_EXISTS_THM]);
 e(Cases_on `x`);
 e(RENAME_TAC [``q:event set``|->``es2:event set``,``r:event set``|->``es1:event set``]);
 e(ssimp[]);
 e(ssimp[assumption2_def]);
 al ``es1:event set``;
 xal ``e1:event``;
 xal ``es2:event set``;
 xal ``e2:event``;
 e(QCUT_TAC `es1 IN E.atomicity /\ es2 IN E.atomicity`); 
  e(ssimp[strict_linearisations_def]);
  e(ssimp[strict_linear_order_def]);
  e(ssimp[DOM_def,RANGE_def,SUBSET_DEF]); 
  e(tac[]);
 e(ssimp[]);
 e(RENAME_TAC [``e':event``|->``e1':event``,``e'':event``|->``e2':event``]);
 e(QCUT_TAC `e1' IN E.events /\ e2' IN E.events`);  e(tac[atomicity_typing]);
 e(QCUT_TAC `(e1'.iiid = e1.iiid) /\ (e2'.iiid = e2.iiid)`);
  e(ssimp[wf_E_def]);
  e(ssimp[wf_atomicity_def]);
  e(ssimp[RES_FORALL_THM]);
  al ``es1:event set``;
  xal ``es2:event set``;
  e(ssimp[]);
  e(ssimp[] THEN NO_TAC);
 e(QCUT_TAC `(e2',e1') IN X.lock_serialization`);
  e(ssimp[]);
  er ``(es2:event set,es1:event set)``;
  e(ssimp[] THEN NO_TAC);
 e(QCUT_TAC `(e2',e1') IN happens_before E X`);
  e(ssimp[happens_before_def] THEN NO_TAC);
 e(QCUT_TAC `(e1',e2') IN preserved_program_order E`);
  e(ssimp[preserved_program_order_def]);
  e(ssimp[po_strict_def]);
  e(ssimp[SPECIFICATION] THEN NO_TAC);
 e(QCUT_TAC `(e1',e2') IN happens_before E X`); 
  e(ssimp[happens_before_def] THEN NO_TAC);
 e(QCUT_TAC `(e1',e1') IN sTC (happens_before E X)`); e(tac[sTC_rules]);
 e(QCUT_TAC `~ (acyclic (happens_before E X))`);
  e(ssimp[acyclic_def]);
  e(tac[]);
 have `e1'.iiid.proc IN E.procs`;
  e(ssimp[proc_def,valid_execution'_def,wf_E_def,wf_events_def,RES_FORALL_THM]);
 e(tac[acyclic_happens_before]);
val here = p();

dropn 1; add(here);
 (* rfmap *)
 e(QCUT_TAC `valid_execution' E X`); e(INIT_TAC);
 e(ASM_CONV_TAC (fn ths => SIMP_CONV (srw_ss()) ([valid_execution'_def])) 0);
 e(elims);
 e(REORDER_TAC ``X.rfmap IN reads_from_map_candidates E``);
 e(ASM_CONV_TAC (fn ths =>SIMP_CONV std_ss [SPECIFICATION]) 0);
 e(ssimp[reads_from_map_candidates_def]);
 e(ssimp[RES_FORALL_THM]);
 xal ``(e2:event,e1:event)``;
 e(ssimp[]);
 e(QCUT_TAC `(e2,e1) IN X.rfmap`); e(INIT_TAC);
 e(QCUT_TAC `(e2,e1) IN (happens_before E X)`); e(ssimp[happens_before_def] THEN NO_TAC);
 e(QCUT_TAC `e2 IN reg_events /\ e1 IN reg_events`); 
  e(Cases_on `l`);
   e(ssimp[reg_events_def,loc_def] THEN NO_TAC);

   e(ssimp[mem_events_def,loc_def] THEN NO_TAC);
 e(QCUT_TAC `(e1,e2) IN preserved_program_order E`);
  e(ssimp[preserved_program_order_def]);
  e(ssimp[po_strict_def]);
  e(ssimp[loc_def]);
  e(Cases_on `l`);
   e(ssimp[] THEN NO_TAC);
   
   e(ssimp[mem_events_def,loc_def] THEN NO_TAC);
 e(QCUT_TAC `(e1,e2) IN (happens_before E X)`); e(ssimp[happens_before_def] THEN NO_TAC);
 e(QCUT_TAC `(e1,e1) IN sTC (happens_before E X)`); e(tac[sTC_rules]);
 e(QCUT_TAC `~ (acyclic (happens_before E X))`);
  e(ssimp[acyclic_def]);
  e(tac[]);
 have `e1.iiid.proc IN E.procs`;
  e(ssimp[proc_def,valid_execution'_def,wf_E_def,wf_events_def,RES_FORALL_THM]);
 e(tac[acyclic_happens_before]);
val reg_reg_happens_before = top_thm();

g `
valid_execution' E X
==> (e1,e2) IN happens_before E X
==> (e1 IN viewed_events E p /\ ~(e2 IN viewed_events E p))
==> e1 IN mem_events UNION locked E
`;
e(intros);
have `e1 IN E.events /\ e2 IN E.events`; 
 e(ASSUME_TAC happens_before_typing);
 e(ssimp[SUBSET_DEF,pairTheory.FORALL_PROD]);
 e(tac[]);
e(ssimp[happens_before_def]);
e(ssimp[SYM (SPEC_ALL DISJ_ASSOC)]);
have `~ (e2.iiid.proc = p)`; e(ssimp[viewed_events_def,proc_def]);
e(MY_DISJL_TAC ``(e1,e2) IN E.intra_causality \/ (e1,e2) IN preserved_program_order E \/
          (e1,e2) IN X.write_serialization \/ (e1,e2) IN X.lock_serialization \/ (e1,e2) IN X.rfmap``);
 (* (e1,e2) IN E.intra_causality *)
 have `e2.iiid = e1.iiid`; 
  e(ssimp[valid_execution'_def,wf_E_def,wf_intra_causality_def,RES_FORALL_THM,pairTheory.FORALL_PROD]);
 have `e1 IN mem_store`;
  e(ssimp[viewed_events_def,proc_def,reverse_SPECIFICATION]); 
 e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);

 (* (e1,e2) IN preserved_program_order E *)
 have `e2.iiid.proc = e1.iiid.proc`; 
  e(ssimp[preserved_program_order_def,po_strict_def]);
 have `e1 IN mem_store`;
  e(ssimp[viewed_events_def,proc_def,reverse_SPECIFICATION]); 
 e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);

 have `(e1 IN mem_store) /\ (e2 IN mem_store)`; e(tac[write_serialization_typing]);
 e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);

 have `e1 IN locked E /\ e2 IN locked E`; e(tac[IN_lock_serialization_typing]);
 e(tac[]);

 (* (e1,e2) IN X.rfmap *)
 have `loc e1 = loc e2`; e(tac[rfmap_typing]);
 have `e1 IN mem_events \/ e1 IN reg_events`; e(tac[DISJOINT_events]);
 dl();
  have `e2 IN mem_events`; e(tac[loc_eq_loc_mem_events_reg_events]);
  e(tac[]);
 
  have `e2 IN reg_events`; e(tac[loc_eq_loc_mem_events_reg_events]);
  have `(e1.iiid.proc = e2.iiid.proc)`; e(tac[rfmap_typing]);
  e(ssimp[viewed_events_def,proc_def]);
  e(ssimp[reverse_SPECIFICATION]);
  e(tac[DISJOINT_events]);
val happens_before_local_foreign_1 = top_thm();

g `
valid_execution' E X
==> (e2,e1) IN happens_before E X
==> (e1 IN viewed_events E p /\ ~(e2 IN viewed_events E p))
==> e1 IN mem_events UNION locked E
`;
e(intros);
have `e1 IN E.events /\ e2 IN E.events`; 
 e(ASSUME_TAC happens_before_typing);
 e(ssimp[SUBSET_DEF,pairTheory.FORALL_PROD]);
 e(tac[]);
e(ssimp[happens_before_def]);
e(ssimp[SYM (SPEC_ALL DISJ_ASSOC)]);
have `~ (e2.iiid.proc = p)`; e(ssimp[viewed_events_def,proc_def]);
e(MY_DISJL_TAC ``(e1,e2) IN E.intra_causality \/ (e1,e2) IN preserved_program_order E \/
          (e1,e2) IN X.write_serialization \/ (e1,e2) IN X.lock_serialization \/ (e1,e2) IN X.rfmap``);
 (* (e1,e2) IN E.intra_causality *)
 have `e2.iiid = e1.iiid`; 
  e(ssimp[valid_execution'_def,wf_E_def,wf_intra_causality_def,RES_FORALL_THM,pairTheory.FORALL_PROD]);
 have `e1 IN mem_store`;
  e(ssimp[viewed_events_def,proc_def,reverse_SPECIFICATION]); 
 e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);

 (* (e1,e2) IN preserved_program_order E *)
 have `e2.iiid.proc = e1.iiid.proc`; 
  e(ssimp[preserved_program_order_def,po_strict_def]);
 have `e1 IN mem_store`;
  e(ssimp[viewed_events_def,proc_def,reverse_SPECIFICATION]); 
 e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);

 have `(e1 IN mem_store) /\ (e2 IN mem_store)`; e(tac[write_serialization_typing]);
 e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);

 have `e1 IN locked E /\ e2 IN locked E`; e(tac[IN_lock_serialization_typing]);
 e(tac[]);

 (* (e1,e2) IN X.rfmap *)
 have `loc e1 = loc e2`; e(tac[rfmap_typing]);
 have `e1 IN mem_events \/ e1 IN reg_events`; e(tac[DISJOINT_events]);
 dl();
  have `e2 IN mem_events`; e(tac[loc_eq_loc_mem_events_reg_events]);
  e(tac[]);
 
  have `e2 IN reg_events`; e(tac[loc_eq_loc_mem_events_reg_events]);
  have `(e1.iiid.proc = e2.iiid.proc)`; e(tac[rfmap_typing]);
  e(ssimp[viewed_events_def,proc_def]);
  e(ssimp[reverse_SPECIFICATION]);
  e(tac[DISJOINT_events]);
val happens_before_local_foreign_2 = top_thm();

g `
valid_execution' E X
==> ((e1,e2) IN happens_before E X \/ (e2,e1) IN happens_before E X)
==> (e1 IN viewed_events E p /\ ~(e2 IN viewed_events E p))
==> e1 IN mem_events UNION locked E
`;
e(tac[happens_before_local_foreign_1,happens_before_local_foreign_2]);
val happens_before_local_foreign = top_thm();

   
(******************************************************************************)
(* prelude *)

val trivial_acyclic_def = Define `
trivial_acyclic r1 r2 = ~ ? x y. (x,y) IN r1 /\ (y,x) IN r2
`;

val reg_events_agree_def = Define `
reg_events_agree vop X_vop = 
  (! e1 e2. (e1,e2) IN vop /\ e1 IN reg_events /\ e2 IN reg_events /\ 
       (e1.iiid.program_order_index = e2.iiid.program_order_index) ==> (e1,e2) IN X_vop)
`;

(* FIXME tidy away the following clauses into defns as reg_events_agree *)
val wf_vop_def = Define `
wf_vop (E,X) p viewed_events_p vop =

  linear_order vop viewed_events_p

  (* FIXME don't we want this a bit stronger? vop is in po order, and
  respects iico; or is this what it says anyway? *)
  (* vop is in po_order, at least for non mem_store events *)
  (* get that respects iico from fact that no back hb edges *)
  /\ RRESTRICT (po_strict E) (viewed_events_p INTER (COMPL mem_store)) SUBSET vop
  
  (* vop is OK so far *)
  /\ check_atomicity' E.atomicity vop

  (* this check is subsumed by a single check on no hb back edges, and a final global check using nasty
  /\ check_causality' vop (happens_before EE XX)
  *)
  /\ trivial_acyclic vop (happens_before E X)

  (* these two subsumed by a stronger requirement- reads and writes match XX.vop exactly
  /\ check_rfmap_initial' viewed_events_p vop XX.rfmap XX.initial_state 
  /\ check_rfmap_written' vop XX.rfmap  
  *)
  (* reads and writes in vop agree with X_vop *)
  (* we strengthen this to include locked events *)
  /\ (vop INTER ((mem_events UNION locked E) CROSS (mem_events UNION locked E)) SUBSET (X.vo p))
  (* don't forget register events *)
  /\ reg_events_agree vop (X.vo p)

  (* finiteness *)
  /\ check_finite_vop vop
`;

(* a limit event is such that, if you include a limit event you
include all X.vo p previous mem_events *)

(* FIXME if we make this <= then we can alter the defn of niceness events to avoid the <= *)
val limit1_def = Define `
limit1 (E,X) p n e1 = 
  e1 IN viewed_events E p 
  /\ (e1.iiid.proc = p) /\ (e1.iiid.program_order_index = n) 
  /\ (e1 IN mem_load \/ locked E e1)
 `;

val limit2_def = Define `
limit2 (E,X) p n e2 = 
  e2 IN viewed_events E p
  /\ (! m e. limit1 (E,X) p m e ==> e.iiid.program_order_index <= n) 
  /\ (e2.iiid.proc = p) /\ (e2.iiid.program_order_index <= n) 
  /\ (e2 IN mem_store)
`;

val limit3_def = Define `
limit3 (E,X) p n e3 = 
  e3 IN viewed_events E p
  /\ (! m e. (limit1 (E,X) p m e \/ limit2 (E,X) p m e) ==> e.iiid.program_order_index <= n)
  /\ (e3.iiid.program_order_index <= n) 
  /\ (e3 IN mem_store)
`;

val limit_event_def = Define `
limit_event (E,X) p n e = 
  (limit1 (E,X) p n e) \/ (limit2 (E,X) p n e) \/ (limit3 (E,X) p n e)
`;

val niceness_events_def = Define `
niceness_events (E,X) p n = 
  E.events INTER
  let X_vop = X.vo p in
  (* all reg_events upto n *)
  { e | e IN reg_events /\ (e.iiid.proc = p) /\ e.iiid.program_order_index <= n }
  UNION
  (* set of mem events that are <=X_vop a mem_load or locked instruction whose po is <= n *)
  (* FIXME don't need m <= in following *)
  { e | e IN mem_events /\ ? m e'. m <=n /\ limit_event (E,X) p m e' /\ (e,e') IN X_vop }
`;

val new_events_def = Define `
(new_events (E,X) p 0 = niceness_events (E,X) p 0)
/\ (new_events (E,X) p (SUC n) = niceness_events (E,X) p (SUC n) DIFF niceness_events (E,X) p n)
`;

(* build up the new view_order piece by piece *)
val po_to_partial_view_order_def = Define `
po_to_partial_view_order (E,X) p n = 
  let old = case n of 0 -> {} || SUC n -> po_to_partial_view_order (E,X) p n in
  let new = RRESTRICT (X.vo p) (new_events (E,X) p n) in
  linear_order_append old new
`;

(* then union together the approximants *)
val po_to_view_order_def = Define `
po_to_view_order (E,X) p = (BIGUNION o (IMAGE (po_to_partial_view_order (E,X) p))) UNIV
`;

(******************************************************************************)
(* begin lemmas limit_event, niceness_event, new_event *)

(*
g `
valid_execution' E X
==> e1 IN mem_events
==> limit_event (E,X) p e2 /\ (e2.iiid.program_order_index <= n) /\ (e1,e2) IN X.vo p
==> ? e2. limit_event (E,X) p e2 /\ (e2.iiid.program_order_index <= SUC n) /\ (e1,e2) IN X.vo p
`;
e(intros);
e(elims);
e(ssimp[limit_event_def]);
e(QCUT_TAC `! x. x <= n ==> x <= SUC n`); e(numLib.ARITH_TAC);
e(tac[]);
val limit_event_mono = top_thm();
*)

g `
valid_execution' E X
==> limit_event (E,X) p n e
==> e IN viewed_events E p /\ ((e IN mem_load \/ e IN locked E) \/ e IN mem_store) /\ (e.iiid.program_order_index <= n)
`;
e(intros);
e(ssimp[limit_event_def,limit1_def,limit2_def,limit3_def]);
e(ssimp[reverse_SPECIFICATION]);
e(tac[less_thms]);
val limit_event_typing = top_thm();

g `
valid_execution' E X
==> FINITE { e' | ? m. m <= n /\ limit_event (E,X) p m e' }
`;
e(intros);
have `! n. FINITE { e | e IN E.events /\ e.iiid.program_order_index <= n }`; e(tac[FINITE_program_order_index_leq]);
have `FINITE { m | m <= n }`; e(tac[FINITE_leq]);
have `! m. FINITE { e' | limit_event (E,X) p m e'}`;
 e(intros);
 have `{ e' | limit_event (E,X) p m e'} SUBSET { e' | e' IN E.events /\ e'.iiid.program_order_index <= m }`;
  e(ssimp[SUBSET_DEF]);
  e(tac[limit_event_typing,viewed_events_SUBSET_E_events,SUBSET_DEF]);
 e(tac[SUBSET_FINITE]);
have `FINITE ((BIGUNION o IMAGE (\ m. { e' | limit_event (E,X) p m e' })) { m | m <= n})`;
 e(MATCH_MP_TAC FINITE_BIGUNION_o_IMAGE);
 e(ssimp[]);
have `((BIGUNION o IMAGE (\ m. { e' | limit_event (E,X) p m e' })) { m | m <= n})
  = { e' | ? m. m <= n /\ limit_event (E,X) p m e' }`;
 e(simp[BIGUNION_o_IMAGE_eq]);
 e(ssimp[]);
e(ssimp[]);
val FINITE_limit_event = top_thm();

g `
niceness_events (E,X) p n SUBSET niceness_events (E,X) p (SUC n)
`;
e(intros);
e(ssimp[SUBSET_DEF]);
e(ssimp[niceness_events_def,LET_THM]);
e(intros);
e(elims);
dl();
  dr1();
  e(tac[less_thms]);

  dr2();
  e(tac[less_thms]);
val niceness_events_SUC = top_thm();

g `
m <= n ==> niceness_events (E,X) p m SUBSET niceness_events (E,X) p n
`;
e(tac[niceness_events_SUC,r_mono]);
val niceness_events_mono = top_thm(); 

g `
valid_execution' E X
==> p IN E.procs
==> niceness_events (E,X) p n SUBSET viewed_events E p
`;
e(intros);
e(ssimp[SUBSET_DEF]);
e(ssimp[niceness_events_def,LET_THM]);
e(intros);
e(elims);
dl();
 e(ssimp[viewed_events_def,proc_def]);

 e(elims);
 e(ssimp[valid_execution'_def,RES_FORALL_THM]);
 xal ``p:proc``;
 e(ssimp[valid_vop_def]);
 e(ssimp[]);
 e(tac[linear_order_typing]);
val niceness_events_viewed_events = top_thm();

g `
valid_execution' E X
==> p IN E.procs
==> e IN E.events
==> e IN COMPL mem_store
==> ((e IN niceness_events (E,X) p n) = ((e.iiid.proc = p) /\ e.iiid.program_order_index <= n))
`;
e(intros);
e(EQR_TAC);
 (* e IN niceness_events (E,X) p n ==> (e.iiid.proc = p) /\ e.iiid.program_order_index <= n *)
 e(ssimp[niceness_events_def,LET_THM]);
 dl(); e(tac[]);
 e(elims);
 e(QCUT_TAC `e IN mem_load`); e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
 e(QCUT_TAC `e IN viewed_events E p`); e(tac[X_vop_typing]);
 e(QCUT_TAC `e.iiid.proc = p`); 
  e(ssimp[viewed_events_def,RES_FORALL_THM,proc_def]);
  e(ASSUME_TAC DISJOINT_mem_load_mem_store);
  e(ssimp[IN_DISJOINT]);
  e(tac[SPECIFICATION]);
 e(QCUT_TAC `e' IN E.events /\ e' IN viewed_events E p`); e(tac[X_vop_typing]);
 e(ssimp[]);
 e(CCONTR_TAC);
 e(ssimp[arithmeticTheory.NOT_LESS_EQUAL]);
 e(QCUT_TAC `limit1 (E,X) p e.iiid.program_order_index e`); e(ssimp[limit1_def]);
 e(QCUT_TAC `limit1 (E,X) p m e'`); 
  e(ssimp[limit_event_def]);
  e(REPEAT DISJL_TAC);
   e(INIT_TAC);
 
   e(ssimp[limit2_def]);
   e(tac[less_thms]);

   e(ssimp[limit3_def]);
   e(tac[less_thms]);
 e(ssimp[limit1_def]);
 e(QCUT_TAC `(e',e) IN preserved_program_order E`); 
  e(ssimp[preserved_program_order_def]);
  e(ssimp[po_strict_def]);
  e(ssimp[reverse_SPECIFICATION]);
  (* but mem_store mem_load are not in ppo *)
  e(tac[less_thms]);
 e(QCUT_TAC `(e',e) IN happens_before E X`); e(ssimp[happens_before_def]);
 e(tac[happens_before_X_vop]);

 have `e IN viewed_events E p`; e(ssimp[viewed_events_def,proc_def]);
 e(ssimp[niceness_events_def,LET_THM]);
 e(QCUT_TAC `e IN reg_events \/ e IN mem_events`); e(tac[reg_events_or_mem_events]);
 dl(); e(ssimp[]);
 dr2();
 e(ssimp[]);
 e(QCUT_TAC `e IN mem_load`); e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
 er ``e.iiid.program_order_index``;
 er ``e:event``;
 e(ssimp[]);
 cr();
  e(ssimp[limit_event_def,limit1_def]);

  e(QCUT_TAC `e IN viewed_events E p`); e(ssimp[viewed_events_def,proc_def]);
  e(ssimp[valid_execution'_def,RES_FORALL_THM]);
  xal ``p:proc``;
  e(ssimp[valid_vop_def]);
  e(tac[linear_order_reflexive]);
val niceness_events_COMPL_mem_store_eq = top_thm();

g `
valid_execution' E X
==> p IN E.procs
==> e1 IN mem_store
==> (e1.iiid.proc = p)
==> e1 IN niceness_events (E,X) p n
==> e1.iiid.program_order_index <= n
`;
e(intros);
e(QCUT_TAC `e1 IN E.events`); e(tac[niceness_events_viewed_events,SUBSET_DEF,viewed_events_SUBSET_E_events]);
e(QCUT_TAC `e1 IN viewed_events E p`); e(ssimp[viewed_events_def,proc_def]);
e(QCUT_TAC `~ (e1 IN reg_events) /\ ~ (e1 IN mem_load)`); e(tac[DISJOINT_events]);
e(ssimp[niceness_events_def,LET_THM]);
e(RENAME_TAC [``e':event``|->``e2:event``]);
e(ssimp[]);
e(QCUT_TAC `e2 IN viewed_events E p /\ e2 IN E.events`); e(tac[X_vop_typing]);
e(CCONTR_TAC);
e(ssimp[arithmeticTheory.NOT_LESS_EQUAL]);
(* problem that e2 could be a limit3, in which case no ppo; but e1 is a later limit2, so not possible *)
e(QCUT_TAC `~ limit3 (E,X) p m e2`);
 ir();
 e(ssimp[limit3_def]);
 al ``e1.iiid.program_order_index``;
 xal ``e1:event``;
 il 0; defer(); e(tac[less_thms]);
 e(ssimp[]);
 e(ssimp[limit1_def,limit2_def]);
 e(tac[less_thms]);
e(QCUT_TAC `((e2 IN mem_load \/ e2 IN locked E) \/ e2 IN mem_store) /\ (e2.iiid.program_order_index <= m)`); e(tac[limit_event_typing]);
e(QCUT_TAC `e2.iiid.proc = p`); 
 e(ssimp[limit_event_def]);
 e(ssimp[limit1_def,limit2_def]);
 e(tac[]);
e(QCUT_TAC `(e2,e1) IN preserved_program_order E`);
 e(ssimp[preserved_program_order_def]);
 e(ssimp[po_strict_def]);
 e(ssimp[reverse_SPECIFICATION]);
 e(ssimp[]);
 e(tac[less_thms]);
e(QCUT_TAC `(e2,e1) IN happens_before E X`); e(ssimp[happens_before_def]);     
e(tac[happens_before_X_vop]);
val niceness_events_local_mem_store_program_order_index = top_thm();

g `
DISJOINT (niceness_events (E,X) p n) (new_events (E,X) p (SUC n))
/\ (niceness_events (E,X) p (SUC n) = niceness_events (E,X) p n UNION (new_events (E,X) p (SUC n))) 
`;
e(intros);
e(ASSUME_TAC niceness_events_SUC);
e(ssimp[new_events_def]);
e(set_tac);
e(tac[]);
val niceness_events_new_events = top_thm();

g `
valid_execution' E X
==> p IN E.procs
==> e IN E.events
==> e IN COMPL mem_store
==> e IN new_events (E,X) p n 
==> (e.iiid.proc = p) /\ (e.iiid.program_order_index=n)
`;
e(intros);
e(Cases_on `n`);
 e(ssimp[new_events_def]);
 e(QCUT_TAC `(e.iiid.proc = p) /\ (e.iiid.program_order_index <= 0)`); 
  e(ssimp[niceness_events_COMPL_mem_store_eq] THEN NO_TAC);
 e(ssimp[] THEN NO_TAC);

 e(ssimp[new_events_def,LET_THM]);
 e(ssimp[niceness_events_COMPL_mem_store_eq]);
 e(tac[less_thms]); 
val new_events_COMPL_mem_store_eq = top_thm();

(* FIXME this essentially says that  new vop agrees with X.vo p on mem_events - shouldn't it say that? *)
(* this is used to show that vop agrees with X_vop on mem_events *)
(* would like to strengthen to locked events as well *)
g `
valid_execution' E X
==> p IN E.procs
==> e1 IN mem_events /\ e2 IN mem_events
==> (e1,e2) IN (X.vo p)
==> e2 IN niceness_events (E,X) p n
==> e1 IN niceness_events (E,X) p n
`;
e(intros);
e(elims);
e(QCUT_TAC `~ (e1 IN reg_events) /\ ~ (e2 IN reg_events)`); e(ssimp[mem_events_eq_mem_load_UNION_mem_store]); e(tac[DISJOINT_events]);
e(ASSUME_TAC X_vop_typing);
e(ssimp[niceness_events_def,LET_THM]);
e(elims);
e(RENAME_TAC [``e':event``|->``e:event``]);
er ``m:num``;
er ``e:event``;
e(ssimp[]);
e(ssimp[valid_execution'_def,RES_FORALL_THM]);
xal ``p:proc``;
e(ssimp[valid_vop_def]);
e(ssimp[linear_order_def]);
e(tac[]);
val niceness_events_mem_events_down_closed = top_thm();

(* FIXME following looks tough to patch in second part *)
(* N.B. the difficult part of the following is for foreign events *)
g `
valid_execution' E X
==> p IN E.procs
==> e1 IN viewed_events E p /\ e2 IN viewed_events E p
==> es IN E.atomicity
==> e1 IN es /\ e2 IN es
==> ((e1 IN niceness_events (E,X) p n) = (e2 IN niceness_events (E,X) p n))
`;
e(QCUT_TAC `! e1 e2.
  valid_execution' E X
==> p IN E.procs
==> e1 IN viewed_events E p /\ e2 IN viewed_events E p
==> es IN E.atomicity
==> e1 IN es /\ e2 IN es
==> (e1 IN niceness_events (E,X) p n) 
==> (e2 IN niceness_events (E,X) p n)
`);
defer(); e(tac[]);
e(intros);
e(QCUT_TAC `e2 IN E.events`); e(tac[atomicity_typing]);
e(QCUT_TAC `locked E e1 /\ locked E e2`); e(ssimp[locked_def]); e(tac[]);
e(QCUT_TAC `(e2.iiid = e1.iiid)`); e(tac[atomicity_typing]);
e(ssimp[niceness_events_def,LET_THM]); 
dl();
 (* e1 IN reg_events /\ (e1.iiid.proc = p) /\ e1.iiid.program_order_index <= n *)
 e(elims);
 e(QCUT_TAC `e2 IN reg_events \/ e2 IN mem_events`); e(tac[reg_events_or_mem_events]);
 dl();
  dr1();
  e(tac[atomicity_typing]);

  dr2();
  e(ssimp[]);
  er ``e2.iiid.program_order_index``;
  er ``e2:event``;
  e(ssimp[]);
  cr();
   e(ssimp[limit_event_def,limit1_def]);

   e(tac[atomicity_typing,X_vop_reflexive]);
val here = p();

dropn 1; add(here);
 (* FIXME following looks tough to patch *)
 (* e1 IN mem_events /\ ? e'. ... *) 
 e(elims);
 e(QCUT_TAC `e' IN E.events /\ e' IN viewed_events E p`); e(tac[X_vop_typing]);
 e(QCUT_TAC `e'.iiid.program_order_index <= m`); e(tac[limit_event_typing]);
 e(QCUT_TAC `e2 IN reg_events \/ e2 IN mem_events`); e(tac[reg_events_or_mem_events]);
 e(PAT_X_ASSUM ``e2 IN reg_events \/ e2 IN mem_events`` (fn th => DISJ_CASES_TAC th));
  (* e2 IN reg_events *)
  e(elims);
  dr1();
  e(ssimp[]);
  e(QCUT_TAC `~(mem_store e2)`);
   e(ssimp[reverse_SPECIFICATION]); 
   e(tac[DISJOINT_events]);
  e(QCUT_TAC `e2.iiid.proc = p`); e(ssimp[viewed_events_def,proc_def]);
  e(ssimp[]);
  e(CCONTR_TAC);
  e(ssimp[arithmeticTheory.NOT_LESS_EQUAL]);
  (* same problem as before, that e' may be a limit3, but then e1 is a later limit *)
  e(QCUT_TAC `~ limit3 (E,X) p m e'`);
   ir();
   e(ssimp[limit3_def]);
   al ``e1.iiid.program_order_index``;
   xal ``e1:event``;
   il 0; defer(); e(tac[less_thms]);
   e(ssimp[]);
   e(ssimp[limit1_def]);
  e(QCUT_TAC `((e' IN mem_load \/ e' IN locked E) \/ e' IN mem_store) /\ (e'.iiid.program_order_index <= m)`); e(tac[limit_event_typing]);
  e(QCUT_TAC `e'.iiid.proc = p`); 
   e(ssimp[limit_event_def]);
   e(ssimp[limit1_def,limit2_def]);
   e(tac[]);
  e(QCUT_TAC `e1.iiid.program_order_index <= e'.iiid.program_order_index`);
   e(CCONTR_TAC);
   e(ssimp[arithmeticTheory.NOT_LESS_EQUAL]);
   e(QCUT_TAC `(e',e1) IN preserved_program_order E`);
    e(ssimp[preserved_program_order_def]);
    e(ssimp[po_strict_def]);
    e(ssimp[reverse_SPECIFICATION]);
    dl();
     e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
     e(tac[]);

     e(ssimp[mem_events_eq_mem_load_UNION_mem_store] THEN NO_TAC);
   e(QCUT_TAC `(e',e1) IN happens_before E X`); e(ssimp[happens_before_def]);
   e(tac[happens_before_X_vop]);
  e(tac[less_thms]);
val here = p();

dropn 1; add(here);
  (* e2 IN mem_events *)
  e(elims);
  dr2();
  e(ssimp[]);
  e(Cases_on `(e2,e') IN X.vo p`); 
   er ``m:num``;
   er ``e':event``;
   e(ssimp[]);
 
   (* again, seems that we have to argue that e' is not a limit3 *)
   (* but in this case it looks like e' could be a limit3 *)
   e(QCUT_TAC `~((e2,e') IN X.vo p)`); e(INIT_TAC);
   e(QCUT_TAC `(e',e2) IN X.vo p`);
    e(ssimp[valid_execution'_def,RES_FORALL_THM]);
    xal ``p:proc``;
    e(ssimp[valid_vop_def]);
    e(ssimp[linear_order_def]);
    e(tac[]);
   e(QCUT_TAC `(e1,e') IN X.vo p`); e(INIT_TAC);
   e(QCUT_TAC `(e1,e2) IN X.vo p`);
    e(ssimp[valid_execution'_def,RES_FORALL_THM]);
    xal ``p:proc``;
    e(ssimp[valid_vop_def]);
    e(ssimp[linear_order_def]);
    e(tac[]);
   e(QCUT_TAC `e' IN es`);
    e(ssimp[valid_execution'_def,RES_FORALL_THM]);
    xal ``p:proc``;
    e(ssimp[valid_vop_def]);
    e(ssimp[check_atomicity'_def,RES_FORALL_THM]);
    e(ssimp[SUBSET_DEF]);
    e(tac[]);
   e(QCUT_TAC `e2.iiid = e'.iiid`); e(tac[atomicity_typing]);
   e(ssimp[]);
   (* g: limit_event (E,X) p e'.iiid.program_order_index e2 *)
   e(Cases_on `e'.iiid.proc = p`);
    er ``e2.iiid.program_order_index``;
    er ``e2:event``;
    cr(); e(tac[less_thms]);
    cr(); defer(); e(tac[X_vop_reflexive]);
    e(QCUT_TAC `limit1 (E,X) p e'.iiid.program_order_index e'`); e(ssimp[limit_event_def,limit1_def]); e(ssimp[locked_def]); e(tac[less_thms]);
    e(ssimp[limit_event_def,limit1_def] THEN NO_TAC);

    er ``m:num``;
    er ``e2:event``;
    e(QCUT_TAC `limit3 (E,X) p m e'`); 
     e(ssimp[limit_event_def]);
     e(ssimp[limit1_def,limit2_def]);
    e(QCUT_TAC `~(e2.iiid.proc = p)`); e(tac[]);
    e(QCUT_TAC `e2 IN mem_store`);  
     e(ssimp[viewed_events_def,reverse_SPECIFICATION,proc_def]);
    cr(); e(tac[less_thms]);
    cr(); defer(); e(tac[X_vop_reflexive]);    
    e(ssimp[limit_event_def,limit3_def]);
    e(tac[less_thms]);
val niceness_events_locked_closed = top_thm();
(* FIXME certainly this is an awful proof, and should be improved *)

g `
valid_execution' E X
==> p IN E.procs
==> assumption2 E
==> e1 IN (mem_events UNION locked E) /\ e2 IN (mem_events UNION locked E)
==> (e1,e2) IN (X.vo p)
==> e2 IN niceness_events (E,X) p n
==> e1 IN niceness_events (E,X) p n
`;
e(intros);
e(elims);
e(ASSUME_TAC X_vop_typing); e(ssimp[]);
e(Cases_on `? es. es IN E.atomicity /\ e1 IN es /\ e2 IN es`); e(tac[niceness_events_locked_closed]);
e(ssimp[UNION_DEF]);
e(QCUT_TAC `
  ? e1' e2'. e1' IN mem_events /\ e2' IN mem_events /\ (e1',e2') IN X.vo p
  /\ (e1' IN niceness_events (E,X) p n = e1 IN niceness_events (E,X) p n)
  /\ (e2' IN niceness_events (E,X) p n = e2 IN niceness_events (E,X) p n)
`); 
 defer(); e(tac[niceness_events_mem_events_down_closed]);
e(Cases_on `e1 IN mem_events`);
 e(ssimp[]);
 er ``e1:event``;
 e(Cases_on `e2 IN mem_events`);
  er ``e2:event``;
  e(ssimp[] THEN NO_TAC);

  (* ~(e2 IN mem_events), e2 IN locked E *)
  e(ssimp[assumption2_def]);
  e(ssimp[IN_locked]);
  e(QCUT_TAC `? e2'. e2' IN es /\ e2' IN mem_load`); e(tac[]);
  e(elims);
  e(QCUT_TAC `e2'.iiid = e2.iiid`); e(tac[atomicity_typing]);
  e(QCUT_TAC `e2.iiid.proc = p`);
   e(ssimp[viewed_events_def]);
   e(ssimp[proc_def]);
   e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
   e(tac[SPECIFICATION]);     
  e(QCUT_TAC `e2' IN E.events`);
   e(REPEAT (TM_THIN_TAC ``! x. P x``));
   e(ssimp[valid_execution'_def]);
   e(ssimp[wf_E_def]);
   e(ssimp[wf_atomicity_def,RES_FORALL_THM]);
   xal ``es:event set``;
   e(ssimp[]);
   e(ssimp[] THEN NO_TAC);   
  e(QCUT_TAC `e2' IN viewed_events E p`); 
   e(ssimp[viewed_events_def]);
   e(ssimp[proc_def] THEN NO_TAC);
  er ``e2':event``;
  e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
  cr();
   (* (e1,e2') IN X.vo p *)
   e(CCONTR_TAC);
   e(QCUT_TAC `(e2',e1) IN X.vo p`);
    e(ssimp[valid_execution'_def,RES_FORALL_THM]);
    xal ``p:proc``;
    e(ssimp[valid_vop_def]);
    e(ssimp[linear_order_def,RES_FORALL_THM]); 
    e(tac[]);
   e(QCUT_TAC `e1 IN es`);
    e(REPEAT (TM_THIN_TAC ``! x. P x``));   
    e(ssimp[valid_execution'_def,RES_FORALL_THM]);
    xal ``p:proc``;
    e(ssimp[valid_vop_def]);
    e(REORDER_TAC ``check_atomicity' E.atomicity (X.vo p)``);
    e(ssimp[check_atomicity'_def,RES_FORALL_THM]);
    e(ASM_CONV_TAC (fn ths => Canon.PRENEX_CONV) 0);
    xal ``es:event set``;
    xal ``e2':event``;
    xal ``e2:event``;
    e(ssimp[]);
    il 0; e(ssimp[linear_order_def]); e(tac[]);
    e(ssimp[]);
    e(QCUT_TAC `e1 IN {e | (e2',e) IN X.vo p /\ (e,e2) IN X.vo p}`);
     e(ssimp[]);
    e(ssimp[SUBSET_DEF]);
   e(tac[]);

   (* e2' IN niceness_events (E,X) p n *)
   e(tac[niceness_events_locked_closed]);
val here = p();

dropn 1; add(here);   
 (* ~(e1 IN mem_events) *)
 e(ssimp[assumption2_def]);
 e(ssimp[IN_locked]);
 e(QCUT_TAC `? e1'. e1' IN es /\ e1' IN mem_load`); e(tac[]);
 e(elims);
 e(QCUT_TAC `e1'.iiid = e1.iiid`); e(tac[atomicity_typing]);
 e(QCUT_TAC `e1.iiid.proc = p`);
  e(ssimp[viewed_events_def]);
  e(ssimp[proc_def]);
  e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
  e(tac[SPECIFICATION]);     
 e(QCUT_TAC `e1' IN E.events`);
  e(REPEAT (TM_THIN_TAC ``! x. P x``));
  e(ssimp[valid_execution'_def]);
  e(ssimp[wf_E_def]);
  e(ssimp[wf_atomicity_def,RES_FORALL_THM]);
  xal ``es:event set``;
  e(ssimp[]);
  e(ssimp[] THEN NO_TAC);   
 e(QCUT_TAC `e1' IN viewed_events E p`); 
  e(ssimp[viewed_events_def]);
  e(ssimp[proc_def] THEN NO_TAC);
 er ``e1':event``;
 e(Cases_on `e2 IN mem_events`);
  er ``e2:event``;
  e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
  cr();
   (* (e1',e2) IN X.vo p *)
   e(CCONTR_TAC);
   e(QCUT_TAC `(e2,e1') IN X.vo p`);
    e(ssimp[valid_execution'_def,RES_FORALL_THM]);
    xal ``p:proc``;
    e(ssimp[valid_vop_def]);
    e(ssimp[linear_order_def,RES_FORALL_THM]); 
    e(tac[]);
   e(QCUT_TAC `e2 IN es`);
    e(REPEAT (TM_THIN_TAC ``! x. P x``));   
    e(ssimp[valid_execution'_def,RES_FORALL_THM]);
    xal ``p:proc``;
    e(ssimp[valid_vop_def]);
    e(REORDER_TAC ``check_atomicity' E.atomicity (X.vo p)``);
    e(ssimp[check_atomicity'_def,RES_FORALL_THM]);
    e(ASM_CONV_TAC (fn ths => Canon.PRENEX_CONV) 0);
    xal ``es:event set``;
    xal ``e1:event``;
    xal ``e1':event``;
    e(ssimp[]);
    il 0; e(ssimp[linear_order_def]); e(tac[]);
    e(ssimp[]);
    e(QCUT_TAC `e2 IN {e | (e1,e) IN X.vo p /\ (e,e1') IN X.vo p}`);
     e(ssimp[]);
    e(ssimp[SUBSET_DEF]);
   e(tac[]);

   (* (e1' IN niceness_events (E,X) p n = e1 IN niceness_events (E,X) p n) *)
   e(tac[niceness_events_locked_closed]);
val here = p();

dropn 1; add(here);   
  (* ~(e2 IN mem_events), e2 IN locked E *)
  e(ssimp[assumption2_def]);
  e(ssimp[IN_locked]);
  e(QCUT_TAC `? e2'. e2' IN es' /\ e2' IN mem_load`); e(tac[]);
  e(elims);
  e(QCUT_TAC `e2'.iiid = e2.iiid`); e(tac[atomicity_typing]);
  e(QCUT_TAC `e2.iiid.proc = p`);
   e(ssimp[viewed_events_def]);
   e(ssimp[proc_def]);
   e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
   e(tac[SPECIFICATION]);     
  e(QCUT_TAC `e2' IN E.events`);
   e(REPEAT (TM_THIN_TAC ``! x. P x``));
   e(ssimp[valid_execution'_def]);
   e(ssimp[wf_E_def]);
   e(ssimp[wf_atomicity_def,RES_FORALL_THM]);
   xal ``es':event set``;
   e(ssimp[]);
   e(ssimp[] THEN NO_TAC);   
  e(QCUT_TAC `e2' IN viewed_events E p`); 
   e(ssimp[viewed_events_def]);
   e(ssimp[proc_def] THEN NO_TAC);
  er ``e2':event``;
  e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
  e(REPEAT CONJR_TAC);
   (* (e1',e2') IN X.vo p *)
   (* seems easiest to resort to lock_serialization here *)
   e(QCUT_TAC `~ (es = es')`); e(tac[]);
   e(QCUT_TAC `valid_execution' E X`); e(INIT_TAC);
   e(ASM_CONV_TAC (fn ths => SIMP_CONV (srw_ss()) ([valid_execution'_def,RES_FORALL_THM])) 0);
   e(ssimp[lock_serialization_candidates_def,LET_THM,RES_FORALL_THM,RES_EXISTS_THM]);
   e(ssimp[strict_linearisations_def]);
   e(ssimp[strict_linear_order_def,RES_FORALL_THM]);
   e(QCUT_TAC `(es,es') IN lin \/ (es',es) IN lin`); 
    e(ASM_CONV_TAC (fn ths => Canon.PRENEX_CONV) 0);
    e(PAT_X_ASSUM ``!x1 x2. x1 IN E.atomicity ==> P x1 x2`` (fn th => ASSUME_TAC (SPECL [``es:event set``,``es':event set``] th)));
    e(ssimp[]);
   dl();
    e(QCUT_TAC `(e1',e2') IN X.lock_serialization`);
     e(ssimp[]);
     er ``(es:event set,es':event set)``;
     e(ssimp[]);
    e(QCUT_TAC `(e1',e2') IN happens_before E X`); 
     e(ssimp[happens_before_def]);
    e(tac[happens_before_X_vop]);
   
    (* (es',es) in lin *)
    e(QCUT_TAC `(e2,e1) IN X.lock_serialization`);
     e(ssimp[]);
     er ``(es':event set,es:event set)``;
     e(ssimp[]);
    e(QCUT_TAC `(e2,e1) IN happens_before E X`); 
     e(ssimp[happens_before_def]);
    e(QCUT_TAC `(e2,e1) IN X.vo p`); e(tac[happens_before_X_vop]);
    e(QCUT_TAC `e2 = e1`); 
     xal ``p:proc``;
     e(ssimp[valid_vop_def]);
     e(ssimp[linear_order_def] THEN NO_TAC);
    e(tac[]);

   (* (e1' IN niceness_events (E,X) p n = e1 IN niceness_events (E,X) p n) *)
   e(tac[niceness_events_locked_closed]);
   
   (* e2' IN niceness_events (E,X) p n *)
   e(tac[niceness_events_locked_closed]);
val niceness_events_mem_events_UNION_locked_down_closed = top_thm();

g `
valid_execution' E X
==> p IN E.procs
==> FINITE (niceness_events (E,X) p n)
`;
e(intros);
e(ssimp[niceness_events_def,LET_THM]);
have `! n. FINITE { e | e IN E.events /\ e.iiid.program_order_index <= n }`; e(tac[FINITE_program_order_index_leq]);

(* reg events first *)
have `FINITE (E.events INTER {e | e IN reg_events /\ (e.iiid.proc = p) /\ e.iiid.program_order_index <= n})`; 
 have `(E.events INTER {e | e IN reg_events /\ (e.iiid.proc = p) /\ e.iiid.program_order_index <= n}) 
   SUBSET { e | e IN E.events /\ e.iiid.program_order_index <= n }`; e(ssimp[SUBSET_DEF]);
 e(tac[SUBSET_FINITE]);

(* mem_events *)
have `? L. L = { e' | ? m. m <= n /\ limit_event (E,X) p m e' }`; e(tac[]);
e(elims);
have `FINITE L`; e(tac[FINITE_limit_event]);
have `! e'. FINITE { e | e IN mem_events /\ (e,e') IN X.vo p }`; 
 have `valid_vop E X (p,X.vo p)`; e(tac[valid_execution'_def]);
 e(ssimp[valid_vop_def]);
 e(ssimp[check_finite_vop_def]);
 have `! e'. { e | e IN mem_events /\ (e,e') IN X.vo p } SUBSET {e | (e,e') IN X.vo p}`; e(ssimp[SUBSET_DEF]);
 e(tac[SUBSET_FINITE]);
have `FINITE ((BIGUNION o IMAGE (\ e'. { e | e IN mem_events /\ (e,e') IN X.vo p })) L)`;
 e(MATCH_MP_TAC FINITE_BIGUNION_o_IMAGE);
 e(ssimp[]);
have `((BIGUNION o IMAGE (\ e'. { e | e IN mem_events /\ (e,e') IN X.vo p })) L)
 = {e | e IN mem_events /\ ?m e'. m <= n /\ limit_event (E,X) p m e' /\ (e,e') IN X.vo p}`;
 e(simp[BIGUNION_o_IMAGE_eq]);
 e(ssimp[]);
 e(tac[]);
have `FINITE (E.events INTER {e | e IN mem_events /\ ?m e'. m <= n /\ limit_event (E,X) p m e' /\ (e,e') IN X.vo p})`;
 e(tac[INTER_FINITE,INTER_COMM]);
e(ssimp[UNION_OVER_INTER]);
val FINITE_niceness_events = top_thm();

(* end lemmas *)
(******************************************************************************)
(* po_to_partial_view_order *)

g `
po_to_partial_view_order (E,X) p (SUC n) = 
  let old = po_to_partial_view_order (E,X) p n in
  let new = RRESTRICT (X.vo p) (new_events (E,X) p (SUC n)) in
  linear_order_append old new
`;
e(CONV_TAC (FORK_CONV (ONCE_REWRITE_CONV [po_to_partial_view_order_def],ALL_CONV)));
e(ssimp[LET_THM]);
val po_to_partial_view_order_SUC_def = top_thm();

g `
(po_to_partial_view_order (E,X) p n) SUBSET (po_to_partial_view_order (E,X) p (SUC n))
`;
e(ssimp[po_to_partial_view_order_SUC_def,LET_THM]);
e(tac[SUBSET_linear_order_append]);
val po_to_partial_view_order_SUC = top_thm();

g `
m <= n ==> (po_to_partial_view_order (E,X) p m) SUBSET (po_to_partial_view_order (E,X) p n)
`;
e(tac[po_to_partial_view_order_SUC,r_mono]);
val po_to_partial_view_order_mono = top_thm();

g `
e' IN niceness_events (E,X) p m
==> (e,e') IN po_to_partial_view_order (E,X) p (SUC m)
==> (e,e') IN po_to_partial_view_order (E,X) p m
`;
e(intros);
e(ssimp[po_to_partial_view_order_SUC_def,LET_THM]);
e(ssimp[new_events_def]);
e(ssimp[]);
e(ssimp[linear_order_append_def]);
e(ssimp[FIELD_def,DOM_def,RANGE_def,EXTENSION]);
e(ssimp[RRESTRICT_def]);
val po_to_partial_view_order_SUC_m_m = top_thm();

g `
e' IN niceness_events (E,X) p m
==> m <= n
==> (e,e') IN po_to_partial_view_order (E,X) p n
==> (e,e') IN po_to_partial_view_order (E,X) p m
`;
e(intros);
e(Induct_on `n - m`);
 e(intros);
 e(ssimp[]);
 have `m=n`; e(tac[less_thms]);
 e(tac[]);

 e(intros);
 e(Cases_on `n`); e(ssimp[]);
 xal ``n':num``;
 xal ``m:num``;
 e(ssimp[]);
 have `(SUC v = SUC n' - m) ==> (v = n' - m)`; e(numLib.ARITH_TAC);
 e(ssimp[]);
 have `(SUC (n' - m) = SUC n' - m) ==> m <= SUC n' ==> m <= n'`; e(numLib.ARITH_TAC);
 e(ssimp[]);
 have `e' IN niceness_events (E,X) p n'`; e(tac[niceness_events_mono,SUBSET_DEF]);
 e(tac[less_thms, po_to_partial_view_order_SUC_m_m]);
val po_to_partial_view_order_rev_mono = top_thm();


g `
check_finite_vop (po_to_partial_view_order (E,X) p m)
==> e' IN niceness_events (E,X) p m
==> FINITE {e | ? n. (e,e') IN po_to_partial_view_order (E,X) p n} 
`;
e(intros);
e(elims);
have `{e | ? n. (e,e') IN po_to_partial_view_order (E,X) p n} SUBSET {e | (e,e') IN po_to_partial_view_order (E,X) p m}`;
 e(ssimp[SUBSET_DEF]);
 e(intros);
 e(elims);
 e(RENAME_TAC [``x:event``|->``e:event``]);
 have `n <= m \/ m <= n`; e(numLib.ARITH_TAC);
 dl(); 
  e(tac[po_to_partial_view_order_mono,SUBSET_DEF]);
  
  (* m <= n *)
  e(tac[po_to_partial_view_order_rev_mono]);  
e(ssimp[check_finite_vop_def]);
e(tac[SUBSET_FINITE,SUBSET_DEF]);
val FINITE_po_to_partial_view_order = top_thm();


(******************************************************************************)
(* niceness base *)

g `
wf_vop (E,X) p {} {}
`;
e(ssimp[wf_vop_def]);
e(ssimp[check_atomicity'_def]);
e(ssimp[trivial_acyclic_def]);
e(ssimp[linear_order_def]);
e(ssimp[RES_FORALL_THM,DOM_def,RANGE_def]);
e(ssimp[RRESTRICT_def,EXTENSION]);
e(ssimp[reg_events_agree_def]);
e(ssimp[check_finite_vop_def]);
val base = top_thm();


(******************************************************************************)
(* begin step *)

dropn 100;
g `
valid_execution' E X 
==> p IN E.procs
==> assumption1 E
==> assumption2 E
==> 
  (let viewed_events_p = case SUCn of 0 -> {} || SUC n -> niceness_events (E,X) p n in
  let vop = case SUCn of 0-> {} || SUC n -> po_to_partial_view_order (E,X) p n in
  wf_vop (E,X) p viewed_events_p vop)
==> 
  (let viewed_events_p' = niceness_events (E,X) p (SUCn) in
  let vop' = po_to_partial_view_order (E,X) p (SUCn) in
  wf_vop (E,X) p viewed_events_p' vop')
`;
e(QCUT_TAC `? viewed_events_p. (case SUCn of 0 -> {} || SUC n -> niceness_events (E,X) p n) = viewed_events_p`); e(tac[]);
e(QCUT_TAC `? vop. (case SUCn of 0-> {} || SUC n -> po_to_partial_view_order (E,X) p n) = vop`); e(tac[]);
e(QCUT_TAC `? viewed_events_p'. niceness_events (E,X) p (SUCn) = viewed_events_p'`); e(tac[]);
e(QCUT_TAC `? vop'. po_to_partial_view_order (E,X) p (SUCn) = vop'`); e(tac[]);
e(QCUT_TAC `? X_vop. X.vo p = X_vop`); e(tac[]);
e(QCUT_TAC `? nes. (new_events (E,X) p (SUCn)) = nes`); e(tac[]);
e(elims);
e(QCUT_TAC `? new. RRESTRICT (X.vo p) nes = new`); e(tac[]);
e(elims);
e(ssimp[LET_THM]);
e(intros);
e(ssimp[wf_vop_def]);
(* obvious disjunction *)
e(QCUT_TAC `! e1 e2. (e1,e2) IN vop' ==> (e1,e2) IN vop \/ (e1 IN FIELD vop /\ e2 IN FIELD new) \/ (e1,e2) IN new`); 
 e(intros);
 e(Cases_on `SUCn`);
  e(ssimp[]);
  e(ssimp[new_events_def]);
  e(ssimp[po_to_partial_view_order_def,LET_THM]);
  e(SYM_TAC ``{} = X``);
  e(SYM_TAC ``{} = X``);
  e(ssimp[]);
  e(ssimp[linear_order_append_empty]);
  e(SYM_TAC ``RRESTRICT X_vop (new_events (E,X) p 0) = vop'``);
  e(ssimp[]);
  e(ssimp[new_events_def]);

  e(ssimp[po_to_partial_view_order_SUC_def,LET_THM]);
  e(SYM_TAC ``linear_order_append vop new = vop'``);
  e(ssimp[linear_order_append_def]);
  e(tac[]);
e(QCUT_TAC `vop SUBSET vop'`); 
 e(Cases_on `SUCn`);
  e(ssimp[]);
  e(SYM_TAC ``{} = vop``);
  e(ssimp[SUBSET_DEF]); 

  e(ssimp[]);
  e(tac[po_to_partial_view_order_SUC]);
(**)
e(QCUT_TAC `DISJOINT viewed_events_p nes /\ (viewed_events_p UNION nes = viewed_events_p')`); 
 e(Cases_on `SUCn`);
  e(ssimp[]);
  e(SYM_TAC ``{}:event set = viewed_events_p``);
  e(ssimp[DISJOINT_DEF]);
  e(ssimp[new_events_def]);

  e(ssimp[]);  
  e(tac[niceness_events_new_events]);
(**)
e(QCUT_TAC `linear_order new nes`);  
 e(QCUT_TAC `linear_order X_vop (viewed_events E p)`); 
  e(ssimp[valid_execution'_def,RES_FORALL_THM]); 
  xal ``p:proc``;
  e(ssimp[valid_vop_def]);
  e(tac[]);
 e(QCUT_TAC `nes SUBSET (viewed_events E p)`); 
  e(QCUT_TAC `viewed_events_p' SUBSET viewed_events E p`); e(tac[niceness_events_viewed_events]);
  e(set_tac);
  e(tac[]);
 e(QCUT_TAC `(viewed_events E p) INTER nes = nes`); 
  e(set_tac);
  e(tac[]);   
 e(tac[linear_order_RRESTRICT]);    
(**)
e(QCUT_TAC `linear_order vop' viewed_events_p'`); 
 e(Cases_on `SUCn`);
  e(ssimp[]);
  e(ssimp[po_to_partial_view_order_def,LET_THM]);
  e(tac[linear_order_linear_order_append_empty,UNION_EMPTY]);

  e(ssimp[po_to_partial_view_order_SUC_def,LET_THM]); 
  e(tac[linear_order_linear_order_append]);
(**)
e(QCUT_TAC `linear_order_append vop new = vop'`);
 e(Cases_on `SUCn`);
  e(ssimp[]);
  e(ssimp[po_to_partial_view_order_def,LET_THM]);

  e(ssimp[]);
  e(ssimp[po_to_partial_view_order_SUC_def,LET_THM]);
val here = p();

dropn 1; add(here);
(* FIXME does FIELD really come into it? *)
e(QCUT_TAC `FIELD vop = viewed_events_p`); e(tac[linear_order_FIELD]);
e(QCUT_TAC `FIELD new = nes`); e(tac[linear_order_FIELD]);
e(QCUT_TAC `vop' INTER ((mem_events UNION locked E) CROSS (mem_events UNION locked E)) SUBSET X_vop`); 
 e(ssimp[SUBSET_DEF]);
 e(intros);
 e(Cases_on `x`);
 e(RENAME_TAC [``q:event``|->``e1:event``,``r:event``|->``e2:event``]);
 xal ``e1:event``;
 xal ``e2:event``;
 e(ssimp[]);
 e(PAT_X_ASSUM ``(e1,e2) IN vop \/ e1 IN viewed_events_p /\ e2 IN nes \/ (e1,e2) IN new`` DISJ_CASES_TAC);
  (* (e1,e2) IN vop *)
  e(PAT_X_ASSUM ``!x.
             x IN vop /\ (FST x IN mem_events \/ FST x IN locked E) /\
             (SND x IN mem_events \/ SND x IN locked E) ==>
             x IN X_vop`` (fn th => ASSUME_TAC (ISPEC ``(e1:event,e2:event)`` th)));
  e(ssimp[] THEN NO_TAC);

  e(PAT_X_ASSUM ``e1 IN viewed_events_p /\ e2 IN nes \/ (e1,e2) IN new`` DISJ_CASES_TAC);
  (* e1 IN viewed_events_p /\ e2 IN nes *) 
  e(CCONTR_TAC);
  e(QCUT_TAC `e1 IN viewed_events_p' /\ e2 IN viewed_events_p'`);
   e(ASSUME_TAC niceness_events_new_events);
   e(set_tac);
   e(tac[]);
  e(QCUT_TAC `viewed_events_p' SUBSET viewed_events E p`); e(tac[niceness_events_viewed_events]);
  e(QCUT_TAC `e1 IN viewed_events E p /\ e2 IN viewed_events E p`); e(set_tac THEN NO_TAC);
  e(QCUT_TAC `(e2,e1) IN X_vop`); 
   e(ssimp[valid_execution'_def,RES_FORALL_THM]);
   xal ``p:proc``;
   e(ssimp[valid_vop_def]);
   e(ssimp[linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def,SUBSET_DEF]);
   e(tac[]);
  e(QCUT_TAC `e2 IN viewed_events_p`); 
   e(Cases_on `SUCn`);
    e(ssimp[]);
    e(SYM_TAC ``{}:event set = viewed_events_p``);
    e(ssimp[]);
    
    e(ssimp[]);
    e(tac[niceness_events_mem_events_UNION_locked_down_closed,IN_UNION]);
  e(set_tac);
  e(tac[]);

  (* (e1,e2) IN new *)
  e(SYM_TAC ``RRESTRICT X_vop nes = new``);
  e(ssimp[RRESTRICT_def] THEN NO_TAC);
(* back to main proof *)
e(cintros);

 (* linear_order vop' viewed_events_p' *)
 e(INIT_TAC);
val here = p();

dropn 1; add(here);
 (* RRESTRICT (po_strict E) (viewed_events_p' INTER COMPL mem_store) SUBSET vop' *)
 e(ssimp[SUBSET_DEF]);
 e(intros);
 e(Cases_on `x`);
 e(RENAME_TAC [``q:event``|->``e1:event``,``r:event``|->``e2:event``]);
 e(ssimp[RRESTRICT_def]);
 e(QCUT_TAC `e1 IN viewed_events E p /\ e2 IN viewed_events E p`); e(tac[niceness_events_viewed_events,SUBSET_DEF]);
 e(ssimp[po_strict_def]); 
 e(CCONTR_TAC);
 (* a ~((e1,e2) IN vop') *)
 e(QCUT_TAC `(e2,e1) IN vop'`); 
  e(ssimp[linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def,SUBSET_DEF]);
  e(tac[]);
 xal ``e2:event``;
 xal ``e1:event``;
 e(ssimp[]);
 e(REPEAT DISJL_TAC);
  (* (e2,e1) IN vop *)
  e(QCUT_TAC `(e1,e2) IN vop`); 
   xal ``(e1:event,e2:event)``;
   il 0; 
   e(ssimp[linear_order_def,RES_FORALL_THM,DOM_def,RANGE_def,SUBSET_DEF]);
   e(tac[]);

   e(INIT_TAC);
  e(QCUT_TAC `(e1,e2) IN vop'`); e(tac[]);
  e(tac[]);

  (* e2 IN viewed_events_p /\ e1 IN nes *)
  e(Cases_on `SUCn`); 
   e(ssimp[]);
   e(tac[IN_EMPTY]);
  e(ssimp[]);
  e(QCUT_TAC `e1.iiid.program_order_index=(SUC n)`); e(tac[new_events_COMPL_mem_store_eq,IN_COMPL,IN_DIFF]);
  e(QCUT_TAC `e2.iiid.program_order_index <= n`); e(tac[niceness_events_COMPL_mem_store_eq,IN_COMPL]);
  e(ssimp[]);
  e(QCUT_TAC `e2.iiid.program_order_index <= n /\ SUC n < e2.iiid.program_order_index ==> F`); e(numLib.ARITH_TAC);
  e(tac[]);  

  (* (e2,e1) IN new *)
  e(QCUT_TAC `e1 IN nes /\ e2 IN nes`); e(tac[linear_order_typing]);
  e(QCUT_TAC `e1.iiid.program_order_index=(SUCn)`); e(tac[new_events_COMPL_mem_store_eq,IN_COMPL]);
  e(QCUT_TAC `e2.iiid.program_order_index=(SUCn)`); e(tac[new_events_COMPL_mem_store_eq,IN_COMPL]);
  e(QCUT_TAC `e1.iiid.program_order_index < e2.iiid.program_order_index`); e(INIT_TAC);
  e(ssimp[] THEN NO_TAC);
val here = p();

dropn 1; add(here);  
 (* check_atomicity' E.atomicity vop' *)
 (* if in old- IH; if in new_events, in X_vop, else (e1,e2) e1:old, e2:new...? see below *)
 e(ssimp[check_atomicity'_def]);
 e(ssimp[RES_FORALL_THM]);
 e(intros);
 e(SIMP_TAC std_ss [SUBSET_DEF]);
 e(intros);
 e(ssimp[]);
 xal ``e1:event``;
 xal ``e2:event``;
 e(ssimp[]); 
 e(REPEAT DISJL_TAC);
  (* (e1,e2) IN vop *)
  e(Cases_on `SUCn`); e(ssimp[]); e(tac[IN_EMPTY]);
  e(ssimp[po_to_partial_view_order_SUC_def,LET_THM]);     
  e(QCUT_TAC `(e1,x) IN vop /\ (x,e2) IN vop`); e(tac[linear_order_append_lower]);
  e(REORDER_TAC ``! es. P es``);
  e(ASM_CONV_TAC (fn ths => Canon.PRENEX_CONV) 0);
  xal ``es:event set``;
  xal ``e1:event``;
  xal ``e2:event``;
  e(ssimp[]);
  e(ssimp[SUBSET_DEF] THEN NO_TAC);

  (* e1 IN viewed_events_p /\ e2 IN nes *)
  e(Cases_on `SUCn`); e(ssimp[]); e(tac[IN_EMPTY]);
  e(ssimp[]);
  e(QCUT_TAC `e1 IN viewed_events E p /\ e2 IN viewed_events E p`); 
   e(ASSUME_TAC (GEN ``n:num`` niceness_events_viewed_events));
   xal ``SUC n``;
   e(ssimp[SUBSET_DEF]);
   e(tac[linear_order_typing]);
  e(QCUT_TAC `e2 IN viewed_events_p`); e(tac[niceness_events_locked_closed]);
  e(ssimp[DISJOINT_DEF,EXTENSION]);
  e(tac[]);

  (* (e1,e2) IN new *)
  e(TM_THIN_TAC ``! es. P es``);
  e(QCUT_TAC `(e1,x) IN new /\ (x,e2) IN new`); 
   e(tac[linear_order_append_upper]);
  e(QCUT_TAC `(e1,e2) IN X_vop /\ (e1,x) IN X_vop /\ (x,e2) IN X_vop`); 
   e(ssimp[RRESTRICT_def,EXTENSION]);
   e(tac[]);
  e(ssimp[valid_execution'_def,RES_FORALL_THM]);
  xal ``p:proc``;
  e(ssimp[valid_vop_def,check_atomicity'_def,RES_FORALL_THM]); 
  e(REORDER_TAC ``! es'. es' IN E.atomicity ==> P es'``);
  e(ASM_CONV_TAC (fn ths => Canon.PRENEX_CONV) 0);
  e(FIRST_X_ASSUM (fn th => ASSUME_TAC 
    (SPECL [(*``p:proc``,*)``es:event set``,``e1:event``,``e2:event``(*,``x:event``*)] th)));
  e(ssimp[]);
  e(ssimp[SUBSET_DEF]);
val here = p();

dropn 1; add(here);  
 (* trivial_acyclic vop' (happens_before E X) *)
 e(ONCE_REWRITE_TAC[trivial_acyclic_def]);
 e(CCONTR_TAC);
 e(ssimp[]);
 e(RENAME_TAC [``x:event``|->``e1:event``,``y:event``|->``e2:event``]); 
 (* a: (e1,e2) IN vop' /\ (e2,e1) IN happens_before E X *)
 e(QCUT_TAC `? hb. happens_before E X = hb`); e(tac[]);
 e(elims THEN ssimp[]);
 xal ``e1:event``;
 xal ``e2:event``;
 e(ssimp[]);
 e(REPEAT DISJL_TAC);
  (* (e1,e2) IN vop *)
  e(ssimp[trivial_acyclic_def]); 
  e(tac[]);

  (* e1 IN viewed_events_p /\ e2 IN nes *)
  e(QCUT_TAC `e1 IN viewed_events E p /\ e2 IN viewed_events E p`); 
   e(ASSUME_TAC (GEN ``n:num`` niceness_events_viewed_events));
   xal ``SUCn:num``;
   e(ssimp[SUBSET_DEF]);
   e(tac[linear_order_typing]);
  e(QCUT_TAC `(e2,e1) IN X_vop /\ ~((e1,e2) IN X_vop)`); e(tac[happens_before_X_vop]);
  e(QCUT_TAC `e1 IN E.events /\ e2 IN E.events`); e(tac[X_vop_typing]);  
  e(Cases_on `e1 IN (mem_events UNION locked E) /\ e2 IN (mem_events UNION locked E)`); 
   e(PAT_X_ASSUM ``vop' INTER ((mem_events UNION locked E) CROSS (mem_events UNION locked E)) SUBSET X_vop`` 
     (fn th => ASSUME_TAC (SIMP_RULE (srw_ss()) [SUBSET_DEF] th)));
   xal ``(e1:event,e2:event)``;
   il 0; e(ssimp[] THEN NO_TAC);
   e(tac[SUBSET_DEF]);
  e(Cases_on `(e1 IN reg_events /\ e2 IN reg_events) \/ (e1 IN reg_events /\ e2 IN mem_load) \/ (e1 IN mem_load /\ e2 IN reg_events)`); 
   e(QCUT_TAC `~(e1 IN mem_events /\ e2 IN mem_events)`); 
    e(ssimp[UNION_DEF]);
    e(tac[]);
   e(QCUT_TAC `(e1.iiid.proc = e2.iiid.proc) /\ e1.iiid.program_order_index<e2.iiid.program_order_index`); 
    e(QCUT_TAC `e1 IN COMPL mem_store /\ e2 IN COMPL mem_store`);    
     e(ssimp[reverse_SPECIFICATION]);
     e(tac[DISJOINT_events]);
    e(Cases_on `SUCn`); e(ssimp[]); e(tac[IN_EMPTY]);
    e(QCUT_TAC `(niceness_events (E,X) p n = viewed_events_p) /\ (po_to_partial_view_order (E,X) p n = vop)`);
     e(ssimp[]);
    e(QCUT_TAC `(e1.iiid.proc = p) /\ e1.iiid.program_order_index <= n /\ (e2.iiid.proc = p) /\ (e2.iiid.program_order_index = SUC n)`);
     e(tac[niceness_events_COMPL_mem_store_eq,new_events_COMPL_mem_store_eq]);
    e(QCUT_TAC `! x. x <= n ==> x < SUC n`); e(numLib.ARITH_TAC);
    e(ssimp[]);
   e(tac[reg_reg_happens_before]);
  e(QCUT_TAC `(e1 IN reg_events /\ e2 IN mem_store) \/ (e1 IN mem_store /\ e2 IN reg_events)`); 
   e(ssimp[reverse_SPECIFICATION]);
   e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
   e(tac[DISJOINT_events]);
  e(elims);
  dl();
   (* (e1 IN reg_events /\ e2 IN mem_store) *)
   e(SYM_TAC ``happens_before E X = hb``);
   e(REORDER_TAC ``(e2,e1) IN hb``);
   e(ASM_CONV_TAC (fn ths => SIMP_CONV (srw_ss()) (ths@[happens_before_def])) 0);
   e(REPEAT DISJL_TAC);   
    (* (e2,e1) IN E.intra_causality *)
    e(ssimp[assumption1_def]);
    e(tac[]);

    (* (e2,e1) IN preserved_program_order E *)
    e(ssimp[preserved_program_order_def,reverse_SPECIFICATION]);
    e(QCUT_TAC `~ (e1 IN mem_load) /\ ~(e1 IN mem_store)`); e(tac[DISJOINT_events]);
    e(ssimp[]);
    e(QCUT_TAC `~ (loc e2 = loc e1)`); 
     e(CCONTR_TAC);
     e(ssimp[reg_events_def,loc_def]);
     e(Cases_on `e1.action` THEN ssimp[]);
     e(Cases_on `e2.action` THEN ssimp[] THEN NO_TAC);
    e(ssimp[]);
    e(QCUT_TAC `e1 IN locked E`); e(tac[]);
    e(QCUT_TAC `e2 IN mem_events`); e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
    e(ssimp[] THEN NO_TAC);

    (* (e2,e1) IN X.write_serialization *)
    e(ssimp[valid_execution'_def]);
    e(ssimp[IN_write_serialization_candidates,write_serialization_candidates_def]);
    e(ssimp[RES_FORALL_THM]);
    xal ``(e2:event,e1:event)``;
    e(ssimp[]);
    e(ssimp[get_l_stores_def]);
    e(ssimp[loc_def]);
    e(ssimp[reverse_SPECIFICATION]);
    e(tac[DISJOINT_events]);

    (* (e2,e1) IN X.lock_serialization *)
    e(QCUT_TAC `e1 IN locked E /\ e2 IN locked E`); e(tac[IN_lock_serialization_typing]);
    e(ssimp[] THEN NO_TAC);

    (* (e2,e1) IN X.rfmap *)
    e(QCUT_TAC `(e1 IN reg_events /\ e2 IN reg_events) \/ (e1 IN mem_events /\ e2 IN mem_events)`);
     e(ssimp[valid_execution'_def]);
     e(ssimp[IN_reads_from_map_candidates,reads_from_map_candidates_def,RES_FORALL_THM]);
     xal ``(e2:event,e1:event)``;
     e(ssimp[]);
     e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
     e(Cases_on `l`);
      e(ssimp[SPECIFICATION,mem_store_def,mem_load_def]);

      e(ssimp[SPECIFICATION,mem_store_def,mem_load_def]);
    e(ssimp[mem_events_eq_mem_load_UNION_mem_store] THEN NO_TAC);
val here = p();

dropn 1; add(here);  
   (* e1 IN mem_store /\ e2 IN reg_events *)
   e(SYM_TAC ``happens_before E X = hb``);
   e(REORDER_TAC ``(e2,e1) IN hb``);
   e(ASM_CONV_TAC (fn ths => SIMP_CONV (srw_ss()) (ths@[happens_before_def])) 0);
   e(REPEAT DISJL_TAC);   
    (* (e2,e1) IN E.intra_causality *)
    e(QCUT_TAC `e2.iiid.proc = p`);
     e(ssimp[viewed_events_def,proc_def,reverse_SPECIFICATION]);
     e(tac[DISJOINT_events]);
    e(QCUT_TAC `e2.iiid = e1.iiid`); 
     e(ssimp[valid_execution'_def,wf_E_def,wf_intra_causality_def]);
     e(ssimp[RES_FORALL_THM]);
     xal ``(e2:event,e1:event)``;
     e(ssimp[]);
    e(Cases_on `SUCn`); e(ssimp[]); e(tac[IN_EMPTY]);
    e(ssimp[]);
    e(QCUT_TAC `e2.iiid.program_order_index = e1.iiid.program_order_index`); e(tac[]);
    e(QCUT_TAC `e1.iiid.program_order_index <= n`); e(tac[niceness_events_local_mem_store_program_order_index]);
    e(QCUT_TAC `e2 IN COMPL mem_store`); e(ssimp[]); e(tac[DISJOINT_events]);
    e(QCUT_TAC `e2.iiid.program_order_index = SUC n`); e(tac[new_events_COMPL_mem_store_eq]);
    e(QCUT_TAC `! x y. x <= n /\ (y = x) /\ (y = SUC n) ==> F`); e(numLib.ARITH_TAC);
    e(tac[]); 

    (* (e2,e1) IN preserved_program_order E *)
    e(ssimp[preserved_program_order_def,reverse_SPECIFICATION]);
    e(QCUT_TAC `~ (e2 IN mem_load) /\ ~(e2 IN mem_store)`); e(tac[DISJOINT_events]);
    e(ssimp[]);
    e(QCUT_TAC `~ (loc e2 = loc e1)`); 
     e(CCONTR_TAC);
     e(ssimp[reg_events_def,loc_def]);
     e(Cases_on `e1.action` THEN ssimp[]);
     e(Cases_on `e2.action` THEN ssimp[] THEN NO_TAC);
    e(ssimp[]);
    e(QCUT_TAC `e2 IN locked E`); e(tac[]);
    e(QCUT_TAC `e1 IN mem_events`); e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
    e(ssimp[] THEN NO_TAC);

    (* (e2,e1) IN X.write_serialization *)
    e(ssimp[valid_execution'_def]);
    e(ssimp[IN_write_serialization_candidates,write_serialization_candidates_def]);
    e(ssimp[RES_FORALL_THM]);
    xal ``(e2:event,e1:event)``;
    e(ssimp[]);
    e(ssimp[get_l_stores_def]);
    e(ssimp[loc_def]);
    e(ssimp[reverse_SPECIFICATION]);
    e(tac[DISJOINT_events]);

    (* (e2,e1) IN X.lock_serialization *)
    (* the following, or use the fact that locked events are preserved in X_vop *)
    e(QCUT_TAC `e1 IN locked E /\ e2 IN locked E`); e(tac[IN_lock_serialization_typing]);
    e(ssimp[] THEN NO_TAC);

    (* rfmap *)
    e(QCUT_TAC `(e1 IN reg_events /\ e2 IN reg_events) \/ (e1 IN mem_events /\ e2 IN mem_events)`);
     e(ssimp[valid_execution'_def]);
     e(ssimp[IN_reads_from_map_candidates,reads_from_map_candidates_def,RES_FORALL_THM]);
     xal ``(e2:event,e1:event)``;
     e(ssimp[]);
     e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);
     e(Cases_on `l`);
      e(ssimp[SPECIFICATION,mem_store_def,mem_load_def]);

      e(ssimp[SPECIFICATION,mem_store_def,mem_load_def]);
    e(ssimp[mem_events_eq_mem_load_UNION_mem_store] THEN NO_TAC);

  (* (e1,e2) IN new *)
  e(QCUT_TAC `(e1,e2) IN X_vop`);
   e(SYM_TAC ``RRESTRICT X_vop nes = new``);
   e(ssimp[RRESTRICT_def]);
  e(QCUT_TAC `e1 IN viewed_events E p /\ e2 IN viewed_events E p`); e(tac[X_vop_typing]);
  e(tac[happens_before_X_vop]);

 (* vop' INTER (mem_events CROSS mem_events) SUBSET X_vop *)
 e(set_tac);
 e(tac[]);
val here8 = p();

dropn 1; add(here8);
 (* reg_events_agree vop' X_vop *)
 e(ssimp[reg_events_agree_def]);
 e(intros);
 have `e1 IN viewed_events E p /\ e2 IN viewed_events E p`; e(tac[niceness_events_viewed_events,linear_order_typing,SUBSET_DEF]);
 have `e1 IN E.events /\ e2 IN E.events`; e(tac[viewed_events_SUBSET_E_events,SUBSET_DEF]);
 have `(e1.iiid.proc = p) /\ (e2.iiid.proc = p)`; 
  e(ssimp[viewed_events_def,proc_def,reverse_SPECIFICATION]);
  e(tac[DISJOINT_events]);
 have `(e1,e2) IN vop \/ (e1,e2) IN new`; 
  e(Cases_on `SUCn`); e(ssimp[]); e(tac[IN_EMPTY]);
  e(ssimp[]);
  have `(e1,e2) IN vop \/ e1 IN viewed_events_p /\ e2 IN nes \/ (e1,e2) IN new`; e(tac[]);
  have `e1 IN COMPL mem_store /\ e2 IN COMPL mem_store`; e(ssimp[]); e(tac[DISJOINT_events]);
  e(tac[niceness_events_COMPL_mem_store_eq,less_thms,IN_DISJOINT]);
 dl(); e(tac[]);
 e(SYM_TAC ``RRESTRICT X_vop nes = new``);
 e(ssimp[RRESTRICT_def]);  
val here9 = p();

dropn 1; add(here9);
 (* check_finite_vop vop' *)
 e(ssimp[check_finite_vop_def]);
 e(intros);
 have `{ e | (e,e') IN vop' } SUBSET viewed_events_p'`;
  e(ssimp[SUBSET_DEF]);
  e(tac[linear_order_typing]);
 have `FINITE viewed_events_p'`; e(tac[FINITE_niceness_events]);
 e(tac[SUBSET_FINITE]);
val step = top_thm();

(* end step *)
(******************************************************************************)
(* combining the approximants *)

g `
valid_execution' E X 
==> p IN E.procs
==> assumption1 E
==> assumption2 E
==> 
  (let viewed_events_p = niceness_events (E,X) p n in
  let vop = po_to_partial_view_order (E,X) p n in
  wf_vop (E,X) p viewed_events_p vop)
`;
e(intros);
e(ASSUME_TAC (ISPECL [``E:event_structure``,``X:execution_witness``,``p:proc``] (MY_GEN_ALL step)));
e(Induct_on `n`);
 e(ssimp[]);
 xal ``0:num``;
 e(ssimp[LET_THM]);
 e(ssimp[base]);

 e(ssimp[]);
val wf_vop_vop = top_thm();

g `
valid_execution' E X
==> p IN E.procs
==> assumption1 E
==> assumption2 E
==> wf_vop (E,X) p (viewed_events E p) (po_to_view_order (E,X) p)
`;
e(ssimp[wf_vop_def]);
e(intros);
e(cintros);
 (******************************************************************************)
 (* linear_order (po_to_view_order (E,X) p) (viewed_events E p) *)
 e(simp[po_to_view_order_def]);
 e(QCUT_TAC `(viewed_events E p) = (BIGUNION o IMAGE (niceness_events (E,X) p)) UNIV`);
  e(ssimp[EXTENSION]);
  e(intros);
  e(RENAME_TAC [``x:event``|->``e:event``]);
  e(EQR_TAC);
   (* a: e IN viewed_events E p g: ?s. e IN s /\ ?x. !x'. x' IN s = x' IN niceness_events (E,X) p x *)
   e(QCUT_TAC `e IN E.events`); e(ssimp[viewed_events_def]);
   e(QCUT_TAC `(e IN reg_events \/ e IN mem_load) \/ e IN mem_store`); e(tac[DISJOINT_events]);
   dl();
    e(QCUT_TAC `(e.iiid.proc = p)`); e(ssimp[viewed_events_def,proc_def]); e(tac[DISJOINT_events,reverse_SPECIFICATION]);
    er ``niceness_events (E,X) p (e.iiid.program_order_index)``;
    e(ssimp[]);
    cr(); defer(); e(tac[]);
    e(QCUT_TAC `e IN COMPL mem_store`); e(ssimp[]); e(tac[DISJOINT_events]);
    e(tac[niceness_events_COMPL_mem_store_eq,less_thms]);
  
    (* e IN mem_store *)
    e(QCUT_TAC `? n. e IN niceness_events (E,X) p n`); defer(); e(tac[]);
    e(ssimp[niceness_events_def,LET_THM]);
    e(QCUT_TAC `~ (e IN reg_events)`); e(tac[DISJOINT_events]);
    e(ssimp[mem_events_eq_mem_load_UNION_mem_store]);     
    (* new goal *)
    e(QCUT_TAC `? n e'. limit_event (E,X) p n e' /\ (e,e') IN X.vo p`); defer(); e(tac[less_thms]);
    e(Cases_on `? n e'. limit1 (E,X) p n e' /\ (e,e') IN X.vo p`); e(tac[limit_event_def]);
    e(Cases_on `? n e'. limit2 (E,X) p n e' /\ (e,e') IN X.vo p`); e(tac[limit_event_def]);
    (* so e is a limit3 *)    
    (* let's calculate! *)
    e(QCUT_TAC `~ ? e'. (? n. limit1 (E,X) p n e') /\ (e,e') IN X.vo p`); e(tac[]);
    e(QCUT_TAC `? A. { e' | (? n. limit1 (E,X) p n e')} = A`); e(tac[]);
    e(elims);
    e(QCUT_TAC `FINITE A`); 
     have `A SUBSET { e' | (e',e) IN X.vo p }`;
      have `! e' n. limit1 (E,X) p n e' ==> ((e,e') IN X.vo p \/ (e',e) IN X.vo p)`; 
       e(ssimp[limit1_def]);
       have `valid_vop E X (p,X.vo p)`; e(ssimp[valid_execution'_def]);
       e(ssimp[valid_vop_def,linear_order_def] THEN NO_TAC);
      e(SYM_TAC ``{e' | ?n. limit1 (E,X) p n e'} = A``);
      e(ssimp[SUBSET_DEF]);
      e(tac[]);
     have `FINITE { e' | (e',e) IN X.vo p }`; e(ssimp[valid_execution'_def,valid_vop_def,check_finite_vop_def]);
     e(tac[SUBSET_FINITE]);
    e(QCUT_TAC `? B. IMAGE (\ e. e.iiid.program_order_index) A = B`); e(tac[]);
    e(elims);
    e(QCUT_TAC `FINITE B`); e(tac[IMAGE_FINITE]);
    e(QCUT_TAC `? m. m IN B UNION {0} /\ (! y. y IN B UNION {0} ==> y <= m)`);
     er ``MAX_SET (B UNION {0})``;
     have `FINITE (B UNION {0})`; e(ssimp[]);
     have `~ (B UNION {0} = {})`; e(ssimp[]);
     e(tac[MAX_SET_DEF]);     
    e(elims);  
    (* so beyond po m, there are no limit1s *)
    e(QCUT_TAC `! e n. limit1 (E,X) p n e ==> e.iiid.program_order_index <= m`);
     e(intros);
     have `e' IN A`; 
      e(SYM_TAC ``{e' | P e' } = A``);
      e(ssimp[]); e(tac[]);
     have `e'.iiid.program_order_index IN B`; 
      e(SYM_TAC ``IMAGE f A = B``);
      e(ssimp[]);
      e(tac[]);
     xal ``e'.iiid.program_order_index``;
     e(ssimp[]);
    e(REMOVE_VAR_TAC ``A:event set``);
    e(REMOVE_VAR_TAC ``B:num set``);
    e(TM_THIN_TAC ``~ ?e'. (?n. limit1 (E,X) p n e') /\ (e,e') IN X.vo p``);    
    e(TM_THIN_TAC ``~ ?n e'. limit1 (E,X) p n e' /\ (e,e') IN X.vo p``);
    (* now consider limit2s *)
val here = p();

dropn 1; add(here);
    e(QCUT_TAC `~ ? e'. (? n. limit2 (E,X) p n e') /\ (e,e') IN X.vo p`); e(tac[]);
    e(QCUT_TAC `? A. { e' | (? n. limit2 (E,X) p n e')} = A`); e(tac[]);
    e(elims);
    e(QCUT_TAC `FINITE A`); 
     have `A SUBSET { e' | (e',e) IN X.vo p }`;
      have `! e' n. limit2 (E,X) p n e' ==> ((e,e') IN X.vo p \/ (e',e) IN X.vo p)`; 
       e(ssimp[limit2_def]);
       have `valid_vop E X (p,X.vo p)`; e(ssimp[valid_execution'_def]);
       e(ssimp[valid_vop_def,linear_order_def] THEN NO_TAC);
      e(SYM_TAC ``{e' | ?n. limit2 (E,X) p n e'} = A``);
      e(ssimp[SUBSET_DEF]);
      e(tac[]);
     have `FINITE { e' | (e',e) IN X.vo p }`; e(ssimp[valid_execution'_def,valid_vop_def,check_finite_vop_def]);
     e(tac[SUBSET_FINITE]);
    e(QCUT_TAC `? B. IMAGE (\ e. e.iiid.program_order_index) A = B`); e(tac[]);
    e(elims);
    e(QCUT_TAC `FINITE B`); e(tac[IMAGE_FINITE]);
    e(QCUT_TAC `? m. m IN B UNION {0} /\ (! y. y IN B UNION {0} ==> y <= m)`);
     er ``MAX_SET (B UNION {0})``;
     have `FINITE (B UNION {0})`; e(ssimp[]);
     have `~ (B UNION {0} = {})`; e(ssimp[]);
     e(tac[MAX_SET_DEF]);     
    e(elims);  
    (* so beyond po m', there are no limit2s *)
    e(QCUT_TAC `! e n. limit2 (E,X) p n e ==> e.iiid.program_order_index <= m'`); 
     e(intros);
     have `e' IN A`; 
      e(SYM_TAC ``{e' | P e' } = A``);
      e(ssimp[]); e(tac[]);
     have `e'.iiid.program_order_index IN B`; 
      e(SYM_TAC ``IMAGE f A = B``);
      e(ssimp[]);
      e(tac[]);
     e(ssimp[]);
    e(REMOVE_VAR_TAC ``A:event set``);
    e(REMOVE_VAR_TAC ``B:num set``);
    e(TM_THIN_TAC ``~ ?e'. (?n. limit2 (E,X) p n e') /\ (e,e') IN X.vo p``);    
    e(TM_THIN_TAC ``~ ?n e'. limit2 (E,X) p n e' /\ (e,e') IN X.vo p``);
val here = p();

dropn 1; add(here);
    (* so beyond m+m', there are no limit1s or limit2s *)
    have `! e n. (limit1 (E,X) p n e \/ limit2 (E,X) p n e) ==> e.iiid.program_order_index <= m+m'`;
     e(ssimp[limit1_def]);
     e(tac[less_thms,arithmeticTheory.ADD_COMM]);
    e(TM_THIN_TAC ``! e n. limit1 (E,X) p n e ==> P n``);
    e(TM_THIN_TAC ``! e n. limit2 (E,X) p n e ==> P n``);
    er ``m + m' + e.iiid.program_order_index``;
    er ``e:event``; 
    cr(); defer(); e(tac[X_vop_reflexive]);
    have `limit3 (E,X) p (m + m' + e.iiid.program_order_index) e`; defer(); e(tac[limit_event_def]);
    e(ssimp[limit3_def]);
    e(tac[less_thms,arithmeticTheory.ADD_COMM]);
 
  (* g: e IN viewed_events E p *)
  e(tac[niceness_events_viewed_events,SUBSET_DEF]);
 e(ssimp[]); 
 (* FIXME INST should really instantiate types if possible *)
 e(ASSUME_TAC (ISPECL [``po_to_partial_view_order (E,X) p``,``niceness_events (E,X) p``] (MY_GEN_ALL linear_order_BIGUNION)));
 il 5;
  e(intros);
  e(RENAME_TAC [``x:num``|->``n:num``]);
  e(ASSUME_TAC wf_vop_vop);
  e(ssimp[wf_vop_def,LET_THM]);
 il 0; 
  (* SUBSET condition *) 
  e(tac[po_to_partial_view_order_mono,niceness_events_mono]);
 e(ssimp[]);
val here = p();

dropn 1; add(here);
 (******************************************************************************)
 (* RRESTRICT (po_strict E) (viewed_events E p INTER COMPL mem_store) SUBSET po_to_view_order (E,X) p *)
 e(ssimp[RRESTRICT_def,SUBSET_DEF]);
 e(intros);
 e(elims);
 e(RENAME_TAC [``x':event``|->``e1:event``,``y:event``|->``e2:event``]);
 e(ssimp[]);
 (* again, use the fact that they both appear in some niceness_events at some point, take the union of both of these, and work from there *)
 e(QCUT_TAC `e1 IN E.events /\ e2 IN E.events`); e(tac[viewed_events_SUBSET_E_events,SUBSET_DEF]);
 e(QCUT_TAC `(e1.iiid.proc = p) /\ (e2.iiid.proc = p)`); 
  e(ssimp[viewed_events_def,proc_def,reverse_SPECIFICATION]);
 e(QCUT_TAC `? n1 n2. (e1.iiid.program_order_index = n1) /\ (e2.iiid.program_order_index = n2)`); e(tac[]);
 e(elims);
 e(QCUT_TAC `e1 IN niceness_events (E,X) p n1 /\ e2 IN niceness_events (E,X) p n2`);
  e(tac[niceness_events_COMPL_mem_store_eq,IN_COMPL,less_thms]);
 e(QCUT_TAC `e1 IN niceness_events (E,X) p (n1+n2) /\ e2 IN niceness_events (E,X) p (n1+n2)`); 
  e(tac[less_thms,arithmeticTheory.ADD_COMM,niceness_events_mono,SUBSET_DEF]);
 e(QCUT_TAC `(e1,e2) IN po_to_partial_view_order (E,X) p (n1+n2)`);
  e(ASSUME_TAC (GENL [``n:num``] wf_vop_vop));
  e(ssimp[LET_THM]);
  xal ``n1+n2:num``;  
  e(ssimp[wf_vop_def]);
  e(ssimp[SUBSET_DEF]);
  xal ``(e1:event,e2:event)``;
  e(ssimp[RRESTRICT_def]);
 e(ssimp[po_to_view_order_def]);
 e(tac[]);
val here = p();

dropn 1; add(here);
 (******************************************************************************)
 (* check_atomicity' *)
 e(ssimp[check_atomicity'_def,RES_FORALL_THM]);
 e(intros);
 e(ssimp[SUBSET_DEF]);
 e(intros);
 e(ssimp[po_to_view_order_def]);
 e(RENAME_TAC [``x':num``|->``n1:num``,``x'':num``|->``n2:num``,``x''':num``|->``n3:num``]);
 e(QCUT_TAC `? l. po_to_partial_view_order (E,X) p (n1+n2+n3) = l`); e(tac[]);
 e(elims);
 e(QCUT_TAC `(e1,x) IN l /\ (x,e2) IN l /\ (e1,e2) IN l`); 
  e(tac[less_thms,po_to_partial_view_order_mono,arithmeticTheory.ADD_COMM,SUBSET_DEF]);
 e(ASSUME_TAC (GENL [``n:num``] wf_vop_vop));
 xal ``n1+n2+n3:num``;
 e(ssimp[]);
 e(ssimp[LET_THM]);
 e(ssimp[wf_vop_def]);
 e(ssimp[check_atomicity'_def,RES_FORALL_THM]);
 e(ssimp[SUBSET_DEF]);
 e(tac[]);  
val here = p();

dropn 1; add(here);
 (******************************************************************************)
 (* trivial_acyclic (po_to_view_order (E,X) p) (happens_before E X) *)
 e(ssimp[trivial_acyclic_def]);
 e(QCUT_TAC `! e1 e2. (e1,e2) IN po_to_view_order (E,X) p ==> ~ ((e2,e1) IN happens_before E X)`); defer(); e(tac[]);
 e(intros);
 e(ssimp[po_to_view_order_def]);
 e(RENAME_TAC [``x:num``|->``n:num``]);
 e(ASSUME_TAC wf_vop_vop);
 e(ssimp[LET_THM]);
 e(ssimp[wf_vop_def]);
 e(ssimp[trivial_acyclic_def]);
 e(tac[]);
val here = p();

dropn 1; add(here);
 (******************************************************************************)
 (* po_to_view_order (E,X) p INTER (mem_events CROSS mem_events) SUBSET X.vo p *)
 e(ssimp[SUBSET_DEF]);
 e(intros);
 e(QCUT_TAC `? e1 e2. x = (e1,e2)`); e(Cases_on `x` THEN tac[]);
 e(elims);
 e(ssimp[]);
 e(ssimp[po_to_view_order_def]);
 e(RENAME_TAC [``x':num``|->``n:num``]);
 e(ASSUME_TAC wf_vop_vop);
 e(ssimp[LET_THM]);
 e(ssimp[wf_vop_def]);
 e(ssimp[SUBSET_DEF]);
 xal ``(e1:event,e2:event)``;
 xal ``(e1:event,e2:event)``;
 e(ssimp[]);
val here = p();

dropn 1; add(here);
 (******************************************************************************)
 (* reg_events_agree (po_to_view_order (E,X) p) (X.vo p) *)
 e(ssimp[reg_events_agree_def]);
 e(intros);
 e(ssimp[po_to_view_order_def]);
 e(RENAME_TAC [``x:num``|->``n:num``]);
 e(ASSUME_TAC wf_vop_vop);
 e(ssimp[LET_THM]);
 e(ssimp[wf_vop_def]);
 e(ssimp[reg_events_agree_def]);
val here = p();

dropn 1; add(here);
 (******************************************************************************)
 (* check_finite_vop (po_to_view_order (E,X) p) *)
 e(ssimp[check_finite_vop_def]);
 (* this needs the fact that new events are added at the end *)
 e(SIMP_TAC (simpLib.empty_ss) [po_to_view_order_def]);
 e(simp[BIGUNION_o_IMAGE_def]);
 e(ssimp[]);
 (* !e'. FINITE {e | ?x'. (e,e') IN po_to_partial_view_order (E,X) p x'} *)
 e(intros);
 e(Cases_on `? n. e' IN niceness_events (E,X) p n`);
  e(elims);
  have `check_finite_vop (po_to_partial_view_order (E,X) p n)`;
   e(ASSUME_TAC wf_vop_vop);
   e(ssimp[wf_vop_def,LET_THM]);
  e(tac[FINITE_po_to_partial_view_order]);
 
  (* ~ ?n. e' IN niceness_events (E,X) p n *)
  have `{e | ?x'. (e,e') IN po_to_partial_view_order (E,X) p x'} = {}`;
   e(ssimp[EXTENSION]);
   e(intros);
   e(RENAME_TAC [``x'':num``|->``n:num``,``x:event``|->``e:event``]);
   xal ``n:num``;
   e(ASSUME_TAC wf_vop_vop);
   e(ssimp[wf_vop_def,LET_THM]);
   e(tac[linear_order_typing]);   
  e(ssimp[]); 
val wf_vop_po_to_view_order = top_thm();

g `
valid_execution' E X
==> p IN E.procs
==> assumption1 E
==> assumption2 E
==> (e1,e2) IN po_to_view_order (E,X) p 
==> e1 IN viewed_events E p /\ e1 IN E.events /\ e2 IN viewed_events E p /\ e2 IN E.events
`;
e(intros);
e(ASSUME_TAC wf_vop_po_to_view_order);
e(ssimp[wf_vop_def]);
e(tac[linear_order_typing,viewed_events_SUBSET_E_events,SUBSET_DEF]);
val po_to_view_order_typing = top_thm();


(* now have to do the tricky argument involving nasty *)
g `
valid_execution' E X
==> p IN E.procs
==> assumption1 E
==> assumption2 E
==> check_causality' (po_to_view_order (E,X) p) (happens_before E X)
`;
e(intros);
e(ssimp[check_causality'_def]);
e(CCONTR_TAC);
e(ASSUME_TAC wf_vop_po_to_view_order);
e(ASSUME_TAC (ISPECL [``happens_before E X``,``strict (po_to_view_order (E,X) p)``,``viewed_events E p``] (MY_GEN_ALL nasty)));
il 6;
 e(cintros);
  e(tac[acyclic_happens_before]);

  have `linear_order (po_to_view_order (E,X) p) (viewed_events E p)`;
   e(ssimp[wf_vop_def]);
  e(tac[strict_linear_order_thm]);

  e(elims);
  e(ssimp[wf_vop_def]);
  e(ssimp[trivial_acyclic_def]);
  e(QCUT_TAC `(x,y) IN (po_to_view_order (E,X) p)`); 
   e(ssimp[strict_def]);
  e(tac[]);

  e(tac[UNION_COMM]);
e(elims);
e(QCUT_TAC `l1 IN mem_events UNION locked E /\ l2 IN mem_events UNION locked E`); e(tac[happens_before_local_foreign]);
e(ssimp[wf_vop_def]);
e(TM_THIN_TAC ``RRESTRICT (po_strict E) (viewed_events E p INTER COMPL mem_store) SUBSET po_to_view_order (E,X) p``);
e(ssimp[SUBSET_DEF,pairTheory.FORALL_PROD]);
xal ``l2:event``;
xal ``l1:event``;
il 0; e(ssimp[strict_def]);
have `(l2,l1) IN (strict(X.vo p))`; e(ssimp[strict_def]);
have `(l1,l2) IN sTC(happens_before E X)`; 
 e(ssimp[IN_RTC]);
 e(tac[sTC_rules]);
have `(l1,l1) IN sTC((strict(X.vo p)) UNION (happens_before E X))`; e(tac[sTC_rules,sTC_UNION_lem,UNION_COMM]);
e(ssimp[valid_execution'_def,RES_FORALL_THM]);
xal ``p:proc``;
e(ssimp[valid_vop_def]);
e(ssimp[check_causality'_def,acyclic_def]);
val check_causality'_po_to_view_order = top_thm();

(* another little lemma to deal with register events in rfmap *)
g `
valid_execution' E X
/\ p IN E.procs
/\ assumption1 E
/\ assumption2 E
/\ ew IN reg_events /\ er IN reg_events /\ (loc ew = loc er)
/\ (ew,er) IN po_to_view_order (E,X) p
==> (ew,er) IN X.vo p
`;
e(intros);
e(elims);
have `ew IN viewed_events E p /\ ew IN E.events /\ er IN viewed_events E p /\ er IN E.events`; e(tac[po_to_view_order_typing]);
have `ew.iiid.proc = er.iiid.proc`; 
 e(ssimp[viewed_events_def,proc_def,reverse_SPECIFICATION]);
 e(tac[DISJOINT_events]);
have `ew.iiid.program_order_index <= er.iiid.program_order_index`;
 e(CCONTR_TAC);
 e(ssimp[arithmeticTheory.NOT_LESS_EQUAL]);
 e(ASSUME_TAC wf_vop_po_to_view_order);
 e(ssimp[wf_vop_def]);
 have `(er,ew) IN po_to_view_order (E,X) p`; 
  e(ssimp[SUBSET_DEF,pairTheory.FORALL_PROD]);
  e(ssimp[RRESTRICT_def]);
  e(ssimp[po_strict_def]);
  e(tac[DISJOINT_events]);
 have `(ew,er) IN po_to_view_order (E,X) p`; e(INIT_TAC);
 have `ew = er`;
  e(ssimp[linear_order_def]);
 e(ssimp[]);
e(ssimp[less_thms]);
dl();
 (* < *)
 have `(ew,er) IN preserved_program_order E`;
  e(ssimp[preserved_program_order_def,po_strict_def,reverse_SPECIFICATION]);
  e(Cases_on `er`); 
  e(ssimp[reg_events_def,loc_def]);
  e(Cases_on `a`); 
  e(ssimp[]);
  e(Cases_on `l` THEN ssimp[] THEN NO_TAC); 
 have `(ew,er) IN happens_before E X`; e(ssimp[happens_before_def]);       
 e(tac[happens_before_X_vop]);

 (* = *)
 e(ASSUME_TAC wf_vop_po_to_view_order);
 e(ssimp[wf_vop_def]);    
 e(ssimp[reg_events_agree_def]);  
val reg_events_po_to_view_order_X_vop = top_thm();

g `
! E X. ? X'. 
valid_execution' E X 
/\ assumption1 E
/\ assumption2 E
==> 
valid_execution' E X' 
/\ (X' with <| vo:=(\ p. {}) |> = X with <| vo:=(\ p. {}) |>) 
/\ nice_execution E X'
`;
e(intros);
er ``X with <| vo:=(\ p. if p IN E.procs then po_to_view_order (E,X) p else {}) |>``;
e(intros);
e(elims);
e(ASSUME_TAC (GEN ``p:proc`` wf_vop_po_to_view_order));
e(ASSUME_TAC (GEN ``p:proc`` check_causality'_po_to_view_order));
e(ssimp[]);
cr();
 (* valid_execution' E (X with vo := (\p. po_to_view_order (E,X) p)) *)
 have `valid_execution' E X`; e(INIT_TAC);
 e(ASM_CONV_TAC (fn ths => SIMP_CONV (srw_ss()) [valid_execution'_def,RES_FORALL_THM]) 0);
 e(SIMP_TAC (srw_ss()) [valid_execution'_def,RES_FORALL_THM]);
 e(ssimp[]);
 e(intros);
 xal ``p:proc``;
 xal ``p:proc``;
 xal ``p:proc``;
 xal ``p:proc``;
 e(ssimp[valid_vop_def]);
 e(cintros);
  (* 7 subgoals *)
  e(ssimp[wf_vop_def]);
  e(tac[linear_order_FIELD]);

  e(ssimp[wf_vop_def]);
  e(tac[linear_order_FIELD]);

  e(ssimp[wf_vop_def]);  

  have `(happens_before E (X with vo := (\p. if p IN E.procs then po_to_view_order (E,X) p else {}))) = happens_before E X`;
   e(ssimp[happens_before_def]);
  e(ssimp[]);
val here = p();

dropn 1; add(here);
  (* check_rfmap_initial' (viewed_events E p) (po_to_view_order (E,X) p) X.rfmap X.initial_state *)
  (* need to show that the order of reg events for a given instruction is the same in X'.vo p as in X *)
  (* also need to show that no intermediate reg events, and initial state is read ok  *)
  (* FIXME really want to remove all quant rescoping from the default simpset *)
  e(ssimp[check_rfmap_initial'_def,RES_FORALL_THM]);
  e(intros);
  e(RENAME_TAC [``ew':event``|->``ew:event``]);
  e(CCONTR_TAC);
  e(ssimp[]);
  have `ew IN viewed_events E p /\ ew IN E.events /\ er IN viewed_events E p /\ er IN E.events`; e(tac[po_to_view_order_typing]);
  (* check that no write comes before the read *)
  xal ``er:event``;
  e(ssimp[]);
  xal ``ew:event``;
  e(ssimp[]);
  have `(ew IN mem_events /\ er IN mem_events) \/ (ew IN reg_events /\ er IN reg_events)`; 
   e(tac[loc_eq_loc_mem_events_reg_events,DISJOINT_events]);
  dl();
   e(ssimp[wf_vop_def]);
   e(ssimp[SUBSET_DEF,pairTheory.FORALL_PROD] THEN NO_TAC);

   (* (ew IN reg_events /\ er IN reg_events) *)
   e(tac[reg_events_po_to_view_order_X_vop]);
val here2 = p();

dropn 1; add(here2);
  (* check_rfmap_written' (po_to_view_order (E,X) p) X.rfmap *)
  e(ssimp[check_rfmap_written'_def,RES_FORALL_THM,pairTheory.FORALL_PROD]);
  e(intros);     
  e(RENAME_TAC [``p_1:event``|->``ew:event``,``p_2:event``|->``er:event``]);
  e(elims);
  xal ``ew:event``;
  xal ``er:event``;
  e(ssimp[]);
  xal ``ew':event``;
  e(ssimp[]);
  e(QCUT_TAC `(ew,ew') IN X.vo p /\ (ew',er) IN X.vo p`); defer(); e(tac[]);
  have `ew IN viewed_events E p /\ ew IN E.events /\ ew' IN viewed_events E p /\ ew' IN E.events /\ er IN viewed_events E p /\ er IN E.events`; 
   e(tac[po_to_view_order_typing]);
  have `loc er = loc ew`; e(tac[rfmap_typing]);
  have `(ew IN mem_events /\ ew' IN mem_events /\ er IN mem_events) \/ (ew IN reg_events /\ ew' IN reg_events /\ er IN reg_events)`; 
   e(tac[loc_eq_loc_mem_events_reg_events,DISJOINT_events]);
  dl();
   e(ssimp[wf_vop_def]);
   e(ssimp[SUBSET_DEF,pairTheory.FORALL_PROD] THEN NO_TAC);

   (* (ew IN reg_events /\ er IN reg_events) *)
   e(tac[reg_events_po_to_view_order_X_vop]);
val here3 = p();

dropn 1; add(here3);
  (* check_finite_vop (po_to_view_order (E,X) p) *)
  e(ssimp[wf_vop_def]);

 (* nice_execution E (X with vo := (\p. po_to_view_order (E,X) p)) *)
 e(ssimp[nice_execution_def,RES_FORALL_THM]);
 e(intros); 
 e(ssimp[wf_vop_def,INTER_COMPL]);
val niceness' = top_thm();

(* FIXME should really remove assumption1 and 2s from previous proofs *)
g `
valid_execution' E X ==> assumption1 E /\ assumption2 E
`;
e(intros);
e(ssimp[valid_execution'_def,wf_E_def,assumption1_def,assumption2_def]);
cr(); 
 e(ssimp[wf_intra_causality_def,reverse_SPECIFICATION,RES_FORALL_THM,pairTheory.FORALL_PROD]);
 e(tac[]);

 e(ssimp[wf_atomicity_def]);
 e(ssimp[RES_FORALL_THM,RES_EXISTS_THM,reverse_SPECIFICATION]);
val valid_execution'_assumption12 = top_thm(); 

g `
! E X. ? X'. 
valid_execution' E X 
==> 
valid_execution' E X' 
/\ (X' with <| vo:=(\ p. {}) |> = X with <| vo:=(\ p. {}) |>) 
/\ nice_execution E X'
`;
e(tac[niceness',valid_execution'_assumption12]);
val niceness'' = top_thm();

g `
! E X. ? X'. 
well_formed_event_structure E
==> valid_execution E X 
==> 
valid_execution E X' 
/\ (X' with <| vo:=(\ p. {}) |> = X with <| vo:=(\ p. {}) |>) 
/\ nice_execution E X'
`;
e(tac[niceness'',valid_execution'_eq_valid_execution]);
val niceness = top_thm();

g `
niceness_thm
`;
e(tac[niceness_thm_def,niceness]);
val niceness_thm = top_thm();

(* end niceness *)
(******************************************************************************)



val _ = export_theory ();
